"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitTemplate = void 0;
exports.cliInit = cliInit;
exports.expandPlaceholders = expandPlaceholders;
exports.availableInitTemplates = availableInitTemplates;
exports.availableInitLanguages = availableInitLanguages;
exports.printAvailableTemplates = printAvailableTemplates;
exports.currentlyRecommendedAwsCdkLibFlags = currentlyRecommendedAwsCdkLibFlags;
const childProcess = require("child_process");
const path = require("path");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const chalk = require("chalk");
const fs = require("fs-extra");
const init_hooks_1 = require("./init-hooks");
const root_dir_1 = require("../../cli/root-dir");
const version_1 = require("../../cli/version");
const util_1 = require("../../util");
const language_1 = require("../language");
const package_manager_1 = require("./package-manager");
/* eslint-disable @typescript-eslint/no-var-requires */ // Packages don't have @types module
// eslint-disable-next-line @typescript-eslint/no-require-imports
const camelCase = require('camelcase');
// eslint-disable-next-line @typescript-eslint/no-require-imports
const decamelize = require('decamelize');
const SUPPORTED_LANGUAGE_NAMES = language_1.SUPPORTED_LANGUAGES.map((l) => l.name);
/**
 * Initialize a CDK package in the current directory
 */
async function cliInit(options) {
    await ensureValidCliInitOptions(options, options.ioHelper);
    const ioHelper = options.ioHelper;
    const canUseNetwork = options.canUseNetwork ?? true;
    const generateOnly = options.generateOnly ?? false;
    const workDir = options.workDir ?? process.cwd();
    // Show available templates only if no fromPath, type, or language provided
    if (!options.fromPath && !options.type && !options.language) {
        await printAvailableTemplates(ioHelper);
        return;
    }
    // Step 1: Load template
    let template;
    if (options.fromPath) {
        template = await loadLocalTemplate(options.fromPath, options.templatePath);
    }
    else {
        template = await loadBuiltinTemplate(ioHelper, options.type, options.language);
    }
    // Step 2: Resolve language
    const language = await resolveLanguage(ioHelper, template, options.language, options.type);
    // Step 3: Initialize project following standard process
    await initializeProject(ioHelper, template, language, canUseNetwork, generateOnly, workDir, options.projectName, options.stackName, options.migrate, options.libVersion, options.packageManager);
}
/**
 * Validate CLI init options and handle invalid or incompatible option combinations
 */
async function ensureValidCliInitOptions(options, ioHelper) {
    if (options.packageManager && !['javascript', 'typescript'].includes(options.language ?? '')) {
        await ioHelper.defaults.warn(`--package-manager option is only applicable for JavaScript and TypeScript projects. Ignoring the provided value: ${options.packageManager}`);
    }
}
/**
 * Load a local custom template from file system path
 * @param fromPath - Path to the local template directory or multi-template repository
 * @param templatePath - Optional path to a specific template within a multi-template repository
 * @returns Promise resolving to the loaded InitTemplate
 */
async function loadLocalTemplate(fromPath, templatePath) {
    try {
        let actualTemplatePath = fromPath;
        // If templatePath is provided, it's a multi-template repository
        if (templatePath) {
            actualTemplatePath = path.join(fromPath, templatePath);
            if (!await fs.pathExists(actualTemplatePath)) {
                throw new toolkit_lib_1.ToolkitError(`Template path does not exist: ${actualTemplatePath}`);
            }
        }
        const template = await InitTemplate.fromPath(actualTemplatePath);
        if (template.languages.length === 0) {
            // Check if this might be a multi-template repository
            if (!templatePath) {
                const availableTemplates = await findPotentialTemplates(fromPath);
                if (availableTemplates.length > 0) {
                    throw new toolkit_lib_1.ToolkitError('Use --template-path to specify which template to use.');
                }
            }
            throw new toolkit_lib_1.ToolkitError('Custom template must contain at least one language directory');
        }
        return template;
    }
    catch (error) {
        const displayPath = templatePath ? `${fromPath}/${templatePath}` : fromPath;
        throw new toolkit_lib_1.ToolkitError(`Failed to load template from path: ${displayPath}. ${error.message}`);
    }
}
/**
 * Load a built-in template by name
 */
async function loadBuiltinTemplate(ioHelper, type, language) {
    const templateType = type || 'default'; // "default" is the default type (and maps to "app")
    const template = (await availableInitTemplates()).find((t) => t.hasName(templateType));
    if (!template) {
        await printAvailableTemplates(ioHelper, language);
        throw new toolkit_lib_1.ToolkitError(`Unknown init template: ${templateType}`);
    }
    return template;
}
/**
 * Resolve the programming language for the template
 * @param ioHelper - IO helper for user interaction
 * @param template - The template to resolve language for
 * @param requestedLanguage - User-requested language (optional)
 * @param type - The template type name for messages
 * @default undefined
 * @returns Promise resolving to the selected language
 */
async function resolveLanguage(ioHelper, template, requestedLanguage, type) {
    return (async () => {
        if (requestedLanguage) {
            return requestedLanguage;
        }
        if (template.languages.length === 1) {
            const templateLanguage = template.languages[0];
            // Only show auto-detection message for built-in templates
            if (template.templateType !== TemplateType.CUSTOM) {
                await ioHelper.defaults.warn(`No --language was provided, but '${type || template.name}' supports only '${templateLanguage}', so defaulting to --language=${templateLanguage}`);
            }
            return templateLanguage;
        }
        await ioHelper.defaults.info(`Available languages for ${chalk.green(type || template.name)}: ${template.languages.map((l) => chalk.blue(l)).join(', ')}`);
        throw new toolkit_lib_1.ToolkitError('No language was selected');
    })();
}
/**
 * Find potential template directories in a multi-template repository
 * @param repositoryPath - Path to the repository root
 * @returns Promise resolving to array of potential template directory names
 */
async function findPotentialTemplates(repositoryPath) {
    const entries = await fs.readdir(repositoryPath, { withFileTypes: true });
    const templateValidationPromises = entries
        .filter(entry => entry.isDirectory() && !entry.name.startsWith('.'))
        .map(async (entry) => {
        try {
            const templatePath = path.join(repositoryPath, entry.name);
            const { languages } = await getLanguageDirectories(templatePath);
            return languages.length > 0 ? entry.name : null;
        }
        catch (error) {
            // If we can't read a specific template directory, skip it but don't fail the entire operation
            return null;
        }
    });
    /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Limited to directory entries
    const validationResults = await Promise.all(templateValidationPromises);
    return validationResults.filter((templateName) => templateName !== null);
}
/**
 * Get valid CDK language directories from a template path
 * @param templatePath - Path to the template directory
 * @returns Promise resolving to array of supported language names
 */
/**
 * Get valid CDK language directories from a template path
 * @param templatePath - Path to the template directory
 * @returns Promise resolving to array of supported language names and directory entries
 * @throws ToolkitError if directory cannot be read or validated
 */
async function getLanguageDirectories(templatePath) {
    try {
        const entries = await fs.readdir(templatePath, { withFileTypes: true });
        const languageValidationPromises = entries
            .filter(directoryEntry => directoryEntry.isDirectory() && SUPPORTED_LANGUAGE_NAMES.includes(directoryEntry.name))
            .map(async (directoryEntry) => {
            const languageDirectoryPath = path.join(templatePath, directoryEntry.name);
            try {
                const hasValidLanguageFiles = await hasLanguageFiles(languageDirectoryPath, (0, language_1.getLanguageExtensions)(directoryEntry.name));
                return hasValidLanguageFiles ? directoryEntry.name : null;
            }
            catch (error) {
                throw new toolkit_lib_1.ToolkitError(`Cannot read language directory '${directoryEntry.name}': ${error.message}`);
            }
        });
        /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Limited to supported CDK languages (7 max)
        const validationResults = await Promise.all(languageValidationPromises);
        return {
            languages: validationResults.filter((languageName) => languageName !== null),
            entries,
        };
    }
    catch (error) {
        throw new toolkit_lib_1.ToolkitError(`Cannot read template directory '${templatePath}': ${error.message}`);
    }
}
/**
 * Iteratively check if a directory contains files with the specified extensions
 * @param directoryPath - Path to search for language files
 * @param extensions - Array of file extensions to look for
 * @returns Promise resolving to true if language files are found
 */
async function hasLanguageFiles(directoryPath, extensions) {
    const dirsToCheck = [directoryPath];
    while (dirsToCheck.length > 0) {
        const currentDir = dirsToCheck.pop();
        try {
            const entries = await fs.readdir(currentDir, { withFileTypes: true });
            for (const entry of entries) {
                if (entry.isFile() && extensions.some(ext => entry.name.endsWith(ext))) {
                    return true;
                }
                else if (entry.isDirectory()) {
                    dirsToCheck.push(path.join(currentDir, entry.name));
                }
            }
        }
        catch (error) {
            throw error;
        }
    }
    return false;
}
/**
 * Returns the name of the Python executable for this OS
 * @returns The Python executable name for the current platform
 */
function pythonExecutable() {
    let python = 'python3';
    if (process.platform === 'win32') {
        python = 'python';
    }
    return python;
}
const INFO_DOT_JSON = 'info.json';
var TemplateType;
(function (TemplateType) {
    TemplateType["BUILT_IN"] = "builtin";
    TemplateType["CUSTOM"] = "custom";
})(TemplateType || (TemplateType = {}));
class InitTemplate {
    static async fromName(templatesDir, name) {
        const basePath = path.join(templatesDir, name);
        const languages = await listDirectory(basePath);
        const initInfo = await fs.readJson(path.join(basePath, INFO_DOT_JSON));
        return new InitTemplate(basePath, name, languages, initInfo, TemplateType.BUILT_IN);
    }
    static async fromPath(templatePath) {
        const basePath = path.resolve(templatePath);
        if (!await fs.pathExists(basePath)) {
            throw new toolkit_lib_1.ToolkitError(`Template path does not exist: ${basePath}`);
        }
        let templateSourcePath = basePath;
        let { languages, entries } = await getLanguageDirectories(basePath);
        if (languages.length === 0) {
            const languageDirs = entries.filter(entry => entry.isDirectory() &&
                SUPPORTED_LANGUAGE_NAMES.includes(entry.name));
            if (languageDirs.length === 1) {
                // Validate that the language directory contains appropriate files
                const langDir = languageDirs[0].name;
                templateSourcePath = path.join(basePath, langDir);
                const hasValidFiles = await hasLanguageFiles(templateSourcePath, (0, language_1.getLanguageExtensions)(langDir));
                if (!hasValidFiles) {
                    // If we found a language directory but it doesn't contain valid files, we should inform the user
                    throw new toolkit_lib_1.ToolkitError(`Found '${langDir}' directory but it doesn't contain the expected language files. Ensure the template contains ${langDir} source files.`);
                }
            }
        }
        const name = path.basename(basePath);
        return new InitTemplate(templateSourcePath, name, languages, null, TemplateType.CUSTOM);
    }
    constructor(basePath, name, languages, initInfo, templateType) {
        this.basePath = basePath;
        this.name = name;
        this.languages = languages;
        this.aliases = new Set();
        this.templateType = templateType;
        // Only built-in templates have descriptions and aliases from info.json
        if (templateType === TemplateType.BUILT_IN && initInfo) {
            this.description = initInfo.description;
            for (const alias of initInfo.aliases || []) {
                this.aliases.add(alias);
            }
        }
    }
    /**
     * @param name - the name that is being checked
     * @returns ``true`` if ``name`` is the name of this template or an alias of it.
     */
    hasName(name) {
        return name === this.name || this.aliases.has(name);
    }
    /**
     * Creates a new instance of this ``InitTemplate`` for a given language to a specified folder.
     *
     * @param language - the language to instantiate this template with
     * @param targetDirectory - the directory where the template is to be instantiated into
     * @param stackName - the name of the stack to create
     * @default undefined
     * @param libVersion - the version of the CDK library to use
     * @default undefined
     */
    async install(ioHelper, language, targetDirectory, projectName, stackName, libVersion, packageManager) {
        if (this.languages.indexOf(language) === -1) {
            await ioHelper.defaults.error(`The ${chalk.blue(language)} language is not supported for ${chalk.green(this.name)} ` +
                `(it supports: ${this.languages.map((l) => chalk.blue(l)).join(', ')})`);
            throw new toolkit_lib_1.ToolkitError(`Unsupported language: ${language}`);
        }
        const projectInfo = {
            name: projectName ? decamelize(projectName) : decamelize(path.basename(path.resolve(targetDirectory))),
            stackName,
            versions: await loadInitVersions(),
        };
        if (libVersion) {
            projectInfo.versions['aws-cdk-lib'] = libVersion;
        }
        let sourceDirectory = path.join(this.basePath, language);
        // For auto-detected single language templates, use basePath directly
        if (this.templateType === TemplateType.CUSTOM && this.languages.length === 1 &&
            path.basename(this.basePath) === language) {
            sourceDirectory = this.basePath;
        }
        if (this.templateType === TemplateType.CUSTOM) {
            // For custom templates, copy files without processing placeholders
            await this.installFilesWithoutProcessing(sourceDirectory, targetDirectory);
        }
        else {
            // For built-in templates, process placeholders as usual
            await this.installFiles(sourceDirectory, targetDirectory, language, projectInfo, packageManager);
            await this.applyFutureFlags(targetDirectory);
            await (0, init_hooks_1.invokeBuiltinHooks)(ioHelper, { targetDirectory, language, templateName: this.name }, {
                substitutePlaceholdersIn: async (...fileNames) => {
                    const fileProcessingPromises = fileNames.map(async (fileName) => {
                        const fullPath = path.join(targetDirectory, fileName);
                        const template = await fs.readFile(fullPath, { encoding: 'utf-8' });
                        await fs.writeFile(fullPath, expandPlaceholders(template, language, projectInfo, packageManager));
                    });
                    /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Processing a small, known set of template files
                    await Promise.all(fileProcessingPromises);
                },
                placeholder: (ph) => expandPlaceholders(`%${ph}%`, language, projectInfo, packageManager),
            });
        }
    }
    async installFiles(sourceDirectory, targetDirectory, language, project, packageManager) {
        for (const file of await fs.readdir(sourceDirectory)) {
            const fromFile = path.join(sourceDirectory, file);
            const toFile = path.join(targetDirectory, expandPlaceholders(file, language, project, packageManager));
            if ((await fs.stat(fromFile)).isDirectory()) {
                await fs.mkdir(toFile);
                await this.installFiles(fromFile, toFile, language, project, packageManager);
                continue;
            }
            else if (file.match(/^.*\.template\.[^.]+$/)) {
                await this.installProcessed(fromFile, toFile.replace(/\.template(\.[^.]+)$/, '$1'), language, project, packageManager);
                continue;
            }
            else if (file.match(/^.*\.hook\.(d.)?[^.]+$/)) {
                // Ignore
                continue;
            }
            else {
                await fs.copy(fromFile, toFile);
            }
        }
    }
    async installProcessed(templatePath, toFile, language, project, packageManager) {
        const template = await fs.readFile(templatePath, { encoding: 'utf-8' });
        await fs.writeFile(toFile, expandPlaceholders(template, language, project, packageManager));
    }
    /**
     * Copy template files without processing placeholders (for custom templates)
     */
    async installFilesWithoutProcessing(sourceDirectory, targetDirectory) {
        await fs.copy(sourceDirectory, targetDirectory, {
            filter: (src) => {
                const filename = path.basename(src);
                return !filename.match(/^.*\.hook\.(d.)?[^.]+$/);
            },
        });
    }
    /**
     * Adds context variables to `cdk.json` in the generated project directory to
     * enable future behavior for new projects.
     */
    async applyFutureFlags(projectDir) {
        const cdkJson = path.join(projectDir, 'cdk.json');
        if (!(await fs.pathExists(cdkJson))) {
            return;
        }
        const config = await fs.readJson(cdkJson);
        config.context = {
            ...config.context,
            ...await currentlyRecommendedAwsCdkLibFlags(),
        };
        await fs.writeJson(cdkJson, config, { spaces: 2 });
    }
    async addMigrateContext(projectDir) {
        const cdkJson = path.join(projectDir, 'cdk.json');
        if (!(await fs.pathExists(cdkJson))) {
            return;
        }
        const config = await fs.readJson(cdkJson);
        config.context = {
            ...config.context,
            'cdk-migrate': true,
        };
        await fs.writeJson(cdkJson, config, { spaces: 2 });
    }
}
exports.InitTemplate = InitTemplate;
function expandPlaceholders(template, language, project, packageManager) {
    const cdkCliVersion = project.versions['aws-cdk'];
    let cdkVersion = project.versions['aws-cdk-lib'];
    let constructsVersion = project.versions.constructs;
    switch (language) {
        case 'java':
        case 'csharp':
        case 'fsharp':
            cdkVersion = (0, util_1.rangeFromSemver)(cdkVersion, 'bracket');
            constructsVersion = (0, util_1.rangeFromSemver)(constructsVersion, 'bracket');
            break;
        case 'python':
            cdkVersion = (0, util_1.rangeFromSemver)(cdkVersion, 'pep');
            constructsVersion = (0, util_1.rangeFromSemver)(constructsVersion, 'pep');
            break;
        case 'go':
            cdkVersion = (0, util_1.stripCaret)(cdkVersion);
            constructsVersion = (0, util_1.stripCaret)(constructsVersion);
            break;
    }
    return template
        .replace(/%name%/g, project.name)
        .replace(/%stackname%/, project.stackName ?? '%name.PascalCased%Stack')
        .replace(/%PascalNameSpace%/, project.stackName ? camelCase(project.stackName + 'Stack', { pascalCase: true }) : '%name.PascalCased%')
        .replace(/%PascalStackProps%/, project.stackName ? camelCase(project.stackName, { pascalCase: true }) + 'StackProps' : 'StackProps')
        .replace(/%name\.camelCased%/g, camelCase(project.name))
        .replace(/%name\.PascalCased%/g, camelCase(project.name, { pascalCase: true }))
        .replace(/%cdk-version%/g, cdkVersion)
        .replace(/%cdk-cli-version%/g, cdkCliVersion)
        .replace(/%constructs-version%/g, constructsVersion)
        .replace(/%cdk-home%/g, (0, util_1.cdkHomeDir)())
        .replace(/%name\.PythonModule%/g, project.name.replace(/-/g, '_'))
        .replace(/%python-executable%/g, pythonExecutable())
        .replace(/%name\.StackName%/g, project.name.replace(/[^A-Za-z0-9-]/g, '-'))
        .replace(/%pm-cmd%/g, (0, package_manager_1.getPmCmdPrefix)(packageManager ?? 'npm'));
}
async function availableInitTemplates() {
    try {
        const templatesDir = path.join((0, root_dir_1.cliRootDir)(), 'lib', 'init-templates');
        const templateNames = await listDirectory(templatesDir);
        const templatePromises = templateNames.map(templateName => InitTemplate.fromName(templatesDir, templateName));
        /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Built-in templates are limited in number
        return await Promise.all(templatePromises);
    }
    catch (error) {
        // Return empty array if templates directory doesn't exist or can't be read
        // This allows the CLI to gracefully handle missing built-in templates
        if (error.code === 'ENOENT' || error.code === 'EACCES') {
            return [];
        }
        throw error;
    }
}
async function availableInitLanguages() {
    const templates = await availableInitTemplates();
    const result = new Set();
    for (const template of templates) {
        for (const language of template.languages) {
            const alias = (0, language_1.getLanguageAlias)(language);
            result.add(language);
            alias && result.add(alias);
        }
    }
    return [...result];
}
/**
 * @param dirPath - is the directory to be listed.
 * @returns the list of file or directory names contained in ``dirPath``, excluding any dot-file, and sorted.
 */
async function listDirectory(dirPath) {
    return ((await fs.readdir(dirPath))
        .filter((p) => !p.startsWith('.'))
        .filter((p) => !(p === 'LICENSE'))
        // if, for some reason, the temp folder for the hook doesn't get deleted we don't want to display it in this list
        .filter((p) => !(p === INFO_DOT_JSON))
        .sort());
}
/**
 * Print available templates to the user
 * @param ioHelper - IO helper for user interaction
 * @param language - Programming language filter
 * @default undefined
 */
async function printAvailableTemplates(ioHelper, language) {
    await ioHelper.defaults.info('Available templates:');
    for (const template of await availableInitTemplates()) {
        if (language && template.languages.indexOf(language) === -1) {
            continue;
        }
        await ioHelper.defaults.info(`* ${chalk.green(template.name)}: ${template.description}`);
        const languageArg = language
            ? chalk.bold(language)
            : template.languages.length > 1
                ? `[${template.languages.map((t) => chalk.bold(t)).join('|')}]`
                : chalk.bold(template.languages[0]);
        await ioHelper.defaults.info(`   └─ ${chalk.blue(`cdk init ${chalk.bold(template.name)} --language=${languageArg}`)}`);
    }
}
async function initializeProject(ioHelper, template, language, canUseNetwork, generateOnly, workDir, projectName, stackName, migrate, cdkVersion, packageManager) {
    // Step 1: Ensure target directory is empty
    await assertIsEmptyDirectory(workDir);
    // Step 2: Copy template files
    await ioHelper.defaults.info(`Applying project template ${chalk.green(template.name)} for ${chalk.blue(language)}`);
    await template.install(ioHelper, language, workDir, projectName, stackName, cdkVersion, packageManager);
    if (migrate) {
        await template.addMigrateContext(workDir);
    }
    if (await fs.pathExists(`${workDir}/README.md`)) {
        const readme = await fs.readFile(`${workDir}/README.md`, { encoding: 'utf-8' });
        await ioHelper.defaults.info(chalk.green(readme));
    }
    if (!generateOnly) {
        // Step 3: Initialize Git repository and create initial commit
        await initializeGitRepository(ioHelper, workDir);
        // Step 4: Post-install steps
        await postInstall(ioHelper, language, canUseNetwork, workDir, packageManager);
    }
    await ioHelper.defaults.info('✅ All done!');
}
/**
 * Validate that a directory exists and is empty (ignoring hidden files)
 * @param workDir - Directory path to validate
 * @throws ToolkitError if directory doesn't exist or is not empty
 */
async function assertIsEmptyDirectory(workDir) {
    try {
        const stats = await fs.stat(workDir);
        if (!stats.isDirectory()) {
            throw new toolkit_lib_1.ToolkitError(`Path exists but is not a directory: ${workDir}`);
        }
        const files = await fs.readdir(workDir);
        const visibleFiles = files.filter(f => !f.startsWith('.'));
        if (visibleFiles.length > 0) {
            throw new toolkit_lib_1.ToolkitError('`cdk init` cannot be run in a non-empty directory!\n' +
                `Found ${visibleFiles.length} visible files in ${workDir}:\n` +
                visibleFiles.map(f => `  - ${f}`).join('\n'));
        }
    }
    catch (e) {
        if (e.code === 'ENOENT') {
            throw new toolkit_lib_1.ToolkitError(`Directory does not exist: ${workDir}\n` +
                'Please create the directory first using: mkdir -p ' + workDir);
        }
        throw new toolkit_lib_1.ToolkitError(`Failed to validate directory ${workDir}: ${e.message}`);
    }
}
async function initializeGitRepository(ioHelper, workDir) {
    if (await isInGitRepository(workDir)) {
        return;
    }
    await ioHelper.defaults.info('Initializing a new git repository...');
    try {
        await execute(ioHelper, 'git', ['init'], { cwd: workDir });
        await execute(ioHelper, 'git', ['add', '.'], { cwd: workDir });
        await execute(ioHelper, 'git', ['commit', '--message="Initial commit"', '--no-gpg-sign'], { cwd: workDir });
    }
    catch {
        await ioHelper.defaults.warn('Unable to initialize git repository for your project.');
    }
}
async function postInstall(ioHelper, language, canUseNetwork, workDir, packageManager) {
    switch (language) {
        case 'javascript':
            return postInstallJavascript(ioHelper, canUseNetwork, workDir, packageManager);
        case 'typescript':
            return postInstallTypescript(ioHelper, canUseNetwork, workDir, packageManager);
        case 'java':
            return postInstallJava(ioHelper, canUseNetwork, workDir);
        case 'python':
            return postInstallPython(ioHelper, workDir);
        case 'go':
            return postInstallGo(ioHelper, canUseNetwork, workDir);
        case 'csharp':
            return postInstallCSharp(ioHelper, canUseNetwork, workDir);
        case 'fsharp':
            return postInstallFSharp(ioHelper, canUseNetwork, workDir);
    }
}
async function postInstallJavascript(ioHelper, canUseNetwork, cwd, packageManager) {
    return postInstallTypescript(ioHelper, canUseNetwork, cwd, packageManager);
}
async function postInstallTypescript(ioHelper, canUseNetwork, cwd, packageManager) {
    const command = packageManager ?? 'npm';
    if (!canUseNetwork) {
        await ioHelper.defaults.warn(`Please run '${command} install'!`);
        return;
    }
    await ioHelper.defaults.info(`Executing ${chalk.green(`${command} install`)}...`);
    try {
        await execute(ioHelper, command, ['install'], { cwd });
    }
    catch (e) {
        await ioHelper.defaults.warn(`${command} install failed: ` + (0, util_1.formatErrorMessage)(e));
    }
}
async function postInstallJava(ioHelper, canUseNetwork, cwd) {
    // Check if this is a Gradle or Maven project
    const hasGradleBuild = await fs.pathExists(path.join(cwd, 'build.gradle'));
    const hasMavenPom = await fs.pathExists(path.join(cwd, 'pom.xml'));
    if (hasGradleBuild) {
        // Gradle project
        const gradleWarning = "Please run './gradlew build'!";
        if (!canUseNetwork) {
            await ioHelper.defaults.warn(gradleWarning);
            return;
        }
        await ioHelper.defaults.info("Executing './gradlew build'");
        try {
            await execute(ioHelper, './gradlew', ['build'], { cwd });
        }
        catch {
            await ioHelper.defaults.warn('Unable to build Gradle project');
            await ioHelper.defaults.warn(gradleWarning);
        }
    }
    else if (hasMavenPom) {
        // Maven project
        const mvnPackageWarning = "Please run 'mvn package'!";
        if (!canUseNetwork) {
            await ioHelper.defaults.warn(mvnPackageWarning);
            return;
        }
        await ioHelper.defaults.info("Executing 'mvn package'");
        try {
            await execute(ioHelper, 'mvn', ['package'], { cwd });
        }
        catch {
            await ioHelper.defaults.warn('Unable to package compiled code as JAR');
            await ioHelper.defaults.warn(mvnPackageWarning);
        }
    }
    else {
        // No recognized build file
        await ioHelper.defaults.warn('No build.gradle or pom.xml found. Please set up your build system manually.');
    }
}
async function postInstallPython(ioHelper, cwd) {
    const python = pythonExecutable();
    // Check if requirements.txt exists
    const hasRequirements = await fs.pathExists(path.join(cwd, 'requirements.txt'));
    if (hasRequirements) {
        await ioHelper.defaults.info(`Executing ${chalk.green('Creating virtualenv...')}`);
        try {
            await execute(ioHelper, python, ['-m', 'venv', '.venv'], { cwd });
            await ioHelper.defaults.info(`Executing ${chalk.green('Installing dependencies...')}`);
            // Install dependencies in the virtual environment
            const pipPath = process.platform === 'win32' ? '.venv\\Scripts\\pip' : '.venv/bin/pip';
            await execute(ioHelper, pipPath, ['install', '-r', 'requirements.txt'], { cwd });
        }
        catch {
            await ioHelper.defaults.warn('Unable to create virtualenv or install dependencies automatically');
            await ioHelper.defaults.warn(`Please run '${python} -m venv .venv && .venv/bin/pip install -r requirements.txt'!`);
        }
    }
    else {
        await ioHelper.defaults.warn('No requirements.txt found. Please set up your Python environment manually.');
    }
}
async function postInstallGo(ioHelper, canUseNetwork, cwd) {
    if (!canUseNetwork) {
        await ioHelper.defaults.warn('Please run \'go mod tidy\'!');
        return;
    }
    await ioHelper.defaults.info(`Executing ${chalk.green('go mod tidy')}...`);
    try {
        await execute(ioHelper, 'go', ['mod', 'tidy'], { cwd });
    }
    catch (e) {
        await ioHelper.defaults.warn('\'go mod tidy\' failed: ' + (0, util_1.formatErrorMessage)(e));
    }
}
async function postInstallCSharp(ioHelper, canUseNetwork, cwd) {
    const dotnetWarning = "Please run 'dotnet restore && dotnet build'!";
    if (!canUseNetwork) {
        await ioHelper.defaults.warn(dotnetWarning);
        return;
    }
    await ioHelper.defaults.info(`Executing ${chalk.green('dotnet restore')}...`);
    try {
        await execute(ioHelper, 'dotnet', ['restore'], { cwd });
        await ioHelper.defaults.info(`Executing ${chalk.green('dotnet build')}...`);
        await execute(ioHelper, 'dotnet', ['build'], { cwd });
    }
    catch (e) {
        await ioHelper.defaults.warn('Unable to restore/build .NET project: ' + (0, util_1.formatErrorMessage)(e));
        await ioHelper.defaults.warn(dotnetWarning);
    }
}
async function postInstallFSharp(ioHelper, canUseNetwork, cwd) {
    // F# uses the same build system as C#
    return postInstallCSharp(ioHelper, canUseNetwork, cwd);
}
/**
 * @param dir - a directory to be checked
 * @returns true if ``dir`` is within a git repository.
 */
async function isInGitRepository(dir) {
    while (true) {
        if (await fs.pathExists(path.join(dir, '.git'))) {
            return true;
        }
        if (isRoot(dir)) {
            return false;
        }
        dir = path.dirname(dir);
    }
}
/**
 * @param dir - a directory to be checked.
 * @returns true if ``dir`` is the root of a filesystem.
 */
function isRoot(dir) {
    return path.dirname(dir) === dir;
}
/**
 * Executes `command`. STDERR is emitted in real-time.
 *
 * If command exits with non-zero exit code, an exception is thrown and includes
 * the contents of STDOUT.
 *
 * @returns STDOUT (if successful).
 */
async function execute(ioHelper, cmd, args, { cwd }) {
    const child = childProcess.spawn(cmd, args, {
        cwd,
        shell: true,
        stdio: ['ignore', 'pipe', 'inherit'],
    });
    let stdout = '';
    child.stdout.on('data', (chunk) => (stdout += chunk.toString()));
    return new Promise((ok, fail) => {
        child.once('error', (err) => fail(err));
        child.once('exit', (status) => {
            if (status === 0) {
                return ok(stdout);
            }
            else {
                return fail(new toolkit_lib_1.ToolkitError(`${cmd} exited with status ${status}`));
            }
        });
    }).catch(async (err) => {
        await ioHelper.defaults.error(stdout);
        throw err;
    });
}
/**
 * Return the 'aws-cdk-lib' version we will init
 *
 * This has been built into the CLI at build time.
 */
async function loadInitVersions() {
    const initVersionFile = path.join((0, root_dir_1.cliRootDir)(), 'lib', 'init-templates', '.init-version.json');
    const contents = JSON.parse(await fs.readFile(initVersionFile, { encoding: 'utf-8' }));
    const ret = {
        'aws-cdk-lib': contents['aws-cdk-lib'],
        'constructs': contents.constructs,
        'aws-cdk': (0, version_1.versionNumber)(),
    };
    for (const [key, value] of Object.entries(ret)) {
        if (!value) {
            throw new toolkit_lib_1.ToolkitError(`Missing init version from ${initVersionFile}: ${key}`);
        }
    }
    return ret;
}
/**
 * Return the currently recommended flags for `aws-cdk-lib`.
 *
 * These have been built into the CLI at build time.
 */
async function currentlyRecommendedAwsCdkLibFlags() {
    const recommendedFlagsFile = path.join((0, root_dir_1.cliRootDir)(), 'lib', 'init-templates', '.recommended-feature-flags.json');
    return JSON.parse(await fs.readFile(recommendedFlagsFile, { encoding: 'utf-8' }));
}
//# sourceMappingURL=data:application/json;base64,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