"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkToolkit = exports.AssetBuildTime = void 0;
exports.displayFlagsMessage = displayFlagsMessage;
const path = require("path");
const util_1 = require("util");
const cxapi = require("@aws-cdk/cloud-assembly-api");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const chalk = require("chalk");
const chokidar = require("chokidar");
const handler_js_1 = require("chokidar/handler.js");
const fs = require("fs-extra");
const uuid = require("uuid");
const io_host_1 = require("./io-host");
const user_configuration_1 = require("./user-configuration");
const api_private_1 = require("../../lib/api-private");
const api_1 = require("../api");
const bootstrap_1 = require("../api/bootstrap");
const cloud_assembly_1 = require("../api/cloud-assembly");
const refactor_1 = require("../api/refactor");
const deploy_1 = require("../commands/deploy");
const list_stacks_1 = require("../commands/list-stacks");
const migrate_1 = require("../commands/migrate");
const cxapp_1 = require("../cxapp");
const util_2 = require("../util");
const collect_telemetry_1 = require("./telemetry/collect-telemetry");
const error_1 = require("./telemetry/error");
const messages_1 = require("./telemetry/messages");
const operations_1 = require("../commands/flags/operations");
// Must use a require() otherwise esbuild complains about calling a namespace
// eslint-disable-next-line @typescript-eslint/no-require-imports,@typescript-eslint/consistent-type-imports
const pLimit = require('p-limit');
/**
 * File events that we care about from chokidar.
 * In chokidar v4, EventName includes additional events like 'error', 'raw', 'ready', 'all'
 * that we need to filter out in the 'all' handler.
 */
const FILE_EVENTS = [handler_js_1.EVENTS.ADD, handler_js_1.EVENTS.ADD_DIR, handler_js_1.EVENTS.CHANGE, handler_js_1.EVENTS.UNLINK, handler_js_1.EVENTS.UNLINK_DIR];
/**
 * Type guard to check if an event is a file event we should process.
 */
function isFileEvent(event) {
    return FILE_EVENTS.includes(event);
}
/**
 * When to build assets
 */
var AssetBuildTime;
(function (AssetBuildTime) {
    /**
     * Build all assets before deploying the first stack
     *
     * This is intended for expensive Docker image builds; so that if the Docker image build
     * fails, no stacks are unnecessarily deployed (with the attendant wait time).
     */
    AssetBuildTime["ALL_BEFORE_DEPLOY"] = "all-before-deploy";
    /**
     * Build assets just-in-time, before publishing
     */
    AssetBuildTime["JUST_IN_TIME"] = "just-in-time";
})(AssetBuildTime || (exports.AssetBuildTime = AssetBuildTime = {}));
/**
 * Custom implementation of the public Toolkit to integrate with the legacy CdkToolkit
 *
 * This overwrites how an sdkProvider is acquired
 * in favor of the one provided directly to CdkToolkit.
 */
class InternalToolkit extends toolkit_lib_1.Toolkit {
    constructor(sdkProvider, options) {
        super(options);
        this._sdkProvider = sdkProvider;
    }
    /**
     * Access to the AWS SDK
     * @internal
     */
    async sdkProvider(_action) {
        return this._sdkProvider;
    }
}
/**
 * Toolkit logic
 *
 * The toolkit runs the `cloudExecutable` to obtain a cloud assembly and
 * deploys applies them to `cloudFormation`.
 */
class CdkToolkit {
    constructor(props) {
        this.props = props;
        this.ioHost = props.ioHost ?? io_host_1.CliIoHost.instance();
        this.toolkitStackName = props.toolkitStackName ?? api_1.DEFAULT_TOOLKIT_STACK_NAME;
        this.toolkit = new InternalToolkit(props.sdkProvider, {
            assemblyFailureAt: this.validateMetadataFailAt(),
            color: true,
            emojis: true,
            ioHost: this.ioHost,
            toolkitStackName: this.toolkitStackName,
            unstableFeatures: ['refactor', 'flags'],
        });
    }
    async metadata(stackName, json) {
        const stacks = await this.selectSingleStackByName(stackName);
        await printSerializedObject(this.ioHost.asIoHelper(), stacks.firstStack.manifest.metadata ?? {}, json);
    }
    async acknowledge(noticeId) {
        const acks = new Set(this.props.configuration.context.get('acknowledged-issue-numbers') ?? []);
        acks.add(Number(noticeId));
        this.props.configuration.context.set('acknowledged-issue-numbers', Array.from(acks));
        await this.props.configuration.saveContext();
    }
    async cliTelemetryStatus(args) {
        const canCollect = (0, collect_telemetry_1.canCollectTelemetry)(args, this.props.configuration.context);
        if (canCollect) {
            await this.ioHost.asIoHelper().defaults.info('CLI Telemetry is enabled. See https://docs.aws.amazon.com/cdk/v2/guide/cli-telemetry.html for ways to disable.');
        }
        else {
            await this.ioHost.asIoHelper().defaults.info('CLI Telemetry is disabled. See https://docs.aws.amazon.com/cdk/v2/guide/cli-telemetry.html for ways to enable.');
        }
    }
    async cliTelemetry(enable) {
        this.props.configuration.context.set('cli-telemetry', enable);
        await this.props.configuration.saveContext();
        await this.ioHost.asIoHelper().defaults.info(`Telemetry ${enable ? 'enabled' : 'disabled'}`);
    }
    async diff(options) {
        const stacks = await this.selectStacksForDiff(options.stackNames, options.exclusively);
        const strict = !!options.strict;
        const contextLines = options.contextLines || 3;
        const quiet = options.quiet || false;
        let diffs = 0;
        const parameterMap = buildParameterMap(options.parameters);
        if (options.templatePath !== undefined) {
            // Compare single stack against fixed template
            if (stacks.stackCount !== 1) {
                throw new toolkit_lib_1.ToolkitError('Can only select one stack when comparing to fixed template. Use --exclusively to avoid selecting multiple stacks.');
            }
            if (!(await fs.pathExists(options.templatePath))) {
                throw new toolkit_lib_1.ToolkitError(`There is no file at ${options.templatePath}`);
            }
            if (options.importExistingResources) {
                throw new toolkit_lib_1.ToolkitError('Can only use --import-existing-resources flag when comparing against deployed stacks.');
            }
            const template = (0, util_2.deserializeStructure)(await fs.readFile(options.templatePath, { encoding: 'UTF-8' }));
            const formatter = new api_1.DiffFormatter({
                templateInfo: {
                    oldTemplate: template,
                    newTemplate: stacks.firstStack,
                },
            });
            if (options.securityOnly) {
                const securityDiff = formatter.formatSecurityDiff();
                // Warn, count, and display the diff only if the reported changes are broadening permissions
                if (securityDiff.permissionChangeType === toolkit_lib_1.PermissionChangeType.BROADENING) {
                    await this.ioHost.asIoHelper().defaults.warn('This deployment will make potentially sensitive changes according to your current security approval level.\nPlease confirm you intend to make the following modifications:\n');
                    await this.ioHost.asIoHelper().defaults.info(securityDiff.formattedDiff);
                    diffs += 1;
                }
            }
            else {
                const diff = formatter.formatStackDiff({
                    strict,
                    contextLines,
                    quiet,
                });
                diffs = diff.numStacksWithChanges;
                await this.ioHost.asIoHelper().defaults.info(diff.formattedDiff);
            }
        }
        else {
            const allMappings = options.includeMoves
                ? await (0, refactor_1.mappingsByEnvironment)(stacks.stackArtifacts, this.props.sdkProvider, true)
                : [];
            // Compare N stacks against deployed templates
            for (const stack of stacks.stackArtifacts) {
                const templateWithNestedStacks = await this.props.deployments.readCurrentTemplateWithNestedStacks(stack, options.compareAgainstProcessedTemplate);
                const currentTemplate = templateWithNestedStacks.deployedRootTemplate;
                const nestedStacks = templateWithNestedStacks.nestedStacks;
                const migrator = new api_1.ResourceMigrator({
                    deployments: this.props.deployments,
                    ioHelper: (0, api_private_1.asIoHelper)(this.ioHost, 'diff'),
                });
                const resourcesToImport = await migrator.tryGetResources(await this.props.deployments.resolveEnvironment(stack));
                if (resourcesToImport) {
                    (0, api_1.removeNonImportResources)(stack);
                }
                let changeSet = undefined;
                if (options.changeSet) {
                    let stackExists = false;
                    try {
                        stackExists = await this.props.deployments.stackExists({
                            stack,
                            deployName: stack.stackName,
                            tryLookupRole: true,
                        });
                    }
                    catch (e) {
                        await this.ioHost.asIoHelper().defaults.debug((0, util_2.formatErrorMessage)(e));
                        if (!quiet) {
                            await this.ioHost.asIoHelper().defaults.info(`Checking if the stack ${stack.stackName} exists before creating the changeset has failed, will base the diff on template differences (run again with -v to see the reason)\n`);
                        }
                        stackExists = false;
                    }
                    if (stackExists) {
                        changeSet = await api_private_1.cfnApi.createDiffChangeSet((0, api_private_1.asIoHelper)(this.ioHost, 'diff'), {
                            stack,
                            uuid: uuid.v4(),
                            deployments: this.props.deployments,
                            willExecute: false,
                            sdkProvider: this.props.sdkProvider,
                            parameters: Object.assign({}, parameterMap['*'], parameterMap[stack.stackName]),
                            resourcesToImport,
                            importExistingResources: options.importExistingResources,
                        });
                    }
                    else {
                        await this.ioHost.asIoHelper().defaults.debug(`the stack '${stack.stackName}' has not been deployed to CloudFormation or describeStacks call failed, skipping changeset creation.`);
                    }
                }
                const mappings = allMappings.find(m => m.environment.region === stack.environment.region && m.environment.account === stack.environment.account)?.mappings ?? {};
                const formatter = new api_1.DiffFormatter({
                    templateInfo: {
                        oldTemplate: currentTemplate,
                        newTemplate: stack,
                        changeSet,
                        isImport: !!resourcesToImport,
                        nestedStacks,
                        mappings,
                    },
                });
                if (options.securityOnly) {
                    const securityDiff = formatter.formatSecurityDiff();
                    // Warn, count, and display the diff only if the reported changes are broadening permissions
                    if (securityDiff.permissionChangeType === toolkit_lib_1.PermissionChangeType.BROADENING) {
                        await this.ioHost.asIoHelper().defaults.warn('This deployment will make potentially sensitive changes according to your current security approval level.\nPlease confirm you intend to make the following modifications:\n');
                        await this.ioHost.asIoHelper().defaults.info(securityDiff.formattedDiff);
                        diffs += 1;
                    }
                }
                else {
                    const diff = formatter.formatStackDiff({
                        strict,
                        contextLines,
                        quiet,
                    });
                    await this.ioHost.asIoHelper().defaults.info(diff.formattedDiff);
                    diffs += diff.numStacksWithChanges;
                }
            }
        }
        await this.ioHost.asIoHelper().defaults.info((0, util_1.format)('\n✨  Number of stacks with differences: %s\n', diffs));
        return diffs && options.fail ? 1 : 0;
    }
    async deploy(options) {
        if (options.watch) {
            return this.watch(options);
        }
        // set progress from options, this includes user and app config
        if (options.progress) {
            this.ioHost.stackProgress = options.progress;
        }
        const startSynthTime = new Date().getTime();
        const stackCollection = await this.selectStacksForDeploy(options.selector, options.exclusively, options.cacheCloudAssembly, options.ignoreNoStacks);
        const elapsedSynthTime = new Date().getTime() - startSynthTime;
        await this.ioHost.asIoHelper().defaults.info(`\n✨  Synthesis time: ${(0, util_2.formatTime)(elapsedSynthTime)}s\n`);
        if (stackCollection.stackCount === 0) {
            await this.ioHost.asIoHelper().defaults.error('This app contains no stacks');
            return;
        }
        const migrator = new api_1.ResourceMigrator({
            deployments: this.props.deployments,
            ioHelper: (0, api_private_1.asIoHelper)(this.ioHost, 'deploy'),
        });
        await migrator.tryMigrateResources(stackCollection, {
            toolkitStackName: this.toolkitStackName,
            ...options,
        });
        const requireApproval = options.requireApproval ?? cloud_assembly_schema_1.RequireApproval.BROADENING;
        const parameterMap = buildParameterMap(options.parameters);
        if (options.deploymentMethod?.method === 'hotswap') {
            await this.ioHost.asIoHelper().defaults.warn('⚠️ The --hotswap and --hotswap-fallback flags deliberately introduce CloudFormation drift to speed up deployments');
            await this.ioHost.asIoHelper().defaults.warn('⚠️ They should only be used for development - never use them for your production Stacks!\n');
        }
        const stacks = stackCollection.stackArtifacts;
        const stackOutputs = {};
        const outputsFile = options.outputsFile;
        const buildAsset = async (assetNode) => {
            await this.props.deployments.buildSingleAsset(assetNode.assetManifestArtifact, assetNode.assetManifest, assetNode.asset, {
                stack: assetNode.parentStack,
                roleArn: options.roleArn,
                stackName: assetNode.parentStack.stackName,
            });
        };
        const publishAsset = async (assetNode) => {
            await this.props.deployments.publishSingleAsset(assetNode.assetManifest, assetNode.asset, {
                stack: assetNode.parentStack,
                roleArn: options.roleArn,
                stackName: assetNode.parentStack.stackName,
                forcePublish: options.force,
            });
        };
        const deployStack = async (stackNode) => {
            const stack = stackNode.stack;
            if (stackCollection.stackCount !== 1) {
                await this.ioHost.asIoHelper().defaults.info(chalk.bold(stack.displayName));
            }
            if (!stack.environment) {
                // eslint-disable-next-line @stylistic/max-len
                throw new toolkit_lib_1.ToolkitError(`Stack ${stack.displayName} does not define an environment, and AWS credentials could not be obtained from standard locations or no region was configured.`);
            }
            if (Object.keys(stack.template.Resources || {}).length === 0) {
                // The generated stack has no resources
                if (!(await this.props.deployments.stackExists({ stack }))) {
                    await this.ioHost.asIoHelper().defaults.warn('%s: stack has no resources, skipping deployment.', chalk.bold(stack.displayName));
                }
                else {
                    await this.ioHost.asIoHelper().defaults.warn('%s: stack has no resources, deleting existing stack.', chalk.bold(stack.displayName));
                    await this.destroy({
                        selector: { patterns: [stack.hierarchicalId] },
                        exclusively: true,
                        force: true,
                        roleArn: options.roleArn,
                        fromDeploy: true,
                    });
                }
                return;
            }
            if (requireApproval !== cloud_assembly_schema_1.RequireApproval.NEVER) {
                const currentTemplate = await this.props.deployments.readCurrentTemplate(stack);
                const formatter = new api_1.DiffFormatter({
                    templateInfo: {
                        oldTemplate: currentTemplate,
                        newTemplate: stack,
                    },
                });
                const securityDiff = formatter.formatSecurityDiff();
                if (requiresApproval(requireApproval, securityDiff.permissionChangeType)) {
                    const motivation = '"--require-approval" is enabled and stack includes security-sensitive updates';
                    await this.ioHost.asIoHelper().defaults.info(securityDiff.formattedDiff);
                    await askUserConfirmation(this.ioHost, api_private_1.IO.CDK_TOOLKIT_I5060.req(`${motivation}: 'Do you wish to deploy these changes'`, {
                        motivation,
                        concurrency,
                        permissionChangeType: securityDiff.permissionChangeType,
                        templateDiffs: formatter.diffs,
                    }));
                }
            }
            // Following are the same semantics we apply with respect to Notification ARNs (dictated by the SDK)
            //
            //  - undefined  =>  cdk ignores it, as if it wasn't supported (allows external management).
            //  - []:        =>  cdk manages it, and the user wants to wipe it out.
            //  - ['arn-1']  =>  cdk manages it, and the user wants to set it to ['arn-1'].
            const notificationArns = (!!options.notificationArns || !!stack.notificationArns)
                ? (options.notificationArns ?? []).concat(stack.notificationArns ?? [])
                : undefined;
            for (const notificationArn of notificationArns ?? []) {
                if (!(0, util_2.validateSnsTopicArn)(notificationArn)) {
                    throw new toolkit_lib_1.ToolkitError(`Notification arn ${notificationArn} is not a valid arn for an SNS topic`);
                }
            }
            const stackIndex = stacks.indexOf(stack) + 1;
            await this.ioHost.asIoHelper().defaults.info(`${chalk.bold(stack.displayName)}: deploying... [${stackIndex}/${stackCollection.stackCount}]`);
            const startDeployTime = new Date().getTime();
            let tags = options.tags;
            if (!tags || tags.length === 0) {
                tags = (0, api_private_1.tagsForStack)(stack);
            }
            // There is already a startDeployTime constant, but that does not work with telemetry.
            // We should integrate the two in the future
            const deploySpan = await this.ioHost.asIoHelper().span(messages_1.CLI_PRIVATE_SPAN.DEPLOY).begin({});
            let error;
            let elapsedDeployTime = 0;
            try {
                let deployResult;
                let rollback = options.rollback;
                let iteration = 0;
                while (!deployResult) {
                    if (++iteration > 2) {
                        throw new toolkit_lib_1.ToolkitError('This loop should have stabilized in 2 iterations, but didn\'t. If you are seeing this error, please report it at https://github.com/aws/aws-cdk/issues/new/choose');
                    }
                    const r = await this.props.deployments.deployStack({
                        stack,
                        deployName: stack.stackName,
                        roleArn: options.roleArn,
                        toolkitStackName: options.toolkitStackName,
                        reuseAssets: options.reuseAssets,
                        notificationArns,
                        tags,
                        execute: options.execute,
                        changeSetName: options.changeSetName,
                        deploymentMethod: options.deploymentMethod,
                        forceDeployment: options.force,
                        parameters: Object.assign({}, parameterMap['*'], parameterMap[stack.stackName]),
                        usePreviousParameters: options.usePreviousParameters,
                        rollback,
                        extraUserAgent: options.extraUserAgent,
                        assetParallelism: options.assetParallelism,
                        ignoreNoStacks: options.ignoreNoStacks,
                    });
                    switch (r.type) {
                        case 'did-deploy-stack':
                            deployResult = r;
                            break;
                        case 'failpaused-need-rollback-first': {
                            const motivation = r.reason === 'replacement'
                                ? `Stack is in a paused fail state (${r.status}) and change includes a replacement which cannot be deployed with "--no-rollback"`
                                : `Stack is in a paused fail state (${r.status}) and command line arguments do not include "--no-rollback"`;
                            if (options.force) {
                                await this.ioHost.asIoHelper().defaults.warn(`${motivation}. Rolling back first (--force).`);
                            }
                            else {
                                await askUserConfirmation(this.ioHost, api_private_1.IO.CDK_TOOLKIT_I5050.req(`${motivation}. Roll back first and then proceed with deployment`, {
                                    motivation,
                                    concurrency,
                                }));
                            }
                            // Perform a rollback
                            await this.rollback({
                                selector: { patterns: [stack.hierarchicalId] },
                                toolkitStackName: options.toolkitStackName,
                                force: options.force,
                            });
                            // Go around through the 'while' loop again but switch rollback to true.
                            rollback = true;
                            break;
                        }
                        case 'replacement-requires-rollback': {
                            const motivation = 'Change includes a replacement which cannot be deployed with "--no-rollback"';
                            if (options.force) {
                                await this.ioHost.asIoHelper().defaults.warn(`${motivation}. Proceeding with regular deployment (--force).`);
                            }
                            else {
                                await askUserConfirmation(this.ioHost, api_private_1.IO.CDK_TOOLKIT_I5050.req(`${motivation}. Perform a regular deployment`, {
                                    concurrency,
                                    motivation,
                                }));
                            }
                            // Go around through the 'while' loop again but switch rollback to true.
                            rollback = true;
                            break;
                        }
                        default:
                            throw new toolkit_lib_1.ToolkitError(`Unexpected result type from deployStack: ${JSON.stringify(r)}. If you are seeing this error, please report it at https://github.com/aws/aws-cdk/issues/new/choose`);
                    }
                }
                const message = deployResult.noOp
                    ? ' ✅  %s (no changes)'
                    : ' ✅  %s';
                await this.ioHost.asIoHelper().defaults.info(chalk.green('\n' + message), stack.displayName);
                elapsedDeployTime = new Date().getTime() - startDeployTime;
                await this.ioHost.asIoHelper().defaults.info(`\n✨  Deployment time: ${(0, util_2.formatTime)(elapsedDeployTime)}s\n`);
                if (Object.keys(deployResult.outputs).length > 0) {
                    await this.ioHost.asIoHelper().defaults.info('Outputs:');
                    stackOutputs[stack.stackName] = deployResult.outputs;
                }
                for (const name of Object.keys(deployResult.outputs).sort()) {
                    const value = deployResult.outputs[name];
                    await this.ioHost.asIoHelper().defaults.info(`${chalk.cyan(stack.id)}.${chalk.cyan(name)} = ${chalk.underline(chalk.cyan(value))}`);
                }
                await this.ioHost.asIoHelper().defaults.info('Stack ARN:');
                await this.ioHost.asIoHelper().defaults.result(deployResult.stackArn);
            }
            catch (e) {
                // It has to be exactly this string because an integration test tests for
                // "bold(stackname) failed: ResourceNotReady: <error>"
                const wrappedError = new toolkit_lib_1.ToolkitError([`❌  ${chalk.bold(stack.stackName)} failed:`, ...(e.name ? [`${e.name}:`] : []), (0, util_2.formatErrorMessage)(e)].join(' '));
                error = {
                    name: (0, error_1.cdkCliErrorName)(wrappedError.name),
                };
                throw wrappedError;
            }
            finally {
                await deploySpan.end({ error });
                if (options.cloudWatchLogMonitor) {
                    const foundLogGroupsResult = await (0, api_1.findCloudWatchLogGroups)(this.props.sdkProvider, (0, api_private_1.asIoHelper)(this.ioHost, 'deploy'), stack);
                    options.cloudWatchLogMonitor.addLogGroups(foundLogGroupsResult.env, foundLogGroupsResult.sdk, foundLogGroupsResult.logGroupNames);
                }
                // If an outputs file has been specified, create the file path and write stack outputs to it once.
                // Outputs are written after all stacks have been deployed. If a stack deployment fails,
                // all of the outputs from successfully deployed stacks before the failure will still be written.
                if (outputsFile) {
                    fs.ensureFileSync(outputsFile);
                    await fs.writeJson(outputsFile, stackOutputs, {
                        spaces: 2,
                        encoding: 'utf8',
                    });
                }
            }
            await this.ioHost.asIoHelper().defaults.info(`\n✨  Total time: ${(0, util_2.formatTime)(elapsedSynthTime + elapsedDeployTime)}s\n`);
        };
        const assetBuildTime = options.assetBuildTime ?? AssetBuildTime.ALL_BEFORE_DEPLOY;
        const prebuildAssets = assetBuildTime === AssetBuildTime.ALL_BEFORE_DEPLOY;
        const concurrency = options.concurrency || 1;
        if (concurrency > 1) {
            // always force "events" progress output when we have concurrency
            this.ioHost.stackProgress = deploy_1.StackActivityProgress.EVENTS;
            // ...but only warn if the user explicitly requested "bar" progress
            if (options.progress && options.progress != deploy_1.StackActivityProgress.EVENTS) {
                await this.ioHost.asIoHelper().defaults.warn('⚠️ The --concurrency flag only supports --progress "events". Switching to "events".');
            }
        }
        const stacksAndTheirAssetManifests = stacks.flatMap((stack) => [
            stack,
            ...stack.dependencies.filter(x => cxapi.AssetManifestArtifact.isAssetManifestArtifact(x)),
        ]);
        const workGraph = new api_1.WorkGraphBuilder((0, api_private_1.asIoHelper)(this.ioHost, 'deploy'), prebuildAssets).build(stacksAndTheirAssetManifests);
        // Unless we are running with '--force', skip already published assets
        if (!options.force) {
            await this.removePublishedAssets(workGraph, options);
        }
        const graphConcurrency = {
            'stack': concurrency,
            'asset-build': 1, // This will be CPU-bound/memory bound, mostly matters for Docker builds
            'asset-publish': (options.assetParallelism ?? true) ? 8 : 1, // This will be I/O-bound, 8 in parallel seems reasonable
        };
        await workGraph.doParallel(graphConcurrency, {
            deployStack,
            buildAsset,
            publishAsset,
        });
    }
    /**
     * Detect infrastructure drift for the given stack(s)
     */
    async drift(options) {
        const driftResults = await this.toolkit.drift(this.props.cloudExecutable, {
            stacks: {
                patterns: options.selector.patterns,
                strategy: options.selector.patterns.length > 0 ? api_1.StackSelectionStrategy.PATTERN_MATCH : api_1.StackSelectionStrategy.ALL_STACKS,
            },
        });
        const totalDrifts = Object.values(driftResults).reduce((total, current) => total + (current.numResourcesWithDrift ?? 0), 0);
        return totalDrifts > 0 && options.fail ? 1 : 0;
    }
    /**
     * Roll back the given stack or stacks.
     */
    async rollback(options) {
        const startSynthTime = new Date().getTime();
        const stackCollection = await this.selectStacksForDeploy(options.selector, true);
        const elapsedSynthTime = new Date().getTime() - startSynthTime;
        await this.ioHost.asIoHelper().defaults.info(`\n✨  Synthesis time: ${(0, util_2.formatTime)(elapsedSynthTime)}s\n`);
        if (stackCollection.stackCount === 0) {
            await this.ioHost.asIoHelper().defaults.error('No stacks selected');
            return;
        }
        let anyRollbackable = false;
        for (const stack of stackCollection.stackArtifacts) {
            await this.ioHost.asIoHelper().defaults.info('Rolling back %s', chalk.bold(stack.displayName));
            const startRollbackTime = new Date().getTime();
            try {
                const result = await this.props.deployments.rollbackStack({
                    stack,
                    roleArn: options.roleArn,
                    toolkitStackName: options.toolkitStackName,
                    orphanFailedResources: options.force,
                    validateBootstrapStackVersion: options.validateBootstrapStackVersion,
                    orphanLogicalIds: options.orphanLogicalIds,
                });
                if (!result.notInRollbackableState) {
                    anyRollbackable = true;
                }
                const elapsedRollbackTime = new Date().getTime() - startRollbackTime;
                await this.ioHost.asIoHelper().defaults.info(`\n✨  Rollback time: ${(0, util_2.formatTime)(elapsedRollbackTime).toString()}s\n`);
            }
            catch (e) {
                await this.ioHost.asIoHelper().defaults.error('\n ❌  %s failed: %s', chalk.bold(stack.displayName), (0, util_2.formatErrorMessage)(e));
                throw new toolkit_lib_1.ToolkitError('Rollback failed (use --force to orphan failing resources)');
            }
        }
        if (!anyRollbackable) {
            throw new toolkit_lib_1.ToolkitError('No stacks were in a state that could be rolled back');
        }
    }
    async watch(options) {
        const rootDir = path.dirname(path.resolve(user_configuration_1.PROJECT_CONFIG));
        const ioHelper = (0, api_private_1.asIoHelper)(this.ioHost, 'watch');
        await this.ioHost.asIoHelper().defaults.debug("root directory used for 'watch' is: %s", rootDir);
        const watchSettings = this.props.configuration.settings.get(['watch']);
        if (!watchSettings) {
            throw new toolkit_lib_1.ToolkitError("Cannot use the 'watch' command without specifying at least one directory to monitor. " +
                'Make sure to add a "watch" key to your cdk.json');
        }
        // For the "include" subkey under the "watch" key, the behavior is:
        // 1. No "watch" setting? We error out.
        // 2. "watch" setting without an "include" key? We default to observing "./**".
        // 3. "watch" setting with an empty "include" key? We default to observing "./**".
        // 4. Non-empty "include" key? Just use the "include" key.
        const watchIncludes = this.patternsArrayForWatch(watchSettings.include, {
            rootDir,
            returnRootDirIfEmpty: true,
        });
        await this.ioHost.asIoHelper().defaults.debug("'include' patterns for 'watch': %s", watchIncludes);
        // For the "exclude" subkey under the "watch" key,
        // the behavior is to add some default excludes in addition to the ones specified by the user:
        // 1. The CDK output directory.
        // 2. Any file whose name starts with a dot.
        // 3. Any directory's content whose name starts with a dot.
        // 4. Any node_modules and its content (even if it's not a JS/TS project, you might be using a local aws-cli package)
        const outputDir = this.props.configuration.settings.get(['output']);
        const watchExcludes = this.patternsArrayForWatch(watchSettings.exclude, {
            rootDir,
            returnRootDirIfEmpty: false,
        }).concat(`${outputDir}/**`, '**/.*', '**/.*/**', '**/node_modules/**');
        await this.ioHost.asIoHelper().defaults.debug("'exclude' patterns for 'watch': %s", watchExcludes);
        // Since 'cdk deploy' is a relatively slow operation for a 'watch' process,
        // introduce a concurrency latch that tracks the state.
        // This way, if file change events arrive when a 'cdk deploy' is still executing,
        // we will batch them, and trigger another 'cdk deploy' after the current one finishes,
        // making sure 'cdk deploy's  always execute one at a time.
        // Here's a diagram showing the state transitions:
        // --------------                --------    file changed     --------------    file changed     --------------  file changed
        // |            |  ready event   |      | ------------------> |            | ------------------> |            | --------------|
        // | pre-ready  | -------------> | open |                     | deploying  |                     |   queued   |               |
        // |            |                |      | <------------------ |            | <------------------ |            | <-------------|
        // --------------                --------  'cdk deploy' done  --------------  'cdk deploy' done  --------------
        let latch = 'pre-ready';
        const cloudWatchLogMonitor = options.traceLogs ? new api_1.CloudWatchLogEventMonitor({
            ioHelper,
        }) : undefined;
        const deployAndWatch = async () => {
            latch = 'deploying';
            await cloudWatchLogMonitor?.deactivate();
            await this.invokeDeployFromWatch(options, cloudWatchLogMonitor);
            // If latch is still 'deploying' after the 'await', that's fine,
            // but if it's 'queued', that means we need to deploy again
            while (latch === 'queued') {
                // TypeScript doesn't realize latch can change between 'awaits',
                // and thinks the above 'while' condition is always 'false' without the cast
                latch = 'deploying';
                await this.ioHost.asIoHelper().defaults.info("Detected file changes during deployment. Invoking 'cdk deploy' again");
                await this.invokeDeployFromWatch(options, cloudWatchLogMonitor);
            }
            latch = 'open';
            await cloudWatchLogMonitor?.activate();
        };
        chokidar
            .watch(watchIncludes, {
            ignored: watchExcludes,
            cwd: rootDir,
        })
            .on('ready', async () => {
            latch = 'open';
            await this.ioHost.asIoHelper().defaults.debug("'watch' received the 'ready' event. From now on, all file changes will trigger a deployment");
            await this.ioHost.asIoHelper().defaults.info("Triggering initial 'cdk deploy'");
            await deployAndWatch();
        })
            .on('all', async (event, filePath) => {
            if (!isFileEvent(event)) {
                return; // Ignore non-file events like 'error', 'raw', 'ready', 'all'
            }
            if (latch === 'pre-ready') {
                await this.ioHost.asIoHelper().defaults.info(`'watch' is observing ${event === 'addDir' ? 'directory' : 'the file'} '%s' for changes`, filePath);
            }
            else if (latch === 'open') {
                await this.ioHost.asIoHelper().defaults.info("Detected change to '%s' (type: %s). Triggering 'cdk deploy'", filePath, event);
                await deployAndWatch();
            }
            else {
                // this means latch is either 'deploying' or 'queued'
                latch = 'queued';
                await this.ioHost.asIoHelper().defaults.info("Detected change to '%s' (type: %s) while 'cdk deploy' is still running. " +
                    'Will queue for another deployment after this one finishes', filePath, event);
            }
        });
    }
    async import(options) {
        const stacks = await this.selectStacksForDeploy(options.selector, true, true, false);
        // set progress from options, this includes user and app config
        if (options.progress) {
            this.ioHost.stackProgress = options.progress;
        }
        if (stacks.stackCount > 1) {
            throw new toolkit_lib_1.ToolkitError(`Stack selection is ambiguous, please choose a specific stack for import [${stacks.stackArtifacts.map((x) => x.id).join(', ')}]`);
        }
        if (!process.stdout.isTTY && !options.resourceMappingFile) {
            throw new toolkit_lib_1.ToolkitError('--resource-mapping is required when input is not a terminal');
        }
        const stack = stacks.stackArtifacts[0];
        await this.ioHost.asIoHelper().defaults.info(chalk.bold(stack.displayName));
        const resourceImporter = new api_1.ResourceImporter(stack, {
            deployments: this.props.deployments,
            ioHelper: (0, api_private_1.asIoHelper)(this.ioHost, 'import'),
        });
        const { additions, hasNonAdditions } = await resourceImporter.discoverImportableResources(options.force);
        if (additions.length === 0) {
            await this.ioHost.asIoHelper().defaults.warn('%s: no new resources compared to the currently deployed stack, skipping import.', chalk.bold(stack.displayName));
            return;
        }
        // Prepare a mapping of physical resources to CDK constructs
        const actualImport = !options.resourceMappingFile
            ? await resourceImporter.askForResourceIdentifiers(additions)
            : await resourceImporter.loadResourceIdentifiers(additions, options.resourceMappingFile);
        if (actualImport.importResources.length === 0) {
            await this.ioHost.asIoHelper().defaults.warn('No resources selected for import.');
            return;
        }
        // If "--create-resource-mapping" option was passed, write the resource mapping to the given file and exit
        if (options.recordResourceMapping) {
            const outputFile = options.recordResourceMapping;
            fs.ensureFileSync(outputFile);
            await fs.writeJson(outputFile, actualImport.resourceMap, {
                spaces: 2,
                encoding: 'utf8',
            });
            await this.ioHost.asIoHelper().defaults.info('%s: mapping file written.', outputFile);
            return;
        }
        // Import the resources according to the given mapping
        await this.ioHost.asIoHelper().defaults.info('%s: importing resources into stack...', chalk.bold(stack.displayName));
        const tags = (0, api_private_1.tagsForStack)(stack);
        await resourceImporter.importResourcesFromMap(actualImport, {
            roleArn: options.roleArn,
            tags,
            deploymentMethod: options.deploymentMethod,
            usePreviousParameters: true,
            rollback: options.rollback,
        });
        // Notify user of next steps
        await this.ioHost.asIoHelper().defaults.info(`Import operation complete. We recommend you run a ${chalk.blueBright('drift detection')} operation ` +
            'to confirm your CDK app resource definitions are up-to-date. Read more here: ' +
            chalk.underline.blueBright('https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/detect-drift-stack.html'));
        if (actualImport.importResources.length < additions.length) {
            await this.ioHost.asIoHelper().defaults.info('');
            await this.ioHost.asIoHelper().defaults.warn(`Some resources were skipped. Run another ${chalk.blueBright('cdk import')} or a ${chalk.blueBright('cdk deploy')} to bring the stack up-to-date with your CDK app definition.`);
        }
        else if (hasNonAdditions) {
            await this.ioHost.asIoHelper().defaults.info('');
            await this.ioHost.asIoHelper().defaults.warn(`Your app has pending updates or deletes excluded from this import operation. Run a ${chalk.blueBright('cdk deploy')} to bring the stack up-to-date with your CDK app definition.`);
        }
    }
    async destroy(options) {
        const ioHelper = this.ioHost.asIoHelper();
        // The stacks will have been ordered for deployment, so reverse them for deletion.
        const stacks = (await this.selectStacksForDestroy(options.selector, options.exclusively)).reversed();
        if (!options.force) {
            const motivation = 'Destroying stacks is an irreversible action';
            const question = `Are you sure you want to delete: ${chalk.blue(stacks.stackArtifacts.map((s) => s.hierarchicalId).join(', '))}`;
            try {
                await ioHelper.requestResponse(api_private_1.IO.CDK_TOOLKIT_I7010.req(question, { motivation }));
            }
            catch (err) {
                if (!toolkit_lib_1.ToolkitError.isToolkitError(err) || err.message != 'Aborted by user') {
                    throw err; // unexpected error
                }
                await ioHelper.notify(api_private_1.IO.CDK_TOOLKIT_E7010.msg(err.message));
                return;
            }
        }
        const action = options.fromDeploy ? 'deploy' : 'destroy';
        for (const [index, stack] of stacks.stackArtifacts.entries()) {
            await ioHelper.defaults.info(chalk.green('%s: destroying... [%s/%s]'), chalk.blue(stack.displayName), index + 1, stacks.stackCount);
            try {
                await this.props.deployments.destroyStack({
                    stack,
                    deployName: stack.stackName,
                    roleArn: options.roleArn,
                });
                await ioHelper.defaults.info(chalk.green(`\n ✅  %s: ${action}ed`), chalk.blue(stack.displayName));
            }
            catch (e) {
                await ioHelper.defaults.error(`\n ❌  %s: ${action} failed`, chalk.blue(stack.displayName), e);
                throw e;
            }
        }
    }
    async list(selectors, options = {}) {
        const stacks = await (0, list_stacks_1.listStacks)(this, {
            selectors: selectors,
        });
        if (options.long && options.showDeps) {
            await printSerializedObject(this.ioHost.asIoHelper(), stacks, options.json ?? false);
            return 0;
        }
        if (options.showDeps) {
            const stackDeps = stacks.map(stack => ({
                id: stack.id,
                dependencies: stack.dependencies,
            }));
            await printSerializedObject(this.ioHost.asIoHelper(), stackDeps, options.json ?? false);
            return 0;
        }
        if (options.long) {
            const long = stacks.map(stack => ({
                id: stack.id,
                name: stack.name,
                environment: stack.environment,
            }));
            await printSerializedObject(this.ioHost.asIoHelper(), long, options.json ?? false);
            return 0;
        }
        // just print stack IDs
        for (const stack of stacks) {
            await this.ioHost.asIoHelper().defaults.result(stack.id);
        }
        return 0; // exit-code
    }
    /**
     * Synthesize the given set of stacks (called when the user runs 'cdk synth')
     *
     * INPUT: Stack names can be supplied using a glob filter. If no stacks are
     * given, all stacks from the application are implicitly selected.
     *
     * OUTPUT: If more than one stack ends up being selected, an output directory
     * should be supplied, where the templates will be written.
     */
    async synth(stackNames, exclusively, quiet, autoValidate, json) {
        const stacks = await this.selectStacksForDiff(stackNames, exclusively, autoValidate);
        // if we have a single stack, print it to STDOUT
        if (stacks.stackCount === 1) {
            if (!quiet) {
                await printSerializedObject(this.ioHost.asIoHelper(), (0, util_2.obscureTemplate)(stacks.firstStack.template), json ?? false);
            }
            await displayFlagsMessage(this.ioHost.asIoHelper(), this.toolkit, this.props.cloudExecutable);
            return undefined;
        }
        // not outputting template to stdout, let's explain things to the user a little bit...
        await this.ioHost.asIoHelper().defaults.info(chalk.green(`Successfully synthesized to ${chalk.blue(path.resolve(stacks.assembly.directory))}`));
        await this.ioHost.asIoHelper().defaults.info(`Supply a stack id (${stacks.stackArtifacts.map((s) => chalk.green(s.hierarchicalId)).join(', ')}) to display its template.`);
        await displayFlagsMessage(this.ioHost.asIoHelper(), this.toolkit, this.props.cloudExecutable);
        return undefined;
    }
    /**
     * Bootstrap the CDK Toolkit stack in the accounts used by the specified stack(s).
     *
     * @param userEnvironmentSpecs - environment names that need to have toolkit support
     *             provisioned, as a glob filter. If none is provided, all stacks are implicitly selected.
     * @param options - The name, role ARN, bootstrapping parameters, etc. to be used for the CDK Toolkit stack.
     */
    async bootstrap(userEnvironmentSpecs, options) {
        const bootstrapper = new bootstrap_1.Bootstrapper(options.source, (0, api_private_1.asIoHelper)(this.ioHost, 'bootstrap'));
        // If there is an '--app' argument and an environment looks like a glob, we
        // select the environments from the app. Otherwise, use what the user said.
        const environments = await this.defineEnvironments(userEnvironmentSpecs);
        const limit = pLimit(20);
        // eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism
        await Promise.all(environments.map((environment) => limit(async () => {
            await this.ioHost.asIoHelper().defaults.info(chalk.green(' ⏳  Bootstrapping environment %s...'), chalk.blue(environment.name));
            try {
                const result = await bootstrapper.bootstrapEnvironment(environment, this.props.sdkProvider, options);
                const message = result.noOp
                    ? ' ✅  Environment %s bootstrapped (no changes).'
                    : ' ✅  Environment %s bootstrapped.';
                await this.ioHost.asIoHelper().defaults.info(chalk.green(message), chalk.blue(environment.name));
            }
            catch (e) {
                await this.ioHost.asIoHelper().defaults.error(' ❌  Environment %s failed bootstrapping: %s', chalk.blue(environment.name), e);
                throw e;
            }
        })));
    }
    /**
     * Garbage collects assets from a CDK app's environment
     * @param options - Options for Garbage Collection
     */
    async garbageCollect(userEnvironmentSpecs, options) {
        const environments = await this.defineEnvironments(userEnvironmentSpecs);
        for (const environment of environments) {
            await this.ioHost.asIoHelper().defaults.info(chalk.green(' ⏳  Garbage Collecting environment %s...'), chalk.blue(environment.name));
            const gc = new api_1.GarbageCollector({
                sdkProvider: this.props.sdkProvider,
                ioHelper: (0, api_private_1.asIoHelper)(this.ioHost, 'gc'),
                resolvedEnvironment: environment,
                bootstrapStackName: options.bootstrapStackName,
                rollbackBufferDays: options.rollbackBufferDays,
                createdBufferDays: options.createdBufferDays,
                action: options.action ?? 'full',
                type: options.type ?? 'all',
                confirm: options.confirm ?? true,
            });
            await gc.garbageCollect();
        }
    }
    async defineEnvironments(userEnvironmentSpecs) {
        // By default, glob for everything
        const environmentSpecs = userEnvironmentSpecs.length > 0 ? [...userEnvironmentSpecs] : ['**'];
        // Partition into globs and non-globs (this will mutate environmentSpecs).
        const globSpecs = (0, util_2.partition)(environmentSpecs, cxapp_1.looksLikeGlob);
        if (globSpecs.length > 0 && !this.props.cloudExecutable.hasApp) {
            if (userEnvironmentSpecs.length > 0) {
                // User did request this glob
                throw new toolkit_lib_1.ToolkitError(`'${globSpecs}' is not an environment name. Specify an environment name like 'aws://123456789012/us-east-1', or run in a directory with 'cdk.json' to use wildcards.`);
            }
            else {
                // User did not request anything
                throw new toolkit_lib_1.ToolkitError("Specify an environment name like 'aws://123456789012/us-east-1', or run in a directory with 'cdk.json'.");
            }
        }
        const environments = [...(0, cxapp_1.environmentsFromDescriptors)(environmentSpecs)];
        // If there is an '--app' argument, select the environments from the app.
        if (this.props.cloudExecutable.hasApp) {
            environments.push(...(await (0, cxapp_1.globEnvironmentsFromStacks)(await this.selectStacksForList([]), globSpecs, this.props.sdkProvider)));
        }
        return environments;
    }
    /**
     * Migrates a CloudFormation stack/template to a CDK app
     * @param options - Options for CDK app creation
     */
    async migrate(options) {
        await this.ioHost.asIoHelper().defaults.warn('This command is an experimental feature.');
        const language = options.language?.toLowerCase() ?? 'typescript';
        const environment = (0, migrate_1.setEnvironment)(options.account, options.region);
        let generateTemplateOutput;
        let cfn;
        let templateToDelete;
        try {
            // if neither fromPath nor fromStack is provided, generate a template using cloudformation
            const scanType = (0, migrate_1.parseSourceOptions)(options.fromPath, options.fromStack, options.stackName).source;
            if (scanType == migrate_1.TemplateSourceOptions.SCAN) {
                generateTemplateOutput = await (0, migrate_1.generateTemplate)({
                    stackName: options.stackName,
                    filters: options.filter,
                    fromScan: options.fromScan,
                    sdkProvider: this.props.sdkProvider,
                    environment: environment,
                    ioHelper: this.ioHost.asIoHelper(),
                });
                templateToDelete = generateTemplateOutput.templateId;
            }
            else if (scanType == migrate_1.TemplateSourceOptions.PATH) {
                const templateBody = (0, migrate_1.readFromPath)(options.fromPath);
                const parsedTemplate = (0, util_2.deserializeStructure)(templateBody);
                const templateId = parsedTemplate.Metadata?.AWSToolsMetrics?.IaC_Generator?.toString();
                if (templateId) {
                    // if we have a template id, we can call describe generated template to get the resource identifiers
                    // resource metadata, and template source to generate the template
                    cfn = new migrate_1.CfnTemplateGeneratorProvider(await (0, migrate_1.buildCfnClient)(this.props.sdkProvider, environment), this.ioHost.asIoHelper());
                    const generatedTemplateSummary = await cfn.describeGeneratedTemplate(templateId);
                    generateTemplateOutput = (0, migrate_1.buildGeneratedTemplateOutput)(generatedTemplateSummary, templateBody, generatedTemplateSummary.GeneratedTemplateId);
                }
                else {
                    generateTemplateOutput = {
                        migrateJson: {
                            templateBody: templateBody,
                            source: 'localfile',
                        },
                    };
                }
            }
            else if (scanType == migrate_1.TemplateSourceOptions.STACK) {
                const template = await (0, migrate_1.readFromStack)(options.stackName, this.props.sdkProvider, environment);
                if (!template) {
                    throw new toolkit_lib_1.ToolkitError(`No template found for stack-name: ${options.stackName}`);
                }
                generateTemplateOutput = {
                    migrateJson: {
                        templateBody: template,
                        source: options.stackName,
                    },
                };
            }
            else {
                // We shouldn't ever get here, but just in case.
                throw new toolkit_lib_1.ToolkitError(`Invalid source option provided: ${scanType}`);
            }
            const stack = (0, migrate_1.generateStack)(generateTemplateOutput.migrateJson.templateBody, options.stackName, language);
            await this.ioHost.asIoHelper().defaults.info(chalk.green(' ⏳  Generating CDK app for %s...'), chalk.blue(options.stackName));
            await (0, migrate_1.generateCdkApp)(this.ioHost.asIoHelper(), options.stackName, stack, language, options.outputPath, options.compress);
            if (generateTemplateOutput) {
                (0, migrate_1.writeMigrateJsonFile)(options.outputPath, options.stackName, generateTemplateOutput.migrateJson);
            }
            if ((0, migrate_1.isThereAWarning)(generateTemplateOutput)) {
                await this.ioHost.asIoHelper().defaults.warn(' ⚠️  Some resources could not be migrated completely. Please review the README.md file for more information.');
                (0, migrate_1.appendWarningsToReadme)(`${path.join(options.outputPath ?? process.cwd(), options.stackName)}/README.md`, generateTemplateOutput.resources);
            }
        }
        catch (e) {
            await this.ioHost.asIoHelper().defaults.error(' ❌  Migrate failed for `%s`: %s', options.stackName, e.message);
            throw e;
        }
        finally {
            if (templateToDelete) {
                if (!cfn) {
                    cfn = new migrate_1.CfnTemplateGeneratorProvider(await (0, migrate_1.buildCfnClient)(this.props.sdkProvider, environment), this.ioHost.asIoHelper());
                }
                if (!process.env.MIGRATE_INTEG_TEST) {
                    await cfn.deleteGeneratedTemplate(templateToDelete);
                }
            }
        }
    }
    async refactor(options) {
        if (options.revert && !options.overrideFile) {
            throw new toolkit_lib_1.ToolkitError('The --revert option can only be used with the --override-file option.');
        }
        try {
            const patterns = options.stacks?.patterns ?? [];
            await this.toolkit.refactor(this.props.cloudExecutable, {
                dryRun: options.dryRun,
                stacks: {
                    patterns: patterns,
                    strategy: patterns.length > 0 ? api_1.StackSelectionStrategy.PATTERN_MATCH : api_1.StackSelectionStrategy.ALL_STACKS,
                },
                force: options.force,
                additionalStackNames: options.additionalStackNames,
                overrides: readOverrides(options.overrideFile, options.revert),
                roleArn: options.roleArn,
            });
        }
        catch (e) {
            await this.ioHost.asIoHelper().defaults.error(e.message);
            throw e;
        }
        return 0;
        function readOverrides(filePath, revert = false) {
            if (filePath == null) {
                return [];
            }
            if (!fs.pathExistsSync(filePath)) {
                throw new toolkit_lib_1.ToolkitError(`The mapping file ${filePath} does not exist`);
            }
            const groups = (0, refactor_1.parseMappingGroups)(fs.readFileSync(filePath).toString('utf-8'));
            return revert
                ? groups.map((group) => ({
                    ...group,
                    resources: Object.fromEntries(Object.entries(group.resources ?? {}).map(([src, dst]) => [dst, src])),
                }))
                : groups;
        }
    }
    async selectStacksForList(patterns) {
        const assembly = await this.assembly();
        const stacks = await assembly.selectStacks({ patterns }, { defaultBehavior: cxapp_1.DefaultSelection.AllStacks });
        // No validation
        return stacks;
    }
    async selectStacksForDeploy(selector, exclusively, cacheCloudAssembly, ignoreNoStacks) {
        const assembly = await this.assembly(cacheCloudAssembly);
        const stacks = await assembly.selectStacks(selector, {
            extend: exclusively ? cloud_assembly_1.ExtendedStackSelection.None : cloud_assembly_1.ExtendedStackSelection.Upstream,
            defaultBehavior: cxapp_1.DefaultSelection.OnlySingle,
            ignoreNoStacks,
        });
        this.validateStacksSelected(stacks, selector.patterns);
        await this.validateStacks(stacks);
        return stacks;
    }
    async selectStacksForDiff(stackNames, exclusively, autoValidate) {
        const assembly = await this.assembly();
        const selectedForDiff = await assembly.selectStacks({ patterns: stackNames }, {
            extend: exclusively ? cloud_assembly_1.ExtendedStackSelection.None : cloud_assembly_1.ExtendedStackSelection.Upstream,
            defaultBehavior: cxapp_1.DefaultSelection.MainAssembly,
        });
        const allStacks = await this.selectStacksForList([]);
        const autoValidateStacks = autoValidate
            ? allStacks.filter((art) => art.validateOnSynth ?? false)
            : new cloud_assembly_1.StackCollection(assembly, []);
        this.validateStacksSelected(selectedForDiff.concat(autoValidateStacks), stackNames);
        await this.validateStacks(selectedForDiff.concat(autoValidateStacks));
        return selectedForDiff;
    }
    async selectStacksForDestroy(selector, exclusively) {
        const assembly = await this.assembly();
        const stacks = await assembly.selectStacks(selector, {
            extend: exclusively ? cloud_assembly_1.ExtendedStackSelection.None : cloud_assembly_1.ExtendedStackSelection.Downstream,
            defaultBehavior: cxapp_1.DefaultSelection.OnlySingle,
        });
        // No validation
        return stacks;
    }
    /**
     * Validate the stacks for errors and warnings according to the CLI's current settings
     */
    async validateStacks(stacks) {
        const failAt = this.validateMetadataFailAt();
        await stacks.validateMetadata(failAt, stackMetadataLogger(this.ioHost.asIoHelper(), this.props.verbose));
    }
    validateMetadataFailAt() {
        let failAt = 'error';
        if (this.props.ignoreErrors) {
            failAt = 'none';
        }
        if (this.props.strict) {
            failAt = 'warn';
        }
        return failAt;
    }
    /**
     * Validate that if a user specified a stack name there exists at least 1 stack selected
     */
    validateStacksSelected(stacks, stackNames) {
        if (stackNames.length != 0 && stacks.stackCount == 0) {
            throw new toolkit_lib_1.ToolkitError(`No stacks match the name(s) ${stackNames}`);
        }
    }
    /**
     * Select a single stack by its name
     */
    async selectSingleStackByName(stackName) {
        const assembly = await this.assembly();
        const stacks = await assembly.selectStacks({ patterns: [stackName] }, {
            extend: cloud_assembly_1.ExtendedStackSelection.None,
            defaultBehavior: cxapp_1.DefaultSelection.None,
        });
        // Could have been a glob so check that we evaluated to exactly one
        if (stacks.stackCount > 1) {
            throw new toolkit_lib_1.ToolkitError(`This command requires exactly one stack and we matched more than one: ${stacks.stackIds}`);
        }
        return assembly.stackById(stacks.firstStack.id);
    }
    assembly(cacheCloudAssembly) {
        return this.props.cloudExecutable.synthesize(cacheCloudAssembly);
    }
    patternsArrayForWatch(patterns, options) {
        const patternsArray = patterns !== undefined ? (Array.isArray(patterns) ? patterns : [patterns]) : [];
        return patternsArray.length > 0 ? patternsArray : options.returnRootDirIfEmpty ? [options.rootDir] : [];
    }
    async invokeDeployFromWatch(options, cloudWatchLogMonitor) {
        const deployOptions = {
            ...options,
            requireApproval: cloud_assembly_schema_1.RequireApproval.NEVER,
            // if 'watch' is called by invoking 'cdk deploy --watch',
            // we need to make sure to not call 'deploy' with 'watch' again,
            // as that would lead to a cycle
            watch: false,
            cloudWatchLogMonitor,
            cacheCloudAssembly: false,
            extraUserAgent: `cdk-watch/hotswap-${options.deploymentMethod?.method === 'hotswap' ? 'on' : 'off'}`,
            concurrency: options.concurrency,
        };
        try {
            await this.deploy(deployOptions);
        }
        catch {
            // just continue - deploy will show the error
        }
    }
    /**
     * Remove the asset publishing and building from the work graph for assets that are already in place
     */
    async removePublishedAssets(graph, options) {
        await graph.removeUnnecessaryAssets(assetNode => this.props.deployments.isSingleAssetPublished(assetNode.assetManifest, assetNode.asset, {
            stack: assetNode.parentStack,
            roleArn: options.roleArn,
            stackName: assetNode.parentStack.stackName,
        }));
    }
}
exports.CdkToolkit = CdkToolkit;
/**
 * Print a serialized object (YAML or JSON) to stdout.
 */
async function printSerializedObject(ioHelper, obj, json) {
    await ioHelper.defaults.result((0, util_2.serializeStructure)(obj, json));
}
function buildParameterMap(parameters) {
    const parameterMap = { '*': {} };
    for (const key in parameters) {
        if (parameters.hasOwnProperty(key)) {
            const [stack, parameter] = key.split(':', 2);
            if (!parameter) {
                parameterMap['*'][stack] = parameters[key];
            }
            else {
                if (!parameterMap[stack]) {
                    parameterMap[stack] = {};
                }
                parameterMap[stack][parameter] = parameters[key];
            }
        }
    }
    return parameterMap;
}
/**
 * Ask the user for a yes/no confirmation
 *
 * Automatically fail the confirmation in case we're in a situation where the confirmation
 * cannot be interactively obtained from a human at the keyboard.
 */
async function askUserConfirmation(ioHost, req) {
    await ioHost.withCorkedLogging(async () => {
        await ioHost.asIoHelper().requestResponse(req);
    });
}
/**
 * Display a warning if there are flags that are different from the recommended value
 *
 * This happens if both of the following are true:
 *
 * - The user didn't configure the value
 * - The default value for the flag (unconfiguredBehavesLike) is different from the recommended value
 */
async function displayFlagsMessage(ioHost, toolkit, cloudExecutable) {
    const flags = await toolkit.flags(cloudExecutable);
    // The "unconfiguredBehavesLike" information got added later. If none of the flags have this information,
    // we don't have enough information to reliably display this information without scaring users, so don't do anything.
    if (flags.every(flag => flag.unconfiguredBehavesLike === undefined)) {
        return;
    }
    const unconfiguredFlags = operations_1.FlagOperations.filterNeedsAttention(flags);
    const numUnconfigured = unconfiguredFlags.length;
    if (numUnconfigured > 0) {
        await ioHost.defaults.warn(`${numUnconfigured} feature flags are not configured. Run 'cdk flags --unstable=flags' to learn more.`);
    }
}
/**
 * Logger for processing stack metadata
 */
function stackMetadataLogger(ioHelper, verbose) {
    const makeLogger = (level) => {
        switch (level) {
            case 'error':
                return [(m) => ioHelper.defaults.error(m), 'Error'];
            case 'warn':
                return [(m) => ioHelper.defaults.warn(m), 'Warning'];
            default:
                return [(m) => ioHelper.defaults.info(m), 'Info'];
        }
    };
    return async (level, msg) => {
        const [logFn, prefix] = makeLogger(level);
        await logFn(`[${prefix} at ${msg.id}] ${msg.entry.data}`);
        if (verbose && msg.entry.trace) {
            logFn(`  ${msg.entry.trace.join('\n  ')}`);
        }
    };
}
/**
 * Determine if manual approval is required or not. Requires approval for
 * - RequireApproval.ANY_CHANGE
 * - RequireApproval.BROADENING and the changes are indeed broadening permissions
 */
function requiresApproval(requireApproval, permissionChangeType) {
    return requireApproval === cloud_assembly_schema_1.RequireApproval.ANYCHANGE ||
        requireApproval === cloud_assembly_schema_1.RequireApproval.BROADENING && permissionChangeType === toolkit_lib_1.PermissionChangeType.BROADENING;
}
//# sourceMappingURL=data:application/json;base64,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