"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CliIoHost = void 0;
const util = require("node:util");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const chalk = require("chalk");
const promptly = require("promptly");
const api_private_1 = require("../../../lib/api-private");
const deploy_1 = require("../../commands/deploy");
const collect_telemetry_1 = require("../telemetry/collect-telemetry");
const messages_1 = require("../telemetry/messages");
const session_1 = require("../telemetry/session");
const endpoint_sink_1 = require("../telemetry/sink/endpoint-sink");
const file_sink_1 = require("../telemetry/sink/file-sink");
const funnel_1 = require("../telemetry/sink/funnel");
const ci_1 = require("../util/ci");
/**
 * A simple IO host for the CLI that writes messages to the console.
 */
class CliIoHost {
    /**
     * Returns the singleton instance
     */
    static instance(props = {}, forceNew = false) {
        if (forceNew || !CliIoHost._instance) {
            CliIoHost._instance = new CliIoHost(props);
        }
        return CliIoHost._instance;
    }
    /**
     * Returns the singleton instance if it exists
     */
    static get() {
        return CliIoHost._instance;
    }
    constructor(props = {}) {
        /**
         * Configure the target stream for notices
         *
         * (Not a setter because there's no need for additional logic when this value
         * is changed yet)
         */
        this.noticesDestination = 'stderr';
        this._progress = deploy_1.StackActivityProgress.BAR;
        // Corked Logging
        this.corkedCounter = 0;
        this.corkedLoggingBuffer = [];
        this.currentAction = props.currentAction ?? 'none';
        this.isTTY = props.isTTY ?? process.stdout.isTTY ?? false;
        this.logLevel = props.logLevel ?? 'info';
        this.isCI = props.isCI ?? (0, ci_1.isCI)();
        this.requireDeployApproval = props.requireDeployApproval ?? cloud_assembly_schema_1.RequireApproval.BROADENING;
        this.stackProgress = props.stackProgress ?? deploy_1.StackActivityProgress.BAR;
        this.autoRespond = props.autoRespond ?? false;
    }
    async startTelemetry(args, context, proxyAgent) {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const config = require('../cli-type-registry.json');
        const validCommands = Object.keys(config.commands);
        const cmd = args._[0];
        if (!validCommands.includes(cmd)) {
            // the user typed in an invalid command - no need for telemetry since the invocation is going to fail
            // imminently anyway.
            await this.asIoHelper().defaults.trace(`Session instantiated with an invalid command (${cmd}). Not starting telemetry.`);
            return;
        }
        let sinks = [];
        const telemetryFilePath = args['telemetry-file'];
        if (telemetryFilePath) {
            try {
                sinks.push(new file_sink_1.FileTelemetrySink({
                    ioHost: this,
                    logFilePath: telemetryFilePath,
                }));
                await this.asIoHelper().defaults.trace('File Telemetry connected');
            }
            catch (e) {
                await this.asIoHelper().defaults.trace(`File Telemetry instantiation failed: ${e.message}`);
            }
        }
        const telemetryEndpoint = process.env.TELEMETRY_ENDPOINT ?? 'https://cdk-cli-telemetry.us-east-1.api.aws/metrics';
        if ((0, collect_telemetry_1.canCollectTelemetry)(args, context) && telemetryEndpoint) {
            try {
                sinks.push(new endpoint_sink_1.EndpointTelemetrySink({
                    ioHost: this,
                    agent: proxyAgent,
                    endpoint: telemetryEndpoint,
                }));
                await this.asIoHelper().defaults.trace('Endpoint Telemetry connected');
            }
            catch (e) {
                await this.asIoHelper().defaults.trace(`Endpoint Telemetry instantiation failed: ${e.message}`);
            }
        }
        else {
            await this.asIoHelper().defaults.trace('Endpoint Telemetry NOT connected');
        }
        if (sinks.length > 0) {
            this.telemetry = new session_1.TelemetrySession({
                ioHost: this,
                client: new funnel_1.Funnel({ sinks }),
                arguments: args,
                context: context,
            });
        }
        await this.telemetry?.begin();
    }
    /**
     * Update the stackProgress preference.
     */
    set stackProgress(type) {
        this._progress = type;
    }
    /**
     * Gets the stackProgress value.
     *
     * This takes into account other state of the ioHost,
     * like if isTTY and isCI.
     */
    get stackProgress() {
        // We can always use EVENTS
        if (this._progress === deploy_1.StackActivityProgress.EVENTS) {
            return this._progress;
        }
        // if a debug message (and thus any more verbose messages) are relevant to the current log level, we have verbose logging
        const verboseLogging = (0, api_private_1.isMessageRelevantForLevel)({ level: 'debug' }, this.logLevel);
        if (verboseLogging) {
            return deploy_1.StackActivityProgress.EVENTS;
        }
        // On Windows we cannot use fancy output
        const isWindows = process.platform === 'win32';
        if (isWindows) {
            return deploy_1.StackActivityProgress.EVENTS;
        }
        // On some CI systems (such as CircleCI) output still reports as a TTY so we also
        // need an individual check for whether we're running on CI.
        // see: https://discuss.circleci.com/t/circleci-terminal-is-a-tty-but-term-is-not-set/9965
        const fancyOutputAvailable = this.isTTY && !this.isCI;
        if (!fancyOutputAvailable) {
            return deploy_1.StackActivityProgress.EVENTS;
        }
        // Use the user preference
        return this._progress;
    }
    get defaults() {
        return this.asIoHelper().defaults;
    }
    asIoHelper() {
        return (0, api_private_1.asIoHelper)(this, this.currentAction);
    }
    /**
     * Executes a block of code with corked logging. All log messages during execution
     * are buffered and only written when all nested cork blocks complete (when CORK_COUNTER reaches 0).
     * The corking is bound to the specific instance of the CliIoHost.
     *
     * @param block - Async function to execute with corked logging
     * @returns Promise that resolves with the block's return value
     */
    async withCorkedLogging(block) {
        this.corkedCounter++;
        try {
            return await block();
        }
        finally {
            this.corkedCounter--;
            if (this.corkedCounter === 0) {
                // Process each buffered message through notify
                for (const ioMessage of this.corkedLoggingBuffer) {
                    await this.notify(ioMessage);
                }
                // remove all buffered messages in-place
                this.corkedLoggingBuffer.splice(0);
            }
        }
    }
    /**
     * Notifies the host of a message.
     * The caller waits until the notification completes.
     */
    async notify(msg) {
        await this.maybeEmitTelemetry(msg);
        if (this.isStackActivity(msg)) {
            if (!this.activityPrinter) {
                this.activityPrinter = this.makeActivityPrinter();
            }
            this.activityPrinter.notify(msg);
            return;
        }
        if (!(0, api_private_1.isMessageRelevantForLevel)(msg, this.logLevel)) {
            return;
        }
        if (this.corkedCounter > 0) {
            this.corkedLoggingBuffer.push(msg);
            return;
        }
        const output = this.formatMessage(msg);
        const stream = this.selectStream(msg);
        stream?.write(output);
    }
    async maybeEmitTelemetry(msg) {
        try {
            if (this.telemetry && isTelemetryMessage(msg)) {
                await this.telemetry.emit({
                    eventType: getEventType(msg),
                    duration: msg.data.duration,
                    error: msg.data.error,
                });
            }
        }
        catch (e) {
            await this.defaults.trace(`Emit Telemetry Failed ${e.message}`);
        }
    }
    /**
     * Detect stack activity messages so they can be send to the printer.
     */
    isStackActivity(msg) {
        return msg.code && [
            'CDK_TOOLKIT_I5501',
            'CDK_TOOLKIT_I5502',
            'CDK_TOOLKIT_I5503',
        ].includes(msg.code);
    }
    /**
     * Detect special messages encode information about whether or not
     * they require approval
     */
    skipApprovalStep(msg) {
        const approvalToolkitCodes = ['CDK_TOOLKIT_I5060'];
        if (!(msg.code && approvalToolkitCodes.includes(msg.code))) {
            false;
        }
        switch (this.requireDeployApproval) {
            // Never require approval
            case cloud_assembly_schema_1.RequireApproval.NEVER:
                return true;
            // Always require approval
            case cloud_assembly_schema_1.RequireApproval.ANYCHANGE:
                return false;
            // Require approval if changes include broadening permissions
            case cloud_assembly_schema_1.RequireApproval.BROADENING:
                return ['none', 'non-broadening'].includes(msg.data?.permissionChangeType);
        }
    }
    /**
     * Determines the output stream, based on message and configuration.
     */
    selectStream(msg) {
        if (isNoticesMessage(msg)) {
            return targetStreamObject(this.noticesDestination);
        }
        return this.selectStreamFromLevel(msg.level);
    }
    /**
     * Determines the output stream, based on message level and configuration.
     */
    selectStreamFromLevel(level) {
        // The stream selection policy for the CLI is the following:
        //
        //   (1) Messages of level `result` always go to `stdout`
        //   (2) Messages of level `error` always go to `stderr`.
        //   (3a) All remaining messages go to `stderr`.
        //   (3b) If we are in CI mode, all remaining messages go to `stdout`.
        //
        switch (level) {
            case 'error':
                return process.stderr;
            case 'result':
                return process.stdout;
            default:
                return this.isCI ? process.stdout : process.stderr;
        }
    }
    /**
     * Notifies the host of a message that requires a response.
     *
     * If the host does not return a response the suggested
     * default response from the input message will be used.
     */
    async requestResponse(msg) {
        // If the request cannot be prompted for by the CliIoHost, we just accept the default
        if (!isPromptableRequest(msg)) {
            await this.notify(msg);
            return msg.defaultResponse;
        }
        const response = await this.withCorkedLogging(async () => {
            // prepare prompt data
            // @todo this format is not defined anywhere, probably should be
            const data = msg.data ?? {};
            const motivation = data.motivation ?? 'User input is needed';
            const concurrency = data.concurrency ?? 0;
            const responseDescription = data.responseDescription;
            // Special approval prompt
            // Determine if the message needs approval. If it does, continue (it is a basic confirmation prompt)
            // If it does not, return success (true). We only check messages with codes that we are aware
            // are requires approval codes.
            if (this.skipApprovalStep(msg)) {
                return true;
            }
            // In --yes mode, respond for the user if we can
            if (this.autoRespond) {
                // respond with yes to all confirmations
                if (isConfirmationPrompt(msg)) {
                    await this.notify({
                        ...msg,
                        message: `${chalk.cyan(msg.message)} (auto-confirmed)`,
                    });
                    return true;
                }
                // respond with the default for all other messages
                if (msg.defaultResponse) {
                    await this.notify({
                        ...msg,
                        message: `${chalk.cyan(msg.message)} (auto-responded with default: ${util.format(msg.defaultResponse)})`,
                    });
                    return msg.defaultResponse;
                }
            }
            // only talk to user if STDIN is a terminal (otherwise, fail)
            if (!this.isTTY) {
                throw new toolkit_lib_1.ToolkitError(`${motivation}, but terminal (TTY) is not attached so we are unable to get a confirmation from the user`);
            }
            // only talk to user if concurrency is 1 (otherwise, fail)
            if (concurrency > 1) {
                throw new toolkit_lib_1.ToolkitError(`${motivation}, but concurrency is greater than 1 so we are unable to get a confirmation from the user`);
            }
            // Basic confirmation prompt
            // We treat all requests with a boolean response as confirmation prompts
            if (isConfirmationPrompt(msg)) {
                const confirmed = await promptly.confirm(`${chalk.cyan(msg.message)} (y/n)`);
                if (!confirmed) {
                    throw new toolkit_lib_1.ToolkitError('Aborted by user');
                }
                return confirmed;
            }
            // Asking for a specific value
            const prompt = extractPromptInfo(msg);
            const desc = responseDescription ?? prompt.default;
            const answer = await promptly.prompt(`${chalk.cyan(msg.message)}${desc ? ` (${desc})` : ''}`, {
                default: prompt.default,
                trim: true,
            });
            return prompt.convertAnswer(answer);
        });
        // We need to cast this because it is impossible to narrow the generic type
        // isPromptableRequest ensures that the response type is one we can prompt for
        // the remaining code ensure we are indeed returning the correct type
        return response;
    }
    /**
     * Formats a message for console output with optional color support
     */
    formatMessage(msg) {
        // apply provided style or a default style if we're in TTY mode
        let message_text = this.isTTY
            ? styleMap[msg.level](msg.message)
            : msg.message;
        // prepend timestamp if IoMessageLevel is DEBUG or TRACE. Postpend a newline.
        return ((msg.level === 'debug' || msg.level === 'trace')
            ? `[${this.formatTime(msg.time)}] ${message_text}`
            : message_text) + '\n';
    }
    /**
     * Formats date to HH:MM:SS
     */
    formatTime(d) {
        const pad = (n) => n.toString().padStart(2, '0');
        return `${pad(d.getHours())}:${pad(d.getMinutes())}:${pad(d.getSeconds())}`;
    }
    /**
     * Get an instance of the ActivityPrinter
     */
    makeActivityPrinter() {
        const props = {
            stream: this.selectStreamFromLevel('info'),
        };
        switch (this.stackProgress) {
            case deploy_1.StackActivityProgress.EVENTS:
                return new api_private_1.HistoryActivityPrinter(props);
            case deploy_1.StackActivityProgress.BAR:
                return new api_private_1.CurrentActivityPrinter(props);
        }
    }
}
exports.CliIoHost = CliIoHost;
/**
 * This IoHost implementation considers a request promptable, if:
 * - it's a yes/no confirmation
 * - asking for a string or number value
 */
function isPromptableRequest(msg) {
    return isConfirmationPrompt(msg)
        || typeof msg.defaultResponse === 'string'
        || typeof msg.defaultResponse === 'number';
}
/**
 * Check if the request is a confirmation prompt
 * We treat all requests with a boolean response as confirmation prompts
 */
function isConfirmationPrompt(msg) {
    return typeof msg.defaultResponse === 'boolean';
}
/**
 * Helper to extract information for promptly from the request
 */
function extractPromptInfo(msg) {
    const isNumber = (typeof msg.defaultResponse === 'number');
    const defaultResponse = util.format(msg.defaultResponse);
    return {
        default: defaultResponse,
        defaultDesc: 'defaultDescription' in msg && msg.defaultDescription ? util.format(msg.defaultDescription) : defaultResponse,
        convertAnswer: isNumber ? (v) => Number(v) : (v) => String(v),
    };
}
const styleMap = {
    error: chalk.red,
    warn: chalk.yellow,
    result: chalk.reset,
    info: chalk.reset,
    debug: chalk.gray,
    trace: chalk.gray,
};
function targetStreamObject(x) {
    switch (x) {
        case 'stderr':
            return process.stderr;
        case 'stdout':
            return process.stdout;
        case 'drop':
            return undefined;
    }
}
function isNoticesMessage(msg) {
    return api_private_1.IO.CDK_TOOLKIT_I0100.is(msg) || api_private_1.IO.CDK_TOOLKIT_W0101.is(msg) || api_private_1.IO.CDK_TOOLKIT_E0101.is(msg) || api_private_1.IO.CDK_TOOLKIT_I0101.is(msg);
}
function isTelemetryMessage(msg) {
    return messages_1.CLI_TELEMETRY_CODES.some((c) => c.is(msg));
}
function getEventType(msg) {
    switch (msg.code) {
        case messages_1.CLI_PRIVATE_IO.CDK_CLI_I1001.code:
            return 'SYNTH';
        case messages_1.CLI_PRIVATE_IO.CDK_CLI_I2001.code:
            return 'INVOKE';
        case messages_1.CLI_PRIVATE_IO.CDK_CLI_I3001.code:
            return 'DEPLOY';
        default:
            throw new toolkit_lib_1.ToolkitError(`Unrecognized Telemetry Message Code: ${msg.code}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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