"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.contextHandler = contextHandler;
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const chalk = require("chalk");
const minimatch_1 = require("minimatch");
const display_version_1 = require("../cli/display-version");
const tables_1 = require("../cli/tables");
const user_configuration_1 = require("../cli/user-configuration");
async function contextHandler(options) {
    const ioHelper = options.ioHelper;
    if (options.clear) {
        options.context.clear();
        await options.context.save(user_configuration_1.PROJECT_CONTEXT);
        await ioHelper.defaults.info('All context values cleared.');
    }
    else if (options.reset) {
        await invalidateContext(ioHelper, options.context, options.reset, options.force ?? false);
        await options.context.save(user_configuration_1.PROJECT_CONTEXT);
    }
    else {
        // List -- support '--json' flag
        if (options.json) {
            /* c8 ignore start */
            const contextValues = options.context.all;
            await ioHelper.defaults.result(JSON.stringify(contextValues, undefined, 2));
            /* c8 ignore stop */
        }
        else {
            await listContext(ioHelper, options.context);
        }
    }
    await (0, display_version_1.displayVersionMessage)(ioHelper);
    return 0;
}
async function listContext(ioHelper, context) {
    const keys = contextKeys(context);
    if (keys.length === 0) {
        await ioHelper.defaults.info('This CDK application does not have any saved context values yet.');
        await ioHelper.defaults.info('');
        await ioHelper.defaults.info('Context will automatically be saved when you synthesize CDK apps');
        await ioHelper.defaults.info('that use environment context information like AZ information, VPCs,');
        await ioHelper.defaults.info('SSM parameters, and so on.');
        return;
    }
    // Print config by default
    const data_out = [[chalk.green('#'), chalk.green('Key'), chalk.green('Value')]];
    for (const [i, key] of keys) {
        const jsonWithoutNewlines = JSON.stringify(context.all[key], undefined, 2).replace(/\s+/g, ' ');
        data_out.push([i, key, jsonWithoutNewlines]);
    }
    await ioHelper.defaults.info('Context found in %s:', chalk.blue(user_configuration_1.PROJECT_CONFIG));
    await ioHelper.defaults.info('');
    await ioHelper.defaults.info((0, tables_1.renderTable)(data_out, process.stdout.columns));
    // eslint-disable-next-line @stylistic/max-len
    await ioHelper.defaults.info(`Run ${chalk.blue('cdk context --reset KEY_OR_NUMBER')} to remove a context key. It will be refreshed on the next CDK synthesis run.`);
}
async function invalidateContext(ioHelper, context, key, force) {
    const i = parseInt(key, 10);
    if (`${i}` === key) {
        // was a number and we fully parsed it.
        key = keyByNumber(context, i);
    }
    // Unset!
    if (context.has(key)) {
        context.unset(key);
        // check if the value was actually unset.
        if (!context.has(key)) {
            await ioHelper.defaults.info('Context value %s reset. It will be refreshed on next synthesis', chalk.blue(key));
            return;
        }
        // Value must be in readonly bag
        await ioHelper.defaults.error('Only context values specified in %s can be reset through the CLI', chalk.blue(user_configuration_1.PROJECT_CONTEXT));
        if (!force) {
            throw new toolkit_lib_1.ToolkitError(`Cannot reset readonly context value with key: ${key}`);
        }
    }
    // check if value is expression matching keys
    const matches = keysByExpression(context, key);
    if (matches.length > 0) {
        matches.forEach((match) => {
            context.unset(match);
        });
        const { unset, readonly } = getUnsetAndReadonly(context, matches);
        // output the reset values
        await printUnset(ioHelper, unset);
        // warn about values not reset
        await printReadonly(ioHelper, readonly);
        // throw when none of the matches were reset
        if (!force && unset.length === 0) {
            throw new toolkit_lib_1.ToolkitError('None of the matched context values could be reset');
        }
        return;
    }
    if (!force) {
        throw new toolkit_lib_1.ToolkitError(`No context value matching key: ${key}`);
    }
}
async function printUnset(ioHelper, unset) {
    if (unset.length === 0)
        return;
    await ioHelper.defaults.info('The following matched context values reset. They will be refreshed on next synthesis');
    for (const match of unset) {
        await ioHelper.defaults.info('  %s', match);
    }
}
async function printReadonly(ioHelper, readonly) {
    if (readonly.length === 0)
        return;
    await ioHelper.defaults.warn('The following matched context values could not be reset through the CLI');
    for (const match of readonly) {
        await ioHelper.defaults.info('  %s', match);
    }
    await ioHelper.defaults.info('');
    await ioHelper.defaults.info('This usually means they are configured in %s or %s', chalk.blue(user_configuration_1.PROJECT_CONFIG), chalk.blue(user_configuration_1.USER_DEFAULTS));
}
function keysByExpression(context, expression) {
    return context.keys.filter(minimatch_1.minimatch.filter(expression));
}
function getUnsetAndReadonly(context, matches) {
    return matches.reduce((acc, match) => {
        if (context.has(match)) {
            acc.readonly.push(match);
        }
        else {
            acc.unset.push(match);
        }
        return acc;
    }, { unset: [], readonly: [] });
}
function keyByNumber(context, n) {
    for (const [i, key] of contextKeys(context)) {
        if (n === i) {
            return key;
        }
    }
    throw new toolkit_lib_1.ToolkitError(`No context key with number: ${n}`);
}
/**
 * Return enumerated keys in a definitive order
 */
function contextKeys(context) {
    const keys = context.keys;
    keys.sort();
    return enumerate1(keys);
}
function enumerate1(xs) {
    const ret = new Array();
    let i = 1;
    for (const x of xs) {
        ret.push([i, x]);
        i += 1;
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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