"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlagOperations = void 0;
exports.isEffectiveValueEqualToRecommended = isEffectiveValueEqualToRecommended;
const os = require("os");
const path = require("path");
const cloudformation_diff_1 = require("@aws-cdk/cloudformation-diff");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const chalk = require("chalk");
const fs = require("fs-extra");
const p_queue_1 = require("p-queue");
const obsolete_flags_1 = require("./obsolete-flags");
const api_1 = require("../../api");
class FlagOperations {
    /**
     * Returns only those feature flags that need configuration
     *
     * That is those flags:
     * - That are unconfigured
     * - That are not obsolete
     * - Whose default value is different from the recommended value
     *
     * The default value being equal to the recommended value sounds odd, but
     * crops up in a number of situtations:
     *
     * - Security-related fixes that we want to force on people, but want to
     *   give them a flag to back out of the changes if they really need to.
     * - Flags that changed their default value in the most recent major
     *   version.
     * - Flags that we've introduced at some point in the past, but have gone
     *   back on.
     */
    static filterNeedsAttention(flags) {
        return flags
            .filter(flag => !obsolete_flags_1.OBSOLETE_FLAGS.includes(flag.name))
            .filter(flag => flag.userValue === undefined)
            .filter(flag => defaultValue(flag) !== flag.recommendedValue);
    }
    constructor(flags, toolkit, ioHelper, cliContextValues = {}) {
        this.flags = flags;
        this.toolkit = toolkit;
        this.ioHelper = ioHelper;
        this.cliContextValues = cliContextValues;
        this.app = '';
        this.baseContextValues = {};
        this.allStacks = [];
        this.queue = new p_queue_1.default({ concurrency: 4 });
    }
    /** Main entry point that routes to either flag setting or display operations */
    async execute(params) {
        if (params.set) {
            if (params.FLAGNAME && params.value) {
                await this.setFlag(params);
            }
            else {
                await this.setMultipleFlags(params);
            }
        }
        else {
            await this.displayFlags(params);
        }
    }
    /** Sets a single specific flag with validation and user confirmation */
    async setFlag(params) {
        const flagName = params.FLAGNAME[0];
        const flag = this.flags.find(f => f.name === flagName);
        if (!flag) {
            await this.ioHelper.defaults.error('Flag not found.');
            return;
        }
        if (!this.isBooleanFlag(flag)) {
            await this.ioHelper.defaults.error(`Flag '${flagName}' is not a boolean flag. Only boolean flags are currently supported.`);
            return;
        }
        const prototypeSuccess = await this.prototypeChanges([flagName], params);
        if (prototypeSuccess) {
            await this.handleUserResponse([flagName], params);
        }
    }
    /** Sets multiple flags (all or unconfigured) with validation and user confirmation */
    async setMultipleFlags(params) {
        if (params.default && !this.flags.some(f => f.unconfiguredBehavesLike)) {
            await this.ioHelper.defaults.error('The --default options are not compatible with the AWS CDK library used by your application. Please upgrade to 2.212.0 or above.');
            return;
        }
        const flagsToSet = this.getFlagsToSet(params);
        const prototypeSuccess = await this.prototypeChanges(flagsToSet, params);
        if (prototypeSuccess) {
            await this.handleUserResponse(flagsToSet, params);
        }
    }
    /** Determines which flags should be set based on the provided parameters */
    getFlagsToSet(params) {
        if (params.all && params.default) {
            return this.flags
                .filter(flag => this.isBooleanFlag(flag))
                .map(flag => flag.name);
        }
        else if (params.all) {
            return this.flags
                .filter(flag => flag.userValue === undefined || !isEffectiveValueEqualToRecommended(flag))
                .filter(flag => this.isBooleanFlag(flag))
                .map(flag => flag.name);
        }
        else {
            return this.flags
                .filter(flag => flag.userValue === undefined)
                .filter(flag => this.isBooleanFlag(flag))
                .map(flag => flag.name);
        }
    }
    /** Sets flags that don't cause template changes */
    async setSafeFlags(params) {
        const cdkJson = await JSON.parse(await fs.readFile(path.join(process.cwd(), 'cdk.json'), 'utf-8'));
        this.app = params.app || cdkJson.app;
        const isUsingTsNode = this.app.includes('ts-node');
        if (isUsingTsNode && !this.app.includes('-T') && !this.app.includes('--transpileOnly')) {
            await this.ioHelper.defaults.info('Repeated synths with ts-node will type-check the application on every synth. Add --transpileOnly to cdk.json\'s "app" command to make this operation faster.');
        }
        const unconfiguredFlags = this.flags.filter(flag => flag.userValue === undefined && this.isBooleanFlag(flag));
        if (unconfiguredFlags.length === 0) {
            await this.ioHelper.defaults.info('All feature flags are configured.');
            return;
        }
        await this.initializeSafetyCheck();
        const safeFlags = await this.batchTestFlags(unconfiguredFlags);
        await this.cleanupSafetyCheck();
        if (safeFlags.length > 0) {
            await this.ioHelper.defaults.info('Flags that can be set without template changes:');
            for (const flag of safeFlags) {
                await this.ioHelper.defaults.info(`- ${flag.name} -> ${flag.recommendedValue}`);
            }
            await this.handleUserResponse(safeFlags.map(flag => flag.name), { ...params, recommended: true });
        }
        else {
            await this.ioHelper.defaults.info('No more flags can be set without causing template changes.');
        }
    }
    /** Initializes the safety check by reading context and synthesizing baseline templates */
    async initializeSafetyCheck() {
        const baseContext = new toolkit_lib_1.CdkAppMultiContext(process.cwd());
        this.baseContextValues = { ...await baseContext.read(), ...this.cliContextValues };
        this.baselineTempDir = fs.mkdtempSync(path.join(os.tmpdir(), 'cdk-baseline-'));
        const mergedContext = new toolkit_lib_1.MemoryContext(this.baseContextValues);
        const baseSource = await this.toolkit.fromCdkApp(this.app, {
            contextStore: mergedContext,
            outdir: this.baselineTempDir,
        });
        const baseCx = await this.toolkit.synth(baseSource);
        const baseAssembly = baseCx.cloudAssembly;
        this.allStacks = baseAssembly.stacksRecursively;
        this.queue = new p_queue_1.default({ concurrency: 4 });
    }
    /** Cleans up temporary directories created during safety checks */
    async cleanupSafetyCheck() {
        if (this.baselineTempDir) {
            await fs.remove(this.baselineTempDir);
            this.baselineTempDir = undefined;
        }
    }
    /** Tests multiple flags together and isolates unsafe ones using binary search */
    async batchTestFlags(flags) {
        if (flags.length === 0)
            return [];
        const allFlagsContext = { ...this.baseContextValues };
        flags.forEach(flag => {
            allFlagsContext[flag.name] = flag.recommendedValue;
        });
        const allSafe = await this.testBatch(allFlagsContext);
        if (allSafe)
            return flags;
        return this.isolateUnsafeFlags(flags);
    }
    /** Tests if a set of context values causes template changes by synthesizing and diffing */
    async testBatch(contextValues) {
        const testContext = new toolkit_lib_1.MemoryContext(contextValues);
        const tempDir = fs.mkdtempSync(path.join(os.tmpdir(), 'cdk-test-'));
        const testSource = await this.toolkit.fromCdkApp(this.app, {
            contextStore: testContext,
            outdir: tempDir,
        });
        const testCx = await this.toolkit.synth(testSource);
        try {
            for (const stack of this.allStacks) {
                const templatePath = stack.templateFullPath;
                const diff = await this.toolkit.diff(testCx, {
                    method: toolkit_lib_1.DiffMethod.LocalFile(templatePath),
                    stacks: {
                        strategy: api_1.StackSelectionStrategy.PATTERN_MUST_MATCH_SINGLE,
                        patterns: [stack.hierarchicalId],
                    },
                });
                for (const stackDiff of Object.values(diff)) {
                    if (stackDiff.differenceCount > 0) {
                        return false;
                    }
                }
            }
            return true;
        }
        finally {
            await fs.remove(tempDir);
        }
    }
    /** Uses binary search to isolate which flags are safe to set without template changes */
    async isolateUnsafeFlags(flags) {
        const safeFlags = [];
        const processBatch = async (batch, contextValues) => {
            if (batch.length === 1) {
                const isSafe = await this.testBatch({ ...contextValues, [batch[0].name]: batch[0].recommendedValue });
                if (isSafe)
                    safeFlags.push(batch[0]);
                return;
            }
            const batchContext = { ...contextValues };
            batch.forEach(flag => {
                batchContext[flag.name] = flag.recommendedValue;
            });
            const isSafeBatch = await this.testBatch(batchContext);
            if (isSafeBatch) {
                safeFlags.push(...batch);
                return;
            }
            const mid = Math.floor(batch.length / 2);
            const left = batch.slice(0, mid);
            const right = batch.slice(mid);
            void this.queue.add(() => processBatch(left, contextValues));
            void this.queue.add(() => processBatch(right, contextValues));
        };
        void this.queue.add(() => processBatch(flags, this.baseContextValues));
        await this.queue.onIdle();
        return safeFlags;
    }
    /** Prototypes flag changes by synthesizing templates and showing diffs to the user */
    async prototypeChanges(flagNames, params) {
        const baseContext = new toolkit_lib_1.CdkAppMultiContext(process.cwd());
        const baseContextValues = { ...await baseContext.read(), ...this.cliContextValues };
        const memoryContext = new toolkit_lib_1.MemoryContext(baseContextValues);
        const cdkJson = await JSON.parse(await fs.readFile(path.join(process.cwd(), 'cdk.json'), 'utf-8'));
        const app = cdkJson.app;
        const source = await this.toolkit.fromCdkApp(app, {
            contextStore: memoryContext,
            outdir: fs.mkdtempSync(path.join(os.tmpdir(), 'cdk-original-')),
        });
        const updateObj = await this.buildUpdateObject(flagNames, params, baseContextValues);
        if (!updateObj)
            return false;
        await memoryContext.update(updateObj);
        const cx = await this.toolkit.synth(source);
        const assembly = cx.cloudAssembly;
        const modifiedSource = await this.toolkit.fromCdkApp(app, {
            contextStore: memoryContext,
            outdir: fs.mkdtempSync(path.join(os.tmpdir(), 'cdk-temp-')),
        });
        const modifiedCx = await this.toolkit.synth(modifiedSource);
        const allStacks = assembly.stacksRecursively;
        for (const stack of allStacks) {
            const templatePath = stack.templateFullPath;
            await this.toolkit.diff(modifiedCx, {
                method: toolkit_lib_1.DiffMethod.LocalFile(templatePath),
                stacks: {
                    strategy: api_1.StackSelectionStrategy.PATTERN_MUST_MATCH_SINGLE,
                    patterns: [stack.hierarchicalId],
                },
            });
        }
        await this.displayFlagChanges(updateObj, baseContextValues);
        return true;
    }
    /** Displays a summary of flag changes showing old and new values */
    async displayFlagChanges(updateObj, baseContextValues) {
        await this.ioHelper.defaults.info('\nFlag changes:');
        for (const [flagName, newValue] of Object.entries(updateObj)) {
            const currentValue = baseContextValues[flagName];
            const currentDisplay = currentValue === undefined ? '<unset>' : String(currentValue);
            await this.ioHelper.defaults.info(`  ${flagName}: ${currentDisplay} → ${newValue}`);
        }
    }
    /** Builds the update object with new flag values based on parameters and current context */
    async buildUpdateObject(flagNames, params, baseContextValues) {
        const updateObj = {};
        if (flagNames.length === 1 && params.value !== undefined) {
            const flagName = flagNames[0];
            const boolValue = params.value === 'true';
            if (baseContextValues[flagName] === boolValue) {
                await this.ioHelper.defaults.info('Flag is already set to the specified value. No changes needed.');
                return null;
            }
            updateObj[flagName] = boolValue;
        }
        else {
            for (const flagName of flagNames) {
                const flag = this.flags.find(f => f.name === flagName);
                if (!flag) {
                    await this.ioHelper.defaults.error(`Flag ${flagName} not found.`);
                    return null;
                }
                const newValue = params.recommended
                    ? flag.recommendedValue
                    : String(defaultValue(flag)) === 'true';
                updateObj[flagName] = newValue;
            }
        }
        return updateObj;
    }
    /** Prompts user for confirmation and applies changes if accepted */
    async handleUserResponse(flagNames, params) {
        const userAccepted = await this.ioHelper.requestResponse({
            time: new Date(),
            level: 'info',
            code: 'CDK_TOOLKIT_I9300',
            message: 'Do you want to accept these changes?',
            data: {
                flagNames,
                responseDescription: 'Enter "y" to apply changes or "n" to cancel',
            },
            defaultResponse: false,
        });
        if (userAccepted) {
            await this.modifyValues(flagNames, params);
            await this.ioHelper.defaults.info('Flag value(s) updated successfully.');
        }
        else {
            await this.ioHelper.defaults.info('Operation cancelled');
        }
        await this.cleanupTempDirectories();
    }
    /** Removes temporary directories created during flag operations */
    async cleanupTempDirectories() {
        const originalDir = path.join(process.cwd(), 'original');
        const tempDir = path.join(process.cwd(), 'temp');
        await fs.remove(originalDir);
        await fs.remove(tempDir);
    }
    /** Actually modifies the cdk.json file with the new flag values */
    async modifyValues(flagNames, params) {
        const cdkJsonPath = path.join(process.cwd(), 'cdk.json');
        const cdkJsonContent = await fs.readFile(cdkJsonPath, 'utf-8');
        const cdkJson = JSON.parse(cdkJsonContent);
        if (flagNames.length === 1 && !params.safe) {
            const boolValue = params.value === 'true';
            cdkJson.context[String(flagNames[0])] = boolValue;
            await this.ioHelper.defaults.info(`Setting flag '${flagNames}' to: ${boolValue}`);
        }
        else {
            for (const flagName of flagNames) {
                const flag = this.flags.find(f => f.name === flagName);
                const newValue = params.recommended || params.safe
                    ? flag.recommendedValue
                    : String(defaultValue(flag)) === 'true';
                cdkJson.context[flagName] = newValue;
            }
        }
        await fs.writeFile(cdkJsonPath, JSON.stringify(cdkJson, null, 2), 'utf-8');
    }
    /** Displays flags in a table format, either specific flags or filtered by criteria */
    async displayFlags(params) {
        const { FLAGNAME, all } = params;
        if (FLAGNAME && FLAGNAME.length > 0) {
            await this.displaySpecificFlags(FLAGNAME);
            return;
        }
        const [flagsToDisplay, header] = all
            ? [this.flags, 'All feature flags']
            : [FlagOperations.filterNeedsAttention(this.flags), 'Unconfigured feature flags'];
        await this.ioHelper.defaults.info(header);
        await this.displayFlagTable(flagsToDisplay);
        // Add helpful message after empty table when not using --all
        if (!all && flagsToDisplay.length === 0) {
            await this.ioHelper.defaults.info('');
            await this.ioHelper.defaults.info('✅ All feature flags are already set to their recommended values.');
            await this.ioHelper.defaults.info('Use \'cdk flags --all --unstable=flags\' to see all flags and their current values.');
        }
    }
    /** Displays detailed information for specific flags matching the given names */
    async displaySpecificFlags(flagNames) {
        const matchingFlags = this.flags.filter(f => flagNames.some(searchTerm => f.name.toLowerCase().includes(searchTerm.toLowerCase())));
        if (matchingFlags.length === 0) {
            await this.ioHelper.defaults.error(`Flag matching "${flagNames.join(', ')}" not found.`);
            return;
        }
        if (matchingFlags.length === 1) {
            const flag = matchingFlags[0];
            await this.ioHelper.defaults.info(`Flag name: ${flag.name}`);
            await this.ioHelper.defaults.info(`Description: ${flag.explanation}`);
            await this.ioHelper.defaults.info(`Recommended value: ${flag.recommendedValue}`);
            await this.ioHelper.defaults.info(`Default value: ${defaultValue(flag)}`);
            await this.ioHelper.defaults.info(`User value: ${flag.userValue}`);
            await this.ioHelper.defaults.info(`Effective value: ${effectiveValue(flag)}`);
            return;
        }
        await this.ioHelper.defaults.info(`Found ${matchingFlags.length} flags matching "${flagNames.join(', ')}"`);
        await this.displayFlagTable(matchingFlags);
    }
    /** Returns sort order for flags */
    getFlagSortOrder(flag) {
        if (flag.userValue === undefined)
            return 3;
        if (isEffectiveValueEqualToRecommended(flag))
            return 1;
        return 2;
    }
    /** Displays flags in a formatted table grouped by module and sorted */
    async displayFlagTable(flags) {
        const sortedFlags = [...flags].sort((a, b) => {
            const orderA = this.getFlagSortOrder(a);
            const orderB = this.getFlagSortOrder(b);
            if (orderA !== orderB)
                return orderA - orderB;
            if (a.module !== b.module)
                return a.module.localeCompare(b.module);
            return a.name.localeCompare(b.name);
        });
        const rows = [['Feature Flag', 'Recommended', 'User', 'Effective']];
        let currentModule = '';
        sortedFlags.forEach((flag) => {
            if (flag.module !== currentModule) {
                rows.push([chalk.bold(`Module: ${flag.module}`), '', '', '']);
                currentModule = flag.module;
            }
            rows.push([
                `  ${flag.name}`,
                String(flag.recommendedValue),
                flag.userValue === undefined ? '<unset>' : String(flag.userValue),
                String(effectiveValue(flag)),
            ]);
        });
        const formattedTable = (0, cloudformation_diff_1.formatTable)(rows, undefined, true);
        await this.ioHelper.defaults.info(formattedTable);
    }
    /** Checks if a flag has a boolean recommended value */
    isBooleanFlag(flag) {
        const recommended = flag.recommendedValue;
        return typeof recommended === 'boolean' ||
            recommended === 'true' ||
            recommended === 'false';
    }
    /** Shows helpful usage examples and available command options */
    async displayHelpMessage() {
        await this.ioHelper.defaults.info('\n' + chalk.bold('Available options:'));
        await this.ioHelper.defaults.info('  cdk flags --interactive     # Interactive menu to manage flags');
        await this.ioHelper.defaults.info('  cdk flags --all             # Show all flags (including configured ones)');
        await this.ioHelper.defaults.info('  cdk flags --set --all --recommended    # Set all flags to recommended values');
        await this.ioHelper.defaults.info('  cdk flags --set --all --default       # Set all flags to default values');
        await this.ioHelper.defaults.info('  cdk flags --set --unconfigured --recommended  # Set unconfigured flags to recommended');
        await this.ioHelper.defaults.info('  cdk flags --set <flag-name> --value <true|false>  # Set specific flag');
        await this.ioHelper.defaults.info('  cdk flags --safe            # Safely set flags that don\'t change templates');
    }
}
exports.FlagOperations = FlagOperations;
/** Checks if the flags current effective value matches the recommended value */
function isEffectiveValueEqualToRecommended(flag) {
    return String(effectiveValue(flag)) === String(flag.recommendedValue);
}
/**
 * Return the effective value of a flag (user value or default)
 */
function effectiveValue(flag) {
    return flag.userValue ?? defaultValue(flag);
}
/**
 * Return the default value for a flag, assume it's `false` if not given
 */
function defaultValue(flag) {
    return flag.unconfiguredBehavesLike?.v2 ?? false;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3BlcmF0aW9ucy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm9wZXJhdGlvbnMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBcWdCQSxnRkFFQztBQXZnQkQseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUU3QixzRUFBMkQ7QUFFM0Qsc0RBQXFGO0FBQ3JGLCtCQUErQjtBQUMvQiwrQkFBK0I7QUFDL0IscUNBQTZCO0FBQzdCLHFEQUFrRDtBQUVsRCxtQ0FBbUQ7QUFHbkQsTUFBYSxjQUFjO0lBQ3pCOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFvQjtRQUNyRCxPQUFPLEtBQUs7YUFDVCxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLCtCQUFjLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUNuRCxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsU0FBUyxLQUFLLFNBQVMsQ0FBQzthQUM1QyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbEUsQ0FBQztJQVFELFlBQ21CLEtBQW9CLEVBQ3BCLE9BQWdCLEVBQ2hCLFFBQWtCLEVBQ2xCLG1CQUF3QyxFQUFFO1FBSDFDLFVBQUssR0FBTCxLQUFLLENBQWU7UUFDcEIsWUFBTyxHQUFQLE9BQU8sQ0FBUztRQUNoQixhQUFRLEdBQVIsUUFBUSxDQUFVO1FBQ2xCLHFCQUFnQixHQUFoQixnQkFBZ0IsQ0FBMEI7UUFFM0QsSUFBSSxDQUFDLEdBQUcsR0FBRyxFQUFFLENBQUM7UUFDZCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsRUFBRSxDQUFDO1FBQzVCLElBQUksQ0FBQyxTQUFTLEdBQUcsRUFBRSxDQUFDO1FBQ3BCLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxpQkFBTSxDQUFDLEVBQUUsV0FBVyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVELGdGQUFnRjtJQUNoRixLQUFLLENBQUMsT0FBTyxDQUFDLE1BQTRCO1FBQ3hDLElBQUksTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDO1lBQ2YsSUFBSSxNQUFNLENBQUMsUUFBUSxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsQ0FBQztnQkFDcEMsTUFBTSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQzdCLENBQUM7aUJBQU0sQ0FBQztnQkFDTixNQUFNLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUN0QyxDQUFDO1FBQ0gsQ0FBQzthQUFNLENBQUM7WUFDTixNQUFNLElBQUksQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDbEMsQ0FBQztJQUNILENBQUM7SUFFRCx3RUFBd0U7SUFDeEUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUE0QjtRQUN4QyxNQUFNLFFBQVEsR0FBRyxNQUFNLENBQUMsUUFBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3JDLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxRQUFRLENBQUMsQ0FBQztRQUV2RCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDVixNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1lBQ3RELE9BQU87UUFDVCxDQUFDO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxTQUFTLFFBQVEsc0VBQXNFLENBQUMsQ0FBQztZQUM1SCxPQUFPO1FBQ1QsQ0FBQztRQUVELE1BQU0sZ0JBQWdCLEdBQUcsTUFBTSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxRQUFRLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUN6RSxJQUFJLGdCQUFnQixFQUFFLENBQUM7WUFDckIsTUFBTSxJQUFJLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxRQUFRLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUNwRCxDQUFDO0lBQ0gsQ0FBQztJQUVELHNGQUFzRjtJQUN0RixLQUFLLENBQUMsZ0JBQWdCLENBQUMsTUFBNEI7UUFDakQsSUFBSSxNQUFNLENBQUMsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsdUJBQXVCLENBQUMsRUFBRSxDQUFDO1lBQ3ZFLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLGlJQUFpSSxDQUFDLENBQUM7WUFDdEssT0FBTztRQUNULENBQUM7UUFFRCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzlDLE1BQU0sZ0JBQWdCLEdBQUcsTUFBTSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRXpFLElBQUksZ0JBQWdCLEVBQUUsQ0FBQztZQUNyQixNQUFNLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxVQUFVLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDcEQsQ0FBQztJQUNILENBQUM7SUFFRCw0RUFBNEU7SUFDcEUsYUFBYSxDQUFDLE1BQTRCO1FBQ2hELElBQUksTUFBTSxDQUFDLEdBQUcsSUFBSSxNQUFNLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDakMsT0FBTyxJQUFJLENBQUMsS0FBSztpQkFDZCxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO2lCQUN4QyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDNUIsQ0FBQzthQUFNLElBQUksTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDO1lBQ3RCLE9BQU8sSUFBSSxDQUFDLEtBQUs7aUJBQ2QsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFNBQVMsS0FBSyxTQUFTLElBQUksQ0FBQyxrQ0FBa0MsQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDekYsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDeEMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzVCLENBQUM7YUFBTSxDQUFDO1lBQ04sT0FBTyxJQUFJLENBQUMsS0FBSztpQkFDZCxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsU0FBUyxLQUFLLFNBQVMsQ0FBQztpQkFDNUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDeEMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzVCLENBQUM7SUFDSCxDQUFDO0lBRUQsbURBQW1EO0lBQ25ELEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBNEI7UUFDN0MsTUFBTSxPQUFPLEdBQUcsTUFBTSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsRUFBRSxVQUFVLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQ25HLElBQUksQ0FBQyxHQUFHLEdBQUcsTUFBTSxDQUFDLEdBQUcsSUFBSSxPQUFPLENBQUMsR0FBRyxDQUFDO1FBRXJDLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ25ELElBQUksYUFBYSxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLENBQUM7WUFDdkYsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsOEpBQThKLENBQUMsQ0FBQztRQUNwTSxDQUFDO1FBRUQsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUNqRCxJQUFJLENBQUMsU0FBUyxLQUFLLFNBQVMsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7UUFFNUQsSUFBSSxpQkFBaUIsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDbkMsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsbUNBQW1DLENBQUMsQ0FBQztZQUN2RSxPQUFPO1FBQ1QsQ0FBQztRQUVELE1BQU0sSUFBSSxDQUFDLHFCQUFxQixFQUFFLENBQUM7UUFDbkMsTUFBTSxTQUFTLEdBQUcsTUFBTSxJQUFJLENBQUMsY0FBYyxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDL0QsTUFBTSxJQUFJLENBQUMsa0JBQWtCLEVBQUUsQ0FBQztRQUVoQyxJQUFJLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFLENBQUM7WUFDekIsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsaURBQWlELENBQUMsQ0FBQztZQUNyRixLQUFLLE1BQU0sSUFBSSxJQUFJLFNBQVMsRUFBRSxDQUFDO2dCQUM3QixNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxLQUFLLElBQUksQ0FBQyxJQUFJLE9BQU8sSUFBSSxDQUFDLGdCQUFnQixFQUFFLENBQUMsQ0FBQztZQUNsRixDQUFDO1lBQ0QsTUFBTSxJQUFJLENBQUMsa0JBQWtCLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLEdBQUcsTUFBTSxFQUFFLFdBQVcsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ3BHLENBQUM7YUFBTSxDQUFDO1lBQ04sTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsNERBQTRELENBQUMsQ0FBQztRQUNsRyxDQUFDO0lBQ0gsQ0FBQztJQUVELDBGQUEwRjtJQUNsRixLQUFLLENBQUMscUJBQXFCO1FBQ2pDLE1BQU0sV0FBVyxHQUFHLElBQUksZ0NBQWtCLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFDMUQsSUFBSSxDQUFDLGlCQUFpQixHQUFHLEVBQUUsR0FBRyxNQUFNLFdBQVcsQ0FBQyxJQUFJLEVBQUUsRUFBRSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO1FBRW5GLElBQUksQ0FBQyxlQUFlLEdBQUcsRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxlQUFlLENBQUMsQ0FBQyxDQUFDO1FBQy9FLE1BQU0sYUFBYSxHQUFHLElBQUksMkJBQWEsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUNoRSxNQUFNLFVBQVUsR0FBRyxNQUFNLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUU7WUFDekQsWUFBWSxFQUFFLGFBQWE7WUFDM0IsTUFBTSxFQUFFLElBQUksQ0FBQyxlQUFlO1NBQzdCLENBQUMsQ0FBQztRQUVILE1BQU0sTUFBTSxHQUFHLE1BQU0sSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDcEQsTUFBTSxZQUFZLEdBQUcsTUFBTSxDQUFDLGFBQWEsQ0FBQztRQUMxQyxJQUFJLENBQUMsU0FBUyxHQUFHLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQztRQUNoRCxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksaUJBQU0sQ0FBQyxFQUFFLFdBQVcsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRCxtRUFBbUU7SUFDM0QsS0FBSyxDQUFDLGtCQUFrQjtRQUM5QixJQUFJLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztZQUN6QixNQUFNLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1lBQ3RDLElBQUksQ0FBQyxlQUFlLEdBQUcsU0FBUyxDQUFDO1FBQ25DLENBQUM7SUFDSCxDQUFDO0lBRUQsaUZBQWlGO0lBQ3pFLEtBQUssQ0FBQyxjQUFjLENBQUMsS0FBb0I7UUFDL0MsSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLENBQUM7WUFBRSxPQUFPLEVBQUUsQ0FBQztRQUVsQyxNQUFNLGVBQWUsR0FBRyxFQUFFLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixFQUFFLENBQUM7UUFDdEQsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUNuQixlQUFlLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQztRQUNyRCxDQUFDLENBQUMsQ0FBQztRQUVILE1BQU0sT0FBTyxHQUFHLE1BQU0sSUFBSSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUN0RCxJQUFJLE9BQU87WUFBRSxPQUFPLEtBQUssQ0FBQztRQUUxQixPQUFPLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQsMkZBQTJGO0lBQ25GLEtBQUssQ0FBQyxTQUFTLENBQUMsYUFBa0M7UUFDeEQsTUFBTSxXQUFXLEdBQUcsSUFBSSwyQkFBYSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3JELE1BQU0sT0FBTyxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUUsV0FBVyxDQUFDLENBQUMsQ0FBQztRQUNwRSxNQUFNLFVBQVUsR0FBRyxNQUFNLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUU7WUFDekQsWUFBWSxFQUFFLFdBQVc7WUFDekIsTUFBTSxFQUFFLE9BQU87U0FDaEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxNQUFNLEdBQUcsTUFBTSxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUVwRCxJQUFJLENBQUM7WUFDSCxLQUFLLE1BQU0sS0FBSyxJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUUsQ0FBQztnQkFDbkMsTUFBTSxZQUFZLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO2dCQUM1QyxNQUFNLElBQUksR0FBRyxNQUFNLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRTtvQkFDM0MsTUFBTSxFQUFFLHdCQUFVLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQztvQkFDMUMsTUFBTSxFQUFFO3dCQUNOLFFBQVEsRUFBRSw0QkFBc0IsQ0FBQyx5QkFBeUI7d0JBQzFELFFBQVEsRUFBRSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7cUJBQ2pDO2lCQUNGLENBQUMsQ0FBQztnQkFFSCxLQUFLLE1BQU0sU0FBUyxJQUFJLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztvQkFDNUMsSUFBSSxTQUFTLENBQUMsZUFBZSxHQUFHLENBQUMsRUFBRSxDQUFDO3dCQUNsQyxPQUFPLEtBQUssQ0FBQztvQkFDZixDQUFDO2dCQUNILENBQUM7WUFDSCxDQUFDO1lBQ0QsT0FBTyxJQUFJLENBQUM7UUFDZCxDQUFDO2dCQUFTLENBQUM7WUFDVCxNQUFNLEVBQUUsQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDM0IsQ0FBQztJQUNILENBQUM7SUFFRCx5RkFBeUY7SUFDakYsS0FBSyxDQUFDLGtCQUFrQixDQUFDLEtBQW9CO1FBQ25ELE1BQU0sU0FBUyxHQUFrQixFQUFFLENBQUM7UUFFcEMsTUFBTSxZQUFZLEdBQUcsS0FBSyxFQUFFLEtBQW9CLEVBQUUsYUFBa0MsRUFBaUIsRUFBRTtZQUNyRyxJQUFJLEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFLENBQUM7Z0JBQ3ZCLE1BQU0sTUFBTSxHQUFHLE1BQU0sSUFBSSxDQUFDLFNBQVMsQ0FDakMsRUFBRSxHQUFHLGFBQWEsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLEVBQUUsQ0FDakUsQ0FBQztnQkFDRixJQUFJLE1BQU07b0JBQUUsU0FBUyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDckMsT0FBTztZQUNULENBQUM7WUFFRCxNQUFNLFlBQVksR0FBRyxFQUFFLEdBQUcsYUFBYSxFQUFFLENBQUM7WUFDMUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRTtnQkFDbkIsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUM7WUFDbEQsQ0FBQyxDQUFDLENBQUM7WUFFSCxNQUFNLFdBQVcsR0FBRyxNQUFNLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLENBQUM7WUFDdkQsSUFBSSxXQUFXLEVBQUUsQ0FBQztnQkFDaEIsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLEtBQUssQ0FBQyxDQUFDO2dCQUN6QixPQUFPO1lBQ1QsQ0FBQztZQUVELE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQztZQUN6QyxNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQztZQUNqQyxNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBRS9CLEtBQUssSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxhQUFhLENBQUMsQ0FBQyxDQUFDO1lBQzdELEtBQUssSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxhQUFhLENBQUMsQ0FBQyxDQUFDO1FBQ2hFLENBQUMsQ0FBQztRQUVGLEtBQUssSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDO1FBQ3ZFLE1BQU0sSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUMxQixPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRUQsc0ZBQXNGO0lBQzlFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFtQixFQUFFLE1BQTRCO1FBQzlFLE1BQU0sV0FBVyxHQUFHLElBQUksZ0NBQWtCLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFDMUQsTUFBTSxpQkFBaUIsR0FBRyxFQUFFLEdBQUcsTUFBTSxXQUFXLENBQUMsSUFBSSxFQUFFLEVBQUUsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQztRQUNwRixNQUFNLGFBQWEsR0FBRyxJQUFJLDJCQUFhLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUUzRCxNQUFNLE9BQU8sR0FBRyxNQUFNLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUFFLFVBQVUsQ0FBQyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDbkcsTUFBTSxHQUFHLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQztRQUV4QixNQUFNLE1BQU0sR0FBRyxNQUFNLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEdBQUcsRUFBRTtZQUNoRCxZQUFZLEVBQUUsYUFBYTtZQUMzQixNQUFNLEVBQUUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxlQUFlLENBQUMsQ0FBQztTQUNoRSxDQUFDLENBQUM7UUFFSCxNQUFNLFNBQVMsR0FBRyxNQUFNLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLEVBQUUsTUFBTSxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFDckYsSUFBSSxDQUFDLFNBQVM7WUFBRSxPQUFPLEtBQUssQ0FBQztRQUU3QixNQUFNLGFBQWEsQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdEMsTUFBTSxFQUFFLEdBQUcsTUFBTSxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM1QyxNQUFNLFFBQVEsR0FBRyxFQUFFLENBQUMsYUFBYSxDQUFDO1FBRWxDLE1BQU0sY0FBYyxHQUFHLE1BQU0sSUFBSSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsR0FBRyxFQUFFO1lBQ3hELFlBQVksRUFBRSxhQUFhO1lBQzNCLE1BQU0sRUFBRSxFQUFFLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1NBQzVELENBQUMsQ0FBQztRQUVILE1BQU0sVUFBVSxHQUFHLE1BQU0sSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDNUQsTUFBTSxTQUFTLEdBQUcsUUFBUSxDQUFDLGlCQUFpQixDQUFDO1FBRTdDLEtBQUssTUFBTSxLQUFLLElBQUksU0FBUyxFQUFFLENBQUM7WUFDOUIsTUFBTSxZQUFZLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO1lBQzVDLE1BQU0sSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFO2dCQUNsQyxNQUFNLEVBQUUsd0JBQVUsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDO2dCQUMxQyxNQUFNLEVBQUU7b0JBQ04sUUFBUSxFQUFFLDRCQUFzQixDQUFDLHlCQUF5QjtvQkFDMUQsUUFBUSxFQUFFLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztpQkFDakM7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDO1FBRUQsTUFBTSxJQUFJLENBQUMsa0JBQWtCLENBQUMsU0FBUyxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFDNUQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQsb0VBQW9FO0lBQzVELEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxTQUFrQyxFQUFFLGlCQUFzQztRQUN6RyxNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQ3JELEtBQUssTUFBTSxDQUFDLFFBQVEsRUFBRSxRQUFRLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUM7WUFDN0QsTUFBTSxZQUFZLEdBQUcsaUJBQWlCLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDakQsTUFBTSxjQUFjLEdBQUcsWUFBWSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDLENBQUM7WUFDckYsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxRQUFRLEtBQUssY0FBYyxNQUFNLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFDdEYsQ0FBQztJQUNILENBQUM7SUFFRCw0RkFBNEY7SUFDcEYsS0FBSyxDQUFDLGlCQUFpQixDQUFDLFNBQW1CLEVBQUUsTUFBNEIsRUFBRSxpQkFBc0M7UUFFdkgsTUFBTSxTQUFTLEdBQTRCLEVBQUUsQ0FBQztRQUU5QyxJQUFJLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEtBQUssU0FBUyxFQUFFLENBQUM7WUFDekQsTUFBTSxRQUFRLEdBQUcsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzlCLE1BQU0sU0FBUyxHQUFHLE1BQU0sQ0FBQyxLQUFLLEtBQUssTUFBTSxDQUFDO1lBQzFDLElBQUksaUJBQWlCLENBQUMsUUFBUSxDQUFDLEtBQUssU0FBUyxFQUFFLENBQUM7Z0JBQzlDLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGdFQUFnRSxDQUFDLENBQUM7Z0JBQ3BHLE9BQU8sSUFBSSxDQUFDO1lBQ2QsQ0FBQztZQUNELFNBQVMsQ0FBQyxRQUFRLENBQUMsR0FBRyxTQUFTLENBQUM7UUFDbEMsQ0FBQzthQUFNLENBQUM7WUFDTixLQUFLLE1BQU0sUUFBUSxJQUFJLFNBQVMsRUFBRSxDQUFDO2dCQUNqQyxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLEtBQUssUUFBUSxDQUFDLENBQUM7Z0JBQ3ZELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztvQkFDVixNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxRQUFRLFFBQVEsYUFBYSxDQUFDLENBQUM7b0JBQ2xFLE9BQU8sSUFBSSxDQUFDO2dCQUNkLENBQUM7Z0JBQ0QsTUFBTSxRQUFRLEdBQUcsTUFBTSxDQUFDLFdBQVc7b0JBQ2pDLENBQUMsQ0FBQyxJQUFJLENBQUMsZ0JBQTJCO29CQUNsQyxDQUFDLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQyxLQUFLLE1BQU0sQ0FBQztnQkFDMUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxHQUFHLFFBQVEsQ0FBQztZQUNqQyxDQUFDO1FBQ0gsQ0FBQztRQUNELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7SUFFRCxvRUFBb0U7SUFDNUQsS0FBSyxDQUFDLGtCQUFrQixDQUFDLFNBQW1CLEVBQUUsTUFBNEI7UUFDaEYsTUFBTSxZQUFZLEdBQUcsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FBQztZQUN2RCxJQUFJLEVBQUUsSUFBSSxJQUFJLEVBQUU7WUFDaEIsS0FBSyxFQUFFLE1BQU07WUFDYixJQUFJLEVBQUUsbUJBQW1CO1lBQ3pCLE9BQU8sRUFBRSxzQ0FBc0M7WUFDL0MsSUFBSSxFQUFFO2dCQUNKLFNBQVM7Z0JBQ1QsbUJBQW1CLEVBQUUsNkNBQTZDO2FBQ25FO1lBQ0QsZUFBZSxFQUFFLEtBQUs7U0FDdkIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxZQUFZLEVBQUUsQ0FBQztZQUNqQixNQUFNLElBQUksQ0FBQyxZQUFZLENBQUMsU0FBUyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBQzNDLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDM0UsQ0FBQzthQUFNLENBQUM7WUFDTixNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQzNELENBQUM7UUFFRCxNQUFNLElBQUksQ0FBQyxzQkFBc0IsRUFBRSxDQUFDO0lBQ3RDLENBQUM7SUFFRCxtRUFBbUU7SUFDM0QsS0FBSyxDQUFDLHNCQUFzQjtRQUNsQyxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUN6RCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUNqRCxNQUFNLEVBQUUsQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDN0IsTUFBTSxFQUFFLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQzNCLENBQUM7SUFFRCxtRUFBbUU7SUFDM0QsS0FBSyxDQUFDLFlBQVksQ0FBQyxTQUFtQixFQUFFLE1BQTRCO1FBQzFFLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ3pELE1BQU0sY0FBYyxHQUFHLE1BQU0sRUFBRSxDQUFDLFFBQVEsQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDL0QsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUUzQyxJQUFJLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQzNDLE1BQU0sU0FBUyxHQUFHLE1BQU0sQ0FBQyxLQUFLLEtBQUssTUFBTSxDQUFDO1lBQzFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsU0FBUyxDQUFDO1lBQ2xELE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGlCQUFpQixTQUFTLFNBQVMsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUNwRixDQUFDO2FBQU0sQ0FBQztZQUNOLEtBQUssTUFBTSxRQUFRLElBQUksU0FBUyxFQUFFLENBQUM7Z0JBQ2pDLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxRQUFRLENBQUUsQ0FBQztnQkFDeEQsTUFBTSxRQUFRLEdBQUcsTUFBTSxDQUFDLFdBQVcsSUFBSSxNQUFNLENBQUMsSUFBSTtvQkFDaEQsQ0FBQyxDQUFDLElBQUksQ0FBQyxnQkFBMkI7b0JBQ2xDLENBQUMsQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDLEtBQUssTUFBTSxDQUFDO2dCQUMxQyxPQUFPLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxHQUFHLFFBQVEsQ0FBQztZQUN2QyxDQUFDO1FBQ0gsQ0FBQztRQUNELE1BQU0sRUFBRSxDQUFDLFNBQVMsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQzdFLENBQUM7SUFFRCxzRkFBc0Y7SUFDdEYsS0FBSyxDQUFDLFlBQVksQ0FBQyxNQUE0QjtRQUM3QyxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsRUFBRSxHQUFHLE1BQU0sQ0FBQztRQUVqQyxJQUFJLFFBQVEsSUFBSSxRQUFRLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxDQUFDO1lBQ3BDLE1BQU0sSUFBSSxDQUFDLG9CQUFvQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQzFDLE9BQU87UUFDVCxDQUFDO1FBRUQsTUFBTSxDQUFDLGNBQWMsRUFBRSxNQUFNLENBQUMsR0FBRyxHQUFHO1lBQ2xDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLENBQUM7WUFDbkMsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSw0QkFBNEIsQ0FBQyxDQUFDO1FBRXBGLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzFDLE1BQU0sSUFBSSxDQUFDLGdCQUFnQixDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRTVDLDZEQUE2RDtRQUM3RCxJQUFJLENBQUMsR0FBRyxJQUFJLGNBQWMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDeEMsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDdEMsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsa0VBQWtFLENBQUMsQ0FBQztZQUN0RyxNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxxRkFBcUYsQ0FBQyxDQUFDO1FBQzNILENBQUM7SUFDSCxDQUFDO0lBRUQsZ0ZBQWdGO0lBQ3hFLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxTQUFtQjtRQUNwRCxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUMxQyxTQUFTLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRXpGLElBQUksYUFBYSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUUsQ0FBQztZQUMvQixNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7WUFDekYsT0FBTztRQUNULENBQUM7UUFFRCxJQUFJLGFBQWEsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDL0IsTUFBTSxJQUFJLEdBQUcsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzlCLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGNBQWMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7WUFDN0QsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO1lBQ3RFLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLHNCQUFzQixJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1lBQ2pGLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGtCQUFrQixZQUFZLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQzFFLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGVBQWUsSUFBSSxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7WUFDbkUsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLGNBQWMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDOUUsT0FBTztRQUNULENBQUM7UUFFRCxNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxTQUFTLGFBQWEsQ0FBQyxNQUFNLG9CQUFvQixTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUM1RyxNQUFNLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQsbUNBQW1DO0lBQzNCLGdCQUFnQixDQUFDLElBQWlCO1FBQ3hDLElBQUksSUFBSSxDQUFDLFNBQVMsS0FBSyxTQUFTO1lBQUUsT0FBTyxDQUFDLENBQUM7UUFDM0MsSUFBSSxrQ0FBa0MsQ0FBQyxJQUFJLENBQUM7WUFBRSxPQUFPLENBQUMsQ0FBQztRQUN2RCxPQUFPLENBQUMsQ0FBQztJQUNYLENBQUM7SUFFRCx1RUFBdUU7SUFDdkUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLEtBQW9CO1FBQ3pDLE1BQU0sV0FBVyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUU7WUFDM0MsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3hDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUV4QyxJQUFJLE1BQU0sS0FBSyxNQUFNO2dCQUFFLE9BQU8sTUFBTSxHQUFHLE1BQU0sQ0FBQztZQUM5QyxJQUFJLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxDQUFDLE1BQU07Z0JBQUUsT0FBTyxDQUFDLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDbkUsT0FBTyxDQUFDLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdEMsQ0FBQyxDQUFDLENBQUM7UUFFSCxNQUFNLElBQUksR0FBZSxDQUFDLENBQUMsY0FBYyxFQUFFLGFBQWEsRUFBRSxNQUFNLEVBQUUsV0FBVyxDQUFDLENBQUMsQ0FBQztRQUNoRixJQUFJLGFBQWEsR0FBRyxFQUFFLENBQUM7UUFFdkIsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFO1lBQzNCLElBQUksSUFBSSxDQUFDLE1BQU0sS0FBSyxhQUFhLEVBQUUsQ0FBQztnQkFDbEMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUM7Z0JBQzlELGFBQWEsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDO1lBQzlCLENBQUM7WUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDO2dCQUNSLEtBQUssSUFBSSxDQUFDLElBQUksRUFBRTtnQkFDaEIsTUFBTSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQztnQkFDN0IsSUFBSSxDQUFDLFNBQVMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUM7Z0JBQ2pFLE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDN0IsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxNQUFNLGNBQWMsR0FBRyxJQUFBLGlDQUFXLEVBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUMxRCxNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBRUQsdURBQXVEO0lBQ3ZELGFBQWEsQ0FBQyxJQUFpQjtRQUM3QixNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUM7UUFDMUMsT0FBTyxPQUFPLFdBQVcsS0FBSyxTQUFTO1lBQ3JDLFdBQVcsS0FBSyxNQUFNO1lBQ3RCLFdBQVcsS0FBSyxPQUFPLENBQUM7SUFDNUIsQ0FBQztJQUVELGlFQUFpRTtJQUNqRSxLQUFLLENBQUMsa0JBQWtCO1FBQ3RCLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUMsQ0FBQztRQUMzRSxNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDO1FBQ3RHLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLDRFQUE0RSxDQUFDLENBQUM7UUFDaEgsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsZ0ZBQWdGLENBQUMsQ0FBQztRQUNwSCxNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQywyRUFBMkUsQ0FBQyxDQUFDO1FBQy9HLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLHlGQUF5RixDQUFDLENBQUM7UUFDN0gsTUFBTSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMseUVBQXlFLENBQUMsQ0FBQztRQUM3RyxNQUFNLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQywrRUFBK0UsQ0FBQyxDQUFDO0lBQ3JILENBQUM7Q0FDRjtBQXBmRCx3Q0FvZkM7QUFFRCxnRkFBZ0Y7QUFDaEYsU0FBZ0Isa0NBQWtDLENBQUMsSUFBaUI7SUFDbEUsT0FBTyxNQUFNLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDLEtBQUssTUFBTSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0FBQ3hFLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsY0FBYyxDQUFDLElBQWlCO0lBQ3ZDLE9BQU8sSUFBSSxDQUFDLFNBQVMsSUFBSSxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUM7QUFDOUMsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxZQUFZLENBQUMsSUFBaUI7SUFDckMsT0FBTyxJQUFJLENBQUMsdUJBQXVCLEVBQUUsRUFBRSxJQUFJLEtBQUssQ0FBQztBQUNuRCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgb3MgZnJvbSAnb3MnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB0eXBlIHsgQ2xvdWRGb3JtYXRpb25TdGFja0FydGlmYWN0IH0gZnJvbSAnQGF3cy1jZGsvY2xvdWQtYXNzZW1ibHktYXBpJztcbmltcG9ydCB7IGZvcm1hdFRhYmxlIH0gZnJvbSAnQGF3cy1jZGsvY2xvdWRmb3JtYXRpb24tZGlmZic7XG5pbXBvcnQgdHlwZSB7IEZlYXR1cmVGbGFnLCBUb29sa2l0IH0gZnJvbSAnQGF3cy1jZGsvdG9vbGtpdC1saWInO1xuaW1wb3J0IHsgQ2RrQXBwTXVsdGlDb250ZXh0LCBNZW1vcnlDb250ZXh0LCBEaWZmTWV0aG9kIH0gZnJvbSAnQGF3cy1jZGsvdG9vbGtpdC1saWInO1xuaW1wb3J0ICogYXMgY2hhbGsgZnJvbSAnY2hhbGsnO1xuaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMtZXh0cmEnO1xuaW1wb3J0IFBRdWV1ZSBmcm9tICdwLXF1ZXVlJztcbmltcG9ydCB7IE9CU09MRVRFX0ZMQUdTIH0gZnJvbSAnLi9vYnNvbGV0ZS1mbGFncyc7XG5pbXBvcnQgdHlwZSB7IEZsYWdPcGVyYXRpb25zUGFyYW1zIH0gZnJvbSAnLi90eXBlcyc7XG5pbXBvcnQgeyBTdGFja1NlbGVjdGlvblN0cmF0ZWd5IH0gZnJvbSAnLi4vLi4vYXBpJztcbmltcG9ydCB0eXBlIHsgSW9IZWxwZXIgfSBmcm9tICcuLi8uLi9hcGktcHJpdmF0ZSc7XG5cbmV4cG9ydCBjbGFzcyBGbGFnT3BlcmF0aW9ucyB7XG4gIC8qKlxuICAgKiBSZXR1cm5zIG9ubHkgdGhvc2UgZmVhdHVyZSBmbGFncyB0aGF0IG5lZWQgY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBUaGF0IGlzIHRob3NlIGZsYWdzOlxuICAgKiAtIFRoYXQgYXJlIHVuY29uZmlndXJlZFxuICAgKiAtIFRoYXQgYXJlIG5vdCBvYnNvbGV0ZVxuICAgKiAtIFdob3NlIGRlZmF1bHQgdmFsdWUgaXMgZGlmZmVyZW50IGZyb20gdGhlIHJlY29tbWVuZGVkIHZhbHVlXG4gICAqXG4gICAqIFRoZSBkZWZhdWx0IHZhbHVlIGJlaW5nIGVxdWFsIHRvIHRoZSByZWNvbW1lbmRlZCB2YWx1ZSBzb3VuZHMgb2RkLCBidXRcbiAgICogY3JvcHMgdXAgaW4gYSBudW1iZXIgb2Ygc2l0dXRhdGlvbnM6XG4gICAqXG4gICAqIC0gU2VjdXJpdHktcmVsYXRlZCBmaXhlcyB0aGF0IHdlIHdhbnQgdG8gZm9yY2Ugb24gcGVvcGxlLCBidXQgd2FudCB0b1xuICAgKiAgIGdpdmUgdGhlbSBhIGZsYWcgdG8gYmFjayBvdXQgb2YgdGhlIGNoYW5nZXMgaWYgdGhleSByZWFsbHkgbmVlZCB0by5cbiAgICogLSBGbGFncyB0aGF0IGNoYW5nZWQgdGhlaXIgZGVmYXVsdCB2YWx1ZSBpbiB0aGUgbW9zdCByZWNlbnQgbWFqb3JcbiAgICogICB2ZXJzaW9uLlxuICAgKiAtIEZsYWdzIHRoYXQgd2UndmUgaW50cm9kdWNlZCBhdCBzb21lIHBvaW50IGluIHRoZSBwYXN0LCBidXQgaGF2ZSBnb25lXG4gICAqICAgYmFjayBvbi5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZmlsdGVyTmVlZHNBdHRlbnRpb24oZmxhZ3M6IEZlYXR1cmVGbGFnW10pOiBGZWF0dXJlRmxhZ1tdIHtcbiAgICByZXR1cm4gZmxhZ3NcbiAgICAgIC5maWx0ZXIoZmxhZyA9PiAhT0JTT0xFVEVfRkxBR1MuaW5jbHVkZXMoZmxhZy5uYW1lKSlcbiAgICAgIC5maWx0ZXIoZmxhZyA9PiBmbGFnLnVzZXJWYWx1ZSA9PT0gdW5kZWZpbmVkKVxuICAgICAgLmZpbHRlcihmbGFnID0+IGRlZmF1bHRWYWx1ZShmbGFnKSAhPT0gZmxhZy5yZWNvbW1lbmRlZFZhbHVlKTtcbiAgfVxuXG4gIHByaXZhdGUgYXBwOiBzdHJpbmc7XG4gIHByaXZhdGUgYmFzZUNvbnRleHRWYWx1ZXM6IFJlY29yZDxzdHJpbmcsIGFueT47XG4gIHByaXZhdGUgYWxsU3RhY2tzOiBDbG91ZEZvcm1hdGlvblN0YWNrQXJ0aWZhY3RbXTtcbiAgcHJpdmF0ZSBxdWV1ZTogUFF1ZXVlO1xuICBwcml2YXRlIGJhc2VsaW5lVGVtcERpcj86IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihcbiAgICBwcml2YXRlIHJlYWRvbmx5IGZsYWdzOiBGZWF0dXJlRmxhZ1tdLFxuICAgIHByaXZhdGUgcmVhZG9ubHkgdG9vbGtpdDogVG9vbGtpdCxcbiAgICBwcml2YXRlIHJlYWRvbmx5IGlvSGVscGVyOiBJb0hlbHBlcixcbiAgICBwcml2YXRlIHJlYWRvbmx5IGNsaUNvbnRleHRWYWx1ZXM6IFJlY29yZDxzdHJpbmcsIGFueT4gPSB7fSxcbiAgKSB7XG4gICAgdGhpcy5hcHAgPSAnJztcbiAgICB0aGlzLmJhc2VDb250ZXh0VmFsdWVzID0ge307XG4gICAgdGhpcy5hbGxTdGFja3MgPSBbXTtcbiAgICB0aGlzLnF1ZXVlID0gbmV3IFBRdWV1ZSh7IGNvbmN1cnJlbmN5OiA0IH0pO1xuICB9XG5cbiAgLyoqIE1haW4gZW50cnkgcG9pbnQgdGhhdCByb3V0ZXMgdG8gZWl0aGVyIGZsYWcgc2V0dGluZyBvciBkaXNwbGF5IG9wZXJhdGlvbnMgKi9cbiAgYXN5bmMgZXhlY3V0ZShwYXJhbXM6IEZsYWdPcGVyYXRpb25zUGFyYW1zKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgaWYgKHBhcmFtcy5zZXQpIHtcbiAgICAgIGlmIChwYXJhbXMuRkxBR05BTUUgJiYgcGFyYW1zLnZhbHVlKSB7XG4gICAgICAgIGF3YWl0IHRoaXMuc2V0RmxhZyhwYXJhbXMpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgYXdhaXQgdGhpcy5zZXRNdWx0aXBsZUZsYWdzKHBhcmFtcyk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGF3YWl0IHRoaXMuZGlzcGxheUZsYWdzKHBhcmFtcyk7XG4gICAgfVxuICB9XG5cbiAgLyoqIFNldHMgYSBzaW5nbGUgc3BlY2lmaWMgZmxhZyB3aXRoIHZhbGlkYXRpb24gYW5kIHVzZXIgY29uZmlybWF0aW9uICovXG4gIGFzeW5jIHNldEZsYWcocGFyYW1zOiBGbGFnT3BlcmF0aW9uc1BhcmFtcyk6IFByb21pc2U8dm9pZD4ge1xuICAgIGNvbnN0IGZsYWdOYW1lID0gcGFyYW1zLkZMQUdOQU1FIVswXTtcbiAgICBjb25zdCBmbGFnID0gdGhpcy5mbGFncy5maW5kKGYgPT4gZi5uYW1lID09PSBmbGFnTmFtZSk7XG5cbiAgICBpZiAoIWZsYWcpIHtcbiAgICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuZXJyb3IoJ0ZsYWcgbm90IGZvdW5kLicpO1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGlmICghdGhpcy5pc0Jvb2xlYW5GbGFnKGZsYWcpKSB7XG4gICAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmVycm9yKGBGbGFnICcke2ZsYWdOYW1lfScgaXMgbm90IGEgYm9vbGVhbiBmbGFnLiBPbmx5IGJvb2xlYW4gZmxhZ3MgYXJlIGN1cnJlbnRseSBzdXBwb3J0ZWQuYCk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY29uc3QgcHJvdG90eXBlU3VjY2VzcyA9IGF3YWl0IHRoaXMucHJvdG90eXBlQ2hhbmdlcyhbZmxhZ05hbWVdLCBwYXJhbXMpO1xuICAgIGlmIChwcm90b3R5cGVTdWNjZXNzKSB7XG4gICAgICBhd2FpdCB0aGlzLmhhbmRsZVVzZXJSZXNwb25zZShbZmxhZ05hbWVdLCBwYXJhbXMpO1xuICAgIH1cbiAgfVxuXG4gIC8qKiBTZXRzIG11bHRpcGxlIGZsYWdzIChhbGwgb3IgdW5jb25maWd1cmVkKSB3aXRoIHZhbGlkYXRpb24gYW5kIHVzZXIgY29uZmlybWF0aW9uICovXG4gIGFzeW5jIHNldE11bHRpcGxlRmxhZ3MocGFyYW1zOiBGbGFnT3BlcmF0aW9uc1BhcmFtcyk6IFByb21pc2U8dm9pZD4ge1xuICAgIGlmIChwYXJhbXMuZGVmYXVsdCAmJiAhdGhpcy5mbGFncy5zb21lKGYgPT4gZi51bmNvbmZpZ3VyZWRCZWhhdmVzTGlrZSkpIHtcbiAgICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuZXJyb3IoJ1RoZSAtLWRlZmF1bHQgb3B0aW9ucyBhcmUgbm90IGNvbXBhdGlibGUgd2l0aCB0aGUgQVdTIENESyBsaWJyYXJ5IHVzZWQgYnkgeW91ciBhcHBsaWNhdGlvbi4gUGxlYXNlIHVwZ3JhZGUgdG8gMi4yMTIuMCBvciBhYm92ZS4nKTtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBjb25zdCBmbGFnc1RvU2V0ID0gdGhpcy5nZXRGbGFnc1RvU2V0KHBhcmFtcyk7XG4gICAgY29uc3QgcHJvdG90eXBlU3VjY2VzcyA9IGF3YWl0IHRoaXMucHJvdG90eXBlQ2hhbmdlcyhmbGFnc1RvU2V0LCBwYXJhbXMpO1xuXG4gICAgaWYgKHByb3RvdHlwZVN1Y2Nlc3MpIHtcbiAgICAgIGF3YWl0IHRoaXMuaGFuZGxlVXNlclJlc3BvbnNlKGZsYWdzVG9TZXQsIHBhcmFtcyk7XG4gICAgfVxuICB9XG5cbiAgLyoqIERldGVybWluZXMgd2hpY2ggZmxhZ3Mgc2hvdWxkIGJlIHNldCBiYXNlZCBvbiB0aGUgcHJvdmlkZWQgcGFyYW1ldGVycyAqL1xuICBwcml2YXRlIGdldEZsYWdzVG9TZXQocGFyYW1zOiBGbGFnT3BlcmF0aW9uc1BhcmFtcyk6IHN0cmluZ1tdIHtcbiAgICBpZiAocGFyYW1zLmFsbCAmJiBwYXJhbXMuZGVmYXVsdCkge1xuICAgICAgcmV0dXJuIHRoaXMuZmxhZ3NcbiAgICAgICAgLmZpbHRlcihmbGFnID0+IHRoaXMuaXNCb29sZWFuRmxhZyhmbGFnKSlcbiAgICAgICAgLm1hcChmbGFnID0+IGZsYWcubmFtZSk7XG4gICAgfSBlbHNlIGlmIChwYXJhbXMuYWxsKSB7XG4gICAgICByZXR1cm4gdGhpcy5mbGFnc1xuICAgICAgICAuZmlsdGVyKGZsYWcgPT4gZmxhZy51c2VyVmFsdWUgPT09IHVuZGVmaW5lZCB8fCAhaXNFZmZlY3RpdmVWYWx1ZUVxdWFsVG9SZWNvbW1lbmRlZChmbGFnKSlcbiAgICAgICAgLmZpbHRlcihmbGFnID0+IHRoaXMuaXNCb29sZWFuRmxhZyhmbGFnKSlcbiAgICAgICAgLm1hcChmbGFnID0+IGZsYWcubmFtZSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiB0aGlzLmZsYWdzXG4gICAgICAgIC5maWx0ZXIoZmxhZyA9PiBmbGFnLnVzZXJWYWx1ZSA9PT0gdW5kZWZpbmVkKVxuICAgICAgICAuZmlsdGVyKGZsYWcgPT4gdGhpcy5pc0Jvb2xlYW5GbGFnKGZsYWcpKVxuICAgICAgICAubWFwKGZsYWcgPT4gZmxhZy5uYW1lKTtcbiAgICB9XG4gIH1cblxuICAvKiogU2V0cyBmbGFncyB0aGF0IGRvbid0IGNhdXNlIHRlbXBsYXRlIGNoYW5nZXMgKi9cbiAgYXN5bmMgc2V0U2FmZUZsYWdzKHBhcmFtczogRmxhZ09wZXJhdGlvbnNQYXJhbXMpOiBQcm9taXNlPHZvaWQ+IHtcbiAgICBjb25zdCBjZGtKc29uID0gYXdhaXQgSlNPTi5wYXJzZShhd2FpdCBmcy5yZWFkRmlsZShwYXRoLmpvaW4ocHJvY2Vzcy5jd2QoKSwgJ2Nkay5qc29uJyksICd1dGYtOCcpKTtcbiAgICB0aGlzLmFwcCA9IHBhcmFtcy5hcHAgfHwgY2RrSnNvbi5hcHA7XG5cbiAgICBjb25zdCBpc1VzaW5nVHNOb2RlID0gdGhpcy5hcHAuaW5jbHVkZXMoJ3RzLW5vZGUnKTtcbiAgICBpZiAoaXNVc2luZ1RzTm9kZSAmJiAhdGhpcy5hcHAuaW5jbHVkZXMoJy1UJykgJiYgIXRoaXMuYXBwLmluY2x1ZGVzKCctLXRyYW5zcGlsZU9ubHknKSkge1xuICAgICAgYXdhaXQgdGhpcy5pb0hlbHBlci5kZWZhdWx0cy5pbmZvKCdSZXBlYXRlZCBzeW50aHMgd2l0aCB0cy1ub2RlIHdpbGwgdHlwZS1jaGVjayB0aGUgYXBwbGljYXRpb24gb24gZXZlcnkgc3ludGguIEFkZCAtLXRyYW5zcGlsZU9ubHkgdG8gY2RrLmpzb25cXCdzIFwiYXBwXCIgY29tbWFuZCB0byBtYWtlIHRoaXMgb3BlcmF0aW9uIGZhc3Rlci4nKTtcbiAgICB9XG5cbiAgICBjb25zdCB1bmNvbmZpZ3VyZWRGbGFncyA9IHRoaXMuZmxhZ3MuZmlsdGVyKGZsYWcgPT5cbiAgICAgIGZsYWcudXNlclZhbHVlID09PSB1bmRlZmluZWQgJiYgdGhpcy5pc0Jvb2xlYW5GbGFnKGZsYWcpKTtcblxuICAgIGlmICh1bmNvbmZpZ3VyZWRGbGFncy5sZW5ndGggPT09IDApIHtcbiAgICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbygnQWxsIGZlYXR1cmUgZmxhZ3MgYXJlIGNvbmZpZ3VyZWQuJyk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgYXdhaXQgdGhpcy5pbml0aWFsaXplU2FmZXR5Q2hlY2soKTtcbiAgICBjb25zdCBzYWZlRmxhZ3MgPSBhd2FpdCB0aGlzLmJhdGNoVGVzdEZsYWdzKHVuY29uZmlndXJlZEZsYWdzKTtcbiAgICBhd2FpdCB0aGlzLmNsZWFudXBTYWZldHlDaGVjaygpO1xuXG4gICAgaWYgKHNhZmVGbGFncy5sZW5ndGggPiAwKSB7XG4gICAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmluZm8oJ0ZsYWdzIHRoYXQgY2FuIGJlIHNldCB3aXRob3V0IHRlbXBsYXRlIGNoYW5nZXM6Jyk7XG4gICAgICBmb3IgKGNvbnN0IGZsYWcgb2Ygc2FmZUZsYWdzKSB7XG4gICAgICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbyhgLSAke2ZsYWcubmFtZX0gLT4gJHtmbGFnLnJlY29tbWVuZGVkVmFsdWV9YCk7XG4gICAgICB9XG4gICAgICBhd2FpdCB0aGlzLmhhbmRsZVVzZXJSZXNwb25zZShzYWZlRmxhZ3MubWFwKGZsYWcgPT4gZmxhZy5uYW1lKSwgeyAuLi5wYXJhbXMsIHJlY29tbWVuZGVkOiB0cnVlIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmluZm8oJ05vIG1vcmUgZmxhZ3MgY2FuIGJlIHNldCB3aXRob3V0IGNhdXNpbmcgdGVtcGxhdGUgY2hhbmdlcy4nKTtcbiAgICB9XG4gIH1cblxuICAvKiogSW5pdGlhbGl6ZXMgdGhlIHNhZmV0eSBjaGVjayBieSByZWFkaW5nIGNvbnRleHQgYW5kIHN5bnRoZXNpemluZyBiYXNlbGluZSB0ZW1wbGF0ZXMgKi9cbiAgcHJpdmF0ZSBhc3luYyBpbml0aWFsaXplU2FmZXR5Q2hlY2soKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgY29uc3QgYmFzZUNvbnRleHQgPSBuZXcgQ2RrQXBwTXVsdGlDb250ZXh0KHByb2Nlc3MuY3dkKCkpO1xuICAgIHRoaXMuYmFzZUNvbnRleHRWYWx1ZXMgPSB7IC4uLmF3YWl0IGJhc2VDb250ZXh0LnJlYWQoKSwgLi4udGhpcy5jbGlDb250ZXh0VmFsdWVzIH07XG5cbiAgICB0aGlzLmJhc2VsaW5lVGVtcERpciA9IGZzLm1rZHRlbXBTeW5jKHBhdGguam9pbihvcy50bXBkaXIoKSwgJ2Nkay1iYXNlbGluZS0nKSk7XG4gICAgY29uc3QgbWVyZ2VkQ29udGV4dCA9IG5ldyBNZW1vcnlDb250ZXh0KHRoaXMuYmFzZUNvbnRleHRWYWx1ZXMpO1xuICAgIGNvbnN0IGJhc2VTb3VyY2UgPSBhd2FpdCB0aGlzLnRvb2xraXQuZnJvbUNka0FwcCh0aGlzLmFwcCwge1xuICAgICAgY29udGV4dFN0b3JlOiBtZXJnZWRDb250ZXh0LFxuICAgICAgb3V0ZGlyOiB0aGlzLmJhc2VsaW5lVGVtcERpcixcbiAgICB9KTtcblxuICAgIGNvbnN0IGJhc2VDeCA9IGF3YWl0IHRoaXMudG9vbGtpdC5zeW50aChiYXNlU291cmNlKTtcbiAgICBjb25zdCBiYXNlQXNzZW1ibHkgPSBiYXNlQ3guY2xvdWRBc3NlbWJseTtcbiAgICB0aGlzLmFsbFN0YWNrcyA9IGJhc2VBc3NlbWJseS5zdGFja3NSZWN1cnNpdmVseTtcbiAgICB0aGlzLnF1ZXVlID0gbmV3IFBRdWV1ZSh7IGNvbmN1cnJlbmN5OiA0IH0pO1xuICB9XG5cbiAgLyoqIENsZWFucyB1cCB0ZW1wb3JhcnkgZGlyZWN0b3JpZXMgY3JlYXRlZCBkdXJpbmcgc2FmZXR5IGNoZWNrcyAqL1xuICBwcml2YXRlIGFzeW5jIGNsZWFudXBTYWZldHlDaGVjaygpOiBQcm9taXNlPHZvaWQ+IHtcbiAgICBpZiAodGhpcy5iYXNlbGluZVRlbXBEaXIpIHtcbiAgICAgIGF3YWl0IGZzLnJlbW92ZSh0aGlzLmJhc2VsaW5lVGVtcERpcik7XG4gICAgICB0aGlzLmJhc2VsaW5lVGVtcERpciA9IHVuZGVmaW5lZDtcbiAgICB9XG4gIH1cblxuICAvKiogVGVzdHMgbXVsdGlwbGUgZmxhZ3MgdG9nZXRoZXIgYW5kIGlzb2xhdGVzIHVuc2FmZSBvbmVzIHVzaW5nIGJpbmFyeSBzZWFyY2ggKi9cbiAgcHJpdmF0ZSBhc3luYyBiYXRjaFRlc3RGbGFncyhmbGFnczogRmVhdHVyZUZsYWdbXSk6IFByb21pc2U8RmVhdHVyZUZsYWdbXT4ge1xuICAgIGlmIChmbGFncy5sZW5ndGggPT09IDApIHJldHVybiBbXTtcblxuICAgIGNvbnN0IGFsbEZsYWdzQ29udGV4dCA9IHsgLi4udGhpcy5iYXNlQ29udGV4dFZhbHVlcyB9O1xuICAgIGZsYWdzLmZvckVhY2goZmxhZyA9PiB7XG4gICAgICBhbGxGbGFnc0NvbnRleHRbZmxhZy5uYW1lXSA9IGZsYWcucmVjb21tZW5kZWRWYWx1ZTtcbiAgICB9KTtcblxuICAgIGNvbnN0IGFsbFNhZmUgPSBhd2FpdCB0aGlzLnRlc3RCYXRjaChhbGxGbGFnc0NvbnRleHQpO1xuICAgIGlmIChhbGxTYWZlKSByZXR1cm4gZmxhZ3M7XG5cbiAgICByZXR1cm4gdGhpcy5pc29sYXRlVW5zYWZlRmxhZ3MoZmxhZ3MpO1xuICB9XG5cbiAgLyoqIFRlc3RzIGlmIGEgc2V0IG9mIGNvbnRleHQgdmFsdWVzIGNhdXNlcyB0ZW1wbGF0ZSBjaGFuZ2VzIGJ5IHN5bnRoZXNpemluZyBhbmQgZGlmZmluZyAqL1xuICBwcml2YXRlIGFzeW5jIHRlc3RCYXRjaChjb250ZXh0VmFsdWVzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+KTogUHJvbWlzZTxib29sZWFuPiB7XG4gICAgY29uc3QgdGVzdENvbnRleHQgPSBuZXcgTWVtb3J5Q29udGV4dChjb250ZXh0VmFsdWVzKTtcbiAgICBjb25zdCB0ZW1wRGlyID0gZnMubWtkdGVtcFN5bmMocGF0aC5qb2luKG9zLnRtcGRpcigpLCAnY2RrLXRlc3QtJykpO1xuICAgIGNvbnN0IHRlc3RTb3VyY2UgPSBhd2FpdCB0aGlzLnRvb2xraXQuZnJvbUNka0FwcCh0aGlzLmFwcCwge1xuICAgICAgY29udGV4dFN0b3JlOiB0ZXN0Q29udGV4dCxcbiAgICAgIG91dGRpcjogdGVtcERpcixcbiAgICB9KTtcblxuICAgIGNvbnN0IHRlc3RDeCA9IGF3YWl0IHRoaXMudG9vbGtpdC5zeW50aCh0ZXN0U291cmNlKTtcblxuICAgIHRyeSB7XG4gICAgICBmb3IgKGNvbnN0IHN0YWNrIG9mIHRoaXMuYWxsU3RhY2tzKSB7XG4gICAgICAgIGNvbnN0IHRlbXBsYXRlUGF0aCA9IHN0YWNrLnRlbXBsYXRlRnVsbFBhdGg7XG4gICAgICAgIGNvbnN0IGRpZmYgPSBhd2FpdCB0aGlzLnRvb2xraXQuZGlmZih0ZXN0Q3gsIHtcbiAgICAgICAgICBtZXRob2Q6IERpZmZNZXRob2QuTG9jYWxGaWxlKHRlbXBsYXRlUGF0aCksXG4gICAgICAgICAgc3RhY2tzOiB7XG4gICAgICAgICAgICBzdHJhdGVneTogU3RhY2tTZWxlY3Rpb25TdHJhdGVneS5QQVRURVJOX01VU1RfTUFUQ0hfU0lOR0xFLFxuICAgICAgICAgICAgcGF0dGVybnM6IFtzdGFjay5oaWVyYXJjaGljYWxJZF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgZm9yIChjb25zdCBzdGFja0RpZmYgb2YgT2JqZWN0LnZhbHVlcyhkaWZmKSkge1xuICAgICAgICAgIGlmIChzdGFja0RpZmYuZGlmZmVyZW5jZUNvdW50ID4gMCkge1xuICAgICAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgcmV0dXJuIHRydWU7XG4gICAgfSBmaW5hbGx5IHtcbiAgICAgIGF3YWl0IGZzLnJlbW92ZSh0ZW1wRGlyKTtcbiAgICB9XG4gIH1cblxuICAvKiogVXNlcyBiaW5hcnkgc2VhcmNoIHRvIGlzb2xhdGUgd2hpY2ggZmxhZ3MgYXJlIHNhZmUgdG8gc2V0IHdpdGhvdXQgdGVtcGxhdGUgY2hhbmdlcyAqL1xuICBwcml2YXRlIGFzeW5jIGlzb2xhdGVVbnNhZmVGbGFncyhmbGFnczogRmVhdHVyZUZsYWdbXSk6IFByb21pc2U8RmVhdHVyZUZsYWdbXT4ge1xuICAgIGNvbnN0IHNhZmVGbGFnczogRmVhdHVyZUZsYWdbXSA9IFtdO1xuXG4gICAgY29uc3QgcHJvY2Vzc0JhdGNoID0gYXN5bmMgKGJhdGNoOiBGZWF0dXJlRmxhZ1tdLCBjb250ZXh0VmFsdWVzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+KTogUHJvbWlzZTx2b2lkPiA9PiB7XG4gICAgICBpZiAoYmF0Y2gubGVuZ3RoID09PSAxKSB7XG4gICAgICAgIGNvbnN0IGlzU2FmZSA9IGF3YWl0IHRoaXMudGVzdEJhdGNoKFxuICAgICAgICAgIHsgLi4uY29udGV4dFZhbHVlcywgW2JhdGNoWzBdLm5hbWVdOiBiYXRjaFswXS5yZWNvbW1lbmRlZFZhbHVlIH0sXG4gICAgICAgICk7XG4gICAgICAgIGlmIChpc1NhZmUpIHNhZmVGbGFncy5wdXNoKGJhdGNoWzBdKTtcbiAgICAgICAgcmV0dXJuO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBiYXRjaENvbnRleHQgPSB7IC4uLmNvbnRleHRWYWx1ZXMgfTtcbiAgICAgIGJhdGNoLmZvckVhY2goZmxhZyA9PiB7XG4gICAgICAgIGJhdGNoQ29udGV4dFtmbGFnLm5hbWVdID0gZmxhZy5yZWNvbW1lbmRlZFZhbHVlO1xuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IGlzU2FmZUJhdGNoID0gYXdhaXQgdGhpcy50ZXN0QmF0Y2goYmF0Y2hDb250ZXh0KTtcbiAgICAgIGlmIChpc1NhZmVCYXRjaCkge1xuICAgICAgICBzYWZlRmxhZ3MucHVzaCguLi5iYXRjaCk7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH1cblxuICAgICAgY29uc3QgbWlkID0gTWF0aC5mbG9vcihiYXRjaC5sZW5ndGggLyAyKTtcbiAgICAgIGNvbnN0IGxlZnQgPSBiYXRjaC5zbGljZSgwLCBtaWQpO1xuICAgICAgY29uc3QgcmlnaHQgPSBiYXRjaC5zbGljZShtaWQpO1xuXG4gICAgICB2b2lkIHRoaXMucXVldWUuYWRkKCgpID0+IHByb2Nlc3NCYXRjaChsZWZ0LCBjb250ZXh0VmFsdWVzKSk7XG4gICAgICB2b2lkIHRoaXMucXVldWUuYWRkKCgpID0+IHByb2Nlc3NCYXRjaChyaWdodCwgY29udGV4dFZhbHVlcykpO1xuICAgIH07XG5cbiAgICB2b2lkIHRoaXMucXVldWUuYWRkKCgpID0+IHByb2Nlc3NCYXRjaChmbGFncywgdGhpcy5iYXNlQ29udGV4dFZhbHVlcykpO1xuICAgIGF3YWl0IHRoaXMucXVldWUub25JZGxlKCk7XG4gICAgcmV0dXJuIHNhZmVGbGFncztcbiAgfVxuXG4gIC8qKiBQcm90b3R5cGVzIGZsYWcgY2hhbmdlcyBieSBzeW50aGVzaXppbmcgdGVtcGxhdGVzIGFuZCBzaG93aW5nIGRpZmZzIHRvIHRoZSB1c2VyICovXG4gIHByaXZhdGUgYXN5bmMgcHJvdG90eXBlQ2hhbmdlcyhmbGFnTmFtZXM6IHN0cmluZ1tdLCBwYXJhbXM6IEZsYWdPcGVyYXRpb25zUGFyYW1zKTogUHJvbWlzZTxib29sZWFuPiB7XG4gICAgY29uc3QgYmFzZUNvbnRleHQgPSBuZXcgQ2RrQXBwTXVsdGlDb250ZXh0KHByb2Nlc3MuY3dkKCkpO1xuICAgIGNvbnN0IGJhc2VDb250ZXh0VmFsdWVzID0geyAuLi5hd2FpdCBiYXNlQ29udGV4dC5yZWFkKCksIC4uLnRoaXMuY2xpQ29udGV4dFZhbHVlcyB9O1xuICAgIGNvbnN0IG1lbW9yeUNvbnRleHQgPSBuZXcgTWVtb3J5Q29udGV4dChiYXNlQ29udGV4dFZhbHVlcyk7XG5cbiAgICBjb25zdCBjZGtKc29uID0gYXdhaXQgSlNPTi5wYXJzZShhd2FpdCBmcy5yZWFkRmlsZShwYXRoLmpvaW4ocHJvY2Vzcy5jd2QoKSwgJ2Nkay5qc29uJyksICd1dGYtOCcpKTtcbiAgICBjb25zdCBhcHAgPSBjZGtKc29uLmFwcDtcblxuICAgIGNvbnN0IHNvdXJjZSA9IGF3YWl0IHRoaXMudG9vbGtpdC5mcm9tQ2RrQXBwKGFwcCwge1xuICAgICAgY29udGV4dFN0b3JlOiBtZW1vcnlDb250ZXh0LFxuICAgICAgb3V0ZGlyOiBmcy5ta2R0ZW1wU3luYyhwYXRoLmpvaW4ob3MudG1wZGlyKCksICdjZGstb3JpZ2luYWwtJykpLFxuICAgIH0pO1xuXG4gICAgY29uc3QgdXBkYXRlT2JqID0gYXdhaXQgdGhpcy5idWlsZFVwZGF0ZU9iamVjdChmbGFnTmFtZXMsIHBhcmFtcywgYmFzZUNvbnRleHRWYWx1ZXMpO1xuICAgIGlmICghdXBkYXRlT2JqKSByZXR1cm4gZmFsc2U7XG5cbiAgICBhd2FpdCBtZW1vcnlDb250ZXh0LnVwZGF0ZSh1cGRhdGVPYmopO1xuICAgIGNvbnN0IGN4ID0gYXdhaXQgdGhpcy50b29sa2l0LnN5bnRoKHNvdXJjZSk7XG4gICAgY29uc3QgYXNzZW1ibHkgPSBjeC5jbG91ZEFzc2VtYmx5O1xuXG4gICAgY29uc3QgbW9kaWZpZWRTb3VyY2UgPSBhd2FpdCB0aGlzLnRvb2xraXQuZnJvbUNka0FwcChhcHAsIHtcbiAgICAgIGNvbnRleHRTdG9yZTogbWVtb3J5Q29udGV4dCxcbiAgICAgIG91dGRpcjogZnMubWtkdGVtcFN5bmMocGF0aC5qb2luKG9zLnRtcGRpcigpLCAnY2RrLXRlbXAtJykpLFxuICAgIH0pO1xuXG4gICAgY29uc3QgbW9kaWZpZWRDeCA9IGF3YWl0IHRoaXMudG9vbGtpdC5zeW50aChtb2RpZmllZFNvdXJjZSk7XG4gICAgY29uc3QgYWxsU3RhY2tzID0gYXNzZW1ibHkuc3RhY2tzUmVjdXJzaXZlbHk7XG5cbiAgICBmb3IgKGNvbnN0IHN0YWNrIG9mIGFsbFN0YWNrcykge1xuICAgICAgY29uc3QgdGVtcGxhdGVQYXRoID0gc3RhY2sudGVtcGxhdGVGdWxsUGF0aDtcbiAgICAgIGF3YWl0IHRoaXMudG9vbGtpdC5kaWZmKG1vZGlmaWVkQ3gsIHtcbiAgICAgICAgbWV0aG9kOiBEaWZmTWV0aG9kLkxvY2FsRmlsZSh0ZW1wbGF0ZVBhdGgpLFxuICAgICAgICBzdGFja3M6IHtcbiAgICAgICAgICBzdHJhdGVneTogU3RhY2tTZWxlY3Rpb25TdHJhdGVneS5QQVRURVJOX01VU1RfTUFUQ0hfU0lOR0xFLFxuICAgICAgICAgIHBhdHRlcm5zOiBbc3RhY2suaGllcmFyY2hpY2FsSWRdLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgYXdhaXQgdGhpcy5kaXNwbGF5RmxhZ0NoYW5nZXModXBkYXRlT2JqLCBiYXNlQ29udGV4dFZhbHVlcyk7XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cblxuICAvKiogRGlzcGxheXMgYSBzdW1tYXJ5IG9mIGZsYWcgY2hhbmdlcyBzaG93aW5nIG9sZCBhbmQgbmV3IHZhbHVlcyAqL1xuICBwcml2YXRlIGFzeW5jIGRpc3BsYXlGbGFnQ2hhbmdlcyh1cGRhdGVPYmo6IFJlY29yZDxzdHJpbmcsIGJvb2xlYW4+LCBiYXNlQ29udGV4dFZhbHVlczogUmVjb3JkPHN0cmluZywgYW55Pik6IFByb21pc2U8dm9pZD4ge1xuICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbygnXFxuRmxhZyBjaGFuZ2VzOicpO1xuICAgIGZvciAoY29uc3QgW2ZsYWdOYW1lLCBuZXdWYWx1ZV0gb2YgT2JqZWN0LmVudHJpZXModXBkYXRlT2JqKSkge1xuICAgICAgY29uc3QgY3VycmVudFZhbHVlID0gYmFzZUNvbnRleHRWYWx1ZXNbZmxhZ05hbWVdO1xuICAgICAgY29uc3QgY3VycmVudERpc3BsYXkgPSBjdXJyZW50VmFsdWUgPT09IHVuZGVmaW5lZCA/ICc8dW5zZXQ+JyA6IFN0cmluZyhjdXJyZW50VmFsdWUpO1xuICAgICAgYXdhaXQgdGhpcy5pb0hlbHBlci5kZWZhdWx0cy5pbmZvKGAgICR7ZmxhZ05hbWV9OiAke2N1cnJlbnREaXNwbGF5fSDihpIgJHtuZXdWYWx1ZX1gKTtcbiAgICB9XG4gIH1cblxuICAvKiogQnVpbGRzIHRoZSB1cGRhdGUgb2JqZWN0IHdpdGggbmV3IGZsYWcgdmFsdWVzIGJhc2VkIG9uIHBhcmFtZXRlcnMgYW5kIGN1cnJlbnQgY29udGV4dCAqL1xuICBwcml2YXRlIGFzeW5jIGJ1aWxkVXBkYXRlT2JqZWN0KGZsYWdOYW1lczogc3RyaW5nW10sIHBhcmFtczogRmxhZ09wZXJhdGlvbnNQYXJhbXMsIGJhc2VDb250ZXh0VmFsdWVzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+KVxuICAgIDogUHJvbWlzZTxSZWNvcmQ8c3RyaW5nLCBib29sZWFuPiB8IG51bGw+IHtcbiAgICBjb25zdCB1cGRhdGVPYmo6IFJlY29yZDxzdHJpbmcsIGJvb2xlYW4+ID0ge307XG5cbiAgICBpZiAoZmxhZ05hbWVzLmxlbmd0aCA9PT0gMSAmJiBwYXJhbXMudmFsdWUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgY29uc3QgZmxhZ05hbWUgPSBmbGFnTmFtZXNbMF07XG4gICAgICBjb25zdCBib29sVmFsdWUgPSBwYXJhbXMudmFsdWUgPT09ICd0cnVlJztcbiAgICAgIGlmIChiYXNlQ29udGV4dFZhbHVlc1tmbGFnTmFtZV0gPT09IGJvb2xWYWx1ZSkge1xuICAgICAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmluZm8oJ0ZsYWcgaXMgYWxyZWFkeSBzZXQgdG8gdGhlIHNwZWNpZmllZCB2YWx1ZS4gTm8gY2hhbmdlcyBuZWVkZWQuJyk7XG4gICAgICAgIHJldHVybiBudWxsO1xuICAgICAgfVxuICAgICAgdXBkYXRlT2JqW2ZsYWdOYW1lXSA9IGJvb2xWYWx1ZTtcbiAgICB9IGVsc2Uge1xuICAgICAgZm9yIChjb25zdCBmbGFnTmFtZSBvZiBmbGFnTmFtZXMpIHtcbiAgICAgICAgY29uc3QgZmxhZyA9IHRoaXMuZmxhZ3MuZmluZChmID0+IGYubmFtZSA9PT0gZmxhZ05hbWUpO1xuICAgICAgICBpZiAoIWZsYWcpIHtcbiAgICAgICAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmVycm9yKGBGbGFnICR7ZmxhZ05hbWV9IG5vdCBmb3VuZC5gKTtcbiAgICAgICAgICByZXR1cm4gbnVsbDtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBuZXdWYWx1ZSA9IHBhcmFtcy5yZWNvbW1lbmRlZFxuICAgICAgICAgID8gZmxhZy5yZWNvbW1lbmRlZFZhbHVlIGFzIGJvb2xlYW5cbiAgICAgICAgICA6IFN0cmluZyhkZWZhdWx0VmFsdWUoZmxhZykpID09PSAndHJ1ZSc7XG4gICAgICAgIHVwZGF0ZU9ialtmbGFnTmFtZV0gPSBuZXdWYWx1ZTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHVwZGF0ZU9iajtcbiAgfVxuXG4gIC8qKiBQcm9tcHRzIHVzZXIgZm9yIGNvbmZpcm1hdGlvbiBhbmQgYXBwbGllcyBjaGFuZ2VzIGlmIGFjY2VwdGVkICovXG4gIHByaXZhdGUgYXN5bmMgaGFuZGxlVXNlclJlc3BvbnNlKGZsYWdOYW1lczogc3RyaW5nW10sIHBhcmFtczogRmxhZ09wZXJhdGlvbnNQYXJhbXMpOiBQcm9taXNlPHZvaWQ+IHtcbiAgICBjb25zdCB1c2VyQWNjZXB0ZWQgPSBhd2FpdCB0aGlzLmlvSGVscGVyLnJlcXVlc3RSZXNwb25zZSh7XG4gICAgICB0aW1lOiBuZXcgRGF0ZSgpLFxuICAgICAgbGV2ZWw6ICdpbmZvJyxcbiAgICAgIGNvZGU6ICdDREtfVE9PTEtJVF9JOTMwMCcsXG4gICAgICBtZXNzYWdlOiAnRG8geW91IHdhbnQgdG8gYWNjZXB0IHRoZXNlIGNoYW5nZXM/JyxcbiAgICAgIGRhdGE6IHtcbiAgICAgICAgZmxhZ05hbWVzLFxuICAgICAgICByZXNwb25zZURlc2NyaXB0aW9uOiAnRW50ZXIgXCJ5XCIgdG8gYXBwbHkgY2hhbmdlcyBvciBcIm5cIiB0byBjYW5jZWwnLFxuICAgICAgfSxcbiAgICAgIGRlZmF1bHRSZXNwb25zZTogZmFsc2UsXG4gICAgfSk7XG5cbiAgICBpZiAodXNlckFjY2VwdGVkKSB7XG4gICAgICBhd2FpdCB0aGlzLm1vZGlmeVZhbHVlcyhmbGFnTmFtZXMsIHBhcmFtcyk7XG4gICAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmluZm8oJ0ZsYWcgdmFsdWUocykgdXBkYXRlZCBzdWNjZXNzZnVsbHkuJyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbygnT3BlcmF0aW9uIGNhbmNlbGxlZCcpO1xuICAgIH1cblxuICAgIGF3YWl0IHRoaXMuY2xlYW51cFRlbXBEaXJlY3RvcmllcygpO1xuICB9XG5cbiAgLyoqIFJlbW92ZXMgdGVtcG9yYXJ5IGRpcmVjdG9yaWVzIGNyZWF0ZWQgZHVyaW5nIGZsYWcgb3BlcmF0aW9ucyAqL1xuICBwcml2YXRlIGFzeW5jIGNsZWFudXBUZW1wRGlyZWN0b3JpZXMoKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgY29uc3Qgb3JpZ2luYWxEaXIgPSBwYXRoLmpvaW4ocHJvY2Vzcy5jd2QoKSwgJ29yaWdpbmFsJyk7XG4gICAgY29uc3QgdGVtcERpciA9IHBhdGguam9pbihwcm9jZXNzLmN3ZCgpLCAndGVtcCcpO1xuICAgIGF3YWl0IGZzLnJlbW92ZShvcmlnaW5hbERpcik7XG4gICAgYXdhaXQgZnMucmVtb3ZlKHRlbXBEaXIpO1xuICB9XG5cbiAgLyoqIEFjdHVhbGx5IG1vZGlmaWVzIHRoZSBjZGsuanNvbiBmaWxlIHdpdGggdGhlIG5ldyBmbGFnIHZhbHVlcyAqL1xuICBwcml2YXRlIGFzeW5jIG1vZGlmeVZhbHVlcyhmbGFnTmFtZXM6IHN0cmluZ1tdLCBwYXJhbXM6IEZsYWdPcGVyYXRpb25zUGFyYW1zKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgY29uc3QgY2RrSnNvblBhdGggPSBwYXRoLmpvaW4ocHJvY2Vzcy5jd2QoKSwgJ2Nkay5qc29uJyk7XG4gICAgY29uc3QgY2RrSnNvbkNvbnRlbnQgPSBhd2FpdCBmcy5yZWFkRmlsZShjZGtKc29uUGF0aCwgJ3V0Zi04Jyk7XG4gICAgY29uc3QgY2RrSnNvbiA9IEpTT04ucGFyc2UoY2RrSnNvbkNvbnRlbnQpO1xuXG4gICAgaWYgKGZsYWdOYW1lcy5sZW5ndGggPT09IDEgJiYgIXBhcmFtcy5zYWZlKSB7XG4gICAgICBjb25zdCBib29sVmFsdWUgPSBwYXJhbXMudmFsdWUgPT09ICd0cnVlJztcbiAgICAgIGNka0pzb24uY29udGV4dFtTdHJpbmcoZmxhZ05hbWVzWzBdKV0gPSBib29sVmFsdWU7XG4gICAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmluZm8oYFNldHRpbmcgZmxhZyAnJHtmbGFnTmFtZXN9JyB0bzogJHtib29sVmFsdWV9YCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGZvciAoY29uc3QgZmxhZ05hbWUgb2YgZmxhZ05hbWVzKSB7XG4gICAgICAgIGNvbnN0IGZsYWcgPSB0aGlzLmZsYWdzLmZpbmQoZiA9PiBmLm5hbWUgPT09IGZsYWdOYW1lKSE7XG4gICAgICAgIGNvbnN0IG5ld1ZhbHVlID0gcGFyYW1zLnJlY29tbWVuZGVkIHx8IHBhcmFtcy5zYWZlXG4gICAgICAgICAgPyBmbGFnLnJlY29tbWVuZGVkVmFsdWUgYXMgYm9vbGVhblxuICAgICAgICAgIDogU3RyaW5nKGRlZmF1bHRWYWx1ZShmbGFnKSkgPT09ICd0cnVlJztcbiAgICAgICAgY2RrSnNvbi5jb250ZXh0W2ZsYWdOYW1lXSA9IG5ld1ZhbHVlO1xuICAgICAgfVxuICAgIH1cbiAgICBhd2FpdCBmcy53cml0ZUZpbGUoY2RrSnNvblBhdGgsIEpTT04uc3RyaW5naWZ5KGNka0pzb24sIG51bGwsIDIpLCAndXRmLTgnKTtcbiAgfVxuXG4gIC8qKiBEaXNwbGF5cyBmbGFncyBpbiBhIHRhYmxlIGZvcm1hdCwgZWl0aGVyIHNwZWNpZmljIGZsYWdzIG9yIGZpbHRlcmVkIGJ5IGNyaXRlcmlhICovXG4gIGFzeW5jIGRpc3BsYXlGbGFncyhwYXJhbXM6IEZsYWdPcGVyYXRpb25zUGFyYW1zKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgY29uc3QgeyBGTEFHTkFNRSwgYWxsIH0gPSBwYXJhbXM7XG5cbiAgICBpZiAoRkxBR05BTUUgJiYgRkxBR05BTUUubGVuZ3RoID4gMCkge1xuICAgICAgYXdhaXQgdGhpcy5kaXNwbGF5U3BlY2lmaWNGbGFncyhGTEFHTkFNRSk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY29uc3QgW2ZsYWdzVG9EaXNwbGF5LCBoZWFkZXJdID0gYWxsXG4gICAgICA/IFt0aGlzLmZsYWdzLCAnQWxsIGZlYXR1cmUgZmxhZ3MnXVxuICAgICAgOiBbRmxhZ09wZXJhdGlvbnMuZmlsdGVyTmVlZHNBdHRlbnRpb24odGhpcy5mbGFncyksICdVbmNvbmZpZ3VyZWQgZmVhdHVyZSBmbGFncyddO1xuXG4gICAgYXdhaXQgdGhpcy5pb0hlbHBlci5kZWZhdWx0cy5pbmZvKGhlYWRlcik7XG4gICAgYXdhaXQgdGhpcy5kaXNwbGF5RmxhZ1RhYmxlKGZsYWdzVG9EaXNwbGF5KTtcblxuICAgIC8vIEFkZCBoZWxwZnVsIG1lc3NhZ2UgYWZ0ZXIgZW1wdHkgdGFibGUgd2hlbiBub3QgdXNpbmcgLS1hbGxcbiAgICBpZiAoIWFsbCAmJiBmbGFnc1RvRGlzcGxheS5sZW5ndGggPT09IDApIHtcbiAgICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbygnJyk7XG4gICAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmluZm8oJ+KchSBBbGwgZmVhdHVyZSBmbGFncyBhcmUgYWxyZWFkeSBzZXQgdG8gdGhlaXIgcmVjb21tZW5kZWQgdmFsdWVzLicpO1xuICAgICAgYXdhaXQgdGhpcy5pb0hlbHBlci5kZWZhdWx0cy5pbmZvKCdVc2UgXFwnY2RrIGZsYWdzIC0tYWxsIC0tdW5zdGFibGU9ZmxhZ3NcXCcgdG8gc2VlIGFsbCBmbGFncyBhbmQgdGhlaXIgY3VycmVudCB2YWx1ZXMuJyk7XG4gICAgfVxuICB9XG5cbiAgLyoqIERpc3BsYXlzIGRldGFpbGVkIGluZm9ybWF0aW9uIGZvciBzcGVjaWZpYyBmbGFncyBtYXRjaGluZyB0aGUgZ2l2ZW4gbmFtZXMgKi9cbiAgcHJpdmF0ZSBhc3luYyBkaXNwbGF5U3BlY2lmaWNGbGFncyhmbGFnTmFtZXM6IHN0cmluZ1tdKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgY29uc3QgbWF0Y2hpbmdGbGFncyA9IHRoaXMuZmxhZ3MuZmlsdGVyKGYgPT5cbiAgICAgIGZsYWdOYW1lcy5zb21lKHNlYXJjaFRlcm0gPT4gZi5uYW1lLnRvTG93ZXJDYXNlKCkuaW5jbHVkZXMoc2VhcmNoVGVybS50b0xvd2VyQ2FzZSgpKSkpO1xuXG4gICAgaWYgKG1hdGNoaW5nRmxhZ3MubGVuZ3RoID09PSAwKSB7XG4gICAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmVycm9yKGBGbGFnIG1hdGNoaW5nIFwiJHtmbGFnTmFtZXMuam9pbignLCAnKX1cIiBub3QgZm91bmQuYCk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgaWYgKG1hdGNoaW5nRmxhZ3MubGVuZ3RoID09PSAxKSB7XG4gICAgICBjb25zdCBmbGFnID0gbWF0Y2hpbmdGbGFnc1swXTtcbiAgICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbyhgRmxhZyBuYW1lOiAke2ZsYWcubmFtZX1gKTtcbiAgICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbyhgRGVzY3JpcHRpb246ICR7ZmxhZy5leHBsYW5hdGlvbn1gKTtcbiAgICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbyhgUmVjb21tZW5kZWQgdmFsdWU6ICR7ZmxhZy5yZWNvbW1lbmRlZFZhbHVlfWApO1xuICAgICAgYXdhaXQgdGhpcy5pb0hlbHBlci5kZWZhdWx0cy5pbmZvKGBEZWZhdWx0IHZhbHVlOiAke2RlZmF1bHRWYWx1ZShmbGFnKX1gKTtcbiAgICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbyhgVXNlciB2YWx1ZTogJHtmbGFnLnVzZXJWYWx1ZX1gKTtcbiAgICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbyhgRWZmZWN0aXZlIHZhbHVlOiAke2VmZmVjdGl2ZVZhbHVlKGZsYWcpfWApO1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbyhgRm91bmQgJHttYXRjaGluZ0ZsYWdzLmxlbmd0aH0gZmxhZ3MgbWF0Y2hpbmcgXCIke2ZsYWdOYW1lcy5qb2luKCcsICcpfVwiYCk7XG4gICAgYXdhaXQgdGhpcy5kaXNwbGF5RmxhZ1RhYmxlKG1hdGNoaW5nRmxhZ3MpO1xuICB9XG5cbiAgLyoqIFJldHVybnMgc29ydCBvcmRlciBmb3IgZmxhZ3MgKi9cbiAgcHJpdmF0ZSBnZXRGbGFnU29ydE9yZGVyKGZsYWc6IEZlYXR1cmVGbGFnKTogbnVtYmVyIHtcbiAgICBpZiAoZmxhZy51c2VyVmFsdWUgPT09IHVuZGVmaW5lZCkgcmV0dXJuIDM7XG4gICAgaWYgKGlzRWZmZWN0aXZlVmFsdWVFcXVhbFRvUmVjb21tZW5kZWQoZmxhZykpIHJldHVybiAxO1xuICAgIHJldHVybiAyO1xuICB9XG5cbiAgLyoqIERpc3BsYXlzIGZsYWdzIGluIGEgZm9ybWF0dGVkIHRhYmxlIGdyb3VwZWQgYnkgbW9kdWxlIGFuZCBzb3J0ZWQgKi9cbiAgYXN5bmMgZGlzcGxheUZsYWdUYWJsZShmbGFnczogRmVhdHVyZUZsYWdbXSk6IFByb21pc2U8dm9pZD4ge1xuICAgIGNvbnN0IHNvcnRlZEZsYWdzID0gWy4uLmZsYWdzXS5zb3J0KChhLCBiKSA9PiB7XG4gICAgICBjb25zdCBvcmRlckEgPSB0aGlzLmdldEZsYWdTb3J0T3JkZXIoYSk7XG4gICAgICBjb25zdCBvcmRlckIgPSB0aGlzLmdldEZsYWdTb3J0T3JkZXIoYik7XG5cbiAgICAgIGlmIChvcmRlckEgIT09IG9yZGVyQikgcmV0dXJuIG9yZGVyQSAtIG9yZGVyQjtcbiAgICAgIGlmIChhLm1vZHVsZSAhPT0gYi5tb2R1bGUpIHJldHVybiBhLm1vZHVsZS5sb2NhbGVDb21wYXJlKGIubW9kdWxlKTtcbiAgICAgIHJldHVybiBhLm5hbWUubG9jYWxlQ29tcGFyZShiLm5hbWUpO1xuICAgIH0pO1xuXG4gICAgY29uc3Qgcm93czogc3RyaW5nW11bXSA9IFtbJ0ZlYXR1cmUgRmxhZycsICdSZWNvbW1lbmRlZCcsICdVc2VyJywgJ0VmZmVjdGl2ZSddXTtcbiAgICBsZXQgY3VycmVudE1vZHVsZSA9ICcnO1xuXG4gICAgc29ydGVkRmxhZ3MuZm9yRWFjaCgoZmxhZykgPT4ge1xuICAgICAgaWYgKGZsYWcubW9kdWxlICE9PSBjdXJyZW50TW9kdWxlKSB7XG4gICAgICAgIHJvd3MucHVzaChbY2hhbGsuYm9sZChgTW9kdWxlOiAke2ZsYWcubW9kdWxlfWApLCAnJywgJycsICcnXSk7XG4gICAgICAgIGN1cnJlbnRNb2R1bGUgPSBmbGFnLm1vZHVsZTtcbiAgICAgIH1cbiAgICAgIHJvd3MucHVzaChbXG4gICAgICAgIGAgICR7ZmxhZy5uYW1lfWAsXG4gICAgICAgIFN0cmluZyhmbGFnLnJlY29tbWVuZGVkVmFsdWUpLFxuICAgICAgICBmbGFnLnVzZXJWYWx1ZSA9PT0gdW5kZWZpbmVkID8gJzx1bnNldD4nIDogU3RyaW5nKGZsYWcudXNlclZhbHVlKSxcbiAgICAgICAgU3RyaW5nKGVmZmVjdGl2ZVZhbHVlKGZsYWcpKSxcbiAgICAgIF0pO1xuICAgIH0pO1xuXG4gICAgY29uc3QgZm9ybWF0dGVkVGFibGUgPSBmb3JtYXRUYWJsZShyb3dzLCB1bmRlZmluZWQsIHRydWUpO1xuICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbyhmb3JtYXR0ZWRUYWJsZSk7XG4gIH1cblxuICAvKiogQ2hlY2tzIGlmIGEgZmxhZyBoYXMgYSBib29sZWFuIHJlY29tbWVuZGVkIHZhbHVlICovXG4gIGlzQm9vbGVhbkZsYWcoZmxhZzogRmVhdHVyZUZsYWcpOiBib29sZWFuIHtcbiAgICBjb25zdCByZWNvbW1lbmRlZCA9IGZsYWcucmVjb21tZW5kZWRWYWx1ZTtcbiAgICByZXR1cm4gdHlwZW9mIHJlY29tbWVuZGVkID09PSAnYm9vbGVhbicgfHxcbiAgICAgIHJlY29tbWVuZGVkID09PSAndHJ1ZScgfHxcbiAgICAgIHJlY29tbWVuZGVkID09PSAnZmFsc2UnO1xuICB9XG5cbiAgLyoqIFNob3dzIGhlbHBmdWwgdXNhZ2UgZXhhbXBsZXMgYW5kIGF2YWlsYWJsZSBjb21tYW5kIG9wdGlvbnMgKi9cbiAgYXN5bmMgZGlzcGxheUhlbHBNZXNzYWdlKCk6IFByb21pc2U8dm9pZD4ge1xuICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbygnXFxuJyArIGNoYWxrLmJvbGQoJ0F2YWlsYWJsZSBvcHRpb25zOicpKTtcbiAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmluZm8oJyAgY2RrIGZsYWdzIC0taW50ZXJhY3RpdmUgICAgICMgSW50ZXJhY3RpdmUgbWVudSB0byBtYW5hZ2UgZmxhZ3MnKTtcbiAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmluZm8oJyAgY2RrIGZsYWdzIC0tYWxsICAgICAgICAgICAgICMgU2hvdyBhbGwgZmxhZ3MgKGluY2x1ZGluZyBjb25maWd1cmVkIG9uZXMpJyk7XG4gICAgYXdhaXQgdGhpcy5pb0hlbHBlci5kZWZhdWx0cy5pbmZvKCcgIGNkayBmbGFncyAtLXNldCAtLWFsbCAtLXJlY29tbWVuZGVkICAgICMgU2V0IGFsbCBmbGFncyB0byByZWNvbW1lbmRlZCB2YWx1ZXMnKTtcbiAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmluZm8oJyAgY2RrIGZsYWdzIC0tc2V0IC0tYWxsIC0tZGVmYXVsdCAgICAgICAjIFNldCBhbGwgZmxhZ3MgdG8gZGVmYXVsdCB2YWx1ZXMnKTtcbiAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmluZm8oJyAgY2RrIGZsYWdzIC0tc2V0IC0tdW5jb25maWd1cmVkIC0tcmVjb21tZW5kZWQgICMgU2V0IHVuY29uZmlndXJlZCBmbGFncyB0byByZWNvbW1lbmRlZCcpO1xuICAgIGF3YWl0IHRoaXMuaW9IZWxwZXIuZGVmYXVsdHMuaW5mbygnICBjZGsgZmxhZ3MgLS1zZXQgPGZsYWctbmFtZT4gLS12YWx1ZSA8dHJ1ZXxmYWxzZT4gICMgU2V0IHNwZWNpZmljIGZsYWcnKTtcbiAgICBhd2FpdCB0aGlzLmlvSGVscGVyLmRlZmF1bHRzLmluZm8oJyAgY2RrIGZsYWdzIC0tc2FmZSAgICAgICAgICAgICMgU2FmZWx5IHNldCBmbGFncyB0aGF0IGRvblxcJ3QgY2hhbmdlIHRlbXBsYXRlcycpO1xuICB9XG59XG5cbi8qKiBDaGVja3MgaWYgdGhlIGZsYWdzIGN1cnJlbnQgZWZmZWN0aXZlIHZhbHVlIG1hdGNoZXMgdGhlIHJlY29tbWVuZGVkIHZhbHVlICovXG5leHBvcnQgZnVuY3Rpb24gaXNFZmZlY3RpdmVWYWx1ZUVxdWFsVG9SZWNvbW1lbmRlZChmbGFnOiBGZWF0dXJlRmxhZyk6IGJvb2xlYW4ge1xuICByZXR1cm4gU3RyaW5nKGVmZmVjdGl2ZVZhbHVlKGZsYWcpKSA9PT0gU3RyaW5nKGZsYWcucmVjb21tZW5kZWRWYWx1ZSk7XG59XG5cbi8qKlxuICogUmV0dXJuIHRoZSBlZmZlY3RpdmUgdmFsdWUgb2YgYSBmbGFnICh1c2VyIHZhbHVlIG9yIGRlZmF1bHQpXG4gKi9cbmZ1bmN0aW9uIGVmZmVjdGl2ZVZhbHVlKGZsYWc6IEZlYXR1cmVGbGFnKSB7XG4gIHJldHVybiBmbGFnLnVzZXJWYWx1ZSA/PyBkZWZhdWx0VmFsdWUoZmxhZyk7XG59XG5cbi8qKlxuICogUmV0dXJuIHRoZSBkZWZhdWx0IHZhbHVlIGZvciBhIGZsYWcsIGFzc3VtZSBpdCdzIGBmYWxzZWAgaWYgbm90IGdpdmVuXG4gKi9cbmZ1bmN0aW9uIGRlZmF1bHRWYWx1ZShmbGFnOiBGZWF0dXJlRmxhZykge1xuICByZXR1cm4gZmxhZy51bmNvbmZpZ3VyZWRCZWhhdmVzTGlrZT8udjIgPz8gZmFsc2U7XG59XG4iXX0=