import type { IoHelper } from '../../api-private';
import { type JsPackageManager } from './package-manager';
export interface CliInitOptions {
    /**
     * Template name to initialize
     * @default undefined
     */
    readonly type?: string;
    /**
     * Programming language for the project
     * @default - Optional/auto-detected if template supports only one language, otherwise required
     */
    readonly language?: string;
    /**
     * @default true
     */
    readonly canUseNetwork?: boolean;
    /**
     * @default false
     */
    readonly generateOnly?: boolean;
    /**
     * @default process.cwd()
     */
    readonly workDir?: string;
    /**
     * @default undefined
     */
    readonly projectName?: string;
    /**
     * @default undefined
     */
    readonly stackName?: string;
    /**
     * @default undefined
     */
    readonly migrate?: boolean;
    /**
     * Override the built-in CDK version
     * @default undefined
     */
    readonly libVersion?: string;
    /**
     * Path to a local custom template directory
     * @default undefined
     */
    readonly fromPath?: string;
    /**
     * Path to a specific template within a multi-template repository.
     * This parameter requires --from-path to be specified.
     * @default undefined
     */
    readonly templatePath?: string;
    /**
     * The package manager to use for installing dependencies. Only applicable for TypeScript and JavaScript projects.
     * @default - If specified language is 'typescript' or 'javascript', 'npm' is selected. Otherwise, no package manager is used.
     */
    readonly packageManager?: JsPackageManager;
    readonly ioHelper: IoHelper;
}
/**
 * Initialize a CDK package in the current directory
 */
export declare function cliInit(options: CliInitOptions): Promise<void>;
interface TemplateInitInfo {
    readonly description: string;
    readonly aliases?: string[];
}
declare enum TemplateType {
    BUILT_IN = "builtin",
    CUSTOM = "custom"
}
export declare class InitTemplate {
    private readonly basePath;
    readonly name: string;
    readonly languages: string[];
    static fromName(templatesDir: string, name: string): Promise<InitTemplate>;
    static fromPath(templatePath: string): Promise<InitTemplate>;
    readonly description?: string;
    readonly aliases: Set<string>;
    readonly templateType: TemplateType;
    constructor(basePath: string, name: string, languages: string[], initInfo: TemplateInitInfo | null, templateType: TemplateType);
    /**
     * @param name - the name that is being checked
     * @returns ``true`` if ``name`` is the name of this template or an alias of it.
     */
    hasName(name: string): boolean;
    /**
     * Creates a new instance of this ``InitTemplate`` for a given language to a specified folder.
     *
     * @param language - the language to instantiate this template with
     * @param targetDirectory - the directory where the template is to be instantiated into
     * @param stackName - the name of the stack to create
     * @default undefined
     * @param libVersion - the version of the CDK library to use
     * @default undefined
     */
    install(ioHelper: IoHelper, language: string, targetDirectory: string, projectName?: string, stackName?: string, libVersion?: string, packageManager?: JsPackageManager): Promise<void>;
    private installFiles;
    private installProcessed;
    /**
     * Copy template files without processing placeholders (for custom templates)
     */
    private installFilesWithoutProcessing;
    /**
     * Adds context variables to `cdk.json` in the generated project directory to
     * enable future behavior for new projects.
     */
    private applyFutureFlags;
    addMigrateContext(projectDir: string): Promise<void>;
}
export declare function expandPlaceholders(template: string, language: string, project: ProjectInfo, packageManager?: JsPackageManager): string;
interface ProjectInfo {
    /** The value used for %name% */
    readonly name: string;
    readonly stackName?: string;
    readonly versions: Versions;
}
export declare function availableInitTemplates(): Promise<InitTemplate[]>;
export declare function availableInitLanguages(): Promise<string[]>;
/**
 * Print available templates to the user
 * @param ioHelper - IO helper for user interaction
 * @param language - Programming language filter
 * @default undefined
 */
export declare function printAvailableTemplates(ioHelper: IoHelper, language?: string): Promise<void>;
interface Versions {
    ['aws-cdk']: string;
    ['aws-cdk-lib']: string;
    constructs: string;
}
/**
 * Return the currently recommended flags for `aws-cdk-lib`.
 *
 * These have been built into the CLI at build time.
 */
export declare function currentlyRecommendedAwsCdkLibFlags(): Promise<any>;
export {};
