// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License. See License.txt in the project root for license information.
import { __awaiter, __generator } from "tslib";
import { flattenResponse } from "@azure/ms-rest-js";
import { createLROPollStrategyFromInitialResponse, createLROPollStrategyFromPollState } from "./lroPollStrategy";
/**
 * An HTTP operation response that provides special methods for interacting with LROs (long running
 * operations).
 */
var LROPoller = /** @class */ (function () {
    /**
     * Create a new HttpLongRunningOperationResponse.
     * @param _lroPollStrategy The LROPollStrategy that this HttpLongRunningOperationResponse will
     * use to interact with the LRO.
     */
    function LROPoller(_lroPollStrategy, _initialResponse) {
        this._lroPollStrategy = _lroPollStrategy;
        this._initialResponse = _initialResponse;
    }
    /**
     * Get the first response that the service sent back when the LRO was initiated.
     */
    LROPoller.prototype.getInitialResponse = function () {
        return this._initialResponse;
    };
    /**
     * Get the most recent response that the service sent back during this LRO.
     */
    LROPoller.prototype.getMostRecentResponse = function () {
        var lroPollStrategy = this._lroPollStrategy;
        return !lroPollStrategy ? this._initialResponse : lroPollStrategy.getMostRecentResponse();
    };
    /**
     * Get whether or not the LRO is finished.
     */
    LROPoller.prototype.isFinished = function () {
        var lroPollStrategy = this._lroPollStrategy;
        return !lroPollStrategy ? true : lroPollStrategy.isFinished();
    };
    /**
     * Get whether or not the LRO is finished and its final state is acceptable. If the LRO has not
     * finished yet, then undefined will be returned. An "acceptable" final state is determined by the
     * LRO strategy that the Azure service uses to perform long running operations.
     */
    LROPoller.prototype.isFinalStatusAcceptable = function () {
        var result;
        var lroPollStrategy = this._lroPollStrategy;
        if (!lroPollStrategy) {
            result = true;
        }
        else if (lroPollStrategy.isFinished()) {
            result = lroPollStrategy.isFinalStatusAcceptable();
        }
        return result;
    };
    /**
     * Get the current status of the LRO.
     */
    LROPoller.prototype.getOperationStatus = function () {
        var lroPollStrategy = this._lroPollStrategy;
        return !lroPollStrategy ? "Succeeded" : lroPollStrategy.getOperationStatus();
    };
    /**
     * If the LRO is finished and in an acceptable state, then return the HttpOperationResponse. If
     * the LRO is finished and not in an acceptable state, then throw the error that the LRO produced.
     * If the LRO is not finished, then return undefined.
     */
    LROPoller.prototype.getOperationResponse = function () {
        var result;
        var lroPollStrategy = this._lroPollStrategy;
        if (!lroPollStrategy) {
            result = Promise.resolve(this._initialResponse);
        }
        else if (!lroPollStrategy.isFinished()) {
            result = Promise.resolve(undefined);
        }
        else if (lroPollStrategy.isFinalStatusAcceptable()) {
            result = lroPollStrategy.getOperationResponse();
        }
        else {
            throw lroPollStrategy.getRestError();
        }
        return result;
    };
    /**
     * Send a single poll request and return the LRO's state.
     */
    LROPoller.prototype.poll = function () {
        var result;
        var lroPollStrategy = this._lroPollStrategy;
        if (!lroPollStrategy) {
            result = Promise.resolve("Succeeded");
        }
        else {
            result = lroPollStrategy.sendPollRequest().then(function () {
                return lroPollStrategy.getOperationStatus();
            });
        }
        return result;
    };
    /**
     * Send poll requests that check the LRO's status until it is determined that the LRO is finished.
     */
    LROPoller.prototype.pollUntilFinished = function () {
        return __awaiter(this, void 0, void 0, function () {
            var result, lroPollStrategy;
            return __generator(this, function (_a) {
                lroPollStrategy = this._lroPollStrategy;
                if (!lroPollStrategy) {
                    result = Promise.resolve(flattenAzureResponse(this._initialResponse));
                }
                else {
                    result = lroPollStrategy.pollUntilFinished().then(function (succeeded) {
                        if (succeeded) {
                            return lroPollStrategy.getOperationResponse().then(flattenAzureResponse);
                        }
                        else {
                            throw lroPollStrategy.getRestError();
                        }
                    });
                }
                return [2 /*return*/, result];
            });
        });
    };
    /**
     * Get an LROPollState object that can be used to poll this LRO in a different context (such as on
     * a different process or a different machine). If the LRO couldn't produce an LRO polling
     * strategy, then this will return undefined.
     */
    LROPoller.prototype.getPollState = function () {
        var lroPollStrategy = this._lroPollStrategy;
        return !lroPollStrategy ? undefined : lroPollStrategy.getPollState();
    };
    return LROPoller;
}());
export { LROPoller };
export function createLROPollerFromInitialResponse(azureServiceClient, initialResponse, options) {
    var lroPollStrategy = createLROPollStrategyFromInitialResponse(initialResponse, azureServiceClient, options);
    return new LROPoller(lroPollStrategy, initialResponse);
}
export function createLROPollerFromPollState(azureServiceClient, lroMemento) {
    var lroPollStrategy = createLROPollStrategyFromPollState(azureServiceClient, lroMemento);
    return new LROPoller(lroPollStrategy, lroMemento.initialResponse);
}
function flattenAzureResponse(response) {
    var _a = response.request, operationResponseGetter = _a.operationResponseGetter, operationSpec = _a.operationSpec;
    return flattenResponse(response, operationResponseGetter && operationSpec && operationResponseGetter(operationSpec, response));
}
//# sourceMappingURL=lroPoller.js.map