/*****************************************************************\
*   BBC BASIC for SDL 2.0 (32-bit Web Assembly)                   *
*   Copyright (c) R. T. Russell, 2000-2023                        *
*                                                                 *
*   BBDATA_WASM.C RAM data definitions                            *
*   Version 1.38a, 03-Sep-2023                                    *
\*****************************************************************/

#include <stdlib.h>

#define MAX_PORTS 4
#define MAX_FILES 8
#define SOUNDQL 20
#define STYPE 24

// Variant:
typedef union tagVAR
{
	double f ;
        struct
        {
          long long n ;
          short t ; // = 0
        } i ;
        struct
        {
          unsigned int p ; // Assumed to be 32 bits
          unsigned int l ; // Must be unsigned for overflow tests in 'math'
          short t ; // = -1
        } s ;
	struct
	{
	  double d ;
	  short t ; // unused (loadn/storen only)
	} d ;
} VAR, *LPVAR ; 

// String descriptor:
typedef struct tagSTR
{
	unsigned int p ; // Assumed to be 32 bits
	int l ;
} STR, *LPSTR ;

typedef struct tagFCB
{
	unsigned char p ; // pointer
	unsigned char o ; // offset  (0-256)
	unsigned char w ; // written (0-256)
	signed char f ;   // bit0: offset<>0, bit7: written<>0
} FCB, *LPFCB ;

// Structure for linked list of string free space
struct node
{
	struct node *next ;
	char *data ;
} ;
typedef struct node node ;

// Type definitions for SYS wrapper:
typedef size_t st ;
typedef double db ;
typedef struct {char *handler; char *proc; unsigned char *flags; } timerparam ;

// Externals:

extern unsigned int palette[256] ;
extern void *TTFcache[] ;

// Functions:

VAR loadn (void *, unsigned char) ;
VAR loads (void *, unsigned char) ;
void storen (VAR, void *, unsigned char) ;
void stores (VAR, void *, unsigned char) ;
void *getvar (unsigned char *) ;
void *putvar (void *, unsigned char *) ;
VAR expr (void) ;
VAR item (void) ;
char *lexan (char *, char *, unsigned char) ;
void token (signed char) ;
void xfloat (VAR *) ;
void xfix (VAR *) ;
int str00 (VAR, char*, int) ;
VAR con (void) ;
void sortup (int, int, int, unsigned int, unsigned int, unsigned int, void *);
void sortdn (int, int, int, unsigned int, unsigned int, unsigned int, void *);
unsigned int hook (unsigned int, timerparam *) ;
VAR xeq (void) ;
int putevt (int, int, int, int) ;
const void *gfxPrimitivesGetFont (void) ;
void gfxPrimitivesSetFont (const void *, unsigned int, unsigned int) ;
int RedefineChar(void *, char, unsigned char *, unsigned int, unsigned int) ;

// Variables used by generic modules (BBMAIN, BBEXEC, BBEVAL, BBASMB):

int stavar[27+54+64] = {0} ;	// Static+dynamic variables, interpreter internal variables
node *flist[33] = {0} ;		// Pointers to string free lists
STR tmps = {0} ; 		// Temp string descriptor

// Text/graphics metrics (pointed to by @vdu% or accessed as @vdu{}):

int vduvar[64] = {0} ;		// First is origx (@vdu%!0), last is prnty (@vdu%!252)

unsigned int lastick = 0 ;

// Sub-structure {x%,y%}:

int ptfmt[7] =  {8,			// size
		12, 0x00002578, 0,	// x% (offset 0)
		 0, 0x04002579, 0 } ;	// y% (offset 4)

// Sub-structure {a&,b&,c&,d&}:

int b4fmt[13] = {4,			// size
		12, 0x00002661, 0,	// a& (offset 0)
		12, 0x01002662, 0,	// b& (offset 1)
		12, 0x02002663, 0,	// c& (offset 2)
		 0, 0x03002664, 0 } ;	// d& (offset 3)

// Structure @vdu{o{x%,y%}, l{x%,y%}, p{x%,y%}, tl%, tr%, tt%, tb%,
//                d{x%,y%}, c{x%,y%}, hf%,  hr%,  g{a&,b&,c&,d&}, 
//                t{a&,b&,c&,d&}, m{a&,b&,c&,d&}, w{a&,b&,c&,d&}}:

int vdustr[55] = { 0x00005000, // vdufmt = vdustr + 1
	0x00001000, 0x007B6F00, (int) ptfmt, 0x00000000, // @vdu.o{}
	0x00001000, 0x007B6C00, (int) ptfmt, 0x00000008, // @vdu.l{}
	0x00000F00, 0x007B7000, (int) ptfmt, 0x00000010, // @vdu.p{}
	0x0000000C, 0x00256C74,              0x00000018, // @vdu.tl%
	0x0000000C, 0x00257274,              0x0000001C, // @vdu.tr%
	0x0000000C, 0x00257474,              0x00000020, // @vdu.tt%
	0x0000000D, 0x00256274,              0x00000024, // @vdu.tb%
	0x00001000, 0x007B6400, (int) ptfmt, 0x00000028, // @vdu.d{}
	0x00000F00, 0x007B6300, (int) ptfmt, 0x00000030, // @vdu.c{}
	0x0000000C, 0x00256668,              0x00000038, // @vdu.hf%
	0x0000000D, 0x00257268,              0x0000003C, // @vdu.hr%
	0x00001000, 0x007B6700, (int) b4fmt, 0x00000040, // @vdu.g{}
	0x00001000, 0x007B7400, (int) b4fmt, 0x00000044, // @vdu.t{}
	0x00001000, 0x007B6D00, (int) b4fmt, 0x00000048, // @vdu.m{}
	0x00000000, 0x007B7700, (int) b4fmt, 0x0000004C};// @vdu.w{}

// Wrapper functions to match SYS signature:

long long BBC_putevt(st handler, st msg, st wparam, st lparam, st i4, st i5, st i6, st i7,
	  st i8, st i9, st i10, st i11, db f0, db f1, db f2, db f3, db f4, db f5, db f6, db f7)
	{ return putevt(handler, msg, wparam, lparam); }

long long GFX_GetFont(st i0, st i1, st i2, st i3, st i4, st i5, st i6, st i7,
	  st i8, st i9, st i10, st i11, db f0, db f1, db f2, db f3, db f4, db f5, db f6, db f7)
	{ return (int) gfxPrimitivesGetFont(); }

long long GFX_SetFont(st fontdata, st cw, st ch, st i3, st i4, st i5, st i6, st i7,
	  st i8, st i9, st i10, st i11, db f0, db f1, db f2, db f3, db f4, db f5, db f6, db f7)
	{ gfxPrimitivesSetFont((void *) fontdata, cw, ch); return 0; }

long long GFX_DefChar(st renderer, st c, st charpos, st width, st height, st i5, st i6, st i7,
	  st i8, st i9, st i10, st i11, db f0, db f1, db f2, db f3, db f4, db f5, db f6, db f7)
	{ return RedefineChar((void *) renderer, (char) c, (unsigned char *) charpos, width, height); }

// Array of function entry points:

void *fntab[24] = {	// fnarr = fntab + 3
	(void *) 0x01000000,	// no. of dimensions = 1	
	(void *) 22, 		// no. of elements
	loadn,		// Load numeric 
	loads,		// Load string
	storen,		// Store numeric
	stores,		// Store string
	getvar,		// Get variable address
	putvar,		// Create variable
	expr,		// Evaluate expression
	item,		// Evaluate item
	lexan,		// Lexical analysis (tokenise)
	token,		// Print character or keyword
	xfloat,		// Convert integer to float
	xfix,		// Convert float to integer
	str00,		// Convert a number to a string
	con,		// Convert a string to a number
	sortup,		// Sort ascending (used by sortlib)
	sortdn,		// Sort descending (used by sortlib)
	hook,		// Timer callback (used by timerlib)
	xeq,		// Return to BASIC
	BBC_putevt,	// Store event in queue
	GFX_GetFont,
	GFX_SetFont,
	GFX_DefChar } ;

// The following variables are organised as a linked-list
// for access via 'system variables' starting with @:

int sysvar[128] = {
	0x00000010, 0x686D656D, 0x00256364, 0x00000000, 		// 0   @memhdc%
	0x00000010, 0x72617077, 0x00256D61, 0x00000000, 		// 4   @wparam%
	0x00000012, 0x7261706C, 0x00256D61, 0x00000000, 		// 8   @lparam%
	0x00100000, 0x77680000, 0x0025646E, 0x00000000,			// 12  @hwnd%
	0x000E0000, 0x70680000, 0x00256C61, (int) palette,		// 16  @hpal%
	0x0000000C, 0x0025786F, 0x00000000,				// 20  @ox%
	0x0000000C, 0x0025796F, 0x00000000,				// 23  @oy%
	0x00000013, 0x6C696668, 0x00282565, (int) stavar+447,		// 26  @hfile%(
	0x10000000, 0x6D000000,	0x00256773, 0x00000000, 		// 30  @msg%
	0x0E000000, 0x76000000, 0x00257564, (int) vduvar,		// 34  @vdu%
	0x00001000, 0x70736900, 0x00256C61, 0x00000000, 		// 38  @ispal%
	0x00001000, 0x616C6600, 0x00257367, 0x00000000, 		// 42  @flags%
	0x00001100, 0x63616300, 0x00256568, (int) TTFcache, 		// 46  @cache%
	0x00110000, 0x6F7A0000, 0x00256D6F, 0x00008000, 		// 50  @zoom%
	0x14000000, 0x64000000, 0x00247269, 0x00000000, 0x00000000,	// 54  @dir$
	0x14000000, 0x6C000000, 0x00246269, 0x00000000, 0x00000000,	// 59  @lib$
	0x14000000, 0x63000000, 0x0024646D, 0x00000000, 0x00000000,	// 64  @cmd$
	0x14000000, 0x75000000, 0x00247273, 0x00000000, 0x00000000,	// 69  @usr$
	0x14000000, 0x74000000, 0x0024706D, 0x00000000, 0x00000000,	// 74  @tmp$
	0x0F000000, 0x68000000, 0x00256F77, 0x00000000,			// 79  @hwo%
	0x00120000, 0x6C700000, 0x6F667461, 0x00256D72, 0x00000000, 	// 83  @platform%
	0x00000012, 0x6D726863, 0x00257061, 0x00000000,			// 88  @chrmap%
	0x00100000, 0x61700000, 0x0025786E, 0x00000000, 		// 92  @panx%
	0x00110000, 0x61700000, 0x0025796E, 0x00000000, 		// 96  @pany%
	0x14000000, 0x76000000, 0x007B7564, (int) vdustr + 1, (int) vduvar, 	// 100 @vdu{
	0x0E000000, 0x66000000, 0x0028256E, (int) fntab + 3, 		// 105 @fn%(
	0x00001000, 0x6B726200, 0x00257470, 0x00000000,    		// 109 @brkpt%
	0x00001100, 0x6B726200, 0x00256968, 0x00000000, 		// 113 @brkhi%
	0x00140000, 0x69730000, 0x007B657A, (int) ptfmt, (int) vduvar + 208, 	// 117 @size{
	0x00000000, 0x68630000, 0x007B7261, (int) ptfmt, (int) vduvar + 216,	// 122 @char{
	0x00000000};	// 127 voices

unsigned char fvtab[7] = { 1, 4, 8, 10, STYPE, 40, 136 } ;	// Fast variable types

unsigned char bbcfont[256*8] = {
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // NUL 
	0x7E,0x81,0xA5,0x81,0xBD,0x99,0x81,0x7E, // SOH 
	0x7E,0xFF,0xDB,0xFF,0xC3,0xE7,0xFF,0x7E, // STX 
	0x6C,0xFE,0xFE,0xFE,0x7C,0x38,0x10,0x00, // ETX 
	0x10,0x38,0x7C,0xFE,0x7C,0x38,0x10,0x00, // EOT 
	0x10,0x38,0x38,0xD6,0xFE,0xD6,0x10,0x38, // ENQ 
	0x10,0x38,0x7C,0xFE,0xFE,0xFE,0x38,0x7C, // ACK 
	0x00,0x00,0x18,0x3C,0x3C,0x18,0x00,0x00, // BEL 
	0xFF,0xFF,0xE7,0xC3,0xC3,0xE7,0xFF,0xFF, // BS 
	0x00,0x3C,0x66,0x42,0x42,0x66,0x3C,0x00, // HT 
	0xFF,0xC3,0x99,0xBD,0xBD,0x99,0xC3,0xFF, // LF 
	0x0F,0x07,0x0F,0x7D,0xCC,0xCC,0xCC,0x78, // VT 
	0x3C,0x66,0x66,0x66,0x3C,0x18,0x7E,0x18, // FF 
	0x3F,0x33,0x3F,0x30,0x30,0x70,0xF0,0xE0, // CR 
	0x7F,0x63,0x7F,0x63,0x63,0x67,0xE6,0xC0, // SO 
	0x99,0x5A,0x3C,0xE7,0xE7,0x3C,0x5A,0x99, // SI 
	0x80,0xE0,0xF8,0xFE,0xF8,0xE0,0x80,0x00, // DLE 
	0x02,0x0E,0x3E,0xFE,0x3E,0x0E,0x02,0x00, // DC1 
	0x18,0x3C,0x7E,0x18,0x18,0x7E,0x3C,0x18, // DC2 
	0x66,0x66,0x66,0x66,0x66,0x00,0x66,0x00, // DC3 
	0x7F,0xDB,0xDB,0x7B,0x1B,0x1B,0x1B,0x00, // DC4 
	0x3E,0x63,0x38,0x6C,0x6C,0x38,0xCC,0x78, // NAK 
	0x00,0x00,0x00,0x00,0x7E,0x7E,0x7E,0x00, // SYN 
	0x18,0x3C,0x7E,0x18,0x7E,0x3C,0x18,0xFF, // ETB 
	0x18,0x3C,0x7E,0x18,0x18,0x18,0x18,0x00, // CAN 
	0x18,0x18,0x18,0x18,0x7E,0x3C,0x18,0x00, // EM 
	0x00,0x18,0x0C,0xFE,0x0C,0x18,0x00,0x00, // SUB 
	0x00,0x30,0x60,0xFE,0x60,0x30,0x00,0x00, // ESC 
	0x00,0x00,0xC0,0xC0,0xC0,0xFE,0x00,0x00, // FS 
	0x00,0x24,0x66,0xFF,0x66,0x24,0x00,0x00, // GS 
	0x00,0x18,0x3C,0x7E,0xFF,0xFF,0x00,0x00, // RS 
	0x00,0xFF,0xFF,0x7E,0x3C,0x18,0x00,0x00, // US 
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, //   
	0x30,0x30,0x30,0x30,0x30,0x00,0x30,0x00, // ! 
	0x6C,0x6C,0x6C,0x00,0x00,0x00,0x00,0x00, // " 
	0x6C,0x6C,0xFE,0x6C,0xFE,0x6C,0x6C,0x00, // # 
	0x18,0x7E,0xD0,0x7C,0x16,0xFC,0x30,0x00, // $ 
	0xC0,0xCC,0x18,0x30,0x60,0xCC,0x0C,0x00, // % 
	0x38,0x6C,0x38,0x76,0xDC,0xCC,0x76,0x00, // & 
	0x18,0x30,0x60,0x00,0x00,0x00,0x00,0x00, // ' 
	0x18,0x30,0x60,0x60,0x60,0x30,0x18,0x00, // ( 
	0x60,0x30,0x18,0x18,0x18,0x30,0x60,0x00, // ) 
	0x00,0x6C,0x38,0xFE,0x38,0x6C,0x00,0x00, // * 
	0x00,0x30,0x30,0xFC,0x30,0x30,0x00,0x00, // + 
	0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x60, // , 
	0x00,0x00,0x00,0xFC,0x00,0x00,0x00,0x00, // - 
	0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00, // . 
	0x00,0x0C,0x18,0x30,0x60,0xC0,0x00,0x00, // / 
	0x78,0xCC,0xDC,0xFC,0xEC,0xCC,0x78,0x00, // 0 
	0x30,0x70,0x30,0x30,0x30,0x30,0xFC,0x00, // 1 
	0x78,0xCC,0x0C,0x18,0x30,0x60,0xFC,0x00, // 2 
	0x78,0xCC,0x0C,0x38,0x0C,0xCC,0x78,0x00, // 3 
	0x18,0x38,0x78,0xD8,0xFC,0x18,0x18,0x00, // 4 
	0xFC,0xC0,0xF8,0x0C,0x0C,0xCC,0x78,0x00, // 5 
	0x38,0x60,0xC0,0xF8,0xCC,0xCC,0x78,0x00, // 6 
	0xFC,0x0C,0x18,0x30,0x60,0x60,0x60,0x00, // 7 
	0x78,0xCC,0xCC,0x78,0xCC,0xCC,0x78,0x00, // 8 
	0x78,0xCC,0xCC,0x7C,0x0C,0x18,0x70,0x00, // 9 
	0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x00, // : 
	0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x60, // ; 
	0x18,0x30,0x60,0xC0,0x60,0x30,0x18,0x00, // < 
	0x00,0x00,0xFC,0x00,0xFC,0x00,0x00,0x00, // = 
	0x60,0x30,0x18,0x0C,0x18,0x30,0x60,0x00, // > 
	0x78,0xCC,0x18,0x30,0x30,0x00,0x30,0x00, // ? 
	0x78,0xCC,0xDC,0xD4,0xDC,0xC0,0x78,0x00, // @ 
	0x78,0xCC,0xCC,0xFC,0xCC,0xCC,0xCC,0x00, // A 
	0xF8,0xCC,0xCC,0xF8,0xCC,0xCC,0xF8,0x00, // B 
	0x78,0xCC,0xC0,0xC0,0xC0,0xCC,0x78,0x00, // C 
	0xF0,0xD8,0xCC,0xCC,0xCC,0xD8,0xF0,0x00, // D 
	0xFC,0xC0,0xC0,0xF8,0xC0,0xC0,0xFC,0x00, // E 
	0xFC,0xC0,0xC0,0xF8,0xC0,0xC0,0xC0,0x00, // F 
	0x78,0xCC,0xC0,0xDC,0xCC,0xCC,0x78,0x00, // G 
	0xCC,0xCC,0xCC,0xFC,0xCC,0xCC,0xCC,0x00, // H 
	0xFC,0x30,0x30,0x30,0x30,0x30,0xFC,0x00, // I 
	0x7C,0x18,0x18,0x18,0x18,0xD8,0x70,0x00, // J 
	0xCC,0xD8,0xF0,0xE0,0xF0,0xD8,0xCC,0x00, // K 
	0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xFC,0x00, // L 
	0xC6,0xEE,0xFE,0xD6,0xD6,0xC6,0xC6,0x00, // M 
	0xCC,0xCC,0xEC,0xFC,0xDC,0xCC,0xCC,0x00, // N 
	0x78,0xCC,0xCC,0xCC,0xCC,0xCC,0x78,0x00, // O 
	0xF8,0xCC,0xCC,0xF8,0xC0,0xC0,0xC0,0x00, // P 
	0x78,0xCC,0xCC,0xCC,0xDC,0x78,0x1C,0x00, // Q 
	0xF8,0xCC,0xCC,0xF8,0xD8,0xCC,0xCC,0x00, // R 
	0x78,0xCC,0xC0,0x78,0x0C,0xCC,0x78,0x00, // S 
	0xFC,0x30,0x30,0x30,0x30,0x30,0x30,0x00, // T 
	0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0x78,0x00, // U 
	0xCC,0xCC,0xCC,0xCC,0xCC,0x78,0x30,0x00, // V 
	0xC6,0xC6,0xD6,0xD6,0xFE,0xEE,0xC6,0x00, // W 
	0xCC,0xCC,0x78,0x30,0x78,0xCC,0xCC,0x00, // X 
	0xCC,0xCC,0xCC,0x78,0x30,0x30,0x30,0x00, // Y 
	0xFC,0x0C,0x18,0x30,0x60,0xC0,0xFC,0x00, // Z 
	0xF8,0xC0,0xC0,0xC0,0xC0,0xC0,0xF8,0x00, // [ 
	0x00,0xC0,0x60,0x30,0x18,0x0C,0x00,0x00, // backslash 
	0x7C,0x0C,0x0C,0x0C,0x0C,0x0C,0x7C,0x00, // ] 
	0x10,0x38,0x6C,0xC6,0x00,0x00,0x00,0x00, // ^ 
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF, // _ 
	0x60,0x30,0x18,0x00,0x00,0x00,0x00,0x00, // ` 
	0x00,0x00,0x78,0x0C,0x7C,0xCC,0x7C,0x00, // a 
	0xC0,0xC0,0xF8,0xCC,0xCC,0xCC,0xF8,0x00, // b 
	0x00,0x00,0x78,0xCC,0xC0,0xCC,0x78,0x00, // c 
	0x0C,0x0C,0x7C,0xCC,0xCC,0xCC,0x7C,0x00, // d 
	0x00,0x00,0x78,0xCC,0xFC,0xC0,0x78,0x00, // e 
	0x38,0x60,0x60,0xF8,0x60,0x60,0x60,0x00, // f 
	0x00,0x00,0x7C,0xCC,0xCC,0x7C,0x0C,0x78, // g 
	0xC0,0xC0,0xF8,0xCC,0xCC,0xCC,0xCC,0x00, // h 
	0x30,0x00,0x70,0x30,0x30,0x30,0x78,0x00, // i 
	0x30,0x00,0x70,0x30,0x30,0x30,0x30,0xE0, // j 
	0xC0,0xC0,0xCC,0xD8,0xF0,0xD8,0xCC,0x00, // k 
	0x70,0x30,0x30,0x30,0x30,0x30,0x78,0x00, // l 
	0x00,0x00,0x6C,0xFE,0xD6,0xD6,0xC6,0x00, // m 
	0x00,0x00,0xF8,0xCC,0xCC,0xCC,0xCC,0x00, // n 
	0x00,0x00,0x78,0xCC,0xCC,0xCC,0x78,0x00, // o 
	0x00,0x00,0xF8,0xCC,0xCC,0xF8,0xC0,0xC0, // p 
	0x00,0x00,0x7C,0xCC,0xCC,0x7C,0x0C,0x0E, // q 
	0x00,0x00,0xD8,0xEC,0xC0,0xC0,0xC0,0x00, // r 
	0x00,0x00,0x7C,0xC0,0x78,0x0C,0xF8,0x00, // s 
	0x60,0x60,0xF8,0x60,0x60,0x60,0x38,0x00, // t 
	0x00,0x00,0xCC,0xCC,0xCC,0xCC,0x7C,0x00, // u 
	0x00,0x00,0xCC,0xCC,0xCC,0x78,0x30,0x00, // v 
	0x00,0x00,0xC6,0xD6,0xD6,0xFE,0x6C,0x00, // w 
	0x00,0x00,0xCC,0x78,0x30,0x78,0xCC,0x00, // x 
	0x00,0x00,0xCC,0xCC,0xCC,0x7C,0x0C,0x78, // y 
	0x00,0x00,0xFC,0x18,0x30,0x60,0xFC,0x00, // z 
	0x18,0x30,0x30,0xE0,0x30,0x30,0x18,0x00, // { 
	0x30,0x30,0x30,0x00,0x30,0x30,0x30,0x00, // | 
	0x60,0x30,0x30,0x1C,0x30,0x30,0x60,0x00, // } 
	0x76,0xDC,0x00,0x00,0x00,0x00,0x00,0x00, // ~ 
	0x00,0x10,0x38,0x6C,0xC6,0xC6,0xFE,0x00, // DEL 
	0x3C,0x62,0xF8,0x60,0xF8,0x62,0x3C,0x00, // &80 euro symbol
	0x00,0x7E,0x7E,0x7E,0x7E,0x7E,0x7E,0x00, // &81 block (teletext)
	0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x30, // &82 single low quotation mark
	0x00,0x0C,0x18,0x18,0x3C,0x18,0x18,0x70, // &83 small letter f with hook
	0x00,0x00,0x00,0x00,0x00,0x6C,0x6C,0xD8, // &84 double low quotation mark
	0x00,0x00,0x00,0x00,0x00,0x00,0x54,0x00, // &85 horizontal ellipsis
	0x18,0x18,0x7E,0x18,0x18,0x18,0x18,0x00, // &86 dagger
	0x18,0x18,0x7E,0x18,0x18,0x7E,0x18,0x18, // &87 double dagger
	0x10,0x38,0x6C,0x00,0x00,0x00,0x00,0x00, // &88 circumflex accent
	0xC6,0xCC,0x18,0x30,0x60,0xDB,0x1B,0x00, // &89 per mille
	0x38,0x7C,0xC6,0x70,0x1C,0xC6,0x7C,0x00, // &8A capital S caron
	0x00,0x18,0x30,0x60,0x30,0x18,0x00,0x00, // &8B left angle quotation mark
	0x7E,0xD8,0xD8,0xDE,0xD8,0xD8,0x7E,0x00, // &8C capital OE ligature
	0x30,0x78,0xFC,0x30,0x30,0x30,0x30,0x00, // &8D up arrow (teletext)
	0x38,0xFE,0x0C,0x18,0x30,0x60,0xFE,0x00, // &8E capital Z caron
	0x00,0x20,0x60,0xFE,0x60,0x20,0x00,0x00, // &8F left arrow (teletext)
	0x00,0x08,0x0C,0xFE,0x0C,0x08,0x00,0x00, // &90 right arrow (teletext)
	0x30,0x30,0x18,0x00,0x00,0x00,0x00,0x00, // &91 left single quotation mark
	0x18,0x18,0x30,0x00,0x00,0x00,0x00,0x00, // &92 right single quotation mark
	0x6C,0x6C,0x36,0x00,0x00,0x00,0x00,0x00, // &93 left double quotation mark
	0x36,0x36,0x6C,0x00,0x00,0x00,0x00,0x00, // &94 right double quotation mark
	0x00,0x00,0x18,0x3C,0x3C,0x18,0x00,0x00, // &95 bullet
	0x00,0x00,0x00,0xFE,0x00,0x00,0x00,0x00, // &96 en dash
	0x00,0x00,0x00,0xFF,0x00,0x00,0x00,0x00, // &97 em dash
	0x36,0x6C,0x00,0x00,0x00,0x00,0x00,0x00, // &98 small tilde
	0xEA,0x4E,0x4A,0x4A,0x00,0x00,0x00,0x00, // &99 trade mark sign
	0x6C,0x38,0x7C,0xC0,0x78,0x0C,0xF8,0x00, // &9A small S caron
	0x00,0x30,0x18,0x0C,0x18,0x30,0x00,0x00, // &9B right angle quotation mark
	0x00,0x00,0x7E,0xDB,0xDF,0xD8,0x7F,0x00, // &9C small OE ligature
	0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0x00, // &9D double line (teletext)
	0x6C,0x38,0x7C,0x18,0x30,0x60,0x7C,0x00, // &9E small Z caron
	0xCC,0x00,0xCC,0xCC,0x78,0x30,0x30,0x00, // &9F capital Y diaeresis
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, //   
	0x18,0x00,0x18,0x18,0x18,0x18,0x18,0x00, //   
	0x00,0x18,0x7E,0xD8,0xD8,0x7E,0x18,0x00, //   
	0x38,0x6C,0x60,0xF0,0x60,0x66,0xFC,0x00, //   
	0x66,0x3C,0x66,0x3C,0x66,0x00,0x00,0x00, //   
	0xC3,0x66,0x3C,0x18,0x3C,0x18,0x18,0x00, //   
	0x18,0x18,0x18,0x00,0x18,0x18,0x18,0x00, //   
	0x3C,0x60,0x3C,0x66,0x66,0x3C,0x06,0x3C, //   
	0x66,0x00,0x00,0x00,0x00,0x00,0x00,0x00, //   
	0x7E,0x81,0x9D,0xB1,0xB1,0x9D,0x81,0x7E, //   
	0x3C,0x6C,0x6C,0x3E,0x00,0x7E,0x00,0x00, //   
	0x00,0x33,0x66,0xCC,0x66,0x33,0x00,0x00, //   
	0x00,0x7E,0x06,0x06,0x00,0x00,0x00,0x00, //   
	0x00,0x00,0x00,0x3C,0x00,0x00,0x00,0x00,
	0x7E,0x81,0xB9,0xA5,0xB9,0xA5,0x81,0x7E, //   
	0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00, //  
	0x3C,0x66,0x3C,0x00,0x00,0x00,0x00,0x00, //   
	0x18,0x18,0x7E,0x18,0x18,0x00,0x7E,0x00, //   
	0x70,0x18,0x30,0x60,0x78,0x00,0x00,0x00, //   
	0x78,0x0C,0x18,0x0C,0x78,0x00,0x00,0x00, //   
	0x0C,0x18,0x30,0x00,0x00,0x00,0x00,0x00, //   
	0x00,0x00,0x66,0x66,0x66,0x7C,0x60,0xC0, //   
	0x3E,0x7A,0x7A,0x3A,0x1A,0x1A,0x1A,0x00, //   
	0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00, //   
	0x00,0x00,0x00,0x00,0x00,0x08,0x18,0x00, //   
	0x30,0x70,0x30,0x30,0x30,0x00,0x00,0x00, //   
	0x38,0x6C,0x6C,0x38,0x00,0x7C,0x00,0x00, //   
	0x00,0xCC,0x66,0x33,0x66,0xCC,0x00,0x00, //   
	0x43,0xC6,0x4C,0x5A,0x36,0x6A,0xCF,0x02, //   
	0x40,0xC6,0x4C,0x5E,0x33,0x66,0xCC,0x0F, //   
	0xC0,0x23,0x66,0x2D,0xDB,0x35,0x67,0x01, //   
	0x18,0x00,0x18,0x30,0x60,0x66,0x3C,0x00, //   
	0x70,0x00,0x3C,0x66,0x7E,0x66,0x66,0x00, //   
	0x0E,0x00,0x3C,0x66,0x7E,0x66,0x66,0x00, //   
	0x18,0x66,0x00,0x3C,0x66,0x7E,0x66,0x00, //   
	0x76,0xDC,0x00,0x3C,0x66,0x7E,0x66,0x00, //   
	0x66,0x00,0x3C,0x66,0x7E,0x66,0x66,0x00, //   
	0x18,0x18,0x00,0x3C,0x66,0x7E,0x66,0x00, //   
	0x3F,0x6C,0xCC,0xFE,0xCC,0xCC,0xCF,0x00, //   
	0x3C,0x66,0x60,0x60,0x60,0x66,0x3C,0x18, //   
	0x70,0x00,0x3C,0x66,0x7E,0x66,0x66,0x00, //   
	0x0E,0x00,0x3C,0x66,0x7E,0x66,0x66,0x00, //   
	0x18,0x66,0x00,0xFE,0xF0,0xC0,0xFE,0x00, //   
	0x66,0x00,0x3C,0x66,0x7E,0x66,0x66,0x00, //   
	0x70,0x00,0x7E,0x18,0x18,0x18,0x7E,0x00, //   
	0x0E,0x00,0x7E,0x18,0x18,0x18,0x7E,0x00, //   
	0x18,0x66,0x00,0x7E,0x18,0x18,0x7E,0x00, //   
	0x66,0x00,0x7E,0x18,0x18,0x18,0x7E,0x00, //   
	0x78,0x6C,0x66,0xF6,0x66,0x6C,0x78,0x00, //   
	0x76,0xDC,0x00,0xC6,0xF6,0xDE,0xC6,0x00, //   
	0x70,0x00,0x7C,0xC6,0xC6,0xC6,0x7C,0x00, //   
	0x0E,0x00,0x7C,0xC6,0xC6,0xC6,0x7C,0x00, //   
	0x18,0x66,0x00,0x7C,0xC6,0xC6,0x7C,0x00, //   
	0x76,0xDC,0x00,0x7C,0xC6,0xC6,0x7C,0x00, //   
	0x66,0x00,0x7C,0xC6,0xC6,0xC6,0x7C,0x00, //   
	0x00,0xC6,0x6C,0x38,0x6C,0xC6,0x00,0x00, //   
	0x3E,0x66,0x6E,0x7E,0x76,0x66,0x7C,0x00, //   
	0x70,0x00,0xC6,0xC6,0xC6,0xC6,0x7C,0x00, //   
	0x0E,0x00,0xC6,0xC6,0xC6,0xC6,0x7C,0x00, //   
	0x18,0x66,0x00,0xC6,0xC6,0xC6,0x7C,0x00, //   
	0x66,0x00,0xC6,0xC6,0xC6,0xC6,0x7C,0x00, //   
	0x0E,0x00,0x66,0x66,0x3C,0x18,0x18,0x00, //   
	0xC0,0xC0,0xFC,0xC6,0xFC,0xC0,0xC0,0x00, //   
	0x3C,0x66,0x66,0x6C,0x66,0x66,0x6C,0x00, //   
	0x70,0x00,0x3C,0x06,0x3E,0x66,0x3E,0x00, //   
	0x0E,0x00,0x3C,0x06,0x3E,0x66,0x3E,0x00, //   
	0x18,0x66,0x00,0x3E,0x66,0xC6,0x7E,0x00, //   
	0x76,0xDC,0x00,0x3E,0x66,0xC6,0x7E,0x00, //   
	0x66,0x00,0x3C,0x06,0x3E,0x66,0x3E,0x00, //   
	0x18,0x18,0x00,0x3E,0x66,0xC6,0x7E,0x00, //   
	0x00,0x00,0x7E,0x1B,0x7F,0xD8,0x77,0x00, //   
	0x00,0x00,0x3C,0x60,0x60,0x60,0x3C,0x18, //   
	0x70,0x00,0x3C,0x66,0x7E,0x60,0x3C,0x00, //   
	0x0E,0x00,0x3C,0x66,0x7E,0x60,0x3C,0x00, //   
	0x18,0x66,0x00,0x3C,0x7E,0x60,0x3C,0x00, //   
	0x66,0x00,0x3C,0x66,0x7E,0x60,0x3C,0x00, //   
	0x70,0x00,0x38,0x18,0x18,0x18,0x3C,0x00, //   
	0x0E,0x00,0x38,0x18,0x18,0x18,0x3C,0x00, //   
	0x18,0x66,0x00,0x38,0x18,0x18,0x3C,0x00, //   
	0x66,0x00,0x38,0x18,0x18,0x18,0x3C,0x00, //   
	0x0C,0x3E,0x0C,0x7C,0xCC,0xCC,0x78,0x00, //   
	0x76,0xDC,0x00,0x7C,0x66,0x66,0x66,0x00, //   
	0x70,0x00,0x3C,0x66,0x66,0x66,0x3C,0x00, //   
	0x0E,0x00,0x3C,0x66,0x66,0x66,0x3C,0x00, //   
	0x18,0x66,0x00,0x3C,0x66,0x66,0x3C,0x00, //   
	0x76,0xDC,0x00,0x3C,0x66,0x66,0x3C,0x00, //   
	0x66,0x00,0x3C,0x66,0x66,0x66,0x3C,0x00, //   
	0x18,0x18,0x00,0x7E,0x00,0x18,0x18,0x00, //   
	0x00,0x02,0x7C,0xCE,0xD6,0xE6,0x7C,0x80, //   
	0x70,0x00,0x66,0x66,0x66,0x66,0x3E,0x00, //   
	0x0E,0x00,0x66,0x66,0x66,0x66,0x3E,0x00, //   
	0x18,0x66,0x00,0x66,0x66,0x66,0x3E,0x00, //   
	0x66,0x00,0x66,0x66,0x66,0x66,0x3E,0x00, //   
	0x0E,0x00,0x66,0x66,0x66,0x3E,0x06,0x3C, //   
	0x60,0x60,0x7C,0x66,0x66,0x7C,0x60,0x60, //   
	0x66,0x00,0x66,0x66,0x66,0x3E,0x06,0x3C } ; //   

unsigned short ttxtfont[256*20] = {
	0x0000,0x0011,0x0011,0x0019,0x0015,0x0013,0x0011,0x0011,0x8800,0x8800,0x8800,0x8800,0x8800,0x8800,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // NU
	0x0000,0x000E,0x0011,0x0011,0x001F,0x0011,0x0011,0x0011,0xF000,0x8800,0x8800,0xF000,0x8800,0x8800,0x8800,0x0000,0x0000,0x0000,0x0000,0x0000, // AR
	0x0000,0x000E,0x0011,0x0011,0x001F,0x0011,0x0011,0x0011,0x7000,0x8800,0x8000,0x9800,0x8800,0x8800,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // AG
	0x0000,0x000E,0x0011,0x0011,0x001F,0x0011,0x0011,0x0011,0x8800,0x8800,0x5000,0x2000,0x2000,0x2000,0x2000,0x0000,0x0000,0x0000,0x0000,0x0000, // AY
	0x0000,0x000E,0x0011,0x0011,0x001F,0x0011,0x0011,0x0011,0xF000,0x8800,0x8800,0xF000,0x8800,0x8800,0xF000,0x0000,0x0000,0x0000,0x0000,0x0000, // AB
	0x0000,0x000E,0x0011,0x0011,0x001F,0x0011,0x0011,0x0011,0x8800,0xD800,0xA800,0xA800,0x8800,0x8800,0x8800,0x0000,0x0000,0x0000,0x0000,0x0000, // AM
	0x0000,0x000E,0x0011,0x0011,0x001F,0x0011,0x0011,0x0011,0x7000,0x8800,0x8000,0x8000,0x8000,0x8800,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // AC
	0x0000,0x000E,0x0011,0x0011,0x001F,0x0011,0x0011,0x0011,0x8800,0x8800,0x8800,0xA800,0xA800,0xD800,0x8800,0x0000,0x0000,0x0000,0x0000,0x0000, // AW
	0x0000,0x001F,0x0010,0x0010,0x001E,0x0010,0x0010,0x0010,0x8000,0x8000,0x8000,0x8000,0x8000,0x8000,0xF800,0x0000,0x0000,0x0000,0x0000,0x0000, // FL
	0x0000,0x000E,0x0011,0x0010,0x000E,0x0001,0x0011,0x000E,0xF800,0x2000,0x2000,0x2000,0x2000,0x2000,0x2000,0x0000,0x0000,0x0000,0x0000,0x0000, // ST
	0x0000,0x001F,0x0010,0x0010,0x001E,0x0010,0x0010,0x001F,0xF000,0x8800,0x8800,0xF000,0x8800,0x8800,0xF000,0x0000,0x0000,0x0000,0x0000,0x0000, // EB
	0x0000,0x000E,0x0011,0x0010,0x000E,0x0001,0x0011,0x000E,0xF000,0x8800,0x8800,0xF000,0x8800,0x8800,0xF000,0x0000,0x0000,0x0000,0x0000,0x0000, // SB
	0x0000,0x0011,0x0011,0x0019,0x0015,0x0013,0x0011,0x0011,0x8800,0x8800,0x8800,0xF800,0x8800,0x8800,0x8800,0x0000,0x0000,0x0000,0x0000,0x0000, // NH
	0x0000,0x001C,0x0012,0x0011,0x0011,0x0011,0x0012,0x001C,0x8800,0x8800,0x8800,0xF800,0x8800,0x8800,0x8800,0x0000,0x0000,0x0000,0x0000,0x0000, // DH
	0x0000,0x000E,0x0011,0x0010,0x000E,0x0001,0x0011,0x000E,0x7000,0x8800,0x8800,0x8800,0x8800,0x8800,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // SO
	0x0000,0x000E,0x0011,0x0010,0x000E,0x0001,0x0011,0x000E,0x7000,0x2000,0x2000,0x2000,0x2000,0x2000,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // SI
	0x0000,0x001C,0x0012,0x0011,0x0011,0x0011,0x0012,0x001C,0x8000,0x8000,0x8000,0x8000,0x8000,0x8000,0xF800,0x0000,0x0000,0x0000,0x0000,0x0000, // DL
	0x0000,0x000E,0x0011,0x0010,0x0013,0x0011,0x0011,0x000E,0xF000,0x8800,0x8800,0xF000,0x8800,0x8800,0x8800,0x0000,0x0000,0x0000,0x0000,0x0000, // GR
	0x0000,0x000E,0x0011,0x0010,0x0013,0x0011,0x0011,0x000E,0x7000,0x8800,0x8000,0x9800,0x8800,0x8800,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // GG
	0x0000,0x000E,0x0011,0x0010,0x0013,0x0011,0x0011,0x000E,0x8800,0x8800,0x5000,0x2000,0x2000,0x2000,0x2000,0x0000,0x0000,0x0000,0x0000,0x0000, // GY
	0x0000,0x000E,0x0011,0x0010,0x0013,0x0011,0x0011,0x000E,0xF000,0x8800,0x8800,0xF000,0x8800,0x8800,0xF000,0x0000,0x0000,0x0000,0x0000,0x0000, // GB
	0x0000,0x000E,0x0011,0x0010,0x0013,0x0011,0x0011,0x000E,0x8800,0xD800,0xA800,0xA800,0x8800,0x8800,0x8800,0x0000,0x0000,0x0000,0x0000,0x0000, // GM
	0x0000,0x000E,0x0011,0x0010,0x0013,0x0011,0x0011,0x000E,0x7000,0x8800,0x8000,0x8000,0x8000,0x8800,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // GC
	0x0000,0x000E,0x0011,0x0010,0x0013,0x0011,0x0011,0x000E,0x8800,0x8800,0x8800,0xA800,0xA800,0xD800,0x8800,0x0000,0x0000,0x0000,0x0000,0x0000, // GW
	0x0000,0x000E,0x0011,0x0010,0x0010,0x0010,0x0011,0x000E,0xE000,0x9000,0x8800,0x8800,0x8800,0x9000,0xE000,0x0000,0x0000,0x0000,0x0000,0x0000, // CD
	0x0000,0x000E,0x0011,0x0010,0x0010,0x0010,0x0011,0x000E,0x7000,0x8800,0x8000,0x9800,0x8800,0x8800,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // CG
	0x0000,0x000E,0x0011,0x0010,0x000E,0x0001,0x0011,0x000E,0x7000,0x8800,0x8000,0x9800,0x8800,0x8800,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // SG
	0x0000,0x001F,0x0010,0x0010,0x001E,0x0010,0x0010,0x001F,0x7000,0x8800,0x8000,0x7000,0x0800,0x8800,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // ES
	0x0000,0x001E,0x0011,0x0011,0x001E,0x0011,0x0011,0x001E,0xF000,0x8800,0x8800,0xF000,0x8800,0x8800,0xF000,0x0000,0x0000,0x0000,0x0000,0x0000, // BB
	0x0000,0x0011,0x0011,0x0019,0x0015,0x0013,0x0011,0x0011,0xF000,0x8800,0x8800,0xF000,0x8800,0x8800,0xF000,0x0000,0x0000,0x0000,0x0000,0x0000, // NB
	0x0000,0x0011,0x0011,0x0011,0x001F,0x0011,0x0011,0x0011,0x7000,0x8800,0x8000,0x9800,0x8800,0x8800,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // HG
	0x0000,0x001E,0x0011,0x0011,0x001E,0x0011,0x0011,0x0011,0x7000,0x8800,0x8000,0x9800,0x8800,0x8800,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000, // RG
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // "!"
	0x0000,0x700E,0x700E,0x700E,0x700E,0x700E,0x700E,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // """
	0x0000,0x700E,0x700E,0x700E,0x700E,0xFC3F,0xFC3F,0x700E,0x700E,0xFC3F,0xFC3F,0x700E,0x700E,0x700E,0x700E,0x0000,0x0000,0x0000,0x0000,0x0000, // "#"
	0x0000,0xF00F,0xF81F,0x9C39,0x8C31,0x8031,0x8039,0xF01F,0xF80F,0x9C01,0x8C01,0x8C31,0x9C39,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "$"
	0x0000,0x003C,0x0C3C,0x1C3C,0x383C,0x7000,0xE000,0xC001,0x8003,0x0007,0x000E,0x3C1C,0x3C38,0x3C30,0x3C00,0x0000,0x0000,0x0000,0x0000,0x0000, // "%"
	0x0000,0x000E,0x001F,0x803B,0x8031,0x803B,0x001F,0x000E,0x001F,0x8C3B,0xDC31,0xF830,0xF838,0xDC1F,0x8C0F,0x0000,0x0000,0x0000,0x0000,0x0000, // "&"
	0x0000,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // "'"
	0x0000,0x6000,0xE000,0xC001,0x8003,0x0007,0x0006,0x0006,0x0006,0x0006,0x0007,0x8003,0xC001,0xE000,0x6000,0x0000,0x0000,0x0000,0x0000,0x0000, // "("
	0x0000,0x0006,0x0007,0x8003,0xC001,0xE000,0x6000,0x6000,0x6000,0x6000,0xE000,0xC001,0x8003,0x0007,0x0006,0x0000,0x0000,0x0000,0x0000,0x0000, // ")"
	0x0000,0x8001,0x8001,0x8C31,0x9C39,0xF81F,0xF00F,0x8001,0x8001,0xF00F,0xF81F,0x9C39,0x8C31,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // "*"
	0x0000,0x0000,0x0000,0x8001,0x8001,0x8001,0x8001,0xF81F,0xF81F,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // "+"
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x8001,0x8001,0x8001,0x8003,0x0007,0x0006,0x0000,0x0000,0x0000, // ","
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // "-"
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // "."
	0x0000,0x0000,0x0C00,0x1C00,0x3800,0x7000,0xE000,0xC001,0x8003,0x0007,0x000E,0x001C,0x0038,0x0030,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // "/"
	0x0000,0xC003,0xE007,0x700E,0x381C,0x1C38,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0x381C,0x700E,0xE007,0xC003,0x0000,0x0000,0x0000,0x0000,0x0000, // "0"
	0x0000,0x8001,0x8001,0x800F,0x800F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "1"
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C00,0x1C00,0xF803,0xF007,0x000E,0x001C,0x0038,0x0030,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // "2"
	0x0000,0xFC3F,0xFC3F,0x1C00,0x3800,0x7000,0xE000,0xF801,0xFC01,0x0C00,0x0C00,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "3"
	0x0000,0x7000,0xF000,0xF001,0xB003,0x3007,0x300E,0x301C,0x3038,0xFC3F,0xFC3F,0x3000,0x3000,0x3000,0x3000,0x0000,0x0000,0x0000,0x0000,0x0000, // "4"
	0x0000,0xFC3F,0xFC3F,0x0030,0x0030,0xF03F,0xF83F,0x1C00,0x0C00,0x0C00,0x0C00,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "5"
	0x0000,0xF001,0xF007,0x000F,0x001C,0x0038,0x0030,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "6"
	0x0000,0xFC3F,0xFC3F,0x0C00,0x1C00,0x3800,0x7000,0xE000,0xC001,0x8003,0x0007,0x0006,0x0006,0x0006,0x0006,0x0000,0x0000,0x0000,0x0000,0x0000, // "7"
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x1C38,0xF81F,0xF81F,0x1C38,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "8"
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0C00,0x1C00,0x3800,0xF000,0xE00F,0x800F,0x0000,0x0000,0x0000,0x0000,0x0000, // "9"
	0x0000,0x0000,0x0000,0x0000,0x0000,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // ":"
	0x0000,0x0000,0x0000,0x0000,0x0000,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x8001,0x8001,0x8001,0x8003,0x0007,0x0006,0x0000,0x0000,0x0000, // ";"
	0x0000,0x6000,0xE000,0xC001,0x8003,0x0007,0x000E,0x001C,0x001C,0x000E,0x0007,0x8003,0xC001,0xE000,0x6000,0x0000,0x0000,0x0000,0x0000,0x0000, // "<"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xFC3F,0xFC3F,0x0000,0x0000,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // "="
	0x0000,0x0006,0x0007,0x8003,0xC001,0xE000,0x7000,0x3800,0x3800,0x7000,0xE000,0xC001,0x8003,0x0007,0x0006,0x0000,0x0000,0x0000,0x0000,0x0000, // ">"
	0x0000,0xE00F,0xF01F,0x3838,0x1830,0x3800,0x7000,0xE000,0xC001,0x8001,0x8001,0x0000,0x0000,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // "?"
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xFC31,0xFC31,0x8C31,0x8C31,0xFC31,0xFC31,0x0030,0x0038,0xF01F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "@"
	0x0000,0x8001,0xC003,0xE007,0x700E,0x381C,0x1C38,0x0C30,0x0C30,0xFC3F,0xFC3F,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // "A"
	0x0000,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x1C30,0xF83F,0xF83F,0x1C30,0x0C30,0x0C30,0x1C30,0xF83F,0xF03F,0x0000,0x0000,0x0000,0x0000,0x0000, // "B"
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "C"
	0x0000,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x1C30,0xF83F,0xF03F,0x0000,0x0000,0x0000,0x0000,0x0000, // "D"
	0x0000,0xFC3F,0xFC3F,0x0030,0x0030,0x0030,0x0030,0xF03F,0xF03F,0x0030,0x0030,0x0030,0x0030,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // "E"
	0x0000,0xFC3F,0xFC3F,0x0030,0x0030,0x0030,0x0030,0xF03F,0xF03F,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0000,0x0000,0x0000,0x0000,0x0000, // "F"
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0030,0x0030,0x0030,0x0030,0x3C30,0x3C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // "G"
	0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0xFC3F,0xFC3F,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // "H"
	0x0000,0xF00F,0xF00F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "I"
	0x0000,0x0C00,0x0C00,0x0C00,0x0C00,0x0C00,0x0C00,0x0C00,0x0C00,0x0C00,0x0C00,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "J"
	0x0000,0x1830,0x3830,0x7030,0xE030,0xC031,0x8033,0x003F,0x003F,0x8033,0xC031,0xE030,0x7030,0x3830,0x1830,0x0000,0x0000,0x0000,0x0000,0x0000, // "K"
	0x0000,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // "L"
	0x0000,0x0C30,0x0C30,0x1C38,0x3C3C,0x7C3E,0xEC37,0xCC33,0x8C31,0x8C31,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // "M"
	0x0000,0x0C30,0x0C30,0x0C38,0x0C3C,0x0C3E,0x0C37,0x8C33,0xCC31,0xEC30,0x7C30,0x3C30,0x1C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // "N"
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "O"
	0x0000,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x1C30,0xF83F,0xF03F,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0000,0x0000,0x0000,0x0000,0x0000, // "P"
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0xCC30,0xEC30,0x7830,0x3838,0xFC1F,0xCC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // "Q"
	0x0000,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x1C30,0xF83F,0xF03F,0xC031,0xE030,0x7030,0x3830,0x1C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // "R"
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0030,0x0038,0xF01F,0xF80F,0x1C00,0x0C00,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "S"
	0x0000,0xFC3F,0xFC3F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // "T"
	0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "U"
	0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0x1818,0x381C,0x300C,0x700E,0xE007,0xC003,0xC003,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // "V"
	0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x8C31,0x8C31,0x8C31,0x8C31,0x8C31,0xFC3F,0xF81F,0x700E,0x0000,0x0000,0x0000,0x0000,0x0000, // "W"
	0x0000,0x0C30,0x0C30,0x1C38,0x381C,0x700E,0xE007,0xC003,0xC003,0xE007,0x700E,0x381C,0x1C38,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // "X"
	0x0000,0x0C30,0x0C30,0x1C38,0x381C,0x700E,0xE007,0xC003,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // "Y"
	0x0000,0xFC3F,0xFC3F,0x1C00,0x3800,0x7000,0xE000,0xC001,0x8003,0x0007,0x000E,0x001C,0x0038,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // "Z"
	0x0000,0xE007,0xE007,0x0006,0x0006,0x0006,0x0006,0x0006,0x0006,0x0006,0x0006,0x0006,0x0006,0xE007,0xE007,0x0000,0x0000,0x0000,0x0000,0x0000, // "["
	0x0000,0x0000,0x0030,0x0038,0x001C,0x000E,0x0007,0x8003,0xC001,0xE000,0x7000,0x3800,0x1C00,0x0C00,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // "\"
	0x0000,0xE007,0xE007,0x6000,0x6000,0x6000,0x6000,0x6000,0x6000,0x6000,0x6000,0x6000,0x6000,0xE007,0xE007,0x0000,0x0000,0x0000,0x0000,0x0000, // "]"
	0x0000,0x8001,0xC003,0xE007,0x700E,0x381C,0x1818,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // "^"
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xFFFF,0xFFFF,0x0000,0x0000, // "_"
	0x0000,0x0006,0x0007,0x8003,0xC001,0xE000,0x6000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // "`"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xF00F,0xF80F,0x1C00,0x0C00,0xFC1F,0xFC3F,0x0C30,0x0C30,0xFC3F,0xFC1F,0x0000,0x0000,0x0000,0x0000,0x0000, // "a"
	0x0000,0x0030,0x0030,0x0030,0x0030,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x0C30,0x0C30,0x1C30,0xF83F,0xF03F,0x0000,0x0000,0x0000,0x0000,0x0000, // "b"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xFC0F,0xFC1F,0x0038,0x0030,0x0030,0x0030,0x0030,0x0038,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // "c"
	0x0000,0x0C00,0x0C00,0x0C00,0x0C00,0xFC0F,0xFC1F,0x0C38,0x0C30,0x0C30,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // "d"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xFC3F,0xFC3F,0x0030,0x0038,0xF01F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "e"
	0x0000,0x3000,0xF000,0xC001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // "f"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xFC0F,0xFC1F,0x0C38,0x0C30,0x0C30,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0C00,0x1C00,0xF80F,0xF00F,0x0000, // "g"
	0x0000,0x0030,0x0030,0x0030,0x0030,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // "h"
	0x0000,0x8001,0x8001,0x0000,0x0000,0x800F,0x800F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "i"
	0x0000,0x8001,0x8001,0x0000,0x0000,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8003,0x000F,0x000C,0x0000, // "j"
	0x0000,0x000C,0x000C,0x000C,0x180C,0x380C,0x700C,0xE00C,0xC00F,0x800F,0xC00D,0xE00C,0x700C,0x380C,0x180C,0x0000,0x0000,0x0000,0x0000,0x0000, // "k"
	0x0000,0x800F,0x800F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "l"
	0x0000,0x0000,0x0000,0x0000,0x0000,0x303E,0xF83F,0xFC33,0x8C31,0x8C31,0x8C31,0x8C31,0x8C31,0x8C31,0x8C31,0x0000,0x0000,0x0000,0x0000,0x0000, // "m"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // "n"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // "o"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x0C30,0x0C30,0x1C30,0xF83F,0xF03F,0x0030,0x0030,0x0030,0x0030,0x0000, // "p"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xFC0F,0xFC1F,0x0C38,0x0C30,0x0C30,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0C00,0x0C00,0x0C00,0x0C00,0x0000, // "q"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xF819,0xF81B,0x001F,0x001E,0x0018,0x0018,0x0018,0x0018,0x0018,0x0018,0x0000,0x0000,0x0000,0x0000,0x0000, // "r"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xFC0F,0xFC1F,0x0030,0x0030,0xF01F,0xF80F,0x0C00,0x0C00,0xF83F,0xF03F,0x0000,0x0000,0x0000,0x0000,0x0000, // "s"
	0x0000,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x8001,0x8001,0x8001,0x8001,0x8001,0xC001,0xF000,0x3000,0x0000,0x0000,0x0000,0x0000,0x0000, // "t"
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // "u"
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0C30,0x0C30,0x0C30,0x1C38,0x1818,0x381C,0x700E,0xE007,0xC003,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // "v"
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x8C31,0x8C31,0x8C31,0xFC3F,0xF81F,0x700E,0x0000,0x0000,0x0000,0x0000,0x0000, // "w"
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0C30,0x1C38,0x781E,0xF00F,0xC003,0xC003,0xF00F,0x781E,0x1C38,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // "x"
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0C00,0x1C00,0xF80F,0xF00F,0x0000, // "y"
	0x0000,0x0000,0x0000,0x0000,0x0000,0xFC3F,0xFC3F,0x3800,0x7000,0xE001,0x8007,0x000E,0x001C,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // "z"
	0x0000,0xF000,0xF001,0x8001,0x8001,0x8003,0x0003,0x000F,0x000F,0x0003,0x8003,0x8001,0x8001,0xF000,0xF000,0x0000,0x0000,0x0000,0x0000,0x0000, // "{"
	0x0000,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // "|"
	0x0000,0x000F,0x800F,0x8001,0x8001,0xC001,0xC000,0xF000,0xF000,0xC000,0xC001,0x8001,0x8001,0x000F,0x000F,0x0000,0x0000,0x0000,0x0000,0x0000, // "}"
	0x0000,0x000E,0x0C1F,0xDC3B,0xF831,0x7000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // "~"
	0x0000,0x8001,0xC003,0xC003,0x6006,0x6006,0x300C,0x300C,0x1818,0x1818,0x0C30,0x0C30,0x0C30,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // &7F DEL
	0x0000,0xF003,0xF807,0x1C0E,0x0C0C,0xC03F,0xC03F,0x000C,0x000C,0x803F,0x803F,0x0C0C,0x1C0E,0xF807,0xF003,0x0000,0x0000,0x0000,0x0000,0x0000, // &80 euro symbol
	0x0000,0xFC3F,0xFC3F,0xFC3F,0xFC3F,0xFC3F,0xFC3F,0xFC3F,0xFC3F,0xFC3F,0xFC3F,0xFC3F,0xFC3F,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // &81 block (teletext)
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x8003,0x8003,0x8001,0x8003,0x0007,0x000E,0x0000,0x0000,0x0000, // &82 single low quotation mark
	0x0000,0x7000,0xF000,0xC001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8003,0x000F,0x000E,0x0000, // &83 small letter f with hook
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x1C0E,0x1C0E,0x0C06,0x1C0E,0x381C,0x7038,0x0000,0x0000,0x0000, // &84 double low quotation mark
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x9819,0x9819,0x0000,0x0000,0x0000,0x0000,0x0000, // &85 horizontal ellipsis
	0x0000,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // &86 dagger
	0x0000,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // &87 double dagger
	0x8001,0xC003,0x6006,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &88 circumflex accent
	0x0000,0x3038,0x706C,0xE06C,0xC039,0x8003,0x0007,0x000E,0x001C,0x0038,0x0070,0x1C67,0xB60D,0xB60D,0x1C07,0x0000,0x0000,0x0000,0x0000,0x0000, // &89 per mille
	0x6006,0xE007,0xC003,0xF00F,0xF81F,0x1C38,0x0C30,0x0038,0xF01F,0xF80F,0x1C00,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // &8A capital S caron
	0x0000,0x0000,0x0000,0x8001,0x8003,0x0007,0x000E,0x000C,0x000E,0x0007,0x8003,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &8B left angle quotation mark
	0x0000,0xFC0F,0xFC1F,0x8039,0x8031,0x8031,0x8031,0xF831,0xF831,0x8031,0x8031,0x8031,0x8039,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // &8C capital OE ligature
	0x0000,0x0000,0x8001,0xC003,0xE007,0xF00F,0xB81D,0x9C39,0x8C31,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &8D up arrow (teletext)
	0x6006,0xE007,0xC003,0xFC3F,0xFC3F,0x3800,0x7000,0xE000,0xC001,0x8003,0x0007,0x000E,0x001C,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // &8E capital Z caron
	0x0000,0x0000,0x8001,0x8003,0x0007,0x000E,0x001C,0xFC3F,0xFC3F,0x001C,0x000E,0x0007,0x8003,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &8F left arrow (teletext)
	0x0000,0x0000,0x8001,0xC001,0xE000,0x7000,0x3800,0xFC3F,0xFC3F,0x3800,0x7000,0xE000,0xC001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &90 right arrow (teletext)
	0x0000,0x7000,0xE000,0xC001,0x8001,0xC001,0xC001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &91 left single quotation mark
	0x0000,0x8003,0x8003,0x8001,0x8003,0x0007,0x000E,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &92 right single quotation mark
	0x0000,0x1C0E,0x381C,0x7038,0x6030,0x7038,0x7038,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &93 left double quotation mark
	0x0000,0x1C0E,0x1C0E,0x0C06,0x1C0E,0x381C,0x7038,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &94 right double quotation mark
	0x0000,0x0000,0x0000,0x0000,0x0000,0x8001,0xC003,0xE007,0xE007,0xC003,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &95 bullet
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xF81F,0xF81F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &96 en dash
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xFE7F,0xFE7F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &97 em dash
	0x6007,0xC00D,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &98 small tilde
	0x0000,0x0000,0xC67E,0xC67E,0xEE18,0xFE18,0xD618,0xD618,0xC618,0xC618,0xC618,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &99 trade mark sign
	0x0000,0x6006,0xE007,0xC003,0x0000,0xFC0F,0xFC1F,0x0038,0x0038,0xF01F,0xF80F,0x1C00,0x1C00,0xF83F,0xF03F,0x0000,0x0000,0x0000,0x0000,0x0000, // &9A small S caron
	0x0000,0x0000,0x0000,0x8001,0xC001,0xE000,0x7000,0x3000,0x7000,0xE000,0xC001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &9B right angle quotation mark
	0x0000,0x0000,0x0000,0x0000,0x0000,0x780E,0xFC1F,0xCC3B,0x8C31,0xFC31,0xFC31,0x8031,0xC03B,0xFC1F,0x7C0E,0x0000,0x0000,0x0000,0x0000,0x0000, // &9C small OE ligature
	0x0000,0x381C,0x381C,0x381C,0x381C,0x381C,0x381C,0x381C,0x381C,0x381C,0x381C,0x381C,0x381C,0x381C,0x381C,0x0000,0x0000,0x0000,0x0000,0x0000, // &9D double line (teletext)
	0x0000,0x6006,0xE007,0xC003,0x0000,0xFC3F,0xFC3F,0x3800,0x7000,0xE001,0x8007,0x000E,0x001C,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // &9E small Z caron
	0x300C,0x300C,0x300C,0x0000,0x0C30,0x1C38,0x381C,0x700E,0xE007,0xC003,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // &9F capital Y diaeresis
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // &A0 non-break space
	0x0000,0x0000,0x0000,0x0000,0x0000,0x8001,0x8001,0x0000,0x0000,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000, // " "
	0x0000,0x0000,0x0000,0x8001,0x8001,0xFC0F,0xFC1F,0x8039,0x8031,0x8031,0x8031,0x8031,0x8039,0xFC1F,0xFC0F,0x8001,0x8001,0x0000,0x0000,0x0000, // " "
	0x0000,0xF003,0xF807,0x1C0E,0x0C0C,0x000C,0x000C,0x803F,0x803F,0x000C,0x000C,0x000C,0x000C,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0C30,0x1C38,0xF81F,0xF00F,0x700E,0x700E,0xF00F,0xF81F,0x1C38,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0C30,0x1C38,0x381C,0x700E,0xFC3F,0xFC3F,0x8001,0x8001,0xFC3F,0xFC3F,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0030,0x0038,0xF01F,0xF81F,0x1C38,0x1C38,0xF81F,0xF80F,0x1C00,0x0C00,0x0C30,0x1C38,0xF81F,0xF00F,0x0000, // " "
	0x1818,0x1818,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xCC33,0xEC37,0x6C36,0x0C36,0x0C36,0x0C36,0x6C36,0xEC37,0xCC33,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000, // " "
	0x0000,0xF00F,0xF80F,0x1C00,0x0C00,0xFC0F,0xFC1F,0x0C38,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x1C0E,0x381C,0x7038,0x7038,0x381C,0x1C0E,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0xFC3F,0xFC3F,0x0C00,0x0C00,0x0C00,0x0C00,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xE007,0xE007,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xCC37,0xEC37,0x6C36,0x6C36,0xCC37,0x8C37,0xCC37,0xEC36,0x6C36,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000, // " "
	0x0000,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0xC003,0xE007,0x700E,0x381C,0x700E,0xE007,0xC003,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x8001,0x8001,0x8001,0x8001,0xFC3F,0xFC3F,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0xC00F,0xE00F,0x7000,0x7000,0xE003,0xC007,0x000E,0x000C,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0xC00F,0xE00F,0x7000,0x7000,0xE00F,0xE00F,0x7000,0x7000,0xE00F,0xC00F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x6000,0xE000,0xC001,0x8003,0x0007,0x0006,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0x6030,0x6030,0x6030,0x6030,0x6030,0x6030,0x6030,0xF830,0xFC3F,0x8C3F,0x0030,0x0030,0x0030,0x0030,0x0000, // " "
	0x0000,0x8C0F,0x8C1F,0x8C39,0x8C31,0x8C31,0x8C39,0x8C1F,0x8C0F,0x8C01,0x8C01,0x8C01,0x8C01,0x8C01,0x8C01,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x8001,0x8001,0x8003,0x0007,0x0006,0x0000, // " "
	0x0000,0x8003,0x8007,0x800F,0x800D,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x7038,0x381C,0x1C0E,0x1C0E,0x381C,0x7038,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x000C,0x000C,0x000C,0x000C,0x000C,0x000C,0x000C,0x000C,0x1C0C,0x3C0C,0x7C00,0xEC00,0xCC01,0x8C01,0xFC01,0xFC01,0x0C00,0x0C00,0x0000, // " "
	0x0000,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0xF831,0xFC31,0x0C00,0x0C00,0x7C00,0xF800,0xC001,0x8001,0xFC01,0xFC01,0x0000, // " "
	0x0000,0x003E,0x003F,0x8003,0x8003,0x003F,0x003F,0x8003,0x8003,0x1C3F,0x3C3E,0x7C00,0xEC00,0xCC01,0x8C01,0xFC01,0xFC01,0x0C00,0x0C00,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0x8001,0x8001,0x0000,0x0000,0x8001,0x8001,0x8003,0x0007,0x000E,0x001C,0x0C18,0x1C1C,0xF80F,0xF00F,0x0000, // " "
	0x0006,0x0007,0x8003,0x8001,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xFC3F,0xFC3F,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xFC3F,0xFC3F,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0xE007,0x6006,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xFC3F,0xFC3F,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x1807,0xB80F,0xF01D,0xE018,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xFC3F,0xFC3F,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x300C,0x300C,0x300C,0x0000,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xFC3F,0xFC3F,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0x6006,0x6006,0xC003,0xF00F,0xF81F,0x1C38,0x0C30,0xFC3F,0xFC3F,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0xFC0F,0xFC1F,0x8039,0x8031,0x8031,0x8031,0xF83F,0xF83F,0x8031,0x8031,0x8031,0x8031,0xFC31,0xFC31,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0030,0x0030,0x0030,0x0030,0x0030,0x0030,0x0C30,0x1C38,0xF81F,0xF00F,0x8001,0x8003,0x0007,0x0006,0x0000, // " "
	0x0006,0x0007,0x8003,0x8001,0x0000,0xFC3F,0xFC3F,0x0030,0x0030,0xF03F,0xF03F,0x0030,0x0030,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0000,0xFC3F,0xFC3F,0x0030,0x0030,0xF03F,0xF03F,0x0030,0x0030,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0xE007,0x6006,0x0000,0xFC3F,0xFC3F,0x0030,0x0030,0xF03F,0xF03F,0x0030,0x0030,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x300C,0x300C,0x300C,0x0000,0x0000,0xFC3F,0xFC3F,0x0030,0x0030,0xF03F,0xF03F,0x0030,0x0030,0xFC3F,0xFC3F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0006,0x0007,0x8003,0x8001,0x0000,0xF00F,0xF00F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0000,0xF00F,0xF00F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0xE007,0x6006,0x0000,0xF00F,0xF00F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x300C,0x300C,0x300C,0x0000,0x0000,0xF00F,0xF00F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0xF00F,0xF80F,0x1C0C,0x0C0C,0x0C0C,0x0C0C,0x8C3F,0x8C3F,0x0C0C,0x0C0C,0x0C0C,0x1C0C,0xF80F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x1807,0xB80F,0xF01D,0xE018,0x0000,0x0C30,0x0C3C,0x0C3E,0x0C37,0x8C33,0xCC31,0xEC30,0x7C30,0x3C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0006,0x0007,0x8003,0x8001,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0xE007,0x6006,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x1807,0xB80F,0xF01D,0xE018,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x300C,0x300C,0x300C,0x0000,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x1818,0x381C,0x700E,0xE007,0xC003,0xE007,0x700E,0x381C,0x1818,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0xF00F,0xF81F,0x1C38,0x3C30,0x7C30,0xEC30,0xCC31,0x8C33,0x0C37,0x0C3E,0x0C3C,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0006,0x0007,0x8003,0x8001,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0xE007,0x6006,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x300C,0x300C,0x300C,0x0000,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0C30,0x1C38,0x381C,0x700E,0xE007,0xC003,0x8001,0x8001,0x8001,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0030,0x0030,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x1C30,0xF83F,0xF03F,0x0030,0x0030,0x0030,0x0030,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0xC00F,0xE01F,0x7038,0x3030,0x3030,0x3030,0xF031,0xF831,0x1C30,0x0C30,0x0C30,0x1C30,0xF831,0xF031,0x0030,0x0030,0x0030,0x0030,0x0000, // " "
	0x0006,0x0007,0x8003,0x8001,0x0000,0xF00F,0xF80F,0x1C00,0x0C00,0xFC0F,0xFC1F,0x0C38,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0000,0xF00F,0xF80F,0x1C00,0x0C00,0xFC0F,0xFC1F,0x0C38,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0xE007,0x6006,0x0000,0xF00F,0xF80F,0x1C00,0x0C00,0xFC0F,0xFC1F,0x0C38,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x1807,0xB80F,0xF01D,0xE018,0x0000,0xF00F,0xF80F,0x1C00,0x0C00,0xFC0F,0xFC1F,0x0C38,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x300C,0x300C,0x300C,0x0000,0xF00F,0xF80F,0x1C00,0x0C00,0xFC0F,0xFC1F,0x0C38,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0x6006,0xC003,0x8001,0xF00F,0xF80F,0x1C00,0x0C00,0xFC0F,0xFC1F,0x0C38,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0x781E,0xFC1F,0x8C03,0x8C01,0xFC1F,0xFC3F,0x8031,0x8031,0xF83F,0xF81F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0xFC0F,0xFC1F,0x0038,0x0030,0x0030,0x0030,0x0030,0x0038,0xFC1F,0xFC0F,0xC000,0xC001,0x8003,0x0003,0x0000, // " "
	0x0006,0x0007,0x8003,0x8001,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xFC3F,0xFC3F,0x0030,0x0038,0xF01F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xFC3F,0xFC3F,0x0030,0x0038,0xF01F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0xE007,0x6006,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xFC3F,0xFC3F,0x0030,0x0038,0xF01F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x300C,0x300C,0x300C,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0xFC3F,0xFC3F,0x0030,0x0038,0xF01F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0006,0x0007,0x8003,0x8001,0x0000,0x800F,0x800F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0000,0x800F,0x800F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0xE007,0x6006,0x0000,0x800F,0x800F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x300C,0x300C,0x300C,0x0000,0x800F,0x800F,0x8001,0x8001,0x8001,0x8001,0x8001,0x8001,0xF00F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x601E,0xE01F,0xC003,0xE007,0x700E,0x380C,0x1C00,0x0C00,0xFC0F,0xFC1F,0x1C38,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x1807,0xB80F,0xF01D,0xE018,0x0000,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0006,0x0007,0x8003,0x8001,0x0000,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0000,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0xE007,0x6006,0x0000,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x1807,0xB80F,0xF01D,0xE018,0x0000,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x300C,0x300C,0x300C,0x0000,0x0000,0xF00F,0xF81F,0x1C38,0x0C30,0x0C30,0x0C30,0x1C38,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x8001,0x8001,0x0000,0x0000,0xFC3F,0xFC3F,0x0000,0x0000,0x8001,0x8001,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x0000,0x0000,0x0000,0x0000,0xF00F,0xF81F,0x7C38,0xEC30,0xCC31,0x8C33,0x0C37,0x1C3E,0xF81F,0xF00F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0006,0x0007,0x8003,0x8001,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x8001,0xC003,0xE007,0x6006,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x0000,0x300C,0x300C,0x300C,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0000,0x0000,0x0000,0x0000,0x0000, // " "
	0x6000,0xE000,0xC001,0x8001,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0C00,0x1C00,0xF80F,0xF00F,0x0000, // " "
	0x0000,0x0030,0x0030,0x0030,0x0030,0xF03F,0xF83F,0x1C30,0x0C30,0x0C30,0x0C30,0x0C30,0x1C30,0xF83F,0xF03F,0x0030,0x0030,0x0030,0x0030,0x0000, // " "
	0x0000,0x300C,0x300C,0x300C,0x0000,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C30,0x0C38,0xFC1F,0xFC0F,0x0C00,0x1C00,0xF80F,0xF00F,0x0000} ; // " "
