SymbolicExpression
====================

A **SymbolicExpression** represents a data value or instruction
operand which should be interpreted as referring to a symbol.


Guaranteed Properties
---------------------

- A **SymbolicExpression** must be one of the following, implemented
  as either an inheritance hierarchy or a tagged union.
  - [SymStackConst](SymStackConst.md)
  - [SymAddrConst](SymAddrConst.md)
  - [SymAddrAddr](SymAddrAddr.md)

Symbolic Expression Attributes
------------------------------

Symbolic expressions have a set of assignable labels (**SymAttribute**) for
architecture and ABI specific relocation types and reference modifiers.

**SymAttribute** labels are organized by architecture with common labels,
stems, and suffixes grouped into a common base set.

For example, a symbol of the form `symbol@GOT` should be assigned the attribute
**gtirb::SymAttribute::GOT**, and a symbol of the form `symbol@GOTPAGE` assigned
both **gtirb::SymAttribute::GOT** and **gtirb::SymAttribute::PAGE**.

Note that we do not generalize or otherwise unify relocation attributes across
architectures and instead prefer an explicit mapping of attribute names to
relocation types as they appear in the assembly for each architecture. As a
result, many labels have similar names (e.g.  `HI`, `HI16`, `H`), but are
duplicated to provide an intuitive mapping.


 | Attribute  | Description                                                                                                                                                                                                                                                                                                                                                                                 |
 |------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
 | **ELF**    |                                                                                                                                                                                                                                                                                                                                                                                             |
 | GOT        | Expression value is the offset of the GOT entry for a symbol from the base of the GOT.                                                                                                                                                                                                                                                                                                      |
 | GOTPC      | Expression value is the PC relative offset to GOT. Used for values referencing an implicit `_GLOBAL_OFFSET_TABLE_` symbol.                                                                                                                                                                                                                                                                  |
 | GOTOFF     | Expression value is the offset to a symbol location from the base of the GOT.                                                                                                                                                                                                                                                                                                               |
 | GOTREL     | Expression value is relative to the GOT.                                                                                                                                                                                                                                                                                                                                                    |
 | PLT        | Expression value is the offset to the PLT entry for a symbol from the current code location.                                                                                                                                                                                                                                                                                                |
 | PLTOFF     | Expression value is the offset to the PLT entry for a symbol from the base of the GOT.                                                                                                                                                                                                                                                                                                      |
 | PCREL      | Expression value is relative to the current code location.                                                                                                                                                                                                                                                                                                                                  |
 | SECREL     | Expression value is relative to the beginning of the section.                                                                                                                                                                                                                                                                                                                               |
 | TLS        | Expression value references Thead Local Storage.                                                                                                                                                                                                                                                                                                                                            |
 | TLSGD      | Expression value is the offset of the GOT entry for a `tls_index` structure. Used in Global Dynamic TLS access model (32 and 64-bit). See [ELF Handling for Thread-Local Storage](https://www.uclibc.org/docs/tls.pdf) for more details.                                                                                                                                                    |
 | TLSLD      | Expression value is the offset of the GOT entry for a `tls_index` structure. Used in Local Dynamic TLS access model (64-bit). See [ELF Handling for Thread-Local Storage](https://www.uclibc.org/docs/tls.pdf) for more details.                                                                                                                                                            |
 | TLSLDM     | Expression value is the offset of the GOT entry for a `tls_index` structure. Used in Local Dynamic TLS access model (32-bit). See [ELF Handling for Thread-Local Storage](https://www.uclibc.org/docs/tls.pdf) for more details.                                                                                                                                                            |
 | TPREL      | Expression value is the offset of a symbol relative to the TLS block. Used in the Local Executable TLS access model. Note that `TPREL` and `TPOFF` are descriptors for the same TLS access models. Use the attribute name equivalent to the syntax compatible with the target assembler. See [ELF Handling for Thread-Local Storage](https://www.uclibc.org/docs/tls.pdf) for more details. |
 | TPOFF      | Expression value is the offset of a symbol relative to the TLS block. Used in the Local Executable TLS access model. Note that `TPREL` and `TPOFF` are descriptors for the same TLS access models. Use the attribute name equivalent to the syntax compatible with the target assembler. See [ELF Handling for Thread-Local Storage](https://www.uclibc.org/docs/tls.pdf) for more details. |
 | DTPREL     | Expression value is the offset of a symbol relative to the TLS block. Used in the Local Dynamic TLS access model. Note that `DTPREL` and `DTPOFF` are descriptors for the same TLS access models. Use the attribute name equivalent to the syntax compatible with the target assembler. See [ELF Handling for Thread-Local Storage](https://www.uclibc.org/docs/tls.pdf) for more details.  |
 | DTPOFF     | Expression value is the offset of a symbol relative to the TLS block. Used in the Local Dynamic TLS access model. Note that `DTPREL` and `DTPOFF` are descriptors for the same TLS access models. Use the attribute name equivalent to the syntax compatible with the target assembler. See [ELF Handling for Thread-Local Storage](https://www.uclibc.org/docs/tls.pdf) for more details.  |
 | DTPMOD     | Expression value is an object identifier for the object containing a TLS symbol. See [ELF Handling for Thread-Local Storage](https://www.uclibc.org/docs/tls.pdf) for more details.                                                                                                                                                                                                         |
 | NTPOFF     | Expression value is a negative offset of a symbol relative to the TLS block. See [ELF Handling for Thread-Local Storage](https://www.uclibc.org/docs/tls.pdf) for more details.                                                                                                                                                                                                             |
 | TLSDESC    | Expression value is a pointer to the `tlsdesc` structure. See [RFC TLSDESC](https://www.fsfla.org/~lxoliva/writeups/TLS/RFC-TLSDESC-x86.txt)                                                                                                                                                                                                                                                |
 | TLSCALL    | Expression value is a call through the `tlsdesc` structure.                                                                                                                                                                                                                                                                                                                                 |
 | PAGE       | Expression value is a PC relative offset within a 4K page.                                                                                                                                                                                                                                                                                                                                  |
 | PAGEOFF    | Expression value is the offset within a 4K page, added to the page address to produce a complete address.                                                                                                                                                                                                                                                                                   |
 | CALL       | Specifies the expression value can only be present in a call instruction.                                                                                                                                                                                                                                                                                                                   |
 | LO         | Expression value is the low half word of an absolute symbol address; structured as [HI,LO] and [HIGHEST,HIGHER,HI,LO] for 32 and 64-bit addresses respectively.                                                                                                                                                                                                                             |
 | HI         | Expression value is the high half word of an absolute symbol address; structured as [HI,LO] and [HIGHEST,HIGHER,HI,LO] for 32 and 64-bit addresses respectively.                                                                                                                                                                                                                            |
 | HIGHER     | Expression value represents the higher half word of a symbol address; structured as [HI,LO] and [HIGHEST,HIGHER,HI,LO] for 32 and 64-bit addresses respectively.                                                                                                                                                                                                                            |
 | HIGHEST    | Expression value represents the highest word of a symbol address; structured as [HI,LO] and [HIGHEST,HIGHER,HI,LO] for 32 and 64-bit addresses respectively.                                                                                                                                                                                                                                |
 |            |                                                                                                                                                                                                                                                                                                                                                                                             |
 | **X86**    |                                                                                                                                                                                                                                                                                                                                                                                             |
 | GOTNTPOFF  | Expression value references a GOT entry with the negative offset relative to the thread-pointer (TLS block).                                                                                                                                                                                                                                                                                |
 | INDNTPOFF  | Position dependent variant of `GOTNTPOFF` (absolute address of GOT entry).                                                                                                                                                                                                                                                                                                                  |
 |            |                                                                                                                                                                                                                                                                                                                                                                                             |
 | **ARM**    |                                                                                                                                                                                                                                                                                                                                                                                             |
 | G0         | Expression value represents bits 0-15 of a 64-bit address.                                                                                                                                                                                                                                                                                                                                  |
 | G1         | Expression value represents bits 16-31 of a 64-bit address.                                                                                                                                                                                                                                                                                                                                 |
 | G2         | Expression value represents bits 32-47 of a 64-bit address.                                                                                                                                                                                                                                                                                                                                 |
 | G3         | Expression value represents bits 48-63 of a 64-bit address.                                                                                                                                                                                                                                                                                                                                 |
 | UPPER16    | Expression value represents upper 16 bits of an address.                                                                                                                                                                                                                                                                                                                                    |
 | LOWER16    | Expression value represents lower 16 bits of an address.                                                                                                                                                                                                                                                                                                                                    |
 | LO12       | Expression value represents bits 0-11 of an address.                                                                                                                                                                                                                                                                                                                                        |
 | LO14       | Expression value represents bits 0-13 of an address.                                                                                                                                                                                                                                                                                                                                        |
 | LO15       | Expression value represents bits 0-14 of an address.                                                                                                                                                                                                                                                                                                                                        |
 | HI12       | Expression value represents bits 13-24 of a 64-bit address.                                                                                                                                                                                                                                                                                                                                 |
 | HI21       | Expression value represents bits 11-31 of an address.                                                                                                                                                                                                                                                                                                                                       |
 | S          | Indicates the expression value is signed.                                                                                                                                                                                                                                                                                                                                                   |
 | PG         | Functionally identical to `PAGE`, but should be used for ARM relocations such as `:pg_hi21:symbol`.                                                                                                                                                                                                                                                                                         |
 | NC         | Indicates the linker should not check for overflow, "no check".                                                                                                                                                                                                                                                                                                                             |
 | ABS        | Indicates the expression value is an absolute address.                                                                                                                                                                                                                                                                                                                                      |
 | PREL       | Indicates the expression value is PC relative.                                                                                                                                                                                                                                                                                                                                              |
 | PREL31     | Expression value represents a 31-bit PC relative offset.                                                                                                                                                                                                                                                                                                                                    |
 | TARGET1    | Target specific relocation interpreted as either `R_ARM_ABS32` or `R_ARM_REL32`. Linker switches `--target1-rel` and `--target1-abs` switches override the default definition.                                                                                                                                                                                                              |
 | TARGET2    | Target specific relocation interpreted as either `R_ARM_ABS32` or `R_ARM_REL32` or `R_ARM_GOT_PREL`. Linker switch `--target2={rel,abs,got-rel}` overrides the default definition.                                                                                                                                                                                                          |
 | SBREL      | Expression value is a static-base-relative offset.                                                                                                                                                                                                                                                                                                                                          |
 | TLSLDO     | Debug relocation for a TLS variable.                                                                                                                                                                                                                                                                                                                                                        |
 |            |                                                                                                                                                                                                                                                                                                                                                                                             |
 | **MIPS**   |                                                                                                                                                                                                                                                                                                                                                                                             |
 | HI16       | Expression value represents the upper 16 bits of a symbol.                                                                                                                                                                                                                                                                                                                                  |
 | LO16       | Expression value represents the lower 16 bits of a symbol.                                                                                                                                                                                                                                                                                                                                  |
 | GPREL      | Expression value represents the offset from the current GP value.                                                                                                                                                                                                                                                                                                                           |
 | DISP       | Composes with `GOT` attribute for `%got_disp` relocation. GOT entry as displacement into the global data section.                                                                                                                                                                                                                                                                           |
 | OFST       | Composes with `GOT` attribute for `%got_ofst` relocation. GOT entry as offset from a page pointer.                                                                                                                                                                                                                                                                                          |
 |            |                                                                                                                                                                                                                                                                                                                                                                                             |
 | **PPC**    |                                                                                                                                                                                                                                                                                                                                                                                             |
 | H          | Expression value represents the high 16 bits of a symbol.                                                                                                                                                                                                                                                                                                                                   |
 | L          | Expression value represents the low 16 bits of a symbol.                                                                                                                                                                                                                                                                                                                                    |
 | HA         | Expression value represents the high 16 bits, _adjusted_ for corresponding signed `symbol@l` relocation.                                                                                                                                                                                                                                                                                    |
 | HIGH       | Expression value represents the high 16 bits of a symbol, with overflow check (PPC64).                                                                                                                                                                                                                                                                                                      |
 | HIGHA      | Expression value represents the hight 16 bits of a symbol, with overflow check (PPC64), _adjusted_ for corresponding signed `symbol@l` relocation.                                                                                                                                                                                                                                          |
 | HIGHERA    | Expression value represents bits 32-47 of an address, _adjusted_ for corresponding signed `symbol@l` relocation.                                                                                                                                                                                                                                                                            |
 | HIGHESTA   | Expression value represents 48-63 of an address, _adjusted_ for corresponding signed `symbol@l` relocation.                                                                                                                                                                                                                                                                                 |
 | TOCBASE    | Expression value represents the TOC base for the current object.                                                                                                                                                                                                                                                                                                                            |
 | TOC        | Expression value is the offset of the TOC entry from the base of the TOC.                                                                                                                                                                                                                                                                                                                   |
 | NOTOC      | Indicates callee clobbers the TOC register.                                                                                                                                                                                                                                                                                                                                                 |


API Implementations
--------------------

The guaranteed functionality is provided as follows.

| Language    | SymbolicExpression Implementation |
|:------------|:----------------------------------|
| C++         | gtirb::SymbolicExpression         |
| Python      | gtirb.SymbolicExpression          |
| Common Lisp | **symbolic-expression**           |


Links
--------------------

- [GTIRB Components](COMPONENTS.md)
- [Using Serialized GTIRB Data](../../PROTOBUF.md)
