/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.memory;

import docking.widgets.OptionDialog;
import ghidra.app.cmd.memory.DeleteBlockCmd;
import ghidra.app.plugin.core.memory.MemoryMapPlugin;
import ghidra.framework.cmd.BackgroundCommand;
import ghidra.framework.cmd.Command;
import ghidra.framework.model.DomainObject;
import ghidra.framework.plugintool.PluginTool;
import ghidra.framework.store.LockException;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOverflowException;
import ghidra.program.model.address.AddressSet;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.address.AddressSpace;
import ghidra.program.model.listing.DataIterator;
import ghidra.program.model.listing.InstructionIterator;
import ghidra.program.model.listing.Listing;
import ghidra.program.model.listing.Program;
import ghidra.program.model.listing.ProgramFragment;
import ghidra.program.model.mem.Memory;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.mem.MemoryBlockException;
import ghidra.program.model.mem.MemoryBlockType;
import ghidra.util.Msg;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.exception.NotFoundException;
import ghidra.util.exception.RollbackException;
import java.awt.Component;
import java.util.Collections;
import java.util.List;

class MemoryMapManager {
    private Program program;
    private MemoryMapPlugin plugin;
    private PluginTool tool;

    MemoryMapManager(MemoryMapPlugin plugin) {
        this.plugin = plugin;
        this.tool = plugin.getTool();
    }

    void splitBlock(MemoryBlock block, Address newStart, String newBlockName) {
        SplitBlockCmd cmd = new SplitBlockCmd(block, newStart, newBlockName);
        if (!this.tool.execute((Command)cmd, (DomainObject)this.program)) {
            Msg.showError(this.getClass(), (Component)this.plugin.getMemoryMapProvider().getComponent(), (String)"Split Block Failed", (Object)cmd.getStatusMsg());
        }
    }

    void mergeBlocks(List<MemoryBlock> blocks) {
        Collections.sort(blocks, (b1, b2) -> b1.getStart().compareTo((Object)b2.getStart()));
        if (!this.goodBlocks(blocks)) {
            return;
        }
        MergeBlocksCmd cmd = new MergeBlocksCmd(blocks);
        if (!this.tool.execute((Command)cmd, (DomainObject)this.program)) {
            Msg.showError(this.getClass(), (Component)this.plugin.getMemoryMapProvider().getComponent(), (String)"Merge Blocks Failed", (Object)cmd.getStatusMsg());
        }
    }

    private static void renameFragment(Program program, Address start, String name) {
        String[] treeNames;
        Listing listing = program.getListing();
        for (String treeName : treeNames = listing.getTreeNames()) {
            boolean duplicate = false;
            int index = 0;
            ProgramFragment frag = listing.getFragment(treeName, start);
            do {
                try {
                    frag.setName("Frag" + index + "-" + name);
                    duplicate = false;
                }
                catch (DuplicateNameException exc) {
                    duplicate = true;
                }
                ++index;
            } while (duplicate);
        }
    }

    private boolean goodBlocks(List<MemoryBlock> blocks) {
        int option;
        Class bc = null;
        long space_size = 0L;
        for (int i = 0; i < blocks.size() - 1; ++i) {
            MemoryBlock blockA = blocks.get(i);
            MemoryBlock blockB = blocks.get(i + 1);
            if (bc == null) {
                bc = blockA.getClass();
            }
            if (bc != blockA.getClass() || bc != blockB.getClass()) {
                Msg.showError((Object)this, (Component)this.plugin.getMemoryMapProvider().getComponent(), (String)"Merge Blocks Failed", (Object)"Can't merge blocks because all block types are not the same");
                return false;
            }
            Address nextStart = blockA.getEnd();
            AddressSpace space = nextStart.getAddressSpace();
            if (space.isOverlaySpace() && space.isNonLoadedMemorySpace()) {
                Msg.showError((Object)this, (Component)this.plugin.getMemoryMapProvider().getComponent(), (String)"Merge Blocks Failed", (Object)"Cannot merge OTHER overlay blocks");
                return false;
            }
            Address blockBstart = blockB.getStart();
            if (!space.isSuccessor(nextStart, blockBstart)) {
                try {
                    Address a = nextStart.addNoWrap(1L);
                    MemoryBlock b = this.program.getMemory().getBlock(a);
                    if (b != null) {
                        Msg.showError((Object)this, (Component)this.plugin.getMemoryMapProvider().getComponent(), (String)"Merge Blocks Failed", (Object)"Can't merge blocks because they are not contiguous");
                        return false;
                    }
                    if (blockA.getType() == MemoryBlockType.BIT_MAPPED) {
                        Msg.showError((Object)this, (Component)this.plugin.getMemoryMapProvider().getComponent(), (String)"Merge Blocks Failed", (Object)"Can't merge Bit Memory Blocks because they do not\nhave successive block end and block start addresses.");
                        return false;
                    }
                }
                catch (AddressOverflowException a) {
                    // empty catch block
                }
            }
            try {
                space_size += blockBstart.subtract(blockA.getEnd());
                continue;
            }
            catch (IllegalArgumentException e) {
                Msg.showError((Object)this, (Component)this.plugin.getMemoryMapProvider().getComponent(), (String)"Merge Blocks Failed", (Object)e.getMessage(), (Throwable)e);
                return false;
            }
        }
        return space_size <= 0x400000L || (option = OptionDialog.showOptionDialog((Component)this.plugin.getMemoryMapProvider().getComponent(), (String)"Merge Memory Blocks", (String)("Merging these blocks will create " + space_size / 1024L + "K extra bytes in memory.\nDo you really want to merge the selected Memory Block(s)?"), (String)"Merge Blocks", (int)3)) != 0;
    }

    void setProgram(Program program) {
        this.program = program;
    }

    void deleteBlocks(List<MemoryBlock> blocks) {
        if (blocks == null || blocks.size() <= 0) {
            return;
        }
        StringBuffer blockNames = new StringBuffer();
        AddressSet set = new AddressSet();
        for (int i = 0; i < blocks.size(); ++i) {
            MemoryBlock block = blocks.get(i);
            blockNames.append(block.getName());
            if (i < blocks.size() - 1) {
                blockNames.append(", ");
            }
            set.addRange(block.getStart(), block.getEnd());
        }
        Object msg = "Do you really want to delete the Memory Block(s)\n   " + String.valueOf(blockNames) + "  ?";
        Listing listing = this.program.getListing();
        InstructionIterator iter = listing.getInstructions((AddressSetView)set, true);
        if (iter.hasNext()) {
            msg = "Code Units exist in selected block(s).\nDo you want to continue?";
        } else {
            DataIterator dIter = listing.getDefinedData((AddressSetView)set, true);
            if (dIter.hasNext()) {
                msg = "Code Units exist in selected block(s).\nDo you want to continue?";
            }
        }
        int option = -1;
        option = OptionDialog.showOptionDialog((Component)this.plugin.getMemoryMapProvider().getComponent(), (String)"Delete Memory Block?", (String)msg, (String)"Yes", (int)3);
        if (option == 0) {
            return;
        }
        Address[] addresses = new Address[blocks.size()];
        for (int i = 0; i < blocks.size(); ++i) {
            MemoryBlock block = blocks.get(i);
            addresses[i] = block.getStart();
        }
        DeleteBlockCmd cmd = new DeleteBlockCmd(addresses, command -> {});
        this.tool.executeBackgroundCommand((BackgroundCommand)cmd, (DomainObject)this.program);
    }

    private static class SplitBlockCmd
    implements Command<Program> {
        private MemoryBlock block;
        private Address newStart;
        private String newBlockName;
        private String msg;

        SplitBlockCmd(MemoryBlock block, Address newStart, String newBlockName) {
            this.block = block;
            this.newStart = newStart;
            this.newBlockName = newBlockName;
        }

        public boolean applyTo(Program program) {
            Memory memory = program.getMemory();
            if (!program.hasExclusiveAccess()) {
                this.msg = "Exclusive access required";
                return false;
            }
            try {
                memory.split(this.block, this.newStart);
            }
            catch (MemoryBlockException e) {
                this.msg = e.getMessage();
                return false;
            }
            catch (IllegalArgumentException e) {
                this.msg = e.getMessage();
                return false;
            }
            catch (NotFoundException e) {
                this.msg = e.getMessage();
                return false;
            }
            catch (LockException e) {
                this.msg = e.getMessage();
                return false;
            }
            MemoryBlock newBlock = memory.getBlock(this.newStart);
            try {
                newBlock.setName(this.newBlockName);
            }
            catch (LockException e) {
                this.msg = e.getMessage();
                return false;
            }
            return true;
        }

        public String getName() {
            return "Split Memory Block";
        }

        public String getStatusMsg() {
            return this.msg;
        }
    }

    private static class MergeBlocksCmd
    implements Command<Program> {
        private String msg;
        private List<MemoryBlock> blocks;

        MergeBlocksCmd(List<MemoryBlock> blocks) {
            this.blocks = blocks;
        }

        public boolean applyTo(Program program) {
            Memory mem = program.getMemory();
            Address min = null;
            Address max = null;
            if (!this.allBlocksInSameSpace()) {
                this.msg = "All memory block must be in the same address space.";
                return false;
            }
            for (MemoryBlock nextBlock : this.blocks) {
                if (min == null || nextBlock.getStart().compareTo((Object)min) < 0) {
                    min = nextBlock.getStart();
                }
                if (max != null && nextBlock.getEnd().compareTo((Object)max) <= 0) continue;
                max = nextBlock.getEnd();
            }
            if (max == null) {
                return false;
            }
            long size = max.subtract(min) + 1L;
            if (size <= 0L || size > Integer.MAX_VALUE) {
                this.msg = "Resulting Memory Block would be too large.";
                return false;
            }
            try {
                MemoryBlock bigBlock = this.blocks.get(0);
                for (int i = 1; i < this.blocks.size(); ++i) {
                    MemoryBlock nextBlock = this.blocks.get(i);
                    Address start = bigBlock.getEnd();
                    start = start.addNoWrap(1L);
                    long length = nextBlock.getStart().subtract(start);
                    if (length != 0L) {
                        MemoryBlock newBlock;
                        if (bigBlock.isInitialized()) {
                            newBlock = mem.createInitializedBlock(bigBlock.getName(), start, length, (byte)0, null, false);
                            newBlock.setRead(bigBlock.isRead());
                            newBlock.setWrite(bigBlock.isWrite());
                            newBlock.setExecute(bigBlock.isExecute());
                            newBlock.setVolatile(bigBlock.isVolatile());
                            newBlock.setArtificial(bigBlock.isArtificial());
                            newBlock.setSourceName("Resized Memory Block");
                        } else {
                            newBlock = mem.createUninitializedBlock(bigBlock.getName(), start, (long)((int)length), false);
                        }
                        newBlock.setComment(bigBlock.getComment());
                        bigBlock = mem.join(bigBlock, newBlock);
                    }
                    MemoryMapManager.renameFragment(program, start, bigBlock.getName());
                    bigBlock = mem.join(bigBlock, nextBlock);
                }
                return true;
            }
            catch (RollbackException e) {
                throw e;
            }
            catch (Exception e) {
                this.msg = e.getMessage();
                if (this.msg == null) {
                    this.msg = "Error merging blocks: " + String.valueOf(e);
                }
            }
            catch (OutOfMemoryError e) {
                this.msg = "Not enough memory to merge blocks";
            }
            throw new RollbackException(this.msg);
        }

        private boolean allBlocksInSameSpace() {
            AddressSpace lastSpace = null;
            for (MemoryBlock block : this.blocks) {
                Address start = block.getStart();
                AddressSpace space = start.getAddressSpace();
                if (lastSpace != null && !lastSpace.equals((Object)space)) {
                    return false;
                }
                lastSpace = space;
            }
            return true;
        }

        public String getName() {
            return "Merge Memory Blocks";
        }

        public String getStatusMsg() {
            return this.msg;
        }
    }
}

