# --------------------------------------------------------------------
# SPDX-License-Identifier: AGPL-3.0-or-later
# © Copyright 2008-2024 José Manuel Rodríguez de la Rosa and contributors.
# See the file CONTRIBUTORS.md for copyright details.
# See https://www.gnu.org/licenses/agpl-3.0.html for details.
# --------------------------------------------------------------------

from typing import Final, NamedTuple


class Opcode(NamedTuple):
    asm: str
    T: int
    size: int
    opcode: str


Z80SET = {
    "ADC A,(HL)": Opcode("ADC A,(HL)", 7, 1, "8E"),
    "ADC A,(IX+N)": Opcode("ADC A,(IX+N)", 19, 3, "DD 8E XX"),
    "ADC A,(IY+N)": Opcode("ADC A,(IY+N)", 19, 3, "FD 8E XX"),
    "ADC A,A": Opcode("ADC A,A", 4, 1, "8F"),
    "ADC A,C": Opcode("ADC A,C", 4, 1, "89"),
    "ADC A,B": Opcode("ADC A,B", 4, 1, "88"),
    "ADC A,E": Opcode("ADC A,E", 4, 1, "8B"),
    "ADC A,D": Opcode("ADC A,D", 4, 1, "8A"),
    "ADC A,H": Opcode("ADC A,H", 4, 1, "8C"),
    "ADC A,L": Opcode("ADC A,L", 4, 1, "8D"),
    "ADC A,N": Opcode("ADC A,N", 7, 2, "CE XX"),
    "ADC HL,BC": Opcode("ADC HL,BC", 15, 2, "ED 4A"),
    "ADC HL,DE": Opcode("ADC HL,DE", 15, 2, "ED 5A"),
    "ADC HL,HL": Opcode("ADC HL,HL", 15, 2, "ED 6A"),
    "ADC HL,SP": Opcode("ADC HL,SP", 15, 2, "ED 7A"),
    "ADD A,(HL)": Opcode("ADD A,(HL)", 7, 1, "86"),
    "ADD A,(IX+N)": Opcode("ADD A,(IX+N)", 19, 3, "DD 86 XX"),
    "ADD A,(IY+N)": Opcode("ADD A,(IY+N)", 19, 3, "FD 86 XX"),
    "ADD A,A": Opcode("ADD A,A", 4, 1, "87"),
    "ADD A,C": Opcode("ADD A,C", 4, 1, "81"),
    "ADD A,B": Opcode("ADD A,B", 4, 1, "80"),
    "ADD A,E": Opcode("ADD A,E", 4, 1, "83"),
    "ADD A,D": Opcode("ADD A,D", 4, 1, "82"),
    "ADD A,H": Opcode("ADD A,H", 4, 1, "84"),
    "ADD A,L": Opcode("ADD A,L", 4, 1, "85"),
    "ADD A,N": Opcode("ADD A,N", 7, 2, "C6 XX"),
    "ADD HL,BC": Opcode("ADD HL,BC", 11, 1, "09"),
    "ADD HL,DE": Opcode("ADD HL,DE", 11, 1, "19"),
    "ADD HL,HL": Opcode("ADD HL,HL", 11, 1, "29"),
    "ADD HL,SP": Opcode("ADD HL,SP", 11, 1, "39"),
    "ADD IX,BC": Opcode("ADD IX,BC", 15, 2, "DD 09"),
    "ADD IX,DE": Opcode("ADD IX,DE", 15, 2, "DD 19"),
    "ADD IX,IX": Opcode("ADD IX,IX", 15, 2, "DD 29"),
    "ADD IX,SP": Opcode("ADD IX,SP", 15, 2, "DD 39"),
    "ADD IY,BC": Opcode("ADD IY,BC", 15, 2, "FD 09"),
    "ADD IY,DE": Opcode("ADD IY,DE", 15, 2, "FD 19"),
    "ADD IY,IY": Opcode("ADD IY,IY", 15, 2, "FD 29"),
    "ADD IY,SP": Opcode("ADD IY,SP", 15, 2, "FD 39"),
    "AND (HL)": Opcode("AND (HL)", 7, 1, "A6"),
    "AND (IX+N)": Opcode("AND (IX+N)", 19, 3, "DD A6 XX"),
    "AND (IY+N)": Opcode("AND (IY+N)", 19, 3, "FD A6 XX"),
    "AND A": Opcode("AND A", 4, 1, "A7"),
    "AND C": Opcode("AND C", 4, 1, "A1"),
    "AND B": Opcode("AND B", 4, 1, "A0"),
    "AND E": Opcode("AND E", 4, 1, "A3"),
    "AND D": Opcode("AND D", 4, 1, "A2"),
    "AND H": Opcode("AND H", 4, 1, "A4"),
    "AND L": Opcode("AND L", 4, 1, "A5"),
    "AND N": Opcode("AND N", 7, 2, "E6 XX"),
    "BIT 0,(HL)": Opcode("BIT 0,(HL)", 12, 2, "CB 46"),
    "BIT 1,(HL)": Opcode("BIT 1,(HL)", 12, 2, "CB 4E"),
    "BIT 2,(HL)": Opcode("BIT 2,(HL)", 12, 2, "CB 56"),
    "BIT 3,(HL)": Opcode("BIT 3,(HL)", 12, 2, "CB 5E"),
    "BIT 4,(HL)": Opcode("BIT 4,(HL)", 12, 2, "CB 66"),
    "BIT 5,(HL)": Opcode("BIT 5,(HL)", 12, 2, "CB 6E"),
    "BIT 6,(HL)": Opcode("BIT 6,(HL)", 12, 2, "CB 76"),
    "BIT 7,(HL)": Opcode("BIT 7,(HL)", 12, 2, "CB 7E"),
    "BIT 0,(IX+N)": Opcode("BIT 0,(IX+N)", 20, 4, "DD CB XX 46"),
    "BIT 1,(IX+N)": Opcode("BIT 1,(IX+N)", 20, 4, "DD CB XX 4E"),
    "BIT 2,(IX+N)": Opcode("BIT 2,(IX+N)", 20, 4, "DD CB XX 56"),
    "BIT 3,(IX+N)": Opcode("BIT 3,(IX+N)", 20, 4, "DD CB XX 5E"),
    "BIT 4,(IX+N)": Opcode("BIT 4,(IX+N)", 20, 4, "DD CB XX 66"),
    "BIT 5,(IX+N)": Opcode("BIT 5,(IX+N)", 20, 4, "DD CB XX 6E"),
    "BIT 6,(IX+N)": Opcode("BIT 6,(IX+N)", 20, 4, "DD CB XX 76"),
    "BIT 7,(IX+N)": Opcode("BIT 7,(IX+N)", 20, 4, "DD CB XX 7E"),
    "BIT 0,(IY+N)": Opcode("BIT 0,(IY+N)", 20, 4, "FD CB XX 46"),
    "BIT 1,(IY+N)": Opcode("BIT 1,(IY+N)", 20, 4, "FD CB XX 4E"),
    "BIT 2,(IY+N)": Opcode("BIT 2,(IY+N)", 20, 4, "FD CB XX 56"),
    "BIT 3,(IY+N)": Opcode("BIT 3,(IY+N)", 20, 4, "FD CB XX 5E"),
    "BIT 4,(IY+N)": Opcode("BIT 4,(IY+N)", 20, 4, "FD CB XX 66"),
    "BIT 5,(IY+N)": Opcode("BIT 5,(IY+N)", 20, 4, "FD CB XX 6E"),
    "BIT 6,(IY+N)": Opcode("BIT 6,(IY+N)", 20, 4, "FD CB XX 76"),
    "BIT 7,(IY+N)": Opcode("BIT 7,(IY+N)", 20, 4, "FD CB XX 7E"),
    "BIT 0,A": Opcode("BIT 0,A", 8, 2, "CB 47"),
    "BIT 1,A": Opcode("BIT 1,A", 8, 2, "CB 4F"),
    "BIT 2,A": Opcode("BIT 2,A", 8, 2, "CB 57"),
    "BIT 3,A": Opcode("BIT 3,A", 8, 2, "CB 5F"),
    "BIT 4,A": Opcode("BIT 4,A", 8, 2, "CB 67"),
    "BIT 5,A": Opcode("BIT 5,A", 8, 2, "CB 6F"),
    "BIT 6,A": Opcode("BIT 6,A", 8, 2, "CB 77"),
    "BIT 7,A": Opcode("BIT 7,A", 8, 2, "CB 7F"),
    "BIT 0,C": Opcode("BIT 0,C", 8, 2, "CB 41"),
    "BIT 1,C": Opcode("BIT 1,C", 8, 2, "CB 49"),
    "BIT 2,C": Opcode("BIT 2,C", 8, 2, "CB 51"),
    "BIT 3,C": Opcode("BIT 3,C", 8, 2, "CB 59"),
    "BIT 4,C": Opcode("BIT 4,C", 8, 2, "CB 61"),
    "BIT 5,C": Opcode("BIT 5,C", 8, 2, "CB 69"),
    "BIT 6,C": Opcode("BIT 6,C", 8, 2, "CB 71"),
    "BIT 7,C": Opcode("BIT 7,C", 8, 2, "CB 79"),
    "BIT 0,B": Opcode("BIT 0,B", 8, 2, "CB 40"),
    "BIT 1,B": Opcode("BIT 1,B", 8, 2, "CB 48"),
    "BIT 2,B": Opcode("BIT 2,B", 8, 2, "CB 50"),
    "BIT 3,B": Opcode("BIT 3,B", 8, 2, "CB 58"),
    "BIT 4,B": Opcode("BIT 4,B", 8, 2, "CB 60"),
    "BIT 5,B": Opcode("BIT 5,B", 8, 2, "CB 68"),
    "BIT 6,B": Opcode("BIT 6,B", 8, 2, "CB 70"),
    "BIT 7,B": Opcode("BIT 7,B", 8, 2, "CB 78"),
    "BIT 0,E": Opcode("BIT 0,E", 8, 2, "CB 43"),
    "BIT 1,E": Opcode("BIT 1,E", 8, 2, "CB 4B"),
    "BIT 2,E": Opcode("BIT 2,E", 8, 2, "CB 53"),
    "BIT 3,E": Opcode("BIT 3,E", 8, 2, "CB 5B"),
    "BIT 4,E": Opcode("BIT 4,E", 8, 2, "CB 63"),
    "BIT 5,E": Opcode("BIT 5,E", 8, 2, "CB 6B"),
    "BIT 6,E": Opcode("BIT 6,E", 8, 2, "CB 73"),
    "BIT 7,E": Opcode("BIT 7,E", 8, 2, "CB 7B"),
    "BIT 0,D": Opcode("BIT 0,D", 8, 2, "CB 42"),
    "BIT 1,D": Opcode("BIT 1,D", 8, 2, "CB 4A"),
    "BIT 2,D": Opcode("BIT 2,D", 8, 2, "CB 52"),
    "BIT 3,D": Opcode("BIT 3,D", 8, 2, "CB 5A"),
    "BIT 4,D": Opcode("BIT 4,D", 8, 2, "CB 62"),
    "BIT 5,D": Opcode("BIT 5,D", 8, 2, "CB 6A"),
    "BIT 6,D": Opcode("BIT 6,D", 8, 2, "CB 72"),
    "BIT 7,D": Opcode("BIT 7,D", 8, 2, "CB 7A"),
    "BIT 0,H": Opcode("BIT 0,H", 8, 2, "CB 44"),
    "BIT 1,H": Opcode("BIT 1,H", 8, 2, "CB 4C"),
    "BIT 2,H": Opcode("BIT 2,H", 8, 2, "CB 54"),
    "BIT 3,H": Opcode("BIT 3,H", 8, 2, "CB 5C"),
    "BIT 4,H": Opcode("BIT 4,H", 8, 2, "CB 64"),
    "BIT 5,H": Opcode("BIT 5,H", 8, 2, "CB 6C"),
    "BIT 6,H": Opcode("BIT 6,H", 8, 2, "CB 74"),
    "BIT 7,H": Opcode("BIT 7,H", 8, 2, "CB 7C"),
    "BIT 0,L": Opcode("BIT 0,L", 8, 2, "CB 45"),
    "BIT 1,L": Opcode("BIT 1,L", 8, 2, "CB 4D"),
    "BIT 2,L": Opcode("BIT 2,L", 8, 2, "CB 55"),
    "BIT 3,L": Opcode("BIT 3,L", 8, 2, "CB 5D"),
    "BIT 4,L": Opcode("BIT 4,L", 8, 2, "CB 65"),
    "BIT 5,L": Opcode("BIT 5,L", 8, 2, "CB 6D"),
    "BIT 6,L": Opcode("BIT 6,L", 8, 2, "CB 75"),
    "BIT 7,L": Opcode("BIT 7,L", 8, 2, "CB 7D"),
    "CALL C,NN": Opcode("CALL C,NN", 17, 3, "DC XX XX"),
    "CALL M,NN": Opcode("CALL M,NN", 17, 3, "FC XX XX"),
    "CALL NC,NN": Opcode("CALL NC,NN", 17, 3, "D4 XX XX"),
    "CALL NN": Opcode("CALL NN", 17, 3, "CD XX XX"),
    "CALL NZ,NN": Opcode("CALL NZ,NN", 17, 3, "C4 XX XX"),
    "CALL P,NN": Opcode("CALL P,NN", 17, 3, "F4 XX XX"),
    "CALL PE,NN": Opcode("CALL PE,NN", 17, 3, "EC XX XX"),
    "CALL PO,NN": Opcode("CALL PO,NN", 17, 3, "E4 XX XX"),
    "CALL Z,NN": Opcode("CALL Z,NN", 17, 3, "CC XX XX"),
    "CCF": Opcode("CCF", 4, 1, "3F"),
    "CP (HL)": Opcode("CP (HL)", 7, 1, "BE"),
    "CP (IX+N)": Opcode("CP (IX+N)", 19, 3, "DD BE XX"),
    "CP (IY+N)": Opcode("CP (IY+N)", 19, 3, "FD BE XX"),
    "CP A": Opcode("CP A", 4, 1, "BF"),
    "CP C": Opcode("CP C", 4, 1, "B9"),
    "CP B": Opcode("CP B", 4, 1, "B8"),
    "CP E": Opcode("CP E", 4, 1, "BB"),
    "CP D": Opcode("CP D", 4, 1, "BA"),
    "CP H": Opcode("CP H", 4, 1, "BC"),
    "CP L": Opcode("CP L", 4, 1, "BD"),
    "CP N": Opcode("CP N", 7, 2, "FE XX"),
    "CPD": Opcode("CPD", 16, 2, "ED A9"),
    "CPDR": Opcode("CPDR", 21, 2, "ED B9"),
    "CPI": Opcode("CPI", 16, 2, "ED A1"),
    "CPIR": Opcode("CPIR", 21, 2, "ED B1"),
    "CPL": Opcode("CPL", 4, 1, "2F"),
    "DAA": Opcode("DAA", 4, 1, "27"),
    "DEC (HL)": Opcode("DEC (HL)", 11, 1, "35"),
    "DEC (IX+N)": Opcode("DEC (IX+N)", 23, 3, "DD 35 XX"),
    "DEC (IY+N)": Opcode("DEC (IY+N)", 23, 3, "FD 35 XX"),
    "DEC A": Opcode("DEC A", 4, 1, "3D"),
    "DEC B": Opcode("DEC B", 4, 1, "05"),
    "DEC BC": Opcode("DEC BC", 6, 1, "0B"),
    "DEC C": Opcode("DEC C", 4, 1, "0D"),
    "DEC D": Opcode("DEC D", 4, 1, "15"),
    "DEC DE": Opcode("DEC DE", 6, 1, "1B"),
    "DEC E": Opcode("DEC E", 4, 1, "1D"),
    "DEC H": Opcode("DEC H", 4, 1, "25"),
    "DEC HL": Opcode("DEC HL", 6, 1, "2B"),
    "DEC IX": Opcode("DEC IX", 10, 2, "DD 2B"),
    "DEC IY": Opcode("DEC IY", 10, 2, "FD 2B"),
    "DEC L": Opcode("DEC L", 4, 1, "2D"),
    "DEC SP": Opcode("DEC SP", 6, 1, "3B"),
    "DI": Opcode("DI", 4, 1, "F3"),
    "DJNZ N": Opcode("DJNZ N", 13, 2, "10 XX"),
    "EI": Opcode("EI", 4, 1, "FB"),
    "EX (SP),HL": Opcode("EX (SP),HL", 19, 1, "E3"),
    "EX (SP),IX": Opcode("EX (SP),IX", 23, 2, "DD E3"),
    "EX (SP),IY": Opcode("EX (SP),IY", 23, 2, "FD E3"),
    "EX AF,AF'": Opcode("EX AF,AF'", 4, 1, "08"),
    "EX DE,HL": Opcode("EX DE,HL", 4, 1, "EB"),
    "EXX": Opcode("EXX", 4, 1, "D9"),
    "HALT": Opcode("HALT", 4, 1, "76"),
    "IM 0": Opcode("IM 0", 8, 2, "ED 46"),
    "IM 1": Opcode("IM 1", 8, 2, "ED 56"),
    "IM 2": Opcode("IM 2", 8, 2, "ED 5E"),
    "IN A,(C)": Opcode("IN A,(C)", 12, 2, "ED 78"),
    "IN A,(N)": Opcode("IN A,(N)", 11, 2, "DB XX"),
    "IN B,(C)": Opcode("IN B,(C)", 12, 2, "ED 40"),
    "IN C,(C)": Opcode("IN C,(C)", 12, 2, "ED 48"),
    "IN D,(C)": Opcode("IN D,(C)", 12, 2, "ED 50"),
    "IN E,(C)": Opcode("IN E,(C)", 12, 2, "ED 58"),
    "IN H,(C)": Opcode("IN H,(C)", 12, 2, "ED 60"),
    "IN L,(C)": Opcode("IN L,(C)", 12, 2, "ED 68"),
    "INC (HL)": Opcode("INC (HL)", 11, 1, "34"),
    "INC (IX+N)": Opcode("INC (IX+N)", 23, 3, "DD 34 XX"),
    "INC (IY+N)": Opcode("INC (IY+N)", 23, 3, "FD 34 XX"),
    "INC A": Opcode("INC A", 4, 1, "3C"),
    "INC B": Opcode("INC B", 4, 1, "04"),
    "INC BC": Opcode("INC BC", 6, 1, "03"),
    "INC C": Opcode("INC C", 4, 1, "0C"),
    "INC D": Opcode("INC D", 4, 1, "14"),
    "INC DE": Opcode("INC DE", 6, 1, "13"),
    "INC E": Opcode("INC E", 4, 1, "1C"),
    "INC H": Opcode("INC H", 4, 1, "24"),
    "INC HL": Opcode("INC HL", 6, 1, "23"),
    "INC IX": Opcode("INC IX", 10, 2, "DD 23"),
    "INC IY": Opcode("INC IY", 10, 2, "FD 23"),
    "INC L": Opcode("INC L", 4, 1, "2C"),
    "INC SP": Opcode("INC SP", 6, 1, "33"),
    "IND": Opcode("IND", 16, 2, "ED AA"),
    "INDR": Opcode("INDR", 21, 2, "ED BA"),
    "INI": Opcode("INI", 16, 2, "ED A2"),
    "INIR": Opcode("INIR", 21, 2, "ED B2"),
    "JP NN": Opcode("JP NN", 10, 3, "C3 XX XX"),
    "JP (HL)": Opcode("JP (HL)", 4, 1, "E9"),
    "JP (IX)": Opcode("JP (IX)", 8, 2, "DD E9"),
    "JP (IY)": Opcode("JP (IY)", 8, 2, "FD E9"),
    "JP C,NN": Opcode("JP C,NN", 10, 3, "DA XX XX"),
    "JP M,NN": Opcode("JP M,NN", 10, 3, "FA XX XX"),
    "JP NC,NN": Opcode("JP NC,NN", 10, 3, "D2 XX XX"),
    "JP NZ,NN": Opcode("JP NZ,NN", 10, 3, "C2 XX XX"),
    "JP P,NN": Opcode("JP P,NN", 10, 3, "F2 XX XX"),
    "JP PE,NN": Opcode("JP PE,NN", 10, 3, "EA XX XX"),
    "JP PO,NN": Opcode("JP PO,NN", 10, 3, "E2 XX XX"),
    "JP Z,NN": Opcode("JP Z,NN", 10, 3, "CA XX XX"),
    "JR N": Opcode("JR N", 12, 2, "18 XX"),
    "JR C,N": Opcode("JR C,N", 12, 2, "38 XX"),
    "JR NC,N": Opcode("JR NC,N", 12, 2, "30 XX"),
    "JR NZ,N": Opcode("JR NZ,N", 12, 2, "20 XX"),
    "JR Z,N": Opcode("JR Z,N", 12, 2, "28 XX"),
    "LD (BC),A": Opcode("LD (BC),A", 7, 1, "02"),
    "LD (DE),A": Opcode("LD (DE),A", 7, 1, "12"),
    "LD (HL),A": Opcode("LD (HL),A", 7, 1, "77"),
    "LD (HL),C": Opcode("LD (HL),C", 7, 1, "71"),
    "LD (HL),B": Opcode("LD (HL),B", 7, 1, "70"),
    "LD (HL),E": Opcode("LD (HL),E", 7, 1, "73"),
    "LD (HL),D": Opcode("LD (HL),D", 7, 1, "72"),
    "LD (HL),H": Opcode("LD (HL),H", 7, 1, "74"),
    "LD (HL),L": Opcode("LD (HL),L", 7, 1, "75"),
    "LD (HL),N": Opcode("LD (HL),N", 10, 2, "36 XX"),
    "LD (IX+N),A": Opcode("LD (IX+N),A", 19, 3, "DD 77 XX"),
    "LD (IX+N),C": Opcode("LD (IX+N),C", 19, 3, "DD 71 XX"),
    "LD (IX+N),B": Opcode("LD (IX+N),B", 19, 3, "DD 70 XX"),
    "LD (IX+N),E": Opcode("LD (IX+N),E", 19, 3, "DD 73 XX"),
    "LD (IX+N),D": Opcode("LD (IX+N),D", 19, 3, "DD 72 XX"),
    "LD (IX+N),H": Opcode("LD (IX+N),H", 19, 3, "DD 74 XX"),
    "LD (IX+N),L": Opcode("LD (IX+N),L", 19, 3, "DD 75 XX"),
    "LD (IX+N),N": Opcode("LD (IX+N),N", 19, 4, "DD 36 XX XX"),
    "LD (IY+N),A": Opcode("LD (IY+N),A", 19, 3, "FD 77 XX"),
    "LD (IY+N),C": Opcode("LD (IY+N),C", 19, 3, "FD 71 XX"),
    "LD (IY+N),B": Opcode("LD (IY+N),B", 19, 3, "FD 70 XX"),
    "LD (IY+N),E": Opcode("LD (IY+N),E", 19, 3, "FD 73 XX"),
    "LD (IY+N),D": Opcode("LD (IY+N),D", 19, 3, "FD 72 XX"),
    "LD (IY+N),H": Opcode("LD (IY+N),H", 19, 3, "FD 74 XX"),
    "LD (IY+N),L": Opcode("LD (IY+N),L", 19, 3, "FD 75 XX"),
    "LD (IY+N),N": Opcode("LD (IY+N),N", 19, 4, "FD 36 XX XX"),
    "LD (NN),A": Opcode("LD (NN),A", 13, 3, "32 XX XX"),
    "LD (NN),BC": Opcode("LD (NN),BC", 20, 4, "ED 43 XX XX"),
    "LD (NN),DE": Opcode("LD (NN),DE", 20, 4, "ED 53 XX XX"),
    "LD (NN),HL": Opcode("LD (NN),HL", 16, 3, "22 XX XX"),
    "LD (NN),IX": Opcode("LD (NN),IX", 20, 4, "DD 22 XX XX"),
    "LD (NN),IY": Opcode("LD (NN),IY", 20, 4, "FD 22 XX XX"),
    "LD (NN),SP": Opcode("LD (NN),SP", 20, 4, "ED 73 XX XX"),
    "LD A,(BC)": Opcode("LD A,(BC)", 7, 1, "0A"),
    "LD A,(DE)": Opcode("LD A,(DE)", 7, 1, "1A"),
    "LD A,(HL)": Opcode("LD A,(HL)", 7, 1, "7E"),
    "LD A,(IX+N)": Opcode("LD A,(IX+N)", 19, 3, "DD 7E XX"),
    "LD A,(IY+N)": Opcode("LD A,(IY+N)", 19, 3, "FD 7E XX"),
    "LD A,(NN)": Opcode("LD A,(NN)", 13, 3, "3A XX XX"),
    "LD A,A": Opcode("LD A,A", 4, 1, "7F"),
    "LD A,C": Opcode("LD A,C", 4, 1, "79"),
    "LD A,B": Opcode("LD A,B", 4, 1, "78"),
    "LD A,E": Opcode("LD A,E", 4, 1, "7B"),
    "LD A,D": Opcode("LD A,D", 4, 1, "7A"),
    "LD A,H": Opcode("LD A,H", 4, 1, "7C"),
    "LD A,L": Opcode("LD A,L", 4, 1, "7D"),
    "LD A,I": Opcode("LD A,I", 9, 2, "ED 57"),
    "LD A,N": Opcode("LD A,N", 7, 2, "3E XX"),
    "LD A,R": Opcode("LD A,R", 4, 2, "ED 5F"),
    "LD B,(HL)": Opcode("LD B,(HL)", 7, 1, "46"),
    "LD B,(IX+N)": Opcode("LD B,(IX+N)", 19, 3, "DD 46 XX"),
    "LD B,(IY+N)": Opcode("LD B,(IY+N)", 19, 3, "FD 46 XX"),
    "LD B,A": Opcode("LD B,A", 4, 1, "47"),
    "LD B,C": Opcode("LD B,C", 4, 1, "41"),
    "LD B,B": Opcode("LD B,B", 4, 1, "40"),
    "LD B,E": Opcode("LD B,E", 4, 1, "43"),
    "LD B,D": Opcode("LD B,D", 4, 1, "42"),
    "LD B,H": Opcode("LD B,H", 4, 1, "44"),
    "LD B,L": Opcode("LD B,L", 4, 1, "45"),
    "LD B,N": Opcode("LD B,N", 7, 2, "06 XX"),
    "LD BC,(NN)": Opcode("LD BC,(NN)", 20, 4, "ED 4B XX XX"),
    "LD BC,NN": Opcode("LD BC,NN", 10, 3, "01 XX XX"),
    "LD C,(HL)": Opcode("LD C,(HL)", 7, 1, "4E"),
    "LD C,(IX+N)": Opcode("LD C,(IX+N)", 19, 3, "DD 4E XX"),
    "LD C,(IY+N)": Opcode("LD C,(IY+N)", 19, 3, "FD 4E XX"),
    "LD C,A": Opcode("LD C,A", 4, 1, "4F"),
    "LD C,C": Opcode("LD C,C", 4, 1, "49"),
    "LD C,B": Opcode("LD C,B", 4, 1, "48"),
    "LD C,E": Opcode("LD C,E", 4, 1, "4B"),
    "LD C,D": Opcode("LD C,D", 4, 1, "4A"),
    "LD C,H": Opcode("LD C,H", 4, 1, "4C"),
    "LD C,L": Opcode("LD C,L", 4, 1, "4D"),
    "LD C,N": Opcode("LD C,N", 7, 2, "0E XX"),
    "LD D,(HL)": Opcode("LD D,(HL)", 7, 1, "56"),
    "LD D,(IX+N)": Opcode("LD D,(IX+N)", 19, 3, "DD 56 XX"),
    "LD D,(IY+N)": Opcode("LD D,(IY+N)", 19, 3, "FD 56 XX"),
    "LD D,A": Opcode("LD D,A", 4, 1, "57"),
    "LD D,C": Opcode("LD D,C", 4, 1, "51"),
    "LD D,B": Opcode("LD D,B", 4, 1, "50"),
    "LD D,E": Opcode("LD D,E", 4, 1, "53"),
    "LD D,D": Opcode("LD D,D", 4, 1, "52"),
    "LD D,H": Opcode("LD D,H", 4, 1, "54"),
    "LD D,L": Opcode("LD D,L", 4, 1, "55"),
    "LD D,N": Opcode("LD D,N", 7, 2, "16 XX"),
    "LD DE,(NN)": Opcode("LD DE,(NN)", 20, 4, "ED 5B XX XX"),
    "LD DE,NN": Opcode("LD DE,NN", 10, 3, "11 XX XX"),
    "LD E,(HL)": Opcode("LD E,(HL)", 7, 1, "5E"),
    "LD E,(IX+N)": Opcode("LD E,(IX+N)", 19, 3, "DD 5E XX"),
    "LD E,(IY+N)": Opcode("LD E,(IY+N)", 19, 3, "FD 5E XX"),
    "LD E,A": Opcode("LD E,A", 4, 1, "5F"),
    "LD E,C": Opcode("LD E,C", 4, 1, "59"),
    "LD E,B": Opcode("LD E,B", 4, 1, "58"),
    "LD E,E": Opcode("LD E,E", 4, 1, "5B"),
    "LD E,D": Opcode("LD E,D", 4, 1, "5A"),
    "LD E,H": Opcode("LD E,H", 4, 1, "5C"),
    "LD E,L": Opcode("LD E,L", 4, 1, "5D"),
    "LD E,N": Opcode("LD E,N", 7, 2, "1E XX"),
    "LD H,(HL)": Opcode("LD H,(HL)", 7, 1, "66"),
    "LD H,(IX+N)": Opcode("LD H,(IX+N)", 19, 3, "DD 66 XX"),
    "LD H,(IY+N)": Opcode("LD H,(IY+N)", 19, 3, "FD 66 XX"),
    "LD H,A": Opcode("LD H,A", 4, 1, "67"),
    "LD H,C": Opcode("LD H,C", 4, 1, "61"),
    "LD H,B": Opcode("LD H,B", 4, 1, "60"),
    "LD H,E": Opcode("LD H,E", 4, 1, "63"),
    "LD H,D": Opcode("LD H,D", 4, 1, "62"),
    "LD H,H": Opcode("LD H,H", 4, 1, "64"),
    "LD H,L": Opcode("LD H,L", 4, 1, "65"),
    "LD H,N": Opcode("LD H,N", 7, 2, "26 XX"),
    "LD HL,(NN)": Opcode("LD HL,(NN)", 20, 3, "2A XX XX"),
    "LD HL,NN": Opcode("LD HL,NN", 10, 3, "21 XX XX"),
    "LD I,A": Opcode("LD I,A", 9, 2, "ED 47"),
    "LD IX,(NN)": Opcode("LD IX,(NN)", 20, 4, "DD 2A XX XX"),
    "LD IX,NN": Opcode("LD IX,NN", 14, 4, "DD 21 XX XX"),
    "LD IY,(NN)": Opcode("LD IY,(NN)", 20, 4, "FD 2A XX XX"),
    "LD IY,NN": Opcode("LD IY,NN", 14, 4, "FD 21 XX XX"),
    "LD L,(HL)": Opcode("LD L,(HL)", 7, 1, "6E"),
    "LD L,(IX+N)": Opcode("LD L,(IX+N)", 19, 3, "DD 6E XX"),
    "LD L,(IY+N)": Opcode("LD L,(IY+N)", 19, 3, "FD 6E XX"),
    "LD L,A": Opcode("LD L,A", 4, 1, "6F"),
    "LD L,C": Opcode("LD L,C", 4, 1, "69"),
    "LD L,B": Opcode("LD L,B", 4, 1, "68"),
    "LD L,E": Opcode("LD L,E", 4, 1, "6B"),
    "LD L,D": Opcode("LD L,D", 4, 1, "6A"),
    "LD L,H": Opcode("LD L,H", 4, 1, "6C"),
    "LD L,L": Opcode("LD L,L", 4, 1, "6D"),
    "LD L,N": Opcode("LD L,N", 7, 2, "2E XX"),
    "LD R,A": Opcode("LD R,A", 4, 2, "ED 4F"),
    "LD SP,(NN)": Opcode("LD SP,(NN)", 20, 4, "ED 7B XX XX"),
    "LD SP,HL": Opcode("LD SP,HL", 6, 1, "F9"),
    "LD SP,IX": Opcode("LD SP,IX", 10, 2, "DD F9"),
    "LD SP,IY": Opcode("LD SP,IY", 10, 2, "FD F9"),
    "LD SP,NN": Opcode("LD SP,NN", 10, 3, "31 XX XX"),
    "LDD": Opcode("LDD", 16, 2, "ED A8"),
    "LDDR": Opcode("LDDR", 21, 2, "ED B8"),
    "LDI": Opcode("LDI", 16, 2, "ED A0"),
    "LDIR": Opcode("LDIR", 21, 2, "ED B0"),
    "NEG": Opcode("NEG", 8, 2, "ED 44"),
    "NOP": Opcode("NOP", 4, 1, "00"),
    "OR (HL)": Opcode("OR (HL)", 7, 1, "B6"),
    "OR (IX+N)": Opcode("OR (IX+N)", 19, 3, "DD B6 XX"),
    "OR (IY+N)": Opcode("OR (IY+N)", 19, 3, "FD B6 XX"),
    "OR A": Opcode("OR A", 4, 1, "B7"),
    "OR C": Opcode("OR C", 4, 1, "B1"),
    "OR B": Opcode("OR B", 4, 1, "B0"),
    "OR E": Opcode("OR E", 4, 1, "B3"),
    "OR D": Opcode("OR D", 4, 1, "B2"),
    "OR H": Opcode("OR H", 4, 1, "B4"),
    "OR L": Opcode("OR L", 4, 1, "B5"),
    "OR N": Opcode("OR N", 7, 2, "F6 XX"),
    "OTDR": Opcode("OTDR", 21, 2, "ED BB"),
    "OTIR": Opcode("OTIR", 21, 2, "ED B3"),
    "OUT (C),A": Opcode("OUT (C),A", 12, 2, "ED 79"),
    "OUT (C),B": Opcode("OUT (C),B", 12, 2, "ED 41"),
    "OUT (C),C": Opcode("OUT (C),C", 12, 2, "ED 49"),
    "OUT (C),D": Opcode("OUT (C),D", 12, 2, "ED 51"),
    "OUT (C),E": Opcode("OUT (C),E", 12, 2, "ED 59"),
    "OUT (C),H": Opcode("OUT (C),H", 12, 2, "ED 61"),
    "OUT (C),L": Opcode("OUT (C),L", 12, 2, "ED 69"),
    "OUT (N),A": Opcode("OUT (N),A", 11, 2, "D3 XX"),
    "OUTD": Opcode("OUTD", 16, 2, "ED AB"),
    "OUTI": Opcode("OUTI", 16, 2, "ED A3"),
    "POP AF": Opcode("POP AF", 10, 1, "F1"),
    "POP BC": Opcode("POP BC", 10, 1, "C1"),
    "POP DE": Opcode("POP DE", 10, 1, "D1"),
    "POP HL": Opcode("POP HL", 10, 1, "E1"),
    "POP IX": Opcode("POP IX", 14, 2, "DD E1"),
    "POP IY": Opcode("POP IY", 14, 2, "FD E1"),
    "PUSH AF": Opcode("PUSH AF", 11, 1, "F5"),
    "PUSH BC": Opcode("PUSH BC", 11, 1, "C5"),
    "PUSH DE": Opcode("PUSH DE", 11, 1, "D5"),
    "PUSH HL": Opcode("PUSH HL", 11, 1, "E5"),
    "PUSH IX": Opcode("PUSH IX", 15, 2, "DD E5"),
    "PUSH IY": Opcode("PUSH IY", 15, 2, "FD E5"),
    "RES 0,(HL)": Opcode("RES 0,(HL)", 15, 2, "CB 86"),
    "RES 1,(HL)": Opcode("RES 1,(HL)", 15, 2, "CB 8E"),
    "RES 2,(HL)": Opcode("RES 2,(HL)", 15, 2, "CB 96"),
    "RES 3,(HL)": Opcode("RES 3,(HL)", 15, 2, "CB 9E"),
    "RES 4,(HL)": Opcode("RES 4,(HL)", 15, 2, "CB A6"),
    "RES 5,(HL)": Opcode("RES 5,(HL)", 15, 2, "CB AE"),
    "RES 6,(HL)": Opcode("RES 6,(HL)", 15, 2, "CB B6"),
    "RES 7,(HL)": Opcode("RES 7,(HL)", 15, 2, "CB BE"),
    "RES 0,(IX+N)": Opcode("RES 0,(IX+N)", 23, 4, "DD CB XX 86"),
    "RES 1,(IX+N)": Opcode("RES 1,(IX+N)", 23, 4, "DD CB XX 8E"),
    "RES 2,(IX+N)": Opcode("RES 2,(IX+N)", 23, 4, "DD CB XX 96"),
    "RES 3,(IX+N)": Opcode("RES 3,(IX+N)", 23, 4, "DD CB XX 9E"),
    "RES 4,(IX+N)": Opcode("RES 4,(IX+N)", 23, 4, "DD CB XX A6"),
    "RES 5,(IX+N)": Opcode("RES 5,(IX+N)", 23, 4, "DD CB XX AE"),
    "RES 6,(IX+N)": Opcode("RES 6,(IX+N)", 23, 4, "DD CB XX B6"),
    "RES 7,(IX+N)": Opcode("RES 7,(IX+N)", 23, 4, "DD CB XX BE"),
    "RES 0,(IY+N)": Opcode("RES 0,(IY+N)", 23, 4, "FD CB XX 86"),
    "RES 1,(IY+N)": Opcode("RES 1,(IY+N)", 23, 4, "FD CB XX 8E"),
    "RES 2,(IY+N)": Opcode("RES 2,(IY+N)", 23, 4, "FD CB XX 96"),
    "RES 3,(IY+N)": Opcode("RES 3,(IY+N)", 23, 4, "FD CB XX 9E"),
    "RES 4,(IY+N)": Opcode("RES 4,(IY+N)", 23, 4, "FD CB XX A6"),
    "RES 5,(IY+N)": Opcode("RES 5,(IY+N)", 23, 4, "FD CB XX AE"),
    "RES 6,(IY+N)": Opcode("RES 6,(IY+N)", 23, 4, "FD CB XX B6"),
    "RES 7,(IY+N)": Opcode("RES 7,(IY+N)", 23, 4, "FD CB XX BE"),
    "RES 0,A": Opcode("RES 0,A", 8, 2, "CB 87"),
    "RES 1,A": Opcode("RES 1,A", 8, 2, "CB 8F"),
    "RES 2,A": Opcode("RES 2,A", 8, 2, "CB 97"),
    "RES 3,A": Opcode("RES 3,A", 8, 2, "CB 9F"),
    "RES 4,A": Opcode("RES 4,A", 8, 2, "CB A7"),
    "RES 5,A": Opcode("RES 5,A", 8, 2, "CB AF"),
    "RES 6,A": Opcode("RES 6,A", 8, 2, "CB B7"),
    "RES 7,A": Opcode("RES 7,A", 8, 2, "CB BF"),
    "RES 0,C": Opcode("RES 0,C", 8, 2, "CB 81"),
    "RES 1,C": Opcode("RES 1,C", 8, 2, "CB 89"),
    "RES 2,C": Opcode("RES 2,C", 8, 2, "CB 91"),
    "RES 3,C": Opcode("RES 3,C", 8, 2, "CB 99"),
    "RES 4,C": Opcode("RES 4,C", 8, 2, "CB A1"),
    "RES 5,C": Opcode("RES 5,C", 8, 2, "CB A9"),
    "RES 6,C": Opcode("RES 6,C", 8, 2, "CB B1"),
    "RES 7,C": Opcode("RES 7,C", 8, 2, "CB B9"),
    "RES 0,B": Opcode("RES 0,B", 8, 2, "CB 80"),
    "RES 1,B": Opcode("RES 1,B", 8, 2, "CB 88"),
    "RES 2,B": Opcode("RES 2,B", 8, 2, "CB 90"),
    "RES 3,B": Opcode("RES 3,B", 8, 2, "CB 98"),
    "RES 4,B": Opcode("RES 4,B", 8, 2, "CB A0"),
    "RES 5,B": Opcode("RES 5,B", 8, 2, "CB A8"),
    "RES 6,B": Opcode("RES 6,B", 8, 2, "CB B0"),
    "RES 7,B": Opcode("RES 7,B", 8, 2, "CB B8"),
    "RES 0,E": Opcode("RES 0,E", 8, 2, "CB 83"),
    "RES 1,E": Opcode("RES 1,E", 8, 2, "CB 8B"),
    "RES 2,E": Opcode("RES 2,E", 8, 2, "CB 93"),
    "RES 3,E": Opcode("RES 3,E", 8, 2, "CB 9B"),
    "RES 4,E": Opcode("RES 4,E", 8, 2, "CB A3"),
    "RES 5,E": Opcode("RES 5,E", 8, 2, "CB AB"),
    "RES 6,E": Opcode("RES 6,E", 8, 2, "CB B3"),
    "RES 7,E": Opcode("RES 7,E", 8, 2, "CB BB"),
    "RES 0,D": Opcode("RES 0,D", 8, 2, "CB 82"),
    "RES 1,D": Opcode("RES 1,D", 8, 2, "CB 8A"),
    "RES 2,D": Opcode("RES 2,D", 8, 2, "CB 92"),
    "RES 3,D": Opcode("RES 3,D", 8, 2, "CB 9A"),
    "RES 4,D": Opcode("RES 4,D", 8, 2, "CB A2"),
    "RES 5,D": Opcode("RES 5,D", 8, 2, "CB AA"),
    "RES 6,D": Opcode("RES 6,D", 8, 2, "CB B2"),
    "RES 7,D": Opcode("RES 7,D", 8, 2, "CB BA"),
    "RES 0,H": Opcode("RES 0,H", 8, 2, "CB 84"),
    "RES 1,H": Opcode("RES 1,H", 8, 2, "CB 8C"),
    "RES 2,H": Opcode("RES 2,H", 8, 2, "CB 94"),
    "RES 3,H": Opcode("RES 3,H", 8, 2, "CB 9C"),
    "RES 4,H": Opcode("RES 4,H", 8, 2, "CB A4"),
    "RES 5,H": Opcode("RES 5,H", 8, 2, "CB AC"),
    "RES 6,H": Opcode("RES 6,H", 8, 2, "CB B4"),
    "RES 7,H": Opcode("RES 7,H", 8, 2, "CB BC"),
    "RES 0,L": Opcode("RES 0,L", 8, 2, "CB 85"),
    "RES 1,L": Opcode("RES 1,L", 8, 2, "CB 8D"),
    "RES 2,L": Opcode("RES 2,L", 8, 2, "CB 95"),
    "RES 3,L": Opcode("RES 3,L", 8, 2, "CB 9D"),
    "RES 4,L": Opcode("RES 4,L", 8, 2, "CB A5"),
    "RES 5,L": Opcode("RES 5,L", 8, 2, "CB AD"),
    "RES 6,L": Opcode("RES 6,L", 8, 2, "CB B5"),
    "RES 7,L": Opcode("RES 7,L", 8, 2, "CB BD"),
    "RET": Opcode("RET", 10, 1, "C9"),
    "RET C": Opcode("RET C", 11, 1, "D8"),
    "RET M": Opcode("RET M", 11, 1, "F8"),
    "RET NC": Opcode("RET NC", 11, 1, "D0"),
    "RET NZ": Opcode("RET NZ", 11, 1, "C0"),
    "RET P": Opcode("RET P", 11, 1, "F0"),
    "RET PE": Opcode("RET PE", 11, 1, "E8"),
    "RET PO": Opcode("RET PO", 11, 1, "E0"),
    "RET Z": Opcode("RET Z", 11, 1, "C8"),
    "RETI": Opcode("RETI", 14, 2, "ED 4D"),
    "RETN": Opcode("RETN", 14, 2, "ED 45"),
    "RL (HL)": Opcode("RL (HL)", 15, 2, "CB 16"),
    "RL A": Opcode("RL A", 8, 2, "CB 17"),
    "RL C": Opcode("RL C", 8, 2, "CB 11"),
    "RL B": Opcode("RL B", 8, 2, "CB 10"),
    "RL E": Opcode("RL E", 8, 2, "CB 13"),
    "RL D": Opcode("RL D", 8, 2, "CB 12"),
    "RL H": Opcode("RL H", 8, 2, "CB 14"),
    "RL L": Opcode("RL L", 8, 2, "CB 15"),
    "RL (IX+N)": Opcode("RL (IX+N)", 23, 4, "DD CB XX 16"),
    "RL (IY+N)": Opcode("RL (IY+N)", 23, 4, "FD CB XX 16"),
    "RLA": Opcode("RLA", 4, 1, "17"),
    "RLC (HL)": Opcode("RLC (HL)", 15, 2, "CB 06"),
    "RLC (IX+N)": Opcode("RLC (IX+N)", 23, 4, "DD CB XX 06"),
    "RLC (IY+N)": Opcode("RLC (IY+N)", 23, 4, "FD CB XX 06"),
    "RLC A": Opcode("RLC A", 8, 2, "CB 07"),
    "RLC C": Opcode("RLC C", 8, 2, "CB 01"),
    "RLC B": Opcode("RLC B", 8, 2, "CB 00"),
    "RLC E": Opcode("RLC E", 8, 2, "CB 03"),
    "RLC D": Opcode("RLC D", 8, 2, "CB 02"),
    "RLC H": Opcode("RLC H", 8, 2, "CB 04"),
    "RLC L": Opcode("RLC L", 8, 2, "CB 05"),
    "RLCA": Opcode("RLCA", 4, 1, "07"),
    "RLD": Opcode("RLD", 18, 2, "ED 6F"),
    "RR (HL)": Opcode("RR (HL)", 15, 2, "CB 1E"),
    "RR A": Opcode("RR A", 8, 2, "CB 1F"),
    "RR C": Opcode("RR C", 8, 2, "CB 19"),
    "RR B": Opcode("RR B", 8, 2, "CB 18"),
    "RR E": Opcode("RR E", 8, 2, "CB 1B"),
    "RR D": Opcode("RR D", 8, 2, "CB 1A"),
    "RR H": Opcode("RR H", 8, 2, "CB 1C"),
    "RR L": Opcode("RR L", 8, 2, "CB 1D"),
    "RR (IX+N)": Opcode("RR (IX+N)", 23, 4, "DD CB XX 1E"),
    "RR (IY+N)": Opcode("RR (IY+N)", 23, 4, "FD CB XX 1E"),
    "RRA": Opcode("RRA", 4, 1, "1F"),
    "RRC (HL)": Opcode("RRC (HL)", 15, 2, "CB 0E"),
    "RRC (IX+N)": Opcode("RRC (IX+N)", 23, 4, "DD CB XX 0E"),
    "RRC (IY+N)": Opcode("RRC (IY+N)", 23, 4, "FD CB XX 0E"),
    "RRC A": Opcode("RRC A", 8, 2, "CB 0F"),
    "RRC C": Opcode("RRC C", 8, 2, "CB 09"),
    "RRC B": Opcode("RRC B", 8, 2, "CB 08"),
    "RRC E": Opcode("RRC E", 8, 2, "CB 0B"),
    "RRC D": Opcode("RRC D", 8, 2, "CB 0A"),
    "RRC H": Opcode("RRC H", 8, 2, "CB 0C"),
    "RRC L": Opcode("RRC L", 8, 2, "CB 0D"),
    "RRCA": Opcode("RRCA", 4, 1, "0F"),
    "RRD": Opcode("RRD", 18, 2, "ED 67"),
    "RST 0H": Opcode("RST 0H", 11, 1, "C7"),
    "RST 8H": Opcode("RST 8H", 11, 1, "CF"),
    "RST 10H": Opcode("RST 10H", 11, 1, "D7"),
    "RST 18H": Opcode("RST 18H", 11, 1, "DF"),
    "RST 20H": Opcode("RST 20H", 11, 1, "E7"),
    "RST 28H": Opcode("RST 28H", 11, 1, "EF"),
    "RST 30H": Opcode("RST 30H", 11, 1, "F7"),
    "RST 38H": Opcode("RST 38H", 11, 1, "FF"),
    "SBC A,(HL)": Opcode("SBC (HL)", 7, 1, "9E"),
    "SBC A,(IX+N)": Opcode("SBC A,(IX+N)", 19, 3, "DD 9E XX"),
    "SBC A,(IY+N)": Opcode("SBC A,(IY+N)", 19, 3, "FD 9E XX"),
    "SBC A,N": Opcode("SBC A,N", 7, 2, "DE XX"),
    "SBC A,A": Opcode("SBC A,A", 4, 1, "9F"),
    "SBC A,B": Opcode("SBC A,B", 4, 1, "98"),
    "SBC A,C": Opcode("SBC A,C", 4, 1, "99"),
    "SBC A,D": Opcode("SBC A,D", 4, 1, "9A"),
    "SBC A,E": Opcode("SBC A,E", 4, 1, "9B"),
    "SBC A,H": Opcode("SBC A,H", 4, 1, "9C"),
    "SBC A,L": Opcode("SBC A,L", 4, 1, "9D"),
    "SBC HL,BC": Opcode("SBC HL,BC", 15, 2, "ED 42"),
    "SBC HL,DE": Opcode("SBC HL,DE", 15, 2, "ED 52"),
    "SBC HL,HL": Opcode("SBC HL,HL", 15, 2, "ED 62"),
    "SBC HL,SP": Opcode("SBC HL,SP", 15, 2, "ED 72"),
    "SCF": Opcode("SCF", 4, 1, "37"),
    "SET 0,(HL)": Opcode("SET 0,(HL)", 15, 2, "CB C6"),
    "SET 1,(HL)": Opcode("SET 1,(HL)", 15, 2, "CB CE"),
    "SET 2,(HL)": Opcode("SET 2,(HL)", 15, 2, "CB D6"),
    "SET 3,(HL)": Opcode("SET 3,(HL)", 15, 2, "CB DE"),
    "SET 4,(HL)": Opcode("SET 4,(HL)", 15, 2, "CB E6"),
    "SET 5,(HL)": Opcode("SET 5,(HL)", 15, 2, "CB EE"),
    "SET 6,(HL)": Opcode("SET 6,(HL)", 15, 2, "CB F6"),
    "SET 7,(HL)": Opcode("SET 7,(HL)", 15, 2, "CB FE"),
    "SET 0,(IX+N)": Opcode("SET 0,(IX+N)", 23, 4, "DD CB XX C6"),
    "SET 1,(IX+N)": Opcode("SET 1,(IX+N)", 23, 4, "DD CB XX CE"),
    "SET 2,(IX+N)": Opcode("SET 2,(IX+N)", 23, 4, "DD CB XX D6"),
    "SET 3,(IX+N)": Opcode("SET 3,(IX+N)", 23, 4, "DD CB XX DE"),
    "SET 4,(IX+N)": Opcode("SET 4,(IX+N)", 23, 4, "DD CB XX E6"),
    "SET 5,(IX+N)": Opcode("SET 5,(IX+N)", 23, 4, "DD CB XX EE"),
    "SET 6,(IX+N)": Opcode("SET 6,(IX+N)", 23, 4, "DD CB XX F6"),
    "SET 7,(IX+N)": Opcode("SET 7,(IX+N)", 23, 4, "DD CB XX FE"),
    "SET 0,(IY+N)": Opcode("SET 0,(IY+N)", 23, 4, "FD CB XX C6"),
    "SET 1,(IY+N)": Opcode("SET 1,(IY+N)", 23, 4, "FD CB XX CE"),
    "SET 2,(IY+N)": Opcode("SET 2,(IY+N)", 23, 4, "FD CB XX D6"),
    "SET 3,(IY+N)": Opcode("SET 3,(IY+N)", 23, 4, "FD CB XX DE"),
    "SET 4,(IY+N)": Opcode("SET 4,(IY+N)", 23, 4, "FD CB XX E6"),
    "SET 5,(IY+N)": Opcode("SET 5,(IY+N)", 23, 4, "FD CB XX EE"),
    "SET 6,(IY+N)": Opcode("SET 6,(IY+N)", 23, 4, "FD CB XX F6"),
    "SET 7,(IY+N)": Opcode("SET 7,(IY+N)", 23, 4, "FD CB XX FE"),
    "SET 0,A": Opcode("SET 0,A", 8, 2, "CB C7"),
    "SET 1,A": Opcode("SET 1,A", 8, 2, "CB CF"),
    "SET 2,A": Opcode("SET 2,A", 8, 2, "CB D7"),
    "SET 3,A": Opcode("SET 3,A", 8, 2, "CB DF"),
    "SET 4,A": Opcode("SET 4,A", 8, 2, "CB E7"),
    "SET 5,A": Opcode("SET 5,A", 8, 2, "CB EF"),
    "SET 6,A": Opcode("SET 6,A", 8, 2, "CB F7"),
    "SET 7,A": Opcode("SET 7,A", 8, 2, "CB FF"),
    "SET 0,C": Opcode("SET 0,C", 8, 2, "CB C1"),
    "SET 1,C": Opcode("SET 1,C", 8, 2, "CB C9"),
    "SET 2,C": Opcode("SET 2,C", 8, 2, "CB D1"),
    "SET 3,C": Opcode("SET 3,C", 8, 2, "CB D9"),
    "SET 4,C": Opcode("SET 4,C", 8, 2, "CB E1"),
    "SET 5,C": Opcode("SET 5,C", 8, 2, "CB E9"),
    "SET 6,C": Opcode("SET 6,C", 8, 2, "CB F1"),
    "SET 7,C": Opcode("SET 7,C", 8, 2, "CB F9"),
    "SET 0,B": Opcode("SET 0,B", 8, 2, "CB C0"),
    "SET 1,B": Opcode("SET 1,B", 8, 2, "CB C8"),
    "SET 2,B": Opcode("SET 2,B", 8, 2, "CB D0"),
    "SET 3,B": Opcode("SET 3,B", 8, 2, "CB D8"),
    "SET 4,B": Opcode("SET 4,B", 8, 2, "CB E0"),
    "SET 5,B": Opcode("SET 5,B", 8, 2, "CB E8"),
    "SET 6,B": Opcode("SET 6,B", 8, 2, "CB F0"),
    "SET 7,B": Opcode("SET 7,B", 8, 2, "CB F8"),
    "SET 0,E": Opcode("SET 0,E", 8, 2, "CB C3"),
    "SET 1,E": Opcode("SET 1,E", 8, 2, "CB CB"),
    "SET 2,E": Opcode("SET 2,E", 8, 2, "CB D3"),
    "SET 3,E": Opcode("SET 3,E", 8, 2, "CB DB"),
    "SET 4,E": Opcode("SET 4,E", 8, 2, "CB E3"),
    "SET 5,E": Opcode("SET 5,E", 8, 2, "CB EB"),
    "SET 6,E": Opcode("SET 6,E", 8, 2, "CB F3"),
    "SET 7,E": Opcode("SET 7,E", 8, 2, "CB FB"),
    "SET 0,D": Opcode("SET 0,D", 8, 2, "CB C2"),
    "SET 1,D": Opcode("SET 1,D", 8, 2, "CB CA"),
    "SET 2,D": Opcode("SET 2,D", 8, 2, "CB D2"),
    "SET 3,D": Opcode("SET 3,D", 8, 2, "CB DA"),
    "SET 4,D": Opcode("SET 4,D", 8, 2, "CB E2"),
    "SET 5,D": Opcode("SET 5,D", 8, 2, "CB EA"),
    "SET 6,D": Opcode("SET 6,D", 8, 2, "CB F2"),
    "SET 7,D": Opcode("SET 7,D", 8, 2, "CB FA"),
    "SET 0,H": Opcode("SET 0,H", 8, 2, "CB C4"),
    "SET 1,H": Opcode("SET 1,H", 8, 2, "CB CC"),
    "SET 2,H": Opcode("SET 2,H", 8, 2, "CB D4"),
    "SET 3,H": Opcode("SET 3,H", 8, 2, "CB DC"),
    "SET 4,H": Opcode("SET 4,H", 8, 2, "CB E4"),
    "SET 5,H": Opcode("SET 5,H", 8, 2, "CB EC"),
    "SET 6,H": Opcode("SET 6,H", 8, 2, "CB F4"),
    "SET 7,H": Opcode("SET 7,H", 8, 2, "CB FC"),
    "SET 0,L": Opcode("SET 0,L", 8, 2, "CB C5"),
    "SET 1,L": Opcode("SET 1,L", 8, 2, "CB CD"),
    "SET 2,L": Opcode("SET 2,L", 8, 2, "CB D5"),
    "SET 3,L": Opcode("SET 3,L", 8, 2, "CB DD"),
    "SET 4,L": Opcode("SET 4,L", 8, 2, "CB E5"),
    "SET 5,L": Opcode("SET 5,L", 8, 2, "CB ED"),
    "SET 6,L": Opcode("SET 6,L", 8, 2, "CB F5"),
    "SET 7,L": Opcode("SET 7,L", 8, 2, "CB FD"),
    "SLA (HL)": Opcode("SLA (HL)", 15, 2, "CB 26"),
    "SLA (IX+N)": Opcode("SLA (IX+N)", 23, 4, "DD CB XX 26"),
    "SLA (IY+N)": Opcode("SLA (IY+N)", 23, 4, "FD CB XX 26"),
    "SLA A": Opcode("SLA A", 8, 2, "CB 27"),
    "SLA C": Opcode("SLA C", 8, 2, "CB 21"),
    "SLA B": Opcode("SLA B", 8, 2, "CB 20"),
    "SLA E": Opcode("SLA E", 8, 2, "CB 23"),
    "SLA D": Opcode("SLA D", 8, 2, "CB 22"),
    "SLA H": Opcode("SLA H", 8, 2, "CB 24"),
    "SLA L": Opcode("SLA L", 8, 2, "CB 25"),
    "SRA (HL)": Opcode("SRA (HL)", 15, 2, "CB 2E"),
    "SRA (IX+N)": Opcode("SRA (IX+N)", 23, 4, "DD CB XX 2E"),
    "SRA (IY+N)": Opcode("SRA (IY+N)", 23, 4, "FD CB XX 2E"),
    "SRA A": Opcode("SRA A", 8, 2, "CB 2F"),
    "SRA C": Opcode("SRA C", 8, 2, "CB 29"),
    "SRA B": Opcode("SRA B", 8, 2, "CB 28"),
    "SRA E": Opcode("SRA E", 8, 2, "CB 2B"),
    "SRA D": Opcode("SRA D", 8, 2, "CB 2A"),
    "SRA H": Opcode("SRA H", 8, 2, "CB 2C"),
    "SRA L": Opcode("SRA L", 8, 2, "CB 2D"),
    "SRL (HL)": Opcode("SRL (HL)", 15, 2, "CB 3E"),
    "SRL (IX+N)": Opcode("SRL (IX+N)", 23, 4, "DD CB XX 3E"),
    "SRL (IY+N)": Opcode("SRL (IY+N)", 23, 4, "FD CB XX 3E"),
    "SRL A": Opcode("SRL A", 8, 2, "CB 3F"),
    "SRL C": Opcode("SRL C", 8, 2, "CB 39"),
    "SRL B": Opcode("SRL B", 8, 2, "CB 38"),
    "SRL E": Opcode("SRL E", 8, 2, "CB 3B"),
    "SRL D": Opcode("SRL D", 8, 2, "CB 3A"),
    "SRL H": Opcode("SRL H", 8, 2, "CB 3C"),
    "SRL L": Opcode("SRL L", 8, 2, "CB 3D"),
    "SUB (HL)": Opcode("SUB (HL)", 7, 1, "96"),
    "SUB (IX+N)": Opcode("SUB (IX+N)", 19, 3, "DD 96 XX"),
    "SUB (IY+N)": Opcode("SUB (IY+N)", 19, 3, "FD 96 XX"),
    "SUB A": Opcode("SUB A", 4, 1, "97"),
    "SUB C": Opcode("SUB C", 4, 1, "91"),
    "SUB B": Opcode("SUB B", 4, 1, "90"),
    "SUB E": Opcode("SUB E", 4, 1, "93"),
    "SUB D": Opcode("SUB D", 4, 1, "92"),
    "SUB H": Opcode("SUB H", 4, 1, "94"),
    "SUB L": Opcode("SUB L", 4, 1, "95"),
    "SUB N": Opcode("SUB N", 7, 2, "D6 XX"),
    "XOR (HL)": Opcode("XOR (HL)", 7, 1, "AE"),
    "XOR (IX+N)": Opcode("XOR (IX+N)", 19, 3, "DD AE XX"),
    "XOR (IY+N)": Opcode("XOR (IY+N)", 19, 3, "FD AE XX"),
    "XOR A": Opcode("XOR A", 4, 1, "AF"),
    "XOR C": Opcode("XOR C", 4, 1, "A9"),
    "XOR B": Opcode("XOR B", 4, 1, "A8"),
    "XOR E": Opcode("XOR E", 4, 1, "AB"),
    "XOR D": Opcode("XOR D", 4, 1, "AA"),
    "XOR H": Opcode("XOR H", 4, 1, "AC"),
    "XOR L": Opcode("XOR L", 4, 1, "AD"),
    "XOR N": Opcode("XOR N", 7, 2, "EE XX"),
    # Undocumented opcodes
    "SLL A": Opcode("SLL A", 8, 2, "CB 37"),
    "SLL C": Opcode("SLL C", 8, 2, "CB 31"),
    "SLL B": Opcode("SLL B", 8, 2, "CB 30"),
    "SLL E": Opcode("SLL E", 8, 2, "CB 33"),
    "SLL D": Opcode("SLL D", 8, 2, "CB 32"),
    "SLL H": Opcode("SLL H", 8, 2, "CB 34"),
    "SLL L": Opcode("SLL L", 8, 2, "CB 35"),
    "SLL (HL)": Opcode("SLL (HL)", 15, 2, "CB 36"),
    "SLL (IX+N)": Opcode("SLL (IX+N)", 19, 4, "DD CB XX 36"),
    "SLL (IY+N)": Opcode("SLL (IY+N)", 19, 4, "FD CB XX 36"),
    "INC IXH": Opcode("INC IXH", 8, 2, "DD 24"),
    "DEC IXH": Opcode("DEC IXH", 8, 2, "DD 25"),
    "INC IXL": Opcode("INC IXL", 8, 2, "DD 2C"),
    "DEC IXL": Opcode("DEC IXL", 8, 2, "DD 2D"),
    "INC IYH": Opcode("INC IYH", 8, 2, "FD 24"),
    "DEC IYH": Opcode("DEC IYH", 8, 2, "FD 25"),
    "INC IYL": Opcode("INC IYL", 8, 2, "FD 2C"),
    "DEC IYL": Opcode("DEC IYL", 8, 2, "FD 2D"),
    "LD IXH,N": Opcode("LD IXH,N", 12, 3, "DD 26 XX"),
    "LD IXL,N": Opcode("LD IXL,N", 12, 3, "DD 2E XX"),
    "LD IYH,N": Opcode("LD IYH,N", 12, 3, "FD 26 XX"),
    "LD IYL,N": Opcode("LD IYL,N", 12, 3, "FD 2E XX"),
    "LD A,IXH": Opcode("LD A,IXH", 8, 2, "DD 7C"),
    "LD A,IXL": Opcode("LD A,IXL", 8, 2, "DD 7D"),
    "LD B,IXH": Opcode("LD B,IXH", 8, 2, "DD 44"),
    "LD B,IXL": Opcode("LD B,IXL", 8, 2, "DD 45"),
    "LD C,IXH": Opcode("LD C,IXH", 8, 2, "DD 4C"),
    "LD C,IXL": Opcode("LD C,IXL", 8, 2, "DD 4D"),
    "LD D,IXH": Opcode("LD D,IXH", 8, 2, "DD 54"),
    "LD D,IXL": Opcode("LD D,IXL", 8, 2, "DD 55"),
    "LD E,IXH": Opcode("LD E,IXH", 8, 2, "DD 5C"),
    "LD E,IXL": Opcode("LD E,IXL", 8, 2, "DD 5D"),
    "LD A,IYH": Opcode("LD A,IYH", 8, 2, "FD 7C"),
    "LD A,IYL": Opcode("LD A,IYL", 8, 2, "FD 7D"),
    "LD B,IYH": Opcode("LD B,IYH", 8, 2, "FD 44"),
    "LD B,IYL": Opcode("LD B,IYL", 8, 2, "FD 45"),
    "LD C,IYH": Opcode("LD C,IYH", 8, 2, "FD 4C"),
    "LD C,IYL": Opcode("LD C,IYL", 8, 2, "FD 4D"),
    "LD D,IYH": Opcode("LD D,IYH", 8, 2, "FD 54"),
    "LD D,IYL": Opcode("LD D,IYL", 8, 2, "FD 55"),
    "LD E,IYH": Opcode("LD E,IYH", 8, 2, "FD 5C"),
    "LD E,IYL": Opcode("LD E,IYL", 8, 2, "FD 5D"),
    "LD IXH,B": Opcode("LD IXH,B", 8, 2, "DD 60"),
    "LD IXH,C": Opcode("LD IXH,C", 8, 2, "DD 61"),
    "LD IXH,D": Opcode("LD IXH,D", 8, 2, "DD 62"),
    "LD IXH,E": Opcode("LD IXH,E", 8, 2, "DD 63"),
    "LD IXH,IXH": Opcode("LD IXH,IXH", 8, 2, "DD 64"),
    "LD IXH,IXL": Opcode("LD IXH,IXL", 8, 2, "DD 65"),
    "LD IXH,A": Opcode("LD IXH,A", 8, 2, "DD 67"),
    "LD IXL,B": Opcode("LD IXL,B", 8, 2, "DD 68"),
    "LD IXL,C": Opcode("LD IXL,C", 8, 2, "DD 69"),
    "LD IXL,D": Opcode("LD IXL,D", 8, 2, "DD 6A"),
    "LD IXL,E": Opcode("LD IXL,E", 8, 2, "DD 6B"),
    "LD IXL,IXH": Opcode("LD IXL,IXH", 8, 2, "DD 6C"),
    "LD IXL,IXL": Opcode("LD IXL,IXL", 8, 2, "DD 6D"),
    "LD IXL,A": Opcode("LD IXL,A", 8, 2, "DD 6F"),
    "LD IYH,B": Opcode("LD IYH,B", 8, 2, "FD 60"),
    "LD IYH,C": Opcode("LD IYH,C", 8, 2, "FD 61"),
    "LD IYH,D": Opcode("LD IYH,D", 8, 2, "FD 62"),
    "LD IYH,E": Opcode("LD IYH,E", 8, 2, "FD 63"),
    "LD IYH,IYH": Opcode("LD IYH,IYH", 8, 2, "DD 64"),
    "LD IYH,IYL": Opcode("LD IYH,IYL", 8, 2, "DD 65"),
    "LD IYH,A": Opcode("LD IYH,A", 8, 2, "FD 67"),
    "LD IYL,B": Opcode("LD IYL,B", 8, 2, "FD 68"),
    "LD IYL,C": Opcode("LD IYL,C", 8, 2, "FD 69"),
    "LD IYL,D": Opcode("LD IYL,D", 8, 2, "FD 6A"),
    "LD IYL,E": Opcode("LD IYL,E", 8, 2, "FD 6B"),
    "LD IYL,IYH": Opcode("LD IYL,IYH", 8, 2, "FD 6C"),
    "LD IYL,IYL": Opcode("LD IYL,IYL", 8, 2, "FD 6D"),
    "LD IYL,A": Opcode("LD IYL,A", 8, 2, "FD 6F"),
    "ADD A,IXH": Opcode("ADD A,IXH", 8, 2, "DD 84"),
    "ADD A,IXL": Opcode("ADD A,IXL", 8, 2, "DD 85"),
    "ADC A,IXH": Opcode("ADC A,IXH", 8, 2, "DD 8C"),
    "ADC A,IXL": Opcode("ADC A,IXL", 8, 2, "DD 8D"),
    "ADD A,IYH": Opcode("ADD A,IYH", 8, 2, "FD 84"),
    "ADD A,IYL": Opcode("ADD A,IYL", 8, 2, "FD 85"),
    "ADC A,IYH": Opcode("ADC A,IYH", 8, 2, "FD 8C"),
    "ADC A,IYL": Opcode("ADC A,IYL", 8, 2, "FD 8D"),
    "SUB IXH": Opcode("SUB IXH", 8, 2, "DD 94"),
    "SUB IXL": Opcode("SUB IXL", 8, 2, "DD 95"),
    "SBC A,IXH": Opcode("SBC A,IXH", 8, 2, "DD 9C"),
    "SBC A,IXL": Opcode("SBC A,IXL", 8, 2, "DD 9D"),
    "SUB IYH": Opcode("SUB IYH", 8, 2, "FD 94"),
    "SUB IYL": Opcode("SUB IYL", 8, 2, "FD 95"),
    "SBC A,IYH": Opcode("SBC A,IYH", 8, 2, "FD 9C"),
    "SBC A,IYL": Opcode("SBC A,IYL", 8, 2, "FD 9D"),
    "AND IXH": Opcode("AND IXH", 8, 2, "DD A4"),
    "AND IXL": Opcode("AND IXL", 8, 2, "DD A5"),
    "AND IYH": Opcode("AND IYH", 8, 2, "FD A4"),
    "AND IYL": Opcode("AND IYL", 8, 2, "FD A5"),
    "XOR IXH": Opcode("XOR IXH", 8, 2, "DD AC"),
    "XOR IXL": Opcode("XOR IXL", 8, 2, "DD AD"),
    "XOR IYH": Opcode("XOR IYH", 8, 2, "FD AC"),
    "XOR IYL": Opcode("XOR IYL", 8, 2, "FD AD"),
    "OR IXH": Opcode("OR IXH", 8, 2, "DD B4"),
    "OR IXL": Opcode("OR IXL", 8, 2, "DD B5"),
    "OR IYH": Opcode("OR IYH", 8, 2, "FD B4"),
    "OR IYL": Opcode("OR IYL", 8, 2, "FD B5"),
    "CP IXH": Opcode("CP IXH", 8, 2, "DD BC"),
    "CP IXL": Opcode("CP IXL", 8, 2, "DD BD"),
    "CP IYH": Opcode("CP IYH", 8, 2, "FD BC"),
    "CP IYL": Opcode("CP IYL", 8, 2, "FD BD"),
    # ZX NEXT extra opcodes
    "LDIX": Opcode("LDIX", 16, 2, "ED A4"),
    "LDWS": Opcode("LDWS", 14, 2, "ED A5"),
    "LDIRX": Opcode("LDIRX", 21, 2, "ED B4"),
    "LDDX": Opcode("LDDX", 16, 2, "ED AC"),
    "LDDRX": Opcode("LDDRX", 21, 2, "ED BC"),
    "LDPIRX": Opcode("LDPIRX", 21, 2, "ED B7"),
    "OUTINB": Opcode("OUTINB", 16, 2, "ED 90"),
    "MUL D,E": Opcode("MUL D,E", 8, 2, "ED 30"),
    "ADD HL,A": Opcode("ADD HL,A", 8, 2, "ED 31"),
    "ADD DE,A": Opcode("ADD DE,A", 8, 2, "ED 32"),
    "ADD BC,A": Opcode("ADD BC,A", 8, 2, "ED 33"),
    "ADD HL,NN": Opcode("ADD HL,NN", 16, 4, "ED 34 XX XX"),
    "ADD DE,NN": Opcode("ADD DE,NN", 16, 4, "ED 35 XX XX"),
    "ADD BC,NN": Opcode("ADD BC,NN", 16, 4, "ED 36 XX XX"),
    "SWAPNIB": Opcode("SWAPNIB", 8, 2, "ED 23"),
    "MIRROR": Opcode("MIRROR", 8, 2, "ED 24"),
    "PUSH NN": Opcode("PUSH NN", 23, 4, "ED 8A XX XX"),
    "NEXTREG N,N": Opcode("NEXTREG N,N", 20, 4, "ED 91 XX XX"),
    "NEXTREG N,A": Opcode("NEXTREG N,A", 17, 3, "ED 92 XX"),
    "PIXELDN": Opcode("PIXELDN", 8, 2, "ED 93"),
    "PIXELAD": Opcode("PIXELAD", 8, 2, "ED 94"),
    "SETAE": Opcode("SETAE", 8, 2, "ED 95"),
    "TEST N": Opcode("TEST N", 11, 3, "ED 27 XX"),
    "BSLA DE,B": Opcode("BSLA DE,B", 8, 2, "ED 28"),
    "BSRA DE,B": Opcode("BSRA DE,B", 8, 2, "ED 29"),
    "BSRL DE,B": Opcode("BSRL DE,B", 8, 2, "ED 2A"),
    "BSRF DE,B": Opcode("BSRF DE,B", 8, 2, "ED 2B"),
    "BRLC DE,B": Opcode("BRLC DE,B", 8, 2, "ED 2C"),
    "JP (C)": Opcode("JP (C)", 13, 2, "ED 98"),
}


# Z80 asm instruction list
Z80INSTR: Final[set[str]] = {x.split()[0] for x in Z80SET}
