// Copyright 2017 The Procyon Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "utf8.h"

#include <string.h>

#define UFFFD_REPLACEMENT_CHARACTER 0xfffd;

static bool is_ascii(uint8_t byte) { return byte < 0200; }
static bool is_continuation(uint8_t byte) { return (0200 <= byte) && (byte < 0300); }
static bool is_header(uint8_t byte) { return (0302 <= byte) && (byte < 0365); }

static size_t continuation_count_for_header(uint8_t byte, pn_rune_t* rune) {
    if (byte >= 0360) {
        *rune = byte & 0007;
        return 3;
    } else if (byte >= 0340) {
        *rune = byte & 0017;
        return 2;
    } else {
        *rune = byte & 0037;
        return 1;
    }
}

int32_t pn_rune(const char* data, size_t size, size_t index) {
    data += index;
    size -= index;

    if (is_ascii(data[0])) {
        return data[0];
    } else if (!is_header(data[0])) {
        return UFFFD_REPLACEMENT_CHARACTER;
    }
    pn_rune_t rune;
    size_t    continuation_count = continuation_count_for_header(data[0], &rune);
    if (continuation_count > size) {
        return UFFFD_REPLACEMENT_CHARACTER;
    }
    while (continuation_count-- > 0) {
        if (!is_continuation(*++data)) {
            return UFFFD_REPLACEMENT_CHARACTER;
        }
        rune = (rune << 6) | (*data & 0077);
    }
    if (((0xd800 <= rune) && (rune < 0xe000)) || (0x110000 < rune)) {
        return UFFFD_REPLACEMENT_CHARACTER;
    }
    return rune;
}

size_t pn_rune_next(const char* data, size_t size, size_t index) {
    if (is_header(data[index])) {
        uint8_t h = data[index];
        if (h < 0340) {  // 2 bytes: U+80 ... U+7FF
            if ((index + 1) >= size) {
                return index + 1;  // invalid: unexpected end of string
            }
            uint8_t c[] = {data[index + 1]};
            if (!is_continuation(c[0])) {
                return index + 1;
            }
            return index + 2;
        } else if (h < 0360) {  // 3 bytes: U+800 ... U+D7FF, U+E000 ... U+FFFF
            if ((index + 2) >= size) {
                return index + 1;
            }
            uint8_t c[] = {data[index + 1], data[index + 2]};
            if (((h == 0340) && (c[0] < 0240)) ||   // U+800 ... U+FFF (no overlong encoding)
                ((h == 0355) && (0240 <= c[0])) ||  // U+D000 ... U+D7FF (no surrogate)
                !(is_continuation(c[0]) && is_continuation(c[1]))) {
                return index + 1;
            }
            return index + 3;
        } else {  // 4 bytes: U+40000 ... U+10FFFF
            if ((index + 3) >= size) {
                return index + 1;
            }
            uint8_t c[] = {data[index + 1], data[index + 2], data[index + 3]};
            if (((h == 0360) && (c[0] < 0220)) ||   // U+10000 ... U+3FFFF (no overlong encoding)
                ((h == 0364) && (0220 <= c[0])) ||  // U+100000 ... U+10FFFF (end of code space)
                !(is_continuation(c[0]) && is_continuation(c[1]) && is_continuation(c[2]))) {
                return index + 1;
            }
            return index + 4;
        }
    } else {
        return index + 1;  // default: ASCII or invalid header, so one byte at a time.
    }
}

size_t pn_rune_prev(const char* data, size_t size, size_t index) {
    if ((index >= 4) && (pn_rune_next(data, size, index - 4) == index)) {
        return index - 4;
    } else if ((index >= 3) && (pn_rune_next(data, size, index - 3) == index)) {
        return index - 3;
    } else if ((index >= 2) && (pn_rune_next(data, size, index - 2) == index)) {
        return index - 2;
    } else {
        return index - 1;
    }
}

void pn_ascchr(uint8_t rune, char* data, size_t* size) {
    if (rune < 0x80) {
        *size = 1;
        *data = rune;
    } else {
        *size = 3;
        memcpy(data, "\357\277\275", 3);
    }
}

void pn_unichr(pn_rune_t rune, char* data, size_t* size) {
    if (((0xd800 <= rune) && (rune < 0xe000)) || (0x110000 <= rune)) {
        *size = 3;
        memcpy(data, "\357\277\275", 3);
    } else if (rune < 0x80) {
        *size = 1;
        *data = rune;
    } else if (rune < 0x800) {
        *size   = 2;
        data[0] = 0xc0 | (rune >> 6);
        data[1] = 0x80 | ((rune >> 0) & 0x3f);
    } else if (rune < 0x10000) {
        *size   = 3;
        data[0] = 0xe0 | (rune >> 12);
        data[1] = 0x80 | ((rune >> 6) & 0x3f);
        data[2] = 0x80 | ((rune >> 0) & 0x3f);
    } else {
        *size   = 4;
        data[0] = 0xf0 | (rune >> 18);
        data[1] = 0x80 | ((rune >> 12) & 0x3f);
        data[2] = 0x80 | ((rune >> 6) & 0x3f);
        data[3] = 0x80 | ((rune >> 0) & 0x3f);
    }
}

enum {
    PN_RUNE_OTHER              = 000,  // C
    PN_RUNE_OTHER_CONTROL      = 000,  // Cc
    PN_RUNE_OTHER_FORMAT       = 001,  // Cf
    PN_RUNE_OTHER_NOT_ASSIGNED = 002,  // Cn
    PN_RUNE_OTHER_PRIVATE_USE  = 003,  // Co
    PN_RUNE_OTHER_SURROGATE    = 004,  // Cs

    PN_RUNE_LETTER           = 010,  // L
    PN_RUNE_LETTER_LOWERCASE = 010,  // Ll
    PN_RUNE_LETTER_MODIFIER  = 011,  // Lm
    PN_RUNE_LETTER_OTHER     = 012,  // Lo
    PN_RUNE_LETTER_TITLECASE = 013,  // Lt
    PN_RUNE_LETTER_UPPERCASE = 014,  // Lu

    PN_RUNE_MODIFIER                   = 020,  // M
    PN_RUNE_MODIFIER_SPACING_COMBINING = 020,  // Mc
    PN_RUNE_MODIFIER_ENCLOSING         = 021,  // Me
    PN_RUNE_MODIFIER_NONSPACING        = 022,  // Mn

    PN_RUNE_NUMBER               = 030,  // N
    PN_RUNE_NUMBER_DECIMAL_DIGIT = 030,  // Nd
    PN_RUNE_NUMBER_LETTER        = 031,  // Nl
    PN_RUNE_NUMBER_OTHER         = 032,  // No

    PN_RUNE_PUNCTUATION               = 040,  // P
    PN_RUNE_PUNCTUATION_CONNECTOR     = 040,  // Pc
    PN_RUNE_PUNCTUATION_DASH          = 041,  // Pd
    PN_RUNE_PUNCTUATION_CLOSE         = 042,  // Pe
    PN_RUNE_PUNCTUATION_FINAL_QUOTE   = 043,  // Pf
    PN_RUNE_PUNCTUATION_INITIAL_QUOTE = 044,  // Pi
    PN_RUNE_PUNCTUATION_OTHER         = 045,  // Po
    PN_RUNE_PUNCTUATION_OPEN          = 046,  // Ps

    PN_RUNE_SYMBOL          = 050,  // S
    PN_RUNE_SYMBOL_CURRENCY = 050,  // Sc
    PN_RUNE_SYMBOL_MODIFIER = 051,  // Sk
    PN_RUNE_SYMBOL_MATH     = 052,  // Sm
    PN_RUNE_SYMBOL_OTHER    = 053,  // So

    PN_RUNE_SEPARATOR           = 060,  // Z
    PN_RUNE_SEPARATOR_LINE      = 060,  // Zl
    PN_RUNE_SEPARATOR_PARAGRAPH = 061,  // Zp
    PN_RUNE_SEPARATOR_SPACE     = 062,  // Zs
};

enum {
    PN_RUNE_HALFWIDTH = 0000,  // A H N Na
    PN_RUNE_FULLWIDTH = 0100,  // F W
};

static const int8_t data_0[][8] = {
        [0]   = {0, 0, 0, 0, 0, 0, 0, 0},
        [1]   = {1, 1, 1, 1, 1, 1, 1, 1},
        [2]   = {1, 1, 1, 1, 1, 1, 42, 42},
        [3]   = {1, 1, 1, 1, 1, 66, 1, 1},
        [4]   = {1, 1, 1, 1, 66, 66, 66, 66},
        [5]   = {1, 1, 1, 18, 18, 18, 18, 18},
        [6]   = {1, 66, 66, 66, 66, 66, 66, 66},
        [7]   = {3, 3, 3, 3, 3, 3, 3, 3},
        [8]   = {3, 3, 3, 3, 3, 3, 66, 66},
        [9]   = {4, 4, 4, 4, 4, 4, 4, 4},
        [10]  = {8, 8, 8, 8, 8, 8, 8, 8},
        [11]  = {8, 8, 8, 8, 8, 8, 8, 12},
        [12]  = {8, 8, 8, 8, 8, 8, 8, 42},
        [13]  = {8, 8, 8, 8, 8, 8, 8, 66},
        [14]  = {8, 8, 8, 8, 8, 8, 12, 8},
        [15]  = {8, 8, 8, 8, 8, 8, 12, 12},
        [16]  = {8, 8, 8, 8, 8, 8, 66, 8},
        [17]  = {8, 8, 8, 8, 8, 8, 66, 66},
        [18]  = {8, 8, 8, 8, 8, 42, 8, 8},
        [19]  = {8, 8, 8, 8, 8, 66, 8, 8},
        [20]  = {8, 8, 8, 8, 9, 9, 9, 9},
        [21]  = {8, 8, 8, 8, 9, 9, 12, 12},
        [22]  = {8, 8, 8, 8, 10, 8, 8, 8},
        [23]  = {8, 8, 8, 8, 12, 8, 42, 12},
        [24]  = {8, 8, 8, 8, 12, 12, 12, 12},
        [25]  = {8, 8, 8, 8, 12, 12, 66, 12},
        [26]  = {8, 8, 8, 8, 12, 66, 12, 12},
        [27]  = {8, 8, 8, 8, 18, 18, 18, 18},
        [28]  = {8, 8, 8, 8, 66, 8, 8, 8},
        [29]  = {8, 8, 8, 8, 66, 66, 8, 8},
        [30]  = {8, 8, 8, 8, 66, 66, 66, 66},
        [31]  = {8, 8, 8, 9, 9, 9, 9, 9},
        [32]  = {8, 8, 8, 37, 9, 8, 8, 8},
        [33]  = {8, 8, 8, 38, 42, 34, 42, 0},
        [34]  = {8, 8, 8, 41, 9, 9, 9, 9},
        [35]  = {8, 8, 8, 42, 8, 8, 8, 8},
        [36]  = {8, 8, 8, 66, 66, 66, 66, 66},
        [37]  = {8, 8, 12, 8, 12, 8, 12, 8},
        [38]  = {8, 8, 12, 8, 66, 66, 24, 24},
        [39]  = {8, 8, 12, 12, 8, 12, 12, 8},
        [40]  = {8, 8, 12, 12, 12, 8, 8, 8},
        [41]  = {8, 8, 12, 12, 12, 12, 12, 12},
        [42]  = {8, 8, 43, 42, 43, 43, 8, 43},
        [43]  = {8, 8, 66, 8, 66, 8, 8, 8},
        [44]  = {8, 12, 8, 8, 12, 8, 12, 12},
        [45]  = {8, 12, 8, 12, 8, 8, 12, 8},
        [46]  = {8, 12, 8, 12, 8, 12, 8, 8},
        [47]  = {8, 12, 8, 12, 8, 12, 8, 12},
        [48]  = {8, 12, 8, 12, 8, 12, 12, 8},
        [49]  = {8, 12, 8, 12, 8, 12, 12, 12},
        [50]  = {8, 12, 8, 12, 12, 12, 12, 8},
        [51]  = {8, 12, 11, 8, 12, 8, 12, 12},
        [52]  = {8, 12, 12, 8, 8, 12, 12, 12},
        [53]  = {8, 12, 12, 8, 12, 8, 12, 12},
        [54]  = {8, 12, 12, 12, 8, 8, 12, 12},
        [55]  = {8, 12, 12, 12, 8, 12, 8, 12},
        [56]  = {8, 12, 12, 12, 12, 12, 12, 12},
        [57]  = {8, 37, 33, 66, 66, 43, 43, 40},
        [58]  = {8, 42, 8, 8, 8, 8, 8, 8},
        [59]  = {8, 66, 66, 66, 66, 66, 66, 66},
        [60]  = {9, 8, 8, 8, 8, 8, 8, 8},
        [61]  = {9, 9, 8, 10, 10, 10, 10, 10},
        [62]  = {9, 9, 9, 8, 8, 8, 8, 8},
        [63]  = {9, 9, 9, 9, 9, 9, 9, 9},
        [64]  = {9, 9, 9, 9, 9, 9, 37, 37},
        [65]  = {9, 9, 9, 9, 9, 9, 66, 66},
        [66]  = {9, 9, 9, 9, 9, 41, 41, 41},
        [67]  = {9, 9, 9, 9, 9, 66, 66, 66},
        [68]  = {9, 9, 9, 9, 37, 43, 66, 66},
        [69]  = {9, 9, 41, 41, 41, 41, 9, 9},
        [70]  = {9, 9, 41, 41, 41, 41, 41, 41},
        [71]  = {9, 10, 10, 10, 10, 10, 10, 10},
        [72]  = {9, 10, 10, 10, 10, 10, 10, 43},
        [73]  = {9, 18, 18, 18, 18, 18, 66, 66},
        [74]  = {9, 41, 41, 12, 8, 12, 8, 10},
        [75]  = {10, 8, 43, 43, 8, 8, 12, 12},
        [76]  = {10, 10, 10, 9, 10, 10, 10, 10},
        [77]  = {10, 10, 10, 10, 9, 37, 37, 37},
        [78]  = {10, 10, 10, 10, 10, 10, 9, 9},
        [79]  = {10, 10, 10, 10, 10, 10, 9, 18},
        [80]  = {10, 10, 10, 10, 10, 10, 10, 10},
        [81]  = {10, 10, 10, 10, 10, 10, 10, 16},
        [82]  = {10, 10, 10, 10, 10, 10, 10, 18},
        [83]  = {10, 10, 10, 10, 10, 10, 10, 43},
        [84]  = {10, 10, 10, 10, 10, 10, 10, 66},
        [85]  = {10, 10, 10, 10, 10, 10, 16, 16},
        [86]  = {10, 10, 10, 10, 10, 10, 18, 16},
        [87]  = {10, 10, 10, 10, 10, 10, 18, 18},
        [88]  = {10, 10, 10, 10, 10, 10, 25, 25},
        [89]  = {10, 10, 10, 10, 10, 10, 26, 26},
        [90]  = {10, 10, 10, 10, 10, 10, 34, 38},
        [91]  = {10, 10, 10, 10, 10, 10, 66, 10},
        [92]  = {10, 10, 10, 10, 10, 10, 66, 37},
        [93]  = {10, 10, 10, 10, 10, 10, 66, 66},
        [94]  = {10, 10, 10, 10, 10, 16, 16, 16},
        [95]  = {10, 10, 10, 10, 10, 16, 18, 16},
        [96]  = {10, 10, 10, 10, 10, 18, 9, 10},
        [97]  = {10, 10, 10, 10, 10, 18, 18, 10},
        [98]  = {10, 10, 10, 10, 10, 18, 18, 18},
        [99]  = {10, 10, 10, 10, 10, 18, 18, 66},
        [100] = {10, 10, 10, 10, 10, 26, 26, 26},
        [101] = {10, 10, 10, 10, 10, 26, 26, 37},
        [102] = {10, 10, 10, 10, 10, 43, 37, 10},
        [103] = {10, 10, 10, 10, 10, 66, 9, 66},
        [104] = {10, 10, 10, 10, 10, 66, 10, 10},
        [105] = {10, 10, 10, 10, 10, 66, 10, 66},
        [106] = {10, 10, 10, 10, 10, 66, 66, 1},
        [107] = {10, 10, 10, 10, 10, 66, 66, 10},
        [108] = {10, 10, 10, 10, 10, 66, 66, 26},
        [109] = {10, 10, 10, 10, 10, 66, 66, 66},
        [110] = {10, 10, 10, 10, 12, 11, 8, 12},
        [111] = {10, 10, 10, 10, 16, 16, 16, 16},
        [112] = {10, 10, 10, 10, 16, 16, 16, 18},
        [113] = {10, 10, 10, 10, 18, 10, 10, 16},
        [114] = {10, 10, 10, 10, 18, 16, 18, 18},
        [115] = {10, 10, 10, 10, 18, 16, 66, 66},
        [116] = {10, 10, 10, 10, 18, 18, 16, 18},
        [117] = {10, 10, 10, 10, 18, 18, 18, 18},
        [118] = {10, 10, 10, 10, 18, 18, 66, 66},
        [119] = {10, 10, 10, 10, 37, 10, 18, 18},
        [120] = {10, 10, 10, 10, 37, 37, 37, 37},
        [121] = {10, 10, 10, 10, 40, 43, 66, 66},
        [122] = {10, 10, 10, 10, 66, 10, 10, 10},
        [123] = {10, 10, 10, 10, 66, 10, 66, 10},
        [124] = {10, 10, 10, 10, 66, 10, 66, 66},
        [125] = {10, 10, 10, 10, 66, 66, 66, 66},
        [126] = {10, 10, 10, 16, 16, 16, 18, 18},
        [127] = {10, 10, 10, 16, 16, 18, 16, 16},
        [128] = {10, 10, 10, 16, 16, 18, 18, 16},
        [129] = {10, 10, 10, 16, 16, 18, 18, 18},
        [130] = {10, 10, 10, 16, 18, 18, 16, 16},
        [131] = {10, 10, 10, 18, 10, 10, 10, 10},
        [132] = {10, 10, 10, 18, 16, 16, 18, 18},
        [133] = {10, 10, 10, 18, 16, 18, 16, 16},
        [134] = {10, 10, 10, 18, 18, 10, 16, 16},
        [135] = {10, 10, 10, 18, 18, 16, 16, 37},
        [136] = {10, 10, 10, 18, 18, 18, 18, 18},
        [137] = {10, 10, 10, 18, 37, 37, 10, 66},
        [138] = {10, 10, 10, 37, 37, 37, 25, 25},
        [139] = {10, 10, 10, 37, 37, 37, 37, 37},
        [140] = {10, 10, 10, 37, 37, 66, 66, 66},
        [141] = {10, 10, 10, 38, 34, 66, 66, 66},
        [142] = {10, 10, 10, 66, 10, 10, 10, 10},
        [143] = {10, 10, 10, 66, 10, 10, 66, 10},
        [144] = {10, 10, 10, 66, 10, 10, 66, 66},
        [145] = {10, 10, 10, 66, 66, 18, 18, 18},
        [146] = {10, 10, 10, 66, 66, 66, 10, 10},
        [147] = {10, 10, 10, 66, 66, 66, 66, 10},
        [148] = {10, 10, 10, 66, 66, 66, 66, 18},
        [149] = {10, 10, 10, 66, 66, 66, 66, 66},
        [150] = {10, 10, 16, 16, 16, 16, 16, 66},
        [151] = {10, 10, 16, 16, 66, 66, 18, 18},
        [152] = {10, 10, 18, 10, 10, 10, 18, 10},
        [153] = {10, 10, 18, 16, 16, 18, 18, 16},
        [154] = {10, 10, 18, 16, 18, 10, 16, 16},
        [155] = {10, 10, 18, 18, 18, 18, 18, 18},
        [156] = {10, 10, 18, 18, 18, 37, 37, 66},
        [157] = {10, 10, 18, 18, 18, 66, 66, 66},
        [158] = {10, 10, 18, 18, 37, 37, 24, 24},
        [159] = {10, 10, 18, 18, 66, 66, 24, 24},
        [160] = {10, 10, 18, 18, 66, 66, 66, 66},
        [161] = {10, 10, 25, 66, 66, 66, 66, 66},
        [162] = {10, 10, 40, 40, 26, 26, 26, 26},
        [163] = {10, 10, 41, 41, 41, 41, 41, 41},
        [164] = {10, 10, 66, 10, 10, 10, 10, 10},
        [165] = {10, 10, 66, 10, 10, 66, 10, 10},
        [166] = {10, 10, 66, 18, 18, 10, 16, 16},
        [167] = {10, 10, 66, 66, 18, 10, 16, 16},
        [168] = {10, 10, 66, 66, 18, 10, 16, 18},
        [169] = {10, 10, 66, 66, 18, 66, 16, 16},
        [170] = {10, 10, 66, 66, 43, 18, 18, 37},
        [171] = {10, 10, 66, 66, 66, 10, 18, 18},
        [172] = {10, 10, 66, 66, 66, 66, 16, 16},
        [173] = {10, 10, 66, 66, 66, 66, 66, 37},
        [174] = {10, 10, 66, 66, 66, 66, 66, 66},
        [175] = {10, 16, 16, 16, 16, 16, 16, 16},
        [176] = {10, 16, 16, 16, 18, 18, 18, 18},
        [177] = {10, 16, 18, 18, 18, 18, 16, 16},
        [178] = {10, 18, 10, 10, 10, 10, 10, 10},
        [179] = {10, 18, 10, 10, 18, 18, 18, 18},
        [180] = {10, 18, 10, 66, 66, 66, 66, 66},
        [181] = {10, 18, 16, 16, 37, 10, 10, 10},
        [182] = {10, 18, 16, 16, 66, 10, 10, 10},
        [183] = {10, 18, 18, 18, 18, 10, 10, 10},
        [184] = {10, 18, 18, 18, 18, 18, 18, 16},
        [185] = {10, 18, 18, 18, 18, 18, 18, 18},
        [186] = {10, 18, 18, 18, 18, 18, 18, 66},
        [187] = {10, 18, 18, 18, 66, 18, 18, 66},
        [188] = {10, 18, 18, 18, 66, 66, 37, 66},
        [189] = {10, 25, 10, 10, 10, 10, 10, 10},
        [190] = {10, 37, 37, 37, 37, 37, 66, 66},
        [191] = {10, 37, 66, 66, 66, 66, 66, 66},
        [192] = {10, 42, 10, 10, 10, 10, 10, 10},
        [193] = {10, 43, 43, 43, 37, 37, 37, 37},
        [194] = {10, 66, 10, 10, 10, 10, 10, 10},
        [195] = {10, 66, 10, 10, 10, 10, 66, 10},
        [196] = {10, 66, 10, 10, 10, 10, 66, 66},
        [197] = {10, 66, 10, 10, 66, 10, 10, 10},
        [198] = {10, 66, 10, 10, 66, 10, 10, 66},
        [199] = {10, 66, 10, 66, 66, 66, 10, 10},
        [200] = {10, 66, 18, 18, 66, 66, 66, 66},
        [201] = {10, 66, 66, 10, 10, 10, 10, 10},
        [202] = {10, 66, 66, 66, 10, 66, 66, 10},
        [203] = {10, 66, 66, 66, 66, 66, 66, 16},
        [204] = {10, 66, 66, 66, 66, 66, 66, 66},
        [205] = {11, 8, 12, 11, 8, 12, 8, 12},
        [206] = {11, 11, 11, 11, 11, 11, 11, 11},
        [207] = {12, 8, 8, 8, 12, 12, 8, 12},
        [208] = {12, 8, 8, 10, 12, 8, 8, 8},
        [209] = {12, 8, 12, 8, 8, 8, 8, 8},
        [210] = {12, 8, 12, 8, 8, 8, 12, 8},
        [211] = {12, 8, 12, 8, 8, 12, 8, 8},
        [212] = {12, 8, 12, 8, 8, 43, 43, 43},
        [213] = {12, 8, 12, 8, 9, 9, 18, 18},
        [214] = {12, 8, 12, 8, 9, 41, 12, 8},
        [215] = {12, 8, 12, 8, 12, 8, 8, 8},
        [216] = {12, 8, 12, 8, 12, 8, 10, 18},
        [217] = {12, 8, 12, 8, 12, 8, 12, 8},
        [218] = {12, 8, 12, 8, 12, 8, 12, 12},
        [219] = {12, 8, 12, 12, 12, 8, 8, 12},
        [220] = {12, 8, 12, 12, 12, 12, 12, 8},
        [221] = {12, 8, 43, 18, 18, 18, 18, 18},
        [222] = {12, 12, 8, 8, 8, 8, 8, 8},
        [223] = {12, 12, 8, 12, 8, 12, 8, 8},
        [224] = {12, 12, 8, 12, 8, 12, 8, 12},
        [225] = {12, 12, 8, 12, 12, 8, 12, 12},
        [226] = {12, 12, 12, 8, 43, 12, 43, 43},
        [227] = {12, 12, 12, 12, 8, 8, 8, 8},
        [228] = {12, 12, 12, 12, 8, 10, 10, 10},
        [229] = {12, 12, 12, 12, 11, 41, 8, 41},
        [230] = {12, 12, 12, 12, 11, 41, 41, 41},
        [231] = {12, 12, 12, 12, 11, 41, 41, 66},
        [232] = {12, 12, 12, 12, 12, 8, 12, 8},
        [233] = {12, 12, 12, 12, 12, 12, 8, 8},
        [234] = {12, 12, 12, 12, 12, 12, 12, 8},
        [235] = {12, 12, 12, 12, 12, 12, 12, 12},
        [236] = {12, 12, 12, 12, 12, 12, 12, 42},
        [237] = {12, 12, 12, 12, 12, 12, 12, 66},
        [238] = {12, 12, 12, 12, 12, 12, 66, 12},
        [239] = {12, 12, 12, 12, 12, 12, 66, 66},
        [240] = {12, 12, 12, 12, 12, 41, 41, 41},
        [241] = {12, 12, 12, 12, 12, 42, 8, 8},
        [242] = {12, 12, 12, 12, 12, 66, 8, 8},
        [243] = {12, 12, 12, 12, 12, 66, 12, 12},
        [244] = {12, 12, 12, 12, 12, 66, 12, 66},
        [245] = {12, 12, 12, 12, 66, 41, 41, 41},
        [246] = {12, 12, 12, 12, 66, 66, 66, 66},
        [247] = {12, 12, 12, 38, 37, 34, 41, 32},
        [248] = {12, 12, 12, 42, 8, 8, 8, 8},
        [249] = {12, 12, 12, 66, 12, 66, 12, 12},
        [250] = {12, 12, 12, 66, 66, 12, 12, 12},
        [251] = {12, 12, 12, 66, 66, 66, 66, 66},
        [252] = {12, 12, 66, 12, 12, 12, 12, 12},
        [253] = {12, 12, 66, 12, 12, 12, 12, 66},
        [254] = {12, 42, 8, 8, 8, 8, 8, 8},
        [255] = {12, 43, 12, 12, 12, 12, 43, 8},
        [256] = {12, 66, 8, 8, 8, 8, 8, 8},
        [257] = {16, 10, 10, 10, 10, 37, 37, 37},
        [258] = {16, 16, 10, 10, 10, 10, 10, 10},
        [259] = {16, 16, 16, 1, 1, 1, 1, 1},
        [260] = {16, 16, 16, 16, 16, 16, 10, 10},
        [261] = {16, 16, 16, 16, 16, 16, 16, 16},
        [262] = {16, 16, 16, 16, 16, 16, 18, 16},
        [263] = {16, 16, 16, 16, 16, 18, 10, 16},
        [264] = {16, 16, 16, 16, 16, 66, 18, 16},
        [265] = {16, 16, 16, 16, 18, 18, 16, 18},
        [266] = {16, 16, 16, 16, 18, 18, 18, 18},
        [267] = {16, 16, 16, 16, 18, 18, 66, 66},
        [268] = {16, 16, 16, 18, 18, 18, 18, 16},
        [269] = {16, 16, 16, 18, 18, 18, 18, 18},
        [270] = {16, 16, 18, 16, 16, 10, 10, 10},
        [271] = {16, 16, 18, 16, 16, 16, 16, 16},
        [272] = {16, 16, 18, 18, 18, 16, 18, 10},
        [273] = {16, 16, 18, 18, 18, 18, 16, 18},
        [274] = {16, 16, 18, 18, 18, 18, 66, 66},
        [275] = {16, 16, 18, 18, 18, 66, 18, 66},
        [276] = {16, 18, 16, 10, 10, 10, 10, 10},
        [277] = {16, 18, 18, 16, 16, 18, 18, 10},
        [278] = {16, 18, 18, 16, 16, 18, 18, 66},
        [279] = {16, 18, 18, 18, 10, 10, 10, 10},
        [280] = {16, 18, 18, 18, 18, 18, 18, 18},
        [281] = {16, 18, 18, 18, 18, 18, 66, 18},
        [282] = {16, 18, 18, 18, 18, 66, 16, 16},
        [283] = {16, 18, 18, 18, 18, 66, 66, 16},
        [284] = {16, 18, 18, 18, 66, 66, 66, 66},
        [285] = {16, 18, 18, 37, 37, 1, 37, 37},
        [286] = {16, 18, 18, 37, 66, 66, 66, 66},
        [287] = {16, 18, 18, 66, 66, 66, 66, 18},
        [288] = {16, 37, 37, 37, 37, 37, 37, 37},
        [289] = {16, 66, 16, 16, 16, 18, 10, 43},
        [290] = {16, 66, 16, 16, 16, 18, 66, 66},
        [291] = {16, 66, 16, 16, 18, 18, 66, 66},
        [292] = {16, 66, 66, 16, 16, 16, 66, 66},
        [293] = {16, 66, 66, 16, 16, 18, 10, 66},
        [294] = {16, 66, 66, 16, 16, 18, 66, 66},
        [295] = {17, 17, 12, 8, 12, 8, 12, 8},
        [296] = {17, 17, 17, 37, 18, 18, 18, 18},
        [297] = {17, 18, 17, 17, 17, 18, 18, 18},
        [298] = {18, 10, 10, 10, 10, 10, 10, 10},
        [299] = {18, 10, 10, 10, 10, 10, 18, 18},
        [300] = {18, 10, 10, 10, 10, 18, 10, 10},
        [301] = {18, 10, 16, 16, 16, 10, 10, 16},
        [302] = {18, 10, 18, 18, 18, 10, 10, 18},
        [303] = {18, 10, 37, 10, 16, 66, 66, 66},
        [304] = {18, 10, 66, 66, 66, 66, 66, 66},
        [305] = {18, 16, 10, 18, 18, 18, 18, 37},
        [306] = {18, 16, 16, 16, 16, 18, 16, 16},
        [307] = {18, 16, 16, 16, 16, 66, 18, 18},
        [308] = {18, 16, 16, 16, 16, 66, 66, 16},
        [309] = {18, 16, 16, 16, 18, 10, 10, 10},
        [310] = {18, 16, 16, 16, 66, 66, 66, 66},
        [311] = {18, 16, 16, 18, 66, 66, 37, 37},
        [312] = {18, 16, 16, 37, 16, 18, 66, 66},
        [313] = {18, 16, 16, 66, 66, 66, 16, 16},
        [314] = {18, 16, 18, 16, 16, 16, 16, 18},
        [315] = {18, 16, 18, 16, 16, 18, 18, 18},
        [316] = {18, 16, 18, 18, 10, 10, 37, 10},
        [317] = {18, 16, 18, 18, 16, 18, 18, 66},
        [318] = {18, 16, 18, 18, 18, 18, 18, 18},
        [319] = {18, 16, 66, 16, 16, 18, 66, 66},
        [320] = {18, 18, 1, 18, 18, 18, 18, 18},
        [321] = {18, 18, 9, 18, 18, 18, 18, 18},
        [322] = {18, 18, 10, 10, 10, 10, 10, 10},
        [323] = {18, 18, 10, 10, 10, 10, 18, 18},
        [324] = {18, 18, 10, 10, 10, 18, 37, 66},
        [325] = {18, 18, 10, 66, 66, 66, 66, 66},
        [326] = {18, 18, 12, 8, 66, 66, 66, 66},
        [327] = {18, 18, 16, 10, 10, 10, 10, 10},
        [328] = {18, 18, 16, 16, 16, 18, 16, 18},
        [329] = {18, 18, 16, 16, 18, 16, 18, 18},
        [330] = {18, 18, 16, 16, 18, 18, 16, 16},
        [331] = {18, 18, 16, 16, 66, 10, 10, 10},
        [332] = {18, 18, 16, 16, 66, 66, 66, 66},
        [333] = {18, 18, 16, 18, 18, 18, 10, 10},
        [334] = {18, 18, 18, 9, 9, 9, 9, 9},
        [335] = {18, 18, 18, 9, 66, 66, 66, 66},
        [336] = {18, 18, 18, 10, 10, 10, 10, 10},
        [337] = {18, 18, 18, 16, 10, 10, 10, 10},
        [338] = {18, 18, 18, 16, 16, 16, 16, 18},
        [339] = {18, 18, 18, 16, 16, 18, 16, 18},
        [340] = {18, 18, 18, 16, 18, 16, 16, 16},
        [341] = {18, 18, 18, 18, 9, 9, 43, 37},
        [342] = {18, 18, 18, 18, 9, 18, 18, 18},
        [343] = {18, 18, 18, 18, 16, 10, 10, 10},
        [344] = {18, 18, 18, 18, 16, 16, 18, 18},
        [345] = {18, 18, 18, 18, 16, 18, 18, 18},
        [346] = {18, 18, 18, 18, 18, 1, 43, 18},
        [347] = {18, 18, 18, 18, 18, 9, 9, 18},
        [348] = {18, 18, 18, 18, 18, 10, 66, 66},
        [349] = {18, 18, 18, 18, 18, 16, 16, 16},
        [350] = {18, 18, 18, 18, 18, 17, 17, 17},
        [351] = {18, 18, 18, 18, 18, 18, 10, 18},
        [352] = {18, 18, 18, 18, 18, 18, 16, 16},
        [353] = {18, 18, 18, 18, 18, 18, 16, 18},
        [354] = {18, 18, 18, 18, 18, 18, 17, 66},
        [355] = {18, 18, 18, 18, 18, 18, 18, 9},
        [356] = {18, 18, 18, 18, 18, 18, 18, 16},
        [357] = {18, 18, 18, 18, 18, 18, 18, 18},
        [358] = {18, 18, 18, 18, 18, 18, 18, 37},
        [359] = {18, 18, 18, 18, 18, 18, 18, 43},
        [360] = {18, 18, 18, 18, 18, 18, 18, 66},
        [361] = {18, 18, 18, 18, 18, 18, 33, 18},
        [362] = {18, 18, 18, 18, 18, 18, 37, 9},
        [363] = {18, 18, 18, 18, 18, 18, 37, 37},
        [364] = {18, 18, 18, 18, 18, 18, 66, 66},
        [365] = {18, 18, 18, 18, 18, 37, 18, 18},
        [366] = {18, 18, 18, 18, 18, 37, 66, 66},
        [367] = {18, 18, 18, 18, 18, 43, 43, 43},
        [368] = {18, 18, 18, 18, 18, 66, 24, 24},
        [369] = {18, 18, 18, 18, 18, 66, 43, 43},
        [370] = {18, 18, 18, 18, 18, 66, 66, 18},
        [371] = {18, 18, 18, 18, 18, 66, 66, 66},
        [372] = {18, 18, 18, 18, 37, 37, 37, 9},
        [373] = {18, 18, 18, 18, 43, 43, 43, 43},
        [374] = {18, 18, 18, 18, 66, 66, 66, 66},
        [375] = {18, 18, 18, 37, 1, 66, 37, 37},
        [376] = {18, 18, 18, 37, 18, 18, 18, 18},
        [377] = {18, 18, 18, 43, 43, 18, 18, 18},
        [378] = {18, 18, 18, 66, 66, 10, 10, 10},
        [379] = {18, 18, 18, 66, 66, 66, 66, 18},
        [380] = {18, 18, 18, 66, 66, 66, 66, 40},
        [381] = {18, 18, 18, 66, 66, 66, 66, 66},
        [382] = {18, 18, 37, 37, 37, 10, 37, 37},
        [383] = {18, 18, 37, 37, 37, 37, 37, 37},
        [384] = {18, 18, 43, 43, 43, 16, 16, 16},
        [385] = {18, 18, 43, 43, 43, 43, 43, 43},
        [386] = {18, 18, 66, 16, 16, 18, 16, 18},
        [387] = {18, 18, 66, 18, 18, 18, 18, 18},
        [388] = {18, 18, 66, 18, 18, 66, 18, 18},
        [389] = {18, 26, 26, 26, 26, 26, 26, 26},
        [390] = {18, 26, 26, 26, 26, 37, 37, 37},
        [391] = {18, 37, 37, 37, 10, 66, 66, 66},
        [392] = {18, 37, 37, 37, 37, 37, 37, 37},
        [393] = {18, 43, 18, 18, 18, 18, 10, 10},
        [394] = {18, 66, 18, 18, 18, 18, 66, 66},
        [395] = {18, 66, 66, 18, 18, 18, 18, 18},
        [396] = {18, 66, 66, 18, 18, 18, 66, 66},
        [397] = {18, 66, 66, 66, 66, 66, 66, 66},
        [398] = {24, 24, 10, 10, 10, 10, 10, 10},
        [399] = {24, 24, 10, 10, 10, 10, 10, 66},
        [400] = {24, 24, 10, 10, 10, 43, 43, 10},
        [401] = {24, 24, 10, 10, 66, 66, 66, 66},
        [402] = {24, 24, 10, 37, 10, 37, 37, 37},
        [403] = {24, 24, 16, 16, 16, 18, 43, 43},
        [404] = {24, 24, 24, 24, 24, 24, 24, 24},
        [405] = {24, 24, 26, 26, 26, 26, 26, 26},
        [406] = {24, 24, 26, 26, 37, 37, 37, 43},
        [407] = {24, 24, 26, 66, 66, 66, 43, 43},
        [408] = {24, 24, 37, 37, 37, 37, 10, 10},
        [409] = {24, 24, 37, 37, 37, 37, 37, 37},
        [410] = {24, 24, 37, 37, 42, 42, 42, 37},
        [411] = {24, 24, 37, 37, 66, 66, 66, 66},
        [412] = {24, 24, 66, 26, 26, 26, 26, 26},
        [413] = {24, 24, 66, 37, 66, 37, 18, 10},
        [414] = {24, 24, 66, 66, 10, 10, 10, 10},
        [415] = {24, 24, 66, 66, 37, 37, 37, 37},
        [416] = {24, 24, 66, 66, 66, 10, 10, 10},
        [417] = {24, 24, 66, 66, 66, 66, 10, 43},
        [418] = {24, 24, 66, 66, 66, 66, 37, 37},
        [419] = {24, 24, 66, 66, 66, 66, 66, 40},
        [420] = {24, 24, 66, 66, 66, 66, 66, 66},
        [421] = {25, 10, 10, 10, 10, 10, 10, 10},
        [422] = {25, 25, 25, 12, 8, 25, 25, 25},
        [423] = {25, 25, 25, 25, 25, 25, 25, 25},
        [424] = {25, 25, 25, 25, 25, 25, 25, 66},
        [425] = {25, 25, 25, 25, 25, 26, 26, 26},
        [426] = {25, 26, 43, 43, 66, 66, 66, 66},
        [427] = {26, 9, 66, 66, 26, 26, 26, 26},
        [428] = {26, 26, 26, 26, 26, 26, 24, 24},
        [429] = {26, 26, 26, 26, 26, 26, 26, 10},
        [430] = {26, 26, 26, 26, 26, 26, 26, 26},
        [431] = {26, 26, 26, 26, 26, 26, 26, 37},
        [432] = {26, 26, 26, 26, 26, 26, 26, 43},
        [433] = {26, 26, 26, 26, 26, 26, 26, 66},
        [434] = {26, 26, 26, 26, 26, 26, 43, 26},
        [435] = {26, 26, 26, 26, 26, 26, 43, 43},
        [436] = {26, 26, 26, 26, 26, 26, 66, 66},
        [437] = {26, 26, 26, 26, 26, 43, 43, 43},
        [438] = {26, 26, 26, 26, 26, 66, 66, 66},
        [439] = {26, 26, 26, 26, 43, 18, 43, 18},
        [440] = {26, 26, 26, 26, 43, 26, 26, 26},
        [441] = {26, 26, 26, 26, 43, 43, 43, 43},
        [442] = {26, 26, 26, 26, 43, 107, 107, 107},
        [443] = {26, 26, 26, 26, 66, 66, 66, 37},
        [444] = {26, 26, 26, 26, 66, 66, 66, 43},
        [445] = {26, 26, 26, 26, 66, 66, 66, 66},
        [446] = {26, 26, 26, 43, 43, 43, 43, 43},
        [447] = {26, 26, 26, 66, 66, 66, 66, 66},
        [448] = {26, 26, 42, 42, 42, 38, 34, 9},
        [449] = {26, 26, 42, 42, 42, 38, 34, 66},
        [450] = {26, 26, 43, 40, 10, 37, 18, 66},
        [451] = {26, 26, 66, 10, 10, 10, 10, 10},
        [452] = {26, 26, 66, 66, 66, 66, 66, 66},
        [453] = {26, 43, 10, 10, 10, 10, 10, 10},
        [454] = {26, 43, 43, 43, 43, 43, 43, 43},
        [455] = {26, 66, 66, 66, 66, 66, 66, 66},
        [456] = {32, 37, 37, 37, 42, 38, 34, 37},
        [457] = {33, 10, 10, 10, 10, 10, 10, 10},
        [458] = {33, 33, 33, 33, 33, 33, 37, 37},
        [459] = {33, 37, 38, 37, 37, 37, 37, 37},
        [460] = {34, 38, 34, 38, 34, 38, 34, 38},
        [461] = {34, 42, 42, 42, 42, 42, 42, 42},
        [462] = {36, 35, 38, 34, 38, 34, 38, 34},
        [463] = {36, 35, 38, 36, 36, 35, 38, 36},
        [464] = {37, 9, 10, 10, 10, 10, 10, 10},
        [465] = {37, 12, 12, 12, 12, 12, 12, 12},
        [466] = {37, 18, 18, 18, 18, 37, 66, 66},
        [467] = {37, 18, 18, 37, 18, 18, 37, 18},
        [468] = {37, 25, 25, 25, 25, 25, 66, 66},
        [469] = {37, 26, 26, 26, 26, 26, 26, 26},
        [470] = {37, 36, 35, 37, 36, 35, 37, 37},
        [471] = {37, 36, 35, 37, 37, 37, 37, 32},
        [472] = {37, 37, 9, 66, 66, 18, 40, 40},
        [473] = {37, 37, 10, 9, 9, 16, 18, 66},
        [474] = {37, 37, 10, 10, 10, 10, 10, 10},
        [475] = {37, 37, 33, 33, 37, 37, 37, 37},
        [476] = {37, 37, 33, 37, 36, 35, 37, 37},
        [477] = {37, 37, 36, 35, 36, 35, 37, 37},
        [478] = {37, 37, 37, 10, 37, 10, 10, 18},
        [479] = {37, 37, 37, 18, 18, 18, 1, 66},
        [480] = {37, 37, 37, 37, 10, 16, 16, 66},
        [481] = {37, 37, 37, 37, 37, 37, 18, 66},
        [482] = {37, 37, 37, 37, 37, 37, 33, 37},
        [483] = {37, 37, 37, 37, 37, 37, 37, 9},
        [484] = {37, 37, 37, 37, 37, 37, 37, 18},
        [485] = {37, 37, 37, 37, 37, 37, 37, 33},
        [486] = {37, 37, 37, 37, 37, 37, 37, 37},
        [487] = {37, 37, 37, 37, 37, 37, 37, 50},
        [488] = {37, 37, 37, 37, 37, 37, 37, 66},
        [489] = {37, 37, 37, 37, 37, 37, 66, 1},
        [490] = {37, 37, 37, 37, 37, 37, 66, 9},
        [491] = {37, 37, 37, 37, 37, 37, 66, 66},
        [492] = {37, 37, 37, 37, 37, 43, 43, 43},
        [493] = {37, 37, 37, 37, 37, 66, 66, 66},
        [494] = {37, 37, 37, 37, 43, 43, 43, 43},
        [495] = {37, 37, 37, 37, 66, 66, 66, 66},
        [496] = {37, 37, 37, 40, 10, 18, 66, 66},
        [497] = {37, 37, 37, 43, 37, 43, 43, 43},
        [498] = {37, 37, 37, 66, 66, 66, 66, 26},
        [499] = {37, 37, 37, 66, 66, 66, 66, 66},
        [500] = {37, 37, 42, 37, 32, 37, 37, 37},
        [501] = {37, 37, 66, 66, 66, 66, 66, 66},
        [502] = {37, 40, 66, 66, 66, 66, 66, 66},
        [503] = {37, 43, 43, 43, 43, 43, 43, 43},
        [504] = {37, 66, 66, 66, 66, 66, 66, 66},
        [505] = {38, 34, 37, 37, 37, 37, 37, 9},
        [506] = {38, 34, 37, 42, 37, 33, 37, 37},
        [507] = {38, 34, 38, 34, 38, 34, 26, 26},
        [508] = {38, 34, 38, 34, 38, 34, 38, 34},
        [509] = {38, 34, 38, 34, 42, 42, 42, 42},
        [510] = {38, 34, 38, 34, 43, 43, 43, 43},
        [511] = {40, 26, 26, 26, 26, 66, 66, 66},
        [512] = {40, 40, 40, 40, 40, 40, 40, 40},
        [513] = {40, 43, 43, 43, 43, 43, 43, 43},
        [514] = {40, 43, 66, 66, 66, 66, 66, 66},
        [515] = {41, 8, 8, 8, 8, 8, 8, 8},
        [516] = {41, 26, 10, 35, 26, 26, 26, 37},
        [517] = {41, 41, 8, 8, 8, 66, 8, 8},
        [518] = {41, 41, 12, 8, 12, 8, 12, 8},
        [519] = {41, 41, 41, 41, 9, 41, 9, 41},
        [520] = {41, 41, 41, 41, 41, 41, 41, 9},
        [521] = {41, 41, 41, 41, 41, 41, 41, 41},
        [522] = {41, 41, 66, 66, 66, 66, 66, 66},
        [523] = {41, 43, 10, 36, 42, 1, 43, 41},
        [524] = {42, 12, 12, 12, 12, 12, 43, 43},
        [525] = {42, 37, 37, 40, 37, 37, 43, 43},
        [526] = {42, 42, 42, 38, 34, 38, 34, 38},
        [527] = {42, 42, 42, 42, 38, 34, 42, 42},
        [528] = {42, 42, 42, 42, 42, 12, 8, 8},
        [529] = {42, 42, 42, 42, 42, 38, 34, 42},
        [530] = {42, 42, 42, 42, 42, 42, 38, 34},
        [531] = {42, 42, 42, 42, 42, 42, 42, 42},
        [532] = {42, 42, 42, 42, 42, 43, 43, 42},
        [533] = {42, 42, 42, 42, 42, 43, 43, 43},
        [534] = {42, 42, 42, 42, 42, 106, 106, 42},
        [535] = {42, 42, 42, 42, 43, 43, 43, 43},
        [536] = {42, 42, 43, 43, 43, 43, 43, 43},
        [537] = {42, 42, 66, 66, 66, 66, 66, 66},
        [538] = {42, 43, 43, 42, 43, 43, 42, 43},
        [539] = {43, 10, 10, 10, 10, 10, 10, 10},
        [540] = {43, 10, 26, 26, 26, 26, 26, 26},
        [541] = {43, 18, 38, 34, 38, 34, 16, 16},
        [542] = {43, 26, 26, 26, 26, 26, 26, 26},
        [543] = {43, 37, 37, 66, 66, 66, 66, 66},
        [544] = {43, 40, 43, 66, 66, 66, 66, 66},
        [545] = {43, 42, 26, 26, 41, 8, 37, 37},
        [546] = {43, 42, 42, 42, 42, 43, 43, 66},
        [547] = {43, 42, 43, 43, 43, 43, 43, 43},
        [548] = {43, 43, 8, 12, 12, 12, 8, 8},
        [549] = {43, 43, 10, 16, 18, 16, 10, 10},
        [550] = {43, 43, 12, 43, 43, 43, 43, 12},
        [551] = {43, 43, 18, 18, 18, 18, 43, 43},
        [552] = {43, 43, 18, 18, 18, 43, 66, 66},
        [553] = {43, 43, 26, 26, 26, 26, 26, 26},
        [554] = {43, 43, 26, 26, 43, 43, 43, 66},
        [555] = {43, 43, 42, 42, 43, 43, 43, 43},
        [556] = {43, 43, 42, 43, 42, 43, 43, 43},
        [557] = {43, 43, 43, 12, 8, 12, 8, 18},
        [558] = {43, 43, 43, 18, 18, 18, 18, 18},
        [559] = {43, 43, 43, 42, 42, 42, 42, 42},
        [560] = {43, 43, 43, 43, 12, 43, 12, 43},
        [561] = {43, 43, 43, 43, 18, 43, 43, 37},
        [562] = {43, 43, 43, 43, 42, 42, 42, 42},
        [563] = {43, 43, 43, 43, 42, 43, 43, 43},
        [564] = {43, 43, 43, 43, 43, 16, 16, 18},
        [565] = {43, 43, 43, 43, 43, 18, 43, 43},
        [566] = {43, 43, 43, 43, 43, 18, 66, 66},
        [567] = {43, 43, 43, 43, 43, 40, 40, 40},
        [568] = {43, 43, 43, 43, 43, 43, 18, 43},
        [569] = {43, 43, 43, 43, 43, 43, 42, 42},
        [570] = {43, 43, 43, 43, 43, 43, 42, 43},
        [571] = {43, 43, 43, 43, 43, 43, 43, 42},
        [572] = {43, 43, 43, 43, 43, 43, 43, 43},
        [573] = {43, 43, 43, 43, 43, 43, 43, 66},
        [574] = {43, 43, 43, 43, 43, 43, 43, 107},
        [575] = {43, 43, 43, 43, 43, 43, 66, 66},
        [576] = {43, 43, 43, 43, 43, 43, 107, 43},
        [577] = {43, 43, 43, 43, 43, 66, 43, 43},
        [578] = {43, 43, 43, 43, 43, 66, 66, 66},
        [579] = {43, 43, 43, 43, 43, 107, 43, 43},
        [580] = {43, 43, 43, 43, 43, 107, 107, 43},
        [581] = {43, 43, 43, 43, 43, 107, 107, 107},
        [582] = {43, 43, 43, 43, 66, 66, 43, 43},
        [583] = {43, 43, 43, 43, 66, 66, 66, 66},
        [584] = {43, 43, 43, 43, 66, 107, 107, 107},
        [585] = {43, 43, 43, 43, 107, 43, 43, 43},
        [586] = {43, 43, 43, 43, 107, 43, 107, 43},
        [587] = {43, 43, 43, 43, 107, 107, 43, 43},
        [588] = {43, 43, 43, 43, 107, 107, 107, 107},
        [589] = {43, 43, 43, 66, 66, 66, 66, 66},
        [590] = {43, 43, 43, 107, 43, 43, 43, 43},
        [591] = {43, 43, 43, 107, 107, 43, 43, 43},
        [592] = {43, 43, 43, 107, 107, 66, 66, 66},
        [593] = {43, 43, 43, 107, 107, 107, 43, 107},
        [594] = {43, 43, 43, 107, 107, 107, 107, 43},
        [595] = {43, 43, 43, 107, 107, 107, 107, 107},
        [596] = {43, 43, 66, 66, 66, 66, 66, 66},
        [597] = {43, 43, 107, 43, 43, 43, 43, 43},
        [598] = {43, 43, 107, 43, 43, 107, 43, 43},
        [599] = {43, 43, 107, 107, 43, 43, 43, 43},
        [600] = {43, 43, 107, 107, 43, 107, 43, 43},
        [601] = {43, 66, 66, 66, 37, 37, 24, 24},
        [602] = {43, 66, 66, 66, 66, 66, 66, 66},
        [603] = {43, 102, 98, 43, 43, 43, 43, 43},
        [604] = {43, 107, 43, 43, 43, 43, 43, 43},
        [605] = {43, 107, 107, 107, 107, 43, 43, 43},
        [606] = {43, 107, 107, 107, 107, 107, 107, 107},
        [607] = {48, 49, 1, 1, 1, 1, 1, 50},
        [608] = {50, 10, 10, 10, 10, 10, 10, 10},
        [609] = {50, 37, 37, 37, 40, 37, 37, 37},
        [610] = {50, 37, 40, 40, 40, 40, 43, 37},
        [611] = {50, 50, 50, 1, 1, 1, 1, 1},
        [612] = {50, 50, 50, 50, 50, 50, 50, 50},
        [613] = {66, 1, 1, 1, 43, 43, 66, 66},
        [614] = {66, 1, 66, 66, 66, 66, 66, 66},
        [615] = {66, 9, 37, 37, 37, 37, 37, 37},
        [616] = {66, 10, 10, 10, 10, 10, 10, 10},
        [617] = {66, 10, 10, 10, 10, 10, 10, 66},
        [618] = {66, 10, 10, 10, 10, 66, 10, 66},
        [619] = {66, 10, 10, 10, 66, 10, 10, 10},
        [620] = {66, 10, 10, 66, 10, 66, 10, 10},
        [621] = {66, 10, 10, 66, 10, 66, 66, 10},
        [622] = {66, 10, 10, 66, 66, 66, 66, 66},
        [623] = {66, 10, 18, 18, 18, 18, 18, 18},
        [624] = {66, 10, 66, 10, 66, 10, 10, 10},
        [625] = {66, 10, 66, 10, 66, 10, 66, 10},
        [626] = {66, 10, 66, 10, 66, 66, 66, 66},
        [627] = {66, 12, 12, 12, 12, 12, 12, 12},
        [628] = {66, 12, 12, 12, 12, 66, 12, 12},
        [629] = {66, 12, 66, 12, 66, 12, 66, 12},
        [630] = {66, 16, 18, 18, 18, 18, 18, 18},
        [631] = {66, 18, 16, 16, 66, 10, 10, 10},
        [632] = {66, 18, 18, 16, 66, 10, 10, 10},
        [633] = {66, 18, 18, 18, 18, 18, 18, 18},
        [634] = {66, 18, 66, 66, 66, 66, 66, 66},
        [635] = {66, 26, 26, 26, 26, 26, 26, 26},
        [636] = {66, 37, 37, 37, 37, 26, 37, 37},
        [637] = {66, 37, 37, 37, 37, 37, 37, 37},
        [638] = {66, 37, 37, 37, 37, 66, 66, 66},
        [639] = {66, 43, 43, 43, 43, 43, 43, 43},
        [640] = {66, 66, 8, 8, 8, 66, 8, 8},
        [641] = {66, 66, 9, 8, 8, 8, 37, 12},
        [642] = {66, 66, 10, 10, 10, 10, 10, 10},
        [643] = {66, 66, 10, 10, 10, 66, 66, 66},
        [644] = {66, 66, 10, 66, 66, 66, 66, 10},
        [645] = {66, 66, 12, 8, 12, 12, 12, 66},
        [646] = {66, 66, 12, 12, 12, 12, 12, 12},
        [647] = {66, 66, 12, 66, 66, 12, 12, 66},
        [648] = {66, 66, 16, 16, 37, 66, 66, 66},
        [649] = {66, 66, 16, 16, 66, 10, 10, 10},
        [650] = {66, 66, 18, 10, 66, 10, 10, 10},
        [651] = {66, 66, 18, 18, 16, 16, 16, 16},
        [652] = {66, 66, 18, 18, 18, 18, 18, 18},
        [653] = {66, 66, 18, 66, 18, 18, 66, 18},
        [654] = {66, 66, 18, 66, 66, 66, 66, 16},
        [655] = {66, 66, 26, 26, 26, 26, 26, 26},
        [656] = {66, 66, 66, 8, 8, 8, 8, 8},
        [657] = {66, 66, 66, 10, 10, 9, 37, 37},
        [658] = {66, 66, 66, 10, 10, 10, 10, 10},
        [659] = {66, 66, 66, 10, 10, 66, 66, 66},
        [660] = {66, 66, 66, 18, 18, 18, 18, 18},
        [661] = {66, 66, 66, 26, 26, 26, 26, 26},
        [662] = {66, 66, 66, 37, 37, 37, 37, 37},
        [663] = {66, 66, 66, 66, 10, 10, 10, 16},
        [664] = {66, 66, 66, 66, 10, 10, 66, 10},
        [665] = {66, 66, 66, 66, 18, 18, 18, 18},
        [666] = {66, 66, 66, 66, 26, 26, 10, 10},
        [667] = {66, 66, 66, 66, 37, 37, 37, 37},
        [668] = {66, 66, 66, 66, 41, 41, 12, 37},
        [669] = {66, 66, 66, 66, 66, 1, 66, 66},
        [670] = {66, 66, 66, 66, 66, 8, 66, 66},
        [671] = {66, 66, 66, 66, 66, 10, 10, 10},
        [672] = {66, 66, 66, 66, 66, 10, 18, 10},
        [673] = {66, 66, 66, 66, 66, 12, 66, 66},
        [674] = {66, 66, 66, 66, 66, 16, 16, 66},
        [675] = {66, 66, 66, 66, 66, 18, 18, 66},
        [676] = {66, 66, 66, 66, 66, 66, 10, 66},
        [677] = {66, 66, 66, 66, 66, 66, 18, 16},
        [678] = {66, 66, 66, 66, 66, 66, 24, 24},
        [679] = {66, 66, 66, 66, 66, 66, 37, 37},
        [680] = {66, 66, 66, 66, 66, 66, 43, 43},
        [681] = {66, 66, 66, 66, 66, 66, 66, 9},
        [682] = {66, 66, 66, 66, 66, 66, 66, 10},
        [683] = {66, 66, 66, 66, 66, 66, 66, 16},
        [684] = {66, 66, 66, 66, 66, 66, 66, 18},
        [685] = {66, 66, 66, 66, 66, 66, 66, 26},
        [686] = {66, 66, 66, 66, 66, 66, 66, 37},
        [687] = {66, 66, 66, 66, 66, 66, 66, 66},
        [688] = {66, 66, 66, 66, 66, 74, 74, 74},
        [689] = {66, 66, 66, 66, 74, 74, 74, 74},
        [690] = {66, 66, 107, 107, 107, 107, 107, 107},
        [691] = {66, 74, 74, 74, 74, 74, 74, 74},
        [692] = {66, 82, 82, 105, 105, 73, 73, 74},
        [693] = {66, 101, 101, 101, 104, 101, 101, 101},
        [694] = {72, 72, 72, 72, 72, 72, 72, 72},
        [695] = {72, 72, 72, 102, 106, 98, 106, 102},
        [696] = {73, 73, 101, 73, 66, 66, 66, 66},
        [697] = {74, 66, 66, 66, 66, 66, 66, 66},
        [698] = {74, 74, 66, 66, 66, 66, 66, 66},
        [699] = {74, 74, 74, 66, 66, 66, 66, 66},
        [700] = {74, 74, 74, 74, 66, 66, 66, 66},
        [701] = {74, 74, 74, 74, 74, 66, 66, 66},
        [702] = {74, 74, 74, 74, 74, 73, 74, 74},
        [703] = {74, 74, 74, 74, 74, 74, 66, 66},
        [704] = {74, 74, 74, 74, 74, 74, 74, 66},
        [705] = {74, 74, 74, 74, 74, 74, 74, 74},
        [706] = {74, 74, 74, 101, 73, 73, 73, 74},
        [707] = {76, 76, 76, 76, 76, 76, 76, 76},
        [708] = {76, 76, 76, 102, 101, 98, 105, 96},
        [709] = {88, 88, 88, 88, 88, 88, 88, 88},
        [710] = {88, 88, 101, 101, 106, 106, 106, 101},
        [711] = {89, 89, 82, 82, 82, 82, 80, 80},
        [712] = {89, 89, 89, 73, 74, 101, 107, 43},
        [713] = {90, 90, 90, 90, 90, 90, 90, 90},
        [714] = {90, 90, 107, 107, 107, 107, 107, 107},
        [715] = {97, 73, 73, 73, 73, 73, 107, 107},
        [716] = {97, 74, 74, 74, 74, 74, 74, 74},
        [717] = {97, 102, 98, 102, 98, 102, 98, 101},
        [718] = {98, 37, 38, 34, 37, 37, 10, 10},
        [719] = {98, 101, 66, 66, 66, 66, 66, 66},
        [720] = {98, 101, 101, 101, 101, 96, 96, 96},
        [721] = {98, 102, 98, 102, 98, 101, 101, 102},
        [722] = {98, 102, 98, 102, 98, 102, 98, 102},
        [723] = {101, 76, 76, 76, 76, 76, 76, 76},
        [724] = {101, 97, 97, 96, 96, 102, 98, 102},
        [725] = {101, 101, 101, 66, 101, 101, 101, 101},
        [726] = {101, 101, 101, 101, 101, 101, 101, 102},
        [727] = {101, 101, 106, 97, 106, 106, 106, 66},
        [728] = {101, 104, 101, 101, 66, 66, 66, 66},
        [729] = {102, 98, 101, 106, 101, 97, 101, 101},
        [730] = {102, 98, 102, 98, 97, 102, 98, 98},
        [731] = {102, 98, 102, 98, 102, 98, 102, 98},
        [732] = {102, 98, 107, 107, 102, 98, 102, 98},
        [733] = {104, 104, 106, 105, 107, 104, 104, 66},
        [734] = {105, 72, 72, 72, 72, 72, 72, 72},
        [735] = {107, 43, 43, 43, 43, 43, 43, 43},
        [736] = {107, 43, 43, 43, 43, 107, 43, 43},
        [737] = {107, 43, 43, 43, 107, 43, 43, 43},
        [738] = {107, 43, 43, 107, 43, 43, 43, 43},
        [739] = {107, 43, 107, 107, 107, 107, 107, 107},
        [740] = {107, 66, 66, 66, 66, 66, 66, 66},
        [741] = {107, 89, 89, 89, 89, 89, 89, 89},
        [742] = {107, 90, 90, 90, 90, 90, 90, 90},
        [743] = {107, 107, 66, 66, 66, 66, 66, 66},
        [744] = {107, 107, 66, 107, 107, 107, 107, 66},
        [745] = {107, 107, 66, 107, 107, 107, 107, 107},
        [746] = {107, 107, 90, 90, 90, 90, 107, 107},
        [747] = {107, 107, 107, 43, 43, 43, 43, 43},
        [748] = {107, 107, 107, 43, 43, 43, 43, 107},
        [749] = {107, 107, 107, 43, 43, 107, 66, 66},
        [750] = {107, 107, 107, 66, 66, 66, 66, 66},
        [751] = {107, 107, 107, 66, 66, 66, 107, 107},
        [752] = {107, 107, 107, 66, 66, 107, 107, 107},
        [753] = {107, 107, 107, 105, 105, 105, 105, 105},
        [754] = {107, 107, 107, 107, 43, 43, 43, 43},
        [755] = {107, 107, 107, 107, 66, 66, 66, 66},
        [756] = {107, 107, 107, 107, 107, 43, 43, 107},
        [757] = {107, 107, 107, 107, 107, 43, 107, 107},
        [758] = {107, 107, 107, 107, 107, 107, 43, 43},
        [759] = {107, 107, 107, 107, 107, 107, 43, 107},
        [760] = {107, 107, 107, 107, 107, 107, 66, 66},
        [761] = {107, 107, 107, 107, 107, 107, 107, 43},
        [762] = {107, 107, 107, 107, 107, 107, 107, 66},
        [763] = {107, 107, 107, 107, 107, 107, 107, 107},
        [764] = {114, 101, 101, 101, 107, 73, 74, 89},
};
static const uint16_t data_1[][8] = {
        [0]   = {0, 0, 0, 0, 609, 506, 404, 410},
        [1]   = {0, 0, 0, 0, 610, 523, 545, 516},
        [2]   = {1, 1, 1, 1, 1, 1, 1, 1},
        [3]   = {2, 525, 357, 375, 80, 80, 80, 80},
        [4]   = {7, 7, 7, 7, 7, 7, 7, 7},
        [5]   = {7, 7, 7, 7, 7, 7, 7, 8},
        [6]   = {9, 9, 9, 9, 9, 9, 9, 9},
        [7]   = {10, 10, 10, 10, 10, 10, 10, 10},
        [8]   = {10, 10, 10, 10, 10, 10, 36, 655},
        [9]   = {10, 10, 10, 10, 10, 20, 63, 63},
        [10]  = {10, 10, 10, 10, 16, 670, 80, 80},
        [11]  = {10, 10, 10, 10, 217, 217, 217, 217},
        [12]  = {10, 10, 10, 10, 404, 405, 447, 682},
        [13]  = {10, 10, 10, 13, 219, 49, 211, 21},
        [14]  = {10, 10, 10, 31, 63, 63, 63, 63},
        [15]  = {10, 10, 10, 34, 10, 687, 10, 10},
        [16]  = {10, 10, 18, 24, 235, 235, 241, 10},
        [17]  = {10, 10, 22, 10, 10, 10, 63, 63},
        [18]  = {10, 10, 24, 235, 235, 233, 10, 10},
        [19]  = {10, 10, 80, 80, 80, 80, 80, 80},
        [20]  = {10, 11, 40, 217, 217, 217, 23, 52},
        [21]  = {10, 12, 15, 235, 235, 236, 10, 10},
        [22]  = {10, 57, 633, 357, 357, 357, 357, 361},
        [23]  = {10, 58, 235, 235, 235, 254, 10, 10},
        [24]  = {10, 59, 235, 235, 235, 235, 235, 250},
        [25]  = {10, 206, 10, 206, 10, 206, 19, 229},
        [26]  = {10, 235, 17, 239, 10, 235, 10, 235},
        [27]  = {10, 235, 235, 235, 222, 10, 10, 24},
        [28]  = {13, 687, 656, 672, 80, 192, 84, 105},
        [29]  = {17, 239, 10, 629, 10, 235, 10, 17},
        [30]  = {25, 250, 243, 242, 10, 10, 10, 253},
        [31]  = {27, 335, 404, 418, 687, 687, 687, 687},
        [32]  = {28, 10, 235, 235, 235, 222, 10, 10},
        [33]  = {35, 38, 404, 404, 404, 404, 404, 404},
        [34]  = {47, 37, 217, 217, 217, 217, 217, 223},
        [35]  = {50, 217, 10, 10, 10, 10, 10, 10},
        [36]  = {53, 54, 225, 207, 218, 44, 55, 208},
        [37]  = {63, 63, 63, 63, 63, 62, 10, 60},
        [38]  = {68, 687, 404, 412, 451, 80, 80, 671},
        [39]  = {69, 63, 70, 521, 66, 519, 521, 521},
        [40]  = {71, 136, 357, 357, 404, 408, 298, 80},
        [41]  = {76, 80, 80, 80, 80, 80, 80, 204},
        [42]  = {79, 358, 404, 411, 687, 687, 687, 687},
        [43]  = {80, 77, 80, 80, 404, 401, 687, 687},
        [44]  = {80, 80, 80, 78, 80, 80, 80, 84},
        [45]  = {80, 80, 80, 80, 80, 80, 80, 80},
        [46]  = {80, 80, 80, 80, 80, 80, 80, 90},
        [47]  = {80, 80, 80, 80, 80, 80, 80, 106},
        [48]  = {80, 80, 80, 80, 80, 80, 80, 149},
        [49]  = {80, 80, 80, 80, 80, 80, 80, 204},
        [50]  = {80, 80, 80, 80, 80, 80, 80, 666},
        [51]  = {80, 80, 80, 80, 80, 80, 84, 687},
        [52]  = {80, 80, 80, 80, 80, 80, 93, 637},
        [53]  = {80, 80, 80, 80, 80, 80, 93, 687},
        [54]  = {80, 80, 80, 80, 80, 80, 94, 357},
        [55]  = {80, 80, 80, 80, 80, 80, 116, 352},
        [56]  = {80, 80, 80, 80, 80, 80, 120, 687},
        [57]  = {80, 80, 80, 80, 80, 80, 163, 521},
        [58]  = {80, 80, 80, 80, 80, 80, 269, 314},
        [59]  = {80, 80, 80, 80, 80, 80, 269, 339},
        [60]  = {80, 80, 80, 80, 80, 80, 302, 299},
        [61]  = {80, 80, 80, 80, 80, 80, 358, 494},
        [62]  = {80, 80, 80, 80, 80, 81, 274, 265},
        [63]  = {80, 80, 80, 80, 80, 84, 1, 6},
        [64]  = {80, 80, 80, 80, 80, 102, 80, 80},
        [65]  = {80, 80, 80, 80, 80, 109, 355, 65},
        [66]  = {80, 80, 80, 80, 80, 112, 357, 286},
        [67]  = {80, 80, 80, 80, 80, 117, 404, 419},
        [68]  = {80, 80, 80, 80, 80, 125, 80, 80},
        [69]  = {80, 80, 80, 80, 80, 129, 318, 277},
        [70]  = {80, 80, 80, 80, 80, 133, 353, 204},
        [71]  = {80, 80, 80, 80, 80, 138, 421, 204},
        [72]  = {80, 80, 80, 80, 80, 149, 80, 109},
        [73]  = {80, 80, 80, 80, 80, 184, 278, 687},
        [74]  = {80, 80, 80, 80, 80, 681, 504, 684},
        [75]  = {80, 80, 80, 80, 80, 687, 80, 80},
        [76]  = {80, 80, 80, 80, 86, 328, 332, 667},
        [77]  = {80, 80, 80, 80, 87, 357, 304, 687},
        [78]  = {80, 80, 80, 80, 88, 423, 383, 687},
        [79]  = {80, 80, 80, 80, 111, 266, 344, 662},
        [80]  = {80, 80, 80, 80, 117, 687, 404, 420},
        [81]  = {80, 80, 80, 80, 125, 686, 687, 687},
        [82]  = {80, 80, 80, 80, 127, 312, 404, 420},
        [83]  = {80, 80, 80, 80, 445, 671, 80, 80},
        [84]  = {80, 80, 80, 82, 269, 381, 404, 420},
        [85]  = {80, 80, 80, 84, 338, 310, 271, 284},
        [86]  = {80, 80, 80, 84, 404, 418, 687, 687},
        [87]  = {80, 80, 80, 84, 685, 430, 687, 687},
        [88]  = {80, 80, 80, 92, 80, 80, 80, 80},
        [89]  = {80, 80, 80, 93, 404, 420, 235, 235},
        [90]  = {80, 80, 80, 100, 429, 687, 80, 80},
        [91]  = {80, 80, 80, 100, 687, 687, 687, 687},
        [92]  = {80, 80, 80, 109, 80, 80, 80, 80},
        [93]  = {80, 80, 80, 145, 273, 374, 404, 406},
        [94]  = {80, 80, 80, 145, 486, 469, 430, 438},
        [95]  = {80, 80, 80, 174, 687, 687, 687, 687},
        [96]  = {80, 80, 80, 188, 80, 149, 687, 687},
        [97]  = {80, 80, 82, 311, 80, 80, 80, 80},
        [98]  = {80, 80, 84, 687, 84, 84, 84, 84},
        [99]  = {80, 80, 87, 321, 342, 73, 486, 488},
        [100] = {80, 80, 89, 443, 80, 80, 80, 173},
        [101] = {80, 80, 92, 430, 80, 80, 83, 542},
        [102] = {80, 80, 93, 430, 80, 80, 149, 430},
        [103] = {80, 80, 93, 687, 80, 687, 687, 687},
        [104] = {80, 80, 95, 360, 315, 349, 269, 370},
        [105] = {80, 80, 119, 346, 347, 393, 404, 400},
        [106] = {80, 80, 160, 687, 80, 104, 200, 687},
        [107] = {80, 80, 164, 80, 80, 112, 329, 481},
        [108] = {80, 80, 174, 638, 687, 635, 687, 687},
        [109] = {80, 80, 176, 651, 303, 687, 687, 687},
        [110] = {80, 80, 196, 80, 80, 80, 80, 80},
        [111] = {80, 80, 204, 687, 389, 430, 430, 445},
        [112] = {80, 80, 572, 596, 235, 235, 235, 235},
        [113] = {80, 80, 642, 80, 80, 80, 80, 80},
        [114] = {80, 80, 652, 357, 357, 630, 317, 687},
        [115] = {80, 80, 687, 687, 687, 687, 687, 687},
        [116] = {80, 93, 80, 93, 687, 687, 687, 687},
        [117] = {80, 104, 157, 687, 80, 80, 156, 687},
        [118] = {80, 122, 80, 80, 84, 80, 80, 143},
        [119] = {80, 148, 175, 261, 261, 261, 261, 261},
        [120] = {80, 150, 386, 204, 404, 420, 687, 687},
        [121] = {80, 155, 356, 382, 499, 687, 687, 687},
        [122] = {80, 164, 80, 125, 619, 164, 80, 125},
        [123] = {80, 174, 404, 407, 572, 572, 572, 572},
        [124] = {80, 194, 80, 80, 80, 81, 360, 353},
        [125] = {80, 196, 80, 80, 80, 80, 196, 84},
        [126] = {80, 196, 84, 196, 80, 80, 80, 80},
        [127] = {80, 204, 80, 170, 4, 687, 687, 687},
        [128] = {80, 204, 687, 687, 687, 687, 687, 687},
        [129] = {80, 539, 80, 80, 99, 661, 488, 687},
        [130] = {80, 616, 80, 80, 80, 109, 633, 356},
        [131] = {80, 687, 687, 687, 687, 687, 80, 121},
        [132] = {82, 357, 332, 686, 705, 705, 705, 701},
        [133] = {84, 84, 84, 84, 357, 357, 357, 357},
        [134] = {84, 164, 80, 80, 80, 80, 186, 653},
        [135] = {84, 195, 80, 91, 80, 191, 80, 80},
        [136] = {84, 654, 275, 261, 678, 404, 648, 687},
        [137] = {84, 658, 80, 80, 80, 80, 80, 125},
        [138] = {84, 687, 687, 687, 687, 687, 687, 687},
        [139] = {87, 357, 390, 501, 687, 687, 687, 687},
        [140] = {93, 194, 80, 80, 80, 80, 91, 202},
        [141] = {97, 80, 80, 80, 80, 180, 80, 80},
        [142] = {103, 364, 404, 414, 687, 687, 687, 687},
        [143] = {108, 430, 360, 687, 687, 687, 687, 687},
        [144] = {110, 205, 47, 45, 217, 217, 51, 217},
        [145] = {122, 80, 80, 80, 621, 616, 142, 626},
        [146] = {125, 80, 468, 687, 687, 687, 687, 687},
        [147] = {125, 687, 687, 687, 687, 687, 687, 687},
        [148] = {131, 115, 404, 415, 80, 80, 72, 549},
        [149] = {152, 131, 80, 80, 128, 583, 435, 514},
        [150] = {153, 263, 404, 403, 235, 235, 235, 235},
        [151] = {165, 80, 80, 80, 80, 80, 80, 80},
        [152] = {180, 687, 687, 657, 80, 130, 473, 687},
        [153] = {181, 104, 194, 80, 80, 194, 122, 168},
        [154] = {182, 107, 201, 80, 80, 194, 199, 167},
        [155] = {185, 336, 80, 80, 80, 80, 136, 305},
        [156] = {187, 665, 122, 616, 80, 80, 93, 379},
        [157] = {189, 161, 80, 80, 80, 80, 87, 381},
        [158] = {190, 687, 404, 405, 430, 438, 474, 80},
        [159] = {193, 486, 497, 385, 404, 405, 439, 541},
        [160] = {196, 80, 84, 80, 80, 80, 80, 80},
        [161] = {217, 74, 210, 217, 217, 220, 232, 217},
        [162] = {217, 217, 215, 14, 217, 217, 217, 217},
        [163] = {217, 217, 217, 213, 80, 80, 80, 80},
        [164] = {217, 217, 217, 217, 212, 557, 326, 636},
        [165] = {217, 217, 217, 217, 217, 216, 296, 362},
        [166] = {217, 217, 217, 217, 217, 217, 60, 48},
        [167] = {217, 217, 217, 217, 217, 217, 209, 39},
        [168] = {217, 217, 217, 217, 217, 217, 217, 47},
        [169] = {217, 217, 217, 217, 217, 217, 217, 217},
        [170] = {217, 217, 217, 217, 217, 217, 627, 235},
        [171] = {221, 295, 217, 217, 217, 217, 217, 217},
        [172] = {222, 10, 10, 26, 647, 628, 233, 43},
        [173] = {224, 46, 217, 217, 217, 217, 217, 217},
        [174] = {233, 10, 10, 10, 235, 235, 235, 222},
        [175] = {235, 222, 10, 10, 17, 235, 235, 235},
        [176] = {235, 233, 19, 10, 10, 235, 235, 235},
        [177] = {235, 235, 233, 10, 10, 10, 235, 235},
        [178] = {235, 235, 235, 222, 10, 10, 24, 235},
        [179] = {235, 235, 235, 235, 10, 10, 10, 10},
        [180] = {235, 235, 235, 235, 222, 10, 10, 10},
        [181] = {235, 235, 235, 235, 235, 10, 10, 10},
        [182] = {235, 235, 235, 235, 235, 235, 10, 10},
        [183] = {235, 235, 235, 235, 235, 235, 239, 17},
        [184] = {235, 235, 235, 235, 235, 235, 251, 687},
        [185] = {235, 235, 235, 235, 235, 237, 10, 10},
        [186] = {235, 235, 236, 234, 10, 10, 12, 10},
        [187] = {235, 235, 237, 615, 10, 10, 10, 10},
        [188] = {235, 235, 246, 10, 10, 10, 10, 30},
        [189] = {238, 673, 10, 10, 10, 10, 10, 32},
        [190] = {244, 646, 256, 10, 10, 24, 235, 235},
        [191] = {254, 10, 10, 35, 41, 235, 235, 248},
        [192] = {257, 466, 404, 402, 635, 430, 438, 687},
        [193] = {258, 80, 80, 80, 80, 80, 111, 261},
        [194] = {261, 684, 334, 63, 687, 687, 687, 687},
        [195] = {262, 280, 372, 496, 404, 420, 430, 452},
        [196] = {264, 291, 674, 676, 159, 404, 622, 687},
        [197] = {267, 679, 404, 420, 357, 357, 322, 478},
        [198] = {270, 125, 404, 409, 503, 558, 373, 578},
        [199] = {272, 139, 404, 413, 687, 687, 687, 687},
        [200] = {276, 80, 80, 80, 80, 80, 80, 357},
        [201] = {280, 306, 185, 80, 158, 404, 464, 80},
        [202] = {281, 319, 204, 687, 159, 404, 502, 623},
        [203] = {282, 289, 663, 429, 159, 404, 430, 453},
        [204] = {283, 293, 683, 664, 159, 404, 162, 450},
        [205] = {283, 294, 677, 664, 159, 404, 540, 687},
        [206] = {287, 396, 634, 618, 678, 404, 324, 687},
        [207] = {288, 490, 404, 418, 96, 80, 404, 399},
        [208] = {307, 394, 675, 149, 159, 404, 686, 432},
        [209] = {308, 292, 203, 671, 151, 371, 371, 687},
        [210] = {309, 104, 194, 80, 80, 194, 80, 171},
        [211] = {313, 290, 203, 687, 678, 404, 446, 544},
        [212] = {316, 687, 404, 420, 687, 687, 687, 687},
        [213] = {327, 80, 80, 80, 80, 80, 126, 356},
        [214] = {327, 80, 80, 80, 80, 80, 268, 285},
        [215] = {327, 80, 80, 80, 177, 333, 404, 398},
        [216] = {331, 104, 194, 80, 80, 80, 80, 134},
        [217] = {331, 107, 201, 80, 80, 194, 197, 166},
        [218] = {336, 80, 80, 80, 82, 345, 368, 404},
        [219] = {337, 80, 80, 80, 80, 80, 80, 154},
        [220] = {337, 80, 80, 80, 80, 80, 132, 330},
        [221] = {343, 80, 80, 80, 80, 80, 114, 340},
        [222] = {351, 687, 404, 420, 91, 194, 80, 80},
        [223] = {357, 357, 357, 357, 357, 357, 214, 641},
        [224] = {357, 357, 357, 357, 357, 357, 357, 357},
        [225] = {357, 357, 357, 357, 357, 357, 357, 387},
        [226] = {357, 357, 357, 357, 357, 357, 359, 558},
        [227] = {357, 357, 357, 357, 357, 357, 687, 687},
        [228] = {357, 357, 357, 357, 357, 367, 565, 572},
        [229] = {357, 357, 726, 719, 357, 357, 724, 722},
        [230] = {357, 378, 80, 80, 80, 80, 80, 80},
        [231] = {358, 491, 655, 430, 428, 404, 687, 684},
        [232] = {360, 357, 357, 395, 388, 381, 687, 687},
        [233] = {365, 98, 357, 633, 357, 357, 357, 369},
        [234] = {377, 373, 572, 572, 572, 551, 572, 572},
        [235] = {391, 687, 404, 420, 486, 493, 687, 687},
        [236] = {392, 486, 486, 118, 687, 687, 687, 687},
        [237] = {404, 398, 80, 80, 80, 136, 341, 472},
        [238] = {404, 398, 80, 80, 87, 363, 80, 80},
        [239] = {404, 409, 85, 323, 301, 260, 183, 80},
        [240] = {404, 416, 404, 398, 80, 80, 80, 64},
        [241] = {404, 417, 687, 687, 687, 687, 687, 687},
        [242] = {404, 420, 404, 420, 483, 491, 357, 354},
        [243] = {404, 420, 687, 687, 687, 687, 687, 687},
        [244] = {422, 426, 533, 555, 538, 570, 572, 572},
        [245] = {423, 423, 423, 423, 423, 423, 423, 423},
        [246] = {423, 423, 423, 423, 423, 423, 425, 454},
        [247] = {423, 423, 423, 423, 423, 424, 493, 687},
        [248] = {430, 430, 430, 430, 430, 440, 511, 687},
        [249] = {430, 430, 430, 441, 572, 572, 572, 572},
        [250] = {430, 430, 431, 499, 687, 687, 687, 687},
        [251] = {430, 430, 437, 567, 513, 572, 596, 686},
        [252] = {430, 430, 442, 572, 572, 572, 735, 574},
        [253] = {430, 430, 655, 430, 430, 430, 430, 430},
        [254] = {430, 438, 572, 572, 572, 572, 572, 572},
        [255] = {430, 449, 63, 67, 512, 512, 512, 512},
        [256] = {430, 455, 486, 504, 80, 80, 80, 101},
        [257] = {456, 486, 500, 487, 3, 1, 427, 448},
        [258] = {457, 80, 80, 80, 80, 80, 80, 80},
        [259] = {459, 486, 687, 687, 687, 687, 687, 687},
        [260] = {465, 235, 235, 247, 515, 10, 10, 33},
        [261] = {467, 687, 80, 80, 80, 147, 140, 687},
        [262] = {477, 470, 485, 476, 462, 505, 486, 475},
        [263] = {480, 687, 80, 80, 80, 80, 137, 687},
        [264] = {482, 479, 404, 420, 80, 80, 80, 80},
        [265] = {484, 687, 184, 279, 80, 80, 80, 80},
        [266] = {486, 489, 178, 80, 80, 80, 357, 357},
        [267] = {486, 687, 376, 357, 318, 300, 113, 325},
        [268] = {498, 430, 430, 430, 430, 430, 444, 572},
        [269] = {501, 669, 80, 80, 80, 204, 404, 420},
        [270] = {517, 230, 29, 245, 10, 240, 640, 231},
        [271] = {521, 521, 520, 63, 518, 217, 37, 217},
        [272] = {522, 687, 658, 80, 80, 80, 80, 80},
        [273] = {526, 460, 460, 461, 531, 531, 531, 531},
        [274] = {528, 42, 430, 430, 423, 423, 423, 423},
        [275] = {529, 531, 531, 531, 530, 508, 531, 531},
        [276] = {531, 531, 531, 509, 531, 531, 531, 527},
        [277] = {531, 531, 531, 531, 531, 531, 531, 531},
        [278] = {532, 533, 736, 572, 572, 572, 582, 572},
        [279] = {547, 572, 572, 572, 572, 572, 572, 534},
        [280] = {550, 548, 226, 524, 560, 255, 228, 75},
        [281] = {552, 687, 687, 687, 687, 687, 687, 687},
        [282] = {561, 495, 687, 660, 633, 357, 687, 687},
        [283] = {568, 577, 492, 543, 687, 687, 687, 687},
        [284] = {572, 510, 572, 599, 536, 603, 572, 572},
        [285] = {572, 554, 572, 583, 602, 687, 687, 687},
        [286] = {572, 569, 556, 572, 572, 572, 562, 531},
        [287] = {572, 572, 572, 559, 531, 531, 535, 572},
        [288] = {572, 572, 572, 562, 536, 605, 738, 572},
        [289] = {572, 572, 572, 572, 564, 384, 259, 5},
        [290] = {572, 572, 572, 572, 572, 553, 430, 430},
        [291] = {572, 572, 572, 572, 572, 572, 571, 572},
        [292] = {572, 572, 572, 572, 572, 572, 572, 563},
        [293] = {572, 572, 572, 572, 572, 572, 572, 572},
        [294] = {572, 572, 572, 572, 572, 572, 572, 595},
        [295] = {572, 572, 572, 572, 572, 572, 575, 687},
        [296] = {572, 572, 572, 572, 572, 572, 583, 687},
        [297] = {572, 572, 572, 572, 572, 578, 572, 572},
        [298] = {572, 572, 572, 572, 572, 602, 687, 687},
        [299] = {572, 572, 572, 572, 573, 639, 572, 572},
        [300] = {572, 572, 572, 572, 573, 687, 687, 687},
        [301] = {572, 572, 572, 591, 572, 572, 531, 531},
        [302] = {572, 572, 572, 602, 763, 755, 687, 687},
        [303] = {572, 572, 573, 687, 430, 430, 430, 455},
        [304] = {572, 572, 575, 572, 572, 572, 572, 572},
        [305] = {572, 572, 580, 572, 585, 572, 572, 572},
        [306] = {572, 572, 583, 687, 572, 573, 639, 572},
        [307] = {572, 572, 583, 687, 572, 575, 755, 750},
        [308] = {572, 572, 587, 572, 572, 572, 572, 572},
        [309] = {572, 572, 590, 572, 604, 599, 572, 580},
        [310] = {572, 576, 606, 747, 572, 578, 687, 687},
        [311] = {572, 583, 572, 572, 572, 572, 572, 572},
        [312] = {572, 584, 763, 763, 763, 763, 763, 763},
        [313] = {572, 586, 593, 572, 572, 508, 507, 430},
        [314] = {572, 589, 687, 687, 430, 430, 430, 430},
        [315] = {572, 594, 763, 763, 763, 572, 572, 597},
        [316] = {572, 687, 572, 572, 572, 575, 687, 687},
        [317] = {572, 687, 572, 596, 572, 572, 572, 572},
        [318] = {572, 763, 754, 572, 572, 571, 572, 574},
        [319] = {579, 599, 572, 572, 572, 735, 572, 572},
        [320] = {585, 572, 572, 572, 572, 583, 572, 572},
        [321] = {587, 576, 585, 572, 572, 597, 600, 598},
        [322] = {601, 404, 80, 80, 80, 93, 109, 687},
        [323] = {608, 80, 80, 141, 80, 80, 80, 80},
        [324] = {612, 611, 458, 463, 486, 607, 486, 471},
        [325] = {614, 687, 687, 687, 1, 1, 1, 1},
        [326] = {616, 80, 80, 80, 80, 80, 179, 380},
        [327] = {617, 617, 617, 687, 84, 84, 10, 10},
        [328] = {620, 142, 80, 80, 123, 80, 179, 348},
        [329] = {631, 107, 201, 80, 80, 194, 197, 168},
        [330] = {632, 91, 164, 80, 80, 194, 197, 167},
        [331] = {632, 147, 201, 80, 80, 194, 198, 169},
        [332] = {635, 430, 430, 430, 430, 434, 430, 436},
        [333] = {639, 574, 639, 572, 572, 572, 575, 687},
        [334] = {642, 642, 642, 643, 733, 546, 687, 613},
        [335] = {644, 624, 621, 625, 621, 142, 142, 618},
        [336] = {645, 687, 687, 687, 687, 687, 682, 61},
        [337] = {649, 80, 84, 642, 80, 80, 164, 124},
        [338] = {650, 146, 196, 620, 659, 146, 80, 172},
        [339] = {668, 249, 56, 235, 252, 227, 10, 10},
        [340] = {687, 687, 80, 80, 80, 80, 80, 80},
        [341] = {687, 687, 80, 80, 80, 93, 366, 687},
        [342] = {687, 687, 357, 350, 297, 357, 397, 687},
        [343] = {687, 687, 572, 572, 572, 572, 572, 566},
        [344] = {687, 687, 660, 357, 320, 357, 357, 357},
        [345] = {687, 687, 687, 687, 80, 80, 84, 687},
        [346] = {687, 687, 687, 687, 80, 80, 104, 93},
        [347] = {687, 687, 687, 687, 80, 80, 135, 504},
        [348] = {687, 687, 687, 687, 80, 80, 144, 661},
        [349] = {687, 687, 687, 687, 80, 642, 80, 80},
        [350] = {687, 687, 687, 687, 235, 235, 235, 235},
        [351] = {687, 687, 687, 687, 430, 430, 430, 433},
        [352] = {687, 687, 687, 687, 430, 430, 445, 687},
        [353] = {687, 687, 687, 687, 680, 572, 572, 572},
        [354] = {687, 687, 687, 687, 687, 687, 537, 687},
        [355] = {687, 687, 687, 687, 687, 687, 635, 430},
        [356] = {687, 687, 687, 687, 687, 687, 687, 687},
        [357] = {687, 687, 687, 687, 696, 687, 687, 687},
        [358] = {687, 687, 699, 687, 689, 687, 705, 705},
        [359] = {688, 705, 705, 705, 705, 705, 691, 705},
        [360] = {691, 705, 705, 705, 705, 705, 705, 705},
        [361] = {693, 729, 709, 710, 723, 707, 707, 708},
        [362] = {705, 701, 763, 763, 763, 763, 763, 763},
        [363] = {705, 704, 746, 763, 705, 705, 705, 699},
        [364] = {705, 705, 702, 705, 705, 705, 705, 705},
        [365] = {705, 705, 704, 687, 687, 687, 687, 687},
        [366] = {705, 705, 704, 692, 716, 705, 705, 705},
        [367] = {705, 705, 705, 698, 687, 687, 687, 687},
        [368] = {705, 705, 705, 703, 687, 687, 687, 687},
        [369] = {705, 705, 705, 703, 705, 705, 705, 705},
        [370] = {705, 705, 705, 704, 687, 687, 687, 687},
        [371] = {705, 705, 705, 705, 80, 80, 80, 80},
        [372] = {705, 705, 705, 705, 697, 687, 687, 687},
        [373] = {705, 705, 705, 705, 698, 687, 705, 705},
        [374] = {705, 705, 705, 705, 700, 687, 80, 80},
        [375] = {705, 705, 705, 705, 705, 703, 705, 705},
        [376] = {705, 705, 705, 705, 705, 705, 687, 687},
        [377] = {705, 705, 705, 705, 705, 705, 699, 687},
        [378] = {705, 705, 705, 705, 705, 705, 701, 687},
        [379] = {705, 705, 705, 705, 705, 705, 703, 687},
        [380] = {705, 705, 705, 705, 705, 705, 705, 687},
        [381] = {705, 705, 705, 705, 705, 705, 705, 700},
        [382] = {705, 705, 705, 705, 705, 705, 705, 705},
        [383] = {705, 705, 705, 705, 705, 705, 705, 706},
        [384] = {713, 714, 763, 763, 763, 763, 742, 713},
        [385] = {721, 720, 725, 717, 727, 728, 104, 80},
        [386] = {734, 694, 694, 695, 718, 80, 71, 80},
        [387] = {739, 763, 763, 763, 763, 763, 763, 763},
        [388] = {750, 687, 760, 687, 687, 687, 687, 687},
        [389] = {750, 687, 763, 763, 763, 763, 763, 755},
        [390] = {758, 585, 749, 687, 572, 592, 588, 750},
        [391] = {762, 687, 80, 80, 80, 80, 80, 64},
        [392] = {763, 430, 742, 713, 763, 763, 763, 763},
        [393] = {763, 740, 743, 687, 760, 687, 687, 687},
        [394] = {763, 748, 754, 572, 763, 763, 737, 753},
        [395] = {763, 752, 763, 763, 763, 763, 763, 763},
        [396] = {763, 763, 572, 572, 572, 572, 572, 572},
        [397] = {763, 763, 754, 572, 763, 763, 763, 763},
        [398] = {763, 763, 760, 687, 687, 687, 763, 755},
        [399] = {763, 763, 763, 745, 763, 763, 763, 763},
        [400] = {763, 763, 763, 762, 713, 714, 763, 763},
        [401] = {763, 763, 763, 763, 735, 581, 759, 763},
        [402] = {763, 763, 763, 763, 752, 751, 763, 763},
        [403] = {763, 763, 763, 763, 755, 687, 705, 705},
        [404] = {763, 763, 763, 763, 763, 763, 744, 690},
        [405] = {763, 763, 763, 763, 763, 763, 755, 687},
        [406] = {763, 763, 763, 763, 763, 763, 763, 756},
        [407] = {763, 763, 763, 763, 763, 763, 763, 757},
        [408] = {763, 763, 763, 763, 763, 763, 763, 758},
        [409] = {763, 763, 763, 763, 763, 763, 763, 761},
        [410] = {763, 763, 763, 763, 763, 763, 763, 763},
        [411] = {764, 731, 732, 730, 741, 711, 715, 712},
};
static const uint16_t data_2[][8] = {
        [0]  = {0, 260, 1, 186, 168, 34, 36, 144},
        [1]  = {3, 40, 45, 105, 266, 230, 77, 237},
        [2]  = {4, 4, 4, 4, 4, 4, 4, 4},
        [3]  = {4, 4, 4, 4, 4, 4, 4, 5},
        [4]  = {4, 4, 4, 4, 382, 382, 382, 382},
        [5]  = {6, 6, 6, 6, 6, 6, 6, 6},
        [6]  = {27, 177, 175, 191, 16, 21, 23, 33},
        [7]  = {43, 165, 163, 78, 271, 166, 161, 336},
        [8]  = {45, 45, 45, 45, 45, 45, 45, 45},
        [9]  = {45, 45, 45, 45, 45, 45, 95, 356},
        [10] = {45, 45, 45, 45, 46, 340, 113, 131},
        [11] = {45, 45, 45, 45, 51, 103, 356, 356},
        [12] = {45, 45, 45, 143, 180, 31, 356, 356},
        [13] = {45, 64, 323, 71, 117, 106, 55, 195},
        [14] = {45, 72, 127, 356, 356, 356, 356, 356},
        [15] = {45, 126, 125, 160, 110, 94, 112, 183},
        [16] = {45, 128, 184, 8, 80, 356, 356, 356},
        [17] = {45, 138, 356, 356, 356, 356, 356, 356},
        [18] = {49, 86, 356, 341, 61, 38, 115, 356},
        [19] = {54, 199, 58, 212, 356, 356, 62, 236},
        [20] = {59, 235, 70, 243, 93, 356, 356, 356},
        [21] = {63, 356, 356, 356, 356, 356, 356, 356},
        [22] = {66, 356, 350, 12, 356, 356, 349, 109},
        [23] = {69, 239, 150, 189, 382, 371, 45, 45},
        [24] = {73, 148, 60, 152, 327, 15, 7, 82},
        [25] = {79, 240, 24, 267, 9, 37, 14, 225},
        [26] = {97, 104, 242, 356, 221, 198, 215, 76},
        [27] = {99, 96, 346, 344, 219, 201, 154, 204},
        [28] = {107, 356, 135, 84, 217, 209, 356, 356},
        [29] = {118, 116, 45, 48, 268, 246, 285, 343},
        [30] = {124, 158, 114, 356, 134, 222, 120, 356},
        [31] = {140, 101, 87, 348, 100, 356, 50, 253},
        [32] = {145, 335, 122, 354, 356, 356, 356, 356},
        [33] = {149, 56, 193, 197, 238, 132, 220, 207},
        [34] = {155, 265, 121, 49, 356, 356, 356, 356},
        [35] = {156, 256, 91, 129, 52, 102, 108, 356},
        [36] = {167, 35, 17, 39, 224, 223, 339, 20},
        [37] = {169, 169, 162, 169, 26, 29, 25, 270},
        [38] = {178, 176, 172, 32, 30, 190, 174, 18},
        [39] = {181, 19, 89, 188, 75, 81, 356, 356},
        [40] = {182, 11, 171, 173, 170, 187, 22, 261},
        [41] = {185, 13, 169, 164, 10, 74, 98, 133},
        [42] = {200, 231, 214, 269, 218, 263, 213, 192},
        [43] = {210, 208, 153, 196, 216, 203, 337, 136},
        [44] = {226, 228, 282, 356, 356, 356, 356, 356},
        [45] = {229, 385, 45, 47, 361, 386, 44, 334},
        [46] = {232, 356, 356, 356, 65, 241, 356, 356},
        [47] = {245, 247, 45, 45, 45, 147, 356, 356},
        [48] = {258, 45, 45, 45, 45, 45, 45, 45},
        [49] = {262, 259, 399, 405, 410, 410, 410, 398},
        [50] = {264, 41, 141, 53, 85, 322, 68, 123},
        [51] = {277, 277, 277, 277, 284, 292, 287, 288},
        [52] = {277, 277, 277, 277, 301, 278, 304, 293},
        [53] = {293, 281, 356, 352, 293, 303, 356, 356},
        [54] = {293, 293, 293, 293, 277, 277, 273, 276},
        [55] = {293, 293, 293, 293, 293, 293, 293, 293},
        [56] = {293, 293, 293, 295, 299, 289, 234, 298},
        [57] = {293, 307, 388, 356, 356, 356, 356, 356},
        [58] = {300, 314, 249, 290, 293, 293, 291, 279},
        [59] = {308, 318, 309, 321, 319, 313, 252, 275},
        [60] = {311, 317, 316, 356, 312, 404, 402, 395},
        [61] = {320, 293, 306, 333, 254, 297, 310, 353},
        [62] = {324, 257, 255, 342, 280, 274, 244, 286},
        [63] = {325, 2, 356, 356, 224, 224, 224, 227},
        [64] = {326, 42, 328, 142, 159, 130, 233, 283},
        [65] = {331, 206, 330, 202, 329, 205, 338, 211},
        [66] = {356, 179, 250, 356, 45, 119, 194, 357},
        [67] = {356, 351, 356, 356, 90, 139, 356, 345},
        [68] = {356, 355, 248, 356, 332, 356, 356, 356},
        [69] = {356, 356, 92, 111, 83, 157, 88, 146},
        [70] = {356, 356, 356, 67, 356, 356, 356, 356},
        [71] = {356, 356, 356, 347, 356, 356, 356, 251},
        [72] = {356, 356, 356, 356, 356, 356, 356, 356},
        [73] = {364, 382, 382, 382, 382, 382, 382, 382},
        [74] = {368, 356, 356, 356, 356, 356, 356, 356},
        [75] = {369, 382, 382, 382, 382, 382, 382, 382},
        [76] = {382, 375, 382, 367, 28, 151, 57, 272},
        [77] = {382, 382, 362, 391, 45, 45, 45, 45},
        [78] = {382, 382, 373, 382, 382, 382, 382, 382},
        [79] = {382, 382, 382, 365, 382, 382, 382, 382},
        [80] = {382, 382, 382, 377, 356, 356, 356, 356},
        [81] = {382, 382, 382, 381, 356, 356, 356, 356},
        [82] = {382, 382, 382, 382, 370, 358, 382, 382},
        [83] = {382, 382, 382, 382, 378, 382, 382, 382},
        [84] = {382, 382, 382, 382, 382, 382, 374, 137},
        [85] = {382, 382, 382, 382, 382, 382, 379, 293},
        [86] = {382, 382, 382, 382, 382, 382, 382, 372},
        [87] = {382, 382, 382, 382, 382, 382, 382, 376},
        [88] = {382, 382, 382, 382, 382, 382, 382, 380},
        [89] = {382, 382, 382, 382, 382, 382, 382, 382},
        [90] = {389, 393, 356, 356, 401, 407, 397, 394},
        [91] = {400, 392, 384, 410, 410, 410, 410, 410},
        [92] = {409, 387, 410, 406, 408, 315, 305, 294},
        [93] = {410, 396, 410, 390, 293, 296, 293, 302},
        [94] = {411, 360, 366, 383, 359, 382, 363, 403},
};
static const uint8_t data_3[][8] = {
        [0] = {0, 36, 40, 1, 27, 65, 43, 64},    [1] = {2, 2, 2, 2, 2, 2, 2, 2},
        [2] = {2, 2, 2, 2, 2, 2, 2, 3},          [3] = {2, 2, 2, 2, 4, 76, 10, 45},
        [4] = {8, 8, 21, 72, 72, 72, 72, 72},    [5] = {8, 9, 47, 72, 72, 72, 72, 72},
        [6] = {23, 15, 48, 13, 50, 26, 25, 37},  [7] = {29, 69, 39, 11, 31, 35, 16, 67},
        [8] = {42, 28, 19, 20, 22, 34, 30, 71},  [9] = {46, 70, 72, 72, 12, 72, 68, 32},
        [10] = {56, 53, 38, 6, 55, 44, 72, 72},  [11] = {61, 90, 92, 93, 60, 57, 72, 72},
        [12] = {62, 51, 58, 59, 54, 52, 41, 49}, [13] = {63, 72, 72, 72, 72, 72, 72, 72},
        [14] = {72, 72, 8, 17, 72, 72, 72, 72},  [15] = {72, 72, 72, 72, 8, 18, 72, 66},
        [16] = {72, 72, 72, 72, 72, 72, 72, 72}, [17] = {72, 72, 72, 72, 89, 74, 72, 72},
        [18] = {73, 89, 77, 7, 33, 24, 89, 89},  [19] = {82, 81, 72, 72, 72, 72, 14, 72},
        [20] = {89, 89, 89, 79, 89, 89, 89, 89}, [21] = {89, 89, 89, 83, 75, 89, 89, 89},
        [22] = {89, 89, 89, 84, 5, 5, 5, 5},     [23] = {89, 89, 89, 88, 89, 80, 72, 72},
        [24] = {89, 89, 89, 89, 89, 86, 72, 72}, [25] = {89, 89, 89, 89, 89, 89, 85, 89},
        [26] = {89, 89, 89, 89, 89, 89, 89, 78}, [27] = {89, 89, 89, 89, 89, 89, 89, 87},
        [28] = {89, 89, 89, 89, 89, 89, 89, 89}, [29] = {94, 91, 89, 89, 89, 89, 89, 89},
};
static const uint8_t data_4[][8] = {
        [0] = {0, 6, 12, 29, 25, 28, 28, 28},   [1] = {1, 1, 1, 1, 1, 1, 1, 1},
        [2] = {1, 1, 1, 1, 1, 1, 1, 2},         [3] = {7, 8, 5, 4, 14, 16, 15, 28},
        [4] = {13, 16, 16, 16, 16, 16, 16, 16}, [5] = {16, 16, 16, 16, 16, 16, 16, 16},
        [6] = {23, 16, 16, 19, 16, 10, 9, 11},  [7] = {28, 27, 18, 28, 28, 22, 1, 3},
        [8] = {28, 28, 20, 21, 26, 28, 24, 17}, [9] = {28, 28, 28, 28, 28, 28, 28, 28},
};
static const uint8_t data_5[][8] = {
        [0] = {0, 7, 3, 6, 9, 8, 5, 5},
        [1] = {1, 2, 0, 0, 0, 0, 0, 0},
        [2] = {5, 5, 5, 5, 4, 5, 1, 2},
        [3] = {5, 5, 5, 5, 5, 5, 5, 5},
};
static const uint8_t data_6[] = {0, 3, 3, 2, 1};

static int8_t pn_rune_data(pn_rune_t rune) {
    uint8_t b6 = (rune & 07000000) >> 18;
    uint8_t b5 = (rune & 00700000) >> 15;
    uint8_t b4 = (rune & 00070000) >> 12;
    uint8_t b3 = (rune & 00007000) >> 9;
    uint8_t b2 = (rune & 00000700) >> 6;
    uint8_t b1 = (rune & 00000070) >> 3;
    uint8_t b0 = (rune & 00000007) >> 0;
    int     x  = data_6[b6];
    x          = data_5[x][b5];
    x          = data_4[x][b4];
    x          = data_3[x][b3];
    x          = data_2[x][b2];
    x          = data_1[x][b1];
    x          = data_0[x][b0];
    return x;
}

size_t pn_rune_width(pn_rune_t rune) {
    if (rune >= 0x110000) {
        return 1;
    }
    int8_t x = pn_rune_data(rune);
    switch (x & 0070) {
        case PN_RUNE_OTHER: return 1;
        case PN_RUNE_MODIFIER: return 0;
    }
    return 1 + (x >> 6);
}

size_t pn_str_width(const char* data, size_t size) {
    size_t total = 0;
    for (size_t i = 0; i < size; i = pn_rune_next(data, size, i)) {
        total += pn_rune_width(pn_rune(data, size, i));
    }
    return total;
}

bool pn_isrune(pn_rune_t r) { return r < 0x110000; }

bool pn_isalnum(pn_rune_t r) {
    if (!pn_isrune(r)) {
        return false;
    }
    switch (pn_rune_data(r) & 0070) {
        case PN_RUNE_LETTER:
        case PN_RUNE_NUMBER: return true;
        default: return false;
    }
}

bool pn_isalpha(pn_rune_t r) {
    return pn_isrune(r) && ((pn_rune_data(r) & 0070) == PN_RUNE_LETTER);
}

bool pn_iscntrl(pn_rune_t r) {
    return pn_isrune(r) && ((pn_rune_data(r) & 0077) == PN_RUNE_OTHER_CONTROL);
}

bool pn_isdigit(pn_rune_t r) {
    return pn_isrune(r) && ((pn_rune_data(r) & 0077) == PN_RUNE_NUMBER_DECIMAL_DIGIT);
}

bool pn_islower(pn_rune_t r) {
    return pn_isrune(r) && ((pn_rune_data(r) & 0077) == PN_RUNE_LETTER_LOWERCASE);
}

bool pn_isnumeric(pn_rune_t r) {
    return pn_isrune(r) && ((pn_rune_data(r) & 0070) == PN_RUNE_NUMBER);
}

bool pn_isprint(pn_rune_t r) {
    return pn_isrune(r) && ((pn_rune_data(r) & 0070) != PN_RUNE_OTHER);
}

bool pn_ispunct(pn_rune_t r) {
    return pn_isrune(r) && ((pn_rune_data(r) & 0070) == PN_RUNE_PUNCTUATION);
}

bool pn_isspace(pn_rune_t r) {
    return pn_isrune(r) && ((pn_rune_data(r) & 0070) == PN_RUNE_SEPARATOR);
}

bool pn_istitle(pn_rune_t r) {
    return pn_isrune(r) && ((pn_rune_data(r) & 0077) == PN_RUNE_LETTER_TITLECASE);
}

bool pn_isupper(pn_rune_t r) {
    return pn_isrune(r) && ((pn_rune_data(r) & 0077) == PN_RUNE_LETTER_UPPERCASE);
}
