#!/usr/bin/env python3

from .error import Error
from .parse_enums import Acc, Emit, Key

LEX_BYTE_CLASSES = bytearray(b"\000\000\000\000\000\000\000\000\000\001\002\000\000\000\000\000"
                             b"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000"
                             b"\001\003\004\005\006\007\007\007\007\007\010\011\012\013\014\015"
                             b"\016\017\020\020\020\020\020\020\021\021\022\007\007\007\023\007"
                             b"\007\024\024\024\025\026\024\027\027\027\027\027\027\027\027\027"
                             b"\027\027\027\027\027\030\027\027\027\027\027\031\032\033\007\027"
                             b"\007\034\035\024\025\036\037\027\027\040\027\027\041\027\042\027"
                             b"\027\027\043\044\045\046\027\027\027\027\027\047\050\051\007\000"
                             b"\052\052\052\052\052\052\052\052\052\052\052\052\052\052\052\052"
                             b"\053\053\053\053\053\053\053\053\053\053\053\053\053\053\053\053"
                             b"\054\054\054\054\054\054\054\054\054\054\054\054\054\054\054\054"
                             b"\054\054\054\054\054\054\054\054\054\054\054\054\054\054\054\054"
                             b"\055\055\056\056\056\056\056\056\056\056\056\056\056\056\056\056"
                             b"\056\056\056\056\056\056\056\056\056\056\056\056\056\056\056\056"
                             b"\057\060\060\060\060\060\060\060\060\060\060\060\060\061\060\060"
                             b"\062\063\063\063\064\055\055\055\055\055\055\055\055\055\055\055")

LEX_TRANSITIONS = [
    bytearray(b"\214\220\201\113\041\125\115\220\117\002\124\003\001\001\004\005\005\005\013\067"
              b"\001\001\001\001\001\120\220\121\001\001\001\024\031\001\014\001\001\020\001\122"
              b"\101\123\215\215\215\215\215\215\215\215\215\215\215"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\001\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\004\005\005\005\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\031\001\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\004\005\005\005\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\034\001\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\324\324\324\324\324\324\324\324\324\001\324\001\006\001\001\001\001\001\013\324"
              b"\001\001\010\001\001\324\324\324\001\001\010\001\001\001\001\001\001\001\001\324"
              b"\324\324\324\324\324\324\324\324\324\324\324\324\324"),
    bytearray(b"\324\324\324\324\324\324\324\324\324\001\324\001\006\001\005\005\005\005\013\324"
              b"\001\001\010\001\001\324\324\324\001\001\010\001\001\001\001\001\001\001\001\324"
              b"\324\324\324\324\324\324\324\324\324\324\324\324\324"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\007\007\007\007\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\001\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\325\325\325\325\325\325\325\325\325\001\325\001\001\001\007\007\007\007\013\325"
              b"\001\001\010\001\001\325\325\325\001\001\010\001\001\001\001\001\001\001\001\325"
              b"\325\325\325\325\325\325\325\325\325\325\325\325\325"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\011\223\011\001\001\012\012\012\012\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\001\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\012\012\012\012\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\001\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\325\325\325\325\325\325\325\325\325\001\325\001\001\001\012\012\012\012\013\325"
              b"\001\001\001\001\001\325\325\325\001\001\001\001\001\001\001\001\001\001\001\325"
              b"\325\325\325\325\325\325\325\325\325\325\325\325\325"),
    bytearray(b"\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322"
              b"\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322\322"
              b"\322\322\322\322\322\322\322\322\322\322\322\322\322"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\037\001\001\001\001\001\001\001\001\001\015\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\016\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\017\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\314\314\314\314\314\314\314\314\314\001\314\001\001\001\001\001\001\001\013\314"
              b"\001\001\001\001\001\314\314\314\001\001\001\001\001\001\001\001\001\001\001\314"
              b"\314\314\314\314\314\314\314\314\314\314\314\314\314"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\001\001\021\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\001\001\001\001\001\022\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\023\001\001\001\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\315\315\315\315\315\315\315\315\315\001\315\001\001\001\001\001\001\001\013\315"
              b"\001\001\001\001\001\315\315\315\001\001\001\001\001\001\001\001\001\001\001\315"
              b"\315\315\315\315\315\315\315\315\315\315\315\315\315"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\025\001\001\001\001\001\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\026\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\001\001\001\027\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\030\001\001\001\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\316\316\316\316\316\316\316\316\316\001\316\001\001\001\001\001\001\001\013\316"
              b"\001\001\001\001\001\316\316\316\001\001\001\001\001\001\001\001\001\001\001\316"
              b"\316\316\316\316\316\316\316\316\316\316\316\316\316"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\001\032\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\033\001\001\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\317\317\317\317\317\317\317\317\317\001\317\001\001\001\001\001\001\001\013\317"
              b"\001\001\001\001\001\317\317\317\001\001\001\001\001\001\001\001\001\001\001\317"
              b"\317\317\317\317\317\317\317\317\317\317\317\317\317"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\001\035\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\036\001\001\001\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\320\320\320\320\320\320\320\320\320\001\320\001\001\001\001\001\001\001\013\320"
              b"\001\001\001\001\001\320\320\320\001\001\001\001\001\001\001\001\001\001\001\320"
              b"\320\320\320\320\320\320\320\320\320\320\320\320\320"),
    bytearray(b"\223\223\223\223\223\223\223\223\223\001\223\001\001\001\001\001\001\001\013\223"
              b"\001\001\001\001\001\223\223\223\001\001\001\001\001\001\040\001\001\001\001\223"
              b"\223\223\223\223\223\223\223\223\223\223\223\223\223"),
    bytearray(b"\321\321\321\321\321\321\321\321\321\001\321\001\001\001\001\001\001\001\013\321"
              b"\001\001\001\001\001\321\321\321\001\001\001\001\001\001\001\001\001\001\001\321"
              b"\321\321\321\321\321\321\321\321\321\321\321\321\321"),
    bytearray(b"\214\041\226\041\065\041\041\041\041\041\041\041\041\041\041\041\041\041\041\041"
              b"\041\041\041\041\041\041\051\041\041\041\041\041\041\041\041\041\041\041\041\041"
              b"\041\041\216\216\216\216\050\042\047\043\044\046\045"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\050\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\050\050\217\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\047\047\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\047\217\217\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\047\047\047\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\050\050\050\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\041\041\041\217\217\217\217\217\217\217\217"),
    bytearray(b"\224\224\226\224\041\224\224\224\224\224\224\224\224\041\224\224\224\224\224\224"
              b"\224\224\224\224\052\224\041\224\224\041\224\041\224\224\041\041\224\041\060\224"
              b"\224\224\224\224\224\224\224\224\224\224\224\224\224"),
    bytearray(b"\225\225\226\225\225\225\225\225\225\225\225\225\225\225\053\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225"),
    bytearray(b"\225\225\226\225\225\225\225\225\225\225\225\225\225\225\054\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225"),
    bytearray(b"\225\225\226\225\225\225\225\225\225\225\225\225\225\225\056\055\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225"),
    bytearray(b"\225\225\226\225\225\225\225\225\225\225\225\225\225\225\057\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225"),
    bytearray(b"\225\225\226\225\225\225\225\225\225\225\225\225\225\225\060\057\057\057\225\225"
              b"\057\057\057\225\225\225\225\225\057\057\057\057\225\225\225\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225"),
    bytearray(b"\225\225\226\225\225\225\225\225\225\225\225\225\225\225\061\061\061\061\225\225"
              b"\061\061\061\225\225\225\225\225\061\061\061\061\225\225\225\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225"),
    bytearray(b"\225\225\226\225\225\225\225\225\225\225\225\225\225\225\061\061\061\061\225\225"
              b"\061\064\061\225\225\225\225\225\061\061\061\061\225\225\225\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225"),
    bytearray(b"\225\225\226\225\225\225\225\225\225\225\225\225\225\225\062\062\062\062\225\225"
              b"\062\062\062\225\225\225\225\225\062\062\062\062\225\225\225\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225"),
    bytearray(b"\225\225\226\225\225\225\225\225\225\225\225\225\225\225\063\063\063\063\225\225"
              b"\063\063\063\225\225\225\225\225\063\063\063\063\225\225\225\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225"),
    bytearray(b"\225\225\226\225\225\225\225\225\225\225\225\225\225\225\041\041\041\041\225\225"
              b"\041\041\041\225\225\225\225\225\041\041\041\041\225\225\225\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225"),
    bytearray(b"\225\225\226\225\225\225\225\225\225\225\225\225\225\225\062\062\062\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225\225"
              b"\225\225\225\225\225\225\225\225\225\225\225\225\225"),
    bytearray(b"\327\327\327\327\327\327\327\327\327\327\327\327\327\327\327\327\327\327\066\327"
              b"\327\327\327\327\327\327\327\327\327\327\327\327\327\327\327\327\327\327\327\327"
              b"\327\327\327\327\327\327\327\327\327\327\327\327\327"),
    bytearray(b"\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323"
              b"\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323\323"
              b"\323\323\323\323\323\323\323\323\323\323\323\323\323"),
    bytearray(b"\214\077\311\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100"
              b"\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100"
              b"\100\100\216\216\216\216\076\070\075\071\072\074\073"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\076\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\076\076\217\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\075\075\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\075\217\217\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\075\075\075\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\076\076\076\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\100\100\100\217\217\217\217\217\217\217\217"),
    bytearray(b"\214\100\311\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100"
              b"\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100"
              b"\100\100\216\216\216\216\076\070\075\071\072\074\073"),
    bytearray(b"\214\100\330\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100"
              b"\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100"
              b"\100\100\216\216\216\216\076\070\075\071\072\074\073"),
    bytearray(b"\214\111\312\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112"
              b"\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112"
              b"\112\112\216\216\216\216\110\102\107\103\104\106\105"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\110\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\110\110\217\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\107\107\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\107\217\217\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\107\107\107\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\110\110\110\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\112\112\112\217\217\217\217\217\217\217\217"),
    bytearray(b"\214\112\312\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112"
              b"\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112"
              b"\112\112\216\216\216\216\110\102\107\103\104\106\105"),
    bytearray(b"\214\112\331\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112"
              b"\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112\112"
              b"\112\112\216\216\216\216\110\102\107\103\104\106\105"),
    bytearray(b"\227\114\313\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227"
              b"\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227"
              b"\227\227\227\227\227\227\227\227\227\227\227\227\227"),
    bytearray(b"\214\227\313\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227"
              b"\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227\227"
              b"\227\227\215\215\215\215\215\215\215\215\215\215\215"),
    bytearray(b"\326\115\326\326\326\326\326\326\326\221\326\221\221\221\116\116\116\116\326\326"
              b"\116\116\116\221\221\326\326\326\116\116\116\116\221\221\221\221\221\221\221\326"
              b"\326\326\326\326\326\326\326\326\326\326\326\326\326"),
    bytearray(b"\222\222\222\222\222\222\222\222\222\221\222\221\221\221\115\115\115\115\222\222"
              b"\115\115\115\221\221\222\222\222\115\115\115\115\221\221\221\221\221\221\221\222"
              b"\222\222\222\222\222\222\222\222\222\222\222\222\222"),
    bytearray(b"\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303"
              b"\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303\303"
              b"\303\303\303\303\303\303\303\303\303\303\303\303\303"),
    bytearray(b"\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304"
              b"\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304\304"
              b"\304\304\304\304\304\304\304\304\304\304\304\304\304"),
    bytearray(b"\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305"
              b"\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305\305"
              b"\305\305\305\305\305\305\305\305\305\305\305\305\305"),
    bytearray(b"\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306"
              b"\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306\306"
              b"\306\306\306\306\306\306\306\306\306\306\306\306\306"),
    bytearray(b"\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307"
              b"\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307\307"
              b"\307\307\307\307\307\307\307\307\307\307\307\307\307"),
    bytearray(b"\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310"
              b"\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310\310"
              b"\310\310\310\310\310\310\310\310\310\310\310\310\310"),
    bytearray(b"\214\125\332\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125"
              b"\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125\125"
              b"\125\125\216\216\216\216\134\126\133\127\130\132\131"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\134\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\134\134\217\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\133\133\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\133\217\217\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\133\133\133\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\134\134\134\217\217\217\217\217\217\217\217"),
    bytearray(b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217\217"
              b"\217\217\125\125\125\217\217\217\217\217\217\217\217"),
]

PARSE_DEFS = [
    (Error.LONG, None, [], [], None),
    (None, None, [1], [], None),
    (None, Emit.NULL, [], [], None),
    (None, Emit.TRUE, [], [], None),
    (None, Emit.FALSE, [], [], None),
    (None, Emit.INF, [], [], None),
    (None, Emit.NEG_INF, [], [], None),
    (None, Emit.NAN, [], [], None),
    (None, Emit.INT, [], [], None),
    (None, Emit.FLOAT, [], [], None),
    (None, Emit.STRING, [], [], None),
    (None, Emit.DATA, [], [], None),
    (None, Emit.SHORT_ARRAY_IN, [17], [], None),
    (None, Emit.SHORT_MAP_IN, [21], [], None),
    (None, None, [2], [], None),
    (None, Emit.NULL, [3], [], None),
    (None, Emit.TRUE, [3], [], None),
    (None, Emit.FALSE, [3], [], None),
    (None, Emit.INF, [3], [], None),
    (None, Emit.NEG_INF, [3], [], None),
    (None, Emit.NAN, [3], [], None),
    (None, Emit.INT, [3], [], None),
    (None, Emit.FLOAT, [3], [], None),
    (None, Emit.STRING, [3], [], None),
    (None, None, [9], [Acc.DATA], None),
    (None, None, [13], [Acc.STRING], None),
    (None, None, [11], [], None),
    (None, None, [15], [], None),
    (None, Emit.SHORT_ARRAY_IN, [3, 17], [], None),
    (None, Emit.LONG_ARRAY_IN, [19, 0], [], None),
    (None, Emit.SHORT_MAP_IN, [3, 21], [], None),
    (None, Emit.LONG_MAP_IN, [24, 0], [], Key.UNQUOTED),
    (None, Emit.LONG_MAP_IN, [24, 0], [], Key.QUOTED),
    (Error.SUFFIX, None, [], [], None),
    (None, None, [2, 5], [], None),
    (None, None, [3, 5], [], None),
    (None, None, [4], [], None),
    (None, None, [], [], None),
    (None, None, [3], [], None),
    (Error.SIBLING, None, [], [], None),
    (Error.CHILD, None, [], [], None),
    (None, None, [6], [], None),
    (None, None, [6, 5], [], None),
    (None, None, [5], [], None),
    (Error.SHORT, None, [], [], None),
    (None, None, [9], [], None),
    (None, None, [9, 5], [], None),
    (None, None, [8], [], None),
    (None, Emit.ACC_DATA, [], [], None),
    (None, None, [13], [Acc.NL, Acc.STRING], None),
    (None, None, [11], [Acc.NL], None),
    (None, None, [11, 5], [], None),
    (None, None, [10], [], None),
    (None, Emit.ACC_STRING, [], [Acc.NL], None),
    (None, None, [13], [], None),
    (None, None, [13], [Acc.SP, Acc.STRING], None),
    (None, None, [13, 5], [], None),
    (None, None, [12], [], None),
    (Error.BANG_LAST, None, [], [], None),
    (None, None, [15, 5], [], None),
    (None, None, [14], [], None),
    (None, Emit.ACC_STRING, [], [], None),
    (Error.ARRAY_END, None, [], [], None),
    (None, None, [16, 7], [], None),
    (None, Emit.SHORT_ARRAY_OUT, [], [], None),
    (None, Emit.NULL, [16], [], None),
    (None, Emit.TRUE, [16], [], None),
    (None, Emit.FALSE, [16], [], None),
    (None, Emit.INF, [16], [], None),
    (None, Emit.NEG_INF, [16], [], None),
    (None, Emit.NAN, [16], [], None),
    (None, Emit.INT, [16], [], None),
    (None, Emit.FLOAT, [16], [], None),
    (None, Emit.STRING, [16], [], None),
    (None, Emit.DATA, [16], [], None),
    (None, Emit.SHORT_ARRAY_IN, [16, 17], [], None),
    (None, Emit.SHORT_MAP_IN, [16, 21], [], None),
    (None, None, [19], [], None),
    (None, None, [19, 0], [], None),
    (None, None, [19, 5], [], None),
    (None, None, [18], [], None),
    (None, Emit.LONG_ARRAY_OUT, [], [], None),
    (Error.MAP_END, None, [], [], None),
    (None, None, [20, 22], [], None),
    (None, Emit.SHORT_MAP_OUT, [], [], None),
    (Error.MAP_KEY, None, [], [], None),
    (None, None, [20, 7], [], Key.UNQUOTED),
    (None, None, [20, 7], [], Key.QUOTED),
    (None, None, [7], [], Key.UNQUOTED),
    (None, None, [7], [], Key.QUOTED),
    (None, None, [24], [], None),
    (None, None, [24, 0], [], Key.UNQUOTED),
    (None, None, [24, 0], [], Key.QUOTED),
    (None, None, [24, 5], [], None),
    (None, None, [23], [], None),
    (None, Emit.LONG_MAP_OUT, [], [], None),
]

PARSE_TABLE = [
    [PARSE_DEFS[1], PARSE_DEFS[0], PARSE_DEFS[0], PARSE_DEFS[0], PARSE_DEFS[12],
     PARSE_DEFS[0], PARSE_DEFS[13], PARSE_DEFS[0], PARSE_DEFS[0], PARSE_DEFS[0],
     PARSE_DEFS[0], PARSE_DEFS[0], PARSE_DEFS[2], PARSE_DEFS[3], PARSE_DEFS[4],
     PARSE_DEFS[5], PARSE_DEFS[6], PARSE_DEFS[7], PARSE_DEFS[0], PARSE_DEFS[0],
     PARSE_DEFS[8], PARSE_DEFS[9], PARSE_DEFS[11], PARSE_DEFS[10], PARSE_DEFS[0],
     PARSE_DEFS[0], PARSE_DEFS[0]],
    [PARSE_DEFS[0], PARSE_DEFS[0], PARSE_DEFS[0], PARSE_DEFS[29], PARSE_DEFS[28],
     PARSE_DEFS[0], PARSE_DEFS[30], PARSE_DEFS[0], PARSE_DEFS[0], PARSE_DEFS[26],
     PARSE_DEFS[26], PARSE_DEFS[27], PARSE_DEFS[15], PARSE_DEFS[16], PARSE_DEFS[17],
     PARSE_DEFS[18], PARSE_DEFS[19], PARSE_DEFS[20], PARSE_DEFS[31], PARSE_DEFS[32],
     PARSE_DEFS[21], PARSE_DEFS[22], PARSE_DEFS[24], PARSE_DEFS[23], PARSE_DEFS[25],
     PARSE_DEFS[25], PARSE_DEFS[14]],
    [PARSE_DEFS[34], PARSE_DEFS[1], PARSE_DEFS[0], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33]],
    [PARSE_DEFS[35], PARSE_DEFS[36], PARSE_DEFS[37], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[38]],
    [PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[0], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[38]],
    [PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[0], PARSE_DEFS[40], PARSE_DEFS[40],
     PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40],
     PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40],
     PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40],
     PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40], PARSE_DEFS[40],
     PARSE_DEFS[40], PARSE_DEFS[41]],
    [PARSE_DEFS[42], PARSE_DEFS[43], PARSE_DEFS[37], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33]],
    [PARSE_DEFS[44], PARSE_DEFS[44], PARSE_DEFS[44], PARSE_DEFS[44], PARSE_DEFS[12],
     PARSE_DEFS[44], PARSE_DEFS[13], PARSE_DEFS[44], PARSE_DEFS[44], PARSE_DEFS[44],
     PARSE_DEFS[44], PARSE_DEFS[44], PARSE_DEFS[2], PARSE_DEFS[3], PARSE_DEFS[4],
     PARSE_DEFS[5], PARSE_DEFS[6], PARSE_DEFS[7], PARSE_DEFS[44], PARSE_DEFS[44],
     PARSE_DEFS[8], PARSE_DEFS[9], PARSE_DEFS[11], PARSE_DEFS[10], PARSE_DEFS[44],
     PARSE_DEFS[44], PARSE_DEFS[44]],
    [PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[24], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[45]],
    [PARSE_DEFS[46], PARSE_DEFS[47], PARSE_DEFS[48], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[45]],
    [PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[50],
     PARSE_DEFS[50], PARSE_DEFS[27], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[49],
     PARSE_DEFS[49], PARSE_DEFS[26]],
    [PARSE_DEFS[51], PARSE_DEFS[52], PARSE_DEFS[53], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33]],
    [PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[50],
     PARSE_DEFS[50], PARSE_DEFS[27], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[55],
     PARSE_DEFS[49], PARSE_DEFS[54]],
    [PARSE_DEFS[56], PARSE_DEFS[57], PARSE_DEFS[53], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33]],
    [PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[58],
     PARSE_DEFS[58], PARSE_DEFS[58], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[58],
     PARSE_DEFS[58], PARSE_DEFS[27]],
    [PARSE_DEFS[59], PARSE_DEFS[60], PARSE_DEFS[61], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33]],
    [PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62],
     PARSE_DEFS[64], PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[63], PARSE_DEFS[62],
     PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62],
     PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62],
     PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62], PARSE_DEFS[62],
     PARSE_DEFS[62], PARSE_DEFS[62]],
    [PARSE_DEFS[44], PARSE_DEFS[44], PARSE_DEFS[44], PARSE_DEFS[44], PARSE_DEFS[75],
     PARSE_DEFS[64], PARSE_DEFS[76], PARSE_DEFS[44], PARSE_DEFS[44], PARSE_DEFS[44],
     PARSE_DEFS[44], PARSE_DEFS[44], PARSE_DEFS[65], PARSE_DEFS[66], PARSE_DEFS[67],
     PARSE_DEFS[68], PARSE_DEFS[69], PARSE_DEFS[70], PARSE_DEFS[44], PARSE_DEFS[44],
     PARSE_DEFS[71], PARSE_DEFS[72], PARSE_DEFS[74], PARSE_DEFS[73], PARSE_DEFS[44],
     PARSE_DEFS[44], PARSE_DEFS[44]],
    [PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[78], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[77]],
    [PARSE_DEFS[79], PARSE_DEFS[80], PARSE_DEFS[81], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[77]],
    [PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82],
     PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[84], PARSE_DEFS[83], PARSE_DEFS[82],
     PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82],
     PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82],
     PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82], PARSE_DEFS[82],
     PARSE_DEFS[82], PARSE_DEFS[82]],
    [PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85],
     PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[84], PARSE_DEFS[85], PARSE_DEFS[85],
     PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85],
     PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[86], PARSE_DEFS[87],
     PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85],
     PARSE_DEFS[85], PARSE_DEFS[85]],
    [PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85],
     PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85],
     PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85],
     PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[88], PARSE_DEFS[89],
     PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85], PARSE_DEFS[85],
     PARSE_DEFS[85], PARSE_DEFS[85]],
    [PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[91], PARSE_DEFS[92],
     PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39], PARSE_DEFS[39],
     PARSE_DEFS[39], PARSE_DEFS[90]],
    [PARSE_DEFS[93], PARSE_DEFS[94], PARSE_DEFS[95], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33], PARSE_DEFS[33],
     PARSE_DEFS[33], PARSE_DEFS[90]],
]
