/*
==============================================================================

                                 DOOM Retro
           The classic, refined DOOM source port. For Windows PC.

==============================================================================

    Copyright © 1993-2024 by id Software LLC, a ZeniMax Media company.
    Copyright © 2013-2024 by Brad Harding <mailto:brad@doomretro.com>.

    This file is a part of DOOM Retro.

    DOOM Retro is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation, either version 3 of the license, or (at your
    option) any later version.

    DOOM Retro is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with DOOM Retro. If not, see <https://www.gnu.org/licenses/>.

    DOOM is a registered trademark of id Software LLC, a ZeniMax Media
    company, in the US and/or other countries, and is used without
    permission. All other trademarks are the property of their respective
    holders. DOOM Retro is in no way affiliated with nor endorsed by
    id Software.

==============================================================================
*/

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "i_system.h"
#include "info.h"
#include "p_local.h"
#include "r_defs.h"
#include "sounds.h"

#define EMPTYMOBJ \
{ \
    /* doomednum            */ -1,          \
    /* spawnstate           */ 0,           \
    /* spawnhealth          */ 0,           \
    /* gibhealth            */ 0,           \
    /* giblevel             */ 0,           \
    /* seestate             */ S_NULL,      \
    /* seesound             */ sfx_none,    \
    /* reactiontime         */ 0,           \
    /* attacksound          */ sfx_none,    \
    /* painstate            */ S_NULL,      \
    /* painchance           */ 0,           \
    /* painsound            */ sfx_none,    \
    /* meleestate           */ S_NULL,      \
    /* missilestate         */ S_NULL,      \
    /* deathstate           */ S_NULL,      \
    /* xdeathstate          */ S_NULL,      \
    /* deathsound           */ sfx_none,    \
    /* droppeditem          */ MT_NULL,     \
    /* speed                */ 0,           \
    /* radius               */ 0,           \
    /* pickupradius         */ 0,           \
    /* height               */ 0,           \
    /* projectilepassheight */ 0,           \
    /* mass                 */ 0,           \
    /* damage               */ 0,           \
    /* activesound          */ sfx_none,    \
    /* flags                */ 0,           \
    /* flags2               */ 0,           \
    /* raisestate           */ S_NULL,      \
    /* frames               */ 0,           \
    /* fullbright           */ false,       \
    /* bloodcolor           */ REDBLOOD,    \
    /* shadowoffset         */ 0,           \
    /* mbf21flags           */ 0,           \
    /* infightinggroup      */ IG_DEFAULT,  \
    /* projectilegroup      */ PG_DEFAULT,  \
    /* splashgroup          */ SG_DEFAULT,  \
    /* ripsound             */ sfx_none,    \
    /* altspeed             */ NO_ALTSPEED, \
    /* meleerange           */ MELEERANGE,  \
    /* name1                */ "",          \
    /* plural1              */ "",          \
    /* name2                */ "",          \
    /* plural2              */ "",          \
    /* name3                */ "",          \
    /* plural3              */ ""           \
}

mobjinfo_t original_mobjinfo[NUMMOBJTYPES] =
{
    // Player (MT_PLAYER)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_PLAY,
        /* spawnhealth          */ 100,
        /* gibhealth            */ -100,
        /* giblevel             */ 0,
        /* seestate             */ S_PLAY_RUN1,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 0,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_PLAY_PAIN,
        /* painchance           */ 255,
        /* painsound            */ sfx_plpain,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_PLAY_ATK1,
        /* deathstate           */ S_PLAY_DIE1,
        /* xdeathstate          */ S_PLAY_XDIE1,
        /* deathsound           */ sfx_pldeth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_DROPOFF | MF_PICKUP | MF_FRIEND),
        /* flags2               */ (MF2_PASSMOBJ | MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "player",
        /* plural1              */ "players",
        /* name2                */ "doomplayer",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Zombieman (MT_POSSESSED)
    {
        /* doomednum            */ Zombieman,
        /* spawnstate           */ S_POSS_STND,
        /* spawnhealth          */ 20,
        /* gibhealth            */ -20,
        /* giblevel             */ 0,
        /* seestate             */ S_POSS_RUN1,
        /* seesound             */ sfx_posit1,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_pistol,
        /* painstate            */ S_POSS_PAIN,
        /* painchance           */ 100,
        /* painsound            */ sfx_popain,
        /* meleestate           */ 0,
        /* missilestate         */ S_POSS_ATK1,
        /* deathstate           */ S_POSS_DIE1,
        /* xdeathstate          */ S_POSS_XDIE1,
        /* deathsound           */ sfx_podth1,
        /* droppeditem          */ MT_CLIP,
        /* speed                */ 8,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_posact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_POSS_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 2 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "zombieman",
        /* plural1              */ "zombiemen",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Shotgun Guy (MT_SHOTGUY)
    {
        /* doomednum            */ ShotgunGuy,
        /* spawnstate           */ S_SPOS_STND,
        /* spawnhealth          */ 30,
        /* gibhealth            */ -30,
        /* giblevel             */ 0,
        /* seestate             */ S_SPOS_RUN1,
        /* seesound             */ sfx_posit2,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_SPOS_PAIN,
        /* painchance           */ 170,
        /* painsound            */ sfx_popain,
        /* meleestate           */ 0,
        /* missilestate         */ S_SPOS_ATK1,
        /* deathstate           */ S_SPOS_DIE1,
        /* xdeathstate          */ S_SPOS_XDIE1,
        /* deathsound           */ sfx_podth2,
        /* droppeditem          */ MT_SHOTGUN,
        /* speed                */ 8,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_posact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_SPOS_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 2 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "shotgun guy",
        /* plural1              */ "shotgun guys",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Arch-vile (MT_VILE)
    {
        /* doomednum            */ ArchVile,
        /* spawnstate           */ S_VILE_STND,
        /* spawnhealth          */ 700,
        /* gibhealth            */ -700,
        /* giblevel             */ 0,
        /* seestate             */ S_VILE_RUN1,
        /* seesound             */ sfx_vilsit,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_VILE_PAIN,
        /* painchance           */ 10,
        /* painsound            */ sfx_vipain,
        /* meleestate           */ 0,
        /* missilestate         */ S_VILE_ATK1,
        /* deathstate           */ S_VILE_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_vildth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 15,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 500,
        /* damage               */ 0,
        /* activesound          */ sfx_vilact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 8 * FRACUNIT,
        /* mbf21flags           */ (MF_MBF21_SHORTMRANGE | MF_MBF21_DMGIGNORED | MF_MBF21_NOTHRESHOLD),
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "arch-vile",
        /* plural1              */ "arch-viles",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Arch-vile Fire (MT_FIRE)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_FIRE1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_NOGRAVITY),
        /* flags2               */ (MF2_TRANSLUCENT | MF2_NOLIQUIDBOB),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "arch-vile fire",
        /* plural1              */ "arch-vile fire",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Revenant (MT_UNDEAD)
    {
        /* doomednum            */ Revenant,
        /* spawnstate           */ S_SKEL_STND,
        /* spawnhealth          */ 300,
        /* gibhealth            */ -300,
        /* giblevel             */ 0,
        /* seestate             */ S_SKEL_RUN1,
        /* seesound             */ sfx_skesit,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_SKEL_PAIN,
        /* painchance           */ 100,
        /* painsound            */ sfx_popain,
        /* meleestate           */ S_SKEL_FIST1,
        /* missilestate         */ S_SKEL_MISS1,
        /* deathstate           */ S_SKEL_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_skedth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 10,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 500,
        /* damage               */ 0,
        /* activesound          */ sfx_skeact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_SKEL_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 4 * FRACUNIT,
        /* mbf21flags           */ (MF_MBF21_LONGMELEE | MF_MBF21_RANGEHALF),
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "revenant",
        /* plural1              */ "revenants",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Revenant Projectile (MT_TRACER)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_TRACER,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_skeatk,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_TRACEEXP1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_barexp,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 10 * FRACUNIT,
        /* radius               */ 11 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 8 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 10,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_NOGRAVITY),
        /* flags2               */ (MF2_TRANSLUCENT | MF2_NOLIQUIDBOB),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "revenant",
        /* plural1              */ "revenants",
        /* name2                */ "revenanttracer",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Smoke (MT_SMOKE)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_SMOKE1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_NOGRAVITY),
        /* flags2               */ (MF2_TRANSLUCENT_33 | MF2_NOLIQUIDBOB),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "smoke",
        /* plural1              */ "smoke",
        /* name2                */ "revenanttracersmoke",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Mancubus (MT_FATSO)
    {
        /* doomednum            */ Mancubus,
        /* spawnstate           */ S_FATT_STND,
        /* spawnhealth          */ 600,
        /* gibhealth            */ -600,
        /* giblevel             */ 0,
        /* seestate             */ S_FATT_RUN1,
        /* seesound             */ sfx_mansit,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_FATT_PAIN,
        /* painchance           */ 80,
        /* painsound            */ sfx_mnpain,
        /* meleestate           */ 0,
        /* missilestate         */ S_FATT_ATK1,
        /* deathstate           */ S_FATT_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_mandth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 8,
        /* radius               */ 48 * FRACUNIT,
        /* pickupradius         */ 48 * FRACUNIT,
        /* height               */ 64 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 1000,
        /* damage               */ 0,
        /* activesound          */ sfx_posact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_FATT_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 2 * FRACUNIT,
        /* mbf21flags           */ MF_MBF21_MAP07BOSS1,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "mancubus",
        /* plural1              */ "mancubi",
        /* name2                */ "fatso",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Mancubus Projectile (MT_FATSHOT)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_FATSHOT1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_firsht,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_FATSHOTX1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_firxpl,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 20 * FRACUNIT,
        /* radius               */ 6 * FRACUNIT,
        /* pickupradius         */ 6 * FRACUNIT,
        /* height               */ 8 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 8,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_NOGRAVITY),
        /* flags2               */ (MF2_TRANSLUCENT | MF2_NOLIQUIDBOB),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "mancubus projectile",
        /* plural1              */ "mancubi projectiles",
        /* name2                */ "fatshot",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Heavy Weapon Dude (MT_CHAINGUY)
    {
        /* doomednum            */ HeavyWeaponDude,
        /* spawnstate           */ S_CPOS_STND,
        /* spawnhealth          */ 70,
        /* gibhealth            */ -70,
        /* giblevel             */ 0,
        /* seestate             */ S_CPOS_RUN1,
        /* seesound             */ sfx_posit2,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_CPOS_PAIN,
        /* painchance           */ 170,
        /* painsound            */ sfx_popain,
        /* meleestate           */ 0,
        /* missilestate         */ S_CPOS_ATK1,
        /* deathstate           */ S_CPOS_DIE1,
        /* xdeathstate          */ S_CPOS_XDIE1,
        /* deathsound           */ sfx_podth2,
        /* droppeditem          */ MT_CHAINGUN,
        /* speed                */ 8,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_posact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP | MF2_NOMIRROREDCORPSE),
        /* raisestate           */ S_CPOS_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 2 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "chaingunner",
        /* plural1              */ "chaingunners",
        /* name2                */ "heavy weapon dude",
        /* plural2              */ "heavy weapon dudes",
        /* name3                */ "chaingunguy",
        /* plural3              */ ""
    },

    // Imp (MT_TROOP)
    {
        /* doomednum            */ Imp,
        /* spawnstate           */ S_TROO_STND,
        /* spawnhealth          */ 60,
        /* gibhealth            */ -60,
        /* giblevel             */ 0,
        /* seestate             */ S_TROO_RUN1,
        /* seesound             */ sfx_bgsit1,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_TROO_PAIN,
        /* painchance           */ 200,
        /* painsound            */ sfx_popain,
        /* meleestate           */ S_TROO_ATK1,
        /* missilestate         */ S_TROO_ATK1,
        /* deathstate           */ S_TROO_DIE1,
        /* xdeathstate          */ S_TROO_XDIE1,
        /* deathsound           */ sfx_bgdth1,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 8,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_bgact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_TROO_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 2 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "imp",
        /* plural1              */ "imps",
        /* name2                */ "doomimp",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Demon (MT_SERGEANT)
    {
        /* doomednum            */ Demon,
        /* spawnstate           */ S_SARG_STND,
        /* spawnhealth          */ 150,
        /* gibhealth            */ -150,
        /* giblevel             */ 0,
        /* seestate             */ S_SARG_RUN1,
        /* seesound             */ sfx_sgtsit,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_sgtatk,
        /* painstate            */ S_SARG_PAIN,
        /* painchance           */ 180,
        /* painsound            */ sfx_dmpain,
        /* meleestate           */ S_SARG_ATK1,
        /* missilestate         */ 0,
        /* deathstate           */ S_SARG_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_sgtdth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 10,
        /* radius               */ 30 * FRACUNIT,
        /* pickupradius         */ 30 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 400,
        /* damage               */ 0,
        /* activesound          */ sfx_dmact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_SARG_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 2 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "pinky demon",
        /* plural1              */ "pinky demons",
        /* name2                */ "demon",
        /* plural2              */ "demons",
        /* name3                */ "pinky",
        /* plural3              */ "pinkies"
    },

    // Spectre (MT_SHADOWS)
    {
        /* doomednum            */ Spectre,
        /* spawnstate           */ S_SARG_STND,
        /* spawnhealth          */ 150,
        /* gibhealth            */ -150,
        /* giblevel             */ 0,
        /* seestate             */ S_SARG_RUN1,
        /* seesound             */ sfx_sgtsit,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_sgtatk,
        /* painstate            */ S_SARG_PAIN,
        /* painchance           */ 180,
        /* painsound            */ sfx_dmpain,
        /* meleestate           */ S_SARG_ATK1,
        /* missilestate         */ 0,
        /* deathstate           */ S_SARG_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_sgtdth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 10,
        /* radius               */ 30 * FRACUNIT,
        /* pickupradius         */ 30 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 400,
        /* damage               */ 0,
        /* activesound          */ sfx_dmact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_FUZZ | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_SARG_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 2 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "spectre",
        /* plural1              */ "spectres",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Cacodemon (MT_HEAD)
    {
        /* doomednum            */ Cacodemon,
        /* spawnstate           */ S_HEAD_STND,
        /* spawnhealth          */ 400,
        /* gibhealth            */ -400,
        /* giblevel             */ 0,
        /* seestate             */ S_HEAD_RUN1,
        /* seesound             */ sfx_cacsit,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_HEAD_PAIN,
        /* painchance           */ 128,
        /* painsound            */ sfx_dmpain,
        /* meleestate           */ 0,
        /* missilestate         */ S_HEAD_ATK1,
        /* deathstate           */ S_HEAD_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_cacdth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 8,
        /* radius               */ 31 * FRACUNIT,
        /* pickupradius         */ 31 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 400,
        /* damage               */ 0,
        /* activesound          */ sfx_dmact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_FLOAT | MF_NOGRAVITY | MF_COUNTKILL),
        /* flags2               */ (MF2_PASSMOBJ | MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_HEAD_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ BLUEBLOOD,
        /* shadowoffset         */ 6 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "cacodemon",
        /* plural1              */ "cacodemons",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Baron of Hell (MT_BRUISER)
    {
        /* doomednum            */ BaronOfHell,
        /* spawnstate           */ S_BOSS_STND,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ -1000,
        /* giblevel             */ 0,
        /* seestate             */ S_BOSS_RUN1,
        /* seesound             */ sfx_brssit,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_BOSS_PAIN,
        /* painchance           */ 50,
        /* painsound            */ sfx_dmpain,
        /* meleestate           */ S_BOSS_ATK1,
        /* missilestate         */ S_BOSS_ATK1,
        /* deathstate           */ S_BOSS_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_brsdth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 8,
        /* radius               */ 24 * FRACUNIT,
        /* pickupradius         */ 24 * FRACUNIT,
        /* height               */ 64 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 1000,
        /* damage               */ 0,
        /* activesound          */ sfx_dmact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_BOSS_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ GREENBLOOD,
        /* shadowoffset         */ 4 * FRACUNIT,
        /* mbf21flags           */ MF_MBF21_E1M8BOSS,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_BARON,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "baron of hell",
        /* plural1              */ "barons of hell",
        /* name2                */ "baron",
        /* plural2              */ "barons",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Baron of Hell and Hell Knight Projectile (MT_BRUISERSHOT)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_BRBALL1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_firsht,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_BRBALLX1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_firxpl,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 15 * FRACUNIT,
        /* radius               */ 6 * FRACUNIT,
        /* pickupradius         */ 6 * FRACUNIT,
        /* height               */ 8 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 8,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_NOGRAVITY),
        /* flags2               */ (MF2_TRANSLUCENT | MF2_NOLIQUIDBOB),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ 20 * FRACUNIT,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hell noble projectile",
        /* plural1              */ "hell noble projectiles",
        /* name2                */ "baronball",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hell Knight (MT_KNIGHT)
    {
        /* doomednum            */ HellKnight,
        /* spawnstate           */ S_BOS2_STND,
        /* spawnhealth          */ 500,
        /* gibhealth            */ -500,
        /* giblevel             */ 0,
        /* seestate             */ S_BOS2_RUN1,
        /* seesound             */ sfx_kntsit,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_BOS2_PAIN,
        /* painchance           */ 50,
        /* painsound            */ sfx_dmpain,
        /* meleestate           */ S_BOS2_ATK1,
        /* missilestate         */ S_BOS2_ATK1,
        /* deathstate           */ S_BOS2_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_kntdth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 8,
        /* radius               */ 24 * FRACUNIT,
        /* pickupradius         */ 24 * FRACUNIT,
        /* height               */ 64 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 1000,
        /* damage               */ 0,
        /* activesound          */ sfx_dmact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_BOS2_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ GREENBLOOD,
        /* shadowoffset         */ 4 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_BARON,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hell knight",
        /* plural1              */ "hell knights",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Lost Soul (MT_SKULL)
    {
        /* doomednum            */ LostSoul,
        /* spawnstate           */ S_SKULL_STND,
        /* spawnhealth          */ 100,
        /* gibhealth            */ -100,
        /* giblevel             */ 0,
        /* seestate             */ S_SKULL_RUN1,
        /* seesound             */ 0,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_sklatk,
        /* painstate            */ S_SKULL_PAIN,
        /* painchance           */ 256,
        /* painsound            */ sfx_dmpain,
        /* meleestate           */ 0,
        /* missilestate         */ S_SKULL_ATK1,
        /* deathstate           */ S_SKULL_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_firxpl,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 8,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 50,
        /* damage               */ 3,
        /* activesound          */ sfx_dmact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_FLOAT | MF_NOGRAVITY | MF_COUNTKILL),
        /* flags2               */ (MF2_PASSMOBJ | MF2_CASTSHADOW | MF2_TRANSLUCENT_REDONLY | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ MF_MBF21_RANGEHALF,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "lost soul",
        /* plural1              */ "lost souls",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Spider Mastermind (MT_SPIDER)
    {
        /* doomednum            */ SpiderMastermind,
        /* spawnstate           */ S_SPID_STND,
        /* spawnhealth          */ 3000,
        /* gibhealth            */ -3000,
        /* giblevel             */ 0,
        /* seestate             */ S_SPID_RUN1,
        /* seesound             */ sfx_spisit,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_shotgn,
        /* painstate            */ S_SPID_PAIN,
        /* painchance           */ 40,
        /* painsound            */ sfx_dmpain,
        /* meleestate           */ 0,
        /* missilestate         */ S_SPID_ATK1,
        /* deathstate           */ S_SPID_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_spidth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 12,
        /* radius               */ 128 * FRACUNIT,
        /* pickupradius         */ 128 * FRACUNIT,
        /* height               */ 100 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 1000,
        /* damage               */ 0,
        /* activesound          */ sfx_dmact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 12 * FRACUNIT,
        /* mbf21flags           */ (MF_MBF21_NORADIUSDMG | MF_MBF21_RANGEHALF | MF_MBF21_FULLVOLSOUNDS | MF_MBF21_E3M8BOSS | MF_MBF21_E4M8BOSS),
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "spider mastermind",
        /* plural1              */ "spider masterminds",
        /* name2                */ "spiderdemon",
        /* plural2              */ "spiderdemons",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Arachnotron (MT_BABY)
    {
        /* doomednum            */ Arachnotron,
        /* spawnstate           */ S_BSPI_STND,
        /* spawnhealth          */ 500,
        /* gibhealth            */ -500,
        /* giblevel             */ 0,
        /* seestate             */ S_BSPI_SIGHT,
        /* seesound             */ sfx_bspsit,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_BSPI_PAIN,
        /* painchance           */ 128,
        /* painsound            */ sfx_dmpain,
        /* meleestate           */ 0,
        /* missilestate         */ S_BSPI_ATK1,
        /* deathstate           */ S_BSPI_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_bspdth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 12,
        /* radius               */ 64 * FRACUNIT,
        /* pickupradius         */ 64 * FRACUNIT,
        /* height               */ 64 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 600,
        /* damage               */ 0,
        /* activesound          */ sfx_bspact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_BSPI_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 4 * FRACUNIT,
        /* mbf21flags           */ MF_MBF21_MAP07BOSS2,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "arachnotron",
        /* plural1              */ "arachnotrons",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Cyberdemon (MT_CYBORG)
    {
        /* doomednum            */ Cyberdemon,
        /* spawnstate           */ S_CYBER_STND,
        /* spawnhealth          */ 4000,
        /* gibhealth            */ -4000,
        /* giblevel             */ 0,
        /* seestate             */ S_CYBER_RUN1,
        /* seesound             */ sfx_cybsit,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_CYBER_PAIN,
        /* painchance           */ 20,
        /* painsound            */ sfx_dmpain,
        /* meleestate           */ 0,
        /* missilestate         */ S_CYBER_ATK1,
        /* deathstate           */ S_CYBER_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_cybdth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 16,
        /* radius               */ 40 * FRACUNIT,
        /* pickupradius         */ 40 * FRACUNIT,
        /* height               */ 110 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 1000,
        /* damage               */ 0,
        /* activesound          */ sfx_dmact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP | MF2_NOMIRROREDCORPSE),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 4 * FRACUNIT,
        /* mbf21flags           */ (MF_MBF21_NORADIUSDMG | MF_MBF21_HIGHERMPROB | MF_MBF21_RANGEHALF | MF_MBF21_FULLVOLSOUNDS | MF_MBF21_E2M8BOSS | MF_MBF21_E4M6BOSS),
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "cyberdemon",
        /* plural1              */ "cyberdemons",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Pain Elemental (MT_PAIN)
    {
        /* doomednum            */ PainElemental,
        /* spawnstate           */ S_PAIN_STND,
        /* spawnhealth          */ 400,
        /* gibhealth            */ -400,
        /* giblevel             */ 0,
        /* seestate             */ S_PAIN_RUN1,
        /* seesound             */ sfx_pesit,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_PAIN_PAIN,
        /* painchance           */ 128,
        /* painsound            */ sfx_pepain,
        /* meleestate           */ 0,
        /* missilestate         */ S_PAIN_ATK1,
        /* deathstate           */ S_PAIN_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_pedth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 8,
        /* radius               */ 31 * FRACUNIT,
        /* pickupradius         */ 31 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 400,
        /* damage               */ 0,
        /* activesound          */ sfx_dmact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_FLOAT | MF_NOGRAVITY | MF_COUNTKILL),
        /* flags2               */ (MF2_PASSMOBJ | MF2_CASTSHADOW | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_PAIN_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "pain elemental",
        /* plural1              */ "pain elementals",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Wolfenstein SS (MT_WOLFSS)
    {
        /* doomednum            */ WolfensteinSS,
        /* spawnstate           */ S_SSWV_STND,
        /* spawnhealth          */ 50,
        /* gibhealth            */ -50,
        /* giblevel             */ 0,
        /* seestate             */ S_SSWV_RUN1,
        /* seesound             */ sfx_sssit,
        /* reactiontime         */ 8,
        /* attacksound          */ 0,
        /* painstate            */ S_SSWV_PAIN,
        /* painchance           */ 170,
        /* painsound            */ sfx_popain,
        /* meleestate           */ 0,
        /* missilestate         */ S_SSWV_ATK1,
        /* deathstate           */ S_SSWV_DIE1,
        /* xdeathstate          */ S_SSWV_XDIE1,
        /* deathsound           */ sfx_ssdth,
        /* droppeditem          */ MT_CLIP,
        /* speed                */ 8,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_posact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_SSWV_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "Wolfenstein SS",
        /* plural1              */ "Wolfenstein SS",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Commander Keen (MT_KEEN)
    {
        /* doomednum            */ CommanderKeen,
        /* spawnstate           */ S_KEENSTND,
        /* spawnhealth          */ 100,
        /* gibhealth            */ -100,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_KEENPAIN,
        /* painchance           */ 256,
        /* painsound            */ sfx_keenpn,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_COMMKEEN,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_keendt,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 67 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 10000000,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "commander keen",
        /* plural1              */ "commander keens",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Boss Brain (MT_BOSSBRAIN)
    {
        /* doomednum            */ BossBrain,
        /* spawnstate           */ S_BRAIN,
        /* spawnhealth          */ 250,
        /* gibhealth            */ -250,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_BRAIN_PAIN,
        /* painchance           */ 255,
        /* painsound            */ sfx_bospn,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_BRAIN_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_bosdth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 10000000,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE),
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "icon of sin",
        /* plural1              */ "icons of sin",
        /* name2                */ "bossbrain",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Monsters Spawner (MT_BOSSSPIT)
    {
        /* doomednum            */ MonsterSpawner,
        /* spawnstate           */ S_BRAINEYE,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_BRAINEYESEE,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 32 * FRACUNIT,
        /* height               */ 32 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_NOSECTOR),
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "monster spawner",
        /* plural1              */ "monster spawners",
        /* name2                */ "bosseye",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Monsters Target (MT_BOSSTARGET)
    {
        /* doomednum            */ MonstersTarget,
        /* spawnstate           */ S_NULL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 32 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_NOSECTOR),
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "monsters target",
        /* plural1              */ "monsters targets",
        /* name2                */ "bosstarget",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Boss Brain Projectile (MT_SPAWNSHOT)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_SPAWN1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_bospit,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_firxpl,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 10 * FRACUNIT,
        /* radius               */ 6 * FRACUNIT,
        /* pickupradius         */ 6 * FRACUNIT,
        /* height               */ 32 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 3,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_NOGRAVITY | MF_NOCLIP),
        /* flags2               */ MF2_CASTSHADOW,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "icon of sin",
        /* plural1              */ "icons of sin",
        /* name2                */ "spawnshot",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Boss Brain Fire (MT_SPAWNFIRE)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_SPAWNFIRE1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_NOGRAVITY),
        /* flags2               */ MF2_TRANSLUCENT,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "icon of sin fire",
        /* plural1              */ "icon of sin fire",
        /* name2                */ "spawnfire",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Barrel (MT_BARREL)
    {
        /* doomednum            */ Barrel,
        /* spawnstate           */ S_BAR1,
        /* spawnhealth          */ 20,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_BEXP,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_barexp,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 10 * FRACUNIT,
        /* pickupradius         */ 10 * FRACUNIT,
        /* height               */ 42 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_NOBLOOD),
        /* flags2               */ (MF2_CASTSHADOW | MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 3,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "barrel",
        /* plural1              */ "barrels",
        /* name2                */ "explosivebarrel",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Imp Projectile (MT_TROOPSHOT)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_TBALL1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_firsht,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_TBALLX1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_firxpl,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 10 * FRACUNIT,
        /* radius               */ 6 * FRACUNIT,
        /* pickupradius         */ 6 * FRACUNIT,
        /* height               */ 8 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 3,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_NOGRAVITY),
        /* flags2               */ MF2_TRANSLUCENT,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ 20 * FRACUNIT,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "imp projectile ",
        /* plural1              */ "imp projectiles",
        /* name2                */ "doomimpball",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Cacodemon Projectile (MT_HEADSHOT)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_RBALL1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_firsht,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_RBALLX1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_firxpl,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 10 * FRACUNIT,
        /* radius               */ 6 * FRACUNIT,
        /* pickupradius         */ 6 * FRACUNIT,
        /* height               */ 8 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 5,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_NOGRAVITY),
        /* flags2               */ MF2_TRANSLUCENT,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ 20 * FRACUNIT,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "cacodemon projectile",
        /* plural1              */ "cacodemon projectiles",
        /* name2                */ "cacodemonball",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Rocket Launcher Projectile (MT_ROCKET)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_ROCKET,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_rlaunc,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_EXPLODE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_barexp,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 20 * FRACUNIT,
        /* radius               */ 11 * FRACUNIT,
        /* pickupradius         */ 11 * FRACUNIT,
        /* height               */ 8 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 20,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_NOGRAVITY),
        /* flags2               */ MF2_CASTSHADOW,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "rocket",
        /* plural1              */ "",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Plasma Rifle Projectile (MT_PLASMA)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_PLASBALL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_plasma,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_PLASEXP,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_firxpl,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 25 * FRACUNIT,
        /* radius               */ 13 * FRACUNIT,
        /* pickupradius         */ 13 * FRACUNIT,
        /* height               */ 8 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 5,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_NOGRAVITY),
        /* flags2               */ MF2_TRANSLUCENT,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "plasma ball",
        /* plural1              */ "",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // BFG-9000 Projectile (MT_BFG)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_BFGSHOT,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ 0,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_BFGLAND,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_rxplod,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 25 * FRACUNIT,
        /* radius               */ 13 * FRACUNIT,
        /* pickupradius         */ 13 * FRACUNIT,
        /* height               */ 8 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 100,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_NOGRAVITY),
        /* flags2               */ MF2_TRANSLUCENT,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "BFG-9000 projectile",
        /* plural1              */ "BFG-9000 projectiles",
        /* name2                */ "bfgball",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Arachnotron Projectile (MT_ARACHPLAZ)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_ARACH_PLAZ,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_plasma,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_ARACH_PLEX,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_firxpl,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 25 * FRACUNIT,
        /* radius               */ 13 * FRACUNIT,
        /* pickupradius         */ 13 * FRACUNIT,
        /* height               */ 8 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 5,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_NOGRAVITY),
        /* flags2               */ MF2_TRANSLUCENT,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "arachnotron's projectile",
        /* plural1              */ "arachnotron projectiles",
        /* name2                */ "arachnotronplasma",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Bullet Puff (MT_PUFF)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_PUFF1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 0,
        /* pickupradius         */ 0,
        /* height               */ 0,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_NOGRAVITY),
        /* flags2               */ MF2_TRANSLUCENT_33,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "bullet puff",
        /* plural1              */ "bullet puffs",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Blood (MT_BLOOD)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_BLOOD1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 0,
        /* pickupradius         */ 0,
        /* height               */ 0,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ 0,
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ 184,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "red blood",
        /* plural1              */ "red blood",
        /* name2                */ "blood",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Teleport Fog (MT_TFOG)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_TFOG,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_NOGRAVITY),
        /* flags2               */ MF2_TRANSLUCENT,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "teleport fog",
        /* plural1              */ "teleport fogs",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Item Fog (MT_IFOG)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_IFOG,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_NOGRAVITY),
        /* flags2               */ MF2_TRANSLUCENT,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "item fog",
        /* plural1              */ "item fogs",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Teleport Destination (MT_TELEPORTMAN)
    {
        /* doomednum            */ TeleportDestination,
        /* spawnstate           */ S_NULL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_NOSECTOR),
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "teleport destination",
        /* plural1              */ "teleport destinations",
        /* name2                */ "teleportdest",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // BFG-9000 Secondary Projectile (MT_EXTRABFG)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_BFGEXP,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_NOGRAVITY),
        /* flags2               */ MF2_TRANSLUCENT,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "BFG-9000 secondary projectile",
        /* plural1              */ "BFG-9000 secondary projectiles",
        /* name2                */ "bfgextra",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Green Armor (MT_MISC0)
    {
        /* doomednum            */ GreenArmor,
        /* spawnstate           */ S_ARM1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 15 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "green armor",
        /* plural1              */ "green armor",
        /* name2                */ "green armour",
        /* plural2              */ "green armour",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Blue Armor (MT_MISC1)
    {
        /* doomednum            */ BlueArmor,
        /* spawnstate           */ S_ARM2,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 15 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "blue armor",
        /* plural1              */ "blue armor",
        /* name2                */ "blue armour",
        /* plural2              */ "blue armour",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Health Bonus (MT_MISC2)
    {
        /* doomednum            */ HealthBonus,
        /* spawnstate           */ S_BON1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 7 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPECIAL | MF_COUNTITEM),
        /* flags2               */ (MF2_CASTSHADOW | MF2_TRANSLUCENT_BLUE_25 | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "health bonus",
        /* plural1              */ "health bonuses",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Armor Bonus (MT_MISC3)
    {
        /* doomednum            */ ArmorBonus,
        /* spawnstate           */ S_BON2,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 8 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPECIAL | MF_COUNTITEM),
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "armor bonus",
        /* plural1              */ "armor bonuses",
        /* name2                */ "armour bonus",
        /* plural2              */ "armour bonuses",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Blue Keycard (MT_MISC4)
    {
        /* doomednum            */ BlueKeycard,
        /* spawnstate           */ S_BKEY,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 7 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "blue keycard",
        /* plural1              */ "blue keycards",
        /* name2                */ "bluecard",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Red Keycard (MT_MISC5)
    {
        /* doomednum            */ RedKeycard,
        /* spawnstate           */ S_RKEY,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 7 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "red keycard",
        /* plural1              */ "red keycards",
        /* name2                */ "redcard",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Yellow Keycard (MT_MISC6)
    {
        /* doomednum            */ YellowKeycard,
        /* spawnstate           */ S_YKEY,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 7 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "yellow keycard",
        /* plural1              */ "yellow keycards",
        /* name2                */ "yellowcard",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Yellow Skull Key (MT_MISC7)
    {
        /* doomednum            */ YellowSkullKey,
        /* spawnstate           */ S_YSKULL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 6 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "yellow skull key",
        /* plural1              */ "yellow skull keys",
        /* name2                */ "yellowskull",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Red Skull Key (MT_MISC8)
    {
        /* doomednum            */ RedSkullKey,
        /* spawnstate           */ S_RSKULL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 6 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "red skull key",
        /* plural1              */ "red skull keys",
        /* name2                */ "redskull",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Blue Skull Key (MT_MISC9)
    {
        /* doomednum            */ BlueSkullKey,
        /* spawnstate           */ S_BSKULL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 6 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "blue skull key",
        /* plural1              */ "blue skull keys",
        /* name2                */ "blueskull",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Stimpack (MT_MISC10)
    {
        /* doomednum            */ Stimpack,
        /* spawnstate           */ S_STIM,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 7 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "stimpack",
        /* plural1              */ "stimpacks",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Medikit (MT_MISC11)
    {
        /* doomednum            */ Medikit,
        /* spawnstate           */ S_MEDI,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 14 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "medikit",
        /* plural1              */ "medikits",
        /* name2                */ "medkit",
        /* plural2              */ "medkits",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // SoulSphere (MT_MISC12)
    {
        /* doomednum            */ SoulSphere,
        /* spawnstate           */ S_SOUL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 12 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPECIAL | MF_COUNTITEM),
        /* flags2               */ (MF2_CASTSHADOW | MF2_TRANSLUCENT_33 | MF2_FLOATBOB),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "soulsphere",
        /* plural1              */ "soulspheres",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Invulnerability (MT_INV)
    {
        /* doomednum            */ Invulnerability,
        /* spawnstate           */ S_PINV,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 12 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPECIAL | MF_COUNTITEM),
        /* flags2               */ (MF2_CASTSHADOW | MF2_TRANSLUCENT_33 | MF2_FLOATBOB),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "invulnerability power-up",
        /* plural1              */ "invulnerability power-ups",
        /* name2                */ "invulnerability",
        /* plural2              */ "invulnerabilities",
        /* name3                */ "invulnerabilitysphere",
        /* plural3              */ ""
    },

    // Berserk (MT_MISC13)
    {
        /* doomednum            */ Berserk,
        /* spawnstate           */ S_PSTR,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 14 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPECIAL | MF_COUNTITEM),
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "berserk power-up",
        /* plural1              */ "berserk power-ups",
        /* name2                */ "berserk",
        /* plural2              */ "berserks",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Partial Invisibility (MT_INS)
    {
        /* doomednum            */ PartialInvisibility,
        /* spawnstate           */ S_PINS,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 12 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPECIAL | MF_COUNTITEM),
        /* flags2               */ (MF2_CASTSHADOW | MF2_TRANSLUCENT_33 | MF2_FLOATBOB),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "partial invisibility power-up",
        /* plural1              */ "partial invisibility power-ups",
        /* name2                */ "partial invisibility",
        /* plural2              */ "partial invisibilities",
        /* name3                */ "blursphere",
        /* plural3              */ ""
    },

    // Radiation Shielding Suit (MT_MISC14)
    {
        /* doomednum            */ RadiationShieldingSuit,
        /* spawnstate           */ S_SUIT,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 12 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FLOATBOB),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "radiation shielding suit power-up",
        /* plural1              */ "radiation shielding suit power-ups",
        /* name2                */ "radiation shielding suit",
        /* plural2              */ "radiation shielding suits",
        /* name3                */ "radsuit",
        /* plural3              */ "radsuits"
    },

    // Computer Area Map (MT_MISC15)
    {
        /* doomednum            */ ComputerAreaMap,
        /* spawnstate           */ S_PMAP,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 14 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPECIAL | MF_COUNTITEM),
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "computer area map power-up",
        /* plural1              */ "computer area map power-ups",
        /* name2                */ "computer area map",
        /* plural2              */ "computer area maps",
        /* name3                */ "allmap",
        /* plural3              */ ""
    },

    // Light Amplification Visor (MT_MISC16)
    {
        /* doomednum            */ LightAmplificationVisor,
        /* spawnstate           */ S_PVIS,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 14 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPECIAL | MF_COUNTITEM),
        /* flags2               */ (MF2_CASTSHADOW | MF2_TRANSLUCENT_REDONLY | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "light amplification visor power-up",
        /* plural1              */ "light amplification visor power-ups",
        /* name2                */ "light amplification visor",
        /* plural2              */ "light amplification visors",
        /* name3                */ "infrared",
        /* plural3              */ ""
    },

    // MegaSphere (MT_MEGA)
    {
        /* doomednum            */ MegaSphere,
        /* spawnstate           */ S_MEGA,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 12 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPECIAL | MF_COUNTITEM),
        /* flags2               */ (MF2_CASTSHADOW | MF2_TRANSLUCENT_33 | MF2_FLOATBOB),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "megasphere",
        /* plural1              */ "megaspheres",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Clip (MT_CLIP)
    {
        /* doomednum            */ Clip,
        /* spawnstate           */ S_CLIP,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 4 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 20,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "clip",
        /* plural1              */ "clips",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Box of Bullets (MT_MISC17)
    {
        /* doomednum            */ BoxOfBullets,
        /* spawnstate           */ S_AMMO,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 14 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "box of bullets",
        /* plural1              */ "boxes of bullets",
        /* name2                */ "clip box",
        /* plural2              */ "clip boxes",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Rockets (MT_MISC18)
    {
        /* doomednum            */ Rocket,
        /* spawnstate           */ S_ROCK,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 6 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 1 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "rockets",
        /* plural1              */ "rockets",
        /* name2                */ "rocketammo",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Box of Rockets (MT_MISC19)
    {
        /* doomednum            */ BoxOfRockets,
        /* spawnstate           */ S_BROK,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "box of rockets",
        /* plural1              */ "boxes of rockets",
        /* name2                */ "rocket box",
        /* plural2              */ "rocket boxes",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Cell (MT_MISC20)
    {
        /* doomednum            */ Cell,
        /* spawnstate           */ S_CELL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 8 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "cells",
        /* plural1              */ "cells",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Cell Pack (MT_MISC21)
    {
        /* doomednum            */ CellPack,
        /* spawnstate           */ S_CELP,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "cell pack",
        /* plural1              */ "cell packs",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Shells (MT_MISC22)
    {
        /* doomednum            */ ShotgunShells,
        /* spawnstate           */ S_SHEL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 7 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "shells",
        /* plural1              */ "shells",
        /* name2                */ "shotgun shells",
        /* plural2              */ "shotgun shells",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Box of Shells (MT_MISC23)
    {
        /* doomednum            */ BoxOfShells,
        /* spawnstate           */ S_SBOX,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "box of shells",
        /* plural1              */ "boxes of shells",
        /* name2                */ "shell box",
        /* plural2              */ "shell boxes",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Backpack (MT_MISC24)
    {
        /* doomednum            */ Backpack,
        /* spawnstate           */ S_BPAK,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 11 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "backpack",
        /* plural1              */ "backpacks",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // BFG-9000 (MT_MISC25)
    {
        /* doomednum            */ BFG9000,
        /* spawnstate           */ S_BFUG,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "BFG-9000",
        /* plural1              */ "BFG-9000s",
        /* name2                */ "BFG",
        /* plural2              */ "BFGs",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Chaingun (MT_CHAINGUN)
    {
        /* doomednum            */ Chaingun,
        /* spawnstate           */ S_MGUN,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 50,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "chaingun",
        /* plural1              */ "chainguns",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Chainsaw (MT_MISC26)
    {
        /* doomednum            */ Chainsaw,
        /* spawnstate           */ S_CSAW,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "chainsaw",
        /* plural1              */ "chainsaws",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Rocket Launcher (MT_MISC27)
    {
        /* doomednum            */ RocketLauncher,
        /* spawnstate           */ S_LAUN,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "rocket launcher",
        /* plural1              */ "rocket launchers",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Plasma Rifle (MT_MISC28)
    {
        /* doomednum            */ PlasmaRifle,
        /* spawnstate           */ S_PLAS,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "plasma rifle",
        /* plural1              */ "plasma rifles",
        /* name2                */ "plasma gun",
        /* plural2              */ "plasma guns",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Shotgun (MT_SHOTGUN)
    {
        /* doomednum            */ Shotgun,
        /* spawnstate           */ S_SHOT,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 40,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "shotgun",
        /* plural1              */ "shotguns",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Super Shotgun (MT_SUPERSHOTGUN)
    {
        /* doomednum            */ SuperShotgun,
        /* spawnstate           */ S_SHOT2,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SPECIAL,
        /* flags2               */ (MF2_CASTSHADOW | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "super shotgun",
        /* plural1              */ "super shotguns",
        /* name2                */ "double-barreled shotgun",
        /* plural2              */ "double-barreled shotguns",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Tall techno floor lamp (MT_MISC29)
    {
        /* doomednum            */ TallTechnoFloorLamp,
        /* spawnstate           */ S_TECHLAMP,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 80 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_TRANSLUCENT_BLUEONLY | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "tall techno floor lamp",
        /* plural1              */ "tall techno floor lamps",
        /* name2                */ "techlamp",
        /* plural2              */ "",
    },

    // Short techno floor lamp (MT_MISC30)
    {
        /* doomednum            */ ShortTechnoFloorLamp,
        /* spawnstate           */ S_TECH2LAMP,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 60 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_TRANSLUCENT_BLUEONLY | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "short techno floor lamp",
        /* plural1              */ "short techno floor lamps",
        /* name2                */ "techlamp2",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Floor lamp (MT_MISC31)
    {
        /* doomednum            */ FloorLamp,
        /* spawnstate           */ S_COLU,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 48 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_TRANSLUCENT_REDONLY | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "floor lamp",
        /* plural1              */ "floor lamps",
        /* name2                */ "column",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Tall green column (MT_MISC32)
    {
        /* doomednum            */ TallGreenColumn,
        /* spawnstate           */ S_TALLGRNCOL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 52 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "tall green column",
        /* plural1              */ "tall green columns",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Short green column (MT_MISC33)
    {
        /* doomednum            */ ShortGreenColumn,
        /* spawnstate           */ S_SHRTGRNCOL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 40 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "short green column",
        /* plural1              */ "short green columns",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Tall red column (MT_MISC34)
    {
        /* doomednum            */ TallRedColumn,
        /* spawnstate           */ S_TALLREDCOL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 52 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "tall red column",
        /* plural1              */ "tall red columns",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Short red column (MT_MISC35)
    {
        /* doomednum            */ ShortRedColumn,
        /* spawnstate           */ S_SHRTREDCOL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 40 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "short red column",
        /* plural1              */ "short red columns",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Short red column with skull (MT_MISC36)
    {
        /* doomednum            */ ShortRedColumnWithSkull,
        /* spawnstate           */ S_SKULLCOL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 40 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "short red column with skull",
        /* plural1              */ "short red columns with skulls",
        /* name2                */ "skullcolumn",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Short green column with beating heart (MT_MISC37)
    {
        /* doomednum            */ ShortGreenColumnWithBeatingHeart,
        /* spawnstate           */ S_HEARTCOL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 40 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "short green column with beating heart",
        /* plural1              */ "short green columns with beating hearts",
        /* name2                */ "heartcolumn",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Evil eye (MT_MISC38)
    {
        /* doomednum            */ EvilEye,
        /* spawnstate           */ S_EVILEYE,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 54 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_TRANSLUCENT_GREENONLY | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 3,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "evil eye",
        /* plural1              */ "evil eyes",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Floating skull (MT_MISC39)
    {
        /* doomednum            */ FloatingSkullRock,
        /* spawnstate           */ S_FLOATSKULL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 26 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_CASTSHADOW | MF2_DECORATION),
        /* raisestate           */ S_NULL,
        /* frames               */ 3,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "floating skull",
        /* plural1              */ "floating skulls",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Torched tree (MT_MISC40)
    {
        /* doomednum            */ TorchedTree,
        /* spawnstate           */ S_TORCHTREE,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "torched tree",
        /* plural1              */ "torched trees",
        /* name2                */ "torchtree",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Tall blue firestick (MT_MISC41)
    {
        /* doomednum            */ TallBlueFirestick,
        /* spawnstate           */ S_BLUETORCH,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 68 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_TRANSLUCENT_BLUEONLY | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "tall blue firestick",
        /* plural1              */ "tall blue firesticks",
        /* name2                */ "bluetorch",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Tall green firestick (MT_MISC42)
    {
        /* doomednum            */ TallGreenFirestick,
        /* spawnstate           */ S_GREENTORCH,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 68 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_TRANSLUCENT_GREENONLY | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "tall green firestick",
        /* plural1              */ "tall green firesticks",
        /* name2                */ "greentorch",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Tall red firestick (MT_MISC43)
    {
        /* doomednum            */ TallRedFirestick,
        /* spawnstate           */ S_REDTORCH,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 68 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_TRANSLUCENT_REDONLY | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "tall red firestick",
        /* plural1              */ "tall red firesticks",
        /* name2                */ "redtorch",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Short blue firestick (MT_MISC44)
    {
        /* doomednum            */ ShortBlueFirestick,
        /* spawnstate           */ S_BTORCHSHRT,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 37 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_TRANSLUCENT_BLUEONLY | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "short blue firestick",
        /* plural1              */ "short blue firesticks",
        /* name2                */ "shortbluetorch",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Short green firestick (MT_MISC45)
    {
        /* doomednum            */ ShortGreenFirestick,
        /* spawnstate           */ S_GTORCHSHRT,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 37 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_TRANSLUCENT_GREENONLY | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "short green firestick",
        /* plural1              */ "short green firesticks",
        /* name2                */ "shortgreentorch",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Short red firestick (MT_MISC46)
    {
        /* doomednum            */ ShortRedFirestick,
        /* spawnstate           */ S_RTORCHSHRT,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 37 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_TRANSLUCENT_REDONLY | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 4,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "short red firestick",
        /* plural1              */ "short red firesticks",
        /* name2                */ "shortredtorch",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Stalagmite (MT_MISC47)
    {
        /* doomednum            */ Stalagmite,
        /* spawnstate           */ S_STALAGMITE,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 40 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "stalagmite",
        /* plural1              */ "stalagmites",
        /* name2                */ "stalagtite",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Tall techno pillar (MT_MISC48)
    {
        /* doomednum            */ TallTechnoPillar,
        /* spawnstate           */ S_TECHPILLAR,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 128 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "tall techno pillar",
        /* plural1              */ "tall techno pillars",
        /* name2                */ "techpillar",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Candlestick (MT_MISC49)
    {
        /* doomednum            */ Candlestick,
        /* spawnstate           */ S_CANDLESTIK,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 14 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ 0,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP | MF2_TRANSLUCENT_REDONLY),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "candlestick",
        /* plural1              */ "candlesticks",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Candelabra (MT_MISC50)
    {
        /* doomednum            */ Candelabra,
        /* spawnstate           */ S_CANDELABRA,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 60 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "candelabra",
        /* plural1              */ "candelabras",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging victim (twitching, blocking) (MT_MISC51)
    {
        /* doomednum            */ HangingVictimTwitchingBlocking,
        /* spawnstate           */ S_BLOODYTWITCH,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 67 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 3,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging victim (twitching, blocking)",
        /* plural1              */ "hanging victims (twitching, blocking)",
        /* name2                */ "bloodytwitch",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging victim (arms out, blocking) (MT_MISC52)
    {
        /* doomednum            */ HangingVictimArmsOutBlocking,
        /* spawnstate           */ S_MEAT2,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 83 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging victim (arms out, blocking)",
        /* plural1              */ "hanging victims (arms out, blocking)",
        /* name2                */ "meat2",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging victim (one-legged, blocking) (MT_MISC53)
    {
        /* doomednum            */ HangingVictimOneLeggedBlocking,
        /* spawnstate           */ S_MEAT3,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 83 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging victim (one-legged, blocking)",
        /* plural1              */ "hanging victims (one-legged, blocking)",
        /* name2                */ "meat3",
        /* plural2              */ "",
        /* name3                */ "hangingcorpse",
        /* plural3              */ ""
    },

    // Hanging pair of legs (blocking) (MT_MISC54)
    {
        /* doomednum            */ HangingPairOfLegsBlocking,
        /* spawnstate           */ S_MEAT4,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 67 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging pair of legs (blocking)",
        /* plural1              */ "hanging pairs of legs (blocking)",
        /* name2                */ "meat4",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging leg (blocking) (MT_MISC55)
    {
        /* doomednum            */ HangingLegBlocking,
        /* spawnstate           */ S_MEAT5,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 51 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging leg (blocking)",
        /* plural1              */ "hanging legs (blocking)",
        /* name2                */ "meat5",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging victim (arms out) (MT_MISC56)
    {
        /* doomednum            */ HangingVictimArmsOut,
        /* spawnstate           */ S_MEAT2,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 83 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging victim (arms out)",
        /* plural1              */ "hanging victims (arms out)",
        /* name2                */ "nonsolidmeat2",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging pair of legs (MT_MISC57)
    {
        /* doomednum            */ HangingPairOfLegs,
        /* spawnstate           */ S_MEAT4,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 67 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging pair of legs",
        /* plural1              */ "hanging pairs of legs",
        /* name2                */ "nonsolidmeat4",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging victim (one-legged) (MT_MISC58)
    {
        /* doomednum            */ HangingVictimOneLegged,
        /* spawnstate           */ S_MEAT3,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 83 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging victim (one-legged)",
        /* plural1              */ "hanging victims (one-legged)",
        /* name2                */ "nonsolidmeat3",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging leg (MT_MISC59)
    {
        /* doomednum            */ HangingLeg,
        /* spawnstate           */ S_MEAT5,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 51 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging leg",
        /* plural1              */ "hanging legs",
        /* name2                */ "nonsolidmeat5",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging victim (twitching) (MT_MISC60)
    {
        /* doomednum            */ HangingVictimTwitching,
        /* spawnstate           */ S_BLOODYTWITCH,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 67 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging victim (twitching)",
        /* plural1              */ "hanging victims (twitching)",
        /* name2                */ "nonsolidtwitch",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Dead Cacodemon (MT_MISC61)
    {
        /* doomednum            */ DeadCacodemon,
        /* spawnstate           */ S_HEAD_DIE6,
        /* spawnhealth          */ 0,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 400,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_CORPSE,
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ 4,
        /* shadowoffset         */ 6 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "dead cacodemon",
        /* plural1              */ "dead cacodemons",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Dead Player (MT_MISC62)
    {
        /* doomednum            */ DeadPlayer,
        /* spawnstate           */ S_PLAY_DIE7,
        /* spawnhealth          */ 0,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_PLAY_XDIE1,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 200,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_CORPSE,
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "dead marine",
        /* plural1              */ "dead marines",
        /* name2                */ "dead player",
        /* plural2              */ "dead players",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Dead Zombieman (MT_MISC63)
    {
        /* doomednum            */ DeadZombieman,
        /* spawnstate           */ S_POSS_DIE5,
        /* spawnhealth          */ 0,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_POSS_XDIE1,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 200,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_CORPSE,
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 2 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "dead zombieman",
        /* plural1              */ "dead zombiemen",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Dead Demon (MT_MISC64)
    {
        /* doomednum            */ DeadDemon,
        /* spawnstate           */ S_SARG_DIE6,
        /* spawnhealth          */ 0,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 400,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_CORPSE,
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 2 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "dead pinky demon",
        /* plural1              */ "dead pinky demons",
        /* name2                */ "dead demon",
        /* plural2              */ "dead demons",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Dead Lost Soul, invisible (MT_MISC65)
    {
        /* doomednum            */ DeadLostSoulInvisible,
        /* spawnstate           */ S_SKULL_DIE6,
        /* spawnhealth          */ 0,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ 0,
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "dead lost soul",
        /* plural1              */ "dead lost souls",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Dead Imp (MT_MISC66)
    {
        /* doomednum            */ DeadImp,
        /* spawnstate           */ S_TROO_DIE5,
        /* spawnhealth          */ 0,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_TROO_XDIE1,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 200,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_CORPSE,
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 2 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "dead imp",
        /* plural1              */ "dead imps",
        /* name2                */ "deaddoomimp",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Dead Shotgun Guy (MT_MISC67)
    {
        /* doomednum            */ DeadShotgunGuy,
        /* spawnstate           */ S_SPOS_DIE5,
        /* spawnhealth          */ 0,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_SPOS_XDIE1,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 200,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_CORPSE,
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 2 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "dead shotgun guy",
        /* plural1              */ "dead shotgun guys",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Bloody mess 1 (MT_MISC68)
    {
        /* doomednum            */ BloodyMess1,
        /* spawnstate           */ S_PLAY_XDIE9,
        /* spawnhealth          */ 0,
        /* gibhealth            */ 0,
        /* giblevel             */ 1,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_GIBS,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 200,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_CORPSE,
        /* flags2               */ (MF2_CASTSHADOW | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "bloody mess 1",
        /* plural1              */ "bloody messes 1",
        /* name2                */ "gibbedmarine",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Bloody mess 2 (MT_MISC69)
    {
        /* doomednum            */ BloodyMess2,
        /* spawnstate           */ S_PLAY_XDIE9,
        /* spawnhealth          */ 0,
        /* gibhealth            */ 0,
        /* giblevel             */ 1,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_GIBS,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 200,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_CORPSE,
        /* flags2               */ (MF2_CASTSHADOW | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "bloody mess 2",
        /* plural1              */ "bloody messes 2",
        /* name2                */ "gibbedmarineextra",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Five skulls shishkebab (MT_MISC70)
    {
        /* doomednum            */ FiveSkullsShishKebab,
        /* spawnstate           */ S_HEADSONSTICK,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 64 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "five skulls shishkebab",
        /* plural1              */ "five skulls shishkebabs",
        /* name2                */ "headsonastick",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Pool of blood and bones (MT_MISC71)
    {
        /* doomednum            */ PoolOfBloodAndBones,
        /* spawnstate           */ S_GIBS,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 2,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_CORPSE,
        /* flags2               */ (MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "pool of blood and bones",
        /* plural1              */ "pools of blood and bones",
        /* name2                */ "gibs",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Skull on a pole (MT_MISC72)
    {
        /* doomednum            */ SkullOnAPole,
        /* spawnstate           */ S_HEADONASTICK,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 56 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "skull on a pole",
        /* plural1              */ "skulls on poles",
        /* name2                */ "headonastick",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Pile of skulls and candles (MT_MISC73)
    {
        /* doomednum            */ PileOfSkullsAndCandles,
        /* spawnstate           */ S_HEADCANDLES,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 42 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ true,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "pile of skulls and candles",
        /* plural1              */ "piles of skulls and candles",
        /* name2                */ "headcandles",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Impaled human (MT_MISC74)
    {
        /* doomednum            */ ImpaledHuman,
        /* spawnstate           */ S_DEADSTICK,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 64 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "impaled human",
        /* plural1              */ "impaled humans",
        /* name2                */ "deadstick",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Twitching impaled human (MT_MISC75)
    {
        /* doomednum            */ TwitchingImpaledHuman,
        /* spawnstate           */ S_LIVESTICK,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 64 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 2,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "twitching impaled human",
        /* plural1              */ "twitching impaled humans",
        /* name2                */ "livestick",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Large brown tree (MT_MISC76)
    {
        /* doomednum            */ LargeBrownTree,
        /* spawnstate           */ S_BIGTREE,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 32 * FRACUNIT,
        /* pickupradius         */ 32 * FRACUNIT,
        /* height               */ 108 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "large brown tree",
        /* plural1              */ "large brown trees",
        /* name2                */ "bigtree",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Burning Barrel (MT_MISC77)
    {
        /* doomednum            */ BurningBarrel,
        /* spawnstate           */ S_BBAR1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 32 * FRACUNIT,
        /* projectilepassheight */ 16 * FRACUNIT,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_SOLID,
        /* flags2               */ (MF2_NOLIQUIDBOB | MF2_TRANSLUCENT_REDONLY | MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 3,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "burning barrel",
        /* plural1              */ "burning barrels",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging victim (guts removed) (MT_MISC78)
    {
        /* doomednum            */ HangingVictimGutsRemoved,
        /* spawnstate           */ S_HANGNOGUTS,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 83 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging victim (guts removed)",
        /* plural1              */ "hanging victims (guts removed)",
        /* name2                */ "hangnoguts",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging victim (guts and brain removed) (MT_MISC79)
    {
        /* doomednum            */ HangingVictimGutsAndBrainRemoved,
        /* spawnstate           */ S_HANGBNOBRAIN,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 83 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging victim (guts and brain removed)",
        /* plural1              */ "hanging victims (guts and brain removed)",
        /* name2                */ "hangbnobrain",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging torso (looking down) (MT_MISC80)
    {
        /* doomednum            */ HangingTorsoLookingDown,
        /* spawnstate           */ S_HANGTLOOKDN,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 59 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging torso (looking down)",
        /* plural1              */ "hanging torsos (looking down)",
        /* name2                */ "hangtlookingdown",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging torso (open skull) (MT_MISC81)
    {
        /* doomednum            */ HangingTorsoOpenSkull,
        /* spawnstate           */ S_HANGTSKULL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 59 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging torso (open skull)",
        /* plural1              */ "hanging torsos (open skull)",
        /* name2                */ "hangtskull",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging torso (looking up) (MT_MISC82)
    {
        /* doomednum            */ HangingTorsoLookingUp,
        /* spawnstate           */ S_HANGTLOOKUP,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 59 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging torso (looking up)",
        /* plural1              */ "hanging torsos (looking up)",
        /* name2                */ "hangtlookingup",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Hanging torso (brain removed) (MT_MISC83)
    {
        /* doomednum            */ HangingTorsoBrainRemoved,
        /* spawnstate           */ S_HANGTNOBRAIN,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16 * FRACUNIT,
        /* pickupradius         */ 16 * FRACUNIT,
        /* height               */ 59 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SOLID | MF_SPAWNCEILING | MF_NOGRAVITY),
        /* flags2               */ MF2_DECORATION,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "hanging torso (brain removed)",
        /* plural1              */ "hanging torsos (brain removed)",
        /* name2                */ "hangtnobrain",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Pool of blood and guts (MT_MISC84)
    {
        /* doomednum            */ PoolOfBloodAndGuts,
        /* spawnstate           */ S_COLONGIBS,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 4 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ 0,
        /* flags2               */ (MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "pool of blood and guts",
        /* plural1              */ "pools of blood and guts",
        /* name2                */ "colongibs",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Pool of blood (MT_MISC85)
    {
        /* doomednum            */ PoolOfBlood,
        /* spawnstate           */ S_SMALLPOOL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 1 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ 0,
        /* flags2               */ (MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "pool of blood",
        /* plural1              */ "pools of blood",
        /* name2                */ "smallbloodpool",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // Pool of brains (MT_MISC86)
    {
        /* doomednum            */ PoolOfBrains,
        /* spawnstate           */ S_BRAINSTEM,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 4 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ 0,
        /* flags2               */ (MF2_DECORATION | MF2_FOOTCLIP),
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "pool of brains",
        /* plural1              */ "pools of brains",
        /* name2                */ "brainstem",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // MT_PUSH
    {
        /* doomednum            */ Pusher,
        /* spawnstate           */ S_TNT1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 8,
        /* pickupradius         */ 8,
        /* height               */ 8,
        /* projectilepassheight */ 8,
        /* mass                 */ 10,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_NOBLOCKMAP,
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "point pusher",
        /* plural1              */ "point pushers",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // MT_PULL
    {
        /* doomednum            */ Puller,
        /* spawnstate           */ S_TNT1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 8,
        /* pickupradius         */ 8,
        /* height               */ 8,
        /* projectilepassheight */ 8,
        /* mass                 */ 10,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_NOBLOCKMAP,
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "point puller",
        /* plural1              */ "point pullers",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // MT_DOGS
    {
        /* doomednum            */ HelperDog,
        /* spawnstate           */ S_DOGS_STND,
        /* spawnhealth          */ 500,
        /* gibhealth            */ -500,
        /* giblevel             */ 0,
        /* seestate             */ S_DOGS_RUN1,
        /* seesound             */ sfx_dgsit,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_dgatk,
        /* painstate            */ S_DOGS_PAIN,
        /* painchance           */ 180,
        /* painsound            */ sfx_dgpain,
        /* meleestate           */ S_DOGS_ATK1,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_DOGS_DIE1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_dgdth,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 10,
        /* radius               */ 12 * FRACUNIT,
        /* pickupradius         */ 12 * FRACUNIT,
        /* height               */ 28 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_dgact,
        /* flags                */ (MF_SOLID | MF_SHOOTABLE | MF_COUNTKILL),
        /* flags2               */ (MF2_CASTSHADOW | MF2_CRUSHABLE | MF2_NOLIQUIDBOB | MF2_FOOTCLIP),
        /* raisestate           */ S_DOGS_RAISE1,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ REDBLOOD,
        /* shadowoffset         */ 4 * FRACUNIT,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "dog",
        /* plural1              */ "dogs",
        /* name2                */ "helper dog",
        /* plural2              */ "helper dogs",
        /* name3                */ "MBF helper dog",
        /* plural3              */ "MBF helper dogs",
    },

    // MT_PLASMA1
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_PLS1BALL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_plasma,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_PLS1EXP,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_firxpl,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 25 * FRACUNIT,
        /* radius               */ 13 * FRACUNIT,
        /* pickupradius         */ 13 * FRACUNIT,
        /* height               */ 8 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 4,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_BOUNCES),
        /* flags2               */ MF2_TRANSLUCENT,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "plasma ball 1",
        /* plural1              */ "plasma balls 1",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // MT_PLASMA2
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_PLS2BALL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_plasma,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_PLS2BALLX1,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_firxpl,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 25 * FRACUNIT,
        /* radius               */ 6 * FRACUNIT,
        /* pickupradius         */ 6 * FRACUNIT,
        /* height               */ 8 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 4,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_MISSILE | MF_DROPOFF | MF_BOUNCES),
        /* flags2               */ MF2_TRANSLUCENT,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ true,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "plasma ball 2",
        /* plural1              */ "plasma balls 2",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // MT_SCEPTRE
    {
        /* doomednum            */ 2016,
        /* spawnstate           */ S_BON3,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 10 * FRACUNIT,
        /* pickupradius         */ 10 * FRACUNIT,
        /* height               */ 16 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPECIAL | MF_COUNTITEM),
        /* flags2               */ MF2_CASTSHADOW,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "evil sceptre",
        /* plural1              */ "evil sceptres",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // MT_BIBLE
    {
        /* doomednum            */ 2017,
        /* spawnstate           */ S_BON4,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 20 * FRACUNIT,
        /* pickupradius         */ 20 * FRACUNIT,
        /* height               */ 10 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_SPECIAL | MF_COUNTITEM),
        /* flags2               */ MF2_CASTSHADOW,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "unholy bible",
        /* plural1              */ "unholy bibles",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // MT_MUSICSOURCE
    {
        /* doomednum            */ MusicSourceMax,
        /* spawnstate           */ S_TNT1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 16,
        /* pickupradius         */ 16,
        /* height               */ 16,
        /* projectilepassheight */ 16,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ MF_NOBLOCKMAP,
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "music changer",
        /* plural1              */ "music changers",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    // MT_GIBDTH
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_TNT1,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 4 * FRACUNIT,
        /* pickupradius         */ 4 * FRACUNIT,
        /* height               */ 4 * FRACUNIT,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_DROPOFF),
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "GIBDTH",
        /* plural1              */ "GIBDTH",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    EMPTYMOBJ,  // MT_BLUEBLOOD (UNUSED)
    EMPTYMOBJ,  // MT_GREENBLOOD (UNUSED)
    EMPTYMOBJ,  // MT_FUZZYBLOOD (UNUSED)

    // Smoke Trail (MT_TRAIL)
    {
        /* doomednum            */ -1,
        /* spawnstate           */ S_TRAIL,
        /* spawnhealth          */ 1000,
        /* gibhealth            */ 0,
        /* giblevel             */ 0,
        /* seestate             */ S_NULL,
        /* seesound             */ sfx_none,
        /* reactiontime         */ 8,
        /* attacksound          */ sfx_none,
        /* painstate            */ S_NULL,
        /* painchance           */ 0,
        /* painsound            */ sfx_none,
        /* meleestate           */ S_NULL,
        /* missilestate         */ S_NULL,
        /* deathstate           */ S_NULL,
        /* xdeathstate          */ S_NULL,
        /* deathsound           */ sfx_none,
        /* droppeditem          */ MT_NULL,
        /* speed                */ 0,
        /* radius               */ 0,
        /* pickupradius         */ 0,
        /* height               */ 0,
        /* projectilepassheight */ 0,
        /* mass                 */ 100,
        /* damage               */ 0,
        /* activesound          */ sfx_none,
        /* flags                */ (MF_NOBLOCKMAP | MF_NOGRAVITY),
        /* flags2               */ 0,
        /* raisestate           */ S_NULL,
        /* frames               */ 0,
        /* fullbright           */ false,
        /* bloodcolor           */ NOBLOOD,
        /* shadowoffset         */ 0,
        /* mbf21flags           */ 0,
        /* infightinggroup      */ IG_DEFAULT,
        /* projectilegroup      */ PG_DEFAULT,
        /* splashgroup          */ SG_DEFAULT,
        /* ripsound             */ sfx_none,
        /* altspeed             */ NO_ALTSPEED,
        /* meleerange           */ MELEERANGE,
        /* name1                */ "rocket trail",
        /* plural1              */ "rocket trails",
        /* name2                */ "",
        /* plural2              */ "",
        /* name3                */ "",
        /* plural3              */ ""
    },

    EMPTYMOBJ,  // MT_EXTRA00
    EMPTYMOBJ,  // MT_EXTRA01
    EMPTYMOBJ,  // MT_EXTRA02
    EMPTYMOBJ,  // MT_EXTRA03
    EMPTYMOBJ,  // MT_EXTRA04
    EMPTYMOBJ,  // MT_EXTRA05
    EMPTYMOBJ,  // MT_EXTRA06
    EMPTYMOBJ,  // MT_EXTRA07
    EMPTYMOBJ,  // MT_EXTRA08
    EMPTYMOBJ,  // MT_EXTRA09
    EMPTYMOBJ,  // MT_EXTRA10
    EMPTYMOBJ,  // MT_EXTRA11
    EMPTYMOBJ,  // MT_EXTRA12
    EMPTYMOBJ,  // MT_EXTRA13
    EMPTYMOBJ,  // MT_EXTRA14
    EMPTYMOBJ,  // MT_EXTRA15
    EMPTYMOBJ,  // MT_EXTRA16
    EMPTYMOBJ,  // MT_EXTRA17
    EMPTYMOBJ,  // MT_EXTRA18
    EMPTYMOBJ,  // MT_EXTRA19
    EMPTYMOBJ,  // MT_EXTRA20
    EMPTYMOBJ,  // MT_EXTRA21
    EMPTYMOBJ,  // MT_EXTRA22
    EMPTYMOBJ,  // MT_EXTRA23
    EMPTYMOBJ,  // MT_EXTRA24
    EMPTYMOBJ,  // MT_EXTRA25
    EMPTYMOBJ,  // MT_EXTRA26
    EMPTYMOBJ,  // MT_EXTRA27
    EMPTYMOBJ,  // MT_EXTRA28
    EMPTYMOBJ,  // MT_EXTRA29
    EMPTYMOBJ,  // MT_EXTRA30
    EMPTYMOBJ,  // MT_EXTRA31
    EMPTYMOBJ,  // MT_EXTRA32
    EMPTYMOBJ,  // MT_EXTRA33
    EMPTYMOBJ,  // MT_EXTRA34
    EMPTYMOBJ,  // MT_EXTRA35
    EMPTYMOBJ,  // MT_EXTRA36
    EMPTYMOBJ,  // MT_EXTRA37
    EMPTYMOBJ,  // MT_EXTRA38
    EMPTYMOBJ,  // MT_EXTRA39
    EMPTYMOBJ,  // MT_EXTRA40
    EMPTYMOBJ,  // MT_EXTRA41
    EMPTYMOBJ,  // MT_EXTRA42
    EMPTYMOBJ,  // MT_EXTRA43
    EMPTYMOBJ,  // MT_EXTRA44
    EMPTYMOBJ,  // MT_EXTRA45
    EMPTYMOBJ,  // MT_EXTRA46
    EMPTYMOBJ,  // MT_EXTRA47
    EMPTYMOBJ,  // MT_EXTRA48
    EMPTYMOBJ,  // MT_EXTRA49
    EMPTYMOBJ,  // MT_EXTRA50
    EMPTYMOBJ,  // MT_EXTRA51
    EMPTYMOBJ,  // MT_EXTRA52
    EMPTYMOBJ,  // MT_EXTRA53
    EMPTYMOBJ,  // MT_EXTRA54
    EMPTYMOBJ,  // MT_EXTRA55
    EMPTYMOBJ,  // MT_EXTRA56
    EMPTYMOBJ,  // MT_EXTRA57
    EMPTYMOBJ,  // MT_EXTRA58
    EMPTYMOBJ,  // MT_EXTRA59
    EMPTYMOBJ,  // MT_EXTRA60
    EMPTYMOBJ,  // MT_EXTRA61
    EMPTYMOBJ,  // MT_EXTRA62
    EMPTYMOBJ,  // MT_EXTRA63
    EMPTYMOBJ,  // MT_EXTRA64
    EMPTYMOBJ,  // MT_EXTRA65
    EMPTYMOBJ,  // MT_EXTRA66
    EMPTYMOBJ,  // MT_EXTRA67
    EMPTYMOBJ,  // MT_EXTRA68
    EMPTYMOBJ,  // MT_EXTRA69
    EMPTYMOBJ,  // MT_EXTRA70
    EMPTYMOBJ,  // MT_EXTRA71
    EMPTYMOBJ,  // MT_EXTRA72
    EMPTYMOBJ,  // MT_EXTRA73
    EMPTYMOBJ,  // MT_EXTRA74
    EMPTYMOBJ,  // MT_EXTRA75
    EMPTYMOBJ,  // MT_EXTRA76
    EMPTYMOBJ,  // MT_EXTRA77
    EMPTYMOBJ,  // MT_EXTRA78
    EMPTYMOBJ,  // MT_EXTRA79
    EMPTYMOBJ,  // MT_EXTRA80
    EMPTYMOBJ,  // MT_EXTRA81
    EMPTYMOBJ,  // MT_EXTRA82
    EMPTYMOBJ,  // MT_EXTRA83
    EMPTYMOBJ,  // MT_EXTRA84
    EMPTYMOBJ,  // MT_EXTRA85
    EMPTYMOBJ,  // MT_EXTRA86
    EMPTYMOBJ,  // MT_EXTRA87
    EMPTYMOBJ,  // MT_EXTRA88
    EMPTYMOBJ,  // MT_EXTRA89
    EMPTYMOBJ,  // MT_EXTRA90
    EMPTYMOBJ,  // MT_EXTRA91
    EMPTYMOBJ,  // MT_EXTRA92
    EMPTYMOBJ,  // MT_EXTRA93
    EMPTYMOBJ,  // MT_EXTRA94
    EMPTYMOBJ,  // MT_EXTRA95
    EMPTYMOBJ,  // MT_EXTRA96
    EMPTYMOBJ,  // MT_EXTRA97
    EMPTYMOBJ,  // MT_EXTRA98
    EMPTYMOBJ   // MT_EXTRA99
};

// DSDHacked
mobjinfo_t  *mobjinfo;
int         nummobjtypes;

void InitMobjInfo(void)
{
  nummobjtypes = NUMMOBJTYPES;
  mobjinfo = original_mobjinfo;
}

void dsdh_EnsureMobjInfoCapacity(const int limit)
{
    static bool first_allocation = true;

    while (limit >= nummobjtypes)
    {
        const int   old_nummobjtypes = nummobjtypes;

        nummobjtypes *= 2;

        if (first_allocation)
        {
            first_allocation = false;
            mobjinfo = I_Malloc(nummobjtypes * sizeof(*mobjinfo));
            memcpy(mobjinfo, original_mobjinfo, old_nummobjtypes * sizeof(*mobjinfo));
        }
        else
            mobjinfo = I_Realloc(mobjinfo, nummobjtypes * sizeof(*mobjinfo));

        memset(mobjinfo + old_nummobjtypes, 0,
            ((size_t)nummobjtypes - old_nummobjtypes) * sizeof(*mobjinfo));

        for (int i = old_nummobjtypes; i < nummobjtypes; i++)
        {
            mobjinfo[i].bloodcolor = REDBLOOD;
            mobjinfo[i].droppeditem = MT_NULL;
            mobjinfo[i].infightinggroup = IG_DEFAULT;
            mobjinfo[i].projectilegroup = PG_DEFAULT;
            mobjinfo[i].splashgroup = SG_DEFAULT;
            mobjinfo[i].altspeed = NO_ALTSPEED;
            mobjinfo[i].meleerange = MELEERANGE;
            M_snprintf(mobjinfo[i].name1, sizeof(mobjinfo[0].name1), "Deh_Actor_%i", i);
            M_snprintf(mobjinfo[i].name2, sizeof(mobjinfo[0].name2), "Deh_Actor_%i", i);
            M_snprintf(mobjinfo[i].name3, sizeof(mobjinfo[0].name3), "Deh_Actor_%i", i);
        }
    }
}
