/*
 * Copyright (c) 2007, Zhang Wei.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the author nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER  CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS  SERVICES; LOSS OF USE, DATA,  PROFITS;  BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY,  TORT (INCLUDING NEGLIGENCE  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include <stdarg.h>
#ifdef WIN32
#include <windows.h>
#include <direct.h>
#else
#include <sys/stat.h>
#endif

#include "sokoban.h"
#include "game.h"


/*
 * Function declarations.
 */
static void set_mode(int, int, int);
static void uninitialize_game(void);


/*
 * Global variables.
 */
 #ifdef WIN32
static const char *config_directory = "zwgames";
#else
static const char *config_directory = ".zwgames";
#endif
static const char *config_file = "/sokowiz";
static const char *icon_file = "./skwd/gfx/icon.bmp";
static SDL_Surface *screen_surface;
static SDL_Surface *icon_surface;
static const int default_height = 600;
static const int default_width = 800;
static const int min_height = 480;
static const int min_width = 640;
static int is_running;
static int is_def;
static int is_verbose;
static int is_fullscreen;
static int screen_height;
static int screen_width;


/*
 * Print verbose information.
 */
void
print_msg(const char *message, ...)
{
	va_list ap;

	if (is_verbose) {
		va_start(ap, message);
		vprintf(message, ap);
		va_end (ap);
	}
}

/*
 * Initialize game system.
 */
int
initialize_game(int def, int fullscreen, int verbose, int height, int width)
{
	int i;
	int map_num;
	FILE *file;
	char *c;
	char *home;
	char theme[256];
	char levelset[256];
	char keys[256];
	char buf[2048];

	is_running = 0;
	is_def = def;
	is_verbose = verbose;
	theme[0] = '\0';
	levelset[0] = '\0';
	keys[0] = '\0';

	print_msg(GAME_NAME" version " GAME_VERSION"\n");

	/* Initialize SDL. */
	print_msg("Initializing system... ");
	if (SDL_Init(SDL_INIT_VIDEO) == 0) {
		screen_height = default_height;
		screen_width = default_width;
		map_num = 1;
		print_msg("[DONE]\n");
	} else {
		print_msg("[FAIL]\n");
		return 1;
	}

	/* Load game settings. */
	if (!def) {
		print_msg("Load settings... ");
		home = getenv("HOME");
		if (home != NULL) {
			strcpy(buf, home);
			strcat(buf, "/");
		} else
			buf[0] = '\0';
		strcat(buf, config_directory);
		strcat(buf, config_file);
		file = fopen(buf, "r");
		if (file == NULL)
			print_msg("[FAIL]\n");
		else {
			i = 0;
			while (!feof(file)) {
				buf[0] = '\0';
				fgets(buf, sizeof(buf), file);
				if (buf[0] <= ' ')
					continue;
				while ((strlen(buf) > 0) && (buf[strlen(buf) - 1] <= ' '))
					buf[strlen(buf) - 1] = '\0';
				c = strchr(buf, '=') + 1;
				if (c == NULL)
					continue;
				while ((c[0] != '\0') && (c[0] <= ' '))
					++c;
				if (c[0] == '\0') {
					++i;
					continue;
				}
				switch (i++) {
					case 0:
						screen_height = atoi(c);
						if (screen_height < min_height)
							screen_height = default_height;
						break;
					case 1:
						screen_width = atoi(c);
						if (screen_width < min_width)
							screen_width = default_width;
						break;
					case 2:
						is_fullscreen = atoi(c);
						break;
					case 3:
						strcpy(theme, c);
						break;
					case 4:
						strcpy(levelset, c);
						break;
					case 5:
						map_num = atoi(c);
						break;
					case 6:
						strcpy(keys, c);
				}
			}
			fclose(file);
			print_msg("[DONE]\n");
		}
	}

	/* Set the icon for the display window. */
	print_msg("Setting icon... ");
	icon_surface = SDL_LoadBMP(icon_file);
	if (icon_surface != NULL) {
		SDL_SetColorKey(icon_surface, SDL_SRCCOLORKEY, SDL_MapRGB(icon_surface->format, 255, 0, 255));
		SDL_WM_SetIcon(icon_surface, NULL);
		print_msg("[DONE]\n");
	} else
		print_msg("[FAIL]\n");

	if (fullscreen)
		is_fullscreen = 1;
	if (min_height <= height)
		screen_height = height;
	if (min_width <= width)
		screen_width = width;
	set_mode(screen_height, screen_width, is_fullscreen);
	if (screen_surface == NULL)
		return 1;

	/* set other parameters */
	SDL_WM_SetCaption(GAME_NAME, NULL);
	SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY - 250 , SDL_DEFAULT_REPEAT_INTERVAL);

	/* Initialize sokoban game */
	if (initialize_sokoban(screen_height, screen_width, map_num, levelset, theme, keys) == 1)
		return 1;

	is_running = 1;
	return 0;
}

/*
 * Set up the specified video mode.
 */
static void
set_mode(int height, int width, int fullscreen)
{
	unsigned int flags;
	int bpp;
	SDL_Surface *screen;

	if ((height < min_height) || (width < min_width)) {
		height = min_height;
		width = min_width;
	}

	print_msg("Setting %dx%d %s mode... ", width, height, !fullscreen ? "window" : "fullscreen");
	flags = SDL_HWSURFACE | SDL_DOUBLEBUF | SDL_RESIZABLE;
	if (fullscreen)
		flags |= SDL_FULLSCREEN;
	bpp = SDL_VideoModeOK(width, height, 32, flags);
	if (bpp != 0) {
		screen = SDL_SetVideoMode(width, height, bpp, flags);
		if (screen != NULL) {
			SDL_FreeSurface(screen_surface);
			screen_surface = screen;
			if (is_running) {
				SDL_Delay(100);
				resize_sokoban(height, width);
			}
			screen_height = height;
			screen_width = width;
			if (fullscreen)
				SDL_ShowCursor(SDL_DISABLE);
			else
				SDL_ShowCursor(SDL_ENABLE);
			print_msg("[DONE]\n");
			return;
		}
	}
	print_msg("[FAIL]\n");
}

/*
 * Handle the events of game.
 */
void
update_game(void)
{
	int redraw;

	redraw = 1;
	while (is_running) {
		SDL_Event event;
		while (SDL_PollEvent(&event) ) {
			redraw = 1;
			switch (event.type) {
			case SDL_KEYDOWN:
				switch (event.key.keysym.sym) {
				case SDLK_F11:
					is_fullscreen = !is_fullscreen;
					set_mode(screen_height, screen_width, is_fullscreen);
					break;
				case SDLK_F12:
					if ((default_height != screen_height) || (default_width != screen_width))
						set_mode(default_height, default_width, is_fullscreen);
					break;
				case SDLK_ESCAPE:
					if (event.key.keysym.mod & KMOD_SHIFT) {
						uninitialize_game();
						return;
					}
				default:
					if (update_sokoban(event.key.keysym) == 1) {
						uninitialize_game();
						return;
					}
				}
				break;
			case SDL_VIDEORESIZE:
				if (((event.resize.h != 20) || (event.resize.h != 32)) && (event.resize.w != 97)) {
					if ((event.resize.h != screen_height) || (event.resize.w != screen_width))
						set_mode(event.resize.h, event.resize.w, 0);
				}
				break;
			case SDL_QUIT:
				uninitialize_game();
				return;
			}
		}
		if (!redraw)
			SDL_Delay(30);
		else {
			draw_sokoban();
			SDL_Flip(screen_surface);
			redraw = 0;
		}
	}
}

/*
 * Uninitialize game.
 */
static void
uninitialize_game(void)
{
	FILE *file;
	char *home;
	char buf[2048];

	/* Save of game settings. */
	if (!is_def) {
		print_msg("Saving settings... ");
		home = getenv("HOME");
		if (home != NULL) {
			strcpy(buf, home);
			strcat(buf, "/");
		} else
			buf[0] = '\0';
		strcat(buf, config_directory);
		strcat(buf, config_file);
		file = fopen(buf, "w");
		if (file == NULL) {
			buf[strrchr(buf, '/') - buf] = '\0';
#ifdef WIN32
			_mkdir(buf);
#else
			mkdir(buf, S_IRWXU);
#endif
			strcat(buf, config_file);
			file = fopen(buf, "w");
		}
		if (file == NULL)
			print_msg("[FAIL]\n");
		else {
			fprintf(file, " %s version %s Configuration file\n", GAME_NAME, GAME_VERSION);
			fprintf(file, "\n [Screen]\n");
			fprintf(file, "Height=%d\n", screen_height);
			fprintf(file, "Width=%d\n", screen_width);
			fprintf(file, "Fullscreen=%d\n", is_fullscreen);
			fprintf(file, "\n [Theme]\n");
			get_sokoban_theme(buf);
			fprintf(file, "Name=%s\n", buf);
			fprintf(file, "\n [Levelset]\n");
			get_sokoban_levelset(buf);
			fprintf(file, "Name=%s\n", buf);
			fprintf(file, "Map=%d\n", get_sokoban_map());
			fprintf(file, "\n [Control]\n");
			get_sokoban_keys(buf);
			fprintf(file, "Keys=%s\n",  buf);
			fclose(file);
			print_msg("[DONE]\n");
		}
	}

	if (icon_surface != NULL) {
		print_msg("Unloading icon... ");
		SDL_FreeSurface(icon_surface);
		print_msg("[DONE]\n");
	}

	uninitialize_sokoban();
	atexit(SDL_Quit);
	is_running = 0;

	print_msg("Thanks for playing!\n");
}


/*
 * Get file size.
 */
int
get_file_size(const char *filename)
{
#ifdef WIN32
	WIN32_FILE_ATTRIBUTE_DATA fad;

	if (GetFileAttributesEx(filename, GetFileExInfoStandard, (void *)&fad)) {
		if ((fad.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 1)
			return fad.nFileSizeLow;
	}
#else
	struct stat sb;

	if ((stat(filename, &sb) == 0) && S_ISREG(sb.st_mode))
		return sb.st_size;
#endif
	return 0;
}

/*
 * Check if the specified file is a directory.
 */
int
is_directory(const char *filename)
{
#ifdef WIN32
	WIN32_FILE_ATTRIBUTE_DATA fad;

	if (GetFileAttributesEx(filename, GetFileExInfoStandard, (void *)&fad))
		return (fad.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY);
#else
	struct stat sb;

	if ((stat(filename, &sb) == 0) && S_ISDIR(sb.st_mode))
		return 1;
#endif
	return 0;
}

/*
 * Load bmp images.
 */
SDL_Surface *
load_bmp(const char *filename)
{
	SDL_Surface *image;
	SDL_Surface *new_image;

	image = SDL_LoadBMP(filename);
	if (image == NULL)
		return NULL;
	SDL_SetColorKey(image, SDL_RLEACCEL|SDL_SRCCOLORKEY, SDL_MapRGB(image->format, 0, 0, 0));
	new_image = SDL_DisplayFormat(image);
	SDL_FreeSurface(image);
	return new_image;
}


/*
 * Blit from the source surface to the destination surface.
 */
void
blit_surface(SDL_Surface *surface, int sprite, int y, int x, int height, int width)
{
	SDL_Rect src;
	SDL_Rect dest;

	if (surface == NULL)
		return;

	src.y = (Sint16)((sprite / (surface->w / width)) * height);
	src.x = (Sint16)((sprite % (surface->w / width)) * width);
	if ((src.y >= surface->h) || (src.x >= surface->w) || (y >= screen_surface->h) || (x >= screen_surface->w))
		return;
	src.h = (Uint16)height;
	src.w = (Uint16)width;
	dest.y = (Sint16)y;
	dest.x = (Sint16)x;
	dest.h = (Uint16)height;
	dest.w = (Uint16)width;
	SDL_BlitSurface(surface, &src, screen_surface, &dest);
}
