/*
 * Copyright (c) 2007, Zhang Wei.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the author nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER  CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS  SERVICES; LOSS OF USE, DATA,  PROFITS;  BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY,  TORT (INCLUDING NEGLIGENCE  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include <string.h>
#ifdef WIN32
#include <windows.h>
#include <io.h>
#else
#include <dirent.h>
#endif

#include "sokoban.h"

#ifndef DATA_PATH
#define DATA_PATH "./skwd"
#endif

#define PATH_MAX_LEN 1024
#define NAME_MAX_LEN 256

#define MAP_MAX_LINES 128
#define MAP_MAX_COLUMNS 128


/*
 * Map structure
 */
typedef struct map_s map;
struct map_s {
	int offset;
	int columns;
	int lines;
	int x;
	int y;
	int blocks;
	int bots;
};

/*
 * Undo structure
 */
typedef struct undo_s undo;
struct undo_s {
	int block;
	int keeper_direction;
	int keeper_y;
	int keeper_x;
	int block_old_y;
	int block_old_x;
	int block_new_y;
	int block_new_x;
	int bots;
};


/*
 * Functions declarations
 */
static int initialize_data(const char *, const char *);
static int load_theme(void);
static int load_levelset(void);
static void set_map(void);
static void draw_text(int, int, const char *);
static void move_keeper(int, int);
static void move_screen(int, int);
static void undo_push(int, int, int);
static void undo_pop(void);
static void redo_pop(void);
static void parse_keys(const char *);
static void reset_keys(void);

/*
 * Global constants
 */
static const char *gfx_path = DATA_PATH"/gfx/";
static const char *themes_path = DATA_PATH"/themes/";
static const char *levelsets_path = DATA_PATH"/levels/";
static const char *gfx_ext_name = ".bmp";
static const char *levels_ext_name = ".xsb";

static const int sprite_size = 32;
static const int font_height = 12;
static const int font_width = 8;
static const int levelset_max_size = 512 * 1024;
static const int map_max_count = 2000;
static const int undo_step = 1024;
static const int key_count = 16;

enum {
	/* Border */
	BORDER = 0,
	/* Warehouse keeper (Sokoban) */
	KEEPER = 3,
	/* Blank field */
	BLANK = 8,
	/* Block */
	BLOCK = 15,
	/* Target area */
	TARGET = 16,
	/* Block on target area */
	BOT = 23,
	/* Wall */
	WALL = 24
};

/*
 * Global variables
 */
static int screen_height;
static int screen_width;

static SDL_Surface *font_surface;
static SDL_Surface *sprites_surface;

static int theme_count;
static int theme_name_len;
static int theme_num;
static int theme_tmp_num;
static char *theme_names;

static int levelset_count;
static int levelset_name_len;
static int levelset_num;
static int levelset_tmp_num;
static char *levelset_names;
static map *levelset_map;
static char *levelset_data;

static int map_count;
static int map_num;
static int map_lines;
static int map_columns;
static int map_blocks;
static int map_bots;
static int keeper_y;
static int keeper_x;
static char map_data[MAP_MAX_LINES][MAP_MAX_COLUMNS];

static int keeper_direction;
static int move_count;
static int push_count;

static int screen_lines;
static int screen_columns;
static int screen_y_count;
static int screen_x_count;
static int screen_y_num;
static int screen_x_num;

static int undo_count;
static int undo_num;
static int redo_count;
static undo *undo_data;

static int key_num;
static int key_wait;
static SDLKey key_left;
static SDLKey key_up;
static SDLKey key_right;
static SDLKey key_down;
static SDLKey key_screen_left;
static SDLKey key_screen_up;
static SDLKey key_screen_right;
static SDLKey key_screen_down;
static SDLKey key_screen_origin;
static SDLKey key_undo;
static SDLKey key_redo;
static SDLKey key_previous_map;
static SDLKey key_next_map;
static SDLKey key_levelsets;
static SDLKey key_themes;
static SDLKey key_restart;

/*
 * Initialize sokoban game.
 */
int
initialize_sokoban(int height, int width, int map, const char *levelset, const char *theme, const char *keys)
{

	screen_height = height;
	screen_width = width;

	theme_count = 0;
	theme_num = -1;
	theme_tmp_num = -1;
	theme_names = NULL;

	levelset_count = 0;
	levelset_num = -1;
	levelset_tmp_num = -1;

	map_count = 0;
	map_num = map - 1;
	map_columns = -1;
	map_lines = -1;
	keeper_x = -1;
	keeper_y = -1;

	keeper_direction = 0;
	move_count = 0;
	push_count = 0;

	undo_count = undo_step;
	undo_num = -1;
	redo_count = 0;
	undo_data = NULL;

	if (initialize_data(theme, levelset) == 1)
		return 1;

	if (load_theme() == 1)
		return 1;

	if (load_levelset() == 1)
		return 1;

	parse_keys(keys);
	return 0;
}

/*
 * Handle keydown event.
 */
int
update_sokoban(SDL_keysym key)
{

	if (map_count <= 0)
		return 0;

	if ((levelset_tmp_num == -1) && (theme_tmp_num == -1) && (key_num == -1)) {
		if (key.sym == key_left)
			move_keeper(0, -1);
		else if (key.sym == key_up)
			move_keeper(-1, 0);
		else if (key.sym == key_right)
			move_keeper(0, 1);
		else if (key.sym == key_down)
			move_keeper(1, 0);
		else if (key.sym == key_screen_left)
			move_screen(0, -1);
		else if (key.sym == key_screen_up)
			move_screen(-1, 0);
		else if (key.sym == key_screen_right)
			move_screen(0, 1);
		else if (key.sym == key_screen_down)
			move_screen(1, 0);
		else if (key.sym == key_screen_origin)
			move_screen(0, 0);
		else if (key.sym == key_undo)
			undo_pop();
		else if (key.sym == key_redo)
			redo_pop();
		else if (key.sym == key_previous_map) {
			if (map_count > 1) {
				if (map_num > 0)
					--map_num;
				else
					map_num = map_count - 1;
				set_map();
			}
		} else if (key.sym == key_next_map) {
			if (map_count > 1) {
				map_num = (map_num + 1) % map_count;
				set_map();
			}
		} else if (key.sym == key_levelsets) {
			if (levelset_count > 1)
				levelset_tmp_num = levelset_num;
		} else if (key.sym == key_themes) {
			if (theme_count > 1)
				theme_tmp_num = theme_num;
		} else if (key.sym == key_restart) {
			if (undo_num >= 0)
				set_map();
		} else if (key.sym == SDLK_F9)
			key_num = 0;
	} else if (levelset_tmp_num != -1) {
		if (key.sym == SDLK_LEFT)
			levelset_tmp_num = 0;
		else if (key.sym == SDLK_UP) {
			if (levelset_tmp_num > 0)
				--levelset_tmp_num;
			else
				levelset_tmp_num = levelset_count - 1;
		} else if (key.sym == SDLK_RIGHT)
			levelset_tmp_num = levelset_count - 1;
		else if (key.sym == SDLK_DOWN) {
			if (levelset_tmp_num < levelset_count - 1)
				++levelset_tmp_num;
			else
				levelset_tmp_num = 0;
		} else if (key.sym == SDLK_RETURN) {
			if (levelset_num != levelset_tmp_num) {
				levelset_num = levelset_tmp_num;
				if (load_levelset() == 1) {
					for (levelset_num = 0; levelset_num < levelset_count; ++levelset_num) {
						if (load_levelset() == 0)
							break;
					}
					if (levelset_num == levelset_count)
						return 1;
				}
			}
			map_num = 0;
			levelset_tmp_num = -1;
		} else if ((key.sym == key_levelsets) || (key.sym == SDLK_ESCAPE))
			levelset_tmp_num = -1;
	} else if (theme_tmp_num != -1) {
		if (key.sym == SDLK_LEFT)
			theme_tmp_num = 0;
		else if (key.sym == SDLK_UP) {
			if (theme_tmp_num > 0)
				--theme_tmp_num;
			else
				theme_tmp_num = theme_count - 1;
		} else if (key.sym == SDLK_RIGHT)
			theme_tmp_num = theme_count - 1;
		else if (key.sym == SDLK_DOWN) {
			if (theme_tmp_num < theme_count - 1)
				++theme_tmp_num;
			else
				theme_tmp_num = 0;
		} else if (key.sym == SDLK_RETURN) {
			if (theme_num != theme_tmp_num) {
				theme_num = theme_tmp_num;
				if (load_theme() == 1) {
					for (theme_num = 0; theme_num < theme_count; ++theme_num) {
						if (load_theme() == 0)
							break;
					}
					if (theme_num == theme_count)
						return 1;
				}
			}
			theme_tmp_num = -1;
		} else if ((key.sym == key_themes) || (key.sym == SDLK_ESCAPE))
			theme_tmp_num = -1;
	} else if (key_num != -1) {
		if (key_wait) {
			if ((key.sym == SDLK_F9) || (key.sym == SDLK_RETURN) || (key.sym == SDLK_ESCAPE)) {
				key_num = -1;
				key_wait = 0;
			} else if (key.sym == SDLK_SPACE)
				key_wait = 0;
			else if ((key.sym != key_left) && (key.sym != key_up) && (key.sym != key_right) && (key.sym != key_down) &&\
			(key.sym != key_screen_left) && (key.sym != key_screen_up) && (key.sym != key_screen_right) && (key.sym != key_screen_down) &&\
			(key.sym != key_screen_origin) && (key.sym != key_undo) && (key.sym != key_redo) && (key.sym != key_previous_map) &&\
			(key.sym != key_next_map) && (key.sym != key_levelsets) && (key.sym != key_themes) && (key.sym != key_restart)) {
				if (((key.sym >= SDLK_1) && (key.sym <= SDLK_9)) || ((key.sym >= SDLK_a) && (key.sym <= SDLK_z)) ||\
				(key.sym == SDLK_BACKSPACE) || (key.sym == SDLK_TAB) || (key.sym == SDLK_QUOTE) || (key.sym == SDLK_COMMA) ||\
				(key.sym == SDLK_MINUS) || (key.sym == SDLK_PERIOD) || (key.sym == SDLK_SLASH) || (key.sym == SDLK_SEMICOLON) ||\
				(key.sym == SDLK_EQUALS) || (key.sym == SDLK_LEFTBRACKET) || (key.sym == SDLK_RIGHTBRACKET) || (key.sym == SDLK_BACKSLASH) ||\
				(key.sym == SDLK_BACKQUOTE) || (key.sym == SDLK_DELETE) || (key.sym == SDLK_UP) || (key.sym == SDLK_DOWN) ||\
				(key.sym == SDLK_RIGHT) || (key.sym == SDLK_LEFT) || (key.sym == SDLK_INSERT) || (key.sym == SDLK_HOME) ||\
				(key.sym == SDLK_END) || (key.sym == SDLK_PAGEUP) || (key.sym == SDLK_PAGEDOWN) || (key.sym == SDLK_RSHIFT) ||\
				(key.sym == SDLK_LSHIFT) || (key.sym == SDLK_RCTRL) || (key.sym == SDLK_LCTRL) || (key.sym == SDLK_RALT) || (key.sym == SDLK_LALT)) {
					if (key_num == 0)
						key_left = key.sym;
					else if (key_num == 1)
						key_up = key.sym;
					else if (key_num == 2)
						key_right = key.sym;
					else if (key_num == 3)
						key_down = key.sym;
					else if (key_num == 4)
						key_screen_left = key.sym;
					else if (key_num == 5)
						key_screen_up = key.sym;
					else if (key_num == 6)
						key_screen_right = key.sym;
					else if (key_num == 7)
						key_screen_down = key.sym;
					else if (key_num == 8)
						key_screen_origin = key.sym;
					else if (key_num == 9)
						key_undo = key.sym;
					else if (key_num == 10)
						key_redo = key.sym;
					else if (key_num == 11)
						key_previous_map = key.sym;
					else if (key_num == 12)
						key_next_map = key.sym;
					else if (key_num == 13)
						key_levelsets = key.sym;
					else if (key_num == 14)
						key_themes = key.sym;
					else if (key_num == 15)
						key_restart = key.sym;
					key_wait = 0;
				}
			}
		} else {
			if (key.sym == SDLK_LEFT)
				key_num = 0;
			else if (key.sym == SDLK_UP) {
				if (key_num > 0)
					--key_num;
				else
					key_num = key_count - 1;
			} else if (key.sym == SDLK_RIGHT)
				key_num = key_count - 1;
			else if (key.sym == SDLK_DOWN) {
				if (key_num < key_count - 1)
					++key_num;
				else
					key_num = 0;
			} else if ((key.sym == SDLK_F9) || (key.sym == SDLK_RETURN) || (key.sym == SDLK_ESCAPE))
				key_num = -1;
			else if (key.sym == SDLK_F10)
				reset_keys();
			else if (key.sym == SDLK_SPACE)
				key_wait = 1;
		}
	}
	return 0;
}

/*
 * Resize screen.
 */
void
resize_sokoban(int height, int width)
{

	if ((height < sprite_size * 6) || (width < sprite_size * 6))
		return;

	/* Set screen size. */
	screen_height = height;
	screen_width = width;

	/* Get the maximum lines and columns in one screen. */
	screen_lines = height / sprite_size - 2;
	screen_columns = width / sprite_size -2;
	if (map_lines < screen_lines)
		screen_lines = map_lines;
	if (map_columns < screen_columns)
		screen_columns = map_columns;

	/* Get the number of screens of this level. */
	screen_y_count = map_lines / screen_lines;
	screen_x_count = map_columns / screen_columns;
	if (map_lines % screen_lines != 0)
		++screen_y_count;
	if (map_columns % screen_columns != 0)
		++screen_x_count;

	/* Get the coordinates of the top left corner of the current screen. */
	screen_y_num = keeper_y / screen_lines * screen_lines;
	screen_x_num = keeper_x / screen_columns * screen_columns;
}

/*
 * Draw screen.
 */
void
draw_sokoban(void)
{
	int y;
	int x;
	int i;
	int j;
	int lines;
	int columns;
	int bits;
	char c;
	char buf[256];

	/* Draw four angles. */
	blit_surface(sprites_surface, BORDER, 0, 0, sprite_size, sprite_size);
	blit_surface(sprites_surface, BORDER, 0, screen_width - sprite_size, sprite_size, sprite_size);
	blit_surface(sprites_surface, BORDER, screen_height - sprite_size, screen_width - sprite_size, sprite_size, sprite_size);
	blit_surface(sprites_surface, BORDER, screen_height - sprite_size, 0, sprite_size, sprite_size);
	/* Draw top and bottom borders. */
	for (i = sprite_size / 2; i < screen_width - sprite_size * 3 / 2; i += sprite_size) {
		blit_surface(sprites_surface, BORDER + 1, 0, i, sprite_size, sprite_size);
		blit_surface(sprites_surface, BORDER + 1, screen_height - sprite_size, i, sprite_size, sprite_size);
	}
	blit_surface(sprites_surface, BORDER + 1, 0, screen_width - sprite_size * 3 / 2, sprite_size, sprite_size);
	blit_surface(sprites_surface, BORDER + 1, screen_height - sprite_size, screen_width - sprite_size * 3 / 2, sprite_size, sprite_size);
	/* Draw left and right borders. */
	for (i = sprite_size / 2; i < screen_height - sprite_size * 3 / 2; i += sprite_size) {
		blit_surface(sprites_surface, BORDER + 2, i, 0, sprite_size, sprite_size);
		blit_surface(sprites_surface, BORDER + 2, i, screen_width - sprite_size, sprite_size, sprite_size);
	}
	blit_surface(sprites_surface, BORDER + 2, screen_height - sprite_size * 3 / 2, 0, sprite_size, sprite_size);
	blit_surface(sprites_surface, BORDER + 2, screen_height - sprite_size * 3 / 2, screen_width - sprite_size, sprite_size, sprite_size);
	/* Draw background area. */
	for (i = sprite_size / 2; i < screen_height - sprite_size * 3 / 2; i += sprite_size) {
		for (j = sprite_size / 2; j < screen_width - sprite_size * 3 / 2; j += sprite_size) {
			blit_surface(sprites_surface, BLANK, i, j, sprite_size, sprite_size);
		}
	}
	for (i = sprite_size / 2; i < screen_height - sprite_size * 3 / 2; i += sprite_size) {
		blit_surface(sprites_surface, BLANK, i, screen_width - sprite_size * 3 / 2, sprite_size, sprite_size);
	}
	for (i = sprite_size / 2; i < screen_width - sprite_size * 3 / 2; i += sprite_size) {
		blit_surface(sprites_surface, BLANK, screen_height - sprite_size * 3 / 2, i, sprite_size, sprite_size);
	}
	blit_surface(sprites_surface, BLANK, screen_height - sprite_size * 3 / 2, screen_width - sprite_size * 3 / 2, sprite_size, sprite_size);

	/* Get the start coordinates of map. */
	y = (screen_height - map_lines * sprite_size) / 2;
	if (y <= 0)
		y = sprite_size;
	x = (screen_width - map_columns * sprite_size) / 2;
	if (x <= 0)
		x = sprite_size;

	/* Draw map */
	if (screen_y_num + screen_lines >= map_lines)
		lines = map_lines;
	else
		lines = screen_y_num + screen_lines;
	if (screen_x_num + screen_columns >= map_columns)
		columns = map_columns;
	else
		columns = screen_x_num + screen_columns;
	for (i = screen_y_num; i < lines; ++i) {
		for (j = screen_x_num; j < columns; ++j) {
			bits = 0;
			if (i > 0) {
				c = map_data[i - 1][j];
				if (((c >= WALL) &&(c <= WALL + 15)) || (c == BLOCK) || (c == BOT))
					bits += 1;
			}
			if (j > 0) {
				c = map_data[i][j - 1];
				if (((c >= WALL) &&(c <= WALL + 15)) || (c == BLOCK) || (c == BOT))
					bits += 4;
			}
			if ((bits != 5) && (i > 0) && (j > 0)) {
				c = map_data[i - 1][j - 1];
				if (((c >= WALL) &&(c <= WALL + 15)) || (c == BLOCK) || (c == BOT))
					bits += 2;
			}
			switch (map_data[i][j]) {
			case BLANK:
				blit_surface(sprites_surface, BLANK + bits, y, x, sprite_size, sprite_size);
				break;
			case TARGET:
				blit_surface(sprites_surface, TARGET + bits, y, x, sprite_size, sprite_size);
				break;
			default:
				blit_surface(sprites_surface, map_data[i][j], y, x, sprite_size, sprite_size);
			}
			x += sprite_size;
		}
		x -= (j - screen_x_num) * sprite_size;
		y += sprite_size;
	}
	y -= (i - screen_y_num) * sprite_size;

	/* Draw keeper. */
	bits = 0;
	if ((keeper_y < screen_y_num) || (keeper_y >= screen_y_num + screen_lines))
		++bits;
	if ((keeper_x < screen_x_num) || (keeper_x >= screen_x_num + screen_columns))
		++bits;
	if (bits == 0) {
		if ((map_bots != map_blocks) && (levelset_tmp_num == -1))
			i = KEEPER + keeper_direction;
		else
			i = KEEPER;
		blit_surface(sprites_surface, i, y + keeper_y % screen_lines * sprite_size, x +  keeper_x % screen_columns * sprite_size, sprite_size, sprite_size);
	}

	/* Draw status bar. */
	if (key_num != -1) {
		sprintf(buf, "Keys: %d/%d", key_num + 1, key_count);
		draw_text(screen_height - sprite_size + 2, screen_width - (int)(strlen(buf) * font_width) - sprite_size / 2 - 2, buf);

		switch (key_num) {
		case 0:
			strcpy(buf, "[ Move the sokoban left:");
			i = key_left;
			break;
		case 1:
			strcpy(buf, "[ Move the sokoban up:");
			i = key_up;
			break;
		case 2:
			strcpy(buf, "[ Move the sokoban right:");
			i = key_right;
			break;
		case 3:
			strcpy(buf, "[ Move the sokoban down:");
			i = key_down;
			break;
		case 4:
			strcpy(buf, "[ Scroll the screen left:");
			i = key_screen_left;
			break;
		case 5:
			strcpy(buf, "[ Scroll the screen up:");
			i = key_screen_up;
			break;
		case 6:
			strcpy(buf, "[ Scroll the screen right:");
			i = key_screen_right;
			break;
		case 7:
			strcpy(buf, "[ Scroll the screen down:");
			i = key_screen_down;
			break;
		case 8:
			strcpy(buf, "[ Reset the screen position:");
			i = key_screen_origin;
			break;
		case 9:
			strcpy(buf, "[ Undo changes:");
			i = key_undo;
			break;
		case 10:
			strcpy(buf, "[ Redo changes:");
			i = key_redo;
			break;
		case 11:
			strcpy(buf, "[ Go to the previous level:");
			i = key_previous_map;
			break;
		case 12:
			strcpy(buf, "[ Go to the next level:");
			i = key_next_map;
			break;
		case 13:
			strcpy(buf, "[ Choose levelsets:");
			i = key_levelsets;
			break;
		case 14:
			strcpy(buf, "[ Choose themes:");
			i = key_themes;
			break;
		case 15:
			strcpy(buf, "[ Restart the current level:");
			i = key_restart;
			break;
		}
		if (key_wait)
			i = SDLK_QUESTION;
		switch (i) {
		case SDLK_QUESTION:
			strcat(buf, " ? ]");
			break;
		case SDLK_BACKSPACE:
			strcat(buf, " Backspace ]");
			break;
		case SDLK_TAB:
			strcat(buf, " Tab ]");
			break;
		case SDLK_QUOTE:
			strcat(buf, " \" ]");
			break;
		case SDLK_COMMA:
			strcat(buf, " , ]");
			break;
		case SDLK_MINUS:
			strcat(buf, " - ]");
			break;
		case SDLK_PERIOD:
			strcat(buf, " . ]");
			break;
		case SDLK_SLASH:
			strcat(buf, " / ]");
			break;
		case SDLK_0:
			strcat(buf, " 0 ]");
			break;
		case SDLK_1:
			strcat(buf, " 1 ]");
			break;
		case SDLK_2:
			strcat(buf, " 2 ]");
			break;
		case SDLK_3:
			strcat(buf, " 3 ]");
			break;
		case SDLK_4:
			strcat(buf, " 4 ]");
			break;
		case SDLK_5:
			strcat(buf, " 5 ]");
			break;
		case SDLK_6:
			strcat(buf, " 6 ]");
			break;
		case SDLK_7:
			strcat(buf, " 7 ]");
			break;
		case SDLK_8:
			strcat(buf, " 8 ]");
			break;
		case SDLK_9:
			strcat(buf, " 9 ]");
			break;
		case SDLK_SEMICOLON:
			strcat(buf, " ; ]");
			break;
		case SDLK_EQUALS:
			strcat(buf, " = ]");
			break;
		case SDLK_LEFTBRACKET:
			strcat(buf, " [ ]");
			break;
		case SDLK_RIGHTBRACKET:
			strcat(buf, " ] ]");
			break;
		case SDLK_BACKSLASH:
			strcat(buf, " \\ ]");
			break;
		case SDLK_BACKQUOTE:
			strcat(buf, " ` ]");
			break;
		case SDLK_a:
			strcat(buf, " a ]");
			break;
		case SDLK_b:
			strcat(buf, " b ]");
			break;
		case SDLK_c:
			strcat(buf, " c ]");
			break;
		case SDLK_d:
			strcat(buf, " d ]");
			break;
		case SDLK_e:
			strcat(buf, " e ]");
			break;
		case SDLK_f:
			strcat(buf, " f ]");
			break;
		case SDLK_g:
			strcat(buf, " g ]");
			break;
		case SDLK_h:
			strcat(buf, " h ]");
			break;
		case SDLK_i:
			strcat(buf, " i ]");
			break;
		case SDLK_j:
			strcat(buf, " j ]");
			break;
		case SDLK_k:
			strcat(buf, " k ]");
			break;
		case SDLK_l:
			strcat(buf, " l ]");
			break;
		case SDLK_m:
			strcat(buf, " m ]");
			break;
		case SDLK_n:
			strcat(buf, " n ]");
			break;
		case SDLK_o:
			strcat(buf, " o ]");
			break;
		case SDLK_p:
			strcat(buf, " p ]");
			break;
		case SDLK_q:
			strcat(buf, " q ]");
			break;
		case SDLK_r:
			strcat(buf, " r ]");
			break;
		case SDLK_s:
			strcat(buf, " s ]");
			break;
		case SDLK_t:
			strcat(buf, " t ]");
			break;
		case SDLK_u:
			strcat(buf, " u ]");
			break;
		case SDLK_v:
			strcat(buf, " v ]");
			break;
		case SDLK_w:
			strcat(buf, " w ]");
			break;
		case SDLK_x:
			strcat(buf, " x ]");
			break;
		case SDLK_y:
			strcat(buf, " y ]");
			break;
		case SDLK_z:
			strcat(buf, " z ]");
			break;
		case SDLK_DELETE:
			strcat(buf, " Delete ]");
			break;
		case SDLK_UP:
			strcat(buf, " Up ]");
			break;
		case SDLK_DOWN:
			strcat(buf, " Down ]");
			break;
		case SDLK_RIGHT:
			strcat(buf, " Right ]");
			break;
		case SDLK_LEFT:
			strcat(buf, " Left ]");
			break;
		case SDLK_INSERT:
			strcat(buf, " Insert ]");
			break;
		case SDLK_HOME:
			strcat(buf, " Home ]");
			break;
		case SDLK_END:
			strcat(buf, " End ]");
			break;
		case SDLK_PAGEUP:
			strcat(buf, " Page Up ]");
			break;
		case SDLK_PAGEDOWN:
			strcat(buf, " Page Down ]");
			break;
		case SDLK_RSHIFT:
			strcat(buf, " Right Shift ]");
			break;
		case SDLK_LSHIFT:
			strcat(buf, " Left Shift ]");
			break;
		case SDLK_RCTRL:
			strcat(buf, " Right Ctrl ]");
			break;
		case SDLK_LCTRL:
			strcat(buf, " Left Ctrl ]");
			break;
		case SDLK_RALT:
			strcat(buf, " Right Alt ]");
			break;
		case SDLK_LALT:
			strcat(buf, " Left Alt ]");
			break;
		default:
			strcat(buf, " ? ]");
			break;
		}
		draw_text(screen_height - sprite_size+ 2, sprite_size / 2, buf);
		return;
	}

	if ((levelset_tmp_num == -1) && (theme_tmp_num == -1)) {
		if (screen_width / font_width > 90) {
			sprintf(buf, "Moves: %d  Pushes: %d  Blocks: %d/%d", move_count, push_count, map_bots, map_blocks);
			if ((screen_lines < map_lines) || (screen_columns < map_columns)) {
				y = screen_y_num / screen_lines + 1;
				x = screen_x_num / screen_columns + 1;
				sprintf(buf + strlen(buf), "  Keeper: %dx%d/%dx%d  Screen: %dx%d/%dx%d", keeper_x + 1, keeper_y + 1, map_columns, map_lines, x, y, screen_x_count, screen_y_count);
			}
			sprintf(buf + strlen(buf), "  Level: %d/%d", map_num + 1, map_count);
		} else {
			sprintf(buf, "M: %d  P: %d  B: %d/%d", move_count, push_count, map_bots, map_blocks);
			if ((screen_lines < map_lines) || (screen_columns < map_columns)) {
				y = screen_y_num / screen_lines + 1;
				x = screen_x_num / screen_columns + 1;
				sprintf(buf + strlen(buf), "  K: %dx%d/%dx%d  S: %dx%d/%dx%d", keeper_x + 1, keeper_y + 1, map_columns, map_lines, x, y, screen_x_count, screen_y_count);
			}
			sprintf(buf + strlen(buf), "  L: %d/%d", map_num + 1, map_count);
		}
	} else if (levelset_tmp_num != -1)
		sprintf(buf, "Levelset: %d/%d", levelset_tmp_num + 1, levelset_count);
	else if (theme_tmp_num != -1)
		sprintf(buf, "Theme: %d/%d", theme_tmp_num + 1, theme_count);
	draw_text(screen_height - sprite_size + 2, screen_width - (int)(strlen(buf) * font_width) - sprite_size / 2 - 2, buf);
	buf[0] = '\0';
	if (levelset_tmp_num != -1) {
		i = levelset_tmp_num * levelset_name_len;
		if (strlen(levelset_names + i) < 50)
			sprintf(buf, "[ %s ]", levelset_names + i);
		else
			sprintf(buf, "[ %.50s... ]", levelset_names + i);
		draw_text(screen_height - sprite_size+ 2, sprite_size / 2, buf);
	} else if (theme_tmp_num != -1) {
		i = theme_tmp_num * theme_name_len;
		if (strlen(theme_names + i) < 50)
			sprintf(buf, "[ %s ]", theme_names + i);
		else
			sprintf(buf, "[ %.50s... ]", theme_names + i);
		draw_text(screen_height - sprite_size+ 2, sprite_size / 2, buf);
	}

}

/*
 * Get the number of current levelset.
 */
int
get_sokoban_map(void)
{
	return map_num + 1;
}

/*
 * Get the name of current levelset.
 */
void
get_sokoban_levelset(char *buf)
{
	strcpy(buf, levelset_names + levelset_num * levelset_name_len);
}

/*
 * Get the name of current theme.
 */
void
get_sokoban_theme(char *buf)
{
	strcpy(buf, theme_names + theme_num * theme_name_len);
}

/*
 * Get control keys.
 */
void
get_sokoban_keys(char *buf)
{
	sprintf(buf, "%03d%03d%03d%03d%03d%03d%03d%03d%03d%03d%03d%03d%03d%03d%03d%03d",\
	key_left, key_up, key_right, key_down, key_screen_left, key_screen_up, key_screen_right, key_screen_down,\
	key_screen_origin, key_undo, key_redo, key_previous_map, key_next_map, key_levelsets, key_themes, key_restart);
}

/*
 * Uninitialize sokoban game.
 */
void
uninitialize_sokoban(void)
{

	if (theme_names != NULL) {
		print_msg("Deallocating memory for data... ");
		free(theme_names);
		theme_names = NULL;
		print_msg("[DONE]\n");
	}
	if (undo_data != NULL) {
		print_msg("Deallocating memory for undo... ");
		free(undo_data);
		undo_data = NULL;
		print_msg("[DONE]\n");
	}
	print_msg("Unloading font... ");
	SDL_FreeSurface(font_surface);
	print_msg("[DONE]\n");
	print_msg("Unloading theme... ");
	SDL_FreeSurface(sprites_surface);
	print_msg("[DONE]\n");
}

/*
 * Initialize data of sokoban.
 * This is the smallest available level in a levelset.
 *  ###
 * #.$@#
 *  ###
 * And we don't handle the levelset file greater than levelset_max_size.
 * It's big enough.
 */
static int
initialize_data(const char *theme, const char *levelset)
{
#ifdef WIN32
	struct _finddata_t fd;
	long file;
#else
	DIR *dp;
	struct dirent *ep;
#endif
	int size;
	int max_size;
	int len;
	const int themes_len = strlen(themes_path);
	const int levelsets_len = strlen(levelsets_path);
	char path[PATH_MAX_LEN];
	char name[NAME_MAX_LEN];

	/* Get data of themes. */
	print_msg("Searching themes... ");
	if (theme[0] == '\0')
		strcpy(name, "default");
	else {
		strcpy(name, theme);
	}
	strcpy(path, themes_path);
#ifdef WIN32
	strcat(path, "*");
	if ((file = _findfirst(path, &fd)) == -1L) {
		print_msg("[FAIL]\n");
		return 1;
	}
	while (file != -1L) {
		if ((strcmp(fd.name, "..") != 0) && (strcmp(fd.name, ".") != 0)) {
			/* Search the position of specified theme. */
			if ((theme_num == -1) && (strcmp(fd.name, name) == 0))
				theme_num = theme_count;
			path[themes_len] = '\0';
			strcat(path, fd.name);
			if (is_directory(path)) {
				/* Get the longest length of filenames. */
				len = strlen(fd.name);
				if (theme_name_len < len)
					theme_name_len = len;
				/* Count the number of themes. */
				++theme_count;
			}
		}
		if (_findnext(file, &fd) == -1L)
			break;
	}
	_findclose(file);
#else
	if ((dp = opendir(themes_path)) == NULL) {
		print_msg("[FAIL]\n");
		return 1;
	}
	while ((ep = readdir(dp)) != NULL) {
		if ((strcmp(ep->d_name, "..") != 0) && (strcmp(ep->d_name, ".") != 0)) {
			if ((theme_num == -1) && (strcmp(ep->d_name, name) == 0))
				theme_num = theme_count;
			path[themes_len] = '\0';
			strcat(path, ep->d_name);
			if (is_directory(path)) {
				len = strlen(ep->d_name);
				if (theme_name_len < len)
					theme_name_len = len;
				++theme_count;
			}
		}
	}
	closedir(dp);
#endif
	if (theme_count == 0) {
		print_msg("[FAIL]\n");
		return 1;
	}
	if (theme_num == -1)
		++theme_num;
	++theme_name_len;
	print_msg("[DONE]\n");

	/* Get data of levelsets. */
	print_msg("Searching levelsets... ");
	if (levelset[0] == '\0')
		strcpy(name, "default");
	else {
		strcpy(name, levelset);
	}
	strcat(name, levels_ext_name);
	strcpy(path, levelsets_path);
	max_size = 0;
#ifdef WIN32
	path[levelsets_len] = '*';
	strcpy(path + levelsets_len + 1, levels_ext_name);
	if ((file = _findfirst(path, &fd)) == -1L) {
		print_msg("[FAIL]\n");
		return 1;
	}
	while (file != -1L) {
		/* Search the position of specified levelset. */
		if ((levelset_num == -1) && (strcmp(fd.name, name) == 0))
			levelset_num = levelset_count;
		path[levelsets_len] = '\0';
		strcat(path, fd.name);
		size = get_file_size(path);
		if ((size >= 15) && (size <= levelset_max_size)) {
			/* Get the maximum size of files. */
			if (max_size < size)
				max_size = size;
			/* Get the longest length of filenames. */
			len = strlen(fd.name);
			if (levelset_name_len < len)
				levelset_name_len = len;
			/* Count the number of themes. */
			++levelset_count;
		}
		if (_findnext(file, &fd) == -1L)
			break;
	}
	_findclose(file);
#else
	if ((dp = opendir(levelsets_path)) == NULL) {
		print_msg("[FAIL]\n");
		return 1;
	}
	while ((ep = readdir(dp)) != NULL) {
		if (strstr(ep->d_name, levels_ext_name) == NULL)
			continue;
		if ((levelset_num == -1) && (strcmp(ep->d_name, name) == 0))
			levelset_num = levelset_count;
		path[levelsets_len] = '\0';
		strcat(path, ep->d_name);
		size = get_file_size(path);
		if ((size >= 15) && (size <= levelset_max_size)) {
			/* Get the maximum size of files. */
			if (max_size < size)
				max_size = size;
			/* Get the longest length of filenames. */
			len = strlen(ep->d_name);
			if (levelset_name_len < len)
				levelset_name_len = len;
			/* Count the number of themes. */
			++levelset_count;
		}
	}
	closedir(dp);
#endif
	if (levelset_count == 0) {
		print_msg("[FAIL]\n");
		return 1;
	}
	if (levelset_num == -1)
		++levelset_num;
	levelset_name_len -= strlen(levels_ext_name) - 1;
	print_msg("[DONE]\n");

	/* Allocate memory for filenames of themes and filenames of levelsets and levels of a levelset. */
	print_msg("Allocating memory for data... ");
	size = theme_count * theme_name_len;
	size += levelset_count * levelset_name_len;
	size += map_max_count * sizeof(map);
	size += max_size + 1;
	theme_names = (char *)malloc(size);
	if (theme_names == NULL) {
		print_msg("[FAIL]\n");
		return 1;
	}
	levelset_names = theme_names + theme_count * theme_name_len;
	levelset_map = (map *)(levelset_names + levelset_count * levelset_name_len);
	levelset_data = levelset_names + levelset_count * levelset_name_len + map_max_count * sizeof(map);
	print_msg("[DONE]\n");

	/* Reserve names of themes. */
	print_msg("Reserving themes... ");
	strcpy(path, themes_path);
#ifdef WIN32
	strcat(path, "*.*");
	if ((file = _findfirst(path, &fd)) == -1L) {
		free(theme_names);
		print_msg("[FAIL]\n");
		return 1;
	}
	while (file != -1L) {
		if ((strcmp(fd.name, "..") != 0) && (strcmp(fd.name, ".") != 0)) {
			path[themes_len] = '\0';
			strcat(path, fd.name);
			if (is_directory(path)) {
				strcpy(theme_names, fd.name);
				theme_names += theme_name_len;
			}
		}
		if (_findnext(file, &fd) == -1L)
			break;
	}
	_findclose(file);
#else
	if ((dp = opendir(themes_path)) == NULL) {
		free(theme_names);
		print_msg("[FAIL]\n");
		return 1;
	}
	while ((ep = readdir(dp)) != NULL) {
		if ((strcmp(ep->d_name, "..") != 0) && (strcmp(ep->d_name, ".") != 0)) {
			path[themes_len] = '\0';
			strcat(path, ep->d_name);
			if (is_directory(path)) {
				strcpy(theme_names, ep->d_name);
				theme_names += theme_name_len;
			}
		}
	}
	closedir(dp);
#endif
	theme_names -= theme_count * theme_name_len;
	print_msg("[DONE]\n");

	/* Reserve names of levelsets. */
	print_msg("Reserving levelsets... ");
	strcpy(path, levelsets_path);
#ifdef WIN32
	path[levelsets_len] = '*';
	strcpy(path + levelsets_len + 1, levels_ext_name);
	if ((file = _findfirst(path, &fd)) == -1L) {
		free(theme_names);
		print_msg("[FAIL]\n");
		return 1;
	}
	while (file != -1L) {
		path[levelsets_len] = '\0';
		strcat(path, fd.name);
		size = get_file_size(path);
		if ((size >= 15) && (size <= levelset_max_size)) {
			fd.name[strlen(fd.name) - strlen(levels_ext_name)] = '\0';
			strcpy(levelset_names, fd.name);
			levelset_names += levelset_name_len;
		}
		if (_findnext(file, &fd) == -1L)
			break;
	}
	_findclose(file);
#else
	if ((dp = opendir(levelsets_path)) == NULL) {
		free(theme_names);
		print_msg("[FAIL]\n");
		return 1;
	}
	while ((ep = readdir(dp)) != NULL) {
		if (strstr(ep->d_name, levels_ext_name) == NULL)
			continue;
		path[levelsets_len] = '\0';
		strcat(path, ep->d_name);
		size = get_file_size(path);
		if ((size >= 15) && (size <= levelset_max_size)) {
			ep->d_name[strlen(ep->d_name) - strlen(levels_ext_name)] = '\0';
			strcpy(levelset_names, ep->d_name);
			levelset_names += levelset_name_len;
		}
	}
	closedir(dp);
#endif
	levelset_names -= levelset_count  * levelset_name_len;
	print_msg("[DONE]\n");

	/* Allocate memory for undo. */
	print_msg("Allocating memory for undo... ");
	undo_data = (undo *)malloc(undo_count * sizeof(undo));
	if (undo_data == NULL) {
		print_msg("[FAIL]\n");
		return 1;
	}
	print_msg("[DONE]\n");
	return 0;
}

/*
 * Load the selected theme.
 */
static int
load_theme(void)
{
	int len;
	SDL_Surface *sprites;
	SDL_Surface *font;
	char filename[PATH_MAX_LEN];

	print_msg("Loading %s theme... ", theme_names + theme_num * theme_name_len);
	strcpy(filename, themes_path);
	strcat(filename, theme_names + theme_num * theme_name_len);
	len = strlen(filename);
	strcat(filename, "/sprites");
	strcat(filename, gfx_ext_name);
	sprites = load_bmp(filename);
	if (sprites == NULL) {
		print_msg("[FAIL]\n");
		return 1;
	}
	strcpy(filename + len, "font");
	strcat(filename, gfx_ext_name);
	font = load_bmp(filename);
	if (font == NULL) {
		strcpy(filename, gfx_path);
		strcat(filename, "font");
		strcat(filename, gfx_ext_name);
		font = load_bmp(filename);
		if (font == NULL) {
			SDL_FreeSurface(sprites);
			print_msg("[FAIL]\n");
			return 1;
		}
	}
	SDL_FreeSurface(sprites_surface);
	sprites_surface = sprites;
	SDL_FreeSurface(font_surface);
	font_surface = font;
	print_msg("[DONE]\n");
	return 0;
}

/*
 * Load the selected levelset.
 */
static int
load_levelset(void)
{
	int i;
	int j;
	int len;
	int len_prev;
	int len_next;
	int bits;
	int y;
	int x;
	int lines;
	int columns;
	int blocks;
	int bots;
	int targets;
	int keepers;
	int offset;
	int count;
	FILE *file;
	map *header;
	char *data;
	char buf[MAP_MAX_LINES][MAP_MAX_COLUMNS + 1];
	char filename[PATH_MAX_LEN];

	print_msg("Loading %s levelset... ", levelset_names + levelset_num * levelset_name_len);
	strcpy(filename, levelsets_path);
	strcat(filename, levelset_names + levelset_num * levelset_name_len);
	strcat(filename, levels_ext_name);
	i = get_file_size(filename);
	if ((i < 15) || (i > levelset_max_size)) {
		print_msg("[FAIL]\n");
		return 1;
	}

	file = fopen(filename, "r");
	if (file == NULL) {
		print_msg("[FAIL]\n");
		return 1;
	}
	y = 0;
	x = 0;
	lines = 0;
	columns = 0;
	blocks = 0;
	bots = 0;
	targets = 0;
	keepers = 0;
	offset = 0;
	count = 0;
	header = levelset_map;
	data = levelset_data;
	while (!feof(file)) {
		buf[lines][0] = '\0';
		fgets(buf[lines], MAP_MAX_COLUMNS + 1, file);
		/* Remove all the non-# characters from end of a line. */
		while ((strlen(buf[lines]) > 0) && (buf[lines][strlen(buf[lines]) - 1] != '#'))
			buf[lines][strlen(buf[lines]) - 1] = '\0';
		len = strlen(buf[lines]);
		if ((len > 1) && ((buf[lines][0] == ' ') || (buf[lines][0] == '#'))) {
			for (i = 0; i < len; ++i) {
				switch (buf[lines][i]) {
				case ' ':
					buf[lines][i] = BLANK;
					break;
				case '#':
					buf[lines][i] = WALL;
					break;
				case '$':
					buf[lines][i] = BLOCK;
					++blocks;
					break;
				case '.':
					buf[lines][i] = TARGET;
					++targets;
					break;
				case '+':
					buf[lines][i] = TARGET;
					++targets;
					++keepers;
					x = i;
					y = lines;
					break;
				case '*':
					buf[lines][i] = BOT;
					++blocks;
					++targets;
					++bots;
					break;
				case '@':
					buf[lines][i] = BLANK;
					++keepers;
					x = i;
					y = lines;
					break;
				default:
					buf[lines][i] = BLANK;
				}
			}
			if (columns < len)
				columns = len;
			++lines;
		} else {
			if ((lines > 2) && (blocks != 0) && (blocks == targets) && (keepers == 1)) {
				/* Set correct wall directions. */
				for (i = 0; i < lines; ++i) {
					len = strlen(buf[i]);
					if (i >= 1)
						len_prev = strlen(buf[i - 1]);
					else
						len_prev = 0;
					if (i < lines - 1)
						len_next = strlen(buf[i + 1]);
					else
						len_next = 0;
					for (j = 0; j < len; ++j) {
						if ((buf[i][j] >= WALL) && (buf[i][j] <= WALL + 15)) {
							bits = 0;
							if ((j < len_prev) && (buf[i - 1][j] >= WALL) && (buf[i - 1][j] <= WALL + 15))
								bits += 1;
							if ((j < len - 1) && (buf[i][j + 1] >= WALL) && (buf[i][j + 1] <= WALL + 15))
								bits += 2;
							if ((j < len_next) && (buf[i + 1][j] >= WALL) && (buf[i + 1][j] <= WALL + 15))
								bits += 4;
							if ((j >= 1) && (buf[i][j - 1] >= WALL) && (buf[i][j - 1] <= WALL + 15))
								bits += 8;
							buf[i][j] += bits;
						}
					}
				}
				/* Move level data from buffer to global area. */
				header->offset = offset;
				header->columns = columns;
				header->lines =  lines;
				header->x = x;
				header->y = y;
				header->blocks = blocks;
				header->bots = bots;
				++header;
				for (i = 0; i < lines; ++i) {
					len = strlen(buf[i]);
					memmove(data, buf[i], len + 1);
					offset += len + 1;
					data += len + 1;
				}
				*(data++) = '\0';
				++offset;
				++count;
			}
			lines = 0;
			columns = 0;
			blocks = 0;
			bots = 0;
			targets = 0;
			keepers = 0;
		}
	}
	fclose(file);

	map_count = count;
	if (map_count <= map_num)
		map_num = 0;

	if (count == 0) {
		print_msg("[FAIL]\n");
		return 1;
	}
	print_msg("[DONE]\n");

	set_map();
	return 0;
}

/*
 * Set data for the selected map.
 */
static void
set_map(void)
{
	int i;
	int len;
	map *header;
	char *data;

	header = levelset_map + map_num;

	/* Fill data to current level. */
	data = levelset_data + header->offset;
	for (i = 0; i < header->lines; ++i) {
		len = strlen(data);
		memmove(map_data[i], data, len);
		memset(map_data[i] + len, BLANK, MAP_MAX_COLUMNS - len);
		data += len + 1;
	}

	/* Set data of this level. */
	map_lines = header->lines;
	map_columns = header->columns;
	map_blocks = header->blocks;
	map_bots = header->bots;
	keeper_y = header->y;
	keeper_x = header->x;
	keeper_direction = 0;
	move_count = 0;
	push_count = 0;
	undo_num = -1;
	redo_count = 0;

	/* Set data of screen. */
	resize_sokoban(screen_height, screen_width);
}

/*
 * Draw text.
 */
static void
draw_text(int y, int x, const char *text)
{
	int i;
	int len;

	len = strlen(text);
	for (i = 0; i < len; i++) {
		blit_surface(font_surface, (int)text[i] - 32, y, x + i * font_width, font_height, font_width);
	}
}

/*
 * Move keeper.
 */
static void
move_keeper(int y, int x)
{

	if ((map_count == 0) || (map_bots == map_blocks))
		return;
	if ((keeper_y + y < 0) || (keeper_y + y >= map_lines))
		return;
	if ((keeper_x + x< 0) || (keeper_x+ x >= map_columns))
		return;

	if (y < 0)
		keeper_direction = 1;
	else if (x>0)
		keeper_direction = 2;
	else if (y>0)
		keeper_direction = 3;
	else if (x<0)
		keeper_direction = 4;
	else
		keeper_direction = 0;

	switch (map_data[keeper_y + y][keeper_x + x]) {
	case BLANK:
	case TARGET:
		undo_push(0, 0 ,0);
		keeper_y += y;
		keeper_x += x;
		screen_y_num = keeper_y / screen_lines * screen_lines;
		screen_x_num = keeper_x / screen_columns * screen_columns;
		++move_count;
		break;
	case BOT:
	case BLOCK:
		switch (map_data[keeper_y + y+ y][keeper_x + x+ x]) {
		case BLANK:
		case TARGET:
			undo_push(1, y, x);
			if (map_data[keeper_y + y][keeper_x + x] == BOT) {
				map_data[keeper_y + y][keeper_x + x] = TARGET;
				--map_bots;
			} else
				map_data[keeper_y + y][keeper_x + x] = BLANK;
			if (map_data[keeper_y + y+ y][keeper_x + x+ x] == TARGET) {
				map_data[keeper_y + y + y][keeper_x + x + x] = BOT;
				++map_bots;
			} else
				map_data[keeper_y + y + y][keeper_x + x + x] = BLOCK;
			keeper_y += y;
			keeper_x += x;
			screen_y_num = keeper_y / screen_lines * screen_lines;
			screen_x_num = keeper_x / screen_columns * screen_columns;
			++move_count;
			++push_count;
			break;
		}
	default:
		break;
	}
}

/*
 * Move screen.
 */
static void
move_screen(int y, int x)
{

	if (map_count == 0)
		return;
	if ((map_lines <= screen_lines) && (map_columns <= screen_columns))
		return;
	if ((y | x ) == 0) {
		screen_y_num = keeper_y / screen_lines * screen_lines;
		screen_x_num = keeper_x / screen_columns * screen_columns;
		return;
	}
	if ((screen_y_num + y * screen_lines < 0) || (screen_y_num + y * screen_lines >= map_lines))
		return;
	if ((screen_x_num + x * screen_columns < 0) || (screen_x_num + x * screen_columns >= map_columns))
		return;

	screen_y_num += y * screen_lines;
	screen_x_num += x * screen_columns;
}

/*
 * Reserve info of keeper and block.
 */
static void
undo_push(int block, int y, int x)
{
	undo* change;

	++undo_num;
	if (undo_count <= undo_num) {
		undo_count += undo_step;
		undo_data = realloc(undo_data, undo_count * sizeof(undo));
	}
	change = undo_data + undo_num;
	change->block = block;
	change->keeper_direction = keeper_direction;
	change->keeper_y = keeper_y;
	change->keeper_x = keeper_x;
	if (block) {
		change->block_old_y = keeper_y + y;
		change->block_old_x= keeper_x + x;
		change->block_new_y = keeper_y + y + y;
		change->block_new_x = keeper_x + x + x;
		change->bots = map_bots;
	}
	redo_count = 0;
}

/*
 * Undo move.
 */
static void
undo_pop(void)
{
	undo* change;

	if (undo_num < 0)
		return;

	change = undo_data + undo_num;

	if (redo_count == 0) {
		undo_push(0, 0, 0);
		(change + 1)->bots = map_bots;
		--undo_num;
	}

	if (change->block) {
		if (map_data[change->block_new_y][change->block_new_x] == BLOCK)
			map_data[change->block_new_y][change->block_new_x] = BLANK;
		else
			map_data[change->block_new_y][change->block_new_x] = TARGET;
		if (map_data[change->block_old_y][change->block_old_x] == BLANK)
			map_data[change->block_old_y][change->block_old_x] = BLOCK;
		else
			map_data[change->block_old_y][change->block_old_x] = BOT;
		map_bots = change->bots;
		--push_count;
	}
	keeper_direction = change->keeper_direction;
	keeper_x = change->keeper_x;
	keeper_y = change->keeper_y;
	screen_y_num = keeper_y / screen_lines * screen_lines;
	screen_x_num = keeper_x / screen_columns * screen_columns;
	--move_count;
	--undo_num;
	++redo_count;
}

/*
 * Redo move.
 */
static void
redo_pop(void)
{
	undo* change;

	if (redo_count <= 0)
		return;

	++undo_num;
	change = undo_data + undo_num;

	if (change->block) {
		if (map_data[change->block_new_y][change->block_new_x] == TARGET)
			map_data[change->block_new_y][change->block_new_x] = BOT;
		else
			map_data[change->block_new_y][change->block_new_x] = BLOCK;
		if (map_data[change->block_old_y][change->block_old_x] == BOT)
			map_data[change->block_old_y][change->block_old_x] = TARGET;
		else
			map_data[change->block_old_y][change->block_old_x] = BLANK;
		map_bots = change->bots;
		++push_count;
	}
	keeper_direction = change->keeper_direction;
	keeper_x = (change + 1)->keeper_x;
	keeper_y = (change + 1)->keeper_y;
	screen_y_num = keeper_y / screen_lines * screen_lines;
	screen_x_num = keeper_x / screen_columns * screen_columns;;
	++move_count;
	--redo_count;
}

/*
 * Parse keys buffer.
 */
static void
parse_keys(const char *keys) {
	int i;
	int j;
	int len;
	int count;
	char buf[4];

	key_num = -1;
	key_wait = 0;

	if (keys == NULL) {
		reset_keys();
		return;
	}

	len = strlen(keys);
	if (len != 16 * 3) {
		reset_keys();
		return;
	}

	buf[3] = '\0';
	i = 0;
	j = 0;
	count = 0;
	while (i <= len) {
		if (j < 3) {
			if ((keys[i] < '0') || (keys[i] > '9'))
				break;
			buf[j++] = keys[i++];
		} else if (j == 3){
			++count;
			if (count == 1)
				key_left = atoi(buf);
			else if (count == 2)
				key_up = atoi(buf);
			else if (count == 3)
				key_right = atoi(buf);
			else if (count == 4)
				key_down = atoi(buf);
			else if (count == 5)
				key_screen_left = atoi(buf);
			else if (count == 6)
				key_screen_up = atoi(buf);
			else if (count == 7)
				key_screen_right = atoi(buf);
			else if (count == 8)
				key_screen_down = atoi(buf);
			else if (count == 9)
				key_screen_origin = atoi(buf);
			else if (count == 10)
				key_undo = atoi(buf);
			else if (count == 11)
				key_redo = atoi(buf);
			else if (count == 12)
				key_previous_map = atoi(buf);
			else if (count == 13)
				key_next_map = atoi(buf);
			else if (count == 14)
				key_levelsets = atoi(buf);
			else if (count == 15)
				key_themes = atoi(buf);
			else if (count == 16)
				key_restart = atoi(buf);
			j = 0;
		}
	}
	if (count != key_count)
		reset_keys();
}

/*
 * Reset all control keys.
 */
static void
reset_keys(void) {
	key_left = SDLK_LEFT;
	key_up = SDLK_UP;
	key_right = SDLK_RIGHT;
	key_down = SDLK_DOWN;
	key_screen_left = SDLK_a;
	key_screen_up = SDLK_w;
	key_screen_right = SDLK_d;
	key_screen_down = SDLK_s;
	key_screen_origin = SDLK_e;
	key_undo = SDLK_DELETE;
	key_redo = SDLK_INSERT;
	key_previous_map = SDLK_PAGEUP;
	key_next_map = SDLK_PAGEDOWN;
	key_levelsets = SDLK_END;
	key_themes = SDLK_HOME;
	key_restart = SDLK_PERIOD;
}
