/*
  Tux Paint Config UI file
  Originally generated by Fast Light User Interface Designer (fluid) v. 1.0104

  Torsten Giebl <wizard@turricane.org>
  Jan Wynholds
  Bill Kendrick <bill@newbreedsoftware.com>
  Martin Fuhrer <mfuhrer@gmail.com>
  and others (see AUTHORS.txt & CHANGES.txt)

  https://tuxpaint.org/

  Formatted with:
  `indent -nbfda -npcs -npsl -bli0 --no-tabs` -bjk 2021.11.12

  November 12, 2002 - October 20, 2024
*/

#include <libintl.h>
#include <math.h>
#include "tuxpaint-config2.h"
#include <FL/fl_ask.H>
#include <FL/fl_message.H>
#include <FL/Fl.H>
#include <FL/Fl_File_Chooser.H>
#include <FL/Fl_Scroll.H>
#include <FL/Fl_Slider.H>
#include <FL/Fl_Pack.H>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <errno.h>
#include <linebreak.h>
#ifdef USE_PAPERSIZES
#include <paper.h>
#endif
#include <pango/pango.h>
#include <pango/pangoft2.h>
#include "version.h"
#include "about.h"
#include "debug.h"  /* must come after FL includes to avoid printf expansion error */

/* Win32 specific stuff to help set the window icon */
#ifdef WIN32
#include <windows.h>
#include "../visualc/resource.h"        /* contains IDI_ICON1 */
#include "win32utils.h"         /* Registry utilities */
extern HINSTANCE fl_display;    /* hInstance on win32 platform */
#endif

/* dont think save_conf will work for win32...
   need to redo with #IFDEF precompiler definitions and better writing */

#ifdef __APPLE__
#include "macos.h"
#endif

/* strcasestr() is not available at least on Windows */
#ifdef WIN32
char *strcasestr(const char *haystack, const char *needle);

char *strcasestr(const char *haystack, const char *needle)
{
  char *uphaystack, *upneedle, *result;
  unsigned int i;

  uphaystack = strdup(haystack);
  upneedle = strdup(needle);

  if (uphaystack == NULL || upneedle == NULL)
    return (NULL);

  for (i = 0; i < strlen(uphaystack); i++)
    uphaystack[i] = toupper(uphaystack[i]);

  for (i = 0; i < strlen(upneedle); i++)
    upneedle[i] = toupper(upneedle[i]);

  result = strstr(uphaystack, upneedle);

  if (result != NULL)
    return (result - uphaystack + (char *)haystack);
  else
    return NULL;
}
#endif

int tab_font_size;              /* font size for tab label text */
int big_font_size;              /* font size of large text */
int smaller_font_size;          /* font size of smaller text */
int small_font_size;            /* font size of small text */
int tab_label_height;           /* height of a tab label */
int tab_height;                 /* height of a tab */
int tab_width;                  /* width of a tab */
int tab_top_y;                  /* position of tabs */
int box_top_y;                  /* top position of top-aligned box */
int full_box_height;            /* height of a full-sized box */
int full_box_width;             /* height of a full-sized box */
int half_box_height;            /* height of a half-sized box */
int half_box_width;             /* height of a half-sized box */
int half_box_l_x;               /* leftmost position of half box */
int half_box_r_x;               /* rightmost position of half box */
int current_box_height;
int padding;                    /* padding between widgets within a box */
int hspace;                     /* left space before the contents */
int border;                     /* border width for tabs */

char conf_filename[256];
int conf_filetype = 0;

char *tpc_locale;

#define NUM_LANGS 131
#define LANG_NAME 0
#define LANG_CODE 1

/* Note: Comments here helps tuxpaint/po/check_translations.sh find these */
const char *langs[NUM_LANGS][2] = {
  {"-", "default"},             /* lang */
  {"Acholi", "acholi"},         /* lang */
  {"Afrikaans", "afrikaans"},   /* lang */
  {"Akan", "akan"},             /* lang */
  {"Albanian", "albanian"},     /* lang */
  {"Amharic", "amharic"},       /* lang */
  {"Arabic", "arabic"},         /* lang */
  {"Aragonese", "aragones"},    /* lang */
  {"Armenian", "armenian"},     /* lang */
  {"Assamese", "assamese"},     /* lang */
  {"Asturian", "asturian"},     /* lang */
  {"Azerbaijani", "azerbaijani"},       /* lang */
  {"Bambara", "bambara"},       /* lang */
  {"Basque", "basque"},         /* lang */
  {"Belarusian", "belarusian"}, /* lang */
  {"Bengali", "bengali"},       /* lang */
  {"Bodo", "bodo"},             /* lang */
  {"Bosnian", "bosnian"},       /* lang */
  {"Breton", "breton"},         /* lang */
  {"Bulgarian", "bulgarian"},   /* lang */
  {"Catalan", "catalan"},       /* lang */
  {"Chinese (Simplified)", "simplified-chinese"},       /* lang */
  {"Chinese (Traditional)", "traditional-chinese"},     /* lang */
  {"Croatian", "croatian"},     /* lang */
  {"Czech", "czech"},           /* lang */
  {"Danish", "danish"},         /* lang */
  {"Dogri", "dogri"},           /* lang */
  {"Dutch", "dutch"},           /* lang */
  {"English (USA)", "american-english"},        /* lang */
  {"English (Australian)", "australian-english"},       /* lang */
  {"English (Britain)", "british"},     /* lang */
  {"English (Canadian)", "canadian-english"},   /* lang */
  {"English (South Africa)", "southafrican-english"},   /* lang */
  {"Esperanto", "esperanto"},   /* lang */
  {"Estonian", "estonian"},     /* lang */
  {"Faroese", "faroese"},       /* lang */
  {"Finnish", "finnish"},       /* lang */
  {"French", "french"},         /* lang */
  {"Fulah", "fulah"},           /* lang */
  {"Gaelic (Ireland)", "gaelic"},       /* lang */
  {"Gaelic (Scotland)", "scottish"},    /* lang */
  {"Galician", "galician"},     /* lang */
  {"Georgian", "georgian"},     /* lang */
  {"German", "german"},         /* lang */
  {"Greek", "greek"},           /* lang */
  {"Gronings", "gronings"},     /* lang */
  {"Gujarati", "gujarati"},     /* lang */
  {"Hebrew", "hebrew"},         /* lang */
  {"Hindi", "hindi"},           /* lang */
  {"Hungarian", "hungarian"},   /* lang */
  {"Icelandic", "icelandic"},   /* lang */
  {"Indonesian", "indonesian"}, /* lang */
  {"Inuktitut", "inuktitut"},   /* lang */
  {"Italian", "italian"},       /* lang */
  {"Japanese", "japanese"},     /* lang */
  {"Kabyle", "kabyle"},         /* lang */
  {"Kannada", "kannada"},       /* lang */
  {"Kashmiri (perso-arabic)", "kashmiri-perso-arabic"}, /* lang */
  {"Kashmiri (devanagari)", "kashmiri-devanagari"},     /* lang */
  {"Khmer", "khmer"},           /* lang */
  {"Kiga", "kiga"},             /* lang */
  {"Kinyarwanda", "kinyarwanda"},       /* lang */
  {"Klingon", "klingon"},       /* lang */
  {"Konkani (Devanagari)", "konkani-devanagari"},       /* lang */
  {"Konkani (Roman)", "konkani-roman"}, /* lang */
  {"Korean", "korean"},         /* lang */
  {"Kurdish", "kurdish"},       /* lang */
  {"Latvian", "latvian"},       /* lang */
  {"Lithuanian", "lithuanian"}, /* lang */
  {"Luganda", "luganda"},       /* lang */
  {"Luxembourgish", "luxembourgish"},   /* lang */
  {"Macedonian", "macedonian"}, /* lang */
  {"Maithili", "maithili"},     /* lang */
  {"Malay", "malay"},           /* lang */
  {"Malayalam", "malayalam"},   /* lang */
  {"Manipuri (Bengali)", "manipuri-bengali"},   /* lang */
  {"Manipuri (Meitei Mayek)", "manipuri-meitei-mayek"}, /* lang */
  {"Marathi", "marathi"},       /* lang */
  {"Mongolian", "mongolian"},   /* lang */
  {"Ndebele", "ndebele"},       /* lang */
  {"Nepali", "nepali"},         /* lang */
  {"Northern Sotho", "northern-sotho"}, /* lang */
  {"Norwegian Bokm�l", "bokmal"},     /* lang */
  {"Norwegian Nynorsk", "nynorsk"},     /* lang */
  {"Occitan", "occitan"},       /* lang */
  {"Ojibwe", "ojibwe"},         /* lang */
  {"Oriya", "oriya"},           /* lang */
  {"Persian", "persian"},       /* lang */
  {"Polish", "polish"},         /* lang */
  {"Portuguese (Portugal)", "portuguese"},      /* lang */
  {"Portuguese (Brazilian)", "brazilian-portuguese"},   /* lang */
  {"Punjabi", "punjabi"},       /* lang */
  {"Romanian", "romanian"},     /* lang */
  {"Russian", "russian"},       /* lang */
  {"Sanskrit", "sanskrit"},     /* lang */
  {"Santali (Devanagari)", "santali-devanagari"},       /* lang */
  {"Santali (Ol-Chiki)", "santali-ol-chiki"},   /* lang */
  {"Sardinian", "sardinian"},   /* lang */
  {"Serbian (cyrillic)", "serbian"},    /* lang */
  {"Serbian (latin)", "serbian-latin"}, /* lang */
  {"Shuswap", "shuswap"},       /* lang */
  {"Sindhi (Devanagari)", "sindhi-devanagari"}, /* lang */
  {"Sindhi (Perso-Arabic)", "sindhi-perso-arabic"},     /* lang */
  {"Sinhala", "sinhala"},       /* lang */
  {"Slovak", "slovak"},         /* lang */
  {"Slovenian", "slovenian"},   /* lang */
  {"Songhay", "songhay"},       /* lang */
  {"Spanish (Mexico)", "mexican-spanish"},      /* lang */
  {"Spanish (Spain)", "spanish"},       /* lang */
  {"Sundanese", "sundanese"},   /* lang */
  {"Swahili", "swahili"},       /* lang */
  {"Swedish", "swedish"},       /* lang */
  {"Tagalog", "tagalog"},       /* lang */
  {"Tamil", "tamil"},           /* lang */
  {"Telugu", "telugu"},         /* lang */
  {"Thai", "thai"},             /* lang */
  {"Tibetan", "tibetan"},       /* lang */
  {"Turkish", "turkish"},       /* lang */
  {"Twi", "twi"},               /* lang */
  {"Ukrainian", "ukrainian"},   /* lang */
  {"Urdu", "urdu"},             /* lang */
  {"Valencian", "valencian"},   /* lang */
  {"Venda", "venda"},           /* lang */
  {"Venetian", "venetian"},     /* lang */
  {"Vietnamese", "vietnamese"}, /* lang */
  {"Walloon", "walloon"},       /* lang */
  {"Welsh", "welsh"},           /* lang */
  {"Wolof", "wolof"},           /* lang */
  {"Xhosa", "xhosa"},           /* lang */
  {"Zapotec (Miahuatlan)", "miahuatlan-zapotec"},       /* lang */
  {"Zulu", "zulu"},             /* lang */
};

#define NUM_COMPLEXITIES 4
const char *complexities[NUM_COMPLEXITIES] = {
  "",
  "advanced",
  "beginner",
  "novice",
};

const char *complexities_names[NUM_COMPLEXITIES] = {
  gettext_noop("(no override)"),
  gettext_noop("Advanced"),
  gettext_noop("Beginner"),
  gettext_noop("Novice"),
};

#define NUM_LAYOUTS 2
#define LAYOUT_NAME 0
const char *layouts[NUM_LAYOUTS][2] = {
  {"abcde", "default.layout"},
  {"qwerty", "qwerty.layout"}
};

#ifdef USE_PAPERSIZES
int num_papersizes;
char **papersizes;
#endif

void measure_and_softwrap(char *&str, int &w, int &h);
void list_fonts(Fl_Hold_Browser *o);
int compare_font_family(const void *a, const void *b);
char * grab_and_dequote_arg(char * config_line, const char * setting_name);

static void construct_conf_filename(int user)
{
  strcpy(conf_filename, "tuxpaint.cfg");        /* default config file */
  conf_filetype = user;

  switch (user)
  {
  case (CURRENT_USER):
#if defined(WIN32)
    {
      char *savedir = 0;
      savedir = GetDefaultSaveDir("TuxPaint");  /* FIXME */
      snprintf(conf_filename, sizeof(conf_filename), "%s/tuxpaint.cfg",
               savedir);
      free(savedir);
    }
#elif defined(__BEOS__) || defined(__HAIKU__)
    if (getenv("HOME") != NULL)
    {
      snprintf(conf_filename, sizeof(conf_filename),
               "%s/tuxpaint.cfg", CONFDIR);
    }
#elif defined(__APPLE__)
    strcpy(conf_filename, macos_filename(MACOS_USER_CONFIG_T));
#else
    /* Linux and other Unixes:  Use 'rc' style (~/.tuxpaintrc) */
    if (getenv("HOME") != NULL)
    {
      snprintf(conf_filename, sizeof(conf_filename),
               "%s/.tuxpaintrc", getenv("HOME"));
    }
#endif
    break;

  case (ALL_USERS):
#if defined(WIN32)
    {
      char *appdir = GetApplicationDir();
      snprintf(conf_filename, sizeof(conf_filename), "%s/tuxpaint.cfg",
               appdir);
      free(appdir);
    }
#elif defined(__BEOS__) || defined(__HAIKU__)
    snprintf(conf_filename, sizeof(conf_filename), "%s/tuxpaint.conf",
             CONFDIR);
#elif defined(__APPLE__)
    strcpy(conf_filename, macos_filename(MACOS_GLOBAL_CONFIG_T));
#else
    snprintf(conf_filename, sizeof(conf_filename), "%s/tuxpaint.conf",
             CONFDIR);
#endif
    break;
  }
}

static void activate_widgets()
{
  /* Video & Sound: */
  /* -------------- */

  CHECKBUTTON_fullscreen->activate();
  if (CHECKBUTTON_fullscreen->value())
  {
    CHECKBUTTON_native->activate();
  }
  CHOICE_windowwidth->activate();
  CHOICE_windowheight->activate();
  CHECKBUTTON_rotate->activate();
  CHECKBUTTON_screensaver->activate();
  CHECKBUTTON_soundeffects->activate();
  CHECKBUTTON_stereo->activate();
  SLIDER_buttonsize->activate();
  CHECKBUTTON_buttonsizeauto->activate();
  SLIDER_colorsrows->activate();

  /* Mouse & Keyboard: */
  /* ----------------- */

  CHECKBUTTON_fancycursorshapes->activate();
  CHECKBUTTON_hidecursor->activate();
  CHECKBUTTON_grabmousepointer->activate();
  CHECKBUTTON_mousewheelsupport->activate();
  CHECKBUTTON_nobuttondistinction->activate();
  CHECKBUTTON_enablekbd->activate();

  /* Simplification: */
  /* --------------- */

  CHECKBUTTON_disableshaperotation->activate();
  CHECKBUTTON_simpleshapeoutlines->activate();
  CHECKBUTTON_showuppercasetextonly->activate();
  CHECKBUTTON_disablequitbutton->activate();
  CHECKBUTTON_disablestampstool->activate();
  CHECKBUTTON_disablestampcontrols->activate();
  CHECKBUTTON_disablestamprotation->activate();
  CHECKBUTTON_disableshapecontrols->activate();
  CHECKBUTTON_disablemagiccontrols->activate();
  CHECKBUTTON_disablemagicsizes->activate();
  CHECKBUTTON_ungroupmagictools->activate();
  CHECKBUTTON_disablelabeltool->activate();
  CHECKBUTTON_disablebrushspacing->activate();
  CHECKBUTTON_disabletemplateexport->activate();
  CHECKBUTTON_disableerase->activate();
  CHOICE_complexity->activate();
  SLIDER_stampsize->activate();

  /* Locale: */
  /* ------- */

  CHOICE_language->activate();
  CHECKBUTTON_mirrorstamps->activate();
  CHECKBUTTON_loadsystemfonts->activate();
  CHECKBUTTON_alllocalefonts->activate();
  BROWSER_uifont->activate();
  if (BROWSER_uifont->value() == UIFONT_MANUAL)
    TEXTINPUT_uifont2->activate();
  else
    TEXTINPUT_uifont2->deactivate();

  /* Printing: */
  /* --------- */

  CHECKBUTTON_allowprinting->activate();
  VALUEINPUT_printdelay->activate();
  CHECKBUTTON_printcfg->activate();
  CHECKBUTTON_usealtprintcmd->activate();
  if (CHECKBUTTON_usealtprintcmd->value())
  {
    TEXTINPUT_printcmd->activate();
  }
  CHECKBUTTON_usealtprintdia->activate();
  if (CHECKBUTTON_usealtprintdia->value())
  {
    TEXTINPUT_printdia->activate();
  }
  ROUNDBUTTON_printdialogmod->activate();
  ROUNDBUTTON_printdialogalways->activate();
  ROUNDBUTTON_printdialognever->activate();

#ifdef USE_PAPERSIZES
  CHOICE_papersize->activate();
#endif

  /* Saving & Exporting: */
  /* ------------------- */

  ROUNDBUTTON_askbeforeoverwriting->activate();
  ROUNDBUTTON_alwaysoverwriteolderversion->activate();
  ROUNDBUTTON_alwayssavenewpicture->activate();
  CHECKBUTTON_usealternatesavedir->activate();
  if (CHECKBUTTON_usealternatesavedir->value())
  {
    FILEINPUT_alternatesavedir->activate();
    BUTTON_browsealternatesavedir->activate();
  }
  CHECKBUTTON_usealternateexportdir->activate();
  if (CHECKBUTTON_usealternateexportdir->value())
  {
    FILEINPUT_alternateexportdir->activate();
    BUTTON_browsealternateexportdir->activate();
  }
  CHECKBUTTON_startblank->activate();
  CHECKBUTTON_newcolorslast->activate();
  CHECKBUTTON_disablesavebutton->activate();
  CHECKBUTTON_autosave->activate();
  CHECKBUTTON_reversesort->activate();

  /* Data: */
  /* ----- */

  CHECKBUTTON_nolockfile->activate();
  CHECKBUTTON_usealternatedatadir->activate();
  if (CHECKBUTTON_usealternatedatadir->value())
  {
    FILEINPUT_alternatedatadir->activate();
    BUTTON_browsealternatedatadir->activate();
  }
  CHECKBUTTON_usecolorfile->activate();
  if (CHECKBUTTON_usecolorfile->value())
  {
    FILEINPUT_colorfile->activate();
    BUTTON_browsecolorfile->activate();
  }

  /* Accessibility */
  /* ------------- */

  CHECKBUTTON_mouse_accessibility->activate();
  CHECKBUTTON_keyboard_mouse->activate();
  CHECKBUTTON_onscreen_keyboard->activate();
  if (CHECKBUTTON_onscreen_keyboard->value() == 1)
  {
    CHOICE_onscreen_keyboard_layout->activate();
    CHECKBUTTON_onscreen_keyboard_disable_change->activate();
  }

  /* Joystick */
  /* -------- */

  SPINNER_joystick_dev->activate();
  SPINNER_joystick_slowness->activate();
  SPINNER_joystick_threshold->activate();
  SPINNER_joystick_maxsteps->activate();
  SPINNER_joystick_hat_slowness->activate();
  SPINNER_joystick_hat_timeout->activate();
  CHECKBUTTON_joystick_buttons_ignore->activate();
  if (CHECKBUTTON_joystick_buttons_ignore->value())
  {
    TEXTINPUT_joystick_buttons_ignore->activate();
  }

  SPINNER_joystick_btn_brush->activate();
  SPINNER_joystick_btn_label->activate();
  SPINNER_joystick_btn_new->activate();
  SPINNER_joystick_btn_print->activate();
  SPINNER_joystick_btn_shapes->activate();
  SPINNER_joystick_btn_undo->activate();
  SPINNER_joystick_btn_eraser->activate();
  SPINNER_joystick_btn_lines->activate();
  SPINNER_joystick_btn_open->activate();
  SPINNER_joystick_btn_redo->activate();
  SPINNER_joystick_btn_stamp->activate();
  SPINNER_joystick_btn_escape->activate();
  SPINNER_joystick_btn_fill->activate();
  SPINNER_joystick_btn_magic->activate();
  SPINNER_joystick_btn_pgsetup->activate();
  SPINNER_joystick_btn_save->activate();
  SPINNER_joystick_btn_text->activate();
}

static void deactivate_widgets()
{
  /* Video & Sound: */
  /* -------------- */

  CHECKBUTTON_fullscreen->deactivate();
  CHECKBUTTON_native->deactivate();
  CHOICE_windowwidth->deactivate();
  CHOICE_windowheight->deactivate();
  CHECKBUTTON_rotate->deactivate();
  CHECKBUTTON_screensaver->deactivate();
  CHECKBUTTON_soundeffects->deactivate();
  CHECKBUTTON_stereo->deactivate();
  SLIDER_buttonsize->deactivate();
  CHECKBUTTON_buttonsizeauto->deactivate();
  SLIDER_colorsrows->deactivate();

  /* Mouse & Keyboard: */
  /* ----------------- */

  CHECKBUTTON_fancycursorshapes->deactivate();
  CHECKBUTTON_hidecursor->deactivate();
  CHECKBUTTON_grabmousepointer->deactivate();
  CHECKBUTTON_mousewheelsupport->deactivate();
  CHECKBUTTON_nobuttondistinction->deactivate();
  CHECKBUTTON_enablekbd->deactivate();

  /* Simplification: */
  /* --------------- */

  CHECKBUTTON_disableshaperotation->deactivate();
  CHECKBUTTON_simpleshapeoutlines->deactivate();
  CHECKBUTTON_showuppercasetextonly->deactivate();
  CHECKBUTTON_disablequitbutton->deactivate();
  CHECKBUTTON_disablestampstool->deactivate();
  CHECKBUTTON_disablestampcontrols->deactivate();
  CHECKBUTTON_disablestamprotation->deactivate();
  CHECKBUTTON_disablemagiccontrols->deactivate();
  CHECKBUTTON_disablemagicsizes->deactivate();
  CHECKBUTTON_ungroupmagictools->deactivate();
  CHECKBUTTON_disableshapecontrols->deactivate();
  CHECKBUTTON_disablelabeltool->deactivate();
  CHECKBUTTON_disablebrushspacing->deactivate();
  CHECKBUTTON_disabletemplateexport->deactivate();
  CHECKBUTTON_disableerase->deactivate();
  CHOICE_complexity->deactivate();
  SLIDER_stampsize->deactivate();

  /* Locale: */
  /* ------- */

  CHOICE_language->deactivate();
  CHECKBUTTON_mirrorstamps->deactivate();
  CHECKBUTTON_loadsystemfonts->deactivate();
  CHECKBUTTON_alllocalefonts->deactivate();
  BROWSER_uifont->deactivate();
  TEXTINPUT_uifont2->deactivate();

  /* Printing: */
  /* --------- */

  CHECKBUTTON_allowprinting->deactivate();
  VALUEINPUT_printdelay->deactivate();
  CHECKBUTTON_printcfg->deactivate();
  TEXTINPUT_printcmd->deactivate();
  CHECKBUTTON_usealtprintcmd->deactivate();
  TEXTINPUT_printdia->deactivate();
  CHECKBUTTON_usealtprintdia->deactivate();
  ROUNDBUTTON_printdialogmod->deactivate();
  ROUNDBUTTON_printdialogalways->deactivate();
  ROUNDBUTTON_printdialognever->deactivate();
#ifdef USE_PAPERSIZES
  CHOICE_papersize->deactivate();
#endif

  /* Saving & Exporting: */
  /* ------------------- */

  ROUNDBUTTON_askbeforeoverwriting->deactivate();
  ROUNDBUTTON_alwaysoverwriteolderversion->deactivate();
  ROUNDBUTTON_alwayssavenewpicture->deactivate();
  FILEINPUT_alternatesavedir->deactivate();
  BUTTON_browsealternatesavedir->deactivate();
  CHECKBUTTON_usealternatesavedir->deactivate();
  FILEINPUT_alternateexportdir->deactivate();
  BUTTON_browsealternateexportdir->deactivate();
  CHECKBUTTON_usealternateexportdir->deactivate();
  CHECKBUTTON_startblank->deactivate();
  CHECKBUTTON_newcolorslast->deactivate();
  CHECKBUTTON_disablesavebutton->deactivate();
  CHECKBUTTON_autosave->deactivate();
  CHECKBUTTON_reversesort->deactivate();

  /* Data: */
  /* ----- */

  FILEINPUT_alternatedatadir->deactivate();
  BUTTON_browsealternatedatadir->deactivate();
  CHECKBUTTON_usealternatedatadir->deactivate();
  CHECKBUTTON_usecolorfile->deactivate();
  FILEINPUT_colorfile->deactivate();
  BUTTON_browsecolorfile->deactivate();

  /* Accessibility */
  /* ------------- */

  CHECKBUTTON_mouse_accessibility->deactivate();
  CHECKBUTTON_keyboard_mouse->deactivate();
  CHECKBUTTON_onscreen_keyboard->deactivate();
  CHOICE_onscreen_keyboard_layout->deactivate();
  CHECKBUTTON_onscreen_keyboard_disable_change->deactivate();

  /* Joystick */
  /* -------- */

  SPINNER_joystick_dev->deactivate();
  SPINNER_joystick_slowness->deactivate();
  SPINNER_joystick_threshold->deactivate();
  SPINNER_joystick_maxsteps->deactivate();
  SPINNER_joystick_hat_slowness->deactivate();
  SPINNER_joystick_hat_timeout->deactivate();
  TEXTINPUT_joystick_buttons_ignore->deactivate();
  CHECKBUTTON_joystick_buttons_ignore->deactivate();

  SPINNER_joystick_btn_brush->deactivate();
  SPINNER_joystick_btn_label->deactivate();
  SPINNER_joystick_btn_new->deactivate();
  SPINNER_joystick_btn_print->deactivate();
  SPINNER_joystick_btn_shapes->deactivate();
  SPINNER_joystick_btn_undo->deactivate();
  SPINNER_joystick_btn_eraser->deactivate();
  SPINNER_joystick_btn_lines->deactivate();
  SPINNER_joystick_btn_open->deactivate();
  SPINNER_joystick_btn_redo->deactivate();
  SPINNER_joystick_btn_stamp->deactivate();
  SPINNER_joystick_btn_escape->deactivate();
  SPINNER_joystick_btn_fill->deactivate();
  SPINNER_joystick_btn_magic->deactivate();
  SPINNER_joystick_btn_pgsetup->deactivate();
  SPINNER_joystick_btn_save->deactivate();
  SPINNER_joystick_btn_text->deactivate();
}

static bool conf_file_exists(int user)
{
  FILE *fd;
  char *flg = (char *) "r";

  construct_conf_filename(user);
  fd = fopen(conf_filename, flg);
  if (fd != NULL)
  {
    fclose(fd);
    return true;
  }
  return false;
}


void save_conf(void)
{
  FILE *fd;
  char *flg = (char *) "w";

  construct_conf_filename(CHOICE_settings->value());

#if defined(__APPLE__)
  switch (conf_filetype)
  {
  case CURRENT_USER:
    macos_mkdir(macos_dirname(conf_filename));
    break;

  case ALL_USERS:
    /*
     * macOS can't write directly to the global file; write to a temp
     * file here, then copy it to global path later.
     */
    strcpy(conf_filename, macos_filename(MACOS_TEMP_CONFIG_T));
    break;
  }
#endif /* __APPLE__ */

  fd = fopen(conf_filename, flg);

  if (fd == NULL)
  {
    fl_message("Cannot write %s: %s", conf_filename, strerror(errno));
    return;
  }

  fprintf(fd, "# Generated by tuxpaint-config version " VER_VERSION "\n");

  /* For all user vs current user setting */
  bool checkall_users_conf_file = conf_file_exists(CURRENT_USER);
  if (checkall_users_conf_file
      && (use_merge_discard_allusersetting->value() ==
          SYS_SETTING_STRATEGY_DISCARD))
  {
    fprintf(fd, "sysconfig=no\n");
  }

  /* Video & Sound: */
  /* -------------- */

  /* Video: */
  if ((CHKBUT_fulls_isdef()) == 0)
  {
    if (CHECKBUTTON_native->value() == 0)
      fprintf(fd, "fullscreen=yes\n");
    else
      fprintf(fd, "fullscreen=native\n");
  }

  if (CHECKBUTTON_native->value() == 1 && CHKBUT_fulls_isdef() == 1)
  {
    fprintf(fd, "native=yes\n");        /* keep track of native when fullscreen is disabled */
  }

  if (CHOICE_windowwidth->value() != -1 &&
      CHOICE_windowheight->value() != -1
      /* squelch "array subscript -1 is below array bounds" warning -bjk 2024.10.20 */)
  {
    if (CHOICE_windowwidth->value() != DEF_WINDOW_WIDTH ||
        CHOICE_windowheight->value() != DEF_WINDOW_HEIGHT)
      fprintf(fd, "windowsize=%sx%s\n",
              window_widths[CHOICE_windowwidth->value()],
              window_heights[CHOICE_windowheight->value()]);
  }

  if (CHECKBUTTON_rotate->value() != 0)
    fprintf(fd, "orient=portrait\n");

  if (CHECKBUTTON_screensaver->value() != 0)
    fprintf(fd, "allowscreensaver=yes\n");

  /* Sound: */
  if (CHECKBUTTON_soundeffects->value() == 0)
    fprintf(fd, "nosound=yes\n");

  /* Stereo: */
  if (CHECKBUTTON_stereo->value() == 0)
    fprintf(fd, "nostereo=yes\n");

  /* Button size: */
  if (CHECKBUTTON_buttonsizeauto->value() != 0)
    fprintf(fd, "buttonsize=auto\n");
  else if (SLIDER_buttonsize->value() != 48)
    fprintf(fd, "buttonsize=%d\n", (int) floor(SLIDER_buttonsize->value()));

  /* # of color palette rows: */
  if (SLIDER_colorsrows->value() != 1)
    fprintf(fd, "colorsrows=%d\n", (int) floor(SLIDER_colorsrows->value()));

  /* Mouse & Keyboard: */
  /* ----------------- */

  /* Mouse: */
  if ((CHKBUT_fancycursors_isdef()) == 0)
    fprintf(fd, "nofancycursors=yes\n");
  if (CHECKBUTTON_hidecursor->value() == 1)
    fprintf(fd, "hidecursor=yes\n");
  if ((CHKBUT_grabmouse_isdef()) == 0)
    fprintf(fd, "grab=yes\n");
  if ((CHKBUT_mousewheelsup_isdef()) == 0)
    fprintf(fd, "nowheelmouse=yes\n");
  if (CHECKBUTTON_nobuttondistinction->value() == 1)
    fprintf(fd, "nobuttondistinction=yes\n");

  /* Keyboard: */
  if (CHECKBUTTON_enablekbd->value() == 0)
    fprintf(fd, "noshortcuts=yes\n");
// if ((CHKBUT_xxx_isdef())==0)
//    fprintf(fd,"keyboard=yes\n");


  /* Simplification: */
  /* --------------- */

  /* Interface: */
  if (CHECKBUTTON_disableshaperotation->value() == 1)
    fprintf(fd, "simpleshapes=yes\n");
  if ((CHKBUT_smploutlns_isdef()) == 0)
    fprintf(fd, "outlines=yes\n");
  if ((CHKBUT_showUConly_isdef()) == 0)
    fprintf(fd, "uppercase=yes\n");

  /* Control: */
  if ((CHKBUT_disablequit_isdef()) == 0)
    fprintf(fd, "noquit=yes\n");
  if (CHECKBUTTON_disablestampstool->value() == 1)
    fprintf(fd, "nostamps=yes\n");
  if (CHECKBUTTON_disablestampcontrols->value() == 1)
    fprintf(fd, "nostampcontrols=yes\n");
  if (CHECKBUTTON_disablestamprotation->value() == 1)
    fprintf(fd, "nostamprotation=yes\n");
  if (CHECKBUTTON_disablemagiccontrols->value() == 1)
    fprintf(fd, "nomagiccontrols=yes\n");
  if (CHECKBUTTON_disablemagicsizes->value() == 1)
    fprintf(fd, "nomagicsizes=yes\n");
  if (CHECKBUTTON_ungroupmagictools->value() == 1)
    fprintf(fd, "ungroupmagictools=yes\n");
  if (CHECKBUTTON_disableshapecontrols->value() == 1)
    fprintf(fd, "noshapecontrols=yes\n");
  if (CHECKBUTTON_disablelabeltool->value() == 1)
    fprintf(fd, "nolabel=yes\n");
  if (CHECKBUTTON_disablebrushspacing->value() == 1)
    fprintf(fd, "nobrushspacing=yes\n");
  if (CHECKBUTTON_disabletemplateexport->value() == 1)
    fprintf(fd, "notemplateexport=yes\n");
  if (CHECKBUTTON_disableerase->value() == 1)
    fprintf(fd, "noerase=yes\n");
  if (CHOICE_complexity->value() != 0)
    fprintf(fd, "complexity=%s\n", complexities[CHOICE_complexity->value()]);

  /* Stamp size */
  if (SLIDER_stampsize->value() != -1)
    fprintf(fd, "stampsize=%d\n", (int) floor(SLIDER_stampsize->value()));

  /* Locale: */
  /* ------- */

  if (CHOICE_language->value() != 0)
    fprintf(fd, "lang=%s\n", langs[CHOICE_language->value()][LANG_CODE]);
  if ((CHKBUT_mirstamps_isdef()) == 0)
    fprintf(fd, "mirrorstamps=yes\n");

  if (CHECKBUTTON_loadsystemfonts->value() == 1)
    fprintf(fd, "sysfonts=yes\n");

  if (CHECKBUTTON_alllocalefonts->value() == 1)
    fprintf(fd, "alllocalefonts=yes\n");

  if ((BROWSER_uifont_isdef()) == 0)
  {
    if (BROWSER_uifont->value() != UIFONT_MANUAL) {
      fprintf(fd, "uifont=\"%s\"\n", BROWSER_uifont->text(BROWSER_uifont->value()));
    } else {
      if (strlen(TEXTINPUT_uifont2->value()) > 0) {
        fprintf(fd, "uifont=\"%s\"\n", TEXTINPUT_uifont2->value());
      }
    }
  }

  /* Printing: */
  /* --------- */

  /* Print Permissions: */
  if ((CHKBUT_allowprint_isdef()) == 0)
    fprintf(fd, "noprint=yes\n");
  if ((VALINP_printdel_isdef()) == 0)
    fprintf(fd, "printdelay=%d\n", (int) VALUEINPUT_printdelay->value());

  /* Print configuration file use: */
  if (CHECKBUTTON_printcfg->value() == 1)
    fprintf(fd, "printcfg=yes\n");

  /* Print Command: */
  if (strlen(TEXTINPUT_printcmd->value()) > 0)
  {
    if (CHECKBUTTON_usealtprintcmd->value() == 1)
      fprintf(fd, "printcommand=\"%s\"\n", TEXTINPUT_printcmd->value());
    else
      fprintf(fd, "# printcommand=\"%s\"\n", TEXTINPUT_printcmd->value());
  }

  /* Print Dialog Alt. Command: */
  if (strlen(TEXTINPUT_printdia->value()) > 0)
  {
    if (CHECKBUTTON_usealtprintdia->value() == 1)
      fprintf(fd, "altprintcommand=\"%s\"\n", TEXTINPUT_printdia->value());
    else
      fprintf(fd, "# altprintcommand=\"%s\"\n", TEXTINPUT_printdia->value());
  }

  /* Show print dialog? */
  if (ROUNDBUTTON_printdialogalways->value() == 1)
    fprintf(fd, "altprint=always\n");
  else if (ROUNDBUTTON_printdialognever->value() == 1)
    fprintf(fd, "altprint=never\n");

#ifdef USE_PAPERSIZES
  /* Specify a paper size? */
  if (CHOICE_papersize->value() != 0)
    fprintf(fd, "papersize=%s\n", papersizes[CHOICE_papersize->value()]);
#endif


  /* Saving & Exporting: */
  /* ------------------- */

  /* Save Over Earlier Work: */
  if (ROUNDBUTTON_alwaysoverwriteolderversion->value() == 1)
    fprintf(fd, "saveover=yes\n");
  else if (ROUNDBUTTON_alwayssavenewpicture->value() == 1)
    fprintf(fd, "saveover=new\n");

  /* Save Directory: */
  if (strlen(FILEINPUT_alternatesavedir->value()) > 0)
  {
    if ((CHKBUT_altsavedir_isdef()) == 0)
      fprintf(fd, "savedir=\"%s\"\n", FILEINPUT_alternatesavedir->value());
    else
      fprintf(fd, "# savedir=\"%s\"\n", FILEINPUT_alternatesavedir->value());
  }

  /* Export Directory: */
  if (strlen(FILEINPUT_alternateexportdir->value()) > 0)
  {
    if ((CHKBUT_altexportdir_isdef()) == 0)
      fprintf(fd, "exportdir=\"%s\"\n", FILEINPUT_alternateexportdir->value());
    else
      fprintf(fd, "# exportdir=\"%s\"\n", FILEINPUT_alternateexportdir->value());
  }

  /* Start blank? */
  if (CHECKBUTTON_startblank->value() == 1)
    fprintf(fd, "startblank=yes\n");

  /* "New" colors last? */
  if (CHECKBUTTON_newcolorslast->value() == 1)
    fprintf(fd, "newcolorslast=yes\n");

  /* Disable saving: */
  if (CHECKBUTTON_disablesavebutton->value() == 1)
    fprintf(fd, "nosave=yes\n");

  /* Auto-save on quit? */
  if (CHECKBUTTON_autosave->value() == 1)
    fprintf(fd, "autosave=yes\n");

  /* Reverse sort order of Open and Slideshow dialogs? */
  if (CHECKBUTTON_reversesort->value() == 1)
    fprintf(fd, "reversesort=yes\n");

  /* Data: */
  /* ----- */

  /* Lockfile: */
  if (CHECKBUTTON_nolockfile->value() == 1)
    fprintf(fd, "nolockfile=yes\n");

  /* Data Directory: */
  if (strlen(FILEINPUT_alternatedatadir->value()) > 0)
  {
    if ((CHKBUT_altdatadir_isdef()) == 0)
      fprintf(fd, "datadir=\"%s\"\n", FILEINPUT_alternatedatadir->value());
    else
      fprintf(fd, "# datadir=\"%s\"\n", FILEINPUT_alternatedatadir->value());
  }

  /* Color file: */
  if (strlen(FILEINPUT_colorfile->value()) > 0)
  {
    if (CHECKBUTTON_usecolorfile->value() == 1)
      fprintf(fd, "colorfile=\"%s\"\n", FILEINPUT_colorfile->value());
    else
      fprintf(fd, "# colorfile=\"%s\"\n", FILEINPUT_colorfile->value());
  }

  /* Accessibility */
  /* ------------- */

  if (CHECKBUTTON_mouse_accessibility->value() == 1)
    fprintf(fd, "mouse-accessibility=yes\n");
  if (CHECKBUTTON_keyboard_mouse->value() == 1)
    fprintf(fd, "keyboard=yes\n");
  if (CHECKBUTTON_onscreen_keyboard->value() == 1)
  {
    fprintf(fd, "onscreen-keyboard=yes\n");

    if (CHOICE_onscreen_keyboard_layout->value() != 0)
      fprintf(fd, "onscreen-keyboard-layout=%s\n",
              layouts[CHOICE_onscreen_keyboard_layout->value()][LANG_CODE]);

    if (CHECKBUTTON_onscreen_keyboard_disable_change->value() == 1)
      fprintf(fd, "onscreen-keyboard-disable-change=yes\n");
  }



  /* Joystick */
  /* -------- */

  if (SPINNER_joystick_dev->value() != 0)
    fprintf(fd, "joystick-dev=%d\n", (int) SPINNER_joystick_dev->value());

  if (SPINNER_joystick_slowness->value() != 15)
    fprintf(fd, "joystick-slowness=%d\n",
            (int) SPINNER_joystick_slowness->value());

  if (SPINNER_joystick_threshold->value() != 3200)
    fprintf(fd, "joystick-threshold=%d\n",
            (int) SPINNER_joystick_threshold->value());

  if (SPINNER_joystick_maxsteps->value() != 7)
    fprintf(fd, "joystick-maxsteps=%d\n",
            (int) SPINNER_joystick_maxsteps->value());

  if (SPINNER_joystick_hat_slowness->value() != 15)
    fprintf(fd, "joystick-hat-slowness=%d\n",
            (int) SPINNER_joystick_hat_slowness->value());

  if (SPINNER_joystick_hat_timeout->value() != 1000)
    fprintf(fd, "joystick-hat-timeout=%d\n",
            (int) SPINNER_joystick_hat_timeout->value());

  /* Disable buttons: */
  if (strlen(TEXTINPUT_joystick_buttons_ignore->value()) > 0)
  {
    if (CHECKBUTTON_joystick_buttons_ignore->value() == 1)
      fprintf(fd, "joystick-buttons-ignore=%s\n",
              TEXTINPUT_joystick_buttons_ignore->value());
    else
      fprintf(fd, "# joystick-buttons-ignore=%s\n",
              TEXTINPUT_joystick_buttons_ignore->value());
  }

  if (SPINNER_joystick_btn_brush->value() != 255)
    fprintf(fd, "joystick-btn-brush=%d\n",
            (int) SPINNER_joystick_btn_brush->value());
  if (SPINNER_joystick_btn_label->value() != 255)
    fprintf(fd, "joystick-btn-label=%d\n",
            (int) SPINNER_joystick_btn_label->value());
  if (SPINNER_joystick_btn_new->value() != 255)
    fprintf(fd, "joystick-btn-new=%d\n",
            (int) SPINNER_joystick_btn_new->value());
  if (SPINNER_joystick_btn_print->value() != 255)
    fprintf(fd, "joystick-btn-print=%d\n",
            (int) SPINNER_joystick_btn_print->value());
  if (SPINNER_joystick_btn_shapes->value() != 255)
    fprintf(fd, "joystick-btn-shapes=%d\n",
            (int) SPINNER_joystick_btn_shapes->value());
  if (SPINNER_joystick_btn_undo->value() != 255)
    fprintf(fd, "joystick-btn-undo=%d\n",
            (int) SPINNER_joystick_btn_undo->value());
  if (SPINNER_joystick_btn_eraser->value() != 255)
    fprintf(fd, "joystick-btn-eraser=%d\n",
            (int) SPINNER_joystick_btn_eraser->value());
  if (SPINNER_joystick_btn_lines->value() != 255)
    fprintf(fd, "joystick-btn-lines=%d\n",
            (int) SPINNER_joystick_btn_lines->value());
  if (SPINNER_joystick_btn_open->value() != 255)
    fprintf(fd, "joystick-btn-open=%d\n",
            (int) SPINNER_joystick_btn_open->value());
  if (SPINNER_joystick_btn_redo->value() != 255)
    fprintf(fd, "joystick-btn-redo=%d\n",
            (int) SPINNER_joystick_btn_redo->value());
  if (SPINNER_joystick_btn_stamp->value() != 255)
    fprintf(fd, "joystick-btn-stamp=%d\n",
            (int) SPINNER_joystick_btn_stamp->value());
  if (SPINNER_joystick_btn_escape->value() != 255)
    fprintf(fd, "joystick-btn-escape=%d\n",
            (int) SPINNER_joystick_btn_escape->value());
  if (SPINNER_joystick_btn_fill->value() != 255)
    fprintf(fd, "joystick-btn-fill=%d\n",
            (int) SPINNER_joystick_btn_fill->value());
  if (SPINNER_joystick_btn_magic->value() != 255)
    fprintf(fd, "joystick-btn-magic=%d\n",
            (int) SPINNER_joystick_btn_magic->value());
  if (SPINNER_joystick_btn_pgsetup->value() != 255)
    fprintf(fd, "joystick-btn-pgsetup=%d\n",
            (int) SPINNER_joystick_btn_pgsetup->value());
  if (SPINNER_joystick_btn_save->value() != 255)
    fprintf(fd, "joystick-btn-save=%d\n",
            (int) SPINNER_joystick_btn_save->value());
  if (SPINNER_joystick_btn_text->value() != 255)
    fprintf(fd, "joystick-btn-text=%d\n",
            (int) SPINNER_joystick_btn_text->value());

  fclose(fd);

#if defined(__APPLE__)
  switch (conf_filetype)
  {
  case ALL_USERS:
    {
      const char *global_conf = macos_filename(MACOS_GLOBAL_CONFIG_T);
      int isok = 0;

      /*
       * macOS can't write directly to the global file so we wrote to a
       * temp file.  Move it into the global path here.
       */
      isok = macos_sudo_cp(conf_filename, global_conf);
      macos_unlink(conf_filename);

      if (!isok)
      {
        fl_message("Cannot write %s: %s", global_conf, macos_error());
        return;
      }

      break;
    }
  }
#endif /* __APPLE__ */

  set_change_status(0);

/*
   FIXME: Add "--keyboard" option to UI and save/load routines
*/
}


void load_conf(void)
{
  FILE *fd;
  char *flg = (char *) "r";
  char line[256];
  bool all_users_conf_file = conf_file_exists(ALL_USERS);

  /* Start with all settings at their default value */
  CB_SetAllTabsToDefaults();

  switch (CHOICE_settings->value())
  {
  case (CURRENT_USER):
    /* We're dealing with the current user's settings */
    use_merge_discard_allusersetting->show();
    if (all_users_conf_file == false)
    {
      /* A system-wide (all-users) config file doesn't exist */
      use_merge_discard_allusersetting->deactivate();
      activate_widgets();
    }
    else
    {
      bool cur_user_conf_file = conf_file_exists(CURRENT_USER);

      /* A system-wide (all-users) config file exists */

      use_merge_discard_allusersetting->activate();
      if (cur_user_conf_file == true)
      {
        /* User conf exists */
        if (use_merge_discard_allusersetting->value() ==
            SYS_SETTING_STRATEGY_USEONLY)
        {
          /* But we want to only use the system ("all users") config file, so deactivate widgets */
          deactivate_widgets();
        }
      }
      else
      {
        /* No user conf exists (a new install/user, or
           perhaps they asked to use all-user settings before?) */
        activate_widgets();
      }
    }
    break;

  case (ALL_USERS):
    /* We're dealing with the system-wide ("All Users") settings */
    use_merge_discard_allusersetting->hide();
    activate_widgets();
    break;
  }

  construct_conf_filename(CHOICE_settings->value());

  fd = fopen(conf_filename, flg);

  if (fd != NULL)
  {
    do
    {
      if (fgets(line, sizeof(line), fd) && !feof(fd))
      {
        /* Trim EOL char(s): */

        while (strlen(line) > 0 &&
               (line[strlen(line) - 1] == '\r' ||
                line[strlen(line) - 1] == '\n'))
        {
          line[strlen(line) - 1] = '\0';
        }


        /* Determine what option it is, and if we should alter the UI
           to correspond */

        /* Video & Sound: */
        if (strcmp(line, "fullscreen=yes") == 0)
          CHECKBUTTON_fullscreen->value(1);

        if (strcmp(line, "fullscreen=native") == 0)
        {
          CHECKBUTTON_fullscreen->value(1);
          CHECKBUTTON_native->value(1);
        }

        if (strcmp(line, "native=yes") == 0)
        {
          CHECKBUTTON_native->value(1);
        }

        if (strstr(line, "windowsize=") == line)
        {
          int i;
          char tmp[32];

          for (i = 0; i < NUM_WINDOW_WIDTHS; i++)
          {
            sprintf(tmp, "windowsize=%sx", window_widths[i]);
            if (strstr(line, tmp) == line)
              CHOICE_windowwidth->value(i);
          }

          for (i = 0; i < NUM_WINDOW_WIDTHS; i++)
          {
            sprintf(tmp, "windowsize=%sx%s",
                    window_widths[CHOICE_windowwidth->value()],
                    window_heights[i]);
            if (strcmp(line, tmp) == 0)
              CHOICE_windowheight->value(i);
          }

          update_canvas_size();
        }
        else if (strcmp(line, "orient=portrait") == 0)
          CHECKBUTTON_rotate->value(1);
        else if (strcmp(line, "allowscreensaver=yes") == 0)
          CHECKBUTTON_screensaver->value(1);
        else if (strcmp(line, "nosound=yes") == 0)
          CHECKBUTTON_soundeffects->value(0);
        else if (strcmp(line, "nostereo=yes") == 0)
          CHECKBUTTON_stereo->value(0);

        /* Mouse & Keyboard: */
        else if (strcmp(line, "nofancycursors=yes") == 0)
          CHECKBUTTON_fancycursorshapes->value(0);
        else if (strcmp(line, "hidecursor=yes") == 0)
          CHECKBUTTON_hidecursor->value(1);
        else if (strcmp(line, "grab=yes") == 0)
          CHECKBUTTON_grabmousepointer->value(1);
        else if (strcmp(line, "nowheelmouse=yes") == 0)
          CHECKBUTTON_mousewheelsupport->value(0);
        else if (strcmp(line, "nobuttondistinction=yes") == 0)
          CHECKBUTTON_nobuttondistinction->value(1);
        else if (strcmp(line, "noshortcuts=yes") == 0)
          CHECKBUTTON_enablekbd->value(0);

        /* Simplifications: */
        else if (strcmp(line, "simpleshapes=yes") == 0)
          CHECKBUTTON_disableshaperotation->value(1);
        else if (strcmp(line, "outlines=yes") == 0)
          CHECKBUTTON_simpleshapeoutlines->value(1);
        else if (strcmp(line, "uppercase=yes") == 0)
          CHECKBUTTON_showuppercasetextonly->value(1);
        else if (strcmp(line, "noquit=yes") == 0)
          CHECKBUTTON_disablequitbutton->value(1);
        else if (strcmp(line, "nostamps=yes") == 0)
          CHECKBUTTON_disablestampstool->value(1);
        else if (strcmp(line, "nostampcontrols=yes") == 0)
          CHECKBUTTON_disablestampcontrols->value(1);
        else if (strcmp(line, "nostamprotation=yes") == 0)
          CHECKBUTTON_disablestamprotation->value(1);
        else if (strcmp(line, "nomagiccontrols=yes") == 0)
          CHECKBUTTON_disablemagiccontrols->value(1);
        else if (strcmp(line, "nomagicsizes=yes") == 0)
          CHECKBUTTON_disablemagicsizes->value(1);
        else if (strcmp(line, "ungroupmagictools=yes") == 0)
          CHECKBUTTON_ungroupmagictools->value(1);
        else if (strcmp(line, "noshapecontrols=yes") == 0)
          CHECKBUTTON_disableshapecontrols->value(1);
        else if (strcmp(line, "nolabel=yes") == 0)
          CHECKBUTTON_disablelabeltool->value(1);
        else if (strcmp(line, "nobrushspacing=yes") == 0)
          CHECKBUTTON_disablebrushspacing->value(1);
        else if (strcmp(line, "notemplateexport=yes") == 0)
          CHECKBUTTON_disabletemplateexport->value(1);
        else if (strcmp(line, "noerase=yes") == 0)
          CHECKBUTTON_disableerase->value(1);
        else if (strstr(line, "complexity=") == line)
        {
          char *complexity;
          int i;

          complexity = strstr(line, "complexity=") + strlen("complexity=");

          for (i = 0; i < NUM_COMPLEXITIES; i++)
          {
            if (strcmp(complexity, complexities[i]) == 0)
              CHOICE_complexity->value(i);
          }
        }
        else if (strstr(line, "stampsize=") == line)
        {
          SLIDER_stampsize->value(atoi(line + 10));
          SLIDER_stampsize->do_callback();
        }
        else if (strcmp(line, "buttonsize=auto") == 0)
        {
          CHECKBUTTON_buttonsizeauto->value(1);
          SLIDER_buttonsize->do_callback();
        }
        else if (strstr(line, "buttonsize=") == line)
        {
          CHECKBUTTON_buttonsizeauto->value(0);
          SLIDER_buttonsize->value(atoi(line + 11));
          SLIDER_buttonsize->do_callback();
        }
        else if (strstr(line, "colorsrows=") == line)
        {
          SLIDER_colorsrows->value(atoi(line + 11));
          SLIDER_colorsrows->do_callback();
        }

        /* Locale: */
        else if (strstr(line, "lang=") == line)
        {
          char *lang;
          int i;

          lang = strstr(line, "lang=") + strlen("lang=");

          for (i = 0; i < NUM_LANGS; i++)
          {
            if (strcmp(lang, langs[i][LANG_CODE]) == 0)
              CHOICE_language->value(i);
          }
        }
        else if (strcmp(line, "mirrorstamps=yes") == 0)
          CHECKBUTTON_mirrorstamps->value();
        else if (strcmp(line, "sysfonts=yes") == 0)
          CHECKBUTTON_loadsystemfonts->value(1);
        else if (strcmp(line, "alllocalefonts=yes") == 0)
          CHECKBUTTON_alllocalefonts->value(1);
        else if (strstr(line, "uifont=") == line)
        {
          char * font_str;
          int i, found;

          font_str = grab_and_dequote_arg(line, "uifont");

          if (font_str[0] != '\0')
          {
            found = 0;
            for (i = 1; i <= BROWSER_uifont->size(); i++) // N.B. array is 1-based!
            {
              if (strcmp(BROWSER_uifont->text(i), font_str) == 0)
              {
                BROWSER_uifont->value(i);
                BROWSER_uifont->make_visible(i);
                found = 1;
              }
            }

            if (found == 0) {
              /* Font listed in the config. was not found in the list
                 (i.e., not found by Pango!)  Set the list to "manual",
                 and fill in and activate the manual type-in font field. */
              BROWSER_uifont->value(UIFONT_MANUAL);
              BROWSER_uifont->make_visible(UIFONT_MANUAL);
              TEXTINPUT_uifont2->value(font_str);
              TEXTINPUT_uifont2->activate();
            }
          }
          free(font_str);
        }

        /* Printing: */
        else if (strcmp(line, "noprint=yes") == 0)
          CHECKBUTTON_allowprinting->value(0);
        else if (strcmp(line, "printcfg=yes") == 0)
          CHECKBUTTON_printcfg->value(1);
        else if (strstr(line, "printdelay=") == line)
        {
          VALUEINPUT_printdelay->value(atoi(strstr(line, "printdelay=") +
                                            strlen("printdelay=")));
        }
        else if (strstr(line, "printcommand=") != NULL)
        {
          char * dequoted_str;

          dequoted_str = grab_and_dequote_arg(line, "printcommand");
          TEXTINPUT_printcmd->value(dequoted_str);
          free(dequoted_str);

          if (strstr(line, "printcommand=") == line)
          {
            /* Active! */

            CHECKBUTTON_usealtprintcmd->value(1);
            TEXTINPUT_printcmd->activate();
          }
          else
          {
            /* Inactive! */

            CHECKBUTTON_usealtprintcmd->value(0);
            TEXTINPUT_printcmd->deactivate();
          }
        }
        else if (strstr(line, "altprintcommand=") != NULL)
        {
          char * dequoted_str;

          dequoted_str = grab_and_dequote_arg(line, "altprintcommand");
          TEXTINPUT_printcmd->value(dequoted_str);
          free(dequoted_str);

          if (strstr(line, "altprintcommand=") == line)
          {
            /* Active! */
            CHECKBUTTON_usealtprintdia->value(1);
            TEXTINPUT_printdia->activate();
          }
          else
          {
            /* Inactive! */
            CHECKBUTTON_usealtprintdia->value(0);
            TEXTINPUT_printdia->deactivate();
          }
        }

        /* Print Dialog: */
        else if (strcmp(line, "altprint=always") == 0)
        {
          ROUNDBUTTON_printdialogmod->value(0);
          ROUNDBUTTON_printdialogalways->value(1);
          ROUNDBUTTON_printdialognever->value(0);
        }
        else if (strcmp(line, "altprint=never") == 0)
        {
          ROUNDBUTTON_printdialogmod->value(0);
          ROUNDBUTTON_printdialogalways->value(0);
          ROUNDBUTTON_printdialognever->value(1);
        }
        else if (strcmp(line, "altprint=mod") == 0)
        {
          ROUNDBUTTON_printdialogmod->value(1);
          ROUNDBUTTON_printdialogalways->value(0);
          ROUNDBUTTON_printdialognever->value(0);
        }

#ifdef USE_PAPERSIZES
        /* Paper Size: */
        else if (strstr(line, "papersize=") == line)
        {
          for (int i = 1; i < num_papersizes; i++)
          {
            if (strcmp(line + strlen("papersize="), papersizes[i]) == 0)
              CHOICE_papersize->value(i);
          }
        }
#endif

        /* Saving: */
        else if (strcmp(line, "saveover=ask") == 0)
        {
          ROUNDBUTTON_askbeforeoverwriting->value(1);
          ROUNDBUTTON_alwaysoverwriteolderversion->value(0);
          ROUNDBUTTON_alwayssavenewpicture->value(0);
        }
        else if (strcmp(line, "saveover=yes") == 0)
        {
          ROUNDBUTTON_askbeforeoverwriting->value(0);
          ROUNDBUTTON_alwaysoverwriteolderversion->value(1);
          ROUNDBUTTON_alwayssavenewpicture->value(0);
        }
        else if (strcmp(line, "saveover=new") == 0)
        {
          ROUNDBUTTON_askbeforeoverwriting->value(0);
          ROUNDBUTTON_alwaysoverwriteolderversion->value(0);
          ROUNDBUTTON_alwayssavenewpicture->value(1);
        }
        else if (strstr(line, "savedir=") != NULL)
        {
          char * dequoted_str;

          dequoted_str = grab_and_dequote_arg(line, "savedir");
          FILEINPUT_alternatesavedir->value(dequoted_str);
          free(dequoted_str);

          if (strstr(line, "savedir=") == line)
          {
            /* Active! */

            CHECKBUTTON_usealternatesavedir->value(1);
            FILEINPUT_alternatesavedir->activate();
            BUTTON_browsealternatesavedir->activate();
          }
          else
          {
            /* Inactive! */

            CHECKBUTTON_usealternatesavedir->value(0);
            FILEINPUT_alternatesavedir->deactivate();
            BUTTON_browsealternatesavedir->deactivate();
          }
        }
        else if (strstr(line, "exportdir=") != NULL)
        {
          char * dequoted_str;

          dequoted_str = grab_and_dequote_arg(line, "exportdir");
          FILEINPUT_alternateexportdir->value(dequoted_str);
          free(dequoted_str);

          if (strstr(line, "exportdir=") == line)
          {
            /* Active! */

            CHECKBUTTON_usealternateexportdir->value(1);
            FILEINPUT_alternateexportdir->activate();
            BUTTON_browsealternateexportdir->activate();
          }
          else
          {
            /* Inactive! */

            CHECKBUTTON_usealternateexportdir->value(0);
            FILEINPUT_alternateexportdir->deactivate();
            BUTTON_browsealternateexportdir->deactivate();
          }
        }
        else if (strstr(line, "datadir=") != NULL)
        {
          char * dequoted_str;

          dequoted_str = grab_and_dequote_arg(line, "datadir");
          FILEINPUT_alternatedatadir->value(dequoted_str);
          free(dequoted_str);

          if (strstr(line, "datadir=") == line)
          {
            /* Active! */

            CHECKBUTTON_usealternatedatadir->value(1);
            FILEINPUT_alternatedatadir->activate();
            BUTTON_browsealternatedatadir->activate();
          }
          else
          {
            /* Inactive! */

            CHECKBUTTON_usealternatedatadir->value(0);
            FILEINPUT_alternatedatadir->deactivate();
            BUTTON_browsealternatedatadir->deactivate();
          }
        }
        else if (strcmp(line, "nolockfile=yes") == 0 ||
                 strcmp(line, "lockfile=no") == 0)
        {
          CHECKBUTTON_nolockfile->value(1);
        }
        else if (strstr(line, "colorfile=") != NULL)
        {
          char * dequoted_str;

          dequoted_str = grab_and_dequote_arg(line, "colorfile");
          FILEINPUT_colorfile->value(dequoted_str);
          free(dequoted_str);

          if (strstr(line, "colorfile=") == line)
          {
            /* Active! */

            CHECKBUTTON_usecolorfile->value(1);
            FILEINPUT_colorfile->activate();
            BUTTON_browsecolorfile->activate();
          }
          else
          {
            /* Inactive! */

            CHECKBUTTON_usecolorfile->value(0);
            FILEINPUT_colorfile->deactivate();
            BUTTON_browsecolorfile->deactivate();
          }
        }
        else if (strcmp(line, "nosave=yes") == 0)
          CHECKBUTTON_disablesavebutton->value(1);
        else if (strcmp(line, "startblank=yes") == 0)
          /* Start blank? */
          CHECKBUTTON_startblank->value(1);
        else if (strcmp(line, "newcolorslast=yes") == 0)
          /* "New" colors last? */
          CHECKBUTTON_newcolorslast->value(1);
        else if (strcmp(line, "autosave=yes") == 0)
          /* Auto-save on quit? */
          CHECKBUTTON_autosave->value(1);
        else if (strcmp(line, "reversesort=yes") == 0)
          /* Reverse sort of Open and Slideshow dialogs? */
          CHECKBUTTON_reversesort->value(1);
        /*Accessibility */
        else if (strcmp(line, "mouse-accessibility=yes") == 0)
          /* Mouse accessibility? */
          CHECKBUTTON_mouse_accessibility->value(1);
        else if (strcmp(line, "keyboard=yes") == 0)
          /* Keyboard mouse? */
          CHECKBUTTON_keyboard_mouse->value(1);
        else if (strcmp(line, "onscreen-keyboard=yes") == 0)
          /* Onscreen keyboard? */
        {
          CHECKBUTTON_onscreen_keyboard->value(1);
          CHOICE_onscreen_keyboard_layout->activate();
          CHECKBUTTON_onscreen_keyboard_disable_change->activate();
        }
        else if (strstr(line, "onscreen-keyboard-layout=") == line)
        {
          char *lay;
          int i;

          lay =
            strstr(line,
                   "onscreen-keyboard-layout=") +
            strlen("onscreen-keyboard-layout=");

          for (i = 0; i < NUM_LAYOUTS; i++)
          {
            if (strcmp(lay, layouts[i][LANG_CODE]) == 0)
              CHOICE_onscreen_keyboard_layout->value(i);
          }
        }

        else if (strcmp(line, "onscreen-keyboard-disable-change=yes") == 0)
          /* Onscreen keyboard? */
          CHECKBUTTON_onscreen_keyboard_disable_change->value(1);

        /* joystick */
        else if (strstr(line, "joystick-dev=") == line)
          SPINNER_joystick_dev->value(atoi
                                      (strstr(line, "joystick-dev=") +
                                       strlen("joystick-dev=")));
        else if (strstr(line, "joystick-slowness=") == line)
          SPINNER_joystick_slowness->value(atoi
                                           (strstr(line, "joystick-slowness=")
                                            + strlen("joystick-slowness=")));
        else if (strstr(line, "joystick-threshold=") == line)
          SPINNER_joystick_threshold->value(atoi
                                            (strstr
                                             (line,
                                              "joystick-threshold=") +
                                             strlen("joystick-threshold=")));
        else if (strstr(line, "joystick-maxsteps=") == line)
          SPINNER_joystick_maxsteps->value(atoi
                                           (strstr(line, "joystick-maxsteps=")
                                            + strlen("joystick-maxsteps=")));
        else if (strstr(line, "joystick-hat-slowness=") == line)
          SPINNER_joystick_hat_slowness->value(atoi
                                               (strstr
                                                (line,
                                                 "joystick-hat-slowness=") +
                                                strlen
                                                ("joystick-hat-slowness=")));
        else if (strstr(line, "joystick-hat-timeout=") == line)
          SPINNER_joystick_hat_timeout->value(atoi
                                              (strstr
                                               (line,
                                                "joystick-hat-timeout=") +
                                               strlen
                                               ("joystick-hat-timeout=")));

        else if (strstr(line, "joystick-buttons-ignore=") != NULL)
        {
          TEXTINPUT_joystick_buttons_ignore->value(strstr
                                                   (line,
                                                    "joystick-buttons-ignore=")
                                                   +
                                                   strlen
                                                   ("joystick-buttons-ignore="));
          if (strstr(line, "joystick-buttons-ignore=") == line)
          {
            /* Active! */
            CHECKBUTTON_joystick_buttons_ignore->value(1);
            TEXTINPUT_joystick_buttons_ignore->activate();
          }
          else
          {
            /* Inactive! */
            CHECKBUTTON_joystick_buttons_ignore->value(0);
            TEXTINPUT_joystick_buttons_ignore->deactivate();
          }
        }

        else if (strstr(line, "joystick-btn-brush=") == line)
          SPINNER_joystick_btn_brush->value(atoi
                                            (strstr
                                             (line,
                                              "joystick-btn-brush=") +
                                             strlen("joystick-btn-brush=")));
        else if (strstr(line, "joystick-btn-label=") == line)
          SPINNER_joystick_btn_label->value(atoi
                                            (strstr
                                             (line,
                                              "joystick-btn-label=") +
                                             strlen("joystick-btn-label=")));
        else if (strstr(line, "joystick-btn-new=") == line)
          SPINNER_joystick_btn_new->value(atoi
                                          (strstr(line, "joystick-btn-new=") +
                                           strlen("joystick-btn-new=")));
        else if (strstr(line, "joystick-btn-print=") == line)
          SPINNER_joystick_btn_print->value(atoi
                                            (strstr
                                             (line,
                                              "joystick-btn-print=") +
                                             strlen("joystick-btn-print=")));
        else if (strstr(line, "joystick-btn-shapes=") == line)
          SPINNER_joystick_btn_shapes->value(atoi
                                             (strstr
                                              (line,
                                               "joystick-btn-shapes=") +
                                              strlen
                                              ("joystick-btn-shapes=")));
        else if (strstr(line, "joystick-btn-undo=") == line)
          SPINNER_joystick_btn_undo->value(atoi
                                           (strstr(line, "joystick-btn-undo=")
                                            + strlen("joystick-btn-undo=")));
        else if (strstr(line, "joystick-btn-eraser=") == line)
          SPINNER_joystick_btn_eraser->value(atoi
                                             (strstr
                                              (line,
                                               "joystick-btn-eraser=") +
                                              strlen
                                              ("joystick-btn-eraser=")));
        else if (strstr(line, "joystick-btn-lines=") == line)
          SPINNER_joystick_btn_lines->value(atoi
                                            (strstr
                                             (line,
                                              "joystick-btn-lines=") +
                                             strlen("joystick-btn-lines=")));
        else if (strstr(line, "joystick-btn-open=") == line)
          SPINNER_joystick_btn_open->value(atoi
                                           (strstr(line, "joystick-btn-open=")
                                            + strlen("joystick-btn-open=")));
        else if (strstr(line, "joystick-btn-redo=") == line)
          SPINNER_joystick_btn_redo->value(atoi
                                           (strstr(line, "joystick-btn-redo=")
                                            + strlen("joystick-btn-redo=")));
        else if (strstr(line, "joystick-btn-stamp=") == line)
          SPINNER_joystick_btn_stamp->value(atoi
                                            (strstr
                                             (line,
                                              "joystick-btn-stamp=") +
                                             strlen("joystick-btn-stamp=")));
        else if (strstr(line, "joystick-btn-escape=") == line)
          SPINNER_joystick_btn_escape->value(atoi
                                             (strstr
                                              (line,
                                               "joystick-btn-escape=") +
                                              strlen
                                              ("joystick-btn-escape=")));
        else if (strstr(line, "joystick-btn-fill=") == line)
          SPINNER_joystick_btn_fill->value(atoi
                                           (strstr
                                            (line,
                                             "joystick-btn-fill=") +
                                            strlen("joystick-btn-fill=")));
        else if (strstr(line, "joystick-btn-magic=") == line)
          SPINNER_joystick_btn_magic->value(atoi
                                            (strstr
                                             (line,
                                              "joystick-btn-magic=") +
                                             strlen("joystick-btn-magic=")));
        else if (strstr(line, "joystick-btn-pgsetup=") == line)
          SPINNER_joystick_btn_pgsetup->value(atoi
                                              (strstr
                                               (line,
                                                "joystick-btn-pgsetup=") +
                                               strlen
                                               ("joystick-btn-pgsetup=")));
        else if (strstr(line, "joystick-btn-save=") == line)
          SPINNER_joystick_btn_save->value(atoi
                                           (strstr(line, "joystick-btn-save=")
                                            + strlen("joystick-btn-save=")));
        else if (strstr(line, "joystick-btn-text=") == line)
          SPINNER_joystick_btn_text->value(atoi
                                           (strstr(line, "joystick-btn-text=")
                                            + strlen("joystick-btn-text=")));

/*
   FIXME: Add "--keyboard" option to UI and save/load routines
*/
      }
    }
    while (!feof(fd));
  }

  set_change_status(0);
}

static void apply_settings(void)
{
  if (use_merge_discard_allusersetting->visible()
      && (use_merge_discard_allusersetting->value() ==
          SYS_SETTING_STRATEGY_USEONLY))

  {
    construct_conf_filename(CURRENT_USER);
    remove(conf_filename);
    set_change_status(0);
  }
  else
  {
    save_conf();
  }
}

Fl_Double_Window *WINDOW_tpc = (Fl_Double_Window *) 0;
Fl_Tabs *TAB_alltabs = (Fl_Tabs *) 0;

Fl_Group *TABE_about = (Fl_Group *) 0;
Fl_Group *GROUP_abouttpc = (Fl_Group *) 0;
Fl_Text_Display *TEXTOUTPUT_abouttext = (Fl_Text_Display *) 0;
Fl_Text_Buffer *TEXTBUFFER_abouttext = new Fl_Text_Buffer();

Fl_Group *TABE_videoandsound = (Fl_Group *) 0;

Fl_Group *GROUP_video = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_fullscreen = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_native = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_startblank = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_newcolorslast = (Fl_Check_Button *) 0;
Fl_Group *GROUP_resolution = (Fl_Group *) 0;
Fl_Choice *CHOICE_windowwidth = (Fl_Choice *) DEF_WINDOW_WIDTH;
Fl_Choice *CHOICE_windowheight = (Fl_Choice *) DEF_WINDOW_HEIGHT;
Fl_Check_Button *CHECKBUTTON_rotate = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_screensaver = (Fl_Check_Button *) 0;
Fl_Box * BOX_canvas_size = (Fl_Box *) 0;

Fl_Group *GROUP_sound = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_soundeffects = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_stereo = (Fl_Check_Button *) 0;

Fl_Group *GROUP_uisize = (Fl_Group *) 0;

Fl_Check_Button *CHECKBUTTON_buttonsizeauto = (Fl_Check_Button *) 0;
Fl_Slider *SLIDER_buttonsize = (Fl_Slider *) 0;
Fl_Slider *SLIDER_colorsrows = (Fl_Slider *) 0;

Fl_Group *TABE_mouseandkeyboard = (Fl_Group *) 0;
Fl_Group *GROUP_mouse = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_fancycursorshapes = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_hidecursor = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_grabmousepointer = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_mousewheelsupport = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_nobuttondistinction = (Fl_Check_Button *) 0;

Fl_Group *GROUP_keyboard = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_enablekbd = (Fl_Check_Button *) 0;

Fl_Group *TABE_simplification = (Fl_Group *) 0;
Fl_Group *GROUP_interfacesimpl = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_disableshaperotation = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_simpleshapeoutlines = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_showuppercasetextonly = (Fl_Check_Button *) 0;
Fl_Group *GROUP_controlsimpl = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_disablequitbutton = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_disablestampstool = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_disablestampcontrols = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_disablestamprotation = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_disablemagiccontrols = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_disablemagicsizes = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_ungroupmagictools = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_disableshapecontrols = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_disablelabeltool = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_disablebrushspacing = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_disabletemplateexport = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_disableerase = (Fl_Check_Button *) 0;
Fl_Choice *CHOICE_complexity = (Fl_Choice *) 0;
Fl_Slider *SLIDER_stampsize = (Fl_Slider *) 0;
Fl_Group *GROUP_stampsize = (Fl_Group *) 0;

Fl_Group *TABE_locale = (Fl_Group *) 0;
Fl_Group *GROUP_locale = (Fl_Group *) 0;
Fl_Choice *CHOICE_language = (Fl_Choice *) 0;
Fl_Check_Button *CHECKBUTTON_mirrorstamps = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_loadsystemfonts = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_alllocalefonts = (Fl_Check_Button *) 0;
Fl_Hold_Browser *BROWSER_uifont = (Fl_Hold_Browser *) 0;
Fl_Input *TEXTINPUT_uifont2 = (Fl_Input *) 0;

Fl_Group *TABE_printing = (Fl_Group *) 0;
Fl_Group *GROUP_printperms = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_allowprinting = (Fl_Check_Button *) 0;
Fl_Value_Input *VALUEINPUT_printdelay = (Fl_Value_Input *) 0;
Fl_Group *GROUP_printcfg = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_printcfg = (Fl_Check_Button *) 0;
Fl_Box *LABEL_seconds = (Fl_Box *) 0;
Fl_Group *GROUP_printcmd = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_usealtprintcmd = (Fl_Check_Button *) 0;
Fl_Group *GROUP_printdia = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_usealtprintdia = (Fl_Check_Button *) 0;

Fl_Group *GROUP_startingout = (Fl_Group *) 0;

Fl_Group *TABE_accessibility = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_mouse_accessibility = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_keyboard_mouse = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_onscreen_keyboard = (Fl_Check_Button *) 0;
Fl_Choice *CHOICE_onscreen_keyboard_layout = (Fl_Choice *) 0;
Fl_Check_Button *CHECKBUTTON_onscreen_keyboard_disable_change =
  (Fl_Check_Button *) 0;

Fl_Group *TABE_joystick = (Fl_Group *) 0;
Fl_Spinner *SPINNER_joystick_dev = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_slowness = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_threshold = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_maxsteps = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_hat_slowness = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_hat_timeout = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_brush = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_label = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_new = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_print = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_shapes = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_undo = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_eraser = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_lines = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_open = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_redo = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_stamp = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_escape = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_fill = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_magic = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_pgsetup = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_save = (Fl_Spinner *) 0;
Fl_Spinner *SPINNER_joystick_btn_text = (Fl_Spinner *) 0;

static void cb_CHECKBUTTON_joystick_buttons_ignore(Fl_Check_Button *, void *)
{
  if (CHECKBUTTON_joystick_buttons_ignore->value() == 1)
  {
    TEXTINPUT_joystick_buttons_ignore->activate();
  }
  else
  {
    TEXTINPUT_joystick_buttons_ignore->deactivate();
  };
  set_change_status(1);
}

static void cb_CHECKBUTTON_onscreen_keyboard(Fl_Check_Button *, void *)
{
  if (CHECKBUTTON_onscreen_keyboard->value() == 1)
  {
    CHOICE_onscreen_keyboard_layout->activate();
    CHECKBUTTON_onscreen_keyboard_disable_change->activate();
  }
  else
  {
    CHOICE_onscreen_keyboard_layout->deactivate();
    CHECKBUTTON_onscreen_keyboard_disable_change->deactivate();
  };
  set_change_status(1);
}

Fl_Check_Button *CHECKBUTTON_joystick_buttons_ignore = (Fl_Check_Button *) 0;
Fl_Input *TEXTINPUT_joystick_buttons_ignore = (Fl_Input *) 0;


static void cb_CHECKBUTTON_usefullscreen(Fl_Check_Button *, void *)
{
  if (CHECKBUTTON_fullscreen->value() == 1)
  {
    CHECKBUTTON_native->activate();
  }
  else
  {
    CHECKBUTTON_native->deactivate();
    CHECKBUTTON_native->value(0);
  }
  set_change_status(1);
}


static void cb_CHECKBUTTON_usealtprintcmd(Fl_Check_Button *, void *)
{
  if (CHECKBUTTON_usealtprintcmd->value() == 1)
  {
    TEXTINPUT_printcmd->activate();
  }
  else
  {
    TEXTINPUT_printcmd->deactivate();
  };
  set_change_status(1);
}

Fl_Input *TEXTINPUT_printcmd = (Fl_Input *) 0;

static void cb_CHECKBUTTON_usealtprintdia(Fl_Check_Button *, void *)
{
  if (CHECKBUTTON_usealtprintdia->value() == 1)
  {
    TEXTINPUT_printdia->activate();
  }
  else
  {
    TEXTINPUT_printdia->deactivate();
  };
  set_change_status(1);
}

Fl_Input *TEXTINPUT_printdia = (Fl_Input *) 0;

Fl_Group *TABE_saving = (Fl_Group *) 0;

Fl_Group *GROUP_saveover = (Fl_Group *) 0;
Fl_Round_Button *ROUNDBUTTON_askbeforeoverwriting = (Fl_Round_Button *) 0;
Fl_Round_Button *ROUNDBUTTON_alwaysoverwriteolderversion =
  (Fl_Round_Button *) 0;
Fl_Round_Button *ROUNDBUTTON_alwayssavenewpicture = (Fl_Round_Button *) 0;

Fl_Round_Button *ROUNDBUTTON_printdialogmod = (Fl_Round_Button *) 0;
Fl_Round_Button *ROUNDBUTTON_printdialogalways = (Fl_Round_Button *) 0;
Fl_Round_Button *ROUNDBUTTON_printdialognever = (Fl_Round_Button *) 0;

#ifdef USE_PAPERSIZES
Fl_Choice *CHOICE_papersize = (Fl_Choice *) 0;
#endif

Fl_Group *GROUP_savedir = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_usealternatesavedir = (Fl_Check_Button *) 0;
Fl_Check_Button *CHECKBUTTON_usealternateexportdir = (Fl_Check_Button *) 0;

Fl_Group *TABE_data = (Fl_Group *) 0;

Fl_Group *GROUP_lockfile = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_nolockfile = (Fl_Check_Button *) 0;

Fl_Group *GROUP_datadir = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_usealternatedatadir = (Fl_Check_Button *) 0;

Fl_Group *GROUP_colorfile = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_usecolorfile = (Fl_Check_Button *) 0;

Fl_Group *GROUP_nosave = (Fl_Group *) 0;
Fl_Check_Button *CHECKBUTTON_disablesavebutton = (Fl_Check_Button *) 0;

Fl_Check_Button *CHECKBUTTON_autosave = (Fl_Check_Button *) 0;

Fl_Check_Button *CHECKBUTTON_reversesort = (Fl_Check_Button *) 0;

static void cb_CHECKBUTTON_usealternatesavedir(Fl_Check_Button *, void *)
{
  if (CHECKBUTTON_usealternatesavedir->value() == 1)
  {
    FILEINPUT_alternatesavedir->activate();
    BUTTON_browsealternatesavedir->activate();
  }
  else
  {
    FILEINPUT_alternatesavedir->deactivate();
    BUTTON_browsealternatesavedir->deactivate();
  }
  set_change_status(1);
}

static void cb_CHECKBUTTON_usealternateexportdir(Fl_Check_Button *, void *)
{
  if (CHECKBUTTON_usealternateexportdir->value() == 1)
  {
    FILEINPUT_alternateexportdir->activate();
    BUTTON_browsealternateexportdir->activate();
  }
  else
  {
    FILEINPUT_alternateexportdir->deactivate();
    BUTTON_browsealternateexportdir->deactivate();
  }
  set_change_status(1);
}

Fl_File_Input *FILEINPUT_alternatesavedir = (Fl_File_Input *) 0;

static void cb_BUTTON_browsealternatesavedir(Fl_Button *, void *)
{
  const char *dirname = "\n";

  dirname =
    fl_dir_chooser( /* Window title for the browser window used when selecting an alternative location to save drawings */ gettext("Save Directory?"), "");
  fl_file_chooser_callback(0);

  if (dirname != NULL && strlen(dirname) > 0)
  {
    FILEINPUT_alternatesavedir->value(dirname);
    WINDOW_tpc->show();
    set_change_status(1);
  }
}

Fl_Button *BUTTON_browsealternatesavedir = (Fl_Button *) 0;

Fl_File_Input *FILEINPUT_alternateexportdir = (Fl_File_Input *) 0;

static void cb_BUTTON_browsealternateexportdir(Fl_Button *, void *)
{
  const char *dirname = "\n";

  dirname =
    fl_dir_chooser( /* Window title for the browser window used when selecting an alternative location to export drawings & slideshow GIFs */ gettext("Export Directory?"), "");
  fl_file_chooser_callback(0);

  if (dirname != NULL && strlen(dirname) > 0)
  {
    FILEINPUT_alternateexportdir->value(dirname);
    WINDOW_tpc->show();
    set_change_status(1);
  }
}

Fl_Button *BUTTON_browsealternateexportdir = (Fl_Button *) 0;


static void cb_CHECKBUTTON_usealternatedatadir(Fl_Check_Button *, void *)
{
  if (CHECKBUTTON_usealternatedatadir->value() == 1)
  {
    FILEINPUT_alternatedatadir->activate();
    BUTTON_browsealternatedatadir->activate();
  }
  else
  {
    FILEINPUT_alternatedatadir->deactivate();
    BUTTON_browsealternatedatadir->deactivate();
  }
  set_change_status(1);
}

Fl_File_Input *FILEINPUT_alternatedatadir = (Fl_File_Input *) 0;

static void cb_BUTTON_browsealternatedatadir(Fl_Button *, void *)
{
  const char *dirname = "\n";

  dirname =
    fl_dir_chooser( /* Window title for the browser window used when selecting an alternative location to find data files */ gettext("Data Directory?"), "");
  fl_file_chooser_callback(0);

  if (dirname != NULL && strlen(dirname) > 0)
  {
    FILEINPUT_alternatedatadir->value(dirname);
    WINDOW_tpc->show();
    set_change_status(1);
  }
}

Fl_Button *BUTTON_browsealternatedatadir = (Fl_Button *) 0;

static void cb_CHECKBUTTON_usecolorfile(Fl_Check_Button *, void *)
{
  if (CHECKBUTTON_usecolorfile->value() == 1)
  {
    FILEINPUT_colorfile->activate();
    BUTTON_browsecolorfile->activate();
  }
  else
  {
    FILEINPUT_colorfile->deactivate();
    BUTTON_browsecolorfile->deactivate();
  }
  set_change_status(1);
}

Fl_File_Input *FILEINPUT_colorfile = (Fl_File_Input *) 0;

static void cb_BUTTON_browsecolorfile(Fl_Button *, void *)
{
  const char *filename = "\n";

  filename =
    fl_file_chooser( /* Window title for the browser window used when selecting an alternative color palette file */ gettext("Color Palette File?"), "*.txt", "", 0);
  fl_file_chooser_callback(0);

  if (filename != NULL && strlen(filename) > 0)
  {
    FILEINPUT_colorfile->value(filename);
    WINDOW_tpc->show();
    set_change_status(1);
  }
}

Fl_Button *BUTTON_browsecolorfile = (Fl_Button *) 0;

Fl_Button *BUTTON_apply = (Fl_Button *) 0;

void cb_BUTTON_apply(Fl_Button *, void *)
{
  apply_settings();
}

static void cb_BUTTON_reset(Fl_Button *, void *)
{
  printf("Loading conf due to RESET\n");
  load_conf();
}

Fl_Button *BUTTON_reset = (Fl_Button *) 0;

Fl_Button *BUTTON_defaults = (Fl_Button *) 0;
Fl_Button *BUTTON_quit = (Fl_Button *) 0;

static void cb_BUTTON_defaults(Fl_Button *, void *)
{
  CB_SetActualTabToDefaults();
  WINDOW_tpc->show();
}

static void cb_status_chng(Fl_Widget *, void *)
{
  set_change_status(1);
}

static void cb_canvas_size(void *, void *)
{
  update_canvas_size();
  cb_status_chng(NULL, NULL);
}

static void cb_uifont_browser_chng(Fl_Hold_Browser * o, void *)
{
  if (o->value() == UIFONT_MANUAL)
  {
    TEXTINPUT_uifont2->activate();
  }
  else
  {
    TEXTINPUT_uifont2->deactivate();
  }
  cb_status_chng(NULL, NULL);
}

static void cb_uifont2_input_chng(Fl_Input * o, void *)
{
  int i, found, exact;

  found = -1;
  exact = 0;
  if (strlen(TEXTINPUT_uifont2->value()) > 0)
  {
    for (i = UIFONT_FIRST_REAL_ENTRY; i <= BROWSER_uifont->size(); i++) // N.B. array is 1-based!
    {
      if (strcasestr(BROWSER_uifont->text(i), TEXTINPUT_uifont2->value()) == BROWSER_uifont->text(i))
      {
        exact = (strcasecmp(BROWSER_uifont->text(i), TEXTINPUT_uifont2->value()) == 0);
        if (found == -1 || exact)
          found = i;
      }
    }
  }

  if (found != -1)
  {
    /* What they typed matched a font in the list, scroll browser to that font */
    BROWSER_uifont->middleline(found);

    /* FIXME: Would be nice to handle FL_UNFOCUS events
       and in that case, do `BROWSER_uifont->value(found)`;
       and `o->deactivate()`.  Doing it while they're still
       typing would be undesirable; imagine typing "SomeFont"
       but wanting "SomeFont Mono" -bjk 2023.06.13 */
  }
}

void confirm_exit_cb(Fl_Widget *, void *)
{
  if (status_chng)
  {
    if (fl_choice(gettext("Quit without applying changes?"),
                  gettext("&No"), gettext("&Yes"), NULL))
    {
      exit(0);
    }
  }
  else
  {
    exit(0);
  }
}

Fl_Choice *CHOICE_settings = (Fl_Choice *) 0;
Fl_Choice *use_merge_discard_allusersetting = (Fl_Choice *) 0;

static void cb_use_merge_discard_allusersetting(Fl_Choice *, void *)
{
  if (use_merge_discard_allusersetting->value() == SYS_SETTING_STRATEGY_MERGE)
  {
    activate_widgets();
  }
  else if (use_merge_discard_allusersetting->value() ==
           SYS_SETTING_STRATEGY_DISCARD)
  {
    activate_widgets();
  }
  else
  {
    deactivate_widgets();
  }
  set_change_status(1);
}

static void cb_CHOICE_settings(Fl_Choice *, void *)
{
  if (status_chng)
  {
    const char *user = 0;

    CHOICE_settings->value(!(CHOICE_settings->value()));

    switch (CHOICE_settings->value())
    {
    case (CURRENT_USER):
      user =
        /* "Apply changes you made to ..." -- saving settings for current user (not all users) */
        gettext("current user");
      break;
    case (ALL_USERS):
      user =
        /* "Apply changes you made to ..." -- saving settings for all users (not current user) */
        gettext("all users");
      break;
    }

    if (fl_choice(gettext("Apply changes you made to %s?"),
                  gettext("No"), gettext("Yes"), NULL, user))
    {
      apply_settings();
    }

    CHOICE_settings->value(!(CHOICE_settings->value()));
  }

  printf("Loading conf due to cb_CHOICE_settings\n");
  load_conf();
}

Fl_Double_Window *Show_TPC_Window()
{
  int boxx, boxy, xx, yy;
  int width, height;
  int work_x, work_y, work_w, work_h;
  float dpi_w, dpi_h;
  int ww, hh, sz;
  char *str;
  int vspace, vspace_s;
  const char *tpc_size = getenv("TPC_SIZE");

  Fl::screen_dpi(dpi_w, dpi_h, 0);
  Fl::screen_work_area(work_x, work_y, work_w, work_h);

  if (((work_w <= 1280 && work_h <= 900)
       || ((tpc_size != NULL) && !strcmp(tpc_size, "small"))) || ((dpi_w < 120
                                                                   && dpi_h <
                                                                   120)
                                                                  &&
                                                                  !((tpc_size
                                                                     != NULL)
                                                                    &&
                                                                    !strcmp
                                                                    (tpc_size,
                                                                     "large"))))
  {
    /* Smaller window with smaller fonts (more like pre-0.0.18) */
    width = 800;
    height = 480;
    tab_font_size = 14;
    big_font_size = 14;
    small_font_size = 12;
    smaller_font_size = 10;     // N.B. for some locales (e.g. Japanese), can't get a font as small as this
    padding = 2;
    border = 2;
    vspace = 8;
    vspace_s = 2;
    hspace = 12;
  }
  else
  {
    /* Larger window with larger fonts (since 0.0.18; for high-res, high-DPI displays) */
    width = 1200;
    height = 720;
    tab_font_size = 20;
    big_font_size = 22;
    small_font_size = 18;
    smaller_font_size = 14;
    padding = 4;
    border = 6;
    vspace = 12;
    vspace_s = 3;
    hspace = 16;
  }
  fprintf(stderr, "Opening %d x %d with LC_ALL=%s\n", width, height,
          tpc_locale == NULL ? "<null>" : tpc_locale);

#ifdef WIN32
  smaller_font_size = small_font_size;
#define FL_ITALIC FL_HELVETICA
#endif

  Fl_Double_Window *w;
  {
    Fl_Double_Window *o;

    o = WINDOW_tpc =
      new Fl_Double_Window(width, height,
                           gettext("Tux Paint Config v" VER_VERSION));
    w = o;
    o->callback((Fl_Callback *) confirm_exit_cb);

    tab_label_height = tab_font_size * 2;
    tab_top_y = tab_label_height + padding;
    tab_height =
      height - tab_label_height - (tab_font_size + small_font_size * 2) -
      (padding * 4);
    tab_width = width - (padding * 2);

    box_top_y = tab_top_y + border;
    full_box_height = tab_height - border * 2;
    full_box_width = tab_width - border * 2;

    half_box_height = (full_box_height - padding) / 2;
    half_box_width = (full_box_width - padding) / 2;
    half_box_l_x = padding + border - 1;
    half_box_r_x = padding + tab_width - border - half_box_width;

    {
      Fl_Tabs *o = TAB_alltabs =
        new Fl_Tabs(padding - 1, tab_top_y - tab_label_height, tab_width,
                    tab_height);
      o->box(FL_PLASTIC_UP_BOX);

      /* TAB: ABOUT */
      {
        Fl_Group *o = TABE_about =
          new Fl_Group(padding, tab_top_y, tab_width, tab_height,
                       /* "About" tab */ gettext("About"));
        o->labelsize(tab_font_size);
        o->labelfont(FL_HELVETICA);
        o->box(FL_PLASTIC_UP_BOX);
        o->color((Fl_Color) 132);
        o->selection_color((Fl_Color) 92);
        o->align(37);
        boxx = half_box_l_x;
        boxy = box_top_y;
        {
          Fl_Group *o = GROUP_abouttpc =
            new Fl_Group(boxx, boxy, full_box_width, full_box_height,
                         gettext("About Tux Paint Config."));
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);

          ww = full_box_width;
          o->measure_label(ww, hh);
          yy = boxy + hh;
          yy += vspace * 2;
          TEXTOUTPUT_abouttext =
            new Fl_Text_Display(boxx + hspace * 2, yy,
                                full_box_width - hspace * 4,
                                full_box_height - hh - vspace * 4);
          /* FIXME: Need to provide a style to get a larger font size */
          TEXTOUTPUT_abouttext->buffer(TEXTBUFFER_abouttext);

          TEXTBUFFER_abouttext->text(gettext(about_text));
          TEXTOUTPUT_abouttext->textsize(big_font_size);
          TEXTOUTPUT_abouttext->wrap_mode(1, 0);

          o->end();
        }
        o->end();
      }

      /* TAB: VIDEO / SOUND */
      yy = tab_top_y;
      {
        Fl_Group *o = TABE_videoandsound =
          new Fl_Group(padding, yy, tab_width, tab_height,
                       /* "Video/Sound tab" */ gettext("Video/Sound"));
        o->labelsize(tab_font_size);
        o->labelfont(FL_HELVETICA);
        o->box(FL_PLASTIC_UP_BOX);
        o->color((Fl_Color) 1);
        o->selection_color((Fl_Color) 1);
        o->align(37);
        yy = box_top_y;

        /* Section: VIDEO */
        boxx = half_box_l_x;
        {
          Fl_Group *o = GROUP_video =
            new Fl_Group(boxx, yy, half_box_width, full_box_height,
                         /* "Video" settings section */ gettext("Video:"));
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          ww = half_box_width;
          o->measure_label(ww, hh);
          yy += hh;

          str =
            /* Checkbox label: Enable/disable fullscreen (vs windowed) */
            gettext("&Fullscreen");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_ITALIC, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_fullscreen =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_ITALIC);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80066);
            o->callback((Fl_Callback *) cb_CHECKBUTTON_usefullscreen);
            yy += hh;
          }

          str =
            gettext
            ("Run Tux Paint in fullscreen mode, rather than in a window.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, smaller_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(smaller_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            hh = o->h();
            yy += hh;
          }

          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_native =
              new Fl_Check_Button(boxx + hspace * 2, yy,
                                  half_box_width - hspace * 3,
                                  big_font_size,
                                  /* Checkbox label: Use native screen/monitor resolution when in fullscreen mode (vs. specific width/height settings) */
                                  gettext("&Native"));
            o->labelfont(FL_HELVETICA);
            o->labelsize(big_font_size);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_canvas_size);
            ww = half_box_width - hspace * 3;
            o->measure_label(ww, hh);
            yy += hh;
          }

          str = gettext("Use native screen resolution in fullscreen mode.");
          ww = half_box_width - hspace * 3;
          hh = 0;
          fl_font(FL_ITALIC, smaller_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace * 2, yy, half_box_width - hspace * 3,
                         hh, str);
            o->labelfont(FL_ITALIC);
            o->labelsize(smaller_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            hh = o->h();
            yy += hh;
          }

          str = gettext("Window size / Full-screen resolution:");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += hh;             /* Push down because label is FL_ALIGN_TOP_LEFT */
          yy += vspace;
          {
            Fl_Choice *o = CHOICE_windowwidth =
              new Fl_Choice(boxx + hspace, yy,
                            (half_box_width - hspace * 2.5) / 2,
                            small_font_size * 2, str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->textsize(small_font_size);
            o->down_box(FL_BORDER_BOX);
            o->align(FL_ALIGN_TOP_LEFT);

            int i;
            for (i = 0; i < NUM_WINDOW_WIDTHS; i++)
            {
              o->add(window_widths[i], "", NULL, NULL, 0);
            }

            o->callback((Fl_Callback *) cb_canvas_size);
          }

          {
            Fl_Choice *o = CHOICE_windowheight =
              new Fl_Choice(boxx + (half_box_width + hspace * 0.5) / 2,
                            yy, (half_box_width - hspace * 2.5) / 2,
                            small_font_size * 2,
                            "");
            o->labelsize(big_font_size);
            o->textsize(small_font_size);
            o->down_box(FL_BORDER_BOX);
            o->align(FL_ALIGN_TOP_LEFT);

            int i;
            for (i = 0; i < NUM_WINDOW_HEIGHTS; i++)
            {
              o->add(window_heights[i], "", NULL, NULL, 0);
            }

            o->callback((Fl_Callback *) cb_canvas_size);

            hh = o->h();
            yy += hh;
          }

          str =
            gettext
            ("Size of the window, or the resolution in fullscreen (in pixels).");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, smaller_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(smaller_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            o->measure_label(ww, hh);
            yy += hh;
          }

          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_rotate =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2,
                                  big_font_size,
                                  /* Checkbox label: Rotate screen orientation (90 degrees/sideways) */
                                  gettext("&Rotate Orientation"));
            o->labelfont(FL_HELVETICA);
            o->labelsize(big_font_size);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_canvas_size);

            ww = half_box_width - hspace * 2 - big_font_size;
            o->measure_label(ww, hh);
            yy += hh;
          }

          str =
            gettext
            ("Swap width and height, to rotate 90 degrees (useful for portrait-display on a tablet PC)");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, smaller_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(smaller_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            o->measure_label(ww, hh);
            yy += hh;
          }
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_screensaver =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2,
                                  big_font_size,
                                  /* Checkbox label: Allow screensaver to take over display while Tux Paint is running */
                                  gettext("Allow S&creensaver"));
            o->labelfont(FL_HELVETICA);
            o->labelsize(big_font_size);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_status_chng);
            ww = half_box_width - hspace * 2 - big_font_size;
            o->measure_label(ww, hh);
            yy += hh;
          }

          str =
            gettext
            ("Don't disable your system's screensaver while Tux Paint is running.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, smaller_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(smaller_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
          }
          o->end();


          /* Canvas size calculation */
          yy += vspace * 4;
          {
            char * tmp_str;

            tmp_str = gettext("Canvas size unknown");

            ww = half_box_width - hspace * 2;
            hh = 0;
            fl_font(FL_ITALIC, big_font_size);
            measure_and_softwrap(tmp_str, ww, hh);

            {
              Fl_Box *o = BOX_canvas_size =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           tmp_str);
              o->labelfont(FL_ITALIC);
              o->labelsize(big_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
            }
            o->end();
          }
        }

        /* Section: SOUND */
        boxx = half_box_r_x;
        yy = box_top_y;
	current_box_height = full_box_height * 0.45;
        {
          Fl_Group *o = GROUP_sound =
            new Fl_Group(boxx, yy, half_box_width, current_box_height,
                         /* Sound settings section */ gettext("Sound:"));
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          ww = half_box_width;
          o->measure_label(ww, hh);
          yy += hh;

          str =
            /* Checkbox label: Enable sound effects */
            gettext("Enable &Sound Effects");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_soundeffects =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80073);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str = gettext("Enable/disable sound effects.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            o->measure_label(ww, hh);
            yy += hh;
          }
          str =
            /* Checkbox label: Enable stereo (vs. mono) sound */
            gettext("Enable S&tereo Sound");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_stereo =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80074);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext("Stereophonic ('stereo') or monaural ('mono') sound.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
          }
          o->end();
        }

        /* Section: INTERFACE SIZE: */
	current_box_height = full_box_height - current_box_height - padding;
	yy = box_top_y + full_box_height - current_box_height;
        {
          Fl_Group *o = GROUP_uisize =
            new Fl_Group(boxx, yy, half_box_width, current_box_height,
                         /* Interface size settings section: */
                         gettext("Interface Size:"));
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelfont(FL_HELVETICA);
          o->labelsize(big_font_size);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          ww = half_box_width;
          o->measure_label(ww, hh);
          yy += hh;
          yy += vspace * 2;
          {
            Fl_Slider *o = SLIDER_buttonsize =
              new Fl_Slider(boxx + half_box_width * 0.05, yy,
                            half_box_width * 0.9, big_font_size, "");
            o->labelsize(small_font_size);
            o->labelfont(FL_HELVETICA);
            o->type(FL_HORIZONTAL);
            o->range(24, 192);
            o->value(48);
            o->label(BTN_SIZE_DEFAULT);
            o->slider_size(0.1);
            o->callback((Fl_Callback *) cb_SLIDER_buttonsize);
            o->align(FL_ALIGN_BOTTOM | FL_ALIGN_WRAP);
            ww = half_box_width;
            o->measure_label(ww, hh);
            yy += hh + o->h();
          }

          str =
            /* Checkbox label: Automatic button size */
            gettext("Automatic button size");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_buttonsizeauto =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80074);
            o->callback((Fl_Callback *) cb_SLIDER_buttonsize);
            yy += hh;
          }

          str =
            gettext
            ("Size of buttons in Tux Paint (in pixels). If chosen size is too large for the chosen window/screen resolution, the largest usable size will be used. Choose 'automatic button size' to scale buttons based window/screen resolution.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, smaller_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(smaller_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            o->measure_label(ww, hh);
            yy += hh;
          }


          yy += vspace;
          {
            Fl_Slider *o = SLIDER_colorsrows =
              new Fl_Slider(boxx + half_box_width * 0.05, yy,
                            half_box_width * 0.9, big_font_size, "");
            o->labelsize(small_font_size);
            o->labelfont(FL_HELVETICA);
            o->type(FL_HORIZONTAL);
            o->range(1, 3);
            o->value(1);
            o->label(COLOR_SIZE_DEFAULT);
            o->slider_size(0.30);
            o->callback((Fl_Callback *) cb_SLIDER_colorsrows);
          }
          o->end();
        }
        o->end();
      }

      /* TAB: MOUSE / KEYBOARD */
      {
        Fl_Group *o = TABE_mouseandkeyboard =
          new Fl_Group(padding, tab_top_y, tab_width, tab_height,
                       /* "Mouse/Keyboard" tab */ gettext("Mouse/Keyboard"));
        o->labelsize(tab_font_size);
        o->labelfont(FL_HELVETICA);
        o->box(FL_PLASTIC_UP_BOX);
        o->color((Fl_Color) 2);
        o->selection_color((Fl_Color) 2);
        o->align(37);
        o->hide();
        boxx = half_box_l_x;
        yy = box_top_y;

        /* Section: CURSOR: */
        {
          Fl_Group *o = GROUP_mouse =
            new Fl_Group(boxx, yy, half_box_width, half_box_height,
                         /* Cursor (mouse pointer) settings section */
                         gettext("Cursor:"));
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          ww = half_box_width;
          o->measure_label(ww, hh);
          yy += hh;

          str =
            /* Checkbox label: Use Tux Paint's "fancy" cursor (mouse pointer) shapes, rather than system's mouse pointer shape */
            gettext("&Fancy Cursor Shapes");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_fancycursorshapes =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80066);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Change the shape of the mouse pointer depending on where the mouse is and what you are doing.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, smaller_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(smaller_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            o->measure_label(ww, hh);
            yy += hh;
          }
          str =
            /* Checkbox label: Hide cursor (mouse pointer) */
            gettext("&Hide Cursor");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_hidecursor =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80066);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Completely hides cursor (useful on touchscreen devices).");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, 45,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            o->measure_label(ww, hh);
            yy += hh;
          }
          o->end();
        }

        /* Section: KEYBOARD: */
        yy = box_top_y + full_box_height - half_box_height;
        {
          Fl_Group *o = GROUP_keyboard =
            new Fl_Group(boxx, yy, half_box_width, half_box_height,
                         /* Keyboard settings section */
                         gettext("Keyboard:"));
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          ww = half_box_width;
          o->measure_label(ww, hh);
          yy += hh;

          str =
            /* Checkbox label: Enable keyboard shortcuts */
            gettext("Enable &Keyboard Shortcuts");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_enablekbd =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x8006b);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Allows key combinations to be used as shortcuts for certain commands. (e.g., Control+S to Save)");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, smaller_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + 12, yy, half_box_width - hspace * 2, hh, str);
            o->labelfont(FL_ITALIC);
            o->labelsize(smaller_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
          }
          o->end();
        }

        /* Section: MOUSE: */
        boxx = half_box_r_x;
        yy = box_top_y;
        {
          Fl_Group *o = GROUP_keyboard =
            new Fl_Group(boxx, boxy, half_box_width, full_box_height,
                         /* Mouse settings section */ gettext("Mouse:"));
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          ww = half_box_width;
          o->measure_label(ww, hh);
          yy += hh;

          str =
            /* Checkbox label: Grab mouse pointer (trap it within Tux Paint's window) */
            gettext("&Grab Mouse Pointer");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_grabmousepointer =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh, str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80067);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Prevents the mouse pointer from leaving the Tux Paint window.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }


          str =
            /* Checkbox label: Allow mouse wheel for scrolling */
            gettext("Mouse &Wheel Support");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_mousewheelsupport =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80077);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Allows a mouse wheel to be used to scroll through items. (e.g., brushes, stamps)");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }

          str =
            /* Checkbox label: Allow all mouse buttons to cause actions (not just primary aka "left mouse button") */
            gettext("No &Button Distinction");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_nobuttondistinction =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Allows middle and right mouse buttons to be used for clicking, too.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }


          /* FIXME: Add 'mouse-accessibilty' (click/release/move/click/release instead of click/drag/release) */
          /* FIXME: Add keymouse option */
          o->end();
        }
        o->end();
      }

      /* TAB: SIMPLIFCIATION */
      {
        Fl_Group *o = TABE_simplification =
          new Fl_Group(padding, tab_top_y, tab_width, tab_height,
                       /* "Simplification" tab */ gettext("Simplification"));
        o->labelsize(tab_font_size);
        o->labelfont(FL_HELVETICA);
        o->box(FL_PLASTIC_UP_BOX);
        o->color((Fl_Color) 3);
        o->selection_color((Fl_Color) 3);
        o->align(37);
        o->hide();

        /* Section: INITIAL STAMP SIZE: [at the bottom] */
        boxx = half_box_l_x;
        str =
          /* Slider label: Initial size given to stamps upon Tux Paint's launch */
          gettext("Initial Stamp Size:");
        ww = half_box_width - hspace;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        sz = hh;
        sz += vspace * 3;
        sz += big_font_size;    /* slider */

        str = STAMP_SIZE_DEFAULT;
        ww = half_box_width - hspace * 2;
        hh = 0;
        fl_font(FL_ITALIC, small_font_size);
        measure_and_softwrap(str, ww, hh);
        sz += hh;
        sz += vspace;
        yy = box_top_y + full_box_height - sz;

        str = gettext("Initial Stamp Size:");   /* FIXME: Repeats the string above */
        {
          Fl_Group *o = GROUP_stampsize =
            new Fl_Group(boxx, yy, half_box_width, sz, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          ww = half_box_width;
          o->measure_label(ww, hh);
          yy += hh;

          str = STAMP_SIZE_DEFAULT;
          yy += vspace * 2;
          {
            Fl_Slider *o = SLIDER_stampsize =
              new Fl_Slider(boxx + half_box_width * 0.05, yy,
                            half_box_width * 0.9, big_font_size, "");
            o->type(FL_HORIZONTAL);
            o->range(-1, 10);
            o->value(-1);
            o->slider_size(0.1);
            o->label(str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->callback((Fl_Callback *) cb_SLIDER_stampsize);
          }
          o->end();
        }

        /* Section: INTERFACE SIMPLIFICATION: */
        boxx = half_box_l_x;
        yy = box_top_y;

        str =
          /* Interface simplification settings section */
          gettext("Interface Simplification:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        {
          Fl_Group *o = GROUP_interfacesimpl =
            new Fl_Group(boxx, yy, half_box_width,
                         full_box_height - sz - padding, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          str =
            /* Checkbox label: Disable the rotation step when using Shapes tool */
            gettext("Disable Shape &Rotation");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_disableshaperotation =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80072);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Shape tool's rotation step is disabled. Useful for younger children.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            o->measure_label(ww, hh);
            yy += hh;
          }


          str =
            /* Checkbox label: Simplified (rectangular, vs. silhouette) outlines for stamps */
            gettext("Simple Stamp &Outlines");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_simpleshapeoutlines =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x8006f);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Draw a rectangle around the cursor when placing stamps, rather than a detailed outline. (For slow computers and thin clients.)");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            o->measure_label(ww, hh);
            yy += hh;
          }


          str =
            /* Checkbox label: UI text only in uppercase */
            gettext("Show &Uppercase Text Only");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_showuppercasetextonly =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80075);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Cause all text in Tux Paint (button labels, dialogs, etc.) to appear in UPPERCASE rather than Mixed Case.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            o->measure_label(ww, hh);
            yy += hh;
          }
          o->end();
        }


        /* Section: CONTROL SIMPLIFICATION: */
        boxx = half_box_r_x;
        yy = box_top_y;

        str =
          /* Control Simplification settings section */
          gettext("Control Simplification:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        {
          Fl_Scroll *o =
            new Fl_Scroll(boxx, yy, half_box_width, full_box_height, "");
          o->type(Fl_Scroll::VERTICAL_ALWAYS);
          {
            Fl_Group *o = GROUP_controlsimpl =
              new Fl_Group(boxx, yy, half_box_width,
                           full_box_height * 2.5 /* FIXME */ , str);
            o->box(FL_PLASTIC_UP_BOX);
            o->labeltype(FL_EMBOSSED_LABEL);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
            yy += hh;

            str =
              /* Checkbox label: Disable standard methods of quitting Tux Paint */
              gettext("Disable \'&Quit\' Button and [Escape] key");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace * 2;
            {
              Fl_Check_Button *o = CHECKBUTTON_disablequitbutton =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80071);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Clicking the window\'s close (X) button in the title bar, or pressing [Alt]+[F4] or [Shift]+[Ctrl]+[Escape] will still quit Tux Paint.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }


            str =
              /* Checkbox label: Disable the "Stamps" tool */
              gettext("Disable \'&Stamps\' Tool");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_disablestampstool =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80073);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Do not load stamps at startup, thus disabling the Stamps tool.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }


            str =
              /* Checkbox label: Disable stamp controls (shrink, mirror, etc.) */
              gettext("Disable Stamp &Controls");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_disablestampcontrols =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80063);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Simplify the \'Stamps\' tool by removing the stamp control buttons (size, mirror and flip)");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }


            str =
              /* Checkbox label: Disable stamp rotation */
              gettext("Disable Stamp Rotation");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_disablestamprotation =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80063);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Stamps tool's rotation step is disabled. Useful for younger children.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }


            str =
              /* Checkbox label: Disable magic tool controls (paint vs fullscreen) */
              gettext("Disable &Magic Controls");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_disablemagiccontrols =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80063);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Simplify the \'Magic\' tools by removing the buttons to switch between freehand and entire-picture modes.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }

            str =
              /* Checkbox label: Disable magic size controls */
              gettext("Disable &Magic Sizes");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_disablemagicsizes =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80063);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Simplify the \'Magic\' tools by removing the size settings.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }


            str =
              /* Checkbox label: Ungroup magic tools */
              gettext("Ungroup Magic &Tools");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_ungroupmagictools =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80063);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Ungroup \'Magic\' tools; place them in one large list.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }



            str =
              /* Checkbox label: Disable shape controls (center- vs corner-based expansion) */
              gettext("Disable Shape Controls");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_disableshapecontrols =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Simplify the \'Shapes\' tool by removing the buttons to switch between center- and corner-based drawing.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }


            str =
              /* Checkbox label: Disable "Label" tool (leaving only "Text" tool) */
              gettext("Disable '&Label' Tool");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_disablelabeltool =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80063);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Disable the \'Label\' text-entry tool (leaving only the \'Text\' tool).");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }

            str =
              /* Checkbox label: Disable "Label" tool */
              gettext("Disable Brush spacing");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_disablebrushspacing =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80063);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Disable the \'Paint\' and \'Lines\' tools\' brush spacing option.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }

            str =
              /* Checkbox label: Disable "Make Template" feature */
              gettext("Disable 'Make Template'");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_disabletemplateexport =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80063);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Disable the option to create new templates from the \'Open\' dialog.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }

            str =
              /* Checkbox label: Disable "Erase" feature */
              gettext("Disable Erase (file deletion)");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_disableerase =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80063);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Disable the \'Erase\' buttons on the \'Open\' dialog (for deleting saved drawings) and \'New\' dialog (for deleting exported templates).");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }

            str = /* Complexity selector's label */ gettext("Complexity:");
            ww = half_box_width - hspace * 2;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);

            yy += hh;             /* Push down because label is FL_ALIGN_TOP_LEFT */
            yy += vspace * 2;
            {
              Fl_Choice *o = CHOICE_complexity =
                new Fl_Choice(boxx + hspace, yy, half_box_width - hspace * 3,
                              small_font_size * 2, str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->textsize(small_font_size);
              o->down_box(FL_BORDER_BOX);
              o->align(FL_ALIGN_TOP_LEFT);
              yy += small_font_size * 2;

              for (int i = 0; i < NUM_COMPLEXITIES; i++)
              {
                o->add(gettext(complexities_names[i]), "", NULL, NULL, 0);
              }

              o->callback((Fl_Callback *) cb_status_chng);
            }

            str =
              gettext
              ("User expertise level, which affects the availability or complexity of some Magic tools.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }

            o->end();
          }
          o->end();
        }
        o->end();
      }

      /* TAB: LANGUAGES */

      /* Section: LANGUAGE */
      {
        Fl_Group *o = TABE_locale =
          new Fl_Group(padding, tab_top_y, tab_width, tab_height,
                       /* "Languages" tab */ gettext("Languages"));
        o->labelsize(tab_font_size);
        o->labelfont(FL_HELVETICA);
        o->box(FL_PLASTIC_UP_BOX);
        o->color((Fl_Color) 4);
        o->selection_color((Fl_Color) 4);
        o->align(37);
        o->hide();

        boxx = half_box_l_x;
        yy = box_top_y;

        str = /* Language selector's label */ gettext("Language:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        {
          Fl_Group *o = GROUP_locale =
            new Fl_Group(boxx, yy, half_box_width, full_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          /* N.B.: Same label as above */
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);

          yy += hh;             /* Push down because label is FL_ALIGN_TOP_LEFT */
          yy += vspace * 2;
          {
            Fl_Choice *o = CHOICE_language =
              new Fl_Choice(boxx + hspace, yy, half_box_width - hspace * 2,
                            small_font_size * 2, str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->textsize(small_font_size);
            o->down_box(FL_BORDER_BOX);
            o->align(FL_ALIGN_TOP_LEFT);
            yy += small_font_size * 2;

            /* Add zeroeth item manually, since we need it to be translated.

               Note: We explicitly send "lang=default", in case we're
               writing to a local config file, and want to _override_
               the system-wide ('all users') config file's setting.
             */
            o->add(gettext("(Use system's setting)"), "default", NULL, NULL,
                   0);
            int i;

            /* Add the rest */
            for (i = 1; i < NUM_LANGS; i++)
            {
              o->add(langs[i][LANG_NAME], "", NULL, NULL, 0);
            }

            o->callback((Fl_Callback *) cb_status_chng);
          }

          str =
            gettext
            ("Run Tux Paint in a particular language (overriding system's settings).");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }


          str =
            /* Checkbox label: Mirror stamps (use the mirror-image versions by default) */
            gettext("&Mirror Stamps");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_mirrorstamps =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x8006d);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Automatically mirror-image all mirror-able stamps. Useful for users who prefer things appearing right-to-left.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
          }
          o->end();
        }

        /* Section: FONTS: */
        yy = box_top_y;
        boxx = half_box_r_x;

        str = /* Font settings section */ gettext("Fonts:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        {
          Fl_Group *o = GROUP_locale =
            new Fl_Group(boxx, yy, half_box_width, full_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          str =
            /* Checkbox label: Load system-wide fonts */
            gettext("Load System &Fonts");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_loadsystemfonts =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->value(0);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Attempt to load more fonts, found elsewhere on your computer. (Note: may cause instability!)");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }


          str =
            /* Checkbox label: Load all locale-specific fonts (not just the current locale's) */
            gettext("Load All &Locale Fonts");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_alllocalefonts =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->value(0);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Load all locale-specific fonts installed in Tux Paint, regardless of the locale Tux Paint is being run under.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
          }
          yy += hh;

          {
            str =
              /* Input label: UI Font */
              gettext("UI Font:");
            ww = (half_box_width - hspace * 2.5) / 2;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);

            #define FONT_BROWSER_HEIGHT 4

            yy += hh;             /* Push down because label is FL_ALIGN_TOP_LEFT */
            yy += vspace;
            {
              Fl_Hold_Browser *o = BROWSER_uifont =
                new Fl_Hold_Browser(boxx + hspace, yy,
                                      (half_box_width - hspace * 2.5),
                                      small_font_size * 2 * FONT_BROWSER_HEIGHT, str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->textsize(small_font_size);
              o->align(FL_ALIGN_TOP_LEFT);
              o->callback((Fl_Callback *) cb_uifont_browser_chng);
              o->when(FL_WHEN_CHANGED);
              list_fonts(o);
	      yy += small_font_size * 2 * FONT_BROWSER_HEIGHT;
            }

            str =
              gettext
              ("Use a different font in Tux Paint's user interface (UI). (Leave blank to not override. Enter 'default' to use the default font.)");
            ww = half_box_width - hspace * 2;
            hh = 0;
            fl_font(FL_ITALIC, small_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(small_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
            }
          }
          o->end();

          yy += hh;             /* Push down because label is FL_ALIGN_TOP_LEFT */
          yy += vspace;
          {
            Fl_Input *o = TEXTINPUT_uifont2 =
                new Fl_Input(boxx + hspace, yy,
                                      (half_box_width - hspace * 2.5),
                                      small_font_size * 2, "");
            o->textsize(small_font_size);
            o->align(FL_ALIGN_TOP_LEFT);
            o->callback((Fl_Callback *) cb_uifont2_input_chng);
            o->when(FL_WHEN_CHANGED);
	    yy += small_font_size * 2 * FONT_BROWSER_HEIGHT;
          }
          o->end();
          TEXTINPUT_uifont2->deactivate();
        }
        o->end();
      }

      /* TAB: PRINTING */
      {
        Fl_Group *o = TABE_printing =
          new Fl_Group(padding, tab_top_y, tab_width, tab_height,
                       /* "Printing" tab */ gettext("Printing"));
        o->labelsize(tab_font_size);
        o->labelfont(FL_HELVETICA);
        o->box(FL_PLASTIC_UP_BOX);
        o->color((Fl_Color) 5);
        o->selection_color((Fl_Color) 5);
        o->align(37);
        o->hide();

        /* Section: SAVE PRINTER CONFIGURATION: [at the bottom] */
        boxx = half_box_l_x;
        str =
          /* Checkbox label: Save printer configuration */
          gettext("Save printer configuration");
        ww = half_box_width - hspace * 2 - big_font_size;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        sz = big_font_size * 3;
        yy = box_top_y + full_box_height - sz;

        {
          Fl_Group *o = GROUP_printcfg =
            new Fl_Group(boxx, yy, half_box_width, sz, "");
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);

          yy += (sz - hh) / 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_printcfg =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_status_chng);
          }

          o->end();
        }

        boxx = half_box_l_x;
        yy = box_top_y;

        /* Section: PRINT PERMISSIONS */
        str = gettext("Print Permissions:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        current_box_height = (full_box_height - sz - padding * 2) * 0.55;
        {
          Fl_Group *o = GROUP_printperms =
            new Fl_Group(boxx, yy, half_box_width, current_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          str =
            /* Checkbox label: Allow/disallow printing */
            gettext("Allow &Printing");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_allowprinting =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80070);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str = gettext("Let users print from within Tux Paint.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }


          str =
            /* Input label: Print delay (in seconds; labeled separately) */
            gettext("Print Delay:");
          ww = (half_box_width - hspace * 2.5) / 2;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);

          yy += hh;             /* Push down because label is FL_ALIGN_TOP_LEFT */
          yy += vspace;
          {
            Fl_Value_Input *o = VALUEINPUT_printdelay =
              new Fl_Value_Input(boxx + hspace, yy,
                                 (half_box_width - hspace * 2.5) / 2,
                                 small_font_size * 2, str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->maximum(3600);
            o->textsize(small_font_size);
            o->step(1);
            o->align(FL_ALIGN_TOP_LEFT);
            o->callback((Fl_Callback *) cb_status_chng);
            o->when(FL_WHEN_CHANGED);
          }

          str =
            /* Input label: Print delay (labeled separately) in seconds (this label) */
            gettext("seconds");
          ww = (half_box_width - hspace * 2.5) / 2;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          {
            Fl_Box *o = LABEL_seconds =
              new Fl_Box(boxx + hspace + (half_box_width - hspace * 2) / 2,
                         yy, (half_box_width - hspace * 2.5) / 2,
                         small_font_size * 2,
                         str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->box(FL_FLAT_BOX);
            o->align(FL_ALIGN_LEFT | FL_ALIGN_INSIDE);
            yy += small_font_size * 2;
          }

          str =
            gettext
            ("Restrict printing to once every N seconds. (Enter '0' to allow unrestricted printing.)");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(hspace, yy, half_box_width - hspace * 2, hh, str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
          }
          o->end();
        }

        /* Section: SHOW PRINTER DIALOG: */
        boxx = half_box_l_x;
        yy = box_top_y + current_box_height + padding;
        current_box_height =
          full_box_height - current_box_height - sz - padding * 2;
        str =
          /* Printer dialog display settings section */
          gettext("Show Printer Dialog:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        {
          Fl_Group *o = GROUP_printperms =
            new Fl_Group(boxx, yy, half_box_width, current_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          str =
            /* Only show printer settings when [Alt] held */
            gettext("Only when [Alt] &modifier key is held");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_HELVETICA, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Round_Button *o = ROUNDBUTTON_printdialogmod =
              new Fl_Round_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2,
                                  hh, str);
            o->labelsize(small_font_size);
            o->labelfont(FL_HELVETICA);
            o->type(102);
            o->down_box(FL_ROUND_DOWN_BOX);
            o->shortcut(0x80061);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            /* Always show printer settings (no need to hold [Alt]) */
            gettext("Always &show printer dialog");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_HELVETICA, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Round_Button *o = ROUNDBUTTON_printdialogalways =
              new Fl_Round_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(small_font_size);
            o->labelfont(FL_HELVETICA);
            o->type(102);
            o->down_box(FL_ROUND_DOWN_BOX);
            o->shortcut(0x80061);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            /* Never show printer dialog... */
            gettext("&Never show printer dialog");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_HELVETICA, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Round_Button *o = ROUNDBUTTON_printdialognever =
              new Fl_Round_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2,
                                  small_font_size, str);
            o->labelsize(small_font_size);
            o->labelfont(FL_HELVETICA);
            o->type(102);
            o->down_box(FL_ROUND_DOWN_BOX);
            o->shortcut(0x80061);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            /* (Never show printer dialog...) even when [Alt] is held */
            gettext("(Even when [Alt] is held.)");
          ww = half_box_width - hspace - small_font_size;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + 24, yy, half_box_width - 36, hh, str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
          }

          o->end();
        }

        /* Section: PRINT COMMANDS: */
        yy = box_top_y;
        boxx = half_box_r_x;

        str = gettext("Print Commands:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        {
          Fl_Group *o = GROUP_printcmd =
            new Fl_Group(boxx, yy, half_box_width, full_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          str =
            /* Checkbox label: Use alternative print command (for normal printing, without e.g. holding [Alt] for a dialog) */
            gettext("Use &Alternative Print Command");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_usealtprintcmd =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80061);
            o->callback((Fl_Callback *) cb_CHECKBUTTON_usealtprintcmd);
            yy += hh;
          }

          str =
            gettext
            ("Override Tux Paint's default setting for print command ('lpr') with another. (Advanced! Unix/Linux only!)");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }

          hh = small_font_size * 2;
          yy += vspace_s;
          {
            Fl_Input *o = TEXTINPUT_printcmd =
              new Fl_Input(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           "");
            o->align(FL_ALIGN_TOP_LEFT);
            o->deactivate();
            o->callback((Fl_Callback *) cb_status_chng);
            o->when(FL_WHEN_CHANGED);
            yy += hh;
          }

          str =
            gettext
            ("Enter the command for printing. It must accept a PostScript format on its standard input (STDIN).");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }
          o->end();


          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);

          str =
            /* Checkbox label: Use alternative print dialog (for when settings should be displayed before printing, e.g. when holding [Alt]) */
            gettext("Use &Alternative Print Dialog");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_usealtprintdia =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80061);
            o->callback((Fl_Callback *) cb_CHECKBUTTON_usealtprintdia);
            yy += hh;
          }

          str =
            gettext
            ("Override Tux Paint's default setting for print dialog ('kprinter') with another. (Advanced! Unix/Linux only!)");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }

          hh = small_font_size * 2;
          yy += vspace_s;
          {
            Fl_Input *o = TEXTINPUT_printdia =
              new Fl_Input(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           "");
            o->align(FL_ALIGN_TOP_LEFT);
            o->deactivate();
            o->callback((Fl_Callback *) cb_status_chng);
            o->when(FL_WHEN_CHANGED);
            yy += hh;
          }

          str =
            gettext
            ("Enter the print dialog command. It must accept a PostScript format on its standard input (STDIN).");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }

#ifdef USE_PAPERSIZES
          str = /* Pulldown label: Paper size */ gettext("Paper Size:");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);

          yy += hh;             /* Push down because label is FL_ALIGN_TOP_LEFT */
          yy += vspace * 2;
          {
            Fl_Choice *o = CHOICE_papersize =
              new Fl_Choice(boxx + hspace, yy, half_box_width - hspace * 2,
                            hh, str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->labeltype(FL_EMBOSSED_LABEL);
            o->textsize(small_font_size);
            o->down_box(FL_BORDER_BOX);
            o->align(FL_ALIGN_TOP_LEFT);

            int i;
            for (i = 0; i < num_papersizes; i++)
            {
              o->add(papersizes[i], "", NULL, NULL, 0);
            }
            o->align(FL_ALIGN_TOP_LEFT);
            o->deactivate();
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }
#endif
          o->end();
        }

        o->end();
      }

      /* TAB: SAVING */
      str = /* "Saving" tab */ gettext("Saving");
      {
        Fl_Group *o = TABE_saving =
          new Fl_Group(padding, tab_top_y, tab_width, tab_height, str);
        o->labelsize(tab_font_size);
        o->labelfont(FL_HELVETICA);
        o->box(FL_PLASTIC_UP_BOX);
        o->color((Fl_Color) 6);
        o->selection_color((Fl_Color) 6);
        o->align(37);
        o->hide();

        boxx = half_box_l_x;
        yy = box_top_y;
        current_box_height = (full_box_height - padding) * 0.6;

        /* Section: SAVE OVER EARLIER WORK */
        str =
          /* "Save over earlier work" settings section */
          gettext("Save Over Earlier Work:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);

        {
          Fl_Group *o = GROUP_saveover =
            new Fl_Group(boxx, yy, half_box_width, current_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          str =
            /* When saving, ask before overwriting previous version of the drawing */
            gettext("&Ask Before Overwriting");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Round_Button *o = ROUNDBUTTON_askbeforeoverwriting =
              new Fl_Round_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->type(102);
            o->down_box(FL_ROUND_DOWN_BOX);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("When re-saving an image, ask whether to overwrite the earlier version, or make a new file.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelsize(small_font_size);
            o->labelfont(FL_ITALIC);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }

          str =
            /* When saving, always overwrite the previous version of the drawing */
            gettext("Always &Overwrite Older Version");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Round_Button *o = ROUNDBUTTON_alwaysoverwriteolderversion =
              new Fl_Round_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->type(102);
            o->down_box(FL_ROUND_DOWN_BOX);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("When re-saving an image, always overwrite the earlier version. (Warning: Potential for lost work!)");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelsize(small_font_size);
            o->labelfont(FL_ITALIC);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }

          str =
            /* When saving, always save a new drawing (never overwrite the earlier versions of the drawing) */
            gettext("Always Save &New Picture");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Round_Button *o = ROUNDBUTTON_alwayssavenewpicture =
              new Fl_Round_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->type(102);
            o->down_box(FL_ROUND_DOWN_BOX);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("When re-saving an image, always make a new file. (Warning: Potential for lots of files!)");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelsize(small_font_size);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
          }
          o->end();
        }

        /* Section: STARTING OUT: */
        yy = box_top_y + current_box_height + padding;
        current_box_height = full_box_height - current_box_height - padding;

        str = /* "Starting out" settings section */ gettext("Starting Out:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        {
          Fl_Group *o = GROUP_startingout =
            new Fl_Group(boxx, yy, half_box_width, current_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          str =
            /* Checkbox label: Start with a blank canvas (vs. loading the drawing that was being worked on the last time Tux Paint was exited-from) */
            gettext("Start with &Blank Canvas");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_startblank =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Always show a blank canvas (not the last open drawing) upon launch.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelsize(small_font_size);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }
          o->end();


          str =
            /* Checkbox label: Place solid colors at the end of the list (below Starters & Templates) in the "New" dialog when starting a new drawing */
            gettext("Colors last in 'New' dialog");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_newcolorslast =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }
          o->end();

          str =
            gettext
            ("Place solid color backgrounds at the end of the \"New\" dialog, promoting pre-drawn Starters and Templates to the top.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelsize(small_font_size);
            o->labelfont(FL_ITALIC);
            o->align(133 | FL_ALIGN_INSIDE);
          }
          o->end();
        }

        /* Section: SAVE AND EXPORT DIRECTORIES: */
        boxx = half_box_r_x;
        yy = box_top_y;

        str =
          /* "Save and Export Directories" settings section */
          gettext("Save and Export Directories:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        current_box_height = full_box_height * 0.66;

        {
          Fl_Group *o = GROUP_savedir =
            new Fl_Group(boxx, yy, half_box_width, current_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          /* Save directory */
          str =
            /* Checkbox label: Save drawings in a specified location (vs, e.g. "~/.tuxpaint/saved/") */
            gettext("Use &Alternative Save Directory");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_usealternatesavedir =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80061);
            o->callback((Fl_Callback *) cb_CHECKBUTTON_usealternatesavedir);
            yy += hh;
          }

          str =
            gettext
            ("Do not save pictures in the standard directory, use the following location:");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }

          str =
            /* Label for file selector: Alternative Save Directory */
            gettext("Alternative Save Directory:");
          ww = ((half_box_width - hspace * 2) * 75) / 100;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += hh;             /* push down to make room for label above widget */
          yy += vspace_s;
          {
            Fl_File_Input *o = FILEINPUT_alternatesavedir =
              new Fl_File_Input(boxx + hspace, yy,
                                ((half_box_width - hspace * 2) * 75) / 100,
                                (big_font_size * 2), str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->align(69);
            o->deactivate();
            o->callback((Fl_Callback *) cb_status_chng);
          }

#define BROWSE_BTN_TXT /* Label for button to browse the filesystem for various purposes (e.g., specifying an alternative location to save drawings, etc.) */ gettext("Browse...")
          str = BROWSE_BTN_TXT;
          ww = ((half_box_width - hspace * 2) * 25) / 100;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          {
            Fl_Button *o = BUTTON_browsealternatesavedir =
              new Fl_Button(boxx + hspace +
                            ((half_box_width - hspace * 2) * 75) / 100, yy,
                            ((half_box_width - hspace * 2) * 25) / 100, hh,
                            str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->shortcut(0x80062);
            o->deactivate();
            o->callback((Fl_Callback *) cb_BUTTON_browsealternatesavedir);
            yy += (hh < (big_font_size * 2) ? (big_font_size * 2) : hh);
          }

          /* Export directory */
          str =
            /* Label for file selector: Alternative Export Directory (vs. e.g. "~/Pictures") */
            gettext("Use Alternative Export Directory");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Check_Button *o = CHECKBUTTON_usealternateexportdir =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_CHECKBUTTON_usealternateexportdir);
            yy += hh;
          }

          str =
            gettext
            ("Do not export pictures and animated GIFs in the standard directory, use the following location:");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }

          str =
            /* Label for file selector: Alternative Export Directory */
            gettext("Alternative Export Directory:");
          ww = ((half_box_width - hspace * 2) * 75) / 100;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += hh;             /* push down to make room for label above widget */
          yy += vspace_s;
          {
            Fl_File_Input *o = FILEINPUT_alternateexportdir =
              new Fl_File_Input(boxx + hspace, yy,
                                ((half_box_width - hspace * 2) * 75) / 100,
                                big_font_size * 2, str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->align(69);
            o->deactivate();
            o->callback((Fl_Callback *) cb_status_chng);
          }

          str = BROWSE_BTN_TXT;
          ww = ((half_box_width - hspace * 2) * 25) / 100;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          {
            Fl_Button *o = BUTTON_browsealternateexportdir =
              new Fl_Button(boxx + hspace +
                            ((half_box_width - hspace * 2) * 75) / 100, yy,
                            ((half_box_width - hspace * 2) * 25) / 100, hh,
                            str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->deactivate();
            o->callback((Fl_Callback *) cb_BUTTON_browsealternateexportdir);
          }
          o->end();
        }

        /* Section: MORE SAVING OPTIONS: */
        current_box_height = full_box_height - current_box_height - padding;
        yy = box_top_y + full_box_height - current_box_height;

        str =
          /* "More Saving Options" section */ gettext("More Saving Options:");
        {
          Fl_Scroll *o =
            new Fl_Scroll(boxx, yy, half_box_width, current_box_height, "");
          o->type(Fl_Scroll::VERTICAL_ALWAYS);

          ww = half_box_width;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          {
            Fl_Group *o = GROUP_nosave =
              new Fl_Group(boxx, yy, half_box_width,
                           full_box_height * 0.5,
                           str);
            o->box(FL_PLASTIC_UP_BOX);
            o->labeltype(FL_EMBOSSED_LABEL);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
            yy += hh;

            str =
              /* Checkbox label: Disable "Save" button */
              gettext("Disable \'&Save\' Button");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace * 2;
            {
              Fl_Check_Button *o = CHECKBUTTON_disablesavebutton =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80071);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }


            str =
              /* Checkbox label: Auto-save when quitting Tux Paint */
              gettext("&Auto-save on Quit");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_autosave =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80071);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Don't ask to save current picture when quitting; just save.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, small_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(small_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
              yy += hh;
            }

            str =
              /* Checkbox label: Reverse sort (of Open and Slideshow dialogs) */
              gettext("Reverse Sort");
            ww = half_box_width - hspace * 3 - big_font_size;
            hh = 0;
            fl_font(FL_HELVETICA, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace;
            {
              Fl_Check_Button *o = CHECKBUTTON_reversesort =
                new Fl_Check_Button(boxx + hspace, yy,
                                    half_box_width - hspace * 2, hh,
                                    str);
              o->labelsize(big_font_size);
              o->labelfont(FL_HELVETICA);
              o->down_box(FL_DOWN_BOX);
              o->shortcut(0x80071);
              o->callback((Fl_Callback *) cb_status_chng);
              yy += hh;
            }

            str =
              gettext
              ("Place newest drawings at the top of Open and Slideshow.");
            ww = half_box_width - hspace * 3;
            hh = 0;
            fl_font(FL_ITALIC, big_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                           str);
              o->labelfont(FL_ITALIC);
              o->labelsize(small_font_size);
              o->align(133 | FL_ALIGN_INSIDE);
            }

            o->end();           // Fl_Group *o
          }
          o->end();
        }
        o->end();
      }

      /* TAB: DATA */
      {
        Fl_Group *o = TABE_data =
          new Fl_Group(padding, tab_top_y, tab_width, tab_height,
                       /* "Data" tab */ gettext("Data"));
        o->labelsize(tab_font_size);
        o->box(FL_PLASTIC_UP_BOX);
        o->color((Fl_Color) 7);
        o->selection_color((Fl_Color) 7);
        o->align(37);
        o->hide();

        boxx = half_box_l_x;
        yy = box_top_y;

        /* Section: LOCKFILE: */
        str = /* Lockfile settings section */ gettext("Lockfile:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);

        {
          Fl_Group *o = GROUP_lockfile =
            new Fl_Group(boxx, yy, half_box_width, half_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          str =
            /* Checkbox label: Don't use a lockfile (allow multiple simultaneous launches of Tux Paint in quick succession) */
            gettext("&Don't use lockfile");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_ITALIC, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_nolockfile =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Do not check for a lockfile. Allow Tux Paint to be launched multiple times. (May be necessary in a networked environment.)");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
          }
          o->end();
        }

        /* Section: DATA DIRECTORY: */
        boxx = half_box_r_x;
        yy = box_top_y;

        str =
          /* Data directory settings section */ gettext("Data Directory:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);

        {
          Fl_Group *o = GROUP_datadir =
            new Fl_Group(boxx, yy, half_box_width, half_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          str =
            /* Checkbox label: use alternative data directory (for loading brushes, stamps, etc.) (vs e.g., /usr/share/tuxpaint/) */
            gettext("Use &Alternative Data Directory");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_usealternatedatadir =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80061);
            o->callback((Fl_Callback *) cb_CHECKBUTTON_usealternatedatadir);
            yy += hh;
          }

          str =
            gettext
            ("Do not load brushes, stamps, etc. from the standard directory, use the following location:");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }

          str =
            /* Label for file selector: Alternative Data Directory */
            gettext("Alternative Data Directory:");
          ww = ((half_box_width - hspace * 2) * 75) / 100;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += hh;             /* push down to make room for label above widget */
          yy += vspace_s;
          {
            Fl_File_Input *o = FILEINPUT_alternatedatadir =
              new Fl_File_Input(boxx + hspace, yy,
                                ((half_box_width - hspace * 2) * 75) / 100,
                                (big_font_size * 2), str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->align(69);
            o->deactivate();
            o->callback((Fl_Callback *) cb_status_chng);
          }

          str = BROWSE_BTN_TXT;
          ww = ((half_box_width - hspace * 2) * 25) / 100;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          {
            Fl_Button *o = BUTTON_browsealternatedatadir =
              new Fl_Button(boxx + hspace +
                            ((half_box_width - hspace * 2) * 75) / 100, yy,
                            ((half_box_width - hspace * 2) * 25) / 100, hh,
                            str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->shortcut(0x80062);
            o->deactivate();
            o->callback((Fl_Callback *) cb_BUTTON_browsealternatedatadir);
            yy += (hh < (big_font_size * 2) ? (big_font_size * 2) : hh);
          }
          o->end();
        }

        yy = box_top_y + full_box_height - half_box_height;

        str =
          /* Color palette settings section */ gettext("Color Palette File:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        {
          Fl_Group *o = GROUP_colorfile =
            new Fl_Group(boxx, yy, half_box_width, half_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE);
          yy += hh;

          str =
            /* Checkbox label: Use Alternative Color Palette */
            gettext("Use &Alternative Color Palette");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_usecolorfile =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80061);
            o->callback((Fl_Callback *) cb_CHECKBUTTON_usecolorfile);
            yy += hh;
          }

          str =
            gettext
            ("Don't use default color palette, use colors defined in the following file:");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(133 | FL_ALIGN_INSIDE);
            yy += hh;
          }

          str =
            /* Label for file selector: Color Palette File */
            gettext("Color Palette File:");
          ww = ((half_box_width - hspace * 2) * 75) / 100;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += hh;             /* push down to make room for label above widget */
          yy += vspace_s;
          {
            Fl_File_Input *o = FILEINPUT_colorfile =
              new Fl_File_Input(boxx + hspace, yy,
                                ((half_box_width - hspace * 2) * 75) / 100,
                                (big_font_size * 2), str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->align(69);
            o->deactivate();
            o->callback((Fl_Callback *) cb_status_chng);
          }

          str = BROWSE_BTN_TXT;
          ww = ((half_box_width - hspace * 2) * 25) / 100;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          {
            Fl_Button *o = BUTTON_browsecolorfile =
              new Fl_Button(boxx + hspace +
                            ((half_box_width - hspace * 2) * 75) / 100, yy,
                            ((half_box_width - hspace * 2) * 25) / 100, hh,
                            str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->shortcut(0x80062);
            o->deactivate();
            o->callback((Fl_Callback *) cb_BUTTON_browsecolorfile);
          }
          o->end();
        }
        o->end();
      }


      /* TAB: ACCESSIBILITY */
      {
        Fl_Group *o = TABE_accessibility =
          new Fl_Group(padding, tab_top_y, tab_width, tab_height,
                       /* "Accessibility" tab */ gettext("Accessibility"));
        o->labelsize(tab_font_size);
        o->labelfont(FL_HELVETICA);
        o->box(FL_PLASTIC_UP_BOX);
        o->color((Fl_Color) 93);
        o->selection_color((Fl_Color) 93);
        o->align(Fl_Align(37));

        boxx = half_box_l_x;
        yy = box_top_y;
        current_box_height = (full_box_height - padding) * 0.4;
        /* Section: MOUSE: */
        str = /* Mouse accessibility settings section */ gettext("Mouse:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);

        {
          Fl_Group *o =
            new Fl_Group(boxx, yy, half_box_width, current_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(Fl_Align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE));
          yy += hh;

          str =
            /* Checkbox label: Sticky mouse clicks -- click once to begin a click/drag, then click again to end ('release' the click) */
            gettext("Sticky mouse clicks");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_mouse_accessibility =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Useful for users who have difficulty clicking and dragging. When enabled, click and release to start painting, move to paint, and click and release again to stop. It can also be combined with joystick- and keyboard-based pointer controls.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
          }
          o->end();
        }

        /* Section: KEYBOARD: */
        boxx = half_box_l_x;
        yy = box_top_y + current_box_height + padding;;
        current_box_height = full_box_height - padding - current_box_height;
        str =
          /* Keyboard accessibility settings section */ gettext("Keyboard:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);

        {
          Fl_Group *o =
            new Fl_Group(boxx, yy, half_box_width, current_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(Fl_Align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE));
          yy += hh;

          str =
            /* Checkbox label: use the keyboard to control the mouse pointer */
            gettext("Keyboard controls the mouse pointer");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_keyboard_mouse =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("When enabled, the arrow keys or numbers can be used to move the mouse pointer. Number 5, Space, Insert or F5 can be used to click. (When in \"Text\" or \"Label\" tools, the numbers and space cannot be used.) Also, F4 cycles the pointer between the \"Tools\" box, \"Colors\" box and drawing canvas, F8 and F7 move up/down inside the \"Tools\" box, and F11 and F12 move left/right inside the \"Tools\" and \"Colors\" boxes.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
          }
          o->end();
        }

        /* Section: ONSCREEN KEYBOARD: */
        boxx = half_box_r_x;
        yy = box_top_y;

        str =
          /* Onscreen keyboard settings section */
          gettext("Onscreen keyboard:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        {
          Fl_Group *o =
            new Fl_Group(boxx, yy, half_box_width, full_box_height, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_ENGRAVED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(Fl_Align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE));
          yy += hh;

          str =
            /* Checkbox label: show the onscreen keyboard */
            gettext("Show a keyboard on the screen");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_onscreen_keyboard =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->callback((Fl_Callback *) cb_CHECKBUTTON_onscreen_keyboard);
            yy += hh;
          }

          str =
            gettext
            ("Display a keyboard on the screen when the \"Text\" and \"Label\" tools are enabled, so you can 'type' with the mouse pointer.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
            yy += hh;
          }


          str =
            /* Menu label: Layout (for onscreen keyboard) */
            gettext("Layout");
          ww = half_box_width;
          hh = 0;
          fl_font(FL_HELVETICA, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Choice *o = CHOICE_onscreen_keyboard_layout =
              new Fl_Choice(boxx + hspace, yy, half_box_width / 2,
                            small_font_size * 2, str);
            o->labelsize(small_font_size);
            o->labelfont(FL_HELVETICA);
            o->textsize(small_font_size);
            o->down_box(FL_BORDER_BOX);
            o->align(Fl_Align(FL_ALIGN_RIGHT));
            o->when(FL_WHEN_CHANGED);
            int i;
            for (i = 0; i < NUM_LAYOUTS; i++)
            {
              o->add(layouts[i][LAYOUT_NAME], "", NULL, NULL, 0);
              o->deactivate();
            }
            o->callback((Fl_Callback *) cb_status_chng);
            yy += small_font_size * 2;
          }

          str = gettext("How are keys organized in the keyboard");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
            yy += hh;
          }


          str =
            /* Checkbox label: Disable layout changes (of the onscreen keyboard) */
            gettext("Disable layout changes");
          ww = half_box_width - hspace * 2 - big_font_size;
          hh = 0;
          fl_font(FL_HELVETICA, big_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Check_Button *o =
              CHECKBUTTON_onscreen_keyboard_disable_change =
              new Fl_Check_Button(boxx + hspace, yy,
                                  half_box_width - hspace * 2, hh,
                                  str);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->deactivate();
            o->callback((Fl_Callback *) cb_status_chng);
            yy += hh;
          }

          str =
            gettext
            ("Disable the buttons that allow changing the keyboard layout.");
          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
          }
          o->end();
        }
        o->end();
        Fl_Group::current()->resizable(o);
      }

      /* TAB: JOYSTICK */
      {
        Fl_Group *o = TABE_joystick =
          new Fl_Group(padding, tab_top_y, tab_width, tab_height,
                       /* "Joystick" tab */ gettext("Joystick"));
        o->labelsize(tab_font_size);
        o->labelfont(FL_HELVETICA);
        o->color((Fl_Color) 234);
        o->selection_color((Fl_Color) 234);
        o->align(Fl_Align(FL_ALIGN_TOP_LEFT));
        o->box(FL_PLASTIC_UP_BOX);

        /* Calculate size we need for spinners */
        int spin_h = (big_font_size * 15) / 10;
        ww = 0;
        hh = spin_h;
        fl_font(FL_HELVETICA, small_font_size);
        fl_measure("WWWW", ww, hh);
        int spin_w = ww + small_font_size /* space for arrows */ ;


        /* Section: BUTTONS TO DISABLE: [at the bottom] */

        /* Measure up from the bottom; lame, but it works in a pinch -bjk 2021.08.07 */
        str =
          /* Checkbox label: Joystick buttons to disable/ignore */
          gettext("Buttons to disable:");

        char *str2 =
          gettext
          ("If for any reason one or more buttons of the joystick are posing problems, you can disable them here using a comma separated list of button numbers. (e.g. 2,3,5)");

        ww = half_box_width - hspace * 2;
        int buttons_disable_h = 0;
        hh = 0;
        fl_font(FL_ITALIC, small_font_size);
        measure_and_softwrap(str2, ww, hh);
        buttons_disable_h += hh;
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        buttons_disable_h += hh + big_font_size + vspace * 3 + vspace_s;

        boxx = half_box_l_x;
        yy = box_top_y + full_box_height - buttons_disable_h;
        /* "yy" is now the top of the "BUTTONS TO DISABLE" section */
        {
          Fl_Group *o =
            new Fl_Group(boxx, yy, half_box_width, buttons_disable_h, str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_ENGRAVED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(Fl_Align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE));
          yy += hh;
          yy += vspace * 2;
          {
            Fl_Check_Button *o = CHECKBUTTON_joystick_buttons_ignore =
              new Fl_Check_Button(boxx + hspace, yy, big_font_size,
                                  big_font_size);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->down_box(FL_DOWN_BOX);
            o->shortcut(0x80066);
            o->callback((Fl_Callback *)
                        cb_CHECKBUTTON_joystick_buttons_ignore);
          }

          {
            Fl_Input *o = TEXTINPUT_joystick_buttons_ignore =
              new Fl_Input(boxx + hspace * 1.5 + big_font_size, yy,
                           half_box_width - (hspace * 2.5 +
                                             big_font_size), big_font_size);
            o->labeltype(FL_NO_LABEL);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->align(Fl_Align(196));
            o->textsize(small_font_size);
            o->callback((Fl_Callback *) cb_status_chng);
            o->when(FL_WHEN_CHANGED);
          }

          yy += big_font_size;

          ww = half_box_width - hspace * 2;
          hh = 0;
          fl_font(FL_ITALIC, small_font_size);
          measure_and_softwrap(str2, ww, hh);
          yy += vspace_s;
          {
            Fl_Box *o =
              new Fl_Box(boxx + hspace, yy, half_box_width - hspace * 2, hh,
                         str2);
            o->labelfont(FL_ITALIC);
            o->labelsize(small_font_size);
            o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
            yy += hh;
          }
          o->end();
        }

        /* Section: HAT: [in the middle] */
        str = /* Joystick hat settings section */ gettext("Hat:");
        int hat_h = 0;
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        hat_h += hh + vspace * 2;

        char *str1 = gettext("Hat slowness. (0-500; default value is 15)");
        int ww1 = half_box_width - hspace * 2 - spin_w;
        int hh1 = 0;
        fl_font(FL_ITALIC, smaller_font_size);
        measure_and_softwrap(str1, ww1, hh1);
        hat_h += hh1;

        str2 =
          gettext
          ("Hat timeout, the number of milliseconds to wait before starting to move continuously. (0-3000; default value is 1000)");
        int ww2 = half_box_width - hspace * 2 - spin_w;
        int hh2 = 0;
        fl_font(FL_ITALIC, smaller_font_size);
        measure_and_softwrap(str2, ww2, hh2);
        hat_h += hh2 + vspace;
        hat_h += vspace;

        yy =
          box_top_y + full_box_height - buttons_disable_h - hat_h - padding;
        {
          Fl_Group *o = new Fl_Group(boxx, yy,
                                     half_box_width,
                                     hat_h,
                                     str);
          o->box(FL_PLASTIC_UP_BOX);
          o->labeltype(FL_EMBOSSED_LABEL);
          o->labelsize(big_font_size);
          o->labelfont(FL_HELVETICA);
          o->align(Fl_Align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE));
          yy += hh;

          /* Hat slowness */
          yy += vspace * 2;
          {
            Fl_Spinner *o = SPINNER_joystick_hat_slowness =
              new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h);
            o->maximum(500);
            o->value(15);
            o->callback((Fl_Callback *) cb_status_chng);
            o->when(FL_WHEN_CHANGED);
          }

          {
            Fl_Box *o = new Fl_Box(boxx + hspace + spin_w, yy,
                                   half_box_width - hspace * 2 - spin_w, hh1,
                                   str1);
            o->labelfont(FL_ITALIC);
            o->labelsize(smaller_font_size);
            o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
            yy += (hh > spin_h ? hh : spin_h);
          }

          /* Hat timeout */
          {
            Fl_Spinner *o = SPINNER_joystick_hat_timeout =
              new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h);
            o->minimum(0);
            o->maximum(3000);
            o->step(100);
            o->value(1000);
            o->callback((Fl_Callback *) cb_status_chng);
            o->when(FL_WHEN_CHANGED);
          }

          {
            Fl_Box *o = new Fl_Box(boxx + hspace + spin_w, yy,
                                   half_box_width - hspace * 2 - spin_w, hh2,
                                   str2);
            o->labelfont(FL_ITALIC);
            o->labelsize(smaller_font_size);
            o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
          }
          o->end();
        }

        boxx = half_box_l_x;
        yy = box_top_y;
        str =
          /* Main joystick device settings section */ gettext("Main device:");
        {
          Fl_Scroll *o =
            new Fl_Scroll(boxx, yy, half_box_width,
                          full_box_height - buttons_disable_h - hat_h -
                          padding * 2, "");
          o->type(Fl_Scroll::VERTICAL_ALWAYS);
          {
            Fl_Group *o = new Fl_Group(boxx, box_top_y, half_box_width,
                                       full_box_height, str);
            o->box(FL_PLASTIC_UP_BOX);
            o->labeltype(FL_EMBOSSED_LABEL);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->align(Fl_Align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE));
            ww = half_box_width - 16;
            o->measure_label(ww, hh);
            yy += hh;

            /* Joystick dev */
            yy += vspace * 2;
            {
              Fl_Spinner *o = SPINNER_joystick_dev =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h);
              o->minimum(0);
              o->value(0);
              o->align(Fl_Align(FL_ALIGN_LEFT | FL_ALIGN_INSIDE));
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
            }

            str =
              gettext("Device number to use. (SDL starts numbering at 0)");
            ww = half_box_width - hspace * 2 - spin_w - 16;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Box *o = new Fl_Box(boxx + hspace + spin_w, yy,
                                     half_box_width - hspace * 2 - spin_w -
                                     16, hh, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
              yy += (hh > spin_h ? hh : spin_h);
            }

            /* Joystick slowness */
            {
              Fl_Spinner *o = SPINNER_joystick_slowness =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h);
              o->minimum(0);
              o->maximum(500);
              o->value(15);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
            }

            str =
              gettext
              ("Joystick slowness. Increase this value for people with slow reactions. (0-500; default value is 15)");
            ww = half_box_width - hspace * 2 - spin_w - 16;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Box *o = new Fl_Box(boxx + hspace + spin_w, yy,
                                     half_box_width - hspace * 2 - spin_w -
                                     16, hh, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
              yy += (hh > spin_h ? hh : spin_h);
            }

            /* Joystick threshold */
            {
              Fl_Spinner *o = SPINNER_joystick_threshold =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h);
              o->minimum(0);
              o->maximum(32766);
              o->step(100);
              o->value(3200);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
            }

            str =
              gettext
              ("SDL events under threshold will be discarded, useful to discard accidental unwanted movements. (0-32766; default value is 3200)");
            ww = half_box_width - hspace * 2 - spin_w - 16;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Box *o = new Fl_Box(boxx + hspace + spin_w, yy,
                                     half_box_width - hspace * 2 - spin_w -
                                     16, hh, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
              yy += (hh > spin_h ? hh : spin_h);
            }

            /* Joystick max steps */
            {
              Fl_Spinner *o = SPINNER_joystick_maxsteps =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h);
              o->maximum(7);
              o->value(7);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
            }

            str =
              gettext("Limit speed when moving. (1-7; default value is 7)");
            ww = half_box_width - hspace * 2 - spin_w - 16;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Box *o = new Fl_Box(boxx + hspace + spin_w, yy,
                                     half_box_width - hspace * 2 - spin_w -
                                     16, hh, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
              yy += (hh > spin_h ? hh : spin_h);
            }

            /* FIXME: Want to resize, but it re-positions the
               widgets within the group, which is unfortunately not
               what I want -bjk 2021.08.07

               o->size(half_box_width, yy - tab_top_y);
             */
            o->end();
          }
          o->end();
        }

        /* Section: BUTTON SHORTCUTS: */
        yy = box_top_y;
        boxx = half_box_r_x;
        str =
          /* Joytick button shortcuts (for switching between tools in Tux Paint) settings section */
          gettext("Button shortcuts:");
        ww = half_box_width;
        hh = 0;
        fl_font(FL_HELVETICA, big_font_size);
        measure_and_softwrap(str, ww, hh);
        str2 =
          gettext
          ("Here you can configure shortcuts for the different buttons of the joystick. (Beware to not assign a shortcut to the button used to draw.)");
        ww2 = half_box_width - hspace * 2 - 16;
        hh2 = 0;
        fl_font(FL_ITALIC, small_font_size);
        // measure_and_softwrap(str2, ww2, hh2);
        measure_and_softwrap(str2, ww2, hh2);
        {
          Fl_Scroll *o =
            new Fl_Scroll(boxx, yy, half_box_width, full_box_height, "");
          o->type(Fl_Scroll::VERTICAL_ALWAYS);
          {
            Fl_Group *o =
              new Fl_Group(boxx, yy, half_box_width, full_box_height * 2,
                           str);
            o->box(FL_PLASTIC_UP_BOX);
            o->labeltype(FL_EMBOSSED_LABEL);
            o->labelsize(big_font_size);
            o->labelfont(FL_HELVETICA);
            o->align(Fl_Align(FL_ALIGN_TOP_LEFT | FL_ALIGN_INSIDE));
            yy += hh;
            fl_font(FL_ITALIC, small_font_size);
            yy += vspace * 2;
            {
              Fl_Box *o =
                new Fl_Box(boxx + hspace, yy,
                           half_box_width - hspace * 2 - 16, hh2, str2);
              o->labelfont(FL_ITALIC);
              o->labelsize(small_font_size);
              o->align(Fl_Align(133 | FL_ALIGN_INSIDE));
              yy += hh2;
            }

            str = gettext("Button number for the Escape key.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            yy += vspace_s;
            {
              Fl_Spinner *o = SPINNER_joystick_btn_escape =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for the Paint tool.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_brush =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for the Stamp tool.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_stamp =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for the Lines tool.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_lines =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for the Shapes tool.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_shapes =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for the Text tool.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_text =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for the Label tool.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_label =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for the Fill tool.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_fill =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for the Magic tool.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_magic =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for Undo.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_undo =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for Redo.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_redo =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for the Eraser tool.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_eraser =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for the New tool.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_new =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for the Open tool.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_open =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for saving.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_save =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for Page Setup.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_pgsetup =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }

            str = gettext("Button number for printing.");
            ww = half_box_width - hspace * 2 - spin_w;
            hh = 0;
            fl_font(FL_ITALIC, smaller_font_size);
            measure_and_softwrap(str, ww, hh);
            {
              Fl_Spinner *o = SPINNER_joystick_btn_print =
                new Fl_Spinner(boxx + hspace, yy, spin_w, spin_h, str);
              o->labelfont(FL_ITALIC);
              o->labelsize(smaller_font_size);
              o->minimum(0);
              o->maximum(255);
              o->callback((Fl_Callback *) cb_status_chng);
              o->when(FL_WHEN_CHANGED);
              o->align(Fl_Align(FL_ALIGN_RIGHT));
              yy += (hh > spin_h ? hh : spin_h);
            }
            o->end();
          }                     // Fl_Group* o
          o->end();
        }                       // Fl_Scroll* o
        o->end();
      }                         // Fl_Group* TABE_joystick

      o->end();                 // TAB_alltabs
    }

    str =
      /* Label for deciding for whom the settings should be saved (current user vs. all users) */
      gettext("Settings for:");

    int ww1;
    ww = width / 3;
    hh = 0;
    fl_font(FL_HELVETICA, big_font_size);
    measure_and_softwrap(str, ww, hh);
    ww1 = width / 3;
    hh = 0;
    fl_font(FL_HELVETICA, small_font_size);
    fl_measure(gettext("Current User"), ww1, hh);
    if (ww1 > ww)
      ww = ww1;
    ww1 = width / 3;
    hh = 0;
    fl_font(FL_HELVETICA, small_font_size);
    fl_measure(gettext("All Users"), ww1, hh);
    if (ww1 > ww)
      ww = ww1;
    ww += small_font_size * 2.5;
    xx = ww + hspace * 1.5;
    {
      Fl_Choice *o = CHOICE_settings =
        new Fl_Choice(hspace, height - (small_font_size * 2) - padding, ww,
                      small_font_size * 2, str);
      o->down_box(FL_BORDER_BOX);
      o->align(FL_ALIGN_TOP_LEFT);
      o->labelsize(big_font_size);
      o->labelfont(FL_HELVETICA);
      o->textsize(small_font_size);

      o->add(gettext("Current User"), "", NULL, NULL, 0);
      o->add(gettext("All Users"), "", NULL, NULL, 0);

      o->value(CURRENT_USER);

      o->callback((Fl_Callback *) cb_CHOICE_settings);
    }

    str =
      /* Label for deciding for whom the settings should be saved (current user vs. all users) */
      gettext("Effects of \"All Users\" Settings");
    ww = width / 3;
    hh = 0;
    fl_font(FL_HELVETICA, big_font_size);
    measure_and_softwrap(str, ww, hh);
    ww1 = width / 3;
    hh = 0;
    fl_font(FL_HELVETICA, small_font_size);
    fl_measure(gettext("Merged with User Settings"), ww1, hh);
    if (ww1 > ww)
      ww = ww1;
    ww1 = width / 3;
    hh = 0;
    fl_font(FL_HELVETICA, small_font_size);
    fl_measure(gettext("Discard; use only User Settings"), ww1, hh);
    if (ww1 > ww)
      ww = ww1;
    ww1 = width / 3;
    hh = 0;
    fl_font(FL_HELVETICA, small_font_size);
    fl_measure(gettext("Use only \"All Users\" Settings"), ww1, hh);
    if (ww1 > ww)
      ww = ww1;
    ww += small_font_size * 2.5;
    {
      Fl_Choice *o = use_merge_discard_allusersetting =
        new Fl_Choice(xx, height - (small_font_size * 2) - padding, ww,
                      small_font_size * 2, str);
      o->down_box(FL_BORDER_BOX);
      o->align(FL_ALIGN_TOP_LEFT);
      o->add(gettext("Merged with User Settings"), "", NULL, NULL, 0);  // SYS_SETTING_STRATEGY_MERGE
      o->add(gettext("Discard; use only User Settings"), "", NULL, NULL, 0);    // SYS_SETTING_STRATEGY_DISCARD
      o->add(gettext("Use only \"All Users\" Settings"), "", NULL, NULL, 0);    // SYS_SETTING_STRATEGY_USEONLY
      o->value(SYS_SETTING_STRATEGY_MERGE);
      o->callback((Fl_Callback *) cb_use_merge_discard_allusersetting);

      o->labelsize(big_font_size);
      o->labelfont(FL_HELVETICA);
      o->textsize(small_font_size);
      o->down_box(FL_DOWN_BOX);
    }

    /* Drawing bottom right buttons from right-to-left */
    int hspace_for_buttons = width - xx - ww - hspace * 2;
    yy = tab_top_y + tab_height + padding;
    xx = width - hspace;

    str = /* Control button: Quit/exit Tux Paint Config. */ gettext("Quit");
    hh = 0;
    fl_font(FL_HELVETICA, big_font_size);
    ww = (hspace_for_buttons - 3 * (hspace / 2)) / 4;
    measure_and_softwrap(str, ww, hh);
    ww += big_font_size * 2;
    xx -= (ww + hspace / 2);
    int button_h = height - yy - padding;

    {
      Fl_Button *o = BUTTON_quit = new Fl_Button(xx, yy, ww, button_h, str);
      o->labelsize(big_font_size);
      o->box(FL_PLASTIC_UP_BOX);
      o->down_box(FL_PLASTIC_DOWN_BOX);
      o->labelfont(FL_BOLD);
      o->callback((Fl_Callback *) confirm_exit_cb);
      o->align(FL_ALIGN_CLIP);
    }
    hspace_for_buttons -= ww + hspace / 2;
    str =
      /* Control button: Reset all settings to their defaults */
      gettext("Defaults");
    hh = 0;
    fl_font(FL_HELVETICA, big_font_size);
    ww = (hspace_for_buttons - 2 * (hspace / 2)) / 3;
    measure_and_softwrap(str, ww, hh);
    ww += big_font_size * 2;
    xx -= (ww + hspace / 2);

    {
      Fl_Button *o = BUTTON_defaults =
        new Fl_Button(xx, yy, ww, button_h, str);
      o->labelsize(big_font_size);
      o->box(FL_PLASTIC_UP_BOX);
      o->down_box(FL_PLASTIC_DOWN_BOX);
      o->labelfont(FL_BOLD);
      o->callback((Fl_Callback *) cb_BUTTON_defaults);
      o->align(FL_ALIGN_CLIP);
    }
    hspace_for_buttons -= ww + hspace / 2;
    str =
      /* Control button: Reset settings to what they were (discard changes) */
      gettext("Reset");
    hh = 0;
    fl_font(FL_HELVETICA, big_font_size);
    ww = (hspace_for_buttons - hspace / 2) / 2;
    measure_and_softwrap(str, ww, hh);
    ww += big_font_size * 2;
    xx -= (ww + hspace / 2);

    {
      Fl_Button *o = BUTTON_reset = new Fl_Button(xx, yy, ww, button_h, str);
      o->labelsize(big_font_size);
      o->box(FL_PLASTIC_UP_BOX);
      o->down_box(FL_PLASTIC_DOWN_BOX);
      o->labelfont(FL_BOLD);
      o->callback((Fl_Callback *) cb_BUTTON_reset);
      o->align(FL_ALIGN_CLIP);
    }
    str =
      /* Control button: Apply changes (save settings to file) */
      gettext("Apply");
    hh = 0;
    fl_font(FL_HELVETICA, big_font_size);
    ww = hspace_for_buttons - ww - hspace / 2;
    measure_and_softwrap(str, ww, hh);
    ww += big_font_size * 2;
    xx -= (ww + hspace / 2);

    {
      Fl_Button *o = BUTTON_apply = new Fl_Button(xx, yy, ww, button_h, str);
      o->labelsize(big_font_size);
      o->box(FL_PLASTIC_UP_BOX);
      o->down_box(FL_PLASTIC_DOWN_BOX);
      o->labelfont(FL_BOLD);
      o->align(FL_ALIGN_CLIP);
      o->callback((Fl_Callback *) cb_BUTTON_apply);
      o->deactivate();
    }

    o->end();
  }


  /* It's kluge to do this here; sorry -bjk 2004-09-04 */

  printf("Loading conf at startup\n");
  load_conf();


  /* Show the window: */

#ifdef WIN32
  /* must set the icon before calling show() */
  WINDOW_tpc->icon((char *) LoadIcon(fl_display, MAKEINTRESOURCE(IDI_ICON1)));
#endif

  /* Make the window resizable up to 1.8 times each direction. */
  WINDOW_tpc->size_range(width, height, width * 1.8, height * 1.8, 0, 0, 0);
  WINDOW_tpc->resizable(WINDOW_tpc);

  WINDOW_tpc->show();

  update_canvas_size();

  return w;
}

/* Like fl_measure(), but will introduce whitespace in `str`
   in case it's too wide.  This happens with languages like
   Japanese and Chinese, which lack explicit whitespace.
   FLTK doesn't appear to soft-wrap those kinds of strings >:-(

   We use `libunibreak` to collect hints about where strings
   should could be safely broken, and "brute-force" various
   attempts until it fits.
*/
void measure_and_softwrap(char *&in_str, int &w, int &h)
{
  int before_w;
  char *brk_str, *str;
  char *brks;
  int break_possibility_cnt;
  size_t input, output, i;
  int last_break_possibility_input, last_break_possibility_output;

  before_w = w;
  fl_measure(in_str, w, h);

  if (w <= before_w)
  {
    /* It fit, no change to `str` required, just return measurements */
    return;
  }

  /* FLTK was unable to word-wrap it */

  if (tpc_locale == NULL)
  {
    /* No locale to send to unibreak; abort! */
    return;
  }

  str = strdup(in_str);

  /* We prefer our word-wrapping, to what FLTK would do
     when it finds any natural spaces in the translated string.
     So we convert them to "_" (so it won't wrap),
     add later will add a hint that we can break here,
     and finally convert any remaining "_" back to " ".

     Motivation: Avoid word-wrapping "XYZXYZ EscapeXYZXYZ."
     like this:
     XYZXYZ
     Escape XYZXYZ.

     And instead allow:
     XYZXYZ Escape
     XYZXYZ.
   */
  for (i = 0; i < strlen(str); i++)
  {
    if (str[i] == ' ')
    {
      str[i] = '_';
    }
  }

  /* Find all the possible break positions */
  brks = (char *) malloc(strlen(str) + 1);
  init_linebreak();
  set_linebreaks_utf8((const utf8_t *) str, strlen(str), tpc_locale, brks);

  break_possibility_cnt = 0;
  for (i = 0; i < strlen(str); i++)
  {
    if (str[i] == '_')
    {
      brks[i] = LINEBREAK_ALLOWBREAK;
    }

    if (brks[i] == LINEBREAK_ALLOWBREAK)
    {
      break_possibility_cnt++;
    }
  }

  if (break_possibility_cnt == 0)
  {
    /* There were none; giving up! */
    return;
  }


  /* We'll assemble a string, and word-wrap as we hit
     the limit of our horizontal space (per fl_measure()) */

  brk_str =
    (char *) malloc(sizeof(char) * (strlen(str) + break_possibility_cnt + 1));

  last_break_possibility_input = -1;
  last_break_possibility_output = -1;

  output = 0;
  for (input = 0; input < strlen(str); input++)
  {
    /* Add the next character */
    brk_str[output++] = str[input];

    if (brks[input] == LINEBREAK_ALLOWBREAK)
    {
      /* Hit a break possibility; See if the text still fits within the available width */
      brk_str[output] = '\0';
      w = before_w;
      h = 0;
      fl_measure(brk_str, w, h);

      if (w > before_w)
      {
        /* It's too wide */
        if (last_break_possibility_input != -1)
        {
          /* Go back and insert a break at our _previous_ break possibility */
          output = last_break_possibility_output;
          brk_str[output++] = ' ';

          /* Rewind input, so we put the rest of the string (up to here) back */
          input = last_break_possibility_input;
        }
        else
        {
          /* No option here! Move on :-( */
          /* No-op */
        }
      }
      else
      {
        /* Keep track of each break possibility as they occur, so we can rewind */
        last_break_possibility_input = input;
        last_break_possibility_output = output;
      }

      brk_str[output] = '\0';
    }
    else
    {
      /* Otherwise, in the middle of a UTF sequence, or within a sequence
         of characters where a break is not permissible */
      /* No-op */
    }
  }
  brk_str[output] = '\0';

  /* Convert any remaining "_" back into " " */
  for (i = 0; i < strlen(brk_str); i++)
  {
    if (brk_str[i] == '_')
    {
      brk_str[i] = ' ';
    }
  }

  /* We have our new string; send it back, along with measurements! */
  in_str = brk_str;
  fl_measure(in_str, w, h);
}

void list_fonts(Fl_Hold_Browser *o) {
  PangoFontMap *fontmap;
  PangoFontFamily **families;
  int i, n_families;
  char * * family_names;
  char locale_font_dir[1024];

  /* Fonts that ship with Tux Paint may not be visible to Tux Paint Config.
     (via Pango, via FontConfig) on certain platforms (e.g., macOS, where
     Tux Paint is a self-contained app, or the Windows Portable ZIP version).

     Therefore we also hard-code that list of fonts into Tux Paint Config.
     They are found in the `tuxpaint` repo under `data/fonts/` and
     `fonts/locale/`.
  */
  // otfinfo --info data/fonts/*.ttf fonts/locale/*.ttf | grep ":Family:" | cut -d ":" -f 3- | sed -e "s/^ \+/    \"/" -e "s/$/\",/" | sort -u
  const char *bundled_fonts[] = {
    "ae_Nice",
    "DejaVu Sans",
    "FreeMono",
    "FreeSans",
    "FreeSerif",
    "Garuda",
    "GJGothicPNSubset",
    "Lohit Gujarati",
    "Nachlieli",
    "Raghindi",
    "SubsetForTuxPaint",
    "Thryomanes",
    "Tsampa Keyboard",
    "TSCu_Comic",
    "TuxPaint Georgian",
    "Vemana2000",
  };
  const size_t n_bundled_fonts = sizeof(bundled_fonts) / sizeof(char *);

  /* In Tux Paint, we ask Pango (via FontConfig) to also look at the
     directory of fonts Tux Paint ships with, so do that here too,
     so we can offer those fonts as options for the `uifont` setting. */
  snprintf(locale_font_dir, sizeof(locale_font_dir), "%s/fonts", TUXPAINT_DATA_PREFIX);
  FcConfigAppFontAddDir(FcConfigGetCurrent(), (const FcChar8 *) locale_font_dir);

  fontmap = pango_ft2_font_map_new();
  pango_font_map_list_families(fontmap, &families, &n_families);

  family_names = (char * *) malloc(sizeof(char *) * (n_families + n_bundled_fonts));
  for (i = 0; i < n_families; i++)
  {
    family_names[i] = strdup(pango_font_family_get_name(families[i]));
    DEBUG_PRINTF("Found font: %s\n", family_names[i]);
  }

  /* Add the list of bundled fonts to the found fonts */
  for(i = 0; i < (int) n_bundled_fonts; i++) {
    family_names[n_families + i] = strdup(bundled_fonts[i]);
    DEBUG_PRINTF("Bundled font: %s\n", bundled_fonts[i]);
  }

  qsort(family_names, n_families + n_bundled_fonts, sizeof(char*), compare_font_family);

  // No "uifont" setting is recorded -- UIFONT_NOCHANGE
  o->add(gettext("[no change]"));
  // User will enter a font name manually -- UIFONT_MANUAL
  o->add(gettext("[manual]"));
  // uifont=default (useful to override something else's "uifont" setting)
  o->add(gettext("default"));
  for (i = 0; i < (int) (n_families + n_bundled_fonts); i++)
  {
    /* If a font is in both found list and bundled list, add it only once.
     * We can do this by just comparing the font name at the current index to
     * the last index because the list is already sorted. */
    if(i > 0 && strcmp(family_names[i], family_names[i-1]) != 0)
    {
      o->add(family_names[i]); // uifont=FONT NAME
      DEBUG_PRINTF("Unique font: %s\n", family_names[i]);
    }
  }
}

int compare_font_family(const void *a, const void *b)
{
  return strcasecmp(*(char * const*) a, *(char * const*) b);
}

/**
 * Grab the value of a config line, trimming quotes from the
 * beginning and end, if they exist.
 *
 * @param char * config_line -- the full config line (e.g., `something="the value"`)
 * @param char * setting_name -- the name of the setting (e.g., `something`)
 * @return -- a new char * containing the value of the setting (e.g., `the value`), sans quotes
 *   (caller must free this!)
 */
char * grab_and_dequote_arg(char * config_line, const char * setting_name) {
  char * str_ptr, * str_return;

  // printf("IN=<%s>, <%s>\n", config_line, setting_name);

  str_ptr = strstr(config_line, setting_name);
  if (str_ptr == NULL)
  {
    /* Shouldn't be possible! Caller should have checked first! */
    return strdup("");
  }

  str_ptr += strlen(setting_name);

  if (str_ptr[0] != '=')
  {
    /* Shouldn't be possible! [ditto] */
    return strdup("");
  }
  str_ptr++; // Eat the '='

  if (str_ptr[0] == '\"')
    str_ptr++; // Eat the (optional) starting '"'

  str_return = strdup(str_ptr);
  if (str_return[strlen(str_return) - 1] == '\"')
  {
    str_return[strlen(str_return) - 1] = '\0'; // Eat the (optional) closing '"'
  }

  // printf("OUT=<%s>\n", str_return);

  return str_return;
}

