///////////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2023 Jon Beniston, M7RCE <jon@beniston.com>                     //
//                                                                               //
// This program is free software; you can redistribute it and/or modify          //
// it under the terms of the GNU General Public License as published by          //
// the Free Software Foundation as version 3 of the License, or                  //
// (at your option) any later version.                                           //
//                                                                               //
// This program is distributed in the hope that it will be useful,               //
// but WITHOUT ANY WARRANTY; without even the implied warranty of                //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the                  //
// GNU General Public License V3 for more details.                               //
//                                                                               //
// You should have received a copy of the GNU General Public License             //
// along with this program. If not, see <http://www.gnu.org/licenses/>.          //
///////////////////////////////////////////////////////////////////////////////////

#include <QDebug>

#include "psk31.h"

// ASCII varicode encoding
// From http://www.aintel.bi.ehu.es/psk31.html
const QStringList PSK31Varicode::m_varicode = {
    "1010101011",
    "1011011011",
    "1011101101",
    "1101110111",
    "1011101011",
    "1101011111",
    "1011101111",
    "1011111101",
    "1011111111",
    "11101111",
    "11101",
    "1101101111",
    "1011011101",
    "11111",
    "1101110101",
    "1110101011",
    "1011110111",
    "1011110101",
    "1110101101",
    "1110101111",
    "1101011011",
    "1101101011",
    "1101101101",
    "1101010111",
    "1101111011",
    "1101111101",
    "1110110111",
    "1101010101",
    "1101011101",
    "1110111011",
    "1011111011",
    "1101111111",
    "1",
    "111111111",
    "101011111",
    "111110101",
    "111011011",
    "1011010101",
    "1010111011",
    "101111111",
    "11111011",
    "11110111",
    "101101111",
    "111011111",
    "1110101",
    "110101",
    "1010111",
    "110101111",
    "10110111",
    "10111101",
    "11101101",
    "11111111",
    "101110111",
    "101011011",
    "101101011",
    "110101101",
    "110101011",
    "110110111",
    "11110101",
    "110111101",
    "111101101",
    "1010101",
    "111010111",
    "1010101111",
    "1010111101",
    "1111101",
    "11101011",
    "10101101",
    "10110101",
    "1110111",
    "11011011",
    "11111101",
    "101010101",
    "1111111",
    "111111101",
    "101111101",
    "11010111",
    "10111011",
    "11011101",
    "10101011",
    "11010101",
    "111011101",
    "10101111",
    "1101111",
    "1101101",
    "101010111",
    "110110101",
    "101011101",
    "101110101",
    "101111011",
    "1010101101",
    "111110111",
    "111101111",
    "111111011",
    "1010111111",
    "101101101",
    "1011011111",
    "1011",
    "1011111",
    "101111",
    "101101",
    "11",
    "111101",
    "1011011",
    "101011",
    "1101",
    "111101011",
    "10111111",
    "11011",
    "111011",
    "1111",
    "111",
    "111111",
    "110111111",
    "10101",
    "10111",
    "101",
    "110111",
    "1111011",
    "1101011",
    "11011111",
    "1011101",
    "111010101",
    "1010110111",
    "110111011",
    "1010110101",
    "1011010111",
    "1110110101",
    "1110111101",
    "1110111111",
    "1111010101",
    "1111010111",
    "1111011011",
    "1111011101",
    "1111011111",
    "1111101011",
    "1111101101",
    "1111101111",
    "1111110101",
    "1111110111",
    "1111111011",
    "1111111101",
    "1111111111",
    "10101010101",
    "10101010111",
    "10101011011",
    "10101011101",
    "10101011111",
    "10101101011",
    "10101101101",
    "10101101111",
    "10101110101",
    "10101110111",
    "10101111011",
    "10101111101",
    "10101111111",
    "10110101011",
    "10110101101",
    "10110101111",
    "10110110101",
    "10110110111",
    "10110111011",
    "10110111101",
    "10110111111",
    "10111010101",
    "10111010111",
    "10111011011",
    "10111011101",
    "10111011111",
    "10111101011",
    "10111101101",
    "10111101111",
    "10111110101",
    "10111110111",
    "10111111011",
    "10111111101",
    "10111111111",
    "11010101011",
    "11010101101",
    "11010101111",
    "11010110101",
    "11010110111",
    "11010111011",
    "11010111101",
    "11010111111",
    "11011010101",
    "11011010111",
    "11011011011",
    "11011011101",
    "11011011111",
    "11011101011",
    "11011101101",
    "11011101111",
    "11011110101",
    "11011110111",
    "11011111011",
    "11011111101",
    "11011111111",
    "11101010101",
    "11101010111",
    "11101011011",
    "11101011101",
    "11101011111",
    "11101101011",
    "11101101101",
    "11101101111",
    "11101110101",
    "11101110111",
    "11101111011",
    "11101111101",
    "11101111111",
    "11110101011",
    "11110101101",
    "11110101111",
    "11110110101",
    "11110110111",
    "11110111011",
    "11110111101",
    "11110111111",
    "11111010101",
    "11111010111",
    "11111011011",
    "11111011101",
    "11111011111",
    "11111101011",
    "11111101101",
    "11111101111",
    "11111110101",
    "11111110111",
    "11111111011",
    "11111111101",
    "11111111111",
    "101010101011",
    "101010101101",
    "101010101111",
    "101010110101",
    "101010110111",
    "101010111011",
    "101010111101",
    "101010111111",
    "101011010101",
    "101011010111",
    "101011011011",
    "101011011101",
    "101011011111",
    "101011101011",
    "101011101101",
    "101011101111",
    "101011110101",
    "101011110111",
    "101011111011",
    "101011111101",
    "101011111111",
    "101101010101",
    "101101010111",
    "101101011011"
};

PSK31Encoder::PSK31Encoder()
{
}

bool PSK31Encoder::encode(QChar c, unsigned &bits, unsigned int &bitCount)
{
    bits = 0;
    bitCount = 0;

    unsigned char ch = (unsigned char) c.toLatin1();
    QString code = PSK31Varicode::m_varicode[ch];

    addCode(bits, bitCount, code);
    return true;
}

void PSK31Encoder::addCode(unsigned& bits, unsigned int& bitCount, const QString& code) const
{
    int codeBits = 0;
    unsigned int codeLen = code.size();

    for (unsigned int i = 0; i < codeLen; i++) {
        codeBits |= (code[i] == '1' ? 1 : 0) << i;
    }

    addStartBits(bits, bitCount);
    addBits(bits, bitCount, codeBits, codeLen);
    addStopBits(bits, bitCount);
}

void PSK31Encoder::addStartBits(unsigned& bits, unsigned int& bitCount) const
{
    // Start bit is 0
    addBits(bits, bitCount, 0, 1);
}

void PSK31Encoder::addStopBits(unsigned& bits, unsigned int& bitCount) const
{
    // Stop bit is 0
    addBits(bits, bitCount, 0, 1);
}

void PSK31Encoder::addBits(unsigned& bits, unsigned int& bitCount, int data, int count) const
{
    bits |= data << bitCount;
    bitCount += count;
}
