/*  display.c
 *
 *  Display/Rendering functions for xfhell
 */

/*
 *  xfhell: An application to transmit and receive
 *  Feld Hell signals using a computer's sound card
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  Draw_Column()
 *
 *  Draws a character's column in the Rx drawing area
 */

  gboolean
Draw_Column( unsigned char *column )
{
  /* Index to current pixel in pixbuf */
  static guchar	*chr_pix;

  /* Char row height, drawing area column and row */
  static int
	scrn_fill,	/* How much of Rx screen is filled */
	row_height,	/* Total height of rows on screen */
	chr_col=0,	/* Current column when painting characters */
	chr_row;	/* Current row when painting characters */

  int
	idx, i, j, temp,
	pix_idx; /* Index to pixel in column */

  static unsigned char *last_column = NULL;


  /* Clear Receive window */
  if( isFlagSet(CLEAR_RX_WINDOW) )
  {
	ClearFlag( CLEAR_RX_WINDOW );
	if( chr_col > 2 ) Save_Pixbuf();
	Clear_Pixbuf( &receive_pixbuf, 0xffffffff );
	row_height = rc_data.num_rows * rc_data.bitmap_height;
	chr_col = 2;
	scrn_fill = 0;

	/* Allocate/clear memory for column buffer */
	if( !mem_realloc( (void **)&last_column,
		  sizeof(unsigned char) * (size_t)rc_data.bitmap_height) )
	{
	  ClearFlag( RECEIVE_MODE );
	  ClearFlag(
		  TRANSMIT_MODE  |
		  TRANSMIT_MACRO |
		  TRANSMIT_TAG   |
		  TRANSMIT_KEYBD );
	  return( FALSE );
	}
	memset( last_column, 0,
		sizeof(unsigned char) * (size_t)rc_data.bitmap_height );

	/* Put cursor and char line at bottom of display */
	chr_row = gbl_drawingarea->allocation.height-2 - row_height;
	chr_pix = receive_pixbuf.pixels +
	  (gbl_drawingarea->allocation.height-2 - row_height) *
	  receive_pixbuf.rowstride + 2*receive_pixbuf.n_channels;
  } /* if( isFlagSet(CLEAR_RX_WINDOW) ) */

  /*** Draw a character column ***/
  /* Number of columns depends on dot size: */
  /* 2 for 2x2, 3 for 3x3 and 4 for 4x4 pixels */
  for( idx = 0; idx < rc_data.dot_size; idx++ )
  {
	/* Draw character dot pixels to column pixbuf */
	for( pix_idx = 0; pix_idx < rc_data.bitmap_height; pix_idx++ )
	{
	  unsigned char grlev; /* dot gray level */

	  i = receive_pixbuf.rowstride *
		(rc_data.bitmap_height - pix_idx - 1);
	  grlev = 0xff - column[pix_idx];
	  chr_pix[i  ] = grlev;
	  chr_pix[i+1] = grlev;
	  chr_pix[i+2] = grlev;
	}

	/* Draw 2nd raw if traditional 2-row mode is in use */
	if( isFlagClear(AUTO_DESKEW) )
	{
	  for( pix_idx = 0; pix_idx < rc_data.bitmap_height; pix_idx++ )
	  {
		unsigned char grlev; /* dot gray level */

		i = receive_pixbuf.rowstride * (row_height - pix_idx - 1);
		grlev = 0xff - last_column[pix_idx];
		chr_pix[i  ] = grlev;
		chr_pix[i+1] = grlev;
		chr_pix[i+2] = grlev;
	  }
	}

	/* Draw new column */
	gtk_widget_queue_draw_area(
		gbl_drawingarea, chr_col, chr_row, 1, row_height );

	chr_col++;
	chr_pix += receive_pixbuf.n_channels;

	/* Move back to beginning of row when reaching end of row */
	if( chr_col >= receive_pixbuf.width - 2 )
	{
	  chr_col = rc_data.bitmap_width + 2;
	  chr_pix =
		receive_pixbuf.pixels +
		chr_col * receive_pixbuf.n_channels +
		chr_row * receive_pixbuf.rowstride;

	  /* Save pixbuf when screen filled */
	  scrn_fill += row_height;
	  if( scrn_fill >= gbl_drawingarea->allocation.height-4 )
	  {
		Save_Pixbuf();
		scrn_fill = 0;
	  }

	  /* Scroll up display one row */
	  gdk_pixbuf_copy_area(
		  receive_pixbuf.pixbuf,
		  2, row_height+2,
		  gbl_drawingarea->allocation.width-4,
		  gbl_drawingarea->allocation.height-4 - row_height,
		  receive_pixbuf.pixbuf, 2, 2 );

	  /* Redraw last block to beginning of new line */
	  gdk_pixbuf_copy_area(
		  receive_pixbuf.pixbuf,
		  receive_pixbuf.width-2 - rc_data.bitmap_width,
		  chr_row,
		  rc_data.bitmap_width,
		  row_height,
		  receive_pixbuf.pixbuf,
		  2,
		  chr_row );

	  /* Clear new row */
	  for( pix_idx = 0; pix_idx < row_height; pix_idx++ )
	  {
		i = (gbl_drawingarea->allocation.height-3 - pix_idx) *
		  receive_pixbuf.rowstride;
		i += chr_col * receive_pixbuf.n_channels;
		temp = gbl_drawingarea->allocation.width-3;
		for( j = chr_col; j < temp; j++ )
		{
		  i += receive_pixbuf.n_channels;
		  receive_pixbuf.pixels[i  ] = 0xff;
		  receive_pixbuf.pixels[i+1] = 0xff;
		  receive_pixbuf.pixels[i+2] = 0xff;
		}
	  }

	  gtk_widget_queue_draw( gbl_drawingarea );

	} /* if( chr_col >= gbl_width-2 ) */

  } /* for( idx = 0; idx < rc_data.dot_wdt; idx++ ) */

  memcpy( last_column, column, (size_t)rc_data.bitmap_height );
  return( TRUE );

} /* Draw_Column() */

/*------------------------------------------------------------------------*/

/* DFT_Bin_Value()
 *
 * Calculates DFT bin values with auto level control
 */
  int
DFT_Bin_Value( int sum_i, int sum_q, gboolean reset )
{
 /* Value of dft output "bin" */
  static int bin_val = 0;

  /* Maximum value of dft bins */
  static int bin_max = 1000, max = 0;


  /* Calculate sliding window average of max bin value */
  if( reset )
  {
	bin_max = max;
	if( !bin_max ) bin_max = 1;
	max = 0;
  }
  else
  {
	/* Calculate average signal power at each frequency (bin) */
	bin_val  = bin_val * AMPL_AVE_MUL;
	bin_val += sum_i * sum_i + sum_q * sum_q;
	bin_val /= AMPL_AVE_WIN;

	/* Record max bin value */
	if( max < bin_val ) max = bin_val;

	/* Scale bin values to 255 depending on max value */
	int ret = ( 255 * bin_val ) / bin_max;
	if( ret > 255 ) ret = 255;
	return( ret );
  }

  return( 0 );
} /* DFT_Bin_Value() */

/*------------------------------------------------------------------------*/

/* Display_Waterfall()
 *
 * Displays audio spectrum as "waterfall"
 */

  void
Display_Waterfall( void )
{
  int
	idh, idv,  /* Index to hor. and vert. position in warterfall */
	pixel_val, /* Greyscale value of pixel derived from dft o/p  */
	dft_idx;   /* Index to dft output array */

  int temp;

  /* Pointer to current pixel */
  static guchar *pix;


  /* Draw a vertical white line in waterfall at detector's freq. */
  pix = wfall_pixbuf.pixels +
	wfall_pixbuf.rowstride + wfall_pixbuf.n_channels;
  pix += wfall_pixbuf.n_channels * rc_data.center_line;
  pix[0] = pix[1] = pix[2] = 0xff;

  /* Copy each line of waterfall to next one */
  temp = wfall_pixbuf.height - 2;
  for( idv = temp; idv > 0; idv-- )
  {
	pix = wfall_pixbuf.pixels +
	  wfall_pixbuf.rowstride * idv + wfall_pixbuf.n_channels;

	for( idh = 0; idh < wfall_pixbuf.width; idh++ )
	{
	  *pix = *(pix - wfall_pixbuf.rowstride); pix++;
	  *pix = *(pix - wfall_pixbuf.rowstride); pix++;
	  *pix = *(pix - wfall_pixbuf.rowstride);
	  pix += wfall_pixbuf.n_channels - 2;
	}
  }

  /* Go to top left of pixbuf */
  pix = wfall_pixbuf.pixels;

  /* First (DC) dft bin not used */
  gbl_bin_ave[0] = 0;

  /* Do the DFT on input array */
  Idft( DFT_INPUT_SIZE, wfall_pixbuf.width );
  for( dft_idx = 0; dft_idx < wfall_pixbuf.width; dft_idx++ )
  {
	/* Calculate vector magnitude of
	 * signal at each freq. ("bin") */
	dft_out_r[dft_idx] /= 1024;
	dft_out_i[dft_idx] /= 1024;
	pixel_val = DFT_Bin_Value(
		dft_out_r[dft_idx], dft_out_i[dft_idx], FALSE );

	/* Calculate average bin values */
	gbl_bin_ave[dft_idx] = pixel_val;

	/* Color code signal strength */
	int n;
	if( pixel_val < 85 )
	{
	  pix[0] = 0;
	  pix[1] = 0;
	  pix[2] = 3 + (guchar)pixel_val * 3;
	}
	else if( pixel_val < 170 )
	{
	  n = pixel_val - 85;
	  pix[0] = 0;
	  pix[1] = 3 + (guchar)n * 3;
	  pix[2] = 255 - (guchar)n * 3;
	}
	else
	{
	  n = pixel_val - 170;
	  pix[0] = (guchar)pixel_val;
	  pix[1] = 255;
	  pix[2] = 0;
	}

	pix += wfall_pixbuf.n_channels;

  } /* for( dft_idx = 0; dft_idx < dft_end; dft_idx++ ) */

  /* Reset function */
  pixel_val = DFT_Bin_Value( 0, 0, TRUE );

  /* At last draw waterfall */
  gtk_widget_queue_draw( gbl_scope );

} /* Display_Waterfall() */

/*------------------------------------------------------------------------*/

/* Create_Pixbuff()
 *
 * Creates a GDK pixbuf for the receive window
 */
  gboolean
Create_Pixbuf( GtkWidget *drawingarea )
{
  /* Last drawing area height */
  static int
	last_wdth = 0,
	last_hght = 0;

  /* Don't create new pixbuf if size has not changed */
  if( (last_hght == drawingarea->allocation.height) &&
	  (last_wdth == drawingarea->allocation.height) )
	return( TRUE );
  last_hght = drawingarea->allocation.height;
  last_wdth = drawingarea->allocation.width;

  /* Destroy existing pixbuff */
  if( receive_pixbuf.pixbuf != NULL )
  {
	g_object_unref( G_OBJECT(receive_pixbuf.pixbuf) );
	receive_pixbuf.pixbuf = NULL;
  }

  /* Create pixbuf */
  receive_pixbuf.pixbuf = gdk_pixbuf_new(
	  GDK_COLORSPACE_RGB, FALSE, 8,
	  drawingarea->allocation.width,
	  drawingarea->allocation.height );
  if( receive_pixbuf.pixbuf == NULL ) return( FALSE );

  receive_pixbuf.pixels =
	gdk_pixbuf_get_pixels( receive_pixbuf.pixbuf );
  receive_pixbuf.width =
	gdk_pixbuf_get_width ( receive_pixbuf.pixbuf )-1;
  receive_pixbuf.height =
	gdk_pixbuf_get_height( receive_pixbuf.pixbuf );
  receive_pixbuf.rowstride =
	gdk_pixbuf_get_rowstride( receive_pixbuf.pixbuf );
  receive_pixbuf.n_channels =
	gdk_pixbuf_get_n_channels( receive_pixbuf.pixbuf );

  Clear_Pixbuf( &receive_pixbuf, 0xffffffff );
  return( TRUE );

} /* Create_Pixbuff() */

/*------------------------------------------------------------------------*/

/*  Clear_Pixbuf()
 *
 *  Initializes pixbuf of Receive window
 */

  void
Clear_Pixbuf( pixbuf_data_t *pixbuf, guint32 fg )
{
  int i, temp;
  guchar *p1, *p2;

  gdk_pixbuf_fill( pixbuf->pixbuf, fg );

  /* Draw a box around pixbuf */
  p1 = pixbuf->pixels;
  p2 = pixbuf->pixels +
	(pixbuf->height-1) * pixbuf->rowstride;
  for( i = 0; i < pixbuf->width; i++ )
  {
	p1[0] = p1[1] = p1[2] = 0;
	p1 += pixbuf->n_channels;
	p2[0] = p2[1] = p2[2] = 0;
	p2 += pixbuf->n_channels;
  }

  p1 = pixbuf->pixels + pixbuf->rowstride;
  p2 = pixbuf->pixels + pixbuf->rowstride +
	(pixbuf->width-1) * pixbuf->n_channels;
  temp = pixbuf->height-1;
  for( i = 0; i < temp; i++ )
  {
	p1[0] = p1[1] = p1[2] = 0;
	p1 += pixbuf->rowstride;
	p2[0] = p2[1] = p2[2] = 0;
	p2 += pixbuf->rowstride;
  }

  gtk_widget_queue_draw( gbl_drawingarea );

} /* Clear_Pixbuf() */

/*------------------------------------------------------------------------*/

/*  Print_Character()
 *
 *  Prints a character to a text view port
 */

void
Print_Character(
	int printchr,
	GtkTextBuffer *text_buffer,
	GtkWidget     *scrolledwindow,
	GtkTextIter   *iter )
{
  GtkAdjustment *adjustment;
  char char2text[2];

  /* Delete a char on backspace */
  if( printchr == BS )
  {
	GtkTextIter start;

	gtk_text_buffer_get_iter_at_offset( text_buffer, &start,
		gtk_text_buffer_get_char_count(text_buffer) -1 );
	gtk_text_buffer_delete( text_buffer, &start, iter );
	return;
  }

  /* Convert char to "text" */
  char2text[0] = (char) printchr;
  char2text[1] = '\0';

  /* Print character */
  gtk_text_buffer_insert(
	  text_buffer, iter,
	  char2text, -1 );

  /* Record to QSO record file */
  if( isFlagSet(RECORD_QSO) )
	fprintf( rc_data.qso_record_fp, "%c", printchr );

  /* Scroll Text View to bottom on LF */
  if( (printchr == LF) ||
	  (printchr == GDK_Return) )
  {
	adjustment = gtk_scrolled_window_get_vadjustment
	  ( GTK_SCROLLED_WINDOW(scrolledwindow) );
	gtk_adjustment_set_value( adjustment, adjustment->upper );
	if( isFlagSet(RECORD_QSO) )
	  fprintf( rc_data.qso_record_fp, "Tx-> " );
  }

} /* Print_Character() */

/*------------------------------------------------------------------------*/

/*  Display_Signal()
 *
 *  Updates the signal scope display */

  void
Display_Signal( int plot )
{
  /* Points to plot */
  static GdkPoint *points = NULL;
  static int points_idx = 0;

  static cairo_t *cr = NULL;


  /* Initialize on first call */
  if( points == NULL )
  {
	if( !mem_alloc((void **)&points,
		  sizeof(GdkPoint) * (size_t)wfall_pixbuf.width) )
	  return;
  }

  /* Needed to re-initialze after Rx window size change */
  if( isFlagSet(INITIALIZE_CAIRO) )
  {
	if( cr != NULL )
	{
	  cairo_destroy( cr );
	  cr = NULL;
	}
	cr = gdk_cairo_create( gbl_scope->window );
	ClearFlag( INITIALIZE_CAIRO );
  }

  /* Save values to be plotted (y is scaled) */
  points[points_idx].y = wfall_pixbuf.height - plot - 1;
  points[points_idx].x = points_idx;

  /* Recycle buffer idx when full and plot */
  if( ++points_idx == wfall_pixbuf.width )
  {
	int idx;

	/* Draw scope background */
	cairo_set_source_rgb( cr, 0.0, .3, 0.0 );
	cairo_rectangle(
		cr, 0.0, 0.0,
		(double)wfall_pixbuf.width,
		(double)wfall_pixbuf.height );
	cairo_fill( cr );

	/* Plot signal graph */
	cairo_set_source_rgb( cr, 0.0, 1.0, 0.0 );
	cairo_move_to( cr,
		(double)points[0].x,
		(double)points[0].y );
	for( idx = 1; idx < points_idx; idx++ )
	  cairo_line_to( cr,
		  (double)points[idx].x,
		  (double)points[idx].y );

	/* Stroke paths */
	cairo_stroke( cr );

	points_idx = 0;

  } /* if( ++points_idx == wfall_pixbuf.width ) */

} /* Display_Signal( void ) */

/*------------------------------------------------------------------------*/

/*  Set_TxRx_Labels()
 *
 *  Sets up the labels in the Tx/Rx frames
 */

  void
Set_TxRx_Labels( void )
{
  char lbl[80];

  /* Set Tx button label */
  if( isFlagSet(TRANSMIT_KEYBD) ||
	  isFlagSet(TRANSMIT_MACRO) ||
	  isFlagSet(KEYBD_BUSY) )
	gtk_label_set_markup(
		GTK_LABEL(lookup_widget(main_window, "xmit_status")),
		XMIT_ON );
  else gtk_label_set_markup(
	  GTK_LABEL(lookup_widget(main_window, "xmit_status")),
	  XMIT_OFF );

  /* Set Tx fonts label */
  gtk_entry_set_text(
	  GTK_ENTRY(lookup_widget(main_window, "font_entry")),
	  rc_data.font_data.font_name );

  /* Set Tx/Rx mode and baud rate */
  if( isFlagSet(MODE_FELDHELL) )
	snprintf( lbl, sizeof(lbl), "FeldHell @ %1.2fBd", rc_data.baud_rate );
  else if( isFlagClear(MODE_FELDHELL) )
	snprintf( lbl, sizeof(lbl), "FMHell @ %1.2fBd", rc_data.baud_rate );
  gtk_entry_set_text(
	  GTK_ENTRY(lookup_widget(main_window, "mode_entry")), lbl );

  /* Set Rx button label */
  if( isFlagSet(RECEIVE_MODE) )
  {
	if( isFlagSet(KEYBD_BUSY) )
	  gtk_label_set_markup( GTK_LABEL(lookup_widget(
			  main_window, "rcve_status")), RECV_SBY );
	else
	{
	  if( isFlagSet(AUTO_DESKEW) )
		snprintf( lbl, sizeof(lbl), RECV_ON, _("Deskew") );
	  else
		snprintf( lbl, sizeof(lbl), RECV_ON, _("Normal") );

	  gtk_label_set_markup( GTK_LABEL(lookup_widget(
			  main_window, "rcve_status")), lbl );
	}
  }
  else gtk_label_set_markup( GTK_LABEL(lookup_widget(
		  main_window, "rcve_status")), RECV_OFF );

  /* Close cat if tx/rx off */
  if( isFlagClear(RECEIVE_MODE) &&
	  isFlagClear(TRANSMIT_MACRO) &&
	  isFlagClear(TRANSMIT_KEYBD) )
	Close_Tcvr_Serial();

} /* Set_TxRx_Labels() */

/*------------------------------------------------------------------------*/

