

#ifndef CGAL_TEST_MODULAR_GCD_H
#define CGAL_TEST_MODULAR_GCD_H

#include <CGAL/Polynomial.h>
#include <CGAL/gen_polynomials.h>
#include <CGAL/Sqrt_extension.h>

//#define WITH_OUTPUT 1


namespace CGAL {
namespace internal {

//fwd
template<class AT>
void test_special_polynomials(Integral_domain_tag);

template<class NT>
void gcd_utcf_test(const NT& f, const NT& g, const NT& d) {
    NT tmp = CGAL::internal::MY_FUNCTION_CALL(f, g);
#ifdef WITH_OUTPUT
    std::cout << "\nf(x) = " << f;
    std::cout << "\ng(x) = " << g;
    std::cout << "\ngcd_utcf(f,g) = " << tmp;
    std::cout << "\nd        = " << CGAL::canonicalize(d) << "\n";
#endif
    assert(
            CGAL::canonicalize(tmp)
            ==
            CGAL::canonicalize(d) );
}

template<class AT>
void test_modular_gcd(Unique_factorization_domain_tag) {

    ::CGAL::IO::set_pretty_mode(std::cout);

    typedef typename AT::Integer Integer;
    typedef Integer NT;

    // testing univariate polynomials with integer coefficients
    typedef CGAL::Polynomial<Integer> int_POLY;
    typedef Integer NT;

    // special cases:
    gcd_utcf_test(int_POLY(0),int_POLY(0),int_POLY(1));
    gcd_utcf_test(int_POLY(0),int_POLY(NT(4),NT(2)),int_POLY(NT(4),NT(2)));
    gcd_utcf_test(int_POLY(NT(4),NT(2)),int_POLY(0),int_POLY(NT(4),NT(2)));

//  std::cout<<" after special cases "<< std::endl;
    int_POLY a, b, c, d, e, ac, bc;
    a = int_POLY(Integer(1), Integer(-5), Integer(1));
    b = int_POLY(Integer(-7), Integer(3));
    c = int_POLY(Integer(6), Integer(4));
    d = int_POLY(Integer(3), Integer(2));
    e = int_POLY(1);
    ac = a*c; bc = b*c;
    // a real gcd exists
    gcd_utcf_test(ac, bc, c);
    // there's no gcd
//        std::cout<<" no gcd "<<std::endl;
    gcd_utcf_test(ac, b, e);
    // g is already a divisor of f
//        std::cout<<" after no gcd "<<std::endl;
    gcd_utcf_test(ac, c, c);
    // testing gcd_utcf
    gcd_utcf_test(ac, bc, d);

    int_POLY  f, g, h, result;
    int l;

    // random polynomials with integer coefficients
    for(l=0;l<2;l++){
//        std::cout<<"run "<<l<< std::endl;
        do{
            f = rand_Poly_int<Integer>(my_random.get_int(10,1000));
            g = rand_Poly_int<Integer>(my_random.get_int(10,1000));
            //      std::cout<<"f "<<f<<std::endl;
            //      std::cout<<"g "<<g<<std::endl;
            result = CGAL::gcd(f, g);
//      std::cout<<" result : "<<result<<std::endl;
        }
        while(result != int_POLY(1));

        d = rand_Poly_int<Integer>(my_random.get_int(10,1000));
        int_POLY p1 = f*d;
        int_POLY p2 = g*d;
        gcd_utcf_test(p1, p2, d);
    }


}

template<class AT>
void test_modular_gcd(Integral_domain_tag) {

    test_modular_gcd<AT>(Unique_factorization_domain_tag());
    test_special_polynomials<AT>(Integral_domain_tag());


    ::CGAL::IO::set_pretty_mode(std::cout);

    typedef typename AT::Integer Integer;
    typedef CGAL::Sqrt_extension<Integer  ,Integer> int_EXT_1;
    typedef CGAL::Sqrt_extension<int_EXT_1,Integer> int_EXT_2;

    {
        // testing univariate polynomials with sqrt-extensions of integers as coefficients
        typedef CGAL::Polynomial<int_EXT_1> int_EXT_1_POLY;
        int_EXT_1_POLY a, b, c, d, e, ac, bc;

        a = int_EXT_1_POLY(int_EXT_1(1), int_EXT_1(-5,-1,2), int_EXT_1(1));
        b = int_EXT_1_POLY(int_EXT_1(-7,-3,2), int_EXT_1(3,-1,2));
        c = int_EXT_1_POLY(int_EXT_1(3), int_EXT_1(2,1,2));
        d = int_EXT_1_POLY(int_EXT_1(6,-3,2), int_EXT_1(2));
        e = int_EXT_1_POLY(1);
        ac = a*c; bc = b*c;

        // there's no gcd
        gcd_utcf_test(a, b, e);
        // a real gcd exists
        gcd_utcf_test(ac, bc, d);
        // g is already a divisor of f
        gcd_utcf_test(ac, c, d);
        int_EXT_1_POLY  f, g, h, result;
        int l;
        Integer root;

        // random polynomials with sqrt-extensions of integers as coefficients
        for(l=0;l<2;l++){
//        std::cout<<"run "<<l<< std::endl;
            root = CGAL::abs(rand_int<Integer>(my_random.get_int(10,1000)));
            do{
                f = rand_Poly_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000), root);
                g = rand_Poly_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000), root);
                //      std::cout<<"f "<<f<<std::endl;
                //      std::cout<<"g "<<g<<std::endl;
                result = CGAL::internal::gcd_utcf(f, g);
                result = CGAL::canonicalize(result);
            }
            while(result != int_EXT_1_POLY(1));

            d = rand_Poly_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000), root);
            int_EXT_1_POLY p1 = f*d;
            int_EXT_1_POLY p2 = g*d;
        }

    }{
        // testing univariate polynomials with nested sqrt-extensions of integers as coefficients
        typedef CGAL::Polynomial<int_EXT_2> int_EXT_2_POLY;
        int_EXT_2_POLY a, b, c, d, e, ac, bc;
        a = int_EXT_2_POLY(
                int_EXT_2(int_EXT_1(1,1,2), int_EXT_1(3,-1,2), 3),
                int_EXT_2(int_EXT_1(5,1,2)),
                int_EXT_2(int_EXT_1(3,1,2), int_EXT_1(1,1,2), 3));
        b = int_EXT_2_POLY(
                int_EXT_2(int_EXT_1(4), int_EXT_1(2,1,2), 3),
                int_EXT_2(int_EXT_1(7,1,2), int_EXT_1(2,-3,2), 3));
        c = int_EXT_2_POLY(
                int_EXT_2(int_EXT_1(3), int_EXT_1(-1,-3,2), 3),
                int_EXT_2(int_EXT_1(2,1,2)));
        d = int_EXT_2_POLY(
                int_EXT_2(int_EXT_1(6,-3,2), int_EXT_1(4,-5,2), 3),
                int_EXT_2(int_EXT_1(2)));
        e = int_EXT_2_POLY(1);
        ac = a*c; bc = b*c;

        // a real gcd exists
        gcd_utcf_test(ac, bc, d);
        // there's no gcd
        gcd_utcf_test(a, b, e);
        // g is already a divisor of f
        gcd_utcf_test(ac, c, d);


        int_EXT_2_POLY  f, g, h, result;
        int l;
        Integer root;
// random polynomials with nested sqrt-extensions of integers as coefficients
        for(l=0;l<2;l++){
            root = CGAL::abs(rand_int<Integer>(my_random.get_int(10,1000)));
            do{
                f = int_EXT_2_POLY(
                        int_EXT_2(rand_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000),root),
                                rand_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000),root),
                                root),
                        int_EXT_2(rand_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000),root),
                                rand_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000),root),
                                root),
                        int_EXT_2(rand_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000),root),
                                rand_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000),root),
                                root));
                g = int_EXT_2_POLY(
                        int_EXT_2(rand_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000),root),
                                rand_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000),root),
                                root),
                        int_EXT_2(rand_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000),root),
                                rand_sqrt<int_EXT_1, Integer>(my_random.get_int(10,1000),root),
                                root));
                result = CGAL::internal::gcd_utcf(f, g);
                result = CGAL::canonicalize(result);
            }
            while(result != int_EXT_2_POLY(1));

            d = int_EXT_2_POLY(
                    int_EXT_2(rand_sqrt<int_EXT_1, Integer>(my_random.get_int(1,10),root),
                            rand_sqrt<int_EXT_1, Integer>(my_random.get_int(1,10),root),
                            root),
                    int_EXT_2(rand_sqrt<int_EXT_1, Integer>(my_random.get_int(1,10),root),
                            rand_sqrt<int_EXT_1, Integer>(my_random.get_int(1,10),root),
                            root));
            int_EXT_2_POLY p1 = f*d;
            int_EXT_2_POLY p2 = g*d;
            gcd_utcf_test(p1, p2, d);
        }
    }
}

template<class AT>
void test_special_polynomials(Integral_domain_tag) {


    ::CGAL::IO::set_pretty_mode(std::cout);

    typedef typename AT::Integer Integer;
    typedef CGAL::Sqrt_extension<Integer  ,Integer> int_EXT_1;
    typedef CGAL::Polynomial<int_EXT_1> int_EXT_1_POLY;

    // testing univariate polynomials with sqrt-extensions of integers as coefficients,
    // where the dfai is needed

    int_EXT_1_POLY P1,P2,G,I,D,W,P;
    {
        std::istringstream
            is_p1("P[4(0,EXT[1191647116146200485090188568127380736530674528772716584920819458323633032566699511811,657936834267592387452731176788217054846573822146742212095985199712,3280386192960530717081986793605014784])(1,EXT[759424558855391695024240575482533939773843120889806256373905023569217133511301781290,419502294919491298013196952655974176267207680459296906162290689505,3280386192960530717081986793605014784])(2,EXT[207121272471980036934512743470553015910295718969711273716423962615753681708330678901,107258620343494557504310989654699004017500109524710005785923153023,3280386192960530717081986793605014784])(3,EXT[33236131552280379517509784575939101833045402729992992622046632417300289693523235822,19224422112534803646410890459067644199273719602134452862082571299,3280386192960530717081986793605014784])(4,EXT[6371372967123254917096854836393843859252905324408991117403750575235323881739717771,0,0])]");
        std::istringstream
            is_p2("P[3(0,EXT[759424558855391695024240575482533939773843120889806256373905023569217133511301781290,419502294919491298013196952655974176267207680459296906162290689505,3280386192960530717081986793605014784])(1,EXT[414242544943960073869025486941106031820591437939422547432847925231507363416661357802,214517240686989115008621979309398008035000219049420011571846306046,3280386192960530717081986793605014784])(2,EXT[99708394656841138552529353727817305499136208189978977866139897251900869080569707466,57673266337604410939232671377202932597821158806403358586247713897,3280386192960530717081986793605014784])(3,EXT[25485491868493019668387419345575375437011621297635964469615002300941295526958871084,0,0])]");
        std::istringstream
            is_g("P[1(0,EXT[1808282469635112112727312,1005929,3280386192960530717081986793605014784])(1,EXT[942631725714432807162528,0,3280386192960530717081986793605014784])]");

        is_p1 >> P1;
        is_p2 >> P2;
        is_g  >> G;
        gcd_utcf_test(P1, P2, G);
//  std::cout<<"poly 1 tested"<<std::endl;
    }

    {
        std::istringstream
            is_p1("P[7(0,EXT[556956381722633028471474499680870980509485402165436111031490053459577842046003465469701569765436445982479846795885536720170424026028719466248467475987356633272611906769676724064147399895354249242946471887158327034346387094514033358904499507941234645980175026099764133345219433283899489941917393237083375257369211052250752850489514117392583851787892292007796557781168600430939353194542728549843404103737814558127050100351256976803414310067236482643088003732722713899765077665701291417698681369067466532335196103638671761010172969148760694794452699257342167838605419328812693730580685201549461421627342965685798982918357496933938585729701994033894401265035229717281678948371322310784084588981470885828021503629216710939674957265424262312420043462207557549625608495473789813662835727656600268920702024891432471929040640,920954918957433157538523809494047890186732711754008307405606748566505735279195857616142015607493627662524038201224721001854212219121174490846195423615135011426984580596510817238993491586420926812922010213448380302890249558982178682249241927867474398303099398279049659610488846439166496353873375486692189498662383573158455147418213611844294013565784939427277399056404430063590928586628416842842847793588101376427410582524130789181984676254738436556774117198277649916050469502179810359852096216539140886127036639003530300263870643065884511064906246089445620617755983215843426492156222878604986873139119394882876010734258699149515493061218084432442496645785313616489672284314477035474346760873136054109282640193621128373302184311040,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(1,EXT[6578739838496720671724852927997595289568968477142837550586069611709633493523128130797031226928621803079355456419317853044585841322416368338727990705740365925571675019485883319835024418544367683240010326541683572885563482183192733590456097184153144883896013627454120008212546017056146768759795699151224264373921748361623658358781829370426778758090881945502374882961699906111432728666103442196575043061095842747850739107131664611044097431145581190131857546636373306378988715151023937459746721987912187228362807535139122272531076097888987424796710896450777926732807973044357599597140192874757389940257551620865682282792602462057353457060278021782122595265682934883518623060694630906231244126101824874193180744790082112497270979889159454908922091685840911624633791894579878335828857041172513248699482285211387974124606768,10878268358503926561095976894970729463128210193163619999250166078647276193290121464981323177141307293547437326247535420494522480343820243911745979540209973515953828732864050800988844413800644196767984102619599362121203519485680908683013806579090247091761098257293825724157994962147913187300318788139280549842655362478271685586748817339875782301248554663177764056321865211828673837883646370613832244519880639775795328584312130776365701873478054214023395257097558445509678515476530461465143987568490623165993698556868708814231589452424516693166008652388750701074029075031872264006559887078480855334244096733013227681598579912362165143583941056491448030502893033419055995094458508097601626359927437872946504549208461552804220652114768,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(2,EXT[20391238058992412289660614269179746590929155836595691712147540408092603081363177958212625469386633665274497121346022553702568275197193277559709922012202532028755999928043279521686343378707248326370252441245266676489988501617618894529117848416773486768015310374671004937656088536700961890006808231531880815875783759097135806527755815217597497164101913814102111070541292960704023758344683000731472747231155743000054249584774983437074430534581606152131123990673851003359649882270992378635186073280354472909182790310005694372484604705769716852192501089694839165833019429160734808594297365053687283932800503817403198405378647261937337739370033450532477508159651713353206071140020520861019314253248460320616985289736461337877988066581989678888790679103078196284479198879992267593438973108183999345109003134153045638690219672,33717947310449381594915542441867268096127272345035375583237842931067342945627964209589528996351888966164925677536542377154554262134695147448416907583859515294702519144138776832526103044943709392777539227354149815903625197267288280482553712561007065286665231832052003052246030899592693653486841369608200824904615019589794688394429686938554423470607152862055909771599560134793106050410519686695882534565408481960617054635658834214400546605433045471423509900634293423758187191178802050683823755385441820228950387795708661111608351503578967431870598565857729503768952520566612017860412014420967695530482138264191978053512678780385220822887446640512227404190130044753499862258761053365768603574488860134505595955630673428828287499280296,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(3,EXT[6049663003593590468377887660300717398816239846823497994706316992353258831415382538635700482360249291997496115905765283278167453775788486712841102322263455894445034106046133967639982705177358833351987667508324684332408074499071496252106238988746828382567853021893256261352981376398183108592476358243635287610353537420098930007611261223306596551131646434720971151771760065898966884441661149671324499198365824023673681282811741251180744197015633636337532224559884001586902996408869307807119923158881545494335253687679364998926376067287013241962618132807954256869201964949202797185409283230244260427093414816399114132290847290617198300236174892916471144011510035873133294046988779743202602111566951910721113393172172710531183573462563366129100176829286030673126753955542280626478260267866755542274971495933524053246157639,10003700396867200714617449402663695482283277651109911406551084145748353104486047416608287323099465778884350217651853397821395263179969523922002910303485841252478313109177688724360046324232160167937280668999670962010489106876583911419834313495380477679586708133861273091566312109123431247501311513885798536203911779663189463312768144134925236759060119000462310572243455413913947173097677466443436478694544953889132751464009644578099222219244091171133867702459325441911165705712765996898284517425985659698073204191667650919861764271369480246553836152948502465323715964069498808337543730086515342013266479901075775202194349697815391348989864063629839778171613156103531347338713318422705635882845714237942071090764845553775505366946561,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(4,EXT[-1694142410850962966382899100972882612587679332829887753995626236270513028003915129215315032825760352917690186698275093941051107695583485997461619672532903680307916648726291457044920767667316590680320983473795717960962271041569877417812523626465778004170193298889531133905870995690797895574077330219953329018246217025675960154505249384870281679944782334393394583795863545953563563927900825555893697880608053332657957777873858133409711940543860297141031754382258901094012223962992497699080996574303007540181863537590551531426045574189814058692386358170024667577787801590362391969151212744592593664866252285535815268438061711918690970670132163958395516021955730427705999884900776760797887946903145482712742763428873106391383478771413662885002982694427250488019364838206163927877749341226746630761233244373571165742290519,-2801201904244925851198103439622430895603156425851717990630753240239932040065705323870081701206361643938285857018495285763532997526516369121691580955724090412430147600210008592068005448507810995430164920913258566130986507685379619521254104372748393685179011009671742689895875664390585316391534929927684086461119733944667986529213857507506926192421917648930957869191973911384873390577738200215248762597806531508117649792150928837194170086805346632519145646844688283137073141851673042155589834672675456373913559507615017969147780875246309188817731855568807654740067251859214992199376342072402846604241723907573232474056637056356806552371461131222541916528988628883163605017361312448703234286891577878043605283488980524666615751704601,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(5,EXT[-581483873597088954404082162174126070751134638387346533366731714604047433104527181134062145737398104611815007079911467518032115546785202154658743504775081687208601531828036114479842308746979854435705124313383461610461671490255302553120730301184922336248814046906109367177501716905326462187583705993632730271065411145253350207749466394759464840519021270034073647655162345337206713921830872240047790937948363015369670171477183212287413665482129107317016921789430547482912949465914546307874624690846714299998155729021501931604628692895554411808611846314541942636368221086920417902029318227132183204411316298526514851275037030557150384344564849383196518494688193889631081174776962680794450150672390900104985380646593364661731118066365658124100946217111553633192676446002636229309995526096993322329549054805747370478343746,-963720055606233065139628700861555000588455063711797915428678687545493038751107530321819376036132764736076027810185034668958039599740038309231986894450064781974548409689349120105359316002583612846790166791689744162502724004340805207555881295592504069326511692446567132810466501859381790231052017550440012908157211899844328049083521029361034197841112761176365810269253222479354869649664950471659572293857388763181474611413407034373476461657858835313115276448958618089041421568613285721659639030798440418566502294296004404421727587595540255970000692647203627911078465405822069779745278719406436190104370648874558916715952346823709776684225056928910245266734881378242284645796852447841710893578141372069375460246543236798448852138350,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(6,EXT[-25280032936943838021895349473577963223781745725641109016384558758841997155685655417371538179135623791335997956838311364140846673595694842151091073140093439569359533323344785239485702979680349128612477049428974624983696848655564806723520532212410819965822302940264964025677405605936246155524701344176035524627935151097409090443713173792428167145538203332433841255461996083064331022750068276521047487538768150004895605739485756822014942900319887132296085084058724914459628822093967347029238820407552001115336658523776159478502961564737285501067352963007604569394119978626337942245678994506927062971550465421488570643084604680039060808649243695795659589102761501235278132208297563517379796754659098984256629627671065487012812989853477841548873145827979454327438831248439397848052284947513632168444372473168857121928632,-45929362006061684701164423169004747366967279418150121351857452290357101132478754531517472358751245937282403029674480321956997168333813550524025950018408349657724697686413099758712558688397950824030576280858064157052327322487052456779148388686004835267064521855678718457763077716709522392364855384635643444031180685842805515798547112772106414832771606740754802698350965564685994530768636779355098458797849951134246045571509429096461656124354545417345084286771228130821899858981017882080338427171669233555288345213166782169471978257491033044522381218867601691119903028314112625575746450473855070224735673921301966237865247518058710988763859936884149609952155068950472290163221997842811847546275718876787324547827735828615272822056,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(7,EXT[1232693318766610527862110165064334138857507594066868845071426405306107903374343723203771897105479971460741854774665682200146894506353926864620681513564387141148931953316032556039649248514960952154069284148766974404115723056514834379993842894954907611144526090789039186051272725279910521493487158478288419795082474365769001970389070351211232092389632456111769329589606652510637810780438412854437312298842636137068805966190166701935450270143388631228584829903108822004233886942262412464297604283506762375246454334118789140861442356174993043726987687334178841560925967153750746338689581087912377773198022121935039600015333146283749719809405671418878132500478913145030818780551594638808014203022636604636625576558427054855106592724339934860392287142196013790833421520158471867349150643428866170011641038670592485795200,0,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])]");
        std::istringstream
            is_p2("P[6(0,EXT[220039133928029798556322030523486200466964509293480318597203844049586859655007404607257415891831435507054330907352218905331206796758379431290504488485442178302864548925142100705984351234115611517439091142742050448467893309516619663511986467416132049279996168056089602124369870505771263120234851971210166365127679132014134902528347792916456041151733168522174656242296755001299135472452334688373554734173683191603457225220279357167015916374139301840543582178721306110660162024596906040823088001528837107033529407254949692652169579678860408898547983334470626166089166051698358875208455364845588930706482508375892533266397355447507499859042612736710575171383515813050953736879305184881076084498273751583446566929278648008049441953965530608301359942632949019568109138801045249769397429920662032680065648916272144021703680,363844097834854672255563992835845506932773575607000750801184235809894710702645227969587623329445178227240027483655740726253146938374781431627256461161386601678216318069336289374667450060611375096749498766246622409828379847126034786872819049125085037570447144101297360410736413325394965640795444431613565362788387883080514221659940181132708937608668069336081707696959300286018663841398163736810456923665635734893036987781670095750318634222908984596719335798276680384875685956995057320879057137541712716293013434925127729262730290854177922681645481196755182235280839646342652004895463328061343782192013154448891670046773877461809426973492571262621123853027297556068694827026071077017227293382445794538291243286811488963609532462080,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(1,EXT[2512161186598281795696558970431838414449710649948384700668147702875870284358638140101474244902035861238911498748150246579427033241322568905640116950597928951005128703044376187990214455069870324054269632326958034667965091789124150112020587892422536370792191740544984891542016751132580302604907259219375157567085862228380113460874230072745858277481020106270848131858889804601916001164265823595794315106295691943082809882895659393298051570423116436621042989863750341536346114680312620021353211887726910351636492963926176157079899472380812141606302935263951775842522846848312909086489329769518493199209943919470920563849400999325918744485077131150611661961464399381226673697129163196228618375173998965589232704134860495397821492249610078587856511047795843638744641458020117539517461078554219888350202186602526243785794496,4153957838058633975714052125344530036474195523144978155585989397011395178333698732811753183123766545190491651528812237760725458791751660706827486283103657392944404018136592709196830750521788096085022725350165741974512900811280024171095848279337998646230975629856593541182947460981972800119586527392796998671427557488380702518909358681971860176318991903127582374849915558319460326307971785876266862021012191743150461079154082474743496532324511505924510601899347797960046624019495034889900547921263801996671354928192425168396629137675712928377662735625620312562902334214481384755133954940609844177935900607171507036899215969439154080156228414450540065589152330371377486377603712756266963120490422508316375542336929090487961050063936,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(2,EXT[7063500635943517357832022952275237538582842697646451829563194801268371298033609613547338683435039987550896072432893895268969728030818867119948343147324122106958043232777438090800866469258907621262699383084857162049597632939214513292157150790332570415136455910135489453082743004039519936972067696707062457849249634723438860155099079429297374852805922996084580154073788302319537896291510911681652355386959803326152012424838968984911513915622595793530059618365328327587226364682799201590577963512375904821616515303055681772068284370487387144584024708441031255607001138064320500168859644902542352459367278948601869621810583311498611972694474627418977073589568673467139684369760422704105500807984215245975071669631120768279967819513407184514059699280837533628214234893606831485737300754255561337911451114706113789256819040,11680027002953029017029944228098271627608722723834894818884205045749107585662934581458376509029250838674328347404905517631212799442075768815320626209623347498490970189878986969890348439925794116974535736414715056301094999099771029691607384804466854791518729950704451135427788761433913748201478457459243760255743624427351423123504475399779217555176364112140459524379159118418567266938430574522256087390064168341830313848466862146315781634048946134959824820908547209738466216312756797370226659052136244374207748402827703696563986809827643914289176537607261387466997009933375956960849388045897943212164526695689738921126411635024847099688772847086409989261140057251971776068918185701802705181809425748061846112398168669605377513601952,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(3,EXT[-400666952975518436865539830078755055961722125762662798400660239971492251584478703755340319460571879370832299824446690213092316975638129699271786338599936282552208392268898871377088716232023153051323493104905186692720340339088866837012745538412808512716818749113400987660767849827434226545135591220634392031357713713214924086222798983130543436774991834531459057174080786522182875567939093887885089176874283664616720116356758614619027979157152524705241081693460958500677708047578123506757308115242877909770367632118263579222463677501615545703447859609069576133063810703846858558548472257188365049707686278152191708279541230218467837629272783098432181869654294953567364969437978330823278613403082862866951013459152753872358936837261366584574447545187396157408156917385496070623820762437798708032485900380916505621624072,-662062782959846813151508511419296009072258497936151363441583195775234613771779545135712034936509034261948425124130291449875756750132452039420633599569656106448166770958348752556103391486637892632012005574095454958417058966032175396417294115144170897103104889169002822847119986977397195164592822878855376148354749212199567355910571011344101329773731507198919957899490248667765366204521675783105505483094801449163762317014346830230619503799423009391477158490395493190373895750447077935384501742875994312974664005258834719606756179004626650121072053093802950849501457829537152156310258298680266519812252943458302620026805528055366738245216516902539359044812124702177835938091369838027158806582429479495540259569963989451846328103872,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(4,EXT[-510639880679300921309577020727976214169297345854333077939812312626450948754780921443649782907129974387069478133261317449719656709460837227366394856672550663481649714150201124213716747607381046886004207956714167295548353019443492108139758342426526145823445619272566855475369529262876448855338289085701816436001326845993766076497515715286129615634349408826244480433045887813039684347624318728847369829850588417097375181610079391420982793023007862155366552085654810697784205164677326529634346338733340106958240293291179284988073519780657774344650354905277213142988023595999190306221605644346581963390559836050393241100953038869453746378815871986406192416899449695947049576090045208155452051680409185615774869498293420577200268580675271349757948484254797789270988592251629647088190445196682206484768560032798462058472828,-845742029666686099319376572474789401897013238766321301184121827986863653836703919972788646309681844739321349156425765704931667358916152815504288202412914945117646585674323510767864867713052009300407391261917519429539715097047920327706748981437438788853017658283339649799815966932760099000836324384887756812380358771433065314786931896807234371094572262901957571051404543532024324655790289085930821052889946092029180897911356551969976123197176880837889137134303940561044791005320093415280567246334771841897247325556453430767567048110998055052299634060617527033304886001363402661360075104708477939949189401681196117958024570117466920564822342620069604104935200847776752637322181536664554982644254737643151226654947511304499566581924,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(5,EXT[-26980411696770616254786927575984506921697930327726740610613159545984254173368445432705512687411255679647866140677871912912805030307357668458364748418648314545556221936206206304214961267450456834035955266650199802027817309866809580378211096292432956033001654964372412955534844598597582369565598786525803236764896774383378352804496580085850436055314268536345502565179996350605012782989088115469684482318598618651857322192194192763306535909241318551618905180885287773184390511870037523408900731023822008078066340939896910154799825229462796554838553833671422788447487197631454432114506584128972738824497943621039950138443518631179286217981451065363206552785148482826203326800903797908234814213140927249507150858875695703262035679355151481637951991408676154139525942918642536585901066575206841787912512768523099534582632,-48277045304098629353759821411788735052813189702779876443320425819385647276344103842752923793722980799019118327439597524972191837987754994844683427994255575074172157481394401559111501651642318018402712540265317883845421658665039726254068039004674156107269858381760049856054175856568509381860879583999909910838455309518475674226869342999188066669307216737786501374606453372010824442070484468768794030078756919969255933412789083027176299238523461227651440259371365334877593875354729420249194794666328551884036032248585321009760850071880883013922060477856703255462367648073886793673563203405498358025931236415319189709686413854196580001756211219482567715515606494562211190430045887508356391484894048769047842794589201117551914033656,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])"
"(6,EXT[1232693318766610527862110165064334138857507594066868845071426405306107903374343723203771897105479971460741854774665682200146894506353926864620681513564387141148931953316032556039649248514960952154069284148766974404115723056514834379993842894954907611144526090789039186051272725279910521493487158478288419795082474365769001970389070351211232092389632456111769329589606652510637810780438412854437312298842636137068805966190166701935450270143388631228584829903108822004233886942262412464297604283506762375246454334118789140861442356174993043726987687334178841560925967153750746338689581087912377773198022121935039600015333146283749719809405671418878132500478913145030818780551594638808014203022636604636625576558427054855106592724339934860392287142196013790833421520158471867349150643428866170011641038670592485795200,0,365733947581262162687387094785295209970272831622209271817795883315575402260802043159872535519941646752114365312873372607556394614910246956500984245333658556509595480220021361])]");

//  std::cout<<"test poly2"<<std::endl;
        is_p1 >> P1;
        is_p2 >> P2;
        G=CGAL::internal::gcd_utcf_Integral_domain(P1,P2);;
        gcd_utcf_test(P1, P2, G);
//  std::cout<<"poly 2 tested"<<std::endl;
    }

    {
        std::istringstream
          is_p1(std::string(
"P[8(0,EXT[1124817735439547117229495095975609275547083585549721716206428681090253712308295250827998341268314611452157772685413006211851837334305751557443159056003520846406233463306801743369213337814852365222103599346526893286926546800540447359361636104139086351941954701029432976263426254188597316570721722883384398342542513471514039959342764847261020532830540349610177795795197908819174213398536941529775706967702105990575236877404905529367188337976919469161228945660431940188724756694344648979595446490212410892713173138685884831976767793716591275509475916110418599314683036296840648672529447512127355129011441040881987505140082024017446462264704572218138339917949203829110522383923667982350647924368203282762424579171689863806931647276487512332008273204872058490478196317413150034770041015269723485711143315437825049206717814461304042285896430906478481153189593816492854798199251817559941409141667431554069666389021879936424024280617605301551527137654663706262133814678128443789794520240506847234070032472594088335419806627655186984834124448111380041539531462134685662531605902439666244500161096155243203312802217312569891875972026963592788374656178580494228747584249099249364671364740627009471899605120375544816911607209683412721347945470422137949404658089646233407377096624845318710021487329827224565895513964416085204228160633632710011468182323017732648617689165321071547877735934236927026996902711414630139606510786453930937356271213681376087023566864424293327405838704643050566451755058830064653321512962758923489608115809083800983322309585238714204392937333757034672807765721512366871539600797825150195916023071867451253829012984426806991144251815885067601542058413034933244793338356933296563119469054867775676825704663230451030108925695001284347710493977668254456042526370827969990455805559966240745934264657066752668416099341944231088349314927792904777895279127485720399154627568110343266060256426131607466528786582081867550431790672305202374515633945224175454559694232835713972125789337893004040120971256263817001975638191414306551493883417429494081738441234907569157146291763747820376179581167833974496977710421674411665003490510666691957082442472500498981782764913574531508938278292630523600702219222197302293155704480063914765056086046512630161123442478580505620490247936960072704939972898888005388049603760565411748464934538797646922660160697613688637583851283284246406163801304713582031386653603853438899054019346037409099690679365194899175944733721942026036855868982318485706336527961830894989356864469625669902413405960651350955281716481447617263880120259193600856485670400704605094735372400248095681795602975613613593797392884533299108796583503342208268439780639012448412221057555330561393449882146585614140996241457160100612737047105357857503473667047569426925301213335686883843365058488074499234689119790826596141220755006180814417474963301643732478212849256028444110438460246404305801026377421009438593634976566274769983516190653376590354054204428698125008558124130886522289774716792577791334795201605545409128948036513424465385342893582283805414569329223912679342497266720669490105007688002446753638736923496852658236729353232195870299067588004158259057716648420951132019789359548957873608738923844046533696779240977148798289936198816283994928771462189429658159090478926610460108494685194908953698551420462667563644251410430238974928682824951044163412959738818940931059824107498907920393628499325376607179503318854808846542410304012220279625017009081433668890929154748248489174736056920310310262331048146655610553516148497187936291199999540038581526123988741600249915244580901833560413281835252887360830876615903659768533670552318656008264757991000228840571363576170118511584130231305019434626589376410974070349691689664518934460431067375429572019301296780724053964559736446937117075022964151545267523760175900467358030686161127160002977288065802007185401734890005202180975599278010229196614133384361971825147021073402740612363035293039581175482768341226138329076550482291362803793010143628586002080597869892449912135610522000917520189253124165306857850858427100961305017458081463469347194535367741068366488399598094028737454917595195438230454043037468244095280937047428931211420015526659878141766015933349981645085057725894707154924504654689092002247922169932855413152538733973657541967751374620171177261786186178834555118220666264335694782654036469203497903158643444454116323260050778708857851995799315254822145276169086328163410360217388648530322423726790383273354023634057970699965311355365319196150188233690291901763505897565991653997063231907178763469905291522033390242818255208504877868682081845708944097152175842807627562343431159521163060007764587498686746150180970005482166301940916505490526905671091012008373976744175764061914272989337808268600120078340704815065206078444231011761686998613608488046024978626133533510810466989459383402968571259336196520092846441938321341423786103819888399345496397237281824934123560879067568828233662643791977197384943895320822692182421826238286450894770014927317574395153368252011121445068845469979888927194943403694231060160864343359980752656599374512965942085731842149935926143750480084378599701751900008068505226407074871273145264010849130843641858900664270541152087840856101736850958955548231374806030514165663262902326019003089104801929796376404424577647546515723003603336462378312474929360955483211811038683222655443983280395781675167749929033395003782181531009510772610943682761203337936185028793373058227463092901547849237340589814281087973504491421471805084530564901992847394614203620014125091026921021877955835654316438182105981930344578381478364225413567995384535858841784161414142589651950309782554240226786295972163924492056083645670525921943665235781840985167551529100282438791216989684509697261714430893572167628052401099747855473343050397211126099283156393901687959030866326662400673730967229932463338517546032483152680919810639708421069886562165255783444000503561387282838273914682215295252742798152682930017660868589101804141524379231553685437217909217405873726887090579102071908817756694137133034822822000201980860132343290666260127753646178065608086386055800613692949550583336344507622471962247691962073344266451667737460519002868544097389506917122969374318856925645866563559616925684789446664345095567129716403577621904378160096245507417000801737970971311154876513356239656970614374322345698973079120830409495051716770761995582265903782850087436723206832566336449207919868128455971215410238383153868270868127994084149261716193341373262456237368120749205651663971096043313264838205638405740483711508070699861709015290698845933175219693388224894766181042348923107043490876844778337764866759938119636555993798883417457867601193848596985235444231777755270647794738199789056,172254529697411160752114058552219025670588875185259742015578526489963400376113445182159979755967120354259034918131984962343934497915721850117729167227631558743132453748003364786832428885172069943204841835538696885674819380217879915482646091661315270637519482200814827949352984382743232189490380637106328243731190394318468317275306020145142831843061751186808306456197855036695458691186791940319713279893048207668533086019305602525892499273867328178035503693661882006047126687798684930530119880648867392347625128708836699455086385292647288408818677016308403013664237155522824656025301876061934864657673072883374318470634702114492020036493157635016623983128534789386757115395843439762032761406327577132259957210664975369514737432331667247278563971420402931078365830502170035778611959336236596533936519070121026786527767712839896028075770492947706239351064963419508927539768367911170836935972315508398814431567356172991753689787674574665290455602490925304694255066061005525663324913500277784134767880465289162303413859435836420031170805568332629137023102640661650919414515543814271047304821776485822553752243065411432603201586947968074721823849447773370403502583472960945518256009516620119681070055294605200183370564456489545359885337097268448599786600328350289816583139532560405509726116021775592290513230670806293790073626136296005931879628259993394621014455379972144466308100398254153675564351304828008214379867903206760360871389881269979521911508448641415841685414491138587831953170974532525299170789658911380836439886506798268894070602907553918327278711410159074963378874123817468832523456844084407467582193486241355846673279832404742293920959279193372815295047401773988762103350877231691519598422975189789633005955853958267827329337611429024262828412921547075860511790279187060766767904582199160299615946687976621852657094736407237361824122576959088558853938310563072362249865182968868676631307732128616265027177163417230290327434711317836429119158842351356393262128870624366424553314221149937938428367988792849594512045958649762341272653047908850146406174391109384168042554586200721800674152434246807208178050247767449076407560078189865333900979735412948347066909681937543656945328833569162531936755450577316512478014955553741852751146838346758457992563515004535496865920258307895179743201083641641786009260509241851974929083281113380749489769717511752320265996940286093750374294936326479788913761991731394882005916614237436620474366513081994881796897122109976100241533284063989814677798908843965208571662809397003413169093761679765704515499758661691203533780113427585508724700330075622118022659929754090038255026603845772669391435491288268960526234771106373155816694349101815948027722062881649289264377776757981805743047005516023135036766312973123321377768420188253955556540718108975975845933526157243101910191827250001646990399062239360914316002911654539606866800984944956651820045255645893976884331229884376262760347907649037392109583794701328239799895705543003482041834539042793742112383247603253528449193551077977566420743669771665097958163911925811766259522988912124195668031049935657802575602840325292377868548272795484475644446606502370000712800534733664711555538779396366441361587116682652510437396839854506950828558766809462824399599730675153143322350090672156175485398753649172976046541913700117400758220903719699983527118447456937766641210040562072522856283639096190014384473679823784787595031434087134127168591460710615531605197298877006025396375373362863869214428627880066166663397105892057654377867559034001148337648397557493290113139599285587726127503401493697969499799413684172440248370154679848077351123219085495682553289811550132046158403945688673117867470166204012835542659839392094824652986813970806899274434577932232955996427305211384307673325464535501886832572159843993180315576987076090081896388409678340359053633161043653291573646330318143161562187436009094125965854013168369370233822088978740025741187692128099586020975689232659204835800838576013162053573053466680154915246714117976948575358471252104025353314402803316878164053317334812534393780168281974093079009622574816564931315100278727794355473828919661273907479873571421594606546723527826235955300452789261369284250939649071639211392451478780725515430463831179113634593046646225561035035857356010133570320703236323203650790710067540008730106420100713016671106931422303306347758524047975823346199661334769736812619671670501618373011802680530827606376516063410293448814131491595814029333326434175581012342186217220250092990684584351098508941518936094455708079372103000249086948038900695053115989386209889840917079567511057913119784608809794273852696605033283148051395671024742934973106361154622441205106055823928091328849786371732255164235886209368052735520218255798690691265537975009146798624508989010606238931316783203116718311224798545463925750098053732167568155043628951884869672224064649265604243944232816060926043784450630296667405751127033832901347312534314391189532274373794947663326327405195193443346127258594544619955781737712292862213779546725858258116332043591953094443133270383922668889911373791867629348346795490851081135836571249971138259563948895326302907710928810962224473113201736188622163599987329742060734776617682512483399770127106189350228572500787587106064807559414670908527306617889656782014931200532525553477653240953119352428783416471541907529706649927890119327737942816823322990515528728013982846586596114589986348073060555587238492709465578930989714457645871664210375947633990397653236452927914755596544240397709546065391341852382460745690926650620415451598795535678327774863474273598450521732055428328303821746490271249351560111936332634300236832262241048155034832294595167196635086386002183279419977000385632372046756861870415897062249635546864435493322841462362496828411152150231132517288258957568151438670555684425791570936602964076262760795589529889823394554798496922362688419447648125025336638208395327938271121524454285605386227379447706405717600318392742426458802878528407669688686326889600387538540735299173075227873504669523596687116278842472317127298042624798134417863963921563352903781404844698376415549786847741126532380999230063276938085358744134784512,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
                            )+ std::string(
"(1,EXT[-6988627372201016695660754383184891024059054585710897320826469932181689350568838659164817649431118755823568116067101253209848236967405148190926072465122332050443271991030598514835564642048385457612577749763851827203859417582340169341304015354273791146435017273356831367407769370331970934616166297490719396604675524270580082818443407860035462241493025772189971111137676378875553244212633494081157012147210162425441323991147953419833566608521415002969269500281398561126092487455398205079793077338546310432506826032172179063328592452957442059747799343124360547118596944598354704615240617631519647391700609531563415030857914990014578051275418519598284296918656785079665640321187072544570604058306131108326186628821474485206682632662444971373080180910358055717662203339497489764633540157797461665325143070524175901005028284973058853877251963826826025463684986892131365304051012804479295202748637449780882267760735464746687395856769910479757036351211611377478149270999765633580261495346789325895270770320028886449412884088072467195718205042344490027410785943267362204566784656034316917362431602943352401826533674224955355528513230629705733818816504886201452707294896181129129586760841008216833062109721321195882162933811364469882235479578684530287844428713925763447508508954920749088193249643014183100559664018784938373531256956801271835385624259146274055933877769008803089380299635671399116770958589256702833930812553674833942215827527709890815597874582298661408635634061748807755015162373532625504843132183384618833917839159206681402714606709119909692114951932711001440665540915232651036789024011421266363133181290776023113054875544979369312649647519016938583931636291602036465355530993754103802225285608858498746862544105631789627296720736765404154099852615455572847019643038719092735017172260765345237139686525658172738006168991043924771869350435258271182449643621535286777491345020588467715615843798038749223644491019839000461275630075837150326998781964991874621423047773337366364614032903892962077600747453460881123243188358846316955968515838707680042047749171614729003773673101839781619721114692460219848355935641720580576438400945062503250397606397103051086361475437438899904180489848109928791118458896776666653861491346043270123395559478701179663912108872633877633329013228299504052867169790236584493380249326794598811863249930985640241311741513335828134257373287312427827726722293812618454742591421258686866719978731483665396171805738303866418463800844110112167751444999862295981992361462681742695072320954863665446722283830364420949249923555326559374988836947758837409444095711467286342024172220915183523663607696316889919126791729695845901116303101033091495824413838579819148945786028441283226772249288683010462962639757421536185607200322609518227321237707123031423925888618817685763352116857189999362168838577209657682023369830147353981326726098525297338630689339320124533966008034820686685431282604523121284133462004549936875203663367395817766413933606889829677241130320599223295012288220182949722347191291157982080726961450627251450844172457913280352085468277235934079393475686430237979991347789285277083116074669086989503144285223340576615451538461904550249796998543189421157262404537439893485061526690448967715812323227349363443875049810602022320997382227700452477197193695070939100245576767173638814240996404016306288726358358958262988326756847583671079310844485158520799755256845154038337056953591092711644375563145757709529199976927531695401699207177158862189804762747963991662944870399559045277413591897650146285760602759887830886207658570263857006498780100330084753062890589172237182173630027885931145644026978232265386857477589645779846766018329424682851272718935569609842008106920125308479701165939309887321271865043456422390087632484171930595312078954984000014174407628639588076634054359168890524188382737493741109971680274246264426523653481835923168647013361982543397769218457117531248965934791584735091856244583438498828194244533797459092605655413946109790014834054350425476439598070400709797107928656635971847417858996077486053940447550614851426728282445280885268395253237621101336672582059407650250396445793709998840308609347713010198495076839435602212472702097954606428774079197678400780823647822755264785932227793196919949191905533121706932992573801002329694380315181817686690857975152570610958693503904995175912151767254876752468105983807577427414724386051783292500336549596990971018966510193285944297433183496863518603307450528010532506401996021155437900481732305444691096982556637402048885884632476855225767958659080055794015617504824851786527898296214022469704773628279629038934048718101593435761969843992558076599372052619287970115651162654340004041087002381775069688142998166132569763866066635777712133915873918395618659168444111779733170184161725663013825372468764919722940810000988166608672841610438496557402331809443442066839638383532450474222300453669305761012875543585268804329753925742950436229987437225941173401020022307237851777309118230718087563208878336957451095771569629129022337076838638633567188118350142019858321598511047379477031314173051090749727763589212806563598454879738522691309453105975580149003635831861744839530032768823271598203174138007649255056673736677004055286688154290549938863249990265967917052215221655103894850858235654653225574558638504269984002928713954991147608920354130157750493021532007769018980377510619920087422500722246155314609533938693816773659720938062973349979634513684050810097257526229596952454746613470571113310204458984017706939704828508694314284205526857512307764060968557343654036448092717351763780504038727411779245988141055279733229689557985658722571986025517611542669740988622246541757849722197360700958221928219302236356636263466212016044465122723342794160450500317464067396616309282280515575359940655843595993257161563525888862162268409033253428223938130428220212153613945137740934301676965769539419060081118174282657113116116631606043455077551902846848557371360780343315467353565276096379245504917330348699243856545165665361109010549416808221639468920039434074212102974533506047486508072231394742450476218975737463849184472551489457547664195874896166953580493492402982279735707796744439353831526190163854455845358669297580815856002796881225862993709133079272788240954589360835174396371537083421876164642052693790950527956310454649025327119067119705971644589988072749939483291141870098038746805869820027663883447069862433326051279220797651402687698391484361750363850691337673883248956607615012703664047874070486158745903706696582903405202747831897444508639730804990617138446882223862857854279017277138242896667163101171286507929053395826620771049086253842650628228628758301247990846386638269922056522346569776578300277795764043281459004084656454818997304871939392,-1070234065687679106953840986921778027029987416210678499160065712603489175217322510453532695568862394283398880950089234236287048109806002548678736532467101600313873301667181088587296026266171956261300222721292570238187440553519112764936582353599516351660423299540343415994467503190866698567537450298390650564414158070234236373448579790399123917684338560389789209939788399640996619506682608609659711225231575137658746101583966944916834656006416117544997945091482349640418261684884936207807928615098482442043607387502363202504363453849127848749822242814541034011816008989502889859846914099264229592468371320700467638173179236087220231730553199877742402406453750267133094542809476914171999812850434483704209060489833231154379031631946956337084385386266570900341604101232509841710847184338393050486534717939632295944509608607523721369780626286065353735004508495155958321967966671843131764049639054568417006208871817872078873756190690635956262103282889425294765024533494430662073588549889470435725426452609087948500277547151611699681654472910445407753234988491588149954152098500204693875550197588593994239313014509829255521304159581075475138123295363752502376955254014483459581907738962239074200328084342780732889808073868240013754729097785866754444104667585586675660495895683923240708669464554785815464598095652759761452289028492720564048577381083566210512590570387700405972081106441483598642511802791977075736980595115389660448608783721867616893044711071171340116477355516159969365174383488437916957853066525829130716747434320264630924379253119800180717938195556531612523090555770113889986937527674295306041998482906856228773255723419775489157281399628187608622785951898397009154920798184496694236253479236191515690549471531778194007036158423128989498801756531751618616887217017236209094369023553964890330365293122820668887425820704647909012781929718721450940181398840292037752146312978618728128062127326844110532446521825616474685774129430312356005874456721276085621801315892128621144190915036869027758321943224292506958188592577887240032582485343822751868358131599704624509819056333017214944108023086738812967801113257383168568741106023142926282062883833285386161130016150303731944655840852266175587315708205458379934471086606452220518518606522189721339871820921549697800692547968201367866201795318960843877576586264391775099799042575175128450673003261913777817651462357086626691422064111589764921180797381032878084706729858651027509506596200308411255602018169170931135447785731240614384745649611717229250721161582250526043986491383217037076737595346009935639383870643570431429561711776675202890784236325855912210810588415622200248384693634880406355458876858534769439815821985563318683641605137893884793932427818797916895943366155540474833436791121232186078424449200633599982021250533305532233329228082187589365844451576602368978563665073049110995825237730634288228947548010325487867443265286035604014031126707509442933693344320015212437657836320870010909596446273155102680679143146533699323496051787503345140989769120118999974156488295576270912772523180894208329347684213706568505334961274794290512230654766440193234853137215470659390579814714404462930521312124117511216777758221120521234849875912798658607821743417933459802093876594147208959574781769593551872762798818834944871956600864417781501446916123798524013158917351189859556894278651365440544969470308578147267981957963341902440700031013433514799021062735124026027501455534923268664760970684341547437473321625074487400750557010875094604334261287183202123193017557974312625275056806412626445747498939807354400746666602045639494572598061709692918901274433580240005910690627246776826237951731001579650483035367666212142949432428797172668785081509111739004237823543958860284583854898640038154816265637503995576951931176763881039439597775205989499575182276063769991854350625452711901668569755324175579930439561632722801130189617458241656658898389861593604376234643030399141771643960046210479653444026310779412659191365242828469824730286593163242703231189014070896295443030064110473922181260854687709499538597357757338807543529623124481275634066175289013351264451042340234063479011869909198990862194436648214725721512971605739985415318205585910580336201748208536839547038904602133156676405231950253747510560866955644795909037911644342326923706438997711012396119784763574122651869214271718838184427064508427231048941024108331930882083047184237095602717463631394549168962813757629195380004860528950854639628404540604397475808695091995997286412773921929125829329209463541455765996324739362206715499864670672829156974987434259025161365510904969037937815259049593740002598874317377917928079174336460004015088412135210706512077328182418570007507670623619352152918220933532105319150558261323936248261069303333412107093532141675513773473721444658138629329972817328819733050832121586527406581724854791368001954812942077520602479745332111231930109555967770178022741288869098281216569119315844233259714970215901009261746706326093680905629765729742796284288828964266752660267098483341756799249318796037576013662742067925662281456221808674930367772456814089608017781393944481994512571488847024605429454344850598782305634962239858715730922087954713802330213363031573927492273482872276510256938127610832057196828466819572040247567372160731900289274589908211998310125591238506916172040842638072085638355570747882361630080690830210581731106002031590308770910609474382378235502210419027018098210531516561321905083114979838248790630272963169902284793628278904794253538560990705816971200753814682532766211710228884030502914721749302542730817070360689482187359900986059108623452162607420167001766385903575536999379136486500337698961157762187013219214279491754489079059054737503078538428914518318861471670511718957556190236450675276548165840482713506059548433025934343059623143500820725294466713799730795931283827759302663417621889868586553646880425971727172441044402373200223235199019672432463807101974213829447072771525226772854885244313226711385457692497118613102412881733451332156355419097907738816117114299812430597022920870551553882848408809538696663547565983762045281602788122935874547270152561229369152284057037297780915272498357818891273664,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(2,EXT[13926823340007403668369677861707021944074035374355927399810269425497016451964245417366988680540366191606349839822664388254717043017648351702086612089442727821743863850266658435866777220056270939154226781474662271068363152994788783307828113135180712181323001439858284192140338567176797718397434701671743467146753867742973390566322340247368059784666925631645435978462286241689143922702179817687204716920047398537561109956816870683105990011814851632197377978900476469105566674267396566870854176887280204352199067738703773616878205718276816458374388277042229585771830280792863261541139720072038174146300248657642392145302501683688157711761922370370511407320509518323817891141442119733777245148292119529000749215628277549564778566385699851610434168575407318255529719845259461687053861878721308418904206803657820112786522727459275888366704631818856431001669905921200059967961598999541423726432318180579621833989659809864846892328993948305557941423855401570842979525277982458833387410554885268732179888793436604457980185871773397549313460110276597485941253122883434332601469576376487683478226596760602629167798554301186031301712723177655344464198427718313036200706836123692355989151585836671046928147378118653771053870889712829642499020762779360200017465493956955613799206171022355901946909693821606338694411561536902861831466232472630604977678106183100091046422831206779464465759533684590790975738142120544625943165866381344415140118312284119833571984918555985019988098539185142743322028688549980405360091608037014494355729387257833567040614095722918736011439678120298723114576895163122983053816600227050099635141603297092901292606798068787364690201600454362645300559945032406698877651176361169191619683363538520430373228635469200718030532488747233752428464508874306784126386642747024261390634772414047716136989949629005828266720016743582359687471473262566306907694271753126816880267518353672683863511914137066667893813634318657347918002476730971623044237996414182468026801135438818289156291902991476421273784872323109616397819131879472311611318708441868752851342560588224225580107463003244681104768112965356042946221487111942681024803872662580432802757760834218644306289546496136216532600078291745865339003737906449594862362561951105656766255769025784811691328214947716273436802923074947739878247323523807408278190826842254595191417590434424893377312679407101126409954585248705826836649707845117927943073721811055213856594810568121973336916003091753506039794123301701529001948560888085214680102252994538033104277571774880813303364750464586651418718367526067949847552449179883581246186517110472719959139050518794466365067425058536489480378464398118318311042440654057149668343189672602170506919929096666786897251241721701717925634338425588799140487204557881452857202058229018369091029027260136050553875295135354923751017386844510486401370040939926603338695811847771878947321397878079706132011464768584054864975810365616557439375137808768861128611578231435502527053319068964515557444319843770881673747201821394090741617846328437810622251541715684405920033185406321200511821779813744773821139760069202651660077057505494174441122364558468900466127692288196306140341445861393009062801643098934506542708198851759582020796703115310900984975245367281200199096279441120924504422906055406263591413973815879665508496012488971536261879240750069738569756947759802558125290107019197163901436560010595538216455666115211616731682208539821614926671590436733296238633899851747566819953821181979724480815873091562660914287455031045545095128411946379544277418567750624238889252334928823474132612472730402910317068874681081345012070186034980435974843448459377638093055903853537653254756541406330734479442266462949328216110308782725619865884492067827092999472442729205601822973938236912566713017363379236847365299735567445381873736734481619258184083149232783274680949948469021190229048390316039953792564937321580016506958273616617305353443948547037332374450424644236355564068396896526581596788940832799412891008685234691876306072071410553930530507792825764264846405612581847042296010403648268276013083317224043648329215060071981076231781713706322166622182563270920959934460055908111768193706536164929599061833135501080687931597025401427196833599611118446587001218617982057647530758370133765560175020434512980583182631374792591876515962477143032081748522231918531932442809100895548508418387419708593851342019185873135946347537913666378912600235128939789249719775258214066840696051074923157581192112915208949665553086836303681014908649347527838684545665651217953067568938036967556762675083406848539483418214219743237274854270517252642908698369981806861761762314231646824214744541886386495731282487732967323694447511489241411043289959652061002870578834803152492489174207502853525909546645199292935759125265647958305786770005857262988995368430510723219132058696607735180451653869540815659087129403377868994789217442187854045230161043060528181098427820061613926906100021323267262479310119943613451849691796344854809491074686595339476169152428474936113247620673926749582924273499154090290577467780624880889864104652300268681101089091709857357645447178679762947426095034619506100441533753168401403194267780106998452831389104724234100152846474393989744535962671425671648832734308326909191303570990531054708887001990789034172465400846090001533224210093595001360738713767067044967096454390040247460339179644964884076930865096827961898158426324345299209139925700151948442933039359034654770930125683362325235164718649242002611322019425901236251892568296880397140524566858169479403485718688728126761011716797723041342287367059320943262444395975540630939538404198288643111154276267985200394245515911236938911160596587007415486741384205172122568863199909306579305004528212799546443124458364390735062350974908313139913534030236922504944798524105857690615087363163328420012483369737954774238080288921521752018994804008156607179737050371843689727143458462026669809147975924908752228385907815362138398276313300385995185630071321276133813687392062834197428537304345848993714361339370403240064915159551772161277351303881581040082906723277869798762696915861873350805877381932806916794831962094067855369292445394602001483143561497573629811944879698756760531698037392480451441764867937788998284894425620151776189775006942678499123319065170870578088023912934146844786537077044837392576794321067269410664457636885230819097957207247240026877684690302682124468361377338525044008051615539345577561102573130405676513995028775956071509704910379058217193309246750102498261455319517811371886555351072485084367225724416763647049196994088343543410020379728060869656721696049696364060523211118207321541772070972878252686911424521080006848792972523180950236844127322967779086178654005089035333426850477561079913216,2132759624841627334689116031794650838361907035478307245401379437481870486268612119649452759958380780240109741876026259838759223224224745154028742114253564067351393356352718281581409830042271986540242759688907399947517446463914777703781412986874216557554692707071792074776054369270246575602533806390708569122645344331065325181981366902969747324895362150089369636033247441984693776423295236827211830404048194563369580717928892355597726196007524520095718459116396024839285288941236892123382851239199537584174039327420804734371955939127962696800012684905868131331700363183588845194822047454155859928580557524182555153844105840581576094357726390382643000258473492608072295869757404494093361950858772282018622649077577464026982201549989537749795383403563372585797854568179550491883285634678210038769433042360170125612819532819495920328531471863621473173440374210463354941229698502204247628101176444033538870202923222319290932739188378082265225473766901257442768144198028929867817274833824365984477590847996465569572966776626390839778761781504990078753259461573386966979894661328760797966285878408210720552062360326226611957100516079143215332420740887750613635426290179717143014524276207895228381144492358592938186792934629338352527119207787865074959514115663557988680019947226547524007333584297244598928754868078721699201736729475159928935215471666429391621324980327377607604128019977723253321621299517136441478865704656426691954929737694017261179656460920552331830680690075743250874018436224344409873014594602251919903870873379165553213935228035842450406876176072189001770201305373173141515217108343981352994963279855072366847221497809285186485790730576196764519519113884715020638288850038414981812349324685472607356921249609139192014281682619111750069934046221455233379839264917621460127398420804041167111969039093902988510785990266633432741731631494522039149579297102708570655641979475965513820569858453884812837812662972146352507624530998428924767233187561947036310154167915941408699128073184637799949309900788884686994272369143933038168185613307183415264723370907798045403359873821186215956554722802127644233028937279847158765785492385747096075224782741830404232449165139426856023468976937099774978544126049762866971907144466419122823596103142369474037107654370737936820587909294995112019475622336929054141871669034983119748119000564748348421731661669135113855994173149660739380616789458187461744475077623756121085885603023192149436877466123426560717770843066802780366193784762544370314676290077653260941428090229727236825265596173899066494090086988467645077629433126359047815442950685473817951662540009855598871639549014497474841279821015288465627135801345669963905625237632665612307403548631228891440754768003174670736785202917103090899434689084380592333521642978450277252882830691510508044844836463831188291307638276109569631730008866843182500841393851943824872715887964582306980735884350183396545155448321923473385161765830724734025995609263841958939373363550146033799209568581393275078693218921511400620865569682338755470642959545159082636200709448097940713700715640674247533555458273054587017724864891369851785149352592906289010353191925243449323158694614019775408799874995108965070801520254044893709174050127716235274006693688854974159219733095647187573866484779869874318187877412044128086420167472307296828273699598202387380766195084785625112934386516789337544013644951660978827959572639381172098661818680034670113341753108664538174772501449184344052085747835336821815571464167072381650853084813166712222544169642775797842609122532878181191943948967985483578981893883212344798391720814238862191388309804427444074036457947690035207369059869928707047500946157511623124719686994704624314934578821250403774885838033420429019131820391666465690460083749975241884978378434017440521835574399842095368423976475319495296037848884390431294507370141783417663615814572531134290868334840911808341528003327719080017499160139811730903883636711603695561045632791208851587677648448648931481991143080640324346739198217650465432102699610885011160631315356551971248169204714963833678635880666073826982025297744199964908307419717232012758895981612948387224577151372244531731657379484062885928746948853640623220552835257541803111186622178682967730477740915433651054254875704937634927367126084432297016532395891910633659233193133873598820816952151166871990733630201177788808934683067419759489065898949718396200419611986458085270397668248769575484420348236063393964319668400323858435177735013035013676525947030880112513143424140327954748070122825774879515089927556927366741624896186585632462820006218402179231510556315282047781774408893335308388522716888814438146792360814172629490940007780060630543040503687455532396760000930319916990217955026747883000992406013879755708258374125038947647382399393643851021583143583813341742009551967108620422852211594507924200728720447130673718585116362389851197097841382762769023138806879373228688840761090137052916522556087480606496109682079669457368977070112322154316412208251039569643609022099880183978632281616928364187844825269440017621935241426597090009486953398606166113147967496144760513141241446199259035535047408018838327036276014682319874348950832841459155730138996996986400033187200372775950556760180768137931922155025114820576037867562840676608216746283917552434917645889294838569995148327405043399811159482131624302936790168113655969676217039292896213865568382352990870222194300422852337206221819551806240357473999981933363623180161124662162142877416763481626074346049668395470967523385015083363583033057047271468304616113373164958168171960373767653887134631894672668657011201634734044315336885287156597151218752793033109395894806888655520886321246013404344795366776474568790337808239036118467307329773151619013060678816293195959624382866039230760418836758435169266099323949755548520003339106704838808464607243446342008857573312063183524176357251916457968920887145629867533758531924082828676261617459035799638567536820368805501278291312782232759380924612727782532816807331889207912283148810166873801864856321154907082347225847971237415917449397388422400202956039450976450226331181372453616673257691965129065778325317397227847670927561243564930560,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"

  )+ std::string(
"(3,EXT[-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,-1521369518212205974387375483133581493973353393069865684722133567614100481274421861321184818531970648425359891001138381982005644247152415866887564105026042218935472211988622136322423756389126511653276303845632227148574225407666278159988144890110130369842705263429052165797888468846184531466696415792763331792889453225891241425720001237560437488594551844768431087688431234920920934844774327114941971016718186118619499678550606149656827150865170815590586910769596346393822930286110780855730734827777705200064328119541181774291723498541758463825505028873200535917578676723183077242561251185501155502923373605078453791390646990691654683086516404210818574045961060610631988464284334522993160874226681984042533619948005291347830677698018782818050860019508402211848936073064797315398850202630121803260572954547890896221449034575623264836128566642906992387476883880229765813565255114484957539539376816258645071297443394803181781249467849450843010141453481913435793108301514622427610718749723404299315171195732993054928061826565247139668191550379334202539484126964993056603815320853734112113146114772319863594269437962869546852774145933435500060112878140495287797512919070135655154799988924915668788972859227554338508012207863922767608976126425693637164541699831824339407836131208168455364592893965012764127025020196293892285320388028143702085044990034889463451524823381954905836135945764659151892688552863676314376262394310539632865569685423929301121192788427191073433228281597909626722079139075000022642311806945880969156132663849100621835225247709445903559328864722132949734983456802275875754963702243292682283201711619589510706167403667312359551878779577665963692396846549762150866531918651764579802718254401983594446017134893734044775533991625695143899209161378388984960868401130192647977244443096280111488010987213311811898218562997349854722950581419737295283464376591462283297989317568797990285085768435125129623290512849069152288270605692330852241153403803901385790846594400525403531321245062858269858357647273792652441404020074575180544271540889692568270185426549101707250613671412245380407563638649571904535528390345488407391416783404620512461861449199230108194723719954663792756943765402965913296189873347254748472050359189005946483883446914258865814854647622360854610569682046256583306977395283210294982803054829463435503553821100065696822959894431977622768830652854949481880434773006785934708197726122167726294171254539540890591150173784177419718050218969013314616938041462000987031569688156627949256902238597345768299707240350424235543120683814020207531040244472387880589938725500924852251377364411365946532101251936677685292659437696322032139749444737554669910460322437455866107905533652434823580722939968335915312539756576026993101196252146666756645748234512524187005982000603456954204368620742939074089532275469163053942208120666134895255319505908550147333627737233882904797771820035715187367857683387547042110408388566674105530263643675771941995999929811544438120886719773334008477940516796294311975864726255498258938830632570047897400933411212208030983712067212070941402523864265155743156574750068923012973131878816813174135561336760258750467063075742011136119464869750929486605594343086037137869158711291904187681383474334701090792544953923473104260013326187047361149570793601849378899734830480988734085567912977320187353700290991348472917553012862129698377460227507178296591486676842633579549419388431058789445205725071852799329940116576126455316738475809065295124020768322134875882265300478607259844039861046091106994799869333772900961210108967818022708220552771078952072825878166745905146918231175131004094749058115179330090083853979566045707429355154543537581620459429942131682666508536143521694107103199407454557243965490476875015002882412230214327940331114597835497471009294530949366788617824321211769860583236049590177924908300570280633158955546358155045778176762131025054208805658613620944488952786842816071309483871777719407882089403349805093085359772917004787931049404518332415291419957879565815595667778659580344372688782631293395207163268931011327127241335089997142385120400096941649575814563286736672857245409394020389977426029758644824457055448019852668400831057066568887147816795490586413028600197614546816036060537994136040824475617199711279172546640995664355541918930267831306660104260708730287750574841492371239022576141361960707791522278182209570486877248400472155015742132191906913637204680663407963802773250027505779009880414295597754423919469917767539437077970163900930049439894758576442418032658826978543350317762948449890349286258236563249724577513370439593851048440680960540517237610770740037297770897630376347903862721940449121676888936704460548269045929378565579013484949674735658065194590251657491316035969535348611016825469738111837806843331769986173599249568569919789132331615914797651913906077598718624272254032810137980241097127278164407762786802185096701549842270934799251526213199196826531012858470512296301965243903900247814266509680523209509709041183365009685084433627542658514940173200344954837936962488741517766479838952399746305172993852751545586187173314952836926092381315160659157392823603555659472052172540912397728520129853019824117595118282306835727241909958585279895686644096735841389723069792657389709723723050581531044866416387289613088249625984172197668646122883702705200515999954610865696999997210671293912064006656697847529874843181072217125268461942799903977936091352344851513256920016849977364701495601057912655039926191053458652583399012792990883266020937571777547644558493204369725903720637823316787477305983723512887362201738591684971211936815189607577125428290272447412074449369425595183720314667828893639302591610990974301035698036933651517254108657030149635856344107499618615700703580404467712172162468749688590432601052539366607290644329288285345792129271023021098909329679587759940767645610860729003502197040009813792964937617612518819899424879545718856658396374062723825692625477055600051613782715647673669550840136969026090171097883463912748011786457167143024003392895848535745611494975740547713612790656563421788068073794221481937139141284920458049196094300785771082813365704304367632086514801339004102064,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(4,EXT[3330267419911093230811531394046527235890381171981064457037992915526462636428376540632053202072859628729120769980459509450870027931098665452572480451570875636513608087290848448183513945526003079484085658261298255478021868127809472841326138394499018888417755760416754416418032527528182008149780586854825712291999999307052515402963158736332361992978113417649440135873358940080394973307290104737115535558144940336202362671589596597911733828371020518481387718612521583134760019204075437364170915462316954258193713018442596445260191362244429474462796613235126166567176859490568608933388336668291131272316297108672420634771128427295641948259192157228396671413378850345273750005722697996118685767029832111168192044635446527597575501310624138876792054867101047476213200803827179322131599576959100824515495555208707343424480322419544671788019015946974879829629424016747900011364079682857615102471639061950992162146943271133655882455049137918748498154558350688857257289655752704491092249878744719698181680214956704964732476790983369899512737150458051895421525151190009645033459497652448845127689535417718512202296679400171961607222962386720047156454065942267773300637229668976521823176889286435717352671241239081780537028106768810861720154618807515026666666413266299672775877689734257695942798517810957128837358633042323931683018468479955978078798303278688532643816707172860859822779087847473588773700464724649444579949743657370847160838266497706077503168599413558878878848118577882536673537665820702033064837318212284050731854870917703653132781643049156297751998669663685885972304781982106633046493622990972291609869977578117692264818066873002425340902972934957348671639466243567401394014496817320476543108898409469758333091793915121112686281232821541471311673008819489370226039313271134787931508216914978046298984892034744756950398537930155033434982932994287404185550648025016789920055804220442935494399116606996452939784144550523745938670157967923586204506477427408582140152410654485412375407007473499861187705951393508818484003394966277923003158354717861438595550765453865785221785194912744807163715841638223908422478507193415229569121753108236182849060135806937748043032077818761886636761426476657460779302577345006498254711110983461239071264605921779917141116881741372123010322116685106899560951066290299784634133738401278788153475075612126566679065062364487391902372927725183069602648583631959754258854021711917149056099654386599388052097279666693972991423182762523280404871060909224543506685678540641183570575406113787881609961966959001001773710216541417032608680474134892769820511943643316256608109430445863469255352643566693072593531966744178547682064761158887197477060819203661349530287659659239010485299953823375344131074140533767774458337691554676447741254455208016302709052895953982997412156203074119548842107486667164609749025437453534356896324553921183798362528441754791900120014409293358163700757633303290311352128003795725822774351260473852075040913145298185485172714137563121557859543350690814848201315646262768903309847799017320972059097846391969789209840783276637027647461626898293890571092008337685414971844435281497009053973663060097662702710375715655635318252949412845880355638487452538669189892555785450866409630759901299654031713302650487287447469518254860637556229405491627575701104071881751370724075882765278910267852501840789290221897721327098256156214352154877606390780170637516640218939101569045323587075791095389027339401203177312036011377542843729659844639961101593489270711952904015391523700494140619581212745618050940796502380430067563928885679337914806750118249412135501035119514682427683978380856214653502471026758178524649216573750646762025589304634370394774415075325447124443251531211150645836292879648388739729638874783471101304489485785402441221633265763555944732085515879548615451452716278215349061557947735373251065601189351439783188619370659804698925644486830895351574660867933087726077046777631278932856331581616100533889853799328574446985839145918710874022266060160720645082189349820106828020804579217887106874030004615927458121592201354583727843635763390897566538695987351364733318269968240352968715615586354948309083570487540629217113576295501138323624118917005138555031059192691055888091768437126870616798322632531109954374350360538595894213642977047271473983183059096147850022710521985958496455227257917670421314337693023374216154343981311581376993787697608498021710919169068760463266077841217415171106390727902061959157634793477853996239234889191737229705937710682304985271946549377660721362531419633189437831688663449340393703661451667536277733701923317235097298028831872693073477575670855159093076306594823893061322973921034254658419102387391560607284269085917565203595980266013407100392643637609867599014854804437667956560533318134612362303131603131019921034035578987110607758601303295805355392927317388469124984165405136641593984843843734159863377938266264750612451359403422734950874659779816987079651147629300047478825417325558236123121769825563222072006665126678738570145351438895047830544676871055184338216407537185973504719517556"
  )+ std::string(
"486163245139982087872828276209372332136811821718528741239706978912881329796336267255757439603541186393008597929825119066199976822359966862464122752111836379376006181630534621722644038056934097366490247950612765650927645259909715678764462806573393903091465721547991938397615601336614933407633205888344770014368583401763297396650798973164942422089126940463012712819257527483740671996350380649786235231013796902444730934076561004199323917065401099713824333998226276930915855796733890723224135295821343397904756189151387135419969520917157689426146256176800432680700694266971368190820801041730675752737411519876481325040020851794394836394877555711253228596076867593706478027612377890833862760076619933522441011649721893531340502748327069277941689424984031481625846464696126083437158283742475217699461750018335268516023073578118345089883496613911945637041703617159746376839404976277058461335984529960019901816397741257957592834973880801721859606811316085711689610848741778749577991317402754766680056940409316883413819840619399773527107359976426627593418357277179077784325636644858806128835437397157143284858286338469622797675972969843107378747285872840003828775511509845745217220278236609409663943316836877212273586640467988263405430846884632256268402434051944439474759618702279972298309331476498059582262574446561070649824599716297797192721420019612092600376435314630378346833985001614353595686438940524134667527912239505670522070514726358027632610582045405653355679922669239082829404051073199681241006085787958075623160675206399581803194248740632945141374762661313013152591361244224235003805720054801162411662955219380899026152828521255628385672512414396115537277248086302240,509956109196088178172162684660020303856463581779241121257763332773637824077502035242042256008372066688919050369765334772442793481924854184895797093705577576468987244528432743227779084318746182852713615432481837231109166279281984001421052268408231812842741358774694954040744760337743428266463530472842962439524861634045011149688310825545955416190392233014270910050145580837628148887912864693224472113010925448925810253117716547203564953198754041948374253918005233543428014079593254543809439581691535099682059365961829919941634515278843998111004256309627604005549918850668703677811047951481596070170151127912828672976568609527556359084518778958856424635334698334784505736087592247249850181983734270025503419940281135032451449732674856316292178013702125043777595941117621405181686232384047890551891675835242957450393543158462388554711554555501419028439600750140938728008718644040219088150754870939561232194714973061328761322707249989766184362749701783906570206412829621178414445922546610512327542565926245458937830723688844272879457475164776204017261772152956504541196909638681757231176100307408821252811368353697432344627033829660110604933246562284540863759125503925406365497739246450357459446085732703036920588214940981149931480690581359472673427546496708747590139589847634867288117317312648763287927008815185053449889306035836697124297766727297789201048899789121553223304956457930103581526750164719740797379696708232655403403493898481524678804059974058850267287271247423825404794165783903283315832285843225256204862864669693713512139181540596805097993071046726439314744821752040604597527581395600358275037196989090532057743443148082981410910151732928668305374070765623329632929868015458860466923517597815548467705082404975138193035634412308803124385123785600397373880843295087075483816452513823903350677926385612357623846028691580425508337138233811852596724854889961279711952010424177298025847796567607437345811187212326072889449998856467222918302481294678470403703420680493852658623505643195465302829149634767718132485898249780481247429260037644331378089179579050708573869552378685699078986567245693143521222511518817469738393369315544332639448883033203675465891217542702085697838829307503445070867039043900988933727283412215669360190399549101663534340649184731218407869694345299581234248423630837013675384626897747504747703561728514195226408428313450078896397413790776856774138516909464465281500793383408019424685448306562496706192566468244867904250090641778184088386005284910841182610509182068521062897512376655927218379873331310255073918545496396291881253095838837626768637184413347013128287069949273270897601381100264553774578562582360277156734996315202208743536317032093730521919392160928628350158825008574752398433691385989362815048938656052092669489142048841295049726019450650445828361858013562041040993600935406031359155245354851336258232309530974644913129797832651587751357727607866599727963957182677532293027428183335580218408519382552981236108006146619698914851219627548798442114431718688968603888408669413744791146627478898014290317683996793714562355487293583228339092277671119005663107420534248025019891177842855356219996471389288962859346142344618746002973303400421637771286484606476809205222829263001525000130499403280214653178735432536179158243168943743530724598837422593187355544557401767261718432610120969159867560340844281709124746821759024334303437501954548409244909242388853235117767827268805797194310707190466657042229687294778739037050958632878480339248709690894963308442009315289257740239171366081814467572095740521847558659635068034033716496027621836730824461450113516538085703390695817372806049723027116155345895585097221385038335695725690163369706856670287341329258589652569666626254720526503353767011018073793095118614905979256274893826256604767183841648447898911444463614469341360346282394763710235875502793925191214594258486179831287447371607315831341390712208005169495090561156329246953004456941338033042190692169724052660719165912710951036165706882659872310380800388862416761458005452696674013926949364783816396016793332816051702808949228271761336584883252858840229563548901502887923828985129978600476410407322424015077070049767512021262407295103782141740517481390417536190515125611776518843394566102166348239713930224365332739653247586079196332092573556753175628920427266718048659269817868891699147698548879215481025471841357191425633076840058341888593257326096625756427005202091741004687915315054536516085828168487623695765070696649782303345721105187864808985842492209285377999732448629323786443109063771094584461291455766677725103164714101528214722809491416738156061466930547587428119135124163106085630296875477257952475811406215499114341843801118254156104470912255336303565803882360320436899844669885443803155020613650276532762054245960390761801197929516147836069188795969134466785778439524477139490043113628417185465978898866036446687326976616535087149319716798170369841718607004027944784620763218039204740917968013488340146639797828728225202234127920873064293440265925380428855493185544289755490371069107115581732456276297563911590257452422644397690582107726475254741583795000462836141290253256695415551903868170261854119684419382637253315144351381433353986001747147698432440790357764761595771499427954182111464871962809891084656564647692202375354994332873543208727422888439568664271215286467121435252537230040066839086965709340159509665047174652862119029370182706571123737560828189378606098702376198563170191110496418416086056204270218812405436843753511534293105019979909236033117412321461925635676831514255284392196762618013357892505694062670037882306354598508127104885662912518282690716895141784420379332701457356262966974912702344361423494796800964415257145250342654994641629861019885493933142721382593143618616510678765339976787606464061156557754180741186228550974405359927108418627852586274160729634585353340278727644416092621693245388378942254998899591818603627424397376204700837870438352144799589881790098765478631750676371836456244250414094347995364976744232546822374899921440760585715472791577285544733296574911223289355437233112451833397232979376354613302171539924234667690579374493431501637416874389456429260154819902605116832,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(5,EXT[-579654738469358490928310228212201602490439784724946679528489555477781966863365235942575161527459843349839577055530511916554105190584960567519156402643419584641662509749960740593822375494246768251041674416806596610629975570696783150778951608175288751741095978975296781341411701544028603987209507037047900013193950832957219859159297987872795333049542984637179931124372825557379339308526480187333398993450064170310306776037811595199361496155997833797362130948314430312194384274190621855512632471771941365875329258075858905459317162537046238782717684574398524437197030760266225926573637843971734611035217088120784544165320644875438237123139826984427661923505602671043741438792854146011801825724062244465063360237384168130258537057776275758595314325969843374968146121231020670490684962049957044090842690553278751294206693101115273645289909417319090966499638122453473898391547545513637743897405223892581377381487070695127226945509227639577219435853979261586853022642537176371249536024716980662392220796763455754773319543287156310808590117607876242541707396020698730998102225283004855247706048253096656404294172589542945343186330084602540003646372076346410850340042932579461344315875547180132549032090614191028454560563257277574398912570132127526158645777627422181367736488636523235391937592936369850689003191505842029459489660493349655222191786868426713979789087171912979155278746831402039801306716179047532426981722155634196976495536698613074320874727193047143997082899243150051838541133820227201085357878126525304264331969442132540652412418234295674359332807621021235756689733775377811856066434827098621771565696976643190350589147949044572141898659263311236144006101991405383249089646802197095393869250917107879585973137937122250857113281559473892143544625763598734315471050226516636432186596180416257404605150216939354782533521304018139306949623414449457396211550694493290461066207639335147658420057483255616444922159839658208306713437637110099847587535287256875436566808332563020391485122438003689660425182402291706095570430173986543495289246799096581989738465901710820471934732449500305635053292180706578900113686460139376437941073737179682291698374405921222368709969315378133877016960488690700055621693277583767523250302869170884394657359136503343418228131598546674719844067067046920974685623367887567520225650911196196212261901657457056265980101021125951035263060426353925990894993523928640036173402437129511451587250592539222312949892455629318601574031209537342852629262586746913137278215272234786089467101818861262303869156068844759985036258025983985814922852385485452245859445557103597787482217013088903742105018247004333348321770391692848945919732061093590556526020475726492151046622203231975422285768841944312960780812149501035739699944706649100304163120617503901004756440551995292273695321259931173856182568735554237885057000837970089445701702612251739454470553080924637686516652247273942772577766487661109039785248465902073968639782653629820272087263970422031925727772563444545205985384827362377594579979139587300490621523823234663871865157799476686483433939973213973316591740658308607044281288956554532895072625281316584518115758414955950510784216702919118790962615769434421076827744070105502710043980516700336438836589342145187616814392363019299764142512483866229143837578466364623145242781433691170690560525577194658006308522159476616802377636583791931288380943980953198753833392742704469915284025238523316637291393086888343360604625753482450231806436971387385518775760000275010302376478630470165614623877312586174847150406812401033700306644686974834908641397599048768043312998358733941196301917094290669718949196894527940723507337856738473616934738857622382827697372729667617257752682885614080448496951706934037509701343255838985582032787673376468780073027857120408361199991604673085343478679967344246138318462218188400959995311168712590068041461700752370697718815968198654730861256194712066195067147922553905363903594016718738091076742828671122187018428576762516772841638977442098035855385178092896497838926052079301324251496632174072618859732847581936126733695435221106094798954054332065601819743050203003901517156924558867745522150345905085072476304265307290361937031744021795861993206719215525555317039529864766644532425384230288887530484279705079226191762490051961548439689229578242817144608035741352293348333413393105364773500899801067329910521375627662250843230327356288659478217601473677343839775255593912205657543671487186029080654369397824752203217564494759800454614675087515358862458405895220886337646596321343924958969154850943165479539092687153490707327296314366490840851874105998981171552385531449829593491676747163218503501565309776906300966193624936854832715710443275291756978270565518508949957081042661120894419766502952141784792382289900238806661428433811886731276280521488014115916933354592090406245139568374267081012678859896925733278372311965626606440057553863735326629319531543816635571072041557709434699842780519281528945145158070794256192650675229704705696257960533754590189581693493594258465297247131382267993540878610870062853572516028821658442446275057412841668244836323859867220292807931195665530683711399582364683444094493958396010027194366319420075695546656047782885959283133902471280767305144034619082801780471394345715489084801923548690401281973512024848465428744668862863930666620220209455896893793821617654164302530231936200943125171352048696713585116437018579166485760085783668675457577441010472404303389398703826046481150711605448818795947146174315028494330460627345316711829130018185636378984954445836644010933013164849481317343290600200113944353554930470649652396978199242023611642874802965461590565737542055945706726761732288171753383101220130451810563756303770041541157310544838495247717049759254203962771410008631587172818437391246573879280992085120439238391920149807621209691475330534916707261081795568856608170803007986696496028175500100607465841674249153949457829420234445837662583769960188107676072448098426517407079286258416024000879253121599834452520892061996272134767029770430959682608997763475568062402046041274803225050955082171995768987777664791908911494657288570648385532077825892083458305809708279649175295591831163385807911863188397349572943081304711690959530877393056209230353047416854875006484531325491941350437800679263036734005374766347855571160283970984197409380276173211191023633690522816154869731334761698118281608584213140161543658163699120170783213618512978581547298997412633145208331294061093824614049585474898249077031675216479977552806727039621904062195708950537842275818782719290730222921136033388942084211568806678342067380903101654953323985877594334351751485731667852206489782682793305560765584881400249572818919753440934558292523236127228408867375045361447939883420,-88785360345723172542431271905583944098202914858851046421679910664580801943250056994864132655821004242286168047218330863574133812647579970817178506849193948249318945052100557348171351516928038689243763687470598618964601030027507989098427350589759915433194152732967561779650034635757876232619942613776046446229048651427054432671440547550868459912244565286609303274378037318231543297893376362935644332284750002860942557630795173474921048832431842280197753571343687388577052985903455304308644522960254020046814297611753480270225672431394007508514778387993540102871193621938679367801465373074844821399685493707198923742935657295524556259734180881437872668943638382195921859387040257545859619703932360302573037465114717668482577461264150666971978090517882339611004124120999585693259897825742553448969252777274344370926113149836818714213058440476261097506550984234093830709618820024830435337969372079166978191936789847747109021800222265055121002220548617716143897486120170374656981457174943768638604969152435156887547989295283559717632805355752915507801604374764153115293218026741923706925367281541090640953018264585903467444405970027341109320869573625961729176720266792420149287707972309139535891715539442831509546238552120414159372322532036336047171318190844384104884601469029869359844532552451159986352421760177777218824749140301048510880361122207401628128531950000865216884410128190732571682200145651666017740684125707398092639158428142865124353079196701564654743922313928789037833115069374481396875442401194161509606937890936695409539940285766949946666962262881100455635448991288553575492916482277404083794526567923819246943546471334637393894362278741179268862389009915161678061120787883606801309407545068361243288982861318328196494850226924162036338954748959426908448517113200011810293652356240928382991473036361574298461123852259407202858419801608833059472662498440117160664798658292786664118444351599198523220514702636142310619120127104515151201632096360016590102679672228431470661754091187348812234816045556853333055414832304587507161704090777055320692120306858360951788387064047511429633745646510484368204264818326521888615202965995350482927903289965062931832579545951692223885988020038226337315912931613459458363483268449867355478466998763637729203492077993950614393666460139179950591879548550740540767644252156727397803728533924532029200047993693905877484105284513471409235748022342024314471117366297668936492516770816636516176009779317849184458673840299307098845857647782332166944973287732136806040442097300668866159074507864658989687045364721841001466862127833250146908849524833489595826800128894316811281148094681745163526234909379612817207355148281251099415912462327566713969569236091633875319765561216080139539118787912026183084165729399082575569104946850444155223411023126647525505799264973276246821583732472797974553253994217421377011881627486355419487185832343372656983978769826620898476711671072980305966568931782175861733663295978384360551632284762346037822470257476450889100498632145068405691963972040871327857441616302645973369118650809733148121685569288863997663965439517459834363206970408229730428065380222037947309547572627433566431092524608261027769466821197318316331686317743547250442173986094767433053187001891578962489910143123912483573717842493884504324099507513697866258003733772184026769003411608905854032276096486480994214519941786683259828901768123678276915322014672297596740365670226428930335175123943080155558817451811385552107969950788266327905280043032888836119684938372587559459952617467985660153760697440922894888652178659320251592895668943753313773325713803848314597847138363330183050137671181320214530836640570943471283788392349043483915556714141249052686496014559877505999139798125105972830807919083365441908547411960202007416022908853538802226225189313838461483295696784892843029052649375880789854444388993347531630058907904317502199777026391917240117144850704594070044331327649062663446783157035428304122377472108148110707750554004894952222159411717300440321892529812466045821587158575034351379323538347737964953662146246412586418806080884589976420765555799644841764987587451278431350687097782870311148814194410690252049716294955573679130248181768629140293826404303672079017924913891142015059913600282588668439498198684411718575215707457604634521451338158523161453964490598430688089476031918776113623544738829644147281001392497345124043205922573062696904720990794744351579564913079697949897704284832520678124878243355849748853373441585171045417984421047241199925669176732140747870032067740094349110865838010342713451803465862345950418729854512493292974875021537269418545778208025444151557994081811882478883561398476695871037381227257593651557926304830204344820494883942527557305832462929727519323471038968490394681316061862410595614796198222606955935994490007983143668704577454449083030958650679467282229053008841488593617718144942152478728840297709806896976666701588904199249041022888524992909490447878994913327654025728170754797088179123663401166540028314017935430331429601699743202900910571142449836020824958627428255973701138391914770903628537669869145141232551103173596582312778217723567495541295003728534838457468493302304888101573883896546536816966877927569141090033661279050127228343777790184842317581817606316233285405081474737158481701306114945326288479746909191986769440787019743255244393668412898409576557900554971331355920947765850059609312061408404108649037254473214077102529709623019952155882427054078319643188103889147253527792001162506314044824367980011756652420773846246776091290081477160137310655252410040304269702882606285718294645804109287938803907148586388498172260147772782461006145948667855219874212647148855393995542355073972246096750222308268288560578587359740318666920869645615425532716011626564837362990050237549580174916618757813545826460883296258448334096816980861235044680074835495984536834949987445231079574406397216249871537109049948237656358608313010349919095047272837366684203899563775655379265086509248292643537355977459892915828319432624365952448737095770564350364974364911731793919297085767723921772617371707706233005221358858573937009896352136796678084252965223792730075333782494569986261003717620,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(6,EXT[53984009516804993076044033238291805006742017043505730457452303046216431629958941836753319237486158869373955606690430258198848869851541042573542557235076556864882818139251105338288490042991747891761123422175206378997400627256556580445836809861835780537136442121633609610444152162764327244040043181458475830768624397527444623501637304607546227741840894642746299006256823601753279386969038169027067720866945856543074251026735523974541968954832966462027071215052694460113431821307462383802013335386450212436959105523782238769595621629113870968503271598640493119496120210610949401008162121874018129526041053215490124026604194714979211036237783750985202120353009408509790830410154008690631499715517730987970524795589093848076800531828410162557701808129179274262466164894512003813537966406234735528871951601516414616922368846598063200173354094667802989863481723287582687031704973446340202465017772169282692906724609260449882890499949806316609145130478797562129382107429321081941625813094016683877354385297127645826744493195579208912295206002284577691163578605953799849484976856771921948419779738289702314037112417308174345475880385732253181126345461881583231340875476243305406065103018316802374652703968491890265372291949335319867503135887734297153816656716226975884440590519402088985665625038870073057161262837695233894600861246182951357420111618853649069891956445152917244258296202753927837712199935729774861140669627371574594714578623310230757105298509113824695522591594850113356236619056276894275284367974574096138210679610645249133491240380190480901714700682895563772970767348806267622103806846265430095891396222972401043930891333929296425392903006611245401215325032344475750536402132534047997098021678876373553423273105101606368644460412236017951673552800636057949333259504582048056451589554352909801489481240560004821671348839392242471208347051733642466834991764794009525000574582918259992395387048804073897618928506382598230194588372054356421826682468945802527502752086126893271091739531464818105539246445605696876899903771790830334552359500077115862031762114660310891758141753269693287316569655483521404109850513916244703569523465395165587032414750854022820876131991465584737244374038326521649884186907313214962734732056052356819576415365673042292307508486615290647205507057421044092468696154562497476582014217752560526736513006373317037074267470555983553587715885306428324106143656878692665658767176594907685248364877391860205704385510396188507484797644145321565046735215668134697718426708060670117119716936382448259141376082958855177273965253115143926740105959591316859660432604337452220293708927312109401461840824586331457828204441288424390594429219311843632435220326954067445140543268932513599793617479135131551287328762554859691025370823990689955009718615538241630572925366330232787511738830048784391285766994856550639031716412598493880255865044212781278508009873405333226452958579551682820914026487534779294946797440839196133996727534569823267464877608566879442012195024095431698432587943227903617686403072696782215445617344363138249815072352944180657074286068971142804931209221579490815748708265914265394737360802189843157315069777647083817578196677794880590703147980515948744591117661315804327423132863126579217962109485832561158722266458399488832371539015439887213203629882489003565895234508419581031935393802011334893239338581909503435431535496874438387972366664772488818353418042812940981631880486336120931599970805395046282587592085904420560194266417827071071974017522001724481624187082804726597773821961688171940805265090730096159836496379735664972999441986820436347658469783985233880275183089371388492577125443076569382874481158318804828447362305512428280048053110301344966505590654265351180881669871456536479411744062787125334665266932395384777467181145009415123894317358710719966884488196423086266866250977321933403058820628941926089526464762894894821606011629881884983167248223956511398799326061772450358817357963895289635128148737802331923759069506903169086799752227802596234498686792898950039622124729411335657071639456854519744631240451746059126825916486245236097580181854408109317579325441302090586702928237145185622221220698401490913792489974814082570519495374007250569206591159073401744758526616664507834662749248107910641687308009442885528477415663421958946451441209241295687633826180640416331254012463854425615829479687207997135717590637315719474137747572699238059296547270937495612076866125023820664670314385832802375724941783640462131413971719946697036279093776086438397401389191369441690535622098227236084927861910811608103407087775903303488280241446169734969823961171363263437886443861710063580689560598428456684874424195408477084100316087271666872573370731065006334320086227794546260544278283511542451578524936332538778434964944036007245222547354515309284007012118093343420035708045963013138683980225636093839054729621898243429581671472250607067468765036180236431733521651937244073208017552783456616992835146295835071569393670054854876567727254394901269751458439157973706200591210659894746949044722544377318394190578509738614021526391456962245841334652283298704350993138905409508452302833924243339702945341645290324487618555291299825658643741542442757868939867018236275895098675997198588792913482943359071608744907956450714472678462015725178561079829880367302329765092714000924009121474195207038653609793089288354898315374232966258415224471738991625068404084121944058245471329608612424903344185015476411510200421265339732497266712480134748743423950153053504263011493380372927174401770813038364736371111532600991272394643867555845982709824662120720850722255524037487499279468185951802311819863757062173633499780901185950255656462447715745889917880515242987374454773037462537575819148611100539635091316941180486648746586764819046978086304160611604369422616783079069235364086824460129460083357556805403353405006195310196377303311753810121108892015315928122330354410708962511277178104108860572980272187352510187130392144460002968124590654805003104361373120398545098686078731213810789810740056776267401696861613908366715826976096489047128641674374124071194458417781639873664768170050150449568794846400866076268245566105368822390149288484093700545267650022185138000095923721363898790042969698707899261828581976465544372139469644059615323134427018920768700789958088824907315144768250279493739212148521980781717872757123839315565472619716396002276955698022519120568246832291381013145101374388110469344237800271652617242200405297664388773653407061204517699544598060492580834624549279198688605725763867592485228078866454377911803465117013631267151989743465403713020626538756283516408842325036216815992983560802971999523353444889178858562336333328864579487676851183497024510378389596170546713547890710332606249535937417878800,8337965519313650949950266962982034227909426852874472749224433639782053411149352279385781781152264994727329191883063661582367907400147989805924878816400276063996499763483134787674832592017062131412832588664282494569359328815396482048526077632207547577367271733847856804221297893404457345841449350984600603257208717108663913646787206396966895611021765269482297786300858080189426199698108747204959712343478935218183919880683038609590556017103825546813097977803779221202606033461507661688630277176379195500406933486253542099252760753310366158413038687037871626003628329159917992413914908071095569780452820704010292829865870886192078611916471678605755803971864209201813321437151925043774201431365157067838831229857398776030596355768015710316951809194483137707577263732632431805126021482478761550041897988705680673011290189462890491601682096412307552073029137060872272242781312191579297183488953416749643136432077142102121484414119954664323871086142288517831695398037119971344316135080322794839534395946134434965443781946757567080415328806405813143480751104765639600108359724079748947269056219763525138998190705542377894050193851595685046103283460427799958444566893181917758953064418409429906898621418048798855697297572038739700078276508157251935603449495253991962714800784953508550311086926993733681328972306496342056926689248538221762651980044026255979453320946638168773981577045706152231776145871096785343333127238870409189149647626201743795776748213520083021381573401040562066339911154602251792756434371938346382227258972673399472910083615013632102576687645367231600112941103525901005627321601464121434212015520728282867371647877669596975114274440292603254385628273072519032696427733224455389080726982971591785247975893167453661644108482096569661796566382592165440648676326359049674759455929416946520393977648238852162532230227843981413346827089184181886990061897228783985871807982524906787912451929483212158525685996864452834033864339039315615984869087623189096058444539586172733157868319024562673754290199348496719747070751120788012407635568545235883150594046925399681418574219748383225431175873329493294673360379845404873041060829732881803029441114945711455916168937921935132619774536130165819865290918988874919620101867772945774899413153839930545667000185577836388028735921439566235810035596029370444097667245974968994699246355498333697395083357668817692832594121212822802371648324706287755552563702291748083403395799531050085570208654228236076613730722714315465826953085926724309897363447211650464623848648304532354501310592361649291264324969178225742425788193851240056948280307174976291004000554829084596425088237968388175664713864725175549218230539463533798182650158884026648920202072150433361422130344235023921878009195935567160487458948231963931206639277341171000972282733211030655645710880124993112756160038558257046935631938819590011911362979111783494374027913798058956350027951100695977077305730565450694461776276145079601933587962546650773971839236355254535959858665804084539710737710522005437806633306509794301261562357291185585769386806798464734549797694119490912734892578674921403452852640775160700041708610629119890906101240099581995042699154101847454007775008950010840094316428001050789284042209442257133753187955708103448824180502864096147140223470675323027446285179792349537401429423542192828371377974530724303531034049220971136151624762803358824219702289574203365786988485919313173732363558448983954366655621678579270220670772263129844126861972356576760841562335163838596092556314400550697305319755801910419982753600880255568490338165234711426085996610946166462403506349342371238353596151321691029587715598621072407658899569100862959440974707595880295297001983526841398448740355769227559932200248461218377750432854995273440395153871295256134955361289429988715803941160751478398115821719139490896251779088289805730728314521974254415823167440887061520709342459925475800022740052901797818800522963674163678735312092792750729565138351875829049239967935553462719866043285061632699728685620217210553352274237640021295753325892804957448643438037492997439857505918739089186884052715499763941085539409475768202549412284353866592563382210906460364179900769009775430372140192251782133507910663375091792131785715901507783232367494224397506501940543540406318423700636357731766720675240817188259111423261259066189912101852820859798267261474986340859174998915710357993795041318300730012260432944355252095899878092942988561639886197809245102630056285093194044025912405497859162237216477187926140559572885890952921859776458210613149962180159175378902344301958239777441703876697125166569474954246654640202763619595353633302859945827233000266822484127972274945326799616975674722905172590437356451297466995441890456997031688943233822416653654479982341323784847680562340624424341416283961628710581477109785990842422379759704108077263121856517881560158719508021146033581812733122133601253779281009832850924475461146318119376629466974264250141980391524311215954857229245627895501938258222307582745235276054922531765919755931958987221622249457486920712320088862101928737809663987597260437144022768593092404076179330230127593134654473019284967913497480653272365682938771099026723634634187987283393341156640824911196979940146204423814698456666177053110034916363563469220148778855377374152806429499917529416125177864857149545843745080720034440477236895029255766095891651705902337438478623785307250993174751335918336777065497619925770209623015513204363851920063389940870182542133343082169570626620865189847694856959782866168062796568302717739523631302081921362549680822981557676064212239631279324251965543186013764506168874617877790956003536904068587550283060849638145279482602974863319437397527445847432387459824410089760277942501488867083710352025737124635737985456978878338010339189445962178233982041236647041855777435003541810009513245709222431683951232910821763364684829826535930365025556323198381615831207853471324756362112147684456124808587853142115032750315400462447395386433191946630077011400949067830701177206476676057690241023895066218771803664065281679395040666150684635013427620153510466812412251501037333037866095792887611357604906215978052872657971802221556841736504800,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(7,EXT[-2553384379946909498664845265276494429380576619558545879836872540748265545315841499655960754985680188804299593589165358732253519003296246180199816026166881138478033065306887879681035970665184820138917077369614305480605524609094515703383175794880508533770796123651922356302030254928368343879425187591209657546362974425769523201451941473742226162901762387657460504016114494795069269767230349752789295027079291566065148647407570226554042259006472771522919304789592053046989110975902526910621128052709799928287796409220296569177702831200906458797031661973505079865789025699398174328497475086856281475311972763753203396023946995389870571130303036092702059561149526624219027297362170625276845944356438411968811916696667217595604711136642112436890204529454381128345042995268720670643896941812442549042146892055153773283926308324790475031483385101252039952430140525286656230989254470790955360690944954055566933289191051754722446752154007162244822059951938890418791359495428558270074196592375717285240209831294651713502128926409678872621385214743511226063142708286754367139155969590290569242540038001906462941445914253257846468489859947521852310902831023496063661108331939823474232059063188636468694495364133522745740172458341555823397353494513779199596926609152123999391344758192764570734280085452120903174381373565792230090528178768101910194715375829909098807277155700472174881087882659723873365677134592989052664902403638045146421504260265692682717676583702490701390994705874149480915071831168754691392561299362326273679117284101390171570944538196970009701676385283114459797490746358039942792268696712158472307037117716438724359580420181589266145624035226337183958026694694006448343124451139906005687426525197097586926238070735890931158113864531220019100734177530367303768969585300617951560226379393297655536311617939715395757085405430754619910706696195250968243859789366412178648391933590859890705927225013548005361532536749732448722527516083049830768630468130559920398168001388527178453759469698002230952420209746649306046335889201221444687247126735727457090702629191881125592275346221252530679997286094010806701362659721015317846167458938370839255884378943961779961101632466962814079837363797030030252943923752712924687563951309262082244526980910202649857723795561841713494837502038040767322016775650774988099956215908101431186486761795404541957477467417884595167957741381664659012608246112261882629309400878460065059811631175895226006065096350998101158569479476506662925212065804622769859307386228214668566588459375782050906843389074563561402060295554152396058384814012976373655575088906592502159612334034933890555871665867303525177371938681390352136166486901496844691273824335076642943683742231589204149730781491686060855164663631377444649679483747721625925961699853693614368889900040765567708490331437547475185403749811166879010866359849014156424119817967083198023737638555074540444044434450335216512837245990547898197126291247798263288163271968419179622013132804568015937709559276223782207952505524370377381530611014595503311612733688193047143963345570798189367610391547789147638051231981838178785841485511307223045662298671748119843619023216876426107201818474631056565368713889438103114790363594903169088495356579113879059078642852674286000160530051154011828032304882160850607262436917957488320042751450101778559166095690217417971669993078553525888591266877446459923749657707670238596808534037796416690844609901025556260784617217077908860235931551469120697359333016857101797236051243831045238256274832613432539208085219156318871896207798432651028435145676872031922469501920443885551341397488039162042993317818521661478681917047547352501676034899405423843622901592749051897162528483083369383908586814808980002833515518930918859273020775187892102136708637549402365159191445049854549677836419868585456154873971381036367531181534290187556097803232461904384162420336828426557035952336445212083675082072681133889510537666959253059129029042362127675904066898116232218100223252391632515727264110429025797394710407323463368523905674547544244412983125381518893832596082964405324466626641423300101202651953036468063134756007529607397941113810546268670654638915810407351920130703684426911718427331898850817842512644679715213484392241774094541648418007882625647466360275085888410629544890694796172064573661382798098377891724791773080809677628843916480367575682610051110586753203833825245478909276891940750513332432814905504535656141577426819366815134401990893346353846583324649972652773273057484993228244981202047356104905854953069974551854316457244694366202531709992853235094346912886523634282070242801312716418175625767814048576428734381156697150062960498046532607434633311609478751791244027484261418775087269725725161494784323294401314715750530270636143353166700366667495107360728842399495146271463954485596791021632126991360934173247952440129050880976768187708319208550795819780901065079281860009314716385672365291803535215621930495074173185917777313449697834569124504291145140596032023381393554531276908846477164091615309968558180738766438268808366667249168748066494392250458647874366707241714515062785754817563926683625656194240562713761844758474381375204580217237358849626024523598767157890150838442846464227014714856798314189354494629654090889727991117037520220380871285474429000030146440827156413825666223019538823007031373372592230859639637411189655299745480376730953946088686543783741666457117947544452411790520274188075413674847759995937158876084223343178044724933942904155327121444172420900549623921893134871866440188524086271432133404613441955564489447883386080929440709148575764317618700498503296514671572738097636061613731589276712939371093066001334620549473684295889357793915093352332790614695476693413489677125106866419183819074931259782827110402821966287905340978527654181301937999855149576875766385681824991668132483101173053601530509384732500583574318312010286132271871973989758154405035378268956883477746404515482156537186682766703477278643011872329911322041438846566604701994693732232589750339654842187671528242163987797162947400917204333669740146375618976993094099650102608236332851576832257981223001159211856176590711602666589587632293195864950127932805858383607542154107302780395667045468196751378698519508739726820426973940627163632204145319299132665713170185296836640360974431191222092919988590704931436190394745127230907887030058508515453622828144976198931578374334476699623291071447677204290795732132884657679880249969420254980912744352619402960520102685063706791139112814126762677175355617636711487816419057556027691928642757487843021458312233244832891435466689542981858870553767705662360519635925048992748128607621461168808749778628685824789649129734644194697330344400270790649971854700118482529044570176882728715417672101875,-402707179675220594267490720820984602603666291378649447284495266506458576639434524613340651884011291970120116092746020525097976093413379248508185350508563977649777716562628029193995421509028335035539425002615636550514023496519681250174472568013169813260537000281782776462371103316684425999725770059022191417703836366931409467554032582604725531799744753468129646144759793000009612701380810045003241331380930831506508264743494585239981374279455692441424687686689909886025598302823484315194020630539185438101710412456138901798146636293769815755378060829121180408325117697003127426572981866033374223974168872413700211849967477799177033018215264833934146412194202441080286661025418296738920799971589568690866654898023709598255737156570470601706879900045998887074612657826479369285063600309669776145139288303037149712238616037567559008577106256060617680796221310820527772769359154697912250829479738786503468121800019037405285492526036620538399235383189574612613661187736914512573703700647617015048899748962499698664728786307555032171204062749205285512617419045360567120600591476475382935682750773544568438398681541619638305378211527124652439980618151271404026592943811771929262191552195784366792008049235587622891062405534510534327696538048174899856904108101596533151968566592973339120548993309504997294227897709953502177611311582206754489911181158488656345668368894007397025055652819665890973047056764178303175730405370897617440957533531056709085841573853112547581882547061768255117796999337443265092820624189705808872632921175516473681354776187257076821751800041910601608410899803326832933811842685277256172270457864349591001001783109278642218573897425585270014789580099706909190968510078575877712308620901267354773735952060992598311750611284994146723422933605849415606533261314298241762761297635446278716414598701222779254397024467477186283965748731991609494812001232693203029395121994675196961671740105956594347029304784815226386215275462913612536525498399416501333076060258563746381748258203359601519577987959459835840458101078575699959443037633232833207999953408116678468159358888087606670133903679589465375507293079963391843485338278398371218512541711427676274311366511070606686007976660739033748756860565912856247897488544767595129646911517952384169771487114774408538470928267842394400084977562862848679070497136504113279852237275208235184356158113647102079245975739594668789448954340515801324438133923008043583473772605614326234086611925923723993982595279189878101818764331472227262155592194879046175370248931119729508757084257163137772273965622921249849390192623345716893612367107630328674498162694640502562352240423291284007202875791868080428023243796006512909121655681247664442746642942676476003563636682004646040866677093700002305130025008765863005833720192282805784185904789794905962859487054745536311737030904542864990490321688093032279127360290684090755341513946459085934750148773440106290911388950044318388018580039890993236408621139206033421692211181655901996424077389794831957114594320799512209593525506030081044521146978701920813659230615293830993664275251488390071775635199780235619126365484999447868084151670143936998093536236205121720246243522625579066263296285654811373029021689327200170528747936176729722285307243845503707953226333915860559894484941086642493354086885076882224647051853664842293622346873670941288327079151581945283013618873914714852371349490704523776869279830628829866022391077153314307775803927065496348522475867368919917301959850942669646574643415637234498438375746066462871068993215223932421535356105604165893498369898893715706942158885819615085328692410803317774059637365296235766397219808030211531221953590029815369354657393376942507122956842725214810553383485129444900634842370187124492801948317400907701194201985306836317439835251545799926599700840820775814099104957574912569667052699482172267701489990674004384056310020326112863973890420876029626584507930330206148363568039102882317813944144736268428734443010832819166789464614302994701559467378654878639926272129957578435590485581032480620728762516335930885418727904576525795404138168503465436721650427405813237514825093546646856332120629134196830159547279758363726729998843841876325865988938952648232698603405427548958135061240152601585574028185642121254204383040296448895334084620672294895351718634165155163906486800142709741008323601011921745166343110496041902575659964109264708121693596191836690888160404074727977220830289162428570791940922887282061918496577097582776045097668231952292697474752915917011784799875257772989928569604997885076801824021293751231604574907274489965992394021023255828698550340726334802584032788200014626515383290216097964828123963868108146580496657916484665661696073837258632506802118563457093433906074953082431695342856365959096777156783334193841914608067275151618697748716971221106974758688223297572981047634867037974761245018737266756125814141383029826889763254972872040746587500455249718387930661351056286249439359730207820533615807373761197644816700703533140169668193522301669183269590204450426785082234571576830915686352563163635304564419536990485113131201071426649908369292664348420425517695667216808656089573190442376505527321171029986316401833877931898123559243236995255709259603630025164925623498633100431462120076712936272315518904911001874723380872418376336780106554671893571794077822873277801078459954373601971168726765187499060442374780105487776295779914280265577734550327232165014913276589487585658700752097128449977441757070389889982459833291976438945585623553880349573787518776489419621368943435775044482090562229415223312677388890868891346534481065673844581895305428364035035226121693668254998369701968526937718179937849554271525890292757438314822223127217925935720789137576262773940460650960620732332621322304793857633877610185736553707452578710120517356956877554654997114691052846331229854643555028129276635002481367277756398422228700318707558085617773043972298005995415908121115671798846403755027904559138914318658082263023776726401107072968863002589913339281717121351982158679748184077877256013471839196204136895173765263972451484679543728366965958623818662691619456646233378288601048063811661609303106559674296980838874859268625,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(8,EXT[99603682433356053932349432252192463567091425161455368452731254808274206315936130500314391415703104182647184341343236183045974578174867219535806697824752146865155913807376176939974588844651332224878305916271996802934828238627021702201729695370130288013983280306383620203387105743350963403403933864987995582956411261343683558447304030239421652569070442843795328022343724495359734682162800262862796914273026520678841645622000565896160175900217967299435816717917559619136500889037777868865420957325032989741889201224655133609094805339639151076069638979610561324415030837067993052644370047050202490270035813758613973364427594731662833475253521745120040707112704763457086268250970745225057799734742584627108094251148272604290659457497811924203628088969694763632184871350500107465868162050337323582733866671566603156021769768648608099787594091787476120703230663854937796521467257280239115760030577732399408947375645407383342589870276077574228546332191051444212223605772348388411390516663553942465858488278681568112952235812645225453766394309092825089760800130065551974234784668746071168741435899614184894217119236472790765113463835197818573683333036787220144145566906123851954462482304198983910570395676045108306835604119089618143337697508008014609507481307054509577243390777628820073366112885629785421271872233632581603680513890622499794113124252200415403496803925654818011186544070357089416229772278995777001811079028980523773775546010930993042394482178511891928854959970327744616382939261394313942452492985550362059752772048800359037894934710323715085562874223479624188272200784143871370694980677079060549979480138261067029016037615100991377179734009727612768851404825413901589928783408117793455770972188174854507402889294802469567116689595520139468955893354023906267762507010598656910913843587558873519280259233018662476609965432380496272083338669868772500607886996155090252001582691418118052791229348226780467825730537827715383995910311579011345723642862509893097913227853270399634840113196818340097157264495108785166476883394569141693725429064839610069130135535786158160719098104602552597591566381985430427545294461462145976676789452147454838822728052138546337310389436180456327856822268324994155643218552264427735269481393761551660542906687869239463171901649635193085888075727555962557448992221747151505857459869945182365266309503450214406101244497918411042769920429798725219884146458857521200952772585050531358803586713487074970419561248965244687949661426939051085196161357342175357997477606985097458477698788802530443963078437101425870014437634342413076185928254008896718765324990952848678722262754977877231544356828212460892715570438910477035375497114392340252873374621945769101757855297704868469933112752692473828839000151708644919296684693546981017346132833047449013881324150410631236836389961841507120971943314410207670141253824308606540737781026525114343466038336602626164117993735985423398820814496468002613162724853046687125853588143953781165625622793657532064208789590143337385643796992173155777325194322205745576097286529813196536222026147286437550736300642712941737091782211576007941103791544660195836596006679370090337045935121196173307557919998479558186886335339241824507499609605731592796590822395782356177929276673291092095142284007604932884957017086389434780469730432565931751302389190912674130601692767495913742579884048274000611627313694750788226310554316536331193568730537546858508718636953492706167764063821417016744144631929535987521163178597474774969273679429357203495226132620565395109813326518321899835169298832113893846565354617049087456515763026526771150251414315471917798030298839244200949151901737763575635638726169421081802148343784557463543888985118110036211285472743404697083459471270859381319101133334000679915058456452418825488027219334518686437610063747931317332214708760251942156971880379606297618837993110578643745705713790067667246137106548422505614178024737293196185853487464720212130217013976350860153574259250483439586810766852743562453618696923629175249936787994769040547813546639379883698247426690246931096118970344469599584349254488734301226349172375402872776020608159261072123932943590922205814630781850700390432914087555742446310019984832346991268704821573969465839265739195375842620034485527942805102227128082219899527370515167675269353970474631676030143769673011757313335266107395166665751009694628963491941530482254370583024188765486526790540810610786903537984649306821649610107561058059885082224800509525935225965597203247783606479514120377944931664591885257190596423795232995515703529113727029251465365427023819344679267841945606719611061381102047372847215906998195563012477267917075429547728699694651651259120068689893028448670681302792782694029093078994350381369335171133330970348245431640946477507639086558120985172202382519790985670277089676735274759099631592298426772144626563546367700921640639796045526657528178846804504121730281628975979245796907552342478341419214739757894317343835332253033711555558109035403413454480310893116207330594957698408888100193676942292167005887136815747700900184902686086460332564726432690036475242069767326682809251552558887358294367364211293053467666754823520256882108229289400108873477329317007865665965362032976218622298466841120833732237916320274603497625623136216662602995347902661323985041439296411316932683463812425879059217791754369300255291417024869301376950548515496600235040051452378285231678898729771079105826383235035958207275487043738588891422813644461645221732847229911363149367426050615118821066243418148830086288668037450061362993485283552105570290154508592622269519243975976847134011170010391666812573324602841721993480554509218160922092469516017189575289009777896229917928190042148728725964785947934952109846350315058962954169105146774285334774490437619277246450773022821280353211406734421394483909443664111610425252374876728167306183451779433496388079177089176549869437851069284534834704640859808408950629697877673652553990698031255294277866236364244975567741974547860111154450222406418570816834995400504371676106226112701832268865826559574360128524963282599063506649465802167108250291361223631604192906389731290759870424761661830712439644599197779415285546625301158649634745764770564793903815217483852346582622423181094583733800514680773186079094139687862524669207488262101743404535725162768768480106862605940560092298884540303816344885120278944619769011023310814497927065289637339124724683263883983303758640964620513789279248956208172660065977245565667128257705292355390724233407667816284266720969665411343167768704203650822689607691630818390549779750887220910138832432738826070565790683811260881113423459669083783127016190596638862725464058830100912669419576136137358236254462439881258843630771600047704465556216740361855113374990000,0,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])]"));
        std::istringstream
            is_p2(
std::string(
"P[7(0,EXT[-6988627372201016695660754383184891024059054585710897320826469932181689350568838659164817649431118755823568116067101253209848236967405148190926072465122332050443271991030598514835564642048385457612577749763851827203859417582340169341304015354273791146435017273356831367407769370331970934616166297490719396604675524270580082818443407860035462241493025772189971111137676378875553244212633494081157012147210162425441323991147953419833566608521415002969269500281398561126092487455398205079793077338546310432506826032172179063328592452957442059747799343124360547118596944598354704615240617631519647391700609531563415030857914990014578051275418519598284296918656785079665640321187072544570604058306131108326186628821474485206682632662444971373080180910358055717662203339497489764633540157797461665325143070524175901005028284973058853877251963826826025463684986892131365304051012804479295202748637449780882267760735464746687395856769910479757036351211611377478149270999765633580261495346789325895270770320028886449412884088072467195718205042344490027410785943267362204566784656034316917362431602943352401826533674224955355528513230629705733818816504886201452707294896181129129586760841008216833062109721321195882162933811364469882235479578684530287844428713925763447508508954920749088193249643014183100559664018784938373531256956801271835385624259146274055933877769008803089380299635671399116770958589256702833930812553674833942215827527709890815597874582298661408635634061748807755015162373532625504843132183384618833917839159206681402714606709119909692114951932711001440665540915232651036789024011421266363133181290776023113054875544979369312649647519016938583931636291602036465355530993754103802225285608858498746862544105631789627296720736765404154099852615455572847019643038719092735017172260765345237139686525658172738006168991043924771869350435258271182449643621535286777491345020588467715615843798038749223644491019839000461275630075837150326998781964991874621423047773337366364614032903892962077600747453460881123243188358846316955968515838707680042047749171614729003773673101839781619721114692460219848355935641720580576438400945062503250397606397103051086361475437438899904180489848109928791118458896776666653861491346043270123395559478701179663912108872633877633329013228299504052867169790236584493380249326794598811863249930985640241311741513335828134257373287312427827726722293812618454742591421258686866719978731483665396171805738303866418463800844110112167751444999862295981992361462681742695072320954863665446722283830364420949249923555326559374988836947758837409444095711467286342024172220915183523663607696316889919126791729695845901116303101033091495824413838579819148945786028441283226772249288683010462962639757421536185607200322609518227321237707123031423925888618817685763352116857189999362168838577209657682023369830147353981326726098525297338630689339320124533966008034820686685431282604523121284133462004549936875203663367395817766413933606889829677241130320599223295012288220182949722347191291157982080726961450627251450844172457913280352085468277235934079393475686430237979991347789285277083116074669086989503144285223340576615451538461904550249796998543189421157262404537439893485061526690448967715812323227349363443875049810602022320997382227700452477197193695070939100245576767173638814240996404016306288726358358958262988326756847583671079310844485158520799755256845154038337056953591092711644375563145757709529199976927531695401699207177158862189804762747963991662944870399559045277413591897650146285760602759887830886207658570263857006498780100330084753062890589172237182173630027885931145644026978232265386857477589645779846766018329424682851272718935569609842008106920125308479701165939309887321271865043456422390087632484171930595312078954984000014174407628639588076634054359168890524188382737493741109971680274246264426523653481835923168647013361982543397769218457117531248965934791584735091856244583438498828194244533797459092605655413946109790014834054350425476439598070400709797107928656635971847417858996077486053940447550614851426728282445280885268395253237621101336672582059407650250396445793709998840308609347713010198495076839435602212472702097954606428774079197678400780823647822755264785932227793196919949191905533121706932992573801002329694380315181817686690857975152570610958693503904995175912151767254876752468105983807577427414724386051783292500336549596990971018966510193285944297433183496863518603307450528010532506401996021155437900481732305444691096982556637402048885884632476855225767958659080055794015617504824851786527898296214022469704773628279629038934048718101593435761969843992558076599372052619287970115651162654340004041087002381775069688142998166132569763866066635777712133915873918395618659168444111779733170184161725663013825372468764919722940810000988166608672841610438496557402331809443442066839638383532450474222300453669305761012875543585268804329753925742950436229987437225941173401020022307237851777309118230718087563208878336957451095771569629129022337076838638633567188118350142019858321598511047379477031314173051090749727763589212806563598454879738522691309453105975580149003635831861744839530032768823271598203174138007649255056673736677004055286688154290549938863249990265967917052215221655103894850858235654653225574558638504269984002928713954991147608920354130157750493021532007769018980377510619920087422500722246155314609533938693816773659720938062973349979634513684050810097257526229596952454746613470571113310204458984017706939704828508694314284205526857512307764060968557343654036448092717351763780504038727411779245988141055279733229689557985658722571986025517611542669740988622246541757849722197360700958221928219302236356636263466212016044465122723342794160450500317464067396616309282280515575359940655843595993257161563525888862162268409033253428223938130428220212153613945137740934301676965769539419060081118174282657113116116631606043455077551902846848557371360780343315467353565276096379245504917330348699243856545165665361109010549416808221639468920039434074212102974533506047486508072231394742450476218975737463849184472551489457547664195874896166953580493492402982279735707796744439353831526190163854455845358669297580815856002796881225862993709133079272788240954589360835174396371537083421876164642052693790950527956310454649025327119067119705971644589988072749939483291141870098038746805869820027663883447069862433326051279220797651402687698391484361750363850691337673883248956607615012703664047874070486158745903706696582903405202747831897444508639730804990617138446882223862857854279017277138242896667163101171286507929053395826620771049086253842650628228628758301247990846386638269922056522346569776578300277795764043281459004084656454818997304871939392,-1070234065687679106953840986921778027029987416210678499160065712603489175217322510453532695568862394283398880950089234236287048109806002548678736532467101600313873301667181088587296026266171956261300222721292570238187440553519112764936582353599516351660423299540343415994467503190866698567537450298390650564414158070234236373448579790399123917684338560389789209939788399640996619506682608609659711225231575137658746101583966944916834656006416117544997945091482349640418261684884936207807928615098482442043607387502363202504363453849127848749822242814541034011816008989502889859846914099264229592468371320700467638173179236087220231730553199877742402406453750267133094542809476914171999812850434483704209060489833231154379031631946956337084385386266570900341604101232509841710847184338393050486534717939632295944509608607523721369780626286065353735004508495155958321967966671843131764049639054568417006208871817872078873756190690635956262103282889425294765024533494430662073588549889470435725426452609087948500277547151611699681654472910445407753234988491588149954152098500204693875550197588593994239313014509829255521304159581075475138123295363752502376955254014483459581907738962239074200328084342780732889808073868240013754729097785866754444104667585586675660495895683923240708669464554785815464598095652759761452289028492720564048577381083566210512590570387700405972081106441483598642511802791977075736980595115389660448608783721867616893044711071171340116477355516159969365174383488437916957853066525829130716747434320264630924379253119800180717938195556531612523090555770113889986937527674295306041998482906856228773255723419775489157281399628187608622785951898397009154920798184496694236253479236191515690549471531778194007036158423128989498801756531751618616887217017236209094369023553964890330365293122820668887425820704647909012781929718721450940181398840292037752146312978618728128062127326844110532446521825616474685774129430312356005874456721276085621801315892128621144190915036869027758321943224292506958188592577887240032582485343822751868358131599704624509819056333017214944108023086738812967801113257383168568741106023142926282062883833285386161130016150303731944655840852266175587315708205458379934471086606452220518518606522189721339871820921549697800692547968201367866201795318960843877576586264391775099799042575175128450673003261913777817651462357086626691422064111589764921180797381032878084706729858651027509506596200308411255602018169170931135447785731240614384745649611717229250721161582250526043986491383217037076737595346009935639383870643570431429561711776675202890784236325855912210810588415622200248384693634880406355458876858534769439815821985563318683641605137893884793932427818797916895943366155540474833436791121232186078424449200633599982021250533305532233329228082187589365844451576602368978563665073049110995825237730634288228947548010325487867443265286035604014031126707509442933693344320015212437657836320870010909596446273155102680679143146533699323496051787503345140989769120118999974156488295576270912772523180894208329347684213706568505334961274794290512230654766440193234853137215470659390579814714404462930521312124117511216777758221120521234849875912798658607821743417933459802093876594147208959574781769593551872762798818834944871956600864417781501446916123798524013158917351189859556894278651365440544969470308578147267981957963341902440700031013433514799021062735124026027501455534923268664760970684341547437473321625074487400750557010875094604334261287183202123193017557974312625275056806412626445747498939807354400746666602045639494572598061709692918901274433580240005910690627246776826237951731001579650483035367666212142949432428797172668785081509111739004237823543958860284583854898640038154816265637503995576951931176763881039439597775205989499575182276063769991854350625452711901668569755324175579930439561632722801130189617458241656658898389861593604376234643030399141771643960046210479653444026310779412659191365242828469824730286593163242703231189014070896295443030064110473922181260854687709499538597357757338807543529623124481275634066175289013351264451042340234063479011869909198990862194436648214725721512971605739985415318205585910580336201748208536839547038904602133156676405231950253747510560866955644795909037911644342326923706438997711012396119784763574122651869214271718838184427064508427231048941024108331930882083047184237095602717463631394549168962813757629195380004860528950854639628404540604397475808695091995997286412773921929125829329209463541455765996324739362206715499864670672829156974987434259025161365510904969037937815259049593740002598874317377917928079174336460004015088412135210706512077328182418570007507670623619352152918220933532105319150558261323936248261069303333412107093532141675513773473721444658138629329972817328819733050832121586527406581724854791368001954812942077520602479745332111231930109555967770178022741288869098281216569119315844233259714970215901009261746706326093680905629765729742796284288828964266752660267098483341756799249318796037576013662742067925662281456221808674930367772456814089608017781393944481994512571488847024605429454344850598782305634962239858715730922087954713802330213363031573927492273482872276510256938127610832057196828466819572040247567372160731900289274589908211998310125591238506916172040842638072085638355570747882361630080690830210581731106002031590308770910609474382378235502210419027018098210531516561321905083114979838248790630272963169902284793628278904794253538560990705816971200753814682532766211710228884030502914721749302542730817070360689482187359900986059108623452162607420167001766385903575536999379136486500337698961157762187013219214279491754489079059054737503078538428914518318861471670511718957556190236450675276548165840482713506059548433025934343059623143500820725294466713799730795931283827759302663417621889868586553646880425971727172441044402373200223235199019672432463807101974213829447072771525226772854885244313226711385457692497118613102412881733451332156355419097907738816117114299812430597022920870551553882848408809538696663547565983762045281602788122935874547270152561229369152284057037297780915272498357818891273664,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(1,EXT[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,4265519249683254669378232063589301676723814070956614490802758874963740972537224239298905519916761560480219483752052519677518446448449490308057484228507128134702786712705436563162819660084543973080485519377814799895034892927829555407562825973748433115109385414143584149552108738540493151205067612781417138245290688662130650363962733805939494649790724300178739272066494883969387552846590473654423660808096389126739161435857784711195452392015049040191436918232792049678570577882473784246765702478399075168348078654841609468743911878255925393600025369811736262663400726367177690389644094908311719857161115048365110307688211681163152188715452780765286000516946985216144591739514808988186723901717544564037245298155154928053964403099979075499590766807126745171595709136359100983766571269356420077538866084720340251225639065638991840657062943727242946346880748420926709882459397004408495256202352888067077740405846444638581865478376756164530450947533802514885536288396057859735634549667648731968955181695992931139145933553252781679557523563009980157506518923146773933959789322657521595932571756816421441104124720652453223914201032158286430664841481775501227270852580359434286029048552415790456762288984717185876373585869258676705054238415575730149919028231327115977360039894453095048014667168594489197857509736157443398403473458950319857870430943332858783242649960654755215208256039955446506643242599034272882957731409312853383909859475388034522359312921841104663661361380151486501748036872448688819746029189204503839807741746758331106427870456071684900813752352144378003540402610746346283030434216687962705989926559710144733694442995618570372971581461152393529039038227769430041276577700076829963624698649370945214713842499218278384028563365238223500139868092442910466759678529835242920254796841608082334223938078187805977021571980533266865483463262989044078299158594205417141311283958951931027641139716907769625675625325944292705015249061996857849534466375123894072620308335831882817398256146369275599898619801577769373988544738287866076336371226614366830529446741815596090806719747642372431913109445604255288466057874559694317531570984771494192150449565483660808464898330278853712046937953874199549957088252099525733943814288932838245647192206284738948074215308741475873641175818589990224038951244673858108283743338069966239496238001129496696843463323338270227711988346299321478761233578916374923488950155247512242171771206046384298873754932246853121435541686133605560732387569525088740629352580155306521882856180459454473650531192347798132988180173976935290155258866252718095630885901370947635903325080019711197743279098028994949682559642030576931254271602691339927811250475265331224614807097262457782881509536006349341473570405834206181798869378168761184667043285956900554505765661383021016089689672927662376582615276552219139263460017733686365001682787703887649745431775929164613961471768700366793090310896643846946770323531661449468051991218527683917878746727100292067598419137162786550157386437843022801241731139364677510941285919090318165272401418896195881427401431281348495067110916546109174035449729782739703570298705185812578020706383850486898646317389228039550817599749990217930141603040508089787418348100255432470548013387377709948318439466191294375147732969559739748636375754824088256172840334944614593656547399196404774761532390169571250225868773033578675088027289903321957655919145278762344197323637360069340226683506217329076349545002898368688104171495670673643631142928334144763301706169626333424445088339285551595685218245065756362383887897935970967157963787766424689596783441628477724382776619608854888148072915895380070414738119739857414095001892315023246249439373989409248629869157642500807549771676066840858038263640783332931380920167499950483769956756868034881043671148799684190736847952950638990592075697768780862589014740283566835327231629145062268581736669681823616683056006655438160034998320279623461807767273423207391122091265582417703175355296897297862963982286161280648693478396435300930864205399221770022321262630713103942496338409429927667357271761332147653964050595488399929816614839434464025517791963225896774449154302744489063463314758968125771857493897707281246441105670515083606222373244357365935460955481830867302108509751409875269854734252168864594033064791783821267318466386267747197641633904302333743981467260402355577617869366134839518978131797899436792400839223972916170540795336497539150968840696472126787928639336800647716870355470026070027353051894061760225026286848280655909496140245651549759030179855113854733483249792373171264925640012436804358463021112630564095563548817786670616777045433777628876293584721628345258981880015560121261086081007374911064793520001860639833980435910053495766001984812027759511416516748250077895294764798787287702043166287167626683484019103934217240845704423189015848401457440894261347437170232724779702394195682765525538046277613758746457377681522180274105833045112174961212992219364159338914737954140224644308632824416502079139287218044199760367957264563233856728375689650538880035243870482853194180018973906797212332226295934992289521026282482892398518071070094816037676654072552029364639748697901665682918311460277993993972800066374400745551901113520361536275863844310050229641152075735125681353216433492567835104869835291778589677139990296654810086799622318964263248605873580336227311939352434078585792427731136764705981740444388600845704674412443639103612480714947999963866727246360322249324324285754833526963252148692099336790941935046770030166727166066114094542936609232226746329916336343920747535307774269263789345337314022403269468088630673770574313194302437505586066218791789613777311041772642492026808689590733552949137580675616478072236934614659546303238026121357632586391919248765732078461520837673516870338532198647899511097040006678213409677616929214486892684017715146624126367048352714503832915937841774291259735067517063848165657352523234918071599277135073640737611002556582625564465518761849225455565065633614663778415824566297620333747603729712642309814164694451695942474831834898794776844800405912078901952900452662362744907233346515383930258131556650634794455695341855122487129861120,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(2,EXT[-29803577322107525114307903716229340576033605989427131706979310836400581024341860850692617448616715244365180564013359898761055411819956281220957211118531259081763321970797311441100445881761814173713590680024997621162059196763643646614267180456543613351380396086998139979208695613589870850031901109420644838373065820133057249764353099128191831141320423795972455654525389699465428659401805095489754500094307682582757227491199568902121075759164584328058466893941419410326191079884534002685061215078397136748565221604340032445967396375619020749188060964421157625434843485104083015827503910014365017985332487772249179904205924177033571566145663511889432273962432196176070879057507511238531973578397481758524915771209632320849265787056750330974739319797804158727914490013358115846083496613997962408938531031210567381381839202798415063695607677805188672546381648918574175411853343689260826264484890531497871529858284365950717715320376675192006617213952612506471538809227276424881439997005995070563618686449878978597546590831944201301258873214819790485969673761878385845276634254572504860890628926344746023106886166225766594416348507635764708050048005278891072846278635511476142543497755273589648978552586350318667798099651582988604285506492253269312617917213923933539505942311564322344121301768906202162815390185800582831534931617426830596770111130821918553538187650092810822883113187917296422800933248783817786046347957001241180579850974210631496553565626897486085393119478568552781097304629283457702851605170455382791817154828555090754341119459787006518672866953530159990313376533296782078009513915444355433761612320294028684591295197286735288221057276282429247533720892825369043938921693641732825660556968695347646885814511007101722183895899358551227989940659881037661397449082970788545359486292703337949350441276104896762416617364219893211303883474937138241828904095876223628176034236008542130687060934874156483654639891020235835483448919661766319925192813941690617172263455654896319997310193795951985461717579566259888958071846333641985469845210442760401376737845187218816669413592435675740634647664665952236140893100137610404483572622803307986936618476381589431870232898715724290059302998688416808555819311720228816850773826543034680095399609464749635139440709295045018074236593247777300447070180156464787973901815374786762890243068178506655564931148870201381038372893604039577172280847611720203767335678590243078235185158012475898895719452862032107014774318988376633447483869901952351523708806763869572955205551159966357445267007555928431026989775893861366252459513273211809038978849053620508291908503833547139058419011159096983754448878756846174991046108646821667936439808439800649925175949356165456486614546070329230694880040451545829192775256362456152235446121595424236895352464204423457405379630744232050414470047447672062366535300451518378928892297401064863355567250093328494939893623343570249289026734144926894130514529299834177970992406573063403797085344213411963468727153792167681501530524448673703341426554582021052198937573437315344443014479055673990208987125060668488795805974149862676732654950400434744995532301518567936352724567606190964606633587644288783896745254315086901364709724497031149432319074760449672845595129155998868592819636490784437925429025429632627550551606081231796096271847780864621121776526952690234741522417590350518710929052226335961232492072943289305431832925124238125078878503466995716999411906248582091971633571801296928836630778655556103649440714666016792918128307224918988836263253445519327920895044868908987975223505898946851297096977764855713965829581387160367347597048763848104788998843711589683583106031672136223562348707083268497283700866941577453615180500126780332021020333253846341795716604212980093644501981984386932088163967900892638448411194196856687460461433194909446351993437050461448074225441970512138601021936499016436965338055662363151040718712846346024393285599844156101228928320815210392583050999289695016453838238684977533958957903207856894853662814133462372121796317887903104722101770014929064319942495195678607453445702218764584964187586139677694585847642498183636668137413344876840361322053053768112704094110883594550516923149572232230157057114332134817719227928025172079090571836114078894918889882772559515675570071856678561989973326153049952240609297009972359801277658150715085060761543998562155959623085476864967318269125337675449183375622233835848887360065570779657516622445756641228448167856610901560721052986362371582269069270178892633526274200237395974411826917925724214316034271719134027626042312163549452223911953027735614346642941742645493511049586092825140090271688951829926053130548797855476157149989093146438421458264708119537150128090257777079287084411909185601625402782654182918544445129558435650102889454863257773241281307783507431997448895962435693469774740319476162800523481617798736320414006461802916439073710170648549891624779746415754235966925462714216097018380146193010232595548370479601593649665464371085653361772836684804435151561860521911494262188925211709093417465802754880306667461285657891576090716601615891754005018759216086619696029576143969746919105135758432035627133758577873586445116528802366908055450901503834047672608483955173515785380432985274584770359276651351996642976903121352859445256139352039624391283463102512344554131399368421205785057776280886921611763407630703277099791381996863462710595399459971553795959198817190522467634003695031968645159010193122322319396672236691688682923127614070206385014626361250089354097235737004814237813971866709521349482459468625062586833409500436452934655385512362317367815617126410925638017748293881058351053907840627846213132343256626193996198844936156289513218471022296245929868919454395935531004090254885421886569922648578153806148881705802795368462711201437443515930371309138562224936626305521149542017073832859150953971142743568961632601530587996852190591985449681716324425068285901189958188327435089453770730528059984603012345201790637662130307282612792775066796083616222015511527697290159562825271424600496656425504971694907072110901219675472723666565465550115272432025143302725167152079019978387944784290645918890195790903123256902261618638233260777460336632692818272445188234858352370707300749480549854201034771170844074118772954086341970904812457565550585772458502970649041467038869449491258461373786112942969446086274023452293606932041502686848714244209363011565119348889945060684848603133470487892475421476537867190871107682314147073338632655575009855077547569067103046715337341038276860231998873864632750565888996307419445318532488413551659855975755920690583893948800544650596455981830791759266248423876247803429961611525624151872280939541577558550405468343478004947573036104049018103056086191078630704,-4564108554636617923162126449400744481920060179209597054166400702842301443823265583963554455595911945276079673003415145946016932741457247600662692315078126656806416635965866408967271269167379534959828911536896681445722676222998834479964434670330391109528115790287156497393665406538553594400089247378289995378668359677673724277160003712681312465783655534305293263065293704762762804534322981344825913050154558355858499035651818448970481452595512446771760732308789039181468790858332342567192204483333115600192984358623545322875170495625275391476515086619601607752736030169549231727683753556503466508770120815235361374171940972074964049259549212632455722137883181831895965392853003568979482622680045952127600859844015874043492033094056348454152580058525206635546808219194391946196550607890365409781718863643672688664347103726869794508385699928720977162430651640689297440695765343454872618618130448775935213892330184409545343748403548352529030424360445740307379324904543867282832156249170212897945513587198979164784185479695741419004574651138002607618452380894979169811445962561202336339438344316959590782808313888608640558322437800306500180338634421485863392538757210406965464399966774747006366918577682663015524036623591768302826928379277080911493625099495473018223508393624505366093778681895038292381075060588881676855961164084431106255134970104668390354574470145864717508407837293977455678065658591028943128787182931618898596709056271787903363578365281573220299684844793728880166237417225000067926935420837642907468397991547301865505675743128337710677986594166398849204950370406827627264891106729878046849605134858768532118502211001937078655636338732997891077190539649286452599595755955293739408154763205950783338051404681202134326601974877085431697627484135166954882605203390577943931733329288840334464032961639935435694655688992049564168851744259211885850393129774386849893967952706393970855257305305375388869871538547207456864811817076992556723460211411704157372539783201576210593963735188574809575072941821377957324212060223725541632814622669077704810556279647305121751841014236736141222690915948715713606585171036465222174250350213861537385584347597690324584171159863991378270831296208897739888569620041764245416151077567017839451650340742776597444563942867082563831709046138769749920932185849630884948409164488390306510661463300197090468879683295932868306491958564848445641304319020357804124593178366503178882513763618622671773450521352532259154150656907039943850814124386002961094709064469883847770706715792037304899121721051272706629362051442060622593120733417163641769816176502774556754132093234097839596303755810033055877978313088966096419248334212664009731380967312367598323716600957304470742168819905007745937619269728080979303588756440000269937244703537572561017946001810370862613105862228817222268596826407489161826624361998404685765958517725650442000883211701648714393315460107145562103573050162641126331225165700022316590790931027315825987999789434633314362660159320002025433821550388882935927594178766494776816491897710143692202800233636624092951136201636212824207571592795467229469724250206769038919395636450439522406684010280776251401189227226033408358394609252788459816783029258111413607476133875712563044150423004103272377634861770419312780039978561142083448712380805548136699204491442966202256703738931960562061100872974045418752659038586389095132380682521534889774460030527900738648258165293176368335617175215558397989820349728379365950215427427195885372062304966404627646795901435821779532119583138273320984399608001318702883630326903454068124661658313236856218477634500237715440754693525393012284247174345537990270251561938698137122288065463630612744861378289826395047999525608430565082321309598222363671731896471430625045008647236690642983820993343793506492413027883592848100365853472963635309581749708148770533774724901710841899476866639074465137334530286393075162626416975840862833466858360528448213928451615333158223646268210049415279256079318751014363793148213554997245874259873638697446787003335978741033118066347893880185621489806793033981381724005269991427155361200290824948727443689860210018571736228182061169932278089275934473371166344059558005202493171199706661443450386471759239085800592843640448108181613982408122473426851599133837517639922986993066625756790803493919980312782126190863251724524477113717067728424085882123374566834546628711460631745201416465047226396575720740911614041990223891408319750082517337029641242886793263271758409753302618311233910491702790148319684275729327254097976480935630050953288845349671047858774709689749173732540111318781553145322042881621551712832312220111893312692891129043711588165821347365030666810113381644807137788135696737040454849024206974195583770754972473948107908606045833050476409214335513420529995309958520797748705709759367396994847744392955741718232796155872816762098430413940723291381834493223288360406555290104649526812804397754578639597590479593038575411536888905895731711700743442799529041569628529127123550095029055253300882627975544820519601034864513810887466224553299439516857199238915518981558254636758561519944858510778277143945481977472178470810666978416156517622737193185560389559059472352785354846920507181725729875755839687059932290207524169169209377972169129171169151744593134599249161868839264748877952516593005938368651108115601547999863832597090999991632013881736192019970093542589624529543216651375805385828399711933808274057034554539770760050549932094104486803173737965119778573160375957750197038378972649798062812715332642933675479613109177711161913469950362431917951170538662086605215775054913635810445568822731376284870817342236223348108276785551160944003486680917907774832972922903107094110800954551762325971090448907569032322498855847102110741213403136516487406249065771297803157618099821871932987864856037376387813069063296727989038763279822302936832582187010506591120029441378894812852837556459698274638637156569975189122188171477077876431166800154841348146943021008652520410907078270513293650391738244035359371501429072010178687545607236834484927221643140838371969690265364204221382664445811417423854761374147588282902357313248440097112913102896259544404017012306192,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(3,EXT[13321069679644372923246125576186108943561524687924257828151971662105850545713506162528212808291438514916483079921838037803480111724394661810289921806283502546054432349163393792734055782104012317936342633045193021912087472511237891365304553577996075553671023041667017665672130110112728032599122347419302849167999997228210061611852634945329447971912453670597760543493435760321579893229160418948462142232579761344809450686358386391646935313484082073925550874450086332539040076816301749456683661849267817032774852073770385781040765448977717897851186452940504666268707437962274435733553346673164525089265188434689682539084513709182567793036768628913586685653515401381095000022890791984474743068119328444672768178541786110390302005242496555507168219468404189904852803215308717288526398307836403298061982220834829373697921289678178687152076063787899519318517696066991600045456318731430460409886556247803968648587773084534623529820196551674993992618233402755429029158623010817964368999514978878792726720859826819858929907163933479598050948601832207581686100604760038580133837990609795380510758141670874048809186717600687846428891849546880188625816263769071093202548918675906087292707557145742869410684964956327122148112427075243446880618475230060106666665653065198691103510758937030783771194071243828515349434532169295726732073873919823912315193213114754130575266828691443439291116351389894355094801858898597778319798974629483388643353065990824310012674397654235515515392474311530146694150663282808132259349272849136202927419483670814612531126572196625191007994678654743543889219127928426532185974491963889166439479910312470769059272267492009701363611891739829394686557864974269605576057987269281906172435593637879033332367175660484450745124931286165885246692035277957480904157253084539151726032867659912185195939568138979027801594151720620133739931731977149616742202592100067159680223216881771741977596466427985811759136578202094983754680631871694344818025909709634328560609642617941649501628029893999444750823805574035273936013579865111692012633418871445754382203061815463140887140779650979228654863366552895633689914028773660918276487012432944731396240543227750992172128311275047546547045705906629843117210309380025993018844443933844956285058423687119668564467526965488492041288466740427598243804265161199138536534953605115152613900302448506266716260249457949567609491710900732278410594334527839017035416086847668596224398617546397552208389118666775891965692731050093121619484243636898174026742714162564734282301624455151526439847867836004007094840866165668130434721896539571079282047774573265026432437721783453877021410574266772290374127866976714190728259044635548789908243276814645398121150638636956041941199815293501376524296562135071097833350766218705790965017820832065210836211583815931989648624812296478195368429946668658438996101749814137427585298215684735193450113767019167600480057637173432654803030533213161245408512015182903291097405041895408300163652581192741940690856550252486231438173402763259392805262585051075613239391196069283888236391385567879156839363133106548110589846507593175562284368033350741659887377741125988036215894652240390650810841502862622541273011797651383521422553949810154676759570223141803465638523039605198616126853210601949149789878073019442550224917621966510302804416287527005482896303531061115641071410007363157160887590885308393024624857408619510425563120682550066560875756406276181294348303164381556109357604812709248144045510171374918639378559844406373957082847811616061566094801976562478324850982472203763186009521720270255715542717351659227000472997648542004140478058729710735913523424858614009884107032714098596866295002587048102357218537481579097660301301788497773006124844602583345171518593554958918555499133884405217957943141609764886533063054223778928342063518194461805810865112861396246231790941493004262404757405759132754477482639218795702577947323581406298643471732350904308187110525115731425326326464402135559415197314297787943356583674843496089064240642882580328757399280427312083218316871548427496120018463709832486368805418334911374543053563590266154783949405458933273079872961411874862462345419793236334281950162516868454305182004553294496475668020554220124236770764223552367073748507482467193290530124439817497401442154383576854571908189085895932732236384591400090842087943833985820909031670681685257350772093496864617375925246325507975150790433992086843676676275041853064311364869660684425562911608247836630539173911415984956939556766948918823750842729219941087786197510642885450125678532757751326754653797361574814645806670145110934807693268940389192115327490772293910302683420636372305226379295572245291895684137018633676409549566242429137076343670260814383921064053628401570574550439470396059419217750671826242133272538449449212526412524079684136142315948442431034405213183221421571709269553876499936661620546566375939375374936639453511753065059002449805437613690939803498639119267948318604590517200189915301669302232944492487079302252888288026660506714954280581405755580191322178707484220737352865630148743894018878070225944652980559928351491313104837489328547247286874114964958827915651525319185345069023029758414164745572034391719300476264799907289439867449856491008447345517504024726522138486890576152227736389465960991802451062603710581039638862715057851226293575612365862886191967753590462405346459733630532823553379080057474333607053189586603195892659769688356507761852050851277030109934962687985401522599144940924055187609778923736306244016797295668261604398855297335992905107723663423186935562892896541183285373591619024756605548541679878083668630757704585024707201730722802777067885472763283204166922703010949646079505925300160083407177579345579510222845012914384307470374825912110449511563335451040306479734089764046598887574125362010993308277111766757699936125926503385858784504333748633134969900870797847000073341074064092294312473380359533986455647782548166814468638985507357619905108233845343938119840079607265590965031830371339895523206887438427245264342846758443394967114998311965269611019066720227761637267533655279362477599094108429439905706510373673429108716311137302546579435224515341749588628573139433145353878491190703891879372429514989143491360015315102046039382980868881112946437638655773267347508849094346561871953053621723387538529025073609736207777757899038474809119889193237325905992238329050297786244282599298398865191188770885680078448370401505741258521513387335940006457414382745755762096538670111648958022682088282058905432110530442328181622613422719690676956331317616204292798724964024343151832302492642700825598327212776994962531780565499050645252052610365444976896940015222880219204649646651820877523596104611314085022513542690049657584462149108992345208960,2039824436784352712688650738640081215425854327116964485031053331094551296310008140968169024033488266755676201479061339089771173927699416739583188374822310305875948978113730972911116337274984731410854461729927348924436665117127936005684209073632927251370965435098779816162979041350973713065854121891371849758099446536180044598753243302183821664761568932057083640200582323350512595551651458772897888452043701795703241012470866188814259812795016167793497015672020934173712056318373018175237758326766140398728237463847319679766538061115375992444017025238510416022199675402674814711244191805926384280680604511651314691906274438110225436338075115835425698541338793339138022944350368988999400727934937080102013679761124540129805798930699425265168712054808500175110383764470485620726744929536191562207566703340971829801574172633849554218846218222005676113758403000563754912034874576160876352603019483758244928778859892245315045290828999959064737450998807135626280825651318484713657783690186442049310170263704981835751322894755377091517829900659104816069047088611826018164787638554727028924704401229635285011245473414789729378508135318640442419732986249138163455036502015701625461990956985801429837784342930812147682352859763924599725922762325437890693710185986834990360558359390539469152469269250595053151708035260740213799557224143346788497191066909191156804195599156486212893219825831720414326107000658878963189518786832930621613613975593926098715216239896235401069149084989695301619176663135613133263329143372901024819451458678774854048556726162387220391972284186905757258979287008162418390110325582401433100148787956362128230973772592331925643640606931714673221496283062493318531719472061835441867694070391262193870820329619900552772142537649235212497540495142401589495523373180348301935265810055295613402711705542449430495384114766321702033348552935247410386899419559845118847808041696709192103391186270429749383244748849304291557799995425868891673209925178713881614813682721975410634494022572781861211316598539070872529943592999121924989717040150577325512356718316202834295478209514742796315946268982772574084890046075269878953573477262177330557795532132814701863564870170808342791355317230013780283468156175603955734909133648862677440761598196406654137362596738924873631478777381198324936993694523348054701538507590990018990814246914056780905633713253800315585589655163107427096554067637857861126003173533632077698741793226249986824770265872979471617000362567112736353544021139643364730442036728274084251590049506623708873519493325241020295674181985585167525012383355350507074548737653388052513148279797093083590405524401058215098314250329441108626939985260808834974145268128374922087677568643714513400635300034299009593734765543957451260195754624208370677956568195365180198904077802601783313447432054248164163974403741624125436620981419405345032929238123898579652519191330606351005430910431466398911855828730710129172109712733342320873634077530211924944432024586478795659404878510195193768457726874755874415553634677654979164586509915592057161270735987174858249421949174332913356369110684476022652429682136992100079564711371421424879985885557155851437384569378474984011893213601686551085145938425907236820891317052006100000521997613120858612714941730144716632972675774974122898395349690372749422178229607069046873730440483876639470241363377126836498987287036097337213750007818193636979636969555412940471071309075223188777242828761866628168918749179114956148203834531513921356994838763579853233768037261157030960956685464327257870288382962087390234638540272136134865984110487346923297845800454066152342813562783269491224198892108464621383582340388885540153342782902760653478827426681149365317034358610278666505018882106013415068044072295172380474459623917025099575305026419068735366593791595645777854457877365441385129579054840943502011175700764858377033944719325149789486429263325365562848832020677980362244625316987812017827765352132168762768678896210642876663650843804144662827530639489241523201555449667045832021810786696055707797459135265584067173331264206811235796913087045346339533011435360918254195606011551695315940519914401905641629289696060308280199070048085049629180415128566962069925561670144762060502447106075373578264408665392958855720897461330958612990344316785328370294227012702515681709066872194637079271475566796590794195516861924101887365428765702532307360233367554373029304386503025708020808366964018751661260218146064343312673950494783060282786599129213382884420751459235943369968837141511998929794517295145772436255084378337845165823066710900412658856406112858891237965666952624245867722190349712476540496652424342521187501909031809903245624861996457367375204473016624417883649021345214263215529441281747599378679541775212620082454601106131048216983841563047204791718064591344276755183876537867143113758097908557960172454513668741863915595464145786749307906466140348597278867192681479366874428016111779138483052872156818963671872053953360586559191314912900808936511683492257173761063701521715421972742177159021961484276428462326929825105190255646361029809690577590762328430905901018966335180001851344565161013026781662207615472681047416478737677530549013260577405525733415944006988590793729763161431059046383085997711816728445859487851239564338626258590768809501419977331494172834909691553758274657084861145868485741010148920160267356347862837360638038660188698611448476117480730826284494950243312757514424394809504794252680764441985673664344224817080875249621747375014046137172420079919636944132469649285847702542707326057021137568787050472053431570022776250680151529225418394032508419542651650073130762867580567137681517330805829425051867899650809377445693979187203857661028581001370619978566519444079541975732570885530372574474466042715061359907150425856244626231016722964744914203897621439708433674511410345096642918538341413361114910577664370486772981553515769019995598367274414509697589504818803351481753408579198359527160395061914527002705487345824977001656377391981459906976930187289499599685763042342861891166309142178933186299644893157421748932449807333588931917505418453208686159696938670762317497973726006549667497557825717040619279610420467328,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(4,EXT[-2898273692346792454641551141061008012452198923624733397642447777388909834316826179712875807637299216749197885277652559582770525952924802837595782013217097923208312548749803702969111877471233841255208372084032983053149877853483915753894758040876443758705479894876483906707058507720143019936047535185239500065969754164786099295796489939363976665247714923185899655621864127786896696542632400936666994967250320851551533880189057975996807480779989168986810654741572151560971921370953109277563162358859706829376646290379294527296585812685231193913588422871992622185985153801331129632868189219858673055176085440603922720826603224377191185615699134922138309617528013355218707193964270730059009128620311222325316801186920840651292685288881378792976571629849216874840730606155103352453424810249785220454213452766393756471033465505576368226449547086595454832498190612267369491957737727568188719487026119462906886907435353475636134727546138197886097179269896307934265113212685881856247680123584903311961103983817278773866597716435781554042950588039381212708536980103493654990511126415024276238530241265483282021470862947714726715931650423012700018231860381732054251700214662897306721579377735900662745160453070955142272802816286387871994562850660637630793228888137110906838682443182616176959687964681849253445015957529210147297448302466748276110958934342133569898945435859564895776393734157010199006533580895237662134908610778170984882477683493065371604373635965235719985414496215750259192705669101136005426789390632626521321659847210662703262062091171478371796664038105106178783448668876889059280332174135493108857828484883215951752945739745222860709493296316556180720030509957026916245448234010985476969346254585539397929865689685611254285566407797369460717723128817993671577355251132583182160932980902081287023025751084696773912667606520090696534748117072247286981057753472466452305331038196675738292100287416278082224610799198291041533567188185550499237937676436284377182834041662815101957425612190018448302125912011458530477852150869932717476446233995482909948692329508554102359673662247501528175266460903532894500568432300696882189705368685898411458491872029606111843549846576890669385084802443453500278108466387918837616251514345854421973286795682516717091140657992733373599220335335234604873428116839437837601128254555980981061309508287285281329900505105629755176315302131769629954474967619643200180867012185647557257936252962696111564749462278146593007870156047686714263146312933734565686391076361173930447335509094306311519345780344223799925181290129919929074614261927427261229297227785517988937411085065444518710525091235021666741608851958464244729598660305467952782630102378632460755233111016159877111428844209721564803904060747505178698499723533245501520815603087519505023782202759976461368476606299655869280912843677771189425285004189850447228508513061258697272352765404623188432583261236369713862888832438305545198926242329510369843198913268149101360436319852110159628638862817222726029926924136811887972899895697936502453107619116173319359325788997383432417169699866069866582958703291543035221406444782772664475363126406582922590578792074779752553921083514595593954813078847172105384138720350527513550219902583501682194182946710725938084071961815096498820712562419331145719187892331823115726213907168455853452802627885973290031542610797383084011888182918959656441904719904765993769166963713522349576420126192616583186456965434441716803023128767412251159032184856936927593878800001375051511882393152350828073119386562930874235752034062005168501533223434874174543206987995243840216564991793669705981509585471453348594745984472639703617536689283692368084673694288111914138486863648338086288763414428070402242484758534670187548506716279194927910163938366882343900365139285602041805999958023365426717393399836721230691592311090942004799976555843562950340207308503761853488594079840993273654306280973560330975335739612769526819517970083593690455383714143355610935092142883812583864208194887210490179276925890464482489194630260396506621257483160870363094298664237909680633668477176105530473994770271660328009098715251015019507585784622794338727610751729525425362381521326536451809685158720108979309966033596077627776585197649323833222662126921151444437652421398525396130958812450259807742198446147891214085723040178706761466741667066965526823867504499005336649552606878138311254216151636781443297391088007368386719198876277969561028287718357435930145403271846989123761016087822473799002273073375437576794312292029476104431688232981606719624794845774254715827397695463435767453536636481571832454204259370529994905857761927657249147967458383735816092517507826548884531504830968124684274163578552216376458784891352827592544749785405213305604472098832514760708923961911449501194033307142169059433656381402607440070579584666772960452031225697841871335405063394299484628666391861559828133032200287769318676633146597657719083177855360207788547173499213902596407644725725790353971280963253376148523528481289802668772950947908467467971292326486235656911339967704393054350314267862580144108292212231375287064208341224181619299336101464039655978327653418556997911823417220472469791980050135971831597100378477733280238914429796415669512356403836525720173095414008902356971728577445424009617743452006409867560124242327143723344314319653333101101047279484468969108088270821512651159681004715625856760243483567925582185092895832428800428918343377287887205052362021516946993519130232405753558027244093979735730871575142471652303136726583559145650090928181894924772229183220054665065824247406586716453001000569721767774652353248261984890996210118058214374014827307952828687710279728533633808661440858766915506100652259052818781518850207705786552724192476238585248796271019813857050043157935864092186956232869396404960425602196191959600749038106048457376652674583536305408977844283040854015039933482480140877500503037329208371245769747289147101172229188312918849800940538380362240492132587035396431292080120004396265607999172262604460309981360673835148852154798413044988817377840312010230206374016125254775410859978844938888323959544557473286442853241927660389129460417291529048541398245876477959155816929039559315941986747864715406523558454797654386965281046151765237084274375032422656627459706752189003396315183670026873831739277855801419854920987046901380866055955118168452614080774348656673808490591408042921065700807718290818495600853916068092564892907736494987063165726041656470305469123070247927374491245385158376082399887764033635198109520310978544752689211379093913596453651114605680166944710421057844033391710336904515508274766619929387971671758757428658339261032448913413966527803827924407001247864094598767204672791462616180636142044336875226807239699417100,-443926801728615862712156359527919720491014574294255232108399553322904009716250284974320663279105021211430840236091654317870669063237899854085892534245969741246594725260502786740856757584640193446218818437352993094823005150137539945492136752948799577165970763664837808898250173178789381163099713068880232231145243257135272163357202737754342299561222826433046516371890186591157716489466881814678221661423750014304712788153975867374605244162159211400988767856718436942885264929517276521543222614801270100234071488058767401351128362156970037542573891939967700514355968109693396839007326865374224106998427468535994618714678286477622781298670904407189363344718191910979609296935201287729298098519661801512865187325573588342412887306320753334859890452589411698055020620604997928466299489128712767244846263886371721854630565749184093571065292202381305487532754921170469153548094100124152176689846860395834890959683949238735545109001111325275605011102743088580719487430600851873284907285874718843193024845762175784437739946476417798588164026778764577539008021873820765576466090133709618534626836407705453204765091322929517337222029850136705546604347868129808645883601333962100746438539861545697679458577697214157547731192760602070796861612660181680235856590954221920524423007345149346799222662762255799931762108800888886094123745701505242554401805611037008140642659750004326084422050640953662858411000728258330088703420628536990463195792140714325621765395983507823273719611569643945189165575346872406984377212005970807548034689454683477047699701428834749733334811314405502278177244956442767877464582411387020418972632839619096234717732356673186969471811393705896344311945049575808390305603939418034006547037725341806216444914306591640982474251134620810181694773744797134542242585566000059051468261781204641914957365181807871492305619261297036014292099008044165297363312492200585803323993291463933320592221757995992616102573513180711553095600635522575756008160481800082950513398361142157353308770455936744061174080227784266665277074161522937535808520453885276603460601534291804758941935320237557148168728232552421841021324091632609443076014829976752414639516449825314659162897729758461119429940100191131686579564658067297291817416342249336777392334993818188646017460389969753071968332300695899752959397742753702703838221260783636989018642669622660146000239968469529387420526422567357046178740111710121572355586831488344682462583854083182580880048896589245922293369201496535494229288238911660834724866438660684030202210486503344330795372539323294948435226823609205007334310639166250734544247624167447979134000644471584056405740473408725817631174546898064086036775741406255497079562311637833569847846180458169376598827806080400697695593939560130915420828646995412877845524734252220776117055115633237627528996324866381234107918662363989872766269971087106885059408137431777097435929161716863284919893849133104492383558355364901529832844658910879308668316479891921802758161423811730189112351287382254445502493160725342028459819860204356639287208081513229866845593254048665740608427846444319988319827197587299171816034852041148652140326901110189736547737863137167832155462623041305138847334105986591581658431588717736252210869930473837165265935009457894812449550715619562417868589212469422521620497537568489331290018668860920133845017058044529270161380482432404971072599708933416299144508840618391384576610073361487983701828351132144651675875619715400777794087259056927760539849753941331639526400215164444180598424691862937797299763087339928300768803487204614474443260893296601257964478344718766568866628569019241572989235691816650915250688355906601072654183202854717356418941961745217419577783570706245263432480072799387529995698990625529864154039595416827209542737059801010037080114544267694011131125946569192307416478483924464215145263246879403949272221944966737658150294539521587510998885131959586200585724253522970350221656638245313317233915785177141520611887360540740553538752770024474761110797058586502201609462649062330229107935792875171756896617691738689824768310731232062932094030404422949882103827778998224208824937937256392156753435488914351555744070972053451260248581474777868395651240908843145701469132021518360395089624569455710075299568001412943342197490993422058592876078537288023172607256690792615807269822452992153440447380159593880568117723694148220736405006962486725620216029612865313484523604953973721757897824565398489749488521424162603390624391216779248744266867207925855227089922105236205999628345883660703739350160338700471745554329190051713567259017329311729752093649272562466464874375107686347092728891040127220757789970409059412394417806992383479355186906136287968257789631524151021724102474419712637786529162314648637596617355194842451973406580309312052978073980991113034779679972450039915718343522887272245415154793253397336411145265044207442968088590724710762393644201488549034484883333507944520996245205114442624964547452239394974566638270128640853773985440895618317005832700141570089677151657148008498716014504552855712249180104124793137141279868505691959573854518142688349345725706162755515867982911563891088617837477706475018642674192287342466511524440507869419482732684084834389637845705450168306395250636141718888950924211587909088031581166427025407373685792408506530574726631442398734545959933847203935098716276221968342064492047882789502774856656779604738829250298046560307042020543245186272366070385512648548115099760779412135270391598215940519445736267638960005812531570224121839900058783262103869231233880456450407385800686553276262050201521348514413031428591473229020546439694019535742931942490861300738863912305030729743339276099371063235744276969977711775369861230483751111541341442802892936798701593334604348228077127663580058132824186814950251187747900874583093789067729132304416481292241670484084904306175223400374177479922684174749937226155397872031986081249357685545249741188281793041565051749595475236364186833421019497818878276896325432546241463217686779887299464579141597163121829762243685478852821751824871824558658969596485428838619608863086858538531165026106794292869685049481760683983390421264826118963650376668912472849931305018588100,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(5,EXT[323904057100829958456264199429750830040452102261034382744713818277298589779753651020519915424916953216243733640142581549193093219109246255441255343410459341189296908835506632029730940257950487350566740533051238273984403763539339482675020859171014683222818652729801657662664912976585963464240259088750854984611746385164667741009823827645277366451045367856477794037540941610519676321814229014162406325201675139258445506160413143847251813728997798772162427290316166760680590927844774302812080012318701274621754633142693432617573729774683225811019629591842958716976721263665696406048972731244108777156246319292940744159625168289875266217426702505911212722118056451058744982460924052143788998293106385927823148773534563088460803190970460975346210848775075645574796989367072022881227798437408413173231709609098487701534213079588379201040124568006817939180890339725496122190229840678041214790106633015696157440347655562699297342999698837899654870782872785372776292644575926491649754878564100103264126311782765874960466959173475253473771236013707466146981471635722799096909861140631531690518678429738213884222674503849046072855282314393519086758072771289499388045252857459832436390618109900814247916223810951341592233751696011919205018815326405782922899940297361855306643543116412533913993750233220438342967577026171403367605167477097708144520669713121894419351738670917503465549777216523567026273199614378649166844017764229447568287471739861384542631791054682948173135549569100680137419714337661365651706207847444576829264077663871494800947442281142885410288204097373382637824604092837605732622841077592580575348377337834406263585348003575778552357418039667472407291950194066854503218412795204287982588130073258241320539638630609638211866762473416107710041316803816347695999557027492288338709537326117458808936887443360028930028093036353454827250082310401854801009950588764057150003447497509559954372322292824443385713571038295589381167530232326138530960094813674815165016512516761359626550437188788908633235478673634181261399422630744982007314157000462695172190572687961865350548850519618159723899417932901128424659103083497468221417140792370993522194488505124136925256791948793508423466244229959129899305121443879289776408392336314140917458492194038253753845050919691743883233042344526264554812176927374984859492085306515363160419078038239902222445604823335901321526295311838569944636861941272155993952603059569446111490189264351161234226313062377131044908785864871929390280411294008808186310560248364020702718301618294689554848256497753131063643791518690863560440635757547901157962595626024713321762253563872656408771044947517988746969226647730546343566575315871061794611321961724404670843259613595081598761704874810789307723972575329158146152224943944139730058311693229449783437552197981396725070432980292706347714601969139303834190298475590963281535190265276687671048059240431999358717751477310096925484158925208675769680784645035176803980365207418939604789265651401276652073170144572590190595527659367421706118418436180693292673704066178829498890434117665083942445716413826856829587255329476944894492249595485592368424164813139058943890418665882502905469180066769283544218887883095692467546705967894825964538797178759475307772656914995366952333598750396932994229234092639323279221779294934021395371407050517486191612362812068009359436031491457020612589212981246630327834199988634932910120508256877645889791282918016725589599824832370277695525552515426523361165598506962426431844105132010346889745122496828359586642931770129031644831590544380576959018978278413989837996651920922618085950818703911403281651098536228330955462752658459416297246886949912828970684173833074569680288318661808069799033543925592107085290019228739218876470464376722752007991601594372308664803086870056490743365904152264319801306929178538517601197505863931600418352923773651556537158788577369368929636069779291309899003489343739068392795956370634702152904147783371737810768892426813991542554417041419014520798513366815577406992120757393700237732748376468013942429836741127118467787442710476354760955498917471416585481091126448655905475952647812543520217569422871113733327324190408945482754939848884495423116972244043503415239546954440410468551159699987047007976495488647463850123848056657313170864493980531753678708647255447774125802957083842497987524074783126553694976878123247982814305543823894316844826485436195428355779283625624973672461196750142923988021886314996814254349650701842772788483830319680182217674562656518630384408335148216650143213732589363416509567171464869648620442526655419820929681448677018409818943767028179580627318663170260381484137363590570740109246545172450862504601896523630001235440224386390038005920517366767277563265669701069254709471149617995232670609789664216043471335284127091855704042072708560060520214248275778078832103881353816563034328377731389460577490028833503642404812590217081418590401129911623464439248105316700739701957010877775010429416362020329129259406363526369407618508750634947842237203547263959368481694268335266263910365143471058431684129158348741773475048007913699792226105958833432457050713817003545460038217672049871741946925711331747798953951862449254656547213639202109417655370592055983191532757480897660154429652469447738704286836070772094351071366478979282203813978590556284005544054728845171242231921658758535730129389892245397797550491346830433949750410424504731664349472827977651674549420065110092858469061202527592038394983600274880808492460543700918321025578068960282237563046410624878230188418226669195605947634367863205335075896258947972724325104333533144224924995676809115710813870919182542373041800998685407115701533938774686294475339507283091457924246728638224775225454914891666603237810547901647082919892479520588914281868517824963669626216535700698474415412184520946760776760500145340832420120430037171861178263819870522860726653352091895568733982126464253775067663068624653163437881633124115061122782352866760017808747543928830018626168238722391270592116472387282864738864440340657604410181169683450200294961856578934282771850046244744427166750506689839241988609020300902697412769078405196457609473396632212934340895730904562203271605900133110828000575542328183392740257818192247395570971491858793266232836817864357691938806562113524612204739748532949443890868609501676962435272891131884690307236542743035893392835718298376013661734188135114723409480993748286078870608246328662816065426801629915703453202431785986332641920442367227106197267588362955485007747295675192131634354583205554911368473198726267470820790702081787602911938460792422278123759232537701098453053950217300895957901364817831997140120669335073151374017999973187476926061107100982147062270337577023280281287344261995637497215624507272800,50027793115881905699701601777892205367456561117246836495346601838692320466896113676314690686913589968363975151298381969494207444400887938835549272898401656383978998580898808726048995552102372788476995531985694967416155972892378892291156465793245285464203630403087140825327787360426744075048696105907603619543252302651983481880723238381801373666130591616893786717805148481136557198188652483229758274060873611309103519284098231657543336102622953280878587866822675327215636200769045970131781663058275173002441600917521252595516564519862196950478232122227229756021769974959507954483489448426573418682716924224061756979195225317152471671498830071634534823831185255210879928622911550262645208588190942407032987379144392656183578134608094261901710855166898826245463582395794590830756128894872569300251387932234084038067741136777342949610092578473845312438174822365233633456687873149475783100933720500497858818592462852612728906484719727985943226516853731106990172388222719828065896810481936769037206375676806609792662691680545402482491972838434878860884506628593837600650158344478493683614337318581150833989144233254267364301163109574110276619700762566799750667401359091506553718386510456579441391728508292793134183785432232438200469659048943511613620696971523951776288804709721051301866521561962402087973833838978052341560135491229330575911880264157535876719925679829012643889462274236913390656875226580712059998763433222455134897885757210462774660489281120498128289440406243372398039466927613510756538606231630078293363553836040396837460501690081792615460125872203389600677646621155406033763929608784728605272093124369697204229887266017581850685646641755619526313769638435114196178566399346732334484361897829550711487855359004721969864650892579417970779398295552992643892057958154298048556735576501679122363865889433112975193381367063888480080962535105091321940371383372703915230847895149440727474711576899272951154115981186717004203186034235893695909214525739134576350667237517036398947209914147376042525741196090980318482424506724728074445813411271415298903564281552398088511445318490299352587055239976959768040162279072429238246364978397290818176646689674268735497013627531610795718647216780994919191745513933249517720611206637674649396478923039583274002001113467018328172415528637397414860213576176222664586003475849813968195478132990002184370500146012906156995564727276936814229889948237726533315382213750488500420374797186300513421251925369416459682384336285892794961718515560345859384180683269902787743091889827194127007863554169895747585949815069354454554729163107440341689681843049857746024003328974507578550529427810329053988283188351053295309383236781202789095900953304159893521212432902600168532782065410143531268055175613402962924753689391783587239835664047026005833696399266183933874265280749958676536960231349542281613791632917540071468177874670700966244167482788353738100167706604175862463834383392704166770657656870477611601527775279904643831035418131527215759151994824507238264426263132032626839799839058765807569374143747113514616320840790788407298786164716945476409355472049528420717115844650964200250251663774719345436607440597491970256194924611084724046650053700065040565898568006304735704253256653542802519127734248620692945083017184576882841340824051938164677711078754097224408576541253156970228267847184345821186204295325826816909748576820152945318213737445220194721930915515879042394181350693903726199933730071475621324024633578779064761171834139460565049374010983031576555337886403304183831918534811462519896521605281533410942028991408268556515979665676998774421038096054227430121576907930146177526293591726434445953397414605177756645848245575281771782011901161048390692442134615365359593201490767310266502597129971640642370923227771536809732167736579932294823646964508870388694930314836945377510674529738834384369887131845526494939004645322369124256054759552854800136440317410786912803137782044982072411872556756504377390830111254974295439807613320776319196259710369796198372113721303263320113645425840127774519955356829744691860628224957984639145035512434535121304316292998583646513236456854609215296473706123199555380293265438762185079404614058652582232841153510692801047463980250550752790714295409046699394204965346385039011643261242437910542203818146390600324051444903129554668539567554397139472611116925158789603568849918045155049993494262147962770247909804380073562597666131512575399268557657931369839317186855470615780337710559164264155474432987154973423298863127556843357437315345717531158658749263678899773080955052273414065811749438664650223260182750999416849725479927841216581717572121799817159674963398001600934904767833649671960797701854048337431035542624138707784801972651342741982190133659402934499921926879894047942709086083374043746546048497703769772263488862658715945054534278558224648463578731139107289360952317048126876201490876398732801607522675686058997105546852766877908716259776801845585500851882349145867295729143375473767373011629549333845496471411656329535190595518535591753923329733496744921524273920533172611572426857983925583562622864136611558554424457075981380765558807926838115709807480984883919634194097632626594160341807805127923700360046939844949467181879640877226542888190739997062318660209498181380815320892673132264244916838576999505176496751067189142897275062470484320206642863421370175534596575349910235414024630871742711843505959048508015510020662392985719554621257738093079226183111520380339645221095252800058493017423759725191139086169141758697197008376779409816306437141787812491528175298084937889346056385273437787675945511793259116082587037013247707266745736021221424411525301698365097828871676895617849179916624385164675084594324758946460538561667655008933202502262112154422747814427912741873270028062035136675773069403892247419882251134664610021250860057079474255334590103707397464930580188108978959215582190153337939190289694987247120827948538172672886106736748851527118852690196501892402774684372318599151679780462068405694406984207063238860056346141446143370397312630821984391690076370243996904107810080565720921062800874473509006223998227196574757325668145629437295868317235947830813329341050419028800,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(6,EXT[-17873690659628366490653916856935461005664036336909821158858107785237858817210890497591725284899761321630097155124157511125774633023073723261398712183168167969346231457148215157767251794656293740972419541587300138364238672263661609923682230564163559736395572865563456494114211784498578407155976313138467602824540820980386662410163590316195583140312336713602223528112801463565484888370612448269525065189555040962456040531852991585878295813045309400660435133527144371328923776831317688374347896368968599498014574864542075984243919818406345211579221633814535559060523179895787220299482325607993970327183809346272423772167628967729093997912121252648914416928046686369533191081535194376937921610495068883781683416876670523169232977956494787058231431706180667898415300966881044694507278592687097843295028244386076412987484158273533325220383695708764279667010983677006593616924781295536687524836614678388968533024337362283057127265078050135713754419663572232931539516467999907890519376146630020996681468819062561994514902484867752108349696503204578582441998958007280569974091787132033984697780266013345240590121399772804925279429019632652966176319817164472445627758323578764319624413442320455280861467548934659220181207208390890763781474461596454397178486264064867995739413307349351995139960598164846322220669614960545610633697251376713371363007630809363691650940089903305224167615178618067113559739942150923368654316825466316024950529821859848779023736085917434909736962941119046366405502818181282839747929095536283915753820988709731200996611767378790067911734696981801218582435224506279599545880876985109306149259824015071070517062941271124863019368246584360287706186862858045138401871157979342039811985676379683108483666495151236518106797051718540133705139242712571126382787097104325660921584655753083588754181325578007770299597838015282339374946873366756777707018525564885250538743535136019234941490575094836037530727757248127141057692612581348815380413276913919442787176009719690249176316287886015616666941468226545142324351224408550112810729887150092199634918404343167879145927423548767714759981002658075646909538618047107224923172212568595874791190652607732459727711427268739698558861546579210211770607466268990472812947659164834575711688866371418549004066568932891994463862514266285371254117429555424916699693511356710018305407332567831793702342271925192166175704189671652613088257722785833178405165806149220455418681418231266582042455674456986708109986356335546640476484460632359389015151703597502679966119215630474356347903723521944929814422068879066772408693698090834615589025622346147515117286338244537233891101661071124676241603570769732464953165408310477912838916770345536500605786195621124429048115470441802425986152645419642112547756386234051381481731898975855300582229300285358973959432320062832326297826248678168153076064518943099094968838725769582386166163469885521783108311041152346515589860721933835287379884038734587843017142903778934257354091929631976111563966914933566475455667538670592641670714277102168523181289135817351330007743418995587325573272740834524033466358623872867251500890398579150561319636090702236838905333162518134982750412729322417395957580997226066721803532545164322183619467496053797153413550499968720002001123710358078082796226134175125954250837058425702418240299260150712449914162669831521925801689951549874681220138868142125219466247603953691670177659738264574916835912269307178893825492320519545362021651520860283844881515331117999712580652358706817316667793923828294027774456596534094232103273454589028557199046019738104223457286513443107198859389782416274134300953224729651630350773419332831467511732244295837966905360311149243363280137699381583585687360107703662860019834608632516432014911145426315244714956960462845816556114340115348981847744854939080098193084117799667254572718270740031312892684622627233330689136942357798985899251666355116484585725574508767937226573763668714771413903203296534893731328468286813625526701562766741427610090848773003180581762972851264243579667339721832809710890881877670632256828172580750837271266386489963100708418563671255276441943292052707251785587796673823880694582472410672851463440914925790988382028991323291955724897588512758006494390745692418661791538926055178379532264521925601218874406814234863573204452015629679586688645242073542411565667743401907415362573029778270357774107272426836776718352364938243585253593327029704338531749592991041987735567705940813936253424476926083272549808569412911402394952597714868414331492734340984671489821862980215200712860563417721969949972645660428390205665439974491699609189014927229380374698340035001140668096880050440723486325728252042433181266351262538708192389829931425610888080076130463490263060809203010253711894453003472166902566672465751525101896796466023900247681399177537151424888939526539212735667080903356166837377313958234459855570738466307455554973020065203014699706557042624746509353513465519212301424441194147884841983871530038015984172224163669754881718938361925340148641307169779907265171365067881658566670744181236465460745753210535120566950692001605439500283722947486785379593359683938996332913309320669626432061520661511947382171665191370105231055869099925249589103003997588199325481462407578636228095937819262641542666098998321003000211025085790094896779663561136771761049219613608145616017477461878327587098218362637116677622620805806486191665199825632811166882533641919316527895723934319971560112132589563402246313074537600329087289850109206946303847367453251944103065081319668603900024933832294093688951426135183702566506084964040030350223330903489523075602701009166683452431296121124936990575597651462009342343846315790071225504557405653466329534302868336853894427739875748064934286733524518818479789772819753764015337386849693579269113565998986047038130364699772774941676927381708211375210713565693127504085020228184072002925903103817928307080835247647882698184344224831608375095760306779366924340950501083106309379254290071925966232913962856125628128252377583895313700697695147914580140631806420430335688181024629332838951658697550718257654329961037825805868561008114482993236134981218666127113426052371054650895529641008685252795078751119462769669318277377259650889636561178087742988817584390145425429017235093928659992191297077856482526821018338554650439920134934520053332763215890616355209210409559608175359797014833392521048620341336897363037500133740430035570124930192603759161749785941784866389210468335820723640718795445947537973789698887338740227489323456980414714933402892193843500499302414901150208185632713830240048266826800873012093876373939636523637451475342949236900253350228181661248450400800773527543908142509362881312410801895534549802982900829377703311991238179101007923704713125,-2818950257726544159872435045746892218225664039650546130991466865545210036476041672293384563188079043790840812649222143675685832653893654739557297453559947843548444015938396204357967950563198345248775975018309455853598164475637768751221307976092188692823759001972479435236597723216790981998080390413155339923926854568519866272878228078233078722598213274276907523013318551000067288909665670315022689319666515820545557853204462096679869619956189847089972813806829369202179188119764390206358144413774298066711972887192972312587026454056388710287646425803848262858275823879021891986010873062233619567819182106895901482949772344594239231127506853837539024885359417087562006627177928077172445599801126980836066584286165967187790160095993294211948159300321992209522288604785355584995445202167688433015975018121260047985670312262972913060039743792424323765573549175743694409385514082885385755806358171505524276852600133261836998447682256343768794647682327022288295628314158401588015925904533319105342298242737497890653101504152885225198428439244436998588321933317523969844204140335327680549779255414811979068790770791337468137647480689872567079864327058899828186150606682403504835340865370490567544056344649113360237436838741573740293875766337224298998328756711175732063779966150813373843842953166534981059595283969674515243279181075447281429378268109420594419678582258051779175389569737661236811329397349248122230112837596283322086702734717396963600891016971787833073177829432377785824578995362102855649744369327940662108430448228615315769483433310799537752262600293374211258876298623287830536682898796940793205893205050447137007012481764950495530017281979096890103527060697948364336779570550031143986160346308871483416151664426948188182254278994959027063960535240945909245732829200087692339329083448123951014902190908559454780779171272340303987760241123941266463684008628852421205765853962726378731702180741696160429205133493706584703506928240395287755678488795915509331532421809946224672237807423517210637045915716218850883206707550029899716101263432629832455999673856816749277115512216613246690937325757126257628551051559743742904397367948788598529587791979993733920179565577494246802055836625173236241298023961389993735282419813373165907528380625666689188400409803420859769296497874896760800594842940039940753493479955528792958965660926457646290493106795529714554721830177162681526142680383610609271066937461056305084316408239300283638606283481466067957878166954329146712731350320305590835089145364153323227591742517838106561299589800141964405917759360448748945731348363420018255286569753412300721487138862483517936465682963038988050420130543076562996162706572045590363851589768733651099226500598735332024945456774032522286066739655900016135910175061361041040836041345979640489301333528564341740016409383218754182159216331800054933432251816651225953891522034788635287390597625213601543251041414080744036379722650310228716130060279236952654860347974442233951845478271591313974968541728563823699802160245596585467154678542210567311648028850913445695614614307056816955649926760418730502429446398461649333884558394996135076589061691007558986654753653435852041723704658379053463843073999583679611203151825290401193701235553237108055997150706918525955672584337411023919261394587606497453478608195538175572529362975653896055356428115696589018289554061073616981095332117403003966599446434931666438084958814401809062156737540073200154430627489458474439657331071582439421113718956598687526022503909460641489068630222465240097482952506567526950747492739229161254488589292256009948595112200737305597300846875623224418417461557073650364780538656211480718553675130208707585482601753638597549860697899076503673873684395906114304443896591309871449613638221806353908359413897147854222078846760820599486197905885745430698693734703024387987669368896375205873910429934718030688394170142282790047817232946132207386091555512311443038544976273720176224697609013153879001141101075829734167526252300120962910916271650584150479483904909703049049133399067227364345101337614351516197931095332035680567828967179524258057051552991840692662603775654826527994324844403939377811116830958308546087109991906893134281061922572668537628890223837992842706945428681068211099018197299494848779430681282075142267338592344706064267462030439156086147345407600998968187058265207083452216164401773472293318029619748764852956851855173342856836217122828523095840545812024136999995543586460210974433429476039683079432315683677623666048882323270411419082493599126804410929499987234985195537612768149056258621232024350921429761946758147162790800889852385084343618088229517400102385607683031512685753796867747076757026063476605415392659631872516860810427547614829944199654037342524671577021867399994561713677440097483339356893402256470926061330884241018798547748823310817563083010867333444069265823328715131160867292880698989681208788228342784810104285226112503186748028715514629457394003746075518111454743735310651616328383513716904924731981187677354656111684282887131431152987495575642001037816409804467942145447131950936758933395791918407499986549358585048650438942978623869670517660592627012333096635538691248197209904214812837145523286864914702658966789964817225410176154479364490431703020234840536990553906208632334377013123063666106928634357460745882703255002558544760112944607549219680615213798181087356312493423096623460738414434070459399961859044141852290625155104392936126413099610905264679899149842092299492729229877218833043835072619099364877162447016512631435425937349582604050425311374633935605906563188741722236082239425741367459716912073267137998548245246582851855677784988587913779688564027259564946879900681232049302068203755561890525481550045523963033839417583224556724345126328349256133557003437143271300155875952168050970843621498698142882584979802837369924318608982504885196904936445017369570944294788955600902230952906599324411307806086041967911356847809702591924826285195331913972400230606575841166437084807749510782041018129393374972019849463875110758237288545140792094302874373428958266216356847807160392756806098568761710366730638841336196523633648020207336446681631265121745917720078865872124014880375,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])"
  )+ std::string(
"(7,EXT[796829459466848431458795458017539708536731401291642947621850038466193650527489044002515131325624833461177474730745889464367796625398937756286453582598017174921247310459009415519796710757210657799026447330175974423478625909016173617613837562961042304111866242451068961627096845946807707227231470919903964663651290090749468467578432241915373220552563542750362624178749795962877877457302402102902375314184212165430733164976004527169281407201743738395486533743340476953092007112302222950923367658600263917935113609797241068872758442717113208608557111836884490595320246696543944421154960376401619922160286510068911786915420757853302667802028173960960325656901638107656690146007765961800462397877940677016864754009186180834325275659982495393629024711757558109057478970804000859726945296402698588661870933372532825248174158149188864798300752734299808965625845310839502372171738058241912926080244621859195271579005163259066740718962208620593828370657528411553697788846178787107291124133308431539726867906229452544903617886501161803630131154472742600718086401040524415793878277349968569349931487196913479153736953891782326120907710681582548589466664294297761153164535248990815635699858433591871284563165408360866454684832952716945146701580064064116876059850456436076617947126221030560586928903085038283370174977869060652829444111124979998352904994017603323227974431405238544089492352562856715329838178231966216014488632231844190190204368087447944339155857428095135430839679762621956931063514091154511539619943884402896478022176390402872303159477682589720684502993787836993506177606273150970965559845416632484399835841106088536232128300920807931017437872077820902150811238603311212719430267264942347646167777505398836059223114358419756536933516764161115751647146832191250142100056084789255287310748700470988154242073864149299812879723459043970176666709358950180004863095969240722016012661531344944422329834785814243742605844302621723071967282492632090765789142900079144783305822826163197078720905574546720777258115960870281331815067156553133549803432518716880553041084286289265285752784836820420780732531055883443420362355691697167813414315617179638710581824417108370698483115489443650622854578146599953245145748418115421882155851150092413284343253502953915705375213197081544687104605820447700459591937773977212046859678959561458922130476027601715248809955983347288342159363438389801759073171670860169607622180680404250870428693707896599763356489991721957503597291415512408681569290858737402863979820855880779667821590310420243551704627496811406960115501074739304609487426032071173750122599927622789429778102039823017852354854625699687141724563511283816283003976915138722022986996975566152814062842381638947759464902021539790630712001213669159354373477548375848138769062664379592111050593203285049894691119694732056967775546515281661361130030594468852325902248212200914747728306692821009312943949887883387190566515971744020905301798824373497006828705151630249325004982349260256513670316721146699085150375937385246218601554577645964608778292238505572289776209178291500405890405141703533896734257692608063528830332357281566692768053434960722696367480969569386460463359987836465495090682713934596059996876845852742372726579166258849423434213386328736761138272060839463079656136691115478243757843460527454010419113527301393044813542139967309940639072386192004893018509558006305810484434532290649548549844300374868069749095627941649342112510571336133953157055436287900169305428779798199754189435434857627961809060964523160878506612146575198681354390656911150772522836936392699652126104212214169202011314523775342384242390713953607593215213902108605085109809355368654417186750276459708351111880944880289690283781947237576667675770166875050552809066672005439320467651619350603904217754676149491500880509983450538657717670082015537255775043036850380950703944884629149965645710320541337969096852387380044913424197898345569486827899717761697041736111810806881228594074003867516694486134821948499628949575389033401999494303958152324382508373115039069585979413521975448768951762755756796674794035909874409810793379003222982208164865274088576991463548727377646517046254805603123463312700445939570480159878658775930149638572591755726714125913563006740960275884223542440817817024657759196218964121341402154831763797053408241150157384094058506682128859161333326008077557031707935532243858034964664193510123892214324326484886295228303877194454573196880860488464479080657798404076207481807724777625982268851836112963023559453316735082057524771390361863964125628232909816234011722923416190554757434142735564853756888491048816378982777727255985564504099818143336603436381829597557213210072960549519144227589365450422342261552232744631954803050954681369066647762785963453127571820061112692464967881377619060158327885362216717413882198072797052738387414177157012508370941607373125118368364213260225430774436032973842253031807833966375260418739826731353717918063154538750682658024269692444464872283227307635842487144929658644759661587271104801549415538337336047097094525981607201479221488691682660517811461520291801936558138613462474012420471098866354938913690344427741334038588162055056865834315200870987818634536062925327722896263809748978387734728966669857903330562196827981004985089733300823962783221290591880331514371290535461467710499407032473742334034954402042331336198954411015604388123972801880320411619026281853431189838168632846611065880287665658203896349908711131382509155693161773862777839290905194939408404920950568529947345190640690309344299600490903947882268416844562321236068740978156153951807814777072089360083133334500586596822733775947844436073745287376739756128137516602312078223169839343425520337189829807718287583479616878770802520471703633352841174194282678195923500954217971606184182570242825691253875371155871275549312892883402018999013825338449467614235467971104633416713412398955502808554276278677637126878467271605037583021389220431925584250042354222929890913959804541935796382880889235601779251348566534679963204034973408849808901614658150926612476594881028199706260792508053195726417336866002330889789052833543251117850326078963398093294645699517156793582235322284373002409269197077966118164518351230521739870818772660979385448756669870404117446185488632753117502900197353659906096813947236285801302150147840854900847524480738391076322430530759080962231556958152088186486515983416522317098712997797466111071866430069127716964110314233991649665381280527817964525337026061642338843125793867261342530274133767757323290745342149633629206581516861533046547124398238007097767281110659461910608564526325470490087048907387677352670265016129524773110901803712470640807301355356609089098865890035699519050070749046172800381635724449733922894840906999920000,0,42640678138738053946605114634301344273377038947988134788819507373737572116494209990135652115970230275554830926835816038786142137555379156498341788343146415485881220095270758074684536062510491094473999682735468925100296324768294433442711776633465255897895270868633282778242696455432714139233639129535820544359561654182198857105782216581607262375961264235537694462853634010355529212429507804602655532224073246808002797213041080927017460485833030116744622435718733449833748910559192887362945355738396299592890513736086980279948207749567125689114769529641189763247911563139927238679697732078932737970678507749242490762685596216701653526047621056527294713295198874266193071676057624314265045351773426877089706685083882703088653044733996535187280227666146360825714379420432618485555560848674054137935975983163992877974938680087539781737067996296406371548450353084755693839192696864436150330783588510591559740131891619253755648054527052705328293205822731796913676840141332791343321005368071557990654860051368397872076633333203124686432220814160888618934911814903992805073690151515820135456537134314012141942607792859802513808630468582929236229089169])]"));
//  std::cout<<" test special poly 3"<<std::endl;
        is_p1 >> P1;
        is_p2 >> P2;
        G=CGAL::internal::gcd_utcf_Integral_domain(P1,P2);;
        gcd_utcf_test(P1, P2, G);
//    std::cout<<" after test special poly 3"<<std::endl;
    }

}


} // namespace internal
} // namespace CGAL



#endif // CGAL_TEST_MODULAR_GCD_H

// EOF
