/*
 * The internal definitions
 *
 * Copyright (C) 2010-2023, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#if !defined( _LIBFSNTFS_INTERNAL_DEFINITIONS_H )
#define _LIBFSNTFS_INTERNAL_DEFINITIONS_H

#include <common.h>
#include <byte_stream.h>

#include "libfsntfs_libfdata.h"

#define LIBFSNTFS_ENDIAN_BIG						_BYTE_STREAM_ENDIAN_BIG
#define LIBFSNTFS_ENDIAN_LITTLE						_BYTE_STREAM_ENDIAN_LITTLE

/* Define HAVE_LOCAL_LIBFSNTFS for local use of libfsntfs
 */
#if !defined( HAVE_LOCAL_LIBFSNTFS )
#include <libfsntfs/definitions.h>

/* The definitions in <libfsntfs/definitions.h> are copied here
 * for local use of libfsntfs
 */
#else
#define LIBFSNTFS_VERSION						20230427

/* The version string
 */
#define LIBFSNTFS_VERSION_STRING					"20230427"

/* The file access
 * bit 1        set to 1 for read access
 * bit 2        set to 1 for write access
 * bit 3-8      not used
 */
enum LIBFSNTFS_ACCESS_FLAGS
{
	LIBFSNTFS_ACCESS_FLAG_READ					= 0x01,
/* Reserved: not supported yet */
	LIBFSNTFS_ACCESS_FLAG_WRITE					= 0x02
};

/* The file access macros
 */
#define LIBFSNTFS_OPEN_READ						( LIBFSNTFS_ACCESS_FLAG_READ )
/* Reserved: not supported yet */
#define LIBFSNTFS_OPEN_WRITE						( LIBFSNTFS_ACCESS_FLAG_WRITE )
/* Reserved: not supported yet */
#define LIBFSNTFS_OPEN_READ_WRITE					( LIBFSNTFS_ACCESS_FLAG_READ | LIBFSNTFS_ACCESS_FLAG_WRITE )

/* The path segment separator
 */
#define LIBFSNTFS_SEPARATOR						'\\'

/* The file attribute flags
 */
enum LIBFSNTFS_FILE_ATTRIBUTE_FLAGS
{
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_READ_ONLY				= 0x00000001UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_HIDDEN				= 0x00000002UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_SYSTEM				= 0x00000004UL,

	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_DIRECTORY				= 0x00000010UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_ARCHIVE				= 0x00000020UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_DEVICE				= 0x00000040UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_NORMAL				= 0x00000080UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_TEMPORARY				= 0x00000100UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_SPARSE_FILE			= 0x00000200UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_REPARSE_POINT			= 0x00000400UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_COMPRESSED			= 0x00000800UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_OFFLINE				= 0x00001000UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_NOT_CONTENT_INDEXED		= 0x00002000UL,
	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_ENCRYPTED				= 0x00004000UL,

	LIBFSNTFS_FILE_ATTRIBUTE_FLAG_VIRTUAL				= 0x00010000UL
};

/* The attribute types
 */
enum LIBFSNTFS_ATTRIBUTE_TYPES
{
	LIBFSNTFS_ATTRIBUTE_TYPE_UNUSED					= 0x00000000UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_STANDARD_INFORMATION			= 0x00000010UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_ATTRIBUTE_LIST				= 0x00000020UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_FILE_NAME				= 0x00000030UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_OBJECT_IDENTIFIER			= 0x00000040UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_SECURITY_DESCRIPTOR			= 0x00000050UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_VOLUME_NAME				= 0x00000060UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_VOLUME_INFORMATION			= 0x00000070UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_DATA					= 0x00000080UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_INDEX_ROOT				= 0x00000090UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_INDEX_ALLOCATION			= 0x000000a0UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_BITMAP					= 0x000000b0UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_REPARSE_POINT				= 0x000000c0UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_EXTENDED_INFORMATION			= 0x000000d0UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_EXTENDED				= 0x000000e0UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_PROPERTY_SET				= 0x000000f0UL,
	LIBFSNTFS_ATTRIBUTE_TYPE_LOGGED_UTILITY_STREAM			= 0x00000100UL,

	LIBFSNTFS_ATTRIBUTE_TYPE_END_OF_ATTRIBUTES			= 0xffffffffUL
};

/* The extent flag definitions
 */
enum LIBFSNTFS_EXTENT_FLAGS
{
	/* The extent is sparse
	 */
	LIBFSNTFS_EXTENT_FLAG_IS_SPARSE					= 0x00000001UL,

	/* The extent is compressed
	 */
	LIBFSNTFS_EXTENT_FLAG_IS_COMPRESSED				= 0x00000002UL
};

#endif /* !defined( HAVE_LOCAL_LIBFSNTFS ) */

/* The attribute flags
 */
enum LIBFSNTFS_ATTRIBUTE_FLAGS
{
	LIBFSNTFS_ATTRIBUTE_FLAG_SPARSE					= 0x8000
};

#define LIBFSNTFS_ATTRIBUTE_FLAG_COMPRESSION_MASK			0x00ff

/* The MFT entry flags
 */
enum LIBFSNTFS_MFT_ENTRY_FLAGS
{
	LIBFSNTFS_MFT_ENTRY_FLAG_IN_USE					= 0x0001,
	LIBFSNTFS_MFT_ENTRY_FLAG_INDEX_PRESENT				= 0x0002,
};

/* The predefined MFT entry index values
 */
enum LIBFSNTFS_MFT_ENTRY_INDEXES
{
	LIBFSNTFS_MFT_ENTRY_INDEX_MFT					= 0,
	LIBFSNTFS_MFT_ENTRY_INDEX_MFT_MIRROR				= 1,

	LIBFSNTFS_MFT_ENTRY_INDEX_VOLUME				= 3,

	LIBFSNTFS_MFT_ENTRY_INDEX_ROOT_DIRECTORY			= 5,
	LIBFSNTFS_MFT_ENTRY_INDEX_BITMAP				= 6,

	LIBFSNTFS_MFT_ENTRY_INDEX_SECURE				= 9,

	LIBFSNTFS_MFT_ENTRY_INDEX_UPPERCASE				= 10,
};

/* The index node flags
 */
enum LIBFSNTFS_INDEX_NODE_FLAGS
{
	LIBFSNTFS_INDEX_NODE_FLAG_HAS_ALLOCATION_ATTRIBUTE		= 0x00000001UL,
};

/* The index value flags
 */
enum LIBFSNTFS_INDEX_VALUE_FLAGS
{
	LIBFSNTFS_INDEX_VALUE_FLAG_HAS_SUB_NODE				= 0x00000001UL,
	LIBFSNTFS_INDEX_VALUE_FLAG_IS_LAST				= 0x00000002UL
};

/* The file name attribute (name string) name space values
 */
enum LIBFSNTFS_FILE_NAME_SPACES
{
	LIBFSNTFS_FILE_NAME_SPACE_POSIX					= 0x00,
	LIBFSNTFS_FILE_NAME_SPACE_WINDOWS				= 0x01,
	LIBFSNTFS_FILE_NAME_SPACE_DOS					= 0x02,
	LIBFSNTFS_FILE_NAME_SPACE_DOS_WINDOWS				= 0x03
};

/* The file entry flags
 */
enum LIBFSNTFS_FILE_ENTRY_FLAGS
{
	/* The file entry is based on MFT metadata only
	 */
	LIBFSNTFS_FILE_ENTRY_FLAGS_MFT_ONLY				= 0x01
};

/* The compression methods
 */
enum LIBFSNTFS_COMPRESSION_METHODS
{
	LIBFSNTFS_COMPRESSION_METHOD_NONE				= 0,
	LIBFSNTFS_COMPRESSION_METHOD_LZNT1				= 1,
	LIBFSNTFS_COMPRESSION_METHOD_LZXPRESS_HUFFMAN			= 2,
	LIBFSNTFS_COMPRESSION_METHOD_LZX				= 3
};

/* Flag to indicate the MFT entry is based on MFT metadata only
 */
#define LIBFSNTFS_MFT_ENTRY_FLAG_MFT_ONLY				LIBFDATA_RANGE_FLAG_USER_DEFINED_1

/* Flag to indicate the index value in the list is stored in the $INDEX_ROOT
 */
#define LIBFSNTFS_INDEX_VALUE_LIST_FLAG_STORED_IN_ROOT			LIBFDATA_RANGE_FLAG_USER_DEFINED_1

#define LIBFSNTFS_DIRECTORY_ENTRIES_TREE_MAXIMUM_NUMBER_OF_SUB_NODES	257
#define LIBFSNTFS_INDEX_TREE_MAXIMUM_NUMBER_OF_SUB_NODES		257

#define LIBFSNTFS_MAXIMUM_CACHE_ENTRIES_COMPRESSED_BLOCKS		2
#define LIBFSNTFS_MAXIMUM_CACHE_ENTRIES_INDEX_NODES			128
#define LIBFSNTFS_MAXIMUM_CACHE_ENTRIES_MFT_ENTRIES			32

#define LIBFSNTFS_MAXIMUM_RECURSION_DEPTH				256

#endif /* !defined( _LIBFSNTFS_INTERNAL_DEFINITIONS_H ) */

