/*************************************************************************
 *                                                                       *
 * Copyright (c) 2021-2023 Howard M. Harte.                              *
 * https://github.com/hharte                                             *
 *                                                                       *
 * Permission is hereby granted, free of charge, to any person obtaining *
 * a copy of this software and associated documentation files (the       *
 * "Software"), to deal in the Software without restriction, including   *
 * without limitation the rights to use, copy, modify, merge, publish,   *
 * distribute, sublicense, and/or sell copies of the Software, and to    *
 * permit persons to whom the Software is furnished to do so, subject to *
 * the following conditions:                                             *
 *                                                                       *
 * The above copyright notice and this permission notice shall be        *
 * included in all copies or substantial portions of the Software.       *
 *                                                                       *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    *
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-            *
 * INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE   *
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN       *
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN     *
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE      *
 * SOFTWARE.                                                             *
 *                                                                       *
 * Except as contained in this notice, the names of The Authors shall    *
 * not be used in advertising or otherwise to promote the sale, use or   *
 * other dealings in this Software without prior written authorization   *
 * from the Authors.                                                     *
 *                                                                       *
 * Module Description:                                                   *
 *     Integrated Business Computers (IBC) Middi Cadet HP and Megastar   *
 * module for SIMH.                                                      *
 *                                                                       *
 * This module is a wrapper around the wd179x FDC module, and adds the   *
 * IBC-specific registers as well as the Boot ROM.                       *
 *                                                                       *
 *************************************************************************/

#include "altairz80_defs.h"
#include "wd179x.h"

#define DEV_NAME                "IBC"

/* Debug flags */
#define ERROR_MSG               (1 << 0)
#define DRIVE_MSG               (1 << 1)
#define PIO_MSG                 (1 << 2)
#define SBD_MSG                 (1 << 3)
#define RTC_MSG                 (1 << 4)
#define BANK_MSG                (1 << 5)
#define UART_MSG                (1 << 6)
#define CACHE_MSG               (1 << 7)
#define UNHANDLED_IO_MSG        (1 << 8)
#define FIFO_MSG                (1 << 9)
#define DIPSW_MSG               (1 << 10)
#define VERBOSE_MSG             (1 << 11)

#define IBC_MAX_UNITS           (4)
#define IBC_NUM_SIO             16      /* Number of serial ports. */
#define IBC_ROM_SIZE            (8 * 1024)
#define IBC_ROM_ADDR_MASK       (IBC_ROM_SIZE - 1)
#define FDC_FIFO_LEN            (1024)
#define FDC_FIFO_MASK           (FDC_FIFO_LEN - 1)

#define IBC_SW_E_DEFAULT_VALUE  0xFF

typedef enum ibc_model {
    ibc_mcc = 0,
    ibc_scc
} ibc_model_t;

typedef struct {
    PNP_INFO    pnp;    /* Plug and Play */
    uint8 rom_disabled; /* TRUE if ROM has been disabled */
    ibc_model_t model;
    uint8 dipsw_E;      /* 8-position DIP switch at location E. */
    uint8* cache;       /* IBC CACHE storage */
    uint8 param;
    uint8 head_sel;
    uint8 autowait;
    uint8 rtc;
    uint8 imask;        /* Interrupt Mask Register */
    uint8 ipend;        /* Interrupt Pending Register */
    uint32 cache_wbase;
    uint32 cache_rbase;
    uint8 cache_index;
    uint8 fdc_fifo[FDC_FIFO_LEN];
    uint16 fdc_fifo_rd_index;
} IBC_INFO;

extern WD179X_INFO_PUB *wd179x_infop;

extern t_stat set_membase(UNIT *uptr, int32 val, CONST char *cptr, void *desc);
extern t_stat show_membase(FILE *st, UNIT *uptr, int32 val, CONST void *desc);
extern t_stat set_iobase(UNIT *uptr, int32 val, CONST char *cptr, void *desc);
extern t_stat show_iobase(FILE *st, UNIT *uptr, int32 val, CONST void *desc);
extern uint32 sim_map_resource(uint32 baseaddr, uint32 size, uint32 resource_type,
                               int32 (*routine)(const int32, const int32, const int32), const char* name, uint8 unmap);
extern void setBankSelect(const int32 b);
extern uint8 GetBYTEWrapper(const uint32 Addr);

extern int32 sio0d(const int32 port, const int32 io, const int32 data);
extern int32 sio0s(const int32 port, const int32 io, const int32 data);

void wd179x_connect_external_fifo(uint16 fifo_len, uint8* storage);
void wd179x_reset_external_fifo(void);

extern uint32 PCX;                              /* external view of PC  */
extern int32  IR_S;                             /* Z80 Interrupt/Refresh register */
extern uint32 vectorInterrupt;                  /* Interrupt pending */
extern uint8 dataBus[MAX_INT_VECTORS];          /* IBC interrupt data bus values  */

#define IBC_CAPACITY            (77*1*26*128)   /* Default SSSD Disk Capacity         */

#define MOTOR_TO_LIMIT          128

static t_stat ibc_reset(DEVICE *ibc_dev);
static t_stat ibc_boot(int32 unitno, DEVICE *dptr);
static t_stat ibc_attach(UNIT *uptr, CONST char *cptr);
static t_stat ibc_detach(UNIT *uptr);
static t_stat ibc_set_model(UNIT* uptr, int32 value, CONST char* cptr, void* desc);

static int32 ibc_sio(const int32 port, const int32 io, const int32 data);
static int32 ibc_pio(const int32 port, const int32 io, const int32 data);
static int32 ibc_param_reg(const int32 port, const int32 io, const int32 data);
static int32 ibc_banksel(const int32 port, const int32 io, const int32 data);
static int32 ibc_rtc(const int32 port, const int32 io, const int32 data);
static int32 ibc_cart(const int32 port, const int32 io, const int32 data);
static int32 ibc_reel(const int32 port, const int32 io, const int32 data);
static int32 ibc_dev31(const int32 port, const int32 io, const int32 data);
static int32 ibc_rom(const int32 port, const int32 io, const int32 data);
static int32 ibc_fdc_data(const int32 port, const int32 io, const int32 data);
static int32 ibc_sc_cache(const int32 port, const int32 io, const int32 data);
static int32 ibc_sc_baud(const int32 port, const int32 io, const int32 data);
static int32 ibc_sc_dtr(const int32 port, const int32 io, const int32 data);
static int32 ibc_unhandled(const int32 port, const int32 io, const int32 data);
static const char* ibc_description(DEVICE *dptr);

/* Disk Control/Flags Register, 0x2a (OUT) */
#define IBC_PARAM_DDENS         (1 << 3)    /* FDC Parameter Register */

/* IBC Middi Cadet I/O Ports */
#define IBC_SIO                 0x04    /* 0x04-0x13: UARTs (using AltairZ80 SIO.)  UARTS 0x00-0x03 are using 2sio. */
#define IBC_FDC_DATA            0x28    /* FDC Data Register */
#define IBC_PARAM               0x2a    /* FDC PARAM Register */
#define IBC_DIPSWE              0x3c    /* CPU Board DIP Switch E */
#define IBC_BANKSEL             0x38    /* Bank Select Register */
#define IBC_FIFO_CTRL           0x3e    /* FDC FIFO Control */
#define IBC_ROM_CTRL            0x3f    /* ROM Control Register */
#define IBC_SCC_BANKSEL         0x58    /* Bank Select register */
#define IBC_SCC_FIFO_CTRL       0x5e    /* FDC FIFO Control */
#define IBC_SCC_ROM_CTRL        0x5f    /* ROM Control Register */
#define IBC_HDC                 0x40    /* 0x40-0x4F: Hard Disk Controller */
#define IBC_CART                0x60    /* 0x60-0x63: Cartridge Tape Controller */
#define IBC_REEL                0x64    /* 0x64-0x67: Reel to Reel Tape Controller */
#define IBC_RTC                 0x70    /* 0x70-0x7F: RTC */
#define IBC_DEV31               0x80    /* 0x80-0x83: SYSTEM.DEV31 ? */
#define IBC_PIO                 0xec    /* 0xEC-0xEF: Centronics Parallel Port */

#define IBC_CACHE_BASE_H        0x00
#define IBC_CACHE_BASE_L_WR     0x01
#define IBC_CACHE_BASE_L_RD     0x02
#define IBC_CACHE_DATA          0x03

/* Z80 Mode 2 Interrupt Requests */
#define IRQ_VI0                 (1 << 0)
#define IRQ_VI1                 (1 << 1)
#define IRQ_VI10                (1 << 10)
#define IRQ_VI15                (1 << 15)

#define IBC_CACHE_SIZE          (256 * 1024)    /* CACHE is 256K */
#define IBC_CACHE_MASK          (IBC_CACHE_SIZE - 1)

#define RST_OPCODE_TO_VECTOR(x) (x & 0x38)

#define DEFAULT_TIMER_DELTA     100         /* default value for timer delta in ms          */

static IBC_INFO ibc_info_data = { { 0x0000, IBC_ROM_SIZE, 0x3, 2 }, 0, ibc_mcc, IBC_SW_E_DEFAULT_VALUE, NULL };
static IBC_INFO* ibc_info = &ibc_info_data;

static uint8 current_baud[IBC_NUM_SIO] = { 0 };
static uint8 baud_unlock[IBC_NUM_SIO] = { 0 };

/* The IBC does not really have RAM associated with it, but for ease of integration with the
 * SIMH/AltairZ80 Resource Mapping Scheme, rather than Map and Unmap the ROM, simply implement our
 * own RAM that can be swapped in when the IBC Boot ROM is disabled.
 */
static uint8 ibc_ram[IBC_ROM_SIZE];

#define IBC_WAIT  16
#define IBC_UDATA(act,fl,wait,u4,name) NULL,act,NULL,NULL,NULL,0,0,(fl),0,(0),0,NULL,0,0,wait,0,u4,0,0,NULL,NULL,0,0,0,NULL,0,0,NULL,0,name

static UNIT ibc_unit[] = {
    { UDATA (NULL, UNIT_FIX + UNIT_ATTABLE + UNIT_DISABLE + UNIT_ROABLE, IBC_CAPACITY), 1024 },
    { UDATA (NULL, UNIT_FIX + UNIT_ATTABLE + UNIT_DISABLE + UNIT_ROABLE, IBC_CAPACITY) },
    { UDATA (NULL, UNIT_FIX + UNIT_ATTABLE + UNIT_DISABLE + UNIT_ROABLE, IBC_CAPACITY) },
    { UDATA (NULL, UNIT_FIX + UNIT_ATTABLE + UNIT_DISABLE + UNIT_ROABLE, IBC_CAPACITY) },
};

static REG ibc_reg[] = {
    { HRDATAD (PARAM,       ibc_info_data.param,          8, "FDC PARAM Register"), },
    { HRDATAD (DIPSWE,      ibc_info_data.dipsw_E,        8, "DIP Switch at location E on the MCC CPU board"), },
    { FLDATAD (PROMDIS,     ibc_info_data.rom_disabled,   1, "PROM Disable"),  },
    { HRDATAD (MODEL,       ibc_info_data.model,          8, "Model (0=MultiStar, 1=MegaStar)"), },
    { HRDATAD (CACHWRBASE,  ibc_info_data.cache_wbase,   32, "CACHE Write Base"), },
    { HRDATAD (CACHERBASE,  ibc_info_data.cache_rbase,   32, "CACHE Read Base"), },
    { HRDATAD (CACHINDEX,   ibc_info_data.cache_index,    8, "CACHE Index"), },
    { NULL }
};

#define IBC_NAME  "IBC Middi Cadet HP (IBC MCC CPU)"

static const char* ibc_description(DEVICE *dptr) {
    if (dptr == NULL) {
        return NULL;
    }

    return IBC_NAME;
}

static t_stat ibc_show_vectable(FILE* st, UNIT* uptr, int32 val, CONST void* desc);

#define UNIT_IBC_V_MCC          (UNIT_V_UF+1)               /* Set model to MCC */
#define UNIT_IBC_MCC            (1 << UNIT_IBC_V_MCC)
#define UNIT_IBC_V_SCC          (UNIT_V_UF+2)               /* Set model to SCC */
#define UNIT_IBC_SCC            (1 << UNIT_IBC_V_SCC)


static MTAB ibc_mod[] = {
    { MTAB_XTD|MTAB_VDV,    0,                  "MEMBASE",  "MEMBASE",
        &set_membase, &show_membase, NULL, "Sets disk controller memory base address"       },
    { MTAB_XTD|MTAB_VDV,    0,                  "IOBASE",   "IOBASE",
        &set_iobase, &show_iobase, NULL, "Sets disk controller I/O base address"            },
    { MTAB_XTD | MTAB_VDV,  0,                  NULL,   "VECTABLE",
        NULL, &ibc_show_vectable, NULL, "Show Z80 Mode 2 Interrupt Vector Table" },
    { MTAB_XTD | MTAB_VDV,  ibc_mcc,    NULL,       "MCC",       &ibc_set_model,
        NULL, NULL, "Set IBC Model to MCC (MultiStar)"            },
    { MTAB_XTD | MTAB_VDV,  ibc_scc,    NULL,       "SCC",       &ibc_set_model,
        NULL, NULL, "Set IBC Model to SCC (MegaStar)"            },
    { 0 }
};

/* Debug Flags */
static DEBTAB ibc_dt[] = {
    { "ERROR",      ERROR_MSG,          "Error messages"            },
    { "BANK",       BANK_MSG,           "Memory Control messages"   },
    { "DRIVE",      DRIVE_MSG,          "Drive messages"            },
    { "VERBOSE",    VERBOSE_MSG,        "Verbose messages"          },
    { "PIO",        PIO_MSG,            "PIO messages"              },
    { "RTC",        RTC_MSG,            "RTC messages"              },
    { "SBD",        SBD_MSG,            "System Board messages"     },
    { "UART",       UART_MSG,           "UART messages"             },
    { "CACHE",      CACHE_MSG,          "CACHE messages"            },
    { "UIO",        UNHANDLED_IO_MSG,   "Unsupported I/O Ports"     },
    { "FIFO",       FIFO_MSG,           "FDC FIFO messages"         },
    { "DIPSW",      DIPSW_MSG,          "DIP Switch messages"       },
    { NULL,         0,                  NULL                        }
};

DEVICE ibc_dev = {
    "IBC", ibc_unit, ibc_reg, ibc_mod,
    IBC_MAX_UNITS,   /* # units */
    10,                 /* address radix */
    31,                 /* address width */
    1,                  /* address increment */
    8,                  /* data radix */
    8,                  /* data width */
    NULL,               /* examine routine */
    NULL,               /* deposit routine */
    &ibc_reset,         /* Reset routine */
    &ibc_boot,          /* boot routine */
    &ibc_attach,        /* attach routine */
    &ibc_detach,        /* detach routine */
    &ibc_info_data,     /* context */
    (DEV_DISABLE | DEV_DIS | DEV_DEBUG), ERROR_MSG,
    ibc_dt, NULL, NULL, NULL, NULL, NULL,
    &ibc_description
};

/*  Synthetic device SIMH for communication
    between Altair and SIMH environment using port 0xfe */
static t_stat ibctimer_svc(UNIT* uptr);
static t_stat ibctimer_reset(DEVICE* dptr);

static UNIT ibctimer_unit = {
    UDATA(&ibctimer_svc, 0, 0), KBD_POLL_WAIT
};

static REG ibctimer_reg[] = {
    { NULL }
};

static MTAB ibctimer_mod[] = {
    /* timer generated interrupts are off   */
    { 0 }
};

const char* ibctimer_description(DEVICE* dptr) {
    if (dptr == NULL) {
        return NULL;
    }

    return "IBC Timer";
}

#define TIMER_MSG   (1 << 1)
#define IN_MSG      (1 << 2)
#define OUT_MSG     (1 << 3)

/* Debug Flags */
static DEBTAB ibctimer_dt[] = {
    { "ERROR",      ERROR_MSG,      "Error messages"    },
    { "TIMER",      TIMER_MSG,      "Timer messages"    },
    { "IN_MSG",     IN_MSG,         "Input messages"    },
    { "OUT_MSG",    OUT_MSG,        "Output messages"   },
    { NULL,         0                                   }
};

DEVICE ibctimer_device = {
    "IBCTIMER", &ibctimer_unit, ibctimer_reg, ibctimer_mod,
    1, 10, 31, 1, 16, 4,
    NULL, NULL, &ibctimer_reset,
    NULL, NULL, NULL,
    NULL, (DEV_DISABLE | DEV_DEBUG), 0,
    ibctimer_dt, NULL, NULL, NULL, NULL, NULL, &ibctimer_description
};

/* Port 0x14 IBC Periodic Timer */
int32 ibctimer_dev(const int32 port, const int32 io, const int32 data) {
    int32 result = 0xFF;
    if (io == 0) {
        sim_debug(IN_MSG, &ibctimer_device, ": " ADDRESS_FORMAT
            " IN(0x%02x)=0x%02x: Clear Status / Reset timer.\n", PCX, port, result);
        sim_cancel(&ibctimer_unit);    /* cancel unit */
    }
    else {
        sim_debug(OUT_MSG, &ibctimer_device, ": " ADDRESS_FORMAT
            "OUT(0x%02x)=0x%02x: Set timer, count=%d.\n", PCX, port, data, data);
        sim_activate_after(&ibctimer_unit, 250);
    }
    return result;
}

static t_stat ibctimer_reset(DEVICE* dptr) {
    if (dptr == NULL) {
        return SCPE_IERR;
    }

    sim_cancel(&ibctimer_unit);    /* cancel unit */
    sim_debug(TIMER_MSG, &ibctimer_device, ": " ADDRESS_FORMAT
        " Reset timer.\n", PCX);
    return SCPE_OK;
}

/* Unit service routine */
static t_stat ibctimer_svc(UNIT* uptr)
{
    if (uptr == NULL) {
        return (SCPE_IERR);
    }

    if (ibc_info->model == ibc_mcc) {
        /* MultiStar */
        vectorInterrupt |= IRQ_VI0;
        dataBus[0] = 0;
    } else {
        /* Megastar */
        if ((IR_S & 0xFF00) == 0x0900) {
//#define THEOS8M
#ifdef THEOS8M
            // THEOS8-M
            vectorInterrupt |= IRQ_VI10;
            dataBus[10] = 20;
#else
            // THEOS8-S
            vectorInterrupt |= IRQ_VI15;
            dataBus[15] = 30;
#endif
        }
        else {
            sim_printf("IBCTIMER - MegaStar - Unhandled!\n");
        }
    }

    return SCPE_OK;
}

/* Port 0x20: MegaStar RTC Timer */
static t_stat ibcrtctimer_svc(UNIT* uptr);
static t_stat ibcrtctimer_reset(DEVICE* dptr);

static UNIT ibcrtctimer_unit = {
    UDATA(&ibcrtctimer_svc, 0, 0), KBD_POLL_WAIT
};

static REG ibcrtctimer_reg[] = {
    { NULL }
};

static MTAB ibcrtctimer_mod[] = {
    /* timer generated interrupts are off   */
    { 0 }
};

const char* ibcrtctimer_description(DEVICE* dptr) {
    if (dptr == NULL) {
        return NULL;
    }

    return "IBC RTC Timer";
}

DEVICE ibcrtctimer_device = {
    "IBCRTCTIMER", &ibcrtctimer_unit, ibcrtctimer_reg, ibcrtctimer_mod,
    1, 10, 31, 1, 16, 4,
    NULL, NULL, &ibcrtctimer_reset,
    NULL, NULL, NULL,
    NULL, (DEV_DISABLE | DEV_DEBUG), 0,
    ibctimer_dt, NULL, NULL, NULL, NULL, NULL, &ibcrtctimer_description
};

/* port 0x20 IBC Super Cadet RTC Timer */
int32 ibcrtctimer_dev(const int32 port, const int32 io, const int32 data) {
    int32 result = 0xFF;
    if (io == 0) {
        sim_debug(IN_MSG, &ibcrtctimer_device, ": " ADDRESS_FORMAT
            " IN(0x%02x)=0x%02x: Clear Status / Reset timer.\n", PCX, port, result);
        sim_cancel(&ibcrtctimer_unit);    /* cancel unit */
    }
    else {
        sim_debug(OUT_MSG, &ibcrtctimer_device, ": " ADDRESS_FORMAT
            "OUT(0x%02x)=0x%02x: Set timer, count=%d.\n", PCX, port, data, data);
        if (!sim_is_active(&ibcrtctimer_unit)) {
            sim_activate(&ibcrtctimer_unit, sim_rtcn_init_unit_ticks(&ibcrtctimer_unit, 20, 0, 1));
        }
    }
    return result;
}

static t_stat ibcrtctimer_reset(DEVICE* dptr) {
    if (dptr == NULL) {
        return SCPE_IERR;
    }

    sim_cancel(&ibcrtctimer_unit);    /* cancel unit */
    sim_debug(TIMER_MSG, &ibcrtctimer_device, ": " ADDRESS_FORMAT
        " Reset timer.\n", PCX);
    return SCPE_OK;
}

/* Unit service routine */
static t_stat ibcrtctimer_svc(UNIT* uptr)
{
    if (uptr == NULL) {
        return SCPE_IERR;
    }

    vectorInterrupt |= IRQ_VI1;
    dataBus[1] = 2;
    sim_rtcn_calb_tick(0);
    sim_activate_after(&ibcrtctimer_unit, 1000000);
    return SCPE_OK;
}


t_stat ibc_show_vectable(FILE* st, UNIT* uptr, int32 val, CONST void* desc)
{
    uint8 i;
    int32 vectable = (IR_S & 0xFF00);
    int32 vector;

    if (uptr == NULL) {
        return SCPE_IERR;
    }

    fprintf(st, "Vector table @0x%04x\n--------------------\n", vectable);

    for (i = 0; i < 64; i++) {
        vector = (GetBYTEWrapper(vectable + (i * 2) + 1) << 8) | GetBYTEWrapper(vectable + (i * 2));
        if (vector != 0) fprintf(st, "%02d @0x%04x=0x%04x\n", i, vectable + (i * 2), vector);
    }
    return SCPE_OK;
}


/* IBC MultiStar Boot ROM DMP011 REV L. SEEQ 2764 */
static uint8 ibc_rom_data[2][IBC_ROM_SIZE] = {
  {
    0xF3, 0xDB, 0x14, 0xDB, 0x24, 0xDB, 0x80, 0xAF, 0xD3, 0x62, 0xD3, 0x40, 0xD3, 0x44, 0xD3, 0x47,
    0x01, 0x00, 0x00, 0x21, 0x00, 0x00, 0x77, 0x23, 0x0D, 0x20, 0xFB, 0x10, 0xF9, 0x3E, 0x03, 0x01,
    0x00, 0x0A, 0xED, 0x79, 0x0C, 0xED, 0x50, 0x0C, 0x10, 0xF8, 0x31, 0xFE, 0xFF, 0x3E, 0xE1, 0xED,
    0x47, 0x11, 0x8A, 0x01, 0x21, 0xF7, 0x02, 0xDB, 0x3C, 0xCB, 0x77, 0x20, 0x26, 0xAF, 0xD3, 0x3E,
    0xD3, 0x28, 0x3C, 0xD3, 0x28, 0x3C, 0xD3, 0x28, 0xD3, 0x3E, 0xDB, 0x28, 0xB7, 0x20, 0x0E, 0xDB,
    0x28, 0xFE, 0x01, 0x20, 0x08, 0xDB, 0x28, 0xFE, 0x02, 0x20, 0x02, 0x18, 0x06, 0x11, 0x40, 0x04,
    0x21, 0xA1, 0x05, 0x18, 0x07, 0x00, 0x00, 0x00, 0xC3, 0xAE, 0x0E, 0x00, 0xD5, 0xAF, 0xD3, 0x1C,
    0xED, 0x52, 0xE5, 0xC1, 0x11, 0x00, 0xE1, 0xE1, 0xED, 0xB0, 0xDB, 0x24, 0x11, 0x00, 0xE1, 0x2A,
    0x04, 0xE1, 0x19, 0x22, 0x04, 0xE1, 0xED, 0x5E, 0xFB, 0x01, 0x11, 0x00, 0x11, 0x22, 0xE4, 0x21,
    0x4D, 0x01, 0xED, 0xB0, 0x3E, 0x11, 0x01, 0x00, 0x0A, 0xED, 0x79, 0x0C, 0x0C, 0x10, 0xFA, 0xDB,
    0x3C, 0xCB, 0x57, 0xCA, 0x65, 0x0A, 0xCB, 0x7F, 0x20, 0x04, 0x3E, 0x01, 0xD3, 0x20, 0xDB, 0x3C,
    0xCB, 0x47, 0x28, 0x13, 0xCB, 0x4F, 0xCA, 0xF7, 0x02, 0x3E, 0x30, 0xD3, 0x2A, 0x32, 0x00, 0xE4,
    0xDB, 0x24, 0xCB, 0x7F, 0xCA, 0x5E, 0x01, 0xAF, 0x32, 0x1A, 0xE4, 0xCD, 0x83, 0x0E, 0x3A, 0x1A,
    0xE4, 0x3C, 0x3C, 0xFE, 0x14, 0x30, 0xC8, 0x18, 0xEF, 0x21, 0x1D, 0x01, 0xCD, 0xD1, 0x0E, 0xCD,
    0xCE, 0x0E, 0xCD, 0x83, 0x0E, 0x28, 0xFB, 0x11, 0xDF, 0x00, 0xD5, 0x67, 0xCD, 0x6F, 0x0E, 0x6F,
    0xE5, 0x06, 0x19, 0x21, 0x6E, 0x08, 0xDB, 0x3C, 0xCB, 0x77, 0x20, 0x03, 0x21, 0xBE, 0x09, 0x56,
    0x23, 0x5E, 0x23, 0xE3, 0xE5, 0xAF, 0xED, 0x52, 0xE1, 0x20, 0x06, 0xE1, 0x5E, 0x23, 0x56, 0xD5,
    0xC9, 0xE3, 0x23, 0x23, 0x10, 0xE9, 0xE1, 0x21, 0xD2, 0x08, 0xC3, 0xA5, 0x0E, 0x2F, 0x0D, 0x0A,
    0x49, 0x42, 0x43, 0x20, 0x4D, 0x75, 0x6C, 0x74, 0x69, 0x53, 0x74, 0x61, 0x72, 0x2A, 0x20, 0x53,
    0x45, 0x52, 0x49, 0x45, 0x53, 0x20, 0x20, 0x20, 0x4C, 0x6F, 0x61, 0x64, 0x65, 0x72, 0x20, 0x50,
    0x52, 0x4F, 0x4D, 0x20, 0x20, 0x56, 0x33, 0x2E, 0x34, 0x20, 0x0D, 0x0A, 0x0A, 0x53, 0x59, 0x53,
    0x54, 0x45, 0x4D, 0x20, 0x20, 0x4E, 0x55, 0x43, 0x4C, 0x45, 0x55, 0x53, 0x20, 0x00, 0xAF, 0xD3,
    0x40, 0xD3, 0x44, 0x06, 0x10, 0xDB, 0x3C, 0xCB, 0x77, 0x28, 0x02, 0x06, 0x1A, 0x78, 0x32, 0x39,
    0xE5, 0xAF, 0x32, 0x15, 0xE4, 0x32, 0x3B, 0xE5, 0xCD, 0x02, 0xE1, 0x06, 0x00, 0x1E, 0x00, 0x21,
    0x00, 0x80, 0xCD, 0x08, 0xE1, 0x38, 0xD7, 0xC3, 0x22, 0x03, 0xAF, 0xC9, 0x18, 0x09, 0x62, 0x01,
    0x18, 0x15, 0x18, 0x09, 0xC3, 0xC0, 0x05, 0x3E, 0x01, 0x32, 0x1C, 0xE4, 0xC9, 0x16, 0x8C, 0xCB,
    0x40, 0x28, 0x0C, 0xCB, 0xCA, 0x18, 0x08, 0x16, 0xAC, 0xCB, 0x40, 0x28, 0x02, 0xCB, 0xCA, 0x0E,
    0x00, 0xE5, 0x21, 0x16, 0xE4, 0x3A, 0x15, 0xE4, 0xE6, 0x03, 0x85, 0x6F, 0x30, 0x01, 0x24, 0xE3,
    0xDD, 0xE1, 0x3A, 0x15, 0xE4, 0xF6, 0x38, 0xCB, 0x6A, 0x28, 0x02, 0xCB, 0xAF, 0xCB, 0x40, 0x28,
    0x02, 0xCB, 0xD7, 0x47, 0xDB, 0x3C, 0xCB, 0x77, 0x20, 0x04, 0xCB, 0xF0, 0x18, 0x1C, 0x3A, 0x39,
    0xE5, 0xFE, 0x0E, 0x38, 0x0E, 0xAF, 0xBB, 0x20, 0x11, 0x3A, 0x3B, 0xE5, 0xB7, 0x28, 0x04, 0xCB,
    0x50, 0x20, 0x07, 0x78, 0x06, 0x80, 0xCB, 0x9F, 0x18, 0x03, 0x78, 0x06, 0x00, 0xD3, 0x2A, 0x32,
    0x00, 0xE4, 0x78, 0x32, 0x1D, 0xE4, 0xDB, 0x24, 0xCB, 0x47, 0x20, 0xFA, 0xCB, 0x7F, 0x20, 0x3B,
    0x3A, 0x1C, 0xE4, 0x1F, 0x30, 0x13, 0x3E, 0x0C, 0xD3, 0x24, 0x18, 0xFE, 0xCB, 0x7F, 0x3E, 0x52,
    0x20, 0x29, 0xAF, 0x32, 0x1C, 0xE4, 0xDD, 0x77, 0x00, 0xDD, 0x7E, 0x00, 0xD3, 0x25, 0x3E, 0x0A,
    0x3D, 0x20, 0xFD, 0xDB, 0x24, 0xCB, 0x7F, 0x3E, 0x73, 0x20, 0x10, 0x7B, 0xDD, 0x77, 0x00, 0xD3,
    0x27, 0x3E, 0x1C, 0xD3, 0x24, 0x18, 0xFE, 0xCB, 0x7F, 0x3E, 0x53, 0x20, 0x7E, 0xCB, 0x52, 0x20,
    0x0C, 0xC5, 0x06, 0xFA, 0xC5, 0x06, 0x19, 0x10, 0xFE, 0xC1, 0x10, 0xF8, 0xC1, 0x3E, 0x05, 0x32,
    0x05, 0xE5, 0xCB, 0x6A, 0x28, 0x0F, 0x3A, 0x00, 0xE4, 0xCB, 0x67, 0xD3, 0x2A, 0xD3, 0x3E, 0xC5,
    0x0E, 0x28, 0xED, 0xB3, 0xC1, 0x3A, 0x00, 0xE4, 0xD3, 0x2A, 0xD3, 0x3E, 0x3A, 0x3B, 0xE5, 0xFE,
    0x09, 0x28, 0x07, 0xFE, 0x05, 0x79, 0x28, 0x04, 0x18, 0x0B, 0x79, 0x87, 0x87, 0x87, 0x81, 0xD6,
    0x1A, 0x30, 0xFC, 0xC6, 0x1A, 0x3C, 0xD3, 0x26, 0x7A, 0xCB, 0x92, 0xD3, 0x24, 0x18, 0xFE, 0xE6,
    0xDC, 0x20, 0x24, 0xCB, 0x6A, 0x20, 0x0F, 0x3A, 0x00, 0xE4, 0xCB, 0xA7, 0xD3, 0x2A, 0xD3, 0x3E,
    0xC5, 0x0E, 0x28, 0xED, 0xB2, 0xC1, 0x0C, 0xDB, 0x3C, 0xCB, 0x77, 0x3E, 0x19, 0x20, 0x02, 0x3E,
    0x0F, 0xB9, 0x30, 0x99, 0xAF, 0x18, 0x24, 0xCB, 0x7F, 0x3E, 0x44, 0x20, 0x09, 0x3A, 0x05, 0xE5,
    0x3D, 0x32, 0x05, 0xE5, 0x20, 0x8C, 0x20, 0x02, 0x3E, 0x44, 0x47, 0x2A, 0x49, 0xE4, 0x2B, 0x7C,
    0xB5, 0x22, 0x49, 0xE4, 0x37, 0x20, 0x04, 0x78, 0xD3, 0x01, 0x37, 0xC9, 0xE3, 0x23, 0x23, 0xE3,
    0xDB, 0x24, 0x32, 0x01, 0xE4, 0xFB, 0xC9, 0x3E, 0x02, 0x32, 0x34, 0xE4, 0x01, 0x03, 0x01, 0x11,
    0x00, 0xE1, 0x21, 0x3D, 0x03, 0xED, 0xB0, 0xAF, 0x32, 0x1D, 0xE4, 0x3E, 0x0A, 0x32, 0x39, 0xE5,
    0xCD, 0x00, 0xE1, 0x38, 0xE2, 0x06, 0x00, 0x11, 0x00, 0x00, 0x21, 0x00, 0x80, 0xCD, 0x08, 0xE1,
    0x38, 0xD5, 0x01, 0x00, 0x01, 0x11, 0x00, 0xC0, 0x21, 0x00, 0x80, 0xED, 0xB0, 0x0E, 0x20, 0x11,
    0x20, 0xE5, 0xED, 0xB0, 0x21, 0xD3, 0x3F, 0x22, 0xFE, 0xBF, 0xC3, 0xFE, 0xBF, 0x18, 0x15, 0x18,
    0x11, 0x4C, 0x03, 0x18, 0x0D, 0x18, 0x0F, 0xC3, 0xC0, 0x05, 0x00, 0x00, 0xF5, 0xDB, 0x24, 0xF1,
    0xFB, 0xC9, 0xAF, 0xC9, 0xAF, 0xC9, 0x3A, 0x39, 0xE5, 0x32, 0x48, 0xE4, 0x3E, 0x01, 0x32, 0x40,
    0xE4, 0xDB, 0x3C, 0xCB, 0x6F, 0x3E, 0x00, 0x20, 0x07, 0x3E, 0x07, 0x32, 0x34, 0xE4, 0x3E, 0x03,
    0x32, 0x41, 0xE4, 0x78, 0x32, 0x45, 0xE4, 0xED, 0x53, 0x42, 0xE4, 0xAF, 0x32, 0x44, 0xE4, 0x3A,
    0x48, 0xE4, 0xFE, 0x0A, 0x38, 0x02, 0x3E, 0x0A, 0x32, 0x46, 0xE4, 0xE5, 0x01, 0x00, 0x00, 0x26,
    0x02, 0xDB, 0x40, 0xE6, 0x10, 0x28, 0x0A, 0x0B, 0x78, 0xB1, 0x20, 0xF5, 0x25, 0x20, 0xF2, 0x18,
    0x4B, 0x21, 0x40, 0xE4, 0xCB, 0xFE, 0x0E, 0x43, 0x23, 0x23, 0x23, 0x06, 0x04, 0x7E, 0xED, 0x79,
    0x2B, 0x0D, 0x10, 0xF9, 0x01, 0x00, 0x00, 0xDB, 0x40, 0xE6, 0x3F, 0xFE, 0x30, 0x28, 0x07, 0x0B,
    0x78, 0xB1, 0x20, 0xF3, 0x18, 0x26, 0x21, 0x44, 0xE4, 0xCB, 0xBE, 0x0E, 0x43, 0x23, 0x23, 0x23,
    0x06, 0x04, 0x7E, 0xED, 0x79, 0x2B, 0x0D, 0x10, 0xF9, 0x1E, 0x50, 0x21, 0x00, 0x00, 0xDB, 0x40,
    0xE6, 0x10, 0x28, 0x10, 0x2B, 0x7C, 0xB5, 0x20, 0xF5, 0x1D, 0x20, 0xEF, 0x18, 0x48, 0x18, 0x8F,
    0x18, 0x99, 0x18, 0x4A, 0xDB, 0x40, 0xE6, 0x0F, 0x28, 0x04, 0xFE, 0x0A, 0x20, 0x38, 0xDB, 0x44,
    0x06, 0x14, 0xDB, 0x40, 0xCB, 0x7F, 0x28, 0x06, 0x10, 0xF8, 0xD3, 0x44, 0x18, 0x28, 0x0E, 0x48,
    0xE1, 0x3A, 0x46, 0xE4, 0x47, 0xC5, 0x06, 0x00, 0xED, 0xB2, 0xC1, 0x10, 0xF8, 0xD3, 0x44, 0x3A,
    0x46, 0xE4, 0x47, 0x3A, 0x44, 0xE4, 0x80, 0x32, 0x44, 0xE4, 0x3A, 0x48, 0xE4, 0x90, 0x32, 0x48,
    0xE4, 0x20, 0xBB, 0xAF, 0x18, 0x09, 0xE1, 0xAF, 0xD3, 0x40, 0xD3, 0x44, 0xD3, 0x47, 0x37, 0xC9,
    0xAF, 0xC9, 0x18, 0x09, 0x00, 0x00, 0x18, 0x15, 0x18, 0x09, 0xC3, 0xC0, 0x05, 0x3E, 0x01, 0x32,
    0x1C, 0xE4, 0xC9, 0x16, 0x8C, 0xCB, 0x40, 0x28, 0x0C, 0xCB, 0xCA, 0x18, 0x08, 0x16, 0xAC, 0xCB,
    0x40, 0x28, 0x02, 0xCB, 0xCA, 0x0E, 0x00, 0xE5, 0x21, 0x16, 0xE4, 0x3A, 0x15, 0xE4, 0xE6, 0x03,
    0x85, 0x6F, 0x30, 0x01, 0x24, 0xE3, 0xDD, 0xE1, 0x3A, 0x15, 0xE4, 0xF6, 0x78, 0xCB, 0x6A, 0x28,
    0x02, 0xCB, 0xAF, 0xCB, 0x40, 0x28, 0x02, 0xCB, 0xD7, 0x06, 0x00, 0xD3, 0x2A, 0x32, 0x00, 0xE4,
    0x78, 0x32, 0x1D, 0xE4, 0xDB, 0x24, 0xCB, 0x47, 0x20, 0xFA, 0xCB, 0x7F, 0x20, 0x54, 0x3A, 0x1C,
    0xE4, 0x1F, 0x30, 0x1D, 0x3E, 0x0C, 0xD3, 0x24, 0xC5, 0x06, 0x0D, 0x10, 0xFE, 0xC1, 0xDB, 0x24,
    0xCB, 0x47, 0x20, 0xFA, 0xCB, 0x7F, 0x3E, 0x52, 0x20, 0x38, 0xAF, 0x32, 0x1C, 0xE4, 0xDD, 0x77,
    0x00, 0xDD, 0x7E, 0x00, 0xD3, 0x25, 0x3E, 0x0A, 0x3D, 0x20, 0xFD, 0xDB, 0x24, 0xCB, 0x7F, 0x3E,
    0x73, 0x20, 0x1F, 0x7B, 0xDD, 0x77, 0x00, 0xD3, 0x27, 0x3E, 0x0A, 0x3D, 0x20, 0xFD, 0x3E, 0x1C,
    0xD3, 0x24, 0xC5, 0x06, 0x0D, 0x10, 0xFE, 0xC1, 0xDB, 0x24, 0xCB, 0x47, 0x20, 0xFA, 0xCB, 0x7F,
    0x3E, 0x53, 0x20, 0x4C, 0xCB, 0x52, 0x20, 0x0C, 0xC5, 0x06, 0xFA, 0xC5, 0x06, 0x19, 0x10, 0xFE,
    0xC1, 0x10, 0xF8, 0xC1, 0x3E, 0x05, 0x32, 0x05, 0xE5, 0xE5, 0x3A, 0x00, 0xE4, 0xCB, 0x67, 0xD3,
    0x2A, 0xC5, 0x3A, 0x00, 0xE4, 0xCB, 0xA7, 0xD3, 0x2A, 0x79, 0x3C, 0xD3, 0x26, 0xCB, 0x6A, 0x28,
    0x23, 0x7A, 0xCB, 0x92, 0xD3, 0x24, 0x06, 0x0C, 0x10, 0xFE, 0xC1, 0xC5, 0x0E, 0x27, 0xD5, 0x16,
    0x02, 0xDB, 0x24, 0xCB, 0x47, 0x28, 0x2A, 0xA2, 0x28, 0xF7, 0xED, 0xA3, 0x20, 0xF3, 0x18, 0x21,
    0x18, 0x3E, 0x18, 0xC5, 0x7A, 0xCB, 0x92, 0xD3, 0x24, 0x06, 0x0C, 0x10, 0xFE, 0xC1, 0xC5, 0x0E,
    0x27, 0xD5, 0x16, 0x02, 0xDB, 0x24, 0xCB, 0x47, 0x28, 0x07, 0xA2, 0x28, 0xF7, 0xED, 0xA2, 0x20,
    0xF3, 0xDB, 0x24, 0xCB, 0x47, 0x20, 0xFA, 0x78, 0xD1, 0xC1, 0xE1, 0xB7, 0x20, 0x0E, 0xE6, 0xDC,
    0x20, 0x0A, 0x24, 0x3E, 0x0F, 0x0C, 0xB9, 0x30, 0x8B, 0xAF, 0x18, 0x24, 0xCB, 0x7F, 0x3E, 0x44,
    0x20, 0x09, 0x3A, 0x05, 0xE5, 0x3D, 0x32, 0x05, 0xE5, 0x20, 0xB7, 0x20, 0x02, 0x3E, 0x44, 0x47,
    0x2A, 0x49, 0xE4, 0x2B, 0x7C, 0xB5, 0x22, 0x49, 0xE4, 0x37, 0x20, 0x04, 0x78, 0xD3, 0x01, 0x37,
    0xC9, 0x21, 0x00, 0x00, 0x1A, 0x13, 0xB7, 0xC8, 0xD6, 0x30, 0xD8, 0xFE, 0x0A, 0x3F, 0xD8, 0xD5,
    0x16, 0x00, 0x5F, 0xD5, 0x29, 0x54, 0x5D, 0x29, 0x29, 0x19, 0xD1, 0x19, 0xD1, 0xD8, 0x18, 0xE4,
    0xC5, 0xD5, 0x47, 0x11, 0x10, 0x00, 0x29, 0x7A, 0x17, 0xB8, 0x38, 0x02, 0x90, 0x2C, 0x57, 0x1D,
    0x20, 0xF4, 0xD1, 0xC1, 0xC9, 0xCD, 0x1F, 0x06, 0x4F, 0xED, 0x78, 0xCD, 0x26, 0x06, 0xC9, 0xCD,
    0x1F, 0x06, 0x4F, 0xED, 0x78, 0xCD, 0x83, 0x0E, 0x18, 0xF9, 0xCD, 0x1F, 0x06, 0x4F, 0xCD, 0x1F,
    0x06, 0xED, 0x79, 0xC9, 0xCD, 0xEA, 0x05, 0x47, 0xCD, 0x83, 0x0E, 0xED, 0x41, 0x18, 0xF9, 0xCD,
    0x1F, 0x06, 0x21, 0x00, 0x80, 0x77, 0x11, 0x01, 0x80, 0x01, 0x00, 0x1A, 0xED, 0xB0, 0xC9, 0x01,
    0x00, 0x1A, 0x21, 0x00, 0x80, 0xAF, 0x77, 0x3C, 0x23, 0x0D, 0x20, 0xFA, 0x10, 0xF8, 0xC9, 0xCD,
    0x13, 0x0F, 0xCD, 0x4E, 0x0E, 0xC9, 0xCD, 0x13, 0x0F, 0xCD, 0xE1, 0x0E, 0xC9, 0x3E, 0x1A, 0x18,
    0x02, 0x3E, 0x0D, 0x32, 0x39, 0xE5, 0xC9, 0xCD, 0x02, 0xE1, 0xC9, 0x3A, 0x01, 0xE4, 0x47, 0x21,
    0x0D, 0x09, 0xCD, 0x4C, 0x06, 0x3A, 0x00, 0xE4, 0x47, 0x21, 0x16, 0x09, 0xCD, 0xD1, 0x0E, 0x78,
    0xCD, 0x26, 0x06, 0xC9, 0x3E, 0x01, 0x18, 0x01, 0xAF, 0x32, 0x20, 0xE4, 0xC9, 0xCD, 0x13, 0x0F,
    0xCD, 0x6F, 0x0E, 0xD6, 0x30, 0x21, 0xD2, 0x08, 0xDA, 0xA5, 0x0E, 0xFE, 0x04, 0xD2, 0xA5, 0x0E,
    0x32, 0x15, 0xE4, 0xC9, 0xCD, 0x92, 0x06, 0xCD, 0x08, 0xE1, 0x21, 0xDD, 0x08, 0xDA, 0xA5, 0x0E,
    0xC9, 0xCD, 0x92, 0x06, 0xCD, 0x83, 0x0E, 0xC5, 0xD5, 0xE5, 0xCD, 0x08, 0xE1, 0xE1, 0xD1, 0xC1,
    0x18, 0xF2, 0x06, 0x02, 0x21, 0x02, 0xE4, 0xE5, 0xD1, 0xCD, 0x2C, 0x0E, 0xCD, 0xA1, 0x05, 0x7D,
    0xFE, 0x00, 0x21, 0xD2, 0x08, 0xDA, 0xA5, 0x0E, 0xFE, 0x4F, 0xD2, 0xA5, 0x0E, 0x5F, 0xDB, 0x3C,
    0xCB, 0x77, 0x28, 0x06, 0x7B, 0xFE, 0x4C, 0xD2, 0xA5, 0x0E, 0x3A, 0x20, 0xE4, 0x47, 0x21, 0x00,
    0x80, 0xC9, 0xFD, 0x21, 0x21, 0x08, 0x06, 0x4D, 0xC5, 0x3A, 0x20, 0xE4, 0x47, 0xFD, 0x5E, 0x00,
    0xFD, 0x23, 0x3E, 0x0D, 0x26, 0x00, 0x6B, 0xCD, 0xFE, 0x0E, 0xCD, 0xDC, 0x0E, 0x21, 0x00, 0x80,
    0xCD, 0x08, 0xE1, 0x30, 0x03, 0xCD, 0xCE, 0x0E, 0xCD, 0x83, 0x0E, 0xC1, 0x10, 0xDA, 0x18, 0xD2,
    0xC9, 0xCD, 0x92, 0x06, 0xCD, 0x06, 0xE1, 0x21, 0xDD, 0x08, 0xDA, 0xA5, 0x0E, 0xC9, 0xCD, 0x92,
    0x06, 0xCD, 0x83, 0x0E, 0xC5, 0xD5, 0xE5, 0xCD, 0x06, 0xE1, 0xE1, 0xD1, 0xC1, 0x18, 0xF2, 0xCD,
    0x45, 0x0E, 0x06, 0x10, 0xC5, 0xE5, 0xCD, 0xCE, 0x0E, 0xE1, 0xCD, 0xDC, 0x0E, 0xCD, 0x10, 0x0F,
    0x06, 0x10, 0xE5, 0x7E, 0xCD, 0xE1, 0x0E, 0x23, 0x10, 0xF9, 0xCD, 0x10, 0x0F, 0xE1, 0x06, 0x10,
    0x7E, 0xCD, 0xF4, 0x0E, 0x23, 0x10, 0xF9, 0xC1, 0x10, 0xDA, 0xC9, 0xCD, 0x13, 0x0F, 0xCD, 0x45,
    0x0E, 0x7E, 0xCD, 0x26, 0x06, 0xCD, 0x13, 0x0F, 0xCD, 0x6F, 0x0E, 0xFE, 0x20, 0x28, 0x0E, 0x06,
    0x00, 0xE5, 0x21, 0x59, 0x07, 0xE3, 0xCD, 0x5C, 0x0E, 0xCD, 0x51, 0x0E, 0x77, 0x23, 0xE5, 0xCD,
    0xCE, 0x0E, 0xE1, 0xCD, 0xDC, 0x0E, 0x18, 0xD9, 0x0E, 0x01, 0xED, 0x78, 0xED, 0x79, 0x0C, 0x0C,
    0x79, 0xFE, 0x15, 0x38, 0xF5, 0xCD, 0x83, 0x0E, 0x18, 0xEE, 0x21, 0x00, 0x00, 0xCD, 0x10, 0x08,
    0xE6, 0x30, 0x28, 0x14, 0x2B, 0x7C, 0xB5, 0x20, 0xF4, 0x21, 0x8F, 0x07, 0xC3, 0xA5, 0x0E, 0x08,
    0x20, 0x54, 0x69, 0x6D, 0x65, 0x6F, 0x75, 0x74, 0x3E, 0x0A, 0xCB, 0xFF, 0xD3, 0x40, 0xCD, 0x10,
    0x08, 0xEE, 0x30, 0xE6, 0x30, 0x20, 0xF7, 0xAF, 0xD3, 0x40, 0xCD, 0x10, 0x08, 0xCB, 0x6F, 0x20,
    0xF9, 0x0E, 0x02, 0x79, 0xD3, 0x40, 0xCB, 0x41, 0x20, 0x0F, 0xDB, 0x00, 0xE6, 0x01, 0x28, 0x09,
    0xDB, 0x01, 0xD3, 0x41, 0xCB, 0xC1, 0x79, 0xD3, 0x40, 0xCD, 0x10, 0x08, 0xCB, 0x67, 0x20, 0x2B,
    0xE6, 0x0F, 0x47, 0xCB, 0xD1, 0xCB, 0x89, 0x79, 0xD3, 0x40, 0xCD, 0x10, 0x08, 0xCB, 0x67, 0x28,
    0xF9, 0xCB, 0x27, 0xCB, 0x27, 0xCB, 0x27, 0xCB, 0x27, 0xB0, 0x47, 0xCB, 0xC9, 0xCB, 0x91, 0x79,
    0xD3, 0x40, 0xDB, 0x00, 0xE6, 0x02, 0x28, 0xFA, 0x78, 0xD3, 0x01, 0xCD, 0x10, 0x08, 0xCB, 0x6F,
    0x28, 0xB4, 0xCB, 0x81, 0x79, 0xD3, 0x40, 0xCD, 0x10, 0x08, 0xCB, 0x6F, 0x20, 0xF9, 0x18, 0xA6,
    0xC5, 0xDB, 0x40, 0x47, 0xDB, 0x40, 0xB8, 0x20, 0xFA, 0xC1, 0xC9, 0xCD, 0x45, 0x0E, 0xE9, 0x6E,
    0x08, 0x4C, 0x25, 0x4B, 0x24, 0x4A, 0x23, 0x49, 0x22, 0x48, 0x21, 0x47, 0x20, 0x46, 0x1F, 0x45,
    0x1E, 0x44, 0x1D, 0x43, 0x1C, 0x42, 0x1B, 0x41, 0x1A, 0x40, 0x19, 0x3F, 0x18, 0x3E, 0x17, 0x3D,
    0x16, 0x3C, 0x15, 0x3B, 0x14, 0x3A, 0x13, 0x39, 0x12, 0x38, 0x11, 0x37, 0x10, 0x36, 0x0F, 0x35,
    0x0E, 0x34, 0x0D, 0x33, 0x0C, 0x32, 0x0B, 0x31, 0x0A, 0x30, 0x09, 0x2F, 0x08, 0x2E, 0x07, 0x2D,
    0x06, 0x2C, 0x05, 0x2B, 0x04, 0x2A, 0x03, 0x29, 0x02, 0x28, 0x01, 0x27, 0x00, 0x26, 0x49, 0x4E,
    0xD5, 0x05, 0x43, 0x49, 0xDF, 0x05, 0x4F, 0x55, 0xEA, 0x05, 0x43, 0x4F, 0xF4, 0x05, 0x46, 0x42,
    0xFF, 0x05, 0x46, 0x49, 0x0F, 0x06, 0x44, 0x44, 0x2D, 0x06, 0x53, 0x44, 0x31, 0x06, 0x54, 0x30,
    0x37, 0x06, 0x53, 0x54, 0x3B, 0x06, 0x48, 0x31, 0x54, 0x06, 0x48, 0x30, 0x58, 0x06, 0x55, 0x53,
    0x5D, 0x06, 0x52, 0x54, 0x74, 0x06, 0x43, 0x52, 0x81, 0x06, 0x52, 0x52, 0xC2, 0x06, 0x57, 0x52,
    0xF1, 0x06, 0x43, 0x57, 0xFE, 0x06, 0x44, 0x4D, 0x0F, 0x07, 0x53, 0x4D, 0x3B, 0x07, 0x42, 0x46,
    0x5E, 0x01, 0x42, 0x53, 0xF7, 0x02, 0x53, 0x49, 0x68, 0x07, 0x53, 0x45, 0x7A, 0x07, 0x47, 0x4F,
    0x1B, 0x08, 0x0A, 0x20, 0x2D, 0x20, 0x69, 0x6E, 0x76, 0x61, 0x6C, 0x69, 0x64, 0x0B, 0x20, 0x2D,
    0x20, 0x64, 0x69, 0x73, 0x6B, 0x20, 0x65, 0x72, 0x72, 0x10, 0x20, 0x2D, 0x20, 0x72, 0x65, 0x73,
    0x74, 0x6F, 0x72, 0x65, 0x2F, 0x73, 0x65, 0x65, 0x6B, 0x20, 0x65, 0x72, 0x72, 0x6F, 0x72, 0x0D,
    0x20, 0x2D, 0x20, 0x72, 0x65, 0x61, 0x64, 0x20, 0x65, 0x72, 0x72, 0x6F, 0x72, 0x08, 0x53, 0x54,
    0x41, 0x54, 0x55, 0x53, 0x20, 0x2D, 0x08, 0x20, 0x50, 0x41, 0x52, 0x41, 0x4D, 0x20, 0x2D, 0x3E,
    0x10, 0x32, 0x39, 0xE5, 0xC9, 0xDB, 0x24, 0x47, 0x21, 0x55, 0x0A, 0xCD, 0x35, 0x09, 0x3A, 0x00,
    0xE4, 0x47, 0x21, 0x5C, 0x0A, 0xCD, 0xD1, 0x0E, 0x78, 0xCD, 0x26, 0x06, 0xC9, 0xFD, 0x21, 0x6E,
    0x09, 0x06, 0x50, 0xC5, 0x3A, 0x20, 0xE4, 0x47, 0xFD, 0x5E, 0x00, 0xFD, 0x23, 0x3E, 0x0D, 0x26,
    0x00, 0x6B, 0xCD, 0xFE, 0x0E, 0xCD, 0xDC, 0x0E, 0x21, 0x00, 0x80, 0xCD, 0x08, 0xE1, 0x30, 0x03,
    0xCD, 0xCE, 0x0E, 0xCD, 0x83, 0x0E, 0xC1, 0x10, 0xDA, 0x18, 0xD2, 0xC9, 0xBE, 0x09, 0x4F, 0x27,
    0x4E, 0x26, 0x4D, 0x25, 0x4C, 0x24, 0x4B, 0x23, 0x4A, 0x22, 0x49, 0x21, 0x48, 0x20, 0x47, 0x1F,
    0x46, 0x1E, 0x45, 0x1D, 0x44, 0x1C, 0x43, 0x1B, 0x42, 0x1A, 0x41, 0x19, 0x40, 0x18, 0x3F, 0x17,
    0x3E, 0x16, 0x3D, 0x15, 0x3C, 0x14, 0x3B, 0x13, 0x3A, 0x12, 0x39, 0x11, 0x38, 0x10, 0x37, 0x0F,
    0x36, 0x0E, 0x35, 0x0D, 0x34, 0x0C, 0x33, 0x0B, 0x32, 0x0A, 0x31, 0x09, 0x30, 0x08, 0x2F, 0x07,
    0x2E, 0x06, 0x2D, 0x05, 0x2C, 0x04, 0x2B, 0x03, 0x2A, 0x02, 0x29, 0x01, 0x28, 0x00, 0x49, 0x4E,
    0xD5, 0x05, 0x43, 0x49, 0xDF, 0x05, 0x4F, 0x55, 0xEA, 0x05, 0x43, 0x4F, 0xF4, 0x05, 0x46, 0x42,
    0xFF, 0x05, 0x46, 0x49, 0x0F, 0x06, 0x44, 0x44, 0x1F, 0x09, 0x54, 0x30, 0x37, 0x06, 0x53, 0x54,
    0x25, 0x09, 0x48, 0x31, 0x54, 0x06, 0x48, 0x30, 0x58, 0x06, 0x55, 0x53, 0x5D, 0x06, 0x52, 0x54,
    0x74, 0x06, 0x43, 0x52, 0x81, 0x06, 0x52, 0x52, 0x3D, 0x09, 0x57, 0x52, 0xF1, 0x06, 0x43, 0x57,
    0xFE, 0x06, 0x44, 0x4D, 0x0F, 0x07, 0x53, 0x4D, 0x3B, 0x07, 0x42, 0x46, 0x5E, 0x01, 0x42, 0x53,
    0xF7, 0x02, 0x53, 0x49, 0x68, 0x07, 0x53, 0x45, 0x7A, 0x07, 0x0A, 0x20, 0x2D, 0x20, 0x69, 0x6E,
    0x76, 0x61, 0x6C, 0x69, 0x64, 0x0B, 0x20, 0x2D, 0x20, 0x64, 0x69, 0x73, 0x6B, 0x20, 0x65, 0x72,
    0x72, 0x10, 0x20, 0x2D, 0x20, 0x72, 0x65, 0x73, 0x74, 0x6F, 0x72, 0x65, 0x2F, 0x73, 0x65, 0x65,
    0x6B, 0x20, 0x65, 0x72, 0x72, 0x6F, 0x72, 0x0D, 0x20, 0x2D, 0x20, 0x72, 0x65, 0x61, 0x64, 0x20,
    0x65, 0x72, 0x72, 0x6F, 0x72, 0x06, 0x41, 0x54, 0x55, 0x53, 0x20, 0x2D, 0x08, 0x20, 0x50, 0x41,
    0x52, 0x41, 0x4D, 0x20, 0x2D, 0x3E, 0x03, 0xD3, 0x00, 0x3E, 0x11, 0xD3, 0x00, 0xFD, 0x21, 0x74,
    0x0A, 0xC3, 0xDC, 0x0B, 0xFD, 0x21, 0x7E, 0x0A, 0x11, 0xF8, 0x0B, 0xC3, 0xAF, 0x0B, 0xDB, 0x01,
    0xE6, 0x7F, 0xFE, 0x41, 0x28, 0x08, 0xFE, 0x30, 0x38, 0xEA, 0xFE, 0x32, 0x30, 0xE6, 0xD9, 0x4F,
    0xD9, 0xFD, 0x21, 0x9B, 0x0A, 0x11, 0x73, 0x0C, 0xC3, 0xAF, 0x0B, 0xDB, 0x01, 0xE6, 0x7F, 0xFE,
    0x41, 0x28, 0x08, 0xFE, 0x30, 0x38, 0xEA, 0xFE, 0x3A, 0x30, 0xE6, 0xD9, 0x6F, 0xD9, 0xD9, 0x06,
    0x30, 0xD9, 0xD9, 0x78, 0xFE, 0x3A, 0xCA, 0x74, 0x0A, 0x7D, 0xFE, 0x41, 0x28, 0x01, 0xB8, 0xD9,
    0xC2, 0x89, 0x0B, 0xFD, 0x21, 0xCD, 0x0A, 0x11, 0x9C, 0x0C, 0xC3, 0xC5, 0x0B, 0xD9, 0x78, 0xD9,
    0xDD, 0x21, 0xD7, 0x0A, 0xC3, 0xD0, 0x0B, 0xD9, 0x78, 0xD9, 0xD3, 0x38, 0x3E, 0x01, 0x32, 0x01,
    0x40, 0x32, 0x02, 0x40, 0x3A, 0x01, 0x40, 0xFE, 0x01, 0x28, 0x11, 0x3A, 0x02, 0x40, 0xFE, 0x01,
    0x28, 0x0A, 0xFD, 0x21, 0x89, 0x0B, 0x11, 0xA4, 0x0C, 0xC3, 0xC5, 0x0B, 0xD9, 0x79, 0xD9, 0xFE,
    0x31, 0x28, 0x1B, 0x0E, 0x00, 0xFD, 0x21, 0x0C, 0x0B, 0xC3, 0x90, 0x0B, 0xFD, 0x21, 0x13, 0x0B,
    0xC3, 0xEE, 0x0B, 0x0C, 0x20, 0xEF, 0xD9, 0x79, 0xD9, 0xFE, 0x41, 0xC2, 0x89, 0x0B, 0x0E, 0x00,
    0xFD, 0x21, 0x27, 0x0B, 0xC3, 0x90, 0x0B, 0xFD, 0x21, 0x35, 0x0B, 0x3E, 0x64, 0xED, 0x47, 0x11,
    0x07, 0x0D, 0xC3, 0xC5, 0x0B, 0x21, 0x00, 0x20, 0x11, 0x00, 0x20, 0x36, 0xFF, 0x7A, 0xB3, 0x28,
    0x23, 0x7A, 0xBC, 0x20, 0x04, 0x7B, 0xBD, 0x28, 0x0C, 0x1A, 0x0E, 0x00, 0x47, 0xB9, 0xFD, 0x21,
    0x55, 0x0B, 0xC2, 0x35, 0x0D, 0x7B, 0xB7, 0x28, 0x07, 0x17, 0x5F, 0x30, 0xE0, 0x14, 0x18, 0xDD,
    0x1E, 0x01, 0x18, 0xD9, 0x46, 0x3E, 0xFF, 0x4F, 0xB8, 0xFD, 0x21, 0x70, 0x0B, 0xC2, 0xA4, 0x0D,
    0xED, 0x57, 0x3D, 0xED, 0x47, 0x3E, 0x64, 0x20, 0x09, 0xED, 0x47, 0xFD, 0x21, 0x82, 0x0B, 0xC3,
    0xEE, 0x0B, 0x36, 0x00, 0x23, 0x7C, 0xB5, 0x20, 0xAF, 0xD9, 0x04, 0x24, 0xD9, 0xC3, 0xB2, 0x0A,
    0x21, 0xFF, 0x1F, 0x23, 0x71, 0x7C, 0xB5, 0x20, 0xFA, 0x21, 0x00, 0x20, 0x7E, 0x47, 0xB9, 0x20,
    0x07, 0x23, 0x7C, 0xB5, 0x20, 0xF6, 0xFD, 0xE9, 0xFD, 0x21, 0x89, 0x0B, 0xC3, 0xA4, 0x0D, 0xDD,
    0x21, 0xB3, 0x0B, 0x1A, 0x13, 0xB7, 0x20, 0x18, 0xDB, 0x00, 0xE6, 0x01, 0x28, 0xFA, 0xDB, 0x01,
    0xD3, 0x01, 0xC3, 0xDC, 0x0B, 0xDD, 0x21, 0xC9, 0x0B, 0x1A, 0x13, 0xB7, 0x20, 0x02, 0xFD, 0xE9,
    0x08, 0xDB, 0x00, 0xE6, 0x02, 0x28, 0xFA, 0x08, 0xD3, 0x01, 0xDD, 0xE9, 0x3E, 0x0D, 0xDD, 0x21,
    0xE4, 0x0B, 0x18, 0xEC, 0x3E, 0x0A, 0xDD, 0x21, 0xEC, 0x0B, 0x18, 0xE4, 0xFD, 0xE9, 0x3E, 0x2E,
    0xDD, 0x21, 0xF6, 0x0B, 0x18, 0xDA, 0xFD, 0xE9, 0x0D, 0x0A, 0x49, 0x42, 0x43, 0x20, 0x4D, 0x49,
    0x44, 0x44, 0x49, 0x2D, 0x43, 0x41, 0x44, 0x45, 0x54, 0x20, 0x4D, 0x65, 0x6D, 0x6F, 0x72, 0x79,
    0x20, 0x54, 0x65, 0x73, 0x74, 0x0D, 0x0A, 0x74, 0x65, 0x73, 0x74, 0x20, 0x23, 0x20, 0x30, 0x20,
    0x2D, 0x20, 0x63, 0x65, 0x6C, 0x6C, 0x20, 0x74, 0x65, 0x73, 0x74, 0x0D, 0x0A, 0x74, 0x65, 0x73,
    0x74, 0x20, 0x23, 0x20, 0x31, 0x20, 0x2D, 0x20, 0x72, 0x6F, 0x77, 0x2F, 0x63, 0x6F, 0x6C, 0x75,
    0x6D, 0x6E, 0x20, 0x73, 0x65, 0x6E, 0x73, 0x69, 0x74, 0x69, 0x76, 0x69, 0x74, 0x79, 0x20, 0x74,
    0x65, 0x73, 0x74, 0x0D, 0x0A, 0x73, 0x65, 0x6C, 0x65, 0x63, 0x74, 0x20, 0x74, 0x65, 0x73, 0x74,
    0x20, 0x28, 0x30, 0x2C, 0x31, 0x20, 0x6F, 0x72, 0x20, 0x41, 0x3C, 0x61, 0x6C, 0x6C, 0x3E, 0x29,
    0x3F, 0x20, 0x00, 0x0D, 0x0A, 0x77, 0x68, 0x69, 0x63, 0x68, 0x20, 0x62, 0x61, 0x6E, 0x6B, 0x20,
    0x74, 0x6F, 0x20, 0x73, 0x65, 0x6C, 0x65, 0x63, 0x74, 0x20, 0x28, 0x30, 0x2D, 0x39, 0x20, 0x6F,
    0x72, 0x20, 0x41, 0x3C, 0x61, 0x6C, 0x6C, 0x3E, 0x29, 0x3F, 0x20, 0x00, 0x0D, 0x0A, 0x42, 0x61,
    0x6E, 0x6B, 0x20, 0x00, 0x20, 0x20, 0x6E, 0x6F, 0x74, 0x20, 0x66, 0x6F, 0x75, 0x6E, 0x64, 0x20,
    0x0D, 0x0A, 0x00, 0x0D, 0x0A, 0x45, 0x52, 0x42, 0x43, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73,
    0x73, 0x3D, 0x00, 0x3B, 0x20, 0x52, 0x65, 0x61, 0x64, 0x3D, 0x00, 0x3B, 0x20, 0x45, 0x78, 0x70,
    0x65, 0x63, 0x74, 0x65, 0x64, 0x3D, 0x00, 0x3B, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x63, 0x65,
    0x6C, 0x6C, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3D, 0x00, 0x3B, 0x20, 0x50, 0x61,
    0x74, 0x74, 0x65, 0x72, 0x6E, 0x3D, 0x00, 0x0D, 0x0A, 0x45, 0x52, 0x54, 0x43, 0x20, 0x61, 0x64,
    0x64, 0x72, 0x65, 0x73, 0x73, 0x3D, 0x00, 0x0D, 0x0A, 0x4E, 0x6F, 0x77, 0x20, 0x65, 0x6E, 0x74,
    0x65, 0x72, 0x69, 0x6E, 0x67, 0x20, 0x52, 0x6F, 0x77, 0x2F, 0x43, 0x6F, 0x6C, 0x75, 0x6D, 0x6E,
    0x20, 0x73, 0x65, 0x6E, 0x73, 0x69, 0x74, 0x69, 0x76, 0x69, 0x74, 0x79, 0x20, 0x74, 0x65, 0x73,
    0x74, 0x73, 0x0D, 0x0A, 0x00, 0xD9, 0xFD, 0xF9, 0xFD, 0x21, 0x42, 0x0D, 0x11, 0xB3, 0x0C, 0xC3,
    0xC5, 0x0B, 0xD9, 0xFD, 0x21, 0x4A, 0x0D, 0xC3, 0xEE, 0x0D, 0xD9, 0xFD, 0x21, 0x55, 0x0D, 0x11,
    0xC3, 0x0C, 0xC3, 0xC5, 0x0B, 0xD9, 0x78, 0xFD, 0x21, 0x5E, 0x0D, 0xC3, 0xDA, 0x0D, 0xD9, 0x11,
    0xCB, 0x0C, 0xFD, 0x21, 0x69, 0x0D, 0xC3, 0xC5, 0x0B, 0xD9, 0x79, 0xFD, 0x21, 0x72, 0x0D, 0xC3,
    0xDA, 0x0D, 0xD9, 0x11, 0xD7, 0x0C, 0xFD, 0x21, 0x7D, 0x0D, 0xC3, 0xC5, 0x0B, 0xD9, 0xEB, 0xFD,
    0x21, 0x86, 0x0D, 0xC3, 0xEE, 0x0D, 0xEB, 0xD9, 0xFD, 0x21, 0x92, 0x0D, 0x11, 0xEC, 0x0C, 0xC3,
    0xC5, 0x0B, 0xD9, 0xFD, 0x21, 0x9B, 0x0D, 0x79, 0xC3, 0xDA, 0x0D, 0xFD, 0x21, 0x00, 0x00, 0xFD,
    0x39, 0xC3, 0xDC, 0x0B, 0xD9, 0xFD, 0xF9, 0xFD, 0x21, 0xB1, 0x0D, 0x11, 0xF7, 0x0C, 0xC3, 0xC5,
    0x0B, 0xD9, 0xEB, 0xFD, 0x21, 0xBA, 0x0D, 0xC3, 0xEE, 0x0D, 0xEB, 0xD9, 0xFD, 0x21, 0xC6, 0x0D,
    0x11, 0xC3, 0x0C, 0xC3, 0xC5, 0x0B, 0xD9, 0x78, 0xFD, 0x21, 0xCF, 0x0D, 0xC3, 0xDA, 0x0D, 0xD9,
    0xFD, 0x21, 0x92, 0x0D, 0x11, 0xCB, 0x0C, 0xC3, 0xC5, 0x0B, 0xD9, 0x47, 0x1F, 0x1F, 0x1F, 0x1F,
    0x21, 0xE5, 0x0D, 0x18, 0x37, 0x78, 0x21, 0xEB, 0x0D, 0x18, 0x31, 0xD9, 0xFD, 0xE9, 0x7A, 0xD9,
    0x1F, 0x1F, 0x1F, 0x1F, 0x21, 0xFA, 0x0D, 0xC3, 0x1C, 0x0E, 0xD9, 0x7A, 0xD9, 0x21, 0x03, 0x0E,
    0xC3, 0x1C, 0x0E, 0xD9, 0x7B, 0xD9, 0x21, 0x10, 0x0E, 0x1F, 0x1F, 0x1F, 0x1F, 0xC3, 0x1C, 0x0E,
    0xD9, 0x7B, 0xD9, 0x21, 0x19, 0x0E, 0xC3, 0x1C, 0x0E, 0xD9, 0xFD, 0xE9, 0xE6, 0x0F, 0xC6, 0x90,
    0x27, 0xCE, 0x40, 0x27, 0xDD, 0x21, 0x2B, 0x0E, 0xC3, 0xD0, 0x0B, 0xE9, 0xCD, 0x13, 0x0F, 0xCD,
    0x6F, 0x0E, 0xFE, 0x0D, 0x28, 0x0C, 0xFE, 0x08, 0x20, 0x04, 0x04, 0x2B, 0x18, 0xF1, 0x77, 0x23,
    0x10, 0xED, 0xAF, 0x77, 0xC9, 0xCD, 0x4E, 0x0E, 0x67, 0xCD, 0x4E, 0x0E, 0x6F, 0xC9, 0xCD, 0x55,
    0x0E, 0x17, 0x17, 0x17, 0x17, 0xC5, 0xE6, 0xF0, 0x47, 0xCD, 0x6F, 0x0E, 0xD6, 0x30, 0x38, 0x42,
    0xFE, 0x0A, 0x38, 0x08, 0xD6, 0x07, 0x38, 0x3A, 0xFE, 0x10, 0x30, 0x36, 0xB0, 0xC1, 0xC9, 0xCD,
    0x78, 0x0E, 0x28, 0xFB, 0xCD, 0x83, 0x0E, 0xC9, 0x3A, 0x1A, 0xE4, 0xC5, 0x4F, 0xED, 0x78, 0xCB,
    0x47, 0xC1, 0xC9, 0xC5, 0xCD, 0x78, 0x0E, 0x28, 0x17, 0x3A, 0x1A, 0xE4, 0x3C, 0x4F, 0xED, 0x78,
    0xE6, 0x7F, 0xFE, 0x61, 0x38, 0x02, 0xCB, 0xAF, 0xFE, 0x1B, 0xCA, 0xA8, 0x0E, 0xCD, 0xFE, 0x0E,
    0xC1, 0xC9, 0x21, 0xD2, 0x08, 0xCD, 0xD1, 0x0E, 0x31, 0xFE, 0xFF, 0xC3, 0xD9, 0x00, 0x11, 0x1B,
    0x0F, 0xFD, 0x21, 0xB8, 0x0E, 0xC3, 0xC5, 0x0B, 0xDB, 0x00, 0xE6, 0x01, 0x28, 0xFA, 0xDB, 0x01,
    0xE6, 0x5F, 0xFE, 0x1B, 0xCA, 0x00, 0x00, 0xFE, 0x4D, 0xCA, 0x65, 0x0A, 0x18, 0xE0, 0x21, 0x3E,
    0x0F, 0xC5, 0x46, 0x23, 0x7E, 0xCD, 0xFE, 0x0E, 0x10, 0xF9, 0xC1, 0xC9, 0x7C, 0xCD, 0xE1, 0x0E,
    0x7D, 0xF5, 0x1F, 0x1F, 0x1F, 0x1F, 0xCD, 0xEA, 0x0E, 0xF1, 0xE6, 0x0F, 0xC6, 0x30, 0xFE, 0x3A,
    0x38, 0x02, 0xC6, 0x07, 0xFE, 0x20, 0x38, 0x04, 0xFE, 0x80, 0x38, 0x02, 0x3E, 0x2E, 0xC5, 0x47,
    0x3A, 0x1A, 0xE4, 0x4F, 0xED, 0x78, 0xCB, 0x4F, 0x28, 0xFA, 0x0C, 0xED, 0x41, 0x78, 0xC1, 0xC9,
    0xCD, 0x13, 0x0F, 0xF5, 0x3E, 0x20, 0xCD, 0xFE, 0x0E, 0xF1, 0xC9, 0x4D, 0x65, 0x6D, 0x6F, 0x72,
    0x79, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20, 0x45, 0x72, 0x72, 0x6F, 0x72, 0x21, 0x21,
    0x21, 0x20, 0x28, 0x45, 0x53, 0x43, 0x20, 0x6F, 0x72, 0x20, 0x4D, 0x29, 0x07, 0x00, 0x03, 0x0D,
    0x0A, 0x2A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
  },
/* IBC MegaStar Boot ROM DMP047 REV A. TMS2764 */
  {
    0xF3, 0xAF, 0xD3, 0x62, 0xDB, 0x14, 0xDB, 0x24, 0xDB, 0x80, 0xAF, 0xD3, 0x40, 0xD3, 0x41, 0x3E,
    0x10, 0xD3, 0x42, 0xD3, 0x47, 0x18, 0x52, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xC3, 0x00, 0x00, 0xC3, 0xF5, 0x0F, 0x16, 0x10, 0xAF, 0x5F, 0x0E, 0x08, 0x7A,
    0x3D, 0x0F, 0x0F, 0x0F, 0xE6, 0x60, 0x6F, 0x7A, 0x3D, 0x0F, 0xE6, 0x06, 0xB5, 0x6F, 0x7B, 0x45,
    0xB7, 0x28, 0x13, 0xFE, 0x04, 0x38, 0x08, 0xCB, 0xC5, 0xD6, 0x04, 0x5F, 0x45, 0x28, 0x07, 0x47,
    0x7D, 0xC6, 0x08, 0x10, 0xFC, 0x47, 0x78, 0xD3, 0x3D, 0x1C, 0x0D, 0x20, 0xE1, 0x15, 0x20, 0xCB,
    0xD3, 0x50, 0x1E, 0x89, 0x06, 0x10, 0x78, 0x3D, 0x0F, 0x0F, 0x0F, 0xE6, 0x60, 0x6F, 0x78, 0x3D,
    0x0F, 0xE6, 0x06, 0xB5, 0xB3, 0xD3, 0x3D, 0x10, 0xED, 0xCB, 0xBF, 0xD3, 0x3D, 0x2E, 0x00, 0xDB,
    0x3C, 0xE6, 0x38, 0x0F, 0x0F, 0x0F, 0xE6, 0x07, 0x28, 0x11, 0xFE, 0x04, 0x38, 0x06, 0xCB, 0xC5,
    0xD6, 0x04, 0x28, 0x07, 0x47, 0x7D, 0xC6, 0x08, 0x10, 0xFC, 0x6F, 0x7D, 0xCB, 0xFF, 0xD3, 0x3D,
    0x06, 0x10, 0x21, 0x07, 0x02, 0x4E, 0x23, 0x3E, 0x03, 0xED, 0x79, 0x3E, 0x11, 0xED, 0x79, 0x0C,
    0xED, 0x50, 0x10, 0xF1, 0x06, 0x08, 0xAF, 0xD3, 0x1E, 0xD3, 0x1F, 0x3C, 0x10, 0xF9, 0x06, 0x08,
    0xAF, 0xD3, 0x1C, 0xD3, 0x1D, 0x3C, 0x10, 0xF9, 0x31, 0x00, 0x00, 0xDB, 0x3C, 0xE6, 0x07, 0xFE,
    0x04, 0x38, 0x37, 0xE6, 0x03, 0x28, 0x0F, 0x31, 0xA0, 0x00, 0x3D, 0x28, 0x09, 0x31, 0x40, 0x01,
    0x3D, 0x28, 0x03, 0x31, 0xE0, 0x01, 0x06, 0x00, 0x0E, 0xC0, 0xD9, 0x06, 0x20, 0xAF, 0xD9, 0xED,
    0x79, 0xCB, 0xE9, 0xED, 0x79, 0xCB, 0xA9, 0x0C, 0xD9, 0x10, 0xF3, 0xD9, 0x3E, 0x08, 0x80, 0x47,
    0x30, 0xE6, 0x06, 0x80, 0x0E, 0xC0, 0x69, 0x60, 0x18, 0x52, 0x21, 0xC1, 0x01, 0x11, 0xE1, 0x01,
    0x06, 0x00, 0x0E, 0xC0, 0xD9, 0x06, 0x20, 0xD9, 0x7E, 0xED, 0x79, 0xED, 0x78, 0xED, 0x78, 0xCB,
    0xE9, 0x1A, 0xED, 0x79, 0xED, 0x78, 0xED, 0x78, 0xCB, 0xA9, 0x0C, 0xD9, 0x10, 0xE9, 0xD9, 0x23,
    0x13, 0x3E, 0x08, 0x80, 0x47, 0x30, 0xDB, 0x31, 0x20, 0x00, 0x06, 0x40, 0xDB, 0x3C, 0x0F, 0x0F,
    0x0F, 0x0F, 0x0F, 0x0F, 0xE6, 0x03, 0x28, 0x0C, 0x06, 0xA0, 0x3D, 0x28, 0x07, 0x06, 0x48, 0x3D,
    0x28, 0x02, 0x06, 0x50, 0x0E, 0xC0, 0x69, 0x60, 0x4D, 0x44, 0x0C, 0x2C, 0xFD, 0x21, 0xA3, 0x01,
    0xC3, 0xAA, 0x02, 0xED, 0x79, 0xCB, 0xE9, 0xFD, 0x21, 0xAE, 0x01, 0xC3, 0xA0, 0x02, 0xED, 0x79,
    0xCB, 0xA9, 0x33, 0x3E, 0x08, 0x80, 0x47, 0x30, 0xE3, 0x0C, 0x79, 0xFE, 0xE0, 0x20, 0xD9, 0x18,
    0x57, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1A, 0x1C,
    0x1E, 0x20, 0x22, 0x24, 0x26, 0x28, 0x2A, 0x2C, 0x2E, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3A, 0x3C,
    0x3E, 0x40, 0x42, 0x44, 0x46, 0x48, 0x4A, 0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10,
    0x12, 0x2C, 0x2E, 0x30, 0x32, 0x34, 0x36, 0xFF, 0xAF, 0xD3, 0x38, 0xDB, 0x3C, 0xE6, 0x07, 0xFE,
    0x04, 0xD2, 0xBC, 0x09, 0x01, 0x00, 0x00, 0x21, 0x00, 0x00, 0xAF, 0x77, 0x23, 0x10, 0xFC, 0x0D,
    0x20, 0xF9, 0x31, 0xFE, 0xFF, 0x3E, 0xE1, 0xED, 0x47, 0x11, 0x7B, 0x03, 0x21, 0xE3, 0x04, 0xAF,
    0xD3, 0x1C, 0xED, 0x52, 0x4D, 0x44, 0x11, 0x00, 0xE1, 0x21, 0x7B, 0x03, 0xED, 0xB0, 0x11, 0x00,
    0xE1, 0x2A, 0x18, 0xE1, 0x19, 0x22, 0x18, 0xE1, 0xDB, 0x24, 0xED, 0x5E, 0xFB, 0x01, 0x11, 0x00,
    0x11, 0x22, 0xE4, 0x21, 0x47, 0x03, 0xED, 0xB0, 0xDB, 0x3C, 0xE6, 0x07, 0x28, 0x0F, 0x3D, 0x28,
    0x1A, 0x3D, 0xCA, 0xE3, 0x04, 0x3D, 0xCA, 0x90, 0x08, 0x3D, 0xCA, 0xBC, 0x09, 0x3E, 0x30, 0xD3,
    0x2A, 0x32, 0x00, 0xE4, 0xDB, 0x24, 0xCB, 0x7F, 0xCA, 0x58, 0x03, 0x21, 0x07, 0x02, 0x7E, 0x23,
    0x32, 0x1A, 0xE4, 0xCD, 0xCA, 0x0F, 0x18, 0xD0, 0x7E, 0x23, 0xFE, 0xFF, 0x28, 0xCA, 0x18, 0xF0,
    0xD9, 0x21, 0x00, 0x00, 0x39, 0x29, 0x7D, 0xD9, 0xFD, 0xE9, 0xD9, 0x21, 0x00, 0x00, 0x39, 0x3E,
    0x80, 0xDD, 0x21, 0xB8, 0x02, 0xC3, 0xC4, 0x02, 0x45, 0xDD, 0x21, 0xC0, 0x02, 0xC3, 0xD6, 0x02,
    0x7D, 0xD9, 0xFD, 0xE9, 0x47, 0x11, 0x10, 0x00, 0x29, 0x7A, 0x17, 0xB8, 0x38, 0x02, 0x90, 0x2C,
    0x57, 0x1D, 0x20, 0xF4, 0xDD, 0xE9, 0x21, 0x01, 0x00, 0x78, 0xB7, 0x28, 0x03, 0x29, 0x10, 0xFD,
    0xDD, 0xE9, 0x21, 0x1D, 0x03, 0xCD, 0x18, 0x10, 0xCD, 0x15, 0x10, 0xCD, 0xCA, 0x0F, 0x28, 0xFB,
    0x11, 0xE8, 0x02, 0xD5, 0x67, 0xCD, 0xB6, 0x0F, 0x6F, 0xE5, 0x06, 0x1A, 0x21, 0x0D, 0x09, 0x56,
    0x23, 0x5E, 0x23, 0xE3, 0xE5, 0xAF, 0xED, 0x52, 0xE1, 0x20, 0x06, 0xE1, 0x5E, 0x23, 0x56, 0xD5,
    0xC9, 0xE3, 0x23, 0x23, 0x10, 0xE9, 0xE1, 0x21, 0x71, 0x09, 0xC3, 0xEC, 0x0F, 0x29, 0x0D, 0x0A,
    0x49, 0x42, 0x43, 0x20, 0x4C, 0x6F, 0x61, 0x64, 0x65, 0x72, 0x20, 0x50, 0x52, 0x4F, 0x4D, 0x20,
    0x2D, 0x20, 0x53, 0x55, 0x50, 0x45, 0x52, 0x2D, 0x43, 0x41, 0x44, 0x45, 0x54, 0x20, 0x20, 0x56,
    0x31, 0x2E, 0x31, 0x20, 0x0D, 0x0A, 0x0A, 0x53, 0x59, 0x53, 0x54, 0x45, 0x4D, 0x20, 0x20, 0x4E,
    0x55, 0x43, 0x4C, 0x45, 0x55, 0x53, 0x20, 0x00, 0xAF, 0xD3, 0x40, 0xD3, 0x44, 0x3E, 0x1A, 0x32,
    0x39, 0xE5, 0xAF, 0x32, 0x15, 0xE4, 0x32, 0x3B, 0xE5, 0xCD, 0x02, 0xE1, 0x06, 0x00, 0x1E, 0x00,
    0x21, 0x00, 0x80, 0xCD, 0x08, 0xE1, 0x38, 0xE0, 0xC3, 0x13, 0x05, 0xAF, 0xC9, 0x18, 0x16, 0x5D,
    0xE2, 0x18, 0x22, 0x18, 0x16, 0xC3, 0xD0, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x5D, 0x01, 0x3E, 0x01, 0x32, 0x1C, 0xE4, 0xC9, 0x16, 0x8C, 0xCB, 0x40, 0x28,
    0x0C, 0xCB, 0xCA, 0x18, 0x08, 0x16, 0xAC, 0xCB, 0x40, 0x28, 0x02, 0xCB, 0xCA, 0x0E, 0x00, 0xE5,
    0x21, 0x16, 0xE4, 0x3A, 0x15, 0xE4, 0xE6, 0x03, 0x85, 0x6F, 0x30, 0x01, 0x24, 0xE3, 0xDD, 0xE1,
    0x3A, 0x15, 0xE4, 0xF6, 0x38, 0xCB, 0x6A, 0x28, 0x02, 0xCB, 0xAF, 0xCB, 0x40, 0x28, 0x02, 0xCB,
    0xD7, 0x47, 0x3A, 0x39, 0xE5, 0xFE, 0x0E, 0x38, 0x0E, 0xAF, 0xBB, 0x20, 0x11, 0x3A, 0x3B, 0xE5,
    0xB7, 0x28, 0x04, 0xCB, 0x50, 0x20, 0x07, 0x78, 0x06, 0x80, 0xCB, 0x9F, 0x18, 0x03, 0x78, 0x06,
    0x00, 0xD3, 0x2A, 0x32, 0x00, 0xE4, 0x78, 0x32, 0x1D, 0xE4, 0xDB, 0x24, 0xCB, 0x47, 0x20, 0xFA,
    0xCB, 0x7F, 0x20, 0x3B, 0x3A, 0x1C, 0xE4, 0x1F, 0x30, 0x13, 0x3E, 0x0C, 0xD3, 0x24, 0x18, 0xFE,
    0xCB, 0x7F, 0x3E, 0x52, 0x20, 0x29, 0xAF, 0x32, 0x1C, 0xE4, 0xDD, 0x77, 0x00, 0xDD, 0x7E, 0x00,
    0xD3, 0x25, 0x3E, 0x0A, 0x3D, 0x20, 0xFD, 0xDB, 0x24, 0xCB, 0x7F, 0x3E, 0x73, 0x20, 0x10, 0x7B,
    0xDD, 0x77, 0x00, 0xD3, 0x27, 0x3E, 0x1C, 0xD3, 0x24, 0x18, 0xFE, 0xCB, 0x7F, 0x3E, 0x53, 0x20,
    0x76, 0xCB, 0x52, 0x20, 0x0C, 0xC5, 0x06, 0xFA, 0xC5, 0x06, 0x19, 0x10, 0xFE, 0xC1, 0x10, 0xF8,
    0xC1, 0x3E, 0x05, 0x32, 0x05, 0xE5, 0xCB, 0x6A, 0x28, 0x0F, 0x3A, 0x00, 0xE4, 0xCB, 0x67, 0xD3,
    0x2A, 0xD3, 0x3E, 0xC5, 0x0E, 0x28, 0xED, 0xB3, 0xC1, 0x3A, 0x00, 0xE4, 0xD3, 0x2A, 0xD3, 0x3E,
    0x3A, 0x3B, 0xE5, 0xFE, 0x09, 0x28, 0x07, 0xFE, 0x05, 0x79, 0x28, 0x04, 0x18, 0x0B, 0x79, 0x87,
    0x87, 0x87, 0x81, 0xD6, 0x1A, 0x30, 0xFC, 0xC6, 0x1A, 0x3C, 0xD3, 0x26, 0x7A, 0xCB, 0x92, 0xD3,
    0x24, 0x18, 0xFE, 0xE6, 0xDC, 0x20, 0x1C, 0xCB, 0x6A, 0x20, 0x0F, 0x3A, 0x00, 0xE4, 0xCB, 0xA7,
    0xD3, 0x2A, 0xD3, 0x3E, 0xC5, 0x0E, 0x28, 0xED, 0xB2, 0xC1, 0x0C, 0x3E, 0x19, 0xB9, 0x30, 0xA1,
    0xAF, 0x18, 0x24, 0xCB, 0x7F, 0x3E, 0x44, 0x20, 0x09, 0x3A, 0x05, 0xE5, 0x3D, 0x32, 0x05, 0xE5,
    0x20, 0x94, 0x20, 0x02, 0x3E, 0x44, 0x47, 0x2A, 0x49, 0xE4, 0x2B, 0x7C, 0xB5, 0x22, 0x49, 0xE4,
    0x37, 0x20, 0x04, 0x78, 0xD3, 0x01, 0x37, 0xC9, 0xE3, 0x23, 0x23, 0xE3, 0xDB, 0x24, 0x32, 0x01,
    0xE4, 0xFB, 0xC9, 0x3E, 0x02, 0x32, 0x34, 0xE4, 0x01, 0x83, 0x01, 0x11, 0x00, 0xE1, 0x21, 0x2E,
    0x05, 0xED, 0xB0, 0xAF, 0x32, 0x1D, 0xE4, 0x3E, 0x0A, 0x32, 0x39, 0xE5, 0xCD, 0x00, 0xE1, 0x38,
    0xE2, 0xCD, 0x02, 0xE1, 0x38, 0xDD, 0x06, 0x00, 0x11, 0x00, 0x00, 0x21, 0x00, 0x80, 0xCD, 0x08,
    0xE1, 0x38, 0xD0, 0x01, 0x00, 0x01, 0x11, 0x00, 0xC0, 0x21, 0x00, 0x80, 0xED, 0xB0, 0x0E, 0x20,
    0x11, 0x20, 0xE5, 0xED, 0xB0, 0x21, 0xD3, 0x5F, 0x22, 0xFE, 0xBF, 0xC3, 0xFE, 0xBF, 0x18, 0x0C,
    0x18, 0x2C, 0x18, 0x07, 0x18, 0x05, 0x18, 0x52, 0xC3, 0xD0, 0x06, 0xC9, 0xAF, 0xD3, 0x40, 0x3E,
    0x10, 0xD3, 0x41, 0x3E, 0x80, 0xD3, 0x42, 0xAF, 0xD3, 0x41, 0x3E, 0x10, 0xD3, 0x42, 0x06, 0x02,
    0x10, 0xFE, 0xDB, 0x47, 0xEE, 0x41, 0xE6, 0x41, 0x28, 0x03, 0x10, 0xF6, 0x37, 0xC9, 0xAF, 0xD3,
    0x40, 0x3E, 0x50, 0xD3, 0x41, 0x3E, 0x80, 0xD3, 0x42, 0xAF, 0xD3, 0x41, 0xD3, 0x42, 0x06, 0x02,
    0x10, 0xFE, 0xDB, 0x47, 0x47, 0xCB, 0x68, 0x20, 0x0F, 0xCB, 0x40, 0x28, 0x0B, 0xEE, 0xD0, 0xE6,
    0xD0, 0x20, 0xEF, 0x78, 0xE6, 0x02, 0x28, 0x01, 0x37, 0xC9, 0xAF, 0xD3, 0x42, 0xD3, 0x40, 0xD3,
    0x41, 0xED, 0x53, 0x02, 0xE5, 0x78, 0x32, 0x01, 0xE5, 0xAF, 0x32, 0x04, 0xE5, 0x0E, 0x44, 0x3A,
    0x01, 0xE5, 0xD3, 0x41, 0xAF, 0xD3, 0x40, 0x3E, 0x40, 0xD3, 0x42, 0x06, 0x04, 0x10, 0xFE, 0xDB,
    0x47, 0xEE, 0x41, 0xE6, 0x41, 0x28, 0x04, 0x10, 0xF6, 0x18, 0x75, 0xAF, 0xD3, 0x42, 0xED, 0x5B,
    0x02, 0xE5, 0x7A, 0xD3, 0x40, 0x7B, 0xD3, 0x41, 0x3E, 0x20, 0xD3, 0x42, 0x06, 0x04, 0x10, 0xFE,
    0xDB, 0x47, 0x47, 0xCB, 0x68, 0x20, 0x71, 0xCB, 0x40, 0x28, 0x55, 0xEE, 0xD0, 0xE6, 0xD0, 0x20,
    0xEF, 0x78, 0xE6, 0x02, 0x20, 0x62, 0x18, 0x02, 0x18, 0xB5, 0xAF, 0x32, 0x06, 0xE5, 0x3E, 0x05,
    0x32, 0x05, 0xE5, 0x3A, 0x39, 0xE5, 0x47, 0xD3, 0x47, 0xAF, 0xD3, 0x40, 0xD3, 0x41, 0xD3, 0x42,
    0xE5, 0x2A, 0x06, 0xE5, 0x26, 0x00, 0x54, 0x5D, 0x29, 0x29, 0x29, 0x19, 0xED, 0x5B, 0x39, 0xE5,
    0x16, 0x00, 0xAF, 0xED, 0x52, 0x30, 0xFC, 0x19, 0x7D, 0xE1, 0xD3, 0x43, 0x32, 0x00, 0xE5, 0x3E,
    0x88, 0xD3, 0x42, 0x06, 0x04, 0x10, 0xFE, 0xDB, 0x47, 0x47, 0xCB, 0x68, 0x20, 0x1A, 0xCB, 0x40,
    0x28, 0x7D, 0xEE, 0xD0, 0xE6, 0xD0, 0x20, 0xEF, 0xCB, 0x50, 0x20, 0x3E, 0x3A, 0x04, 0xE5, 0xB7,
    0x28, 0x1D, 0xD3, 0x47, 0xED, 0x78, 0xFE, 0xAA, 0x20, 0x65, 0xED, 0x50, 0xED, 0x58, 0xED, 0x53,
    0x02, 0xE5, 0xED, 0x78, 0x32, 0x01, 0xE5, 0xAF, 0x32, 0x04, 0xE5, 0x18, 0x8B, 0x18, 0x98, 0xC5,
    0xE5, 0x06, 0x04, 0x21, 0x03, 0xE5, 0xED, 0x78, 0xBE, 0x20, 0x03, 0x2B, 0x10, 0xF8, 0xE1, 0xC1,
    0x20, 0xD0, 0xC5, 0x06, 0x00, 0xED, 0xB2, 0xC1, 0x18, 0x1A, 0x3A, 0x05, 0xE5, 0x3D, 0x32, 0x05,
    0xE5, 0x20, 0xDA, 0x3A, 0x04, 0xE5, 0x3C, 0x28, 0x0B, 0x3A, 0x06, 0xE5, 0xB7, 0x20, 0x20, 0x3E,
    0xFF, 0x32, 0x04, 0xE5, 0x3A, 0x06, 0xE5, 0x3C, 0x32, 0x06, 0xE5, 0x57, 0x3A, 0x39, 0xE5, 0xBA,
    0x28, 0x07, 0x3E, 0x05, 0x32, 0x05, 0xE5, 0x18, 0xB4, 0x3A, 0x05, 0xE5, 0xB7, 0x20, 0x01, 0x37,
    0xC9, 0x21, 0x00, 0x00, 0x1A, 0x13, 0xB7, 0xC8, 0xD6, 0x30, 0xD8, 0xFE, 0x0A, 0x3F, 0xD8, 0xD5,
    0x16, 0x00, 0x5F, 0xD5, 0x29, 0x54, 0x5D, 0x29, 0x29, 0x19, 0xD1, 0x19, 0xD1, 0xD8, 0x18, 0xE4,
    0xC5, 0xD5, 0x47, 0x11, 0x10, 0x00, 0x29, 0x7A, 0x17, 0xB8, 0x38, 0x02, 0x90, 0x2C, 0x57, 0x1D,
    0x20, 0xF4, 0xD1, 0xC1, 0xC9, 0x21, 0x01, 0x00, 0x78, 0xB7, 0x28, 0x03, 0x29, 0x10, 0xFD, 0xC9,
    0xCD, 0x3A, 0x07, 0x4F, 0xED, 0x78, 0xCD, 0x41, 0x07, 0xC9, 0xCD, 0x3A, 0x07, 0x4F, 0xED, 0x78,
    0xCD, 0xCA, 0x0F, 0x18, 0xF9, 0xCD, 0x3A, 0x07, 0x4F, 0xCD, 0x3A, 0x07, 0xED, 0x79, 0xC9, 0xCD,
    0x05, 0x07, 0x47, 0xCD, 0xCA, 0x0F, 0xED, 0x41, 0x18, 0xF9, 0xCD, 0x3A, 0x07, 0x21, 0x00, 0x80,
    0x77, 0x11, 0x01, 0x80, 0x01, 0x00, 0x1A, 0xED, 0xB0, 0xC9, 0x01, 0x00, 0x1A, 0x21, 0x00, 0x80,
    0xAF, 0x77, 0x3C, 0x23, 0x0D, 0x20, 0xFA, 0x10, 0xF8, 0xC9, 0xCD, 0x5A, 0x10, 0xCD, 0x95, 0x0F,
    0xC9, 0xCD, 0x5A, 0x10, 0xCD, 0x28, 0x10, 0xC9, 0x3E, 0x1A, 0x18, 0x02, 0x3E, 0x0D, 0x32, 0x39,
    0xE5, 0xC9, 0xCD, 0x02, 0xE1, 0xC9, 0x3A, 0x01, 0xE4, 0x47, 0x21, 0xAC, 0x09, 0xCD, 0x67, 0x07,
    0x3A, 0x00, 0xE4, 0x47, 0x21, 0xB3, 0x09, 0xCD, 0x18, 0x10, 0x78, 0xCD, 0x41, 0x07, 0xC9, 0x3E,
    0x01, 0x18, 0x01, 0xAF, 0x32, 0x20, 0xE4, 0xC9, 0xCD, 0x5A, 0x10, 0xCD, 0xB6, 0x0F, 0xD6, 0x30,
    0x21, 0x71, 0x09, 0xDA, 0xEC, 0x0F, 0xFE, 0x04, 0xD2, 0xEC, 0x0F, 0x32, 0x15, 0xE4, 0xC9, 0xCD,
    0xAD, 0x07, 0xCD, 0x08, 0xE1, 0x21, 0x7C, 0x09, 0xDA, 0xEC, 0x0F, 0xC9, 0xCD, 0xAD, 0x07, 0xCD,
    0xCA, 0x0F, 0xC5, 0xD5, 0xE5, 0xCD, 0x08, 0xE1, 0xE1, 0xD1, 0xC1, 0x18, 0xF2, 0x06, 0x02, 0x21,
    0x02, 0xE4, 0xE5, 0xD1, 0xCD, 0x73, 0x0F, 0xCD, 0xB1, 0x06, 0x7D, 0xFE, 0x00, 0x21, 0x71, 0x09,
    0xDA, 0xEC, 0x0F, 0xFE, 0x4C, 0xD2, 0xEC, 0x0F, 0x5F, 0x3A, 0x20, 0xE4, 0x47, 0x21, 0x00, 0x80,
    0xC9, 0xFD, 0x21, 0xC0, 0x08, 0x06, 0x4D, 0xC5, 0x3A, 0x20, 0xE4, 0x47, 0xFD, 0x5E, 0x00, 0xFD,
    0x23, 0x3E, 0x0D, 0x26, 0x00, 0x6B, 0xCD, 0x45, 0x10, 0xCD, 0x23, 0x10, 0x21, 0x00, 0x80, 0xCD,
    0x08, 0xE1, 0x30, 0x03, 0xCD, 0x15, 0x10, 0xCD, 0xCA, 0x0F, 0xC1, 0x10, 0xDA, 0x18, 0xD2, 0xC9,
    0xCD, 0xAD, 0x07, 0xCD, 0x06, 0xE1, 0x21, 0x7C, 0x09, 0xDA, 0xEC, 0x0F, 0xC9, 0xCD, 0xAD, 0x07,
    0xCD, 0xCA, 0x0F, 0xC5, 0xD5, 0xE5, 0xCD, 0x06, 0xE1, 0xE1, 0xD1, 0xC1, 0x18, 0xF2, 0xCD, 0x8C,
    0x0F, 0x06, 0x10, 0xC5, 0xE5, 0xCD, 0x15, 0x10, 0xE1, 0xCD, 0x23, 0x10, 0xCD, 0x57, 0x10, 0x06,
    0x10, 0xE5, 0x7E, 0xCD, 0x28, 0x10, 0x23, 0x10, 0xF9, 0xCD, 0x57, 0x10, 0xE1, 0x06, 0x10, 0x7E,
    0xCD, 0x3B, 0x10, 0x23, 0x10, 0xF9, 0xC1, 0x10, 0xDA, 0xC9, 0xCD, 0x5A, 0x10, 0xCD, 0x8C, 0x0F,
    0x7E, 0xCD, 0x41, 0x07, 0xCD, 0x5A, 0x10, 0xCD, 0xB6, 0x0F, 0xFE, 0x20, 0x28, 0x0E, 0x06, 0x00,
    0xE5, 0x21, 0x68, 0x08, 0xE3, 0xCD, 0xA3, 0x0F, 0xCD, 0x98, 0x0F, 0x77, 0x23, 0xE5, 0xCD, 0x15,
    0x10, 0xE1, 0xCD, 0x23, 0x10, 0x18, 0xD9, 0x0E, 0x01, 0xED, 0x78, 0xED, 0x79, 0x0C, 0x0C, 0x79,
    0xFE, 0x15, 0x38, 0xF5, 0x20, 0x04, 0x0E, 0x2D, 0x18, 0xEF, 0xFE, 0x38, 0x38, 0xEB, 0x18, 0xE7,
    0x0E, 0x01, 0x06, 0x08, 0xED, 0x78, 0xED, 0x79, 0x0C, 0x0C, 0x10, 0xF8, 0x18, 0xF2, 0x21, 0x00,
    0x80, 0x0E, 0xC0, 0x06, 0x00, 0xED, 0x78, 0x77, 0x23, 0xCB, 0xE9, 0xED, 0x78, 0x77, 0x23, 0xCB,
    0xA9, 0x3E, 0x08, 0x80, 0x47, 0x30, 0xEE, 0x0C, 0x79, 0xFE, 0xE0, 0x38, 0xE6, 0xC9, 0x0D, 0x09,
    0x4C, 0x25, 0x4B, 0x24, 0x4A, 0x23, 0x49, 0x22, 0x48, 0x21, 0x47, 0x20, 0x46, 0x1F, 0x45, 0x1E,
    0x44, 0x1D, 0x43, 0x1C, 0x42, 0x1B, 0x41, 0x1A, 0x40, 0x19, 0x3F, 0x18, 0x3E, 0x17, 0x3D, 0x16,
    0x3C, 0x15, 0x3B, 0x14, 0x3A, 0x13, 0x39, 0x12, 0x38, 0x11, 0x37, 0x10, 0x36, 0x0F, 0x35, 0x0E,
    0x34, 0x0D, 0x33, 0x0C, 0x32, 0x0B, 0x31, 0x0A, 0x30, 0x09, 0x2F, 0x08, 0x2E, 0x07, 0x2D, 0x06,
    0x2C, 0x05, 0x2B, 0x04, 0x2A, 0x03, 0x29, 0x02, 0x28, 0x01, 0x27, 0x00, 0x26, 0x49, 0x4E, 0xF0,
    0x06, 0x43, 0x49, 0xFA, 0x06, 0x4F, 0x55, 0x05, 0x07, 0x43, 0x4F, 0x0F, 0x07, 0x46, 0x42, 0x1A,
    0x07, 0x46, 0x49, 0x2A, 0x07, 0x44, 0x44, 0x48, 0x07, 0x53, 0x44, 0x4C, 0x07, 0x54, 0x30, 0x52,
    0x07, 0x53, 0x54, 0x56, 0x07, 0x48, 0x31, 0x6F, 0x07, 0x48, 0x30, 0x73, 0x07, 0x55, 0x53, 0x78,
    0x07, 0x52, 0x54, 0x8F, 0x07, 0x43, 0x52, 0x9C, 0x07, 0x52, 0x52, 0xD1, 0x07, 0x57, 0x52, 0x00,
    0x08, 0x43, 0x57, 0x0D, 0x08, 0x44, 0x4D, 0x1E, 0x08, 0x53, 0x4D, 0x4A, 0x08, 0x42, 0x46, 0x58,
    0x03, 0x42, 0x53, 0xE3, 0x04, 0x53, 0x49, 0x77, 0x08, 0x54, 0x52, 0x9E, 0x08, 0x4D, 0x54, 0xBC,
    0x09, 0x0A, 0x20, 0x2D, 0x20, 0x69, 0x6E, 0x76, 0x61, 0x6C, 0x69, 0x64, 0x0B, 0x20, 0x2D, 0x20,
    0x64, 0x69, 0x73, 0x6B, 0x20, 0x65, 0x72, 0x72, 0x10, 0x20, 0x2D, 0x20, 0x72, 0x65, 0x73, 0x74,
    0x6F, 0x72, 0x65, 0x2F, 0x73, 0x65, 0x65, 0x6B, 0x20, 0x65, 0x72, 0x72, 0x6F, 0x72, 0x0D, 0x20,
    0x2D, 0x20, 0x72, 0x65, 0x61, 0x64, 0x20, 0x65, 0x72, 0x72, 0x6F, 0x72, 0x06, 0x41, 0x54, 0x55,
    0x53, 0x20, 0x2D, 0x08, 0x20, 0x50, 0x41, 0x52, 0x41, 0x4D, 0x20, 0x2D, 0x3E, 0x03, 0xD3, 0x00,
    0x3E, 0x11, 0xD3, 0x00, 0xFD, 0x21, 0xCB, 0x09, 0xC3, 0x35, 0x0B, 0xFD, 0x21, 0xD5, 0x09, 0x11,
    0x51, 0x0B, 0xC3, 0x08, 0x0B, 0xDB, 0x01, 0xE6, 0x7F, 0xFE, 0x41, 0x28, 0x08, 0xFE, 0x30, 0x38,
    0xEA, 0xFE, 0x32, 0x30, 0xE6, 0xD9, 0x4F, 0xD9, 0xFD, 0x21, 0xF2, 0x09, 0x11, 0xCC, 0x0B, 0xC3,
    0x08, 0x0B, 0xDB, 0x01, 0xE6, 0x7F, 0xFE, 0x41, 0x28, 0x08, 0xFE, 0x30, 0x38, 0xEA, 0xFE, 0x3A,
    0x30, 0xE6, 0xD9, 0x6F, 0xD9, 0xD9, 0x06, 0x30, 0xD9, 0xD9, 0x78, 0xFE, 0x3A, 0xCA, 0xCB, 0x09,
    0x7D, 0xFE, 0x41, 0x28, 0x01, 0xB8, 0xD9, 0xC2, 0xE2, 0x0A, 0xFD, 0x21, 0x24, 0x0A, 0x11, 0xF5,
    0x0B, 0xC3, 0x1E, 0x0B, 0xD9, 0x78, 0xD9, 0xDD, 0x21, 0x2E, 0x0A, 0xC3, 0x29, 0x0B, 0xD9, 0x78,
    0xD9, 0xE6, 0x0F, 0xD3, 0x38, 0x3E, 0x01, 0x32, 0x01, 0x80, 0x32, 0x02, 0x80, 0x3A, 0x01, 0x80,
    0xFE, 0x01, 0x28, 0x11, 0x3A, 0x02, 0x80, 0xFE, 0x01, 0x28, 0x0A, 0xFD, 0x21, 0xE2, 0x0A, 0x11,
    0xFD, 0x0B, 0xC3, 0x1E, 0x0B, 0xD9, 0x79, 0xD9, 0xFE, 0x31, 0x28, 0x1B, 0x0E, 0x00, 0xFD, 0x21,
    0x65, 0x0A, 0xC3, 0xE9, 0x0A, 0xFD, 0x21, 0x6C, 0x0A, 0xC3, 0x47, 0x0B, 0x0C, 0x20, 0xEF, 0xD9,
    0x79, 0xD9, 0xFE, 0x41, 0xC2, 0xE2, 0x0A, 0x0E, 0x00, 0xFD, 0x21, 0x80, 0x0A, 0xC3, 0xE9, 0x0A,
    0xFD, 0x21, 0x8E, 0x0A, 0x3E, 0x64, 0xED, 0x47, 0x11, 0x60, 0x0C, 0xC3, 0x1E, 0x0B, 0x21, 0x00,
    0x80, 0x11, 0x00, 0x80, 0x36, 0xFF, 0x7A, 0xB3, 0x28, 0x23, 0x7A, 0xBC, 0x20, 0x04, 0x7B, 0xBD,
    0x28, 0x0C, 0x1A, 0x0E, 0x00, 0x47, 0xB9, 0xFD, 0x21, 0xAE, 0x0A, 0xC2, 0x8E, 0x0C, 0x7B, 0xB7,
    0x28, 0x07, 0x17, 0x5F, 0x30, 0xE0, 0x14, 0x18, 0xDD, 0x1E, 0x01, 0x18, 0xD9, 0x46, 0x3E, 0xFF,
    0x4F, 0xB8, 0xFD, 0x21, 0xC9, 0x0A, 0xC2, 0x63, 0x0D, 0xED, 0x57, 0x3D, 0xED, 0x47, 0x3E, 0x64,
    0x20, 0x09, 0xED, 0x47, 0xFD, 0x21, 0xDB, 0x0A, 0xC3, 0x47, 0x0B, 0x36, 0x00, 0x23, 0x7C, 0xB5,
    0x20, 0xAF, 0xD9, 0x04, 0x24, 0xD9, 0xC3, 0x09, 0x0A, 0x21, 0xFF, 0x7F, 0x23, 0x71, 0x7C, 0xB5,
    0x20, 0xFA, 0x21, 0x00, 0x80, 0x7E, 0x47, 0xB9, 0x20, 0x07, 0x23, 0x7C, 0xB5, 0x20, 0xF6, 0xFD,
    0xE9, 0xFD, 0x21, 0xE2, 0x0A, 0xC3, 0x63, 0x0D, 0xDD, 0x21, 0x0C, 0x0B, 0x1A, 0x13, 0xB7, 0x20,
    0x18, 0xDB, 0x00, 0xE6, 0x01, 0x28, 0xFA, 0xDB, 0x01, 0xD3, 0x01, 0xC3, 0x35, 0x0B, 0xDD, 0x21,
    0x22, 0x0B, 0x1A, 0x13, 0xB7, 0x20, 0x02, 0xFD, 0xE9, 0x08, 0xDB, 0x00, 0xE6, 0x02, 0x28, 0xFA,
    0x08, 0xD3, 0x01, 0xDD, 0xE9, 0x3E, 0x0D, 0xDD, 0x21, 0x3D, 0x0B, 0x18, 0xEC, 0x3E, 0x0A, 0xDD,
    0x21, 0x45, 0x0B, 0x18, 0xE4, 0xFD, 0xE9, 0x3E, 0x2E, 0xDD, 0x21, 0x4F, 0x0B, 0x18, 0xDA, 0xFD,
    0xE9, 0x0D, 0x0A, 0x49, 0x42, 0x43, 0x20, 0x4D, 0x49, 0x44, 0x44, 0x49, 0x2D, 0x43, 0x41, 0x44,
    0x45, 0x54, 0x20, 0x4D, 0x65, 0x6D, 0x6F, 0x72, 0x79, 0x20, 0x54, 0x65, 0x73, 0x74, 0x0D, 0x0A,
    0x74, 0x65, 0x73, 0x74, 0x20, 0x23, 0x20, 0x30, 0x20, 0x2D, 0x20, 0x63, 0x65, 0x6C, 0x6C, 0x20,
    0x74, 0x65, 0x73, 0x74, 0x0D, 0x0A, 0x74, 0x65, 0x73, 0x74, 0x20, 0x23, 0x20, 0x31, 0x20, 0x2D,
    0x20, 0x72, 0x6F, 0x77, 0x2F, 0x63, 0x6F, 0x6C, 0x75, 0x6D, 0x6E, 0x20, 0x73, 0x65, 0x6E, 0x73,
    0x69, 0x74, 0x69, 0x76, 0x69, 0x74, 0x79, 0x20, 0x74, 0x65, 0x73, 0x74, 0x0D, 0x0A, 0x73, 0x65,
    0x6C, 0x65, 0x63, 0x74, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x28, 0x30, 0x2C, 0x31, 0x20, 0x6F,
    0x72, 0x20, 0x41, 0x3C, 0x61, 0x6C, 0x6C, 0x3E, 0x29, 0x3F, 0x20, 0x00, 0x0D, 0x0A, 0x77, 0x68,
    0x69, 0x63, 0x68, 0x20, 0x62, 0x61, 0x6E, 0x6B, 0x20, 0x74, 0x6F, 0x20, 0x73, 0x65, 0x6C, 0x65,
    0x63, 0x74, 0x20, 0x28, 0x30, 0x2D, 0x39, 0x20, 0x6F, 0x72, 0x20, 0x41, 0x3C, 0x61, 0x6C, 0x6C,
    0x3E, 0x29, 0x3F, 0x20, 0x00, 0x0D, 0x0A, 0x42, 0x61, 0x6E, 0x6B, 0x20, 0x00, 0x20, 0x20, 0x6E,
    0x6F, 0x74, 0x20, 0x66, 0x6F, 0x75, 0x6E, 0x64, 0x20, 0x0D, 0x0A, 0x00, 0x0D, 0x0A, 0x45, 0x52,
    0x42, 0x43, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3D, 0x00, 0x3B, 0x20, 0x52, 0x65,
    0x61, 0x64, 0x3D, 0x00, 0x3B, 0x20, 0x45, 0x78, 0x70, 0x65, 0x63, 0x74, 0x65, 0x64, 0x3D, 0x00,
    0x3B, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x63, 0x65, 0x6C, 0x6C, 0x20, 0x61, 0x64, 0x64, 0x72,
    0x65, 0x73, 0x73, 0x3D, 0x00, 0x3B, 0x20, 0x50, 0x61, 0x74, 0x74, 0x65, 0x72, 0x6E, 0x3D, 0x00,
    0x0D, 0x0A, 0x45, 0x52, 0x54, 0x43, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3D, 0x00,
    0x0D, 0x0A, 0x4E, 0x6F, 0x77, 0x20, 0x65, 0x6E, 0x74, 0x65, 0x72, 0x69, 0x6E, 0x67, 0x20, 0x52,
    0x6F, 0x77, 0x2F, 0x43, 0x6F, 0x6C, 0x75, 0x6D, 0x6E, 0x20, 0x73, 0x65, 0x6E, 0x73, 0x69, 0x74,
    0x69, 0x76, 0x69, 0x74, 0x79, 0x20, 0x74, 0x65, 0x73, 0x74, 0x73, 0x0D, 0x0A, 0x00, 0xD9, 0xFD,
    0xF9, 0xFD, 0x21, 0x98, 0x0C, 0xC3, 0xF2, 0x0D, 0xFD, 0x21, 0xA2, 0x0C, 0x11, 0x0C, 0x0C, 0xC3,
    0x1E, 0x0B, 0xD9, 0xFD, 0x21, 0xAA, 0x0C, 0xC3, 0xB4, 0x0D, 0xD9, 0xFD, 0x21, 0xB5, 0x0C, 0x11,
    0x1C, 0x0C, 0xC3, 0x1E, 0x0B, 0xD9, 0x78, 0xFD, 0x21, 0xBE, 0x0C, 0xC3, 0xA0, 0x0D, 0xD9, 0x11,
    0x24, 0x0C, 0xFD, 0x21, 0xC9, 0x0C, 0xC3, 0x1E, 0x0B, 0xD9, 0x79, 0xFD, 0x21, 0xD2, 0x0C, 0xC3,
    0xA0, 0x0D, 0xD9, 0x11, 0x30, 0x0C, 0xFD, 0x21, 0xDD, 0x0C, 0xC3, 0x1E, 0x0B, 0xD9, 0xEB, 0xFD,
    0x21, 0xE6, 0x0C, 0xC3, 0xB4, 0x0D, 0xEB, 0xD9, 0xFD, 0x21, 0xF2, 0x0C, 0x11, 0x45, 0x0C, 0xC3,
    0x1E, 0x0B, 0xD9, 0xFD, 0x21, 0xFB, 0x0C, 0x79, 0xC3, 0xA0, 0x0D, 0xD9, 0x06, 0x00, 0x0E, 0xE0,
    0xED, 0x78, 0x57, 0xDD, 0x21, 0x0A, 0x0D, 0xC3, 0x54, 0x0E, 0x0F, 0x0F, 0x0F, 0x0F, 0xE6, 0x0F,
    0xB2, 0x57, 0xDD, 0x21, 0x19, 0x0D, 0xC3, 0x54, 0x0E, 0x59, 0xDD, 0x21, 0x21, 0x0D, 0xC3, 0x54,
    0x0E, 0x0F, 0x0F, 0x0F, 0x0F, 0xE6, 0x0F, 0xB3, 0x5F, 0xDD, 0x21, 0x30, 0x0D, 0xC3, 0x54, 0x0E,
    0x67, 0xDD, 0x21, 0x38, 0x0D, 0xC3, 0x54, 0x0E, 0x0F, 0x0F, 0x0F, 0x0F, 0xE6, 0x0F, 0xB4, 0x67,
    0xDD, 0x21, 0x47, 0x0D, 0xC3, 0x54, 0x0E, 0x6F, 0xDD, 0x21, 0x4F, 0x0D, 0xC3, 0x54, 0x0E, 0x0F,
    0x0F, 0x0F, 0x0F, 0xE6, 0x0F, 0xB5, 0x6F, 0x42, 0x4B, 0xD9, 0xFD, 0x21, 0x00, 0x00, 0xFD, 0x39,
    0xC3, 0x35, 0x0B, 0xD9, 0xFD, 0xF9, 0xFD, 0x21, 0x6D, 0x0D, 0xC3, 0xF2, 0x0D, 0xFD, 0x21, 0x77,
    0x0D, 0x11, 0x50, 0x0C, 0xC3, 0x1E, 0x0B, 0xD9, 0xEB, 0xFD, 0x21, 0x80, 0x0D, 0xC3, 0xB4, 0x0D,
    0xEB, 0xD9, 0xFD, 0x21, 0x8C, 0x0D, 0x11, 0x1C, 0x0C, 0xC3, 0x1E, 0x0B, 0xD9, 0x78, 0xFD, 0x21,
    0x95, 0x0D, 0xC3, 0xA0, 0x0D, 0xD9, 0xFD, 0x21, 0xF2, 0x0C, 0x11, 0x24, 0x0C, 0xC3, 0x1E, 0x0B,
    0xD9, 0x47, 0x1F, 0x1F, 0x1F, 0x1F, 0x21, 0xAB, 0x0D, 0x18, 0x37, 0x78, 0x21, 0xB1, 0x0D, 0x18,
    0x31, 0xD9, 0xFD, 0xE9, 0x7A, 0xD9, 0x1F, 0x1F, 0x1F, 0x1F, 0x21, 0xC0, 0x0D, 0xC3, 0xE2, 0x0D,
    0xD9, 0x7A, 0xD9, 0x21, 0xC9, 0x0D, 0xC3, 0xE2, 0x0D, 0xD9, 0x7B, 0xD9, 0x21, 0xD6, 0x0D, 0x1F,
    0x1F, 0x1F, 0x1F, 0xC3, 0xE2, 0x0D, 0xD9, 0x7B, 0xD9, 0x21, 0xDF, 0x0D, 0xC3, 0xE2, 0x0D, 0xD9,
    0xFD, 0xE9, 0xE6, 0x0F, 0xC6, 0x90, 0x27, 0xCE, 0x40, 0x27, 0xDD, 0x21, 0xF1, 0x0D, 0xC3, 0x29,
    0x0B, 0xE9, 0x50, 0x59, 0x0E, 0xE0, 0x06, 0x00, 0x7A, 0xDD, 0x21, 0x00, 0x0E, 0xC3, 0x4A, 0x0E,
    0x7A, 0x07, 0x07, 0x07, 0x07, 0xDD, 0x21, 0x0C, 0x0E, 0xC3, 0x4A, 0x0E, 0x7B, 0xDD, 0x21, 0x14,
    0x0E, 0xC3, 0x4A, 0x0E, 0x7B, 0x07, 0x07, 0x07, 0x07, 0xDD, 0x21, 0x20, 0x0E, 0xC3, 0x4A, 0x0E,
    0x7C, 0xDD, 0x21, 0x28, 0x0E, 0xC3, 0x4A, 0x0E, 0x7C, 0x07, 0x07, 0x07, 0x07, 0xDD, 0x21, 0x34,
    0x0E, 0xC3, 0x4A, 0x0E, 0x7D, 0xDD, 0x21, 0x3C, 0x0E, 0xC3, 0x4A, 0x0E, 0x7D, 0x07, 0x07, 0x07,
    0x07, 0xDD, 0x21, 0x48, 0x0E, 0xC3, 0x4A, 0x0E, 0xFD, 0xE9, 0xE6, 0xF0, 0xED, 0x79, 0x3E, 0x08,
    0x80, 0x47, 0xDD, 0xE9, 0x3E, 0x08, 0x80, 0x47, 0xED, 0x78, 0xDD, 0xE9, 0xFD, 0x21, 0x66, 0x0E,
    0x11, 0x22, 0x0F, 0xC3, 0x1E, 0x0B, 0x26, 0x55, 0x31, 0x6E, 0x0E, 0xC3, 0x82, 0x0E, 0x31, 0x74,
    0x0E, 0xC3, 0x99, 0x0E, 0x26, 0xAA, 0x31, 0x7C, 0x0E, 0xC3, 0x82, 0x0E, 0x31, 0x00, 0x00, 0xC3,
    0x99, 0x0E, 0x0E, 0xC0, 0x06, 0x00, 0xED, 0x61, 0x3E, 0x08, 0x80, 0x47, 0x30, 0xF8, 0x0C, 0x20,
    0xF3, 0xFD, 0x21, 0x00, 0x00, 0xFD, 0x39, 0xFD, 0xE9, 0x0E, 0xC0, 0x06, 0x00, 0xED, 0x78, 0xBC,
    0x20, 0x11, 0x3E, 0x08, 0x80, 0x47, 0x30, 0xF5, 0x0C, 0x20, 0xF0, 0xFD, 0x21, 0x00, 0x00, 0xFD,
    0x39, 0xFD, 0xE9, 0x7C, 0xD9, 0x57, 0xD9, 0x11, 0x42, 0x0F, 0xFD, 0x21, 0xC1, 0x0E, 0xC3, 0x1E,
    0x0B, 0x79, 0xE6, 0x1F, 0xFD, 0x21, 0xCB, 0x0E, 0xC3, 0xA0, 0x0D, 0x79, 0xFE, 0xE0, 0x11, 0x6B,
    0x0F, 0x38, 0x03, 0x11, 0x63, 0x0F, 0xFD, 0x21, 0xDD, 0x0E, 0xC3, 0x1E, 0x0B, 0x11, 0x4A, 0x0F,
    0xFD, 0x21, 0xE7, 0x0E, 0xC3, 0x1E, 0x0B, 0x78, 0x0F, 0x0F, 0x0F, 0xE6, 0x1F, 0xFD, 0x21, 0xF4,
    0x0E, 0xC3, 0xA0, 0x0D, 0x11, 0x52, 0x0F, 0xFD, 0x21, 0xFE, 0x0E, 0xC3, 0x1E, 0x0B, 0xED, 0x78,
    0xFD, 0x21, 0x07, 0x0F, 0xC3, 0xA0, 0x0D, 0x11, 0x57, 0x0F, 0xFD, 0x21, 0x11, 0x0F, 0xC3, 0x1E,
    0x0B, 0xD9, 0x7A, 0xD9, 0xFD, 0x21, 0x1B, 0x0F, 0xC3, 0xA0, 0x0D, 0xD9, 0x7A, 0xD9, 0x67, 0xC3,
    0x1F, 0x0F, 0x0D, 0x0A, 0x54, 0x72, 0x61, 0x6E, 0x73, 0x6C, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x20,
    0x52, 0x61, 0x6D, 0x20, 0x4D, 0x65, 0x6D, 0x6F, 0x72, 0x79, 0x20, 0x74, 0x65, 0x73, 0x74, 0x0D,
    0x0A, 0x00, 0x0D, 0x0A, 0x62, 0x61, 0x6E, 0x6B, 0x20, 0x00, 0x20, 0x62, 0x6C, 0x6F, 0x63, 0x6B,
    0x20, 0x00, 0x20, 0x69, 0x73, 0x20, 0x00, 0x20, 0x73, 0x68, 0x6F, 0x75, 0x6C, 0x64, 0x20, 0x62,
    0x65, 0x20, 0x00, 0x20, 0x75, 0x70, 0x70, 0x65, 0x72, 0x20, 0x00, 0x20, 0x6C, 0x6F, 0x77, 0x65,
    0x72, 0x20, 0x00, 0xCD, 0x5A, 0x10, 0xCD, 0xB6, 0x0F, 0xFE, 0x0D, 0x28, 0x0C, 0xFE, 0x08, 0x20,
    0x04, 0x04, 0x2B, 0x18, 0xF1, 0x77, 0x23, 0x10, 0xED, 0xAF, 0x77, 0xC9, 0xCD, 0x95, 0x0F, 0x67,
    0xCD, 0x95, 0x0F, 0x6F, 0xC9, 0xCD, 0x9C, 0x0F, 0x17, 0x17, 0x17, 0x17, 0xC5, 0xE6, 0xF0, 0x47,
    0xCD, 0xB6, 0x0F, 0xD6, 0x30, 0x38, 0x42, 0xFE, 0x0A, 0x38, 0x08, 0xD6, 0x07, 0x38, 0x3A, 0xFE,
    0x10, 0x30, 0x36, 0xB0, 0xC1, 0xC9, 0xCD, 0xBF, 0x0F, 0x28, 0xFB, 0xCD, 0xCA, 0x0F, 0xC9, 0x3A,
    0x1A, 0xE4, 0xC5, 0x4F, 0xED, 0x78, 0xCB, 0x47, 0xC1, 0xC9, 0xC5, 0xCD, 0xBF, 0x0F, 0x28, 0x17,
    0x3A, 0x1A, 0xE4, 0x3C, 0x4F, 0xED, 0x78, 0xE6, 0x7F, 0xFE, 0x61, 0x38, 0x02, 0xCB, 0xAF, 0xFE,
    0x1B, 0xCA, 0xEF, 0x0F, 0xCD, 0x45, 0x10, 0xC1, 0xC9, 0x21, 0x71, 0x09, 0xCD, 0x18, 0x10, 0x31,
    0xFE, 0xFF, 0xC3, 0xE2, 0x02, 0x11, 0x62, 0x10, 0xFD, 0x21, 0xFF, 0x0F, 0xC3, 0x1E, 0x0B, 0xDB,
    0x00, 0xE6, 0x01, 0x28, 0xFA, 0xDB, 0x01, 0xE6, 0x5F, 0xFE, 0x1B, 0xCA, 0x00, 0x00, 0xFE, 0x4D,
    0xCA, 0xBC, 0x09, 0x18, 0xE0, 0x21, 0x85, 0x10, 0xC5, 0x46, 0x23, 0x7E, 0xCD, 0x45, 0x10, 0x10,
    0xF9, 0xC1, 0xC9, 0x7C, 0xCD, 0x28, 0x10, 0x7D, 0xF5, 0x1F, 0x1F, 0x1F, 0x1F, 0xCD, 0x31, 0x10,
    0xF1, 0xE6, 0x0F, 0xC6, 0x30, 0xFE, 0x3A, 0x38, 0x02, 0xC6, 0x07, 0xFE, 0x20, 0x38, 0x04, 0xFE,
    0x80, 0x38, 0x02, 0x3E, 0x2E, 0xC5, 0x47, 0x3A, 0x1A, 0xE4, 0x4F, 0xED, 0x78, 0xCB, 0x4F, 0x28,
    0xFA, 0x0C, 0xED, 0x41, 0x78, 0xC1, 0xC9, 0xCD, 0x5A, 0x10, 0xF5, 0x3E, 0x20, 0xCD, 0x45, 0x10,
    0xF1, 0xC9, 0x4D, 0x65, 0x6D, 0x6F, 0x72, 0x79, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20,
    0x45, 0x72, 0x72, 0x6F, 0x72, 0x21, 0x21, 0x21, 0x20, 0x28, 0x45, 0x53, 0x43, 0x20, 0x6F, 0x72,
    0x20, 0x4D, 0x29, 0x07, 0x00, 0x03, 0x0D, 0x0A, 0x2A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
  }
};

typedef struct io_resource_list {
    int32(*routine)(const int32, const int32, const int32);
    uint32 baseaddr;
    uint32 size;
    uint32 resource_type;
    char* name;
} IO_RESOURCE_LIST;

IO_RESOURCE_LIST IBC_MCC_RESOURCES[] = {
    { &ibc_sio,         IBC_SIO,        16, RESOURCE_TYPE_IO, "ibc_sio"},
    { &ibctimer_dev,    0x14,           1,  RESOURCE_TYPE_IO, "ibctimer_dev"},
    { &ibc_fdc_data,    IBC_FDC_DATA,   1,  RESOURCE_TYPE_IO, "ibc_sbd" },
    { &ibc_param_reg,   IBC_PARAM,      1,  RESOURCE_TYPE_IO, "ibc_param_reg" },
    { &ibc_banksel,     IBC_BANKSEL,    1,  RESOURCE_TYPE_IO, "ibc_banksel" },
    { &ibc_banksel,     IBC_DIPSWE,     4,  RESOURCE_TYPE_IO, "ibc_banksel" },
    { &ibc_cart,        IBC_CART,       4,  RESOURCE_TYPE_IO, "ibc_cart" },
    { &ibc_reel,        IBC_REEL,       4,  RESOURCE_TYPE_IO, "ibc_reel" },
    { &ibc_dev31,       IBC_DEV31,      4,  RESOURCE_TYPE_IO, "ibc_dev31" },
    { &ibc_rtc,         IBC_RTC,        16, RESOURCE_TYPE_IO, "ibc_rtc" },
    { &ibc_pio,         IBC_PIO,        4,  RESOURCE_TYPE_IO, "ibc_pio" },
    { NULL }
};

IO_RESOURCE_LIST IBC_SCC_RESOURCES[] = {
    { &ibc_sio,         IBC_SIO,        16, RESOURCE_TYPE_IO, "ibc_sio"},
    { &ibctimer_dev,    0x14,           1,  RESOURCE_TYPE_IO, "ibctimer_dev"},
    { &ibc_fdc_data,    IBC_FDC_DATA,   1,  RESOURCE_TYPE_IO, "ibc_sbd" },
    { &ibc_param_reg,   IBC_PARAM,      1,  RESOURCE_TYPE_IO, "ibc_param_reg" },
    { &ibc_banksel,     IBC_BANKSEL,    1,  RESOURCE_TYPE_IO, "ibc_banksel" },
    { &ibc_banksel,     IBC_DIPSWE,     4,  RESOURCE_TYPE_IO, "ibc_banksel" },
    { &ibc_cart,        IBC_CART,       4,  RESOURCE_TYPE_IO, "ibc_cart" },
    { &ibc_reel,        IBC_REEL,       4,  RESOURCE_TYPE_IO, "ibc_reel" },
    { &ibc_dev31,       IBC_DEV31,      4,  RESOURCE_TYPE_IO, "ibc_dev31" },
    { &ibc_rtc,         IBC_RTC,        16, RESOURCE_TYPE_IO, "ibc_rtc" },
    { &ibc_pio,         IBC_PIO,        4,  RESOURCE_TYPE_IO, "ibc_pio" },
    { &ibc_sc_dtr,      0x1c,           4,  RESOURCE_TYPE_IO, "ibc_sc_1C-1F"   },
    { &ibcrtctimer_dev, 0x20,           1,  RESOURCE_TYPE_IO, "ibcrtctimer_dev"},
    { &ibc_sio,         0x2c,           12, RESOURCE_TYPE_IO, "ibc_sc_sio"     }, /* UARTS 11-16 */
    { &ibc_sc_baud,     0x3d,           1,  RESOURCE_TYPE_IO, "ibc_sc_3D"      },
    { &ibc_pio,         0x4c,           4,  RESOURCE_TYPE_IO, "ibc_sc_pio"     },
    { &ibc_unhandled,   0x50,           12, RESOURCE_TYPE_IO, "ibc_sc_50-5B"   },
    { &ibc_banksel,     0x5c,           4,  RESOURCE_TYPE_IO, "ibc_sc_banksel" },
    { &ibc_unhandled,   0x70,           1,  RESOURCE_TYPE_IO, "ibc_sc_70"      },
    { &ibc_unhandled,   0xC0,           16, RESOURCE_TYPE_IO, "ibc_sc_C0-CF"   },
    { &ibc_sc_cache,    0xE0,           4,  RESOURCE_TYPE_IO, "ibc_sc_cache"   },
    { &ibc_unhandled,   0xE4,           12, RESOURCE_TYPE_IO, "ibc_sc_E4-EF"   },
    { &ibc_unhandled,   0xF0,           16, RESOURCE_TYPE_IO, "ibc_sc_F0-FF"   },   /* Take over the SIMH Pseudo device ports as the boot ROM accesses them. */
    { NULL }
};

static t_stat ibc_set_model(UNIT* uptr, int32 value, CONST char* cptr, void* desc) {
    IO_RESOURCE_LIST* resources;

    if (value == ibc_info->model) {
        sim_printf("IBC model unchanged\n");
        return SCPE_OK;
    }

    if (value > ibc_scc) {
        return SCPE_ARG;
    }

    /* Unmap current model I/O ports */
    resources = (ibc_info->model == ibc_mcc) ? IBC_MCC_RESOURCES : IBC_SCC_RESOURCES;
    while (resources->routine != NULL) {
        sim_debug(VERBOSE_MSG, &ibc_dev, "Unmapping I/O at 0x%02x-0x%02x: %s\n",
            resources->baseaddr, resources->baseaddr + resources->size - 1, resources->name);
        sim_map_resource(resources->baseaddr, resources->size, resources->resource_type, resources->routine, resources->name, TRUE);
        resources++;
    }

    ibc_info->model = value;

    /* Map new model I/O ports */
    resources = (ibc_info->model == ibc_mcc) ? IBC_MCC_RESOURCES : IBC_SCC_RESOURCES;
    while (resources->routine != NULL) {
        sim_debug(VERBOSE_MSG, &ibc_dev, "Mapping I/O at 0x%02x-0x%02x: %s\n",
            resources->baseaddr, resources->baseaddr + resources->size - 1, resources->name);
        sim_map_resource(resources->baseaddr, resources->size, resources->resource_type, resources->routine, resources->name, FALSE);
        resources++;
    }

    return SCPE_OK;
}

/* Reset routine */
static t_stat ibc_reset(DEVICE* dptr)
{
    PNP_INFO* pnp = (PNP_INFO*)dptr->ctxt;
    int i;
    uint8 unmap = dptr->flags & DEV_DIS;
    IO_RESOURCE_LIST* resources = (ibc_info->model == ibc_mcc) ? IBC_MCC_RESOURCES : IBC_SCC_RESOURCES;

    /* Connect IBC ROM at base address */
    sim_map_resource(pnp->mem_base, pnp->mem_size, RESOURCE_TYPE_MEMORY, &ibc_rom, "ibc_rom", unmap);

    while (resources->routine != NULL) {
        sim_debug(VERBOSE_MSG, &ibc_dev, "%s I/O at 0x%02x-0x%02x: %s\n",
            unmap ? "Unmapping" : "Mapping", resources->baseaddr, resources->baseaddr + resources->size - 1, resources->name);
        sim_map_resource(resources->baseaddr, resources->size, resources->resource_type, resources->routine, resources->name, unmap);
        resources++;
    }

    for (i = 0; i < IBC_MAX_UNITS; i++) {
        ibc_unit[i].u4 = i;
    }

    /* Reset FDC PARAM register */
    ibc_info->param = 0;
    ibc_info->rom_disabled = FALSE;
    ibc_info->cache_wbase = 0;
    ibc_info->cache_rbase = 0;
    ibc_info->cache_index = 0;

    if (unmap) {
        wd179x_infop->intenable = 0;
        wd179x_connect_external_fifo(0, NULL);
    }
    else {
        wd179x_connect_external_fifo(FDC_FIFO_LEN, ibc_info->fdc_fifo);
        ibc_info->fdc_fifo_rd_index = 0;
        wd179x_infop->intenable = 1;
        wd179x_infop->intvector = 2;
    }

    /* If memory was allocated for the CACHE, free it. */
    if (ibc_info->cache != NULL) {
        sim_debug(CACHE_MSG, &ibc_dev, "Freeing CACHE.\n");
        free(ibc_info->cache);
        ibc_info->cache = NULL;
    }

    return SCPE_OK;
}

static t_stat ibc_boot(int32 unitno, DEVICE *dptr)
{
    if ((dptr == NULL) || (unitno > 0)) {
        return SCPE_IERR;
    }

    sim_debug(VERBOSE_MSG, &ibc_dev, "Booting IBC Controller\n");

    /* Set the PC to 0000H, and go. */
    *((int32 *) sim_PC->loc) = 0000;
    return SCPE_OK;
}

/* Attach routine */
static t_stat ibc_attach(UNIT *uptr, CONST char *cptr)
{
    t_stat r = SCPE_IERR;

    if (uptr->u4 < 4) {
        r = wd179x_attach(uptr, cptr);
    }

    return r;
}

/* Detach routine */
static t_stat ibc_detach(UNIT *uptr)
{
    t_stat r = SCPE_IERR;

    if (uptr->u4 < 4) {
    r = wd179x_detach(uptr);
    }

    return r;
}

static int32 ibc_rom(const int32 Addr, const int32 write, const int32 data)
{
    if(write) {
        ibc_ram[Addr & IBC_ROM_ADDR_MASK] = (uint8)data;
        return 0;
    } else {
        if(ibc_info->rom_disabled == FALSE) {
            return(ibc_rom_data[ibc_info->model][Addr & IBC_ROM_ADDR_MASK]);
        } else {
            return(ibc_ram[Addr & IBC_ROM_ADDR_MASK]);
        }
    }
}

/* IBC Cadet SIO ports */
static int32 ibc_sio(const int32 port, const int32 io, const int32 data)
{
    int32 result = 0xFF;
    int32 uart = port >> 1;

    if (port >=0x2c && port <= 0x37) {
        uart -= 12;
    }

    if (io) {    /* Write */
        if (port & 1) { /* Data port */
            result = sio0d(port, 1, data);
            sim_debug(UART_MSG, &ibc_dev, ADDRESS_FORMAT " WR UART[%d] DATA=0x%02x\n", PCX, uart, data);
        }
        else { /* Status port */
            result = sio0s(port, 1, data);
            sim_debug(UART_MSG, &ibc_dev, ADDRESS_FORMAT " WR UART[%d] STAT=0x%02x\n", PCX, uart, data);
        }
    }
    else {    /* Read */
        if (port & 1) { /* Data port */
            result = sio0d(port, 0, 0);
            sim_debug(UART_MSG, &ibc_dev, ADDRESS_FORMAT " RD UART[%d] DATA=0x%02x\n", PCX, uart, result);
        }
        else { /* Status port */
            result = sio0s(port, 0, 0);
            sim_debug(UART_MSG, &ibc_dev, ADDRESS_FORMAT " RD UART[%d] STAT=0x%02x\n", PCX, uart, result);
        }
    }
    return result;
}

/* Disk Control/Flags Register, 0x2a */
static int32 ibc_param_reg(const int32 port, const int32 io, const int32 data)
{
    int32 result = 0;
    if(io) { /* I/O Write */

        ibc_info->param = (uint8)data;
        /* Disk drive select bits 1:0 */
        wd179x_infop->sel_drive = data & 0x03;

        if (data & IBC_PARAM_DDENS) {
            wd179x_infop->ddens = 1;
        }
        else {
            wd179x_infop->ddens = 0;
        }

        sim_debug(DRIVE_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                  " WR CTRL 0x%02x: sel_drive=%d, drivetype=%d, head_sel=%d, dens=%d, aw=%d\n",
                  PCX, port, wd179x_infop->sel_drive,
                  wd179x_infop->drivetype, ibc_info->head_sel,
                  wd179x_infop->ddens, ibc_info->autowait);
    } else { /* I/O Read */
        result = wd179x_infop->drq ? 0xFF : 0;
        if (wd179x_infop->intrq)
            result &= 0x7F;

        sim_debug(DRIVE_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
            " RD DISK STATUS: sel_drive=%d: Status=0x%02x\n", PCX, wd179x_infop->sel_drive, result);
    }

    return result;
}

/* DIP Switch E (0x3c) / ROM Control (0x3f) */
static int32 ibc_banksel(const int32 port, const int32 io, const int32 data)
{
    int32 result = 0xFF;

    if (io) {    /* Write */
        switch(port) {
        case IBC_BANKSEL:
        case IBC_SCC_BANKSEL:
            if (data < 16) {
                setBankSelect(data);
                sim_debug(BANK_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                    " WR Bank Select Port: 0x%02x=0x%02x\n", PCX, port, data);
            }
            else {
                sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                    " Invalid RAM bank 0x%02x\n", PCX, data);
            }
            break;
        case IBC_ROM_CTRL:
        case IBC_SCC_ROM_CTRL:
            sim_debug(BANK_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " WR ROM Ctrl Port: 0x%02x=0x%02x (Disable ROM.)\n", PCX, port, data);
            ibc_info->rom_disabled = TRUE;
            break;
        case IBC_FIFO_CTRL:
        case IBC_SCC_FIFO_CTRL:
            sim_debug(DRIVE_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " WR FDC FIFO Ctrl Port: 0x%02x=0x%02x\n", PCX, port, data);
            if (data & 0x10) { /* FIFO Reset */
                wd179x_reset_external_fifo();
                ibc_info->fdc_fifo_rd_index = 0;
            }
            break;
        default:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                      " WR Unhandled Port: 0x%02x=0x%02x\n", PCX, port, data);
            break;
        }
    } else {    /* Read */
        switch(port) {
        case IBC_DIPSWE:
            /* These are 8-position DIP switch at location E on the IBC CPU board.
             *
             * 0xff: Boot from floppy (All off.)
             * Switch 1 - 0xfe: Boot into ROM monitor (Switch 1 on.)
             * Switch 2 - 0xfd: Boot from hard disk (Switch 2 on.)
             * Switch 7 - 0xbe: OFF = Use FDC Interrupts, ON = Poll FDC instead.
             */
            result = ibc_info->dipsw_E;

            if (ibc_info->model == ibc_mcc) {
                sim_debug(DIPSW_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                    " RD DIPSW E=0x%02x\n", PCX, result);
            }
            else {
                sim_debug(DIPSW_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                    " Boot mode: %d, baud=%d, Bank %s, %s Floppy\n", PCX, (result & 0x07), (result & 0x38) >> 3, (result & 0x40) ? "Base 16K / Bank 48K" : "Base 40K / Bank 24K", (result & 0x80) ? "1.6MB" : "1MB");
            }
            break;
        case IBC_ROM_CTRL:
        case 0x5F:
            sim_debug(BANK_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " RD ROM Ctrl Port: 0x%02x=0x%02x (Enable ROM.)\n", PCX, port, result);
            ibc_info->rom_disabled = FALSE;
            break;
        default:
            sim_debug(SBD_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                      " RD attempt from write-only 0x%02x=0x%02x\n", PCX, port, result);
            break;
        }
    }
    return result;
}

/* IBC FDC Data Port (0x28) */
static int32 ibc_fdc_data(const int32 port, const int32 io, const int32 data)
{
    uint8 fifodata = 0xFF;

    if (io) { /* Write */
        ibc_info->fdc_fifo[ibc_info->fdc_fifo_rd_index] = (uint8)data;
        ibc_info->fdc_fifo_rd_index++;
        ibc_info->fdc_fifo_rd_index &= FDC_FIFO_MASK;
    } else { /* Read */
        fifodata = ibc_info->fdc_fifo[ibc_info->fdc_fifo_rd_index];
        ibc_info->fdc_fifo_rd_index++;
        ibc_info->fdc_fifo_rd_index &= FDC_FIFO_MASK;
    }

    sim_debug(FIFO_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
        " %s FIFO: 0x%02x=%02x\n", PCX, io ? "WR" : "RD", port, fifodata);

    return (fifodata);
}

static struct tm currentTime;

/* IBC MCC RTC (0x70-0x7F)
 *
 * Uses the National Semiconductor MM58174A
 * Microprocessor-Compatible Real-Time Clock
 */
static int32 ibc_rtc(const int32 port, const int32 io, const int32 data)
{
    int32 result;
    time_t now;

    if (io) {    /* Write */
        switch (port) {
        default:
            sim_debug(RTC_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " WR Unhandled RTC Port: 0x%02x=0x%02x\n", PCX, port, data);
            break;
        }
        result = 0;
    }
    else {    /* Read */
        sim_get_time(&now);
        now += 0; // ss1_rtc[0].clockDelta;
        currentTime = *localtime(&now);

        switch (port & 0x0F) {
        case 1:
            result = 0;
            break;
        case 2:
            result = currentTime.tm_sec % 10;
            break;
        case 3:
            result = currentTime.tm_sec / 10;
            break;
        case 4:
            result = currentTime.tm_min % 10;
            break;
        case 5:
            result = currentTime.tm_min / 10;
            break;
        case 6:
            result = currentTime.tm_hour % 10;
            break;
        case 7:
            result = currentTime.tm_hour / 10;
            break;
        case 8:
            result = currentTime.tm_mday % 10;
            break;
        case 9:
            result = currentTime.tm_mday / 10;
            break;
        case 10:
            result = currentTime.tm_wday;
            break;
        case 11:
            result = (currentTime.tm_mon + 1) % 10;
            break;
        case 12:
            result = (currentTime.tm_mon + 1) / 10;
            break;
        default:
            result = 0xFF;
            break;
        }
    }
    return result;
}

/* IBC Cartridge Tape Controller (0x60-0x63) */
static int32 ibc_cart(const int32 port, const int32 io, const int32 data)
{
    int32 result;
    if (io) {    /* Write */
        switch (port) {
        case IBC_CART:
        case IBC_CART + 1:
        case IBC_CART + 2:
        case IBC_CART + 3:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " WR CART Port: 0x%02x=0x%02x\n", PCX, port, data);
            break;
        default:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " WR Unhandled CART Port: 0x%02x=0x%02x\n", PCX, port, data);
            break;
        }
        result = 0;
    }
    else {    /* Read */
        result = 0xFF;
        switch (port) {
        case IBC_CART:
        case IBC_CART + 1:
        case IBC_CART + 2:
        case IBC_CART + 3:
            sim_debug(DRIVE_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " RD CART Port: 0x%02x=0x%02x\n", PCX, port, result);
            break;
        default:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " RD Unhandled SBD CART: 0x%02x=0x%02x\n", PCX, port, result);
            break;
        }
    }
    return result;
}

/* IBC Reel to Reel Tape Controller (0x64-0x67) */
static int32 ibc_reel(const int32 port, const int32 io, const int32 data)
{
    int32 result;
    if (io) {    /* Write */
        switch (port) {
        case IBC_REEL:
        case IBC_REEL + 1:
        case IBC_REEL + 2:
        case IBC_REEL + 3:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " WR REEL Port: 0x%02x=0x%02x\n", PCX, port, data);
            break;
        default:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " WR Unhandled REEL Port: 0x%02x=0x%02x\n", PCX, port, data);
            break;
        }
        result = 0;
    }
    else {    /* Read */
        result = 0xFF;
        switch (port) {
        case IBC_REEL:
        case IBC_REEL + 1:
        case IBC_REEL + 2:
        case IBC_REEL + 3:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " RD REEL Port: 0x%02x=0x%02x\n", PCX, port, result);
            break;
        default:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " RD Unhandled SBD CART: 0x%02x=0x%02x\n", PCX, port, result);
            break;
        }
    }
    return result;
}

/* IBC DEV31 (Unknown) (0x80-0x83) */
static int32 ibc_dev31(const int32 port, const int32 io, const int32 data)
{
    int32 result;
    if (io) {    /* Write */
        switch (port) {
        case IBC_DEV31:
        case IBC_DEV31 + 1:
        case IBC_DEV31 + 2:
        case IBC_DEV31 + 3:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " WR DEV31 Port: 0x%02x=0x%02x\n", PCX, port, data);
            break;
        default:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " WR Unhandled DEV31 Port: 0x%02x=0x%02x\n", PCX, port, data);
            break;
        }
        result = 0;
    }
    else {    /* Read */
        result = 0xFF;
        switch (port) {
        case IBC_DEV31:
        case IBC_DEV31 + 1:
        case IBC_DEV31 + 2:
        case IBC_DEV31 + 3:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " RD DEV31 Port: 0x%02x=0x%02x\n", PCX, port, result);
            break;
        default:
            sim_debug(ERROR_MSG, &ibc_dev, DEV_NAME ": " ADDRESS_FORMAT
                " RD Unhandled SBD CART: 0x%02x=0x%02x\n", PCX, port, result);
            break;
        }
    }
    return result;
}

/* IBC Cadet PIO port */
static int32 ibc_pio(const int32 port, const int32 io, const int32 data)
{
    static int32 result = 0xFF;
    if (io) { /* Write */
        switch (port & 0x03) {
        case 0x00:
            sim_debug(PIO_MSG, &ibc_dev, ADDRESS_FORMAT
                " WR 0x%02x PIOA DATA=0x%02x\n", PCX, port, data);
            break;
        case 0x01:
            sim_debug(PIO_MSG, &ibc_dev, ADDRESS_FORMAT
                " WR 0x%02x PIOB DATA=0x%02x\n", PCX, port, data);
            break;
        case 0x02:
            sim_debug(PIO_MSG, &ibc_dev, ADDRESS_FORMAT
                " WR 0x%02x PIOC CTRL=0x%02x\n", PCX, port, data);
            break;
        default:
            sim_debug(ERROR_MSG, &ibc_dev, ADDRESS_FORMAT
                " WR Unhandled Port: 0x%02x=0x%02x\n", PCX, port, data);
            break;
        }
    }
    else { /* Read */
        result = 0x00;
        switch (port & 0x03) {
        case 0x00:
            sim_debug(PIO_MSG, &ibc_dev, ADDRESS_FORMAT
                " RD %02x PIOA DATA=0x%02x\n", PCX, port, result);
            break;
        default:
            sim_debug(ERROR_MSG, &ibc_dev, ADDRESS_FORMAT
                " RD Unhandled Port: 0x%02x=0x%02x\n", PCX, port, result);
            break;
        }
    }
    return result;
}

/* IBC Cadet CACHE */
static int32 ibc_sc_cache(const int32 port, const int32 io, const int32 data)
{
    static int32 result = 0xFF;

    /* The first time the CACHE is accessed, allocate memory for the CACHE */
    if (ibc_info->cache == NULL) {
        ibc_info->cache = calloc(IBC_CACHE_SIZE, 1);
        if (ibc_info->cache != NULL) {
            sim_debug(CACHE_MSG, &ibc_dev, ADDRESS_FORMAT
                " Allocated %d bytes for CACHE\n", PCX, IBC_CACHE_SIZE);
        }
        else {
            return sim_messagef(SCPE_MEM, "Cannot allocate %d bytes for CACHE.\n", IBC_CACHE_SIZE);
        }
    }

    if (io) { /* Write */
        switch (port & 0x03) {
        case IBC_CACHE_BASE_H:
            /* Set CACHE Write Base */
            ibc_info->cache_wbase = ((ibc_info->cache_wbase) & 0xFF00) | (data << 16);
            ibc_info->cache_wbase &= IBC_CACHE_MASK;

            /* Set CACHE Read Base */
            ibc_info->cache_rbase = ibc_info->cache_wbase;

            /* Reset CACHE Index */
            ibc_info->cache_index = 0;
            sim_debug(CACHE_MSG, &ibc_dev, ADDRESS_FORMAT
                " WR CACHE Port: 0x%02x=0x%02x, wbase=0x%06x, rbase=0x%06x\n", PCX, port, data, ibc_info->cache_wbase, ibc_info->cache_rbase);
            break;
        case IBC_CACHE_BASE_L_WR:
            ibc_info->cache_wbase = ((ibc_info->cache_wbase) & 0xFF0000) | (data << 8);
            ibc_info->cache_wbase &= IBC_CACHE_MASK;

            ibc_info->cache_index = 0;
            sim_debug(CACHE_MSG, &ibc_dev, ADDRESS_FORMAT
                " WR CACHE Port: 0x%02x=0x%02x, wbase=0x%06x\n", PCX, port, data, ibc_info->cache_wbase);
            break;
        case IBC_CACHE_BASE_L_RD:
            ibc_info->cache_rbase = ((ibc_info->cache_rbase) & 0xFF0000) | (data << 8);
            ibc_info->cache_rbase &= IBC_CACHE_MASK;
            ibc_info->cache_index = 0;
            sim_debug(CACHE_MSG, &ibc_dev, ADDRESS_FORMAT
                " WR CACHE Port: 0x%02x=0x%02x, wbase=0x%06x\n", PCX, port, data, ibc_info->cache_rbase);
            break;
        case IBC_CACHE_DATA:
            sim_debug(CACHE_MSG, &ibc_dev, ADDRESS_FORMAT
                " WR CACHE Data: 0x%02x=0x%02x, wbase=0x%06x\n", PCX, port, data, ibc_info->cache_rbase);
            ibc_info->cache[ibc_info->cache_wbase + ibc_info->cache_index] = (uint8)data;
            ibc_info->cache_index++;
            break;
        }
    }
    else { /* Read */
        switch (port & 0x03) {
        case IBC_CACHE_DATA:
            result = ibc_info->cache[ibc_info->cache_rbase + ibc_info->cache_index];
            sim_debug(CACHE_MSG, &ibc_dev, ADDRESS_FORMAT
                " RD CACHE Data: 0x%02x=0x%02x, rbase=0x%06x, index=0x%02x\n", PCX, port, result, ibc_info->cache_rbase, ibc_info->cache_index);
            ibc_info->cache_index++;
            break;
        default:
            sim_debug(CACHE_MSG, &ibc_dev, ADDRESS_FORMAT
                " RD CACHE Port: 0x%02x=0x%02x, rbase=0x%06x, index=0x%02x\n", PCX, port, result, ibc_info->cache_rbase, ibc_info->cache_index);
            break;
        }
    }
    return result;
}

/* IBC Super Cadet Baud Rate Control port */
static int32 ibc_sc_baud(const int32 port, const int32 io, const int32 data)
{
    uint8 sioport = ((data & 0x60) >> 5) | ((data & 0x06) << 1);
    uint8 baudrate = ((data & 0x18) >> 3) | ((data & 0x1) << 2);
    uint8 baudset = ((data & 0x80) >> 7);
    static int32 result = 0xFF;

    if (io) { /* Write */
        sim_debug(UART_MSG, &ibc_dev, ADDRESS_FORMAT
            " WR Baud Rate Port 0x%02x=0x%02x: SIO: %d, baud: 0x%x, set=%d, current_baud=%d, unlock=%d\n", PCX, port, data, sioport, baudrate, baudset, current_baud[sioport], baud_unlock[sioport]);

        if (baudset == 0) {
            if (baudrate == current_baud[sioport]) {
                baud_unlock[sioport] = 1;
            }
        }
        else {
            if (baud_unlock[sioport] == 1) {
                sim_debug(UART_MSG, &ibc_dev, ADDRESS_FORMAT "     Setting SIO %d baud: %d\n", PCX, sioport, baudrate);
                current_baud[sioport] = baudrate;
                baud_unlock[sioport] = 0;
            }
            else {
                sim_debug(ERROR_MSG, &ibc_dev, ADDRESS_FORMAT "     Error Setting SIO %d baud: %d (locked.)\n", PCX, sioport, baudrate);
            }
        }
    }
    else { /* Read */
        sim_debug(UART_MSG, &ibc_dev, ADDRESS_FORMAT
            " RD Baud Port: 0x%02x=0x%02x\n", PCX, port, result);
    }
    return result;
}

/* IBC Super Cadet DTR Control ports */
static int32 ibc_sc_dtr(const int32 port, const int32 io, const int32 data)
{
    uint8 sioport = (data & 0x07) | ((port & 1) << 3);
    uint8 deasserted = (port & 0x02) >> 1;
    static int32 result = 0xFF;

    if (io) { /* Write */
        sim_debug(UART_MSG, &ibc_dev, ADDRESS_FORMAT
            " WR DTR Port 0x%02x=0x%02x: SIO: %d: %s DTR\n", PCX, port, data, sioport, deasserted == 1 ? "De-assert" : "Assert");
    }
    else { /* Read */
        sim_debug(UART_MSG, &ibc_dev, ADDRESS_FORMAT
            " RD DTR Port: 0x%02x=0x%02x\n", PCX, port, result);
    }
    return result;
}

/* IBC Cadet Unhandled ports */
static int32 ibc_unhandled(const int32 port, const int32 io, const int32 data)
{
    static int32 result = 0xFF;
    if (io) { /* Write */
        sim_debug(UNHANDLED_IO_MSG, &ibc_dev, ADDRESS_FORMAT
            " WR Unhandled Port: 0x%02x=0x%02x\n", PCX, port, data);
    }
    else { /* Read */
        sim_debug(UNHANDLED_IO_MSG, &ibc_dev, ADDRESS_FORMAT
            " RD Unhandled Port: 0x%02x=0x%02x\n", PCX, port, result);
    }
    return result;
}
