// Copyright (c) 2010-2021 The Bitcoin Core developers
// Distributed under the MIT software license, see the accompanying
// file COPYING or http://www.opensource.org/licenses/mit-license.php.

//! @file node/types.h is a home for public enum and struct type definitions
//! that are used by internally by node code, but also used externally by wallet
//! or GUI code.
//!
//! This file is intended to define only simple types that do not have external
//! dependencies. More complicated types should be defined in dedicated header
//! files.

#ifndef BITCOIN_NODE_TYPES_H
#define BITCOIN_NODE_TYPES_H

#include <policy/feerate.h>
#include <policy/policy.h>

#include <cstddef>

namespace node {
enum class TransactionError {
    OK, //!< No error
    MISSING_INPUTS,
    ALREADY_IN_UTXO_SET,
    MEMPOOL_REJECTED,
    MEMPOOL_ERROR,
    MAX_FEE_EXCEEDED,
    MAX_BURN_EXCEEDED,
    INVALID_PACKAGE,
};

static const bool DEFAULT_PRINT_MODIFIED_FEE = false;

struct BlockCreateOptions {
    /**
     * Set false to omit mempool transactions
     */
    bool use_mempool{true};
    /**
     * The maximum additional size which the miner will add to the coinbase
     * scriptSig, witness and outputs. This must include any additional
     * size needed for larger CompactSize encoded lengths.
     */
    size_t coinbase_max_additional_size{1000};
    /**
     * The maximum additional weight which the pool will add to the coinbase
     * scriptSig, witness and outputs. This must include any additional
     * weight needed for larger CompactSize encoded lengths.
     */
    size_t coinbase_max_additional_weight{4000};
    /**
     * The maximum additional sigops which the pool will add in coinbase
     * transaction outputs.
     */
    size_t coinbase_output_max_additional_sigops{400};

    // Configuration parameters for the block size
    size_t nBlockMaxWeight{DEFAULT_BLOCK_MAX_WEIGHT};
    size_t nBlockMaxSize{DEFAULT_BLOCK_MAX_SIZE};
    CFeeRate blockMinFeeRate{DEFAULT_BLOCK_MIN_TX_FEE};
    // Whether to call TestBlockValidity() at the end of CreateNewBlock().
    bool test_block_validity{true};
    bool print_modified_fee{DEFAULT_PRINT_MODIFIED_FEE};

    BlockCreateOptions Clamped() const;

    friend bool operator==(const BlockCreateOptions& a, const BlockCreateOptions& b) noexcept = default;
};
} // namespace node

#endif // BITCOIN_NODE_TYPES_H
