/*
Copyright 2025 Element Creations Ltd.

SPDX-License-Identifier: AGPL-3.0-only OR GPL-3.0-only OR LicenseRef-Element-Commercial
Please see LICENSE files in the repository root for full details.
*/

// Gathers all the translation keys from element-web's en_EN.json into a TypeScript type definition file
// that exports a type `TranslationKey` which is a union of all supported translation keys.
// This prevents having to import the json file and make typescript do the work as this results in vite-dts
// generating an import to the json file in the .d.ts which doesn't work at runtime: this way, the type
// gets put into the bundle.
// XXX: It should *not* be in the 'src' directory, being a generated file, but if it isn't then the type
// bundler won't bundle the types and will leave the file as a relative import, which will break.

import * as fs from "fs";
import * as path from "path";
import { dirname } from "node:path";
import { fileURLToPath } from "node:url";

const __dirname = dirname(fileURLToPath(import.meta.url));
const i18nStringsPath = path.resolve(__dirname, "../../../src/i18n/strings/en_EN.json");
const outPath = path.resolve(__dirname, "../src/i18nKeys.d.ts");

function gatherKeys(obj: any, prefix: string[] = []): string[] {
    if (typeof obj !== "object" || obj === null) return [];
    let keys: string[] = [];
    for (const key of Object.keys(obj)) {
        const value = obj[key];

        // add the path (for both leaves and intermediates as then we include plurals)
        keys.push([...prefix, key].join("|"));
        if (typeof value === "object" && value !== null) {
            // If the value is an object, recurse
            keys = keys.concat(gatherKeys(value, [...prefix, key]));
        }
    }
    return keys;
}

function main() {
    const json = JSON.parse(fs.readFileSync(i18nStringsPath, "utf8"));
    const keys = gatherKeys(json);
    const typeDef =
        "/*\n" +
        " * Copyright 2025 Element Creations Ltd.\n" +
        " *\n" +
        " * SPDX-License-Identifier: AGPL-3.0-only OR GPL-3.0-only OR LicenseRef-Element-Commercial\n" +
        " * Please see LICENSE files in the repository root for full details.\n" +
        " */\n" +
        "\n" +
        "// This file is auto-generated by gatherTranslationKeys.ts\n" +
        "// Do not edit manually.\n\n" +
        "export type TranslationKey =\n" +
        keys.map((k) => `    | \"${k}\"`).join("\n") +
        ";\n";
    fs.mkdirSync(path.dirname(outPath), { recursive: true });
    fs.writeFileSync(outPath, typeDef, "utf8");
    console.log(`Wrote ${keys.length} keys to ${outPath}`);
}

if (import.meta.url.startsWith("file:")) {
    const modulePath = fileURLToPath(import.meta.url);
    if (process.argv[1] === modulePath) {
        main();
    }
}
