/*
Copyright 2024, 2025 New Vector Ltd.
Copyright 2022 The Matrix.org Foundation C.I.C.

SPDX-License-Identifier: AGPL-3.0-only OR GPL-3.0-only OR LicenseRef-Element-Commercial
Please see LICENSE files in the repository root for full details.
*/

import React from "react";
import { fireEvent, render, screen, waitFor, waitForElementToBeRemoved, within } from "jest-matrix-react";
import { EventType, getHttpUriForMxc, MatrixEvent, Room } from "matrix-js-sdk/src/matrix";
import fetchMock from "fetch-mock-jest";
import encrypt from "matrix-encrypt-attachment";
import { mocked } from "jest-mock";
import fs from "fs";
import path from "path";
import userEvent from "@testing-library/user-event";

import MImageBody from "../../../../../src/components/views/messages/MImageBody";
import { RoomPermalinkCreator } from "../../../../../src/utils/permalinks/Permalinks";
import {
    getMockClientWithEventEmitter,
    mockClientMethodsCrypto,
    mockClientMethodsDevice,
    mockClientMethodsServer,
    mockClientMethodsUser,
    withClientContextRenderOptions,
} from "../../../../test-utils";
import { MediaEventHelper } from "../../../../../src/utils/MediaEventHelper";
import SettingsStore from "../../../../../src/settings/SettingsStore";
import { MediaPreviewValue } from "../../../../../src/@types/media_preview";

jest.mock("matrix-encrypt-attachment", () => ({
    decryptAttachment: jest.fn(),
}));

describe("<MImageBody/>", () => {
    const ourUserId = "@user:server";
    const senderUserId = "@other_use:server";
    const deviceId = "DEADB33F";
    const cli = getMockClientWithEventEmitter({
        ...mockClientMethodsUser(ourUserId),
        ...mockClientMethodsServer(),
        ...mockClientMethodsDevice(deviceId),
        ...mockClientMethodsCrypto(),
        getRooms: jest.fn().mockReturnValue([]),
        getRoom: jest.fn(),
        getIgnoredUsers: jest.fn(),
        getVersions: jest.fn().mockResolvedValue({
            unstable_features: {
                "org.matrix.msc3882": true,
                "org.matrix.msc3886": true,
            },
        }),
    });
    const url = "https://server/_matrix/media/v3/download/server/encrypted-image";
    // eslint-disable-next-line no-restricted-properties
    cli.mxcUrlToHttp.mockImplementation(
        (mxcUrl: string, width?: number, height?: number, resizeMethod?: string, allowDirectLinks?: boolean) => {
            return getHttpUriForMxc("https://server", mxcUrl, width, height, resizeMethod, allowDirectLinks);
        },
    );
    const encryptedMediaEvent = new MatrixEvent({
        event_id: "$foo:bar",
        room_id: "!room:server",
        sender: senderUserId,
        type: EventType.RoomMessage,
        content: {
            body: "alt for a test image",
            info: {
                w: 40,
                h: 50,
                mimetype: "image/png",
            },
            file: {
                url: "mxc://server/encrypted-image",
            },
        },
    });

    const props = {
        onMessageAllowed: jest.fn(),
        permalinkCreator: new RoomPermalinkCreator(new Room(encryptedMediaEvent.getRoomId()!, cli, cli.getUserId()!)),
    };

    beforeEach(() => {
        jest.spyOn(SettingsStore, "getValue").mockRestore();
        fetchMock.mockReset();
    });

    afterEach(() => {
        SettingsStore.reset();
        mocked(encrypt.decryptAttachment).mockReset();
    });

    it("should show a thumbnail while image is being downloaded", async () => {
        fetchMock.getOnce(url, { status: 200 });

        const { container } = render(
            <MImageBody
                {...props}
                mxEvent={encryptedMediaEvent}
                mediaEventHelper={new MediaEventHelper(encryptedMediaEvent)}
            />,
            withClientContextRenderOptions(cli),
        );

        // thumbnail with dimensions present
        expect(container).toMatchSnapshot();
    });

    it("should show error when encrypted media cannot be downloaded", async () => {
        fetchMock.getOnce(url, { status: 500 });

        render(
            <MImageBody
                {...props}
                mxEvent={encryptedMediaEvent}
                mediaEventHelper={new MediaEventHelper(encryptedMediaEvent)}
            />,
            withClientContextRenderOptions(cli),
        );

        expect(fetchMock).toHaveBeenCalledWith(url);

        await screen.findByText("Error downloading image");
    });

    it("should show error when encrypted media cannot be decrypted", async () => {
        fetchMock.getOnce(url, "thisistotallyanencryptedpng");
        mocked(encrypt.decryptAttachment).mockRejectedValue(new Error("Failed to decrypt"));

        render(
            <MImageBody
                {...props}
                mxEvent={encryptedMediaEvent}
                mediaEventHelper={new MediaEventHelper(encryptedMediaEvent)}
            />,
            withClientContextRenderOptions(cli),
        );

        await screen.findByText("Error decrypting image");
    });

    describe("with image previews/thumbnails disabled", () => {
        beforeEach(() => {
            const origFn = SettingsStore.getValue;
            jest.spyOn(SettingsStore, "getValue").mockImplementation((setting, ...args) => {
                if (setting === "mediaPreviewConfig") {
                    return { invite_avatars: MediaPreviewValue.Off, media_previews: MediaPreviewValue.Off };
                }
                return origFn(setting, ...args);
            });
        });

        it("should not download image", async () => {
            fetchMock.getOnce(url, { status: 200 });

            render(
                <MImageBody
                    {...props}
                    mxEvent={encryptedMediaEvent}
                    mediaEventHelper={new MediaEventHelper(encryptedMediaEvent)}
                />,
                withClientContextRenderOptions(cli),
            );

            expect(screen.getByText("Show image")).toBeInTheDocument();

            expect(fetchMock).not.toHaveFetched(url);
        });

        it("should render hidden image placeholder", async () => {
            fetchMock.getOnce(url, { status: 200 });

            render(
                <MImageBody
                    {...props}
                    mxEvent={encryptedMediaEvent}
                    mediaEventHelper={new MediaEventHelper(encryptedMediaEvent)}
                />,
                withClientContextRenderOptions(cli),
            );

            expect(screen.getByText("Show image")).toBeInTheDocument();

            fireEvent.click(screen.getByRole("button"));

            expect(fetchMock).toHaveFetched(url);

            // Show image is asynchronous since it applies through a settings watcher hook, so
            // be sure to wait here.
            await waitFor(() => {
                // spinner while downloading image
                expect(screen.getByRole("progressbar")).toBeInTheDocument();
            });
        });
    });

    it("should fall back to /download/ if /thumbnail/ fails", async () => {
        const thumbUrl = "https://server/_matrix/media/v3/thumbnail/server/image?width=800&height=600&method=scale";
        const downloadUrl = "https://server/_matrix/media/v3/download/server/image";

        const event = new MatrixEvent({
            room_id: "!room:server",
            sender: senderUserId,
            type: EventType.RoomMessage,
            content: {
                body: "alt for a test image",
                info: {
                    w: 40,
                    h: 50,
                },
                url: "mxc://server/image",
            },
        });

        const { container } = render(
            <MImageBody {...props} mxEvent={event} mediaEventHelper={new MediaEventHelper(event)} />,
            withClientContextRenderOptions(cli),
        );

        const img = container.querySelector(".mx_MImageBody_thumbnail")!;
        expect(img).toHaveProperty("src", thumbUrl);

        fireEvent.error(img);
        expect(img).toHaveProperty("src", downloadUrl);
    });

    it("should generate a thumbnail if one isn't included for animated media", async () => {
        Object.defineProperty(global.Image.prototype, "src", {
            set(src) {
                window.setTimeout(() => this.onload?.());
            },
        });
        Object.defineProperty(global.Image.prototype, "height", {
            get() {
                return 600;
            },
        });
        Object.defineProperty(global.Image.prototype, "width", {
            get() {
                return 800;
            },
        });

        mocked(global.URL.createObjectURL).mockReturnValue("blob:generated-thumb");

        fetchMock.getOnce(
            "https://server/_matrix/media/v3/download/server/image",
            {
                body: fs.readFileSync(path.resolve(__dirname, "..", "..", "..", "images", "animated-logo.webp")),
            },
            { sendAsJson: false },
        );

        const event = new MatrixEvent({
            room_id: "!room:server",
            sender: senderUserId,
            type: EventType.RoomMessage,
            content: {
                body: "alt for a test image",
                info: {
                    w: 40,
                    h: 50,
                    mimetype: "image/webp",
                },
                url: "mxc://server/image",
            },
        });

        const { container } = render(
            <MImageBody {...props} mxEvent={event} mediaEventHelper={new MediaEventHelper(event)} />,
            withClientContextRenderOptions(cli),
        );

        // Wait for spinners to go away
        await waitForElementToBeRemoved(screen.getAllByRole("progressbar"));
        // thumbnail with dimensions present
        expect(container).toMatchSnapshot();
    });

    it("should show banner on hover", async () => {
        const event = new MatrixEvent({
            room_id: "!room:server",
            sender: senderUserId,
            type: EventType.RoomMessage,
            content: {
                body: "alt for a test image",
                info: {
                    w: 40,
                    h: 50,
                },
                url: "mxc://server/image",
            },
        });

        const { container } = render(
            <MImageBody {...props} mxEvent={event} mediaEventHelper={new MediaEventHelper(event)} />,
            withClientContextRenderOptions(cli),
        );

        const img = container.querySelector(".mx_MImageBody_thumbnail")!;
        await userEvent.hover(img);

        expect(container.querySelector(".mx_MImageBody_banner")).toHaveTextContent("...alt for a test image");
    });

    it("should render MFileBody for svg with no thumbnail", async () => {
        const event = new MatrixEvent({
            room_id: "!room:server",
            sender: senderUserId,
            type: EventType.RoomMessage,
            content: {
                info: {
                    w: 40,
                    h: 50,
                    mimetype: "image/svg+xml",
                },
                file: {
                    url: "mxc://server/encrypted-svg",
                },
            },
        });

        const { container, asFragment } = render(
            <MImageBody {...props} mxEvent={event} mediaEventHelper={new MediaEventHelper(event)} />,
            withClientContextRenderOptions(cli),
        );

        expect(container.querySelector(".mx_MFileBody")).toHaveTextContent("Attachment");
        expect(asFragment()).toMatchSnapshot();
    });

    it("should open ImageView using thumbnail for encrypted svg", async () => {
        const url = "https://server/_matrix/media/v3/download/server/encrypted-svg";
        fetchMock.getOnce(url, { status: 200 });
        const thumbUrl = "https://server/_matrix/media/v3/download/server/svg-thumbnail";
        fetchMock.getOnce(thumbUrl, { status: 200 });

        const event = new MatrixEvent({
            room_id: "!room:server",
            sender: senderUserId,
            type: EventType.RoomMessage,
            origin_server_ts: 1234567890,
            content: {
                info: {
                    w: 40,
                    h: 50,
                    mimetype: "image/svg+xml",
                    thumbnail_file: {
                        url: "mxc://server/svg-thumbnail",
                    },
                    thumbnail_info: { mimetype: "image/png" },
                },
                file: {
                    url: "mxc://server/encrypted-svg",
                },
            },
        });

        const mediaEventHelper = new MediaEventHelper(event);
        mediaEventHelper.thumbnailUrl["prom"] = Promise.resolve(thumbUrl);
        mediaEventHelper.sourceUrl["prom"] = Promise.resolve(url);

        const { findByRole } = render(
            <MImageBody {...props} mxEvent={event} mediaEventHelper={mediaEventHelper} />,
            withClientContextRenderOptions(cli),
        );

        fireEvent.click(await findByRole("link"));

        const dialog = await screen.findByRole("dialog");
        await expect(within(dialog).findByRole("img")).resolves.toHaveAttribute(
            "src",
            "https://server/_matrix/media/v3/download/server/svg-thumbnail",
        );
        expect(dialog).toMatchSnapshot();
    });
});
