// Copyright (c) Tailscale Inc & AUTHORS
// SPDX-License-Identifier: BSD-3-Clause

// Package eventbustest provides helper methods for testing an [eventbus.Bus].
//
// # Usage
//
// A [Watcher] presents a set of generic helpers for testing events.
//
// To test code that generates events, create a [Watcher] from the [eventbus.Bus]
// used by the code under test, run the code to generate events, then use the watcher
// to verify that the expected events were produced. In outline:
//
//	bus := eventbustest.NewBus(t)
//	tw := eventbustest.NewWatcher(t, bus)
//	somethingThatEmitsSomeEvent()
//	if err := eventbustest.Expect(tw, eventbustest.Type[EventFoo]()); err != nil {
//	  t.Error(err.Error())
//	}
//
// As shown, [Expect] checks that at least one event of the given type occurs
// in the stream generated by the code under test.
//
// The following functions all take an any parameter representing a function.
// This function will take an argument of the expected type and is used to test
// for the events on the eventbus being of the given type. The function can
// take the shape described in [Expect].
//
// [Type] is a helper for only testing event type.
//
// To check for specific properties of an event, use [Expect], and pass a function
// as the second argument that tests for those properties.
//
// To test for multiple events, use [Expect], which checks that the stream
// contains the given events in the given order, possibly with other events
// interspersed.
//
// To test the complete contents of the stream, use [ExpectExactly], which
// checks that the stream contains exactly the given events in the given order,
// and no others.
//
// To test for the absence of events, use [ExpectExactly] without any
// expected events, along side [testing/synctest] to avoid waiting for timers
// to ensure that no events are produced. This will look like:
//
//	synctest.Test(t, func(t *testing.T) {
//		bus := eventbustest.NewBus(t)
//		tw := eventbustest.NewWatcher(t, bus)
//		somethingThatShouldNotEmitsSomeEvent()
//		synctest.Wait()
//		if err := eventbustest.ExpectExactly(tw); err != nil {
//			t.Errorf("Expected no events or errors, got %v", err)
//		}
//	})
//
// See the [usage examples].
//
// [usage examples]: https://github.com/tailscale/tailscale/blob/main/util/eventbus/eventbustest/examples_test.go
package eventbustest
