# Copyright 2018-2026 Jérôme Dumonteil
# Copyright (c) 2009-2013 Ars Aperta, Itaapy, Pierlis, Talend.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
#
# Authors (odfdo project): jerome.dumonteil@gmail.com
# The odfdo project is a derivative work of the lpod-python project:
# https://github.com/lpod/lpod-python
"""Command-line script to shrink tables in an ODF spreadsheet.

This script optimizes the width and height of all tables in a spreadsheet
document by removing empty trailing rows and columns.
"""

from __future__ import annotations

from argparse import ArgumentParser, Namespace

from odfdo import __version__
from odfdo.utils.script_utils import read_document, save_document

PROG = "odfdo-table-shrink"


def configure_parser() -> ArgumentParser:
    description = (
        "Optimize the width and height of tables in an ODF spreadsheet "
        "by removing empty trailing rows and columns."
    )
    epilog = (
        "This tool helps reduce the size and improve the presentation of "
        "spreadsheet documents by automatically cleaning up unused table space. "
        "It's particularly useful for spreadsheets generated by applications "
        "that might include excessive empty cells."
    )
    parser = ArgumentParser(prog=PROG, description=description, epilog=epilog)

    parser.add_argument(
        "--version",
        action="version",
        version=f"{PROG} v{__version__}",
    )
    parser.add_argument(
        "-i",
        "--input",
        action="store",
        dest="input_file",
        metavar="INPUT",
        required=False,
        help="input file. if option not present, read from stdin",
    )
    parser.add_argument(
        "-o",
        "--output",
        action="store",
        dest="output_file",
        metavar="OUTPUT",
        required=False,
        help="output file. if option not present, write to stdout",
    )
    return parser


def parse_cli_args(cli_args: list[str] | None = None) -> Namespace:
    parser = configure_parser()
    return parser.parse_args(cli_args)


def main() -> None:
    args: Namespace = parse_cli_args()
    main_shrink(args)


def main_shrink(args: Namespace) -> None:
    try:
        shrink_tables(
            args.input_file,
            args.output_file,
        )
    except Exception as e:
        configure_parser().print_help()
        print()
        print(f"Error: {e.__class__.__name__}, {e}")
        raise SystemExit(1) from None


def shrink_tables(
    input_path: str | None,
    output_path: str | None,
) -> None:
    document = read_document(input_path)
    if document.get_type() not in {"spreadsheet", "spreadsheet-template"}:
        raise TypeError("Document must be a Spreadsheet type.")
    for table in document.body.tables:
        table.optimize_width()
    save_document(document, output_path)


if __name__ == "__main__":
    main()
