package sboRepo

import (
	"fmt"

	"gitlab.com/M0M097/pkg/lib/utils"
)

// PkgCreator is a struct that holds the necessary fields to create a new
// Package instance. It can be used as a annonymous struct to conveniently
// call its Create method to generate a new Package instance without having
// to manually set each field. All omitted fields will be set to an empty
// string.
type PkgCreator struct {
	Name,
	Path,
	Version,
	Homepage,
	Download,
	Md5sum,
	Download64,
	Md5sum64,
	Requires,
	Maintainer,
	Email,
	SlackBuildContent,
	ReadmeContent,
	SlackDescContent string
}

// Create creates a new Package instance based on the provided PkgCreator
// fields. It generates the necessary directory structure and files for the,
// package including the .info, README, and slack-desc files. It returns a
// pointer to the generated Package instance.
func (pc PkgCreator) Create() *Package {
	p := Package{
		name: pc.Name,
		path: pc.Path,
	}
	utils.Mkdir(p.Path())
	utils.Content2File(p.SlackBuild(), pc.SlackBuildContent)
	utils.Content2File(p.Readme(), pc.ReadmeContent)
	utils.Content2File(p.SlackDesc(), pc.SlackDescContent)
	f := utils.Create(p.InfoFile())
	defer f.Close()
	fmt.Fprintf(f, `PRGNAM="%s"
VERSION="%s"
HOMEPAGE="%s"
DOWNLOAD="%s"
MD5SUM="%s"
DOWNLOAD_x86_64="%s"
MD5SUM_x86_64="%s"
REQUIRES="%s"
MAINTAINER="%s"
EMAIL="%s"
`,
		p.name, pc.Version, pc.Homepage, pc.Download, pc.Md5sum,
		pc.Download64, pc.Md5sum64, pc.Requires, pc.Maintainer, pc.Email,
	)
	return &p
}
