package sboRepo

import (
	"strings"
	"testing"
)

const (
	DUMMY_PKG1                  = "pkg1"
	DUMMY_PKG2                  = "pkg2"
	DUMMY_PKG3                  = "pkg3"
	DUMMY_PKG3_DOWNLOAD         = "foo.tar.gz"
	DUMMY_VERSION               = "1.0"
	DUMMY_DOWNLOAD_URL          = "http://example.com/foo.tar.gz"
	DUMMY_DOWNLOAD_URL64        = "http://example.com/foo64.tar.gz"
	DUMMY_MD5SUM                = "1234567890abcdef1234567890abcdef"
	DUMMY_MD5SUM64              = "abcdef1234567890abcdef1234567890"
	DUMMY_HOMEPAGE              = "http://example.com"
	DUMMY_MAINTAINER            = "Maintainer Name"
	DUMMY_EMAIL                 = "john[at]example.com"
	DUMMY_BLACKLISTED_PKG       = "blacklisted_pkg"
	DUMMY_GREYLISTED_PKG        = "greylisted_pkg"
	DUMMY_OTHER_BLACKLISTED_PKG = "other_blacklisted_pkg"
	MOCK_PKG_SLACKBUILD_CONTENT = "exit 1"
	MOCK_PKG_README_CONTENT     = "README file\n"
	MOCK_PKG_SLACK_DES_CONTENT  = `# HOW TO EDIT THIS FILE:
# The \"handy ruler\" below makes it easier to edit a package description.  Line
# up the first '|' above the ':' following the base package name, and the '|'
# on the right side marks the last column you can put a character in.  You must
# make exactly 11 lines for the formatting to be correct.  It's also
# customary to leave one space after the ':'.

      |-----handy-ruler------------------------------------------------------|
name: name Description
name:
name:
name:
name:
name:
name:
name:
name:
name:
name:
`
)

func mustBeEqual(t *testing.T, got, expected interface{}) {
	t.Helper()
	if got != expected {
		t.Fatalf("Expected %v, got %v", expected, got)
	}
}

func mustBeEqualStringSlices(t *testing.T, got, expected []string) {
	t.Helper()
	if len(got) != len(expected) {
		t.Fatalf("Expected %v and %v to have the same length, got %d and %d",
			expected, got, len(expected), len(got))
	}
	for i := range got {
		mustBeEqual(t, got[i], expected[i])
	}
}

func mustContain(t *testing.T, output string, expected ...string) {
	t.Helper()
	for _, s := range expected {
		if !strings.Contains(output, s) {
			t.Errorf("Expected %v to be in %v", s, output)
		}
	}
}

func mustNotContain(t *testing.T, output string, expected ...string) {
	t.Helper()
	for _, s := range expected {
		if strings.Contains(output, s) {
			t.Errorf("Expected %v not to be in %v", s, output)
		}
	}
}

func createPkgTestfilesInDir(p *Package, t *testing.T, dir string) {
	t.Helper()
	createTestfileArgsInDir(p, t,
		dir,
		DUMMY_DOWNLOAD_URL, DUMMY_MD5SUM,
		"", "",
		"%README% dependee1",
	)
}

func createPkgTestfiles(p *Package, t *testing.T) {
	t.Helper()
	createPkgTestfilesInDir(p, t, t.TempDir())
}

func createTestfileArgs(p *Package, t *testing.T, download, md5sum, download64, md5sum64, requires string) {
	t.Helper()
	createTestfileArgsInDir(p, t, t.TempDir(), download, md5sum, download64, md5sum64, requires)
}

func createTestfileArgsInDir(p *Package, t *testing.T, tdir, download, md5sum, download64, md5sum64, requires string) {
	t.Helper()
	*p = *PkgCreator{
		Name:              p.Name(),
		Path:              tdir,
		Version:           DUMMY_VERSION,
		Download:          download,
		Md5sum:            md5sum,
		Download64:        download64,
		Md5sum64:          md5sum64,
		Requires:          requires,
		SlackBuildContent: MOCK_PKG_SLACKBUILD_CONTENT,
		ReadmeContent:     MOCK_PKG_README_CONTENT,
		SlackDescContent:  MOCK_PKG_SLACK_DES_CONTENT,
	}.Create()
}

// This helper modifies the .info file contents so that it has the given REQUIRES line.
// A typical info file creation might look like this:
func createPkgTestfilesWithRequires(p *Package, t *testing.T, requiresLine string) {
	t.Helper()
	createTestfileArgs(p, t,
		"http://example.com/foo.tar.gz", "1234567890",
		"", "",
		requiresLine,
	)
}

func MockPkg() *Package {
	var (
		head       = &Package{}
		tail       = head
		prev, next = &Package{name: "prev"}, &Package{name: "next"}
		p          = &Package{
			name:      "name",
			path:      "category",
			dependees: []*Package{prev, next},
		}
	)
	tail = prev.Append(tail)
	tail = p.Append(tail)
	tail = next.Append(tail)
	return p
}
