package sboRepo

import (
	"os"
	"path/filepath"
	"strings"
	"testing"

	"gitlab.com/M0M097/pkg/lib/utils"
)

func TestNewPackage(t *testing.T) {
	pkg := NewPackage("pkg_name", "pkg_path")
	if pkg.name != "pkg_name" || pkg.path != "pkg_path" {
		t.Error("NewPackage() did not construct a package type correctly.")
	}
}

func TestDependees(t *testing.T) {
	pkg := MockPkg()
	if len(pkg.Dependees()) != 2 {
		t.Error("Dependees() returned an incorrect value")
	}
}

func TestDependencies(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t)
	if len(pkg.Dependencies()) != 2 {
		t.Error("Dependencies() returned an incorrect value")
	}
}

func TestInfoFile(t *testing.T) {
	pkg := MockPkg()
	if pkg.InfoFile() != "category/name/name.info" {
		t.Error("InfoFile() returned an incorrect value")
	}
}

func TestName(t *testing.T) {
	pkg := MockPkg()
	if pkg.Name() != "name" {
		t.Error("Name() returned an incorrect value")
	}
}

func TestPath(t *testing.T) {
	pkg := MockPkg()
	if pkg.Path() != "category/name" {
		t.Error("Path() returned an incorrect value")
	}
}

func TestReadme(t *testing.T) {
	pkg := MockPkg()
	if pkg.Readme() != "category/name/README" {
		t.Error("Readme() returned an incorrect value")
	}
}

func TestSlackBuild(t *testing.T) {
	pkg := MockPkg()
	if pkg.SlackBuild() != "category/name/name.SlackBuild" {
		t.Error("SlackBuild() returned an incorrect value")
	}
}

func TestVersion(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t)
	if pkg.Version() != "1.0" {
		t.Error("Version() expected 1.0, got ", pkg.Version())
	}
}

func TestFullName(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t)
	if pkg.FullName() != "name-1.0" {
		t.Error("FullName() expected name-1.0, got ", pkg.FullName())
	}
}

func TestIsReadmeRequired(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t)
	if !pkg.IsReadmeRequired() {
		t.Error("IsReadmeRequired() returned an incorrect value")
	}
}

func TestAllReadmes(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t)
	// Create a README.Slackware file
	utils.Create(filepath.Join(pkg.Path(), "README.Slackware"))

	if pkg.AllReadmes() != pkg.Readme()+" "+pkg.Path()+"/README.Slackware" {
		t.Error("AllReadmes() returned an incorrect value")
	}
}

func TestDownloadUrls(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t)
	if len(pkg.DownloadUrls()) != 1 ||
		pkg.DownloadUrls()[0] != "http://example.com/foo.tar.gz" {
		t.Error("DownloadUrls() returned an incorrect value")
	}
}

func TestMd5sums(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t) // Only MD5SUM (no MD5SUM_x86_64)
	mustBeEqual(t, len(pkg.Md5sums()), 1)
	mustBeEqual(t, pkg.Md5sums()[0], DUMMY_MD5SUM)

	createTestfileArgs(pkg, t, "", "", "", "8664", "") // MD5SUM_x86_64
	mustBeEqual(t, len(pkg.Md5sums()), 1)
	mustBeEqual(t, pkg.Md5sums()[0], "8664")

	createTestfileArgs(pkg, t, "", "1234567890", "", "8664", "") // both
	mustBeEqual(t, len(pkg.Md5sums()), 1)
	mustBeEqual(t, pkg.Md5sums()[0], "8664")
}

func TestDownloadNames(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t) // Only DOWNLOAD (no DOWNLOAD_x86_64)
	if len(pkg.DownloadNames()) != 1 || pkg.DownloadNames()[0] != "foo.tar.gz" {
		t.Error("Incorrect value, with only DOWNLOAD")
	}
	createTestfileArgs(pkg, t, "", "", "foo_x86_64", "", "") // DOWNLOAD_x86_64
	if len(pkg.DownloadNames()) != 1 || pkg.DownloadNames()[0] != "foo_x86_64" {
		t.Error("Incorrect value, with only DOWNLOAD_x86_64")
	}
	createTestfileArgs(pkg, t, "foo.tar.gz", "", "foo_x86_64", "", "") // both
	if len(pkg.DownloadNames()) != 1 || pkg.DownloadNames()[0] != "foo_x86_64" {
		t.Error("Incorrect value, when both DOWNLOAD_x86_64 and DOWNLOAD are present")
	}
	createTestfileArgs(pkg, t, "foo.tar.gz \\\n\tbar.tar.gz\\\n    baz.tar.gz", "", "", "", "") // multiple
	if len(pkg.DownloadNames()) != 3 ||
		pkg.DownloadNames()[0] != "foo.tar.gz" ||
		pkg.DownloadNames()[1] != "bar.tar.gz" ||
		pkg.DownloadNames()[2] != "baz.tar.gz" {
		t.Error("Incorrect value, with multiple DOWNLOAD")
	}
}

func TestIsUpToDate(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t)
	docDir := t.TempDir()
	if pkg.IsUpToDate(docDir, "random_kernel_version") {
		t.Error("Returns true although the package is not installed")
	}
	utils.Mkdir(docDir + "/name-1.0")
	utils.Content2File(docDir+"/name-1.0/name.SlackBuild", "differs")
	if pkg.IsUpToDate(docDir, "random_kernel_version") {
		t.Error("Returns true although the package is not up-to-date")
	}
	utils.Content2File(docDir+"/name-1.0/name.SlackBuild", MOCK_PKG_SLACKBUILD_CONTENT)
	if !pkg.IsUpToDate(docDir, "random_kernel_version") {
		t.Error("Returns false although the package is up-to-date")
	}
}

func TestIsUpToDateKernelVersion(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t)
	docDir := t.TempDir()
	utils.Mkdir(docDir + "/name-1.0_random_kernel_version")
	utils.Content2File(docDir+"/name-1.0_random_kernel_version/name.SlackBuild", MOCK_PKG_SLACKBUILD_CONTENT)
	if !pkg.IsUpToDate(docDir, "random_kernel_version") {
		t.Error("Returns false although the package is up-to-date")
	}
}

func TestAddDependee(t *testing.T) {
	pkg := MockPkg()
	dependee := MockPkg()
	pkg.AddDependee(dependee)
	if len(pkg.dependees) != 3 {
		t.Error("AddDependee() did not add the dependee")
	}
}

func TestQueryInfoFilePanic(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t)

	// Test that requesting a non-existing package panics
	defer func() {
		if r := recover(); r == nil {
			t.Errorf("expected a panic for non-existing package, got no panic")
		}
	}()
	pkg.queryInfoFile("non-existing-key")
}

func TestPackage_FindBuildPkg(t *testing.T) {
	t.Run("Success_MatchingFile", func(t *testing.T) {
		// 1) Create a package with a known FullName "name-1.0"
		pkg := MockPkg()
		createPkgTestfiles(pkg, t)

		// 2) Create a temp directory
		tmpDir := t.TempDir()

		// 3) Make a file in tmpDir that matches the regex: pkg.FullName()+".*_SBo.*"
		//    e.g. "foo-1.0-x86_64-1_SBo.tgz"
		fileName := pkg.FullName() + "-x86_64-1_SBo.tgz" // "foo-1.0-x86_64-1_SBo.tgz"
		fullPath := filepath.Join(tmpDir, fileName)
		if err := os.WriteFile(fullPath, []byte("dummy content"), 0o644); err != nil {
			t.Fatalf("failed to create slackpkg: %v", err)
		}

		// 4) Call InstallCmd and ensure it returns "installpkg /tempDir/foo-1.0-x86_64-1_SBo.tgz"
		got := pkg.FindBuildPkg(tmpDir)
		want := fullPath
		if got != want {
			t.Errorf("expected %q, got %q", want, got)
		}
	})

	t.Run("Panic_NoMatchingFile", func(t *testing.T) {
		// 1) Package with some full name
		pkg := MockPkg()
		createPkgTestfiles(pkg, t)

		// 2) Temp directory has a file that DOES NOT match the package's regex
		tmpDir := t.TempDir()
		if err := os.WriteFile(filepath.Join(tmpDir, "randomfile_SBo.tgz"), []byte(""), 0o644); err != nil {
			t.Fatalf("failed to create a dummy file: %v", err)
		}

		// 3) Expect panic because no file matches "bar-2.5.*_SBo.*"
		defer func() {
			if r := recover(); r == nil {
				t.Error("expected a panic but did not get one")
			} else {
				// Optionally check the panic message
				msg, ok := r.(string)
				if ok && !strings.Contains(msg, "No package to install") {
					t.Errorf("expected panic message to contain 'No package to install', got %q", msg)
				}
			}
		}()

		_ = pkg.FindBuildPkg(tmpDir) // should panic
		t.Error("InstallCmd should have panicked before this line")
	})
}

func TestIsInList(t *testing.T) {
	pkg := MockPkg()
	if !pkg.IsInList() {
		t.Error("IsInList() returned an incorrect value")
	}
	pkg.prev = nil
	if pkg.IsInList() {
		t.Error("IsInList() returned an incorrect value")
	}
}

func TestDescription(t *testing.T) {
	pkg := MockPkg()
	createPkgTestfiles(pkg, t)
	if pkg.Description() != pkg.Name()+" Description" {
		t.Error("Description() returned an incorrect value")
	}
	// Now we check an ill-formatted slack-desc file
	pkg = PkgCreator{
		Name:             pkg.name,
		Path:             filepath.Join(t.TempDir(), "category"),
		SlackDescContent: "foobarbaz", // slackDescContent
	}.Create()
	mustBeEqual(t, pkg.Description(), pkg.Name())
}

func TestAppend(t *testing.T) {
	pkg := MockPkg()
	newPkg := &Package{name: "new"}
	newPkg.Append(pkg)
	if pkg.next != newPkg || newPkg.prev != pkg {
		t.Error("Append() did not append the package")
	}
}

func TestDel(t *testing.T) {
	pkg := MockPkg()
	prev, next := pkg.prev, pkg.next
	pkg.Del()
	if pkg.prev != nil || prev.next.name != "next" || next.prev.name != "prev" {
		t.Error("Delete() did not delete the package")
	}
}

func TestWrite(t *testing.T) {
	pkg := MockPkg()
	var output strings.Builder
	pkg.Write(&output)
	if output.String() != pkg.Name()+"\n" {
		t.Error("Write() did not write the package name correctly")
	}
}
