#!/usr/bin/python3
# -*- coding: utf-8 -*-

import json
import sys
from pathlib import Path
from typing import Any

import tomlkit

BASE_REPO_DIR: Path = Path('/var/lib/slpkg/repos')
REPOSITORIES_TOML_PATH: Path = Path('/etc/slpkg/repositories.toml')


def get_default_repo_from_toml() -> str:
    """
    Reads the default repository name from the repositories.toml file.
    Returns 'slack' as a fallback if the file cannot be read or 'toml' is not available.
    """
    if not REPOSITORIES_TOML_PATH.is_file():
        # Fallback if the TOML file does not exist.
        return 'sbo'

    try:
        with open(REPOSITORIES_TOML_PATH, 'r', encoding='utf-8') as f:
            config: dict[str, str] = tomlkit.load(f)
            # Check if '[DEFAULT]' section and 'REPOSITORY' key exist
            if 'DEFAULT' in config and 'REPOSITORY' in config['DEFAULT']:
                return config['DEFAULT']['REPOSITORY']  # type: ignore
    except Exception:  # pylint: disable=[W0718]
        pass

    # Fallback if anything goes wrong during TOML parsing.
    return 'sbo'


# Set DEFAULT_REPO_NAME dynamically
DEFAULT_REPO_NAME: str = get_default_repo_from_toml()


def get_repo_packages(repo_name: str) -> dict[str, dict[str, str]]:
    """Loads package list from a specific repository's JSON cache file.

    Args:
        repo_name (str): Repository name.

    Returns:
        list[str]: List of packages.
    """
    repo_file: Path = BASE_REPO_DIR / repo_name / 'data.json'

    if not repo_file.is_file():
        return {}

    try:
        with open(repo_file, 'r', encoding='utf-8') as f:
            loaded_data: Any = json.load(f)

            if not isinstance(loaded_data, dict):
                return {}
            return loaded_data
    except json.JSONDecodeError:
        return {}
    except Exception:  # pylint: disable=[W0718]
        return {}


def get_available_packages(current_command_words: list[str], prefix: str = "") -> list[str]:
    """
    Fetches and filters available packages, considering the selected repository.

    Args:
        current_command_words (list[str]): The current commands on argv readline.
        prefix (str, optional): The begin prefix of words.

    Returns:
        list[str]: Available packages.
    """
    selected_repo: str = DEFAULT_REPO_NAME

    try:
        o_index = current_command_words.index('-o')
        if o_index + 1 < len(current_command_words):
            selected_repo = current_command_words[o_index + 1]
    except ValueError:
        try:
            repo_index: int = current_command_words.index('--repository')
            if repo_index + 1 < len(current_command_words):
                selected_repo = current_command_words[repo_index + 1]
        except ValueError:
            pass

    repo_available_packages: dict[str, dict[str, str]] = get_repo_packages(selected_repo)
    return [pkg for pkg in repo_available_packages if pkg.startswith(prefix)]


if __name__ == '__main__':
    try:
        if len(sys.argv) < 2:
            sys.exit(1)

        cword_index: int = int(sys.argv[1])
        command_words: list[str] = sys.argv[2:]

        # CURRENT_WORD_PREFIX is the fragment of the word being completed.
        CURRENT_WORD_PREFIX = ''
        if len(command_words) > cword_index:
            CURRENT_WORD_PREFIX = command_words[cword_index]

        completions = []

        ALL_SLPKG_COMMANDS: list[str] = [
            "update", "upgrade", "config", "repo-info", "clean-tmp",
            "self-check", "build", "install", "remove", "download",
            "list", "info", "search", "dependees", "tracking",
            "changelog", "version", "help"
        ]

        if cword_index == 0:
            # Case: slpkg <TAB>
            # cword_index is 0, command_words will contain only 'slpkg'.
            # We want to list all primary commands. The prefix for matching is empty.
            for cmd in ALL_SLPKG_COMMANDS:
                completions.append(cmd)
        elif cword_index == 1:
            # Case: slpkg u<TAB> or slpkg in<TAB>
            # cword_index is 1, command_words might be ['slpkg', 'u'] or ['slpkg', 'in'].
            # CURRENT_WORD_PREFIX will be 'u' or 'in'.
            # We need to filter the main commands based on this prefix.
            for cmd in ALL_SLPKG_COMMANDS:
                if cmd.startswith(CURRENT_WORD_PREFIX):
                    completions.append(cmd)
        else:
            subcommand: str = command_words[1] if len(command_words) > 1 else ""

            package_commands: list[str] = ["install", "remove", "build", "download", "info", "search", "list", "dependees", "tracking"]

            if subcommand in package_commands:
                # If it's a package-related command and the current word is not an option flag.
                if not CURRENT_WORD_PREFIX.startswith('-'):
                    completions = get_available_packages(command_words, CURRENT_WORD_PREFIX)
        for comp in completions:
            print(comp)

    except Exception:  # pylint: disable=[W0718]
        # Catch all exceptions silently, as completion scripts should not crash the shell.
        pass

    sys.exit(0)
