/*
 *  linux/fs/read_write.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

#include <linux/types.h>
#include <linux/errno.h>
#include <linux/stat.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/fileio.h>

#include <asm/segment.h>

/*
 * Count is not yet used: but we'll probably support reading several entries
 * at once in the future. Use count = 1 in the library for future expansions.
 */
asmlinkage int sys_readdir(unsigned int fd, struct dirent *dirent, unsigned int count)
{
   int error;
   struct file *filp;
   struct inode *inode;

   if (fd >= NR_OPEN || !(filp = current->filp[fd]) ||
       !(inode = filp->f_inode))
      return -EBADF;
   error = -ENOTDIR;
   if (filp->f_op && filp->f_op->readdir) {
      error = verify_area(VERIFY_WRITE, dirent, sizeof (*dirent));
      if (!error)
         error = filp->f_op->readdir(inode, filp, dirent, count);
   }
   return error;
}

asmlinkage int sys_lseek(unsigned int fd, off_t offset, unsigned int origin)
{
   struct file *filp;
   int tmp = -1;

   if (fd >= NR_OPEN || !(filp = current->filp[fd]) || !(filp->f_inode))
      return -EBADF;
   if (origin > 2)
      return -EINVAL;
   if (filp->f_op && filp->f_op->lseek)
      return filp->f_op->lseek(filp->f_inode,filp,offset,origin);

   /*
    * This is the default handler if no lseek handler is present.
    */
   switch (origin) {
      case 0:
         tmp = offset;
         break;
      case 1:
         tmp = filp->f_pos + offset;
         break;
      case 2:
         if (!filp->f_inode)
            return -EINVAL;
         tmp = filp->f_inode->i_size + offset;
         break;
      default:
         break;
   }
   if (tmp < 0)
      return -EINVAL;
   filp->f_pos = tmp;
   filp->f_reada = 0;
   return filp->f_pos;
}

asmlinkage int sys_read(unsigned int fd, char *buf, unsigned int count)
{
   int error;
   struct file *filp;
   struct inode *inode;

   if (fd >= NR_OPEN || !(filp = current->filp[fd]) || !(inode = filp->f_inode))
      return -EBADF;
   if (!(filp->f_mode & 1))
      return -EBADF;
   if (!filp->f_op || !filp->f_op->read)
      return -EINVAL;
   if (!count)
      return 0;
   error = verify_area(VERIFY_WRITE, buf, count);
   if (error)
      return error;
   return filp->f_op->read(inode, filp, buf, count);
}

asmlinkage int sys_write(unsigned int fd, char *buf, unsigned int count)
{
   int error;
   struct file *filp;
   struct inode *inode;
   
   if (fd >= NR_OPEN || !(filp = current->filp[fd]) || !(inode = filp->f_inode))
      return -EBADF;
   if (!(filp->f_mode & 2))
      return -EBADF;
   if (!filp->f_op || !filp->f_op->write)
      return -EINVAL;
   if (!count)
      return 0;
   error = verify_area(VERIFY_READ,buf,count);
   if (error)
      return error;
   return vfs_write(inode, filp, buf, count);
}
