/* This file is part of the KDE project
   Copyright (C) 2002-2005 David Faure <faure@kde.org>
   Copyright (C) 2002-2004 Clarence Dang <dang@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "kofiledialog.h"
#include "koDocument.h"
#include <kfilefiltercombo.h>
#include <klocale.h>
#include <kdiroperator.h>
#include <kdebug.h>

KoFileDialog::KoFileDialog(const QString& startDir, const QString& filter,
                           QWidget *parent, const char *name,
                           bool modal)
    : KFileDialog( startDir, filter, parent, name, modal )
{
    connect( filterWidget, SIGNAL( activated( int) ),
             this, SLOT( slotChangedfilter( int ) ) );
}

void KoFileDialog::slotChangedfilter( int index )
{
    if ( index == KoDocument::SaveAsDirectoryStore ){
        ops->setMode(KFile::Directory);
    } else {
        ops->setMode(KFile::File);
    }
    updateAutoSelectExtension();
}

void KoFileDialog::setSpecialMimeFilter( QStringList& mimeFilter,
                                         const QString& currentFormat, const int specialOutputFlag,
                                         const QString& nativeFormat,
                                         int supportedSpecialFormats )
{
    Q_ASSERT( !mimeFilter.isEmpty() );
    Q_ASSERT( mimeFilter[0] == nativeFormat );

    bool add1dot1 = supportedSpecialFormats & KoDocument::SaveAsKOffice1dot1;
    bool addUncompressed = supportedSpecialFormats & KoDocument::SaveAsDirectoryStore;

    int idxSpecialOutputFlag = 0;
    int numSpecialEntries = 0;
    if ( add1dot1 ) {
        ++numSpecialEntries;
        m_specialFormats.append( KoDocument::SaveAsKOffice1dot1 );
        if ( specialOutputFlag == KoDocument::SaveAsKOffice1dot1 )
            idxSpecialOutputFlag = numSpecialEntries;
    }
    if ( addUncompressed ) {
        ++numSpecialEntries;
        m_specialFormats.append( KoDocument::SaveAsDirectoryStore );
        if ( specialOutputFlag == KoDocument::SaveAsDirectoryStore )
            idxSpecialOutputFlag = numSpecialEntries;
    }

    // Insert numSpecialEntries entries with native mimetypes, for the special entries.
    QStringList::Iterator mimeFilterIt = mimeFilter.at( 1 );
    mimeFilter.insert( mimeFilterIt /* before 1 -> after 0 */, numSpecialEntries, nativeFormat );

    // Fill in filter combo
    // Note: if currentFormat doesn't exist in mimeFilter, filterWidget
    //       will default to the first item (native format)
    setMimeFilter( mimeFilter, currentFormat.isEmpty() ? nativeFormat : currentFormat );

    // To get a different description in the combo, we need to change its entries afterwards
    KMimeType::Ptr type = KMimeType::mimeType( nativeFormat );
    int idx = 1; // 0 is the native format
    if ( add1dot1 )
        filterWidget->changeItem( i18n("%1 (KOffice-1.1 Format)").arg( type->comment() ), idx++ );
    if ( addUncompressed )
        filterWidget->changeItem( i18n("%1 (Uncompressed XML Files)").arg( type->comment() ), idx++ );
    // if you add an entry here, update numSpecialEntries above and specialEntrySelected() below

    // For native format...
    if (currentFormat == nativeFormat || currentFormat.isEmpty())
    {
        // KFileFilterCombo selected the _last_ "native mimetype" entry, select the correct one
        filterWidget->setCurrentItem( idxSpecialOutputFlag );
        slotChangedfilter( filterWidget->currentItem() );
    }
    // [Mainly KWord] Tell MS Office users that they can save in RTF!
    int i = 0;
    for (mimeFilterIt = mimeFilter.begin (); mimeFilterIt != mimeFilter.end (); ++mimeFilterIt, i++)
    {
        KMimeType::Ptr mime = KMimeType::mimeType (*mimeFilterIt);
        QString compatString = mime->property ("X-KDE-CompatibleApplication").toString ();
        if (!compatString.isEmpty ())
            filterWidget->changeItem (i18n ("%1 (%2 Compatible)").arg (mime->comment ()).arg (compatString), i);
    }
}

int KoFileDialog::specialEntrySelected()
{
    int i = filterWidget->currentItem();
    // Item 0 is the native format, the following ones are the special formats
    if ( i >= 1 && i <= (int)m_specialFormats.count() )
        return m_specialFormats[i-1];
    return 0;
}

#include "kofiledialog.moc"
