------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                N A M E T                                 --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.46 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Limits;  use Limits;
with System;  use System;
with Types;   use Types;
with Table;

package Namet is

--  WARNING: There is a C version of this package. Any changes to this
--  source file must be properly reflected in the C header file namet.h
--  which is created manually from namet.ads and namet.adb.

--  This package contains routines for handling the names table. The table
--  is used to store character strings for identifiers and operator symbols,
--  as well as other string values such as unit names and file names.

--  The forms of the entries are as follows:

--    Identifiers        Stored with upper case letters folded to lower case.
--                       Upper half (16#80# bit set) and wide characters are
--                       stored in an encoded form (Uhh for upper half and
--                       Whhhh for wide characters, as provided by the routine
--                       Store_Encoded_Character, where hh are hex digits for
--                       the character code using lower case a-f). Other
--                       internally generated names use upper case letters
--                       (other than O,Q,U,W) to ensure that they do not clash
--                       with identifier names in the source program. The body
--                       of this package contains a catalog of the use of these
--                       upper case letters.

--    Operator symbols   Stored with an initial letter O, and the remainder
--                       of the name is the lower case characters XXX where
--                       the name is Name_Op_XXX, see Snames spec for a full
--                       list of the operator names.

--    Character literals Character literals have names that are used only for
--                       debugging and error message purposes. The form is a
--                       upper case Q followed by a single letter, or by a Uxx
--                       or Wxxxx encoding as described for identifiers. The
--                       Set_Character_Literal_Name procedure should be used
--                       to construct these encodings.

--    Unit names         Stored with upper case letters folded to lower case,
--                       using Uhh/Whhhh encoding as described for identifiers,
--                       and a %s or %b suffix for specs/bodies. See package
--                       Uname for further details.

--    File names         Are stored in the form provided by Osint. Typically
--                       they may include wide character escape sequences and
--                       upper case characters (in non-encoded form). Casing
--                       is also derived from the external environment. Note
--                       that file names provided by Osint must generally be
--                       consistent with the names from Fname.Get_File_Name.

--    Other strings      The names table is also used as a convenient storage
--                       location for other variable length strings such as
--                       error messages etc. There are no restrictions on what
--                       characters may appear for such entries.

--  Note: the encodings Uhh (upper half characters), Whhhh (wide characters),
--  and Qx (character literal names) are described in the spec, since they
--  are visible throughout the system (e.g. in debugging output). However,
--  no code should depend on these particular encodings, so it should be
--  possible to change the encodings by making changes only to the Namet
--  specification (to change these comments) and the body (which actually
--  implements the encodings).

--  The names are hashed so that a given name appears only once in the table,
--  except that names entered with Name_Enter as opposed to Name_Find are
--  omitted from the hash table.

--  The first 26 entries in the names table (with Name_Id values in the range
--  First_Name_Id .. First_Name_Id + 25) represent names which are the one
--  character lower case letters in the range a-z, and these names are created
--  and initialized by the Initialize procedure.

--  Two values, one of type Int and one of type Byte, are stored with each
--  names table entry and subprograms are provided for setting and retrieving
--  these associated values. The usage of these values is up to the client.
--  In the compiler, the Int field is used to point to a chain of potentially
--  visible entities (see Sem.Ch8 for details), and the Byte field is used
--  to hold the Token_Type value for reserved words (see Sem for details).
--  In the binder, the Byte field is unused, and the Int field is used in
--  various ways depending on the name involved (see binder documentation).

   Name_Buffer : String (1 .. Max_Name_Length);
   --  This buffer is used to set the name to be stored in the table for the
   --  Name_Find call, and to retrieve the name for the Get_Name_String call.

   Name_Len : Natural;
   --  Length of name stored in Name_Buffer. Used as an input parameter for
   --  Name_Find, and as an output value by Get_Name_String, or Write_Name.

   -----------------
   -- Subprograms --
   -----------------

   procedure Finalize;
   --  Called at the end of a use of the Namet package (before a subsequent
   --  call to Initialize). Currently this routine is only used to generate
   --  debugging output.

   procedure Get_Name_String (Id : Name_Id);
   --  Get_Name_String is used to retrieve the string associated with an entry
   --  in the names table. The resulting string is stored in Name_Buffer
   --  and Name_Len is set. It is an error to call Get_Name_String with one
   --  of the special name Id values (No_Name, Error_Name, or Child_Name).

   procedure Get_Decoded_Name_String (Id : Name_Id);
   --  Same calling sequence an interface as Get_Name_String, except that the
   --  result is decoded, so that upper half characters and wide characters
   --  procedure decodes the name so that upper case characters and wide
   --  appear as originally found in the source program text, operators have
   --  their source forms (special characters and enclosed in quotes), and
   --  character literals appear surrounded by apostrophes.

   function Get_Name_Table_Byte (Id : Name_Id) return Byte;
   pragma Inline (Get_Name_Table_Byte);

   function Get_Name_Table_Info (Id : Name_Id) return Int;
   pragma Inline (Get_Name_Table_Info);
   --  Fetches the Int value associated with the given name
   --  Fetches the Byte value associated with the given name

   procedure Initialize;
   --  Initializes the names table, including initializing the first 26
   --  entries in the table (for the 1-character lower case names a-z)

   function Is_Internally_Generated_Name return Boolean;
   --  True for any name except an operator symbol name, a name of an entity
   --  in Standard, a character literal, or a program identifier, i.e. true
   --  for names generated by the expander and implicit types created by the
   --  semantic analyzer. The name to be checked is stored in Name_Buffer,
   --  with Name_Len indicating the length in encoded form (e.g. as provided
   --  by a call to Get_Name_String).

   function Length_Of_Name (Id : Name_Id) return Nat;
   pragma Inline (Length_Of_Name);
   --  Returns length of given name in characters (this is the length of the
   --  encoded name, as stored in the names table, the result is equivalent to
   --  calling Get_Name_String and reading Name_Len, except that a call to
   --  Length_Of_Name does not affect the contents of Name_Len and Name_Buffer.

   function Name_Chars_Address return System.Address;
   --  Return starting address of name characters table (used in Back_End
   --  call to Gigi).

   function Name_Enter return Name_Id;
   --  Name_Enter has the same calling interface as Name_Find. The difference
   --  is that it does not search the table for an existing match, and also
   --  subsequent Name_Find calls using the same name will not locate the
   --  entry created by this call. Thus multiple calls to Name_Enter with the
   --  same name will create multiple entries in the name table with different
   --  Name_Id values. This is useful in the case of created names, which are
   --  never expected to be looked up. Note: Name_Enter should never be used
   --  for one character names, since these are efficiently located without
   --  hashing by Name_Find in any case.

   function Name_Entries_Address return System.Address;
   --  Return starting address of names table. Used in Back_End call to Gigi.

   function Name_Entries_Count return Nat;
   --  Return current number of entries in the names table. Used in Back_End
   --  call to Gigi.

   function Name_Find return Name_Id;
   --  Name_Find is called with a string stored in Name_Buffer whose length
   --  is in Name_Len (i.e. the characters of the name are in subscript
   --  positions 1 to Name_Len in Name_Buffer). It searches the names
   --  table to see if the string has already been stored. If so the Id of
   --  the existing entry is returned. Otherwise a new entry is created with
   --  its Name_Table_Info field set to zero.

   function Is_OK_Internal_Letter (C : Character) return Boolean;
   pragma Inline (Is_OK_Internal_Letter);
   --  Returns true if C is a suitable character for using as a prefix or a
   --  suffix of an internally generated name, i.e. it is an upper case letter
   --  other than one of the ones used for encoding source names (currently
   --  the set of reserved letters is Q, U, W)

   procedure Reset_Name_Table;
   --  This procedure is used when there are multiple source files to reset
   --  the name table info entries associated with current entries in the
   --  names table. There is no harm in keeping the names entries themselves
   --  from one compilation to another, but we can't keep the entity info,
   --  since this refers to tree nodes, which are destroyed between each
   --  main source file.

   procedure Set_Character_Literal_Name (C : Char_Code);
   --  This procedure sets the proper encoded name for the character literal
   --  for the given character code. On return Name_Buffer and Name_Len are
   --  set to reflect the stored name.

   procedure Set_Name_Table_Info (Id : Name_Id; Val : Int);
   pragma Inline (Set_Name_Table_Info);
   --  Sets the Int value associated with the given name

   procedure Set_Name_Table_Byte (Id : Name_Id; Val : Byte);
   pragma Inline (Set_Name_Table_Byte);
   --  Sets the Byte value associated with the given name

   procedure Store_Encoded_Character (C : Char_Code);
   --  Stores given character code at the end of Name_Buffer, updating the
   --  value in Name_Len appropriately. Lower case letters and digits are
   --  stored unchanged. Other 8-bit characters are stored using the Uhh
   --  encoding (hh = hex code), and other 16-bit wide-character values
   --  are stored using the Whhhh (hhhh = hex code) encoding. Note that
   --  this procedure does not fold upper case letters (they are stored
   --  using the Uhh encoding). If folding is required, it must be done
   --  by the caller prior to the call.

   procedure Write_Name (Id : Name_Id);
   --  Write_Name writes the characters of the specified name using the
   --  standard output procedures in package Output. No end of line is
   --  written, just the characters of the name. On return Name_Buffer and
   --  Name_Len are set as for a call to Get_Name_String. The name is written
   --  in encoded form (i.e. including Uhh, Whhh, Qx, _op as they appear in
   --  the name table). If Id is Error_Name, or No_Name, no text is output.

   procedure Write_Name_Decoded (Id : Name_Id);
   --  Like Write_Name, except that the name written is the decoded name, as
   --  described for Name_Decode, and the resulting value stored in Name_Len
   --  and Name_Buffer is the decoded name.

end Namet;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.44
--  date: Sat Feb 12 13:28:59 1994;  author: dewar
--  Remove unused with's
--  ----------------------------
--  revision 1.45
--  date: Tue Jun  7 07:27:21 1994;  author: dewar
--  (Reset_Names_Table): New procedure
--  ----------------------------
--  revision 1.46
--  date: Tue Jul  5 19:00:09 1994;  author: dewar
--  Character literals encoding is slightly changed to avoid confusion between
--   upper case letters and encoding characters.
--  (Store_Encoded_Character): Document different encoding rules
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
