------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ A T T R                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.158 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Errout;   use Errout;
with Features; use Features;
with Namet;    use Namet;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Opt;      use Opt;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Ch4;  use Sem_Ch4;
with Sem_Ch5;  use Sem_Ch5;
with Sem_Ch6;  use Sem_Ch6;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Ch13; use Sem_Ch13;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Type; use Sem_Type;
with Sem_Util; use Sem_Util;
with Stand;    use Stand;
with Sinfo;    use Sinfo;
with Sinput;   use Sinput;
with Snames;   use Snames;
with Stand;
with Stringt;  use Stringt;
with Table;
with Ttypes;   use Ttypes;
with Ttypef;   use Ttypef;
with Tbuild;   use Tbuild;
with Uintp;    use Uintp;
with Urealp;   use Urealp;
with Widechar; use Widechar;

package body Sem_Attr is

   type Attribute_Class_Array is array (Attribute_Id) of Boolean;
   --  Type used to build attribute classification flag arrays

   Attribute_83 : Attribute_Class_Array := Attribute_Class_Array'(
      Attribute_Address           |
      Attribute_Aft               |
      Attribute_Alignment         |
      Attribute_Base              |
      Attribute_Callable          |
      Attribute_Constrained       |
      Attribute_Count             |
      Attribute_Delta             |
      Attribute_Digits            |
      Attribute_Emax              |
      Attribute_Epsilon           |
      Attribute_First             |
      Attribute_First_Bit         |
      Attribute_Fore              |
      Attribute_Image             |
      Attribute_Large             |
      Attribute_Last              |
      Attribute_Last_Bit          |
      Attribute_Leading_Part      |
      Attribute_Length            |
      Attribute_Machine_Emax      |
      Attribute_Machine_Emin      |
      Attribute_Machine_Mantissa  |
      Attribute_Machine_Overflows |
      Attribute_Machine_Radix     |
      Attribute_Machine_Rounds    |
      Attribute_Mantissa          |
      Attribute_Pos               |
      Attribute_Position          |
      Attribute_Pred              |
      Attribute_Range             |
      Attribute_Safe_Emax         |
      Attribute_Safe_Large        |
      Attribute_Safe_Small        |
      Attribute_Size              |
      Attribute_Small             |
      Attribute_Storage_Size      |
      Attribute_Succ              |
      Attribute_Terminated        |
      Attribute_Val               |
      Attribute_Value             |
      Attribute_Width             => True,
      others                      => False);
   --  Array indicating attributes defined in the Ada 83 RM

   Attribute_Impl_Def : Attribute_Class_Array := Attribute_Class_Array'(
      Attribute_Abort_Signal             |
      Attribute_Address_Size             |
      Attribute_Enum_Rep                 |
      Attribute_Img                      |
      Attribute_Max_Interrupt_Priority   |
      Attribute_Max_Priority             |
      Attribute_Passed_By_Reference      |
      Attribute_Storage_Unit             |
      Attribute_Tick                     |
      Attribute_Universal_Literal_String |
      Attribute_Unrestricted_Access      |
      Attribute_Word_Size                => True,
      others                             => False);
   --  Array indicating GNAT implementation dependent attributes

   Bad_Attribute : exception;
   --  Exception raised if an error is detected during attribute processing,
   --  used so that we can abandon the processing so we don't run into
   --  trouble with cascaded errors.

   -----------------------
   -- Analyze_Attribute --
   -----------------------

   procedure Analyze_Attribute (N : Node_Id) is
      Loc         : constant Source_Ptr   := Sloc (N);
      Aname       : constant Name_Id      := Attribute_Name (N);
      P           : constant Node_Id      := Prefix (N);
      Exprs       : constant List_Id      := Expressions (N);
      Attr_Id     : constant Attribute_Id := Get_Attribute_Id (Aname);
      E1          : Node_Id;
      E2          : Node_Id;

      P_Type : Entity_Id;
      --  Type of prefix after analysis

      P_Base_Type : Entity_Id;
      --  Base type of prefix after analysis

      P_Root_Type : Entity_Id;
      --  Root type of prefix after analysis

      -----------------------
      -- Local Subprograms --
      -----------------------

      procedure Access_Attribute;
      --  Used for Access, Unchecked_Access, Unrestricted_Access attributes.
      --  Internally, Id distinguishes which of the three cases is involved.

      procedure Check_Array_Or_Scalar_Type;
      --  Common procedure used by First, Last, Range attribute to check
      --  that the prefix is a constrained array or scalar type, or a name
      --  of an array object, and that an argument appears only if appropriate
      --  (i.e. only in the array case).

      procedure Check_Array_Type;
      --  Common semantic checks for all array attributes. Checks that the
      --  prefix is a constrained array type or the name of an array object.

      procedure Check_Decimal_Fixed_Point_Type;
      --  Check that prefix of attribute N is a decimal fixed-point type

      procedure Check_Discrete_Attribute;
      --  Common processing for attributes operating on discrete types

      procedure Check_Discrete_Type;
      --  Verify that prefix of attribute N is a discrete type

      procedure Check_E0;
      --  Check that no attribute arguments are present

      procedure Check_E0_Or_E1;
      --  Check that at most one attribute argument is present

      procedure Check_E1;
      --  Check that exactly one attribute argument is present

      procedure Check_E2;
      --  Check that two attribute arguments are present

      procedure Check_Enumeration_Type;
      --  Verify that prefix of attribute N is an enumeration type

      procedure Check_Fixed_Point_Type;
      --  Verify that prefix of attribute N is a fixed type

      procedure Check_Fixed_Point_Type_0;
      --  Verify that prefix of attribute N is a fixed type and that
      --  no attribute expressions are present

      procedure Check_Floating_Point_Type;
      --  Verify that prefix of attribute N is a float type

      procedure Check_Floating_Point_Type_0;
      --  Verify that prefix of attribute N is a float type and that
      --  no attribute expressions are present

      procedure Check_Floating_Point_Type_1;
      --  Verify that prefix of attribute N is a float type and that
      --  exactly one attribute expression is present

      procedure Check_Floating_Point_Type_2;
      --  Verify that prefix of attribute N is a float type and that
      --  two attribute expressions are present

      procedure Check_Real_Type;
      --  Verify that prefix of attribute N is fixed or float type

      procedure Check_Scalar_Type;
      --  Verify that prefix of attribute N is a scalar type

      procedure Check_Standard_Prefix;
      --  Verify that prefix of attribute N is package Standard

      procedure Check_Task_Prefix;
      --  Verify that prefix of attribute N is a task or task type

      procedure Check_Type;
      --  Verify that the prefix of attribute N is a type

      procedure Error_Attr (Msg : String; Error_Node : Node_Id);
      --  Posts error using Error_Msg_N at given node, sets type of attribute
      --  node to Any_Type, and then raises Bad_Attribute to avoid any further
      --  semantic processing.

      procedure Standard_Attribute (Val : Int);
      --  Used to process attributes whose prefix is package Standard which
      --  yield values of type Universal_Integer. The attribute reference
      --  node is rewritten with an integer literal of the given value.

      procedure Unexpected_Argument (En : Node_Id);
      --  Signal unexpected attribute argument (En is the argument)

      procedure Unimplemented_Attribute;
      --  Give error message for unimplemented attribute

      procedure Validate_Non_Static_Attribute_Function_Call;
      --  Called when processing an attribute that is a function call to a
      --  non-static function, i.e. an attribute function that either takes
      --  non-scalar arguments or returns a non-scalar result. Verifies that
      --  such a call does not appear in a preelaborable context.

      ----------------------
      -- Access_Attribute --
      ----------------------

      procedure Access_Attribute is
         Index    : Interp_Index;
         It       : Interp;
         Acc_Type : Entity_Id;

         function Valid_Aliased_View (Obj : Node_Id) return Boolean is
            E : Entity_Id;

         begin
            if Is_Entity_Name (Obj) then
               E := Entity (Obj);

               return Is_Aliased (E)
                 or else (Present (Renamed_Object (E))
                           and then Valid_Aliased_View (Renamed_Object (E)))

                 or else ((Ekind (E) = E_In_Out_Parameter
                             or else Ekind (E) = E_Generic_In_Out_Parameter)
                           and then Is_Tagged_Type (Etype (E)))

                 or else ((Ekind (E) = E_Task_Type
                             or else Ekind (E) = E_Protected_Type)
                           and then In_Open_Scopes (E))

                  --  Access discriminant constraint

                 or else (Is_Type (E) and then E = Current_Scope)
                 or else (Is_Incomplete_Or_Private_Type (E)
                           and then Full_View (E) = Current_Scope);


            elsif Nkind (Obj) = N_Selected_Component then
               return Is_Aliased (Entity (Selector_Name (Obj)));

            elsif Nkind (Obj) = N_Indexed_Component then
               return (Is_Aliased (Etype (Prefix (Obj)))
                 or else Is_Access_Type (Etype (Prefix (Obj))));

            elsif Nkind (Obj) = N_Unchecked_Type_Conversion
              or else Nkind (Obj) = N_Type_Conversion
            then
               return Is_Tagged_Type (Etype (Obj));

            elsif Nkind (Obj) = N_Explicit_Dereference then
               return True;  --  more precise test needed???

            else
               return False;
            end if;
         end Valid_Aliased_View;

      --  Start of processing for Access_Attribute

      begin
         Check_E0;

         --  In the case of an access to subprogram, use the name of the
         --  subprogram itself as the designated type. Type-checking in
         --  this case compares the signatures of the designated types.

         if Is_Entity_Name (P)
           and then Is_Overloadable (Entity (P))
         then
            if not Is_Overloaded (P) then
               Acc_Type :=
                 New_Internal_Entity
                   (E_Access_Subprogram_Type, Current_Scope, Loc, 'A');
               Set_Etype (Acc_Type,  Acc_Type);
               Set_Directly_Designated_Type (Acc_Type, Entity (P));
               Set_Etype (N, Acc_Type);

            else
               Get_First_Interp (P, Index, It);
               Set_Etype (N, Any_Type);

               while Present (It.Nam) loop
                  Acc_Type :=
                    New_Internal_Entity
                      (E_Access_Subprogram_Type, Current_Scope, Loc, 'A');
                  Set_Etype (Acc_Type,  Acc_Type);
                  Set_Directly_Designated_Type (Acc_Type, It.Nam);
                  Add_One_Interp (N,  Acc_Type,  Acc_Type);
                  Get_Next_Interp (Index, It);
               end loop;
            end if;

         elsif (Nkind (P) = N_Selected_Component
          and then Is_Overloadable (Entity (Selector_Name (P))))
         then
            Unimplemented (N,  "access to protected operations");

         --  Case of access to object

         else
            Acc_Type :=
              New_Internal_Entity (E_Allocator_Type, Current_Scope, Loc, 'A');
            Set_Etype (Acc_Type,  Acc_Type);
            Set_Directly_Designated_Type (Acc_Type, P_Type);
            Set_Etype (N, Acc_Type);

            --  Check for aliased view unless unrestricted case

            if Aname /= Name_Unrestricted_Access
              and then not Valid_Aliased_View (P)
            then
               Error_Attr ("prefix of % attribute must be aliased view", P);
            end if;
         end if;

      end Access_Attribute;

      --------------------------------
      -- Check_Array_Or_Scalar_Type --
      --------------------------------

      procedure Check_Array_Or_Scalar_Type is
         Index_Type : Entity_Id;

         D : Int;
         --  Dimension number for array attributes.

      begin
         if Is_Scalar_Type (P_Type) then
            Check_Type;

            if Present (E1) then
               Error_Attr ("invalid argument in % attribute", E1);
            else
               Set_Etype (N, Base_Type (P_Type));
               return;
            end if;

         else
            Check_Array_Type;

            --  We know prefix is an array type, or the name of an array
            --  object, and that the expression, if present, is static
            --  and within the range of the dimensions of the type.

            if Is_Array_Type (P_Type) then
               Index_Type := First_Index (P_Type);

            elsif Is_Access_Type (P_Type) then
               Index_Type := First_Index (Designated_Type (P_Type));
            end if;

            if No (E1) then

               --  First dimension assumed

               Set_Etype (N, Etype (Index_Type));

            else
               D := UI_To_Int (Intval (E1));

               for I in 1 .. D - 1 loop
                  Index_Type := Next_Index (Index_Type);
               end loop;

               Set_Etype (N, Etype (Index_Type));
               Set_Etype (E1, Standard_Integer);
            end if;
         end if;
      end Check_Array_Or_Scalar_Type;

      ----------------------
      -- Check_Array_Type --
      ----------------------

      procedure Check_Array_Type is
         D : Int;
         --  Dimension number for array attributes.

      begin
         Check_E0_Or_E1;

         if Is_Array_Type (P_Type) then
            if not Is_Constrained (P_Type)
              and then Is_Entity_Name (P)
              and then Is_Type (Entity (P))
            then
               Error_Attr
                 ("prefix for % attribute must be constrained array", P);
            end if;

            D := Number_Dimensions (P_Type);

         elsif Is_Access_Type (P_Type)
           and then Is_Array_Type (Designated_Type (P_Type))
         then
            if Is_Entity_Name (P) and then Is_Type (Entity (P)) then
               Error_Attr ("prefix of % attribute cannot be access type", P);
            end if;

            D := Number_Dimensions (Designated_Type (P_Type));

         else
            Error_Attr ("prefix for % attribute must be array", P);
         end if;

         if Present (E1) then
            Resolve (E1, Any_Integer);
            Set_Etype (E1, Standard_Integer);

            if not Is_Static_Expression (E1) then
               Error_Attr ("expression for dimension must be static", E1);

            elsif  UI_To_Int (Intval (E1)) > D
              or else UI_To_Int (Intval (E1)) < 1
            then
               Error_Attr ("invalid dimension number for array type", E1);
            end if;
         end if;
      end Check_Array_Type;

      ------------------------------------
      -- Check_Decimal_Fixed_Point_Type --
      ------------------------------------

      procedure Check_Decimal_Fixed_Point_Type is
      begin
         Check_Type;

         if not Is_Decimal_Fixed_Point_Type (P_Type) then
            Error_Attr
              ("prefix of % attribute must be decimal type", P);
         end if;
      end Check_Decimal_Fixed_Point_Type;

      ------------------------------
      -- Check_Discrete_Attribute --
      ------------------------------

      procedure Check_Discrete_Attribute is
      begin
         Check_Discrete_Type;
         Check_E1;
         Resolve (E1, P_Type);
      end Check_Discrete_Attribute;

      -------------------------
      -- Check_Discrete_Type --
      -------------------------

      procedure Check_Discrete_Type is
      begin
         Check_Type;

         if not Is_Discrete_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be discrete type", P);
         end if;
      end Check_Discrete_Type;

      --------------
      -- Check_E0 --
      --------------

      procedure Check_E0 is
      begin
         if Present (E1) then
            Unexpected_Argument (E1);
         end if;
      end Check_E0;

      --------------------
      -- Check_E0_Or_E1 --
      --------------------

      procedure Check_E0_Or_E1 is
      begin
         if Present (E2) then
            Unexpected_Argument (E2);
         end if;
      end Check_E0_Or_E1;

      --------------
      -- Check_E1 --
      --------------

      procedure Check_E1 is
      begin
         Check_E0_Or_E1;

         if No (E1) then
            Error_Attr ("missing argument for % attribute", N);
         end if;
      end Check_E1;

      --------------
      -- Check_E2 --
      --------------

      procedure Check_E2 is
      begin
         if No (E1) then
            Error_Attr ("missing arguments for % attribute (2 required)", N);
         elsif No (E2) then
            Error_Attr ("missing argument for % attribute (2 required)", N);
         end if;
      end Check_E2;

      ----------------------------
      -- Check_Enumeration_Type --
      ----------------------------

      procedure Check_Enumeration_Type is
      begin
         Check_Type;

         if not Is_Enumeration_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be enumeration type", P);
         end if;
      end Check_Enumeration_Type;

      ----------------------------
      -- Check_Fixed_Point_Type --
      ----------------------------

      procedure Check_Fixed_Point_Type is
      begin
         Check_Type;

         if not Is_Fixed_Point_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be fixed point type", P);
         end if;
      end Check_Fixed_Point_Type;

      ------------------------------
      -- Check_Fixed_Point_Type_0 --
      ------------------------------

      procedure Check_Fixed_Point_Type_0 is
      begin
         Check_Fixed_Point_Type;
         Check_E0;
      end Check_Fixed_Point_Type_0;

      -------------------------------
      -- Check_Floating_Point_Type --
      -------------------------------

      procedure Check_Floating_Point_Type is
      begin
         Check_Type;

         if not Is_Floating_Point_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be float type", P);
         end if;
      end Check_Floating_Point_Type;

      ---------------------------------
      -- Check_Floating_Point_Type_0 --
      ---------------------------------

      procedure Check_Floating_Point_Type_0 is
      begin
         Check_Floating_Point_Type;
         Check_E0;
      end Check_Floating_Point_Type_0;

      ---------------------------------
      -- Check_Floating_Point_Type_1 --
      ---------------------------------

      procedure Check_Floating_Point_Type_1 is
      begin
         Check_Floating_Point_Type;
         Check_E1;
      end Check_Floating_Point_Type_1;

      ---------------------------------
      -- Check_Floating_Point_Type_2 --
      ---------------------------------

      procedure Check_Floating_Point_Type_2 is
      begin
         Check_Floating_Point_Type;
         Check_E2;
      end Check_Floating_Point_Type_2;

      ---------------------
      -- Check_Real_Type --
      ---------------------

      procedure Check_Real_Type is
      begin
         Check_Type;

         if not Is_Real_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be real type", P);
         end if;
      end Check_Real_Type;

      -----------------------
      -- Check_Scalar_Type --
      -----------------------

      procedure Check_Scalar_Type is
      begin
         Check_Type;

         if not Is_Scalar_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be scalar type", P);
         end if;
      end Check_Scalar_Type;

      ---------------------------
      -- Check_Standard_Prefix --
      ---------------------------

      procedure Check_Standard_Prefix is
      begin
         Check_E0;

         if Nkind (P) /= N_Identifier or else Chars (P) /= Name_Standard then
            Error_Attr ("only allowed prefix for % attribute is Standard", P);
         end if;
      end Check_Standard_Prefix;

      -----------------------
      -- Check_Task_Prefix --
      -----------------------

      procedure Check_Task_Prefix is
      begin
         Analyze (P);

         if Is_Task_Type (Etype (P))
           or else (Is_Access_Type (Etype (P))
              and then Is_Task_Type (Designated_Type (Etype (P))))
         then
            Resolve (P, Etype (P));
         else
            Error_Attr ("prefix of % attribute must be a task", P);
         end if;
      end Check_Task_Prefix;

      ----------------
      -- Check_Type --
      ----------------

      --  The possibilities are an entity name denoting a type, or an
      --  attribute reference that denotes a type (Base or Class)

      procedure Check_Type is
      begin
         if not Is_Entity_Name (P)
           or else not Is_Type (Entity (P))
         then
            Error_Attr ("prefix of % attribute must be a type", P);
         end if;
      end Check_Type;

      ----------------
      -- Error_Attr --
      ----------------

      procedure Error_Attr (Msg : String; Error_Node : Node_Id) is
      begin
         Error_Msg_N (Msg, Error_Node);
         Set_Etype (N, Any_Type);
         Set_Entity (N, Any_Type);
         raise Bad_Attribute;
      end Error_Attr;

      ------------------------
      -- Standard_Attribute --
      ------------------------

      procedure Standard_Attribute (Val : Int) is
      begin
         Check_Standard_Prefix;
         Rewrite_Substitute_Tree (N,
           Make_Integer_Literal (Loc, UI_From_Int (Val)));
         Analyze (N);
      end Standard_Attribute;

      -------------------------
      -- Unexpected Argument --
      -------------------------

      procedure Unexpected_Argument (En : Node_Id) is
      begin
         Error_Attr ("unexpected argument for % attribute", En);
      end Unexpected_Argument;

      -----------------------------
      -- Unimplemented_Attribute --
      -----------------------------

      procedure Unimplemented_Attribute is
      begin
         Error_Attr ("% attribute not implemented yet", N);
      end Unimplemented_Attribute;

      -------------------------------------------------
      -- Validate_Non_Static_Attribute_Function_Call --
      -------------------------------------------------

      procedure Validate_Non_Static_Attribute_Function_Call is
      begin
         if Inside_Preelaborated_Unit (N)
           and then not Inside_Subprogram_Unit (N)
         then
            Error_Msg_N ("?non-static function call in preelaborated unit", N);
         end if;
      end Validate_Non_Static_Attribute_Function_Call;

   -----------------------
   -- Analyze_Attribute --
   -----------------------

   begin
      Error_Msg_Name_1 := Aname;

      --  Immediate return if unrecognized attribute (already diagnosed
      --  by parser, so there is nothing more that we need to do)

      if not Is_Attribute_Name (Aname) then
         raise Bad_Attribute;
      end if;

      --  Deal with Ada 83 and Featues issues

      if not Attribute_83 (Attr_Id) then
         if Ada_83 then
            Error_Msg_N ("(Ada 83) attribute% is not recognized", N);
         end if;

         if Attribute_Impl_Def (Attr_Id) then
            Note_Feature (Implementation_Dependent_Attributes, Loc);
         else
            Note_Feature (New_Attributes, Loc);
         end if;
      end if;

      --  Analyze prefix and exit if error in analysis

      Analyze (P);
      P_Type := Etype (P);

      if P_Type = Any_Type then
         raise Bad_Attribute;
      end if;

      P_Base_Type := Base_Type (P_Type);
      P_Root_Type := Root_Type (P_Base_Type);

      --  Analyze expressions that may be present, exiting if an error occurs

      if No (Exprs) then
         E1 := Empty;
         E2 := Empty;

      else
         E1 := First (Exprs);
         Analyze (E1);

         if Etype (E1) = Any_Type then
            raise Bad_Attribute;
         end if;

         E2 := Next (E1);

         if Present (E2) then
            Analyze (E2);

            if Etype (E2) = Any_Type then
               raise Bad_Attribute;
            end if;

            if Present (Next (E2)) then
               Unexpected_Argument (Next (E2));
            end if;
         end if;
      end if;

      if Is_Overloaded (P)
        and then Aname /= Name_Access
        and then Aname /= Name_Address
        and then Aname /= Name_Unchecked_Access
      then
         Error_Attr ("ambiguous prefix for % attribute", P);
      end if;

      Error_Msg_Name_1 := Aname;

      case Attr_Id is

      ------------------
      -- Abort_Signal --
      ------------------

      when Attribute_Abort_Signal =>
         Check_Standard_Prefix;
         Rewrite_Substitute_Tree (N,
           New_Reference_To (Stand.Abort_Signal, Loc));
         Analyze (N);

      ------------
      -- Access --
      ------------

      when Attribute_Access =>
         Access_Attribute;

      -------------
      -- Address --
      -------------

      when Attribute_Address =>
         Check_E0;

         if Is_Entity_Name (P) and then Is_Type (Entity (P)) then
            Error_Attr ("prefix of % attribute cannot be a type", P);
         end if;

         Set_Etype (N, RTE (RE_Address));

      ------------------
      -- Address_Size --
      ------------------

      when Attribute_Address_Size =>
         Standard_Attribute (Ttypes.System_Address_Size);

      --------------
      -- Adjacent --
      --------------

      when Attribute_Adjacent =>
         Check_Floating_Point_Type_2;
         Unimplemented_Attribute;

      ---------
      -- Aft --
      ---------

      when Attribute_Aft =>
         Check_Fixed_Point_Type_0;
         Set_Etype (N, Universal_Integer);

      ---------------
      -- Alignment --
      ---------------

      when Attribute_Alignment =>
         Check_E0;
         Unimplemented_Attribute;

      ----------
      -- Base --
      ----------

      when Attribute_Base => Base :
      begin
         Check_E0_Or_E1;
         Find_Type (P);
         Set_Etype (N, Base_Type (Entity (P)));

         if Present (Exprs) then

            --  Attribute is the subtype mark of a conversion.

            declare
               New_N : Node_Id;

            begin
               New_N :=
                 Make_Type_Conversion (Loc,
                   Subtype_Mark => New_Occurrence_Of (Etype (N), Loc),
                   Expression => New_Copy (E1));
               Rewrite_Substitute_Tree (N,  New_N);
               Analyze (N);
            end;

         else
            Set_Entity (N, Base_Type (Entity (P)));
         end if;
      end Base;

      ---------------
      -- Bit_Order --
      ---------------

      when Attribute_Bit_Order =>
         Check_E0;
         Unimplemented_Attribute;

      ------------------
      -- Body_Version --
      ------------------

      when Attribute_Body_Version =>
         Check_E0;
         Unimplemented_Attribute;

      --------------
      -- Callable --
      --------------

      when Attribute_Callable =>
         Check_E0;
         Set_Etype (N, Standard_Boolean);
         Check_Task_Prefix;

      ------------
      -- Caller --
      ------------

      when Attribute_Caller =>
         Check_E0;
         Unimplemented_Attribute;

      -------------
      -- Ceiling --
      -------------

      when Attribute_Ceiling =>
         Check_Floating_Point_Type_1;
         Unimplemented_Attribute;

      -----------
      -- Class --
      -----------

      when Attribute_Class => Class :
      begin
         Note_Feature (Class_Wide_Types, Loc);
         Check_E0_Or_E1;
         Find_Type (N);

         if Present (E1) then

            --  This is a conversion not an attribute : T'Class (X)

            Rewrite_Substitute_Tree (N, Make_Type_Conversion (Loc,
              Subtype_Mark => New_Occurrence_Of (Etype (N), Loc),
              Expression => New_Copy (E1)));

            Analyze (N);
         end if;

      end Class;

      --------------------
      -- Component_Size --
      --------------------

      when Attribute_Component_Size =>
         Check_E0;
         Set_Etype (N, Universal_Integer);

         --  Note: unlike other array attributes, unconstrained arrays are OK

         if Is_Array_Type (P_Type) and then not Is_Constrained (P_Type) then
            null;
         else
            Check_Array_Type;
         end if;

      -------------
      -- Compose --
      -------------

      when Attribute_Compose =>
         Check_Floating_Point_Type_2;
         Unimplemented_Attribute;

      -----------------
      -- Constrained --
      -----------------

      when Attribute_Constrained =>
         Check_E0;
         Set_Etype (N, Standard_Boolean);
         Unimplemented_Attribute;

      ---------------
      -- Copy_Sign --
      ---------------

      when Attribute_Copy_Sign =>
         Check_Floating_Point_Type_2;
         Unimplemented_Attribute;

      -----------
      -- Count --
      -----------

      when Attribute_Count => Count :
      declare
         Ent : Entity_Id;
         S   : Entity_Id;

      begin
         Check_E0;

         if Nkind (P) = N_Identifier
           or else Nkind (P) = N_Expanded_Name
         then
            Ent := Entity (P);

            if Ekind (Ent) /= E_Entry then
               Error_Attr ("invalid entry name",  N);
            end if;

         elsif Nkind (P) = N_Indexed_Component then
            Ent := Entity (Prefix (P));

            if Ekind (Ent) /= E_Entry_Family then
               Error_Attr ("invalid entry family name",  P);
               return;
            end if;

         else
            Error_Attr ("invalid entry name",  N);
            return;
         end if;

         for J in reverse 0 .. Scope_Stack.Last loop
            S := Scope_Stack.Table (J).Entity;

            if S = Scope (Ent) then
               exit;

            elsif Ekind (Scope (Ent)) in Task_Kind
              and then Ekind (S) /= E_Loop
              and then Ekind (S) /= E_Block
              and then Ekind (S) /= E_Entry
              and then Ekind (S) /= E_Entry_Family
            then
               Error_Attr ("Count cannot appear in inner unit", N);
            end if;
         end loop;

         Set_Etype (N, Universal_Integer);
      end Count;

      --------------
      -- Definite --
      --------------

      when Attribute_Definite =>
         Check_E0;
         Unimplemented_Attribute;

      -----------
      -- Delta --
      -----------

      when Attribute_Delta =>
         Check_Fixed_Point_Type_0;
         Set_Etype (N, Universal_Real);

      ------------
      -- Denorm --
      ------------

      when Attribute_Denorm =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Standard_Boolean);

      ------------
      -- Digits --
      ------------

      when Attribute_Digits =>
         Check_E0;
         Check_Type;

         if not Is_Floating_Point_Type (P_Type)
           and then not Is_Decimal_Fixed_Point_Type (P_Type)
         then
            Error_Attr
              ("prefix of % attribute must be float or decimal type", P);
         end if;

         Set_Etype (N, Universal_Integer);

      ----------
      -- Emax --
      ----------

      when Attribute_Emax =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Integer);

      --------------
      -- Enum_Rep --
      --------------

      when Attribute_Enum_Rep => Enum_Rep : declare
      begin
         Check_E1;
         Check_Enumeration_Type;
         Resolve (E1, Base_Type (P_Type));
         Set_Etype (N, Universal_Integer);
      end Enum_Rep;

      -------------
      -- Epsilon --
      -------------

      when Attribute_Epsilon =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Real);

      --------------
      -- Exponent --
      --------------

      when Attribute_Exponent =>
         Check_Floating_Point_Type_1;
         Unimplemented_Attribute;

      ------------------
      -- External_Tag --
      ------------------

      when Attribute_External_Tag =>
         Check_E0;
         Unimplemented_Attribute;

      -----------
      -- First --
      -----------

      when Attribute_First =>
         Check_Array_Or_Scalar_Type;

      ---------------
      -- First_Bit --
      ---------------

      when Attribute_First_Bit =>
         Check_E0;
         Set_Etype (N, Universal_Integer);
         Unimplemented_Attribute;

      -----------
      -- Floor --
      -----------

      when Attribute_Floor =>
         Check_Floating_Point_Type_1;
         Unimplemented_Attribute;

      ----------
      -- Fore --
      ----------

      when Attribute_Fore =>
         Check_Fixed_Point_Type_0;
         Set_Etype (N, Universal_Integer);

      --------------
      -- Fraction --
      --------------

      when Attribute_Fraction =>
         Check_Floating_Point_Type_1;
         Unimplemented_Attribute;

      ------------------
      -- Huge_Integer --
      ------------------

      when Attribute_Huge_Integer =>
         Check_Standard_Prefix;
         Rewrite_Substitute_Tree (N, New_Occurrence_Of (Huge_Integer, Loc));

      --------------
      -- Identity --
      --------------

      when Attribute_Identity =>
         Check_E0;
         Unimplemented_Attribute;

      -----------
      -- Image --
      -----------

      when Attribute_Image => Image :
      begin
         Set_Etype (N, Standard_String);

         if Is_Real_Type (P_Type) then
            Check_Type;
            Note_Feature (Image_Attribute_For_Real, Loc);

            if Ada_83 then
               Error_Msg_N
                 ("(Ada 83) % attribute not allowed for real types", N);
            end if;
         else
            Check_Discrete_Attribute;
         end if;
      end Image;

      ---------
      -- Img --
      ---------

      when Attribute_Img => Img :
      begin
         Set_Etype (N, Standard_String);

         --  Must be scalar type

         if Is_Scalar_Type (P_Type) then

            --  Variable is OK

            if Is_Variable (P) then
               return;

            --  So is constant (or in parameter)

            elsif Is_Entity_Name (P) then
               if Ekind (Entity (P)) = E_Constant
                 or else Ekind (Entity (P)) = E_In_Parameter
               then
                  return;
               end if;
            end if;
         end if;

         --  Fall through on error

         Error_Attr ("prefix of % attribute must be scalar object name", N);
      end Img;

      -----------
      -- Input --
      -----------

      when Attribute_Input =>
         Check_E2;
         Validate_Non_Static_Attribute_Function_Call;
         Unimplemented_Attribute;

      -----------
      -- Large --
      -----------

      when Attribute_Large =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Real);

      ----------
      -- Last --
      ----------

      when Attribute_Last =>
         Check_Array_Or_Scalar_Type;

      --------------
      -- Last_Bit --
      --------------

      when Attribute_Last_Bit =>
         Check_E0;
         Set_Etype (N, Universal_Integer);
         Unimplemented_Attribute;

      ------------------
      -- Leading_Part --
      ------------------

      when Attribute_Leading_Part =>
         Check_Floating_Point_Type_2;
         Unimplemented_Attribute;

      ------------
      -- Length --
      ------------

      when Attribute_Length =>
         Check_Array_Type;
         Set_Etype (N, Universal_Integer);

      -------------
      -- Machine --
      -------------

      when Attribute_Machine =>
         Check_Floating_Point_Type_1;
         Unimplemented_Attribute;

      ------------------
      -- Machine_Emax --
      ------------------

      when Attribute_Machine_Emax =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Integer);

      ------------------
      -- Machine_Emin --
      ------------------

      when Attribute_Machine_Emin =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Integer);

      ----------------------
      -- Machine_Mantissa --
      ----------------------

      when Attribute_Machine_Mantissa =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Integer);

      -----------------------
      -- Machine_Overflows --
      -----------------------

      when Attribute_Machine_Overflows =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Standard_Boolean);

      -------------------
      -- Machine_Radix --
      -------------------

      when Attribute_Machine_Radix =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Integer);

      --------------------
      -- Machine_Rounds --
      --------------------

      when Attribute_Machine_Rounds =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Standard_Boolean);

      --------------
      -- Mantissa --
      --------------

      when Attribute_Mantissa =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Integer);

      ---------
      -- Max --
      ---------

      when Attribute_Max =>
         Check_E2;
         Check_Scalar_Type;
         Resolve (E1, Base_Type (P_Type));
         Resolve (E2, Base_Type (P_Type));
         Set_Etype (N, Base_Type (P_Type));

      ----------------------------
      -- Max_Interrupt_Priority --
      ----------------------------

      when Attribute_Max_Interrupt_Priority =>
         Standard_Attribute (Ttypes.System_Max_Interrupt_Priority);

      ------------------
      -- Max_Priority --
      ------------------

      when Attribute_Max_Priority =>
         Standard_Attribute (Ttypes.System_Max_Priority);

      ----------------------------------
      -- Max_Size_In_Storage_Elements --
      ----------------------------------

      when Attribute_Max_Size_In_Storage_Elements =>
         Check_E0;
         Unimplemented_Attribute;

      ---------
      -- Min --
      ---------

      when Attribute_Min =>
         Check_E2;
         Check_Scalar_Type;
         Resolve (E1, Base_Type (P_Type));
         Resolve (E2, Base_Type (P_Type));
         Set_Etype (N, Base_Type (P_Type));

      -----------
      -- Model --
      -----------

      when Attribute_Model =>
         Check_Floating_Point_Type_1;
         Unimplemented_Attribute;

      ----------------
      -- Model_Emin --
      ----------------

      when Attribute_Model_Emin =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Integer);

      -------------------
      -- Model_Epsilon --
      -------------------

      when Attribute_Model_Epsilon =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Real);

      --------------------
      -- Model_Mantissa --
      --------------------

      when Attribute_Model_Mantissa =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Integer);

      -----------------
      -- Model_Small --
      -----------------

      when Attribute_Model_Small =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Real);

      -------------
      -- Modulus --
      -------------

      when Attribute_Modulus =>
         Check_Type;

         if not Is_Modular_Integer_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be modular type", P);
         end if;

      ------------
      -- Output --
      ------------

      when Attribute_Output =>
         Check_E2;
         Validate_Non_Static_Attribute_Function_Call;
         Unimplemented_Attribute;

      ------------------
      -- Partition_Id --
      ------------------

      when Attribute_Partition_Id => Partition_Id :
      begin
         Check_E0;

         if P_Type /= Any_Type then
            if not Is_Library_Level_Entity (Entity (P)) then
               Error_Attr
                 ("prefix of % attribute must be library-level entity", P);

            elsif Inside_Pure_Unit (Entity (P)) then
               Error_Attr
                 ("prefix of % attribute must not be declared pure", P);
            end if;
         end if;

         --  For the moment we have only one possible value, zero, and we
         --  set it here, being sure to reset the potentially static flag
         --  on the literal, since the result of this attribute cannot be
         --  static, since it is not a scalar attribute.

         Rewrite_Substitute_Tree (N, Make_Integer_Literal (Loc, Uint_0));
         Analyze (N);
         Set_Potentially_Static (N, False);

      end Partition_Id;

      -------------------------
      -- Passed_By_Reference --
      -------------------------

      when Attribute_Passed_By_Reference =>
         Check_E0;
         Check_Type;
         Set_Etype (N, Standard_Boolean);

      ---------
      -- Pos --
      ---------

      when Attribute_Pos =>
         Check_Discrete_Attribute;
         Set_Etype (N, Universal_Integer);

      --------------
      -- Position --
      --------------

      when Attribute_Position =>
         Check_E0;
         Set_Etype (N, Universal_Integer);

      ----------
      -- Pred --
      ----------

      when Attribute_Pred =>
         Check_Scalar_Type;
         Check_E1;
         Resolve (E1, P_Type);
         Set_Etype (N, Base_Type (P_Type));

         if Is_Floating_Point_Type (P_Type) then
            Error_Attr ("% attribute not implemented for float types", N);
         end if;

      ---------------------
      -- Range_Attribute --
      ---------------------

      when Attribute_Range =>
         Check_Array_Or_Scalar_Type;

      ----------
      -- Read --
      ----------

      when Attribute_Read =>
         Check_E2;
         Validate_Non_Static_Attribute_Function_Call;
         Unimplemented_Attribute;

      ---------------
      -- Remainder --
      ---------------

      when Attribute_Remainder =>
         Check_Floating_Point_Type_2;
         Unimplemented_Attribute;

      -----------
      -- Round --
      -----------

      when Attribute_Round =>
         Check_E1;
         Check_Decimal_Fixed_Point_Type;
         Unimplemented_Attribute;

      --------------
      -- Rounding --
      --------------

      when Attribute_Rounding =>
         Check_Floating_Point_Type_1;
         Unimplemented_Attribute;

      ---------------
      -- Safe_Emax --
      ---------------

      when Attribute_Safe_Emax =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Integer);

      ----------------
      -- Safe_First --
      ----------------

      when Attribute_Safe_First =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Real);

      ----------------
      -- Safe_Large --
      ----------------

      when Attribute_Safe_Large =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Real);

      ---------------
      -- Safe_Last --
      ---------------

      when Attribute_Safe_Last =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Real);

      ----------------
      -- Safe_Small --
      ----------------

      when Attribute_Safe_Small =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Universal_Real);

      -----------
      -- Scale --
      -----------

      when Attribute_Scale =>
         Check_E0;
         Check_Decimal_Fixed_Point_Type;

      -------------
      -- Scaling --
      -------------

      when Attribute_Scaling =>
         Check_Floating_Point_Type_2;
         Unimplemented_Attribute;

      ------------------
      -- Signed_Zeros --
      ------------------

      when Attribute_Signed_Zeros =>
         Check_Floating_Point_Type_0;
         Set_Etype (N, Standard_Boolean);

      ----------
      -- Size --
      ----------

      when Attribute_Size =>
         Check_E0;
         Set_Etype (N, Universal_Integer);

      -----------
      -- Small --
      -----------

      when Attribute_Small =>
         Check_Fixed_Point_Type_0;
         Set_Etype (N, Universal_Real);

      ------------------
      -- Storage_Pool --
      ------------------

      when Attribute_Storage_Pool =>
         Check_E0;
         Unimplemented_Attribute;

      ------------------
      -- Storage_Size --
      ------------------

      when Attribute_Storage_Size =>

         if Is_Task_Type (P_Type) then
            Check_E0;
            Set_Etype (N, Universal_Integer);

         elsif Is_Access_Type (P_Type) then
            Check_Type;
            Unimplemented_Attribute;

         else
            Error_Attr
              ("prefix of % attribute must be access or task type", P);
         end if;

      ------------------
      -- Storage_Unit --
      ------------------

      when Attribute_Storage_Unit =>
         Standard_Attribute (Ttypes.System_Storage_Unit);

      ----------
      -- Succ --
      ----------

      when Attribute_Succ =>
         Check_Scalar_Type;
         Check_E1;
         Resolve (E1, P_Type);
         Set_Etype (N, Base_Type (P_Type));

         if Is_Floating_Point_Type (P_Type) then
            Error_Attr ("% attribute not implemented for float types", N);
         end if;

      ---------
      -- Tag --
      ---------

      when Attribute_Tag =>
         Check_E0;
         if not Is_Tagged_Type (P_Type) then
            Error_Attr ("prefix of % attribute must be tagged", P);
         end if;

         Set_Etype (N, RTE (RE_Tag));

      ----------------
      -- Terminated --
      ----------------

      when Attribute_Terminated =>
         Check_E0;
         Set_Etype (N, Standard_Boolean);
         Check_Task_Prefix;

      ----------
      -- Tick --
      ----------

      when Attribute_Tick =>
         Check_Standard_Prefix;
         Rewrite_Substitute_Tree (N,
           Make_Real_Literal (Loc,
             UR_From_Components (
               Num   => UI_From_Int (Ttypes.System_Tick_Nanoseconds),
               Den   => UI_From_Int (9),
               Rbase => 10)));
         Analyze (N);

      ----------------
      -- Truncation --
      ----------------

      when Attribute_Truncation =>
         Check_Floating_Point_Type_1;
         Resolve (E1, P_Base_Type);
         Set_Etype (N, P_Base_Type);

      -----------------------
      -- Unbiased_Rounding --
      -----------------------

      when Attribute_Unbiased_Rounding =>
         Check_Floating_Point_Type_1;
         Unimplemented_Attribute;

      ----------------------
      -- Unchecked_Access --
      ----------------------

      when Attribute_Unchecked_Access =>
         Access_Attribute;

      ------------------------------
      -- Universal_Literal_String --
      ------------------------------

      --  This is a GNAT specific attribute whose prefix must be a named
      --  number where the expression is either a single numeric literal,
      --  or a numeric literal immediately preceded by a minus sign. The
      --  result is equivalent to a string literal containing the text of
      --  the literal as it appeared in the source program with a possible
      --  leading minus sign.

      when Attribute_Universal_Literal_String => Universal_Literal_String :
      begin
         Check_E0;

         if not Is_Entity_Name (P)
           or else Ekind (Entity (P)) not in Named_Kind
         then
            Error_Attr ("prefix for % attribute must be named number", P);

         else
            declare
               Expr     : Node_Id;
               Negative : Boolean;
               S        : Source_Ptr;
               Src      : Source_Buffer_Ptr;

            begin
               Expr := Original_Node (Expression (Parent (Entity (P))));

               if Nkind (Expr) = N_Op_Minus then
                  Negative := True;
                  Expr := Original_Node (Right_Opnd (Expr));
               else
                  Negative := False;
               end if;

               if Nkind (Expr) /= N_Integer_Literal
                 and then Nkind (Expr) /= N_Real_Literal
               then
                  Error_Attr
                    ("named number for % attribute must be simple literal", N);
               end if;

               --  Build string literal corresponding to source literal text

               Start_String;

               if Negative then
                  Store_String_Char (Get_Char_Code ('-'));
               end if;

               S := Sloc (Expr);
               Src := Source_Text (Get_Source_File_Index (S));

               while Src (S) /= ';' and then Src (S) /= ' ' loop
                  Store_String_Char (Get_Char_Code (Src (S)));
                  S := S + 1;
               end loop;

               --  Now we rewrite the attribute with the string literal

               Rewrite_Substitute_Tree (N,
                 Make_String_Literal (Loc, End_String));
               Analyze (N);
            end;
         end if;
      end Universal_Literal_String;

      -------------------------
      -- Unrestricted_Access --
      -------------------------

      --  This is a GNAT specific attribute which is like Access except that
      --  all scope checks and checks for aliased views are omitted.

      when Attribute_Unrestricted_Access =>
         Access_Attribute;

      ---------
      -- Val --
      ---------

      when Attribute_Val => Val : declare
      begin
         Check_E1;
         Check_Discrete_Type;

         if not Is_Integer_Type (Etype (E1)) then
            Error_Attr ("argument of % attribute is not integer type", N);

         else
            Resolve (E1, Etype (E1));
         end if;

         Set_Etype (N, P_Type);
      end Val;

      -----------
      -- Valid --
      -----------

      when Attribute_Valid =>
         Check_E0;
         Unimplemented_Attribute;

      -----------
      -- Value --
      -----------

      when Attribute_Value => Value :
      begin
         Check_E1;
         Check_Scalar_Type;

         if Is_Floating_Point_Type (P_Type) then
            Note_Feature (Value_Attribute_For_Real, Loc);
         end if;

         Resolve (E1, Standard_String);
         Set_Etype (N, P_Type);
         Validate_Non_Static_Attribute_Function_Call;
      end Value;

      -------------
      -- Version --
      -------------

      when Attribute_Version =>
         Check_E0;
         Unimplemented_Attribute;

      ----------------
      -- Wide_Image --
      ----------------

      when Attribute_Wide_Image => Wide_Image :
      begin
         Set_Etype (N, Standard_Wide_String);

         if Is_Real_Type (P_Type) then
            Check_Type;
         else
            Check_Discrete_Attribute;
         end if;

         Validate_Non_Static_Attribute_Function_Call;
      end Wide_Image;

      ----------------
      -- Wide_Value --
      ----------------

      when Attribute_Wide_Value => Wide_Value :
      begin
         Check_E1;
         Check_Discrete_Type;
         Resolve (E1, Standard_Wide_String);
         Set_Etype (N, P_Type);

         if Is_Modular_Integer_Type (P_Type)
           or else Is_Real_Type (P_Type)
         then
            Unimplemented_Attribute;
         end if;

         Validate_Non_Static_Attribute_Function_Call;
      end Wide_Value;

      -----------
      -- Width --
      -----------

      when Attribute_Width =>
         Check_E0;
         Check_Scalar_Type;
         Set_Etype (N, Universal_Integer);

      ---------------
      -- Word_Size --
      ---------------

      when Attribute_Word_Size =>
         Standard_Attribute (System_Word_Size);

      -----------
      -- Write --
      -----------

      when Attribute_Write =>
         Check_E2;
         Validate_Non_Static_Attribute_Function_Call;
         Unimplemented_Attribute;

      end case;

   --  All errors raise Bad_Attribute, so that we get out before any further
   --  damage occurs when an error is detected (for example, if we check for
   --  one attribute expression, and the check succeeds, we want to be able
   --  to proceed securely assuming that an expression is in fact present.

   exception
      when Bad_Attribute =>
         Set_Etype (N, Any_Type);
         return;

   end Analyze_Attribute;

   --------------------
   -- Eval_Attribute --
   --------------------

   procedure Eval_Attribute (N : Node_Id) is
      Aname  : constant Name_Id      := Attribute_Name (N);
      Id     : constant Attribute_Id := Get_Attribute_Id (Aname);
      P      : constant Node_Id      := Prefix (N);

      E1 : Node_Id;
      --  First expression, or Empty if none

      E2 : Node_Id;
      --  Second expression, or Empty if none

      P_Entity : Entity_Id;
      --  Entity denoted by prefix

      P_Type : Entity_Id;
      --  The type of the prefix

      P_Root_Type : Entity_Id;
      --  The root type of type of the prefix

      Static : Boolean;
      --  True if prefix type is static

      Lo_Bound, Hi_Bound : Node_Id;
      --  Expressions for low and high bounds of type or array index referenced
      --  by First, Last, or Length attribute for array, set by Set_Bounds.

      function Aft_Value return Nat;
      --  Computes Aft value for current attribute prefix (used by Aft itself
      --  and also by Width for computing the Width of a fixed point type).

      procedure Check_Expressions;
      --  In case where the attribute is not foldable, the expressions, if
      --  any, of the attribute, are in a non-static context. This procedure
      --  performs the required additional checks.

      procedure Float_Attribute_Boolean
        (Short_Float_Val     : Boolean;
         Float_Val           : Boolean;
         Long_Float_Val      : Boolean;
         Long_Long_Float_Val : Boolean);
      --  This procedure evaluates a float attribute with no arguments that
      --  returns a Boolean result. The four parameters are the Boolean result
      --  values for the four possible floating-point root types. The prefix
      --  type is a floating-point type (and is thus not a generic type).

      procedure Float_Attribute_Universal_Integer
        (Short_Float_Val     : Int;
         Float_Val           : Int;
         Long_Float_Val      : Int;
         Long_Long_Float_Val : Int);
      --  This procedure evaluates a float attribute with no arguments that
      --  returns a universal integer result. All such results are easily
      --  within Int range, and the four parameters are the result values
      --  for the four possible floating-point root types. The prefix type
      --  is a floating-point type (and is thus not a generic type).

      procedure Float_Attribute_Universal_Real
        (Short_Float_Val     : String;
         Float_Val           : String;
         Long_Float_Val      : String;
         Long_Long_Float_Val : String);
      --  This procedure evaluates a float attribute with no arguments that
      --  returns a universal real result. The four parameters are strings
      --  that contain representations of the values required in normal
      --  real literal format with a possible leading minus sign. The prefix
      --  type is a floating-point type (and is thus not a generic type)

      function Fore_Value return Nat;
      --  Computes the Fore value for the current attribute prefix, which is
      --  known to be a static fixed-point type. Used by Fore and Width.

      procedure Set_Bounds;
      --  Used for First, Last and Length attributes applied to an array or
      --  array subtype. Sets the variables Index_Lo and Index_Hi to the low
      --  and high bound expressions for the index referenced by the attribute
      --  designator (i.e. the first index if no expression is present, and
      --  the N'th index if the value N is present as an expression).

      procedure Unimplemented_Attribute;
      --  Called if an attribute is unimplemented, which should never happen,
      --  since in this case, the unimplemented error message should have been
      --  signalled by Analyze_Attribute, causing the type to be Error_Type.

      ---------------
      -- Aft_Value --
      ---------------

      function Aft_Value return Nat is
         Result    : Nat;
         Delta_Val : Ureal;

      begin
         Result := 1;
         Delta_Val := Delta_Value (P_Type);

         while UR_Lt (Delta_Val, Ureal_Tenth) loop
            Delta_Val := UR_Product (Delta_Val, Ureal_10);
            Result := Result + 1;
         end loop;

         return Result;
      end Aft_Value;

      -----------------------
      -- Check_Expressions --
      -----------------------

      procedure Check_Expressions is
         E : Node_Id := E1;

      begin
         while Present (E) loop
            Check_Non_Static_Context (E);
            E := Next (E);
         end loop;
      end Check_Expressions;

      -----------------------------
      -- Float_Attribute_Boolean --
      -----------------------------

      procedure Float_Attribute_Boolean
        (Short_Float_Val     : Boolean;
         Float_Val           : Boolean;
         Long_Float_Val      : Boolean;
         Long_Long_Float_Val : Boolean)
      is
         Val    : Boolean;
         Result : Node_Id;

      begin
         if P_Root_Type = Standard_Short_Float then
            Val := Short_Float_Val;
         elsif P_Root_Type = Standard_Float then
            Val := Float_Val;
         elsif P_Root_Type = Standard_Long_Float then
            Val := Long_Float_Val;
         else
            pragma Assert (P_Root_Type = Standard_Long_Long_Float);
            Val := Long_Long_Float_Val;
         end if;

         if Val then
            Fold_Uint (N, Uint_1);
         else
            Fold_Uint (N, Uint_0);
         end if;
      end Float_Attribute_Boolean;

      ---------------------------------------
      -- Float_Attribute_Universal_Integer --
      ---------------------------------------

      procedure Float_Attribute_Universal_Integer
        (Short_Float_Val     : Int;
         Float_Val           : Int;
         Long_Float_Val      : Int;
         Long_Long_Float_Val : Int)
      is
         Val : Int;

      begin
         if P_Root_Type = Standard_Short_Float then
            Val := Short_Float_Val;
         elsif P_Root_Type = Standard_Float then
            Val := Float_Val;
         elsif P_Root_Type = Standard_Long_Float then
            Val := Long_Float_Val;
         else
            pragma Assert (P_Root_Type = Standard_Long_Long_Float);
            Val := Long_Long_Float_Val;
         end if;

         Fold_Uint (N, UI_From_Int (Val));
      end Float_Attribute_Universal_Integer;

      ------------------------------------
      -- Float_Attribute_Universal_Real --
      ------------------------------------

      procedure Float_Attribute_Universal_Real
        (Short_Float_Val     : String;
         Float_Val           : String;
         Long_Float_Val      : String;
         Long_Long_Float_Val : String)
      is
         Result : Node_Id;

      begin
         if P_Root_Type = Standard_Short_Float then
            Result := Real_Convert (Short_Float_Val);
         elsif P_Root_Type = Standard_Float then
            Result := Real_Convert (Float_Val);
         elsif P_Root_Type = Standard_Long_Float then
            Result := Real_Convert (Long_Float_Val);
         else
            pragma Assert (P_Root_Type = Standard_Long_Long_Float);
            Result := Real_Convert (Long_Long_Float_Val);
         end if;

         Rewrite_Substitute_Tree (N, Result);
         Analyze (N);
      end Float_Attribute_Universal_Real;

      ----------------
      -- Fore_Value --
      ----------------

      --  Note that the Fore calculation is based on the actual values
      --  of the bounds, and does not take into account possible rounding.

      function Fore_Value return Nat is
         Lo      : constant Uint  := Expr_Value (Type_Low_Bound (P_Type));
         Hi      : constant Uint  := Expr_Value (Type_High_Bound (P_Type));
         Small   : constant Ureal := Small_Value (P_Type);
         Lo_Real : constant Ureal := UR_Product (UR_From_Uint (Lo), Small);
         Hi_Real : constant Ureal := UR_Product (UR_From_Uint (Hi), Small);
         T       : Ureal;
         R       : Nat;

      begin
         --  Bounds are given in terms of small units, so first compute
         --  proper values as reals.

         T := UR_Max (UR_Abs (Lo_Real), UR_Abs (Hi_Real));
         R := 2;

         --  Loop to compute proper value if more than one digit required

         while UR_Ge (T, Ureal_10) loop
            R := R + 1;
            T := UR_Quotient (T, Ureal_10);
         end loop;

         return R;
      end Fore_Value;

      ----------------
      -- Set_Bounds --
      ----------------

      procedure Set_Bounds is
         N    : Nat;
         Indx : Node_Id;
         Ityp : Entity_Id;

      begin
         --  For non-array case, just get bounds of scalar type

         if Is_Scalar_Type (P_Type) then
            Ityp := P_Type;

         --  For array case, get type of proper index

         else
            if No (E1) then
               N := 1;
            else
               N := UI_To_Int (Expr_Value (E1));
            end if;

            Indx := First_Index (P_Type);
            while N > 1 loop
               Indx := Next_Index (Indx);
               N := N - 1;
            end loop;

            Ityp := Etype (Indx);
         end if;

         Lo_Bound := Type_Low_Bound (Ityp);
         Hi_Bound := Type_High_Bound (Ityp);

      end Set_Bounds;

      -----------------------------
      -- Unimplemented_Attribute --
      -----------------------------

      procedure Unimplemented_Attribute is
      begin
         pragma Assert (False);
         null;
      end Unimplemented_Attribute;

   --------------------
   -- Eval_Attribute --
   --------------------

   begin
      --  Acquire first two expressions (at the moment, no attributes
      --  take more than two expressions in any case).

      if Present (Expressions (N)) then
         E1 := First (Expressions (N));
         E2 := Next (E1);
      else
         E1 := Empty;
         E2 := Empty;
      end if;

      --  Attribute definitely is not foldable if prefix is not an entity

      if not Is_Entity_Name (P) then
         Check_Expressions;
         return;
      else
         P_Entity := Entity (P);
      end if;

      --  First foldable possibility is a scalar or array type (RM 4.9(7))
      --  that is not generic (generic types are eliminated by RM 4.9(25)).
      --  Note we allow non-static non-generic types at this stage as further
      --  described below.

      if Is_Type (P_Entity)
        and then (Is_Scalar_Type (P_Entity) or Is_Array_Type (P_Entity))
        and then (not Is_Generic_Type (P_Entity))
      then
         P_Type := P_Entity;

      --  Second foldable possibility is an array object (RM 4.9(8))

      elsif (Ekind (P_Entity) = E_Variable
               or else Ekind (P_Entity) = E_Constant)
        and then Is_Array_Type (Etype (P_Entity))
        and then (not Is_Generic_Type (P_Entity))
      then
         P_Type := Etype (P_Entity);

      --  No other cases are foldable (they certainly aren't static, and at
      --  the moment we don't try to fold any cases other than the two above)

      else
         Check_Expressions;
         return;
      end if;

      --  Scalar subtype case. We have not yet enforced the static requirement
      --  of (RM 4.9(7)) and we don't intend to just yet, since there are cases
      --  of non-static attribute references (e.g. S'Digits for a non-static
      --  floating-point type, which we can compute at compile time).

      --  Note: this folding of non-static attributes is not simply a case of
      --  optimization. For many of the attributes affected, Gigi cannot handle
      --  the attribute and depends on the front end having folded them away.

      --  Note: although we don't require staticness at this stage, we do set
      --  the Static variable to record the staticness, for easy reference by
      --  those attributes where it matters (e.g. Succ and Pred), and also to
      --  be used to ensure that non-static folded things are not marked as
      --  being potentially static (a check that is done right at the end).

      P_Root_Type := Root_Type (P_Type);

      if Is_Scalar_Type (P_Type) then
         Static := Is_Static_Subtype (P_Type);

      --  Array case. We enforce the constrained requirement of (RM 4.9(7-8))
      --  since we can't do anything with unconstrained arrays. In addition,
      --  only the First, Last and Length attributes are foldable.

      else
         if not Is_Constrained (P_Type)
           or else (Id /= Attribute_First
                     and then Id /= Attribute_Last
                     and then Id /= Attribute_Length)
         then
            Check_Expressions;
            return;
         end if;

         --  The rules in (RM 4.9(7,8)) require a static array, but as in the
         --  scalar case, we hold off on enforcing staticness, since there are
         --  cases which we can fold at compile time even though they are not
         --  static (e.g. 'Length applied to a static index, even though other
         --  non-static indexes make the array type non-static). This is only
         --  ab optimization, but it falls out essentially free, so why not.
         --  Again we compute the variable Static for easy reference later
         --  (note that no array attributes are static in Ada 83).

         Static := Ada_9X;

         declare
            N : Node_Id;

         begin
            N := First_Index (P_Type);
            while Present (N) loop
               Static := Static and Is_Static_Subtype (Etype (N));
               N := Next_Index (N);
            end loop;
         end;
      end if;

      --  Check any expressions that are present. Note that these expressions,
      --  depending on the particular attribute type, are either part of the
      --  attribute designator, or they are arguments in a case where the
      --  attribute reference returns a function. In the latter case, the
      --  rule in (RM 4.9(22)) applies and in particular requires the type
      --  of the expressions to be scalar in order for the attribute to be
      --  considered to be potentially static.

      declare
         E : Node_Id;

      begin
         E := E1;
         while Present (E) loop

            --  If expression is not potentially static, then the attribute
            --  reference certainly is neither foldable nor potentially
            --  static, so we can quit immediately. We can also quit if
            --  the expression is not of a scalar type as noted above.

            if not Potentially_Static (E)
              or else not Is_Scalar_Type (Etype (E))
            then
               Check_Expressions;
               return;

            --  If the expression raises a constraint error, then so does
            --  the attribute reference. We keep going in this case because
            --  we are still interested in whether the attribute reference
            --  is potentially static even if it is not static.

            elsif Raises_Constraint_Error (E) then
               Set_Raises_Constraint_Error (N);
            end if;

            E := Next (E);
         end loop;
      end;

      --  Deal with the case of a potentially static attribute reference
      --  that raises constraint error. The Raises_Constraint_Error flag
      --  will already have been set, and the Static flag shows whether
      --  the attribute reference is potentially static. In any case we
      --  certainly can't fold such an attribute reference.

      if Raises_Constraint_Error (N) then
         Set_Potentially_Static (N, Static);
         Check_Expressions;
         return;
      end if;

      --  At this point we have a potentially foldable attribute reference.
      --  If Static is set, then the attribute reference definitely obeys
      --  the requirements in (RM 4.9(7,8,22)), and it definitely can be
      --  folded. If Static is not set, then the attribute may or may not
      --  be foldable, and the individual attribute processing routines
      --  test Static as required in cases where it makes a difference.

      case Id is

      --------------
      -- Adjacent --
      --------------

      when Attribute_Adjacent =>
         Unimplemented_Attribute;

      ---------
      -- Aft --
      ---------

      when Attribute_Aft =>
         Fold_Uint (N, UI_From_Int (Aft_Value));

      ---------------
      -- Alignment --
      ---------------

      when Attribute_Alignment =>
         Unimplemented_Attribute;

      -------------
      -- Ceiling --
      -------------

      when Attribute_Ceiling =>
         Unimplemented_Attribute;

      -------------
      -- Compose --
      -------------

      when Attribute_Compose =>
         Unimplemented_Attribute;

      ---------------
      -- Copy_Sign --
      ---------------

      when Attribute_Copy_Sign => Copy_Sign :
      declare
         Val : constant Ureal := Expr_Value (E1);
         Num : Uint           := UI_Abs (Numerator (Val));
         Den : constant Uint  := Denominator (Val);
         Bas : constant Nat   := Rbase (Val);

      begin
         if UR_Is_Negative (Expr_Value (E2)) then
            Num := UI_Negate (Num);
         end if;

         Fold_Ureal (N, UR_From_Components (Num, Den, Bas));
      end Copy_Sign;

      -----------
      -- Delta --
      -----------

      when Attribute_Delta =>
         Fold_Ureal (N, Delta_Value (P_Type));

      ------------
      -- Denorm --
      ------------

      when Attribute_Denorm =>
         Float_Attribute_Boolean (
           Short_Float_Attr_Denorm,
           Float_Attr_Denorm,
           Long_Float_Attr_Denorm,
           Long_Long_Float_Attr_Denorm);

      ------------
      -- Digits --
      ------------

      when Attribute_Digits =>
         Fold_Uint (N, Digits_Value (P_Type));

      ----------
      -- Emax --
      ----------

      when Attribute_Emax =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Emax,
           Float_Attr_Emax,
           Long_Float_Attr_Emax,
           Long_Long_Float_Attr_Emax);

      -------------
      -- Epsilon --
      -------------

      when Attribute_Epsilon =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Epsilon'Universal_Literal_String,
           Float_Attr_Epsilon'Universal_Literal_String,
           Long_Float_Attr_Epsilon'Universal_Literal_String,
           Long_Long_Float_Attr_Epsilon'Universal_Literal_String);

      --------------
      -- Exponent --
      --------------

      when Attribute_Exponent =>
         Unimplemented_Attribute;

      -----------
      -- First --
      -----------

      when Attribute_First => First_Attr :
      begin
         Set_Bounds;

         if Is_Static_Expression (Lo_Bound) then
            if Is_Real_Type (P_Type) then
               Fold_Ureal (N, Expr_Value (Lo_Bound));
            else
               Fold_Uint  (N, Expr_Value (Lo_Bound));
            end if;
         end if;
      end First_Attr;

      -----------
      -- Floor --
      -----------

      when Attribute_Floor =>
         Unimplemented_Attribute;

      ----------
      -- Fore --
      ----------

      when Attribute_Fore =>
         if Static then
            Fold_Uint (N, UI_From_Int (Fore_Value));
         end if;

      --------------
      -- Fraction --
      --------------

      when Attribute_Fraction =>
         Unimplemented_Attribute;

      -----------
      -- Image --
      -----------

      --  Image is a scalar attribute, but is never static, because it is
      --  not a static function (having a non-scalar argument (RM 4.9(22))

      when Attribute_Image =>
         null;

      ---------
      -- Img --
      ---------

      --  Img is a scalar attribute, but is never static, because it is
      --  not a static function (having a non-scalar argument (RM 4.9(22))

      when Attribute_Img =>
         null;

      -----------
      -- Large --
      -----------

      when Attribute_Large =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Large'Universal_Literal_String,
           Float_Attr_Large'Universal_Literal_String,
           Long_Float_Attr_Large'Universal_Literal_String,
           Long_Long_Float_Attr_Large'Universal_Literal_String);

      ----------
      -- Last --
      ----------

      when Attribute_Last => Last :
      begin
         Set_Bounds;

         if Is_Static_Expression (Hi_Bound) then
            if Is_Real_Type (P_Type) then
               Fold_Ureal (N, Expr_Value (Hi_Bound));
            else
               Fold_Uint  (N, Expr_Value (Hi_Bound));
            end if;
         end if;
      end Last;

      ------------
      -- Length --
      ------------

      when Attribute_Length => Length :
      begin
         Set_Bounds;

         if Is_Static_Expression (Lo_Bound)
           and then Is_Static_Expression (Hi_Bound)
         then
            Fold_Uint (N,
              UI_Max (
                Uint_0,
                UI_Sum (
                  Uint_1,
                  UI_Difference (
                    Expr_Value (Hi_Bound),
                    Expr_Value (Lo_Bound)))));
         end if;
      end Length;

      -------------
      -- Machine --
      -------------

      when Attribute_Machine =>
         Unimplemented_Attribute;

      ------------------
      -- Machine_Emax --
      ------------------

      when Attribute_Machine_Emax =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Machine_Emax,
           Float_Attr_Machine_Emax,
           Long_Float_Attr_Machine_Emax,
           Long_Long_Float_Attr_Machine_Emax);

      ------------------
      -- Machine_Emin --
      ------------------

      when Attribute_Machine_Emin =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Machine_Emin,
           Float_Attr_Machine_Emin,
           Long_Float_Attr_Machine_Emin,
           Long_Long_Float_Attr_Machine_Emin);

      ----------------------
      -- Machine_Mantissa --
      ----------------------

      when Attribute_Machine_Mantissa =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Machine_Mantissa,
           Float_Attr_Machine_Mantissa,
           Long_Float_Attr_Machine_Mantissa,
           Long_Long_Float_Attr_Machine_Mantissa);

      -----------------------
      -- Machine_Overflows --
      -----------------------

      when Attribute_Machine_Overflows =>
         Float_Attribute_Boolean (
           Short_Float_Attr_Machine_Overflows,
           Float_Attr_Machine_Overflows,
           Long_Float_Attr_Machine_Overflows,
           Long_Long_Float_Attr_Machine_Overflows);

      -------------------
      -- Machine_Radix --
      -------------------

      when Attribute_Machine_Radix =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Machine_Radix,
           Float_Attr_Machine_Radix,
           Long_Float_Attr_Machine_Radix,
           Long_Long_Float_Attr_Machine_Radix);

      --------------------
      -- Machine_Rounds --
      --------------------

      when Attribute_Machine_Rounds =>
         Float_Attribute_Boolean (
           Short_Float_Attr_Machine_Rounds,
           Float_Attr_Machine_Rounds,
           Long_Float_Attr_Machine_Rounds,
           Long_Long_Float_Attr_Machine_Rounds);

      --------------
      -- Mantissa --
      --------------

      when Attribute_Mantissa =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Mantissa,
           Float_Attr_Mantissa,
           Long_Float_Attr_Mantissa,
           Long_Long_Float_Attr_Mantissa);

      ---------
      -- Max --
      ---------

      when Attribute_Max => Max :
      begin
         if Is_Real_Type (P_Type) then
            Fold_Ureal (N, UR_Max (Expr_Value (E1), Expr_Value (E2)));
         else
            Fold_Uint  (N, UI_Max (Expr_Value (E1), Expr_Value (E2)));
         end if;
      end Max;

      ----------------------------------
      -- Max_Size_In_Storage_Elements --
      ----------------------------------

      when Attribute_Max_Size_In_Storage_Elements =>
         Unimplemented_Attribute;

      ---------
      -- Min --
      ---------

      when Attribute_Min => Min :
      begin
         if Is_Real_Type (P_Type) then
            Fold_Ureal (N, UR_Min (Expr_Value (E1), Expr_Value (E2)));
         else
            Fold_Uint (N, UI_Min (Expr_Value (E1), Expr_Value (E2)));
         end if;
      end Min;

      -----------
      -- Model --
      -----------

      when Attribute_Model =>
         Unimplemented_Attribute;

      ----------------
      -- Model_Emin --
      ----------------

      when Attribute_Model_Emin =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Model_Emin,
           Float_Attr_Model_Emin,
           Long_Float_Attr_Model_Emin,
           Long_Long_Float_Attr_Model_Emin);

      -------------------
      -- Model_Epsilon --
      -------------------

      when Attribute_Model_Epsilon =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Model_Epsilon'Universal_Literal_String,
           Float_Attr_Model_Epsilon'Universal_Literal_String,
           Long_Float_Attr_Model_Epsilon'Universal_Literal_String,
           Long_Long_Float_Attr_Model_Epsilon'Universal_Literal_String);

      --------------------
      -- Model_Mantissa --
      --------------------

      when Attribute_Model_Mantissa =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Model_Mantissa,
           Float_Attr_Model_Mantissa,
           Long_Float_Attr_Model_Mantissa,
           Long_Long_Float_Attr_Model_Mantissa);

      -----------------
      -- Model_Small --
      -----------------

      when Attribute_Model_Small =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Model_Small'Universal_Literal_String,
           Float_Attr_Model_Small'Universal_Literal_String,
           Long_Float_Attr_Model_Small'Universal_Literal_String,
           Long_Long_Float_Attr_Model_Small'Universal_Literal_String);

      -------------
      -- Modulus --
      -------------

      when Attribute_Modulus =>
         Fold_Uint (N, Modulus (P_Type));

      -------------------------
      -- Passed_By_Reference --
      -------------------------

      --  Scalar types are never passed by reference

      when Attribute_Passed_By_Reference =>
         Fold_Uint (N, Uint_0);

      ---------
      -- Pos --
      ---------

      when Attribute_Pos =>
         Fold_Uint (N, Expr_Value (E1));

      ----------
      -- Pred --
      ----------

      when Attribute_Pred => Pred :
      begin
         if Static then
            if UI_Eq (Expr_Value (E1),
              Expr_Value (Type_Low_Bound (Base_Type (P_Type))))
            then
               Constraint_Error_Warning (N, "Pred of type'First?!");
               Set_Potentially_Static (N, True);
               Check_Expressions;
               return;
            else
               Fold_Uint (N, UI_Difference (Expr_Value (E1), Uint_1));
            end if;
         end if;
      end Pred;

      -----------
      -- Range --
      -----------

      --  No processing required, because by this stage, Range has been
      --  replaced by First .. Last, so this branch can never be taken.

      when Attribute_Range =>
         pragma Assert (False); null;

      ---------------
      -- Remainder --
      ---------------

      when Attribute_Remainder =>
         Unimplemented_Attribute;

      -----------
      -- Round --
      -----------

      when Attribute_Round =>
         Unimplemented_Attribute;

      --------------
      -- Rounding --
      --------------

      when Attribute_Rounding =>
         Unimplemented_Attribute;

      ---------------
      -- Safe_Emax --
      ---------------

      when Attribute_Safe_Emax =>
         Float_Attribute_Universal_Integer (
           Short_Float_Attr_Safe_Emax,
           Float_Attr_Safe_Emax,
           Long_Float_Attr_Safe_Emax,
           Long_Long_Float_Attr_Safe_Emax);

      ----------------
      -- Safe_First --
      ----------------

      when Attribute_Safe_First =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Safe_First'Universal_Literal_String,
           Float_Attr_Safe_First'Universal_Literal_String,
           Long_Float_Attr_Safe_First'Universal_Literal_String,
           Long_Long_Float_Attr_Safe_First'Universal_Literal_String);

      ----------------
      -- Safe_Large --
      ----------------

      when Attribute_Safe_Large =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Safe_Large'Universal_Literal_String,
           Float_Attr_Safe_Large'Universal_Literal_String,
           Long_Float_Attr_Safe_Large'Universal_Literal_String,
           Long_Long_Float_Attr_Safe_Large'Universal_Literal_String);

      ---------------
      -- Safe_Last --
      ---------------

      when Attribute_Safe_Last =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Safe_Last'Universal_Literal_String,
           Float_Attr_Safe_Last'Universal_Literal_String,
           Long_Float_Attr_Safe_Last'Universal_Literal_String,
           Long_Long_Float_Attr_Safe_Last'Universal_Literal_String);

      ----------------
      -- Safe_Small --
      ----------------

      when Attribute_Safe_Small =>
         Float_Attribute_Universal_Real (
           Short_Float_Attr_Safe_Small'Universal_Literal_String,
           Float_Attr_Safe_Small'Universal_Literal_String,
           Long_Float_Attr_Safe_Small'Universal_Literal_String,
           Long_Long_Float_Attr_Safe_Small'Universal_Literal_String);

      -----------
      -- Scale --
      -----------

      when Attribute_Scale => Scale :
      declare
         D : Ureal := Delta_Value (P_Type);
         S : Int   := 0;

      begin
         while UR_Gt (D, Ureal_1) loop
            D := UR_Quotient (D, Ureal_10);
            S := S - 1;
         end loop;

         while UR_Lt (D, Ureal_1) loop
            D := UR_Product (D, Ureal_10);
            S := S + 1;
         end loop;

         Fold_Uint (N, UI_From_Int (S));
      end Scale;

      -------------
      -- Scaling --
      -------------

      when Attribute_Scaling =>
         Unimplemented_Attribute;

      ------------------
      -- Signed_Zeros --
      ------------------

      when Attribute_Signed_Zeros =>
         Float_Attribute_Boolean (
           Short_Float_Attr_Signed_Zeros,
           Float_Attr_Signed_Zeros,
           Long_Float_Attr_Signed_Zeros,
           Long_Long_Float_Attr_Signed_Zeros);

      ----------
      -- Size --
      ----------

      --  Size attribute just returns the size (note that we can only
      --  fold the size for static types, where the size is known in
      --  the front end. Dynamic cases of size are left to Gigi.

      when Attribute_Size =>
         Fold_Uint (N, Esize (P_Type));

      -----------
      -- Small --
      -----------

      when Attribute_Small =>
         Fold_Ureal (N, Small_Value (P_Type));

      ----------
      -- Succ --
      ----------

      when Attribute_Succ => Succ :
      begin
         if Static then
            if UI_Eq (Expr_Value (E1),
                      Expr_Value (Type_High_Bound (Base_Type (P_Type))))
            then
               Constraint_Error_Warning (N, "Succ of type'Last?!");
               Set_Potentially_Static (N, True);
               Check_Expressions;
               return;
            else
               Fold_Uint (N, UI_Sum (Expr_Value (E1), Uint_1));
            end if;
         end if;
      end Succ;

      ----------------
      -- Truncation --
      ----------------

      when Attribute_Truncation =>
         if Static then
            Fold_Ureal (N, UR_From_Uint (UR_Trunc (Expr_Value (E1))));
         end if;

      -----------------------
      -- Unbiased_Rounding --
      -----------------------

      when Attribute_Unbiased_Rounding =>
         Unimplemented_Attribute;

      ---------
      -- Val --
      ---------

      when Attribute_Val => Val :
      begin
         if Static then
            if UI_Lt (Expr_Value (E1),
                      Expr_Value (Type_Low_Bound (Base_Type (P_Type))))
              or else
               UI_Gt (Expr_Value (E1),
                      Expr_Value (Type_High_Bound (Base_Type (P_Type))))
            then
               Constraint_Error_Warning (N, "pos out of range?!");
               Set_Potentially_Static (N, True);
               Check_Expressions;
               return;
            else
               Fold_Uint (N, Expr_Value (E1));
            end if;
         end if;
      end Val;

      -----------
      -- Width --
      -----------

      when Attribute_Width => Width :
      begin
         if Static then

            --  Real types

            if Is_Real_Type (P_Type) then

               --  Width is zero for a null range (RM 3.5 (38))

               if UR_Lt (Expr_Value (Type_High_Bound (P_Type)),
                          Expr_Value (Type_Low_Bound (P_Type)))
               then
                  Fold_Uint (N, Uint_0);

               --  The non-null case depends on the specific real type

               else
                  if Is_Floating_Point_Type (P_Type) then

                  --  For floating-point, we have +N.dddE+nnn where  length
                  --  of ddd is determined by type'Digits - 1 (but is one
                  --  if Digits is one (RM 3.5 (33))

                     Fold_Uint (N,
                       UI_From_Int (7 +
                         Int'Max (2, UI_To_Int (Digits_Value (P_Type)))));

                  --  For fixed-point type width is Fore + 1 + Aft (RM 3.5(34))

                  else -- Is_Fixed_Point_Type (P_Type)
                     Fold_Uint (N, UI_From_Int (Fore_Value + 1 + Aft_Value));
                  end if;
               end if;

            --  Discrete types

            else
               declare
                  R  : constant Entity_Id := Root_Type (P_Type);
                  Lo : constant Uint :=
                         Expr_Value (Type_Low_Bound (P_Type));
                  Hi : constant Uint :=
                         Expr_Value (Type_High_Bound (P_Type));
                  W  : Nat := 0;
                  T  : Uint;
                  L  : Node_Id;

               begin
                  --  Empty ranges

                  if UI_Gt (Lo, Hi) then
                     W := 0;

                  --  Width for types derived from Standard.Character
                  --  and Standard.Wide_Character.

                  elsif R = Standard_Character
                    or else R = Standard_Wide_Character
                  then
                     W := 2;

                     for J in UI_To_Int (Lo) .. UI_To_Int (Hi) loop

                        --  Assume all wide-character escape sequences are
                        --  same length, so we can quit when we reach one.

                        if J > 255 then
                           if Length_Wide > W then
                              W := Length_Wide;
                           end if;

                           exit;

                        else
                           --  Test for all cases where Character'Image
                           --  yields an image that is longer than three
                           --  characters. First the cases of Reserved_xxx
                           --  names (length = 12).

                           if J in 128 .. 131 or else J in 152 .. 154 then
                              W := 12;

                           --  Then the ordinary cases of three characters,
                           --  either from control character names, or from
                           --  the standard quote-character-quote output.

                           else
                              if W = 2
                                and then
                                 (J          in   0 ..   7   -- NUL .. BEL
                                   or else J in  16 ..  24   -- DEL .. CAN
                                   or else J in  26 ..  27   -- SUB .. ESC
                                   or else J in  32 .. 126   -- lower graphics
                                   or else J =  127          -- DEL
                                   or else J in 132 .. 140   -- IND .. PLU
                                   or else J in 142 .. 148   -- SS2 .. CCH
                                   or else J in 150 .. 151   -- SPA .. EPA
                                   or else J =  155          -- CSI
                                   or else J =  157          -- OSC
                                   or else J =  159          -- APC
                                   or else J in 160 .. 255)  -- upper graphics
                              then
                                 W := 3;
                              end if;
                           end if;
                        end if;
                     end loop;

                  --  Width for types derived from Standard.Boolean

                  elsif R = Standard_Boolean then
                     if UI_Eq (Lo, Uint_0) then
                        W := 5; -- FALSE
                     else
                        W := 4; -- TRUE
                     end if;

                  --  Width for integer types

                  elsif Is_Integer_Type (P_Type) then
                     T := UI_Max (UI_Abs (Lo), UI_Abs (Hi));

                     W := 2;
                     while UI_Ge (T, Uint_10) loop
                        W := W + 1;
                        T := UI_Quotient (T, Uint_10);
                     end loop;

                  --  Only remaining possibility is user declared enum type

                  else
                     pragma Assert (Is_Enumeration_Type (P_Type));

                     W := 0;
                     L := First_Literal (P_Type);

                     while Present (L) loop
                        Get_Decoded_Name_String (Chars (L));

                        if Int (Name_Len) > W
                          and then UI_Ge (Enumeration_Pos (L), Lo)
                          and then UI_Le (Enumeration_Pos (L), Hi)
                        then
                           W := Int (Name_Len);
                        end if;

                        L := Next_Literal (L);
                     end loop;
                  end if;

                  Fold_Uint (N, UI_From_Int (W));
               end;
            end if;
         end if;
      end Width;

      --  The following attributes can never be folded, and furthermore we
      --  should not even have entered the case statement for any of these.

      when Attribute_Abort_Signal             |
           Attribute_Access                   |
           Attribute_Address                  |
           Attribute_Address_Size             |
           Attribute_Base                     |
           Attribute_Bit_Order                |
           Attribute_Body_Version             |
           Attribute_Callable                 |
           Attribute_Caller                   |
           Attribute_Class                    |
           Attribute_Component_Size           |
           Attribute_Constrained              |
           Attribute_Count                    |
           Attribute_Definite                 |
           Attribute_Enum_Rep                 |
           Attribute_External_Tag             |
           Attribute_First_Bit                |
           Attribute_Huge_Integer             |
           Attribute_Identity                 |
           Attribute_Input                    |
           Attribute_Last_Bit                 |
           Attribute_Leading_Part             |
           Attribute_Max_Interrupt_Priority   |
           Attribute_Max_Priority             |
           Attribute_Output                   |
           Attribute_Partition_Id             |
           Attribute_Position                 |
           Attribute_Read                     |
           Attribute_Storage_Pool             |
           Attribute_Storage_Size             |
           Attribute_Storage_Unit             |
           Attribute_Tag                      |
           Attribute_Terminated               |
           Attribute_Tick                     |
           Attribute_Unchecked_Access         |
           Attribute_Universal_Literal_String |
           Attribute_Unrestricted_Access      |
           Attribute_Valid                    |
           Attribute_Value                    |
           Attribute_Wide_Value               |
           Attribute_Version                  |
           Attribute_Wide_Image               |
           Attribute_Word_Size                |
           Attribute_Write                    =>

         pragma Assert (False); null;

      end case;

      --  At the end of the case, one more check. If we did a static evaluation
      --  so that the result is now an integer or real constant, then set the
      --  Potentially_Static flag in this literal only if the prefix type is a
      --  static subtype. For non-static subtypes, the replacement is still OK,
      --  but cannot be considered to be static.

      if Nkind (N) /= N_Attribute_Reference then
         Set_Potentially_Static (N, Static);

      --  If this is still an attribute reference, then it has not been folded
      --  and that means that its expressions are in a non-static context.

      else
         Check_Expressions;
      end if;

   end Eval_Attribute;

   -----------------------
   -- Resolve_Attribute --
   -----------------------

   procedure Resolve_Attribute (N : Node_Id; Typ : Entity_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      P     : constant Node_Id    := Prefix (N);
      Index : Interp_Index;
      It    : Interp;

   begin
      --  If attribute was universal type, reset to actual type

      if Etype (N) = Universal_Integer
        or else Etype (N) = Universal_Real
      then
         Set_Etype (N, Typ);
      end if;

      --  Remaining processing depends on attribute

      case Get_Attribute_Id (Attribute_Name (N)) is

         --  For these attributes, if the prefix denotes an entity, it is
         --  interpreted as a name, never as a call. It may be overloaded,
         --  in which case resolution uses the profile of the context type.
         --  Otherwise prefix must be resolved.

         when Attribute_Access
            | Attribute_Unchecked_Access =>

            if Is_Entity_Name (P) then
               if Is_Overloaded (P) then
                  Get_First_Interp (P, Index, It);

                  while Present (It.Nam) loop

                     if Type_Conformant (It.Nam, Designated_Type (Typ)) then
                        Set_Entity (P, It.Nam);
                        exit;
                     end if;

                     Get_Next_Interp (Index, It);
                  end loop;

               elsif not Is_Overloadable (Entity (P))
                 and then not Is_Type (Entity (P))
               then
                  Resolve (P, Etype (P));
               end if;

               if Is_Abstract (Entity (P)) then
                  Error_Msg_N
                    ("prefix of % attribute cannot be abstract subprogram",
                      P);
                  Set_Etype (N, Any_Type);
               end if;

            else
               Resolve (P, Etype (P));
            end if;

            Set_Etype (N, Typ);

         when Attribute_Address =>

            if not Is_Entity_Name (P)
               or else not Is_Overloadable (Entity (P))
            then
               Resolve (P, Etype (P));

            elsif Is_Overloaded (P) then
               Get_First_Interp (P, Index, It);
               Get_Next_Interp (Index, It);

               if Present (It.Nam) then
                  Error_Msg_N ("Prefix of 'Address cannot be overloaded", N);
               end if;
            end if;

         when Attribute_Range =>

            if not Is_Entity_Name (P)
              or else not Is_Type (Entity (P)) then
               Resolve (P, Etype (P));
            end if;

            --  We now replace the Range attribute node with a range expression
            --  whose bounds are the 'First and 'Last attributes applied to the
            --  same prefix. The reason that we do this transformation here
            --  instead of in the expander is that it simplifies other parts of
            --  the semantic analysis which assume that the Range has been
            --  replaced; thus it must be done even when in semantic-only mode
            --  (note that the RM specifically mentions this equivalence, we
            --  take care that the prefix is only evaluated once).

            Set_Evaluate_Once (P, True);
            Rewrite_Substitute_Tree (N,
              Make_Range (Loc,
                Low_Bound =>
                  Make_Attribute_Reference (Loc,
                    Prefix         => P,
                    Attribute_Name => Name_First,
                    Expressions    => Expressions (N)),
                High_Bound =>
                  Make_Attribute_Reference (Loc,
                    Prefix         => P,
                    Attribute_Name => Name_Last,
                    Expressions    => Expressions (N))));
            Analyze (N);
            Resolve (N, Typ);

            --  Normally after resolving attribute nodes, Eval_Attribute
            --  is called to do any possible static evaluation of the node.
            --  However, here since the Range attribute has just been
            --  transformed into a range expression it is no longer an
            --  attribute node and therefore the call needs to be avoided
            --  and is accomplished by simply returning from the procedure.

            return;

         --  For other attributes, resolve prefix if it is not a type mark.

         when others =>
            if not Is_Entity_Name (P)
              or else not Is_Type (Entity (P))
            then
               Resolve (P, Etype (P));
            end if;

      end case;

      --  Normally the Freezing is done by Resolve but sometimes the Prefix is
      --  not resolved, in which case the freezing must be done.

      Freeze_Expression (P);

      Eval_Attribute (N);
   end Resolve_Attribute;

end Sem_Attr;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.156
--  date: Tue Aug 23 14:07:23 1994;  author: schonber
--  (Resolve_Attribute): extend fix 1.154 to 'Access.
--  ----------------------------
--  revision 1.157
--  date: Thu Aug 25 16:33:36 1994;  author: dewar
--  (Fore_Value): Fix for bounds given in integer, not real form
--  ----------------------------
--  revision 1.158
--  date: Wed Aug 31 00:05:32 1994;  author: schonber
--  (Access_Attribute): access to protected operations is still unimplemented.
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
