/* {{{ Copyright notice */

/* View file module for the Midnight Commander
   Copyright (C) 1994 Miguel de Icaza
   Copyright (C) 1994, 1995 Janne Kukonlehto
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* }}} */
/* {{{ Declarations */

#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <ncurses.h>	/* Includes stdio.h */
#include <sys/stat.h>
#ifndef AUX
#include <sys/mman.h>
#endif
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>	/* For toupper() */
#include <stdlib.h>	/* atoi() */
#include <malloc.h>
#include <errno.h>
#include "mad.h"
#include "win.h"
#include "color.h"
#include "util.h"
#include "input.h"
#include "dialog.h"
#include "file.h"
#include "mouse.h"
#include "global.h"
#include "zip.h"
#include "help.h"
#include "key.h"	/* For mi_getch() */
#include "layout.h"
#include "view.h"
#include "main.h"	/* For the externs */

/* Needed for current_panel and other_panel */
#include <sys/param.h>
#include "dir.h"
#include "panel.h"

#ifndef MAP_FILE
#define MAP_FILE 0
#endif

/* Block size for reading files in parts */
#define READ_BLOCK 4096

#if defined(AIX) || defined(_AIX) || defined(__aix__)
    #define IFAIX(x) case (x):
#else
    #define IFAIX(x)
#endif

int max_dirt_limit = 10;	/* Maxlimit for skipping updates */
int wrap_mode = 1;	/* Wrap mode */

/* }}} */
/* {{{ Local variables */

static WINDOW *view_win;	/* Window for the file */
static int width, height;	/* Size of the view_win window */
static WINDOW *fkeys;		/* Fkey window for real view */
static char *filename = 0;	/* Name of the file */
static int view_active = 0;
static int have_frame;

static unsigned char *data;	/* Memory area for the file to be viewed */
static int file;		/* File descriptor (for mmap and munmap) */
static FILE *stdfile;		/* Stdio struct for reading file in parts */
static int reading_pipe;	/* Flag: Reading from pipe(use popen/pclose) */
static long bytes_read;		/* How much of file is read */
static long last;		/* Last byte showed */
static long last_byte;		/* Last byte of file */
static long first;		/* First byte in file */
static long start_display;	/* First char displayed */
static struct stat s;		/* stat for file */
static int  start_col = 0;	/* First displayed column, negative */
static int dirty = 0;		/* Number of skipped updates */
static int hex_mode = 0;	/* Hexadecimal mode flag */
static int bytes_per_line;	/* Number of bytes per line in hex mode */
static int view_quit = 0;		/* Quit flag */
static int viewer_magic_flag=0; /* Selected viewer */
static int mmaping = 0;		/* Did we use mmap on the file? */
static long int gziped = 0;	/* Size of gziped file or 0 */
static int do_pop_refresh;	/* Flag: Should pop the refresh function? */
static char *search_exp;	/* The search expression */

/* Pointer to the last search command */
static void (*last_search)(char *) = 0;

static char hex_char[] = "0123456789ABCDEF";
static char rcsid [] = "$Id: view.c,v 1.18 1995/01/27 02:36:46 miguel Exp $";

/* }}} */
/* {{{ Clean-up functions */

void free_file (void)
{
#ifdef HAVE_MMAP
    if (mmaping){
	munmap (data, s.st_size);
	close (file);
    } else
#endif /* HAVE_MMAP */
    {
	free (data);
	if (reading_pipe){
	    /* Check error messages */
	    if (!have_frame)
		check_error_pipe ();
	    /* Close pipe */
	    pclose (stdfile);
	    /* Ignore errors because we don't want to hear about broken pipe */
	    close_error_pipe (-1, NULL);
	} else
	    fclose (stdfile);
    }
}

/* Both views */

void view_done (void)
{
    if (view_active){
	free_file ();
	free (filename);
    }
    view_active = 0;
}

/* }}} */
/* {{{ Initialization and loading functions */

/* Loads enough blocks so that value at offset is known */
char load_byte (long offset, int able_to_recover)
{
    int readsize, n;

    if (offset >= s.st_size){
#ifdef MCDEBUG
	fprintf (stderr, "Internal failure: Trying to read beyond the end of file\r\n");
	abort ();
#else
	return '\n';
#endif
    }
    readsize = offset - bytes_read + 1;
    readsize = (readsize + READ_BLOCK) & ~READ_BLOCK;
    if (readsize + bytes_read > s.st_size)
	readsize = s.st_size - bytes_read;
    n = fread (data + bytes_read, 1, readsize, stdfile);
    bytes_read += n;
    if (n < readsize){
	/* Size was shorter than guess */
	s.st_size = bytes_read;
	last_byte = first + bytes_read;
	data = realloc (data, bytes_read);
    }
    if (bytes_read <= offset){
#ifdef MCDEBUG
	if (!able_to_recover)
	{
	    fprintf (stderr, "Internal failure: Can't read file up to requested offset\r\n");
	    abort ();
	} else
#endif
	    return '\n';
    }
    return data [offset];
}

#define get_byte(from) ((from >= bytes_read) ? (load_byte (from, 0)) : (data[from]))

/* Load filename into core */
/* returns 1 if used mmap, 0 if used malloc */
/* -1 on failure */
static int load_view_file (char *filename, char **error_string)
{
    char *gzip_cmd = "gzip -dc ";
    char *cmd;
    int  return_val;
#ifndef HAVE_MMAP
    int count;
#endif

    data = NULL;
    return_val = -1;
    
    if ((file = open (filename, O_RDONLY)) < 0){
	*error_string = copy_strings (" Can't open file \"",
				     filename, "\"\n ",
				     unix_error_string (errno), " ", 0);
	if (have_frame){
	    data = *error_string;
	    return 0;
	}
	return -1;
    }
    if (fstat (file, &s) < 0){
	*error_string = copy_strings (" Can't stat file \n ",
				     unix_error_string (errno), " ", 0);
	if (have_frame){
	    data = *error_string;
	    return_val = 0;
	}
	close (file);
	return return_val;
    }
    if (S_ISDIR (s.st_mode) || S_ISSOCK (s.st_mode) || S_ISFIFO (s.st_mode)){
	*error_string = copy_strings (" Can't view: not a regular file ", 0);
	if (have_frame){
	    data = *error_string;
	    return_val = 0;
	}
	close (file);
	return return_val;
    }
    if (s.st_size < 1) {
	*error_string = strdup (" Can't view empty file ");
	if (have_frame){
	    data = *error_string;
	    return_val = 0;
	}
	close (file);
	return return_val;
    }

    /* First, try to open a compressed file */
    if (!viewer_magic_flag && (gziped = is_gunzipable (file)) != 0){
	close (file);
	s.st_size = gziped;

	cmd = copy_strings (gzip_cmd, "'", filename, "'", 0);

	if ((data = (char *) malloc (gziped)) == NULL){
	    free (cmd);
	    *error_string = strdup (" Not enough memory for unziped file ");
	    if (have_frame){
		data = *error_string;
		return_val = 0;
	    }
	    return return_val;
	}
	first = 0;
	bytes_read = 0;
	reading_pipe = 1;

	open_error_pipe ();
	if ((stdfile = popen (cmd, "r")) == NULL){
	    free (cmd);
	    free (data);
	    data = NULL;

	    if (have_frame){
		data = strdup (" Can't spawn child gzip program ");
		return_val = 0;
	    }
	    close_error_pipe (have_frame?-1:1, data);
	    return return_val;
	}
	free (cmd);

	/* First, check is gzip produced any output */
	load_byte (0, 1);
	if (bytes_read <= 0){
	    free (data);
	    data = NULL;
	    pclose (stdfile);

	    if (have_frame){
		data = strdup (" Empty output from child gzip ");
		return_val = 0;
	    }
	    close_error_pipe (have_frame?-1:1, data);
	    return return_val;
	}

	/* Yes, we can view this file */
	return 0;
    } else
	gziped = 0;

    /* Otherwise, the file wasn't compressed */
#ifdef HAVE_MMAP
    data = mmap (0, s.st_size, PROT_READ, MAP_FILE | MAP_SHARED, file, 0);
    if ((caddr_t) data == (caddr_t) -1){
	close (file);
	*error_string = copy_strings (" Can't mmap file \n ",
				      unix_error_string (errno), " ", 0);
	
	if (have_frame){
	    data = *error_string;
	    return_val = 0;
	}
	return return_val;
    }
    first = 0;
    bytes_read = s.st_size;
    return 1;
#else
    if ((data = malloc (s.st_size)) == NULL){
	close (file);
	*error_string = strdup (" Machine without mmap: \n"
				" no free memory to load file in RAM ");
	if (have_frame){
	    data = *error_string;
	    return_val = 0;
	} 
	return return_val;
    }
    lseek (file, 0, SEEK_SET);
    stdfile = fdopen (file, "r");
    if (!stdfile){
	free (data);
	close (file);
	*error_string = copy_strings (" Failed to reopen file \n ",
				      unix_error_string (errno), " ", 0);
	if (have_frame){
	    data = *error_string;
	    return_val = 0;
	}
	return return_val;
    }
    /*close (file);*/
    first = 0;
    bytes_read = 0;
    reading_pipe = 0;

    load_byte (0, 1);
    if (bytes_read <= 0){
	free (data);
	data = NULL;
	fclose (stdfile);
	*error_string = copy_strings (" Can't read file \n ",
				      unix_error_string (errno), " ", 0);
	if (have_frame){
	    data = *error_string;
	    return_val = 0;
	}
	return return_val;
    }
    return 0;
#endif
}

/* Return zero on success, -1 on failure */
int do_view_init (char *_file)
{
    char *error = 0;
    
    if (view_active)
	view_done ();
    if (!have_frame){
	/* hex_mode = 0; */
	wrap_mode = 1;
	start_col = 0;
    }
    mmaping = load_view_file (_file, &error);

    if (mmaping == -1){
	if (!have_frame){
	    message (1, " Error ", error);
	    free (error);
	    return -1;
	}
	return -1;
    }

    view_active = 1;
    filename = strdup (_file);
    start_display = first;
    start_col = 0;
    last_search = 0;            /* Start a new search */

    /* Special case: The data points to the error message */
    if (mmaping == 0 && data == (unsigned char *) error)
	s.st_size = bytes_read = strlen (data);
    
    last_byte = first + s.st_size;

    return 0;
}

/* Both views */
/* Return zero on success, -1 on failure */
int view_init (WINDOW *w, WINDOW *f, int _width, int _height, char *_file, int _have_frame)
{
    view_win = w;
    fkeys = f;
    width = _width;
    height = _height;
    have_frame = _have_frame;
    bytes_per_line = 2 * (width - 7) / 9;
    bytes_per_line &= 0xfffc;
    dirty = 1;
    if (!view_active || strcmp (_file, filename))
	return do_view_init (_file);
    else
	return 0;
}

void view_init_windows(WINDOW *w, WINDOW *f) /*Didn't find a better name :-) */
{
    view_win = w;
    fkeys = f;
}

/* }}} */
/* {{{ Screen update functions */

/* Real view only */
static void view_refresh (void *unused)
{
    if (keybar_visible){
	touchwin (fkeys);
	wrefresh (fkeys);
    }
    touchwin (view_win);
    wrefresh (view_win);
}

static void view_status (void)
{
    int w = width - have_frame;
    int i;

    wattron (view_win, SELECTED_COLOR);
    wmove (view_win, have_frame, have_frame);
    whline (view_win, ' ', w);
    if (w > 6){
    	i = w > 24 ? 18 : w - 6;
    	wprintw (view_win, "File: %s", name_trunc (filename?filename:"", i));
    	if (w > 30){
    	    wmove (view_win, have_frame, 26);
    	    wprintw (view_win, "Col %d", -start_col);
    	}
    	if (w > 60){
	    wmove (view_win, have_frame, 42);
	    wprintw (view_win, "%s bytes", size_trunc (s.st_size));
        }
        if (w - i > 4){
            wmove (view_win, have_frame, width - 5);
            wprintw (view_win, "%3d%% ", last_byte == last ? 100 :
	             (start_display-first)*100 / s.st_size);
	}
    }
    wattroff (view_win, SELECTED_COLOR);
}

/* Shows the file pointed to by *start_display on view_win */
static long display (void)
{
    int col = 0 + have_frame;
    int row = 1 + have_frame;
    long from;
    int c;
    
    from = start_display;
    wattron (view_win, NORMAL_COLOR);
    if (have_frame){
	int i;
	for (i = 1; i < height; i++)
	    mvwprintw (view_win, i, 1, "%*s", width-1, "");
    } else {
	werase (view_win);
	wclr (view_win);
    }
    if (hex_mode){
        char hex_buff[10];   /* A temporary buffer for sprintf and mvwaddstr */
        int bytes;	     /* Number of bytes already printed on the line */
        int text_start = width - bytes_per_line - 1 + have_frame;  /* Start of text column */
	
        for (;row < height && from < last_byte; row++){
            /* Print the hex offset */
            sprintf (hex_buff, "%05X", (int) (from - first));
            mvwaddstr (view_win, row, have_frame, hex_buff);
	    
            /* Hex dump starts from column seven */
            col = 7;
	    
            /* Each hex number is two digits */
            hex_buff[2] = 0;
            for (bytes = 0; bytes < bytes_per_line && from < last_byte; bytes++, from++){
                c = (unsigned char) get_byte (from);
		
                /* Print a hex number (sprintf is too slow) */
                hex_buff [0] = hex_char [(c >> 4)];
                hex_buff [1] = hex_char [c & 15];
                mvwaddstr (view_win, row, col, hex_buff);
                col += 3;
                if ((bytes & 3) == 3 && bytes + 1 < bytes_per_line){
		    
                    /* Hex numbers are printed in the groups of four */
                    /* Groups are separated by a vline */
		    
                    waddch (view_win, ' ');
                    wvline (view_win, 0, 1);
                    col += 2;
                }
                /* Print the corresponding ascii character */
                if (is_printable (c))
                    mvwaddch (view_win, row, text_start + bytes, c);
                else
                    mvwaddch (view_win, row, text_start + bytes, '.');
            }
        }
    } else {
    	for (; row < height && from < last_byte; from++){
	    c = get_byte (from);
    	    if ((c == '\n') || (col == width && wrap_mode)){
       	        col = have_frame;
       	        row++;
		if (c == '\n' || row >= height)
		    continue;
       	    }
	    if (c == '\r')
		continue;
       	    if (c == '\t'){
       	        col = ((col - have_frame)/8)*8 + 8 + have_frame;
       	        continue;
       	    }
       	    if (col >= have_frame-start_col && col < width-start_col){
       		if (is_printable (c))
       		    mvwaddch (view_win, row, col+start_col, c);
       		else
       		    mvwaddch (view_win, row, col+start_col, '.');
       	    } 
	    col++;
        }
    }
    last = from;
    return from;
}

void view_update (void)
{
    static int dirt_limit = 1;

    if (dirty > dirt_limit){
	/* Too many updates skipped -> force a update */
	display ();
	dirty = 0;
	/* Raise the update skipping limit */
	dirt_limit++;
	if (dirt_limit > max_dirt_limit) dirt_limit = max_dirt_limit;
    }
    if (dirty){
	if (is_idle ()){
	    /* We have time to update the screen properly */
	    display ();
	    view_status ();
	    dirty = 0;
	    if (dirt_limit > 1)
		dirt_limit--;
	} else {
	    /* We are busy -> skipping full update,
	       only the status line is updated */
	    view_status ();
	}
	wrefresh (view_win);
    }
}

/* Both views */
void view_labels (WINDOW *fkeys)
{
    set_label_text (fkeys, 2, hex_mode ? "" : wrap_mode ? "Unwrap" : "Wrap");
    set_label_text (fkeys, 4, hex_mode ? "Ascii" : "Hex");
    set_label_text (fkeys, 5, "Line");
    set_label_text (fkeys, 6, hex_mode ? "" : "RxSrch");
    set_label_text (fkeys, 7, hex_mode ? "HxSrch" : "Search");
    set_label_text (fkeys, 8, viewer_magic_flag ? "Gunzip" : "Raw");
}

/* }}} */
/* {{{ Movement functions */

static long move_forward2 (long current, int lines)
{
    long p;
    int  line;
    int  col = 0;

    if (last == last_byte)
	return current;
    
    if (hex_mode){
        int i;
        for (i = 0, p = current; i < lines * bytes_per_line; p++, i++)
            if (p >= last_byte)
		return current;
        return p;
    } else {
        for (line = col = 0, p = current; p < last_byte; p++){
	    if (line >= lines)
	        return p;
	    if (wrap_mode){
		col++;
		if (col == width){
		    col = 0;
		    line++;
		}
	    }
	    if (get_byte (p) == '\n'){
	        line++;
		col = 0;
	    }
        }
    }
    return current;
}

/* returns the new current pointer */
static long move_backward2 (long current, int lines)
{
    long p;
    int line, col;

    if (current == first)
	return current;
    
    if (hex_mode){
        int i;
        for (i=0,p=current; p > first && i < lines * bytes_per_line; p--, i++)
	    ;
    } else {
        for (p = current-1; p > first; p--)
	    if (get_byte (p) == '\n'){
	        p--;
	        break;
	    }
        /* p points to the char before the new line */
        if (p <= first)
	    return current;
    
        for (line = col = 0; p > first; p--){
	    if (get_byte (p) == '\n')
	        line++;
	    if (line == lines)
	        return p+1;
        }
    }
    return p;
}

static void move_backward (int i)
{
    start_display = move_backward2 (start_display, i);
    dirty++;
}

static void move_forward (int i)
{
    start_display = move_forward2 (start_display, i);
    dirty++;
}

#if 0
/* FIXME -- remove these fns? */
/* R: They could be left here as a reference for the future */
static void half_up (void)
{
    move_backward (height/2 - 1);
    view_update ();
}

static void half_down (void)
{
    move_forward (height/2 - 1);
}
#endif

static void move_to_top (int dummy)
{
    start_display = first;
    dirty++;
}

static void move_to_bottom (int dummy)
{
    start_display = move_backward2 (last_byte, height-1);
    dirty++;
}

/* Scroll left/right the view panel functions */
static void move_right (void)
{
    if (wrap_mode)
	return;
    
    if (--start_col > 0)
	start_col = 0;
    dirty++;
}

static void move_left (void)
{
    if (wrap_mode)
	return;
    
    if (++start_col > 0)
	start_col = 0;
    dirty++;
}

/* }}} */
/* {{{ Search routines */

/* Case insensitive search of text in data */
static int icase_search_p (char *text, char *data, int nothing)
{
    int p = icase_search (text, data) != 0; 
    return p;
}

static void search (char *text, int (*search)(char *, char *, int))
{
    char *s;
    long p;

    if (verbose)
	message (D_INSERT, " Search ", "Searching %s", text);
    
    for (p = start_display; p < last_byte; ){
	for (; p < last_byte; p++)
	    if (get_byte (p) == '\n'){
		p++;
		break;
	    }
	s = extract_line ((data + p), (data + last_byte));
	if ((*search) (text, s, match_normal)){
	    start_display = p;
	    break;
	}
    }
    if (verbose)
	destroy_dialog ();
    if (p >= last_byte)
	message (0, " Search ", " Search string not found ");
}

/* Search buffer (it's size is len) in the complete buffer */
/* returns the position where the block was found */
static long block_search (char *buffer, int len)
{
    char *d = buffer;
    long e = start_display;

    for (; e < last_byte; e++){
	if (*d == get_byte (e))
	    d++;
	else
	    d = buffer;
	if (d - buffer == len)
	    return e - len;
    }
    return 0;
}

/* States of our funny recognizer */
enum {normal, inside_quotes, zero, hex1, hex2, oct1};

/* This routine doesn't report all the user mistakes, it just ignores them */
static void hex_search (char *text)
{
    char buffer [120];		/* Where we hold the information */
    int  i, block_len;
    int  v = 0;
    long pos;			/* Where did we found the string */
    char *p;			/* Temporary */
    int  state = normal;	/* Initial state of the micro-scanner */
    
    /* First convert the string to a stream of bytes */
    for (i = block_len = 0; text [i] && block_len < sizeof (buffer); i++){
	switch (state){
	case inside_quotes:
	    if (text [i] == '"')
		state = normal;
	    else
		buffer [block_len++] = text [i];
	    break;

	case normal:
	    if (text [i] == '"'){
		state = inside_quotes;
		break;
	    }
	    if (text [i] == '0'){
		state = zero;
		break;
	    }
	    if (text [i] == 'x'){
		state = hex1;
		break;
	    }
	    break;

	case zero:
	    if (text [i] == 'x')
		state = hex1;
	    break;

	case hex1:
	    v = 0;
	    text [i] = toupper (text [i]);
	    if ((p = strchr (hex_char, text [i])) != 0){
		v = (p - hex_char) << 4;
		state = hex2;
	    }
	    break;

	case hex2:
	    if ((p = strchr (hex_char, text [i])) != 0){
		v |= (p - hex_char);
		state = normal;
	    }
	    buffer [block_len++] = v;
	    break;
	}
    }
    /* Then start the search */
    pos = block_search (buffer, block_len);
    if (!pos){
	message (0, " Search ", " Search string not found ");
	return;
    }
    
    /* Adjust the file offset */
    start_display = (pos & (~(bytes_per_line-1)));
}

static void do_regexp_search (char *regexp)
{
    search_exp = regexp;
    search (regexp, regexp_match);
    touchwin (view_win);
    dirty++;
    view_update ();
}

static void do_normal_search (char *text)
{
    search_exp = text;
    if (hex_mode)
	hex_search (text);
    else 
	search (text, icase_search_p);
    touchwin (view_win);
    dirty++;
    view_update ();
}

/* }}} */
/* {{{ Mouse and keyboard handling */

/* Real view only */
static void help_cmd (void)
{
    interactive_display (LIBDIR "mc.hlp", "[Internal File Viewer]");
    view_refresh (0);
}

/* Both views */
void toggle_wrap_mode (void)
{
    if (hex_mode)
	return;
    wrap_mode = 1 - wrap_mode;
    if (wrap_mode)
	start_col = 0;
    view_labels (fkeys);
    dirty++;
    view_update ();
}

/* Both views */
void toggle_hex_mode (void)
{
    hex_mode = 1 - hex_mode;
    view_labels (fkeys);
    dirty++;
    view_update ();
}

/* Both views */
void goto_line (void)
{
    char *line, prompt [100];
    int i, oldline = 1;

    for (i = first; i < start_display; i++)
	if (get_byte (i) == '\n')
	    oldline ++;
    sprintf (prompt, "The current line number is %d.\n"
	             "Enter the new line number:", oldline);
    line = input_dialog (" Goto line ", prompt, "");
    if (line){
	if (*line){
	    move_to_top (0);
	    move_forward (atoi (line) - 1);
	}
	free (line);
    }
    dirty++;
    view_update ();
}

/* Both views */
void regexp_search (void)
{
    char *regexp = "";
    static char *old = 0;

    if (hex_mode)
	return;
    
    regexp = old ? old : regexp;
    regexp = input_dialog (" Search ", "Enter regexp:", regexp);
    if ((!regexp) || (!*regexp)){
	regexp = "";
	return;
    }
    if (old) free (old);
    old = strdup (regexp);
    if (bytes_read < last_byte)
	load_byte (last_byte-1, 1);	/* Get the whole file in to memory */
    do_regexp_search (regexp);
    last_search = do_regexp_search;
}

/* Both views */
void normal_search (void)
{
    static char *old;
    char *exp = "";

    exp = old ? old : exp;
    exp = input_dialog (" Search ", "Enter search string:", exp);
    if ((!exp) || (!*exp)){
	exp = "";
	return;
    }
    if (old)
	free (old);
    old = strdup (exp);
/* FIXME: add this after 2.0: free (exp) */
    if (bytes_read < last_byte)
	load_byte (last_byte-1, 1);	/* Get the whole file in to memory */
    do_normal_search (exp);
    last_search = do_normal_search;
}

void change_viewer (void)
{
    char *s = strdup (filename);

    view_done ();
    viewer_magic_flag = !viewer_magic_flag;
    view_init (view_win, fkeys, width, height, s, have_frame);
    free (s);
    view_labels (fkeys);
}

/* Real view only */
static int quit_cmd (void)
{
    return view_quit = 1;
}

/* Both views */
static int check_left_right_keys (int c)
{
    if (c == KEY_LEFT)
	move_left ();
    else if (c == KEY_RIGHT)
	move_right ();
    else return 0;

    return 1;
}

/* Both views */
int view_check_key (int c, int additional)
{
    if (check_fkeys (c))
	return 1;
    else if (check_left_right_keys (c))
	return 1;
    else if (check_movement_keys (c, additional, height,
				  move_backward, move_forward,
				  move_to_top, move_to_bottom))
	return 1;
    else switch (c * additional){

    case '/':
	regexp_search ();
	return 1;

	/* Continue search */
    case XCTRL('s'):
    case 'n':
	if (last_search){
	    (*last_search)(search_exp);
	} else {
	    /* if not... then ask for an expression */
	    normal_search ();
	}
	return 1;

    case XCTRL('l'):
	view_refresh (0);
	return 1;

    case ESC_CHAR:
	view_quit = 1;
	return 1;

    }
    /* Key not used */
    return 0;
}

/* Both views */
int view_event (Gpm_Event *event, int *result)
{
    *result = MOU_NORMAL;
    if (event->type & (GPM_DOWN|GPM_DRAG)){
    	if (!wrap_mode){
    	    if (event->x < width / 4){
    	    	move_left ();
    	    	*result = MOU_REPEAT;
    	    	return 1;
    	    }
    	    if (event->x > 3 * width / 4){
    	    	move_right ();
    	    	*result = MOU_REPEAT;
    	    	return 1;
    	    }
	}
	if (event->y < height / 3){
	    if (mouse_move_pages)
	    	move_backward (height / 2 - 1);
	    else
	    	move_backward (1);
	    *result = MOU_REPEAT;
	    return 1;
	}
	else if (event->y > 2 * height /3){
	    if (mouse_move_pages)
	    	move_forward (height / 2 - 1);
	    else
	    	move_forward (1);
	    *result = MOU_REPEAT; 
	    return 1;
	}
    }
    return 0;
}

/* Real view only */
int real_view_event (Gpm_Event *event, void *x)
{
    int result;
    
    if (view_event (event, &result))
    	view_update ();
    return result;
}

/* }}} */
/* {{{ Window creation, destruction and a driver stub for real view */

/* Real view only */
static void create_windows (void)
{
    do_pop_refresh = 0;
#ifdef BUGGY_CURSES
    /* See key.c: mi_getch /BUGGY_CURSES/ */
    touchwin (stdscr);
#endif
    view_win = newwin (LINES-keybar_visible, COLS, 0, 0);
    wattron (view_win, NORMAL_COLOR);
    push_frame (0, 0, 0);

    fkeys  = push_fkey (1);

    define_label (fkeys, 1, "Help", help_cmd);
    define_label (fkeys, 2, "Unwrap", toggle_wrap_mode);
    define_label (fkeys, 3, "", 0);
    define_label (fkeys, 4, hex_mode ? "Ascii" : "Hex", toggle_hex_mode);
    define_label (fkeys, 5, "Line", goto_line);
    define_label (fkeys, 6, "RxSrch", regexp_search);
    define_label (fkeys, 7, "Search", normal_search);
#if defined(AIX) || defined(_AIX) || defined(__aix__)
    define_label (fkeys, 8, "Top", move_to_top);
    define_label (fkeys, 9, "Bottom", move_to_bottom);
#else
    define_label (fkeys, 8, "Raw", change_viewer);
    define_label (fkeys, 9, "", 0);
#endif
    define_label_quit (fkeys, 10, "Quit", quit_cmd);
    view_labels (fkeys);

    /* workaround ncurses 1.8.5 bug */
    if (keybar_visible){
	wmove (fkeys, 0, 0);
	waddch (fkeys, '1');
	wrefresh (fkeys);
    }
    clearok (view_win, TRUE);

    /* At least, we can set the update function */
    push_refresh (view_refresh, 0, REFRESH_COVERS_ALL);
    do_pop_refresh = 1;

    view_quit = 0;

    push_event (1, 1, COLS, LINES-1, real_view_event, 0,
		event_use_frame);
}

/* Real view only */
static void destroy_windows (void)
{
    delwin (view_win);
    pop_fkey (fkeys);
    pop_frame ();
    if (do_pop_refresh)
        pop_refresh ();
}

/* Real view only */
void view (char *_file)
{
    int error, old_keybar_visible;

    old_keybar_visible = keybar_visible;
    keybar_visible = 1;
    create_windows ();
    dirty = max_dirt_limit + 1;	/* To force refresh */
    error = view_init (view_win, fkeys, COLS, LINES-keybar_visible, _file, 0);
    if (!error){
	while (!view_quit){
	    view_update ();

	    /* NCurses bug workaround */
	    view_status ();
	    wrefresh (view_win);

	    view_check_key (mi_getch (), 1);
	}
    }
    if (current_panel->view_type != view_quick
	&& other_panel->view_type != view_quick)
	view_done ();
    destroy_windows ();
    keybar_visible = old_keybar_visible;
}

/* }}} */
/* {{{ Emacs local variables */

/*
   Cause emacs to enter folding mode for this file:
   Local variables:
   folded-file: t
   end:
   */

/* }}} */
