//  LAST EDIT: Mon Aug  8 15:30:49 1994 by Barth(@prakinf.tu-ilmenau.de)
#ifndef __BASE_H__
#define __BASE_H__
////////////////////////////////////////////////////////////////////////////////
// Base classes for Scientific Visualization (levels: Base, Geometry, and     //
// Topology)                                                                  //  
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//                      barth@prakinf.tu-ilmenau.de                           //
//                      ekki@prakinf.tu-ilmenau.de                            //
//                                                                            //
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include <primitiv.h>
#include <quadmesh.h>


#define MAX_PARAM 10
// max. number of parameters at vertex
#define MAX_MAPS 4
// max. number of mappings per parameter
#define MAXISO 30
// max. number of isolines

typedef enum { RTSC_DOUBLE, RTSC_INTEGER, RTSC_FLOAT, RTSC_CHAR } RT_ScType;
typedef enum { NO_MAP, Z_MAP, COLOR_MAP, ISO_MAP } RT_MapType;
typedef enum { NONE, HOLLOW, FILLED } RT_GridMode;
typedef enum { NO_ISO, ISO_LINES, ISO_AREAS } RT_IsoMode;
typedef enum { BACK_PLANE, FRONT_PLANE, LEFT_PLANE, RIGHT_PLANE, BOTTOM_PLANE, TOP_PLANE } RT_LatticePlaneType;

class MapRec {
public:
   int mapNr;
   RT_MapType mtype[MAX_MAPS];
};

class ParamSet {
public:
   RT_ScType paramType;
   MapRec paramMap[ MAX_MAPS ];
   int mappings[ MAX_MAPS ];
   int *iData;
   float *fData;
   double *dData;
   char *cData;
   void **params;
   int loaded;
   double Pmin, Pmax;
   RT_String description;
};


class IsoRec {
public:
   RT_IsoMode mode;
   int nr;
   double levels[ MAXISO ];
   RT_Color colors[ MAXISO ];
   int occupied[ MAXISO ];
   double offset;
   int realZ;
};


//////////// base level:

#define RTN_SCIENTIFIC "Scientific"

void rt_initScientCommands( Tcl_Interp *);

class RT_Scientific: public RT_Primitive {
  static RT_ParseEntry table[];
  //#### the statics for parameter parsing:
  static int gmF, gmV, gmG, imF, imV, imG, ilF, ilG, ioF, ioG, pmapF, pmapG, punmapF, izF, izV, izG, cmF, cmG, inrG, imaxG, pmaxG, mmaxG, lpG, pscrF;
  static char *ilV, *pmapV, *punmapV, *cmV, *pscrV;
  static double ioV;

protected:
  int coord;
  // number of coordinates at each vertex
  // in most cases = 3 (x,y,z)
  int dims[4];
  // dimensions of the 4D geometry
  // and temporary products needed for access
  
  double *geometry;
  // the geometry values
  int currParNr;
  // parameter to access

  double hueMin, hueMax, saturation, value;
  int hslChanged;

  RT_GridMode gmode;
  // 0 = no grid, 1 = grid, 2 = filled grid;
  IsoRec iso;
  ParamSet paramData[ MAX_PARAM ];
  
  RT_Scientific( char *a ): RT_Primitive( a ) {
      for ( int i=0; i<MAX_PARAM; i++ ) {
	  paramData[i].params = 0;
	  paramData[i].loaded = 0;
	  for ( int j=0; j<MAX_MAPS; j++ ) {
	      paramData[i].mappings[j] = NO_MAP;
	      paramData[i].description = RT_String( 0 );
	  }
      }
      geometry = 0; coord = 3; 
      dims[0] = dims[1] = dims[2] = dims[3] = 1;
      gmode = HOLLOW;
      iso.mode = NO_ISO;
      iso.nr = 0;
      for ( i = 0; i< MAXISO; i++ ) {
	  iso.levels[i] = 0.0;
	  iso.colors[i] = RT_Color(1, 1, 1);
	  iso.occupied[i] = 0;
      }
      iso.offset = 0.0;
      iso.realZ = 0;
      hueMin = 0.01; hueMax = 0.9;
      saturation = 1.0;
      value = 0.5;
      hslChanged = 0;
      currParNr = 0;
  }

  virtual ~RT_Scientific() {
      if (geometry) delete [] geometry;
      destroyParams();      
  }

  void init_vars();
  virtual void createGeometry() = 0;
  // create the geometry from file or numerical algorithms
  virtual void createParameters() = 0;
  // create the parameters from file or numerical algorithms
  void newGeometry() {
      // allocate a new field for geometry according 
      // to the values in dims and the coord value
      if (geometry) delete [] geometry;
      if (coord) {
	  geometry = new double[ coord * dims[0] * dims[1] * dims[2]];
      }
      else geometry = 0;
  }
  void newParameters();
  // allocate a new field for parameters according
  // to the values in dims and the coord value
  void destroyParams();
  // release memory from parameters

  virtual void printParams( FILE * )const { }
  virtual void printGeometry( FILE * )const { }

public:
  //##### public methods:
  int isA(const char *)const;
  
  int get_xdim() { return dims[0]; }
  int get_ydim() { return dims[1]; }
  int get_zdim() { return dims[2]; }
  
  virtual void gridmode( RT_GridMode _gm ) { gmode = _gm; geomChanged(); }
  int get_gridmode() { return gmode; }
  virtual void isomode( RT_IsoMode _im ) { iso.mode = _im; geomChanged(); }
  int get_isomode() { return iso.mode; }
  virtual void isolevel( int _nr, double _lev, RT_Color _c ) {
      iso.nr = ( _nr+1 > iso.nr ) ? _nr+1 : iso.nr;
      iso.levels[_nr] = _lev;  iso.colors[_nr] = _c;
      iso.occupied[_nr] = 1;
      geomChanged();
  }
  void get_isolevels( int *nr, double *lev, RT_Color *c ) {
      for ( int i=0; i<MAXISO; i++ ) 
	  if ( iso.occupied[ i ] ) {
	      nr[i] = i;
	      lev[i] = iso.levels[i];
	      c[i] = iso.colors[i];
	  }
  }
  int get_isonr()const { return iso.nr; }
  int get_maxiso() { return MAXISO; }
  void isooffset( double _io ) { iso.offset = _io; geomChanged(); }
  double get_isooffset() { return iso.offset; }
  void isobase( int _iz ) { iso.realZ = _iz; geomChanged(); }
  int get_isobase() { return iso.realZ; }
  virtual void hslmap( double _hmin, double _hmax, double _sat, double _val ) {
      hueMin = _hmin; hueMax = _hmax; saturation = _sat;
      value = _val; hslChanged = 1;
      geomChanged();
  }
  void get_hslmap( double *_hmin, double *_hmax, double *_sat, double *_val ) {
      *_hmin = hueMin; *_hmax = hueMax; *_sat = saturation; *_val = value;
  }

  void setParamType( int _pnr, RT_ScType _pt );
  void paramdescription( int _pnr, RT_String _dscr ) {
      if ( _pnr < MAX_PARAM ) paramData[_pnr].description = _dscr;
  }
  virtual void map( int _pnr, RT_MapType _mt );
  virtual void unmap( int _pnr, RT_MapType _mt );
  void get_map( int _pnr, RT_MapType *_mt );
  int get_maxmaps() { return MAX_MAPS; }
  int get_maxparams() { return MAX_PARAM; }
  //#### the tcl commands:
  int objectCMD(char *argv[]);
  void print( FILE *f )const {
      RT_Primitive::print( f );
      printGeometry( f );
      printParams( f );
      fprintf( f, "\n%s ", get_name() );
      if ( gmode != HOLLOW ) fprintf( f, " -gridmode %i ", (int)gmode );
      if ( iso.mode != NO_ISO ) fprintf( f, " -isomode %i ", (int)iso.mode );
      if ( iso.offset != 0.0 ) fprintf( f, " -isooffset %lf ", iso.offset );
      if ( iso.realZ ) fprintf( f, " -isobase %i ", iso.realZ );
      if ( hslChanged ) fprintf( f, " -hslmap { %lf %lf %lf %lf }", hueMin, hueMax, saturation, value );
      for ( int i=0; i<get_isonr(); i++ ) 
	  if ( iso.occupied[i] ) fprintf( f, "\n%s -isolevel { %i %lf %lf %lf } ", get_name(), i, iso.levels[i], iso.colors[i].r, iso.colors[i].g, iso.colors[i].b );
      for ( i=0; i<MAX_PARAM; i++ ) 
	  for ( int j=0; j<MAX_MAPS; j++ ) 
	      if (paramData[i].mappings[j]) fprintf( f, "\n%s -map { %i %i } ", get_name(), i, j );
      for ( i=0; i<MAX_PARAM; i++ ) 
	  if (paramData[i].description.length()) fprintf( f, "\n%s -paramdescription { %i { %s } } ", get_name(), i, paramData[i].description.getValue() );
  }
};


//////////// geometry level:
class RT_Geometry {
protected:
  RT_Geometry() { }
  int geomDim;
public:
  virtual int get_geomDim()const { return geomDim; }
};

class RT_1DGeometry: public RT_Geometry {
protected:
  RT_1DGeometry(): RT_Geometry() { geomDim = 1; }
};

class RT_2DGeometry: public RT_Geometry {
protected:
  RT_2DGeometry(): RT_Geometry() { geomDim = 2; }
};

class RT_3DGeometry: public RT_Geometry {
protected:
  RT_3DGeometry(): RT_Geometry() { geomDim = 3; }
};

//////////// topology level:


class RT_Lattice: 
public RT_Scientific {
protected:
  RT_Lattice(char *_name): RT_Scientific(_name) {}
  
  virtual void getVertex( int *, double*)const = 0;
  virtual void createParameters() = 0;
public:
  void setParameters( int *geo, double *p );
  void newMatrix() { RT_Primitive::newMatrix(); }
};

#endif
