////////////////////////////////////////////////////////////////////////////////
//  Source code of Classes for Scientific Visualization.                      //  
//  LAST EDIT: Thu Aug 11 14:15:53 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//              yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include <scient/2d_lat.h>

// functions to access several kinds of parameters
void l2DgetDfunc( const ParamSet &pS, int _x, int _y, int dimY, double *p ) {
    *p = pS.dData[ _x * dimY + _y ];
}

void l2DgetFfunc( const ParamSet &pS, int _x, int _y, int dimY, double *p ) {
    *p = pS.fData[ _x * dimY + _y ];
}

void l2DgetIfunc( const ParamSet &pS, int _x, int _y, int dimY, double *p ) {
    *p = pS.iData[ _x * dimY + _y ];
}

void l2DgetCfunc( const ParamSet &pS, int _x, int _y, int dimY, double *p ) {
    *p = pS.cData[ _x * dimY + _y ];
}

void l2DgetDefaultfunc( const ParamSet &, int , int , int , double * ) {
    rt_Output->errorVar(" No parameters given. ",0 );
}


void RT_2DLattice::create() {
    if ( (!dims[0]) || (!dims[1]) ) {
	rt_Output->errorVar( get_name(), " No geometry initialized. ", 0 );
	return;
    }
    if (quadmesh.get_x() != dims[1]
	|| quadmesh.get_y() != dims[0]) quadmesh.newGeometry( dims[1], dims[0] );

    int co[2]; double vt[3];
    double pm[1];
    vt[2] = 0.0;
    for (int i = 0; i < dims[1]; i++ ) {
	co[1] = i;
	for (int j = 0; j < dims[0]; j++ ) {
	    co[0] = j;
	    getVertex( co, vt );
	    quadmesh.vtPoint( i, j, (RT_Vector&)vt );
	}
    }

    int isopar = -1;
    int cpar = -1;
    for (i=0; i<MAX_PARAM; i++ ) {
	if (paramData[i].mappings[ Z_MAP ]) zmap( i );
	if (paramData[i].mappings[ COLOR_MAP ]) cpar = i;
	if (paramData[i].mappings[ ISO_MAP ]) isopar = i;
    }
    if ( cpar > -1 ) cmap( cpar, hueMin, hueMax, saturation, value );
        else delete_cmap();
    if ( isopar > -1 ) drawIso( isopar );
        else delete_isolines();
    switch ( gmode ) {
      case NONE: 
	quadmesh.invisible();
	break;
      case HOLLOW:
	quadmesh.visible();
	quadmesh.fillstyle( 0 );
	break;
      case FILLED:
	quadmesh.visible();
	quadmesh.fillstyle( 1 );
	break;
    }
}

void RT_2DLattice::createGeometry() {
}
 
void RT_2DLattice::createParameters() {
}

void RT_2DLattice::setVertex(int *co, double *vt) {
    if (!geometry) {
        rt_Output->errorVar( get_name(), ": Bad vertex access! No geometry field initialized.", 0 );
        return;
    }
    int offset = 0; 
    switch (get_geomDim()) {
      case 3: offset += co[2] * dims[0] * dims[1];   // tdims[1];
      case 2: offset += co[1] * dims[0];             // tdims[0];
      case 1: offset += co[0];
    break;
      default:
      rt_Output->errorVar( get_name(), ": Bad geometrical dimensions specified!", 0 );
    };
    double *tmp = &geometry[offset * coord];
    for (int i = 0; i < coord; i++) *tmp++ = *vt++;
}

void RT_2DLattice::getVertex(int *co, double *vt)const {
    if (!geometry) {
        rt_Output->errorVar( get_name(), ": Bad vertex access! No geometry field initialized.", 0 );
        return;
    }
    int offset = 0; 
    switch (get_geomDim()) {
      case 3: offset += co[2] * dims[0] * dims[1];    // tdims[1];
      case 2: offset += co[1] * dims[0];              // tdims[0];
      case 1: offset += co[0];
    break;
      default:
      rt_Output->errorVar( get_name(), ": Bad geometrical dimensions specified!", 0 );
    };
    double *tmp = &geometry[offset * coord];
    for (int i = 0; i < coord; i++) *vt++ = *tmp++;
}


void RT_2DLattice::zmap( int _pnr ) {
    int co[2];
    double vt[3];
    double p;
    if (!paramData[_pnr].loaded) return;
    for (int i = 0; i < dims[1]; i++ ) {
	co[1] = i;
	for (int j = 0; j < dims[0]; j++ ) {
	    co[0] = j;
	    getVertex( co, vt );
	    if ( _pnr ) {
		getParameter( _pnr, j, i, &p );
		vt[2] = p;
	    } else vt[2] = 0.0;
	    quadmesh.vtPoint( i, j, (RT_Vector&)vt );
	}
    }
}

void RT_2DLattice::getParExtremes( int _pnr, double *pmin, double *pmax ) {
    double p;
    getParameter( _pnr, 0, 0, pmin );
    getParameter( _pnr, 0, 0, pmax );
    for ( int i=0; i<dims[1]; i++ ) {
	for ( int j=0; j<dims[0]; j++ ) {
	    getParameter( _pnr, j, i, &p );
	    *pmin = ( *pmin < p ) ? *pmin : p;
	    *pmax = ( *pmax > p ) ? *pmax : p;
	}
    }
}

void RT_2DLattice::setParExtremes( int _pnr, double _pmin, double _pmax ) {
    paramData[_pnr].Pmin = _pmin;
    paramData[_pnr].Pmax = _pmax;
}

void RT_2DLattice::updateParExtremes( int _pnr ) {
    double pmin, pmax;
    getParExtremes( _pnr, &pmin, &pmax );
    setParExtremes( _pnr, pmin, pmax );
}

void RT_2DLattice::setParameter(int pnr, int _x, int _y, double _p ) {
    paramData[pnr].dData[ _x * dims[1] + _y ] = _p;
}

void RT_2DLattice::setParameter(int pnr, int _x, int _y, float _p ) {
    paramData[pnr].fData[ _x * dims[1] + _y ] = _p;
}
void RT_2DLattice::setParameter(int pnr, int _x, int _y, int _p ) {
    paramData[pnr].iData[ _x * dims[1] + _y ] = _p;
}
void RT_2DLattice::setParameter(int pnr, int _x, int _y, char _p ) {
    paramData[pnr].cData[ _x * dims[1] + _y ] = _p;
}

void RT_2DLattice::getParameter(int pnr, int _x, int _y, double *p )const {
    l2Dgetfunc[pnr]( paramData[pnr], _x, _y, dims[1], p );
}

void RT_2DLattice::setParamType( int _np, RT_ScType _pt ) {
    paramData[ _np ].paramType = _pt;
    switch ( _pt ) {
      case RTSC_DOUBLE: l2Dgetfunc[_np] = l2DgetDfunc; break;
      case RTSC_FLOAT: l2Dgetfunc[_np] = l2DgetFfunc; break;
      case RTSC_INTEGER: l2Dgetfunc[_np] = l2DgetIfunc; break;
      case RTSC_CHAR: l2Dgetfunc[_np] = l2DgetCfunc; break;
    }
}

void RT_2DLattice::loadParameters( int _np, RT_ScType _pt, double *_p ) {
    if ( _pt != RTSC_DOUBLE ) {
	rt_Output->warningVar( get_name(), ": RTSC_DOUBLE is not the type of the corresponding parameter-array. This may cause problems later. ", 0 );
	return;
    }
    if ( _np > MAX_PARAM ) {
	char tmp[20];
	sprintf( tmp, "%d", MAX_PARAM );
	rt_Output->errorVar( get_name(), ": Sorry, only ", tmp, " parameters per vertex allowed.", 0 );
	return;
    }
    if ( paramData[ _np ].loaded ) {
	char tmp[20];
	sprintf( tmp, "%d", _np );
	rt_Output->warningVar( get_name(),": Parameter number ", tmp, " was in use. Now it is overloaded", 0 );
    }
    setParamType( _np, _pt );
    currParNr = _np;
    paramData[ currParNr ].loaded = 1;
    newParameters();
    for ( int i=0; i < dims[0]; i++ ) {
	for ( int j=0; j < dims[1]; j++ ) {
	    setParameter( _np, i, j, _p[ i*dims[1] + j] );
	}
    }
    updateParExtremes( _np );
    geomChanged();
}

void RT_2DLattice::loadParameters( int _np, RT_ScType _pt, float *_p ) {
    if ( _pt != RTSC_FLOAT ) {
	rt_Output->warningVar(get_name(), ": RTSC_FLOAT is not the type of the corresponding parameter-array. This may cause problems later. ", 0 );
	return;
    }
    if ( _np > MAX_PARAM ) {
	char tmp[20];
	sprintf( tmp, "%d", MAX_PARAM );
	rt_Output->errorVar( get_name(), ": Sorry, only ", tmp, " parameters per vertex allowed.", 0 );
	return;
    }
    if ( paramData[ _np ].loaded ) {
	char tmp[20];
	sprintf( tmp, "%d", _np );
	rt_Output->warningVar( get_name(), ": Parameter number ", tmp, " was in use. Now it is overloaded", 0 );
    }
    setParamType( _np, _pt );
    currParNr = _np;
    paramData[ currParNr ].loaded = 1;
    newParameters();
    for ( int i=0; i < dims[0]; i++ ) {
	for ( int j=0; j < dims[1]; j++ ) {
	    setParameter( _np, i, j, _p[ i*dims[1] + j] );
	}
    }
    updateParExtremes( _np );
    geomChanged();
}

void RT_2DLattice::loadParameters( int _np, RT_ScType _pt, int *_p ) {
    if ( _pt != RTSC_INTEGER ) {
	rt_Output->warningVar(get_name(),": RTSC_INTEGER is not the type of the corresponding parameter-array. This may cause problems later. ", 0 );
	return;
    }
    if ( _np > MAX_PARAM ) {
	char tmp[20];
	sprintf( tmp, "%d", MAX_PARAM );
	rt_Output->errorVar(get_name(), ": Sorry, only ", tmp, " parameters per vertex allowed.", 0 );
	return;
    }
    if ( paramData[ _np ].loaded ) {
	char tmp[20];
	sprintf( tmp, "%d", _np );
	rt_Output->warningVar( get_name(), ": Parameter number ", tmp, " was in use. Now it is overloaded", 0 );
    }
    setParamType( _np, _pt );
    currParNr = _np;
    paramData[ currParNr ].loaded = 1;
    newParameters();
    for ( int i=0; i < dims[0]; i++ ) {
	for ( int j=0; j < dims[1]; j++ ) {
	    setParameter( _np, i, j, _p[ i*dims[1] + j] );
	}
    }
    updateParExtremes( _np );
    geomChanged();
}

void RT_2DLattice::loadParameters( int _np, RT_ScType _pt, char *_p ) {
    if ( _pt != RTSC_CHAR ) {
	rt_Output->warningVar(get_name(),": RTSC_CHAR is not the type of the corresponding parameter-array. This may cause problems later. ", 0 );
	return;
    }
    if ( _np > MAX_PARAM ) {
	char tmp[20];
	sprintf( tmp, "%d", MAX_PARAM );
	rt_Output->errorVar( get_name(), ": Sorry, only ", tmp, " parameters per vertex allowed.", 0 );
	return;
    }
    if ( paramData[ _np ].loaded ) {
	char tmp[20];
	sprintf( tmp, "%d", _np );
	rt_Output->warningVar( get_name(), ": Parameter number ", tmp, " was in use. Now it is overloaded", 0 );
    }
    setParamType( _np, _pt );
    currParNr = _np;
    paramData[ currParNr ].loaded = 1;
    newParameters();
    for ( int i=0; i < dims[0]; i++ ) {
	for ( int j=0; j < dims[1]; j++ ) {
	    setParameter( _np, i, j, _p[ i*dims[1] + j] );
	}
    }
    updateParExtremes( _np );
    geomChanged();
}

int RT_2DLattice::parF;
char *RT_2DLattice::parV;

RT_ParseEntry RT_2DLattice::table[] = {
    { "-loadParameters", RTP_STRING, (char*)&parV, &parF, "Specify the {ARG 1 ParameterIdentifier} and the {ARG 2 ParameterList} for the lattice.",RTPS_STRING },
	{ 0, RTP_END, 0, 0, 0, 0 }
};

int RT_2DLattice::objectCMD(char *argv[]) {
    int t = RT_Scientific::objectCMD( argv );
    RT_parseTable( argv, table );
    if (parF) { 
	char **xargv; int xargc;
	int _parId; double *_p = 0;
	if ( Tcl_SplitList( rt_Ip, parV, &xargc, &xargv) == TCL_OK) {
	    if ( xargc == 2 ) {
		RT_string2int( xargv[0], _parId );
		char **xxargv; int xxargc;
		if ( Tcl_SplitList( rt_Ip, xargv[1], &xxargc, &xxargv) == TCL_OK) {
		    _p = new double[ xxargc ];
		    for ( int i=0; i<xxargc; i++ ) RT_string2double( xxargv[i], _p[i] );
		    loadParameters( _parId, RTSC_DOUBLE, _p );
		    t++;
		    free((char*)xxargv);
		}
	    }
	    free((char*)xargv);
	    if (_p) delete _p;
	}
    }
    return t;
}

/////////////////////////////////////////////////////////////////////////////////////
//     Implementration of a 2D perimetric lattice
/////////////////////////////////////////////////////////////////////////////////////

void RT_2DPerimetricLattice::createGeometry() {
}
 
void RT_2DPerimetricLattice::createParameters() {
}

void RT_2DPerimetricLattice::findParameters() {
}

void RT_2DPerimetricLattice::loadGeometry( int _nx, int _ny, double *_x, double *_y ) {
    dims[0] = _nx;
    dims[1] = _ny;
    newGeometry();
    int indices[3];
    double vtx[4];
    for ( int i=0; i < _nx; i++ ) {
	for ( int j=0; j < _ny; j++ ) {
	    indices[0] = i; indices[1] = j;
	    vtx[0] = _x[i]; 
	    vtx[1] = _y[j];
	    vtx[2] = vtx[3] = 0.0;
	    setVertex( indices, vtx );
	}
    }
}

int RT_2DPerimetricLattice::pgeoF;
char *RT_2DPerimetricLattice::pgeoV;

RT_ParseEntry RT_2DPerimetricLattice::table[] = {
    { "-loadGeometry", RTP_STRING, (char*)&pgeoV, &pgeoF, "Specify {ARG 1 DimX}, {ARG 2 DimY}, {ARG 3 X-vector}, {ARG 4 Y-vector} for the geometry of the lattice.",RTPS_STRING },
	{ 0, RTP_END, 0, 0, 0, 0 }
};

int RT_2DPerimetricLattice::objectCMD(char *argv[]) {
    int t = RT_2DLattice::objectCMD( argv );
    RT_parseTable( argv, table );
    if (pgeoF) { 
	char **xargv; int xargc;
	int nx, ny;
	double *xvec = 0; double *yvec = 0;
	if ( Tcl_SplitList( rt_Ip, pgeoV, &xargc, &xargv) == TCL_OK) {
	    if ( xargc == 4 ) {
		RT_string2int( xargv[0], nx );
		RT_string2int( xargv[1], ny );
		xvec = new double[ nx ]; yvec = new double[ ny ];
		char **xxargv; int xxargc;
		if ( Tcl_SplitList( rt_Ip, xargv[2], &xxargc, &xxargv) == TCL_OK) {
		    if ( xxargc == nx ) for ( int i=0; i<nx; i++ ) RT_string2double( xxargv[i], xvec[i] );
		    free((char*)xxargv);
		}
		if ( Tcl_SplitList( rt_Ip, xargv[3], &xxargc, &xxargv) == TCL_OK) {
		    if ( xxargc == ny ) for ( int i=0; i<ny; i++ ) RT_string2double( xxargv[i], yvec[i] );
		    free((char*)xxargv);
		}
		loadGeometry( nx, ny, xvec, yvec );
		if (xvec) delete xvec;
		if (yvec) delete yvec;
		t++;
	    }
	    free((char*)xargv);
	}
    }
    return t;
}

int RT_2DPerimetricLattice::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", RTN_2D_P_LATTICE, " { String } {  Creates a 2D perimetric lattice. {ARG 1 Name} of the new object.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	if (argc != 2) { 
	    Tcl_AppendResult( ip, "Bad number of arguments. Try \"", RTN_2D_P_LATTICE, " ?\".", 0 ); 
	    return TCL_ERROR;
	}
	new RT_2DPerimetricLattice( argv[1] );
	return TCL_OK;
    }
    else return TCL_ERROR;
}


int RT_2DPerimetricLattice::isA(const char *_c)const {
    if (RTM_isA(_c, RTN_2D_P_LATTICE )) return 1;
    return RT_Scientific::isA( _c );
}

///////////////////////////////////////////////////////////////////////////////
//     Implementation of a 2D uniform lattice
///////////////////////////////////////////////////////////////////////////////

void RT_2DUniformLattice::createGeometry() {
}
 
void RT_2DUniformLattice::createParameters() {
}

void RT_2DUniformLattice::findParameters() {
}

void RT_2DUniformLattice::loadGeometry( int _nx, int _ny, double _x0, double _y0, double _dx, double _dy ) {
    dims[0] = _nx;
    dims[1] = _ny;
    newGeometry();
    int indices[3];
    double vtx[4];
    double x = x0 = _x0; 
    double y = y0 = _y0;
    dx = _dx; dy = _dy;
    for ( int i=0; i < _nx; i++ ) {
	for ( int j=0; j < _ny; j++ ) {
	    indices[0] = i; indices[1] = j;
	    vtx[0] = x; 
	    vtx[1] = y;
	    vtx[2] = vtx[3] = 0.0;
	    setVertex( indices, vtx );
	    y += _dy;
	}
	x += _dx;
	y = _y0;
    }
}

int RT_2DUniformLattice::ugeoF;
char *RT_2DUniformLattice::ugeoV;

RT_ParseEntry RT_2DUniformLattice::table[] = {
    { "-loadGeometry", RTP_STRING, (char*)&ugeoV, &ugeoF, "Specify {ARG 1 DimX}, {ARG 2 DimY}, {ARG 3 X0}, {ARG 4 Y0}, {ARG 5 DX}, {ARG 6 DY} for the geometry of the lattice.",RTPS_STRING },
	{ 0, RTP_END, 0, 0, 0, 0 }
};

int RT_2DUniformLattice::objectCMD(char *argv[]) {
    int t = RT_2DLattice::objectCMD( argv );
    RT_parseTable( argv, table );
    if (ugeoF) { 
	char **xargv; int xargc;
	int nx, ny;
	double x0, y0, dx, dy;
	if ( Tcl_SplitList( rt_Ip, ugeoV, &xargc, &xargv) == TCL_OK) {
	    if ( xargc == 6 ) {
		RT_string2int( xargv[0], nx );
		RT_string2int( xargv[1], ny );
		RT_string2double( xargv[2], x0 );
		RT_string2double( xargv[3], y0 );
		RT_string2double( xargv[4], dx );
		RT_string2double( xargv[5], dy );
		loadGeometry( nx, ny, x0, y0, dx, dy );
		t++;
	    }
	    free((char*)xargv);
	}
    }
    return t;
}

int RT_2DUniformLattice::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", RTN_2D_U_LATTICE, " { String } {  Creates a 2D uniform lattice. {ARG 1 Name} of the new object.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	if (argc != 2) { 
	    Tcl_AppendResult( ip, "Bad number of arguments. Try \"", RTN_2D_U_LATTICE, " ?\".", 0 ); 
	    return TCL_ERROR;
	}
	new RT_2DUniformLattice( argv[1] );
	return TCL_OK;
    }
    else return TCL_ERROR;
}


int RT_2DUniformLattice::isA(const char *_c)const {
    if (RTM_isA(_c, RTN_2D_U_LATTICE )) return 1;
    return RT_Scientific::isA( _c );
}

//////////////////////////////////////////////////////////////////////////////
//      Implementation of a 2D curvilinear lattice
//////////////////////////////////////////////////////////////////////////////

void RT_2DCurvilinearLattice::createGeometry() {
}
 
void RT_2DCurvilinearLattice::createParameters() {
}

void RT_2DCurvilinearLattice::findParameters() {
}

void RT_2DCurvilinearLattice::loadGeometry( int _nx, int _ny, double *_x, double *_y ) {
    dims[0] = _nx;
    dims[1] = _ny;
    newGeometry();
    int indices[3];
    double vtx[4];
    for ( int i=0; i < _nx; i++ ) {
	for ( int j=0; j < _ny; j++ ) {
	    indices[0] = i; indices[1] = j;
	    vtx[0] = _x[i * _ny + j]; 
	    vtx[1] = _y[i * _ny + j];
	    vtx[2] = vtx[3] = 0.0;
	    setVertex( indices, vtx );
	}
    }
}

int RT_2DCurvilinearLattice::cgeoF;
char *RT_2DCurvilinearLattice::cgeoV;

RT_ParseEntry RT_2DCurvilinearLattice::table[] = {
    { "-loadGeometry", RTP_STRING, (char*)&cgeoV, &cgeoF, "Specify {ARG 1 DimX}, {ARG 2 DimY}, {ARG 3 X-vector}, {ARG 4 Y-vector} for the geometry of the lattice.",RTPS_STRING },
	{ 0, RTP_END, 0, 0, 0, 0 }
};

int RT_2DCurvilinearLattice::objectCMD(char *argv[]) {
    int t = RT_Scientific::objectCMD( argv );
    RT_parseTable( argv, table );

    if (cgeoF) { 
	char **xargv; int xargc;
	int nx, ny;
	double *xvec = 0; double *yvec = 0;
	if ( Tcl_SplitList( rt_Ip, cgeoV, &xargc, &xargv) == TCL_OK) {
	    if ( xargc == 4 ) {
		RT_string2int( xargv[0], nx );
		RT_string2int( xargv[1], ny );
		int nxy = nx * ny;
		xvec = new double[ nxy ]; yvec = new double[ nxy ];
		char **xxargv; int xxargc;
		if ( Tcl_SplitList( rt_Ip, xargv[2], &xxargc, &xxargv) == TCL_OK) {
		    if ( xxargc == nxy ) for ( int i=0; i<nxy; i++ ) RT_string2double( xxargv[i], xvec[i] );
		    free((char*)xxargv);
		}
		if ( Tcl_SplitList( rt_Ip, xargv[3], &xxargc, &xxargv) == TCL_OK) {
		    if ( xxargc == nxy ) for ( int i=0; i<nxy; i++ ) RT_string2double( xxargv[i], yvec[i] );
		    free((char*)xxargv);
		}
		loadGeometry( nx, ny, xvec, yvec );
		if (xvec) delete xvec;
		if (yvec) delete yvec;
		t++;
	    }
	    free((char*)xargv);
	}
    }
    return t;
}

int RT_2DCurvilinearLattice::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", RTN_2D_C_LATTICE, " { String } {  Creates a 2D curvilinear lattice. {ARG 1 Name} of the new object.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	if (argc != 2) { 
	    Tcl_AppendResult( ip, "Bad number of arguments. Try \"", RTN_2D_C_LATTICE, " ?\".", 0 ); 
	    return TCL_ERROR;
	}
	new RT_2DCurvilinearLattice( argv[1] );
	return TCL_OK;
    }
    else return TCL_ERROR;
}

int RT_2DCurvilinearLattice::isA(const char *_c)const {
    if (RTM_isA(_c, RTN_2D_C_LATTICE )) return 1;
    return RT_Scientific::isA( _c );
}
