////////////////////////////////////////////////////////////////////////////////
// It\'s a demonstration of some features of 3D-lattices                      //
// A 3D-lattice itself is father of 9 2D-lattices ( six for the border        //
// planes and three for walkplanes in each direction).                        //
//  LAST EDIT: Thu Jun 30 16:42:30 1994 by Barth(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include <yart.h>
#include <scient/scient.h>
#include <unistd.h>
#include <math.h>

#define NX 30
#define NY 25
#define NZ 10
#define DX 0.09
#define DY 0.1
#define DZ 0.4

// This program demonstrates two kinds of 3D-lattices, uniform and perimetric.
// for further information on lattices please read the documentation.
// Both lattices will have the same geometry and the same parameters. The 
// difference is only in loading the geometry. To choose the kind of lattice,
// please comment/uncomment the following two lines:
#define PERIMETRIC_DEMO
// #define UNIFORM_DEMO

#ifdef PERIMETRIC_DEMO
 RT_3DPerimetricLattice *ball;
#endif

#ifdef UNIFORM_DEMO
 RT_3DUniformLattice *ball;
#endif

RT_Scene *sc;


void initIsoLines() {
// Level and color for each isoline are specified here:
    double pmin, pmax, p, diff;
    ball->getParExtremes( 1, &pmin, &pmax );
    diff = ( pmax - pmin ) / 11.0;
    p = pmin + diff;
    ball->isolevel(0, p, RT_Color(1, 0, 0 ) ); p += diff;
    ball->isolevel(1, p, RT_Color(0, 1, 0 ) ); p += diff;
    ball->isolevel(2, p, RT_Color(0, 1, 0 ) ); p += diff;
    ball->isolevel(3, p, RT_Color(1, 1, 0 ) ); p += diff;
    ball->isolevel(4, p, RT_Color(0, 1, 1 ) ); p += diff;
    ball->isolevel(5, p, RT_Color(1, 0, 1 ) ); p += diff;
    ball->isolevel(6, p, RT_Color(0, 1, 0 ) ); p += diff;
    ball->isolevel(7, p, RT_Color(0.5, 1, 0.5 ) ); p += diff;
    ball->isolevel(8, p, RT_Color(1, 0, 0.5 ) ); p += diff;
    ball->isolevel(9, p, RT_Color(0.5, 1, 0 ) );
}

void drawText() {
// Some text strings for explanation are drawn here. They become
// childs of the lattice.
    RT_Text *l, *d, *w;
    sc->insert( l = new RT_Text("lenText", "length", RT_Vector(0, 0, 0)));
    l->height(0.35);
    l->width( 1.0 );
    l->translate( RT_Vector( 2, 0, 4.0 ));
    l->rotate( RT_Vector( -M_PI_2, 0, 0 ));
    l->father(ball);

    sc->insert( w = new RT_Text("widthText", "depth", RT_Vector(0, 0, 0)));
    w->height(0.35);
    w->width( 1.0 );
    w->translate( RT_Vector( 3.8, 0, 3.5 ));
    w->rotate( RT_Vector( -M_PI_2, 0, M_PI_2 ));
    w->father(ball);

    sc->insert( d = new RT_Text("depthText", "  width", RT_Vector(0, 0, 0)));
    d->height(0.35);
    d->width( 1.0 );
    d->translate( RT_Vector( 0, 0, 3.8 ));
    d->rotate( RT_Vector( M_PI_2, M_PI_2, 0 ));
    d->father(ball);
}

void drawCoords() {
// The coordinate system
    RT_Coord *cx, *cy, *cz;
    RT_AxisType at = X;
    sc->insert( cx = new RT_Coord( "cx", X) );
    RT_AxisType yt = Y;
    sc->insert( cy = new RT_Coord( "cy", yt) );
    sc->insert( cz = new RT_Coord( "cz", Z) );
    cx->start( 0.0 );
    cx->len( 4.0 );
    cx->incr( 1 );
    cx->labels( OFF );

    cy->len( 3.0 );
    cy->start( 0.0 );
    cy->incr( 1.0 );
    cy->labels( OFF );

    cz->len( 4.0 );
    cz->incr( 1.0 );
}

void mapping() {
// Three of the outer planes of the lattice will be invisible, because
// you want to have a look "in" the parameter volume
    ball->pl2d[FRONT_PLANE]->invisible();
    ball->pl2d[TOP_PLANE]->invisible();
    ball->pl2d[RIGHT_PLANE]->invisible();
// The parameter no. 1 ( which is the only one in this example ) will
// be mapped to color for all planes of the lattice
    ball->map( 1, COLOR_MAP );
    ball->gridmode( FILLED );
}

int main() {
    int i,j,k;

    RT_init();
    RT_PixmapDisplay *px = new RT_PixmapDisplay( "px", 500, 500 );
    px->singlebuffer();

    sc = new RT_Scene("sc");

#ifdef UNIFORM_DEMO
// A uniform lattice will be created an gets the data for the geometry
    sc->insert( ball = new RT_3DUniformLattice("ball") );
    ball->loadGeometry( NX, NY, NZ, 0, 0, 0, DX, DY, DZ );
#endif

#ifdef PERIMETRIC_DEMO
// The same lattice. Of course, it\'s senceless to fill a perimetric
// lattice with uniform geometry, but for a demo ...
    sc->insert( ball = new RT_3DPerimetricLattice("ball") );
    double *gx = new double[NX]; 
    double *gy = new double[NY]; 
    double *gz = new double[NZ];
    for (i=0; i<NX; i++ ) gx[i] = (double)i * DX;
    for (i=0; i<NY; i++ ) gy[i] = (double)i * DY;
    for (i=0; i<NZ; i++ ) gz[i] = (double)i * DZ;
    ball->loadGeometry( NX, NY, NZ, gx, gy, gz );
    delete gx; delete gy; delete gz;
#endif

    int d3 = NX * NY * NZ;
    double *dummy = new double[ d3 ];
    for ( i=0; i<d3; i++ ) dummy[i] = 0.0;
    ball->loadParameters( 1, RTSC_DOUBLE, dummy );

    double x, y, z;
    x = y = z = 0.0;
    for ( i=0; i<NZ; i++ ) {
	for ( j=0; j<NY; j++ ) {
	    for ( k=0; k<NX; k++ ) {
		ball->setParameter(1, k,j,i,sqrt(x*x + y*y + z*z));
		x += DX;
	    }
	    y += DY; x = 0.0;
	}
	z += DZ; x = 0.0; y = 0.0;
    }

    ball->setPlaneParameters(1);

// Now, the first picture can be generated:
    initIsoLines();
    mapping();
    drawText();
    drawCoords();

    RT_Light *ambSource;
    sc->insert( ambSource = new RT_AmbientLight( "al" ) );

    RT_LookatCamera *cam = new RT_LookatCamera("cam", RT_Vector( 7, 5, 10), RT_Vector(0,0,0));
    cam->zfar( 15 ); cam->angle( 35 );

    cam->pixmap( px );
    cam->scene( sc );

    cam->rendering();

    char cmd[100];
    fprintf( stdout, "Type \"run\" to have a little animation.\n" );
    while (1) {
	fprintf( stdout, "-> " );
	fgets( cmd, 100, stdin ); 
	if (!strcmp( cmd, "run\n" )) break;
	fprintf( stdout, "%s\n", RT_eval( cmd ) );
	cam->rendering();
    }
    while(1) {
	ball->gridmode( HOLLOW );
	ball->unmap( 1, COLOR_MAP );
	ball->map( 1, ISO_MAP );
	ball->pl2d[ BACK_PLANE ]->isomode( ISO_LINES );
	ball->pl2d[ BOTTOM_PLANE ]->isomode( ISO_LINES );
	ball->pl2d[ LEFT_PLANE ]->isomode( ISO_LINES );

// send the "walkplanes" through the volume:
	ball->walkPlane[ XY ]->visible();
	ball->walkPlane[ XY ]->gridmode( FILLED );
	ball->walkPlane[ XY ]->map( 1, COLOR_MAP );
	for ( i=0; i<NZ; i++ ) {
	    ball->walkto( XY, i );
	    cam->rendering();
	    sleep(5);
	}
	ball->walkPlane[ XY ]->invisible();

	ball->walkPlane[ XZ ]->visible();
	ball->walkPlane[ XZ ]->gridmode( FILLED );
	ball->walkPlane[ XZ ]->map( 1, COLOR_MAP );
	for ( i=0; i<NY; i += 2 ) {
	    ball->walkto( XZ, i );
	    cam->rendering();
	    sleep(5);
	}
	ball->walkPlane[ XZ ]->invisible();

	ball->walkPlane[ YZ ]->visible();
	ball->walkPlane[ YZ ]->gridmode( FILLED );
	ball->walkPlane[ YZ ]->map( 1, COLOR_MAP );
	for ( i=0; i<NX; i += 3 ) {
	    ball->walkto( YZ, i );
	    cam->rendering();
	    sleep(5);
	}
	ball->walkPlane[ YZ ]->invisible();

	ball->unmap( 1, ISO_MAP );
	ball->gridmode( HOLLOW );
	ball->unmap( 1, COLOR_MAP );
	ball->pl2d[ BACK_PLANE ]->map( 1, COLOR_MAP );
	ball->pl2d[ BACK_PLANE ]->gridmode( FILLED );
	for ( i=0; i<2; i++ ) {
	    ball->walkPlane[ i ]->map( 1, ISO_MAP );
	    ball->walkPlane[ i ]->gridmode( FILLED );
	    ball->walkPlane[ i ]->isomode( ISO_LINES );
	    ball->walkPlane[ i ]->visible();
	}
	ball->walkPlane[ XZ ]->isooffset( -0.05 );
	ball->walkPlane[ XY ]->isooffset( 0.05 );
	i = j = 0;
	while (( i<NY ) && ( j<NZ )) {
	    ball->walkto( XZ, i );
	    ball->walkto( XY, j );
	    i += 3; j += 1;
	    cam->rendering();
	    sleep(4);
	}
	ball->walkPlane[ XZ ]->invisible();
	ball->walkPlane[ YZ ]->invisible();
	ball->walkPlane[ XZ ]->unmap( 1, ISO_MAP );
	ball->walkPlane[ YZ ]->unmap( 1, ISO_MAP );
    }
}


