//  LAST EDIT: Mon Feb 14 01:13:34 1994 by ekki(@prakinf.tu-ilmenau.de)
#ifndef _DrawTreeNode_h_
#define _DrawTreeNode_h_

/*
 * DrawTreeNode.h - class definitions for drawing tree nodes
 *
 * -----------------------------------------------------------------------------
 * Copyright 1993 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  Allan
 * Brighton make no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 * -----------------------------------------------------------------------------
 */
 
#include <OS/string.h>
#include <Polygon.h>
#include <OS/list.h>

declarePtrList(PolylinePtrList, Polyline);

/*
 * This class defines routines for drawing trees.  The window system
 * specific parts must be implemented in a derived class.
 * 
 * The basic structure and algorithm is taken from:
 *     IEEE Software: July 1990 "Drawing Dynamic Trees")
 */
class DrawTreeNode {
private:

    CopyString tag_;					// node's label
    
    DrawTreeNode *parent_, *child_, *sibling_;
    int width_, height_, border_;
    Point pos_, prev_pos_, offset_;
    Polygon contour_;
    
    // memory management
    PolylinePtrList polylines_;			// used for garbage collection
    Polyline* newPolyline(int x, int y, Polyline* pl = 0);
    
    // Routines for calculating the positions of the nodes in the tree
    void layout();
    void layout_leaf();
    void attach_parent(int h);
    int join();
    int merge(Polygon&, Polygon&);
    int offset(int p1, int p2, int a1, int a2, int b1, int b2);
    Polyline* bridge(Polyline* line1, int x1, int y1, Polyline* line2, int x2, int y2);
    void branch(DrawTreeNode* child, DrawTreeNode* sibling);
    void unzip();
    void zip();
    
    // calculate the absolute positions of the tree nodes
    void setXY(int& x1, int& y1, int& x2, int& y2);
    void setPosFrom(DrawTreeNode* pred, int& x1, int& y1, int& x2, int& y2);
    
    // draw the tree, calling virtual drawNode for each node
    void draw();
    
    
    // member functions redefined in derived classes (but only called from this class)

    // draw a tree node
    virtual void drawNode() const = 0;
    
    // remove a node from the tree
    virtual void removeNode() const = 0;
    
    // redraw the line to the parent node at the given coords
    virtual void drawLine(const Point& p1, const Point& p2) const = 0;
    
    // remove a line fom the nodes parent
    virtual void removeLine() const = 0;
    
    // called with the bounding box of the tree before drawing
    virtual void setDrawArea(int x1, int y1, int x2, int y2) const = 0;
    
    // distance from parent node
    virtual int parentDistance() const {return 30;}
    
    // tree's orientation 
    virtual int horizontal() const {return 1;}
    int vertical() const {return !horizontal();}
    
public:
    
    // constructor 
    DrawTreeNode(const String& tag, int x=0, int y=0, int w=0, int h=0, int border=0);
    
    // clean up the mess...	
    virtual ~DrawTreeNode();
    
    // insert the node in the tree
    void addLink(DrawTreeNode*);
    
    // remove and delete this node and its subnodes
    void rmLink();
    
    // unlink this subtree from the tree
    void unLink ();
    
    // remove and delete this nodes children
    void prune();
    
    // calculate the node positions and draw the tree
    void drawTree();
    
    // find node with tag
    DrawTreeNode* find(const String& tag);
    
    // member access
    const String& tag() const { return tag_; }
    const Point& pos() const { return pos_; }
    const Point& prev_pos() const { return prev_pos_; }
    int width() const { return width_; }
    int height() const { return height_; }
    int border() const { return border_; }
    DrawTreeNode* parent() const { return parent_; }
    DrawTreeNode* child() const { return child_; }
    DrawTreeNode* sibling() const { return sibling_; }

    // set the bounding box of a node
    void box(int x1, int y1, int x2, int y2);
    
    // set the position
    void pos(const Point& p) { pos_ = p; }
    
    // set the border width
    void border(int i) { border_ = i; }
};

#endif 
