################################################################################
##  Utility class for working with trees.                                     ##  
##  LAST EDIT: Thu Aug  4 13:54:50 1994 by ekki(@prakinf.tu-ilmenau.de)
################################################################################
##  This file belongs to the YART implementation. Copying, distribution and   ##
##  legal info is in the file COPYRGHT which should be distributed with this  ##
##  file. If COPYRGHT is not available or for more info please contact:       ##
##                                                                            ##  
##		yart@prakinf.tu-ilmenau.de                                    ##
##                                                                            ##  
## (C) Copyright 1994 YART team                                               ##
################################################################################

# -----------------------------------------------------------------------------
# Copyright 1993 Allan Brighton.
# 
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies.  Allan
# Brighton make no representations about the suitability of this software
# for any purpose.  It is provided "as is" without express or implied
# warranty.
# -----------------------------------------------------------------------------

# create a tree with the given callback actions
# text_action is evaluated when a text node is double-clicked and
# bitmap_action when a bitmap is double-clicked

proc tree_create {tree {text_action ""} {bitmap_action} {menu}} {
    global tree_private
    tree $tree

    set tree_private($tree,text_action) $text_action
    set tree_private($tree,bitmap_action) $bitmap_action
    set tree_private($tree,menu) $menu

    tree_set_default_bindings $tree

    return $tree
}
        
# set the bindings for the tree canvas
    
proc tree_set_default_bindings {tree} {
    global tree_private
    set canvas [$tree canvas]
    $canvas bind text <1> "focus %W; tree_selectNode $tree"
    $canvas bind bitmap <1> "tree_selectBitmap $tree"
    
    if [string length $tree_private($tree,menu)] {
	popup_to_canvas $canvas text $tree_private($tree,menu)
	popup_to_canvas $canvas bitmap $tree_private($tree,menu)
    }
    
    $canvas bind text <Double-Button-1> "after 1 $tree_private($tree,text_action)"
    $canvas bind bitmap <Double-Button-1> "$tree_private($tree,bitmap_action)"
    
    bind $canvas <ButtonPress-2> "$canvas scan mark %x %y"
    bind $canvas <B2-Motion> "$canvas scan dragto %x %y"
}
   
# make sure the tree is visible in the canvas window by scrolling to center
# the tree in the view

proc tree_center {tree} {
    set canvas [$tree canvas]
    set scrollincrement [utilGetConfigValue $canvas -scrollincrement]
    set scrollregion [utilGetConfigValue $canvas -scrollregion]
    scan $scrollregion "%d %d %d %d" x1 y1 x2 y2
    
    if {[utilGetConfigValue $tree -layout] == "horizontal"} {
	set height [utilGetConfigValue $canvas -height]
	$canvas yview [expr "(($y2 - $y1)/2 - $height/2)/$scrollincrement"]
	$canvas xview 0
    } else {
	set width [utilGetConfigValue $canvas -width]
	$canvas xview [expr "(($x2 - $x1)/2 - $width/2)/$scrollincrement"]
	$canvas yview 0
    }
}

# layout the components of the given node depending on whether
# the tree is vertical or horizontal

proc tree_layoutNode {tree node} {
    set canvas [$tree canvas]
    set label $node:text
    set bitmap $node:bitmap
    
    if {[utilGetConfigValue $tree -layout] == "horizontal"} {
	scan [$canvas bbox $label] "%d %d %d %d" x1 y1 x2 y2
	$canvas itemconfig $bitmap -anchor se
	$canvas coords $bitmap $x1 $y2
    } else {
	scan [$canvas bbox $bitmap] "%d %d %d %d" x1 y1 x2 y2
	$canvas itemconfig $label -anchor n
	$canvas coords $label [expr "$x1+($x2-$x1)/2"] $y2
    }
}
    
# add the given node to the tree under the given parent node 
# (if not already there)  using the given bitmap and
# the given label, if provided.

proc tree_addNode {tree parent node bitmap {label ""}} {
    set canvas [$tree canvas]
    
    if {"$label" == ""} {
	set label $node
    }
    
    # don't add the node if its already there
    if ![llength [$canvas find withtag $node]] {
	$canvas create bitmap 0 0 -bitmap "@$bitmap" \
		-foreground black \
		-tags [list $node bitmap $node:bitmap]
	
	$canvas create text 0 0 \
		-fill black \
		-text "$label" \
		-tags [list $node text $node:text]
	
	set line [$canvas create line 0 0 0 0 \
		-fill black \
		-tag "line" \
		-width 1 \
		-capstyle round ]
	tree_layoutNode $tree $node
	$tree addlink $parent $node $line -border 2
    } {
	if {"$parent"!="[$tree parent $node]"} {
	    $tree movelink $node $parent
	}
    }	
}

# select the current node's label
    
proc tree_selectNode {tree} {
    set canvas [$tree canvas]
    $canvas select from current 0
    $canvas select to current 	[string length [lindex [$canvas itemconf current -text] 4]]
    tree_deSelectBitmap $tree
    set current [tree_getCurrent $tree]
    if {$current==[lindex [$canvas gettags marked] 0]} {tree_deMarkBitmap $tree}
}
    
# de-select all node labels
    
proc tree_deselectNode {tree} {
    set canvas [$tree canvas]
    $canvas select clear
}

# highlight the node's bitmap
    
proc tree_selectBitmap {tree } {
    set canvas [$tree canvas]
    focus none
    set current [lindex [$canvas gettags current] 0]
    tree_deselectNode $tree

    if {$current==[lindex [$canvas gettags marked] 0]} {tree_deMarkBitmap $tree}
	
    tree_deSelectBitmap $tree
    set oldTags [$canvas gettags $current]
    regsub selected $oldTags "" oldTags


    $canvas itemconfig $current:bitmap \
	-background yellow\
	-tags "$oldTags selected" 
}

proc tree_selectNextBitmap {tree} {
    set canvas [$tree canvas]
    focus none
    set current [lindex [$canvas gettags current] 0]

    # unmark the node if it is marked
    if {$current==[lindex [$canvas gettags marked] 0]} {tree_deMarkBitmap $tree}

    # remove the selected tag of the current tag list
    set oldTags [$canvas gettags $current:bitmap]
    regsub selected $oldTags "" oldTags

    # if the node is selected then deselect the node
    if {-1!=[lsearch [$canvas gettags $current] selected]} {
    $canvas itemconfig $current:bitmap \
	-tags "$oldTags" \
	-background [utilGetConfigValue $canvas -background]
	return
    }

    # select the new node
    $canvas itemconfig $current:bitmap \
	-background yellow\
	-tags "$oldTags selected" 
}

proc tree_selectNodeBitmap {tree name} {
    set canvas [$tree canvas]

    if {$name==[lindex [$canvas gettags marked] 0]} {tree_deMarkBitmap $tree}
	
    set oldTags [$canvas gettags $name]
    regsub selected $oldTags "" oldTags

    $canvas itemconfig $name:bitmap \
	-background yellow\
	-tags "$oldTags selected" 
}

# stop highlighting the node's bitmap
    
proc tree_deSelectBitmap {tree} {
    set canvas [$tree canvas]
    $canvas itemconfig bitmap \
	-background [utilGetConfigValue $canvas -background]\
	-foreground black
    $canvas itemconfig marked \
	-background red
    $canvas dtag selected
}
   
# mark the node's bitmap
    
proc tree_markBitmap {tree {demark 1}} {
    set canvas [$tree canvas]
    focus none
    set current [lindex [$canvas gettags current] 0]
    tree_deselectNode $tree
    if $demark {tree_deMarkBitmap $tree}
    $canvas itemconfig $current:bitmap \
	-foreground black\
	-background red\
	-tags "[$canvas gettags $current:bitmap] marked" 
}

# stop marking the node's bitmap
    
proc tree_deMarkBitmap {tree} {
    set canvas [$tree canvas]
    $canvas itemconfig marked \
	-background [utilGetConfigValue $canvas -background]
    $canvas dtag marked
}
   
# return the text of the item currently selected
    
proc tree_getCurrent {tree} {
    set canvas [$tree canvas]
    set id [$canvas select item]
    if {"$id" == ""} {
	return [lindex [$canvas gettags selected] 0]
    }
    return [lindex [$canvas gettags $id] 0]
}

# Toggle the layout of the tree between vertical and horizontal

proc tree_toggleLayout {tree} {
    set canvas [$tree canvas]
    set scrollincrement [utilGetConfigValue $canvas -scrollincrement]
    if {[utilGetConfigValue $tree -layout] == "horizontal"} {
        $tree config -layout vertical
    } else {
        $tree config -layout horizontal
    }
    
    # change the layout of the nodes so that the bitmap is on top for
    # vertical trees and at left for horizontal trees
    foreach i [$canvas find withtag text] {
        set node [lindex [$canvas gettags $i] 0]
        tree_layoutNode $tree $node
        $tree nodeconfig $node
    }
    
    $tree draw
    tree_center $tree
}

# return the value of the given tk option for the given widget

proc utilGetConfigValue {w option} {
    return [lindex [$w config $option] 4]
}




