////////////////////////////////////////////////////////////////////////////////
//  implementations of analytical classes                                     //  
//  LAST EDIT: Wed Mar  8 10:35:27 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "complex.h"
#include "analytic.h"
#include "primitiv.h"
#include "poly.h"
#include "intrsect.h"
#include "macros.h"

// the sphere:

const char *RTN_SPHERE = "Sphere";

double RT_Sphere::radV;
int RT_Sphere::radF, RT_Sphere::radG;

RT_Point RT_Sphere::inverse(const RT_Vector &v) const {
    RT_Vector sn = v.UNITIZE();
    RT_Point p;
    double a1 = acos( sn.y );
    p.y = a1 * M_1_PI;
    if (p.y == 0.0 || p.y == 1.0) {
	p.x = 0;
	return p;
    }
    double a2 = acos( sn.x / sin( a1 ) ) * M_1_PI * 0.5;
    if (sn.z > 0) p.x = 1 - a2;
    else p.x = a2;
    p.y = 1 - p.y;
    return p;
}

RT_Bounds RT_Sphere::get_bounds() {
    RT_Bounds b;
    b.max.x = b.max.y = b.max.z = xrad;
    b.min.x = b.min.y = b.min.z = -xrad;
    return b;
} 

void RT_Sphere::resolution(double s) { 
    xsolution = (RT_ResolutionAttribute*)attributes->get( RTN_RESOLUTION_ATTR );
    if (!xsolution) 
	attributes->insert( xsolution = new RT_ResolutionAttribute);
    xsolution->resolution( s );
    geomChanged();
}

void RT_Sphere::createReferences(const RT_AttributeList &list) {
    // create resolution reference:
    RT_ResolutionAttribute *tmp = xsolution;
    xsolution = (RT_ResolutionAttribute*)list.retraverse( RTN_RESOLUTION_ATTR, this );
    if ( (tmp != xsolution) || (*tmp != *xsolution)) geomChanged();
}

int RT_Sphere::copy(RT_Primitive *p) const {
    if (p->isA( RTN_SPHERE )) {
	RT_Sphere *s = (RT_Sphere*)p;
	s->radius( get_radius() );
	return( 1 );
    }
    if (p->isA( RTN_QUADMESH )) {
	if (!qum->get_x()) {
	    rt_Output->errorVar( get_name(), ": First update the object, then copy again.", 0 );
	    return( 0 );
	}
	return( qum->copy( p ) );
    }
    return( RT_Primitive::copy( p ) );
}

void RT_Sphere::create() {
    // max dim: 50, min dim: 4
    int dim =(int)( 46.0 *xsolution->get_resolution() + 4);
    // must a new geometry created?
    if (qum->get_x() != dim) qum->newGeometry( dim, dim );
    RT_Vector tmp;
    double tmpr, tmpy;
    double dim2 = (dim-1)/2.0;
    double pidim = 2*asin(1)/(dim-1);
    for (int i = 0; i < dim; i++) {
	tmpr = xrad * cos( (i-dim2)*pidim );
	tmpy = xrad * sin( (i-dim2)*pidim );
	for (int j = 0; j < dim; j++) {
	    tmp.x = tmpr * sin( j*2*pidim ); 
	    tmp.y = tmpy;
	    tmp.z = tmpr * cos( j*2*pidim ); 
	    qum->vtPoint(i,j, tmp);
	    tmp = tmp.UNITIZE();
	    qum->vtNormal( i,j, tmp );
	}
    }
}

RT_ParseEntry RT_Sphere::table[] = {
    { "-radius", RTP_DOUBLE, (char*)&radV, &radF, "Specify a new {ARG 1 Radius} for the sphere.", RTPS_DOUBLE },
    { "-get_radius", RTP_NONE, 0, &radG, "Get the radius of the sphere.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_Sphere::objectCMD(char *argv[]) {
    int r = RT_Primitive::objectCMD( argv );
    RT_parseTable( argv, table );
    if (radF) { radius( radV ); r++; }
    if (radG) {
	r++; 
	char tmp[20];
	RT_double2string( get_radius(), tmp );
	RT_Object::result( tmp );
    }
    return r;
}

int RT_Sphere::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Double} {Creates a new sphere called {ARG 1 Name} with an inital {ARG 2 Radius}.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if (argc != 3) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <rad>. ", 0 );
	    return TCL_ERROR;
	}
	double rad;
	if (! RT_string2double( argv[2], rad )) {
	    Tcl_AppendResult( ip, "Bad parameter. <rad> must be a float. ", NULL );
	    return TCL_ERROR;
	}
	new RT_Sphere( argv[1], rad ); 
	RTM_classReturn;
    }
    return res; 
}

// the cone:

const char *RTN_CONE = "Cone";

double RT_Cone::radV;
int RT_Cone::radF, RT_Cone::radG;
double RT_Cone::lenV;
int RT_Cone::lenF, RT_Cone::lenG;
int RT_Cone::opF, RT_Cone::clF, RT_Cone::opG;

int RT_Cone::intersect(const RT_Ray &ray, RT_InterSectionList &list) {
    RT_Ray xray;
    RTM_wcRay2mcRay( ray, xray );
    
    double x0 = xray.pt.x; double x02 = x0*x0; 
    double y0 = xray.pt.y; double y02 = y0*y0;
    double z0 = xray.pt.z; double z02 = z0*z0;
    double x1 = xray.dir.x; double x12 = x1*x1;
    double y1 = xray.dir.y; double y12 = y1*y1;
    double z1 = xray.dir.z; double z12 = z1*z1;
    
    double a0= x02+y02-xrad*xrad*z02/xlen/xlen;
    double a1= 2*x0*x1+2*y0*y1-2*xrad*xrad*z0*z1/xlen/xlen;
    double a2= x12+y12-xrad*xrad*z12/xlen/xlen;
    double det=a1*a1/a2/a2/4-a0/a2;
    double hits[2];
    int n=0;
    
    if ( det < rt_RayEps && det >= 0) {
	double t=-a1/a2/2;
	if ((z0+t*z1>=0)&&(z0+t*z1<=xlen)) hits[n++]=t; 
    }
    else if ( det > 0 ) {
	det = sqrt(det);
	double t1 = -a1/a2/2 + det;
	double t2 = -a1/a2/2 - det;
	double zschnitt1 = z0+t1*z1;
	double zschnitt2 = z0+t2*z1;
	if ((zschnitt1 >= 0) && (zschnitt1 <= xlen)) hits[n++] = t1;
	if ((zschnitt2 >= 0) && (zschnitt2 <= xlen)) hits[n++] = t2;
    }
    
    if ( !xop && z1 ) {    
	double top = (xlen-z0)/z1;
	if ((x0+top*x1)*(x0+top*x1)+(y0+top*y1)*(y0+top*y1) <= xrad*xrad)
	    hits[n++] = top;
    }
    
    int ret = 0;
    for (int ii = 0; ii < n; ii++) {
	double tt;
	RTM_mcT2wcT( ray, xray, tt, hits[ii]);
	if (tt >= 0) hits[ret++] = tt;
    }
    
    for ( ii = 0; ii < (ret-1); ii++ ) {
	int mk = ii;
	double min = hits[ii];
	for ( int jj = (ii+1); jj < ret; jj++ ) {
	    if (hits[jj] < min) { min = hits[jj]; mk = jj; }
	}
	hits[mk] = hits[ii]; hits[ii] = min;
    }
    
    int inside = ret%2;
    int nh = 0;
    
    for ( ii = 0; ii < ret; ii++ ) {
	if (ray.valid( hits[ii] )) {
	    list.add( new RT_InterSection( this, hits[ii], !inside, &get_surface() ));
	    nh++;
	}
	inside = inside ? 0 : 1;
    }
    return nh;
}

int RT_Cone::copy(RT_Primitive *p) const {
    if (p->isA( RTN_CONE )) {
	RT_Cone *c = (RT_Cone*)p;
	c->radius( get_radius() );
	c->length( get_length() );
	if (get_open()) c->open(); else c->close();
	return( 1 );
    }
    return( RT_Primitive::copy( p ) );
}

void RT_Cone::resolution(double s) { 
    xsolution = (RT_ResolutionAttribute*)attributes->get( RTN_RESOLUTION_ATTR );
    if (!xsolution) 
	attributes->insert( xsolution = new RT_ResolutionAttribute);
    xsolution->resolution( s );
    geomChanged();
}

RT_Bounds RT_Cone::get_bounds() {
    RT_Bounds b;
    b.max.x = xrad; b.max.y = xrad; b.max.z = xlen;
    b.min.x = -xrad; b.min.y = -xrad; b.min.z = 0;
    return b;
}

void RT_Cone::normal(const RT_Vector &wc, RT_Vector &n) {
    RT_Vector mc = wc2mc( wc );
    RT_Vector mcn;
    if (fabs(mc.z - xlen) < rt_RayEps) {
	mcn.x = 0; mcn.y = 0; mcn.z = 1;
    } else {
	mcn.x = mc.z;
	mcn.y = mc.z*mc.y/mc.x;
	mcn.z = -mc.y*mc.y/mc.x-mc.x;
    }
    RTM_mcN2wcN( wc, mcn, n);
} 

RT_Point RT_Cone::inverse(const RT_Vector &v) const {
    if (v.z < rt_RayEps) return RT_Point( -1, -1 );
    double rt = v.z/xlen * xrad;
    RT_Point p;
    p.y =  v.z/xlen;
    if (fabs( v.z - xlen) > rt_RayEps ) {
	p.x =  asin( v.x / rt ) * 0.5 / M_PI;
	if (v.y < 0) p.x = 0.5 - p.x;
	if (p.x < 0) p.x += 1;
	return p;
    }
    return RT_Point( -1, -1 );
}

void RT_Cone::createReferences(const RT_AttributeList &list) {
    RT_Primitive::createReferences( list );
    // create resolution reference:
    RT_ResolutionAttribute *tmp = xsolution;
    xsolution = (RT_ResolutionAttribute*)list.retraverse( RTN_RESOLUTION_ATTR, this );
    if ( (tmp != xsolution) || (*tmp != *xsolution)) geomChanged();
}

void RT_Cone::create() {
    // max dim: 50, min dim: 4
    int dim =(int)( 46.0 *xsolution->get_resolution() + 4);
    // must a new geometry created?
    if (xmesh->get_x() != dim) xmesh->newGeometry( dim, dim );

    double fak = 2 * M_PI / (double)(dim - 1);;

    // compute a simple circle:
    RT_Vector *circle = new RT_Vector[dim];
    RT_Vector *c = circle;
    for (int i = 0; i < dim; i++) {
	double r = i * fak;
	c->x = cos( r ); 
	c->y = sin( r );
	c->z = xlen;
	c++;
    }

    double fak2 = 1.0/(dim-1);
    // draw the cone:
    RT_Vector tmp;
    for (i = 0; i < dim; i++) {
	c = circle;
	double r = i * fak2;
	tmp.z = xlen * r;
	for (int j = 0; j < dim; j++) {
	    tmp.x = r * xrad * c->x;
	    tmp.y = r * xrad * c->y;
	    xmesh->vtPoint(i,j, tmp);
	    c++;
	}
    }
    xmesh->computeNormals();
    
    // draw the cover:
    if (xcover) {
	xcover->father( 0 );
	delete xcover;
    }
    if (xop) xcover = 0;
    else {
	c = circle;
	for (int i = 0; i < dim; i++) {
	    c->x *= xrad;
	    c->y *= xrad;
	    c++;
	}
	// resort the vertices in circle to be conform
	// to the right-hand rule:
	int ii = dim/2; RT_Vector tv;
	for (i = 0; i < ii; i++) {
	    tv = circle[i]; circle[i] = circle[dim-i-1]; circle[dim-i-1] = tv;
	}
	xcover = new RT_Polygon( 0, dim-1, circle );
	xcover->gnormal( RT_Vector(0, 0, 1));
	xcover->father( this );
	xcover->createReferences( *attributes );
    }
    delete circle;
}

RT_ParseEntry RT_Cone::table[] = {
    { "-radius", RTP_DOUBLE, (char*)&radV, &radF, "Specify the {ARG 1 Radius} for the cone.", RTPS_DOUBLE },
    { "-get_radius", RTP_NONE, 0, &radG, "Get the radius of the cone.", RTPS_NONE },
    { "-length", RTP_DOUBLE, (char*)&lenV, &lenF, "Set the {ARG 1 Length} for the cone.", RTPS_DOUBLE },
    { "-get_length", RTP_NONE, 0, &lenG, "Inquire the length of the cone.", RTPS_NONE },
    { "-open", RTP_NONE, 0, &opF, "Open the cone.", RTPS_NONE },
    { "-get_open", RTP_NONE, 0, &opG, "Return current open/close state.", RTPS_NONE },
    { "-close", RTP_NONE, 0, &clF, "Close the cone.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_Cone::objectCMD(char *argv[]) {
    int r = RT_Primitive::objectCMD( argv );
    RT_parseTable( argv, table );
    if (radF) { radius( radV ); r++; }
    if (radG) {
	r++; 
	char tmp[20];
	RT_double2string( get_radius(), tmp );
	RT_Object::result( tmp );
    }
    if (lenF) { length( lenV ); r++; }
    if (lenG) {
	r++; 
	char tmp[20];
	RT_double2string( get_length(), tmp );
	RT_Object::result( tmp );
    }
    if (opF) { open(); r++; }
    if (clF) { close(); r++; }
    if (opG) { RT_Object::result( get_open() ? "1" : "0" ); r++; }
    return r;
}

int RT_Cone::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Double Double} {Creates a new {ARG 1 Cone} with an inital {ARG 2 Radius} and {ARG 3 Length}. The cone is created around the positive z axis. The top of the cone lies in the origin of the modelling coordinates system.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if (argc != 4) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <radius> <length>. ", 0 );
	    return TCL_ERROR;
	}
	double rad, len;
	if ( RT_string2double( argv[2], rad )) 
	    if ( RT_string2double( argv[3], len )) {
		new RT_Cone( argv[1], rad, len ); 
		RTM_classReturn;
	    }
	Tcl_AppendResult( ip, "Bad parameter. Radius and Length must be floats. ", NULL );
	RTM_classReturn;
    }
    return res; 
}

// the torus:

int RT_Torus::copy(RT_Primitive *p) const {
    if (p->isA( RTN_TORUS )) {
	RT_Torus *t = (RT_Torus*)p;
	t->iradius( get_iradius() );
	t->oradius( get_oradius() );
	return( 1 );
    }
    return( RT_Primitive::copy( p ) );
}

const char *RTN_TORUS = "Torus";

double RT_Torus::iradV, RT_Torus::oradV;
int RT_Torus::iradF, RT_Torus::iradG, RT_Torus::oradF, RT_Torus::oradG ;

void RT_Torus::createReferences(const RT_AttributeList &list) {
    RT_Quadmesh::createReferences( list );
    // create resolution reference:
    RT_ResolutionAttribute *tmp = xsolution;
    xsolution = (RT_ResolutionAttribute*)list.retraverse( RTN_RESOLUTION_ATTR, this );
    if ( (tmp != xsolution) || (*tmp != *xsolution )) geomChanged();
}

RT_Bounds RT_Torus::get_bounds() {
    RT_Bounds b;
    b.max.x = orad + irad; b.max.y = orad + irad; b.max.z = orad;
    b.min.x = -orad - irad; b.min.y = -orad - irad; b.min.z = -orad;
    return b;
} 

void RT_Torus::resolution(double s) { 
    xsolution = (RT_ResolutionAttribute*)attributes->get( RTN_RESOLUTION_ATTR );
    if (!xsolution) 
	attributes->insert( xsolution = new RT_ResolutionAttribute);
    xsolution->resolution( s );
    geomChanged();
}

void RT_Torus::create() {
    // max dim: 50, min dim: 4
    int dim =(int)( 46.0 *xsolution->get_resolution() + 4);
    // must a new geometry created?
    if (get_x() != dim) newGeometry( dim, dim );
    double r, a1, a2;
    RT_Vector tmp;
    for (int i = 0; i < dim; i++) {
	a1 = 2.0 * M_PI * i / (dim - 1); 
	for (int j = 0; j < dim; j++) {
	    a2 = 2.0 * M_PI * j / (dim - 1);
	    r = irad + orad * cos(a2);
	    tmp.x = r * sin(a1);
	    tmp.y = r * cos(a1);
	    tmp.z = orad * sin(a2);
	    vtPoint(i,j, tmp);
	}
    }
    computeNormals();
}

int RT_Torus::intersect(const RT_Ray &ray, RT_InterSectionList &list) { 
    RT_Ray xray;
    RTM_wcRay2mcRay( ray, xray );

    double x0 = xray.pt.x; double x02 = x0 * x0; 
    double y0 = xray.pt.y; double y02 = y0 * y0; 
    double z0 = xray.pt.z; double z02 = z0 * z0; 
    double x1 = xray.dir.x; double x12 = x1 * x1;
    double y1 = xray.dir.y; double y12 = y1 * y1;
    double z1 = xray.dir.z; double z12 = z1 * z1;

    double aa = irad; double aa2 = aa * aa;
    double bb = orad; double bb2 = bb * bb;

    // QUARTIC FOR 't' (by substituting the ray equation):
    // somebody could optimize this stuff :=
    double a4 = x12 + y12 + z12; 
    double a3 = 4. * (x0 * x1 + y0 * y1 + z0 * z1 ) * a4; a4 *= a4;
    double a2 = -aa2 * x12 - bb2 * x12 + 3. * x02 * x12 + x12 * z02 + 4. * x0 * x1 * z0 * z1
	+ aa2 * z12 - bb2 * z12 + x02 * z12 + 3. * z02 * z12 + x12 * y02 + z12 * y02
	    + 4. * x0 * x1 * y0 * y1 + 4. * z0 * z1 * y0 * y1 - aa2 * y12 - bb2 * y12 + x02 * y12 
		+ z02 * y12 + 3. * y02 * y12; a2 *= 2.;
    double a1 = -aa2 * x0 *x1 - bb2 * x0 * x1 + (x02 * x0) * x1 + x0 * x1 * z02 + aa2 * z0 * z1 
	- bb2 * z0 * z1 + x02 * z0 * z1 + (z02 * z0) * z1 + x0 * x1 * y02 + z0 * z1 * y02 - aa2 * y0 * y1
	    - bb2 * y0 * y1 + x02 * y0 * y1 + z02 * y0 * y1 + (y02 * y0) * y1; a1 *= 4.;
    double a0 = aa2 - bb2; a0 *= a0;
    a0 += -2. * aa2 * x02 - 2. * bb2 * x02 + (x02 * x02) + 2. * aa2 * z02 - 2. * bb2 * z02 + 2. * x02 * z02 + (z02 * z02) -2. * aa2 * y02 -2. * bb2 * y02 + 2. * x02 * y02 + 2. * z02 * y02 + (y02 * y02);	

    double co1 = - a3 / (4. * a4);
    double co2 = co1 * co1;
    double co3 = co2 * co1;
    double co4 = co3 * co1;

    // REDUCED QUARTIC (b4 = 1, b3 = 0):
    double b2 = 6. * co2 * a4 + 3. * a3 * co1 + a2; 
    double b1 = 4. * co3 * a4 + 3. * a3 * co2 + 2. * a2 * co1 + a1; 
    double b0 =      co4 * a4 + 1. * a3 * co3 +      a2 * co2 + a1 * co1 + a0;  
    b2 /= a4; b1 /= a4; b0 /= a4;

    // CUBIC RESOLVEMENT:
    double c2 = 2. * b2;
    double c1 = b2 * b2 - 4. * b0;
    double c0 = - b1 * b1;

    // REDUCED (CARDANIC) EQUATION:
    double d1 = c1 - c2 * c2 / 3.;
    double d0 = 2. * c2 * c2 * c2 / 27. - c2 * c1 / 3. + c0;

    // the discriminant:
    double disc = d1 * d1 * d1 / 27. + d0 * d0 / 4.;
    RT_Complex u, v;
    if (disc >= 0) u.setReal( sqrt( disc));
    else u.setImag( sqrt( -disc ) ); 

    u.addReal( - 0.5 * d0 ); u.pow( 1./3. ); 
    v = u; v.pow( -1 ); v.mult(-d1/3.);
    
    RT_Complex tmpc1 = u + v;
    tmpc1.mult( -0.5 );
    RT_Complex tmpc2 = (u - v) * RT_Complex( 0, sqrt( 3 )/2 ); 

    RT_Complex v1, v2, v3;
    v1 = u + v; v1.addReal( -c2/3. ); v1.pow( 0.5 );  
    v2 = tmpc1 + tmpc2; v2.addReal( -c2/3. ); v2.pow( 0.5 ); 
    v3 = tmpc1 - tmpc2; v3.addReal( -c2/3. ); v3.pow( 0.5 ); 

    // change the signs if necessary:
    RT_Complex xxx = v1 * v2;
    xxx = xxx * v3;
    if (xxx.getReal() < 0 && b1 > 0) v2.mult( -1); 
    if (xxx.getReal() > 0 && b1 < 0) { v1.mult( -1); v2.mult( -1); v3.mult( -1); }
    
    RT_Complex w1 = v1 + v2 - v3; w1.mult( 0.5 ); w1.addReal( -0.25 * a3/a4 );    
    RT_Complex w2 = v1 - v2 + v3; w2.mult( 0.5 ); w2.addReal( -0.25 * a3/a4 );
    RT_Complex w3 = v2 + v3 - v1; w3.mult( 0.5 ); w3.addReal( -0.25 * a3/a4 );
    RT_Complex w4 = v1 + v2 + v3; w4.mult( -0.5 ); w4.addReal( -0.25 * a3/a4 );


#ifdef NEVER

    // the following stuff was used to check the computed solutions
    // of the 4th graduate equation
    {
	RT_Complex ctmp;
	RT_Complex xres;

	RT_Complex *ws[4];
	ws[0] = &w1; ws[1] = &w2; ws[2] = &w3; ws[3] = &w4;  

	for ( int i = 0; i < 4; i++ ) { 
	    xres.setReal( 0 ); xres.setImag( 0 );
	    ctmp = *ws[i]; ctmp.pow( 4 ); ctmp.mult( a4 ); xres = xres + ctmp;
	    ctmp = *ws[i]; ctmp.pow( 3 ); ctmp.mult( a3 ); xres = xres + ctmp;
	    ctmp = *ws[i]; ctmp.pow( 2 ); ctmp.mult( a2 ); xres = xres + ctmp;
	    ctmp = *ws[i];                ctmp.mult( a1 ); xres = xres + ctmp;
	    ctmp.setReal( a0 ); ctmp.setImag( 0 );     xres = xres + ctmp;

	    double err = xres.getReal() * xres.getReal();
	    err += xres.getImag() * xres.getImag();
	    if (!i && err > 0.001) {
		fprintf( stderr, "Error in torus computing: f() = %lf\n", err );
		fprintf( stderr, "q: %lf --- ", b1 );
		xxx.print( stderr); 
		fprintf( stderr, "*" );
	    }
	}
    }
#endif

    double hits[4];
    int ret = 0;
    // put the valid hits in an array:
    if (fabs( w1.getImag()) <  0.001 && w1.getReal() > rt_RayEps ) 
	hits[ret++] = w1.getReal();
    if (fabs( w2.getImag()) <  0.001 && w2.getReal() > rt_RayEps ) 
	hits[ret++] = w2.getReal();
    if (fabs( w3.getImag()) <  0.001 && w3.getReal() > rt_RayEps ) 
	hits[ret++] = w3.getReal();
    if (fabs( w4.getImag()) <  0.001 && w4.getReal() > rt_RayEps ) 
	hits[ret++] = w4.getReal();


    for ( int ii = 0; ii < ret; ii++ ) {
	double tt;
	RTM_mcT2wcT( ray, xray, tt, hits[ii] );
	hits[ii] = tt;
    }

    // sort it - nearest hit at first:
    for ( ii = 0; ii < (ret-1); ii++ ) {
	int mk = ii;
	double min = hits[ii];
	for ( int jj = (ii+1); jj < ret; jj++ ) {
	    if (hits[jj]<min) { min = hits[jj]; mk = jj; }
	}
	hits[mk] = hits[ii]; hits[ii] = min;
    }
    
    int inside = ret%2;

    // if we have an odd number of intersections - we are outside the torus

    int xret = 0;
    for ( ii = 0; ii < ret; ii++ ) {
	if ( ray.valid( hits[ii] )) {
	    list.add( new RT_InterSection( this, hits[ii], !inside, &get_surface() ));
	    xret++;
	}
	inside = inside ? 0 : 1;
    }
    return xret; 
}

void RT_Torus::normal(const RT_Vector &wc, RT_Vector &n) {
    RT_Vector mc = wc2mc( wc );
    RT_Vector mcn;
    double co = mc.x * mc.x + mc.y * mc.y + mc.z * mc.z 
	-irad * irad - orad *orad;
    mcn.x = 4 * mc.x * co;
    mcn.y = 4 * mc.y * co;
    mcn.z = 4 * mc.z * ( 2 * irad * irad + co);
    RTM_mcN2wcN( wc, mcn, n );
}

RT_ParseEntry RT_Torus::table[] = {
    { "-iradius", RTP_DOUBLE, (char*)&iradV, &iradF, "Specify a new inner {ARG 1 Radius} for the torus.", RTPS_DOUBLE },
    { "-get_iradius", RTP_NONE, 0, &iradG, "Get the inner radius of the torus.", RTPS_NONE },
    { "-oradius", RTP_DOUBLE, (char*)&oradV, &oradF, "Specify a new outer {ARG 1 Radius} for the torus.", RTPS_DOUBLE },
    { "-get_oradius", RTP_NONE, 0, &oradG, "Get the outer radius of the torus.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_Torus::objectCMD(char *argv[]) {
    int r = RT_Primitive::objectCMD( argv );
    RT_parseTable( argv, table );
    if (iradF) { iradius( iradV ); r++; }
    if (iradG) {
	r++; 
	char tmp[20];
	RT_double2string( get_iradius(), tmp );
	RT_Object::result( tmp );
    }
    if (oradF) { oradius( oradV ); r++; }
    if (oradG) {
	r++; 
	char tmp[20];
	RT_double2string( get_oradius(), tmp );
	RT_Object::result( tmp );
    }
    return r;
}

int RT_Torus::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Double Double} {Creates a new torus called {ARG 1 Name} with an inital {ARG 2 Inner} and {ARG 3 Outer} Radius. The torus is centrically built around the z axis}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if (argc != 4) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <inner rad> <outer rad>. ", 0 );
	    return TCL_ERROR;
	}
	double iirad, oorad;
	if ( RT_string2double( argv[2], iirad )) 
	    if ( RT_string2double( argv[3], oorad )) {
		new RT_Torus( argv[1], iirad, oorad ); 
		RTM_classReturn;
	    }
	Tcl_AppendResult( ip, "Bad parameter. The radius values must be floats. ", 0 );
	return TCL_ERROR;
    }
    return res; 
}

///// the plane:

const char *RTN_PLANE = "Plane";

int RT_Plane::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Double Double Double Double} {Creates a new plane called {ARG 1 Name} with {ARG 2 A} * x + {ARG 3 B} * y + {ARG 4 C} * z + {ARG 5 D} = 0.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if (argc != 6) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <a> <b> <c> <d>. ", 0 );
	    return TCL_ERROR;
	}
	double d[4];
	int err = 0;
	for (int i = 0; i < 4; i++) 
	    if (! RT_string2double( argv[i+2], d[i])) {
		err = 1;
		break;
	    }
	if (err) {
	    Tcl_AppendResult( ip, "Bad parameters. <a> <b> <c> <d> must be floats. ", NULL );
	    return TCL_ERROR;
	}
	new RT_Plane( argv[1], d[0], d[1], d[2], d[3] ); 
	RTM_classReturn;
    }
    return res; 
}

int RT_Plane::intersect(const RT_Ray &wray, RT_InterSectionList &list) {
    RT_Ray ray;
    RTM_wcRay2mcRay( wray, ray );

    RT_Vector v = quTerms * 2;
    double vd = v.DOT( ray.dir );
    if (fabs(vd) < rt_RayEps ) return 0;
    double v0 = v.DOT( ray.pt ) + quConstant;
    double t = - v0/vd;
    
    double wt;
    RTM_mcT2wcT( wray, ray, wt, t ); 

    if ( wray.valid( wt )) {
	list.add( new RT_InterSection( this, wt, 1,  &get_surface() ));
	return 1;
    }
    return 0;
} 

void RT_Plane::normal(const RT_Vector &wc, RT_Vector &n) {
    RTM_mcN2wcN( wc, quTerms, n);
} 

///// the ellipsoid:

const char *RTN_ELLIPSOID = "Ellipsoid";

int RT_Ellipsoid::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Double Double Double} {Creates a new ellipsoid called {ARG 1 Name}; {ARG 2 X} Radius, {ARG 3 Y} Radius, {ARG 4 Z} Radius}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if (argc != 5) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <x> <y> <z>. ", 0 );
	    return TCL_ERROR;
	}
	double d[3];
	int err = 0;
	for (int i = 0; i < 3; i++) 
	    if (! RT_string2double( argv[i+2], d[i])) {
		err = 1;
		break;
	    }
	if (err) {
	    Tcl_AppendResult( ip, "Bad parameters. <x> <y> <z> must be floats. ", NULL );
	    return TCL_ERROR;
	}
	new RT_Ellipsoid( argv[1], d[0], d[1], d[2] ); 
	RTM_classReturn;
    }
    return res; 
}

///// the quader:

const char *RTN_QUADER = "Quader";

double RT_Quader::xV, RT_Quader::yV, RT_Quader::zV;
int RT_Quader::xF, RT_Quader::yF, RT_Quader::zF;
int RT_Quader::xG, RT_Quader::yG, RT_Quader::zG;

RT_ParseEntry RT_Quader::table[] = {
    { "-xdim", RTP_DOUBLE, (char*)&xV, &xF, "Specify a {ARG 1 X} dimension for the block.", RTPS_DOUBLE },
    { "-get_xdim", RTP_NONE, 0, &xG, "Get the x dimension of the block.", RTPS_NONE },
    { "-ydim", RTP_DOUBLE, (char*)&yV, &yF, "Specify a {ARG 1 Y} dimension for the block.", RTPS_DOUBLE },
    { "-get_ydim", RTP_NONE, 0, &yG, "Get the y dimension of the block.", RTPS_NONE },
    { "-zdim", RTP_DOUBLE, (char*)&zV, &zF, "Specify a {ARG 1 Z} dimension for the block.", RTPS_DOUBLE },
    { "-get_zdim", RTP_NONE, 0, &zG, "Get the z dimension of the block.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

void RT_Quader::create() {
    double ddx = dx * 0.5;
    double ddy = dy * 0.5;
    double ddz = dz * 0.5;
    RT_Vector vt, n;
    
    // plus x plane:
    n.x = 1; n.y = 0; n.z = 0; qpx->gnormal( n );
    vt.x = ddx; vt.y = ddy; vt.z = -ddz; qpx->vtPoint( 0, 0, vt );
    vt.z += dz; qpx->vtPoint( 0, 1, vt );
    vt.z -= dz; vt.y -= dy; qpx->vtPoint( 1, 0, vt );
    vt.z += dz; qpx->vtPoint( 1, 1, vt );
    
    // minus x plane:
    n.x = -1; qmx->gnormal( n );
    vt.x = -ddx; vt.y = ddy; vt.z = ddz; qmx->vtPoint( 0, 0, vt );
    vt.z -= dz; qmx->vtPoint( 0, 1, vt );
    vt.z += dz; vt.y -= dy; qmx->vtPoint( 1, 0, vt );
    vt.z -= dz; qmx->vtPoint( 1, 1, vt );
    
    // plus y plane:
    n.x = 0; n.y = 1; n.z = 0; qpy->gnormal( n );
    vt.x = -ddx; vt.y = ddy; vt.z = -ddz; qpy->vtPoint( 0, 0, vt );
    vt.z += dz; qpy->vtPoint( 0, 1, vt );
    vt.z -= dz; vt.x += dx; qpy->vtPoint( 1, 0, vt );
    vt.z += dz; qpy->vtPoint( 1, 1, vt );
    
    // minus y plane:
    n.y = -1; qmy->gnormal( n );
    vt.x = -ddx; vt.y = -ddy; vt.z = ddz; qmy->vtPoint( 0, 0, vt );
    vt.z -= dz; qmy->vtPoint( 0, 1, vt );
    vt.z += dz; vt.x += dx; qmy->vtPoint( 1, 0, vt );
    vt.z -= dz; qmy->vtPoint( 1, 1, vt );
    
    // plus z plane:
    n.x = 0; n.y = 0; n.z = 1; qpz->gnormal( n );
    vt.x = ddx; vt.y = ddy; vt.z = ddz; qpz->vtPoint( 0, 0, vt );
    vt.x -= dx; qpz->vtPoint( 0, 1, vt );
    vt.x += dx; vt.y -= dy; qpz->vtPoint( 1, 0, vt );
    vt.x -= dx; qpz->vtPoint( 1, 1, vt );
    
    // minus z plane:
    n.z = -1; qmz->gnormal( n );
    vt.x = -ddx; vt.y = ddy; vt.z = -ddz; qmz->vtPoint( 0, 0, vt );
    vt.x += dx; qmz->vtPoint( 0, 1, vt );
    vt.x -= dx; vt.y -= dy; qmz->vtPoint( 1, 0, vt );
    vt.x += dx; qmz->vtPoint( 1, 1, vt );
}

RT_Bounds RT_Quader::get_bounds() {
    RT_Bounds b;
    b.max.x = dx * 0.5; b.max.y = dy * 0.5; b.max.z = dz * 0.5;
    b.min.x = -dx * 0.5; b.min.y = -dy * 0.5; b.min.z = -dz * 0.5;
    return b;
} 

RT_Point RT_Quader::inverse(const RT_Vector &v) const {
    if (v.z < (dz * 0.5 - rt_RayEps)) return RT_Point(-1, -1);
    return RT_Point( 0.5 + v.x/dx,  0.5 + v.y/dy );
}

int RT_Quader::intersect(const RT_Ray &ray, RT_InterSectionList &list) {
    RT_Ray xray;
    RTM_wcRay2mcRay( ray, xray );
    RT_Vector phits[2], nhits[2];
    double hits[2];
    double t, tx, ty, tz;
    double ddx = 0.5 * dx;
    double ddy = 0.5 * dy;
    double ddz = 0.5 * dz;
    int nbhs = 0;
    if (xray.dir.x != 0.) {
	// +x plane:
	t = -(-ddx + xray.pt.x)/xray.dir.x;
	tx = xray.pt.x + t * xray.dir.x;
	ty = xray.pt.y + t * xray.dir.y;
	tz = xray.pt.z + t * xray.dir.z;
	if ( ty >= -ddy && ty < ddy &&
	    tz >= -ddz && tz < ddz) {
	    hits[nbhs] = t; nhits[nbhs] = RT_Vector( 1,0,0 ); 
	    phits[nbhs].x = tx;
	    phits[nbhs].y = ty;
	    phits[nbhs].z = tz;
	    nbhs++;
	}
	// -x plane:
	t = -(ddx + xray.pt.x)/xray.dir.x;
	tx = xray.pt.x + t * xray.dir.x;
	ty = xray.pt.y + t * xray.dir.y;
	tz = xray.pt.z + t * xray.dir.z;
	if ( ty >= -ddy && ty < ddy &&
	    tz >= -ddz && tz < ddz) {
	    hits[nbhs] = t; nhits[nbhs] = RT_Vector( -1,0,0 ); 
	    phits[nbhs].x = tx;
	    phits[nbhs].y = ty;
	    phits[nbhs].z = tz;
	    nbhs++;
	}
    }
    if (xray.dir.y != 0. && nbhs < 2) {
	// +y plane:
	t = -(-ddy + xray.pt.y)/xray.dir.y;
	tx = xray.pt.x + t * xray.dir.x;
	ty = xray.pt.y + t * xray.dir.y;
	tz = xray.pt.z + t * xray.dir.z;
	if ( tx >= -ddx && tx < ddx &&
	    tz >= -ddz && tz < ddz) {
	    hits[nbhs] = t; nhits[nbhs] = RT_Vector( 0,1,0 ); 
	    phits[nbhs].x = tx;
	    phits[nbhs].y = ty;
	    phits[nbhs].z = tz;
	    nbhs++;
	}
	if (nbhs < 2) {
	    // -y plane:
	    t = -(ddy + xray.pt.y)/xray.dir.y;
	    tx = xray.pt.x + t * xray.dir.x;
	    ty = xray.pt.y + t * xray.dir.y;
	    tz = xray.pt.z + t * xray.dir.z;
	    if ( tx >= -ddx && tx < ddx &&
		tz >= -ddz && tz < ddz) {
		hits[nbhs] = t; nhits[nbhs] = RT_Vector( -1,0,0 ); 
		phits[nbhs].x = tx;
		phits[nbhs].y = ty;
		phits[nbhs].z = tz;
		nbhs++;
	    }
	}
    }
    if (xray.dir.z != 0. && nbhs < 2) {
	// +z plane:
	t = -(-ddz + xray.pt.z)/xray.dir.z;
	tx = xray.pt.x + t * xray.dir.x;
	ty = xray.pt.y + t * xray.dir.y;
	tz = xray.pt.z + t * xray.dir.z;
	if ( tx >= -ddx && tx < ddx &&
	    ty >= -ddy && ty < ddy) {
	    hits[nbhs] = t; nhits[nbhs] = RT_Vector( 0,0,1 ); 
	    phits[nbhs].x = tx;
	    phits[nbhs].y = ty;
	    phits[nbhs].z = tz;
	    nbhs++;
	}
	if (nbhs < 2) {
	    // -z plane:
	    t = -(ddz + xray.pt.z)/xray.dir.z;
	    tx = xray.pt.x + t * xray.dir.x;
	    ty = xray.pt.y + t * xray.dir.y;
	    tz = xray.pt.z + t * xray.dir.z;
	    if ( tx >= -ddx && tx < ddx &&
		ty >= -ddy && ty < ddy) {
		hits[nbhs] = t; nhits[nbhs] = RT_Vector( 0,0,-1 ); 
		phits[nbhs].x = tx;
		phits[nbhs].y = ty;
		phits[nbhs].z = tz;
		nbhs++;
	    }    
	}
    }
    int ret = 0;

    if (!nbhs) return 0; // no hits!
    if (nbhs == 1) { // one - MC - hit:
	double wt;
	mcnorm = nhits[0];
	RTM_mcT2wcT( ray, xray, wt, hits[0] );
	if (ray.valid( wt) ) {
	    list.add( new RT_InterSection( this, wt, 0,  &get_surface() ));
	    ret++;
	}
	
    }
    else { // two - MC - hits:
	double wchits[2];
	RTM_mcT2wcT( ray, xray, wchits[0], hits[0] );
	RTM_mcT2wcT( ray, xray, wchits[1], hits[1] );

	if (ray.valid( wchits[0])) {
	    if (ray.valid( wchits[1])) 
		mcnorm = wchits[0] < wchits[1] ? nhits[0] : nhits[1];
	    else mcnorm = nhits[0];
	}
	else mcnorm = nhits[1];    
	    

	if (ray.valid( wchits[0] )) {
	    list.add( new RT_InterSection( this, wchits[0], wchits[0] < wchits[1],  &get_surface() ));
	    ret++;
	}
	if (ray.valid( wchits[1] )) {
	    list.add( new RT_InterSection( this, wchits[1], wchits[1] < wchits[0],  &get_surface() ));
	    ret++;
	}
    }
    return ret;
}

int RT_Quader::copy(RT_Primitive *p) const {
    if (p->isA( RTN_QUADER )) {
	RT_Quader *q = (RT_Quader*)p;
	q->xdim( get_xdim() );
	q->ydim( get_ydim() );
	q->zdim( get_zdim() );
	return( 1 );
    }
    return( RT_Primitive::copy( p ) );
}

void RT_Quader::normal(const RT_Vector &w, RT_Vector &n) { 
    RTM_mcN2wcN( w, mcnorm, n);
}

int RT_Quader::objectCMD(char *argv[]) {
    int r = 0;
    if (RT_Primitive::objectCMD( argv )) r++;
    RT_parseTable( argv, table );
    if (xF) { xdim( xV ); r++; }
    if (yF) { ydim( yV ); r++; }
    if (zF) { zdim( zV ); r++; }
    if (xG) {
	char tmp[20]; r++;
	RT_double2string( get_xdim(), tmp );
	RT_Object::result( tmp );
    }
    if (yG) {
	char tmp[20]; r++;
	RT_double2string( get_ydim(), tmp );
	RT_Object::result( tmp );
    }
    if (zG) {
	char tmp[20]; r++;
	RT_double2string( get_zdim(), tmp );
	RT_Object::result( tmp );
    }
    return r;
}

int RT_Quader::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Double Double Double} {Creates a new block called {ARG 1 Name}. Additional arguments specify the dimensions: {ARG 2 dx}, {ARG 3 dy}, and {ARG 4 dz}.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if (argc != 5) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <dx> <dy> <dz>. ", 0 );
	    return TCL_ERROR;
	}
	RT_Vector v;
	if (!RT_strings2vector( (const char**)&argv[2], v )) {
	    Tcl_AppendResult( ip, "Bad parameters. <dx> <dy> <dz> must be floats. ", NULL );
	    return TCL_ERROR;
	}
	new RT_Quader( argv[1], v.x,v.y,v.z ); 
	RTM_classReturn;
    }
    return res; 
}

// cylinder:

const char *RTN_CYLINDER = "Cylinder";

double RT_Cylinder::radV;
int RT_Cylinder::radF, RT_Cylinder::radG;
double RT_Cylinder::lenV;
int RT_Cylinder::lenF, RT_Cylinder::lenG;
int RT_Cylinder::opF, RT_Cylinder::clF, RT_Cylinder::opG;

RT_Cylinder::intersect(const RT_Ray &ray, RT_InterSectionList &list) {
    RT_Ray xray;
    RTM_wcRay2mcRay( ray, xray);

    double x0 = xray.pt.x;  double x02 = x0*x0;
    double y0 = xray.pt.y;  double y02 = y0*y0;
    double z0 = xray.pt.z;  double z02 = z0*z0;
    double x1 = xray.dir.x; double x12 = x1*x1;
    double y1 = xray.dir.y; double y12 = y1*y1;
    double z1 = xray.dir.z; double z12 = z1*z1;   

    double len2 = xlen/2;
    double f =  x12+y12;
    if (f == 0.) f = rt_RayEps; 
    double fac = (x0*x1+y0*y1) / f;
    double rt = fac*fac - (x02+y02-xrad*xrad) / f;

    int num = 0;
    double hits[4]; 
    
    if (rt == 0) { 
      hits[num] = -fac; 
      if ( fabs( z0+z1*hits[num] ) <= len2 ) num++;
    }     

    if (rt>0) {
      rt = sqrt(rt);
      hits[num] = -fac+rt; 
      if ( fabs( z0+z1*hits[num] ) <= len2 ) num++;
      hits[num] = -fac-rt; 
      if ( fabs( z0+z1*hits[num] ) <= len2 ) num++;
    }

    if ( (!xop) && (z1 != 0.) ) {
      hits[num] = (len2-z0) / z1;
      if ( (pow(x0+hits[num]*x1,2) + pow(y0+hits[num]*y1,2)) <= pow(xrad,2) ) num++;
      hits[num] = (-len2-z0) / z1;
      if ( (pow(x0+hits[num]*x1,2) + pow(y0+hits[num]*y1,2)) <= pow(xrad,2) ) num++;
    }

    int newnum = 0;

    for ( int i=0; i<num; i++ ) {
	double t;
	RTM_mcT2wcT( ray, xray, t, hits[i] );
        if ( t>=0 ) { hits[newnum] = t; newnum++; }
    }

    num = newnum;

    for ( i=0; i<(num-1); i++ ) {
        int mk = i;
        double min = hits[i];
        for ( int j=(i+1); j<num; j++ ) {
            if (hits[j]<min) { min = hits[j]; mk=j; }
        } 
        hits[mk] = hits[i]; hits[i] = min;
    }
    
    int inside = num%2;
    int ret = 0; 

    for ( i=0; i<num; i++ ) {
	if (ray.valid( hits[i] )) {
	    list.add( new RT_InterSection( this, hits[i], !inside, &get_surface() ));
            ret++;
	}
	inside = inside ? 0 : 1;
    }
    return ret;
}   

int RT_Cylinder::copy(RT_Primitive *p) const {
    if (p->isA( RTN_CYLINDER )) {
	RT_Cylinder *c = (RT_Cylinder*)p;
	c->radius( get_radius() );
	c->length( get_length() );
	if (get_open()) c->open(); else c->close();
	return( 1 );
    }
    return( RT_Primitive::copy( p ) );
}

RT_Bounds RT_Cylinder::get_bounds() {
    RT_Bounds b;
    b.max.x = xrad; b.max.y = xrad; b.max.z = xlen * 0.5;
    b.min.x = -xrad; b.min.y = -xrad; b.min.z = -xlen * 0.5;
    return b;
} 

void RT_Cylinder::resolution(double s) { 
    xsolution = (RT_ResolutionAttribute*)attributes->get( RTN_RESOLUTION_ATTR );
    if (!xsolution) 
	attributes->insert( xsolution = new RT_ResolutionAttribute);
    xsolution->resolution( s );
    geomChanged();
}

void RT_Cylinder::normal(const RT_Vector &wc, RT_Vector &n) {
    RT_Vector mc = wc2mc( wc);
    RT_Vector mcn;
    if (fabs(fabs(mc.z)-xlen/2) < rt_RayEps ) {
	mcn.x = 0; mcn.y = 0; mcn.z = -mc.z; 
    } else {
	mcn.x = mc.x; mcn.y = mc.y; mcn.z = 0; 
    }
    RTM_mcN2wcN( wc, mcn, n );
}

RT_Point RT_Cylinder::inverse(const RT_Vector &v) const {
    if (fabs( v.x * v.x + v.y * v.y - xrad * xrad ) < rt_RayEps ) {
	RT_Point p;
	p.y = 0.5 + v.z/xlen;
	p.x =  asin( v.x / xrad )* 0.5 / M_PI;
	if (v.y < 0) p.x = 0.5 - p.x;
	if (p.x < 0) p.x += 1;
	return p;
    }
    return RT_Point( -1, -1 );
}

void RT_Cylinder::createReferences(const RT_AttributeList &list) {
    RT_Primitive::createReferences( list );
    // create resolution reference:
    RT_ResolutionAttribute *tmp = xsolution;
    xsolution = (RT_ResolutionAttribute*)list.retraverse( RTN_RESOLUTION_ATTR, this );
    if ( (tmp != xsolution) || (*tmp != *xsolution )) geomChanged();
}

void RT_Cylinder::create() {
    // max dim: 50, min dim: 4
    int dim =(int)( 46.0 *xsolution->get_resolution() + 4);
    // must a new geometry created?
    if (xmesh->get_x() != dim) xmesh->newGeometry( dim, dim );

    double fak = 2 * M_PI / (double)(dim - 1);

    double xlen2 = xlen/2;

    // compute a simple circle:
    RT_Vector *circle = new RT_Vector[dim];
    RT_Vector *c = circle;
    for (int i = 0; i < dim; i++) {
	double r = i * fak;
	c->x = xrad * cos( r ); 
	c->y = xrad * sin( r );
	c->z = xlen2;
	c++;
    }

    double fak2 = 1.0/(dim-1);
    // draw the cylinder:
    RT_Vector tmp;
    for (i = 0; i < dim; i++) {
	c = circle;
	tmp.z = xlen * i * fak2 - xlen2;
	for (int j = 0; j < dim; j++) {
	    tmp.x = c->x;
	    tmp.y = c->y;
	    xmesh->vtPoint(i,j, tmp);
	    c++;
	}
    }
    xmesh->computeNormals();
    
    // draw the cover:
    if (xcover1) {
	xcover1->father( 0 );
	delete xcover1;
	xcover2->father( 0 );
	delete xcover2;
    }
    if (xop) { xcover1 = 0; xcover2 = 0; }
    else {
	c = circle; int i;
	for ( i = 0; i < dim; i++) { c->z -= xlen; c++ ; }
	// cover at -z axis:
	xcover1 = new RT_Polygon( 0, dim-1, circle );
	xcover1->gnormal( RT_Vector(0, 0, -1));
	xcover1->father( this );
	xcover1->createReferences( *attributes );

	c = circle;
	for ( i = 0; i < dim; i++) { c->z += xlen; c++ ; }
	// resort the vertices in circle to be conform
	// to the right-hand rule:
	int ii = dim/2; RT_Vector tv;
	for (i = 0; i < ii; i++) {
	    tv = circle[i]; circle[i] = circle[dim-i-1]; circle[dim-i-1] = tv;
	}
	// cover at +z axis:
	xcover2 = new RT_Polygon( 0, dim-1, circle );
	xcover2->gnormal( RT_Vector(0, 0, 1));
	xcover2->father( this );
	xcover2->createReferences( *attributes );
    }
    delete circle;
}

RT_ParseEntry RT_Cylinder::table[] = {
    { "-radius", RTP_DOUBLE, (char*)&radV, &radF, "Specify the {ARG 1 Radius} for the cylinder.", RTPS_DOUBLE },
    { "-get_radius", RTP_NONE, 0, &radG, "Get the radius of the cylinder.", RTPS_NONE },
    { "-length", RTP_DOUBLE, (char*)&lenV, &lenF, "Set the {ARG 1 Length} for the cylinder.", RTPS_DOUBLE },
    { "-get_length", RTP_NONE, 0, &lenG, "Inquire the length of the cylinder.", RTPS_NONE },
    { "-open", RTP_NONE, 0, &opF, "Open the cylinder.", RTPS_NONE },
    { "-get_open", RTP_NONE, 0, &opG, "Return current open/close state.", RTPS_NONE },
    { "-close", RTP_NONE, 0, &clF, "Close the cylinder.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_Cylinder::objectCMD(char *argv[]) {
    int r = RT_Primitive::objectCMD( argv );
    RT_parseTable( argv, table );
    if (radF) { radius( radV ); r++; }
    if (radG) {
	r++; 
	char tmp[20];
	RT_double2string( get_radius(), tmp );
	RT_Object::result( tmp );
    }
    if (lenF) { length( lenV ); r++; }
    if (lenG) {
	r++; 
	char tmp[20];
	RT_double2string( get_length(), tmp );
	RT_Object::result( tmp );
    }
    if (opF) { open(); r++; }
    if (clF) { close(); r++; }
    if (opG) { RT_Object::result( get_open() ? "1" : "0" ); r++; }
    return r;
}

int RT_Cylinder::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Double Double} {Creates a new cylinder called {ARG 1 Name} with an inital {ARG 2 Radius} and {ARG 3 Length}. The cylinder is built around the z-axis. The middle point of the cylinder is z = 0.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if (argc != 4) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <radius> <length>. ", 0 );
	    return TCL_ERROR;
	}
	double rad, len;
	if ( RT_string2double( argv[2], rad )) 
	    if ( RT_string2double( argv[3], len )) {
		new RT_Cylinder( argv[1], rad, len ); 
		RTM_classReturn;
	    }
	Tcl_AppendResult( ip, "Bad parameter. Radius and Length must be floats. ", 0 );
	return TCL_ERROR;
    }
    return res; 
}

