////////////////////////////////////////////////////////////////////////////////
//  Analytical classes such as Sphere, Quader, Ellipsoid, Plane               //  
//  LAST EDIT: Wed Mar  8 09:13:06 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#ifndef __ANALYTICAL_H__
#define __ANALYTICAL_H__

#include "quadric.h"
#include "quadmesh.h"
#include "predattr.h"

///// the class SPHERE

extern const char *RTN_SPHERE;

class RT_Sphere: public RT_Primitive, public RT_ResolutionAttrImpl {
    void create();
    // create it!
    RT_ResolutionAttribute *xsolution;
    double xrad; 
    RT_Quadric *qu;
    RT_Quadmesh *qum;
    static RT_ParseEntry table[];
    //##### statics for parameter parsing:
    static double radV;
    static int radF, radG;
  public:
    //#### the Tcl/C++ methods:
    RT_Sphere( char *_name, double _r) : RT_Primitive( _name ) {
	xrad = _r; xsolution = 0; 
	qu = new RT_Quadric( 0, RT_Vector(1,1,1), RT_Vector(0,0,0), RT_Vector(0,0,0), -_r*_r);
	qu->father( this );
	qum = new RT_Quadmesh( 0, 0, 0, 0 );
	qum->father( this );
    }

    void printCon(FILE *f) const { 
	RT_Primitive::printCon( f ); fprintf( f, "%lf ", xrad ); 
    }

    const char *get_class() const { return RTN_SPHERE; }
    const char *get_description() const { return "A sphere with changeable radius."; }
    int isA(const char *_c) const { return RT_Primitive::isA( _c ) || RTM_isA(_c, RTN_SPHERE );}

    int copy(RT_Primitive *) const;

    RT_Bounds get_bounds();

    void radius(double r) { 
	xrad = r;
	// quadric:
	qu->setConst( -xrad*xrad);
	// quadmesh:
	geomChanged();
    }
    double get_radius() const { return xrad; }

    //#### ray-tracing interface:
    int intersect(const RT_Ray &a, RT_InterSectionList &b) { return qu->intersect( a, b ); } 
    void normal(const RT_Vector &w, RT_Vector &n) { qu->normal( w, n ); } 

    RT_Point inverse(const RT_Vector &v) const;

    //#### generic primitive methods:
    void checkAttributes() {
	if (xsolution->isChanged()) geomChanged(); 
    }
    void createReferences(const RT_AttributeList &);

    //#### interface of Resolution:
    void resolution(double); 
    double get_resolution() const { return xsolution->get_resolution(); }

    //#### the Tcl commands:
    static int classCMD(ClientData, Tcl_Interp *, int, char *[]); 
    int objectCMD(char *[]);
};

///// the class Cone

extern const char *RTN_CONE;

class RT_Polygon;

class RT_Cone: public RT_Primitive, public RT_ResolutionAttrImpl {
    void create();
    RT_ResolutionAttribute *xsolution;
    double xrad, xlen;
    RT_Quadmesh *xmesh;
    RT_Polygon *xcover;
    int xop;
    // default value: 0 (meaning the cone is closed)
    static RT_ParseEntry table[];

    //##### statics for parameter parsing:
    static double radV;
    static int radF, radG;
    static double lenV;
    static int lenF, lenG;
    static int opF, clF, opG;
  public:
    //#### Tcl/C++ methods:
    RT_Cone( char *_name, double _r, double _l): RT_Primitive(_name) { 
	xrad = _r; xsolution = 0; xlen = _l; xop = 0; xcover = 0; 
	xmesh = new RT_Quadmesh( 0, 0, 0, 0 );
	xmesh->father( this );
    }
    const char *get_class() const { return RTN_CONE; }
    const char *get_description() const { return "A cone with changeable dimensions: radius and length."; }
    int isA(const char *_c) const { return RT_Primitive::isA( _c ) || RTM_isA(_c, RTN_CONE );}
    
    int copy(RT_Primitive *) const;

    RT_Bounds get_bounds();

    void radius(double r) { xrad = r; geomChanged(); }
    double get_radius() const { return xrad; }
    void length(double l) { xlen = l; geomChanged(); }
    double get_length() const { return xlen; }
    void open() { xop = 1; geomChanged(); }
    void close() { xop = 0; geomChanged(); }
    int get_open() const { return xop; }

    void printCon(FILE *f) const {
	RT_Primitive::printCon( f );
	fprintf( f, "%lf %lf ", xrad, xlen ); 
    }
    void print(FILE *f) const {
	RT_Primitive::print( f );
	if (xop) fprintf( f, "%s -open\n", get_name() );
    }
    //#### interface of Resolution:
    void resolution(double); 
    double get_resolution() const { return xsolution->get_resolution(); }

    //#### ray-tracer interface:
    int intersect(const RT_Ray&, RT_InterSectionList &); 
    void normal(const RT_Vector&, RT_Vector &);

    void checkAttributes() {
	if (xsolution->isChanged()) geomChanged(); 
    }
    void createReferences(const RT_AttributeList &);
    RT_Point inverse(const RT_Vector &) const;

    //#### the Tcl commands:
    static int classCMD(ClientData, Tcl_Interp *, int, char *[]); 
    int objectCMD(char *[]);
};

///// the class Torus

extern const char *RTN_TORUS;

class RT_Torus: public RT_Quadmesh, public RT_ResolutionAttrImpl {
    void create();
    // create it!
    RT_ResolutionAttribute *xsolution;
    double irad; 
    double orad; 
    static RT_ParseEntry table[];
    //##### statics for parameter parsing:
    static double iradV, oradV;
    static int iradF, iradG, oradF, oradG;
  public:
    //#### ray-tracer interface:
    int intersect(const RT_Ray&, RT_InterSectionList &);
    void normal(const RT_Vector&, RT_Vector&);

    //#### generic primitive methods:
    void checkAttributes() {
	RT_Quadmesh::checkAttributes();
	if (xsolution->isChanged()) geomChanged(); 
    }
    void createReferences(const RT_AttributeList &);

    //#### Tcl/C++ methods:
    RT_Torus( char *_name, double _ir, double _or ): RT_Quadmesh(_name, 0, 0, 0 ) { 
	irad = _ir; orad = _or; xsolution = 0; 
    }

    int copy(RT_Primitive *) const;

    const char *get_class() const { return RTN_TORUS; }
    const char *get_description() const { return "A torus with two changeable radiuses."; }
    int isA(const char *_c) const { return RT_Primitive::isA( _c ) || RTM_isA(_c, RTN_TORUS );}

    void printCon(FILE *f) const { RT_Primitive::printCon( f ); fprintf( f, "%lf %lf", irad, orad ); }
    void print(FILE *f) const { RT_Primitive::print( f ); }

    RT_Bounds get_bounds();

    void iradius(double r) { 
	irad = r;
	geomChanged();
    }
    double get_iradius() const { return irad; }

    void oradius(double r) { 
	orad = r;
	geomChanged();
    }
    double get_oradius() const { return orad; }

    //#### interface of Resolution:
    void resolution(double); 
    double get_resolution() const { return xsolution->get_resolution(); }

    //#### the Tcl commands:
    static int classCMD(ClientData, Tcl_Interp *, int, char *[]); 
    int objectCMD(char *[]);
};

///// the class PLANE

extern const char *RTN_PLANE;

class RT_Plane: public RT_Quadric {
    double a, b, c, d;
  public:
    //#### Tcl/C++ methods:
    RT_Plane( char *_name, double _a, double _b, double _c, double _d): 
    RT_Quadric(_name, RT_Vector(0,0,0), RT_Vector(0,0,0), 
	       RT_Vector(_a * 0.5, _b * 0.5, _c * 0.5),_d), 
    a(_a), b(_b), c(_c), d(_d) {}
    
    const char *get_description() const { return "An infinite plane only visible in ray-tracing mode."; }
    const char *get_class() const { return RTN_PLANE; }
    int isA(const char *_c) const { return RT_Primitive::isA( _c ) || RTM_isA(_c, RTN_PLANE );}

    void printCon(FILE *f) const { 
	RT_Primitive::printCon( f ); 
	fprintf( f, "%lf %lf %lf %lf ", a, b, c, d);
    }
    void print(FILE *f) const { RT_Primitive::print( f ); }

    //#### ray-tracer interface:
    int intersect(const RT_Ray&, RT_InterSectionList &); 
    void normal(const RT_Vector&, RT_Vector &bb);

    //#### the Tcl commands:
    static int classCMD(ClientData, Tcl_Interp *, int, char *[]); 
    int objectCMD(char *argv[]) { return RT_Primitive::objectCMD( argv ); }
};

///// the class ELLIPSOID

extern const char *RTN_ELLIPSOID;

class RT_Ellipsoid: public RT_Primitive {
    double a, b, c;
    RT_Sphere *sp;
  public:
    //#### Tcl/C++ methods:
    RT_Ellipsoid( char *_name, double _a, double _b, double _c) : RT_Primitive(_name),
    a(_a), b(_b), c(_c) {
	sp = new RT_Sphere( 0, 1 );
	sp->father( this );
	sp->scale( RT_Vector( a, b, c ) );
    }
    const char *get_description() const { return "An ellipsoid. The dimensions cannot be changed after construction."; }
    const char *get_class() const { return RTN_ELLIPSOID; }
    int isA(const char *_c) const { return RT_Primitive::isA( _c ) || RTM_isA(_c, RTN_ELLIPSOID );}

    void printCon(FILE *f) const { 
	RT_Primitive::printCon( f ); 
	fprintf( f, "%lf %lf %lf ", a, b, c);
    }

    int copy(RT_Primitive *p) const {
	return sp->copy( p );
    }
    //#### the Tcl commands:
    static int classCMD(ClientData, Tcl_Interp *, int, char *[]); 
};

///// the class QUADER

extern const char *RTN_QUADER;

class RT_Quader: public RT_Primitive {
    static RT_ParseEntry table[];
    RT_Quadmesh *qpx, *qmx, *qpy, *qmy, *qpz, *qmz;
    RT_Vector mcnorm; // the normal vector of the nearest hit
    double dx, dy, dz;
    void create();
    //#### the statics for parameter parsing:
    static double xV, yV, zV; 
    static int xF, yF, zF, xG, yG, zG;
  public:
    //#### the Tcl/C++ methods:
    RT_Quader( char *_name, double _dx, double _dy, double _dz): RT_Primitive(_name),
    dx( _dx ), dy( _dy ), dz( _dz ) {
	qpx = new RT_Quadmesh(0,2,2,0); qpx->father( this );
	qpy = new RT_Quadmesh(0,2,2,0); qpy->father( this );
	qpz = new RT_Quadmesh(0,2,2,0); qpz->father( this );
	qmx = new RT_Quadmesh(0,2,2,0); qmx->father( this );
	qmy = new RT_Quadmesh(0,2,2,0); qmy->father( this );
	qmz = new RT_Quadmesh(0,2,2,0); qmz->father( this );
    }
    const char *get_class() const { return RTN_QUADER; }
    int isA(const char *_c) const { return RT_Primitive::isA( _c ) || RTM_isA(_c, RTN_QUADER );}
    const char *get_description() const { return "A quader with changeable dimensions. This quader maps a potential image to only one plane: the positive Z axis. You may change this in subclasses of the quader."; }

    int copy(RT_Primitive *) const;

    void xdim(double _x) { dx = _x; geomChanged(); }
    double get_xdim() const { return dx; };
    void ydim(double _y) { dy = _y; geomChanged(); }
    double get_ydim() const { return dy; };
    void zdim(double _z) { dz = _z; geomChanged(); }
    double get_zdim() const { return dz; };

    void printCon( FILE *f) const {
	RT_Object::printCon( f);
	fprintf( f, "%lf %lf %lf", dx, dy, dz);
    }
    void print(FILE *f) const { RT_Primitive::print( f ); }

    RT_Bounds get_bounds();

    //##### ray-tracing interface:
    int intersect(const RT_Ray &, RT_InterSectionList &);
    void normal(const RT_Vector&, RT_Vector &);
    RT_Point inverse(const RT_Vector &) const;

    //#### the Tcl commands:
    static int classCMD(ClientData, Tcl_Interp *, int, char *[]); 
    int objectCMD(char *[]);
};

///// the class Cylinder

extern const char *RTN_CYLINDER;

class RT_Cylinder: public RT_Primitive, public RT_ResolutionAttrImpl {
    void create();
    RT_ResolutionAttribute *xsolution;
    double xrad, xlen;
    RT_Polygon *xcover1, *xcover2;
    RT_Quadmesh *xmesh;
    int xop;
    // default value: 0 (meaning the cylinder is closed)
    static RT_ParseEntry table[];
    //##### statics for parameter parsing:
    static double radV;
    static int radF, radG;
    static double lenV;
    static int lenF, lenG;
    static int opF, clF, opG;
  public:
    void checkAttributes() {
	if (xsolution->isChanged()) geomChanged(); 
    }
    void createReferences(const RT_AttributeList &);

    //#### the Tcl/C++ methods:
    RT_Cylinder( char *_name, double _r, double _l) : RT_Primitive(_name) {
	xrad = _r; xsolution = 0; xlen = _l; xop = 0; xcover1 = 0; xcover2 = 0; 
	xmesh = new RT_Quadmesh( 0, 0, 0, 0 );
	xmesh->father( this );
    }
    const char *get_class() const { return RTN_CYLINDER; }
    int isA(const char *_c) const { return RT_Primitive::isA( _c ) || RTM_isA(_c, RTN_CYLINDER );}
    const char *get_description() const { return "A cylinder with changeable dimensions: radius and length."; }

    int copy(RT_Primitive *) const;

    void printCon(FILE *f) const {
	RT_Primitive::printCon( f );
	fprintf( f, "%lf %lf ", xrad, xlen ); 
    }
    void print(FILE *f) const {
	RT_Primitive::print( f );
	if (xop) fprintf( f, "%s -open\n", get_name() );
    }

    void radius(double r) { xrad = r; geomChanged(); }
    double get_radius() const { return xrad; }

    void length(double l) { xlen = l; geomChanged(); }
    double get_length() const { return xlen; }

    void open() { xop = 1; geomChanged(); }
    void close() { xop = 0; geomChanged(); }
    int get_open() const { return xop; }

    RT_Bounds get_bounds();

    //#### interface of Resolution:
    void resolution(double); 
    double get_resolution() const { return xsolution->get_resolution(); }

    //#### ray-tracer interface
    int intersect(const RT_Ray&, RT_InterSectionList&); 
    void normal(const RT_Vector&, RT_Vector&);
    RT_Point inverse(const RT_Vector &) const;

    //#### the Tcl commands:
    static int classCMD(ClientData, Tcl_Interp *, int, char *[]); 
    int objectCMD(char *[]);
};

#endif


