#ifndef __LOOKAT_CAMERA_H__
#define __LOOKAT_CAMERA_H__

////////////////////////////////////////////////////////////////////////////////
//  Two cameras based on the lookat model                                     //  
//  LAST EDIT: Fri Aug  5 08:55:00 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "../camera.h"
#include "../image.h"

extern const char *RTN_LOOKAT_CAMERA;
extern const char *RTSN_LC;

typedef enum {
    RTE_LC_SHADING = 0,  
    // render scene using a simple gouraud shader
    RTE_LC_RAY_TRACING, 
    // render scene using recursive raytracing
#ifdef RTD_RSY
    RTE_LC_RADIOSITY,  
    // render radiosity scene using the whole computed intensity (diffuse + non-diffuse parts)
    RTE_LC_DIFFUSE_RADIOSITY,
    //render radiosity scene using diffuse components only
    RTE_LC_WIREFRAME_RADIOSITY
    //draw radiosity scene geometry as a wireframe
#endif
} RT_LookatCameraMode;

class RT_LookatCamera:public RT_RayCamera {
    //##### statics for parameter parsing:
    static RT_Vector vpV; static int vpF, vpG;
    static RT_Vector rpV; static int rpF, rpG;
    static int moV, moF, moG;
    static int twV, twF, twG;
    static double anV; static int anF, anG;
    static double nrV; static int nrF, nrG;
    static double frV; static int frF, frG;
    //##### privates:
    static RT_ParseEntry table[];
  protected:
    void shading(); // platform dependent rendering (shading)
#ifdef RTD_RSY
    void RSYshading();
#endif
    RT_Vector vp, rp; // viewpoint and ref point
    int xtwist; // right hand rotation about the z axis in the eye coordinate system
    RT_LookatCameraMode xmode;
    double xangle, xnear, xfar;
    RT_Matrix lam;
    // the lookat matrix
    void computeLookatMatrix();
  public:
    RT_Primitive *getPrimitive(const RT_Ray &r) { return RT_RayCamera::getPrimitive( r ); }
    const RT_Matrix &getMatrix() { return lam; }
    void buildRay( double, double, RT_Ray &);
    virtual void event(RT_Event &);

    //#### Tcl/C++ methods:
    RT_LookatCamera( char *, const RT_Vector &VIEW, const RT_Vector &REF);
#ifndef RTD_RSY
    const char *get_description() const { return "A combined camera for shading/ray-tracing in one and the same pixmap. For final ray-tracing images use the supersampling camera (RayImageLookatCamera)."; }
#else
    const char *get_description() const { return "A combined camera for shading/ray-tracing and radiosity in one and the same pixmap. For final ray-tracing images use the supersampling camera (RayImageLookatCamera)"; }
#endif
    const char *get_keywords() const { return "Shading Ray-Tracing Supersampling Lookat Radiosity"; }
    const char *get_class() const { return RTN_LOOKAT_CAMERA; }
    int isA(const char *c) const { return RT_RayCamera::isA( c ) || RTM_isA( c, RTN_LOOKAT_CAMERA ); }

    void rendering();
    void refresh() {
	RT_LookatCameraMode m = get_mode();
	mode( RTE_LC_SHADING ); 
	rendering(); 
	mode( m );
    }

    void printCon( FILE *f) const { RT_Object::printCon( f); vp.print( f ); rp.print( f ); }
    void print(FILE *f) const {
	RT_Camera::print(f); 
	fprintf(f, "%s -angle %lf -znear %lf -zfar %lf -mode %i -twist %i\n", get_name(), xangle, xnear, xfar, xmode, xtwist );
    }

    void viewpoint(const RT_Vector &_vp) { vp = _vp; computeLookatMatrix(); }
    const RT_Vector &get_viewpoint() const { return vp; }

    void refpoint(const RT_Vector &_rp) { rp = _rp; computeLookatMatrix(); }
    const RT_Vector &get_refpoint() const { return rp; }
    void mode( RT_LookatCameraMode _m) { xmode = _m; }
    RT_LookatCameraMode get_mode() const { return xmode; }

    void twist(int _tw) { xtwist = _tw; computeLookatMatrix(); }
    int get_twist() const { return xtwist; }

    void angle(double _a) { 
	if (_a < 1) {
	    _a = 1;
	    rt_Output->errorVar( get_name(), ": Angle is smaller than one degree, thus set to 1 degree.", 0 );
	}
	if (_a >= 180) {
	    _a = 179;
	    rt_Output->errorVar( get_name(), ": Angle is bigger than or equal to 180 degrees, thus set to 179 degrees.", 0 );
	}
	xangle = _a; computeLookatMatrix(); 
    }
    double get_angle() const { return xangle; }

    void znear(double _n) { xnear = _n; computeLookatMatrix(); }
    double get_znear() const { return xnear; }

    void zfar(double _f) { xfar = _f; computeLookatMatrix(); }
    double get_zfar() const { return xfar; }
    //#### the Tcl commands: 
    static int classCMD(ClientData, Tcl_Interp *, int, char *[]);  
    int objectCMD(char *[]);  
};

extern const char *RTN_RAY_IMAGE_LOOKAT_CAMERA;
extern const char *RTSN_RILC;

class RT_RayImageLookatCamera:public RT_LookatCamera {
    //##### privates:
    static RT_ParseEntry table[];
  protected:
    //#### the statics for parameter parsing:
    static int imgF, imgG, nsF, nsG, jtF, jtG;
    static char *imgV;
    static int nsV;
    static double jtV;
    double xjitter;
    int xsamples; // Number of samples per pixel in each direction
    RT_Image *ximage; // the current image 
  public:
    void buildRay( double, double, RT_Ray &);

    void objectKilled(RT_Object *a) {
	if (a == ximage) image( (RT_Image*)-1 );
	RT_Camera::objectKilled( a );
    }
    //#### Tcl/C++ methods:
    RT_RayImageLookatCamera( char *_n, const RT_Vector &_vp, const RT_Vector &_rp): RT_LookatCamera( _n , _vp, _rp ) {
	xsamples = 3; xjitter = 0.3; ximage = 0;
    }
    const char *get_description() const { return "A realistic camera for ray-tracing to an image."; }  
    const char *get_class() const { return RTN_RAY_IMAGE_LOOKAT_CAMERA; }
    int isA(const char *c) const { return RT_LookatCamera::isA( c ) || RTM_isA( c, RTN_RAY_IMAGE_LOOKAT_CAMERA ); }
    virtual void rendering();
    void samples(int _ns) { xsamples = _ns; }
    int get_samples() const { return xsamples; }
    void jitter(double _j) { xjitter = _j; }
    double get_jitter() const { return xjitter; }

    void print(FILE *f) const {
	RT_LookatCamera::print(f); 
	fprintf( f, "%s -jitter %lf -samples %i", get_name(), xjitter, xsamples );
	if (ximage) fprintf( f, " -image %s ", ximage->get_name() );
	fprintf( f, "\n" );
    }

    void image(RT_Image *); 
    RT_Image *get_image() const { return ximage; }

    //#### the Tcl commands: 
    static int classCMD(ClientData, Tcl_Interp *, int, char *[]);  
    int objectCMD(char *[]);  
};

#endif


