////////////////////////////////////////////////////////////////////////////////
//  VertexManipulator manipulation                                            //  
//  LAST EDIT: Wed Mar  8 17:10:03 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////


#include "vpick.h"
#include "../poly.h"
#include "../polyhdrn.h"
#include "../quadmesh.h"
#include "../high/height.h"
#include "../high/analytic.h"

const char *RTN_VERTEX_ENTRY = "VertexEntry";

// these are generic interfaces to the vertex classes

static RT_Vertex *getVertex(const RT_Primitive *p, int i) {
    if (p->isA(RTN_POLYVERTEX))
	return ((RT_Polyvertex *)p)->get(i);
    else
	if (p->isA(RTN_POLYHEDRON))
	    return ((RT_Polyhedron *)p)->get(i);
	else
	    if (p->isA(RTN_QUADMESH))
		return ((RT_Quadmesh *)p)->get(i);
	    else
		if (p->isA(RTN_HEIGHT_FIELD))
		    return ((RT_HeightField *)p)->getQuadmesh()->get(i);
    return 0;
    
}

static int get_number_of_vertices(const RT_Primitive *p) {
    if (p->isA(RTN_POLYVERTEX))
	return ((RT_Polyvertex *)p)->get_number();
    else
	if (p->isA(RTN_POLYHEDRON))
	    return ((RT_Polyhedron *)p)->get_npoints();
	else
	    if (p->isA(RTN_QUADMESH))
		return ((RT_Quadmesh *)p)->get_x() * ((RT_Quadmesh *)p)->get_y();
	    else
		if (p->isA(RTN_HEIGHT_FIELD))
		    return ((RT_HeightField *)p)->get_x() * ((RT_HeightField *)p)->get_y();
    return -1;
    
}

const RT_Color VC_NORMAL_COLOR(1,1,0);
const RT_Color VC_SELECTED_COLOR(1,0,0);

void RT_VertexEntry::get(const RT_Primitive *p) {
    RT_Vertex *v = getVertex(p, getIdx());
    if (!v) {delete this; return;}
    matrix(RT_IDENTITY.TRANSLATION(v->getPoint()));
    xchanged=0;
}

void RT_VertexEntry::put(RT_Primitive *p) {
    RT_Vertex *v = getVertex(p, getIdx());
    if (!v) {delete this; return;}
    RT_Vector _v = getModel()->get_matrix() * RT_Vector(0,0,0);
    if (p->isA(RTN_HEIGHT_FIELD)) {
	double h = _v.y;
	_v = v->getPoint();
	_v.y = h;
	matrix(RT_IDENTITY.TRANSLATION(_v));
    }
    v->point(_v);
    p->geomChanged();
    xchanged=0;
}

void RT_VertexEntry::setIdx(const RT_Primitive *p, int i) {
    int n = get_number_of_vertices(p);
    if (i >= 0 && i < n) index = i;
    get(p);
}

RT_Primitive* RT_VertexEntry::get_Primitive() {
    RT_VertexCollection *vc = (RT_VertexCollection *)get_father();
    return (vc && vc->isA(RTN_VERTEX_COLLECTION)) ? vc->get_Primitive() : 0;
}


const char *RTN_VERTEX_COLLECTION = "VertexCollection";
    
RT_VertexCollection::RT_VertexCollection(RT_Primitive *_p, int idx) : RT_Primitive(0) {
    prim = _p;
    prim->addRelatedObject(this);
    addRelatedObject(prim);
    if (idx != -1) addVertexEntry(idx);
    else {
	int n = get_number_of_vertices(_p);
	for(int i=0;i<n;i++)
	    (new RT_VertexEntry(prim, i))->father(this);
    }
    sync();
}

void RT_VertexCollection::addVertexEntry(int _i) {
    if (_i >= 0 && _i < get_number_of_vertices(prim) && !isVertexInList(_i))
	(new RT_VertexEntry(prim, _i))->father(this);
}

void RT_VertexCollection::removeVertexEntry(int idx) {
    class LFunctoid : public RT_GeneralListFunctoid {
      public:
	void exec(RT_GeneralListEntry *tmp, void *idx) {
	    if (((RT_VertexEntry *)tmp)->getIdx() == *(int *)idx)
		delete tmp;
	}
    } func;
    parts.doWithElements(&func,&idx);
}

void RT_VertexCollection::invertCollection() {
    int n = get_number_of_vertices(prim);
    char *flags = new char[n];
    bzero(flags, n * sizeof(char));
    class LFunctoid : public RT_GeneralListFunctoid {
      public:
	void exec(RT_GeneralListEntry *tmp, void *flags) {
	    ((char *)flags)[((RT_VertexEntry *)tmp)->getIdx()] = 1;
	    delete tmp;
	}
    } F;
    parts.doWithElements(&F,flags);
    for(int i = 0; i < n; i++)
	if (!flags[i])
	    (new RT_VertexEntry(prim, i))->father(this);
    delete flags;
}

int RT_VertexCollection::isVertexInList(int idx) const {    
    class LFunctoid : public RT_GeneralListFunctoid {
	int idx;
      public:
	LFunctoid(int _i) {idx = _i;}
	void exec(RT_GeneralListEntry *tmp, void *flag) {
	    if (((RT_VertexEntry *)tmp)->getIdx() == idx)
		*(int *)flag = 1;
	}
    } func(idx);
    int flag = 0;
    parts.doWithElements(&func, &flag);
    return flag;
}

void RT_VertexCollection::sync() {
    matrix(prim->get_worldMatrix());
    class LFunctoid : public RT_GeneralListFunctoid {
      public:
	void exec(RT_GeneralListEntry *tmp, void *prim) {
	    ((RT_VertexEntry *)tmp)->sync((RT_Primitive *)prim);
	}
    } func;
    parts.doWithElements(&func,prim);    
}

void RT_VertexCollection::doFixedCoordTrans(const RT_Vector &v) {
    class LFunctoid: public RT_GeneralListFunctoid {
	RT_Primitive *p; RT_Vector v;
      public:
	LFunctoid(RT_Primitive *_p, const RT_Vector &_v) {p = _p; v = _v;}
	void exec(RT_GeneralListEntry *tmp, void *) {
	    RT_fixedCoordTrans((RT_VertexEntry *)tmp, v);
	    ((RT_VertexEntry *)tmp)->changed();
	}
    } func(prim,v);
    parts.doWithElements(&func);    
}

RT_VertexEntry *RT_VertexCollection::getEntry(int idx) const {
    class LFunctoid: public RT_GeneralListFunctoid {
	int idx;
      public:
	LFunctoid(int _i) {idx = _i;}
	void exec(RT_GeneralListEntry *tmp, void *vc) {
	    if (((RT_VertexEntry *)tmp)->getIdx() == idx)
		*(RT_VertexEntry **)vc = (RT_VertexEntry *)tmp;
	}
    } func(idx);
    RT_VertexEntry *vc = 0;
    parts.doWithElements(&func,&vc);
    return vc;
}   

void RT_VertexCollection::insertVertex(int i) const {
    if (!prim->isA(RTN_POLYVERTEX)) return;
    RT_Polyvertex *pv = (RT_Polyvertex *)prim;
    int n = pv->get_number();
    if (i < 0 || i >= n) return;
    if (i >= n-1) i = n-2;
    pv->vtInsert(i+1,(pv->get_vtPoint(i)+pv->get_vtPoint(i+1))*0.5);    
    class LFunctoid : public RT_GeneralListFunctoid {
	RT_Primitive *p;
	int idx;
      public:
	LFunctoid(RT_Primitive *_p, int _i) {p = _p, idx = _i;}
	void exec(RT_GeneralListEntry *tmp, void *) {
	    RT_VertexEntry *ve = (RT_VertexEntry *)tmp;
	    int i = ve->getIdx();
	    if (i > idx) ve->setIdx(p,i+1);
	}
    } func(prim, i);
    parts.doWithElements(&func);
}

void RT_VertexCollection::deleteVertex(int i) const {
    if (!prim->isA(RTN_POLYVERTEX)) return;
    RT_Polyvertex *pv = (RT_Polyvertex *)prim;
    pv->vtDelete(i);
    class LFunctoid: public RT_GeneralListFunctoid {
	RT_Primitive *p;
	int idx;
      public:
	LFunctoid(RT_Primitive *_p, int _i) {p = _p, idx = _i;}
	void exec(RT_GeneralListEntry *tmp, void *) {
	    RT_VertexEntry *ve = (RT_VertexEntry *)tmp;
	    int i = ve->getIdx();
	    if (i == idx) delete ve;
	    else
		if (i > idx) ve->setIdx(p,i-1);
	}
    } func(prim, i);
    parts.doWithElements(&func);
}

const char *RTN_PRIMITIVE_VERTEX_ASSOCIATION = "PrimitiveVertexAssociation";

RT_PrimitiveVertexAssociation::RT_PrimitiveVertexAssociation() : RT_Primitive(0) {
    ambient(VC_NORMAL_COLOR); diffuse(VC_NORMAL_COLOR);
}

class RT_FindCollectionFunctoid : public RT_GeneralListFunctoid {
    const RT_Primitive *p;
  public:
    RT_FindCollectionFunctoid(const RT_Primitive *_p) {p = _p;}
    void exec(RT_GeneralListEntry *tmp, void *vc) {
	if (((RT_VertexCollection *)tmp)->get_Primitive() == p)
	    *(RT_VertexCollection **)vc = (RT_VertexCollection *)tmp;
    }
};
    
int RT_PrimitiveVertexAssociation::isPrimitiveInList(const RT_Primitive *_p) const {
    RT_FindCollectionFunctoid func(_p);
    RT_VertexCollection *vc = 0;
    parts.doWithElements(&func, &vc);
    return (vc != 0);
}    

void RT_PrimitiveVertexAssociation::addVertexEntry(RT_Primitive *_p, int _i) {
    RT_FindCollectionFunctoid func(_p);
    RT_VertexCollection *vc = 0;
    parts.doWithElements(&func,&vc);
    if (vc) vc->addVertexEntry(_i);
    else (new RT_VertexCollection(_p,_i))->father(this);
}

void RT_PrimitiveVertexAssociation::addAllVertexEntrys(RT_Primitive *_p) {
    removeAllVertexEntrys(_p);
    (new RT_VertexCollection(_p,-1))->father(this);
}

RT_VertexEntry *RT_PrimitiveVertexAssociation::getEntry(const RT_Primitive *_p, int _i) const {
    RT_FindCollectionFunctoid func(_p);
    RT_VertexCollection *vc = 0;
    parts.doWithElements(&func,&vc);
    if (vc) return vc->getEntry(_i);
    else return 0;
}
    
void RT_PrimitiveVertexAssociation::removeVertexEntry(RT_Primitive *_p, int _i) {
    RT_FindCollectionFunctoid func(_p);
    RT_VertexCollection *vc = 0;
    parts.doWithElements(&func,&vc);
    if (vc) vc->removeVertexEntry(_i);
}

void RT_PrimitiveVertexAssociation::removeAllVertexEntrys(RT_Primitive *_p) {
    RT_FindCollectionFunctoid func(_p);
    RT_VertexCollection *vc = 0;
    parts.doWithElements(&func,&vc);
    if (vc) delete vc;
}

void RT_PrimitiveVertexAssociation::invertVertexCollection(RT_Primitive *_p) {
    RT_FindCollectionFunctoid func(_p);
    RT_VertexCollection *vc = 0;
    parts.doWithElements(&func,&vc);
    if (vc) vc->invertCollection();
    else if (_p) (new RT_VertexCollection(_p,-1))->father(this);
}

void RT_PrimitiveVertexAssociation::insertVertex(RT_Primitive *_p, int _i) {
    RT_FindCollectionFunctoid func(_p);
    RT_VertexCollection *vc = 0;
    parts.doWithElements(&func,&vc);
    if (vc) vc->insertVertex(_i);
}

void RT_PrimitiveVertexAssociation::deleteVertex(RT_Primitive *_p, int _i) {
    RT_FindCollectionFunctoid func(_p);
    RT_VertexCollection *vc = 0;
    parts.doWithElements(&func,&vc);
    if (vc) vc->deleteVertex(_i);
}

void RT_PrimitiveVertexAssociation::sync(RT_Primitive *_p) {
    if (!_p) {
	class LFunctoid: public RT_GeneralListFunctoid {
	  public:
	    void exec(RT_GeneralListEntry *tmp, void *) {
		((RT_VertexCollection *)tmp)->sync();
	    }
	} func;
	parts.doWithElements(&func);
    } else {
	RT_FindCollectionFunctoid func(_p);
	RT_VertexCollection *vc = 0;
	parts.doWithElements(&func,&vc);
	if (vc) vc->sync();
    }
}

void RT_PrimitiveVertexAssociation::doFixedCoordTrans(RT_Primitive *_p, const RT_Vector &v) {
    RT_FindCollectionFunctoid func(_p);
    RT_VertexCollection *vc = 0;
    parts.doWithElements(&func,&vc);
    if (vc) vc->doFixedCoordTrans(v);
}

// the following functoid could be declared locally
// however g++ 2.3.3 runs into problems if the inlined
// exec function creates an object of its own class

class RT_PVACSFunctoid: public RT_GeneralListFunctoid {
    const RT_Primitive *p;
  public:
    RT_PVACSFunctoid(const RT_Primitive *_p) {p = _p;}
    void exec(RT_GeneralListEntry *, void *);
};

void RT_PVACSFunctoid::exec(RT_GeneralListEntry *tmp, void *flag) {
    if (!*(int *)flag && ((RT_Object *)tmp)->isA(RTN_PRIMITIVE)) {
	const RT_Primitive *_p = (RT_Primitive *)tmp;
	if (p == _p) *(int *)flag = 1;
	else {
	    RT_PVACSFunctoid func(p);
	    _p->get_children().doWithElements(&func,flag);
	}
    }
}

void RT_PrimitiveVertexAssociation::checkScene(const RT_Scene *s) {    
    class LFunctoid1: public RT_GeneralListFunctoid {
	const RT_Scene *s;
      public:
	LFunctoid1(const RT_Scene *_s) {s = _s;}
	void exec(RT_GeneralListEntry *tmp, void *) {
	    const RT_Primitive *p = ((RT_VertexCollection *)tmp)->get_Primitive();
	    int flag = 0;
	    RT_PVACSFunctoid func( p );
	    s->doWithElements(&func, &flag);
	    if (!flag) delete (RT_VertexCollection *)tmp;
	    // delete VertexCollections if ass. Primitive not in Scene
	}
    } func(s);
    parts.doWithElements(&func); // all parts must be VertexCollections!!!
}

const char *RTN_VERTEX_MANIPULATOR = "VertexManipulator";

int RT_VertexManipulator::insertF, RT_VertexManipulator::deleteF;
int RT_VertexManipulator::pickF;
char *RT_VertexManipulator::pickS; 
int RT_VertexManipulator::addVCBF, RT_VertexManipulator::remVCBF;
char *RT_VertexManipulator::addVCBS, *RT_VertexManipulator::remVCBS;
int RT_VertexManipulator::coordF, RT_VertexManipulator::coordGF;
int RT_VertexManipulator::wcoordGF;
RT_Vector RT_VertexManipulator::coordV;
RT_Matrix RT_VertexManipulator::wcoordGM;
int RT_VertexManipulator::chkscF;
int RT_VertexManipulator::idxGF;
int RT_VertexManipulator::lockF, RT_VertexManipulator::unlockF, RT_VertexManipulator::lockGF;
int RT_VertexManipulator::surfGF, RT_VertexManipulator::surfF;
char *RT_VertexManipulator::surfS;

RT_ParseEntry RT_VertexManipulator::table[] = {
    {"-insert", RTP_NONE, 0, &insertF, "Insert a vertex after the selected vertex.", RTPS_NONE},
    {"-delete", RTP_NONE, 0, &deleteF, "Delete the selected vertex.", RTPS_NONE },
    {"-pick", RTP_STRING, (char *)&pickS, &pickF, "Pick {ARG1 NONE, ALL or INVERT(all other)} vertices of the current object.", RTPS_NONE },
    {"-addVertexCB", RTP_STRING, (char*)&addVCBS, &addVCBF, "Add a new {ARG 1 vertex Callback} to the vertex callback list.", "Procedure"},
    {"-removeVertexCB", RTP_STRING, (char *)&remVCBS, &remVCBF, "Remove a {ARG 1 vertex Callback} from the vertex callback list.", "Procedure"},
    { "-VertexCoords", RTP_VECTOR, (char*)&coordV, &coordF, "Set the {ARG 1 modell coords} of the picked Vertex.", RTPS_VECTOR },
    { "-get_VertexCoords", RTP_NONE, 0, &coordGF, "Get the coords of the selected vertex in modell coords.", RTPS_NONE },
    { "-VertexWCoords", RTP_SPECIAL, 0, 0, "Set the {ARG 1 world coords} of the selected Vertex using {ARG2 World Matrix}.", "Vector Matrix" },
    { "-get_VertexWCoords", RTP_MATRIX, (char *)&wcoordGM, &wcoordGF, "Get the coords of the selected vertex in world coords.", RTPS_MATRIX },
    { "-VertexSurface", RTP_STRING, (char *)&surfS, &surfF, "Set the {ARG 1 surface} of the selected Vertex.", RTPS_SURFACE },
    { "-get_VertexSurface", RTP_NONE, 0, &surfGF, "Get the surface of the picked vertex.", RTPS_NONE },
    {"-checkScene", RTP_NONE, 0, &chkscF, "Check the current scene if all objects with picked verticies are in the scene. Remove the vertex cursors if not.", RTPS_NONE},
    {"-lock", RTP_NONE, 0, &lockF, "Lock the current object.", RTPS_NONE},
    {"-unlock", RTP_NONE, 0, &unlockF, "Unlock the current object.", RTPS_NONE},
    {"-get_lock", RTP_NONE, 0, &lockGF, "Get the lock state for current object.", RTPS_NONE},
    {"-get_idx", RTP_NONE, 0, &idxGF, "Get the index of the selected vertex in current object.", RTPS_NONE},
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_VertexManipulator::objectCMD(char *argv[]) { 
    int ret = RT_Manipulator::objectCMD( argv );

    RT_parseTable(argv, table);

    if (insertF) {
	RT_Primitive *p;
	if (vc && (p = vc->get_Primitive()) && p->isA(RTN_POLYVERTEX)) {
	    int idx = vc->getIdx();
	    int n = get_number_of_vertices(p);
	    if (idx >= n-1) idx = n-2; // no insert after the last vertex
	    pva->insertVertex(p, idx);
	    pva->addVertexEntry(p, idx+1);
	    select(pva->getEntry(p, idx+1));
	}
	ret++;
    }

    if (deleteF) {
	RT_Primitive *p;
	if (vc && (p = vc->get_Primitive()) && p->isA(RTN_POLYVERTEX)) {
	    if (get_number_of_vertices(p) > 3) {
		int idx = vc->getIdx();
		pva->deleteVertex(p, idx);
		updateFeedback();
	    }
	}
	ret++;
    }
    
    if (pickF) {
	int mode = -1;
	if (!strcmp(pickS, "NONE"))
	    mode = 0;
	else
	    if (!strcmp(pickS, "ALL"))
		mode = 1;
	    else
		if (!strcmp(pickS, "INVERT"))
		    mode = 2;
	pick(mode);
    }
	
	

    if (addVCBF) {
	if (addVertexTclCB( addVCBS )) ret++;
	else Tcl_AppendResult(rt_Ip, "There is already a vertex callback \"", addVCBS, "\" in device \"", get_name(), "\". ", NULL );
    }

    if (remVCBF) {
	if (removeVertexTclCB( remVCBS )) ret++; 
	else Tcl_AppendResult(rt_Ip, "There is no such vertex callback \"", remVCBS, "\" in device \"", get_name(), "\". ", NULL );
    }

    if (coordF) {
	VertexCoords(coordV);
	ret++;
    }

    if (coordGF) {
	char tmp[100];
	RT_vector2string(get_VertexCoords(),tmp);
	RT_Object::result( tmp );
	ret++;
    }

    {  	int start, diff1 = 0, diff2 = 0;
	RT_Vector v; RT_Matrix m;
	if ((start = RT_findString( argv, "-VertexWCoords" )) >= 0) { 
	    if ( RT_getVector( &argv[ start + 1 ], v, diff1) ) 
		if ( RT_getMatrix( &argv[ start +2 ], m, diff2) ) {
		    VertexCoords(m.INVERS() * v);
		    ret++;
		}
	    RT_clearArgv( argv, start, diff1 + diff2 );
	}
    }

    if (wcoordGF) {
	char tmp[100];
	RT_vector2string( wcoordGM * get_VertexCoords() , tmp);
	RT_Object::result( tmp );
	ret++;
    }

    // surface I/O
    if (surfF) {
	RT_Surface s;
	s.set(surfS);
	VertexSurface(s);
	ret++;
    }

    if (surfGF) {
	RT_Object::result( get_VertexSurface().get() );
	ret++;
    }
    
    if (chkscF) {
	checkScene();
	ret++;
    }

    if (lockF) {
	lock();
	ret++;
    }
    if (unlockF) {
	unlock();
	ret++;
    }
    if (lockGF) {
	char tmp[20];
	RT_int2string(get_lock(),tmp);
	RT_Object::result( tmp );
	ret++;
    }			      

    if (idxGF) {
	char tmp[20];
	RT_int2string(get_currentIndex(),tmp);
	RT_Object::result( tmp );
	ret++;
    }			      
    return( ret );
}

// don't forget: the vertex entry is in the current object!

void RT_VertexManipulator::select(RT_VertexEntry *_vc) {
    if (!_vc) return;
    if (vc == _vc) delete vc; // delete if double click, vc = 0 in objectKilled!
    else {
	deselect(1); // no update
	vc = _vc;
	vc->color(VC_SELECTED_COLOR);
	vc->addRelatedObject(this);
    }
    updateFeedback();
}

void RT_VertexManipulator::deselect(int no_update) {
    if (vc) {
	vc->color(VC_NORMAL_COLOR);
	vc->removeRelatedObject(this);
	vc=0;
	if (!no_update) updateFeedback();
    }
}

void RT_VertexManipulator::objectKilled(RT_Object *o) {
    if (o == vc) vc = 0;
    if (o == last_camera) last_camera = 0;
    if (o == last_scene) last_scene = 0;
    RT_Manipulator::objectKilled(o);
}

void RT_VertexManipulator::updateFeedback() {
    RT_Camera *r;
    if (r = get_camera()) {
	if (r != last_camera) {
	    if (last_camera) {
		last_camera->feedback->remove(pva);
		last_camera->refresh();
		last_camera->removeRelatedObject(this);
	    }
	    last_camera = r;
	    r->feedback->insert(pva);
	    // all input deviced are related with the father camera!
	    r->addRelatedObject(this);
	}
	RT_Scene *s;
	if (s = last_camera->get_scene()) {
	    if (s != last_scene) {
		if (last_scene)
		    last_scene->removeRelatedObject(this);
		last_scene = s;
		pva->checkScene(s);
		s->addRelatedObject(this);
	    }
	    pva->sync();
	    last_scene->update();
	    // the object can move her verticies in the create method
	    pva->sync(); // resync
	    callVertexCBs();
	}
    }
    RT_Manipulator::updateFeedback();
}


#define PICK_DISTANCE 0.1
int RT_VertexManipulator::pickVertex(const RT_Primitive *p, const RT_Ray &ray) {
    int ret = -1, n = get_number_of_vertices(p);
    if (n <= 0) return -1;

    double mdist = DBL_MAX;
    RT_Ray mray;
    mray.pt = p->wc2mc( ray.pt );
    mray.dir = (p->wc2mc( ray.pt + ray.dir ) - mray.pt).UNITIZE();

    for (int i=0; i<n; i++) {
	RT_Vertex *v = getVertex(p,i);
	if (!v) continue;
	double d = ((v->getPoint() - mray.pt).CROSS(mray.dir)).ABS();
	if (d <= PICK_DISTANCE && d < mdist) {
	    ret = i; mdist = d;
	}
    }
    return ret;
}

/***********************************
the magic hot keys:
 ctrl: move the selected vertex
 ctrl+alt: move all picked verticies
*************************************/

void RT_VertexManipulator::event(RT_Event *ev) {
    if (!ev || !get_camera()) {
	RT_Manipulator::event(ev); // report the event(?) to the father	
	return; // nothing more to do
    }
    // save the current object if you want to pick a fairing object vertex(eg. splines)
    RT_Primitive *old_cob = get_currentObject();
    
    // check the control key state
    int ctrl = (ev->isA(RTN_BUTTON_EVENT) && ((RT_ButtonEvent *)ev)->ctrl || ev->isA(RTN_MOTION_EVENT) && ((RT_MotionEvent *)ev)->ctrl);
    // check the shift key state
    int shift = (ev->isA(RTN_BUTTON_EVENT) && ((RT_ButtonEvent *)ev)->shift || ev->isA(RTN_MOTION_EVENT) && ((RT_MotionEvent *)ev)->shift);    

    // if the control key is pressed, report only left button events without shift to the father
    if (!ctrl || ev->isA(RTN_BUTTON_EVENT) && ((RT_ButtonEvent *)ev)->left && !((RT_ButtonEvent *)ev)->middle && !((RT_ButtonEvent *)ev)->right && !shift)
	RT_Manipulator::event(ev);

    // get the new current object
    RT_Primitive *cob = get_currentObject();

    // assume(not set!) the old current object as current object
    // so you can pick fairing object verticies
    if (!cob && old_cob) cob = old_cob;
    
    if ( ev->isA(RTN_BUTTON_EVENT) && cob && cob == old_cob) {
	// process the left button pick without checking the control key
	RT_ButtonEvent &bev = *(RT_ButtonEvent *)ev;
	if (bev.left && !bev.middle && !bev.right) {
	    // some stuff from RT_Manipulator::event
	    RT_Ray r;
	    oldx =  bev.x/((double) bev.w);
	    oldy =  bev.y/((double) bev.h);
	    get_camera()->buildRay(oldx, oldy, r);
	    // find the index of the picked vertex
	    int idx = pickVertex(cob,r);
	    if (idx >= 0) {
		// try the add a new vertex Entry
		// checks existing if the vertex Entry automatic
		pva->addVertexEntry(cob,idx);
		// select these vertex Entry
		select(pva->getEntry(cob,idx));
		// the vertex entry must be in the current object
		currentObject(cob);
	    }
	}
    }
    if (ev->isA( RTN_MOTION_EVENT ) && (vc || ((RT_MotionEvent *)ev)->alt)) {
	RT_MotionEvent &mev = *(RT_MotionEvent *)ev;
	// some stuff from RT_Manipulator::event
	double dx,dy,x,y;
	x = mev.x/((double) mev.w );
	y = mev.y/((double) mev.h );
	dx = (oldx-x);
	dy = (oldy-y);
	oldx = x;
	oldy = y;
	if (mev.ctrl && !mev.left && !mev.middle && mev.right) {
	    RT_Vector v;
	    if (mev.shift) checkAxis( v,0.0,0.0,dy,get_axis());
	    else checkAxis( v, -dx, -dy, 0.0, get_axis());
	    v = v * RTD_MANIPULATOR_M_TRANS_SENS*get_sensibility();
	    if (mev.alt)
		// move all picked verticies
		pva->doFixedCoordTrans(get_currentObject(),v);
	    else {
		// move the selected vertex only
		RT_fixedCoordTrans(vc,v);
		((RT_VertexEntry *)vc)->changed();
	    }
	    updateFeedback();
	}
    }
}

int RT_VertexManipulator::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String} {Creates a {ARG 1 Manipulator} object. The Manipulator device has to be coupled to a camera object.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if (argc != 2) {
	    Tcl_AppendResult( ip, argv[0], ": Need exactly one argument.", 0 );
	    return TCL_ERROR;
	}
	new RT_VertexManipulator( argv[1] ); 
	RTM_classReturn;
    }
    return res; 
}

void RT_VertexManipulator::pick(int mode) {
    RT_Primitive *p = get_currentObject();
    if (!p) return;
    switch(mode) {
      case 0:
	pva->removeAllVertexEntrys(p);
	break;
      case 1:
	pva->addAllVertexEntrys(p);
	break;
      case 2:
	pva->invertVertexCollection(p);
    }
    updateFeedback();
}

int RT_VertexManipulator::addVertexTclCB(char *proc) {
    RT_FindTclCallbackFunctoid func( proc );
    cbList.doWithElements( &func );
    if ( func.get() ) return 0;
    RT_Callback *cb = new RT_TclCallback( proc );
    cbList.append( cb );
    cb->setList( &cbList );
    return 1;
}

int RT_VertexManipulator::removeVertexTclCB(char *proc) {
    RT_FindTclCallbackFunctoid func( proc );
    cbList.doWithElements( &func );
    RT_Callback *cb = func.get();
    if (!cb) return 0;
    cbList.remove( cb );
    delete cb;
    return 1;
}

void RT_VertexManipulator::callVertexCBs() {
    class LFunctoid: public RT_GeneralListFunctoid {
      public:
	void exec(RT_GeneralListEntry *e, void *d) {
	    if ( e->isA( RTN_CALLBACK )) ((RT_Callback*)e)->exec( (RT_InputDevice*)d);
	}
    } func;
    cbList.doWithElements( &func, this );
}

void RT_VertexManipulator::VertexCoords(const RT_Vector& c) {
    if (vc) {
	vc->matrix(RT_IDENTITY.TRANSLATION(c));
	vc->changed();
	updateFeedback();
    }
}

RT_Vector RT_VertexManipulator::get_VertexCoords(RT_Vector* wc) {
    RT_Vector ret,wret;
    if (vc) {
	RT_Primitive *p;
	RT_Vertex *v;
	if ((p = vc->get_Primitive()) && (v = getVertex(p,vc->getIdx()))) {
	    ret = v->getPoint();
	    wret = p->mc2wc(ret);
	}
	
    }
    if (wc) *wc=wret;
    return ret;
}
    
RT_Surface RT_VertexManipulator::get_VertexSurface() {
    RT_Primitive *p;
    RT_Vertex *v;
    RT_Surface s;
    if (vc && (p = vc->get_Primitive()))
	if ((v = getVertex(p,vc->getIdx())) && v->getSurface())
	    s = *v->getSurface();
	else
	    s = p->get_surface();    
    return s;
}

void RT_VertexManipulator::VertexSurface(const RT_Surface& s) {
    if (vc) {
	RT_Primitive *p;
	RT_Vertex *v;
	if ((p = vc->get_Primitive()) && (v = getVertex(p,vc->getIdx()))) {
	    v->surface(s);
	    p->geomChanged();
	    updateFeedback();
	}
    }
}
