////////////////////////////////////////////////////////////////////////////////
//  Implemenation of point set based output primitives. All point list editing//
//  functionality is placed in the (abstract) class Polyvertex.               //  
//  The render-methods are usually platform-dependent, thus look in the       //
//  system subdir.                                                            //  
//  LAST EDIT: Wed Mar  8 08:52:48 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "poly.h"
#include "intrsect.h"

const char *RTN_POLYVERTEX = "Polyvertex";

int RT_Polyvertex::vtDelSuV, RT_Polyvertex::vtDelSuF, RT_Polyvertex::getNumF;
int RT_Polyvertex::vtGetSuGV, RT_Polyvertex::vtGetSuGF, RT_Polyvertex::pointGF; 
int RT_Polyvertex::delV, RT_Polyvertex::delF, RT_Polyvertex::pointGV;
int RT_Polyvertex::vtGetNrGF, RT_Polyvertex::vtGetNrGV;
int RT_Polyvertex::vtDelNrV, RT_Polyvertex::vtDelNrF;
RT_Vector RT_Polyvertex::normV; int RT_Polyvertex::normF, RT_Polyvertex::normG;

RT_ParseEntry RT_Polyvertex::table[] = {
    { "-gnormal", RTP_VECTOR, (char*)&normV, &normF, "Set the global {ARG 1 Normal} vector.", RTPS_VECTOR },
    { "-get_gnormal", RTP_NONE, 0, &normG, "Get the global normal vector.", RTPS_NONE },
    { "-get_number", RTP_NONE, 0, &getNumF, "Get the number of vertexes.", RTPS_NONE },

    { "-vtInsert", RTP_SPECIAL, 0, 0, "Insert vertex {ARG 2 Point} at {ARG 1 Position} and specify {ARG 3 Surface} and {ARG 4 Normal}. For surface and normal vector empty strings can be specified.", "Integer Vector Surface Vector" },
    { "-vtDelete", RTP_INTEGER, (char*)&delV, &delF, "Delete vertex {ARG 1 N}.", RTPS_INTEGER },

    { "-vtPoint", RTP_SPECIAL, 0, 0, "Set new {ARG 2 Geometry} to vertex {ARG 1 N}.", "Integer Vector" },
    { "-get_vtPoint", RTP_INTEGER, (char*)&pointGV, &pointGF, "Get point of vertex {ARG 1 N}.", RTPS_INTEGER },

    { "-vtSurface", RTP_SPECIAL, 0, 0, "Set {ARG 2 Surface} for specified vertex {ARG 1 Number}.", "Integer Surface" },
    { "-get_vtSurface", RTP_INTEGER, (char*)&vtGetSuGV, &vtGetSuGF, "Get Surface of specified vertex {ARG 1 Number}. If there isnt a surface at the vertex return the object default surface.", RTPS_INTEGER },
    { "-vtDeleteSurface", RTP_INTEGER, (char*)&vtDelSuV, &vtDelSuF, "Clear surface at {ARG 1 Position}.", RTPS_INTEGER },

    { "-vtNormal", RTP_SPECIAL, 0, 0, "Set {ARG 2 Normal} for specified vertex {ARG 1 Number}.", "Integer Vector" },
    { "-get_vtNormal", RTP_INTEGER, (char*)&vtGetNrGV, &vtGetNrGF, "Get normal of specified vertex {ARG 1 Number}. If there isnt a normal at the vertex return the object default normal.", RTPS_INTEGER },
    { "-vtDeleteNormal", RTP_INTEGER, (char*)&vtDelNrV, &vtDelNrF, "Clear normal at {ARG 1 Position}.", RTPS_INTEGER },

    { 0, RTP_END, 0, 0, 0, 0 }
};

void RT_Polyvertex::print(FILE *f) const {
    RT_Primitive::print( f ); 
    fprintf( f, "%s -gnormal ", get_name() );
    get_gnormal().print( f );
    fprintf( f, "\n" );
}

RT_Bounds RT_Polyvertex::get_bounds() {
    RT_Bounds b;
    RT_GeneralListElem *tmp;
    tmp = root;
    int i = 0;
    if (tmp) {
	do {
	    const RT_Vector &v = ((RT_Vertex*)tmp->elem)->getPoint();
	    if (i) {
		if (v.x < b.min.x) b.min.x = v.x;
		if (v.x > b.max.x) b.max.x = v.x;
		if (v.y < b.min.y) b.min.y = v.y;
		if (v.y > b.max.y) b.max.y = v.y;
		if (v.z < b.min.z) b.min.z = v.z;
		if (v.z > b.max.z) b.max.z = v.z;
	    }
	    else {
		b.max.x = b.min.x = v.x;
		b.max.y = b.min.y = v.y;
		b.max.z = b.min.z = v.z;
		i++;
	    }
	}
	while ( tmp = tmp->next );
    }
    return b;
} 

void RT_Polyvertex::printCon(FILE *f) const {
    RT_Object::printCon( f );
    fprintf( f, "%i { ", get_number() );
    RT_GeneralListElem *tmp;
    // save the points:
    tmp = root;
    if (tmp) {
	do ((RT_Vertex*)tmp->elem)->getPoint().print( f );
	while ( tmp = tmp->next );
    }
    fprintf( f, "} {" );

    // save the surfaces:
    tmp = root;
    if (tmp) {
	do {
	    const RT_Surface *su = ((RT_Vertex*)tmp->elem)->getSurface();
	    if (su) su->print( f );
	    else fprintf( f, "{} " );
	}
	while ( tmp = tmp->next );
    }
    fprintf( f, "} {" );

    // save the normals:
    tmp = root;
    if (tmp) {
	do {
	    const RT_Vector *n = ((RT_Vertex*)tmp->elem)->getNormal();
	    if (n) n->print( f );
	    else fprintf( f, "{} " );
	}
	while ( tmp = tmp->next );
    }
    fprintf( f, "} " );
}

int RT_Polyvertex::objectCMD(char *argv[])  { 
    int ret = RT_Primitive::objectCMD( argv );
    RT_parseTable( argv, table );
    if (normF) { gnormal( normV ); ret++; }

    if (normG) {
	static char tmp[30]; ret++;
	RT_vector2string( get_gnormal(), tmp );
	RT_Object::result( tmp );
    }

    if (getNumF) {
	static char tmp[10]; ret++;
	RT_int2string( get_number(), tmp );
	RT_Object::result( tmp );
    }

    { 	int start, diff1 = 0, diff2 = 0, diff3 = 0, diff4 = 0;
	int pos;
	RT_Vector v;
	RT_Surface s = RT_NULL_SURFACE;
	RT_Vector n = RT_NULL_NORMAL;
	if ((start = RT_findString( argv, "-vtInsert" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], pos, diff1) ) 
		if ( RT_getVector( &argv[ start + 2 ], v, diff2) ) 
		    if ( (argv[ start + 3 ] && !strcmp( argv[ start + 3 ], "" ))
			|| RT_getSurface( &argv[ start + 3 ], s, diff3) )  
			if ( (argv[ start + 4 ] && !strcmp( argv[ start + 4 ], "" )) 
			    || RT_getVector( &argv[ start + 4 ], n, diff4) )  {
			    vtInsert( pos, v, s == RT_NULL_SURFACE ? 0 : &s, n == RT_NULL_NORMAL ? 0 : &n); 
			    ret++;
			}
	RT_clearArgv( argv, start, diff1 + diff2 + diff3 + diff4 );
    }

    if (pointGF) {
	static char tmp[60]; ret++;
	RT_vector2string( get_vtPoint(pointGV), tmp );
	RT_Object::result( tmp );
    }

    if (vtGetSuGF) {
	static char tmp[200]; ret++;
	RT_surface2string( get_vtSurface(vtGetSuGV), tmp );
	RT_Object::result( tmp );
    }

    if (vtGetNrGF) {
	static char tmp[100]; ret++;
	RT_vector2string( get_vtNormal(vtGetNrGV), tmp );
	RT_Object::result( tmp );
    }

    if (vtDelSuF) { vtDeleteSurface( vtDelSuV ); ret++; }

    if (vtDelNrF) { vtDeleteNormal( vtDelNrV ); ret++; }

    if (delF) { vtDelete( delV ); ret++; }

    { 	int start, diff1 = 0, diff2 = 0;
	int pos; RT_Vector v;
	if ((start = RT_findString( argv, "-vtPoint" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], pos, diff1) ) 
		if ( RT_getVector( &argv[ start +2 ], v, diff2) ) { vtPoint( pos, v ); ret++; }
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int pos; RT_Surface s;
	if ((start = RT_findString( argv, "-vtSurface" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], pos, diff1) ) 
		if ( RT_getSurface( &argv[ start +2 ], s, diff2) ) { vtSurface( pos, s ); ret++; }
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int pos; RT_Vector n;
	if ((start = RT_findString( argv, "-vtNormal" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], pos, diff1) ) 
		if ( RT_getVector( &argv[ start +2 ], n, diff2) ) { vtNormal( pos, n ); ret++; }
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    return ret;
}

int RT_Polyvertex::copy(RT_Primitive *p) const {
    if (p->isA( RTN_POLYVERTEX)) {
	RT_Polyvertex *v = (RT_Polyvertex*)p;
	v->removeAll();
	v->gnormal( get_gnormal() );
	class LFunctoid: public RT_GeneralListFunctoid {
	    RT_Polyvertex *v;
	  public:
	    LFunctoid(RT_Polyvertex *_v): v(_v) {}
	    void exec(RT_GeneralListEntry *e, void* = 0) {
		if (e->isA( RTN_VERTEX )) v->append( new RT_Vertex((RT_Vertex*)e));
	    }
	} lfunc(v);
	doWithElements( &lfunc );
	return( 1 );
    }
    return( RT_Primitive::copy(p) );
}

RT_Vertex *RT_Polyvertex::get(int i) const {
    // get the i th entry from the list (i >= 0)
    RT_Vertex *v = 0;
    RT_GeneralListElem *tmp = root;
    if (!tmp) return( 0 );
    do if (!i--) {
	v = (RT_Vertex*)tmp->elem;
	break;
    }
    while ( tmp = tmp->next );
    return( v );
}

RT_Polyvertex::RT_Polyvertex( char *_name, int n, RT_Vector *v, RT_Surface *sf, RT_Vector *nv): RT_Primitive( _name ) { 
    RT_Surface *ts; RT_Vector *tn;
    for (int i = 0; i < n; i++) { 
	if (sf) {
	    ts = *sf == RT_NULL_SURFACE ? 0 : sf;
	    sf++;
	}
	else ts = 0;
	if (nv) {
	    tn = (*nv == RT_NULL_NORMAL) ? 0 : nv;
	    nv++;
	}
	else tn = 0;
	add( *v++, ts, tn );
    }

    gnormal( RT_Vector(0,0,1) );
}

void RT_Polyvertex::vtSurface(int i, const RT_Surface &s) {
    RT_Vertex *vr;
    if (!(vr = get( i ))) return;
    vr->surface( s );
    geomChanged();
}

void RT_Polyvertex::vtPoint(int i, const RT_Vector &p) {
    RT_Vertex *vr;
    if (!(vr = get( i ))) return;
    vr->point( p );
    geomChanged();
}

void RT_Polyvertex::vtNormal(int i, const RT_Vector &p) {
    RT_Vertex *vr;
    if (!(vr = get( i ))) return;
    vr->normal( p );
    geomChanged();
}

int RT_Polyvertex::get_number() const {
    int i = 0;
    RT_GeneralListElem *tmp = root;
    if (!tmp) return i;
    do 	i++;
    while(tmp = tmp->next);
    return i;
}

void RT_Polyvertex::vtDelete(int i) {
    RT_Vertex *vr;
    if (!(vr = get( i ))) return;
    remove( vr );
    geomChanged();
}

void RT_Polyvertex::vtDeleteSurface(int i) {
    RT_Vertex *vr;
    if (!(vr = get( i ))) return;
    vr->deleteSurface();
    geomChanged();
}

void RT_Polyvertex::vtDeleteNormal(int i) {
    RT_Vertex *vr;
    if (!(vr = get( i ))) return;
    vr->deleteNormal();
    geomChanged();
}

void RT_Polyvertex::vtInsert(int i, const RT_Vector &v, const RT_Surface *s, const RT_Vector *n) {
    if (!root || !i) appendAtBegin( new RT_Vertex(v, s, n));
    else {
	RT_GeneralListElem *tmp = root;
	do if (!i) break; else i--;
	while ( tmp = tmp->next );
	if (!i && tmp) {
	    RT_GeneralListElem *el = new RT_GeneralListElem;
	    // insert the new element:
	    *el = *tmp;
	    el->prev->next = el;
	    el->elem = new RT_Vertex(v, s, n);
	    el->next = tmp;
	    tmp->prev = el;
	}
	else append( new RT_Vertex(v, s, n));
    }
    geomChanged();
}

const RT_Vector &RT_Polyvertex::get_vtPoint(int i) const {
    RT_Vertex *vr;
    if (!(vr = get( i ))) rt_Output->fatalVar( "Access to a point of ", get_name(), ", that does not exist.", 0);
    return vr->getPoint();
}

const RT_Surface &RT_Polyvertex::get_vtSurface(int i) const {
    RT_Vertex *vr;
    if (!(vr = get( i )) || !vr->getSurface()) return get_surface();
    return *vr->getSurface();
}

const RT_Vector &RT_Polyvertex::get_vtNormal(int i) const {
    RT_Vertex *vr;
    if (!(vr = get( i )) || !vr->getNormal()) return get_gnormal();
    return *vr->getNormal();
}

RT_Polyvertex::~RT_Polyvertex() {
    RT_GeneralListElem *tmp = root;
    if (!tmp) return;
    do delete tmp->elem;
    while(tmp = tmp->next);
}

int RT_Polyvertex::createPoly(Tcl_Interp *ip, int argc, char **argv, int &nr, RT_Vector *&vecs, RT_Surface *&surfs, RT_Vector *&norms) {
    vecs = 0; surfs = 0; norms = 0;
    if (argc != 6) {
	Tcl_AppendResult( ip, "Bad number of arguments. Try \"", argv[0], " ?\".", 0 ); return TCL_ERROR; 
    }
    if ( !RT_string2int( argv[2], nr)) {
	Tcl_AppendResult( ip, "Arg 2 should be an integer.", 0 ); return TCL_ERROR; 
    }
    if (nr < 0) { Tcl_AppendResult( ip, "Dimension should be a non-negative number!.", 0 ); return TCL_ERROR; }
    
    if (nr > 0) {
	vecs = new RT_Vector[nr];
	// initialize points:
	for (int i = 0; i < nr; i++) vecs[i] = RT_NULL_NORMAL; 
	
	int xargc; char **xargv; int dum, cnt;
	if (Tcl_SplitList( ip, argv[3], &xargc, &xargv) == TCL_OK) {
	    if (xargc < nr) rt_Output->warningVar( argv[0], ": not enough vertices specified. Using defaults!", 0);
	    cnt = xargc < nr ? xargc : nr;
	    for ( i = 0; i < cnt; i++) 
		if (!RT_getVector( &xargv[i], vecs[i], dum)) 
		    rt_Output->warningVar( argv[0], ": found a bad point value. Using default value!", 0);
	    free((char*)xargv);
	    
	    if (Tcl_SplitList( ip, argv[4], &xargc, &xargv) == TCL_OK) {
		surfs = new RT_Surface[nr];	
		// initialize surfaces:
		for ( i = 0; i < nr; i++) surfs[i] = RT_NULL_SURFACE;
		if (xargc < nr && xargc) rt_Output->warningVar( argv[0], ": not enough surfaces specified. Using defaults!", 0);
		cnt = xargc < nr ? xargc : nr;
		for ( i = 0; i < cnt; i++) 
		    if (strlen( xargv[i]) && !RT_getSurface( &xargv[i], surfs[i], dum)) 
			rt_Output->warningVar( argv[0], ": found a bad surface value. Using default value!", 0);
		free((char*)xargv);
	    }
	
	    if (Tcl_SplitList( ip, argv[5], &xargc, &xargv) == TCL_OK) {
		norms = new RT_Vector[nr];
		// initialize normals:
		for ( i = 0; i < nr; i++) norms[i] = RT_NULL_NORMAL;
		if (xargc < nr && xargc) rt_Output->warningVar( argv[0], ": not enough normals specified. Using defaults!", 0);
		cnt = xargc < nr ? xargc : nr;
		for ( i = 0; i < cnt; i++) if (strlen( xargv[i]) && !RT_getVector( &xargv[i], norms[i], dum)) 
		    rt_Output->warningVar( argv[0], ": found a bad normal value. Using default value!", 0);
		free((char*)xargv);
	    }
	}
    }
    return TCL_OK;
}

// class Polymarker:

const char *RTN_POLYMARKER = "Polymarker";

int RT_Polymarker::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " { String Integer Vector_List Surface_List Vector_List} {  Creates a new polymarker. {ARG 1 Name} of the new object, {ARG 2 Number} of initial elements, list of {ARG 3 Points}, list of {ARG 4 Surfaces}, list of {ARG 5 Normals}. The surface and normals list can be empty.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	int ret = TCL_ERROR; int nr;
	RT_Vector  *vecs; RT_Surface *surfs; RT_Vector *norms;
	if ( createPoly( ip, argc, argv, nr, vecs, surfs, norms) == TCL_OK ) {
	    new RT_Polymarker( argv[1], nr, vecs, surfs, norms );
	    RTM_classReturn;
	}	    
	else rt_Output->errorVar( argv[0], ": Could not create a polymarker!", 0);
	if (vecs) delete [] vecs; if (surfs) delete [] surfs; if (norms) delete [] norms;
	return ret;
    }
    return res;
}

// class Polyline:

const char *RTN_POLYLINE = "Polyline";

int RT_Polyline::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " { String Integer Vector_List Surface_List Vector_List} {  Creates a new polyline. {ARG 1 Name} of the new object, {ARG 2 Number} of initial elements, list of {ARG 3 Points}, list of {ARG 4 Surfaces}, list of {ARG 5 Normals}. The surface and normals list can be empty.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	int ret = TCL_ERROR; int nr;
	RT_Vector  *vecs; RT_Surface *surfs; RT_Vector *norms;
	if ( createPoly( ip, argc, argv, nr, vecs, surfs, norms) == TCL_OK ) {
	    new RT_Polyline( argv[1], nr, vecs, surfs, norms );
	    RTM_classReturn;
	}	    
	else rt_Output->errorVar( argv[0], ": Could not create a polyline!", 0);
	if (vecs) delete [] vecs; if (surfs) delete [] surfs; if (norms) delete [] norms;
	return ret;
    }
    return res;
}

// the ray-tracing interface of polyline ignores vertex surfaces and normals!

#define LineWidth ((double)0.01)
int RT_Polyline::intersect(const RT_Ray &ray, RT_InterSectionList &il) {
    int r = 0;
    RT_Ray mray;
    RTM_wcRay2mcRay(ray,mray);
    int segs = get_number() - 1;    
    for(int s=0; s<segs; s++) {	
	RT_Vector x0 = get(s)->getPoint();
	RT_Vector v = get(s+1)->getPoint() - x0;
	double abs_v = v.ABS();
	if (abs_v < rt_RayEps) continue;
	v = v * (1.0 / abs_v);
	RT_Vector a = (mray.pt - x0).CROSS(v);
	RT_Vector b = mray.dir.CROSS(v);
	double aa = a.DOT(a), bb = b.DOT(b);
	double _p_2 = a.DOT(b) / -bb;
	double q = (aa - (LineWidth * LineWidth)) / bb;
	double tmp = _p_2 * _p_2 - q;
	if (tmp < 0.0) continue;
	double m = _p_2 - sqrt(tmp);
	if (m < 0.0) continue;
	a = mray.pt + mray.dir * m;
	double t = v.DOT(a - x0);
	if (t < 0.0 || t > abs_v) {
	    m = _p_2 + sqrt(tmp);
	    if (m < 0.0) continue;
	    a = mray.pt + mray.dir * m;
	    t = v.DOT(a - x0);
	    if (t < 0.0 || t > abs_v) continue;
	}
	double wt;
	RTM_mcT2wcT(ray,mray,wt,m);
	if (ray.valid(wt)) {
	    il.add(new RT_InterSection(this,wt,1,&get_surface()));
	    r++;
	}
    }
    return r;
}

void RT_Polyline::normal(const RT_Vector &wc, RT_Vector &n) {
    RTM_mcN2wcN( wc, get_gnormal(), n);
}

RT_Bounds RT_Polyline::get_bounds() {
    RT_Bounds b = RT_Polyvertex::get_bounds();
    b.max.x += LineWidth;
    b.max.y += LineWidth;
    b.max.z += LineWidth;
    b.min.x -= LineWidth;
    b.min.y -= LineWidth;
    b.min.z -= LineWidth;
    return b;
}

// class Polygon:

const char *RTN_POLYGON = "Polygon";

RT_Polygon::RT_Polygon( char *a, int b, RT_Vector *c, RT_Surface *d, RT_Vector *e):
RT_Polyvertex( a, b, c, d, e) {
    polygon = 0;
    poly_segs = 0;
    xfillstyle = 0;
    poly_valid = 0;
    gnormal( RT_NULL_NORMAL );
    // create() should be called by update
}

RT_Polygon::~RT_Polygon() {
    if (polygon) delete polygon;
}

int RT_Polygon::intersect(const RT_Ray &ray, RT_InterSectionList &il) {
    if (!poly_valid) return 0;
    RT_Ray mray;
    // transform the ray into model coordinats
    RTM_wcRay2mcRay(ray,mray);
    // the distance of ray from the plane
    double t = _normal.DOT(mray.dir);
    if (fabs(t) < rt_RayEps) return 0;
    t = (dist - _normal.DOT(mray.pt)) / t;
    // the intersection point with the plane
    if (t < 0.0) return 0;
    RT_Vector p = mray.pt + mray.dir * t;
    RT_GeneralListElem *tmp = root;
    int s = 0, i = 0; // number of intersections
    do {
	const RT_Vector *p1 = &((RT_Vertex *)tmp->elem)->getPoint();
        i += ifunc(p,*p1,polygon[s]);
	tmp = tmp->next;
	s++;
    } while (s < poly_segs);
    if (i & 1) {
	double wt;
	RTM_mcT2wcT(ray,mray,wt,t); // macro compute p again !
	if (ray.valid(wt)) {
	    RT_Surface surf;
	    RT_Color zero_color(0,0,0);
	    surf.ambi = zero_color; surf.diff = zero_color;
	    surf.spec = zero_color; surf.emmi = zero_color;
	    surf.tran = surf.refr = surf.shin = 0;
	    tmp = root;
	    double sum = 0;
	    while (tmp) {
		const RT_Surface *_surf = ((RT_Vertex *)tmp->elem)->getSurface();
		if (!_surf) _surf = &get_surface();
		double a = (((RT_Vertex *)tmp->elem)->getPoint() - p).ABS();
		if (fabs(a) < rt_RayEps) {
		    surf = *_surf;
		    sum = 1;
		    break;
		}
		a = 1.0 / (a * a);
		sum += a;
		surf.ambi = surf.ambi + _surf->ambi * a;
		surf.diff = surf.diff + _surf->diff * a;
		surf.spec = surf.spec + _surf->spec * a;
		surf.emmi = surf.emmi + _surf->emmi * a;
		surf.tran += _surf->tran * a;
		surf.refr += _surf->refr * a;
		surf.shin += _surf->shin * a;		
		tmp = tmp->next;
	    }
	    sum = 1.0 / sum;
	    surf.ambi = surf.ambi * sum;
	    surf.diff = surf.diff * sum;
	    surf.spec = surf.spec * sum;
	    surf.emmi = surf.emmi * sum;
	    surf.tran *= sum;
	    surf.refr *= sum;
	    surf.shin *= sum;
	    il.add(new RT_InterSection(this,wt,_normal.DOT(mray.dir)<0,&surf));
	    return 1;
	}
    }
    return 0;
}

void RT_Polygon::normal(const RT_Vector &R, RT_Vector &N) {
    RT_Vector *used_normal = (use_xnormal) ? &xnormal : &_normal;
    if (!poly_valid) {
	RTM_mcN2wcN(R,*used_normal,N); // normal vector should be unitized
	return;
    }
    RT_Vector r = wc2mc(R);
    RT_GeneralListElem *tmp = root;
    RT_Vector n;
    do {
	RT_Vector const *_n = ((RT_Vertex *)tmp->elem)->getNormal();
	if (!_n) _n = used_normal;
	double a = (((RT_Vertex *)tmp->elem)->getPoint() - r).ABS();
	if (fabs(a) < rt_RayEps) {
	    n = *_n;
	    break;
	} else 
	    n = n + *_n * (1.0 / (a * a));
    } while (tmp = tmp->next);
    RTM_mcN2wcN(R,n,N); // normal vector should be unitized
}

void RT_Polygon::create () {
    _normal = RT_NULL_NORMAL; 
    if (polygon) delete [] polygon; polygon = 0;
    poly_valid = 0;
    poly_segs = get_number();
    if (poly_segs < 3) {
	rt_Output->errorVar(get_name(),": not enough vertices for creating polygon.",0);
	poly_segs = 0;
	return;
    }
    // allocte some data space:
    polygon = new RT_Vector [poly_segs];
    RT_Vector *v = new RT_Vector[poly_segs]; // only temporary
    v[0] = get(0)->getPoint();
    int i = 1;
    while (i < poly_segs) {
	polygon[i-1] = (v[i] = get(i)->getPoint()) - v[i-1];
	i++;
    }
    polygon[poly_segs-1] = v[0] - v[poly_segs-1];
    RT_calcNorm(poly_segs,v,_normal);
    delete [] v;
    double nabs = _normal.ABS(); 
    if (nabs == 0.0) {
	rt_Output->warningVar(get_name(),": normal vector of polygon is zero.",0);
	delete [] polygon; polygon = 0; poly_segs = 0;// cleanup
	return;
    }
    // use a point in the plane to compute distance from 0
    dist = _normal.DOT(get(0)->getPoint()) / nabs ;
    _normal = _normal * (1 / nabs);
    ifunc = &RT_intersect_yz;
    if (fabs(_normal.y) > fabs(_normal.x))
	ifunc = &RT_intersect_xz;
    if (fabs(_normal.z) > fabs(_normal.x) && fabs(_normal.z) > fabs(_normal.y))
	ifunc = &RT_intersect_xy;
    poly_valid = 1;
}

int RT_Polygon::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Integer Vector_List Surface_List Vector_List} { Creates a new polygon. {ARG 1 Name} of the new object, {ARG 2 Number} of initial elements, list of {ARG 3 Points}, list of {ARG 4 Surfaces}, list of {ARG 5 Normals}. The surface and normals list can be empty.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	int ret = TCL_ERROR; int nr;
	RT_Vector  *vecs; RT_Surface *surfs; RT_Vector *norms;
	if ( createPoly( ip, argc, argv, nr, vecs, surfs, norms) == TCL_OK ) {
	    new RT_Polygon( argv[1], nr, vecs, surfs, norms );
	    RTM_classReturn;
	}	    
	else rt_Output->errorVar( argv[0], ": Could not create a polygon!.", 0);
	if (vecs) delete [] vecs; if (surfs) delete [] surfs; if (norms) delete [] norms;
	return ret;
    }
    return res;
}

