////////////////////////////////////////////////////////////////////////////////
//  Quadmesh implementation.                                                  //  
//  LAST EDIT: Wed Mar  8 16:47:46 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "quadmesh.h"

RT_IntersectionCache::~RT_IntersectionCache() {
    RT_IntersectionCacheEntry *e = root;
    while (e) {
	RT_IntersectionCacheEntry *t = e->next;
	delete e;
	e = t;
    }
}

void RT_IntersectionCache::add(const RT_IntersectionCacheEntry& e) {
    RT_IntersectionCacheEntry *t;
    if (!pos)
	if (!root)
	    root = pos = new RT_IntersectionCacheEntry(e);
	else {
	    pos = root;
	    t = pos->next;
	    *pos = e;
	    pos->next = t;
	}
    else
	if (!pos->next) {
	    pos->next = new RT_IntersectionCacheEntry(e);
	    pos = pos->next;
	} else {
	    pos = pos->next;
	    t = pos->next;
	    *pos = e;
	    pos->next = t;
	}
    end = pos->next == 0;
}

const RT_IntersectionCacheEntry *RT_IntersectionCache::next() {
    if (!pos) pos = root;
    else pos = pos->next;
    end = pos->next == 0;
    return pos;
}

const char *RTN_QUADMESH = "Quadmesh";

int RT_Quadmesh::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " { String Integer Integer Vector_List Surface_List Vector_List} {  Creates a new quadmesh. Arguments arre: the {ARG 1 Name} of the new object, number of {ARG 2 Rows}, number of {ARG 3 Columns}, list of {ARG 4 Points}, list of {ARG 5 Surfaces} and the list of {ARG 6 Normals}.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	if (argc != 7) { Tcl_AppendResult( ip, "Bad number of arguments. Try \"", RTN_QUADMESH, " ?\".", 0 ); return TCL_ERROR; }
	int x; 
	if ( !RT_string2int( argv[2], x)) { Tcl_AppendResult( ip, "Arg 2 should be an integer.", 0 ); return TCL_ERROR; }
	if ( x <= 0 ) { Tcl_AppendResult( ip, "X dimension should be a positive number!.", 0 ); return TCL_ERROR; }
	int y;
	if ( !RT_string2int( argv[3], y)) { Tcl_AppendResult( ip, "Arg 3 should be an integer.", 0 ); return TCL_ERROR; }
	if ( y <= 0 ) { Tcl_AppendResult( ip, "Y dimension should be a positive number!.", 0 ); return TCL_ERROR; }
	
	int nr = x * y;
	RT_Vector  *vecs = new RT_Vector[nr];
	RT_Surface *surfs = 0;
	RT_Vector *norms = 0;
	
	// initialize points
	for (int i = 0; i < nr; i++) vecs[i] = RT_NULL_NORMAL; 
	
	int xargc; char **xargv; int dum, cnt;
	int ret = TCL_ERROR;
	if (Tcl_SplitList( ip, argv[4], &xargc, &xargv) == TCL_OK) {
	    if (xargc < nr) rt_Output->warningVar( argv[0], ": not enough vertices specified. Using defaults!", 0);
	    cnt = xargc < nr ? xargc : nr;
	    for ( i = 0; i < cnt; i++) 
		if (!RT_getVector( &xargv[i], vecs[i], dum)) 
		    rt_Output->warningVar( argv[0], ": found a bad point value. Using default value!", 0);
	    free((char*)xargv);

	    if (Tcl_SplitList( ip, argv[5], &xargc, &xargv) == TCL_OK) {
		surfs = new RT_Surface[nr];	
		// initialize surfaces:
		for ( i = 0; i < nr; i++) surfs[i] = RT_NULL_SURFACE;
		if (xargc < nr && xargc) rt_Output->warningVar( argv[0], ": not enough surfaces specified. Using defaults!", 0);
		cnt = xargc < nr ? xargc : nr;
		for ( i = 0; i < cnt; i++) 
		    if (strlen( xargv[i]) && !RT_getSurface( &xargv[i], surfs[i], dum)) 
			rt_Output->warningVar( argv[0], ": found a bad surface value. Using default value!", 0);
		free((char*)xargv);
	    }
	    
	    if (Tcl_SplitList( ip, argv[6], &xargc, &xargv) == TCL_OK) {
		norms = new RT_Vector[nr];
		// initialize normals:
		for ( i = 0; i < nr; i++) norms[i] = RT_NULL_NORMAL;
		if (xargc < nr && xargc) rt_Output->warningVar( argv[0], ": not enough normals specified. Using defaults!", 0);
		cnt = xargc < nr ? xargc : nr;
		for ( i = 0; i < cnt; i++) if (strlen( xargv[i]) && !RT_getVector( &xargv[i], norms[i], dum)) 
		    rt_Output->warningVar( argv[0], ": found a bad normal value. Using default value!", 0);
		free((char*)xargv);
	    }
	    new RT_Quadmesh( argv[1], x, y, vecs, surfs, norms );
	    RTM_classReturn;
	}
	else rt_Output->errorVar( argv[0], ": Could not create a quadmesh! No vertices found.", 0);
	delete [] vecs; if (surfs) delete [] surfs; if (norms) delete [] norms;
	return ret;
    }
    return res;
}

RT_Quadmesh::RT_Quadmesh( char *a, int _x, int _y, const RT_Vector *c, const RT_Surface *d, const RT_Vector *e) :  RT_Primitive( a), xnormal(RT_NULL_NORMAL) {
    xfillstyle = 0;
    // _x and _y should be non-negative!!!
    x = _x; y = _y;
    int p = x*y; 
    if ( x && y ) vertices = new RT_Vertex[ p ];
    else vertices = 0;
    if (!vertices) return;
    RT_Vertex *tmp = vertices;
    for (int i = 0; i < p; i++) {
	if (c) { tmp->point( *c ); c++; }
	if (d) { if (*d == RT_NULL_SURFACE) ; else tmp->surface( *d ); d++; }
	if (e) { if (*e == RT_NULL_NORMAL) ; else tmp->normal( *e ); e++; }
	tmp++;
    }
}

void RT_Quadmesh::newGeometry(int _x, int _y) {
    x = _x; y = _y;
    if (vertices) delete [] vertices;
    if ( x && y ) vertices = new RT_Vertex[ x * y ];
    else vertices = 0;
}

RT_Vertex *RT_Quadmesh::get( int _x, int _y) const {
    if ( _x <0 || _x >= x || _y <0 || _y >= y ) {
	rt_Output->errorVar( get_name(), ": indices out of area!", 0 );
	return 0;
    }
    return &vertices[x*_y +_x];
}

RT_Vertex *RT_Quadmesh::get( int xy) const {
    if ( xy <0 || xy >= (get_x() * get_y())) {
	rt_Output->errorVar( get_name(), ": indices out of area!", 0 );
	return 0;
    }
    return &vertices[xy];
}

int RT_Quadmesh::normF, RT_Quadmesh::normG, RT_Quadmesh::cnF; 
int RT_Quadmesh::getXF, RT_Quadmesh::getYF;
RT_Vector RT_Quadmesh::normV;

RT_ParseEntry RT_Quadmesh::table[] = {
    { "-get_x", RTP_NONE, 0, &getXF, "Get the number of vertexes in x dimension.", RTPS_NONE },
    { "-get_y", RTP_NONE, 0, &getYF, "Get the number of vertexes in y dimension.", RTPS_NONE },

    { "-gnormal", RTP_VECTOR, (char*)&normV, &normF, "Set the global {ARG 1 Normal} vector.", RTPS_VECTOR },
    { "-get_gnormal", RTP_NONE, 0, &normG, "Get the global normal vector.", RTPS_NONE },

    { "-vtPoint", RTP_SPECIAL, 0, 0, "Set new {ARG 3 Geometry} to vertex: {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer Vector" },
    { "-get_vtPoint", RTP_SPECIAL, 0, 0, "Get point of vertex: {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer" },

    { "-vtSurface", RTP_SPECIAL, 0, 0, "Set {ARG 3 Surface} for specified vertex: {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer Surface" },
    { "-get_vtSurface", RTP_SPECIAL, 0, 0, "Get Surface of specified vertex: {ARG 1 Row}, {ARG 2 Column}. If there isnt a surface at the vertex return the object default surface.", "Integer Integer" },
    { "-vtDeleteSurface", RTP_SPECIAL, 0, 0, "Clear surface at {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer" },

    { "-vtNormal", RTP_SPECIAL, 0, 0, "Set {ARG 3 Normal} for specified vertex: {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer Vector" },
    { "-get_vtNormal", RTP_SPECIAL, 0, 0, "Get normal of specified vertex: {ARG 1 Row}, {ARG 2 Column}. If there isnt a normal at the vertex return the object default normal.", "Integer Integer" },
    { "-vtDeleteNormal", RTP_SPECIAL, 0, 0, "Clear normal at {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer" },

    { "-computeNormals", RTP_NONE, 0, &cnF, "For each vertex a normal vector will be computed. Existing normals will be overwritten.", RTPS_NONE },

    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_Quadmesh::objectCMD(char *argv[])  { 
    int ret =  RT_Primitive::objectCMD( argv );
    RT_parseTable( argv, table );

    if (normF) { gnormal( normV ); ret++; }

    if (normG) {
	static char tmp[30]; ret++;
	RT_vector2string( get_gnormal(), tmp );
	RT_Object::result( tmp );
    }

    if (getXF) {
	char tmp[10]; ret++;
	RT_int2string( get_x(), tmp );
	RT_Object::result( tmp );
    }

    if (getYF) {
	char tmp[10]; ret++;
	RT_int2string( get_y(), tmp );
	RT_Object::result( tmp );
    }

    { 	int start, diff1 = 0, diff2 = 0, diff3 = 0;
	int posx, posy; RT_Surface s;
	if ((start = RT_findString( argv, "-vtSurface" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) 
		    if ( RT_getSurface( &argv[ start + 3 ], s, diff3) )	{ 
			vtSurface( posx, posy, s ); ret++; 
		    }
	RT_clearArgv( argv, start, diff1 + diff2 + diff3 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-get_vtSurface" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    static char tmp[250]; ret++;
		    RT_surface2string( get_vtSurface(posx, posy), tmp );
		    RT_Object::result( tmp );
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-vtDeleteSurface" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) 
		    vtDeleteSurface(posx, posy);
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0, diff3 = 0;
	int posx, posy; RT_Vector n;
	if ((start = RT_findString( argv, "-vtNormal" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) 
		    if ( RT_getVector( &argv[ start + 3 ], n, diff3) ) {
			vtNormal( posx, posy, n ); ret++;
		    }
	RT_clearArgv( argv, start, diff1 + diff2 + diff3 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-get_vtNormal" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    char tmp[60]; ret++;
		    RT_vector2string( get_vtNormal(posx, posy), tmp );
		    RT_Object::result( tmp );
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-vtDeleteNormal" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    vtDeleteNormal(posx, posy); ret++;
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0, diff3 = 0;
	int posx, posy; RT_Vector p;
	if ((start = RT_findString( argv, "-vtPoint" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) 
		    if ( RT_getVector( &argv[ start + 3 ], p, diff3) ) {
			vtPoint( posx, posy, p ); ret++;
		    }
	RT_clearArgv( argv, start, diff1 + diff2 + diff3 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-get_vtPoint" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    char tmp[60]; ret++;
		    RT_vector2string( get_vtPoint(posx, posy), tmp );
		    RT_Object::result( tmp );
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    if (cnF) { computeNormals(); ret++; }

    return ret;
}

void RT_Quadmesh::vtPoint(int _x, int _y, const RT_Vector &p) {
    RT_Vertex *vr;
    if ( !(vr = get( _x, _y ))) return;
    vr->point( p );
    geomChanged();
}

const RT_Vector &RT_Quadmesh::get_vtPoint(int _x, int _y) const {
    RT_Vertex *vr;
    if ( !(vr = get( _x, _y ))) return RT_BAD_VECTOR;
    return vr->getPoint();
}

void RT_Quadmesh::vtSurface(int _x, int _y, const RT_Surface &s) {
    RT_Vertex *vr;
    if ( !(vr = get( _x, _y ))) return;
    vr->surface( s );
    geomChanged();
}

const RT_Surface &RT_Quadmesh::get_vtSurface(int _x, int _y ) const {
    RT_Vertex *vr;
    if ( !(vr = get( _x, _y ))) return get_surface();
    return vr->getSurface() ? *vr->getSurface() : get_surface();
}

void RT_Quadmesh::vtDeleteSurface(int _x, int _y) {
    RT_Vertex *vr;
    if ( !(vr = get( _x, _y ))) return;
    vr->deleteSurface();
    geomChanged();
}

void RT_Quadmesh::vtNormal(int _x, int _y, const RT_Vector &p) {
    RT_Vertex *vr;
    if ( !(vr = get( _x, _y ))) return;
    vr->normal( p );
    geomChanged();
}

const RT_Vector &RT_Quadmesh::get_vtNormal(int _x, int _y) const {
    RT_Vertex *vr;
    if ( !(vr = get( _x, _y ))) return get_gnormal();
    return vr->getNormal() ? *vr->getNormal() : get_gnormal();
}

void RT_Quadmesh::vtDeleteNormal( int _x, int _y ) {
    RT_Vertex *vr;
    if ( !(vr = get( _x, _y ))) return;
    vr->deleteNormal();
    geomChanged();
}

void RT_Quadmesh::printCon(FILE *f) const {
    RT_Object::printCon( f );
    fprintf( f, "%i %i { ", get_x(),get_y() );
    int p = x * y, i;
    RT_Vertex *tmp = vertices;
    // save points:
    for ( i = 0; i < p; i++) { tmp->getPoint().print( f ); tmp++; }
    fprintf( f, "} {" );
    tmp = vertices;
    // save the surfaces:
    for ( i = 0; i < p; i++) {
	const RT_Surface *su = tmp->getSurface();
	if (su) su->print( f );
	else fprintf( f, "{} " );
	tmp++;
    }
    fprintf( f, "} {" );
    tmp = vertices;
    // save the normals:
    for ( i = 0; i < p; i++) {
	const RT_Vector *no = tmp->getNormal();
	if (no) no->print( f );
	else fprintf( f, "{} " );
	tmp++;
    }
    fprintf( f, "} " );
}

void RT_Quadmesh::computeNormals() {
    RT_Vector tmp[3];
    RT_Vector norm;

    for ( int j = 0; j < get_y(); j++ )
	for ( int i = 0; i < get_x(); i++ ) {
	    if ( i ) {
		if ( j ) {
		    tmp[0] = get_vtPoint( i, j  - 1 );
		    tmp[1] = get_vtPoint( i , j );
		    tmp[2] = get_vtPoint( i - 1 , j );
		}
		else {
		    tmp[0] = get_vtPoint( i , j );
		    tmp[1] = get_vtPoint( i , j + 1 );
		    tmp[2] = get_vtPoint( i - 1 , j + 1);
		}
	    }
	    else {
		if ( j ) {
		    tmp[0] = get_vtPoint( i + 1 , j - 1 );
		    tmp[1] = get_vtPoint( i + 1 , j );
		    tmp[2] = get_vtPoint( i , j );
		}
		else {
		    tmp[0] = get_vtPoint( i + 1 , j + 1 );
		    tmp[1] = get_vtPoint( i + 1 , j );
		    tmp[2] = get_vtPoint( i , j );
		}
	    }
	    RT_calcNorm( 3, tmp, norm );
	    vtNormal( i, j, norm );
	}
    // invert the first normal vector:
    norm = get_vtNormal( 0, 0 );
    norm = norm * -1;
    vtNormal( 0, 0, norm );

    // correct normals of qmesh is closed:
    int dim = get_x();
    int last = get_y() - 1;
    for ( int k = 0; k < get_x(); k++ ) 
	if (get_vtPoint( k, 0) == get_vtPoint( k, last ))
	    vtNormal( k, last, get_vtNormal( k, 0 ));
    dim = get_y();
    last = get_x() - 1;
    for ( k = 0; k < dim; k++ ) 
	if (get_vtPoint( 0, k) == get_vtPoint( last, k ))
	    vtNormal( last, k, get_vtNormal( 0, k ));
}

RT_Bounds RT_Quadmesh::get_bounds() {
    RT_Bounds b;
    int p = x * y;
    int fl = 0;
    RT_Vertex *tmp = vertices;
    for ( int i = 0; i < p; i++) {
	const RT_Vector &v = tmp->getPoint();
	if (fl) {
	    if (v.x < b.min.x) b.min.x = v.x;
	    if (v.x > b.max.x) b.max.x = v.x;
	    if (v.y < b.min.y) b.min.y = v.y;
	    if (v.y > b.max.y) b.max.y = v.y;
	    if (v.z < b.min.z) b.min.z = v.z;
	    if (v.z > b.max.z) b.max.z = v.z;
	}
	else {
	    b.max.x = b.min.x = v.x;
	    b.max.y = b.min.y = v.y;
	    b.max.z = b.min.z = v.z;
	    fl++;
	}
	tmp++;
    }
    return b;
} 

void RT_Quadmesh::print(FILE *f) const {
    RT_Primitive::print( f ); 
    if (!(get_gnormal() == RT_NULL_NORMAL )) {
	fprintf( f, "%s -gnormal ", get_name() );
	get_gnormal().print( f );
	fprintf( f, "\n" );
    }
}

#ifdef RTD_RSY

void RT_Quadmesh::toRSY(RT_RSYScene* rsc) {
    RT_Vertex *rt_vtx;        //quadmesh vertex
    RT_Vector rt_pnt, rt_nrm; //quadmesh point, normal
    const RT_Surface *sf;           //quadmesh surface attributes
    
    RT_RS_Scene *rscene = rsc->get_rscene(); //the radiosity scene
    RT_RS_DiffVtx* rs_dvtx;   //diffuse vertex
    RT_RS_InsVtx* rs_ivtx;    //non-diffuse vertex
    RT_RS_3DVector rs_pnt, rs_nrm, rs_tgn; //point, normal, tangent
    RT_RS_DiffRGB rs_drefl, rs_irefl,      //diffuse, non-diffuse reflection values
    rs_demi, rs_iemi;        //diffuse, non-diffuse emission values
    int gnorm = !(get_gnormal() == RT_NULL_NORMAL),  //global normal defined?
    diff = 0, ins = 0, brdf_idx;
    long x, y;
    long p_icnt = rscene->Points->get_inscnt();
    long p_dcnt = rscene->Points->get_diffcnt(); //the number of points already inserted
    long xn = get_x(), yn = get_y();                //quadmesh dimensions
    RT_String specfunc, fname;   //BRDF-name
    vertex_type vt;   //vertex-type
    
    //#### the points:
    for (y = 0; y < yn; y++)
	for (x = 0; x < xn; x++) {
	    //get quadmesh vertex:
	    rt_vtx = get((int)x, (int)y);

	    //#### point:
	    rt_pnt = mc2wc(rt_vtx->getPoint());
	    rs_pnt.x = rt_pnt.x; rs_pnt.y = rt_pnt.y; rs_pnt.z = rt_pnt.z;

	    //#### normal:
	    if (rt_vtx->getNormal()) 
		if(gnorm) {  //???
		    rt_Output->errorVar( get_class(), ":Vertex normal AND global normal defined!", NULL);
		    rscene->clear(); return;
		}
		else rt_nrm = *rt_vtx->getNormal();
	    else if (!gnorm) {
		rt_Output->errorVar( get_class(), ": Missing normal!", 0 );
		rscene->clear(); return;
	    } else rt_nrm = get_gnormal();  //seems to be a planar surface
	    rt_nrm = mc2wc(rt_vtx->getPoint() + rt_nrm) - rt_pnt;
	    rs_nrm.x = rt_nrm.x; rs_nrm.y = rt_nrm.y; rs_nrm.z = rt_nrm.z;

	    //#### surface attributes:
	    //# reflection:
	    sf = rt_vtx->getSurface(); if(!sf) sf = &get_surface();
	    if (strlen( sf->specfunc)) {
		ins = 1; specfunc = sf->specfunc;  //this mesh has non-diffuse vertices
	    } else diff = 1;  //this mesh has diffuse vertices
	    if(diff && ins) { //cannot be mixed in a mesh
		rt_Output->errorVar(get_class(), ": Diffuse and non-diffuse vertices mixed!", NULL);
		rscene->clear(); return;
	    }
	    if (ins) {  //search for the associated BRDF
		brdf_idx = rscene->BRDFs->get_brdfidx((char*)specfunc);
		if (brdf_idx < 0) { 
		    // this function is not loaded yet:
		    fname = getenv( RT_GOOD_ROOT_DIR );
		    fname += "/YART/funcs/";
		    fname += specfunc;
		    fname += ".brf";

		    RT_RS_BRDFMatrix* brdf = new RT_RS_BRDFMatrix((char*)fname);
		    if(strcmp(brdf->get_name(), (char*)specfunc)) { 
			rt_Output->errorVar(get_name(), ": Cannot load reflection function ", 
					    (char*)specfunc, " from file ", (char*)fname, "!", 0 );
			delete brdf; rscene->clear(); return;
		    }
		    rscene->BRDFs->insert(brdf);  
		    // insert the function into function-container
		    brdf_idx = rscene->BRDFs->get_brdfidx((char*)specfunc);
		}
		// non-diffuse reflection, must be 0..1 !
		rs_irefl.r = sf->spec.r; rs_irefl.g = sf->spec.g; rs_irefl.b = sf->spec.b;
	    }
	    // diffuse reflection, must be 0..1 !
	    rs_drefl.r = sf->diff.r; rs_drefl.g = sf->diff.g; rs_drefl.b = sf->diff.b;
	    //sum must be 0..1 !
	    if (rs_drefl.r + rs_irefl.r > 1. || rs_drefl.g + rs_irefl.g > 1. || rs_drefl.b + rs_irefl.b > 1.) {
		rt_Output->errorVar(get_class(), ": The amount of reflected light cannot be greater than the amount of incident light!", NULL);
		rscene->clear(); return;
	    }
	    // #emission:
	    rs_demi.r = sf->emmi.r; rs_demi.g = sf->emmi.g; rs_demi.b = sf->emmi.b;
	    rs_demi = rs_demi * 1000; 
	    //scale to obtain useful values
	    if(ins) {
		// emission is not specified! - 
		// see the reflection values for diffuse and non-diffuse components:
		rs_iemi = rs_demi * rs_irefl; rs_demi = rs_demi * rs_drefl;
	    } // (else treat as diffuse emission)
	    
	    //#### vertex types:
	    if(gnorm) {                                             // planar surface: 
		if(x == 0 && y == 0) vt = corner0;                  // the first corner 
		else if(x == xn - 1 && y == 0) vt = corner1;        // the second corner
		else if(x == xn - 1 && y == xn - 1) vt = corner2;   // the third corner
		else if(x == 0 && y == xn - 1) vt = corner3;        // the fourth corner
		else vt = inner;                                    // an inner vertex
	    } else vt = curved;                                     // a vertex of a curved surface
	    
	    if (ins) {
		//#### create a new non-diffuse vertex:

		RT_RS_brdfRGB brdfRGB(brdf_idx, brdf_idx, brdf_idx);  //Note:
		//Today, the BRDF is the same for r, g, b. This will be changed in later versions.

		// create the vertex
		rs_ivtx = new RT_RS_InsVtx(rs_pnt, rs_nrm, rs_tgn, rs_drefl, rs_irefl, brdfRGB, vt);

		// insert the new vertex into points-container:
		rscene->Points->insert(rs_ivtx, &brdfRGB, &rs_demi, &rs_iemi);

	    } else {
		//#### create a new diffuse vertex:
		rs_dvtx = new RT_RS_DiffVtx(rs_pnt, rs_nrm, rs_drefl, vt);

		// insert the new vertex into points-container
		rscene->Points->insert(rs_dvtx, &rs_demi);
	    }
	}
    
    RT_RS_DiffArea* rs_dar;  //diffuse patch
    RT_RS_InsArea* rs_iar;   //non-diffuse patch
    long vidx[4], neighbors[4];

    //#### the patches:
    //get a new mesh number
    long mesh_nr = rscene->Areas->get_new_mesh_nr();
    long p_cnt = ins ? p_icnt : p_dcnt;
    for(y = 0; y < yn-1; y++)
	for(x = 0; x < xn-1; x++) {
	    //#### the 4 vertices of the new patch:
	    vidx[0] = (y+1)*xn + x + p_cnt;
	    vidx[1] = y*xn + x + p_cnt;
	    vidx[2] = y*xn + x+1 + p_cnt;
	    vidx[3] = (y+1)*xn + x+1 + p_cnt;
	  
	    //#### get the patch index:
	    long ar_cnt = ins ? rscene->Areas->get_inscnt() : rscene->Areas->get_diffcnt();

	    //#### the 4 neighbors of the new patch
	    neighbors[0] = x > 0 ? ar_cnt - 1 : -1;
	    neighbors[1] = y > 0 ? ar_cnt - xn + 1 : -1;
	    neighbors[2] = x < xn - 2 ? ar_cnt + 1 : -1;
	    neighbors[3] = y < yn - 2 ? ar_cnt + xn - 1 : -1;
	    
	    if(ins) {
		//#### create a new non-diffuse patch
		rs_iar = new RT_RS_InsArea(vidx[0], vidx[1], vidx[2], vidx[3],
					   neighbors[0], neighbors[1], neighbors[2],
					   neighbors[3], mesh_nr);
		// insert the new patch into areas-container
		rscene->Areas->insert(rs_iar);
	    }
	    else {
		//#### create a new diffuse patch
		rs_dar = new RT_RS_DiffArea(vidx[0], vidx[1], vidx[2], vidx[3],
					    neighbors[0], neighbors[1], neighbors[2],
					    neighbors[3], mesh_nr);
		//insert the new patch into areas-container
		rscene->Areas->insert(rs_dar);
	    }
	}
}

#endif

int RT_Quadmesh::intersect(const RT_Ray& ray, RT_InterSectionList& il) {
    int ret=0;
    RT_Ray mray;
    RTM_wcRay2mcRay(ray,mray);	    
    icache.restart();
    for(int _x = 0; _x < x-1; _x++)
	for(int _y = 0; _y < y-1; _y++)
	    ret += _intersect(_x,_y,mray,ray,il);
    return ret;
}

int RT_Quadmesh::_intersect(int _x,int _y,const RT_Ray& mray, const RT_Ray& ray,RT_InterSectionList& il) {
    int ret = 0;
    RT_Vertex *vert[2][3];
    vert[0][0] = vert[1][0] = get(_x,_y);
    vert[0][1] = get(_x,_y+1);
    vert[0][2] = vert[1][1] = get(_x+1,_y+1);
    vert[1][2] = get(_x+1,_y); 
    for(int tri = 0; tri < 2; tri++) {
	const RT_Vector *A = &vert[tri][0]->getPoint();
	const RT_Vector *B = &vert[tri][1]->getPoint();
	const RT_Vector *C = &vert[tri][2]->getPoint();
	
	RT_Vector a = *B - *A;
	
	RT_Vector n = a.CROSS(*C - *A);
	
	double t = n.DOT(mray.dir);
	if (fabs(t) < rt_RayEps) continue;
	t = (n.DOT(*A) - n.DOT(mray.pt)) / t;
	if (t < rt_RayEps) continue;
	RT_Vector p = mray.pt + mray.dir * t;
	int i = 0;
	RT_ifunc_ptr ifunc = &RT_intersect_yz;
	if (fabs(n.y) > fabs(n.x))
	    ifunc = &RT_intersect_xz;
	if (fabs(n.z) > fabs(n.x) && fabs(n.z) > fabs(n.y))
	    ifunc = &RT_intersect_xy;
	i += ifunc(p,*A,a);
	i += ifunc(p,*B,*C - *B);
	i += ifunc(p,*C,*A - *C);
	if (i & 1) {
	    double wt;
	    RTM_mcT2wcT(ray,mray,wt,t);
	    if (ray.valid(wt)) {
		RT_Surface surf;
		RT_Color zero_color(0,0,0);
		surf.ambi = zero_color; surf.diff = zero_color;
		surf.spec = zero_color; surf.emmi = zero_color;
		surf.tran = surf.refr = surf.shin = 0;
		double sum = 0;
		for (int v=0; v<3; v++) {
		    const RT_Surface *_surf = vert[tri][v]->getSurface();
		    if (!_surf) _surf = &get_surface();
		    double a = (vert[tri][v]->getPoint() - p).ABS();
		    if (fabs(a) < rt_RayEps) {
			surf = *_surf;
			sum = 1;
			break;
		    }
		    a = 1.0 / (a * a);
		    sum += a;
		    surf.ambi = surf.ambi + _surf->ambi * a;
		    surf.diff = surf.diff + _surf->diff * a;
		    surf.spec = surf.spec + _surf->spec * a;
		    surf.emmi = surf.emmi + _surf->emmi * a;
		    surf.tran += _surf->tran * a;
		    surf.refr += _surf->refr * a;
		    surf.shin += _surf->shin * a;		
		}
		sum = 1.0 / sum;
		surf.ambi = surf.ambi * sum;
		surf.diff = surf.diff * sum;
		surf.spec = surf.spec * sum;
		surf.emmi = surf.emmi * sum;
		surf.tran *= sum;
		surf.refr *= sum;
		surf.shin *= sum;
		int enter = n.DOT(mray.dir) < 0;
		il.add(new RT_InterSection(this,wt,enter,&surf));
		RT_Vector wp = ray.pt + ray.dir * wt;
		RT_IntersectionCacheEntry entry(wp,_x,_y,tri,n.UNITIZE());
		icache.add(entry);
		ret++;
	    }
	}
    }
    return ret;
}

void RT_Quadmesh::normal(const RT_Vector& R, RT_Vector &N) {
    icache.restart();
    while (!icache.at_end()) {
	const RT_IntersectionCacheEntry *e = icache.next();
	if (e->GetPoint() == R) {
	    int _x = e->Geti(); int _y = e -> Getj();
	    const RT_Vector *norm = &e -> GetNormal();
	    RT_Vertex *v;
	    RT_Vector r = wc2mc(R);
	    double a;
	    RT_Vector n;
	    RT_Vector const *_n;
	    if ( !e -> GetTriangle() ) {
		v = get(_x,_y);
		_n = v->getNormal(); if (!_n) _n = norm;
		a = (v->getPoint() - r).ABS();
		if (a < rt_RayEps) a = rt_RayEps;
		n = *_n * (1.0 / (a * a));

		v = get(_x,_y+1);
		_n = v ->getNormal(); if (!_n) _n = norm;
		a = (v->getPoint() - r).ABS();
		if (a < rt_RayEps) a = rt_RayEps;
		n = n + *_n * (1.0 / (a * a));

		v = get(_x+1,_y+1);
		_n = v->getNormal(); if (!_n) _n = norm;
		a = (v->getPoint() - r).ABS();
		if (a < rt_RayEps) a = rt_RayEps;
		n = n + *_n * (1.0 / (a * a));

	    } else {
		v = get(_x,_y);
		_n = v->getNormal(); if (!_n) _n = norm;
		a = (v->getPoint() - r).ABS();
		if (a < rt_RayEps) a = rt_RayEps;
		n = *_n * (1.0 / (a * a));

		v = get(_x+1,_y+1);
		_n = v->getNormal(); if (!_n) _n = norm;
		a = (v->getPoint() - r).ABS();
		if (a < rt_RayEps) a = rt_RayEps;
		n = n + *_n * (1.0 / (a * a));

		v = get(_x+1,_y);
		_n = v->getNormal(); if (!_n) _n = norm;
		a = (v->getPoint() - r).ABS();
		if (a < rt_RayEps) a = rt_RayEps;
		n = n + *_n * (1.0 / (a * a));
				
	    }
	    RTM_mcN2wcN(R,n,N);
	    return;
	}
    }
    rt_Output->warningVar(get_name(),": ask for normal at a non intersection point.",0);
    N = RT_NULL_NORMAL;
}

int RT_Quadmesh::copy(RT_Primitive *p) const {
    if (p->isA( RTN_QUADMESH)) {
	RT_Quadmesh *q = (RT_Quadmesh*)p;
	q->gnormal( get_gnormal() );

	int xs = get_x();
	int ys = get_y();
	q->newGeometry( xs, ys );
	RT_Vertex *tv = vertices;
	RT_Vertex *qv = q->vertices;
	for (int i = 0; i < xs; i++)
	    for (int j = 0; j < ys; j++)
		*qv++ = *tv++;
	return( 1 );
    }
    return( RT_Primitive::copy(p) );
}
