////////////////////////////////////////////////////////////////////////////////
//          Bidirectional Reflectance Distribution Function classes.          //   
//  LAST EDIT: Fri Aug  5 08:55:07 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//              yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <values.h>
#include "rs_brdf.h"
#include "rs_io.h"
#include "../global.h"
#include "../error.h"

//////////////////////  THE RT_RS_SpCoeff MEMBER FUNCTIONS /////////////////////////

void RT_RS_SpCoeff::alloch(int npnts, float f, char atemp)
{
  nelem = nc = nr = npnts;
  temp = atemp;

  x = new RT_RS_Vector(nelem);
  y = new RT_RS_Vector(nelem);
  b = new RT_RS_Vector(nelem-1);
  c = new RT_RS_Vector(nelem-1);
  d = new RT_RS_Vector(nelem-1);
  set(f);
  return;
}

void RT_RS_SpCoeff::freeh()
{
  if(x != NULL) delete x; x = NULL;
  if(y != NULL) delete y; y = NULL;
  if(b != NULL) delete b; b = NULL;
  if(c != NULL) delete c; c = NULL;
  if(d != NULL) delete d; d = NULL;
}

RT_RS_SpCoeff::RT_RS_SpCoeff(char *filename, FILE *fp)
{
  x = y = b = c = d = NULL;
  if(!read(filename, fp))
    freeh();
}

RT_RS_SpCoeff::RT_RS_SpCoeff(RT_RS_SpCoeff& a)
{
#ifdef RS_DEBUG
  if(a.x == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in copy-constructor", NULL);
#endif
  temp = a.temp;
  nelem = nc = nr = a.nelem;
  if(temp == 'y') {
    x = a.x; a.x = NULL;
    y = a.y; a.y = NULL;
    b = a.b; a.b = NULL;
    c = a.c; a.c = NULL;
    d = a.d; a.d = NULL;
  }
  else {
    x = new RT_RS_Vector(*a.x);
    y = new RT_RS_Vector(*a.y);
    b = new RT_RS_Vector(*a.b);
    c = new RT_RS_Vector(*a.c);
    d = new RT_RS_Vector(*a.d);
  }
}

void RT_RS_SpCoeff::set(float f)
{
#ifdef RS_DEBUG
  if(x == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in set()", NULL);
#endif
  for(int i = 0; i < nelem; i++)
    (*x)(i) = (*y)(i) = f;
  isZero = f < epsilon;
}

void RT_RS_SpCoeff::set_spl(RT_RS_Vector& ax, RT_RS_Vector& ay)
{
  float sum = .0;
  int i;
#ifdef RS_DEBUG
  if(x == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in set_spl()", NULL);
  if(ax.get_elements() < nelem || ay.get_elements() < nelem)
    rt_Output->fatalVar(get_class(), ":Dimensions do not match in set_spl()", NULL);
  if((ax(0) > epsilon) || (fabs(ax(nelem-1) - (float)M_PI_2) > epsilon5))
    rt_Output->fatalVar(get_class(), ":Illegal x-vector in set_spl()", NULL);
#endif
  *x = ax; (*x)(nelem-1) = M_PI_2; *y = ay;
  for(i=0;i<nelem;i++) sum += (*y)(i);
  isZero = sum < epsilon;
  comp_spl();
}

void RT_RS_SpCoeff::comp_spl()
{
  float sum = .0;
  for(int i=0;i<nelem;i++) sum += (*y)(i);
  isZero = sum < epsilon;
  nspline(*x, *y, *b, *c, *d);
}

RT_RS_SpCoeff& RT_RS_SpCoeff::operator = (RT_RS_SpCoeff& a)
{
#ifdef RS_DEBUG
  if(x == NULL || a.x == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator = ", NULL);
  if(nelem != a.nelem)
    rt_Output->fatalVar(get_class(), ":Dimensions do not match in operator =", NULL);
#endif
  *x = *(a.x); *y = *(a.y); *b = *(a.b); *c = *(a.c); *d = *(a.d);
  a.freet();
  return *this;
}

float RT_RS_SpCoeff::eval(float theta_in)
{
#ifdef RS_DEBUG
  if(x == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in eval()", NULL);
#endif
  if(isZero) return .0;
  else return spval(theta_in, *y, *b, *c, *d, *x);
}


boolean RT_RS_SpCoeff::read(char *filename, FILE *fp)
{
  int _npnts;
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getint(filename, fp, _npnts, RS_s, clmn);
  if(Ok) Ok = checkdim(filename, _npnts, 4, 1000);
  if(Ok) {
    freeh();
    alloch(_npnts, .0, 'n');
    Ok = x->read(filename, fp);
  }
  if(Ok) Ok = y->read(filename, fp);

  if(aln) fclose(fp);
  if(Ok)  set_spl(*x, *y);
  return Ok;
}

boolean RT_RS_SpCoeff::write(char *filename, FILE *fp)
{
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(x == NULL) return FALSE;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s %d\n", get_class(), nr);
    Ok = !ferror(fp);
  }
  if(Ok) Ok = x->write(filename, fp);
  if(Ok) Ok = y->write(filename, fp);

  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

void RT_RS_SpCoeff::print(FILE *f, char *n, int width, int decimalPlaces)
{
  int i, nperline;

  if(x == NULL) return;
  if(n) fprintf(f, "%s", n);
  if(isZero) fprintf(f, " is zero ");
  else {
    fprintf(f, "\n");
    nperline = 79/width;
    sprintf(fmt,"[%s%d.%df %s%d.%df]", "%", width, decimalPlaces,
                                       "%", width, decimalPlaces);
    for(i = 0; i < nelem; i++) {
      fprintf(f, fmt, (*x)(i),(*y)(i));
      if(i % nperline == nperline - 1) fprintf(f, "\n");
    }
  }
}
//////////////////////  THE RT_RS_BRDFMatrix MEMBER FUNCTIONS /////////////////////////

void RT_RS_BRDFMatrix::alloch(int n, int anpnts, float f, char atemp)
{
  nelem = n * n;
#ifdef RS_DEBUG
  if(nelem > RS_MAX_COEFF)
    rt_Output->errorVar(get_class(), ":Illegal number of elements", NULL);
#endif
  nr = nc = n;
  temp = atemp;
  npnts = anpnts;
  scm = new RT_RS_SpCoeff* [nelem];
  int l, m, i;
  for(i = 0; i < nelem; i++) scm[i] = NULL;
  for(l = 1; l < nr; l++)
    for(m = 0; m <= l; m++)    // coeffs with negative m are zero
      if((l+m) & 1)        // coeffs with l+m even are zero
        scm[l * nc + m] = new RT_RS_SpCoeff(npnts, f, temp);
  avg = new RT_RS_SpCoeff(npnts, f, temp);
}

void RT_RS_BRDFMatrix::freeh()
{
  if(scm) {
    for(int i = 0; i < nelem; i++)
      if(scm[i] != NULL)
         delete scm[i];
    delete scm; scm = NULL;
  }
  if(avg) { delete avg; avg = NULL; }
  if(name) { delete name; name = NULL; }
  if(description) { delete description; description = NULL; }
}

RT_RS_BRDFMatrix::RT_RS_BRDFMatrix(char *filename, FILE *fp)
{
  name = description = NULL;
  scm = NULL; avg = NULL;
  if(!read(filename, fp))
    freeh();
}

RT_RS_BRDFMatrix::RT_RS_BRDFMatrix(RT_RS_BRDFMatrix& a)
{
#ifdef RS_DEBUG
  if(a.scm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in copy-constructor", NULL);
#endif
  int l,m,i;
  nr = a.nr;
  nc = a.nc;
  temp = a.temp;
  nelem = a.nelem;
  npnts = a.npnts;

  if(a.temp == 'y') {
    scm = a.scm; a.scm = NULL;
    avg = a.avg; a.avg = NULL;
    name = a.name; a.name = NULL;
    description = a.description; a.description = NULL;
  }
  else {
    scm = new RT_RS_SpCoeff* [nelem];
    for(i = 0; i < nelem; i++) scm[i] = NULL;
    for(l = 1; l < nr; l++)
      for(m = 0; m <= l; m++)    // coeffs with negative m are zero
        if((l+m) & 1) {      // coeffs with l+m even are zero
          i = l * nc + m;
          scm[i] = new RT_RS_SpCoeff(*a.scm[i]);
        }
    avg = new RT_RS_SpCoeff(*a.avg);
    if(a.name) {
      name = new char[strlen(a.name) + 1];
      strcpy(name, a.name);
    } else name = NULL;
    if(a.description) {
      description = new char[strlen(a.description) + 1];
      strcpy(description, a.description);
    } else description = NULL;
  }
}

void RT_RS_BRDFMatrix::set(float f)
{
#ifdef RS_DEBUG
  if(scm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in set()", NULL);
#endif
  for(int i = 0; i < nelem; i++)
    if(scm[i] != NULL) scm[i]->set(f);
  avg->set(f);
}

void RT_RS_BRDFMatrix::set_scm(RT_RS_lmMatrix& mx, RT_RS_lmMatrix& my, int pnt)
{
  int l,m,i;
#ifdef RS_DEBUG
  if(scm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in set_scm()", NULL);
  if(mx.get_rows() != nr || mx.get_columns() != nc ||
     my.get_rows() != nr || my.get_columns() != nc ||
     pnt < 0 || pnt >= npnts)
     rt_Output->fatalVar(get_class(), ":Dimensions do not match in set_scm()", NULL);
#endif
  for(l=1;l<nr;l++)
    for(m=0;m<=l;m++)
      if((l+m)&1) {
        i=l*nc+m;
        scm[i]->get_x()(pnt)=mx(l,m);
        scm[i]->get_y()(pnt)=my(l,m);
      }
}

void RT_RS_BRDFMatrix::set_avg(float x, float y, int pnt)
{
#ifdef RS_DEBUG
  if(avg == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in set_avg()", NULL);
  if(x < 0 || pnt < 0 || pnt >= npnts)
     rt_Output->fatalVar(get_class(), ":Illegal arguments in set_avg()", NULL);
#endif
  avg->get_x()(pnt) = x;
  avg->get_y()(pnt) = y;
}

void RT_RS_BRDFMatrix::comp_scm()
{
#ifdef RS_DEBUG
  if(scm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in comp_scm()", NULL);
#endif
  for(int i=0; i<nelem; i++)
    if(scm[i] != NULL)
      scm[i]->comp_spl();
  avg->comp_spl();
}

RT_RS_InsMatrix RT_RS_BRDFMatrix::eval(float theta_in, float phi_in, float scale_val)
{
  int l,m;

#ifdef RS_DEBUG
  if(scm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in eval()", NULL);
  if(theta_in < 0 || theta_in-M_PI_2 > epsilon5 ||
     phi_in < 0 || phi_in-M_2PI > epsilon5 ||
     scale_val < 0)
     rt_Output->fatalVar(get_class(), ":Illegal arguments in eval()", NULL);
#endif

  RT_RS_InsMatrix ITemp(nc, 0., 'y');

  //interpolate and scale
  for(l=1;l<nr;l++)
    for(m=0;m<=l;m++)    //negative subscripted coefficients are zero
      if((l+m)&1)        //coefficients for wich l+m is even are zero
        ITemp(l,m)=scm[l*nc+m]->eval(theta_in) * scale_val;
  ITemp.avg  = avg->eval(theta_in) * scale_val;

  //rotate
  for(l=1;l<nc;l++)
    for(m=-l;m<0;m++)
      if((l+m)&1)
        ITemp(l,m)=ITemp(l,-m)*sin(-m*phi_in);
  for(l=1;l<nc;l++)
    for(m=1;m<=l;m++)
      if((l+m)&1)
        ITemp(l,m)*=cos(m*phi_in);

  return ITemp;
}

RT_RS_SpCoeff& RT_RS_BRDFMatrix::operator () (int i, int j)
{
#ifdef RS_DEBUG
  if(scm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator ()", NULL);
  if(i < 1 || i >= nr)
    rt_Output->fatalVar(get_class(), ":Illegal row index in operator ()", NULL);
  if(j < 0 || j > i)
    rt_Output->fatalVar(get_class(), ":Illegal column index in operator ()", NULL);
  if(!((j+i) & 1))
    rt_Output->fatalVar(get_class(), ":Illegal index sum in operator ()", NULL);
#endif
  return *scm[i * nc + j];
}

boolean RT_RS_BRDFMatrix::read(char *filename, FILE *fp)
{
  int _nr, _npnts, l, m;
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getint(filename, fp, _nr, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, _npnts, RS_s, clmn);
  if(Ok) {
    freeh();
    Ok = getline(filename, fp, RS_s); clmn = 0;
  }
  if(Ok) Ok = getlinestr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) {
    if(strstr(RS_s1, NONAME)) name = NULL;
    else {
      name = new char[strlen(RS_s1) + 1];
      strcpy(name, RS_s1);
    }
    Ok = getlinestr(filename, fp, RS_s, RS_s1, clmn);
  }
  if(Ok) {
    if(strstr(RS_s1, NODESCRIPTION))
      description = NULL;
    else {
      description = new char[strlen(RS_s1) + 1];
      strcpy(description, RS_s1);
    }
    alloch(_nr, _npnts, 0, 'n');
    for(l=1;l<nr && Ok;l++)
      for(m=0;m<=l && Ok;m++)
        if((l+m)&1)
          Ok = ((*this)(l,m)).read(filename,fp);
  }
  if(Ok) Ok = avg->read(filename,fp);
  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_BRDFMatrix::write(char *filename, FILE *fp)
{
  int l,m;
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s %d %d\n%s\n%s\n", get_class(), nr, npnts,
            get_name(), get_description());
    Ok = !ferror(fp);
  }
  for(l=1;l<nr && Ok;l++)
    for(m=0;m<=l && Ok;m++)
      if((l+m)&1)
        Ok = ((*this)(l,m)).write(filename,fp);
  if(Ok) Ok = avg->write(filename,fp);
  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, ".", NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}
#pragma argsused
void RT_RS_BRDFMatrix::print(FILE *f, char *n, int, int) {
  int l, m, i = 0;
  char ctitle[30];

  if(scm == NULL) return;
  if(n) fprintf(f, "%s\n", n);
  fprintf(f, "name: %s\ndescription: %s\n", get_name(), get_description());
  for(l=1;l<nr;l++)
    for(m=0;m<=l;m++)
      if((l+m)&1) {
#ifdef RS_PRINT_ALL
        sprintf(ctitle,"coefficient l=%d m=%d", l, m);
         ((*this)(l,m)).print(f, ctitle, width, decimalPlaces);
#endif  
        i++;
       }
  fprintf(f, "coefficients: %d\ncoefficient control points: %d\n", i, npnts);
#ifdef RS_PRINT_ALL
  avg->print(f, "avg", width, decimalPlaces);
#endif
}
//////////////////////  THE _RS_BRDFs MEMBER FUNCTIONS /////////////////////////
void RT_RS_BRDFs::freeh()
{
  for(int i = 0; i < cnt; i++) delete BRDFs[i];
  cnt = 0;
}

RT_RS_BRDFs::RT_RS_BRDFs(char *filename, FILE *fp)
{
  cnt = 0;
  if(!read(filename, fp))
    freeh();
}

boolean RT_RS_BRDFs::insert(RT_RS_BRDFMatrix* a)
{
  if(cnt >= RS_MAX_BRDFS) return FALSE;
  BRDFs[cnt] = a;
  cnt++;
  return TRUE;
}

RT_RS_BRDFMatrix* RT_RS_BRDFs::get_at(int n)
{
#ifdef RS_DEBUG
  if(n < 0 || n >= cnt)
    rt_Output->fatalVar(get_class(), ":Illegal index in get_at()", NULL);
#endif
  return BRDFs[n];
}

RT_RS_BRDFMatrix* RT_RS_BRDFs::get_brdf(char* name)
{
  int i = get_brdfidx(name);
  return i < 0 ? NULL : BRDFs[i];
}

int RT_RS_BRDFs::get_brdfidx(char* name) // get index of BRDF with name
{
  for(int i=0; i<cnt; i++)
    if(!strcmp(BRDFs[i]->get_name(), name))
      return i;
  return -1;
}

boolean RT_RS_BRDFs::read(char *filename, FILE *fp)  // Read ASCII from a file.
{
  int _cnt, i;
  RT_RS_BRDFMatrix* brdf;
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getint(filename, fp, _cnt, RS_s, clmn);
  if(Ok) Ok = checkdim(filename, _cnt, 0, RS_MAX_BRDFS);
  if(Ok) {
    freeh();
    for(i = 0; i < _cnt && Ok; i++) {
      brdf = new RT_RS_BRDFMatrix(filename, fp);
      insert(brdf);
    }
  }
  if(aln) fclose(fp);
  return(Ok);
}

boolean RT_RS_BRDFs::write(char *filename, FILE *fp) // Write ASCII file
{
  int i;
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
    fprintf(fp,"%s %d\n", get_class(), cnt);
    Ok = !ferror(fp);
  }
  if(Ok)
    for(i = 0; i < cnt && Ok; i++)
      Ok = BRDFs[i]->write(filename, fp);
  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

#pragma argsused
void RT_RS_BRDFs::print(FILE *f, char *n, int width, int decimalPlaces)
{
  char stitle[30];

  if(n) fprintf(f, "%s\n", n);
  fprintf(f, "functions: %d\n", cnt);
  for(int i = 0; i < cnt; i++) {
    sprintf(stitle, "BRDF nr. %d:", i);
    get_at(i)->print(f, stitle, width, decimalPlaces);
  }
}
