////////////////////////////////////////////////////////////////////////////////
//                      Intensity function class.                             //   
//  LAST EDIT: Fri Aug  5 08:55:06 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//              yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>  
#include "rs_inmat.h"
#include "rs_sphr.h"
#include "rs_io.h"
#include "../global.h"
#include "../error.h"

//////////////////////  THE RT_RS_InsMatrix MEMBER FUNCTIONS /////////////////////////

void RT_RS_InsMatrix::alloch(int n, float cf, char atemp)
{
  if(!n) {
    fm = NULL; cm = NULL;
    return;
  }
#ifdef RS_DEBUG
  if(!(n&1))
    rt_Output->fatalVar(get_class(), ":Number of elements n is even", NULL);
#endif
   nr= n;
   nc= (n - 1) >> 1;
   nelem=nr*nc;
#ifdef RS_DEBUG
  if(nelem > RS_MAX_COEFF)
    rt_Output->fatalVar(get_class(), ":Illegal number of elements", NULL);
#endif
  temp = atemp;
  cm = NULL;
  fm = new float[nelem];
  set(cf);
}

void RT_RS_InsMatrix::freeh()
{
  if(cm != NULL) { delete cm; cm = NULL; }
  if(fm != NULL) { delete fm; fm = NULL; }
}

RT_RS_InsMatrix::RT_RS_InsMatrix(char *filename, FILE* fp)
{
  fm = NULL; cm = NULL;
  if(!read(filename, fp))
    freeh();
}

RT_RS_InsMatrix::RT_RS_InsMatrix(RT_RS_InsMatrix& a)
{
#ifdef RS_DEBUG
  if(a.fm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in copy-constructor", NULL);
#endif
  nr = a.nr;
  nc = a.nc;
  temp = a.temp;
  nelem = a.nelem;
  cm = NULL;

  if(a.temp == 'y') {
    fm = a.fm;
    a.fm = NULL;
  }
  else {
    fm = new float[nelem];
#ifdef RS_USE_MEM_FUNCS
    memcpy(fm, a.fm, nelem * sizeof(float));
#else
    for(int i = 0; i < nelem; i++) fm[i] = a.fm[i];
#endif
  }
  avg = a.avg;
}

void RT_RS_InsMatrix::set(float f)
{
#ifdef RS_DEBUG
  if(fm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in set()", NULL);
#endif
  for(int i = 0; i < nelem; i++) fm[i] = f;
  avg = f;
}

void RT_RS_InsMatrix::compress()
{
  int l,m,i;
  int nCnt = 0;
  int j = 0;
  int n = 0;
  float maxmin, thres, mmdiv;
  unsigned int tcm[RS_MAX_COEFF];
  unsigned char *pcm;

#ifdef RS_DEBUG
  if(fm == NULL || cm != NULL)
    rt_Output->fatalVar(get_class(), " is deleted or already compressed", NULL);
#endif

  max = -BIGFLOAT;
  min =  BIGFLOAT;
  for(i=0; i<nelem;i++) {
    if(fm[i]>max) max = fm[i];
    if(fm[i]<min) min = fm[i];
  }
  maxmin = max-min;
  thres = maxmin / 65534.;   //2*32767
  mmdiv = 32767./maxmin;
  for(l = 1; l < nr; l++){
    for(m = -l; m <= l; m++) {
      if((l+m)&1) {
        if(fabs((*this)(l,m))<thres) n++;
        else {
          if(n) {
            while (n>127) {
              tcm[j++] = 0x80FF;
              nCnt++;
              n-=127;
            }
            tcm[j++] = 0x8080 | n;
            nCnt++;
            n = 0;
          }
          tcm[j++] = (unsigned)floor(((*this)(l,m)-min)*mmdiv+.5);
        }
      }
    }
  }
  if(n) {
    while (n>127) {
      tcm[j++] = 0x80FF;
      nCnt++;
      n-=127;
    }
    tcm[j++] = 0x8080 | n;
    nCnt++;
  }

  pcm = cm = new unsigned char [nelem = j*2-nCnt];
  for(i = 0; i<j; i++) {
    if(!(tcm[i] & 0x8000))
      *pcm++ = (unsigned char)(tcm[i] >> 8);
    *pcm++ = (unsigned char)tcm[i];
  }
  delete fm; fm = NULL;
}

void RT_RS_InsMatrix::expand()
{
  int l,m;
  int n = 0;
  unsigned x;
  float maxmin = max-min;
  float mmdiv = maxmin / 32767.;
  unsigned char* pcm = cm;

#ifdef RS_DEBUG
  if(fm != NULL || cm == NULL)
    rt_Output->fatalVar(get_class(), " is deleted or not compressed", NULL);
#endif

  nelem = nr*nc;
  fm = new float[nelem];

  for(l = 0; l < nr; l++)
    for(m = -l; m <= l; m++) {
      if((l+m)&1) {
        if(!n)
          if(*pcm & 0x80)
            n = *pcm++ & 0x7F;
          else {
            x = *pcm++ << 8;
            x |= *pcm++;
            (*this)(l,m) = (float)x * mmdiv + min;
          }
        if(n) {
          (*this)(l,m) = .0;
          n--;
        }
      }
    }
  delete cm; cm = NULL;
}

float RT_RS_InsMatrix::eval(float theta_out, float phi_out)
{
  int l, m;
  float f=.0;
  float Coeff;

#ifdef RS_DEBUG
  if(fm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in eval()", NULL);
  if(theta_out < 0 || theta_out-M_PI > epsilon5 ||
     phi_out < 0 || phi_out-M_2PI > epsilon5)
    rt_Output->fatalVar(get_class(), ":Illegal arguments in eval()", NULL);
#endif
  for(l=0;l<nr;l++)
    for(m=-l;m<=l;m++)
      if((l+m)&1)
        if(fabs(Coeff=(*this)(l,m))>epsilon)
          f+=Coeff*comp_SpherHarms(l,m,theta_out,phi_out);
  if(f<0) f=.0;
  return f;
}

RT_RS_InsMatrix& RT_RS_InsMatrix::operator = ( RT_RS_InsMatrix& a) 
{
#ifdef RS_DEBUG
  if(fm == NULL || a.fm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator = ", NULL);
  if(nelem != a.nelem)
    rt_Output->fatalVar(get_class(), ":Equating matrices of unequal size", NULL);
#endif
#ifdef RS_USE_MEM_FUNCS
  memcpy(fm, a.fm, nelem * sizeof(float));
#else
  for(int i = 0; i < nelem; i++) fm[i] = a.fm[i];
#endif
  avg = a.avg;
  a.freet();
  return *this;
}

float& RT_RS_InsMatrix::operator () (int i, int j)
{
#ifdef RS_DEBUG
  if(fm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator ()", NULL);
  if(i < 1 || i >= nr)
    rt_Output->fatalVar(get_class(), ":Illegal row index in operator ()", NULL);
  if(j < -i || j > i)
    rt_Output->fatalVar(get_class(), ":Illegal column index in operator ()", NULL);
  if(!((j+i) & 1))
    rt_Output->fatalVar(get_class(), ":Illegal index sum in operator ()", NULL);
#endif
  if(j < 0) {
    int temp = i;
    i = -1 - j;
    j = temp - 1;
  }
  if(!(i&1)) j--;
  j >>= 1;
  return fm[i * nc + j];
}


RT_RS_InsMatrix RT_RS_InsMatrix::operator + ( RT_RS_InsMatrix& a) 
{
#ifdef RS_DEBUG
  if(fm == NULL || a.fm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator +", NULL);
  if(nelem != a.nelem)
    rt_Output->fatalVar(get_class(), ":Dimensions do not match in operator +", NULL);
#endif
  RT_RS_InsMatrix Temp(nr, nc, 'y');
  for(int i = 0; i < nelem; i++)
    Temp.fm[i] = fm[i] + a.fm[i];
  Temp.avg = avg + a.avg;
  freet();
  a.freet();
  return Temp;
}

RT_RS_InsMatrix RT_RS_InsMatrix::operator - ( RT_RS_InsMatrix& a) {
#ifdef RS_DEBUG
  if(fm == NULL || a.fm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator -", NULL);
  if(nelem != a.nelem)
    rt_Output->fatalVar(get_class(), ":Dimensions do not match in operator -", NULL);
#endif
  RT_RS_InsMatrix Temp(nr, nc, 'y');
  for(int i = 0; i < nelem; i++)
    Temp.fm[i] = fm[i] - a.fm[i];
  Temp.avg = avg - a.avg;
  freet();
  a.freet();
  return Temp;
}

RT_RS_InsMatrix RT_RS_InsMatrix::operator * (float f) {
  int i;
#ifdef RS_DEBUG
  if(fm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator * (float)", NULL);
#endif
  RT_RS_InsMatrix Temp(nr, nc, 'y');
  for(i = 0; i < nelem; i++)
    Temp.fm[i] = f * fm[i];
  Temp.avg = f * avg;
  freet();
  return Temp;
}

RT_RS_InsMatrix RT_RS_InsMatrix::operator / (float f) 
{
#ifdef RS_DEBUG
  if(fm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator /", NULL);
#endif
  if(f < epsilon)
    rt_Output->fatalVar(get_class(), ":Division by zero in operator /", NULL);
  float recip = 1. / f;
  RT_RS_InsMatrix Temp(nr, nc, 'y');
  for(int i = 0; i < nelem; i++)
    Temp.fm[i] = fm[i] * recip;
  Temp.avg = avg * recip;
  freet();
  return Temp;
}

int RT_RS_InsMatrix::read(char* filename, FILE* fp)
{
  int _nr, l, m;
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getint(filename, fp, _nr, RS_s, clmn);
  if(Ok) {
    freeh();
    alloch(_nr, _nr, 'n');
    for(l=0;l<nr && Ok;l++)
      for(m=-l;m<=l && Ok;m++)
        if((l+m)&1)
          Ok = getfloat(filename, fp, (*this)(l,m), RS_s, clmn);
  }
  if(Ok) Ok = getfloat(filename, fp, avg, RS_s, clmn);
  if(aln) fclose(fp);
  return Ok;
}

int RT_RS_InsMatrix::write(char *filename, FILE *fp)
{
#ifdef RS_DEBUG
  if(fm == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), NULL);
#endif
  int l,m;
  int aln = fp == NULL;
  boolean Ok = TRUE;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
    fprintf(fp,"%s %d\n", get_class(), nr);
    Ok = !ferror(fp);
  }
  for(l=0;l<nr && Ok;l++)
    for(m=-l;m<=l && Ok;m++)
      if((l+m)&1) {
        fprintf(fp, "% .6f\n", (*this)(l,m));
        Ok = !ferror(fp);
      }
  if(Ok) {
    fprintf(fp, "% .6f\n", avg);
    Ok = !ferror(fp);
  }
  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

void RT_RS_InsMatrix::print(FILE *f, char *n, int width, int decimalPlaces)
{
  int l, m, nperline;

  if(fm == NULL) return;
  if(n) fprintf(f, "%s\n", n);
  nperline = 79/width;
  sprintf(fmt,"%s%d.%df", "%",width,decimalPlaces);
  for(l = 0; l < nr; l++){
    for(m = -l; m <= l; m++)
      if((l+m)&1) {
        fprintf(f, fmt, (*this)(l,m));
        if(((m+l) % nperline == nperline -1) && m < l)
          fprintf(f, "\n");
      }
       fprintf(f, "\n");
  }
  sprintf(fmt, "avg: %s%d.%df\n", "%", width, decimalPlaces);
  fprintf(f, fmt, avg);
}

