////////////////////////////////////////////////////////////////////////////////
//                      Point related classes.                                //   
//  LAST EDIT: Sat Aug 20 18:35:48 1994 by H. Fischer
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//              ekki@prakinf.tu-ilmenau.de                                    //
//                                                                            //
// (C) Copyright 1993 YART team                                               //
////////////////////////////////////////////////////////////////////////////////


#ifndef __RS_PNTS_H__
#define __RS_PNTS_H__

#include "rs_scene.h"
#include "rs_inmat.h"
#include "rs_brdf.h"
#include "rs_io.h"

//a rgb-triple of scalar values
class RT_RS_DiffRGB :public RT_RS_IO {
  public:
    float r, g, b;  //red, green, blue

    char* get_class() { return "RS_DiffRGB"; }

    //constructors:
    RT_RS_DiffRGB(float ar, float ag, float ab) : r(ar), g(ag), b(ab) {}
    RT_RS_DiffRGB(float af) { set(af); }
    RT_RS_DiffRGB() { r = g = b = 0.; }
    RT_RS_DiffRGB(const RT_RS_DiffRGB &v) {r = v.r; b = v.b; g = v.g; }
    
    float comp_sum() { return r + g + b; }  //return the sum of r, g, b
    void set(float f) { r = g = b = f; }

    RT_RS_DiffRGB operator = (const RT_RS_DiffRGB& v) {
	r = v.r; g = v.g; b = v.b; return *this; }
    RT_RS_DiffRGB operator = (float f) { set(f); return *this; }
    RT_RS_DiffRGB operator + (const RT_RS_DiffRGB& v) const
  { RT_RS_DiffRGB c(r + v.r, g + v.g, b + v.b); return c; }
    RT_RS_DiffRGB operator - (const RT_RS_DiffRGB& v) const
  { RT_RS_DiffRGB c(r - v.r, g - v.g, b - v.b); return c; }
    RT_RS_DiffRGB operator * (const float f) const
   { RT_RS_DiffRGB c(f * r, f * g, f * b); return c; }
    RT_RS_DiffRGB operator * (const RT_RS_DiffRGB& v) const
  { RT_RS_DiffRGB c(r * v.r, g * v.g, b * v.b); return c; }

    //in/out functions
    boolean read(char *filename, FILE *fp = NULL); // Read from ASCII file.
    boolean write(char *filename, FILE *fp = NULL); // Write to ASCII file.
    void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};

extern RT_RS_DiffRGB rt_DefaultRGB;

//vertex attributes:
enum vertex_type { corner0, corner1, corner2, corner3, inner, curved };
  // corner0, corner1, corner2, corner3 : first, second, ... corner of a planar surface
  // inner: inner point of a planar surface (default)
  // curved: point of a curved surface                              

enum visibility { unknown, visible, hide }; //used to speed up rendering

//the vertex
class RT_RS_Vtx : public RT_RS_IO {
  friend RT_RS_Points; //the container
protected:
  RT_RS_Scene* scene; //the scene to wich this vertex belongs to
  float ar;           //the "point's area"
  visibility vis;    
  vertex_type vtx_type; 
public:
  RT_RS_3DVector pnt;   //the coordinate in 3D-space
  RT_RS_3DVector nrm;   //the point's normal

  char* get_class() { return "RS_Vtx"; }

  RT_RS_Vtx(RT_RS_Scene* aScene, char *filename, FILE *fp = NULL) :
    scene(aScene), ar(0) { read(filename, fp); }
  RT_RS_Vtx(RT_RS_Scene* aScene = NULL) : scene(aScene), ar(0), vtx_type(inner) {}
  RT_RS_Vtx(const RT_RS_3DVector& apnt, const RT_RS_3DVector& anrm, vertex_type avtx_type = inner,
            RT_RS_Scene* aScene = NULL);
  RT_RS_Vtx(RT_RS_Vtx& v) : scene(v.scene), pnt(v.pnt), nrm(v.nrm), ar(v.ar),
           vtx_type(v.vtx_type) {}

  void set_scene(RT_RS_Scene* ascene) { scene = ascene; }
  void set_vtx_type(vertex_type type) { vtx_type = type; }
  vertex_type get_vtx_type() { return vtx_type; }
  void set_ar(float f) { ar = f; }
  float get_ar() { return ar; }
  virtual float get_acc() { return .0; };
  virtual float get_ush() { return .0; };
  virtual boolean get_view_val(RT_RS_DiffRGB&, RT_RS_Vtx&,
                               boolean = TRUE) { return FALSE; }
  float comp_ctheta(RT_RS_3DVector &_pnt); //compute the angle theta to vtx
  float comp_theta(RT_RS_3DVector &) { return acos(comp_ctheta(pnt)); }

  //in/out functions
  boolean read(char *filename, FILE *fp = NULL); // Read from ASCII file.
  boolean write(char *filename, FILE *fp = NULL); // Write to ASCII file.
  void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};
                                                                                                                                              
//a vertex of a diffuse patch
class RT_RS_DiffVtx : public RT_RS_Vtx {
public:
  RT_RS_DiffRGB d_acc_ins, d_ush_ins, //accumulated intensity, unshot intensity
                d_p,      // diffuse reflectivity
                view_val; //used during rendering

  char* get_class() { return "RS_DiffVtx"; }

  RT_RS_DiffVtx(RT_RS_Scene* aScene, char *filename, FILE *fp = NULL) :
                RT_RS_Vtx(aScene) { read(filename, fp); }
  RT_RS_DiffVtx(RT_RS_Scene* aScene = NULL) : RT_RS_Vtx(aScene) {}
  RT_RS_DiffVtx(const RT_RS_3DVector &apnt, RT_RS_3DVector &anrm,
                RT_RS_DiffRGB &ad_p = rt_DefaultRGB,
                vertex_type avtx_type = inner, RT_RS_Scene* aScene = NULL) :
                RT_RS_Vtx(apnt, anrm, avtx_type, aScene), d_p(ad_p) {}
  RT_RS_DiffVtx(RT_RS_DiffVtx& v) : RT_RS_Vtx(v),
                d_acc_ins(v.d_acc_ins), d_ush_ins(v.d_ush_ins), d_p(v.d_p) {}

  float get_acc() { return d_acc_ins.comp_sum(); }
  float get_ush() { return d_ush_ins.comp_sum(); }
  boolean get_view_val(RT_RS_DiffRGB& val, RT_RS_Vtx& view_dir, boolean eval_ins = TRUE);
  //in/out functions
  boolean read(char *filename, FILE *fp = NULL); // Read from ASCII file.
  boolean write(char *filename, FILE *fp = NULL); // Write to ASCII file.
  void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};


//BRDF-index-triple
class RT_RS_brdfRGB : public RT_RS_IO {
public:
  int r, g, b;     //red, green, blue index in BRDF-container

  char* get_class() { return "RS_brdfRGB"; }

  RT_RS_brdfRGB() { r = g = b = -1; }
  RT_RS_brdfRGB(int ar, int ag, int ab) : r(ar), g(ag), b(ab) {}
  RT_RS_brdfRGB(RT_RS_brdfRGB& v) : r(v.r), g(v.g), b(v.b) {}

  //in/out functions
  boolean read(char *filename, FILE *fp = NULL); // Read from ASCII file.
  boolean write(char *filename, FILE *fp = NULL); // Write to ASCII file.
  void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};

extern RT_RS_brdfRGB rt_BrdfRGB;

//a rgb-triple of BRDFs
class RT_RS_InsRGB : public RT_RS_IO {
public:
  void alloch(int r_nr = 0, int g_nr = 0, int b_nr = 0); //allocate heap memory
  void freeh(); //free heap

  RT_RS_InsMatrix *r, *g, *b;  //red, green, blue

  char* get_class() { return "RS_InsRGB"; }

  RT_RS_InsRGB() { r = g = b = NULL; }
  RT_RS_InsRGB(const RT_RS_InsRGB& v);
  RT_RS_InsRGB( RT_RS_InsMatrix &ar, RT_RS_InsMatrix &ag, RT_RS_InsMatrix &ab);
  RT_RS_InsRGB(int r_nr, int g_nr, int b_nr) { alloch(r_nr, g_nr, b_nr); }
  virtual ~RT_RS_InsRGB() { freeh(); }

  RT_RS_DiffRGB eval(float theta_out, float phi_out); //evaluate the functions in outgoing direction theta, phi
  void re_alloch(int r_nr, int g_nr, int b_nr)
    { freeh(); alloch(r_nr, g_nr, b_nr); }
  float comp_sum() { return (r != NULL && g != NULL && b != NULL) ?
    r->avg + g->avg + b->avg : .0; }   //return the average sum

  void set(float f) { r->set(f); g->set(f); b->set(f); }
  void set_temp(char temp) { r->temp = g->temp = b->temp = temp; } //change the temporary state of tze functions

  RT_RS_InsRGB operator = (const RT_RS_InsRGB&);
  RT_RS_InsRGB operator + (const RT_RS_InsRGB&);
  RT_RS_InsRGB operator - (const RT_RS_InsRGB&);
  RT_RS_InsRGB operator * (float f);
  RT_RS_InsRGB operator * (const RT_RS_DiffRGB&);

  //in/out functions
  boolean read(char *filename, FILE *fp = NULL); // Read from ASCII file.
  boolean write(char *filename, FILE *fp = NULL); // Write to ASCII file.
  void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};

extern RT_RS_3DVector rt_Tangent;
                                                                          
                                                                          
//a vertex of a non-diffuse patch
class RT_RS_InsVtx : public RT_RS_DiffVtx
{
  void alloch(int r_nr = 0, int g_nr = 0, int b_nr = 0)
    { i_acc_ins.alloch(r_nr, g_nr, b_nr); i_ush_ins.alloch(r_nr, g_nr, b_nr); }
  void freeh() { i_acc_ins.freeh(); i_ush_ins.freeh(); }
public:
  RT_RS_3DVector tgn;                  //the point's tangent
  RT_RS_InsRGB i_acc_ins, i_ush_ins;   //accumulated and unshot intensity
  RT_RS_DiffRGB i_p;               //non-diffuse reflectivity of the patch
  RT_RS_brdfRGB brdf;              //index to the associated BRDF

  char* get_class() { return "RS_InsVtx"; }

  RT_RS_InsVtx(RT_RS_Scene* aScene, char *filename, FILE *fp = NULL)
    : RT_RS_DiffVtx(aScene) { if(!read(filename, fp)) freeh(); }
  RT_RS_InsVtx(RT_RS_Scene* aScene = NULL) : RT_RS_DiffVtx(aScene) {}
  RT_RS_InsVtx(RT_RS_3DVector& apnt, RT_RS_3DVector& anrm,
               RT_RS_3DVector& atgn = rt_Tangent,
               RT_RS_DiffRGB& ad_p  = rt_DefaultRGB,
               RT_RS_DiffRGB& ai_p  = rt_DefaultRGB,
               RT_RS_brdfRGB& abrdf = rt_BrdfRGB,
               vertex_type avtx_type = inner, RT_RS_Scene* aScene = NULL);
  RT_RS_InsVtx(RT_RS_InsVtx& v) : RT_RS_DiffVtx(v), tgn(v.tgn),
    i_acc_ins(v.i_acc_ins), i_ush_ins(v.i_ush_ins), i_p(v.i_p), brdf(v.brdf) {}
  virtual ~RT_RS_InsVtx() { freeh(); }

  void re_alloch();
  float comp_phi(RT_RS_3DVector& _pnt);  //compute the angle to vtx

  RT_RS_InsRGB comp_dI(float theta_in, float phi_in, RT_RS_DiffRGB& PHI);

  void get_brdf(RT_RS_BRDFMatrix*& r, RT_RS_BRDFMatrix*& g, RT_RS_BRDFMatrix*& b);

  float get_acc() { return i_acc_ins.comp_sum() + d_acc_ins.comp_sum(); }
  float get_ush() { return i_ush_ins.comp_sum() + d_ush_ins.comp_sum(); }
  boolean get_view_val(RT_RS_DiffRGB& val, RT_RS_Vtx& view_dir, boolean eval_ins = TRUE);

  //in/out functions
  boolean read(char *filename, FILE *fp = NULL); // Read from ASCII file.
  boolean write(char *filename, FILE *fp = NULL); // Write to ASCII file.
  void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};


class RT_RS_HyDiffVtx :  public RT_RS_Vtx {
public:
  RT_RS_DiffRGB d_ush_ins; // unshot intensity

  char* get_class() { return "RS_HyDiffVtx"; }
                                    

  RT_RS_HyDiffVtx(RT_RS_Scene* aScene, char *filename, FILE *fp = NULL) :
                  RT_RS_Vtx(aScene) { read(filename, fp); }
  RT_RS_HyDiffVtx(RT_RS_DiffVtx& dvtx) : RT_RS_Vtx(dvtx.pnt, dvtx.nrm), 
                  d_ush_ins(dvtx.d_ush_ins) { ar = dvtx.get_ar(); }

  RT_RS_HyDiffVtx(RT_RS_Scene* aScene = NULL) : RT_RS_Vtx(aScene) {}
  RT_RS_HyDiffVtx(const RT_RS_3DVector &apnt, RT_RS_3DVector &anrm,
                  vertex_type avtx_type = inner, RT_RS_Scene* aScene = NULL) :
                  RT_RS_Vtx(apnt, anrm, avtx_type, aScene) {} 
  RT_RS_HyDiffVtx(RT_RS_HyDiffVtx& v) : RT_RS_Vtx(v), d_ush_ins(v.d_ush_ins) {}
    
  //in/out functions
  boolean read(char *filename, FILE *fp = NULL);  // Read from ASCII file.
  boolean write(char *filename, FILE *fp = NULL); // Write to ASCII file.
  void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};                                                                                        


//a vertex of a non-diffuse patch
class RT_RS_HyInsVtx : public RT_RS_HyDiffVtx
{
  void alloch(int r_nr = 0, int g_nr = 0, int b_nr = 0)
    { i_ush_ins.alloch(r_nr, g_nr, b_nr); }
  void freeh() { i_ush_ins.freeh(); }
public:
  RT_RS_3DVector tgn;                  //the point's tangent
  RT_RS_InsRGB i_ush_ins;           //accumulated and unshot intensity

  char* get_class() { return "RS_HyInsVtx"; }

  RT_RS_HyInsVtx(RT_RS_Scene* aScene, char *filename, FILE *fp = NULL)
    : RT_RS_HyDiffVtx(aScene) { if(!read(filename, fp)) freeh(); }
  RT_RS_HyInsVtx(RT_RS_InsVtx& ivtx) : RT_RS_HyDiffVtx(ivtx.pnt, ivtx.nrm), tgn(ivtx.tgn),
                  i_ush_ins(ivtx.i_ush_ins) { 
                    ar = ivtx.get_ar(); d_ush_ins = ivtx.d_ush_ins; i_ush_ins.set_temp('n');
                  }
  RT_RS_HyInsVtx(RT_RS_Scene* aScene = NULL) : RT_RS_HyDiffVtx(aScene) {}

  //in/out functions
  boolean read(char *filename, FILE *fp = NULL); // Read from ASCII file.
  boolean write(char *filename, FILE *fp = NULL); // Write to ASCII file.
  void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};



//a volume list element
class RT_RS_Voxel : public RT_RS_IO {
  void freeh();
public:
  RT_RS_Voxel* next;  //next element in list
  long vidx[4];       //defines a patch
  char* get_class() { return "RS_Voxel"; }

  RT_RS_Voxel(long* avidx);
  virtual ~RT_RS_Voxel() { freeh(); }

  void append(long* avidx);  //append a new list element

  //in/out functions
  boolean read(char *, FILE * = NULL) { return FALSE; }
  boolean write(char *, FILE * = NULL) { return FALSE; }
  void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};

//the spatial partitioning structure
class RT_RS_SpatPar : public RT_RS_IO {
  int dim;                //dim*dim*dim volume elements
  RT_RS_Voxel ****Space;  //3D-array of pointers to RT_RS_Voxel (lists)
public:
  RT_RS_3DVector bnd_p1, bnd_p2, d_bnd, rd_bnd; //bounding box and increments

  char* get_class() { return "RS_SpatPar"; }

  RT_RS_SpatPar(int dim);   //create a new structure with Dimension dim in each direction
  virtual ~RT_RS_SpatPar();

  double get_volume(); //return the volume of the bounding-box
  void insert(long* vidx, RT_RS_3DVector* pnt); //insert a patch defined by vidx, pnt
  RT_RS_Voxel* get(int* vox);      // get the list associated with voxel vox
  void get_voxidx(const RT_RS_3DVector& v, int* vox); //get the (3D) voxel-index of point v

  //in/out functions
  boolean read(char *, FILE * = NULL) { return FALSE; }
  boolean write(char *, FILE * = NULL) { return FALSE; }
  void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};


extern RT_RS_Vtx rt_PointsVtx;
extern RT_RS_3DVector rt_PointsPRP; 

//contains the vertices of the scene
class RT_RS_Points : public RT_RS_IO
{
protected:
  RT_RS_Scene* scene;    //points to the scene
  long dcnt, icnt, Historydcnt, Historyicnt;
  int dspns, ispns, Historydspns, Historyispns;

  RT_RS_DiffVtx** dPnts[1024]; //the diffuse points of the scene
  RT_RS_InsVtx** iPnts[1024];  //the non-diffuse points of the scene

  RT_RS_HyDiffVtx** HistorydPnts[1024]; //the diffuse history points of the scene
  RT_RS_HyInsVtx** HistoryiPnts[1024];  //the non-diffuse history points of the scene

  RT_RS_DiffVtx* cur_src_dvtx; //current shooting diffuse vertex
  RT_RS_InsVtx* cur_src_ivtx;  //current shooting non-diffuse vertex

  void dallc();        //allocate memory for diffuse vertices
  void Historydallc(); //allocate memory for diffuse history vertices
  void iallc();        //allocate memory for non-diffuse vertices
  void Historyiallc(); //allocate memory for non-diffuse history vertices
  void freeh();        //free heap
public:
#ifdef RS_SHOW
  long  cur_src_didx, cur_src_iidx;
#endif
  int shoots, spar_dim, nr_shoots;
  boolean cur_eval_ins, fast;
  float max_unshoot,
        fast_max_unshoot,
        fast_max_area,
        view_scale,
        max_scale;
  RT_RS_SpatPar* SpatPar;  //the spatial partitioning structure

  char* get_class() { return "RS_Points"; }

  //constructors
  RT_RS_Points(RT_RS_Scene* ascene, char *filename, FILE *fp = NULL) :
    scene(ascene), dcnt(0), icnt(0), dspns(0), ispns(0), shoots(0),
    cur_eval_ins(TRUE), cur_src_dvtx(NULL), cur_src_ivtx(NULL)
    {  if(!read(filename, fp)) freeh(); }
  RT_RS_Points(RT_RS_Scene* ascene);
  //destructor
  virtual ~RT_RS_Points() { freeh(); }

  void init_spatpar();  //initialize spatial partitioning

  long get_diffcnt() { return dcnt; }//returns the number of diffuse vertices
  long get_inscnt() { return icnt; } //returns the number of non-diffuse vertices

  void insert(RT_RS_DiffVtx* a, RT_RS_DiffRGB* a_diff = NULL); //insert a diffuse vertex
  void insertHy(RT_RS_HyDiffVtx* a); //insert a diffuse history vertex
  void insertHy(RT_RS_HyInsVtx* a); //insert a diffuse history vertex                      
    
  void insert(RT_RS_InsVtx* a, RT_RS_brdfRGB* ai_p = NULL,
    RT_RS_DiffRGB* a_diff = NULL, RT_RS_DiffRGB* a_ins = NULL);//insert a non-diffuse vertex
  void insert_ip(int n, long* vtxidx, RT_RS_DiffVtx*& new_vtx,
                 long& new_vtxidx); //interpolate and insert a new diff. vertex from n (vtxidx) vertices
  void insert_ip(int n, long* vtxidx, RT_RS_InsVtx*& new_vtx,
                 long& new_vtxidx); //interpolate and insert a new non-diff. vertex from n (vtxidx) vertices
  long get_idx(RT_RS_DiffVtx* dvtx); //get index of diffuse vertex dvtx
  long get_idx(RT_RS_InsVtx* ivtx);  //get index of non-diffuse vertex ivtx

  void get(long n, RT_RS_DiffVtx*& dvtx);     //get the diffuse vertex at index n
  void getHy(long n, RT_RS_HyDiffVtx*& Historydvtx);     //get the diffuse history vertex at index n

  void get(long n, RT_RS_InsVtx*& ivtx);      //get the non-diffuse vertex at index n
  void getHy(long n, RT_RS_HyInsVtx*& Historyivtx);     //get the non-diffuse history vertex at
  
  int get_neighbors_of_vtx(long vn, RT_RS_DiffVtx** neighbors); //get the neighbor-vertices of diff. vertex vn  
  int get_neighbors_of_vtx(long vn, RT_RS_InsVtx** neighbors); //get the neighbor-vertices of non-diff. vertex vn

  RT_RS_DiffVtx* get_diff(long n) { RT_RS_DiffVtx* dv; get(n, dv); return dv; }
  RT_RS_HyDiffVtx* get_Hydiff(long n) { RT_RS_HyDiffVtx* dv; getHy(n, dv); return dv; }
  
  RT_RS_InsVtx* get_ins(long n) { RT_RS_InsVtx* iv; get(n, iv); return iv; }
  RT_RS_HyInsVtx* get_Hyins(long n) { RT_RS_HyInsVtx* dv; getHy(n, dv); return dv; }
    
  void set_ar0();   //set all "points-areas" 0

  void shoot(); //the shooting loop
  //propagate the accumulated energy of a diffuse patch into the environment
  //receiving diffuse vertices are start_didx...end_didx
  //receiving non-diffuse vertices are start_iidx...end_iidx
  void shoot_diff(long start_didx,
    long end_didx, long start_iidx, long end_iidx, boolean forward = TRUE);
  //propagate the accumulated energy of a non-diffuse patch into the environment
  //receiving diffuse vertices are start_didx...end_didx
  //receiving non-diffuse vertices are start_iidx...end_iidx
   void shoot_ins(long start_didx,
    long end_didx, long start_iidx, long end_iidx, boolean forward = TRUE);
  RT_RS_DiffVtx* get_maxd(float& ins_max, long& idx); //return diffuse vertex with max. unshoot value
  RT_RS_InsVtx* get_maxi(float& ins_max, long& idx); //return max. non-diffuse unshoot (average) value
  float comp_dff(RT_RS_Vtx* vtx1, RT_RS_Vtx* vtx2);  //compute form-factor
  boolean is_occluded(RT_RS_Vtx* vtx1, RT_RS_Vtx* vtx2);  //occlusion test
  boolean is_visible(RT_RS_Vtx* src_vtx, RT_RS_Vtx* rec_vtx,
    float& ctheta_out, float& ctheta_in, boolean do_occ_test = TRUE); //visibility test

  void reset_vis(); //set visibility of all vertices to unknown
  void scale_view_vals(); //scale viewing values of the vertices
  void comp_max_scale();  //compute max. scale bound
  //get max. viewing value of all visible vertices
  float get_viewmax(float max_val = BIGFLOAT, RT_RS_Vtx& view_dir = rt_PointsVtx,
    RT_RS_3DVector& PRP = rt_PointsPRP, boolean eval_ins = FALSE);

  //in/out functions
  boolean read(char *filename, FILE *fp = NULL); // Read from ASCII file.
  boolean write(char *filename, FILE *fp = NULL); // Write to ASCII file.
  void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};

#endif

